/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the constants used when utilizing the fuel
 * database file
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef FUEL_DB_CONSTANTS_H_
#define FUEL_DB_CONSTANTS_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "sms_api.h"
#include "location_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* Database Table Names */
#define VERSION_TABLE_NAME "Version"
#define LOGO_TABLE_NAME "Logo"
#define TEXT_TABLE_NAME "Text"
#define STATION_TABLE_NAME "Stations"
#define PRICE_TABLE_NAME "Prices"
#define REGION_TABLE_NAME "Regions"
#define STATE_TABLE_NAME "States"

/* Fixed-Point configuration */
#define FUEL_FIXED_CONSTANT (1 << LOCATION_BINPOINT)

/* Database Commands / Queries */

// Version Table

// Select all from version
#define FUEL_SELECT_ALL_DB_VERSION \
    "select * from "VERSION_TABLE_NAME";"

// Update version row
#define FUEL_UPDATE_DB_VERSION_ROW \
    "update "VERSION_TABLE_NAME \
    " set version=%11i " \
    "where type=%3u;"

// Text Table

// Select all text rows
#define FUEL_SELECT_ALL_TEXT \
    "select * from "TEXT_TABLE_NAME";"

// Clear entire table
#define FUEL_CLEAR_TEXT_TABLE \
    "delete from "TEXT_TABLE_NAME";"

// Region Table
#define FUEL_INSERT_REGION_ROW \
    "insert into "REGION_TABLE_NAME" " \
    "values (?);"

#define FUEL_SELECT_ALL_REGIONS \
    "select * from "REGION_TABLE_NAME";"

#define FUEL_SELECT_REGION \
    "select count(1) from "REGION_TABLE_NAME" " \
    "where ID = %10u;"

// Station Table

// Create a new table
#define FUEL_CREATE_STATION_TABLE            \
    "create table "STATION_TABLE_NAME"%u ( " \
    "StationID int,brand varchar,"           \
    "name varchar,addr varchar,"             \
    "city varchar,state int,"                \
    "ZIP varchar, phone int,"                \
    "lat int, lon int,"                      \
    "amenities int,"                         \
    "primary key (StationID) );"

// Insert / Replace a row
#define FUEL_INSERT_REPLACE_STATION_ROW_HEADER \
    "insert or replace into "STATION_TABLE_NAME"%u" \
    " (StationID, brand, name, addr, " \
    "city, state, ZIP, phone, lat, lon, amenities) " \
    "values ( "
#define FUEL_INSERT_REPLACE_STATION_ROW_BEGIN \
    FUEL_INSERT_REPLACE_STATION_ROW_HEADER \
    "%10u, \'"
#define FUEL_INSERT_REPLACE_STATION_ROW_LEN (236)
#define FUEL_UPDATE_ROW_BEGIN \
    "update "STATION_TABLE_NAME"%u set "
#define FUEL_UPDATE_BRAND \
    " brand=\'"
#define FUEL_UPDATE_NAME \
    " name=\'"
#define FUEL_UPDATE_ADDR \
    " addr=\'"
#define FUEL_UPDATE_CITY \
    " city=\'"
#define FUEL_UPDATE_STATE \
    " state=%d"
#define FUEL_UPDATE_ZIP \
    " ZIP=\'"
#define FUEL_UPDATE_PHONE \
    " phone=%lld"
#define FUEL_UPDATE_LOC \
    " lat=%d, lon=%d"
#define FUEL_UPDATE_AMENITIES \
    " amenities=%u"
#define FUEL_UPDATE_ROW_END \
    " where StationID=%10u;"

// Delete a fuel station
#define FUEL_DEL_STATION_DESC \
    "delete from "STATION_TABLE_NAME"%u " \
    "where StationID=%5u;"

#define FUEL_SELECT_STATION_DESC \
    "select * from "STATION_TABLE_NAME"%u as Stations " \
    "where Stations.StationID=%5u;"

// Select fuel stations by location -- ensure
// there is recent data in the price table for these entries,
// order them by their distance from a central point
// and limit to the first x resultant rows
#define FUEL_SELECT_STATION_DESC_BY_LOCATION \
    "select * from "STATION_TABLE_NAME"%u as Stations where " \
    "Stations.lat between %d and %d and " \
    "Stations.lon between %d and %d;"

#define FUEL_SELECT_STATION_EXISTS \
    "select count(1) from "STATION_TABLE_NAME"%u as Stations " \
    "where Stations.StationID=%5u;"

// Price Table

#define PRICE_INSERT_UPDATE_PRICE_ROW_LEN (230)

#define FUEL_FLUSH_PRICE_ROWS \
    "delete from "PRICE_TABLE_NAME";"

#define FUEL_AGEOUT_PRICE_ROWS \
    "delete from "PRICE_TABLE_NAME" where priceAgeInEpochSeconds<=%10u;"

#define FUEL_SELECT_PRICES_STATIONID \
    "select * from "PRICE_TABLE_NAME" where " \
    "Prices.priceAgeInEpochSeconds > %10u and " \
    "regionID = %d and StationID = %d"

#define FUEL_SELECT_PRICES \
    "select * from "PRICE_TABLE_NAME" where " \
    "priceAgeInEpochSeconds > %10u;"

#define FUEL_CREATE_VERSION_TABLE \
    "create table "VERSION_TABLE_NAME" ( " \
        "type tinyint, " \
        "version int  " \
        " ); "

#define FUEL_INSERT_VERSION_ROW \
    "INSERT INTO "VERSION_TABLE_NAME" VALUES(%d, %d);"

// Constant used to size buffers used to perform
// DB queries.  Size is based on largest string
// size (all defines above specify field lengths)
#define FUEL_MAX_SQL_STRING_LENGTH (340)

/* General Definitions */
#define FUEL_AMENITIES_ARRAY_SIZE    (8)
#define FUEL_REFUELING_TYPES_ARRAY_SIZE (8)
#define FUEL_INVALID_BRAND_ID        (255)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef N16 FUEL_REGION;
#define FUEL_INVALID_REGION (FUEL_REGION)-1
typedef N32 FUEL_STATION_ID;
#define FUEL_INVALID_STATION_ID (FUEL_STATION_ID)-1

// Enumeration specifying all the available fields
// in the region table
typedef enum fuel_db_region_fields_enum
{
    FUEL_DB_REGION_ID = 0,
    FUEL_DB_REGION_MAX_FIELDS

} FUEL_DB_REGION_FIELDS_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Structure which specifies all attributes
// of a single database logo row that is
// available within the fuel database
typedef struct fuel_logo_row_struct
{
    UN16 un16LogoId;
    FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType;
    UN8 un8LogoTableVer;
    UN8 un8LogoVer;
    BOOLEAN bIsBaselineLogo;

} FUEL_LOGO_ROW_STRUCT;

// Structure which specifies all attributes
// of a single database text row that is
// available within the fuel database
typedef struct fuel_text_row_struct
{
    UN8 un8TextId;      // The ID of this text row
    BOOLEAN bBrandText; // A flag indicating if this is brand text
    STRING_OBJECT hText;// The text itself (via a STRING handle)
    STRING_OBJECT hLongText; // A longer version of the text

} FUEL_TEXT_ROW_STRUCT;

// Structure which describes a permanent refueling
// position for a fuel station
typedef struct fuel_refueling_type_struct
{
    UN8 un8NumPositions;
    UN8 un8FuelType;
    FUEL_TYPE_ENUM eFuelType;
    STRING_OBJECT hLongFuelName;
    STRING_OBJECT hShortFuelName;

} FUEL_REFUELING_TYPE_STRUCT;

// Structure which specifies all attributes
// of a single database station row that is
// available within the fuel database
typedef struct fuel_station_row_struct
{
    FUEL_REGION tRegion;
    FUEL_STATION_ID tStationId;
    STRING_OBJECT hDesc;
    STRING_OBJECT hBrand;
    STRING_OBJECT hName;
    STRING_OBJECT hAddr;
    STRING_OBJECT hCity;
    STATE_ID tStateID;
    STRING_OBJECT hZIP;
    N64 n64Phone;
    N16 n16LogoId;
    N32 n32Lat;
    N32 n32Lon;
    // Previous row coordinates for handling LOCtype=0
    N32 n32PrevLat;
    N32 n32PrevLon;

    UN32 un32Amenities;

    // The number of known permanent refueling types
    UN8 un8NumPermanentTypes;
    FUEL_REFUELING_TYPE_STRUCT *psTypes;
    // Boolean flags for processing station updates (for EV Charging only)
    BOOLEAN bUpdateTypes;
    BOOLEAN bUpdateNumbers;

} FUEL_STATION_ROW_STRUCT;

// Structure which specifies all attributes
// of a single price entry in the
// price list (blob in the DB)
typedef struct fuel_price_entry_struct
{
    UN8 un8FuelType;
    FUEL_TYPE_ENUM eFuelType;
    STRING_OBJECT hFuelType;
    FUEL_AVAILABILITY_ENUM eAvailable;
    UN32 un32FuelPrice;
    UN32 un32PriceAgeUTCSeconds;

} FUEL_PRICE_ENTRY_STRUCT;

// Structure which specifies all attributes
// of a single database price row that is
// available within the fuel database that
// is not already described in the station row
typedef struct fuel_price_row_struct
{
    size_t tNumPrices;
    FUEL_PRICE_ENTRY_STRUCT *pasPrices;
    UN32 un32YoungestPriceAgeUTCSeconds;

} FUEL_PRICE_ROW_STRUCT;

// Structure used to pair station & price
// row data to aid in the update of the
// price table
typedef struct fuel_db_update_price_table_struct
{
    FUEL_STATION_ROW_STRUCT sStationRow;
    UN8 aun8RawRefuelingTypeData[FUEL_REFUELING_TYPES_ARRAY_SIZE];
    FUEL_PRICE_ROW_STRUCT sPriceRow;

} FUEL_DB_UPDATE_PRICE_TABLE_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif // FUEL_DB_CONSTANTS_H_
