/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:PROGRAM implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <standard.h>
#include <string.h>

#include "osal.h"
#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "sms_api_debug.h"

#include "epg_mgr_obj.h"
#include "_eprogram_obj.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
 *
 * hChannel
 *
 *******************************************************************************/
static SERVICE_ID tChannelSID (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return SERVICE_INVALID_ID;
    }

    return psEProg->tServiceId;
}

/*******************************************************************************
 *
 * tSeriesId
 *
 *******************************************************************************/
static SERIES_ID tSeriesId (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return SERIES_INVALID_ID;
    }

    return psEProg->tSeriesId;
}

/*******************************************************************************
 *
 * tProgramId
 *
 *******************************************************************************/
static PROGRAM_ID tProgramId (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return PROGRAM_INVALID_ID;
    }

    return psEProg->tProgramId;
}

/*******************************************************************************
 *
 * un32StartTime
 *
 *******************************************************************************/
static TIME_T tStartTime (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return 0;
    }

    return psEProg->tStartTime;
}

/*******************************************************************************
 *
 * un32EndTime
 *
 *******************************************************************************/
static TIME_T tEndTime (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return 0;
    }

    return psEProg->tEndTime;
}

/*******************************************************************************
 *
 * un8SchSeg
 *
 *******************************************************************************/
static PROGRAM_DAY_ID tEventDay (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    PROGRAM_DAY_ID tDay = PROGRAM_INVALID_DAY_ID;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == TRUE )
    {
        if( psEProg->tSchSeg >= PROGRAM_DAY_DAY0 && psEProg->tSchSeg <= PROGRAM_DAY_MAX )
        {
            tDay = (PROGRAM_DAY_ID)psEProg->tSchSeg;
        }
    }

    return tDay;
}

/*******************************************************************************
 *
 * tPFlags
 *
 *******************************************************************************/
static PROGRAM_FLAGS tPFlags (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return EPG_PROGRAM_UNDEFINED;
    }

    return psEProg->tPFlags;
}

/*******************************************************************************
 *
 * bGetShortName
 *
 *******************************************************************************/
static BOOLEAN bGetShortName (
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT hShortName
            )
{
    BOOLEAN bResult;
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;

    // Verify SMS Objects
    bResult = (SMSO_bValid((SMS_OBJECT) hEpgEvent) == TRUE &&
               SMSO_bValid((SMS_OBJECT) hShortName) == TRUE);
    if (bResult == TRUE)
    {
            bResult = bLoadString(hEpgEvent,
                                  (UN8)psEProg->tSchSeg,
                                  psEProg->un32PNameShortIdx,
                                  hShortName);
    }

    return bResult;
}

/*******************************************************************************
 *
 * bGetLongName
 *
 *******************************************************************************/
static BOOLEAN bGetLongName (
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT hLongName
            )
{
    BOOLEAN bResult;
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;

    // Verify SMS Objects
    bResult = (SMSO_bValid((SMS_OBJECT) hEpgEvent) == TRUE &&
               SMSO_bValid((SMS_OBJECT) hLongName) == TRUE);
    if (bResult == TRUE)
    {
        bResult = bLoadString(hEpgEvent,
                              (UN8)psEProg->tSchSeg,
                              psEProg->un32PNameLongIdx,
                              hLongName);
    }

    return bResult;
}

/*******************************************************************************
 *
 * bGetSeriesDescription
 *
 *******************************************************************************/
static BOOLEAN bGetSeriesDescription (
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT hSeriesDescription
            )
{
    BOOLEAN bResult;
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;

    // Verify SMS Objects
    bResult = (SMSO_bValid((SMS_OBJECT) hEpgEvent) == TRUE &&
               SMSO_bValid((SMS_OBJECT) hSeriesDescription) == TRUE);
    if (bResult == TRUE)
    {
            bResult = bLoadString(hEpgEvent,
                                  (UN8)psEProg->tSchSeg,
                                  psEProg->un32SeriesDescIdx,
                                  hSeriesDescription);
    }

    return bResult;
}

/*******************************************************************************
 *
 * bGetProgramDescription
 *
 *******************************************************************************/
static BOOLEAN bGetProgramDescription (
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT hProgramDescription
            )
{
    BOOLEAN bResult;
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;

    // Verify SMS Objects
    bResult = (SMSO_bValid((SMS_OBJECT) hEpgEvent) == TRUE &&
               SMSO_bValid((SMS_OBJECT) hProgramDescription) == TRUE);
    if (bResult == TRUE)
    {
        bResult = bLoadString(hEpgEvent,
                              (UN8)psEProg->tSchSegProgramDescription,
                              psEProg->un32ProgramDescIdx,
                              hProgramDescription);
    }

    return bResult;
}

/*******************************************************************************
 *
 * un16OrigDate
 *
 *******************************************************************************/
static UN16 un16OrigDate (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return 0;
    }

    return psEProg->un16OrigDate;
}

/*******************************************************************************
 *
 * hEpgTopics
 *
 *******************************************************************************/
static EPG_TOPICS_LIST hEpgTopics (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return EPG_INVALID_TOPICS_LIST_OBJECT;
    }

    if( psEProg->hEpgTopics != EPG_INVALID_TOPICS_LIST_OBJECT )
    {
        return psEProg->hEpgTopics;
    }

    return EPG_INVALID_TOPICS_LIST_OBJECT;
}

/*******************************************************************************
 *
 * bPrintEpgTopic
 *
 *******************************************************************************/
static BOOLEAN bPrintEpgTopic (
        void *pvData,
        void *pvArg
            )
{
    TOPIC_OBJECT hTopic = (TOPIC_OBJECT) pvData;
    PRINTF_STRUCT *psPrintf = (PRINTF_STRUCT*) pvArg;
    N32 n32Count = 0;

    if( pvData == NULL )
    {
        return FALSE;
    }

    n32Count = TOPIC.n32FPrintf(hTopic, psPrintf->psFile);
    if (n32Count > 0)
    {
        psPrintf->n32CCount += n32Count;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   n32FPrintf
 *
 *****************************************************************************/
static N32 n32FPrintf (
        PROGRAM_OBJECT hEpgEvent,
        FILE *psFile
            )
{
    N32 n32Return = 0;
    PROG_EVENT_STRUCT *psObj = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid, bResult;
    char acTmpStrBuf[EPG_LONG_DESCRIPTION_LEN];
    STRING_OBJECT hString = STRING_INVALID_OBJECT;

    // Determine if the handle is valid
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if (( bValid == FALSE ) || ( psFile == NULL ))
    {
        return EOF;
    }

    // Print EPG event
    n32Return += fprintf(psFile, "\nEPG Event:\n\thEpgEvent = %p\n", psObj);

    n32Return += fprintf(psFile,
                "\tServiceId = %d\n",
                psObj->tServiceId);

    n32Return += fprintf(psFile,
                "\tSeries ID = %d, Program ID = %d\n",
                psObj->tSeriesId,
                psObj->tProgramId);

    n32Return += fprintf(psFile, "\tSegment number = %d\n", psObj->tSchSeg);
    n32Return += fprintf(psFile, "\tProgram Description segment number = %d\n", psObj->tSchSegProgramDescription);

    OSAL.ctime_r((TIME_T*) &psObj->tStartTime, acTmpStrBuf);
    n32Return += fprintf(psFile, "\tShow time [%s]", acTmpStrBuf);

    OSAL.ctime_r((TIME_T*) &psObj->tEndTime, acTmpStrBuf);
    n32Return += fprintf(psFile, " - [%s]\n", acTmpStrBuf);

    n32Return += fprintf(psFile, "\tFlags: ");

    if( psObj->tPFlags & EPG_PROGRAM_FEATURED )
    {
        n32Return += fprintf(psFile, " 'FEATURED' ");
    }

    if( psObj->tPFlags & EPG_PROGRAM_HIGHLIGHTED )
    {
        n32Return += fprintf(psFile, " 'HIGHLITED' ");
    }

    if( psObj->tPFlags & EPG_PROGRAM_LIVE )
    {
        n32Return += fprintf(psFile, " 'LIVE' ");
    }

    if( psObj->tPFlags & EPG_PROGRAM_NEW_THIS )
    {
        n32Return += fprintf(psFile, " 'NEW' ");
    }

    n32Return += fprintf(psFile, "\n\tRecording Option: ");

    switch(psObj->eRecordingOption)
    {
        case EPG_PROGRAM_RECORD_NONE:
        {
            n32Return += fprintf(psFile, " 'NO RECORDING' ");
        }
        break;
        case EPG_PROGRAM_RECORD_APPROVED:
        {
            n32Return += fprintf(psFile, " 'APPROVED RECORDING' ");
        }
        break;
        case EPG_PROGRAM_RECORD_RESTRICTED:
        {
            n32Return += fprintf(psFile, " 'RESTRICTED RECORDING' ");
        }
        break;
        case EPG_PROGRAM_RECORD_ALWAYS:
        {
            n32Return += fprintf(psFile, " 'ALWAYS RECORDABLE' ");
        }
        break;
        default:
        {
            n32Return += fprintf(psFile, " !  I N V A L I D  ! ");
        }
        break;
    }

    n32Return += fprintf(psFile, "\n");
    hString = STRING.hCreate("",0);

    bResult= bGetShortName(hEpgEvent, hString);
    if( bResult == TRUE )
    {
        n32Return += fprintf(psFile,
                    "\tShort name: %s\n",
                    STRING.pacCStr(hString));
    }

    bResult = bGetLongName(hEpgEvent, hString);
    if( bResult == TRUE )
    {
        n32Return += fprintf(psFile,
                    "\tLong name: %s\n",
                    STRING.pacCStr(hString));
    }

    bResult = bGetSeriesDescription(hEpgEvent, hString);
    if( bResult == TRUE )
    {
        n32Return += fprintf(psFile,
                    "\tSeries description: %s\n",
                    STRING.pacCStr(hString));
    }

    bResult = bGetProgramDescription(hEpgEvent, hString);
    if( bResult == TRUE )
    {
        n32Return += fprintf(psFile,
                    "\tProgram description: %s\n",
                    STRING.pacCStr(hString));
    }

    STRING_vDestroy( hString );

    if( psObj->hEpgTopics != EPG_INVALID_TOPICS_LIST_OBJECT )
    {
        SMSAPI_RETURN_CODE_ENUM eRet;
        PRINTF_STRUCT sPrintf;
        sPrintf.n32CCount = 0;
        sPrintf.psFile = psFile;
        n32Return += fprintf(psFile, "\tAssigned topics:\n");

        eRet = PROGRAM.eIterateTopics(psObj->hEpgTopics,
                    (EPG_TOPIC_ITERATOR_CALLBACK) bPrintEpgTopic,
                    &sPrintf);
        if( eRet != SMSAPI_RETURN_CODE_SUCCESS )
        {
            n32Return += fprintf(psFile,
                        "Topic list iterator error %d.\n",
                        eRet);
        }
        else
        {
            n32Return += sPrintf.n32CCount;
        }
    }
    else
    {
        n32Return += fprintf(psFile, "\tNo topics.\n");
    }

    return n32Return;
}

/*****************************************************************************
 *
 *  eIterateTopics
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateTopics (
        EPG_TOPICS_LIST hEpgTopics,
        EPG_TOPIC_ITERATOR_CALLBACK bEpgTopicIterator,
        void *pvIteratorArg
            )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    bOwner = SMSO_bOwner((SMS_OBJECT) hEpgTopics);
    if( bOwner == TRUE )
    {
        EPG_TOPICS_LIST_STRUCT *psObj = (EPG_TOPICS_LIST_STRUCT *) hEpgTopics;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR_UNKNOWN;

        if( psObj == NULL )
        {
            return eReturnCode;
        }

        // Populate a local iterator structure with info needed to call
        // the provided callback to the caller along with their provided argument.

        // Iterate the list and execute caller's callback function for
        // each entry in the list.
        eOsalReturnCode = OSAL.eLinkedListIterate(psObj->hTopicsList,
                (OSAL_LL_ITERATOR_HANDLER) bEpgTopicIterator,
                pvIteratorArg);

        if( (eOsalReturnCode == OSAL_SUCCESS) || (eOsalReturnCode
                == OSAL_NO_OBJECTS) )
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            printf("EPG topics iterate: error code = %d \n", eOsalReturnCode);
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *  eRecordingOption
 *
 *****************************************************************************/
static EPG_PROGRAM_RECORDING_OPTION_ENUM eRecordingOption (
        PROGRAM_OBJECT hEpgEvent
            )
{
    PROG_EVENT_STRUCT *psEProg = (PROG_EVENT_STRUCT *) hEpgEvent;
    BOOLEAN bValid;

    // Verify SMS Object
    bValid = SMSO_bValid((SMS_OBJECT) hEpgEvent);
    if( bValid == FALSE )
    {
        return EPG_PROGRAM_RECORD_UNDEFINED;
    }

    return psEProg->eRecordingOption;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/* ------------------- PROGRAM object related functions -------------------- */

/*****************************************************************************
 *
 *  EPG_PROGRAM_psCreate
 *
 *****************************************************************************/
PROG_EVENT_STRUCT *EPG_PROGRAM_psCreate (
        SMS_OBJECT hEpgChannel
            )
{
    PROG_EVENT_STRUCT *psProgramEvent = NULL;

    psProgramEvent = (PROG_EVENT_STRUCT*) SMSO_hCreate(EPG_PROGRAM_EVENT_OBJECT_NAME,
                                                       sizeof(PROG_EVENT_STRUCT),
                                                       hEpgChannel,
                                                       FALSE);
    if ((SMS_OBJECT) psProgramEvent == SMS_INVALID_OBJECT)
    {
        psProgramEvent = NULL;
    }
    else
    {
        // Set default values
        psProgramEvent->tSchSeg = PROGRAM_INVALID_DAY_ID;
        psProgramEvent->tSchSegProgramDescription = PROGRAM_INVALID_DAY_ID;
        psProgramEvent->tServiceId = SERVICE_INVALID_ID;
        psProgramEvent->tSeriesId = SERIES_INVALID_ID;
        psProgramEvent->tProgramId = PROGRAM_INVALID_ID;
        psProgramEvent->tPFlags = EPG_PROGRAM_INVALID_FLAGS;
        psProgramEvent->eRecordingOption = EPG_PROGRAM_RECORD_UNDEFINED;
        psProgramEvent->tStartTime = 0;
        psProgramEvent->tEndTime = 0;
        psProgramEvent->un32PNameShortIdx = UN32_MAX;
        psProgramEvent->un32PNameLongIdx = UN32_MAX;
        psProgramEvent->un32SeriesDescIdx = UN32_MAX;
        psProgramEvent->un32ProgramDescIdx = UN32_MAX;
        psProgramEvent->un16OrigDate = UN16_MAX;
        psProgramEvent->hEpgTopics = EPG_INVALID_TOPICS_LIST_OBJECT;
        psProgramEvent->hEpgChannel = hEpgChannel;
    }

    return psProgramEvent;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_psDuplicate
 *
 *****************************************************************************/
PROG_EVENT_STRUCT *EPG_PROGRAM_psDuplicate (
        PROG_EVENT_STRUCT *psProgramEvent,
        SMS_OBJECT hEpgChannel,
        TIME_T tNewStartTime
            )
{
    PROG_EVENT_STRUCT *psNewProgramEvent = NULL;
    EPG_TOPICS_LIST_STRUCT *psNewTopicsList = NULL;

    do
    {
        // Create empty (invalid) event
        psNewProgramEvent = EPG_PROGRAM_psCreate(hEpgChannel);
        if (psNewProgramEvent == NULL)
        {
            break;
        }

        // Copy common fields
        psNewProgramEvent->tSchSeg = psProgramEvent->tSchSeg;
        psNewProgramEvent->tSchSegProgramDescription = psProgramEvent->tSchSegProgramDescription;
        psNewProgramEvent->tServiceId = psProgramEvent->tServiceId;
        psNewProgramEvent->tSeriesId = psProgramEvent->tSeriesId;
        psNewProgramEvent->tProgramId = psProgramEvent->tProgramId;
        psNewProgramEvent->tPFlags = psProgramEvent->tPFlags;
        psNewProgramEvent->eRecordingOption = psProgramEvent->eRecordingOption;
        psNewProgramEvent->un32PNameShortIdx = psProgramEvent->un32PNameShortIdx;
        psNewProgramEvent->un32PNameLongIdx = psProgramEvent->un32PNameLongIdx;
        psNewProgramEvent->un32SeriesDescIdx = psProgramEvent->un32SeriesDescIdx;
        psNewProgramEvent->un32ProgramDescIdx = psProgramEvent->un32ProgramDescIdx;
        psNewProgramEvent->un16OrigDate = psProgramEvent->un16OrigDate;

        // Set new time
        psNewProgramEvent->tStartTime = tNewStartTime;
        psNewProgramEvent->tEndTime =
            tNewStartTime + (psProgramEvent->tEndTime - psProgramEvent->tStartTime);

        // Copy topics
        if (psProgramEvent->hEpgTopics != EPG_INVALID_TOPICS_LIST_OBJECT)
        {
            psNewTopicsList = EPG_TOPICS_LIST_psDuplicate(psProgramEvent->hEpgTopics,
                                                          (SMS_OBJECT) psNewProgramEvent);
            if (psNewTopicsList == NULL)
            {
                break;
            }

            psNewProgramEvent->hEpgTopics = (EPG_TOPICS_LIST) psNewTopicsList;
        }

        // Everything's OK
        return psNewProgramEvent;

    } while (FALSE);

    // Error case:
    EPG_PROGRAM_vDestroy(psNewProgramEvent);
    return NULL;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_vDestroy
 *
 *****************************************************************************/
void EPG_PROGRAM_vDestroy (
        PROG_EVENT_STRUCT *psProgramEvent
            )
{
    if (psProgramEvent == NULL)
    {
        return;
    }

    EPG_TOPICS_LIST_vDestroy((EPG_TOPICS_LIST_STRUCT *) psProgramEvent->hEpgTopics);

    SMSO_vDestroy((SMS_OBJECT) psProgramEvent);

    return;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_bAddTopic
 *
 *****************************************************************************/
BOOLEAN EPG_PROGRAM_bAddTopic (
        PROG_EVENT_STRUCT *psProgramEvent,
        EPG_TOPIC_STRUCT *psTopic
            )
{
    BOOLEAN bResult = FALSE;
    EPG_TOPICS_LIST_STRUCT *psTopicsList = NULL;

    do
    {
        // Check input
        if (psTopic == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid topic object.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

        // First, create Topics List object (if not created).
        if (psProgramEvent->hEpgTopics == EPG_INVALID_TOPICS_LIST_OBJECT)
        {
            psTopicsList = EPG_TOPICS_LIST_psCreate((SMS_OBJECT) psProgramEvent);
            if (psTopicsList == NULL)
            {
                bResult = FALSE;
                break;
            }

            psProgramEvent->hEpgTopics = (EPG_TOPICS_LIST) psTopicsList;
        }
        else
        {
            psTopicsList = (EPG_TOPICS_LIST_STRUCT *) psProgramEvent->hEpgTopics;
        }

        // Add Topic
        bResult = EPG_TOPICS_LIST_bAddTopic(psTopic, psTopicsList);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to add Topic to a Program Event.\n"PC_RESET);
            break;
        }

        // Topic is successfully added
        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_n16CompareEpgEventAndStartTime
 *
 *****************************************************************************/
N16 EPG_PROGRAM_n16CompareEpgEventAndStartTime (
        PROG_EVENT_STRUCT *psProgEvent,
        TIME_T *ptStartTime
            )
{
    if (psProgEvent->tStartTime < *ptStartTime)
    {
        return -1;
    }

    if (psProgEvent->tStartTime > *ptStartTime)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_n16CompareEpgEventAndEndTime
 *
 *****************************************************************************/
N16 EPG_PROGRAM_n16CompareEpgEventAndEndTime (
        PROG_EVENT_STRUCT *psProgEvent,
        TIME_T *ptEndTime
            )
{
    if (psProgEvent->tEndTime < *ptEndTime)
    {
        return -1;
    }

    if (psProgEvent->tEndTime > *ptEndTime)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_n16CompareEpgEventsByStartTime
 *
 *****************************************************************************/
N16 EPG_PROGRAM_n16CompareEpgEventsByStartTime (
        PROG_EVENT_STRUCT *psProgEvent1,
        PROG_EVENT_STRUCT *psProgEvent2
            )
{
    return EPG_PROGRAM_n16CompareEpgEventAndStartTime(psProgEvent1,
                                                      &(psProgEvent2->tStartTime));
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_n16CompareEpgEventsById
 *
 *****************************************************************************/
N16 EPG_PROGRAM_n16CompareEpgEventsById (
        PROG_EVENT_STRUCT *psProgEvent1,
        PROG_EVENT_STRUCT *psProgEvent2
            )
{
    if (psProgEvent1->tSeriesId != SERIES_INVALID_ID ||
        psProgEvent2->tSeriesId != SERIES_INVALID_ID)
    {
        // This is repeatable program with Series Id
        if (psProgEvent1->tSeriesId < psProgEvent2->tSeriesId)
        {
            return -1;
        }
        if (psProgEvent1->tSeriesId > psProgEvent2->tSeriesId)
        {
            return 1;
        }
    }
    else
    {
        // Program does not have Series Id, Program Id should be used
        if (psProgEvent1->tProgramId < psProgEvent2->tProgramId)
        {
            return -1;
        }
        if (psProgEvent1->tProgramId > psProgEvent2->tProgramId)
        {
            return 1;
        }
    }

    return 0;
}

/*****************************************************************************
 *
 *  EPG_PROGRAM_n16CompareEpgEventAndSeriesId
 *
 *****************************************************************************/
N16 EPG_PROGRAM_n16CompareEpgEventAndSeriesId (
        PROG_EVENT_STRUCT *psProgEvent,
        SERIES_ID *ptSeriesId
            )
{
    if (psProgEvent->tSeriesId < *ptSeriesId)
    {
        return -1;
    }

    if (psProgEvent->tSeriesId > *ptSeriesId)
    {
        return 1;
    }

    return 0;
}

/* ------------------- CHANNEL object related functions -------------------- */

/*****************************************************************************
 *
 *  EPG_CHANNEL_psCreate
 *
 *****************************************************************************/
EPG_CHANNEL_OBJECT_STRUCT *EPG_CHANNEL_psCreate (
        SERVICE_ID tServiceId,
        SMS_OBJECT hParent
            )
{
    EPG_CHANNEL_OBJECT_STRUCT *psChannel = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    do
    {
        psChannel =
            (EPG_CHANNEL_OBJECT_STRUCT *) SMSO_hCreate(EPG_CHANNEL_OBJECT_NAME,
                                                       sizeof(EPG_CHANNEL_OBJECT_STRUCT),
                                                       hParent,
                                                       FALSE);
        if (psChannel == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create EPG Channel.\n"PC_RESET);
            break;
        }

        // Create events list
        eReturnCode = OSAL.eLinkedListCreate(&psChannel->hEpgChannelEvents,
                                             EPG_CHANNEL_EVENTS_LL_NAME,
                                             (OSAL_LL_COMPARE_HANDLER) EPG_PROGRAM_n16CompareEpgEventsByStartTime,
                                             OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create events LL.\n"PC_RESET);
            break;
        }

        // Create unique programs list
        eReturnCode = OSAL.eLinkedListCreate(&psChannel->hUniquePrograms,
                                             EPG_CHANNEL_UNIQUE_EVENTS_LL_NAME,
                                             (OSAL_LL_COMPARE_HANDLER) EPG_PROGRAM_n16CompareEpgEventsById,
                                             OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create unique programms LL.\n"PC_RESET);
            break;
        }

        psChannel->tServiceId = tServiceId;

        // Everything's OK
        return psChannel;

    } while (FALSE);

    // Error case:
    EPG_CHANNEL_vDestroy(psChannel);
    return NULL;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_vDestroy
 *
 *****************************************************************************/
void EPG_CHANNEL_vDestroy (
        EPG_CHANNEL_OBJECT_STRUCT *psChannel
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    if (psChannel == NULL)
    {
        return;
    }

    if (psChannel->hEpgChannelEvents != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psChannel->hEpgChannelEvents,
                                                (OSAL_LL_RELEASE_HANDLER) EPG_PROGRAM_vDestroy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to remove Program Events.\n");
        }

        eReturnCode = OSAL.eLinkedListDelete(psChannel->hEpgChannelEvents);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to delete Program Events LL.\n");
        }
    }

    if (psChannel->hUniquePrograms != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psChannel->hUniquePrograms, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to remove Unique Program LL entries.\n");
        }

        eReturnCode = OSAL.eLinkedListDelete(psChannel->hUniquePrograms);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to delete Unique Program Events LL.\n");
        }
    }

    SMSO_vDestroy((SMS_OBJECT) psChannel);

    return;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_un32GetShowsCount
 *
 *****************************************************************************/
UN32 EPG_CHANNEL_un32GetShowsCount (
        EPG_CHANNEL_OBJECT_STRUCT *psChannel
            )
{
    UN32 un32ShowsCount = 0;

    OSAL.eLinkedListItems(psChannel->hEpgChannelEvents,
                          &un32ShowsCount);

    return un32ShowsCount;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_bAddProgramEventToEpgChannel
 *
 *****************************************************************************/
BOOLEAN EPG_CHANNEL_bAddProgramEventToEpgChannel (
        EPG_CHANNEL_OBJECT_STRUCT *psChannel,
        PROG_EVENT_STRUCT *psProgramEvent
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY,
        hEventEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    BOOLEAN bResult = FALSE;

    do
    {
        eReturnCode = OSAL.eLinkedListAdd(psChannel->hEpgChannelEvents,
                                          &hEventEntry,
                                          psProgramEvent);
        if (eReturnCode != OSAL_SUCCESS)
        {
            if (eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 1,
                    "WARNING: Error adding program to list.\n");
            }
            else
            {
                // Attempt to add duplicated entry (program start time is the same)
                SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 1,
                    "WARNING: Non unique record ignored.\n");
            }

            break;
        }

        eReturnCode = OSAL.eLinkedListAdd(psChannel->hUniquePrograms,
                                          &hEntry,
                                          psProgramEvent);
        if (eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
        {
            if (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                PROG_EVENT_STRUCT *psProgEventOld = NULL;

                psProgEventOld = (PROG_EVENT_STRUCT *) OSAL.pvLinkedListThis(hEntry);
                if (psProgEventOld->tStartTime < psProgramEvent->tStartTime)
                {
                    eReturnCode = OSAL.eLinkedListReplaceEntry(
                            psChannel->hUniquePrograms,
                            hEntry,
                            psProgramEvent);
                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                            PC_RED": failed to Replace the "
                            "entry in the list (%s)\n"PC_RESET,
                            OSAL.pacGetReturnCodeName(eReturnCode));
                        // The function will return FALSE here, so we need
                        // to remove previously added event entry
                        (void)OSAL.eLinkedListRemove(hEventEntry);
                        break;
                    }
                }
            }
        }

        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_bAddProgramCopy
 *
 *****************************************************************************/
BOOLEAN EPG_CHANNEL_bAddProgramCopy (
        EPG_CHANNEL_OBJECT_STRUCT *psChannel,
        PROG_EVENT_STRUCT *psProgramEvent,
        TIME_T tNewStartTime
            )
{
    PROG_EVENT_STRUCT *psNewProgramEvent = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Create new Program Event with new start time
        psNewProgramEvent = EPG_PROGRAM_psDuplicate(psProgramEvent,
                                                    (SMS_OBJECT) psChannel,
                                                    tNewStartTime);
        if (psNewProgramEvent == NULL)
        {
            break;
        }

        // Add Program Event to the Channel
        bResult = EPG_CHANNEL_bAddProgramEventToEpgChannel(psChannel,
                                                           psNewProgramEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error case:
    EPG_PROGRAM_vDestroy(psNewProgramEvent);
    return FALSE;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_n16CompareEpgChannelAndSid
 *
 *****************************************************************************/
N16 EPG_CHANNEL_n16CompareEpgChannelAndSid (
        EPG_CHANNEL_OBJECT_STRUCT *psEpgChannel,
        SERVICE_ID *ptServiceId
            )
{
    if (psEpgChannel->tServiceId < *ptServiceId)
    {
        return -1;
    }

    if (psEpgChannel->tServiceId > *ptServiceId)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
 *
 *  EPG_CHANNEL_n16CompareEpgChannelsBySid
 *
 *****************************************************************************/
N16 EPG_CHANNEL_n16CompareEpgChannelsBySid (
        EPG_CHANNEL_OBJECT_STRUCT *psChannel1,
        EPG_CHANNEL_OBJECT_STRUCT *psChannel2
            )
{
    if( psChannel1->tServiceId < psChannel2->tServiceId )
    {
        return -1;
    }
    if( psChannel1->tServiceId > psChannel2->tServiceId )
    {
        return 1;
    }
    return 0;
}

/* ------------------- TOPICS_LIST object related functions ---------------- */

/*****************************************************************************
 *
 *  EPG_TOPICS_LIST_psCreate
 *
 *****************************************************************************/
EPG_TOPICS_LIST_STRUCT *EPG_TOPICS_LIST_psCreate (
        SMS_OBJECT hParent
            )
{
    EPG_TOPICS_LIST_STRUCT *psTopicsList = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    do
    {
        psTopicsList =
            (EPG_TOPICS_LIST_STRUCT *) SMSO_hCreate(EPG_TOPICS_LIST_OBJECT_NAME,
                                                    sizeof(EPG_TOPICS_LIST_STRUCT),
                                                    hParent,
                                                    FALSE);
        if ((SMS_OBJECT) psTopicsList == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create Topics List object.\n"PC_RESET);
            psTopicsList = NULL;
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&psTopicsList->hTopicsList,
                                             EPG_TOPICS_LL_NAME,
                                             (OSAL_LL_COMPARE_HANDLER) EPG_TOPIC_n16CompareEpgTopicsByTid,
                                             OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create Topics LL.\n"PC_RESET);
            break;
        }

        psTopicsList->un8NumTopics = 0;

        // Everything's OK
        return psTopicsList;

    } while (FALSE);

    // Error case
    EPG_TOPICS_LIST_vDestroy(psTopicsList);
    return NULL;
}

/*****************************************************************************
 *
 *  EPG_TOPICS_LIST_psDuplicate
 *
 *****************************************************************************/
EPG_TOPICS_LIST_STRUCT *EPG_TOPICS_LIST_psDuplicate (
        EPG_TOPICS_LIST_STRUCT *psTopicsList,
        SMS_OBJECT hParent
            )
{
    EPG_TOPICS_LIST_STRUCT *psNewTopicsList = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    do
    {
        psNewTopicsList = EPG_TOPICS_LIST_psCreate(hParent);
        if (psNewTopicsList == NULL)
        {
            break;
        }

        if (psTopicsList->hTopicsList != OSAL_INVALID_OBJECT_HDL &&
            psTopicsList->un8NumTopics > 0)
        {
            // Copy topics
            eReturnCode = OSAL.eLinkedListIterate(psTopicsList->hTopicsList,
                                                  (OSAL_LL_ITERATOR_HANDLER) EPG_TOPICS_LIST_bAddTopic,
                                                  psNewTopicsList);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }

        // Everything's OK
        return psNewTopicsList;

    } while (FALSE);

    //Error case:
    EPG_TOPICS_LIST_vDestroy(psNewTopicsList);
    return NULL;
}

/*****************************************************************************
 *
 *  EPG_TOPICS_LIST_vDestroy
 *
 *****************************************************************************/
void EPG_TOPICS_LIST_vDestroy (
        EPG_TOPICS_LIST_STRUCT *psTopicsList
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    if (psTopicsList == NULL)
    {
        return;
    }

    if (psTopicsList->hTopicsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(psTopicsList->hTopicsList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to remove Topics LL entries.\n");
        }

        eReturnCode = OSAL.eLinkedListDelete(psTopicsList->hTopicsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 0,
                "ASSERT: Failed to delete Topics LL.\n");
        }
    }

    SMSO_vDestroy((SMS_OBJECT) psTopicsList);

    return;
}

/*****************************************************************************
 *
 *  EPG_TOPICS_LIST_bAddTopic
 *
 *****************************************************************************/
BOOLEAN EPG_TOPICS_LIST_bAddTopic (
        EPG_TOPIC_STRUCT *psTopic,
        EPG_TOPICS_LIST_STRUCT *psTopicsList
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bResult = FALSE;

    eReturnCode = OSAL.eLinkedListAdd(psTopicsList->hTopicsList,
                                      OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                      psTopic);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Topic is successfully added
        psTopicsList->un8NumTopics += 1;
        bResult = TRUE;
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_PROGRAM_DBG_PREFIX, 1,
            PC_RED" ERROR: Failed to add Topic to a Topics LL.\n"PC_RESET);
        bResult = FALSE;
    }

    return bResult;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *  bLoadString
 *
 *****************************************************************************/
static BOOLEAN bLoadString (
        PROGRAM_OBJECT hEProgEvent,
        UN8 un8SegNum,
        UN32 un32Index,
        STRING_OBJECT hText
            )
{
    BOOLEAN bResult = FALSE;

    do
    {
        PROG_EVENT_STRUCT *psObj = (PROG_EVENT_STRUCT *)hEProgEvent;

        if (un32Index == UN32_MAX)
        {
            break;
        }

        bResult = SMSO_bValid(psObj->hEpgChannel);
        if (bResult == FALSE)
        {
            break;
        }

        bResult = EPG_MGR_bLoadString(
            ((EPG_CHANNEL_OBJECT_STRUCT*)psObj->hEpgChannel)->hEpgService,
            un8SegNum,
            un32Index,
            hText);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 INLINE FUNCTIONS
 *****************************************************************************/
