/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  Electronic Program Guide (EPG) dataservice.
 *  This module contains the Object:EPG_MGR implementation for the
 *  Sirius Module Services (SMS).
 *  This module implements API to parse received EPG Access Units and
 *  build-up full EPG schedule to be used in CHANNEL_OBJECT.
 *
 ******************************************************************************/
#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_event.h"

#include "decoder_obj.h"
#include "channel_obj.h"
#include "dataservice_mgr_impl.h"
#include "ds_util.h"
#include "epg_mgr_obj.h"
#include "_epg_mgr_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *  hStartExt
 *
 *  This API function will create all the basics needed for this service to
 *  operate.  However, all initial processing to actually get this service
 *  running is done at a later time.
 *
 *****************************************************************************/
static EPG_SERVICE_OBJECT hStart (
        const char *pacSRHDriverName,
        UN8 un8NumSegsForRecieve,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvAppEventCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = NULL;
    DATASERVICE_CREATE_STRUCT sCreate;
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_GREEN" EPG.hStartExt()\n"PC_RESET);

    // Process variable arguments list options
    bResult = DATASERVICE_IMPL_bProcessOptions(EPG_SUPPORTED_OPTIONS,
                                               psOptions,
                                               &sOptionValues
                                              );
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Bad EPG service options.");

        return EPG_SERVICE_INVALID_OBJECT;
    }

    DATASERVICE_IMPL_vInitCreateStruct(&sCreate);

    // Populate data service creation structure
    sCreate.pacSRHDriverName = pacSRHDriverName;
    sCreate.pacServiceObjectName = EPG_MGR_OBJECT_NAME;
    sCreate.tServiceObjectSize = sizeof(EPG_MGR_OBJECT_STRUCT);
    sCreate.tDataID = (DATASERVICE_ID)GsEpgIntf.tDSI;
    sCreate.tSuggestedOTABufferByteSize = GsEpgIntf.tOTABufferByteSize;
    sCreate.bTimeRequired = FALSE;
    // Enable all data streams for this service
    sCreate.bEnableAll = TRUE;
    // Configure the data service's static event attributes
    sCreate.vEventCallback = vEventHandler;
    sCreate.tEventRequestMask = (DATASERVICE_EVENT_ALL
            | DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED
            | DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED
            | DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC );

    // Ask the data service manager controller to
    // create our manager object and do everything
    // necessary to create the underlying objects required
    // in order to support this service
    psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) DATASERVICE_IMPL_hCreateNewService(&sCreate);
    if (psEpgMgrObj == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to create EPG service.");

        return EPG_SERVICE_INVALID_OBJECT;
    }

    // Save maximum allowed number of segments to collect
    if (un8NumSegsForRecieve == 0)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Number of segments must not be 0.");

        vUninitEpgMgrObject(psEpgMgrObj);
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psEpgMgrObj);

        return EPG_SERVICE_INVALID_OBJECT;
    }
    else if (un8NumSegsForRecieve > GsEpgIntf.un8MaxTotalSegments)
    {
        // Set maximum number of segments
        psEpgMgrObj->un8NumSegmentsToRecieve = GsEpgIntf.un8MaxTotalSegments;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": WARNING: Number of segments is set to %u.",
            psEpgMgrObj->un8NumSegmentsToRecieve);
    }
    else
    {
        psEpgMgrObj->un8NumSegmentsToRecieve = un8NumSegsForRecieve;
    }

    // Clear stopping flag
    psEpgMgrObj->bStopping = FALSE;

    // Current EPG Schedule is initially uninitialized
    psEpgMgrObj->psEpgOwner = NULL;
    psEpgMgrObj->bInitialEpgLoaded = FALSE;

    // EPG LOADER task is initially uninitialized
    psEpgMgrObj->psEpgLoader = NULL;
    psEpgMgrObj->eEpgLoaderState = EPG_LOADER_STATE_UNINITIALIZED;

    // Initialize asynchronous update configuration
    SMSU_vInitialize(&psEpgMgrObj->sEvent,
                     psEpgMgrObj,
                     DATASERVICE_EVENT_NONE,
                     tEventRequestMask,
                     (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallback,
                     pvAppEventCallbackArg);

    // The service may now start
    bResult = DATASERVICE_IMPL_bStart((DATASERVICE_IMPL_HDL)psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to start EPG service.");

        vUninitEpgMgrObject(psEpgMgrObj);
        DATASERVICE_IMPL_vDestroy((DATASERVICE_IMPL_HDL)psEpgMgrObj);
        return EPG_SERVICE_INVALID_OBJECT;
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 1,
        PC_GREEN" EPG MGR SUCCESSFULLY CREATED.\n"PC_RESET);

    return (EPG_SERVICE_OBJECT)psEpgMgrObj;
}

/*****************************************************************************
 *
 *  vStop
 *
 *****************************************************************************/
static void vStop (
    EPG_SERVICE_OBJECT hEpgService
        )
{
    SMS_EVENT_HDL hEvent;
    EPG_MGR_EVENT_STRUCT *psStopEvent = (EPG_MGR_EVENT_STRUCT *)NULL;
    BOOLEAN bPosted = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 5,
        PC_GREEN"EPG.vStop().\n"PC_RESET);

    // Allocate an event to perform this stop
    hEvent = DATASERVICE_IMPL_hAllocateEvent(
        (DATASERVICE_IMPL_HDL)hEpgService, (void **)&psStopEvent);

    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        // Indicate this is a stop prepare event
        psStopEvent->eEvent = EPG_MGR_STOP_PREPARE_EVENT;

        // Post it now
        bPosted = SMSE_bPostEvent(hEvent);
    }

    if (bPosted == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Unable to post ePG stop event.");
    }

    return;
}

/*****************************************************************************
 *
 *  eIteratePrograms
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIteratePrograms (
        EPG_SERVICE_OBJECT hEpgService,
        PROGRAM_ITERATOR_CALLBACK bEpgIterator,
        void *pvIteratorArg,
        EPG_FILTER_STRUCT *psEpgFilter
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hEpgService;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR_UNKNOWN;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    EPG_EVENTS_ITERATOR_STRUCT sIterator;
    OSAL_OBJECT_HDL hChannelsList = OSAL_INVALID_OBJECT_HDL;
    BOOLEAN bValid;

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_GREEN"EPG.eIteratePrograms()\n"PC_RESET);

    // Check input
    bValid = DATASERVICE_IMPL_bIsValid((DATASERVICE_IMPL_HDL)hEpgService);
    if ((bValid == FALSE) || (bEpgIterator == NULL))
    {
        // Do not allow calling this API without callback provided
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eIteratePrograms(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
            PC_BLUE"EPG.eIteratePrograms(): Failed to get facing object...\n"PC_RESET);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eIteratePrograms(): --> Facing object locked.\n"PC_RESET);

    // Iterate the list and execute caller's callback function for
    // each entry in the list.
    do
    {
        // Checking if there is any schedule root available
        if (psEpgOwner->psCurrentSchedule != NULL)
        {
            // Check if at least one segment is available
            if (psEpgOwner->psCurrentSchedule->un8NumSegmentsCollected > 0)
            {
                hChannelsList = psEpgOwner->psCurrentSchedule->hEpgChannels;
            }
        }

        if (hChannelsList == OSAL_INVALID_OBJECT_HDL)
        {
            // No schedule available
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
            break;
        }

        // Populate a local iterator structure with info needed to call
        // the provided callback to the caller along with their provided argument.
        sIterator.bEpgIterator = bEpgIterator;
        sIterator.pvIteratorArg = pvIteratorArg;
        sIterator.psEpgFilter = psEpgFilter;
        sIterator.bContinue = TRUE;
        sIterator.un8NumSegmentsAvailable =
            psEpgOwner->psCurrentSchedule->un8NumSegmentsCollected;

        eOsalReturnCode =
            OSAL.eLinkedListIterate(hChannelsList,
                                    (OSAL_LL_ITERATOR_HANDLER) bEpgChannelsIterator,
                                    &sIterator);
        if (eOsalReturnCode == OSAL_SUCCESS ||
            eOsalReturnCode == OSAL_NO_OBJECTS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": EPG iterate: error code = %d.",
                eOsalReturnCode);
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eIteratePrograms(): <-- Facing object unlocked.\n"PC_RESET);

    return eReturnCode;
}

/*****************************************************************************
 *
 *  eScheduleStates
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eScheduleStates (
        EPG_SERVICE_OBJECT hEpgService,
        SCHEDULE_STATE_STRUCT *psCurrentState,
        SCHEDULE_STATE_STRUCT *psInProcessState
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hEpgService;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bValid;

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_GREEN" EPG.eScheduleStates()\n"PC_RESET);

    // Check input
    bValid = DATASERVICE_IMPL_bIsValid((DATASERVICE_IMPL_HDL)hEpgService);
    if ((bValid == FALSE) ||
        ((psCurrentState == NULL) && (psInProcessState == NULL)))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Cannot pass invalid manager object or "
           "both state pointers as NULL.");
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eScheduleStates(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
            PC_BLUE"EPG.eScheduleStates(): Failed to get facing object...\n"PC_RESET);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eScheduleStates(): --> Facing object locked.\n"PC_RESET);

    if (psEpgOwner->psProcessingDescriptor != NULL)
    {
        psProcessingSchedule =
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    }

    if (psCurrentState != NULL)
    {
        psCurrentState->un8NumSegments = psEpgMgrObj->un8NumSegmentsToRecieve;

        // Check if in-process schedule is available only
        if (psEpgOwner->psCurrentSchedule == psProcessingSchedule)
        {
            eReturnCode = eScheduleState(NULL,
                                         psCurrentState);
        }
        else
        {
            eReturnCode = eScheduleState(psEpgOwner->psCurrentSchedule,
                                         psCurrentState);
        }

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Cannot retrieve Current schedule's state.");
        }
    }

    if (psInProcessState != NULL)
    {
        psInProcessState->un8NumSegments = psEpgMgrObj->un8NumSegmentsToRecieve;

        eReturnCode = eScheduleState(psProcessingSchedule,
                                     psInProcessState);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Cannot retrieve In-Progress schedule's state.");
        }
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eScheduleStates(): <-- Facing object unlocked.\n"PC_RESET);

    return eReturnCode;
}

/*****************************************************************************
 *
 *  eEnableCheckVersionsBySegments
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableCheckVersionsBySegments (
        EPG_SERVICE_OBJECT hEpgService,
        BOOLEAN bEnable
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hEpgService;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    EPG_RETURN_CODES_ENUM eEpgRet = EPG_RET_OK;
    BOOLEAN bValid;
    UN32 un32Options = 0;

    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_GREEN" EPG.eEnableCheckVersionsBySegments()\n"PC_RESET);

    // Check input
    bValid = DATASERVICE_IMPL_bIsValid((DATASERVICE_IMPL_HDL)hEpgService);
    if (bValid == FALSE)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eEnableCheckVersionsBySegments(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
            PC_BLUE"EPG.eEnableCheckVersionsBySegments(): Failed to get facing object...\n"PC_RESET);
        return SMSAPI_RETURN_CODE_NOT_OWNER;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eEnableCheckVersionsBySegments(): --> Facing object locked.\n"PC_RESET);

    do
    {
        eEpgRet = GsEpgIntf.eGetOptions(psEpgMgrObj->hEpgParserObject,
                                        &un32Options);
        if (eEpgRet != EPG_RET_OK)
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        if (bEnable == TRUE)
        {
            un32Options |= EPG_PARSER_OPTION_CHECK_SEG_VERSION;
        }
        else
        {
            un32Options &= ~EPG_PARSER_OPTION_CHECK_SEG_VERSION;
        }

        eEpgRet = GsEpgIntf.eSetOptions(psEpgMgrObj->hEpgParserObject,
                                        un32Options);
        if (eEpgRet != EPG_RET_OK)
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_BLUE"EPG.eEnableCheckVersionsBySegments(): <-- Facing object unlocked.\n"PC_RESET);

    return eReturnCode;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   EPG_MGR_bLoadString
 *
 *****************************************************************************/
BOOLEAN EPG_MGR_bLoadString (
        EPG_SERVICE_OBJECT hEpgService,
        UN8 un8SegNum,
        UN32 un32StringIndex,
        STRING_OBJECT hResultString
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    EPG_TEXT_CACHE_STRUCT *psCacheObj = NULL;
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tReadSize = 0;
    EPG_TEXT_FILE_HEADER_STRUCT sTextFileHeader;
    BOOLEAN bResult = FALSE;

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"EPG_MGR_bLoadString(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject((EPG_MGR_OBJECT_STRUCT *) hEpgService);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"EPG_MGR_bLoadString(): Failed to get facing object...\n"PC_RESET);
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"EPG_MGR_bLoadString(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Text cache is saved for Current Schedule only
        if (psEpgOwner->psCurrentSchedule == NULL)
        {
            break;
        }

        // Get Text Cache object
        psCacheObj = psEpgOwner->psCurrentSchedule->psTextCache;
        if (psCacheObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Text cache is not created.");
            break;
        }

        // Check if segment is already cached
        bResult = EPG_TEXT_CACHE_bIsSegmentCached(psCacheObj, un8SegNum);
        if (bResult == FALSE)
        {
            // Segment's text is not cached. Cache it.

            // Get appropriate text file name
            pacFilePath = pcGetFilePath(psEpgOwner,
                                        EPG_FILE_CACHE_BIN,
                                        un8SegNum);
            if (pacFilePath == NULL)
            {
                break;
            }

            // Open file
            pFile = fopen(pacFilePath, "rb");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open CT file: %s.",
                    pacFilePath);
                break;
            }

            // Read file header
            tReadSize = fread(&sTextFileHeader,
                              sizeof(EPG_TEXT_FILE_HEADER_STRUCT),
                              1,
                              pFile);
            if (tReadSize != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to read CT header from %s.",
                    pacFilePath);
                break;
            }

            // Add segment's text from file to the cache
            bResult = EPG_TEXT_CACHE_bLoadCompressedTextFile(psCacheObj,
                                                             un8SegNum,
                                                             pFile,
                                                             sTextFileHeader.un32CompressedSize,
                                                             sTextFileHeader.un32DecompressedSize);
            if (bResult == FALSE)
            {
                break;
            }
        }

        // Read string from cache
        bResult = EPG_TEXT_CACHE_bGetString(psCacheObj,
                                            un8SegNum,
                                            un32StringIndex,
                                            hResultString);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"EPG_MGR_bLoadString(): <-- Facing object unlocked.\n"PC_RESET);

    return bResult;
}

/*****************************************************************************
 *
 *  EPG_MGR_vDecoderEventHandler
 *
 *  This friend function is to be called from DECODER event handle while
 *  processing SMS_EVENT_EPG event.
 *
 *****************************************************************************/
void EPG_MGR_vDecoderEventHandler (
        DECODER_OBJECT hDecoder,
        SMS_EVENT_EPG_STRUCT const *psEpg
            )
{
    static BOOLEAN bNeedToSendDirectedUnsub = FALSE;
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj =
        (EPG_MGR_OBJECT_STRUCT *) psEpg->hEpgService;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
        PC_GREEN"DECODER EVENT HANDLER. STATUS: %d\n"PC_RESET,
           psEpg->eStatus);

    switch (psEpg->eStatus)
    {
        case SMS_EVENT_EPG_STATUS_START:
        {
            // Whenever we start the service we
            // know we'll eventually need to send the unsub
            bNeedToSendDirectedUnsub = TRUE;
        }
        break;

        case SMS_EVENT_EPG_STATUS_UPDATE:
        {
            bAttachChannelsEpgData(hDecoder, psEpgMgrObj);
        }
        break;

        case SMS_EVENT_EPG_STATUS_STOP:
        {
            bDetachChannelsEpgData(hDecoder, psEpgMgrObj);

            // Do we need to send a "directed unsubscribe"
            // to the manager?
            if (TRUE == bNeedToSendDirectedUnsub)
            {
                BOOLEAN bPosted;

                // Send a "directed unsubscribe" to this manager
                bPosted = DATASERVICE_IMPL_bPostEvent(
                    (DATASERVICE_IMPL_HDL) psEpg->hEpgService,
                    DATASERVICE_FW_EVENT_DECODER_UNSUBSCRIBED,
                    hDecoder);

                // If the event was sent then we mark
                // this as false since we don't want to send it again
                bNeedToSendDirectedUnsub = (bPosted == TRUE) ? FALSE : TRUE;
            }
        }
        break;
    };

    return;
}

/*****************************************************************************
 *
 *   bAttachChannelsEpgData
 *
 *****************************************************************************/
BOOLEAN bAttachChannelsEpgData (
        DECODER_OBJECT hDecoder,
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    OSAL_LINKED_LIST_ENTRY hEpgChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    EPG_CHANNEL_OBJECT psEpgChannel = EPG_INVALID_CHANNEL_OBJECT;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    UN32 un32TotalShowsCount = 0;
    static EPG_FULL_SCHEDULE_ROOT *psLastPushedSchedule = NULL;
    BOOLEAN bResult;

    // Check input parameters
    bResult = SMSO_bOwner((SMS_OBJECT) hDecoder);

    if (bResult == TRUE)
    {
        bResult = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL) psEpgMgrObj);
    }

    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": INVALID INPUT.");
        return FALSE;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bAttachChannelsEpgData(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bAttachChannelsEpgData(): Failed to get facing object.\n"PC_RESET);
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bAttachChannelsEpgData(): --> Facing object locked.\n"PC_RESET);

    do
    {
        CCACHE_OBJECT hCCache = CCACHE_INVALID_OBJECT;

        if (psEpgMgrObj->psEpgOwner->psCurrentSchedule == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                "Schedule is not available.\n");
            bResult = FALSE;
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            "Current schedule is used.\n");

        if (psLastPushedSchedule == psEpgMgrObj->psEpgOwner->psCurrentSchedule)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_GREEN"Schedule is not changed since last update. Skipped.\n"PC_RESET);
            bResult = TRUE;
            break;
        }

        hEpgChannelEntry =
            OSAL.hLinkedListFirst(psEpgMgrObj->psEpgOwner->psCurrentSchedule->hEpgChannels,
                                  (void **)NULL);

        if (hEpgChannelEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                "No schedule events to load.\n");
            bResult = FALSE;
            break;
        }

        // Get the owning DECODER's channel cache
        hCCache = DECODER_hCCache(hDecoder);
        if (hCCache != CCACHE_INVALID_OBJECT)
        {
            do
            {
                psEpgChannel = (EPG_CHANNEL_OBJECT_STRUCT*) OSAL.pvLinkedListThis(hEpgChannelEntry);
                hChannel = CCACHE_hChannelFromIds(hCCache,
                                                  psEpgChannel->tServiceId,
                                                  CHANNEL_INVALID_ID,
                                                  FALSE);
                if (hChannel != CHANNEL_INVALID_OBJECT)
                {
                    UN32 un32ChannelShowsCount = EPG_CHANNEL_un32GetShowsCount(psEpgChannel);
                    if (un32ChannelShowsCount > 0)
                    {
                        un32TotalShowsCount += un32ChannelShowsCount;
                    }
                    else
                    {
                        psEpgChannel = EPG_INVALID_CHANNEL_OBJECT;
                    }

                    CHANNEL_vSetProgramsList(hChannel,
                                             psEpgChannel);
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                        PC_GREY"Channel sid %d not found in cache.\n"PC_RESET,
                               psEpgChannel->tServiceId);
                }
                hEpgChannelEntry = OSAL.hLinkedListNext(hEpgChannelEntry, NULL);

            } while (hEpgChannelEntry != OSAL_INVALID_LINKED_LIST_ENTRY);
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bAttachChannelsEpgData(): <-- Facing object unlocked.\n"PC_RESET);

    if (un32TotalShowsCount > 0)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            "Total shows in list: %u\n",
            un32TotalShowsCount);
        DECODER_vGenericUpdateEventMask(hDecoder, DECODER_OBJECT_EVENT_EPG_UPDATE_DONE);
        psLastPushedSchedule = psEpgMgrObj->psEpgOwner->psCurrentSchedule;
        bResult = TRUE;
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bDetachChannelsEpgData
 *
 *****************************************************************************/
BOOLEAN bDetachChannelsEpgData (
        DECODER_OBJECT hDecoder,
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    BOOLEAN bResult = FALSE;
    CCACHE_OBJECT hCCache = CCACHE_INVALID_OBJECT;

    do
    {
        // Check input
        bResult = SMSO_bOwner((SMS_OBJECT) hDecoder);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": INVALID INPUT.");
            break;
        }

        // Get the owning DECODER's channel cache
        hCCache = DECODER_hCCache(hDecoder);
        if (hCCache == CCACHE_INVALID_OBJECT)
        {
            break;
        }

        CCACHE_vCleanChannelsEpgData(hCCache);

        // Everything's OK
        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/* EPG Client interface implementation (used by EPG Parser) */

/*****************************************************************************
 *
 *  bEpgClientProcessParserEvent
 *
 *****************************************************************************/
static BOOLEAN bEpgClientProcessParserEvent (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    BOOLEAN bResult = TRUE;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> bEpgClientProcessParserEvent()\n"PC_RESET);

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessParserEvent(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bEpgClientProcessParserEvent(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< bEpgClientProcessParserEvent(): %u\n"PC_RESET,
               FALSE);
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessParserEvent(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Ensure everything is ready for processing
        bResult = bInitScheduleProcessing(psEpgOwner,
                    psEpgMgrObj->un8NumSegmentsToRecieve);
        if (bResult == FALSE)
        {
            break;
        }

        bResult = bProcessParserEvent(psEpgMgrObj,
                                      psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessParserEvent(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< bEpgClientProcessParserEvent(): %u\n"PC_RESET,
           bResult);
    return bResult;
}

/*****************************************************************************
 *
 *  bEpgClientProcessCompressedTextPayload
 *
 *****************************************************************************/
static BOOLEAN bEpgClientProcessCompressedTextPayload (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    BOOLEAN bResult = TRUE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> bEpgClientProcessCompressedTextPayload()\n"PC_RESET);

    // Process only if initial schedule is loaded
    bResult = bIsEpgInitialized(psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bEpgClientProcessCompressedTextPayload(): Not initialized yet.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< bEpgClientProcessCompressedTextPayload(): %u\n"PC_RESET,
               FALSE);
        return FALSE;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessCompressedTextPayload(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bEpgClientProcessCompressedTextPayload(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< bEpgClientProcessCompressedTextPayload(): %u\n"PC_RESET,
               FALSE);
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessCompressedTextPayload(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Ensure everything is ready for processing
        bResult = bInitScheduleProcessing(psEpgOwner,
                    psEpgMgrObj->un8NumSegmentsToRecieve);
        if (bResult == FALSE)
        {
            break;
        }

        bResult = bSaveCompressedTextPartToFile(psEpgOwner,
                                                psAuData);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgClientProcessCompressedTextPayload(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< bEpgClientProcessCompressedTextPayload(): %u\n"PC_RESET,
        bResult);
    return bResult;
}

/*****************************************************************************
 *
 *  vEpgClientSaveAuData
 *
 *****************************************************************************/
static void vEpgClientSaveAuData (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientSaveAuData()\n"PC_RESET);

    // Process only if initial schedule is loaded
    bResult = bIsEpgInitialized(psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSaveAuData(): Not initialized yet.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSaveAuData()\n"PC_RESET);
        return;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSaveAuData(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSaveAuData(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSaveAuData()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSaveAuData(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Save GRID AU payload
        if (psAuData->eAuDataType == EPG_DATA_TYPE_GRID)
        {
            bSaveGridPayloadPartToFile(psEpgOwner,
                                       psAuData);
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSaveAuData(): <-- Facing object locked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientSaveAuData()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientSegmentGridCompleted
 *
 *****************************************************************************/
static void vEpgClientSegmentGridCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    char *pacFilePath = NULL;
    BOOLEAN bResult = FALSE;
    FILE *pFile = NULL;
    UN8 un8FileAttr = 0;
    EPG_GRID_FILE_HEADER_STRUCT *psGridFileHeader = NULL;
    size_t tBytesWritten = 0;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientSegmentGridCompleted()\n"PC_RESET);

    // Process only if initial schedule is loaded
    bResult = bIsEpgInitialized(psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSegmentGridCompleted(): Not initialized yet.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSegmentGridCompleted()\n"PC_RESET);
        return;
    }

    // Update appropriate segment GRID file header.

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentGridCompleted(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSegmentGridCompleted(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSegmentGridCompleted()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentGridCompleted(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Get appropriate grid file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_GRID_TMP,
                                    un8SegmentNumber);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Check file existence
        bResult = OSAL.bFileSystemGetFileAttributes(pacFilePath, &un8FileAttr);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": File does not exist: %s.",
                pacFilePath);
            break;
        }

        // Check appropriate file descriptor
        psGridFileHeader =
            &(psEpgOwner->psProcessingDescriptor->asGridFileDesc[un8SegmentNumber]);
        if (psGridFileHeader->un32AusCount == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": File descriptor is incomplete: %s.",
                pacFilePath);
            break;
        }

        // Write updated descriptor to the file
        // Open file for update
        pFile = fopen(pacFilePath, "rb+");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to open CT file for update: %s.",
                pacFilePath);
            break;
        }

        fseek(pFile, 0, SEEK_SET);

        // Write data
        tBytesWritten = fwrite(psGridFileHeader,
                               sizeof(EPG_GRID_FILE_HEADER_STRUCT),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write CT header to %s.",
                pacFilePath);
            break;
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentGridCompleted(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientSegmentGridCompleted()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientSegmentTextCompleted
 *
 *****************************************************************************/
static void vEpgClientSegmentTextCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber,
        UN16 un16StringsCount,
        UN32 un32DecompressedFileSize
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    char *pacFilePath = NULL;
    BOOLEAN bResult = FALSE;
    FILE *pFile = NULL;
    UN8 un8FileAttr = 0;
    EPG_TEXT_FILE_HEADER_STRUCT *psTextFileHeader = NULL;
    size_t tBytesWritten = 0;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6, PC_GREEN">> vEpgClientSegmentTextCompleted()\n"PC_RESET);

    // Process only if initial schedule is loaded
    bResult = bIsEpgInitialized(psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSegmentTextCompleted(): Not initialized yet.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSegmentTextCompleted()\n"PC_RESET);
        return;
    }

    // Update appropriate segment TEXT file header.

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentTextCompleted(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientSegmentTextCompleted(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientSegmentTextCompleted()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentTextCompleted(): --> Facing object locked.\n"PC_RESET);

    do
    {
        // Get appropriate text file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_TEXT_TMP,
                                    un8SegmentNumber);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Check file existence
        bResult = OSAL.bFileSystemGetFileAttributes(pacFilePath, &un8FileAttr);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": File does not exist: %s.",
                pacFilePath);
            break;
        }

        // Check appropriate file descriptor
        psTextFileHeader =
            &(psEpgOwner->psProcessingDescriptor->asTextFileDesc[un8SegmentNumber]);
        if (psTextFileHeader->un32CompressedSize == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": File descriptor is incomplete: %s.",
                pacFilePath);
            break;
        }

        // Update descriptor
        psTextFileHeader->un16StringsCount = un16StringsCount;
        psTextFileHeader->un32DecompressedSize = un32DecompressedFileSize;

        // Write updated descriptor to the file
        // Open file for update
        pFile = fopen(pacFilePath, "rb+");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to open CT file for update: %s.",
                pacFilePath);
            break;
        }

        fseek(pFile, 0, SEEK_SET);

        // Write data
        tBytesWritten = fwrite(psTextFileHeader,
                               sizeof(EPG_TEXT_FILE_HEADER_STRUCT),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write CT header to %s.",
                pacFilePath);
            break;
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientSegmentTextCompleted(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientSegmentTextCompleted()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientWholeSegmentCompleted
 *
 *****************************************************************************/
static void vEpgClientWholeSegmentCompleted (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientWholeSegmentCompleted()\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
        PC_GREEN"Segment %u is completed.\n"PC_RESET,
           un8SegmentNumber);

    // Update only if initial schedule is loaded
    bResult = bIsEpgInitialized(psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientWholeSegmentCompleted(): Not initialized yet.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientWholeSegmentCompleted()\n"PC_RESET);
        return;
    }

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientWholeSegmentCompleted(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientWholeSegmentCompleted(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientWholeSegmentCompleted()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientWholeSegmentCompleted(): --> Facing object locked.\n"PC_RESET);

    do
    {
        EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

        // Update segment information
        psProcessingSchedule->aeSegState[un8SegmentNumber] = EPG_SEG_STATE_READY;
        psProcessingSchedule->un8NumSegmentsCollected++;

        // Rename grid and text files from .tmp to .bin
        bResult = bMoveProcessingFilesToReady(psEpgOwner, un8SegmentNumber);
        if (bResult == TRUE)
        {
            psProcessingSchedule->aeSegState[un8SegmentNumber] = EPG_SEG_STATE_SAVED;
        }

        // Rename only if no schedule was previously loaded
        if (psEpgOwner->psCurrentSchedule == psProcessingSchedule)
        {
            bCopyEpgFile(psEpgOwner,
                         EPG_FILE_TEXT_BIN,
                         un8SegmentNumber,
                         EPG_FILE_CACHE_BIN,
                         un8SegmentNumber,
                         TRUE);
        }

        // When segment 0 is completed,
        // all midnight programs should be saved.
        // Move midnight file from .tmp to .bin.
        if (un8SegmentNumber == 0)
        {
            bMoveProcessingMidnightFileToReady(psEpgOwner);
        }

        // Check if a whole schedule is complete
        if (psProcessingSchedule->un8NumSegmentsCollected
            != psProcessingSchedule->un8NumSegmentsTotal)
        {
            // Schedule is incomplete yet
            psProcessingSchedule->eState = SCHEDULE_STATE_PROGRESS;
        }
        else
        {
            // Schedule is complete
            vHandleReceivedScheduleCompletion(psEpgMgrObj);
        }

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientWholeSegmentCompleted(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientWholeSegmentCompleted()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientGetVersion
 *
 *****************************************************************************/
static void vEpgClientGetVersion (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule = NULL;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientGetVersion()\n"PC_RESET);

    if (psVersion == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": psVersion parameter is NULL.");
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientGetVersion()\n"PC_RESET);
        return;
    }

    // Clean version data
    psVersion->un16Epoch = 0;
    psVersion->un8Version = 0;

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientGetVersion(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientGetVersion(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientGetVersion()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientGetVersion(): --> Facing object locked.\n"PC_RESET);

    if (psEpgOwner->psProcessingDescriptor != NULL)
    {
        psProcessingSchedule =
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    }

    // First try to get the newest schedule version
    if (psProcessingSchedule != NULL)
    {
        *psVersion = psProcessingSchedule->sVersion;
    }
    // If there is no new schedule being processed,
    // return current schedule version.
    else if (psEpgOwner->psCurrentSchedule != NULL)
    {
        *psVersion = psEpgOwner->psCurrentSchedule->sVersion;
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientGetVersion(): <-- Facing object locked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientGetVersion()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientVersionChanged
 *
 *****************************************************************************/
static void vEpgClientVersionChanged (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psNewVersion
            )
{
    BOOLEAN bResult = FALSE;
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule = NULL;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientVersionChanged()\n"PC_RESET);

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientVersionChanged(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientVersionChanged(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientVersionChanged()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientVersionChanged(): --> Facing object locked.\n"PC_RESET);

    // Ensure everything is ready for processing
    bResult = bInitScheduleProcessing(psEpgOwner,
                psEpgMgrObj->un8NumSegmentsToRecieve);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientVersionChanged()\n"PC_RESET);
        return;
    }

    psProcessingSchedule = psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
        "Schedule Version is changed: [epoch: %u, version: %u] -> "
        "[epoch: %u, version: %u].\n",
        psProcessingSchedule->sVersion.un16Epoch,
        psProcessingSchedule->sVersion.un8Version,
        psNewVersion->un16Epoch, psNewVersion->un8Version);

    psProcessingSchedule->sVersion = *psNewVersion;

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientVersionChanged(): <-- Facing object locked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientVersionChanged()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientRestartGridProcessing
 *
 *****************************************************************************/
static void vEpgClientRestartGridProcessing (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;

    DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": Restart grid processing.\n");

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientRestartGridProcessing()\n"PC_RESET);

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartGridProcessing(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientRestartGridProcessing(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientRestartGridProcessing()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartGridProcessing(): --> Facing object locked.\n"PC_RESET);

    if (psEpgOwner->psProcessingDescriptor != NULL)
    {
        // Clean schedule data (NOT object itself)
        vCleanEpgSchedule(
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule);

        // Clean appropriate processing descriptors data (NOT object itself)
        vCleanGridDescriptors(psEpgOwner->psProcessingDescriptor);
    }
    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartGridProcessing(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientRestartGridProcessing()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  vEpgClientRestartTextProcessing
 *
 *****************************************************************************/
static void vEpgClientRestartTextProcessing (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) hClientObj;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;

    DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": Restart text processing.\n");

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN">> vEpgClientRestartTextProcessing()\n"PC_RESET);

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartTextProcessing(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgClientRestartTextProcessing(): Failed to get facing object.\n"PC_RESET);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"<< vEpgClientRestartTextProcessing()\n"PC_RESET);
        return;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartTextProcessing(): --> Facing object locked.\n"PC_RESET);

    // Clean appropriate processing descriptors data (NOT object itself)
    if (psEpgOwner->psProcessingDescriptor != NULL)
    {
        vCleanTextDescriptors(psEpgOwner->psProcessingDescriptor);
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgClientRestartTextProcessing(): <-- Facing object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgClientRestartTextProcessing()\n"PC_RESET);
    return;
}

/* General */

/*****************************************************************************
 *
 *  vUninitEpgMgrObject
 *
 *****************************************************************************/
static void vUninitEpgMgrObject (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> DEINIT EPG MGR OBJECT\n"PC_RESET);

    vEpgLoaderTaskUninitialize(psEpgMgrObj);

    // Destroy EPG Interface object
    if (psEpgMgrObj->hEpgParserObject != EPG_PARSER_INTERFACE_INVALID_OBJECT)
    {
        GsEpgIntf.vUninit(psEpgMgrObj->hEpgParserObject);
    }

    if (psEpgMgrObj->psEpgOwner != NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"DESTROY EPG OWNER.\n"PC_RESET);
        vDestroyEpgOwner(psEpgMgrObj->psEpgOwner);
    }

    // Destroy the SMS Update object
    SMSU_vDestroy(&psEpgMgrObj->sEvent);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< EPG MGR DEINITIALIZED.\n"PC_RESET);

    return;
}

/*****************************************************************************
 *
 *   bIsEpgInitialized
 *
 *****************************************************************************/
static BOOLEAN bIsEpgInitialized (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    BOOLEAN bResult = FALSE;
    DATASERVICE_STATE_ENUM eState = DATASERVICE_STATE_INVALID;
    BOOLEAN bInitialEpgLoaded = FALSE;
    BOOLEAN bStopping = FALSE;

    do
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bIsEpgInitialized(): Try to lock EPG MGR...\n"PC_RESET);
        bResult = DATASERVICE_IMPL_bLock((DATASERVICE_IMPL_HDL)psEpgMgrObj);
        if (bResult == FALSE)
        {
            break;
        }
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bIsEpgInitialized(): --> EPG MGR locked.\n"PC_RESET);

        eState = DATASERVICE_IMPL_eState((DATASERVICE_IMPL_HDL)psEpgMgrObj);
        bInitialEpgLoaded = psEpgMgrObj->bInitialEpgLoaded;
        bStopping = psEpgMgrObj->bStopping;

        DATASERVICE_IMPL_vUnlock((DATASERVICE_IMPL_HDL)psEpgMgrObj);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bIsEpgInitialized(): <-- EPG MGR unlocked.\n"PC_RESET);

        if (eState != DATASERVICE_STATE_READY)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"EPG DS IS NOT YET READY.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

        if (bInitialEpgLoaded == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"INITIAL EPG IS NOT YET LOADED.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

        if (bStopping == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"EPG IS STOPPING NOW.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   bPrepareForInitialProcessing
 *
 *****************************************************************************/
static BOOLEAN bPrepareForInitialProcessing (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    BOOLEAN bResult = FALSE;

    do
    {
        // Create new EpgOwner
        psEpgMgrObj->psEpgOwner = psCreateEpgOwner();
        if (psEpgMgrObj->psEpgOwner == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create new owner.");
            break;
        }

        // Build File Path for copy
        bResult = bBuildEpgFilePath(psEpgMgrObj->psEpgOwner);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Cannot build EPG directory.");
            break;
        }

        // Create EPG Parser Interface
        gsEpgClientInterface.hClientObj = (EPG_CLIENT_INTERFACE_OBJECT)psEpgMgrObj;
        psEpgMgrObj->hEpgParserObject = GsEpgIntf.hInit(&gsEpgClientInterface,
                                                        psEpgMgrObj->un8NumSegmentsToRecieve,
                                                        EPG_PARSER_OPTION_CHECK_SEG_VERSION);
        if (psEpgMgrObj->hEpgParserObject == EPG_PARSER_INTERFACE_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create EPG Parser Interface object.");
            break;
        }

        // Everything's done
        return TRUE;

    } while (FALSE);

    // Error case:
    // Release allocated resources
    if (psEpgMgrObj->hEpgParserObject != EPG_PARSER_INTERFACE_INVALID_OBJECT)
    {
        GsEpgIntf.vUninit(psEpgMgrObj->hEpgParserObject);
    }

    if (psEpgMgrObj->psEpgOwner != NULL)
    {
        vDestroyEpgOwner(psEpgMgrObj->psEpgOwner);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   psCreateEpgOwner
 *
 *****************************************************************************/
static EPG_OWNER_STRUCT *psCreateEpgOwner( void )
{
    EPG_OWNER_STRUCT *psEpgOwner = NULL;

    // Create object (initially locked)
    psEpgOwner = (EPG_OWNER_STRUCT *) SMSO_hCreate(EPG_OWNER_OBJECT_NAME,
                                                   sizeof(EPG_OWNER_STRUCT),
                                                   SMS_INVALID_OBJECT, // No parent
                                                   TRUE); // Self-locked
    if ((SMS_OBJECT) psEpgOwner != SMS_INVALID_OBJECT)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
    }
    else
    {
        psEpgOwner = NULL;
    }

    return psEpgOwner;
}

/*****************************************************************************
 *
 *  vDestroyEpgOwner
 *
 *****************************************************************************/
static void vDestroyEpgOwner (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    BOOLEAN bLocked = FALSE;

    if (psEpgOwner == NULL)
    {
        return;
    }

    bLocked = SMSO_bLock((SMS_OBJECT) psEpgOwner, OSAL_OBJ_TIMEOUT_NONE);
    if (bLocked == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to lock EPG Owner during deinitialization!");
    }

    // Destroy entities required during processing (if exist)
    vUninitScheduleProcessing(psEpgOwner);

    // Destroy assembled schedule (if exists)
    vDestroyEpgSchedule(psEpgOwner->psCurrentSchedule);

    // Destroy EPG Path string
    if (psEpgOwner->pacEpgDirectoryPath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psEpgOwner->pacEpgDirectoryPath);
    }

    // Destroy object itself
    SMSO_vDestroy((SMS_OBJECT) psEpgOwner);

    return;
}

/*****************************************************************************
 *
 *   bBuildEpgFilePath
 *
 *   This function generates the epg file path based on the SMS file path.
 *   This information is used in order to locate the epg database
 *   and compressed text files.
 *
 *****************************************************************************/
static BOOLEAN bBuildEpgFilePath (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    size_t tEpgPathLen = 0;
    const char *pacSmsPath = NULL;
    UN8 un8FAttr = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        pacSmsPath = SMS_pacGetPath();
        if (pacSmsPath == NULL)
        {
            break;
        }

        // Calculate required string length
        tEpgPathLen = strlen(pacSmsPath) + strlen(EPG_DIRECTORY_NAME) + 2; // + '\' + '\0'

        // Allocate/reallocate the object
        if (psEpgOwner->pacEpgDirectoryPath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psEpgOwner->pacEpgDirectoryPath);
        }
        psEpgOwner->pacEpgDirectoryPath = (char *) SMSO_hCreate(EPG_DIRECTORY_PATH_OBJECT_NAME,
                                                                tEpgPathLen,
                                                                (SMS_OBJECT) psEpgOwner, FALSE);
        if ((SMS_OBJECT) psEpgOwner->pacEpgDirectoryPath == SMS_INVALID_OBJECT)
        {
            psEpgOwner->pacEpgDirectoryPath = NULL;
            break;
        }

        // Construct full path
        snprintf(psEpgOwner->pacEpgDirectoryPath,
                 tEpgPathLen,
                 "%s/%s",
                 pacSmsPath,
                 EPG_DIRECTORY_NAME);

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"EPG DIR: %s\n"PC_RESET,
            psEpgOwner->pacEpgDirectoryPath);

        bResult = OSAL.bFileSystemGetFileAttributes(psEpgOwner->pacEpgDirectoryPath,
                                                    &un8FAttr);
        if (bResult == TRUE)
        {
            // Directory already exists
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"EPG DIR DOES NOT EXIST. TRY TO CREATE...\n"PC_RESET);
        bResult = OSAL.bFileSystemMakeDir(psEpgOwner->pacEpgDirectoryPath);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Cannot create directory: %s",
                psEpgOwner->pacEpgDirectoryPath);
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   psGetAppFacingObject
 *
 *****************************************************************************/
static EPG_OWNER_STRUCT *psGetAppFacingObject (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    BOOLEAN bResult = FALSE;
    EPG_OWNER_STRUCT *psEpgOwner = NULL;

    do
    {
        psEpgOwner = psEpgMgrObj->psEpgOwner;
        if (psEpgOwner == NULL)
        {
            break;
        }

        // Object is already created. Try to lock it.
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"psGetAppFacingObject: Try to lock EPG Owner...\n"PC_RESET);
        bResult = SMSO_bLock((SMS_OBJECT) psEpgOwner, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,
                                         __LINE__,
                                         EPG_MGR_OBJECT_NAME
                                         ": Failed to lock Facing object (%p)\n",
                                         psEpgMgrObj->psEpgOwner);
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
                PC_BLUE"psGetAppFacingObject: Failed to lock EPG Owner.\n"PC_RESET);
            psEpgOwner = NULL;
            break;
        }

    } while (FALSE);

    return psEpgOwner;
}

/*******************************************************************************
 *
 *  pcGetFilePath
 *
 *  NOTE: Returns pointer to newly allocated string.
 *        Caller must clean allocated object using SMSO_vDestroy().
 *
 *******************************************************************************/
static char *pcGetFilePath (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_FILE_TYPE_ENUM eFileType,
        UN8 un8SegNum
            )
{
    size_t tFileNameLen = 0;
    char *pacFilePath = NULL;

    // Calculate required string length
    tFileNameLen = strlen(psEpgOwner->pacEpgDirectoryPath) +
        EPG_FILE_NAME_LEN_MAX + EPG_FILE_EXT_LEN_MAX + 2; // + '\' + '\0'

    // Allocate the object
    pacFilePath = (char *) SMSO_hCreate(EPG_FILE_PATH_OBJECT_NAME,
                                        tFileNameLen,
                                        SMS_INVALID_OBJECT,
                                        FALSE);
    if ((SMS_OBJECT) pacFilePath == SMS_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to allocate file path string.");

        return NULL;
    }

    // Construct full path depending on file type
    switch (eFileType)
    {
    case EPG_FILE_GRID_BIN:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_GRID_BIN_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_GRID_TMP:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_GRID_TMP_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_TEXT_BIN:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_TEXT_BIN_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_TEXT_TMP:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_TEXT_TMP_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_CACHE_BIN:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_CACHE_BIN_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_CACHE_TMP:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_CACHE_TMP_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath,
                 un8SegNum);
        break;

    case EPG_FILE_MIDNIGHT_BIN:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_MIDNIGHT_BIN_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath);
        break;

    case EPG_FILE_MIDNIGHT_TMP:
        snprintf(pacFilePath,
                 tFileNameLen,
                 "%s/"EPG_MIDNIGHT_TMP_FILE_NAME_LABEL,
                 psEpgOwner->pacEpgDirectoryPath);
        break;

    default:
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
        pacFilePath = NULL;
        break;
    }

    return pacFilePath;
}

/*******************************************************************************
 *
 *  bSaveCompressedTextPartToFile
 *
 *******************************************************************************/
static BOOLEAN bSaveCompressedTextPartToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_AU_INFO_STRUCT *psAuData
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tBytesWritten = 0;
    EPG_TEXT_FILE_HEADER_STRUCT *psTextFileHeader = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Verify input data
        if (psAuData == NULL)
        {
            break;
        }

        if (psAuData->eAuDataType != EPG_DATA_TYPE_COMPRESSED_TEXT ||
            psAuData->pvData == NULL ||
            psAuData->tDataSize == 0)
        {
            break;
        }

        // Get appropriate text file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_TEXT_TMP,
                                    psAuData->un8SegmentNumber);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Get saved file descriptor
        psTextFileHeader =
            &(psEpgOwner->psProcessingDescriptor->asTextFileDesc[psAuData->un8SegmentNumber]);

        // If this is the first part of the compressed text file,
        // overwrite the file, otherwise append data to the end.
        if (psTextFileHeader->un32CompressedSize == 0)
        {
            EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
                psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

            // Open for writing
            pFile = fopen(pacFilePath, "wb");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open CT file %s for writing.",
                    pacFilePath);
                break;
            }

            // Write file header
            psTextFileHeader->un16Epoch =
                psProcessingSchedule->sVersion.un16Epoch + psAuData->un8SegmentNumber;
            psTextFileHeader->un8Version = psProcessingSchedule->sVersion.un8Version;
            psTextFileHeader->un16StringsCount = 0;     // Will be updated when all parts are saved
            psTextFileHeader->un32DecompressedSize = 0; // Will be updated when all parts are saved
            psTextFileHeader->un32CompressedSize = 0;   // Will be updated when all parts are saved

            tBytesWritten = fwrite(psTextFileHeader,
                                   sizeof(EPG_TEXT_FILE_HEADER_STRUCT),
                                   1,
                                   pFile);
            if (tBytesWritten != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to write CT header to %s.",
                    pacFilePath);
                break;
            }
        }
        else
        {
            // Open for appending
            pFile = fopen(pacFilePath, "ab");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open CT file %s for appending.",
                    pacFilePath);
                break;
            }
        }

        // Write compressed text data
        tBytesWritten = fwrite(psAuData->pvData,
                               1,
                               psAuData->tDataSize,
                               pFile);
        if (tBytesWritten != psAuData->tDataSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to write CT part %u:%u to %s.",
                    psAuData->un8SegmentNumber, psAuData->un32AuNumber, pacFilePath);
            break;
        }

        // Data is successfully saved
        psTextFileHeader->un32CompressedSize += psAuData->tDataSize;

        bResult = TRUE;

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bSaveGridPayloadPartToFile
 *
 *******************************************************************************/
static BOOLEAN bSaveGridPayloadPartToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_AU_INFO_STRUCT *psAuData
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tBytesWritten = 0;
    EPG_GRID_FILE_HEADER_STRUCT *psGridFileHeader = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Verify input data
        if (psAuData == NULL)
        {
            break;
        }

        if (psAuData->eAuDataType != EPG_DATA_TYPE_GRID ||
            psAuData->pvData == NULL ||
            psAuData->tDataSize == 0)
        {
            break;
        }

        // Get appropriate GRID file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_GRID_TMP,
                                    psAuData->un8SegmentNumber);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Get saved file descriptor
        psGridFileHeader =
            &(psEpgOwner->psProcessingDescriptor->asGridFileDesc[psAuData->un8SegmentNumber]);

        // If this is the first part of the compressed text file,
        // overwrite the file, otherwise append data to the end.
        if (psGridFileHeader->un32AusCount == 0)
        {
            EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
                psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

            // Open for writing
            pFile = fopen(pacFilePath, "wb");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open GRID file %s for writing.",
                    pacFilePath);
                break;
            }

            // Write file header
            psGridFileHeader->un16Epoch =
                psProcessingSchedule->sVersion.un16Epoch + psAuData->un8SegmentNumber;
            psGridFileHeader->un8Version = psProcessingSchedule->sVersion.un8Version;
            psGridFileHeader->un32AusCount = 0; // Will be updated when all parts are saved

            tBytesWritten = fwrite(psGridFileHeader,
                                   sizeof(EPG_GRID_FILE_HEADER_STRUCT),
                                   1,
                                   pFile);
            if (tBytesWritten != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to write GRID header to %s.",
                    pacFilePath);
                break;
            }
        }
        else
        {
            // Open for appending
            pFile = fopen(pacFilePath, "ab");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open GRID file %s for appending.",
                    pacFilePath);
                break;
            }
        }

        // Write AU number
        tBytesWritten = fwrite(&(psAuData->un32AuNumber),
                               sizeof(UN32),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write AU number to %s.",
                pacFilePath);
            break;
        }

        // Write AU size
        tBytesWritten = fwrite(&(psAuData->tDataSize),
                               sizeof(size_t),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write AU size to %s.",
                pacFilePath);
            break;
        }

        // Write Grid payload
        tBytesWritten = fwrite(psAuData->pvData,
                               1,
                               psAuData->tDataSize,
                               pFile);
        if (tBytesWritten != psAuData->tDataSize)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write GRID payload %u:%u to %s.",
                psAuData->un8SegmentNumber, psAuData->un32AuNumber, pacFilePath);
            break;
        }

        // Access Unit Data is successfully saved
        psGridFileHeader->un32AusCount++;

        bResult = TRUE;

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bCopyEpgFile
 *
 *******************************************************************************/
static BOOLEAN bCopyEpgFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        EPG_FILE_TYPE_ENUM eSrcFileType,
        UN8 un8SrcSegNum,
        EPG_FILE_TYPE_ENUM eDstFileType,
        UN8 un8DstSegNum,
        BOOLEAN bRename
            )
{
    char *pcSrcFilePath = NULL;
    char *pcDstFilePath = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        pcSrcFilePath = pcGetFilePath(psEpgOwner,
                                      eSrcFileType,
                                      un8SrcSegNum);
        pcDstFilePath = pcGetFilePath(psEpgOwner,
                                      eDstFileType,
                                      un8DstSegNum);
        if (pcSrcFilePath == NULL || pcDstFilePath == NULL)
        {
            break;
        }

        bResult = DS_UTIL_bCopyFile(pcSrcFilePath,
                                    pcDstFilePath,
                                    TRUE,
                                    bRename);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
                PC_RED"WARNING: Failed to rename file %s to %s.\n"PC_RESET,
                pcSrcFilePath, pcDstFilePath);
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 4,
            PC_GREEN"File %s from %s to %s.\n"PC_RESET,
            bRename == TRUE ? "renamed" : "copied",
            pcSrcFilePath, pcDstFilePath);

        // Everything's done
        bResult = TRUE;

    } while (FALSE);

    if (pcSrcFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pcSrcFilePath);
    }

    if (pcDstFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pcDstFilePath);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bMoveProcessingFilesToReady
 *
 *******************************************************************************/
static BOOLEAN bMoveProcessingFilesToReady (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegmentNumber
            )
{
    BOOLEAN bResult = TRUE;

    do
    {
        // Rename grid file
        bResult = bCopyEpgFile(psEpgOwner,
                               EPG_FILE_GRID_TMP,
                               un8SegmentNumber,
                               EPG_FILE_GRID_BIN,
                               un8SegmentNumber,
                               TRUE);
        if (bResult == FALSE)
        {
            break;
        }

        // Rename text file
        bResult = bCopyEpgFile(psEpgOwner,
                               EPG_FILE_TEXT_TMP,
                               un8SegmentNumber,
                               EPG_FILE_TEXT_BIN,
                               un8SegmentNumber,
                               TRUE);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  eScheduleState
 *
 *******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eScheduleState (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        SCHEDULE_STATE_STRUCT *psState
            )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    UN8 un8Index = 0;

    do
    {
        if (psState == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        if (psSchedule != NULL)
        {
            psState->un8NumSegments = psSchedule->un8NumSegmentsTotal;
            psState->eState = psSchedule->eState;
            psState->un8Version = psSchedule->sVersion.un8Version;
            psState->un16Epoch = psSchedule->sVersion.un16Epoch;

            for (un8Index=0; un8Index < psState->un8NumSegments; un8Index++)
            {
                psState->aeSegState[un8Index] = psSchedule->aeSegState[un8Index];
            }
        }
        else
        {
            psState->eState = SCHEDULE_STATE_INVALID;
            psState->un8Version = 0;
            psState->un16Epoch = 0;

            for (un8Index=0; un8Index < psState->un8NumSegments; un8Index++)
            {
                psState->aeSegState[un8Index] = EPG_SEG_STATE_WAIT;
            }
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
 *
 *  bLoadStoredSegmentGridVersion
 *
 *******************************************************************************/
static BOOLEAN bLoadStoredSegmentGridVersion (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegNum,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tReadSize = 0;
    EPG_GRID_FILE_HEADER_STRUCT sGridFileHeader;
    BOOLEAN bResult = FALSE;

    do
    {
        // Check input
        if (psVersion == NULL)
        {
            break;
        }

        // Get segment grid file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_GRID_BIN,
                                    un8SegNum);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Open file
        pFile = fopen(pacFilePath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"WARNING: Failed to open file: %s.\n"PC_RESET,
                pacFilePath);
            break;
        }

        // Read file header
        tReadSize = fread(&sGridFileHeader,
                          sizeof(EPG_GRID_FILE_HEADER_STRUCT),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read GRID header from %s.",
                pacFilePath);
            break;
        }

        // Copy data
        psVersion->un16Epoch = sGridFileHeader.un16Epoch;
        psVersion->un8Version = sGridFileHeader.un8Version;

        // Everything's OK
        bResult = TRUE;

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  eLoadStoredSchedule
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eLoadStoredSchedule (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = NULL;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule = NULL;
    UN8 un8NumSegmentsTotal = psEpgMgrObj->un8NumSegmentsToRecieve;
    EPG_SCHEDULE_VERSION_STRUCT sVersion = {0,0};
    BOOLEAN bResult = FALSE;
    UN8 un8SegNum = 0;
    UN8 un8TargetSegNum = 0;
    UN16 un16TargetEpoch = 0;
    BOOLEAN bUnbrokenSchedule = TRUE;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;

    // Get locked EPG Owner
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"eLoadStoredSchedule(): Try to get facing object...\n"PC_RESET);
    psEpgOwner = psGetAppFacingObject(psEpgMgrObj);
    if (psEpgOwner == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"eLoadStoredSchedule(): Failed to get facing object.\n"PC_RESET);
        return EPG_RET_FAIL;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"eLoadStoredSchedule(): --> Facing object locked.\n"PC_RESET);

    // Ensure everything is ready for processing
    bResult = bInitScheduleProcessing(psEpgOwner,
                                      psEpgMgrObj->un8NumSegmentsToRecieve);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to init schedule processing.");

        SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"eLoadStoredSchedule(): <-- Facing object unlocked.\n"PC_RESET);

        return EPG_RET_FAIL;
    }

    psProcessingSchedule = psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

    // Before loading of segments, move all ready text files to cache files.
    vMoveReadyTextFilesToCache(psEpgOwner);

    un8TargetSegNum = 0;

    for (un8SegNum = 0; un8SegNum < un8NumSegmentsTotal; un8SegNum++)
    {
        // Try to load stored segment version
        bResult = bLoadStoredSegmentGridVersion(psEpgOwner,
                                                un8SegNum,
                                                &sVersion);
        if (bResult == FALSE)
        {
            if (un8SegNum == 0)
            {
                // It seems we have no stored schedule, so just skip loading.
                eResult = EPG_RET_OK;
                break;
            }
            else
            {
                continue;
            }
        }

        // Check if stored schedule is too old
        if (un8SegNum == 0)
        {
            UN32 un32UTCsec = 0;
            UN32 un32LastSegTimeSeconds = 0;
            OSAL.eTimeGet(&un32UTCsec); // Note: this is UTC
            un32LastSegTimeSeconds = (sVersion.un16Epoch + un8NumSegmentsTotal) * SECONDS_IN_DAY;
            if (un32LastSegTimeSeconds <= un32UTCsec)
            {
                SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                    PC_BLUE"STORED SCHEDULE IS TOO OLD. NO NEED TO LOAD IT.\n"PC_RESET);
                eResult = EPG_RET_OK;
                break;
            }

            // Save version and epoch from segment 0
            psProcessingSchedule->sVersion = sVersion;
            un16TargetEpoch = sVersion.un16Epoch;
        }
        else if (sVersion.un16Epoch != un16TargetEpoch)
        {
            // Schedule is compiled from different epochs
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"STORED SCHEDULE INTEGRITY BROKEN. LAST FULL SEGMENT NUMBER = %u. CURRENT EPOCH: %u. NEXT EPOCH: %u\n"PC_RESET,
                un8TargetSegNum, sVersion.un16Epoch, un16TargetEpoch);
            bUnbrokenSchedule = FALSE;
            continue;
        }

        if (sVersion.un8Version != psProcessingSchedule->sVersion.un8Version)
        {
            // A schedule is compiled from different versions
            bUnbrokenSchedule = FALSE;
        }

        // If some files were skipped,
        // it is required to move (rename) files
        // for proper text strings indexing.
        if (un8TargetSegNum != un8SegNum)
        {
            // Move GRID file
            bResult = bCopyEpgFile(psEpgOwner,
                                   EPG_FILE_GRID_BIN,
                                   un8SegNum,
                                   EPG_FILE_GRID_BIN,
                                   un8TargetSegNum,
                                   TRUE);
            if (bResult == FALSE)
            {
                eResult = EPG_RET_FAIL;
                break;
            }

            // Move TEXT file
            bResult = bCopyEpgFile(psEpgOwner,
                                   EPG_FILE_CACHE_BIN,
                                   un8SegNum,
                                   EPG_FILE_CACHE_BIN,
                                   un8TargetSegNum,
                                   TRUE);
            if (bResult == FALSE)
            {
                eResult = EPG_RET_FAIL;
                break;
            }
        }

        // Load TEXT data with version verification:
        // do not load segment which has different GRID and TEXT versions.
        eResult = eLoadStoredSegmentTextData(psEpgMgrObj,
                                             &sVersion,
                                             un8TargetSegNum);
        if (eResult == EPG_RET_OK)
        {
            // Load GRID
            eResult = eLoadStoredSegmentGridData(psEpgMgrObj,
                                                 un8TargetSegNum);
        }

        if (eResult != EPG_RET_OK)
        {
            if (eResult == EPG_RET_CANCELLED)
            {
                SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
                    PC_BLUE"Segment loading is cancelled. Stop loading.\n"PC_RESET);
                break;
            }
            else if (un8TargetSegNum == 0)
            {
                // Schedule cannot be collected without segment 0.
                // Stop loading.
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to load segment 0.");
                break;
            }
            else
            {
                // Just skip an error
                eResult = EPG_RET_OK;
            }
        }

        // Segment is successfully loaded. Update segment information.
        psProcessingSchedule->aeSegState[un8TargetSegNum] = EPG_SEG_STATE_LOADED;
        psProcessingSchedule->un8NumSegmentsCollected++;

        un16TargetEpoch++;
        un8TargetSegNum++;
    }

    if (eResult == EPG_RET_OK)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"***** SCHEDULE LOADING IS FINISHED (%u segments loaded; %s) *****\n"PC_RESET,
            psProcessingSchedule->un8NumSegmentsCollected,
            bUnbrokenSchedule == TRUE ? "UNBROKEN" : "BROKEN");

        // Check if something is loaded
        if (psProcessingSchedule->un8NumSegmentsCollected > 0)
        {
            vHandleLoadedScheduleCompletion(psEpgMgrObj, bUnbrokenSchedule);
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"***** SCHEDULE LOADING IS %s. KEEP EMPTY SCHEDULE. *****\n"PC_RESET,
            eResult == EPG_RET_CANCELLED ? "CANCELLED" : "FAILED");
    }

    // Delete entities required during processing.
    // They will be created when appropriate event from parser is received.
    vUninitScheduleProcessing(psEpgOwner);

     SMSO_vUnlock((SMS_OBJECT) psEpgOwner);
     SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
         PC_BLUE"eLoadStoredSchedule(): <-- Facing object unlocked.\n"PC_RESET);

    return eResult;
}

/*******************************************************************************
 *
 *  eLoadStoredSegmentGridData
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eLoadStoredSegmentGridData (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        UN8 un8SegNum
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = psEpgMgrObj->psEpgOwner;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tReadSize = 0;
    EPG_GRID_FILE_HEADER_STRUCT sGridFileHeader;
    UN32 un32AuIndex = 0;
    EPG_AU_INFO_STRUCT sAuData;
    size_t tAllocatedDataSize = 0;

    do
    {
        // Get segment grid file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_GRID_BIN,
                                    un8SegNum);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Open file
        pFile = fopen(pacFilePath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to open file: %s.",
                pacFilePath);
            break;
        }

        // Read file header
        tReadSize = fread(&sGridFileHeader,
                          sizeof(EPG_GRID_FILE_HEADER_STRUCT),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read GRID header from %s.",
                pacFilePath);
            break;
        }

        // Init data descriptor
        sAuData.eAuDataType = EPG_DATA_TYPE_GRID;
        sAuData.un8SegmentNumber = un8SegNum;
        sAuData.pvData = NULL;
        sAuData.tDataSize = 0;

        // Read AUs
        for (un32AuIndex = 0; un32AuIndex < sGridFileHeader.un32AusCount; un32AuIndex++)
        {
            // Read Au number
            tReadSize = fread(&(sAuData.un32AuNumber),
                              sizeof(UN32),
                              1,
                              pFile);
            if (tReadSize != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to read AU number from %s.",
                    pacFilePath);
                break;
            }

            // Read AU size
            tReadSize = fread(&(sAuData.tDataSize),
                              sizeof(size_t),
                              1,
                              pFile);
            if (tReadSize != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to read AU size from %s.",
                    pacFilePath);
                break;
            }

            // Reallocate buffer if required
            if (sAuData.pvData != NULL && sAuData.tDataSize > tAllocatedDataSize)
            {
                SMSO_vDestroy((SMS_OBJECT) sAuData.pvData);
                sAuData.pvData = NULL;
            }

            if (sAuData.pvData == NULL)
            {
                sAuData.pvData = (void*) SMSO_hCreate(EPG_TMP_PAYLOAD_BUF_OBJECT_NAME,
                                                      sAuData.tDataSize,
                                                      SMS_INVALID_OBJECT,
                                                      FALSE);
                if ((SMS_OBJECT) sAuData.pvData == SMS_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        EPG_MGR_OBJECT_NAME": Failed to create AU payload.");
                    break;
                }

                tAllocatedDataSize = sAuData.tDataSize;
            }

            // Read Grid payload
            tReadSize = fread(sAuData.pvData,
                              1,
                              sAuData.tDataSize,
                              pFile);
            if (tReadSize != sAuData.tDataSize)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to read GRID payload %u:%u from %s.",
                    un8SegNum, sAuData.un32AuNumber, pacFilePath);
                break;
            }

            // Process data
            eResult = GsEpgIntf.eLoadAuData(psEpgMgrObj->hEpgParserObject,
                                            &sAuData,
                                            bEpgLoadingCancellationCheckCallback,
                                            psEpgMgrObj);
            if (eResult != EPG_RET_OK)
            {
                break;
            }
        }

        if (sAuData.pvData != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) sAuData.pvData);
            sAuData.pvData = NULL;
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eLoadStoredSegmentTextData
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eLoadStoredSegmentTextData (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion,
        UN8 un8SegNum
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = psEpgMgrObj->psEpgOwner;
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tReadSize = 0;
    EPG_TEXT_FILE_HEADER_STRUCT sFileHeader;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;

    do
    {
        // Get appropriate text file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_CACHE_BIN,
                                    un8SegNum);
        if (pacFilePath == NULL)
        {
            eResult = EPG_RET_FAIL;
            break;
        }

        // Open file
        pFile = fopen(pacFilePath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to open CT file: %s.",
                pacFilePath);

            eResult = EPG_RET_FAIL;

            break;
        }

        // Read file header
        tReadSize = fread(&sFileHeader,
                          sizeof(EPG_TEXT_FILE_HEADER_STRUCT),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read CT header from %s.",
                pacFilePath);

            eResult = EPG_RET_FAIL;

            break;
        }

        // Check header data
        if (sFileHeader.un16StringsCount == 0 ||
            sFileHeader.un32CompressedSize == 0 ||
            sFileHeader.un32DecompressedSize == 0)
        {
            eResult = EPG_RET_INVALID_DATA;
            break;
        }

        // Verify version
        if (psVersion != NULL)
        {
            if (sFileHeader.un16Epoch != psVersion->un16Epoch ||
                sFileHeader.un8Version != psVersion->un8Version)
            {
                eResult = EPG_RET_INVALID_DATA;
                break;
            }
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return eResult;
}

/*******************************************************************************
 *
 *  bCreateMidnightEventCopy
 *
 *******************************************************************************/
static BOOLEAN bCreateMidnightEventCopy (
        EPG_OWNER_STRUCT *psEpgOwner,
        SERVICE_ID tServiceId,
        TIME_T tEventEndTime
            )
{
    PROG_EVENT_STRUCT *psProgramEvent = NULL;
    BOOLEAN bResult = TRUE;

    do
    {
        if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule
            == psEpgOwner->psCurrentSchedule)
        {
            // Schedule data is being collected to the Current Schedule
            // so no need to copy an event.
            bResult = FALSE;
            break;
        }

        // Try to get copy of midnight event found in the Current loaded Schedule.
        psProgramEvent = psGetMidnightEventCopy(psEpgOwner, tServiceId, tEventEndTime);
        if (psProgramEvent == NULL)
        {
            bResult = FALSE;
            break;
        }

        // Now it is required to copy text files from current schedule
        // to in-process with new numbers (higher than protocol's max number)
        // for proper midnight programs text strings indexing.
        // Copy file from epg_cache_<from>.bin to epg_text_<from+max>.bin.
        bResult = bCopyYesterdayTextFile(psEpgOwner,
                                         (UN8) psProgramEvent->tSchSeg,
                                         (UN8*) &(psProgramEvent->tSchSeg));
        if (bResult == FALSE)
        {
            break;
        }
        bResult = bCopyYesterdayTextFile(psEpgOwner,
                                         (UN8) psProgramEvent->tSchSegProgramDescription,
                                         (UN8*) &(psProgramEvent->tSchSegProgramDescription));
        if (bResult == FALSE)
        {
            break;
        }

        // Save midnight program to file
        bResult = bSaveMidnightProgramToFile(psEpgOwner,
                                             psProgramEvent);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    // Delete allocated event
    if (psProgramEvent != NULL)
    {
        EPG_PROGRAM_vDestroy(psProgramEvent);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  psGetMidnightEventCopy
 *
 *  The function tries to find Program Event in the current schedule
 *  by Service ID and End Time.
 *  Returns pointer to the event duplicated from found event.
 *
 *******************************************************************************/
PROG_EVENT_STRUCT *psGetMidnightEventCopy (
        EPG_OWNER_STRUCT *psEpgOwner,
        SERVICE_ID tServiceId,
        TIME_T tEventEndTime
            )
{
    EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor =
                psEpgOwner->psProcessingDescriptor;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
        psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    EPG_CHANNEL_OBJECT_STRUCT *psChannel = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    OSAL_LINKED_LIST_ENTRY hProgramEventEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    PROG_EVENT_STRUCT *psProgramEvent = NULL;

    // Try to find existing Program Event from currently loaded schedule  by end time
    // If Event is found, create its copy.

    do
    {
        // Check if current schedule is loaded
        if (psEpgOwner->psCurrentSchedule == NULL)
        {
            break;
        }

        // Current schedule's epoch should be older than processing
        if (psEpgOwner->psCurrentSchedule->sVersion.un16Epoch
            == psProcessingSchedule->sVersion.un16Epoch)
        {
            break;
        }

        // Get channel from Current schedule
        psChannel = psGetEpgChannel(psEpgOwner->psCurrentSchedule,
                                    tServiceId,
                                    FALSE);
        if (psChannel == NULL)
        {
            break;
        }

        // Find Event by end time
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psChannel->hEpgChannelEvents,
            &hProgramEventEntry,
            (OSAL_LL_COMPARE_HANDLER)EPG_PROGRAM_n16CompareEpgEventAndEndTime,
            &tEventEndTime);
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Get instance
        psProgramEvent = (PROG_EVENT_STRUCT *) OSAL.pvLinkedListThis(hProgramEventEntry);
        if (psProgramEvent == NULL)
        {
            break;
        }

        // Create new Program Event (copy of found event)
        // with currently processing channel as a parent.
        psProgramEvent = EPG_PROGRAM_psDuplicate(psProgramEvent,
                                                 (SMS_OBJECT) psDescriptor->psCurrentChannel,
                                                 psProgramEvent->tStartTime);
    } while (FALSE);

    // Everything is done
    return psProgramEvent;
}

/*******************************************************************************
 *
 *  bCopyYesterdayTextFile
 *
 *******************************************************************************/
static BOOLEAN bCopyYesterdayTextFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8SegNumCopyFrom,
        UN8 *pun8SegNumCopiedTo
            )
{
    EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor =
                    psEpgOwner->psProcessingDescriptor;
    UN8 un8DstSegNum = un8SegNumCopyFrom + psDescriptor->un8NumSegmentsMax;
    BOOLEAN bResult = TRUE;

    do
    {
        if (psDescriptor->abYesterdayTextSaved[un8SegNumCopyFrom] == TRUE)
        {
            // OK. Segment's text is already copied.
            break;
        }

        // Copy text file
        bResult = bCopyEpgFile(psEpgOwner,
                               EPG_FILE_CACHE_BIN,
                               un8SegNumCopyFrom,
                               EPG_FILE_TEXT_BIN,
                               un8DstSegNum,
                               FALSE);
        if (bResult == FALSE)
        {
            break;
        }

        psDescriptor->abYesterdayTextSaved[un8SegNumCopyFrom] = TRUE;

    } while (FALSE);

    if (bResult == TRUE && pun8SegNumCopiedTo != NULL)
    {
        *pun8SegNumCopiedTo = un8DstSegNum;
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bSaveMidnightProgramToFile
 *
 *******************************************************************************/
static BOOLEAN bSaveMidnightProgramToFile (
        EPG_OWNER_STRUCT *psEpgOwner,
        PROG_EVENT_STRUCT *psProgramEvent
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tBytesWritten = 0;
    EPG_MIDNIGHT_FILE_HEADER_STRUCT *psMidnightFileHeader = NULL;
    UN8 un8NumTopics = 0;
    EPG_TOPICS_LIST_STRUCT *psTopicsList = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Get appropriate file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_MIDNIGHT_TMP,
                                    0);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Get saved file descriptor
        psMidnightFileHeader =
            &(psEpgOwner->psProcessingDescriptor->sMidnightFileDesc);

        // If there are no saved midnight programs for currently processing schedule,
        // overwrite the file, otherwise append data to the end.
        if (psMidnightFileHeader->un32NumPrograms == 0)
        {
            EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
                psEpgOwner->psProcessingDescriptor->psProcessingSchedule;

            // Open for writing
            pFile = fopen(pacFilePath, "wb");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open MIDNIGHT file %s for writing.",
                    pacFilePath);
                break;
            }

            // Write file header
            psMidnightFileHeader->un16Epoch = psProcessingSchedule->sVersion.un16Epoch;
            psMidnightFileHeader->un8Version = psProcessingSchedule->sVersion.un8Version;
            psMidnightFileHeader->un8ProgramSize = sizeof(PROG_EVENT_STRUCT);
            psMidnightFileHeader->un32NumPrograms = 0; // Will be updated when all programs are saved

            tBytesWritten = fwrite(psMidnightFileHeader,
                                   sizeof(EPG_MIDNIGHT_FILE_HEADER_STRUCT),
                                   1,
                                   pFile);
            if (tBytesWritten != 1)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to write MIDNIGHT header to %s.",
                    pacFilePath);
                break;
            }
        }
        else
        {
            // Open for appending
            pFile = fopen(pacFilePath, "ab");
            if (pFile == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to open MIDNIGHT file %s for appending.",
                    pacFilePath);
                break;
            }
        }

        // Write Program Event struct
        tBytesWritten = fwrite(psProgramEvent,
                               sizeof(PROG_EVENT_STRUCT),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write Program to %s.",
                pacFilePath);
            break;
        }

        // Get number of topics
        if (psProgramEvent->hEpgTopics != EPG_INVALID_TOPICS_LIST_OBJECT)
        {
            psTopicsList = (EPG_TOPICS_LIST_STRUCT*) psProgramEvent->hEpgTopics;
            un8NumTopics = psTopicsList->un8NumTopics;
        }

        // Write number of topics
        tBytesWritten = fwrite(&un8NumTopics,
                               sizeof(UN8),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write Topics Count to %s.",
                pacFilePath);
            break;
        }

        // Write Topics IDs
        if (psTopicsList != NULL &&
            psTopicsList->hTopicsList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eRetCode;

            eRetCode = OSAL.eLinkedListIterate(psTopicsList->hTopicsList,
                                (OSAL_LL_ITERATOR_HANDLER) bSaveTopicToFile,
                                pFile);
            if (eRetCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to save topics to %s.",
                    pacFilePath);
                break;
            }
        }

        // Program Event is successfully saved
        psMidnightFileHeader->un32NumPrograms++;

        bResult = TRUE;

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*****************************************************************************
 *
 *  bSaveTopicToFile
 *
 *****************************************************************************/
static BOOLEAN bSaveTopicToFile (
        TOPIC_OBJECT hTopic,
        FILE *pFile
            )
{
    EPG_TOPIC_STRUCT *psTopic = (EPG_TOPIC_STRUCT *) hTopic;
    size_t tWritten = 0;

    tWritten = fwrite(&(psTopic->tTopicId),
                      sizeof(TOPIC_ID),
                      1,
                      pFile);
    if (tWritten != 1)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to write Topic Id to file.");
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *  bLoadMidnightPrograms
 *
 *****************************************************************************/
static BOOLEAN bLoadMidnightPrograms (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    size_t tReadSize = 0;
    EPG_FULL_SCHEDULE_ROOT *psProcessingSchedule =
        psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    EPG_MIDNIGHT_FILE_HEADER_STRUCT sMidnightFileHeader;
    BOOLEAN bResult = FALSE;

    do
    {
        // Get appropriate file name
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_MIDNIGHT_BIN,
                                    0);
        if (pacFilePath == NULL)
        {
            break;
        }

        // Open file for reading
        pFile = fopen(pacFilePath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
                PC_RED"WARNING: Failed to open MIDNIGHT file for reading (%s).\n"PC_RESET,
                pacFilePath);
            break;
        }

        // Read file header
        tReadSize = fread(&sMidnightFileHeader,
                          sizeof(EPG_MIDNIGHT_FILE_HEADER_STRUCT),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read MIDNIGHT header from %s.",
                pacFilePath);
            break;
        }

        // Verify header data
        if (sMidnightFileHeader.un16Epoch !=
            psProcessingSchedule->sVersion.un16Epoch)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1, PC_RED"WARNING: Midnight file has different epoch. Not loaded.\n"PC_RESET);
            break;
        }
        if (sMidnightFileHeader.un8ProgramSize != sizeof(PROG_EVENT_STRUCT))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Read Midnight Program Event size is invalid.");
            break;
        }

        // Load Program Events to currently processing schedule
        bResult = bLoadProgramEventsFromFile(pFile,
                                             psProcessingSchedule,
                                             sMidnightFileHeader.un32NumPrograms);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read Midnight Programs from %s.",
                pacFilePath);
            break;
        }

        bResult = TRUE;

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*****************************************************************************
 *
 *  bLoadProgramEventsFromFile
 *
 *****************************************************************************/
static BOOLEAN bLoadProgramEventsFromFile (
        FILE *pFile,
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        UN32 un32NumProgramsToRead
            )
{
    size_t tReadSize = 0;
    PROG_EVENT_STRUCT sReadProgramEvent;
    EPG_CHANNEL_OBJECT_STRUCT *psChannel = NULL;
    PROG_EVENT_STRUCT *psNewProgramEvent = NULL;
    BOOLEAN bResult = TRUE;
    UN8 un8NumTopics = 0;

    while (un32NumProgramsToRead > 0)
    {
        // Read Program Event struct
        tReadSize = fread(&sReadProgramEvent,
                          sizeof(PROG_EVENT_STRUCT),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read Program Event from file.");

            bResult = FALSE;

            break;
        }
        sReadProgramEvent.hEpgTopics = EPG_INVALID_TOPICS_LIST_OBJECT;

        // Find channel
        psChannel = psGetEpgChannel(psSchedule,
                                    sReadProgramEvent.tServiceId,
                                    FALSE);
        if (psChannel == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Channel (SID=%d) is not found.",
                sReadProgramEvent.tServiceId);

            bResult = FALSE;

            break;
        }

        // Read Topics Count
        tReadSize = fread(&un8NumTopics,
                          sizeof(UN8),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read Topics Count from file.");

            bResult = FALSE;

            break;
        }

        // Duplicate event
        psNewProgramEvent = EPG_PROGRAM_psDuplicate(&sReadProgramEvent,
                                                    (SMS_OBJECT) psChannel,
                                                    sReadProgramEvent.tStartTime);
        if (psNewProgramEvent == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to duplicate Program Event.");

            bResult = FALSE;

            break;
        }

        // Read topics
        bResult = bReadTopicsFromFile(pFile,
                                      psSchedule,
                                      psNewProgramEvent,
                                      un8NumTopics);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read Topics from file.");
            break;
        }

        // Add Program Event to Channel
        bResult = EPG_CHANNEL_bAddProgramEventToEpgChannel(psChannel,
                                                           psNewProgramEvent);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to add Program Event to Channel.");
            break;
        }

        // Event is successfully added
        psNewProgramEvent = NULL;
        un32NumProgramsToRead--;
    };

    // If failed, delete allocated event
    if (psNewProgramEvent != NULL)
    {
        EPG_PROGRAM_vDestroy(psNewProgramEvent);
    }

    return bResult;
}

/*****************************************************************************
 *
 *  bReadTopicsFromFile
 *
 *****************************************************************************/
static BOOLEAN bReadTopicsFromFile (
        FILE *pFile,
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        UN8 un8NumTopicsToRead
            )
{
    size_t tReadSize = 0;
    TOPIC_ID tReadTopicId = TOPIC_INVALID_ID;
    BOOLEAN bResult = TRUE;

    // Delete existing topics
    EPG_TOPICS_LIST_vDestroy((EPG_TOPICS_LIST_STRUCT *) psProgramEvent->hEpgTopics);
    psProgramEvent->hEpgTopics = EPG_INVALID_TOPICS_LIST_OBJECT;

    while (un8NumTopicsToRead > 0)
    {
        // Read topic Id
        tReadSize = fread(&tReadTopicId,
                          sizeof(TOPIC_ID),
                          1,
                          pFile);
        if (tReadSize != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to read Topic Id from file.");

            bResult = FALSE;

            break;
        }

        // Save Topic
        bResult = bSaveTopicInfo(psSchedule,
                                 psProgramEvent,
                                 tReadTopicId,
                                 NULL);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to save Topic Id.");
            break;
        }

        un8NumTopicsToRead--;
    }

    return bResult;
}

/*****************************************************************************
 *
 *  bMoveProcessingMidnightFileToReady
 *
 *****************************************************************************/
static BOOLEAN bMoveProcessingMidnightFileToReady (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    char *pacFilePath = NULL;
    FILE *pFile = NULL;
    UN8 un8FileAttr = 0;
    BOOLEAN bResult = TRUE;
    EPG_MIDNIGHT_FILE_HEADER_STRUCT *psMidnightFileHeader = NULL;
    size_t tBytesWritten = 0;

    // 1) Update MIDNIGHT file header
    // 2) Rename file from .tmp to .bin

    do
    {
        // Get appropriate file path
        pacFilePath = pcGetFilePath(psEpgOwner,
                                    EPG_FILE_MIDNIGHT_TMP, 0);
        if (pacFilePath == NULL)
        {
            bResult = FALSE;
            break;
        }

        // Check file existence
        bResult = OSAL.bFileSystemGetFileAttributes(pacFilePath, &un8FileAttr);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                "WARNING: File does not exist: %s.\n",
                pacFilePath);
            break;
        }

        // Check appropriate file descriptor
        psMidnightFileHeader =
            &(psEpgOwner->psProcessingDescriptor->sMidnightFileDesc);
        if (psMidnightFileHeader->un32NumPrograms == 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Midnight file descriptor is incomplete: %s.",
                pacFilePath);

            bResult = FALSE;

            break;
        }

        // Write updated descriptor to the file
        // Open file for update
        pFile = fopen(pacFilePath, "rb+");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to open file for update: %s.",
                pacFilePath);

            bResult = FALSE;

            break;
        }

        fseek(pFile, 0, SEEK_SET);

        // Write data
        tBytesWritten = fwrite(psMidnightFileHeader,
                               sizeof(EPG_MIDNIGHT_FILE_HEADER_STRUCT),
                               1,
                               pFile);
        if (tBytesWritten != 1)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to write MIDNIGHT header to %s.",
                pacFilePath);

            bResult = FALSE;

            break;
        }

        // Close file
        fclose(pFile);
        pFile = NULL;

        // Rename file from .tmp to .bin
        bResult = bCopyEpgFile(psEpgOwner,
                               EPG_FILE_MIDNIGHT_TMP, 0,
                               EPG_FILE_MIDNIGHT_BIN, 0,
                               TRUE);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to rename MIDNIGHT file.");

            bResult = FALSE;

            break;
        }

    } while (FALSE);

    // Close file if it was opened
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    // Delete string if it was allocated
    if (pacFilePath != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pacFilePath);
    }

    return bResult;
}

/*****************************************************************************
 *
 *  vMoveReadyTextFilesToCache
 *
 *****************************************************************************/
static void vMoveReadyTextFilesToCache (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    UN8 un8SegNum = 0;

    // Rename all existing epg_text_<num>.bin file to epg_cache_<num>.bin

    for (un8SegNum = 0;
         un8SegNum < (2 * GsEpgIntf.un8MaxTotalSegments);
         un8SegNum++)
    {
        bCopyEpgFile(psEpgOwner,
                     EPG_FILE_TEXT_BIN,
                     un8SegNum,
                     EPG_FILE_CACHE_BIN,
                     un8SegNum,
                     TRUE);
    }

    return;
}

/* Events handlers */

/*****************************************************************************
 *
 *   vEventHandler
 *
 *   This function runs in the context of an SMS resource which has been
 *   assigned to this service.
 *
 *****************************************************************************/
static void vEventHandler (
        DATASERVICE_MGR_OBJECT hDataService,
        DATASERVICE_EVENT_MASK tCurrentEvent,
        void *pvEventArg,
        void *pvEventCallbackArg
            )
{
    BOOLEAN bResult = FALSE;
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *)pvEventCallbackArg;
    SMSAPI_EVENT_MASK tEventMask = DATASERVICE_EVENT_NONE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE EPG EVENT\n"PC_RESET);

    // Only handle events for valid object
    bResult = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL)psEpgMgrObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Invalid EPG MGR object.");
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5, PC_GREEN"<<< EPG EVENT HANDLED.\n\n"PC_RESET);
        return;
    }

    switch (tCurrentEvent)
    {
        // Handle EPG Service events here...
        // State has been changed
        case DATASERVICE_EVENT_STATE:
        {
            BOOLEAN bStateChanged = FALSE;
            DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange =
                (DATASERVICE_STATE_CHANGE_STRUCT const *)pvEventArg;
            if (psStateChange == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": No state change data.");
                break;
            }
            // Process the state transition
            bStateChanged = DATASERVICE_IMPL_bStateFSM(
                    (DATASERVICE_IMPL_HDL)psEpgMgrObj,
                    psStateChange,
                    &GsEPGStateHandlers,
                    (void *)psEpgMgrObj);
            if (bStateChanged == TRUE)
            {
                // The state has been updated
                tEventMask |= DATASERVICE_EVENT_STATE;
            }
        }
        break;

        // A decoder has subscribed to this service
        case DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED:
        {
            DECODER_OBJECT hDecoder = (DECODER_OBJECT)pvEventArg;
            vHandleDecoderSubscribed(psEpgMgrObj, hDecoder);
        }
        break;

        // A decoder has unsubscribed from this service
        case DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED:
        {
            DECODER_OBJECT hDecoder = (DECODER_OBJECT)pvEventArg;
            vHandleDecoderUnsubscribed(psEpgMgrObj, hDecoder);
        }
        break;

        // This service has a message to process
        case DATASERVICE_EVENT_NEW_DATA:
        {
            OSAL_BUFFER_HDL hPayload = (OSAL_BUFFER_HDL)pvEventArg;
            bHandleMessageReception(psEpgMgrObj, hPayload);
        }
        break;

        // Handle the ePG service specific event type
        case DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC:
        {
            EPG_MGR_EVENT_STRUCT *psEvent = (EPG_MGR_EVENT_STRUCT *)pvEventArg;

            switch (psEvent->eEvent)
            {
                case EPG_MGR_STOP_PREPARE_EVENT:
                {
                    vHandleServiceStopRequest(psEpgMgrObj);
                }
                break;

                case EPG_MGR_LOADER_EVENT:
                {
                    bHandleEpgLoaderReport(psEpgMgrObj, psEvent->uEvent.sLoader.eLoaderResult);
                }
                break;

                default:
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        EPG_MGR_OBJECT_NAME": Unknown event.");
                }
                break;
            }
        }
        break;

        default:
            break;
    }

    if (tEventMask != DATASERVICE_EVENT_NONE)
    {
        // Update event mask with any relevant events which have occurred
        SMSU_tUpdate(&psEpgMgrObj->sEvent, tEventMask);
        // Notify of any change via any registered callback which may be present
        SMSU_bNotify(&psEpgMgrObj->sEvent);
    }

    if (tCurrentEvent == DATASERVICE_EVENT_STATE)
    {
        DATASERVICE_STATE_ENUM eFinalState;

        // If we're stopped, don't allow any more events to be generated
        eFinalState = DATASERVICE_IMPL_eState((DATASERVICE_IMPL_HDL)psEpgMgrObj);
        if (eFinalState == DATASERVICE_STATE_STOPPED)
        {
            // Filter out all further updates
            SMSU_tFilter(&psEpgMgrObj->sEvent, DATASERVICE_EVENT_ALL);

            vUninitEpgMgrObject(psEpgMgrObj);
        }
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< EPG EVENT HANDLED.\n\n"PC_RESET);

    return;
}

/*****************************************************************************
 *
 *  vHandleServiceStopRequest
 *
 *****************************************************************************/
static void vHandleServiceStopRequest (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"vHandleServiceStopRequest()\n"PC_RESET);

    // The flag indicates the service is currently stopping.
    // Required for proper service termination.
    psEpgMgrObj->bStopping = TRUE;

    // If Loader task is currently running, cancel processing.
    // Service stop will be initiated on event from the Loader task
    // when processing is cancelled.
    if (psEpgMgrObj->eEpgLoaderState == EPG_LOADER_STATE_PROCESSING)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"vHandleServiceStopRequest(): EPG LOADER IS CURRENTLY RUNNING. CANCEL PROCESSING.\n"PC_RESET);

        vEpgLoaderTaskCancel(psEpgMgrObj);
    }
    else
    {
        // Initiate service STOP now
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN"vHandleServiceStopRequest(): EPG LOADER IS NOT RUNNING. INITIATE EPG DS STOP NOW.\n"PC_RESET);

        vInitiateServiceStop(psEpgMgrObj);
    }

    return;
}

/*****************************************************************************
 *
 *  vInitiateServiceStop
 *
 *****************************************************************************/
static void vInitiateServiceStop (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    SMSAPI_DEBUG_vPrint(EPG_MGR_API_DBG_PREFIX, 6,
        PC_GREEN"vInitiateServiceStop()\n"PC_RESET);

    // No decoders are subscribed, so proceed with the stop now.
    DATASERVICE_IMPL_vLog(
        EPG_MGR_OBJECT_NAME
        ": INITIATE EPG DS STOP NOW.\n");

    DATASERVICE_IMPL_vStopAutonomous((DATASERVICE_IMPL_HDL)psEpgMgrObj);

    return;
}

/*****************************************************************************
 *
 *   bHandleServiceReady
 *
 *   This function is called when SMS is ready for the service to startup.
 *   At this time, the service has a context in which to operate (SMS
 *   assigned a resource to us). When this call is made, this service will
 *   perform all of its time-intensive startup procedures.
 *
 *****************************************************************************/
static BOOLEAN bHandleServiceReady (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    BOOLEAN bResult = TRUE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE SERVICE_READY EVENT\n"PC_RESET);

    do
    {
        // Check if EPG is already stopping before receipt of READY event.
        if (psEpgMgrObj->bStopping == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
                PC_BLUE"EPG IS CURRENTLY STOPPING, SO IGNORE SERVICE_READY EVENT.\n"PC_RESET);
            break;
        }

        bResult = bPrepareForInitialProcessing(psEpgMgrObj);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": FAILED TO INITIALIZE EPG MGR PROCESSING.");
            break;
        }

        bResult = bEpgLoaderTaskInitialize(psEpgMgrObj);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to ititialize EPG LOADER task.");
            break;
        }

        bResult = bEpgLoaderTaskStart(psEpgMgrObj);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to start EPG LOADER task.");

            vEpgLoaderTaskUninitialize(psEpgMgrObj);

            break;
        }

    } while (FALSE);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< SERVICE_READY EVENT HANDLED (%u).\n"PC_RESET, bResult);

    return bResult;
}

/*****************************************************************************
 *
 *   vHandleDecoderSubscribed
 *
 *   This function is called when a decoder subscribes to this service
 *   after it has already started.  This information is provided to give
 *   us a chance to handle any late-comers.
 *
 *****************************************************************************/
static void vHandleDecoderSubscribed (
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bSubscribed;
    BOOLEAN bInitialized = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE DECODER_SUBSCRIBED EVENT\n"PC_RESET);

    // Tell recently added decoder that EPG is started
    bSubscribed = bSendUpdate(psEpgMgrObj,
                              hDecoder,
                              EPG_SUB_UPDATE_TYPE_START);
    if (TRUE == bSubscribed)
    {
        // Tell the DSM that this decoder is now subscribed
        DATASERVICE_IMPL_vDecoderSubscribed(
            (DATASERVICE_IMPL_HDL)psEpgMgrObj, hDecoder);

        bInitialized = bIsEpgInitialized(psEpgMgrObj);
        if (TRUE == bInitialized)
        {
            // Provide recently added decoder with EPG schedule
            bSendUpdate(psEpgMgrObj,
                        hDecoder,
                        EPG_SUB_UPDATE_TYPE_PEM_CHANGE);
        }
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        "<<< DECODER_SUBSCRIBED EVENT HANDLED "
        "(subscribed: %u, initialized: %u)\n",
        bSubscribed, bInitialized);

    return;
}

/*****************************************************************************
 *
 *   vHandleDecoderUnsubscribed
 *
 *   This function is called when a decoder unsubscribes from this service.
 *
 *****************************************************************************/
static void vHandleDecoderUnsubscribed (
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
    DECODER_OBJECT hDecoder
        )
{
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE DECODER_UNSUBSCRIBED EVENT\n"PC_RESET);

    bSendUpdate(psEpgMgrObj, hDecoder, EPG_SUB_UPDATE_TYPE_STOP);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< DECODER_UNSUBSCRIBED EVENT HANDLED.\n"PC_RESET);

    return;
}

/*****************************************************************************
 *
 *   bHandleMessageReception
 *
 *   This function is called when a new message is received by the
 *   epg service.
 *
 *****************************************************************************/
static BOOLEAN bHandleMessageReception (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = TRUE;
    EPG_RETURN_CODES_ENUM eEpgRetCode = EPG_RET_OK;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE NEW_DATA EVENT\n"PC_RESET);

    do
    {
        if (hPayload == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Invalid payload data.");

            bResult = FALSE;

            break;
        }

        bResult = bIsEpgInitialized(psEpgMgrObj);
        if (bResult == FALSE)
        {
            break;
        }

        eEpgRetCode = GsEpgIntf.eProcessMessage(psEpgMgrObj->hEpgParserObject,
                                                hPayload,
                                                NULL,
                                                NULL);
        if (eEpgRetCode != EPG_RET_OK)
        {
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    if (hPayload != OSAL_INVALID_BUFFER_HDL)
    {
        DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< NEW_DATA EVENT HANDLED (%u).\n"PC_RESET, bResult);

    return bResult;
}

/*****************************************************************************
 *
 *   bHandleEpgLoaderReport
 *
 *****************************************************************************/
static BOOLEAN bHandleEpgLoaderReport (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_LOADER_RESULT_ENUM eLoaderResult
            )
{
    BOOLEAN bResult = TRUE;
    EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader = psEpgMgrObj->psEpgLoader;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> HANDLE LOADER_REPORT EVENT (loader result=%d)\n"PC_RESET,
        eLoaderResult);

    if (psEpgMgrObj->eEpgLoaderState != EPG_LOADER_STATE_PROCESSING)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": EPG Loader Task must be in PROCESSING state.");

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
            PC_GREEN"<<< LOADER_REPORT EVENT HANDLED (%u)\n"PC_RESET,
            FALSE);

        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bHandleEpgLoaderReport(): Try to lock EPG Loader...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgLoader, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to lock EPG LOADER.");

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
            PC_GREEN"<<< LOADER_REPORT EVENT HANDLED (%u)\n"PC_RESET,
            FALSE);

        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bHandleEpgLoaderReport(): --> EPG Loader locked.\n"PC_RESET);

    switch (eLoaderResult)
    {
    case EPG_LOADER_RESULT_PROCESSING_FINISHED:
    {
        // If processing was cancelled while posting of this rvent, ignore the event.
        if (psEpgLoader->bCancel == TRUE)
        {
            DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": SCHEDULE LOADING IS ALREADY CANCELLED. IGNORE LOADER REPORT.\n");
        }
        else
        {
            DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": SCHEDULE LOADING IS SUCCESSFULY FINISHED. NOTIFY EPG PEM_CHANGE.\n");

            psEpgMgrObj->bInitialEpgLoaded = TRUE;

            bSendUpdate(psEpgMgrObj,
                        DECODER_INVALID_OBJECT, // Notify all decoders
                        EPG_SUB_UPDATE_TYPE_PEM_CHANGE);
        }
    }
    break;

    case EPG_LOADER_RESULT_PROCESSING_CANCELLED:
        DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": SCHEDULE LOADING IS CANCELLED.\n");
        break;
    case EPG_LOADER_RESULT_PROCESSING_ERROR:
        DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": SCHEDULE LOADING IS FAILED.\n");
        break;
    default:
        break;
    }

    // Reset cancellation flag
    psEpgLoader->bCancel = FALSE;

    SMSO_vUnlock((SMS_OBJECT) psEpgLoader);
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bHandleEpgLoaderReport(): <-- EPG Loader unlocked.\n"PC_RESET);

    psEpgMgrObj->eEpgLoaderState = EPG_LOADER_STATE_IDLE;

    // If EPG is currently stopping, initiate service stop.
    if (psEpgMgrObj->bStopping == TRUE)
    {
        vInitiateServiceStop(psEpgMgrObj);
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< LOADER_REPORT EVENT HANDLED (%u)\n"PC_RESET,
        bResult);

    return bResult;
}

/*****************************************************************************
 *
 *  bSendUpdate
 *
 *****************************************************************************/
static BOOLEAN bSendUpdate (
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
    DECODER_OBJECT hDecoder,
    EPG_SUB_UPDATE_TYPE_ENUM eUpdateType
        )
{
    EPG_SUB_UPDATE_STRUCT sUpdate;
    BOOLEAN bResult = TRUE;

    DATASERVICE_IMPL_vLog(
        EPG_MGR_OBJECT_NAME": Updating subscribed decoder(s) (type = %u)\n",
        eUpdateType);

    // Populate the update structure
    sUpdate.psEpgMgrObj = psEpgMgrObj;
    sUpdate.eUpdateType = eUpdateType;
    sUpdate.bSuccess = TRUE;

    // Do we have a specific destination for this event?
    if (DECODER_INVALID_OBJECT != hDecoder)
    {
        // Send the event direct to this decoder.  Saves
        // a bit of overhead and also is useful when we are
        // starting a new subscription because the destination
        // decoder would not yet be in our subscribed list, which
        // means that DATASERVICE_IMPL_bUpdateSubscribed is
        // not going to help us in that situation.
        bResult = bIterateSubscribedForEvent(hDecoder, &sUpdate);
    }
    else
    {
        // Iterate subscribed decoders
        bResult = DATASERVICE_IMPL_bUpdateSubscribed(
            (DATASERVICE_IMPL_HDL)psEpgMgrObj,
            hDecoder, // All subscribed if hDecoder == DECODER_INVALID_OBJECT
            bIterateSubscribedForEvent,
            &sUpdate);
    }

    // Put 'em together and what have you got?
    bResult &= sUpdate.bSuccess;

    if (FALSE == bResult)
    {
        DATASERVICE_IMPL_vLog(
            EPG_MGR_OBJECT_NAME": Failed to update subscribed decoders.\n");
    }

    return bResult;
}

/*****************************************************************************
 *
 *   bUpdateChannelsEpgData
 *
 *****************************************************************************/
static BOOLEAN bSendEpgUpdateEvent (
        DECODER_OBJECT hDecoder,
        SMS_EVENT_EPG_STRUCT *psUpdateData
            )
{
    BOOLEAN bResult;

    bResult = SMSO_bValid((SMS_OBJECT) hDecoder);
    if (bResult == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData = NULL;

        // Allocate an event
        hEvent = DECODER_hAllocateEvent(hDecoder,
                                        SMS_EVENT_EPG,
                                        SMS_EVENT_OPTION_NONE,
                                        &puEventData);
        if (hEvent != SMS_INVALID_EVENT_HDL)
        {
            puEventData->uDecoder.sEpg = *psUpdateData;

            bResult = SMSE_bPostEvent(hEvent);
        }
    }

    return bResult;
}

/* Linked List Iterators / Searchers */

/*******************************************************************************
 *
 *   bIterateSubscribedForEvent
 *
 *******************************************************************************/
static BOOLEAN bIterateSubscribedForEvent (
        DECODER_OBJECT hDecoder,
        void *pvArg
            )
{
    EPG_SUB_UPDATE_STRUCT *psUpdate = (EPG_SUB_UPDATE_STRUCT *) pvArg;
    SMS_EVENT_EPG_STRUCT sUpdateData;
    BOOLEAN bContinue = TRUE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        "Enter Iterate subscribed.\n");

    sUpdateData.hEpgService =
        (EPG_SERVICE_OBJECT) psUpdate->psEpgMgrObj;

    switch (psUpdate->eUpdateType)
    {
        case EPG_SUB_UPDATE_TYPE_START:
        {
            sUpdateData.eStatus = SMS_EVENT_EPG_STATUS_START;

            DATASERVICE_IMPL_vLog(
                EPG_MGR_OBJECT_NAME
                ": Sending START event to subscribed decoder (0x%X).\n",
                hDecoder);
        }
        break;

        case EPG_SUB_UPDATE_TYPE_PEM_CHANGE:
        {
            sUpdateData.eStatus = SMS_EVENT_EPG_STATUS_UPDATE;

            DATASERVICE_IMPL_vLog(
                EPG_MGR_OBJECT_NAME
                ": Sending UPDATE event to subscribed decoder (0x%X).\n",
                hDecoder);
        }
        break;

        case EPG_SUB_UPDATE_TYPE_STOP:
        {
            sUpdateData.eStatus = SMS_EVENT_EPG_STATUS_STOP;

            DATASERVICE_IMPL_vLog(
                EPG_MGR_OBJECT_NAME
                ": Sending STOP event to subscribed decoder (0x%X).\n",
                hDecoder);
        }
        break;

        case EPG_SUB_UPDATE_INVALID_TYPE:
        // Fall thru
        default:
        {
            bContinue = FALSE;
            psUpdate->bSuccess = FALSE;
        }
        break;
    }

    if (bContinue == TRUE)
    {
        BOOLEAN bResult;

        bResult = bSendEpgUpdateEvent(hDecoder, &sUpdateData);
        if (FALSE == bResult)
        {
            psUpdate->bSuccess = FALSE;
        }
    }

    return bContinue;
}

/*****************************************************************************
 *
 *   bEpgChannelsIterator
 *   - does filter epg channels
 *
 *****************************************************************************/
static BOOLEAN bEpgChannelsIterator (
        EPG_CHANNEL_OBJECT hEpgChannel,
        EPG_EVENTS_ITERATOR_STRUCT *psIterator
            )
{
    EPG_CHANNEL_OBJECT_STRUCT *psEpgChannel = (EPG_CHANNEL_OBJECT_STRUCT *)hEpgChannel;
    BOOLEAN bMatch = FALSE;

    if (psIterator == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
            "ERROR: Cease iterating. The caller didn't provide an Iterator.\n");
        return FALSE;
    }

    if (hEpgChannel == EPG_INVALID_CHANNEL_OBJECT)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
            "ERROR: Cease iterating. The caller didn't provide a EPG Channel.\n");
        psIterator->bContinue = FALSE;
        return FALSE;
    }

    if (psIterator->bEpgIterator == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
            "ERROR: Cease iterating. The caller didn't provide a Callback.\n");
        psIterator->bContinue = FALSE;
        return FALSE;
    }

    psIterator->bContinue = TRUE;

    if (psIterator->psEpgFilter == NULL)
    {
        bMatch = TRUE;
    }
    else if (psIterator->psEpgFilter->u8NumChannels == 0)
    {
        bMatch = TRUE;
    }
    else if (psIterator->psEpgFilter->ptChannels == NULL)
    {
        bMatch = TRUE;
    }
    else
    {
        UN8 un8Chan = 0;
        for (un8Chan = 0; un8Chan < psIterator->psEpgFilter->u8NumChannels; un8Chan++)
        {
            if (psEpgChannel->tServiceId == psIterator->psEpgFilter->ptChannels[un8Chan])
            {
                bMatch = TRUE;
                break;
            }
        }
    }

    // If there we found an EPG event, call the iterator, otherwise keep iterating.
    if (bMatch == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR_UNKNOWN;

        if (psIterator->psEpgFilter != NULL && psIterator->psEpgFilter->bUnique == TRUE)
        {
            eOsalReturnCode = OSAL.eLinkedListIterate(
                    psEpgChannel->hUniquePrograms,
                    (OSAL_LL_ITERATOR_HANDLER)bProgramsIterator,
                    psIterator);
        }
        else
        {
            eOsalReturnCode = OSAL.eLinkedListIterate(
                    psEpgChannel->hEpgChannelEvents,
                    (OSAL_LL_ITERATOR_HANDLER)bProgramsIterator,
                    psIterator);
        }

        if (eOsalReturnCode != OSAL_SUCCESS &&
            eOsalReturnCode != OSAL_NO_OBJECTS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": EPG Channels iterate error code = %d.",
                eOsalReturnCode);
        }
    }

    return psIterator->bContinue;
}

/*****************************************************************************
 *
 *   bProgramsIterator
 *   - does filter programs
 *
 *****************************************************************************/
static BOOLEAN bProgramsIterator (
        PROGRAM_OBJECT hProgEvent,
        EPG_EVENTS_ITERATOR_STRUCT *psIterator
            )
{
    PROG_EVENT_STRUCT *psEpgEvent = (PROG_EVENT_STRUCT*)hProgEvent;
    BOOLEAN bMatch = TRUE;
    STRING_OBJECT hOrigString = STRING_INVALID_OBJECT;

    if (psIterator == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
            "ERROR: Cease iterating. The caller didn't provide an Iterator.\n");
        return FALSE;
    }

    if (psEpgEvent == NULL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
            "ERROR: Cease iterating. The caller didn't provide a Program Event.\n");
        psIterator->bContinue = FALSE;
        return FALSE;
    }

    if (psIterator->bEpgIterator == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME
            ": Cease iterating. The caller didn't provide a Callback.");

        psIterator->bContinue = FALSE;

        return FALSE;
    }

    psIterator->bContinue = TRUE;

    // Checking through filter fields only if filter is provided
    // otherwise, assuming current entry matches
    do
    {
        if (psEpgEvent->tSchSeg >= psIterator->un8NumSegmentsAvailable)
        {
            // Ignore programs in incomplete segments
            bMatch = FALSE;
            break;
        }

        if (psIterator->psEpgFilter == NULL)
        {
            break;
        }

        /* TIME check */
        if (psIterator->psEpgFilter->tStartTime > 0)
        {
            if (psEpgEvent->tEndTime < psIterator->psEpgFilter->tStartTime)
            {
                bMatch = FALSE;
                break;
            }
        }

        if (psIterator->psEpgFilter->tEndTime > 0)
        {
            if (psEpgEvent->tStartTime > psIterator->psEpgFilter->tEndTime)
            {
                bMatch = FALSE;
                break;
            }
        }

        /* FLAGs check */
        if ((psIterator->psEpgFilter->tEpgFlags & EPG_PROGRAM_ALL_FLAGS) != 0)
        {
            if ((psEpgEvent->tPFlags & psIterator->psEpgFilter->tEpgFlags)
                != psIterator->psEpgFilter->tEpgFlags)
            {
                bMatch = FALSE;
                break;
            }
        }

        /* Recording Options check */
        if (psIterator->psEpgFilter->eRecordingOption !=
            EPG_PROGRAM_RECORD_UNDEFINED)
        {
            if (psEpgEvent->eRecordingOption !=
                psIterator->psEpgFilter->eRecordingOption)
            {
                bMatch = FALSE;
                break;
            }
        }

        /* STRINGs check */
        if (psIterator->psEpgFilter->hSearchSubstring != STRING_INVALID_OBJECT)
        {
            BOOLEAN bOk;
            char *pcSearchString = NULL;
            N16 n16Pos = N16_MIN;

            pcSearchString = (char*)STRING.pacCStr(psIterator->psEpgFilter->hSearchSubstring);
            hOrigString = STRING.hCreate("", 0);
            bOk = PROGRAM.bGetShortName(hProgEvent, hOrigString);
            if (bOk == TRUE)
            {
                n16Pos = STRING.n16SubstringCStr(hOrigString, pcSearchString, TRUE);
            }
            if (n16Pos < 0)
            {
                bOk = PROGRAM.bGetLongName(hProgEvent, hOrigString);
                if (bOk == TRUE)
                {
                    n16Pos = STRING.n16SubstringCStr(hOrigString, pcSearchString, TRUE);
                }
                if (n16Pos < 0)
                {
                    bOk = PROGRAM.bGetSeriesDescription(hProgEvent, hOrigString);
                    if (bOk == TRUE)
                    {
                        n16Pos = STRING.n16SubstringCStr(hOrigString, pcSearchString, TRUE);
                    }
                    if (n16Pos < 0)
                    {
                        bOk = PROGRAM.bGetProgramDescription(hProgEvent, hOrigString);
                        if (bOk == TRUE)
                        {
                            n16Pos = STRING.n16SubstringCStr(hOrigString, pcSearchString, TRUE);
                        }
                        if (n16Pos < 0)
                        {
                            bMatch = FALSE;
                            break;
                        }
                    }
                }
            }
        }

        /* SERIES check */
        if (psIterator->psEpgFilter->u16NumSeries > 0)
        {
            if (psIterator->psEpgFilter->ptSeries != NULL)
            {
                UN16 un16ProgIdx;
                bMatch = FALSE;
                for (un16ProgIdx = 0;
                     un16ProgIdx < psIterator->psEpgFilter->u16NumSeries;
                     un16ProgIdx++)
                {
                    if (psEpgEvent->tSeriesId == psIterator->psEpgFilter->ptSeries[un16ProgIdx])
                    {
                        bMatch = TRUE;
                        break;
                    }
                }

                if (bMatch == FALSE)
                {
                    // Exiting loop if Series ID does not match
                    break;
                }
            }
        }

        /* PROGRAMS check */
        if (psIterator->psEpgFilter->u16NumPrograms > 0)
        {
            if (psIterator->psEpgFilter->ptPrograms != NULL)
            {
                UN16 un16EpIdx;
                bMatch = FALSE;
                for (un16EpIdx = 0;
                     un16EpIdx < psIterator->psEpgFilter->u16NumPrograms;
                     un16EpIdx++)
                {
                    if (psEpgEvent->tProgramId == psIterator->psEpgFilter->ptPrograms[un16EpIdx])
                    {
                        bMatch = TRUE;
                        break;
                    }
                }

                if (bMatch == FALSE)
                {
                    // Exiting loop if Program ID does not match
                    break;
                }
            }
        }

#if 0 // Currently not supported. Reserved for future use.
        /* TOPICS check */
        if (psIterator->psEpgFilter->u16NumTopics > 0)
        {
            if (psIterator->psEpgFilter->ptTopics != NULL)
            {
                bMatch = FALSE;

                if (psEpgEvent->hEpgTopics == EPG_INVALID_TOPICS_LIST_OBJECT)
                {
                    // Topic list was set as search condition but no topics found for this program
                    bMatch = FALSE;
                    break;
                }

                {
                    UN16 un16TopicIdx = 0;
                    EPG_TOPICS_LIST_STRUCT *psTopics;

                    psTopics = (EPG_TOPICS_LIST_STRUCT*) psEpgEvent->hEpgTopics;
                    for (un16TopicIdx = 0; un16TopicIdx < psIterator->psEpgFilter->u16NumTopics; un16TopicIdx++)
                    {
                        bMatch = bCheckIfTopicInList(psTopics,
                            psIterator->psEpgFilter->ptTopics[un16TopicIdx]);
                        if (bMatch == TRUE)
                        {
                            break;
                        }
                    }//end "for" topics
                }

                if (bMatch == FALSE)
                {
                    // Exiting loop if Topics do not match
                    break;
                }
            }
        }//end topics check
#endif
    } while (FALSE);

    if (hOrigString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hOrigString);
    }

    if (bMatch == TRUE)
    {
        psIterator->bContinue = psIterator->bEpgIterator(psEpgEvent,
                                                         psIterator->pvIteratorArg);
    }

    return psIterator->bContinue;
}

/* EPG schedule loader task functions */

/*****************************************************************************
 *
 *  bEpgLoaderTaskInitialize
 *
 *****************************************************************************/
static BOOLEAN bEpgLoaderTaskInitialize (
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
        )
{
    EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader = psEpgMgrObj->psEpgLoader;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"INITIALIZE EPG LOADER TASK...\n"PC_RESET);

    /* Check if EPG Loader is already initialized */
    if (psEpgLoader != NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": EPG LOADER is already initialized.");

        return FALSE;
    }

    // Create an instance of the object (initially locked)
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgLoaderTaskInitialize(): Try to create locked EPG Loader...\n"PC_RESET);
    psEpgLoader = (EPG_MGR_LOADER_CONTROL_STRUCT *)SMSO_hCreate(EPG_LOADER_OBJECT_NAME,
                                                    sizeof(EPG_MGR_LOADER_CONTROL_STRUCT),
                                                    SMS_INVALID_OBJECT, // No parent
                                                    TRUE); // Self-locked
    if (psEpgLoader == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Unable to allocate memory");
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgTaskInitialize(): --> EPG Loader created (locked).\n"PC_RESET);

    psEpgLoader->hEpgService = (EPG_SERVICE_OBJECT) psEpgMgrObj;

    // Initialize event to send notification to the EPG MGR
    SMSU_vInitialize(&(psEpgLoader->sLoaderEvent),
                     psEpgLoader,
                     EPG_LOADER_EVENT_NONE,
                     EPG_LOADER_EVENT_ALL,
                     vEpgLoaderEventCallback,
                     psEpgMgrObj);

    // Install SMS Task now.
    // Note: successful installation of this task will cause the
    // provided SMS object (psEpgLoader) to be owned by the SMS task.
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgLoaderTaskInitialize(): Try to create Loader Task...\n"PC_RESET);
    psEpgLoader->hLoaderTask = SMST_hInstall((SMS_OBJECT)psEpgLoader,
                                             &gsEPGMTaskConfiguration,
                                             vEpgLoaderTaskEventHandler,
                                             &psEpgLoader->hEventHandler);
    if (psEpgLoader->hLoaderTask == SMS_INVALID_TASK_HANDLE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": EPG Loader task could not be installed.");

        SMSU_vDestroy(&(psEpgLoader->sLoaderEvent));
        SMSO_vDestroy((SMS_OBJECT)psEpgLoader);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"bEpgLoaderTaskInitialize(): EPG LOADER DESTROYED.\n"PC_RESET);
        return FALSE;
    }
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"bEpgLoaderTaskInitialize(): <-- EPG Loader unlocked.\n"PC_RESET);

    psEpgMgrObj->psEpgLoader = psEpgLoader;
    psEpgMgrObj->eEpgLoaderState = EPG_LOADER_STATE_IDLE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"EPG LOADER TASK IS INITIALIZED.\n"PC_RESET);

    return TRUE;
}

/*****************************************************************************
 *
 *  vEpgLoaderTaskUninitialize
 *
 *  NOTE: All task's activities must be surely stopped before this call.
 *****************************************************************************/
static void vEpgLoaderTaskUninitialize (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"DEINIT EPG LOADER TASK...\n"PC_RESET);

    if (psEpgMgrObj->psEpgLoader != NULL)
    {
        SMSU_vDestroy(&(psEpgMgrObj->psEpgLoader->sLoaderEvent));

        // NOTE: psEpgLoader object will be destroyed upon task destruction since it's owned by the task
        SMST_vUninstall(psEpgMgrObj->psEpgLoader->hLoaderTask);

        psEpgMgrObj->psEpgLoader = NULL;
        psEpgMgrObj->eEpgLoaderState = EPG_LOADER_STATE_UNINITIALIZED;

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_GREEN"EPG LOADER TASK DEINITIALIZED.\n"PC_RESET);
    }

    return;
}

/*****************************************************************************
 *
 *  bEpgLoaderTaskStart
 *
 *****************************************************************************/
static BOOLEAN bEpgLoaderTaskStart (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"SIGNAL EPG LOADER TASK TO START PROCESSING...\n"PC_RESET);

    do
    {
        SMS_EVENT_DATA_UNION *puEventData = NULL;

        // New processing cannot be started during ongoing processing
        if (psEpgMgrObj->eEpgLoaderState != EPG_LOADER_STATE_IDLE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Invalid call: Current EpgLoaderState=%d.",
                psEpgMgrObj->eEpgLoaderState);
            break;
        }

        hEvent = SMSE_hAllocateEvent(psEpgMgrObj->psEpgLoader->hEventHandler,
                                     SMS_EVENT_DATASERVICE,
                                     &puEventData,
                                     SMS_EVENT_OPTION_NONE);
        if (hEvent == SMS_INVALID_EVENT_HDL)
        {
            break;
        }

        puEventData->sData.eDataServiceEvent =
            (DATASERVICE_FW_EVENT_ENUM) EPG_FW_EVENT_LOAD_INITIAL_SCHEDULE;

        bResult = SMSE_bPostEvent(hEvent);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Unable to start Loader task.");
            break;
        }

        psEpgMgrObj->eEpgLoaderState = EPG_LOADER_STATE_PROCESSING;

    } while (FALSE);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"EPG LOADER TASK START POSTING RESULT: %u.\n"PC_RESET, bResult);

    return bResult;
}

/*****************************************************************************
 *
 *  vEpgLoaderTaskCancel
 *
 *****************************************************************************/
static void vEpgLoaderTaskCancel (
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
        )
{
    BOOLEAN bLocked = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"CANCEL EPG LOADER TASK PROCESSING...\n"PC_RESET);

    if (psEpgMgrObj->eEpgLoaderState != EPG_LOADER_STATE_PROCESSING)
    {
        return;
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgTaskCancel(): Try to lock EPG Loader...\n"PC_RESET);
    bLocked = SMSO_bLock((SMS_OBJECT) psEpgMgrObj->psEpgLoader, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgTaskCancel(): --> EPG Loader locked.\n"PC_RESET);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to lock EPG LOADER.");
    }

    psEpgMgrObj->psEpgLoader->bCancel = TRUE;

    if (bLocked == TRUE)
    {
        SMSO_vUnlock((SMS_OBJECT) psEpgMgrObj->psEpgLoader);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgTaskCancel(): <-- EPG Loader unlocked.\n"PC_RESET);
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
        PC_GREEN"CANCEL FLAG IS SET.\n"PC_RESET);

    return;
}

/*****************************************************************************
 *
 *  vEpgLoaderEventCallback
 *
 *****************************************************************************/
static void vEpgLoaderEventCallback (
    void *pvObj,
    SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj =
        (EPG_MGR_OBJECT_STRUCT *)pvEventCallbackArg;
    BOOLEAN bPosted = FALSE;
    EPG_LOADER_RESULT_ENUM eLoaderResult;
    EPG_MGR_EVENT_STRUCT *psLoaderEvent;
    SMS_EVENT_HDL hEvent;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"EPG LOADER: vEpgLoaderEventCallback(tEventMask=0x%X)\n"PC_RESET,
        tEventMask);

    switch (tEventMask)
    {
        case EPG_LOADER_EVENT_PROCESSING_FINISHED:
        {
            eLoaderResult = EPG_LOADER_RESULT_PROCESSING_FINISHED;
        }
        break;

        case EPG_LOADER_EVENT_PROCESSING_CANCELLED:
        {
            eLoaderResult = EPG_LOADER_RESULT_PROCESSING_CANCELLED;
        }
        break;

        case EPG_LOADER_EVENT_PROCESSING_ERROR:
        {
            eLoaderResult = EPG_LOADER_RESULT_PROCESSING_ERROR;
        }
        break;

        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Invalid EPG LOADER event.");
            return;
        }
    }

    // Allocate an event for this
    hEvent = DATASERVICE_IMPL_hAllocateEvent(
        (DATASERVICE_IMPL_HDL)psEpgMgrObj, (void **)&psLoaderEvent);

    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        // Indicate this is a loader event
        psLoaderEvent->eEvent = EPG_MGR_LOADER_EVENT;

        // Populate the event argument
        psLoaderEvent->uEvent.sLoader.eLoaderResult = eLoaderResult;

        // Post it now
        bPosted = SMSE_bPostEvent(hEvent);
    }

    if (bPosted == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to post EPG LOADER event.");
    }

    return;
}

/*******************************************************************************
 *
 *  vEpgLoaderTaskEventHandler
 *
 *  This function runs in the context of the EPG LOADER task
 *
 *******************************************************************************/
static void vEpgLoaderTaskEventHandler (
        SMS_OBJECT hEpgLoader,
        SMS_EVENT_HDL hEvent
            )
{
    EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader =
                    (EPG_MGR_LOADER_CONTROL_STRUCT *)hEpgLoader;
    SMS_EVENT_DATA_UNION const *puEventData = NULL;
    SMS_EVENT_TYPE_ENUM eEventType;

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_GREEN"EPG LOADER: LOADER TASK HANDLER (EVENT=%d)...\n"PC_RESET,
        eEventType);

    if (eEventType == SMS_EVENT_DATASERVICE)
    {
        const EPG_FW_EVENT_ENUM eDataServiceEvent =
            (const EPG_FW_EVENT_ENUM)puEventData->sData.eDataServiceEvent;
        if (eDataServiceEvent == EPG_FW_EVENT_LOAD_INITIAL_SCHEDULE)
        {
            vEpgTaskLoadInitialSchedule(psEpgLoader);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  vEpgTaskLoadInitialSchedule
 *
 *****************************************************************************/
static void vEpgTaskLoadInitialSchedule (
        EPG_MGR_LOADER_CONTROL_STRUCT *psEpgLoader
            )
{
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *)(psEpgLoader->hEpgService);
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;
    SMSAPI_EVENT_MASK tEventMask = EPG_LOADER_EVENT_NONE;
    BOOLEAN bLocked = FALSE;

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN">>> LOAD INITIAL SCHEDULE\n"PC_RESET);

    eResult = eLoadStoredSchedule(psEpgMgrObj);

    // Handle result
    if (eResult == EPG_RET_CANCELLED || psEpgLoader->bCancel == TRUE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN" PROCESSING IS CANCELLED. SEND PROCESSING_CANCELLED EVENT.\n"PC_RESET);
        tEventMask = EPG_LOADER_EVENT_PROCESSING_CANCELLED;
    }
    else if (eResult == EPG_RET_OK)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_GREEN" PROCESSING IS OK. SEND PROCESSING_FINISHED EVENT.\n"PC_RESET);
        tEventMask = EPG_LOADER_EVENT_PROCESSING_FINISHED;
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_BLUE"EPG LOADER: PROCESSING ERROR. SEND PROCESSING_ERROR EVENT.\n"PC_RESET);
        tEventMask = EPG_LOADER_EVENT_PROCESSING_ERROR;
    }

    // Try to lock EPG Loader
    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
        PC_BLUE"vEpgTaskLoadInitialSchedule(): Try to lock EPG Loader...\n"PC_RESET);
    bLocked = SMSO_bLock((SMS_OBJECT) psEpgLoader, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgTaskLoadInitialSchedule(): --> EPG Loader locked.\n"PC_RESET);

        // Notify EPG LOADER task's initiator
        SMSU_tUpdate(&psEpgLoader->sLoaderEvent, tEventMask);
        SMSU_bNotify(&psEpgLoader->sLoaderEvent);

        SMSO_vUnlock((SMS_OBJECT) psEpgLoader);
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 6,
            PC_BLUE"vEpgTaskLoadInitialSchedule(): <-- EPG Loader unlocked.\n"PC_RESET);
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Cannot lock EPG Loader! Loader report will not be posted!");
    }

    SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 5,
        PC_GREEN"<<< INITIAL SCHEDULE LOADING FINISHED.\n"PC_RESET);

    return;
}

/*****************************************************************************
 *
 *  bEpgLoadingCancellationCheckCallback
 *
 *****************************************************************************/
static BOOLEAN bEpgLoadingCancellationCheckCallback (
        void *pvCallbackArg
            )
{
    BOOLEAN bResult;
    EPG_MGR_OBJECT_STRUCT *psEpgMgrObj = (EPG_MGR_OBJECT_STRUCT *) pvCallbackArg;

    bResult = SMSO_bLock((SMS_OBJECT) psEpgMgrObj->psEpgLoader, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EPG_MGR_OBJECT_NAME": Failed to lock EPG LOADER. Return CANCEL=TRUE to cancel processing.");
        return TRUE;
    }

    bResult = psEpgMgrObj->psEpgLoader->bCancel;

    SMSO_vUnlock((SMS_OBJECT) psEpgMgrObj->psEpgLoader);

    if (bResult == TRUE)
    {
        SMSAPI_DEBUG_vPrint(EPG_LOADER_DBG_PREFIX, 1,
            PC_GREEN" CATCHED CANCELLATION.\n"PC_RESET);
    }

    return bResult;
}

/* Data processing */

/*******************************************************************************
 *
 *  un32CalculateEventDuration
 *
 *******************************************************************************/
static UN32 un32CalculateEventDuration (
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        TIME_T tStartTime
            )
{
    TIME_T tTimeVal = 0;

    // Calculate duration in seconds

    switch (psParserEvent->eDurationCode)
    {
        case EPG_DURATION_CODE_30MIN:
            tTimeVal = 30 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_1HOUR:
            tTimeVal = 60 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_2HOURS:
            tTimeVal = 120 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_3HOURS:
            tTimeVal = 180 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_4HOURS:
            tTimeVal = 240 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_5HOURS:
            tTimeVal = 300 * SECONDS_IN_MINUTE;
            break;
        case EPG_DURATION_CODE_END_OF_SEGMENT:
            // Time in seconds to the end of day since program event start
            // ( = start time of the next segment - event start time).
            // Calculate Epoch
            tTimeVal = tStartTime / SECONDS_IN_DAY;
            // Calculate start time of the next segment
            tTimeVal = (tTimeVal + 1) * SECONDS_IN_DAY;
            // Calculate duration value in seconds
            tTimeVal = tTimeVal - tStartTime;
            break;
        case EPG_DURATION_CODE_ESCAPE_TO_ODUR:
            tTimeVal = psParserEvent->un32OptionalDuration;
            break;
        default:
            tTimeVal = 0;
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid duration.\n"PC_RESET);
            break;
    }

    return (UN32) tTimeVal;
}

/*******************************************************************************
 *
 *  bInitScheduleProcessing
 *
 *******************************************************************************/
static BOOLEAN bInitScheduleProcessing (
        EPG_OWNER_STRUCT *psEpgOwner,
        UN8 un8NumSegments
            )
{
    do
    {
        // Check if processing descriptor is already allocated
        if (psEpgOwner->psProcessingDescriptor == NULL)
        {
            psEpgOwner->psProcessingDescriptor =
                psCreateProcessingDescriptor((SMS_OBJECT) psEpgOwner, GsEpgIntf.un8MaxTotalSegments);
            if (psEpgOwner->psProcessingDescriptor == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to create processing descriptor.");
                break;
            }
        }

        // Check if processing schedule object is already created
        if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule == NULL)
        {
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule =
                psCreateEpgSchedule((SMS_OBJECT) psEpgOwner, un8NumSegments);
            if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to create processing schedule.");
                break;
            }

            if (psEpgOwner->psCurrentSchedule == NULL)
            {
                // Allow EPG data to be accessed even during first schedule assembling.
                psEpgOwner->psCurrentSchedule =
                    psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
            }
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error case:
    vUninitScheduleProcessing(psEpgOwner);
    return FALSE;
}

/*******************************************************************************
 *
 *  vUninitScheduleProcessing
 *
 *******************************************************************************/
static void vUninitScheduleProcessing (
        EPG_OWNER_STRUCT *psEpgOwner
            )
{
    if (psEpgOwner->psProcessingDescriptor != NULL)
    {
        if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule != NULL)
        {
            if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule
                == psEpgOwner->psCurrentSchedule)
            {
                psEpgOwner->psCurrentSchedule = NULL;
            }

            vDestroyEpgSchedule(psEpgOwner->psProcessingDescriptor->psProcessingSchedule);
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule = NULL;
        }

        vDestroyProcessingDescriptor(psEpgOwner->psProcessingDescriptor);
        psEpgOwner->psProcessingDescriptor = NULL;
    }

    return;
}

/*******************************************************************************
 *
 *  psCreateEpgSchedule
 *
 *******************************************************************************/
static EPG_FULL_SCHEDULE_ROOT *psCreateEpgSchedule (
        SMS_OBJECT hParent,
        UN8 un8NumSegmentsTotal
            )
{
    EPG_FULL_SCHEDULE_ROOT *psSchedule = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    UN8 un8Index = 0;

    do
    {
        // Create schedule object
        psSchedule =
            (EPG_FULL_SCHEDULE_ROOT *) SMSO_hCreate(EPG_SCHEDULE_ROOT_OBJECT_NAME,
                                                    sizeof(EPG_FULL_SCHEDULE_ROOT),
                                                    hParent,
                                                    FALSE);
        if ((SMS_OBJECT) psSchedule == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create EPG Schedule object.");

            psSchedule = NULL;

            break;
        }

        psSchedule->hEpgChannels = OSAL_INVALID_OBJECT_HDL;
        psSchedule->hFullTopicsList = OSAL_INVALID_OBJECT_HDL;

        // Create channels list
        eReturnCode =
            OSAL.eLinkedListCreate(&(psSchedule->hEpgChannels),
                                   EPG_SCHEDULE_CHANNELS_LIST_NAME,
                                   (OSAL_LL_COMPARE_HANDLER) EPG_CHANNEL_n16CompareEpgChannelsBySid,
                                   OSAL_LL_OPTION_UNIQUE | OSAL_LL_OPTION_LINEAR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create EPG Channels list.");
            break;
        }

        // Create topics list
        eReturnCode =
            OSAL.eLinkedListCreate(&(psSchedule->hFullTopicsList),
                                   EPG_SCHEDULE_TOPICS_LIST_NAME,
                                   (OSAL_LL_COMPARE_HANDLER) EPG_TOPIC_n16CompareEpgTopicsByTid,
                                   OSAL_LL_OPTION_UNIQUE | OSAL_LL_OPTION_LINEAR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create full EPG Topics list.");
            break;
        }

        // Create text cache//TODO: Create on demand?
        psSchedule->psTextCache = EPG_TEXT_CACHE_psCreate((SMS_OBJECT) psSchedule,
                                                          (UN8)(2 * GsEpgIntf.un8MaxTotalSegments),
                                                          (UN8)(un8NumSegmentsTotal + 1));
        if (psSchedule->psTextCache == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create Text Cache object.");
            break;
        }

        // Create segments states array
        psSchedule->aeSegState =
            (EPG_SEG_STATE_ENUM *) SMSO_hCreate(EPG_SCHEDULE_STATES_ARRAY_NAME,
                                                sizeof(EPG_SEG_STATE_ENUM) * un8NumSegmentsTotal,
                                                (SMS_OBJECT) psSchedule, FALSE);
        if ((SMS_OBJECT) psSchedule->aeSegState == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create Segments States array.");
            break;
        }

        // Set initial values
        for (un8Index = 0; un8Index < un8NumSegmentsTotal; un8Index++)
        {
            psSchedule->aeSegState[un8Index] = EPG_SEG_STATE_WAIT;
        }

        psSchedule->eState = SCHEDULE_STATE_INVALID;

        psSchedule->un8NumSegmentsTotal = un8NumSegmentsTotal;

        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 1,
            PC_CYAN"New Program Schedule Root created: %p\n"PC_RESET,
            psSchedule);

        // Everything's OK
        return psSchedule;

    } while (FALSE);

    // Error case:
    vDestroyEpgSchedule(psSchedule);
    return NULL;
}

/*******************************************************************************
 *
 *  vCleanEpgSchedule
 *
 *******************************************************************************/
static void vCleanEpgSchedule (
        EPG_FULL_SCHEDULE_ROOT *psSchedule
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    UN8 un8Index = 0;

    if (psSchedule == NULL)
    {
        return;
    }

    // Remove all EPG channels (NOT Channels list)
    if (psSchedule->hEpgChannels != OSAL_INVALID_OBJECT_HDL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 11, "Remove EPG Channels...\n");
        eReturnCode = OSAL.eLinkedListRemoveAll(psSchedule->hEpgChannels,
            (OSAL_LL_RELEASE_HANDLER) EPG_CHANNEL_vDestroy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": ASSERT: Failed to Remove EPG Channels!");
        }
    }

    // Remove all topics (NOT Topics list)
    if (psSchedule->hFullTopicsList != OSAL_INVALID_OBJECT_HDL)
    {
        SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 11, "Clean topics pool...\n");
        eReturnCode = OSAL.eLinkedListRemoveAll(psSchedule->hFullTopicsList,
                                                (OSAL_LL_RELEASE_HANDLER) EPG_TOPIC_vDestroy);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": ASSERT: Failed to clean Topics pool!");
        }
    }

    // Clean Text Cache
    if (psSchedule->psTextCache != NULL)
    {
        EPG_TEXT_CACHE_vCleanWholeCache(psSchedule->psTextCache);
    }

    // Set initial values
    for (un8Index = 0;
         un8Index < psSchedule->un8NumSegmentsTotal;
         un8Index++)
    {
        psSchedule->aeSegState[un8Index] = EPG_SEG_STATE_WAIT;
    }

    psSchedule->un8NumSegmentsCollected = 0;

    psSchedule->eState = SCHEDULE_STATE_INVALID;

    return;
}

/*******************************************************************************
 *
 *  vDestroyEpgSchedule
 *
 *******************************************************************************/
static void vDestroyEpgSchedule (
        EPG_FULL_SCHEDULE_ROOT *psSchedule
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    if (psSchedule == NULL)
    {
        return;
    }

    vCleanEpgSchedule(psSchedule);

    // Delete Channels List
    if (psSchedule->hEpgChannels != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListDelete(psSchedule->hEpgChannels);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": ASSERT: Failed to delete EPG Channels list!");
        }
    }

    // Delete Topics List
    if (psSchedule->hFullTopicsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListDelete(psSchedule->hFullTopicsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": ASSERT: Failed to delete Topics pool!");
        }
    }

    // Delete Text Cache
    if (psSchedule->psTextCache != NULL)
    {
        EPG_TEXT_CACHE_vDestroy(psSchedule->psTextCache);
    }

    // Delete States array
    if (psSchedule->aeSegState != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psSchedule->aeSegState);
    }

    // Delete Schedule Object
    SMSO_vDestroy((SMS_OBJECT) psSchedule);

    return;
}

/*******************************************************************************
 *
 *  psCreateProcessingDescriptor
 *
 *******************************************************************************/
static EPG_PROCESSING_DESCRIPTOR_STRUCT *psCreateProcessingDescriptor (
        SMS_OBJECT hParent,
        UN8 un8NumSegmentsMax
            )
{
    EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor = NULL;

    do
    {
        // Create Descriptor object
        psDescriptor =
            (EPG_PROCESSING_DESCRIPTOR_STRUCT *)
                SMSO_hCreate(EPG_PROC_DESC_OBJECT_NAME,
                             sizeof(EPG_PROCESSING_DESCRIPTOR_STRUCT),
                             (SMS_OBJECT) hParent,
                             FALSE);
        if ((SMS_OBJECT) psDescriptor == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create Processing Descriptor object.");

            psDescriptor = NULL;

            break;
        }

        // Set initial values
        psDescriptor->un8CurrentSegment = UN8_MAX;
        psDescriptor->un8NumSegmentsMax = un8NumSegmentsMax;

        // Create arrays

        psDescriptor->asGridFileDesc =
            (EPG_GRID_FILE_HEADER_STRUCT *)
                SMSO_hCreate(EPG_GRID_DESC_OBJECT_NAME,
                             sizeof(EPG_GRID_FILE_HEADER_STRUCT) * un8NumSegmentsMax,
                             (SMS_OBJECT) psDescriptor, FALSE);
        if ((SMS_OBJECT) psDescriptor->asGridFileDesc == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create GRID descriptors array.");

            psDescriptor->asGridFileDesc = NULL;

            break;
        }

        psDescriptor->asTextFileDesc =
            (EPG_TEXT_FILE_HEADER_STRUCT *)
                SMSO_hCreate(EPG_TEXT_DESC_OBJECT_NAME,
                             sizeof(EPG_TEXT_FILE_HEADER_STRUCT) * un8NumSegmentsMax,
                             (SMS_OBJECT) psDescriptor, FALSE);
        if ((SMS_OBJECT) psDescriptor->asTextFileDesc == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create TEXT descriptors array.");

            psDescriptor->asTextFileDesc = NULL;

            break;
        }

        psDescriptor->abYesterdayTextSaved =
            (BOOLEAN *) SMSO_hCreate(EPG_YESTERDAY_TEXT_DESC_OBJECT_NAME,
                                     sizeof(BOOLEAN) * un8NumSegmentsMax,
                                     (SMS_OBJECT) psDescriptor, FALSE);
        if ((SMS_OBJECT) psDescriptor->abYesterdayTextSaved == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create YESTERDAY TEXT descriptors array.");

            psDescriptor->abYesterdayTextSaved = NULL;

            break;
        }

        // Everything's OK
        return psDescriptor;

    } while (FALSE);

    // Error case:
    vDestroyProcessingDescriptor(psDescriptor);
    return FALSE;
}

/*******************************************************************************
 *
 *  vCleanGridDescriptors
 *
 *******************************************************************************/
static void vCleanGridDescriptors (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    if (psDescriptor == NULL)
    {
        return;
    }

    psDescriptor->psCurrentChannel = NULL;
    psDescriptor->tCumulativeTime = 0;
    psDescriptor->un8CurrentSegment = UN8_MAX;

    // Clean Grid data files descriptors
    OSAL.bMemSet(psDescriptor->asGridFileDesc,
        0, sizeof(EPG_GRID_FILE_HEADER_STRUCT) * psDescriptor->un8NumSegmentsMax);

    // Clean Midnight Events files descriptors
    OSAL.bMemSet(&(psDescriptor->sMidnightFileDesc),
        0, sizeof(EPG_MIDNIGHT_FILE_HEADER_STRUCT));
    OSAL.bMemSet(psDescriptor->abYesterdayTextSaved,
        0, sizeof(BOOLEAN) * psDescriptor->un8NumSegmentsMax);

    return;
}

/*******************************************************************************
 *
 *  vCleanTextDescriptors
 *
 *******************************************************************************/
static void vCleanTextDescriptors (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    if (psDescriptor == NULL)
    {
        return;
    }

    OSAL.bMemSet(psDescriptor->asTextFileDesc,
        0, sizeof(EPG_TEXT_FILE_HEADER_STRUCT) * psDescriptor->un8NumSegmentsMax);

    return;
}

/*******************************************************************************
 *
 *  vDestroyProcessingDescriptor
 *
 *******************************************************************************/
static void vDestroyProcessingDescriptor (
         EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    if (psDescriptor == NULL)
    {
        return;
    }

    // Delete schedule
    if (psDescriptor->psProcessingSchedule != NULL)
    {
        vDestroyEpgSchedule(psDescriptor->psProcessingSchedule);
    }

    // Delete arrays
    if (psDescriptor->asGridFileDesc != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psDescriptor->asGridFileDesc);
    }

    if (psDescriptor->asTextFileDesc != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psDescriptor->asTextFileDesc);
    }

    if (psDescriptor->abYesterdayTextSaved != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psDescriptor->abYesterdayTextSaved);
    }

    // Delete Descriptor object
    SMSO_vDestroy((SMS_OBJECT) psDescriptor);

    return;
}

/*******************************************************************************
 *
 *  psGetEpgChannel
 *
 *******************************************************************************/
static EPG_CHANNEL_OBJECT_STRUCT *psGetEpgChannel (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        SERVICE_ID tServiceId,
        BOOLEAN bCreate
            )
{
    OSAL_LINKED_LIST_ENTRY hChannelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    EPG_CHANNEL_OBJECT_STRUCT *psChannel = NULL;

    do
    {
        // Check if a channel is already in the list
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psSchedule->hEpgChannels,
            &hChannelEntry,
            (OSAL_LL_COMPARE_HANDLER)EPG_CHANNEL_n16CompareEpgChannelAndSid,
            &tServiceId);
        if (eReturnCode == OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 12,
                "EPG Channel with SID=%d is already in the list.\n",
                tServiceId);
            // Get instance
            psChannel = (EPG_CHANNEL_OBJECT_STRUCT *)
                OSAL.pvLinkedListThis(hChannelEntry);
            if (psChannel == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to get Channel instance (SID=%d).",
                    tServiceId);
                break;
            }

            // Return existing channel
            return psChannel;
        }

        // Channel is not in the list
        if (bCreate == FALSE)
        {
            // Cahnnel should not be created. Return.
            break;
        }

        // Cahnnel should not created and added to the list.

        // Create new channel object
        psChannel = EPG_CHANNEL_psCreate(tServiceId, (SMS_OBJECT) psSchedule);
        if (psChannel == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create EPG Channel.");
            break;
        }

        // Add EPG channel to the list
        eReturnCode = OSAL.eLinkedListAdd(psSchedule->hEpgChannels,
                                          OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                          psChannel);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to add EPG Channel.");
            break;
        }

        // Everything's OK
        return psChannel;

    } while (FALSE);

    // Error case:
    EPG_CHANNEL_vDestroy(psChannel);
    return NULL;
}

/*******************************************************************************
 *
 *  bProcessParserEvent
 *
 *******************************************************************************/
static BOOLEAN bProcessParserEvent (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor =
        psEpgMgrObj->psEpgOwner->psProcessingDescriptor;
    EPG_FULL_SCHEDULE_ROOT *psSchedule =
        psDescriptor->psProcessingSchedule;
    SERVICE_ID tCurrentServiceId = SERVICE_INVALID_ID;
    EPG_CHANNEL_OBJECT_STRUCT *psChannel = NULL;
    BOOLEAN bResult = TRUE;

    do
    {
        // Handle segment change
        if (psParserEvent->un8SegmentNumber !=
            psDescriptor->un8CurrentSegment)
        {
            // Set new segment number
            psDescriptor->un8CurrentSegment = psParserEvent->un8SegmentNumber;

            // Set start time to the beginning of the current segment
            psDescriptor->tCumulativeTime =
                (UN32)((psSchedule->sVersion.un16Epoch +
                        psParserEvent->un8SegmentNumber) *
                        SECONDS_IN_DAY);
        }

        // Handle channel change
        if (psDescriptor->psCurrentChannel != NULL)
        {
            tCurrentServiceId =
                psDescriptor->psCurrentChannel->tServiceId;
        }
        if (psParserEvent->tServiceId != tCurrentServiceId)
        {
            // Get (create) required channel entry by SID
            psChannel = psGetEpgChannel(psSchedule,
                                        psParserEvent->tServiceId,
                                        TRUE);
            if (psChannel == NULL)
            {
                bResult = FALSE;
                break;
            }

            psChannel->hEpgService = (EPG_SERVICE_OBJECT) psEpgMgrObj;

            // Update current channel pointer
            psDescriptor->psCurrentChannel = psChannel;

            // Set start time to the beginning of the current segment
            psDescriptor->tCumulativeTime =
                (UN32)((psSchedule->sVersion.un16Epoch +
                        psParserEvent->un8SegmentNumber) *
                        SECONDS_IN_DAY);
        }

        // Process event
        bResult = bUnwrapParserEvent(psSchedule, psParserEvent, psDescriptor);
        if (bResult == FALSE)
        {
            break;
        }

        // Try to find and save midnight event
        // from currently loaded schedule
        // to currently processed schedule.
        if (psParserEvent->un8SegmentNumber == 0 &&
            psParserEvent->ePemType == EPG_PTYPE_DURATION_ONLY_EVENT)
        {
            // Duration event is just processed,
            // so cumulative time holds event's end time.
            bCreateMidnightEventCopy(psEpgMgrObj->psEpgOwner,
                                     psParserEvent->tServiceId,
                                     psDescriptor->tCumulativeTime);
            // No need to check function result
        }

        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bUnwrapParserEvent
 *
 *******************************************************************************/
static BOOLEAN bUnwrapParserEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    BOOLEAN bResult = FALSE;

    switch (psParserEvent->ePemType)
    {
        case EPG_PTYPE_FULL_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 12,
                "Unwrap FULL event...\n");
            bResult = bUnwrapFullEvent(psSchedule,
                                       psParserEvent,
                                       psDescriptor);
            break;

        case EPG_PTYPE_MODIFY_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 12,
                "Unwrap MODIFY event...\n");
            bResult = bUnwrapModifyEvent(psSchedule,
                                         psParserEvent,
                                         psDescriptor);
            break;

        case EPG_PTYPE_ADD_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 12,
                "Unwrap ADD event...\n");
            bResult = bUnwrapAddEvent(psSchedule,
                                      psParserEvent,
                                      psDescriptor);
            break;

        case EPG_PTYPE_DURATION_ONLY_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_MGR_DBG_PREFIX, 12,
                "Unwrap DURATION event...\n");
            bResult = bUnwrapDurationEvent(psSchedule,
                                           psParserEvent,
                                           psDescriptor);
            break;

        default:
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Invalid PEM Type = %u.",
                psParserEvent->ePemType);

            bResult = FALSE;

            break;
    };

    return bResult;
}

/*******************************************************************************
 *
 *  bUnwrapDurationEvent
 *
 *******************************************************************************/
static BOOLEAN bUnwrapDurationEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    UN32 un32Duration = 0;
    BOOLEAN bResult = FALSE;

    // Just get duration
    un32Duration = un32CalculateEventDuration(psParserEvent,
                                              psDescriptor->tCumulativeTime);
    if (un32Duration != 0)
    {
        psDescriptor->tCumulativeTime += un32Duration;
        bResult = TRUE;
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bUnwrapModifyEvent
 *
 *******************************************************************************/
static BOOLEAN bUnwrapModifyEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    OSAL_LINKED_LIST_ENTRY hProgramEventEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    PROG_EVENT_STRUCT *psProgramEvent = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Find existing Program Event by start time
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDescriptor->psCurrentChannel->hEpgChannelEvents,
            &hProgramEventEntry,
            (OSAL_LL_COMPARE_HANDLER)EPG_PROGRAM_n16CompareEpgEventAndStartTime,
            &(psDescriptor->tCumulativeTime));
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        psProgramEvent = (PROG_EVENT_STRUCT *) OSAL.pvLinkedListThis(hProgramEventEntry);
        if (psProgramEvent == NULL)
        {
            break;
        }

        // Copy data from "MODIFY" Parser Event to found Program Event
        psProgramEvent->tProgramId = psParserEvent->tProgramId;
        if (psParserEvent->tProgramFlags != EPG_PROGRAM_INVALID_FLAGS)
        {
            psProgramEvent->tPFlags = psParserEvent->tProgramFlags;
        }
        if (psParserEvent->eRecordingOption != EPG_PROGRAM_RECORD_UNDEFINED)
        {
            psProgramEvent->eRecordingOption = psParserEvent->eRecordingOption;
        }
        if (psParserEvent->un32ProgramDescIdx != UN32_MAX)
        {
            psProgramEvent->un32ProgramDescIdx
                = psParserEvent->un32ProgramDescIdx;

            psProgramEvent->tSchSegProgramDescription
                = (PROGRAM_DAY_ID) psParserEvent->un8SegmentNumber;
        }
        if (psParserEvent->un16OriginalDate != UN16_MAX)
        {
            psProgramEvent->un16OrigDate = psParserEvent->un16OriginalDate;
        }
        // Copy topics
        if (psParserEvent->un8NumTopics > 0)
        {
            bResult = bCopyTopics(psSchedule,
                                  psProgramEvent,
                                  psParserEvent->atTopicId,
                                  psParserEvent->un8NumTopics);
            if (bResult == FALSE)
            {
                break;
            }
        }

        // Update Program time and cumulative time
        if (psParserEvent->eDurationCode != EPG_DURATION_CODE_UNDEFINED)
        {
            UN32 un32Duration = 0;

            un32Duration = un32CalculateEventDuration(psParserEvent,
                                                      psDescriptor->tCumulativeTime);
            if (un32Duration == 0)
            {
                break;
            }

            psDescriptor->tCumulativeTime += un32Duration;
            psProgramEvent->tEndTime = psDescriptor->tCumulativeTime;
        }
        else
        {
            psDescriptor->tCumulativeTime +=
                (psProgramEvent->tEndTime - psProgramEvent->tStartTime);
        }

        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bUnwrapAddEvent
 *
 *******************************************************************************/
static BOOLEAN bUnwrapAddEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    OSAL_LINKED_LIST_ENTRY hProgramEventEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    PROG_EVENT_STRUCT *psProgramEvent = NULL;
    BOOLEAN bResult = FALSE;
    UN32 un32Duration = 0;
    BOOLEAN bProgramSaved = FALSE;

    do
    {
        // Find existing Program Event by Series ID
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psDescriptor->psCurrentChannel->hEpgChannelEvents,
            &hProgramEventEntry,
            (OSAL_LL_COMPARE_HANDLER)EPG_PROGRAM_n16CompareEpgEventAndSeriesId,
            &(psParserEvent->tSeriesId));
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        psProgramEvent = (PROG_EVENT_STRUCT *) OSAL.pvLinkedListThis(hProgramEventEntry);
        if (psProgramEvent == NULL)
        {
            break;
        }

        // Create new Program Event (copy of found event but with new time)
        psProgramEvent = EPG_PROGRAM_psDuplicate(psProgramEvent,
                                                 (SMS_OBJECT) psDescriptor->psCurrentChannel,
                                                 psDescriptor->tCumulativeTime);
        if (psProgramEvent == NULL)
        {
            break;
        }

        // Copy data from "ADD" Parser Event to new Program Event
        psProgramEvent->tProgramId = psParserEvent->tProgramId;
        if (psParserEvent->tProgramFlags != EPG_PROGRAM_INVALID_FLAGS)
        {
            psProgramEvent->tPFlags = psParserEvent->tProgramFlags;
        }
        if (psParserEvent->eRecordingOption != EPG_PROGRAM_RECORD_UNDEFINED)
        {
            psProgramEvent->eRecordingOption = psParserEvent->eRecordingOption;
        }
        if (psParserEvent->un32ProgramDescIdx != UN32_MAX)
        {
            psProgramEvent->un32ProgramDescIdx =
                psParserEvent->un32ProgramDescIdx;

            psProgramEvent->tSchSegProgramDescription =
                (PROGRAM_DAY_ID) psParserEvent->un8SegmentNumber;
        }
        if (psParserEvent->un16OriginalDate != UN16_MAX)
        {
            psProgramEvent->un16OrigDate = psParserEvent->un16OriginalDate;
        }
        // Copy topics
        if (psParserEvent->un8NumTopics > 0)
        {
            bResult = bCopyTopics(psSchedule,
                                  psProgramEvent,
                                  psParserEvent->atTopicId,
                                  psParserEvent->un8NumTopics);
            if (bResult == FALSE)
            {
                break;
            }
        }

        if (psParserEvent->eDurationCode != EPG_DURATION_CODE_UNDEFINED)
        {
            un32Duration = un32CalculateEventDuration(psParserEvent,
                                                      psDescriptor->tCumulativeTime);
            if (un32Duration == 0)
            {
                break;
            }

            psProgramEvent->tEndTime = psProgramEvent->tStartTime + un32Duration;
        }
        else
        {
            un32Duration = (UN32) (psProgramEvent->tEndTime - psProgramEvent->tStartTime);
        }

        // Update cumulative time
        psDescriptor->tCumulativeTime += un32Duration;

        // Add original program instance
        bResult = EPG_CHANNEL_bAddProgramEventToEpgChannel(psDescriptor->psCurrentChannel,
                                                           psProgramEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Now original program instance is saved
        // so it should not be deleted on error.
        bProgramSaved = TRUE;

        // Add additional program instances
        bResult = bAddExtraProgramInstances(psSchedule,
                                            psDescriptor->psCurrentChannel,
                                            psProgramEvent,
                                            psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error case:
    if (bProgramSaved == FALSE)
    {
        EPG_PROGRAM_vDestroy(psProgramEvent);
    }
    return FALSE;
}

/*******************************************************************************
 *
 *  bUnwrapFullEvent
 *
 *******************************************************************************/
static BOOLEAN bUnwrapFullEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        EPG_PROCESSING_DESCRIPTOR_STRUCT *psDescriptor
            )
{
    PROG_EVENT_STRUCT *psProgramEvent = NULL;
    BOOLEAN bResult = TRUE;
    BOOLEAN bProgramSaved = FALSE;

    do
    {
        // Create Program Event from Parser Event
        psProgramEvent = psCreateNewProgramEvent(psSchedule,
                                                 psParserEvent,
                                                 psDescriptor->tCumulativeTime,
                                                 (SMS_OBJECT) psDescriptor->psCurrentChannel);
        if (psProgramEvent == NULL)
        {
            break;
        }

        // Add original program instance
        bResult = EPG_CHANNEL_bAddProgramEventToEpgChannel(psDescriptor->psCurrentChannel,
                                                           psProgramEvent);
        if (bResult == FALSE)
        {
            break;
        }

        bProgramSaved = TRUE;

        // Add additional program instances
        bResult = bAddExtraProgramInstances(psSchedule,
                                            psDescriptor->psCurrentChannel,
                                            psProgramEvent,
                                            psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Update cumulative time
        psDescriptor->tCumulativeTime +=
            un32CalculateEventDuration(psParserEvent,
                                       psDescriptor->tCumulativeTime);

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error case:
    if (bProgramSaved == FALSE)
    {
        EPG_PROGRAM_vDestroy(psProgramEvent);
    }
    return FALSE;
}

/*******************************************************************************
 *
 *  psCreateNewProgramEvent
 *
 *******************************************************************************/
static PROG_EVENT_STRUCT *psCreateNewProgramEvent (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        TIME_T tStartTime,
        SMS_OBJECT hParent
            )
{
    PROG_EVENT_STRUCT *psProgramEvent = NULL;
    UN32 un32Duration = 0;

    do
    {
        // Create empty (invalid) Program Event
        psProgramEvent = EPG_PROGRAM_psCreate(hParent);
        if (psProgramEvent == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EPG_MGR_OBJECT_NAME": Failed to create Program Event.");
            break;
        }

        // Copy common fields
        psProgramEvent->tSchSeg = (PROGRAM_DAY_ID) psParserEvent->un8SegmentNumber;
        psProgramEvent->tSchSegProgramDescription = (PROGRAM_DAY_ID) psParserEvent->un8SegmentNumber;
        psProgramEvent->tServiceId = psParserEvent->tServiceId;
        psProgramEvent->tSeriesId = psParserEvent->tSeriesId;
        psProgramEvent->tProgramId = psParserEvent->tProgramId;
        psProgramEvent->tPFlags = psParserEvent->tProgramFlags;
        psProgramEvent->eRecordingOption = psParserEvent->eRecordingOption;
        psProgramEvent->un32PNameShortIdx = psParserEvent->un32ProgNameShortIdx;
        psProgramEvent->un32PNameLongIdx = psParserEvent->un32ProgNameLongIdx;
        psProgramEvent->un32ProgramDescIdx = psParserEvent->un32ProgramDescIdx;
        psProgramEvent->un32SeriesDescIdx = psParserEvent->un32SeriesDescIdx;
        psProgramEvent->un16OrigDate = psParserEvent->un16OriginalDate;

        // Calculate times
        un32Duration = un32CalculateEventDuration(psParserEvent,
                                                  tStartTime);
        if (un32Duration == 0)
        {
            break;
        }
        psProgramEvent->tStartTime = tStartTime;
        psProgramEvent->tEndTime = tStartTime + un32Duration;

        // Copy topics (optional)
        if (psParserEvent->un8NumTopics > 0)
        {
            BOOLEAN bResult;

            bResult = bCopyTopics(
                    psSchedule,
                    psProgramEvent,
                    psParserEvent->atTopicId,
                    psParserEvent->un8NumTopics);
            if (bResult == FALSE)
            {
                break;
            }
        }

        // Everything's OK
        return psProgramEvent;

    } while (FALSE);

    // Error case:
    EPG_PROGRAM_vDestroy(psProgramEvent);
    return NULL;
}

/*******************************************************************************
 *
 *  bSaveTopicInfo
 *
 *******************************************************************************/
static BOOLEAN bSaveTopicInfo (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        TOPIC_ID tTopicId,
        char *pcTopicName
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    EPG_TOPIC_STRUCT *psEpgTopic = NULL;
    BOOLEAN bIsTopicInPool = FALSE;
    BOOLEAN bResult = FALSE;

    do
    {
        // Check if Topic is already saved in the topics pool
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psSchedule->hFullTopicsList,
            &hEntry,
            (OSAL_LL_COMPARE_HANDLER)EPG_TOPIC_n16CompareEpgTopicAndTid,
            &tTopicId);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // Topic is found
            psEpgTopic = (EPG_TOPIC_STRUCT*) OSAL.pvLinkedListThis(hEntry);
            if (psEpgTopic == NULL)
            {
                break;
            }

            bIsTopicInPool = TRUE;
        }
        else
        {
            // Not found. Create new instance.
            psEpgTopic = EPG_TOPIC_psCreate(SMS_INVALID_OBJECT, tTopicId, pcTopicName);
            if (psEpgTopic == NULL)
            {
                break;
            }

            // Add newly created instance to the topics pool
            eReturnCode = OSAL.eLinkedListAdd(psSchedule->hFullTopicsList,
                                              &hEntry,
                                              psEpgTopic);
            if (eReturnCode != OSAL_SUCCESS &&
                eReturnCode != OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EPG_MGR_OBJECT_NAME": Failed to add Topic to the pool.");
                break;
            }

            bIsTopicInPool = TRUE;
        }

        // Add Topic to a Program Event
        bResult = EPG_PROGRAM_bAddTopic(psProgramEvent, psEpgTopic);
        if (bResult == FALSE)
        {
            break;
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error case:
    if (bIsTopicInPool == FALSE)
    {
        EPG_TOPIC_vDestroy(psEpgTopic);
    }
    return FALSE;
}

/*******************************************************************************
 *
 *  bAddExtraProgramInstances
 *
 *******************************************************************************/
static BOOLEAN bAddExtraProgramInstances (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        EPG_CHANNEL_OBJECT_STRUCT *psChannel,
        PROG_EVENT_STRUCT *psProgramEvent,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = TRUE;
    UN8 un8ExtraStartTimeCounter = 0;
    // Set original program start time first
    TIME_T tNewStartTime = psProgramEvent->tStartTime;
    UN8 un8ExtraDayNumber = 0;
    UN16 un16CurrentSegmentEpoch =
        psSchedule->sVersion.un16Epoch + psParserEvent->un8SegmentNumber;

    // Add additional program instances
    // depending on additional start times and daily repeat presence.

    do // while (un8ExtraStartTimeCounter <= psParserEvent->un8NumAdditionalStartTimes)
    {
        // Add additional program instances to the current channel
        // in subsequent segments if daily repeat is defined.
        if (psParserEvent->un8DailyRepeat != 0)
        {
            un8ExtraDayNumber = 0;
            while (un8ExtraDayNumber <
                  (psSchedule->un8NumSegmentsTotal - psParserEvent->un8SegmentNumber))
            {
                // Move time by one day
                tNewStartTime += SECONDS_IN_DAY;

                // Check if program instance should be added to a certain day
                if ((psParserEvent->un8DailyRepeat & (1 << un8ExtraDayNumber)) != 0)
                {
                    bResult = EPG_CHANNEL_bAddProgramCopy(psChannel,
                                                          psProgramEvent,
                                                          tNewStartTime);
                    if (bResult == FALSE)
                    {
                        return FALSE;
                    }
                }

                un8ExtraDayNumber++;
            };
        }

        if (un8ExtraStartTimeCounter == psParserEvent->un8NumAdditionalStartTimes)
        {
            // No more additional times are defined
            break;
        }

        // Calculate start time for additional program instance in the current segment.
        // New time = current segment start time + program's additional start time.
        tNewStartTime = (un16CurrentSegmentEpoch * SECONDS_IN_DAY) +
                               psParserEvent->aun32AdditionalTimes[un8ExtraStartTimeCounter];

        // Add program instance to the current channel in the current segment
        bResult = EPG_CHANNEL_bAddProgramCopy(psChannel,
                                              psProgramEvent,
                                              tNewStartTime);
        if (bResult == FALSE)
        {
           return FALSE;
        }

        un8ExtraStartTimeCounter++;

    } while (un8ExtraStartTimeCounter <= psParserEvent->un8NumAdditionalStartTimes);

    return TRUE;
}

/*****************************************************************************
 *
 *  bCopyTopics
 *
 *****************************************************************************/
static BOOLEAN bCopyTopics (
        EPG_FULL_SCHEDULE_ROOT *psSchedule,
        PROG_EVENT_STRUCT *psProgramEvent,
        TOPIC_ID *atTopicId,
        UN8 un8NumTopics
            )
{
    UN8 un8Idx = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Check input
        if (atTopicId == NULL || un8NumTopics == 0)
        {
            break;
        }

        // Delete existing topics
        EPG_TOPICS_LIST_vDestroy((EPG_TOPICS_LIST_STRUCT *) psProgramEvent->hEpgTopics);
        psProgramEvent->hEpgTopics = EPG_INVALID_TOPICS_LIST_OBJECT;

        // Add new topics
        for (un8Idx = 0; un8Idx < un8NumTopics; un8Idx++)
        {
            bResult = bSaveTopicInfo(psSchedule,
                                     psProgramEvent,
                                     atTopicId[un8Idx],
                                     NULL);
            if (bResult == FALSE)
            {
                break;
            }
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  vHandleReceivedScheduleCompletion
 *
 *****************************************************************************/
static void vHandleReceivedScheduleCompletion (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = psEpgMgrObj->psEpgOwner;

    DATASERVICE_IMPL_vLog(EPG_MGR_OBJECT_NAME": =*=*=*=*=*= SCHEDULE IS COMPLETED =*=*=*=*=*=\n");

    // Move all ready text files to cache files.
    vMoveReadyTextFilesToCache(psEpgOwner);

    // Load midnight programs from file to processing schedule
    bLoadMidnightPrograms(psEpgOwner);

    psEpgOwner->psProcessingDescriptor->psProcessingSchedule->eState
         = SCHEDULE_STATE_READY_FULL;

    // Replace current schedule with processing one
    if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule
        != psEpgOwner->psCurrentSchedule)
    {
        vDestroyEpgSchedule(psEpgOwner->psCurrentSchedule);
        psEpgOwner->psCurrentSchedule =
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    }
    psEpgOwner->psProcessingDescriptor->psProcessingSchedule = NULL;

    // Delete entities required during processing.
    // They will be created when appropriate event from parser is received.
    vUninitScheduleProcessing(psEpgOwner);

    // Set option to ignore all AUs of currently assembled schedule
    GsEpgIntf.eSetOptions(psEpgMgrObj->hEpgParserObject,
                          EPG_PARSER_OPTION_CHECK_SEG_VERSION |
                          EPG_PARSER_OPTION_ACCEPT_NEW_VERSION_ONLY);

    // Update Decoders
    bSendUpdate(psEpgMgrObj,
                DECODER_INVALID_OBJECT, // Notify all decoders
                EPG_SUB_UPDATE_TYPE_PEM_CHANGE);

    return;
}

/*****************************************************************************
 *
 *  vHandleLoadedScheduleCompletion
 *
 *****************************************************************************/
static void vHandleLoadedScheduleCompletion (
        EPG_MGR_OBJECT_STRUCT *psEpgMgrObj,
        BOOLEAN bUnbrokenSchedule
            )
{
    EPG_OWNER_STRUCT *psEpgOwner = psEpgMgrObj->psEpgOwner;

    // Load midnight programs from file to processing schedule
    bLoadMidnightPrograms(psEpgOwner);

    // Replace current schedule with processing one
    if (psEpgOwner->psProcessingDescriptor->psProcessingSchedule
        != psEpgOwner->psCurrentSchedule)
    {
        vDestroyEpgSchedule(psEpgOwner->psCurrentSchedule);
        psEpgOwner->psCurrentSchedule =
            psEpgOwner->psProcessingDescriptor->psProcessingSchedule;
    }
    psEpgOwner->psProcessingDescriptor->psProcessingSchedule = NULL;

    // Check if a whole schedule is completed
    if (psEpgOwner->psCurrentSchedule->un8NumSegmentsCollected
        == psEpgOwner->psCurrentSchedule->un8NumSegmentsTotal
        && bUnbrokenSchedule == TRUE)
    {
        psEpgOwner->psCurrentSchedule->eState = SCHEDULE_STATE_READY_FULL;

        // Set option to ignore all AUs of currently assembled schedule
        GsEpgIntf.eSetOptions(psEpgMgrObj->hEpgParserObject,
                              EPG_PARSER_OPTION_CHECK_SEG_VERSION |
                              EPG_PARSER_OPTION_ACCEPT_NEW_VERSION_ONLY);
    }
    else
    {
        psEpgOwner->psCurrentSchedule->eState = SCHEDULE_STATE_READY_PARTIAL;
    }

    return;
}

#ifdef SUPPORT_CUNIT
#include <epg_mgr_obj.cunit>
#endif
