/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This file defines the interface used for service-specific EPG parsers.
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef EPG_INTERFACE_H_
#define EPG_INTERFACE_H_

  /**************/
 /** INCLUDES **/
/**************/
#include "sms_api.h"
#include "sms_obj.h"

#include "radio.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

#define EPG_PARSER_OPTION_NONE                      (0)
// If following option is set, EPG parser consumes minimum required memory while parsing the payload.
// Each segments is parsed consequently beginning from access unit 0 of segment 0.
// Only one AU is allocated at a time. So schedule download time can be increased
// but memory consumption is reduced.
// IMPORTANT NOTE: If EPG_PARSER_OPTION_CHECK_SEG_VERSION is set, this option will be ignored.
#define EPG_PARSER_OPTION_MEMORY_SAVING             (1)
// If following option is set, then Schedule Segment Versioning is supported
// as described in the EPG Protocol Specification v.1.0.
// IMPORTANT NOTE: If this option is set, EPG_PARSER_OPTION_MEMORY_SAVING option will be ignored.
#define EPG_PARSER_OPTION_CHECK_SEG_VERSION         (2)
// If following option is set, all received data is ignored until version is changed
// comparing to the initial version.
#define EPG_PARSER_OPTION_ACCEPT_NEW_VERSION_ONLY   (4)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Handle to the EPG parser's client object.
// Used to bring parsed data to a parser's user.
typedef void * EPG_CLIENT_INTERFACE_OBJECT;
#define EPG_CLIENT_INTERFACE_INVALID_OBJECT ((EPG_CLIENT_INTERFACE_OBJECT) NULL)

// Handle to the EPG parser object
typedef void * EPG_PARSER_INTERFACE_OBJECT;
#define EPG_PARSER_INTERFACE_INVALID_OBJECT ((EPG_PARSER_INTERFACE_OBJECT) NULL)

// Processing cancellation callback.
// The callback should be called periodically
// inside the EPG parser inplementation.
// to check if processing is canncelled by processing initiator.
typedef BOOLEAN (*CANCELLATION_CHECK_CALLBACK)(void *pvCallbackArg);

// EPG general data types
typedef enum epg_data_type_enum
{
    EPG_DATA_TYPE_GRID,
    EPG_DATA_TYPE_COMPRESSED_TEXT,

} EPG_DATA_TYPE_ENUM;

// EPG interface return codes
typedef enum epg_return_codes_enum
{
    EPG_RET_OK,
    EPG_RET_FAIL,
    EPG_RET_INVALID_DATA,
    EPG_RET_NOT_SUPPORTED,

    EPG_RET_IGNORED,
    EPG_RET_CANCELLED,

} EPG_RETURN_CODES_ENUM;

// Program Event Message Types
typedef enum epg_pem_type_enum
{
    EPG_PTYPE_FULL_EVENT,
    EPG_PTYPE_MODIFY_EVENT,
    EPG_PTYPE_ADD_EVENT,
    EPG_PTYPE_DURATION_ONLY_EVENT,

} EPG_PEM_TYPE_ENUM;

// Event Duration Code
typedef enum epg_pem_duration_code_enum
{
    EPG_DURATION_CODE_30MIN,
    EPG_DURATION_CODE_1HOUR,
    EPG_DURATION_CODE_2HOURS,
    EPG_DURATION_CODE_3HOURS,
    EPG_DURATION_CODE_4HOURS,
    EPG_DURATION_CODE_5HOURS,
    EPG_DURATION_CODE_END_OF_SEGMENT,
    EPG_DURATION_CODE_ESCAPE_TO_ODUR,
    EPG_DURATION_CODE_UNDEFINED,

} EPG_PEM_DURATION_CODE_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

// Access Unit data description
typedef struct epg_au_info_struct
{
    // AU info
    EPG_DATA_TYPE_ENUM eAuDataType;
    UN8 un8SegmentNumber;
    UN32 un32AuNumber;

    // AU payload data
    void *pvData;
    size_t tDataSize;

} EPG_AU_INFO_STRUCT;

// Schedule version
typedef struct epg_schedule_version_struct
{
    UN8 un8Version;
    UN16 un16Epoch;

} EPG_SCHEDULE_VERSION_STRUCT;

// Full PEM data provided by the parser
typedef struct epg_parser_event_struct
{
    UN8 un8SegmentNumber;
    SERVICE_ID tServiceId;

    EPG_PEM_TYPE_ENUM ePemType;

    SERIES_ID tSeriesId;
    PROGRAM_ID tProgramId;
    PROGRAM_FLAGS tProgramFlags;
    EPG_PROGRAM_RECORDING_OPTION_ENUM eRecordingOption;
    EPG_PEM_DURATION_CODE_ENUM eDurationCode;
    UN32 un32OptionalDuration;      // In seconds
    UN32 un32ProgNameShortIdx;
    UN32 un32ProgNameLongIdx;
    UN32 un32SeriesDescIdx;
    UN32 un32ProgramDescIdx;
    UN16 un16OriginalDate;
    UN8 un8NumAdditionalStartTimes; // Size of aun16AdditionalTimes array
    UN32 *aun32AdditionalTimes;     // Dynamic array of additional times in seconds
    UN8 un8DailyRepeat;
    UN8 un8NumTopics;               // Size of atTopicId array
    TOPIC_ID *atTopicId;            // Dynamic array of topic IDs

    //NOTE: Extra data is not included. Can be implemented in future if required.

} EPG_PARSER_EVENT_STRUCT;

// Callback interface used to bring parsed data to a parser's user.
// Should be implemented by EPG parser's client.
typedef struct epg_client_interface_struct
{
    EPG_CLIENT_INTERFACE_OBJECT hClientObj;

    BOOLEAN (*bProcessParserEvent) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            );

    BOOLEAN (*bProcessCompressedTextPayload) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            );

    void (*vSaveAuData) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_AU_INFO_STRUCT *psAuData
            );

    void (*vSegmentGridCompleted) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            );

    void (*vSegmentTextCompleted) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber,
        UN16 un16StringsCount,
        UN32 un32DecompressedFileSize
            );

    void (*vWholeSegmentCompleted) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        UN8 un8SegmentNumber
            );

    void (*vGetVersion) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            );

    void (*vVersionChanged) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj,
        EPG_SCHEDULE_VERSION_STRUCT *psNewVersion
            );

    void (*vRestartGridProcessing) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            );

    void (*vRestartTextProcessing) (
        EPG_CLIENT_INTERFACE_OBJECT hClientObj
            );

} EPG_CLIENT_INTERFACE_STRUCT;

// EPG Parser Interface
typedef struct epg_parser_interface_struct
{
    const PVN tPVN;
    const DSI tDSI;
    const size_t tOTABufferByteSize;
    const UN8 un8MaxTotalSegments;

    EPG_PARSER_INTERFACE_OBJECT (*hInit) (
        EPG_CLIENT_INTERFACE_STRUCT *psClientIntf,
        UN8 un8NumSegments,
        UN32 un32Options
            );

    void (*vUninit) (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj
            );

    EPG_RETURN_CODES_ENUM (*eProcessMessage) (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        OSAL_BUFFER_HDL hPayload,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            );

    EPG_RETURN_CODES_ENUM (*eLoadAuData) (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        EPG_AU_INFO_STRUCT *psAuData,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            );

    EPG_RETURN_CODES_ENUM (*eGetOptions) (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 *pun32Options
            );

    EPG_RETURN_CODES_ENUM (*eSetOptions) (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 un32Options
            );

} EPG_PARSER_INTERFACE_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

// Interface into the EPG plugin
extern const EPG_PARSER_INTERFACE_STRUCT GsEpgIntf;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#endif /* EPG_INTERFACE_H_ */
