/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DSRL_TARGET implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "location_obj.h"
#include "stock_symbol_obj.h"

#include "dsrl_target_obj.h"
#include "_dsrl_target_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eType
*
* This object interface method is used by the caller to retrieve the type
* this Data Target is.
*
* Inputs:
*
*   hTarget - The DSRL_TARGET_OBJECT to retreive the type for
*
* Outputs:
*
*   A valid DSRL_TARGET_TYPE_ENUM on success
*   A DSRL_TARGET_TYPE_UNKNOWN on failure
*
*****************************************************************************/
static DSRL_TARGET_TYPE_ENUM eType (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    DSRL_TARGET_TYPE_ENUM eType = DSRL_TARGET_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return DSRL_TARGET_TYPE_UNKNOWN;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        eType = psObj->eType;
    }

    return eType;
}

/*****************************************************************************
*
*   hLocation
*
* This object interface method returns a handle to a valid LOCATION_OBJECT if
* the type of this object is a DSRL_TARGET_TYPE_LOCATION
*
* Inputs:
*
*   hTarget - The DSRL_TARGET_OBJECT to get a location handle from
*
* Outputs:
*
*   A valid LOCATION_OBJECT handle on success
*   A LOCATION_INVALID_OBJECT on failure
*
*****************************************************************************/
static LOCATION_OBJECT hLocation (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    LOCATION_OBJECT hLocation = LOCATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return LOCATION_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_TARGET_TYPE_LOCATION)
        {
            hLocation = (LOCATION_OBJECT)hTarget;
        }
    }

    return hLocation;
}

/*****************************************************************************
*
*   hStockSymbol
*
* This object interface method returns a handle to a valid STOCK_SYMBOL_OBJECT if
* the type of this object is a DSRL_TARGET_TYPE_STOCK_SYMBOL
*
* Inputs:
*
*   hTarget - The DSRL_TARGET_OBJECT to get a location handle from
*
* Outputs:
*
*   A valid STOCK_SYMBOL_OBJECT handle on success
*   A STOCK_SYMBOL_INVALID_OBJECT on failure
*
*****************************************************************************/
static STOCK_SYMBOL_OBJECT hStockSymbol (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    STOCK_SYMBOL_OBJECT hStockSymbolTarget = STOCK_SYMBOL_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return STOCK_SYMBOL_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_TARGET_TYPE_STOCK_SYMBOL)
        {
            hStockSymbolTarget = (STOCK_SYMBOL_OBJECT)hTarget;
        }
    }

    return hStockSymbolTarget;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   DSRL_TARGET_hCreate
*
*****************************************************************************/
DSRL_TARGET_OBJECT DSRL_TARGET_hCreate (
    const char *pacName,
    DSRL_TARGET_TYPE_ENUM eType,
    size_t tSize,
    SMS_OBJECT hParent,
    BOOLEAN bLock
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Verify inputs
    if(pacName == NULL)
    {
        // Error!
        return DSRL_TARGET_INVALID_OBJECT;
    }

    // Construct an appropriate name for memory to be allocated
    snprintf(&acName[0], sizeof(acName),
        DSRL_TARGET_PREFIX":%s", pacName);

    // Allocate an Data Target object plus the requested memory
    psObj = (DSRL_TARGET_OBJECT_STRUCT *)SMSO_hCreate(&acName[0],
        sizeof(DSRL_TARGET_OBJECT_STRUCT) + tSize,
        hParent, bLock);
    if(psObj == NULL)
    {
        return DSRL_TARGET_INVALID_OBJECT;
    }

    // Set the type
    psObj->eType = eType;

    // return the pointer to the user space
    return (DSRL_TARGET_OBJECT)(psObj + 1);
}

/*****************************************************************************
*
*   DSRL_TARGET_hDuplicate
*
*****************************************************************************/
DSRL_TARGET_OBJECT DSRL_TARGET_hDuplicate (
    SMS_OBJECT hOwner,
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    BOOLEAN bOwner = FALSE;
    DSRL_TARGET_OBJECT hDuplicateTarget = DSRL_TARGET_INVALID_OBJECT;

    // Verify input
    bOwner =  SMSO_bOwner(hOwner);
    if ((hOwner != SMS_INVALID_OBJECT) && (bOwner == FALSE ))
    {
        // Error!
        return DSRL_TARGET_INVALID_OBJECT;
    }

    switch (psObj->eType)
    {
        case DSRL_TARGET_TYPE_LOCATION:
        {
            hDuplicateTarget = (DSRL_TARGET_OBJECT)
                LOCATION_hDuplicate(hOwner, (LOCATION_OBJECT)hTarget);
        }
        break;
        case DSRL_TARGET_TYPE_STOCK_SYMBOL:
        {
            hDuplicateTarget = (DSRL_TARGET_OBJECT)
                STOCK_SYMBOL_hDuplicate(hOwner, (STOCK_SYMBOL_OBJECT)hTarget);
        }
        break;
        case DSRL_TARGET_TYPE_UNKNOWN:
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DSRL_TARGET_PREFIX": Attempting to duplicate an unknown DSRL_TARGET object");
        }
        break;
    }

    return hDuplicateTarget;
}

/*****************************************************************************
*
*   DSRL_TARGET_bCompare
*
*****************************************************************************/
BOOLEAN DSRL_TARGET_bCompare(
    DSRL_TARGET_OBJECT hTarget1,
    DSRL_TARGET_OBJECT hTarget2
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        DSRL_TARGET_OBJECT_STRUCT *psObj1 =
            (DSRL_TARGET_OBJECT_STRUCT *)hTarget1 - 1;
        DSRL_TARGET_OBJECT_STRUCT *psObj2 =
            (DSRL_TARGET_OBJECT_STRUCT *)hTarget2 - 1;

        if ((hTarget1 == NULL) || (hTarget2 == NULL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DSRL_TARGET_PREFIX": Invalid DSRL_TARGET object");
            break;
        }

        if ((psObj1->eType != psObj2->eType))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DSRL_TARGET_PREFIX": Different types of DSRL_TARGET objects to compare");
            break;
        }

        switch (psObj1->eType)
        {
            case DSRL_TARGET_TYPE_LOCATION:
            {
                bResult = LOCATION_bCompare(hTarget1, hTarget2);
            }
            break;
            case DSRL_TARGET_TYPE_STOCK_SYMBOL:
            {
                N16 n16Result = 0;
                n16Result = STOCK_SYMBOL_n16Compare(hTarget1, hTarget2);
                bResult = (n16Result == 0) ? TRUE : FALSE;
            }
            break;
            case DSRL_TARGET_TYPE_UNKNOWN:
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DSRL_TARGET_PREFIX": Attempting to compare an unknown"
                    " DSRL_TARGET object"
                    );
            }
            break;
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   DSRL_TARGET_hGetSMSObject
*
*****************************************************************************/
SMS_OBJECT DSRL_TARGET_hGetSMSObject (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    SMS_OBJECT hObject = SMS_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return SMS_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        hObject = (SMS_OBJECT)psObj;
    }

    return hObject;
}

/*****************************************************************************
*
*   DSRL_TARGET_bOwner
*
*****************************************************************************/
BOOLEAN DSRL_TARGET_bOwner (
    DSRL_TARGET_OBJECT hTarget
        )
{
    BOOLEAN bOwner = FALSE;

    if (hTarget != DSRL_TARGET_INVALID_OBJECT)
    {
        DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    }

    return bOwner;
}

/*****************************************************************************
*
*   DSRL_TARGET_vDestroy
*
*****************************************************************************/
void DSRL_TARGET_vDestroy (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        psObj->eType = DSRL_TARGET_TYPE_UNKNOWN;

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_TARGET_vDestroyByType
*
*****************************************************************************/
void DSRL_TARGET_vDestroyByType (
    DSRL_TARGET_OBJECT hTarget
        )
{
    DSRL_TARGET_OBJECT_STRUCT *psObj =
        (DSRL_TARGET_OBJECT_STRUCT *)hTarget - 1;
    BOOLEAN bOwner;

    if (hTarget == DSRL_TARGET_INVALID_OBJECT)
    {
        return;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        switch (psObj->eType)
        {
            case DSRL_TARGET_TYPE_LOCATION:
            {
                LOCATION.vDestroy((LOCATION_OBJECT)hTarget);
            }
            break;
            case DSRL_TARGET_TYPE_STOCK_SYMBOL:
            {
                STOCK_SYMBOL.vDestroy((STOCK_SYMBOL_OBJECT)hTarget);
            }
            break;
            case DSRL_TARGET_TYPE_UNKNOWN:
            default:
            {
                // Don't do anything
            }
            break;
        }
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/


