/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DSRL_ENTRY implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "dsrl_obj.h"
#include "dsrl_target_obj.h"
#include "dataservice_mgr_impl.h"
#include "stock_symbol_obj.h"

#include "dsrl_entry_obj.h"
#include "_dsrl_entry_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eType
*
* This object interface method is used by the caller to retrieve the type
* this Data Entry is.
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to retreive the type for
*
* Outputs:
*
*   A valid DSRL_ENTRY_TYPE_ENUM on success
*   A DSRL_ENTRY_TYPE_UNKNOWN on failure
*
*****************************************************************************/
static DSRL_ENTRY_TYPE_ENUM eType (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    DSRL_ENTRY_TYPE_ENUM eType = DSRL_ENTRY_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return DSRL_ENTRY_TYPE_UNKNOWN;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        eType = psObj->eType;
    }

    return eType;
}

/*****************************************************************************
*
*   hTrafficMsg
*
* This object interface method returns a handle to a valid TRAFFIC_MSG_OBJECT if
* the type of this object is a DSRL_ENTRY_TYPE_TRAFFIC_MSG
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a location handle from
*
* Outputs:
*
*   A valid TRAFFIC_MSG_OBJECT handle on success
*   A TRAFFIC_MSG_INVALID_OBJECT on failure
*
*****************************************************************************/
static TRAFFIC_MSG_OBJECT hTrafficMsg (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    TRAFFIC_MSG_OBJECT hTrafficMsg = TRAFFIC_MSG_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return TRAFFIC_MSG_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_TRAFFIC_MSG)
        {
            hTrafficMsg = (TRAFFIC_MSG_OBJECT)hEntry;
        }
    }

    return hTrafficMsg;
}

/*****************************************************************************
*
*   hFuelStation
*
* This object interface method returns a handle to a valid FUEL_STATION_OBJECT
* if the type of this object is a DSRL_ENTRY_TYPE_FUEL_STATION
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a fuel station handle from
*
* Outputs:
*
*   A valid FUEL_STATION_OBJECT handle on success
*   A FUEL_STATION_INVALID_OBJECT on failure
*
*****************************************************************************/
static FUEL_STATION_OBJECT hFuelStation (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    FUEL_STATION_OBJECT hFuelStation = FUEL_STATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return FUEL_STATION_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_FUEL_STATION)
        {
            hFuelStation = (FUEL_STATION_OBJECT)hEntry;
        }
    }

    return hFuelStation;
}

/*****************************************************************************
*
*   hTheater
*
* This object interface method returns a handle to a valid THEATER_OBJECT if
* the type of this object is a DSRL_ENTRY_TYPE_THEATER
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a location handle from
*
* Outputs:
*
*   A valid THEATER_OBJECT handle on success
*   A THEATER_INVALID_OBJECT on failure
*
*****************************************************************************/
static THEATER_OBJECT hTheater (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    THEATER_OBJECT hTheater = THEATER_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return THEATER_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_THEATER)
        {
            hTheater = (THEATER_OBJECT)hEntry;
        }
    }

    return hTheater;
}

/*****************************************************************************
*
*   hAgwTile
*
* This object interface method returns a handle to a valid AGW_TILE if
* the type of this object is a DSRL_ENTRY_TYPE_AGW_TILE
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a AGW Tile handle from
*
* Outputs:
*
*   A valid AGW_TILE_OBJECT handle on success
*   A AGW_TILE_INVALID_OBJECT on failure
*
*****************************************************************************/
static AGW_TILE_OBJECT hAgwTile (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    AGW_TILE_OBJECT hAgwTile = AGW_TILE_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

        if (bOwner == TRUE)
        {
            if (psObj->eType == DSRL_ENTRY_TYPE_AGW_TILE)
            {
                hAgwTile = (AGW_TILE_OBJECT)hEntry;
            }
        }
    }

    return hAgwTile;
}

/*****************************************************************************
*
*   hAgwShape
*
* This object interface method returns a handle to a valid AGW_SHAPE if
* the type of this object is a DSRL_ENTRY_TYPE_AGW_SHAPE
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a AGW Shape handle from
*
* Outputs:
*
*   A valid AGW_SHAPE_OBJECT handle on success
*   A AGW_SHAPE_INVALID_OBJECT on failure
*
*****************************************************************************/
static AGW_SHAPE_OBJECT hAgwShape (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    AGW_SHAPE_OBJECT hAgwShape = AGW_SHAPE_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

        if (bOwner == TRUE)
        {
            if (psObj->eType == DSRL_ENTRY_TYPE_AGW_SHAPE)
            {
                hAgwShape = (AGW_SHAPE_OBJECT)hEntry;
            }
        }
    }

    return hAgwShape;
}

/*****************************************************************************
*
*   hWeatherMsg
*
* This object interface method returns a handle to a valid WEATHER_MSG_OBJECT if
* the type of this object is a DSRL_ENTRY_TYPE_WEATHER_MSG
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a Weather message handle from
*
* Outputs:
*
*   A valid WEATHER_MSG_OBJECT handle on success
*   A WEATHER_MSG_INVALID_OBJECT on failure
*
*****************************************************************************/
static WEATHER_MSG_OBJECT hWeatherMsg (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    WEATHER_MSG_OBJECT hWeatherMsg = WEATHER_MSG_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return WEATHER_MSG_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_WEATHER_MSG)
        {
            hWeatherMsg = (WEATHER_MSG_OBJECT)hEntry;
        }
    }

    return hWeatherMsg;
}

/*****************************************************************************
*
*   hStockMsg
*
* This object interface method returns a handle to a valid STOCK_MSG_OBJECT if
* the type of this object is a DSRL_ENTRY_TYPE_STOCK_MSG
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a Stock message handle from
*
* Outputs:
*
*   A valid STOCK_MSG_OBJECT handle on success
*   A STOCK_MSG_INVALID_OBJECT on failure
*
*****************************************************************************/
static STOCK_MSG_OBJECT hStockMsg (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    STOCK_MSG_OBJECT hStockMsg = STOCK_MSG_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return STOCK_MSG_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_STOCK_MSG)
        {
            hStockMsg = (STOCK_MSG_OBJECT)hEntry;
        }
    }

    return hStockMsg;
}


/*****************************************************************************
*
*   hSafeCamLoc
*
* This object interface method returns a handle to a valid
* SAFETY_CAMERAS_LOCATION_OBJECT if the type of this object is a
* DSRL_ENTRY_TYPE_SAFETY_CAMERAS_LOCATION
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a Safety Camera Location handle from
*
* Outputs:
*
*   A valid SAFETY_CAMERAS_LOCATION_OBJECT handle on success
*   A SAFETY_CAMERAS_LOCATION_INVALID_OBJECT on failure
*
*****************************************************************************/
static SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc = SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return SAFETY_CAMERAS_LOCATION_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_SAFETY_CAMERAS_LOCATION)
        {
            hSafeCamLoc = (SAFETY_CAMERAS_LOCATION_OBJECT)hEntry;
        }
    }

    return hSafeCamLoc;
}

/*****************************************************************************
*
*   hWSAlertMsg
*
* This object interface method returns a handle to a valid
* WS_ALERT_MSG_OBJECT if the type of this object is a
* DSRL_ENTRY_TYPE_WS_ALERTS_MSG
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to get a WS Alerts message handle from
*
* Outputs:
*
*   A valid WS_ALERT_MSG_OBJECT handle on success
*   A WS_ALERT_MSG_INVALID_OBJECT on failure
*
*****************************************************************************/
static WS_ALERT_MSG_OBJECT hWSAlertMsg (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    WS_ALERT_MSG_OBJECT hWSAlertMsg = WS_ALERT_MSG_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return WS_ALERT_MSG_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        if (psObj->eType == DSRL_ENTRY_TYPE_WS_ALERTS_MSG)
        {
            hWSAlertMsg = (WS_ALERT_MSG_OBJECT)hEntry;
        }
    }

    return hWSAlertMsg;
}

/*****************************************************************************
                             PUBLIC FUNCTIONS
                           FAVORITES INTERFACE
*****************************************************************************/
/*****************************************************************************
*
*   eSetFavorite
*
* This FAVORITES Interface methodd is used by the caller to set the favorite
* status of this object.
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to configure
*   bIsFavorite - A flag indicating if this entry is a favorite
*
* Outputs:
*
*   A valid DSRL_ENTRY_TYPE_ENUM on success
*   A DSRL_ENTRY_TYPE_UNKNOWN on failure
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM eSetFavorite (
    DSRL_ENTRY_OBJECT hEntry,
    BOOLEAN bIsFavorite
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bOwner;
    BOOLEAN bSuccess;
    DSRL_ARG_STRUCT *psEventArg;
    DSRL_MODIFY_OPERATION_ENUM eModifyOp = DSRL_MODIFY_OPERATION_ADD;
    DSRL_TARGET_OBJECT hTarget = DSRL_TARGET_INVALID_OBJECT;
    UN8 un8NumOfTargets = 1;
    LOCATION_OBJECT hLocation;
    LOCID_OBJECT hLocID;

    do
    {

        if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Check ownership
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner != TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (psObj->bIsFavorite == bIsFavorite)
        {
            // Nothing to do, so tell 'em it worked
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            break;
        }

        // Extract the target from the entry object
        switch (psObj->eType)
        {
            case DSRL_ENTRY_TYPE_FUEL_STATION:
            {
                // Extract the location from the station
                hLocation =
                    FUEL_STATION.hLocation( (FUEL_STATION_OBJECT)hEntry );

                // Extract the locId
                hLocID = LOCATION.hLocID(hLocation);

                // Create a new location object
                hLocation = LOCATION.hCreateForLocID(
                            OSAL_FIXED_INVALID_OBJECT,
                            OSAL_FIXED_INVALID_OBJECT, hLocID );

                // Use it as our target
                hTarget = (DSRL_TARGET_OBJECT)hLocation;
            }
            break;

            case DSRL_ENTRY_TYPE_THEATER:
            {
                // Extract the location from the theater
                hLocation =
                            THEATER.hLocation((THEATER_OBJECT)hEntry);

                // Extract the locId
                hLocID = LOCATION.hLocID(hLocation);

                // Create a new location object
                hLocation = LOCATION.hCreateForLocID(
                            OSAL_FIXED_INVALID_OBJECT,
                            OSAL_FIXED_INVALID_OBJECT, hLocID );

                // Use it as our target
                hTarget = (DSRL_TARGET_OBJECT)hLocation;
            }
            break;

            case DSRL_ENTRY_TYPE_WEATHER_MSG:
            {
                // Extract the location from the weather station
                hLocation = WEATHER_MSG.hLocation((WEATHER_MSG_OBJECT)hEntry);

                // Extract the locId
                hLocID = LOCATION.hLocID(hLocation);

                // Create a new location object
                hLocation = LOCATION.hCreateForLocID(
                        OSAL_FIXED_INVALID_OBJECT,
                        OSAL_FIXED_INVALID_OBJECT, hLocID );

                // Use it as our target
                hTarget = (DSRL_TARGET_OBJECT)hLocation;
            }
            break;

            case DSRL_ENTRY_TYPE_STOCK_MSG:
            {
                STOCK_SYMBOL_OBJECT hStockSymbol;
                STOCK_SYMBOL_OBJECT hStockSymbolDup;

                // Extract the stock symbol from the entry
                hStockSymbol = STOCK_MSG.hStockSymbol((STOCK_MSG_OBJECT) hEntry);

                hStockSymbolDup = STOCK_SYMBOL_hDuplicate(SMS_INVALID_OBJECT, hStockSymbol);

                // Use it as our target
                hTarget = (DSRL_TARGET_OBJECT)hStockSymbolDup;
            }
            break;

            default:
            {
            }
            break;
        }

        if (hTarget == DSRL_TARGET_INVALID_OBJECT)
        {
            // This object is a type which does not support
            // this feature
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }

        // Update this entry's favorite status
        psObj->bIsFavorite = bIsFavorite;

        // Use the remove operation if we are
        // no longer considering this to be a favorite
        if (bIsFavorite == FALSE)
        {
            eModifyOp = DSRL_MODIFY_OPERATION_REMOVE;
        }

        // Create memory for our argument to be passed along with the event
        psEventArg = DSRL_psCreateArg(un8NumOfTargets);

        if (psEventArg == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Set arguments
        psEventArg->eAction = DSRL_ACTION_MODIFY;
        psEventArg->eDSRLType = DSRL_TYPE_FAVORITES;
        psEventArg->hDSRL = DSRL_INVALID_OBJECT;
        psEventArg->uAction.sModify.eModifyType = eModifyOp;
        psEventArg->uAction.sModify.bForceDSRLStateChange = TRUE;
        psEventArg->tNumTargets = un8NumOfTargets;
        psEventArg->ahTargetList[0] = hTarget;

        // Inform the data service of the DSRL update
        bSuccess = DATASERVICE_IMPL_bDSRLUpdate(
            psObj->hService, psEventArg);

        if (bSuccess == FALSE)
        {
            // can't post the event
            // This function will free hTarget for us
            DSRL_vDestroyArg(psEventArg);

            // This service doesn't support this feature
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   bIsFavorite
*
* This FAVORITES interface method is used by the caller to query an entry for
* its favorite status.
*
* Inputs:
*
*   hEntry - The DSRL_ENTRY_OBJECT to query
*
* Outputs:
*
*   TRUE if the entry is a favorite, FALSE if not or if an error was
*   experienced
*
*****************************************************************************/
BOOLEAN bIsFavorite (
    DSRL_ENTRY_OBJECT hEntry
  	    )
{
    BOOLEAN bFavorite = FALSE;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        BOOLEAN bOwner;
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        // Check ownership
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

        if (bOwner == TRUE)
        {
            bFavorite = psObj->bIsFavorite;
        }
    }

    return bFavorite;
}

/*****************************************************************************
*
*   vClearAllFavorites
*
* This object interface method is used by the caller to clear all favorites
* from a data service.
*
* Inputs:
*
*   hManager - The DATASERVICE_MGR_OBJECT handle which represents
*       the service which must clear its favorites
*
* Outputs:
*
*   None
*
*****************************************************************************/
void vClearAllFavorites (
    DATASERVICE_MGR_OBJECT hManager
  	    )
{
    DSRL_ARG_STRUCT *psEventArg;
    BOOLEAN bSuccess;

    // Create memory for our argument to be passed along with the event
    psEventArg = DSRL_psCreateArg( 0 );
    if (psEventArg == NULL)
    {
        return;
    }

    // Set arguments
    psEventArg->eAction = DSRL_ACTION_MODIFY;
    psEventArg->uAction.sModify.eModifyType = DSRL_MODIFY_OPERATION_REMOVEALL;
    psEventArg->uAction.sModify.bForceDSRLStateChange = TRUE;
    psEventArg->eDSRLType = DSRL_TYPE_FAVORITES;
    psEventArg->hDSRL = DSRL_INVALID_OBJECT;
    psEventArg->tNumTargets = 0;

    // Inform the data service of the DSRL update
    bSuccess = DATASERVICE_IMPL_bDSRLUpdate(
        (DATASERVICE_IMPL_HDL)hManager,
        psEventArg);

    if (bSuccess == FALSE)
    {
        // can't post the event
        DSRL_vDestroyArg(psEventArg);
    }

    return;
}

/*****************************************************************************
*
*   eSetFavoriteByLocID
*
* This object interface method is used by the caller to set a particular
* LOC_ID_OBJECT as a favorite
*
* Inputs:
*
*   LOCID_OBJECT hLocID - A Handle to the LOCID object to update the favorite
*                         status on
*   DATASERVICE_MGR_OBJECT hManager - The DATASERVICE_MGR_OBJECT handle which
*                         represents the service which will have a favorite
*                         set
*   BOOLEAN bFavorite   - TRUE - mark as favorite
*                         FALSE - remove from favorite list
*
* Outputs:
*
*   None
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetFavoriteByLocID (
    LOCID_OBJECT hLocID,
    DATASERVICE_MGR_OBJECT hManager,
    BOOLEAN bIsFavorite
        )
{
    LOCATION_OBJECT hLocation;
    DSRL_TARGET_OBJECT hTarget = DSRL_TARGET_INVALID_OBJECT;
    DSRL_MODIFY_OPERATION_ENUM eModifyOp = DSRL_MODIFY_OPERATION_ADD;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bSuccess, bManagerValid;
    DSRL_ARG_STRUCT *psEventArg;
    UN8 un8NumOfTargets = 1;

    do
    {
        // Check input
        bManagerValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL) hManager);
        if (bManagerValid == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Extract the target from the entry object
        // Create a new location object
        hLocation = LOCATION.hCreateForLocID(
                            OSAL_FIXED_INVALID_OBJECT,
                            OSAL_FIXED_INVALID_OBJECT, hLocID );

        // Use it as our target
        hTarget = (DSRL_TARGET_OBJECT)hLocation;

        if (hTarget == DSRL_TARGET_INVALID_OBJECT)
        {
            // This object is a type which does not support
            // this feature
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Use the remove operation if we are
        // no longer considering this to be a favorite
        if (bIsFavorite == FALSE)
        {
            eModifyOp = DSRL_MODIFY_OPERATION_REMOVE;
        }

        // Create memory for our argument to be passed along with the event
        psEventArg = DSRL_psCreateArg(un8NumOfTargets);
        if (psEventArg == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Set arguments
        psEventArg->eAction = DSRL_ACTION_MODIFY;
        psEventArg->uAction.sModify.eModifyType = eModifyOp;
        psEventArg->hDSRL = DSRL_INVALID_OBJECT;
        psEventArg->eDSRLType = DSRL_TYPE_FAVORITES;
        psEventArg->uAction.sModify.bForceDSRLStateChange = TRUE;
        psEventArg->tNumTargets = un8NumOfTargets;
        psEventArg->ahTargetList[0] = hTarget;

        // Inform the data service of the DSRL update
        bSuccess = DATASERVICE_IMPL_bDSRLUpdate(
            (DATASERVICE_IMPL_HDL)hManager, psEventArg);

        if (bSuccess == FALSE)
        {
            // can't post the event
            // This function will free hTarget for us
            DSRL_vDestroyArg(psEventArg);

            // This service doesn't support this feature
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eSetFavoriteByStockSymbol
*
* This object interface method is used by the caller to set a particular
* STOCK_SYMBOL_OBJECT as a favorite
*
* Inputs:
*
*   STOCK_SYMBOL_OBJECT hStockSymbol
*                       - A Handle to the STOCK_SYMBOL object to update the favorite
*                         status on
*   DATASERVICE_MGR_OBJECT hManager - The DATASERVICE_MGR_OBJECT handle which
*                         represents the service which will have a favorite
*                         set
*   BOOLEAN bFavorite   - TRUE - mark as favorite
*                         FALSE - remove from favorite list
*
* Outputs:
*
*   None
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetFavoriteByStockSymbol (
    STOCK_SYMBOL_OBJECT hStockSymbol,
    DATASERVICE_MGR_OBJECT hManager,
    BOOLEAN bIsFavorite
        )
{
    STOCK_SYMBOL_OBJECT hStockSymbolTarget;
    DSRL_TARGET_OBJECT hTarget = DSRL_TARGET_INVALID_OBJECT;
    DSRL_MODIFY_OPERATION_ENUM eModifyOp = DSRL_MODIFY_OPERATION_ADD;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bSuccess, bManagerValid, bTargetOwned;
    DSRL_ARG_STRUCT *psEventArg;
    UN8 un8NumOfTargets = 1;

    do
    {
        // Check input
        bManagerValid = DATASERVICE_IMPL_bValid((DATASERVICE_IMPL_HDL) hManager);
        bTargetOwned = DSRL_TARGET_bOwner((DSRL_TARGET_OBJECT)hStockSymbol);
        if ((bManagerValid == FALSE) || (bTargetOwned == FALSE))
        {
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Extract the target from the entry object
        // Create a new stock symbol object
        hStockSymbolTarget = STOCK_SYMBOL_hDuplicate(SMS_INVALID_OBJECT, hStockSymbol);
        if (hStockSymbolTarget == STOCK_SYMBOL_INVALID_OBJECT)
        {
            eReturnCode = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        // Use it as our target
        hTarget = (DSRL_TARGET_OBJECT)hStockSymbolTarget;

        // Use the remove operation if we are
        // no longer considering this to be a favorite
        if (bIsFavorite == FALSE)
        {
            eModifyOp = DSRL_MODIFY_OPERATION_REMOVE;
        }

        // Create memory for our argument to be passed along with the event
        psEventArg = DSRL_psCreateArg(un8NumOfTargets);
        if (psEventArg == NULL)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Set arguments
        psEventArg->eAction = DSRL_ACTION_MODIFY;
        psEventArg->uAction.sModify.eModifyType = eModifyOp;
        psEventArg->hDSRL = DSRL_INVALID_OBJECT;
        psEventArg->eDSRLType = DSRL_TYPE_FAVORITES;
        psEventArg->uAction.sModify.bForceDSRLStateChange = TRUE;
        psEventArg->tNumTargets = un8NumOfTargets;
        psEventArg->ahTargetList[0] = hTarget;

        // Inform the data service of the DSRL update
        bSuccess = DATASERVICE_IMPL_bDSRLUpdate(
            (DATASERVICE_IMPL_HDL)hManager, psEventArg);

        if (bSuccess == FALSE)
        {
            // can't post the event
            // This function will free hTarget for us
            DSRL_vDestroyArg(psEventArg);

            // This service doesn't support this feature
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
            break;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   DSRL_ENTRY_hCreate
*
*****************************************************************************/
DSRL_ENTRY_OBJECT DSRL_ENTRY_hCreate (
    const char *pacName,
    DSRL_ENTRY_TYPE_ENUM eType,
    size_t tObjectSize,
    size_t tServiceDataSize,
    SMS_OBJECT hParent,
    BOOLEAN bLock
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Verify inputs
    if ((pacName == NULL) || (eType >= DSRL_ENTRY_TYPE_UNKNOWN))
    {
        // Error!
        return DSRL_ENTRY_INVALID_OBJECT;
    }

    // Construct an appropriate name for memory to be allocated
    snprintf(&acName[0], sizeof(acName),
        DSRL_ENTRY_PREFIX":%s", pacName);

    // Allocate an Data Entry object plus the requested memory
    psObj = (DSRL_ENTRY_OBJECT_STRUCT *)SMSO_hCreate(&acName[0],
        sizeof(DSRL_ENTRY_OBJECT_STRUCT) + tObjectSize + tServiceDataSize,
        hParent, bLock);
    if(psObj == NULL)
    {
        return DSRL_ENTRY_INVALID_OBJECT;
    }

    // Set the type
    psObj->eType = eType;

    // Clear the other fields
    psObj->hService = DATASERVICE_IMPL_INVALID_HDL;
    psObj->bIsFavorite = FALSE;

    if (tServiceDataSize != 0)
    {
        psObj->pvServiceData = (UN8*)(psObj + 1) + tObjectSize;
    }

    // return the pointer to the user space
    return (DSRL_ENTRY_OBJECT)(psObj + 1);
}

/*****************************************************************************
*
*   DSRL_ENTRY_hGetSMSObject
*
*****************************************************************************/
SMS_OBJECT DSRL_ENTRY_hGetSMSObject (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    SMS_OBJECT hObject = SMS_INVALID_OBJECT;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return SMS_INVALID_OBJECT;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        hObject = (SMS_OBJECT)psObj;
    }

    return hObject;
}

/*****************************************************************************
*
*   DSRL_ENTRY_bLock
*
*****************************************************************************/
BOOLEAN DSRL_ENTRY_bLock (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    BOOLEAN bLocked = FALSE;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        // Verify and lock SMS Object
        bLocked =
            SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    }

    return bLocked;
}

/*****************************************************************************
*
*   DSRL_ENTRY_vUnLock
*
*****************************************************************************/
void DSRL_ENTRY_vUnLock (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_ENTRY_bOwner
*
*****************************************************************************/
BOOLEAN DSRL_ENTRY_bOwner (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    BOOLEAN bOwner = FALSE;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    }

    return bOwner;
}

/*****************************************************************************
*
*   DSRL_ENTRY_hParent
*
*****************************************************************************/
SMS_OBJECT DSRL_ENTRY_hParent (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    SMS_OBJECT hParent = SMS_INVALID_OBJECT;

    if (hEntry != DSRL_ENTRY_INVALID_OBJECT)
    {
        hParent = SMSO_hParent((SMS_OBJECT)psObj);
    }

    return hParent;
}

/*****************************************************************************
*
*   DSRL_ENTRY_vSetService
*
*****************************************************************************/
void DSRL_ENTRY_vSetService (
    DSRL_ENTRY_OBJECT hEntry,
    DATASERVICE_IMPL_HDL hService
        )
{
    BOOLEAN bOwner;

    // Ensure we own this DSRL & entry
    bOwner = DSRL_ENTRY_bOwner(hEntry);
    if (bOwner == TRUE)
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        if (hService != DATASERVICE_IMPL_INVALID_HDL)
        {
            psObj->hService = hService;
        }
    }

    return;
}

/*****************************************************************************
*
*   DSRL_ENTRY_vSetFavorite
*
*****************************************************************************/
void DSRL_ENTRY_vSetFavorite (
    DSRL_ENTRY_OBJECT hEntry,
    BOOLEAN bIsFavorite
        )
{
    BOOLEAN bOwner;

    // Ensure we own this DSRL & entry
    bOwner = DSRL_ENTRY_bOwner(hEntry);
    if (bOwner == TRUE)
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;

        psObj->bIsFavorite = bIsFavorite;
    }

    return;
}

/*****************************************************************************
*
*   DSRL_ENTRY_vDestroy
*
*****************************************************************************/
void DSRL_ENTRY_vDestroy (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);

    if (bOwner == TRUE)
    {
        psObj->eType = DSRL_ENTRY_TYPE_UNKNOWN;

        if (psObj->hDataList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eLinkedListRemoveAll(psObj->hDataList, 
                OSAL.vLinkedListMemoryFree);

            OSAL.eLinkedListDelete(psObj->hDataList);
        }

        if (psObj->psEntryData != NULL)
        {
            OSAL.vLinkedListMemoryFree(psObj->psEntryData);
        }
        

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   DSRL_ENTRY_pvServiceData
*
*****************************************************************************/
void *DSRL_ENTRY_pvServiceData (
    DSRL_ENTRY_OBJECT hEntry
        )
{
    DSRL_ENTRY_OBJECT_STRUCT *psObj =
        (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
    BOOLEAN bOwner;

    if (hEntry == DSRL_ENTRY_INVALID_OBJECT)
    {
        return NULL;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if (bOwner != TRUE)
    {
        return NULL;
    }

    return psObj->pvServiceData;
}

/*****************************************************************************
*
*   DSRL_ENTRY_hDSRLListEntry
*
*****************************************************************************/
OSAL_LINKED_LIST_ENTRY DSRL_ENTRY_hDSRLListEntry (
    DSRL_ENTRY_OBJECT hEntry,
    DSRL_OBJECT hDSRL
        )
{
    OSAL_LINKED_LIST_ENTRY hResult = OSAL_INVALID_LINKED_LIST_ENTRY;

    do 
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
        BOOLEAN bOwner;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT) || 
            (hDSRL == DSRL_INVALID_OBJECT))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner != TRUE)
        {
            break;
        }

        if ((psObj->psEntryData != NULL) && 
            (psObj->psEntryData->hDSRL == hDSRL))
        {
            // we don't use list yet and single entry data struct
            // contains LL handle for requested DSRL
            hResult = psObj->psEntryData->hDSRLListEntry;
        }
        else
        {
            // we use list already so search in list is required

            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_LINKED_LIST_ENTRY hListEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            DSRL_ENTRY_DATA_STRUCT sSearchCriteria, *psEntryData;

            sSearchCriteria.hDSRL = hDSRL;

            eReturnCode = OSAL.eLinkedListSearch(psObj->hDataList, &hListEntry,
                &sSearchCriteria);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            psEntryData = (DSRL_ENTRY_DATA_STRUCT *)OSAL.pvLinkedListThis(
                hListEntry);
            hResult = psEntryData->hDSRLListEntry;
        }

    } while (FALSE);

    return hResult;
}

/*****************************************************************************
*
*   DSRL_ENTRY_bAddToDSRL
*
*****************************************************************************/
BOOLEAN DSRL_ENTRY_bAddToDSRL (
    DSRL_ENTRY_OBJECT hEntry,
    DSRL_OBJECT hDSRL,
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry
        )
{
    BOOLEAN bResult = FALSE;

    do 
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
        BOOLEAN bOwner;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        DSRL_ENTRY_DATA_STRUCT *psEntryData = NULL;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT) || 
            (hDSRL == DSRL_INVALID_OBJECT) || 
            (hDSRLListEntry == OSAL_INVALID_LINKED_LIST_ENTRY))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner != TRUE)
        {
            break;
        }

        // pre-check before to verify if this entry is already in DSRL
        if (psObj->hDataList == OSAL_INVALID_OBJECT_HDL)
        {
            if (psObj->psEntryData == NULL)
            {
                // this is 1st time when entry is added to DSRL
                psObj->psEntryData = 
                    (DSRL_ENTRY_DATA_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
                    DSRL_ENTRY_PREFIX":DSRLEntryData", 
                    sizeof(*psObj->psEntryData), TRUE);
                if (psObj->psEntryData == NULL)
                {
                    //error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                        DSRL_ENTRY_PREFIX": Failed to allocate memory for "
                        "DSRLEntryData");
                    break;
                }

                psObj->psEntryData->hDSRL = hDSRL;
                psObj->psEntryData->hDSRLListEntry = hDSRLListEntry;
                bResult = TRUE;
                break;
            }
            else if (psObj->psEntryData->hDSRL == hDSRL)
            {
                // currently, data list is not created but entry in DSRL already
                // it should not happen - we must avoid adding the entry
                // to DSRL twice
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    DSRL_ENTRY_PREFIX": Entry %p belongs to DSRL %p already",
                    hEntry, hDSRL);
                bResult = FALSE;
                break;
            }
            else if (psObj->psEntryData->hDSRL == DSRL_INVALID_OBJECT)
            {
                // we could reuse this entry data
                psObj->psEntryData->hDSRL = hDSRL;
                psObj->psEntryData->hDSRLListEntry = hDSRLListEntry;
                bResult = TRUE;
                break;
            }
            else //ok, lets create the list then
            {
                // need to create the list
                eReturnCode = OSAL.eLinkedListCreate(&psObj->hDataList, 
                    DSRL_ENTRY_PREFIX":DSRLDataList", 
                    (OSAL_LL_COMPARE_HANDLER)n16CompareDSRLs,
                    OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
                    OSAL_LL_OPTION_UNIQUE);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                        DSRL_ENTRY_PREFIX": Failed to create DSRLDataList "
                        "for entry %p", hEntry);
                    break;
                }

                // adding the existing entry to list
                eReturnCode = OSAL.eLinkedListAdd(psObj->hDataList,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psObj->psEntryData);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    // looks like list is not working, lets remove it
                    OSAL.eLinkedListDelete(psObj->hDataList);
                    psObj->hDataList = OSAL_INVALID_OBJECT_HDL;

                    // invalid case
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                        DSRL_ENTRY_PREFIX": Failed to add entry-to-DSRL link"
                        " to list for entry %p, DSRL %p (%s)", hEntry, 
                        psObj->psEntryData->hDSRL,
                        OSAL.pacGetReturnCodeName(eReturnCode));
                    break;
                }

                // skipping the existing entry data to avoid any issues
                psObj->psEntryData = NULL;
            }
        }

        // creating the structure for this DSRL
        psEntryData = (DSRL_ENTRY_DATA_STRUCT *)OSAL.pvLinkedListMemoryAllocate(
            DSRL_ENTRY_PREFIX":DSRLEntryData", sizeof(*psEntryData), TRUE);
        if (psEntryData == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                DSRL_ENTRY_PREFIX": Failed to allocate memory for "
                "entry-to-DSRL link for entry %p, DSRL %p", hEntry, hDSRL);
            break;
        }

        psEntryData->hDSRL = hDSRL;
        psEntryData->hDSRLListEntry = hDSRLListEntry;

        eReturnCode = OSAL.eLinkedListAdd(psObj->hDataList, 
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psEntryData);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                DSRL_ENTRY_PREFIX": Failed to add entry-to-DSRL link"
                " to list for entry %p, DSRL %p (%s)", hEntry, hDSRL,
                OSAL.pacGetReturnCodeName(eReturnCode));

            OSAL.vLinkedListMemoryFree(psEntryData);
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   DSRL_ENTRY_bRemoveFromDSRL
*
*****************************************************************************/
BOOLEAN DSRL_ENTRY_bRemoveFromDSRL (
    DSRL_ENTRY_OBJECT hEntry,
    DSRL_OBJECT hDSRL
        )
{
    BOOLEAN bResult = FALSE;

    do 
    {
        DSRL_ENTRY_OBJECT_STRUCT *psObj =
            (DSRL_ENTRY_OBJECT_STRUCT *)hEntry - 1;
        BOOLEAN bOwner;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        if ((hEntry == DSRL_ENTRY_INVALID_OBJECT) || 
            (hDSRL == DSRL_INVALID_OBJECT))
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner != TRUE)
        {
            break;
        }


        if (psObj->hDataList != OSAL_INVALID_OBJECT_HDL)
        {
            // we have the list of entries

            DSRL_ENTRY_DATA_STRUCT sSearchCriteria;
            OSAL_LINKED_LIST_ENTRY hListEntry = 
                OSAL_INVALID_LINKED_LIST_ENTRY;
            DSRL_ENTRY_DATA_STRUCT *psEntryData;

            sSearchCriteria.hDSRL = hDSRL;

            eReturnCode = OSAL.eLinkedListSearch(psObj->hDataList, 
                &hListEntry, &sSearchCriteria);
            if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
            {
                // looks like entry is not linked to DSRL
                printf(DSRL_ENTRY_PREFIX": Entry %p not in DSRL %p\n", hEntry,
                    hDSRL);
                break;
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    DSRL_ENTRY_PREFIX": Failed to search for entry-to-DSRL "
                    "link for entry %p, DSRL %p (%s)", hEntry, hDSRL,
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            psEntryData = (DSRL_ENTRY_DATA_STRUCT *)OSAL.pvLinkedListThis(
                hListEntry);

            eReturnCode = OSAL.eLinkedListRemove(hListEntry);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    DSRL_ENTRY_PREFIX": Failed to remove entry-to-DSRL "
                    "link from list for entry %p, DSRL %p (%s)", hEntry, hDSRL,
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }

            OSAL.vLinkedListMemoryFree(psEntryData);
        }
        else if ((psObj->psEntryData != NULL) &&
                 (psObj->psEntryData->hDSRL == hDSRL))
        {
            // we have a single DSRL for this entry
            psObj->psEntryData->hDSRL = DSRL_INVALID_OBJECT;
        }
        else
        {
            // nothing to do here
            printf(DSRL_ENTRY_PREFIX": Entry %p not in DSRL %p\n", hEntry,
                hDSRL);
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16CompareDSRLs
*
*****************************************************************************/
static N16 n16CompareDSRLs (
    DSRL_ENTRY_DATA_STRUCT *psEntry1,
    DSRL_ENTRY_DATA_STRUCT *psEntry2
        )
{
    return COMPARE(psEntry1->hDSRL, psEntry2->hDSRL);
}
