/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains utility functions
 *  used in some data services implementation.
 *
 ******************************************************************************/
#include <stdio.h>
#include "standard.h"
#include "osal.h"
#include "sms_obj.h"

#include "_ds_util.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *  DS_UTIL_bExtractBitField
 *
 *****************************************************************************/
BOOLEAN DS_UTIL_bExtractBitField (
    UN32 *pun32Value,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8FieldBitLen
        )
{
    UN32 un32Val = 0;
    BOOLEAN bResult = FALSE;

    bResult = OSAL.bBufferReadBitsToUN32( hPayload, &un32Val, un8FieldBitLen);
    if (bResult == TRUE)
    {
        *pun32Value = un32Val;
    }

    return bResult;
}

/*****************************************************************************
 *
 *  DS_UTIL_bExtractOptionalBitField
 *
 *****************************************************************************/
BOOLEAN DS_UTIL_bExtractOptionalBitField (
    UN32 *pun32Value,
    OSAL_BUFFER_HDL hPayload,
    UN8 un8FieldBitLen,
    UN32 un32DefaultValue
        )
{
    UN32 un32Val = 0;
    BOOLEAN bResult = TRUE;

    do
    {
        // Check field presence flag (1 bit)
        bResult = OSAL.bBufferReadBitsToUN32(hPayload, &un32Val, 1);
        if (bResult == FALSE)
        {
            break;
        }

        if (un32Val == 0)
        {
            // Field is not present. Set default value.
            *pun32Value = un32DefaultValue;
            break;
        }

        // Field is present. Read field value.
        bResult = OSAL.bBufferReadBitsToUN32( hPayload, &un32Val, un8FieldBitLen);
        if (bResult == FALSE)
        {
            break;
        }

        *pun32Value = un32Val;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  DS_UTIL_bExtractBytes
 *
 *****************************************************************************/
BOOLEAN DS_UTIL_bExtractBytes (
    void *pvDst,
    OSAL_BUFFER_HDL hPayload,
    size_t tSize
        )
{
    size_t tBytesRead = 0;

    tBytesRead = OSAL.tBufferReadHead( hPayload, pvDst, tSize);
    if (tBytesRead != tSize)
    {
        SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
            PC_RED"ERROR: Failed to read %u bytes.\n"PC_RESET, tSize);
        return FALSE;
    }

    return TRUE;
}

/*******************************************************************************
 *
 *  DS_UTIL_bCutCRC
 *
 *******************************************************************************/
BOOLEAN DS_UTIL_bCutCRC (
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tNumBytes = 0;
    BOOLEAN bResult = TRUE;

    // Cut CRC
    tNumBytes = OSAL.tBufferSeekTail(hPayload, AU_CRC_BYTELEN);
    if (tNumBytes != AU_CRC_BYTELEN)
    {
        bResult = FALSE;
    }

    return bResult;
}

/*******************************************************************************
 *
 *  DS_UTIL_bIsCRCValid
 *
 *******************************************************************************/
BOOLEAN DS_UTIL_bIsCRCValid (
    OSAL_OBJECT_HDL hCRC,
    OSAL_BUFFER_HDL hPayload,
    OSAL_CRC_RESULT *ptCalculatedCrc
        )
{
    OSAL_CRC_RESULT tCRC = 0;
    OSAL_CRC_RESULT tPayloadCRC = 0;
    size_t tBufferSizeInBits = 0;
    size_t tNumBytesToCRC = 0;
    size_t tCRCOffsetInBits = 0;
    size_t tBytesProcessed = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Get the bit-size of the entire message
        tBufferSizeInBits = OSAL.tBufferGetSizeInBits(hPayload);
        if (tBufferSizeInBits <= AU_CRC_BITLEN)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: AU too small to read CRC.\n"PC_RESET);
            break;
        }

        // Calculate the number of bytes to check with a CRC
        tNumBytesToCRC = OSAL.tBufferGetSize(hPayload) - AU_CRC_BYTELEN;

        // Our offset starts at the last byte
        tCRCOffsetInBits = tBufferSizeInBits - AU_CRC_BITLEN;

        // Peek the CRC from the payload
        bResult = OSAL.bBufferPeekBitsToUN32(hPayload, &tPayloadCRC,
                                            AU_CRC_BITLEN, tCRCOffsetInBits);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Error reading payload during CRC check.\n"PC_RESET);
            break;
        }

        // Prepare CRC holder
        if (ptCalculatedCrc == NULL)
        {
            ptCalculatedCrc = &tCRC;
        }

        // Initialize the CRC
        bResult = OSAL.bInitializeCRC(hCRC, ptCalculatedCrc);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to initialize OSAL CRC with handle %p!\n"PC_RESET,
                   hCRC);
            break;
        }

        // Compute the CRC of the entire payload
        *ptCalculatedCrc = OSAL.tComputeCRC(hCRC,
            *ptCalculatedCrc,
            hPayload,
            0,
            tNumBytesToCRC,
            &tBytesProcessed);
        if (tBytesProcessed != tNumBytesToCRC)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: CRC processed length mismatches payload length!\n"PC_RESET);
            bResult = FALSE;
            break;
        }

        // Invert the CRC bits (RX 185, Section 4.3.3)
        *ptCalculatedCrc ^= DS_UTIL_AU_CRC_INVERSION_VALUE;

        // The packet is valid if the calculated CRC is
        // equal to the CRC field in the message
        bResult = (*ptCalculatedCrc == tPayloadCRC) ? TRUE : FALSE;
    } while (FALSE);

    SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 11,
        "CRC check result = %u\n", bResult);

    return bResult;
}

/*****************************************************************************
 *
 *  DS_UTIL_bCopyPayload
 *
 *****************************************************************************/
BOOLEAN DS_UTIL_bCopyPayload (
    OSAL_BUFFER_HDL hPayload,
    void *pvDstData,
    size_t *ptDataSize
        )
{
    size_t tBufSizeBytes = 0;
    size_t tBufSizeBits = 0;
    size_t tBitsRead = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        if (pvDstData == NULL || ptDataSize == NULL)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid output data param(s).\n"PC_RESET);
            break;
        }

        tBufSizeBytes = OSAL.tBufferGetSize(hPayload);
        tBufSizeBits = OSAL.tBufferGetSizeInBits(hPayload);

        if ((tBufSizeBits % BITS_IN_BYTE) != 0)
        {
            tBufSizeBytes++;
        }

        if ((*ptDataSize) < tBufSizeBytes)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Output data buffer size is not enough.\n"PC_RESET);
            break;
        }

        tBitsRead = OSAL.tBufferPeekBits(hPayload,
            pvDstData,
            0,
            tBufSizeBits,
            0);
        if (tBitsRead != tBufSizeBits)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to copy payload data.\n"PC_RESET);
            break;
        }

        *ptDataSize = tBufSizeBytes;

        // Everything's OK
        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  DS_UTIL_pvCopyPayload
 *
 *****************************************************************************/
void *DS_UTIL_pvCopyPayload (
    OSAL_BUFFER_HDL hPayload,
    size_t *ptDataSize
        )
{
    size_t tBufSizeBytes = 0;
    size_t tBufSizeBits = 0;
    void *pvData = NULL;
    size_t tBitsRead = 0;

    do
    {
        if (ptDataSize == NULL)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid output param to set data size.\n"PC_RESET);
            break;
        }

        tBufSizeBytes = OSAL.tBufferGetSize(hPayload);
        tBufSizeBits = OSAL.tBufferGetSizeInBits(hPayload);

        if ((tBufSizeBits % BITS_IN_BYTE) != 0)
        {
            tBufSizeBytes++;
        }

        pvData = (void*) SMSO_hCreate("PayloadData",
            tBufSizeBytes,
            SMS_INVALID_OBJECT, // No parent
            FALSE);             // No lock
        if ((SMS_OBJECT)pvData == SMS_INVALID_OBJECT)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Can't create AU payload.\n"PC_RESET);
            pvData = NULL;
            break;
        }

        SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 11,
            "Payload copy created: %p with size %u.\n",
            pvData, tBufSizeBytes);

        tBitsRead = OSAL.tBufferPeekBits(hPayload,
            pvData,
            0,
            tBufSizeBits,
            0);
        if (tBitsRead != tBufSizeBits)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to copy payload data.\n"PC_RESET);
            SMSO_vDestroy((SMS_OBJECT)pvData);
            pvData = NULL;
            break;
        }

        *ptDataSize = tBufSizeBytes;

    } while (FALSE);

    return pvData;
}

/*****************************************************************************
 *
 *  DS_UTIL_bCopyFile
 *
 *****************************************************************************/
BOOLEAN DS_UTIL_bCopyFile (
    char *pcSrcFilePath,
    char *pcDstFilePath,
    BOOLEAN bReplace,
    BOOLEAN bRename
        )
{
    N32 nError = 0;
    UN8 un8FileAttr = 0;
    BOOLEAN bResult = TRUE;

    do
    {
        if (pcSrcFilePath == NULL || pcDstFilePath == NULL)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                PC_RED"ERROR: File name is not specified.\n");
            bResult = FALSE;
            break;
        }

        bResult = OSAL.bFileSystemGetFileAttributes(pcSrcFilePath, &un8FileAttr);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 6,
                PC_RED"WARNING: Source file does not exist: %s\n"PC_RESET,
                pcSrcFilePath);
            bResult = FALSE;
            break;
        }

        // Remove existing file
        bResult = OSAL.bFileSystemGetFileAttributes(pcDstFilePath, &un8FileAttr);
        if (bResult == TRUE)
        {
            // Destination file exists
            if (bReplace == FALSE)
            {
                // Destination file should be replaced
                SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 1,
                    PC_RED"WARNING: Destination file exists: %s.\n"PC_RESET,
                    pcDstFilePath);
                bResult = FALSE;
                break;
            }

            // Destination file should be replaced
            nError = remove(pcDstFilePath);
            if (nError != 0)
            {
                SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to remove file %s.\n"PC_RESET,
                    pcDstFilePath);
                bResult = FALSE;
                break;
            }
        }

        if (bRename == TRUE)
        {
            nError = rename(pcSrcFilePath, pcDstFilePath);
            if (nError != 0)
            {
                SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to rename file %s to %s.\n"PC_RESET,
                    pcSrcFilePath, pcDstFilePath);
                bResult = FALSE;
                break;
            }
        }
        else
        {
            nError = OSAL.iCopyFile(pcSrcFilePath, pcDstFilePath, TRUE);
            if (nError != 0)
            {
                SMSAPI_DEBUG_vPrint(DS_UTIL_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to copy file %s to %s.\n"PC_RESET,
                    pcSrcFilePath, pcDstFilePath);
                bResult = FALSE;
                break;
            }
        }

        // Everything's done
        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 INLINE FUNCTIONS
 *****************************************************************************/
