/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:Detailed Signal Quality implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "sms_obj.h"
#include "_detailed_signal_quality_obj.h"
#include "detailed_signal_quality_obj.h"
#include "engineering_data_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   hSigStrength
*
*****************************************************************************/
static STRING_OBJECT hSigStrength (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hSigStrength;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hTunerStatus
*
*****************************************************************************/
static STRING_OBJECT hTunerStatus (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hTunerStatus;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hENSALockStatus
*
*****************************************************************************/
static STRING_OBJECT hENSALockStatus (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hENSALockStatus;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hENSBLockStatus
*
*****************************************************************************/
static STRING_OBJECT hENSBLockStatus (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hENSBLockStatus;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hBERS1
*
*****************************************************************************/
static STRING_OBJECT hBERS1 (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hBERS1;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hBERS2
*
*****************************************************************************/
static STRING_OBJECT hBERS2 (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hBERS2;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hBERT
*
*****************************************************************************/
static STRING_OBJECT hBERT (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hBERT;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hCNS1A
*
*****************************************************************************/
static STRING_OBJECT hCNS1A (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hCNS1A;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hCNS1B
*
*****************************************************************************/
static STRING_OBJECT hCNS1B (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hCNS1B;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hCNS2A
*
*****************************************************************************/
static STRING_OBJECT hCNS2A (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hCNS2A;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hCNS2B
*
*****************************************************************************/
static STRING_OBJECT hCNS2B (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hCNS2B;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hRSErrsWord
*
*****************************************************************************/
static STRING_OBJECT hRSErrsWord (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hRSErrsWord;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hRSErrsSatSymb
*
*****************************************************************************/
static STRING_OBJECT hRSErrsSatSymb (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hRSErrsSatSymb;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hRSErrsTerrSymb
*
*****************************************************************************/
static STRING_OBJECT hRSErrsTerrSymb (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hRSErrsTerrSymb;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hTunerCarrierFreqOffset
*
*****************************************************************************/
static STRING_OBJECT hTunerCarrierFreqOffset (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hTunerCarrierFreqOffset;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   hRSSI
*
*****************************************************************************/
static STRING_OBJECT hRSSI (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            )
{
    BOOLEAN bOwner;
    STRING_OBJECT hStringReturn = STRING_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);
    if ( bOwner == TRUE )
    {
        DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
        psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *) hDetSigQuality;
        hStringReturn = psObj->hRSSI;
    }

    return hStringReturn;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality,
    FILE *psFile
        )

{
    N32 n32Return = 0;
    BOOLEAN bOwner = FALSE;

    if (  ( psFile == NULL )
        ||( hDetSigQuality == DETAILED_SIGNAL_QUALITY_INVALID_OBJECT )
       )
    {
        return EOF;
    }
    bOwner = SMSO_bOwner((SMS_OBJECT)hDetSigQuality);

    if ( bOwner==TRUE )
    {
        STRING_OBJECT hStr;
        n32Return += fprintf(psFile, "DETAILED_SIGNAL_QUALITY_OBJECT\n");

        //hSigStrength
        hStr = DETAILED_SIGNAL_QUALITY.hSigStrength(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "SigStrength=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hTunerStatus
        hStr = DETAILED_SIGNAL_QUALITY.hTunerStatus(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "TunerStatus=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hENSALockStatus
        hStr = DETAILED_SIGNAL_QUALITY.hENSALockStatus(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "ENSALockStatus=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hENSBLockStatus
        hStr = DETAILED_SIGNAL_QUALITY.hENSBLockStatus(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "ENSBLockStatus=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hBERS1
        hStr = DETAILED_SIGNAL_QUALITY.hBERS1(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "BERS1=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hBERS2
        hStr = DETAILED_SIGNAL_QUALITY.hBERS2(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "BERS2=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hBERT
        hStr = DETAILED_SIGNAL_QUALITY.hBERT(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "BERT=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hCNS1A
        hStr = DETAILED_SIGNAL_QUALITY.hCNS1A(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "CNS1A=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hCNS1B
        hStr = DETAILED_SIGNAL_QUALITY.hCNS1B(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "CNS1B=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hCNS2A
        hStr = DETAILED_SIGNAL_QUALITY.hCNS2A(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "CNS2A=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hCNS2B
        hStr = DETAILED_SIGNAL_QUALITY.hCNS2B(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "CNS2B=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hRSErrsWord
        hStr = DETAILED_SIGNAL_QUALITY.hRSErrsWord(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "RSErrsWord=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hRSErrsSatSymb
        hStr = DETAILED_SIGNAL_QUALITY.hRSErrsSatSymb(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "RSErrsSatSymb=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hRSErrsTerrSymb
        hStr = DETAILED_SIGNAL_QUALITY.hRSErrsTerrSymb(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "RSErrsTerrSymb=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hTunerCarrierFreqOffset
        hStr = DETAILED_SIGNAL_QUALITY.hTunerCarrierFreqOffset(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "TunerCarrierFreqOffset=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }

        //hRSSI
        hStr = DETAILED_SIGNAL_QUALITY.hRSSI(hDetSigQuality);
        if ( hStr != STRING_INVALID_OBJECT )
        {
            n32Return += fprintf(psFile, "RSSI=%s\n",
                    STRING.pacCStr(hStr));
        }
        else
        {
           SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	            DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	            ": Invalid STRING_OBJECT");
        }
    }
    else
    {
       SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	        DETAILED_SIGNAL_QUALITY_OBJECT_NAME
	        ": Not owner");
	   return EOF;
    }
    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   DETAILED_SIGNAL_QUALITY_bUpdateDetailedSignalQuality
*
*****************************************************************************/
BOOLEAN DETAILED_SIGNAL_QUALITY_bUpdateDetailedSignalQuality(
    DECODER_OBJECT hDecoder,
    DETAILED_SIGNAL_QUALITY_STRUCT const *psDetailedSignalQuality
        )

{
    DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bReturn = FALSE;
    BOOLEAN bFirstUpdateReceived = FALSE;

    if ( hDecoder == DECODER_INVALID_OBJECT )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": Invalid DECODER_OBJECT");
        return bReturn;
    }

    eReturn = eAccessObjectFromDecoder(hDecoder,
                    (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT **)&psObj);
    if ( eReturn != SMSAPI_RETURN_CODE_SUCCESS )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": Unable to access DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT");
        return bReturn;
    }


    bFirstUpdateReceived = psObj->bFirstUpdateReceived;

    //Determine if any values have changed and update those values

    //n16RSSI
    if (  ( psObj->sCurrentSignalQuality.n16RSSI
          != psDetailedSignalQuality->n16RSSI
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        if ( (psDetailedSignalQuality->n16RSSI <= MAX_RSSI_DISPLAY) &&
             (psDetailedSignalQuality->n16RSSI >= MIN_RSSI_DISPLAY)
           )
        {
            snprintf(pStr, RSSI_SIZE, "%4d", psDetailedSignalQuality->n16RSSI);
        }
        else
        {
            snprintf(pStr, RSSI_SIZE, INVALID_4);
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": RSSI value out of range %d",
                    psDetailedSignalQuality->n16RSSI);
        }
        STRING.bModifyCStr(psObj->hRSSI, pStr);
        bReturn = TRUE;

    }

    //n16TunerCarrierFreqOffset
    if (  ( psObj->sCurrentSignalQuality.n16TunerCarrierFreqOffset
          != psDetailedSignalQuality->n16TunerCarrierFreqOffset
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        if (psDetailedSignalQuality->n16TunerCarrierFreqOffset
                > MAX_OFFSET_DISPLAY
           )
        {
            snprintf(pStr, CARRIER_FREQ_OFFSET_SIZE, "%s",
                MAX_OFFSET_STRING);
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": TUNER Carrier Freq Offset value out of display range %d",
                    psDetailedSignalQuality->n16TunerCarrierFreqOffset);
        }
        else if (psDetailedSignalQuality->n16TunerCarrierFreqOffset
                  < MIN_OFFSET_DISPLAY
                )
        {
            snprintf(pStr, CARRIER_FREQ_OFFSET_SIZE, "%s",
                MIN_OFFSET_STRING);
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": TUNER Carrier Freq Offset value out of display range %d",
                    psDetailedSignalQuality->n16TunerCarrierFreqOffset);
        }
        else
        {
             snprintf(pStr, CARRIER_FREQ_OFFSET_SIZE, "%6d",
                psDetailedSignalQuality->n16TunerCarrierFreqOffset * 100);
        }

        STRING.bModifyCStr( psObj->hTunerCarrierFreqOffset, pStr );
        bReturn = TRUE;
    }

    //un16BERS1
    if (  ( psObj->sCurrentSignalQuality.un16BERS1
          != psDetailedSignalQuality->un16BERS1
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hBERS1, psDetailedSignalQuality->un16BERS1);
        bReturn = TRUE;
    }

    //un16BERS2
    if (  ( psObj->sCurrentSignalQuality.un16BERS2
          != psDetailedSignalQuality->un16BERS2
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hBERS2, psDetailedSignalQuality->un16BERS2);
        bReturn = TRUE;
    }

    //un16BERT
    if (  ( psObj->sCurrentSignalQuality.un16BERT
          != psDetailedSignalQuality->un16BERT
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hBERT, psDetailedSignalQuality->un16BERT);
        bReturn = TRUE;
    }

    //un16RRSErrsTerrSymb
    if (  ( psObj->sCurrentSignalQuality.un16RRSErrsTerrSymb
          != psDetailedSignalQuality->un16RRSErrsTerrSymb
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hRSErrsTerrSymb,
                psDetailedSignalQuality->un16RRSErrsTerrSymb);
        bReturn = TRUE;
    }

    //un16RSErrsSatSymb
    if (  ( psObj->sCurrentSignalQuality.un16RSErrsSatSymb
          != psDetailedSignalQuality->un16RSErrsSatSymb
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hRSErrsSatSymb,
                psDetailedSignalQuality->un16RSErrsSatSymb);
        bReturn = TRUE;
    }

    //un16RSErrsWords
    if (  ( psObj->sCurrentSignalQuality.un16RSErrsWords
          != psDetailedSignalQuality->un16RSErrsWords
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        vPercentStringModify(psObj->hRSErrsWord,
                psDetailedSignalQuality->un16RSErrsWords);
        bReturn = TRUE;
    }

    //un8CNS1A
    if (  ( psObj->sCurrentSignalQuality.un8CNS1A
          != psDetailedSignalQuality->un8CNS1A
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, CN_SIZE, "%02u.%02u",
            (UN8)(psDetailedSignalQuality->un8CNS1A / 4),
            (UN8)((psDetailedSignalQuality->un8CNS1A % 4) * 25));
        STRING.bModifyCStr(psObj->hCNS1A, pStr);
        bReturn = TRUE;
    }

    //un8CNS1B
    if (  ( psObj->sCurrentSignalQuality.un8CNS1B
          != psDetailedSignalQuality->un8CNS1B
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, CN_SIZE, "%02u.%02u",
            (UN8)(psDetailedSignalQuality->un8CNS1B / 4),
            (UN8)((psDetailedSignalQuality->un8CNS1B % 4) * 25));
        STRING.bModifyCStr(psObj->hCNS1B, pStr);
        bReturn = TRUE;
    }

    //un8CNS2A
    if (  ( psObj->sCurrentSignalQuality.un8CNS2A
          != psDetailedSignalQuality->un8CNS2A
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];

        snprintf(pStr, CN_SIZE, "%02u.%02u",
            (UN8)(psDetailedSignalQuality->un8CNS2A / 4),
            (UN8)((psDetailedSignalQuality->un8CNS2A % 4) * 25));
        STRING.bModifyCStr( psObj->hCNS2A, pStr );
        bReturn = TRUE;
    }

    //un8CNS2B
    if (  ( psObj->sCurrentSignalQuality.un8CNS2B
          != psDetailedSignalQuality->un8CNS2B
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, CN_SIZE, "%02u.%02u",
            (UN8)(psDetailedSignalQuality->un8CNS2B / 4),
            (UN8)((psDetailedSignalQuality->un8CNS2B % 4) * 25));
        STRING.bModifyCStr( psObj->hCNS2B, pStr );
        bReturn = TRUE;
    }

    //un8ENSALockStatus
    if (  ( psObj->sCurrentSignalQuality.un8ENSALockStatus
          != psDetailedSignalQuality->un8ENSALockStatus
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, BYTEFIELD_SIZE, "x%02x",
                psDetailedSignalQuality->un8ENSALockStatus);
        STRING.bModifyCStr(psObj->hENSALockStatus, pStr);
        bReturn = TRUE;
    }

    //un8ENSBLockStatus
    if (  ( psObj->sCurrentSignalQuality.un8ENSBLockStatus
          != psDetailedSignalQuality->un8ENSBLockStatus
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, BYTEFIELD_SIZE, "x%02x",
                psDetailedSignalQuality->un8ENSBLockStatus);
        STRING.bModifyCStr( psObj->hENSBLockStatus, pStr );
        bReturn = TRUE;
    }

    //un8SignalStrength
    if (  ( psObj->sCurrentSignalQuality.eSignalStrength
          != psDetailedSignalQuality->eSignalStrength
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        if (psDetailedSignalQuality->eSignalStrength
             <= DETAILED_SIGNAL_QUALITY_GOOD
           )
        {
            snprintf(pStr, BYTEFIELD_SIZE, "x%02x",
                    psDetailedSignalQuality->eSignalStrength);
        }
        else
        {
            snprintf(pStr, BYTEFIELD_SIZE, "%s",
                    INVALID_3);
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": TUNER Signal Strength out of range %d",
                    psDetailedSignalQuality->eSignalStrength);
        }
        STRING.bModifyCStr(psObj->hSigStrength, pStr);
        bReturn = TRUE;
    }

    //un8TunerStatus
    if (  ( psObj->sCurrentSignalQuality.un8TunerStatus
          != psDetailedSignalQuality->un8TunerStatus
          )
       || ( bFirstUpdateReceived == FALSE )
       )
    {
        char pStr[MAX_STRING_SIZE];
        snprintf(pStr, BYTEFIELD_SIZE, "x%02x",
                psDetailedSignalQuality->un8TunerStatus);
        STRING.bModifyCStr(psObj->hTunerStatus, pStr);
        bReturn = TRUE;
    }

    psObj->sCurrentSignalQuality = *psDetailedSignalQuality;
    psObj->bFirstUpdateReceived = TRUE;
    return bReturn;

}

/*****************************************************************************
*
*   DETAILED_SIGNAL_QUALITY_bInitializeObject
*
*****************************************************************************/
void DETAILED_SIGNAL_QUALITY_vInitializeObject(
    DETAILED_SIGNAL_QUALITY_OBJECT hDetailedSignalQuality
        )
{
    DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;

    psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *)hDetailedSignalQuality;

    STRING_vDestroy(psObj->hBERS1);
    STRING_vDestroy(psObj->hBERS2);
    STRING_vDestroy(psObj->hBERT);
    STRING_vDestroy(psObj->hCNS1A);
    STRING_vDestroy(psObj->hCNS1B);
    STRING_vDestroy(psObj->hCNS2A);
    STRING_vDestroy(psObj->hCNS2B);
    STRING_vDestroy(psObj->hENSALockStatus);
    STRING_vDestroy(psObj->hENSBLockStatus);
    STRING_vDestroy(psObj->hRSErrsSatSymb);
    STRING_vDestroy(psObj->hRSErrsTerrSymb);
    STRING_vDestroy(psObj->hRSErrsWord);
    STRING_vDestroy(psObj->hRSSI);
    STRING_vDestroy(psObj->hSigStrength);
    STRING_vDestroy(psObj->hTunerCarrierFreqOffset);
    STRING_vDestroy(psObj->hTunerStatus);

    // Initialize Object values
	*psObj = gsObjectDefaults;

    //Create strings and set values to one obviously not from decoder.
    psObj->hSigStrength = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_3,
                            BYTEFIELD_SIZE , BYTEFIELD_SIZE );
    psObj->hTunerStatus = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_3,
                            BYTEFIELD_SIZE , BYTEFIELD_SIZE );
    psObj->hENSALockStatus = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_3,
                            BYTEFIELD_SIZE , BYTEFIELD_SIZE );
    psObj->hENSBLockStatus = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_3,
                            BYTEFIELD_SIZE , BYTEFIELD_SIZE );
    psObj->hBERS1 = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hBERS2 = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hBERT = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hCNS1A = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            CN_SIZE, CN_SIZE);
    psObj->hCNS1B = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            CN_SIZE, CN_SIZE);
    psObj->hCNS2A = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            CN_SIZE, CN_SIZE);
    psObj->hCNS2B = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            CN_SIZE, CN_SIZE);
    psObj->hRSErrsWord = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hRSErrsSatSymb = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hRSErrsTerrSymb = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            PERCENT_SIZE, PERCENT_SIZE);
    psObj->hTunerCarrierFreqOffset =
                            STRING_hCreate( (SMS_OBJECT)psObj, INVALID_5,
                            CARRIER_FREQ_OFFSET_SIZE,
                            CARRIER_FREQ_OFFSET_SIZE);
    psObj->hRSSI = STRING_hCreate( (SMS_OBJECT)psObj, INVALID_4,
                            RSSI_SIZE, RSSI_SIZE);
    return;
}


/*****************************************************************************
*
*   DETAILED_SIGNAL_QUALITY_hCreate
*
*****************************************************************************/
DETAILED_SIGNAL_QUALITY_OBJECT DETAILED_SIGNAL_QUALITY_hCreate(
    ENGINEERING_DATA_OBJECT hOwner
        )
{
    DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;
    BOOLEAN bOwner;

    // Verify ownership of parent object
    bOwner = SMSO_bOwner((SMS_OBJECT)hOwner);
    if (bOwner == FALSE)
    {
        return DETAILED_SIGNAL_QUALITY_INVALID_OBJECT;
    }

    // Create an instance of this object
    psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *)
         SMSO_hCreate(
            DETAILED_SIGNAL_QUALITY_OBJECT_NAME,
            sizeof(DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT),
            (SMS_OBJECT)hOwner, FALSE);

    if( psObj == NULL )
    {
        // Error!
        return DETAILED_SIGNAL_QUALITY_INVALID_OBJECT;
    }

    DETAILED_SIGNAL_QUALITY_vInitializeObject(
            (DETAILED_SIGNAL_QUALITY_OBJECT) psObj);

    if (  ( psObj->hSigStrength == STRING_INVALID_OBJECT )
       || ( psObj->hTunerStatus == STRING_INVALID_OBJECT )
       || ( psObj->hENSALockStatus == STRING_INVALID_OBJECT )
       || ( psObj->hENSBLockStatus == STRING_INVALID_OBJECT )
       || ( psObj->hBERS1 == STRING_INVALID_OBJECT )
       || ( psObj->hBERS2 == STRING_INVALID_OBJECT )
       || ( psObj->hBERT == STRING_INVALID_OBJECT )
       || ( psObj->hCNS1A == STRING_INVALID_OBJECT )
       || ( psObj->hCNS1B == STRING_INVALID_OBJECT )
       || ( psObj->hCNS2A == STRING_INVALID_OBJECT )
       || ( psObj->hCNS2B == STRING_INVALID_OBJECT )
       || ( psObj->hRSErrsWord == STRING_INVALID_OBJECT )
       || ( psObj->hRSErrsSatSymb == STRING_INVALID_OBJECT )
       || ( psObj->hRSErrsTerrSymb == STRING_INVALID_OBJECT )
       || ( psObj->hTunerCarrierFreqOffset == STRING_INVALID_OBJECT )
       || ( psObj->hRSSI == STRING_INVALID_OBJECT )
       )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                ": Unable to create a DETAILED_SIGNAL_QUALITY_OBJECT component");
        DETAILED_SIGNAL_QUALITY_vDestroy(
                (DETAILED_SIGNAL_QUALITY_OBJECT)psObj);
        return DETAILED_SIGNAL_QUALITY_INVALID_OBJECT;
    }

	psObj->bFirstUpdateReceived = FALSE;
	return (DETAILED_SIGNAL_QUALITY_OBJECT)psObj;
}


/*****************************************************************************
*
*   DETAILED_SIGNAL_QUALITY_vDestroy
*
*****************************************************************************/
void DETAILED_SIGNAL_QUALITY_vDestroy (
    DETAILED_SIGNAL_QUALITY_OBJECT hDetailedSignalQuality
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDetailedSignalQuality);
    if( bOwner == FALSE )
    {
        // Error!
        return;
    }

    vReleaseStrings(hDetailedSignalQuality);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hDetailedSignalQuality);
    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   vPercentStringModify
*
*****************************************************************************/
static void vPercentStringModify(
    STRING_OBJECT hStrTargetObj,
    UN16 un16PercentValue
        )
{
    char pStr[MAX_STRING_SIZE];

    //undocumented feature in SXi....
    if ( un16PercentValue == NO_ACTIVE_PIPE )
    {
        snprintf(pStr, PERCENT_SIZE, NO_ACTIVE_PIPE_STR);
        STRING.bModifyCStr(hStrTargetObj, pStr);
    }
    else
    {
        UN16 un16PercentInteger;
        UN16 un16PercentFraction;

        un16PercentInteger = un16PercentValue/DECIMAL_BASE;
        un16PercentFraction = un16PercentValue%DECIMAL_BASE;

        //Special formatting for 100%
        if ( un16PercentInteger == MAX_PERCENT )
        {
            snprintf(pStr, PERCENT_SIZE, "%u ", MAX_PERCENT);
            STRING.bModifyCStr(hStrTargetObj, pStr);
        }
               //Convert to xx.x format
        else if ( un16PercentInteger < MAX_PERCENT )
        {
            snprintf( pStr, PERCENT_SIZE, "%02u.%01u",
                      un16PercentInteger,
                      un16PercentFraction );
            STRING.bModifyCStr(hStrTargetObj, pStr);
        }
        else
        {
            snprintf(pStr, PERCENT_SIZE, INVALID_4);
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DETAILED_SIGNAL_QUALITY_OBJECT_NAME
                    ": Invalid percentage received.");
            STRING.bModifyCStr(hStrTargetObj, pStr);
        }
    }
}

/*****************************************************************************
*
*   eAccessObjectFromDecoder
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAccessObjectFromDecoder(
    DECODER_OBJECT hDecoder,
    DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT **ppsObj
        )
{
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;

    // Verify caller is already the owner of the object.
    // If not this function may not be called.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if( bOwner == TRUE )
    {
        do
        {
            *ppsObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *)
                       ENGINEERING_DATA.hSignalQualityDetailed(hDecoder);
            if( *ppsObj == NULL )
            {
                eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
                break;
            }

            // If we made it this far all is well
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        } while(0);
    } else {
        eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vReleaseStrings
*
*****************************************************************************/
static void vReleaseStrings(
    DETAILED_SIGNAL_QUALITY_OBJECT hDetailedSignalQuality
        )
{
    DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *psObj;

    psObj = (DETAILED_SIGNAL_QUALITY_OBJECT_STRUCT *)hDetailedSignalQuality;

    STRING_vDestroy(psObj->hBERS1);
    psObj->hBERS1 = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hBERS2);
    psObj->hBERS2 = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hBERT);
    psObj->hBERT = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hCNS1A);
    psObj->hCNS1A = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hCNS1B);
    psObj->hCNS1B = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hCNS2A);
    psObj->hCNS2A = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hCNS2B);
    psObj->hCNS2B = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hENSALockStatus);
    psObj->hENSALockStatus = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hENSBLockStatus);
    psObj->hENSBLockStatus = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hRSErrsSatSymb);
    psObj->hRSErrsSatSymb = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hRSErrsTerrSymb);
    psObj->hRSErrsTerrSymb = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hRSErrsWord);
    psObj->hRSErrsWord = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hRSSI);
    psObj->hRSSI = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hSigStrength);
    psObj->hSigStrength = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hTunerCarrierFreqOffset);
    psObj->hTunerCarrierFreqOffset = STRING_INVALID_OBJECT;

    STRING_vDestroy(psObj->hTunerStatus);
    psObj->hTunerStatus = STRING_INVALID_OBJECT;

    return;
}
