/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DECODER implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_version.h"

#include "sms.h"
#include "sms_obj.h"
#include "sms_task.h"

#include "sms_update.h"
#include "sms.h"
#include "srm_obj.h"
#include "module_obj.h"
#include "channel_obj.h"
#include "channellist_obj.h"
#include "ccache.h"
#include "category_obj.h"
#include "song_obj.h"
#include "songlist_obj.h"
#include "playback_obj.h"
#include "smart_favorites_obj.h"
#include "cme.h"
#include "cdo_obj.h"
#include "browse_obj.h"
#include "cm.h"
#include "seek.h"
#include "sport_zone.h"
#include "league_obj.h"
#include "report_obj.h"
#include "sub_notification.h"
#include "sports_flash_obj.h"
#include "tw_now_obj.h"

#include "engineering_data_obj.h"

#include "dataservice_mgr_obj.h"
#include "channel_art_mgr_obj.h"
#include "epg_mgr_obj.h"

#include "radio.h"
#include "srh.h"

#include "decoder_obj.h"
#include "_decoder_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   hGet
 *
 *****************************************************************************/
static DECODER_OBJECT hGet(
    SRM_OBJECT hSRM, 
    const char *pacDecoderName,
    DECODER_EVENT_MASK tEventRequestMask,
    DECODER_OBJECT_EVENT_CALLBACK vEventCallback, 
    void *pvEventCallbackArg
        )
{
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
    MODULE_OBJECT hModule;
    BOOLEAN bResult;
    MODULE_ID tId;
    DECODER_OBJECT_STRUCT *psObj;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "DECODER.%s(hSRM: %p, pacDecoderName: %s, tEventRequestMask: 0x%X, "
        "vEventCallback: %p, pvEventCallbackArg: %p)\n",
        __FUNCTION__, hSRM, pacDecoderName, tEventRequestMask,
        vEventCallback, pvEventCallbackArg);

    do
    {
        // Verify inputs
        bResult = SMSO_bValid((SMS_OBJECT)hSRM);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Invalid SRM object.");
            break;
        }

        if (NULL == pacDecoderName)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": pacDecoderName is not specified.");
            break;
        }

        // Is the library initialized?
        bResult = SMS_bIsInitialized();
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": SMS is not initialized.");
            break;
        }

        // Based on input DECODER name, find the unique MODULE Id
        // Acquire a unique module id for the named decoder provided
        tId = RADIO_tGetUniqueModuleId(hSRM, pacDecoderName);
        if (MODULE_ID_INVALID == tId)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": No such decoder exists '%s'.",
                pacDecoderName);
            break;
        }

        // Now we have the Module Id, but has a
        // corresponding MODULE object already been created?  Ask the SRM...
        hModule = SRM_hModule(hSRM, tId);
        if (MODULE_INVALID_OBJECT == hModule)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Cannot get MODULE.");
            break;
        }

        // Now, we have the MODULE and it is locked. We also have the DECODER
        // name which was provided by the caller, but has a corresponding
        // DECODER object already been created?  Ask the MODULE...
        hDecoder = MODULE_hDecoder(hModule, pacDecoderName);

        // Unlock MODULE. We don't need it anymore.
        SMSO_vUnlock((SMS_OBJECT)hModule);

        if (DECODER_INVALID_OBJECT != hDecoder)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": DECODER already exists.");
            hDecoder = DECODER_INVALID_OBJECT;
            break;
        }

        // Create a new Decoder object.
        psObj = psCreateObject (
            hSRM, pacDecoderName, tId, hModule,
            tEventRequestMask, vEventCallback, pvEventCallbackArg);
        if (NULL == psObj)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to create DECODER object.");
            break;
        }

        // Save owning MODULE handle
        psObj->hModule = hModule;

        // Set initial reference counter
        psObj->un16RefCounter = 1;

        // If at this point all goes well, then we can add ourself
        // to the Module's management of Decoders.
        bResult = MODULE_bAddDecoder(hModule, (DECODER_OBJECT)psObj);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": DECODER could not be added to MODULE.");

            // Destroy object
            vDestroyObject(psObj);
            break;
        }

        // Set approprite flag
        psObj->tCurrentModes |= SMS_MODE_PARENT_ADDED;

        // Success! Added to Module's management.
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: Added to MODULE.\n", psObj->pacObjectName);

        // Use this one, it's all good.
        hDecoder = (DECODER_OBJECT)psObj;

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: Created DECODER: %p.\n", psObj->pacObjectName, hDecoder);

    } while (FALSE);

    return hDecoder;
}

/*****************************************************************************
 *
 *   vRelease
 *
 *   Release the App's access to this DECODER
 *   This may cause object destruction
 *
 *****************************************************************************/
static void vRelease(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bResult;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "DECODER.%s(hDecoder: %p)\n", __FUNCTION__, hDecoder);

    // Verify SMS Object
    bResult = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (TRUE == bResult)
    {
        // Start application initiated release
        bInitiateObjectRelease((DECODER_OBJECT_STRUCT *)hDecoder,
            SMS_OBJECT_RELEASE_BY_APP);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Invalid DECODER object.");
    }

    return;
}

/*****************************************************************************
 *
 *   eState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eState(DECODER_OBJECT hDecoder)
{
    DECODER_STATE_ENUM eState;

    eState = DECODER_eState(hDecoder);

    return eState;
}

/*****************************************************************************
 *
 *   eErrorCode
 *
 *****************************************************************************/
static DECODER_ERROR_CODE_ENUM eErrorCode(DECODER_OBJECT hDecoder)
{
    DECODER_ERROR_CODE_ENUM eErrorCode = DECODER_ERROR_CODE_INVALID_OBJECT;

    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract object error code
        eErrorCode = psObj->eErrorCode;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to lock DECODER.");
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *   tEventMask
 *
 *****************************************************************************/
static DECODER_EVENT_MASK tEventMask(DECODER_OBJECT hDecoder)
{
    DECODER_EVENT_MASK tEventMask = DECODER_OBJECT_EVENT_NONE;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        tEventMask = SMSU_tMask(&psObj->sEvent);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to lock DECODER.");
    }

    return tEventMask;
}

/*****************************************************************************
 *
 *   eModifyRegisteredEventMask
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask(
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification)
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if ( eModification < SMSAPI_MODIFY_EVENT_MASK_START ||
         eModification >= SMSAPI_MODIFY_EVENT_MASK_INVALID )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Does anything we are about to do impact engineering data?
        if(tEventMask & DECODER_OBJECT_EVENT_ENGINEERING_DATA)
        {
            eReturnCode = MODULE_eModifyEngineeringData(
                psObj->hModule, hDecoder, eModification);
            if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // Error!
                return eReturnCode;
            }
        }

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                    SMS_EVENT_MODIFY_EVENT_MASK,
                    &puEventData,
                    SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uDecoder.sDecoderEventStruct.eModification =
                    eModification;
            puEventData->uDecoder.sDecoderEventStruct.tDecoderEventMask =
                    tEventMask;
            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                //Error!!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME
                    ": Unable to post modify decoder event mask.");
            }
            else
            {
                // We're done. All is well.
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   bUpdate
 *
 *****************************************************************************/
static BOOLEAN bUpdate(DECODER_OBJECT hDecoder, DECODER_EVENT_MASK tMask)
{
    BOOLEAN bPosted = FALSE, bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_UPDATE_OBJECT, &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sUpdate.tEventMask = tMask;
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   eTuneDirect
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTuneDirect(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN bOverride)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    SMS_EVENT_TUNE_STRUCT sTune;
    BOOLEAN bLocked, bValid;

    // Populate tune struct
    sTune.tChannelId = tChannelId;
    sTune.tServiceId = SERVICE_INVALID_ID;
    sTune.bLockedOverride = bOverride;
    sTune.bMatureOverride = bOverride;
    sTune.bSkippedOverride = bOverride;
    sTune.bPlayUnrestricted = FALSE;

    // Verify SMS Object is locked
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        eReturnCode = eQualifyTuneRequest(psObj, &sTune);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if((bValid == TRUE) && (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS))
    {
        BOOLEAN bPosted;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        bPosted = bPostTune(psObj, &sTune);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post tune event");
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eTuneState
 *
 *****************************************************************************/
static TUNE_STATE_ENUM eTuneState(DECODER_OBJECT hDecoder)
{
    TUNE_STATE_ENUM eTuneState = TUNE_STATE_UNKNOWN;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Return current tune state
        eTuneState = psObj->eTuneState;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eTuneState;
}

/*****************************************************************************
 *
 *   eSignalQuality
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSignalQuality(
    DECODER_OBJECT hDecoder, SIGNAL_QUALITY_STRUCT *psSignalQuality)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // If they did not provide a structure to fill, this means
    // they want to poll from the receiver.
    if(psSignalQuality == NULL)
    {
        BOOLEAN bPosted = FALSE, bValid;

        // Verify SMS Object is valid
        bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
        if(bValid == TRUE)
        {
            // De-reference object
            DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
            bPosted = SMSE_bPostSignal(psObj->hEventHdlr,
                SMS_EVENT_UPDATE_SIGNAL, SMS_EVENT_OPTION_NONE);
            if(bPosted == TRUE)
            {
                // Success
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        // Lock object for exclusive access
        BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // De-reference object
            DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

            // They want the current value, populate response from cache
            *psSignalQuality = psObj->sSignalQuality;

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hDecoder);

            // Success
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   n8NumAntennas
 *
 *****************************************************************************/
static N8 n8NumAntennas(DECODER_OBJECT hDecoder)
{
    N8 n8NumAntennas = -1;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // Retrieve number of antennas
        n8NumAntennas = RADIO_n8GetNumAntennas(hDecoder);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return n8NumAntennas;
}

/*****************************************************************************
 *
 *   eAntennaState
 *
 *****************************************************************************/
static ANTENNA_STATE_ENUM eAntennaState(DECODER_OBJECT hDecoder, N8 n8Antenna)
{
    ANTENNA_STATE_ENUM eAntennaState = ANTENNA_STATE_UNKNOWN;
    BOOLEAN bLocked;

    // index 1 based, so must be > 0
    if (n8Antenna > 0)
    {
        // Lock object for exclusive access
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if(bLocked == TRUE)
        {
            // Retrieve current antenna state (convert from 1 based index to 0 based)
            eAntennaState = RADIO_eGetAntennaState(hDecoder, (N8)(n8Antenna - 1));

            // Unlock object
            SMSO_vUnlock((SMS_OBJECT)hDecoder);
        }
    }

    return eAntennaState;
}

/*****************************************************************************
 *
 *   eUpdateProgress
 *
 *   THIS FUNCTON MAY ONLY BE CALLED IN THE CONTEXT OF THE DECODER CALLBACK
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUpdateProgress(
    DECODER_OBJECT hDecoder, UN8 *pun8UpdateProgress)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOwner;

    // Verify the inputs
    if(pun8UpdateProgress == NULL)
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Verify and check ownership of object
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // return the real progress
        *pun8UpdateProgress = psObj->un8TotalUpdateProgress;

        // Success
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   tGetChannelId
 *
 *****************************************************************************/
static CHANNEL_ID tGetChannelId(DECODER_OBJECT hDecoder, SERVICE_ID tServiceId)
{
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // fetch the service ID from the CCACHE
        tChannelId = CCACHE_tChannelIdFromServiceId(psObj->hCCache, tServiceId);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tChannelId;
}

/*****************************************************************************
 *
 *   tCurrentChannelId
 *
 *****************************************************************************/
static CHANNEL_ID tCurrentChannelId(DECODER_OBJECT hDecoder)
{
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract current channelId
        tChannelId = CHANNEL.tChannelId(psObj->hTunedChannel);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tChannelId;
}

/*****************************************************************************
 *
 *   tCurrentServiceId
 *
 *****************************************************************************/
static SERVICE_ID tCurrentServiceId(DECODER_OBJECT hDecoder)
{
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;
    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract current serviceId
        tServiceId = CHANNEL.tServiceId(psObj->hTunedChannel);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tServiceId;
}

/*****************************************************************************
 *
 *   tCurrentCategoryId
 *
 *****************************************************************************/
static CATEGORY_ID tCurrentCategoryId(DECODER_OBJECT hDecoder)
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract current categoryId
        tCategoryId = CATEGORY.tGetCategoryId(psObj->hTunedCategory);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *   tLastTunedChannelId
 *
 *****************************************************************************/
static CHANNEL_ID tLastTunedChannelId(DECODER_OBJECT hDecoder)
{
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // extract the last tuned channel id
        tChannelId = CHANNEL.tChannelId(psObj->hLastTunedChannel);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tChannelId;
}

/*****************************************************************************
 *
 *   tLastTunedCategoryId
 *
 *****************************************************************************/
static CATEGORY_ID tLastTunedCategoryId(DECODER_OBJECT hDecoder)
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // extract the last tuned category id
        tCategoryId = CATEGORY.tGetCategoryId(psObj->hLastTunedCategory);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *   tBrowsedChannelId
 *
 *****************************************************************************/
static CHANNEL_ID tBrowsedChannelId(DECODER_OBJECT hDecoder)
{
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // extract the browsed channel id
        tChannelId = CHANNEL.tChannelId(psObj->hBrowsedChannel);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tChannelId;
}

/*****************************************************************************
 *
 *   tBrowsedServiceId
 *
 *****************************************************************************/
static SERVICE_ID tBrowsedServiceId(DECODER_OBJECT hDecoder)
{
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // extract the browsed service id
        tServiceId = CHANNEL.tServiceId(psObj->hBrowsedChannel);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tServiceId;
}

/*****************************************************************************
 *
 *   tBrowsedCategoryId
 *
 *****************************************************************************/
static CATEGORY_ID tBrowsedCategoryId(DECODER_OBJECT hDecoder)
{
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // extract the browsed category id
        tCategoryId = CATEGORY.tGetCategoryId(psObj->hBrowsedCategory);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return tCategoryId;
}

/*****************************************************************************
 *
 *       eBrowseChannelStyleSet
 *
 *       This API is used set the current channel browsing style for this
 *       decoder. If a handler is not provided by the application, the default
 *       functionality is to include all channels in the browse.  If a handler is
 *       provided, that will be used while browsing channels to determine if that
 *       channel is to be provided or not.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the caller
 *               wishes to set the current channel browse style.
 *           bBrowseChannelCompareHandler - The optional compare handler that
 *               will be used during the browsing process to determine if a
 *               channel should or should not be included.
 *           pvBrowseChannelCompareArg - The pointer to an optional argument
 *               provided to the compare handler to help make the determination
 *               of whether or not the channel is to be included in the browse.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success or SMSAPI_RETURN_CODE_ERROR
 *               on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseChannelStyleSet(
    DECODER_OBJECT hDecoder,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        BOOLEAN bOk;

        // Set the callback and handler into the object
        bOk = BROWSE_bSetChannelBrowseStyle(psObj->hBrowse,
            bBrowseChannelCompareHandler, pvBrowseChannelCompareArg);

        if(bOk == TRUE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *   eBrowseChannel
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseChannel(
    DECODER_OBJECT hDecoder, SMSAPI_DIRECTION_ENUM eDirection)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMS_EVENT_BROWSE_STRUCT sBrowse;

    // Validate eDirection range
    if (eDirection < SMSAPI_DIRECTION_PREVIOUS
        || eDirection > SMSAPI_DIRECTION_NEXT)
    {
        return eReturnCode;
    }

    // Configure browse
    sBrowse.eBrowseType = BROWSE_TYPE_ALL_CHANNELS;
    sBrowse.eDirection = eDirection;
    sBrowse.uId.tChannel = CHANNEL_INVALID_ID;

    // Browse per configuration
    eReturnCode = eBrowseLocal(hDecoder, &sBrowse);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eBrowseToChannel
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseToChannel(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    SMS_EVENT_BROWSE_STRUCT sBrowse;

    // Configure browse
    sBrowse.eBrowseType = BROWSE_TYPE_ALL_CHANNELS;
    sBrowse.eDirection = SMSAPI_DIRECTION_DIRECT;
    sBrowse.uId.tChannel = tChannelId;

    // Browse per configuration
    eReturnCode = eBrowseLocal(hDecoder, &sBrowse);

    return eReturnCode;
}

/*****************************************************************************
 *
 *       eBrowseCategoryStyleSet
 *
 *       This API is used set the current category browsing style for this
 *       decoder. The default value is to include every Category, both user and
 *       broadcast, in the browse.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the caller
 *               wishes to set the current category browse style.
 *           bBrowseCategoryCompareHandler - The optional compare handler that
 *               will be used during the browsing process to determine if a
 *               category should or should not be included.
 *           pvBrowseCategoryCompareArg - The pointer to an optional argument
 *               provided to the compare handler to help make the determination
 *               of whether or not the category is to be included in the browse.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success or SMSAPI_RETURN_CODE_ERROR
 *               on error.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseCategoryStyleSet(
    DECODER_OBJECT hDecoder,
    BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
    void *pvBrowseCategoryCompareArg)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        BOOLEAN bOk;

        // Set the callback and handler into the object
        bOk = BROWSE_bSetCategoryBrowseStyle(psObj->hBrowse,
            bBrowseCategoryCompareHandler, pvBrowseCategoryCompareArg);

        if(bOk == TRUE)
        {
            eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eErrorCode = SMSAPI_RETURN_CODE_ERROR;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *   eBrowseCategory
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseCategory(
    DECODER_OBJECT hDecoder, SMSAPI_DIRECTION_ENUM eDirection)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMS_EVENT_BROWSE_STRUCT sBrowse;

    // Validate eDirection range
    if (eDirection < SMSAPI_DIRECTION_PREVIOUS
        || eDirection > SMSAPI_DIRECTION_NEXT)
    {
        return eReturnCode;
    }

    // Configure browse
    sBrowse.eBrowseType = BROWSE_TYPE_CATEGORY;
    sBrowse.eDirection = eDirection;
    sBrowse.uId.tChannel = CHANNEL_INVALID_ID;

    // Browse per configuration
    eReturnCode = eBrowseLocal(hDecoder, &sBrowse);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eBrowseToCategory
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseToCategory(
    DECODER_OBJECT hDecoder, CATEGORY_ID tCategoryId)
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    SMS_EVENT_BROWSE_STRUCT sBrowse;

    // Configure browse
    sBrowse.eBrowseType = BROWSE_TYPE_CATEGORY;
    sBrowse.eDirection = SMSAPI_DIRECTION_DIRECT;
    sBrowse.uId.tCategory = tCategoryId;

    // Browse per configuration
    eReturnCode = eBrowseLocal(hDecoder, &sBrowse);

    return eReturnCode;
}

/*****************************************************************************
 *
 *       eUseChannel
 *
 *       This API is used to perform some type of action on a CHANNEL object.
 *       This API allows a caller to provide a function and argument which
 *       allows the direct manipulation of a CHANNEL object by first providing
 *       a channel id. Manipulation of the CHANNEL object is performed while
 *       safely observing exclusive access rules.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel which is to be
 *               manipulated.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on
 *               error, or SMSAPI_RETURN_CODE_NOT_FOUND if the channel id cannot
 *               be found/invalid.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUseChannel(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId,
    DECODER_CHANNEL_ACCESS_CALLBACK vChannelAccessCallback,
    void *pvChannelAccessCallbackArg)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Check if the input channel id is CHANNEL_INVALID_ID.
        if(tChannelId == CHANNEL_INVALID_ID)
        {
            // Use the currently browsed channel and skip the search
            hChannel = psObj->hBrowsedChannel;
        }
        else
        {
            // Search for the specified channel Id in the cache
            hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        }

        // If the channel is invalid
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // Use caller's function to manipulate this channel if one exists
            if(vChannelAccessCallback != NULL)
            {
                // Call function.
                vChannelAccessCallback(hDecoder, hChannel,
                    pvChannelAccessCallbackArg);
                eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                // Input error. No callback
                eErrorCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       eUseCategory
 *
 *       This API is used to perform some type of action on a CATEGORY object.
 *       This API allows a caller to provide a function and argument which
 *       allows the direct manipulation of a CATEGORY object by first providing
 *       a category id. Manipulation of the CATEGORY object is performed while
 *       safely observing exclusive access rules.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified category belongs to.
 *           tCategoryId - Category identifier for the CATEGORY which is to be
 *               manipulated.
 *           vCategoryAccessCallback - A user-provided callback where the user can
 *              do things to the category
 *           pvCategoryAccessCallbackArg - A user-provide argument that is used
 *              like all the rest of our callback args are used
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on
 *               error, or SMSAPI_RETURN_CODE_NOT_FOUND if the CATEGORY id cannot
 *               be found/invalid.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eUseCategory(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    DECODER_CATEGORY_ACCESS_CALLBACK vCategoryAccessCallback,
    void *pvCategoryAccessCallbackArg)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CATEGORY_OBJECT hCategory;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Check if the input category id is CATEGORY_INVALID_ID.
        if(tCategoryId == CATEGORY_INVALID_ID)
        {
            // Use the currently browsed category and skip the search
            tCategoryId = CATEGORY.tGetCategoryId(psObj->hBrowsedCategory);
            hCategory = psObj->hBrowsedCategory;
        }
        else
        {
            // Search for the specified channel Id in the cache
            hCategory = CCACHE_hCategory(psObj->hCCache, &tCategoryId, 0);
        }

        // If the category is invalid
        if(hCategory == CATEGORY_INVALID_OBJECT)
        {
            // We failed to find the category
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The category is valid
        else
        {
            // Use caller's function to manipulate this category if one exists
            if(vCategoryAccessCallback != NULL)
            {
                // Call function.
                vCategoryAccessCallback(hDecoder, hCategory,
                    pvCategoryAccessCallbackArg);
                eErrorCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                // Input error. No callback
                eErrorCode = SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       bLockChannel
 *
 *       This API is used to set the channel attribute for the specified channel
 *       to LOCKED.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel to lock
 *
 *       Outputs:
 *           TRUE upon successful request.
 *
 *****************************************************************************/
static BOOLEAN bLockChannel(DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId)
{
    BOOLEAN bRetVal = FALSE, bPost = FALSE;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        return FALSE;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            SERVICE_ID tServiceId;

            tServiceId = CHANNEL.tServiceId(hChannel);

            // we need to make sure the channel we're going to lock isn't the
            // safe or default channel
            if ( (tServiceId != GsRadio.sChannel.tDefaultServiceID) &&
                 (tServiceId != GsRadio.sChannel.tSafeServiceID) )
            {
                CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus;

                eStatus = CHANNEL_eSetUnconfirmedAttribute(
                    hChannel, CHANNEL_OBJECT_ATTRIBUTE_LOCKED);

                if(eStatus == CHANNEL_ATTRIBUTE_STATUS_CHANGED)
                {
                    TAG_OBJECT hTag;
                    SMSAPI_RETURN_CODE_ENUM eReturn;

                    eReturn = TAG_eGet(LOCKED_TAG_NAME, psObj->hTag, &hTag,
                        NULL, FALSE);
                    if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        bRetVal = bAddServiceIdTag(hTag, hChannel);
                    }
                    else
                    {
                        bRetVal = FALSE;
                    }
                }
                else if(eStatus == CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE)
                {
                    bRetVal = TRUE;
                }
                else
                {
                    bRetVal = FALSE;
                }
            }
            else
            {
                // we can't lock this channel because it is either the default
                // or safe channel
                bRetVal = FALSE;
            }

            if ((bRetVal == TRUE) && 
                ((psObj->tPendingAttrs & CHANNEL_OBJECT_ATTRIBUTE_LOCKED) != 
                  CHANNEL_OBJECT_ATTRIBUTE_LOCKED))
            {
                // Set pending attribute flag
                psObj->tPendingAttrs |= CHANNEL_OBJECT_ATTRIBUTE_LOCKED;
                bPost = TRUE;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if (bPost == TRUE)
        {
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Sending event to update channel attributes in radio module
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr, SMS_EVENT_CHANNEL, &puEventData, SMS_EVENT_OPTION_NONE);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                // LOCKED attribute is to be updated
                puEventData->uDecoder.sChannel.bLocked = TRUE;

                bRetVal = SMSE_bPostEvent(hEvent);
            }
        }
    }

    return bRetVal;
}

/*****************************************************************************
 *
 *       bUnlockChannel
 *
 *       This API is used to set the channel attribute for the specified channel
 *       to UNLOCKED.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel to unlock
 *
 *       Outputs:
 *           TRUE upon successful request.
 *
 *****************************************************************************/
static BOOLEAN bUnlockChannel(DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId)
{
    BOOLEAN bRetVal = FALSE, bPost = FALSE;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        return FALSE;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus;

            eStatus = CHANNEL_eClearUnconfirmedAttribute(
                hChannel, CHANNEL_OBJECT_ATTRIBUTE_LOCKED);

            if(eStatus == CHANNEL_ATTRIBUTE_STATUS_CHANGED)
            {
                TAG_OBJECT hTag;
                SMSAPI_RETURN_CODE_ENUM eReturn;
                CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT sIteratorStruct;

                sIteratorStruct.psDecoderObj = NULL;
                sIteratorStruct.tServiceId = CHANNEL.tServiceId(hChannel);
                sIteratorStruct.bRemove = TRUE; // remove the tag
                sIteratorStruct.bLock = FALSE;
                sIteratorStruct.bSkip = FALSE;

                eReturn = TAG_eGet(LOCKED_TAG_NAME, psObj->hTag, &hTag, NULL,
                    FALSE);
                if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    eReturn = TAG_eIterateChildren(hTag,
                        bChannelAttributeTagIterator, &sIteratorStruct);
                    if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        eReturn = CM_eCommitChangesToFile();
                    }
                }

                bRetVal =
                    (eReturn == SMSAPI_RETURN_CODE_SUCCESS ? TRUE : FALSE);
            }
            else if(eStatus == CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE)
            {
                bRetVal = TRUE;
            }
            else
            {
                bRetVal = FALSE;
            }

            if ((bRetVal == TRUE) &&
                ((psObj->tPendingAttrs & CHANNEL_OBJECT_ATTRIBUTE_LOCKED) != 
                  CHANNEL_OBJECT_ATTRIBUTE_LOCKED))
            {
                psObj->tPendingAttrs |= CHANNEL_OBJECT_ATTRIBUTE_LOCKED;
                bPost = TRUE;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if (bPost == TRUE)
        {
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Sending event to update channel attributes in radio module
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr, SMS_EVENT_CHANNEL, &puEventData, SMS_EVENT_OPTION_NONE);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                // LOCKED attribute is to be updated
                puEventData->uDecoder.sChannel.bLocked = TRUE;

                bRetVal = SMSE_bPostEvent(hEvent);
            }
        }
    }

    return bRetVal;
}

/*****************************************************************************
 *
 *       bSkipChannel
 *
 *       This API is used to set the channel attribute for the specified channel
 *       to SKIPPED.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel to skip
 *
 *       Outputs:
 *           TRUE upon successful request.
 *
 *****************************************************************************/
static BOOLEAN bSkipChannel(DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId)
{
    BOOLEAN bRetVal = FALSE, bPost = FALSE;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        return FALSE;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus;

            eStatus = CHANNEL_eSetUnconfirmedAttribute(
                hChannel, CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);

            if(eStatus == CHANNEL_ATTRIBUTE_STATUS_CHANGED)
            {
                TAG_OBJECT hTag;
                SMSAPI_RETURN_CODE_ENUM eReturn;

                eReturn = TAG_eGet(SKIPPED_TAG_NAME, psObj->hTag, &hTag, NULL,
                    FALSE);
                if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    bRetVal = bAddServiceIdTag(hTag, hChannel);
                }
                else
                {
                    bRetVal = FALSE;
                }
            }
            else if(eStatus == CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE)
            {
                bRetVal = TRUE;
            }
            else
            {
                bRetVal = FALSE;
            }

            if ((bRetVal == TRUE) &&
                ((psObj->tPendingAttrs & CHANNEL_OBJECT_ATTRIBUTE_SKIPPED) != 
                  CHANNEL_OBJECT_ATTRIBUTE_SKIPPED))
            {
                psObj->tPendingAttrs |= CHANNEL_OBJECT_ATTRIBUTE_SKIPPED;
                bPost = TRUE;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if (bPost == TRUE)
        {
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Sending event to update channel attributes in radio module
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr, SMS_EVENT_CHANNEL, &puEventData, SMS_EVENT_OPTION_NONE);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                // SKIPPED attribute is to be updated
                puEventData->uDecoder.sChannel.bSkipped = TRUE;

                bRetVal = SMSE_bPostEvent(hEvent);
            }
        }
    }

    return bRetVal;
}

/*****************************************************************************
 *
 *       bUnskipChannel
 *
 *       This API is used to set the channel attribute for the specified channel
 *       to UNSKIPPED.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel to unskip
 *
 *       Outputs:
 *           TRUE upon successful request.
 *
 *****************************************************************************/
static BOOLEAN bUnskipChannel(DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId)
{
    BOOLEAN bRetVal = FALSE, bPost = FALSE;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        return FALSE;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            CHANNEL_ATTRIBUTE_STATUS_ENUM eStatus;

            // unskip
            eStatus = CHANNEL_eClearUnconfirmedAttribute(
                hChannel, CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);

            if(eStatus == CHANNEL_ATTRIBUTE_STATUS_CHANGED)
            {
                TAG_OBJECT hTag;
                SMSAPI_RETURN_CODE_ENUM eReturn;
                CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT sIteratorStruct;

                sIteratorStruct.psDecoderObj = NULL;
                sIteratorStruct.tServiceId = CHANNEL.tServiceId(hChannel);
                sIteratorStruct.bRemove = TRUE; // remove the tag
                sIteratorStruct.bLock = FALSE;
                sIteratorStruct.bSkip = FALSE;

                // get the skipped tag
                eReturn = TAG_eGet(SKIPPED_TAG_NAME, psObj->hTag, &hTag, NULL,
                    FALSE);
                if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    // iterate,
                    eReturn = TAG_eIterateChildren(hTag,
                        bChannelAttributeTagIterator, &sIteratorStruct);
                    if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        eReturn = CM_eCommitChangesToFile();
                    }
                }
                bRetVal = (eReturn == SMSAPI_RETURN_CODE_SUCCESS ? TRUE
                                    : FALSE);
            }
            else if(eStatus == CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE)
            {
                bRetVal = TRUE;
            }
            else
            {
                bRetVal = FALSE;
            }

            if ((bRetVal == TRUE) &&
                ((psObj->tPendingAttrs & CHANNEL_OBJECT_ATTRIBUTE_SKIPPED) != 
                  CHANNEL_OBJECT_ATTRIBUTE_SKIPPED))
            {
                psObj->tPendingAttrs |= CHANNEL_OBJECT_ATTRIBUTE_SKIPPED;
                bPost = TRUE;
            }
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        if (bPost == TRUE)
        {
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Sending event to update channel attributes in radio module
            hEvent = SMSE_hAllocateEvent(
                psObj->hEventHdlr, SMS_EVENT_CHANNEL, &puEventData, SMS_EVENT_OPTION_NONE);
            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                // SKIPPED attribute is to be updated
                puEventData->uDecoder.sChannel.bSkipped = TRUE;

                bRetVal = SMSE_bPostEvent(hEvent);
            }
        }
    }

    return bRetVal;
}

/*****************************************************************************
 *
 *       eIsChannelLocked
 *
 *       This API is used to check if a specific channel is locked.  This call
 *       may be made asynchronously from the application without regard to
 *       whether or not it is made in the context of the Decoder's Event
 *       Callback.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel whose lock attribute
 *               is being checked
 *           pbLocked - Pointer to a BOOLEAN.  This function will set the value
 *               to TRUE if the channel is locked or FALSE if the channel is
 *               unlocked.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
 *           of type SMSAPI_RETURN_CODE_ENUM
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsChannelLocked(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN *pbLocked)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if((tChannelId == CHANNEL_INVALID_ID) || (pbLocked == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // See if the channel is locked
            eErrorCode = CHANNEL.eIsLocked(hChannel, pbLocked);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       eIsChannelSkipped
 *
 *       This API is used to check if a specific channel is skipped.  This call
 *       may be made asynchronously from the application without regard to
 *       whether or not it is made in the context of the Decoder's Event
 *       Callback.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel whose skip attribute
 *               is being checked
 *           pbLocked - Pointer to a BOOLEAN.  This function will set the value
 *               to TRUE if the channel is skipped or FALSE if the channel is
 *               unskipped.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
 *           of type SMSAPI_RETURN_CODE_ENUM
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsChannelSkipped(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN *pbSkipped)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if((tChannelId == CHANNEL_INVALID_ID) || (pbSkipped == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // See if the channel is skipped
            eErrorCode = CHANNEL.eIsSkipped(hChannel, pbSkipped);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       eIsChannelSubscribed
 *
 *       This API is used to check if a specific channel is subscribed. This call
 *       may be made asynchronously from the application without regard to
 *       whether or not it is made in the context of the Decoder's Event
 *       Callback.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel whose subscribed
 *               attribute is being checked
 *           pbSubscribed - Pointer to a BOOLEAN.  This function will set the
 *               value to TRUE if the channel is subscribed or FALSE if the
 *               channel is unsubscribed.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
 *           of type SMSAPI_RETURN_CODE_ENUM
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsChannelSubscribed(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN *pbSubscribed)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if((tChannelId == CHANNEL_INVALID_ID) || (pbSubscribed == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // See if the channel is subscribed
            eErrorCode = CHANNEL.eIsSubscribed(hChannel, pbSubscribed);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       eIsChannelMature
 *
 *       This API is used to check if a specific channel is mature. This call
 *       may be made asynchronously from the application without regard to
 *       whether or not it is made in the context of the Decoder's Event
 *       Callback.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel whose mature
 *               attribute is being checked
 *           pbMature - Pointer to a BOOLEAN.  This function will set the
 *               value to TRUE if the channel is mature or FALSE if the
 *               channel is not.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
 *           of type SMSAPI_RETURN_CODE_ENUM
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsChannelMature(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN *pbMature)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if((tChannelId == CHANNEL_INVALID_ID) || (pbMature == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // See if the channel is mature
            eErrorCode = CHANNEL.eIsMature(hChannel, pbMature);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *       eIsChannelFreeToAir
 *
 *       This API is used to check if a specific channel is free-to-air
 *       (always available regardless of the current subscription state).
 *       This call may be made asynchronously from the application without
 *       regard to whether or not it is made in the context of the Decoder's
 *       Event Callback.
 *
 *       Inputs:
 *           hDecoder - A handle to a valid DECODER object for which the
 *               specified channel belongs to.
 *           tChannelId - Channel identifier for the channel whose mature
 *               attribute is being checked
 *           pbFreeToAir - Pointer to a BOOLEAN.  This function will set the
 *               value to TRUE if the channel is free-to-air or FALSE if the
 *               channel is not.
 *
 *       Outputs:
 *           SMSAPI_RETURN_CODE_SUCCESS on success; otherwise an error code
 *           of type SMSAPI_RETURN_CODE_ENUM
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsChannelFreeToAir(
    DECODER_OBJECT hDecoder, CHANNEL_ID tChannelId, BOOLEAN *pbFreeToAir)
{
    SMSAPI_RETURN_CODE_ENUM eErrorCode = SMSAPI_RETURN_CODE_ERROR;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bLocked;

    // Verify the inputs
    if((tChannelId == CHANNEL_INVALID_ID) || (pbFreeToAir == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Search for the specified channel Id in the cache
        hChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, SERVICE_INVALID_ID, tChannelId, FALSE);
        if(hChannel == CHANNEL_INVALID_OBJECT)
        {
            // We failed to find the channel
            eErrorCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
        // The channel is valid
        else
        {
            // See if the channel is free-to-air
            eErrorCode = CHANNEL.eIsFreeToAir(hChannel, pbFreeToAir);
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eErrorCode;
}

/*****************************************************************************
 *
 *   hPlayback
 *
 *****************************************************************************/
static PLAYBACK_OBJECT hPlayback(DECODER_OBJECT hDecoder)
{
    PLAYBACK_OBJECT hPlayback = PLAYBACK_INVALID_OBJECT;

    // Lock object for exclusive access
    BOOLEAN bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract current playback handle
        hPlayback = psObj->hPlayback;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hPlayback;
}

/*****************************************************************************
 *
 *   eSetUnsubscribedText
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSetUnsubscribedText(
    DECODER_OBJECT hDecoder, const char *pacUnsubscribedText,
    CDO_FIELD_MASK tMask)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    BOOLEAN bLocked = FALSE;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);

    // Make sure the CDO_FIELD_MASK only contains fields we understand
    // and passed DECODER handle is valid.
    if(((tMask & ((CDO_FIELD_MASK)~CDO_FIELD_ALL)) != CDO_FIELD_NONE) ||
      (bValid == FALSE))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CHANNEL_OBJECT hUnSubscribedChannel;

        // Extract the CCACHE unsubscribed channel
        hUnSubscribedChannel = CCACHE_hUnsubscribedChannel(psObj->hCCache);
        if(hUnSubscribedChannel != CHANNEL_INVALID_OBJECT)
        {
            // Set Unsubscribed Channel's Text
            CDO_vUpdate(hUnSubscribedChannel, pacUnsubscribedText, tMask);

            // store text across power cycles
            bStoreUnsubscribedText(psObj, pacUnsubscribedText, tMask);

            // Unsubscribed text changed, we need to iterate the ccache,
            // looking for unsubscribed channels and set events for all
            // unsubscribed channels
            CCACHE_bIterateChannels (
                psObj->hCCache, FALSE,
                bCCacheUnsubscribedChansIterator,
                NULL
                    );
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
 *
 *   eStartToneGeneration
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStartToneGeneration (
        DECODER_OBJECT hDecoder,
        UN32 un32ToneFreqHz,
        N8 n8Volume,
        DECODER_TONE_GENERATION_BALANCE_ENUM eBalance )
{
    BOOLEAN bValid, bPosted = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_TONE_GENERATION;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sToneGeneration.eAction = DECODER_TONE_GEN_START;
            puEventData->uDecoder.sToneGeneration.un32ToneFreqHz = un32ToneFreqHz;
            puEventData->uDecoder.sToneGeneration.n8Volume = n8Volume;
            puEventData->uDecoder.sToneGeneration.eBalance = eBalance;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   eStopToneGeneration
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStopToneGeneration (
        DECODER_OBJECT hDecoder )
{
    BOOLEAN bValid, bPosted = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_TONE_GENERATION;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sToneGeneration.eAction = DECODER_TONE_GEN_STOP;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   eAlertToneGeneration
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAlertToneGeneration(
             DECODER_OBJECT hDecoder,
              N8 n8Volume,
              int iCmd,
              ...)
{
    BOOLEAN bValid, bPosted = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_TONE_GENERATION;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sToneGeneration.eAction = DECODER_TONE_GEN_ALERT;
            puEventData->uDecoder.sToneGeneration.n8Volume = n8Volume;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   eAudio
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAudio(
             DECODER_OBJECT hDecoder,
             N16 n16Level)
{
    BOOLEAN bValid, bPosted = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_AUDIO_CONTROL;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sAudioControl.eAction = DECODER_AUDIO_CONTROL_VOLUME;
            puEventData->uDecoder.sAudioControl.n16Level = n16Level;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        eReturn = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturn;
}

/*****************************************************************************
 *
 *   eTuneScanConfigure
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTuneScanConfigure (
    DECODER_OBJECT hDecoder,
    UN8 un8PlaySeconds,
    BOOLEAN bScanLockedMature,
    BOOLEAN bScanSkipped,
    DECODER_TUNE_SCAN_STYLE_ENUM eScanStyle
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // For now we don't support this
    if(eScanStyle == DECODER_TUNE_SCAN_STYLE_ALL_CHANNELS)
    {
        return SMSAPI_RETURN_CODE_UNSUPPORTED_API;
    }

    // Verify and lock object
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (bValid == TRUE)
    {
        BOOLEAN bSupported;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_TUNE_SCAN;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Verify Advanced IR is supported
        bSupported = MODULE_bIsAdvancedIrSupported(psObj->hModule);
        if (bSupported == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Advanced IR is not supported by the Module.\n");
            return SMSAPI_RETURN_CODE_UNSUPPORTED_API;
        }

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_CFG;

            puEventData->uDecoder.sTuneScan.bScanLockedMature = bScanLockedMature;
            puEventData->uDecoder.sTuneScan.bScanSkipped = bScanSkipped;
            puEventData->uDecoder.sTuneScan.un8PlaySeconds = un8PlaySeconds;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == TRUE)
            {
                BOOLEAN bLocked;

                // Lock decoder to synchronize Tune Scan config
                bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
                if (bLocked == TRUE)
                {
                    // Scan was configured
                    psObj->sTuneScan.bConfigured = TRUE;
                    psObj->sTuneScan.eScanStyle = eScanStyle;

                    // Relinquish ownership
                    SMSO_vUnlock((SMS_OBJECT)hDecoder);
                }

                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    else
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eTuneScan
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTuneScan (
    DECODER_OBJECT hDecoder,
    DECODER_TUNE_SCAN_CMD_ENUM eScanCmd
        )
{
    BOOLEAN bValid;
    BOOLEAN bLocked = FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    do
    {
        BOOLEAN bPosted;
        BOOLEAN bSupported;
        SMS_EVENT_HDL hEvent;
        DECODER_OBJECT_STRUCT *psObj;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_TUNE_SCAN;

        // Verify input command
        if (eScanCmd > DECODER_TUNE_SCAN_CMD_SKIP_BACKWARD)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Requested command is not supported yet or invalid.\n");

            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // Verify object
        bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
        if (bValid == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Invalid DECODER object.");

            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        // De-reference object
        psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Verify Advanced IR is supported
        bSupported = MODULE_bIsAdvancedIrSupported(psObj->hModule);
        if (bSupported == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Advanced IR is not supported by the Module.\n",
                psObj->pacObjectName);

            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        // Lock decoder
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            // SMSO_bLock prints error anyway
            break;
        }

        // For this type now this method is not supported
        if (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_ALL_CHANNELS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Tune Scan Style All Channel is not Supported right Now.\n",
                psObj->pacObjectName);

            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        // Application should configure Tune Scan first
        if (psObj->sTuneScan.bConfigured == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Tune Scan is not configured.\n",
                psObj->pacObjectName);

            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);

        // Resource unlocked
        bLocked = FALSE;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData, SMS_EVENT_OPTION_NONE);
        if (hEvent == SMS_INVALID_EVENT_HDL)
        {
            // SMSE_hAllocateEvent prints error anyway
            break;
        }

        // Extract event information and populate it
        if (eScanCmd == DECODER_TUNE_SCAN_CMD_ABORT)
        {
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_ABORT;
        }
        else if (eScanCmd == DECODER_TUNE_SCAN_CMD_SKIP_FORWARD)
        {
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_FORWARD;
        }
        else if (eScanCmd == DECODER_TUNE_SCAN_CMD_SKIP_BACKWARD)
        {
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_BACKWARD;
        }
        else if (eScanCmd == DECODER_TUNE_SCAN_CMD_START)
        {
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_START;
        }
        else if (eScanCmd == DECODER_TUNE_SCAN_CMD_STOP)
        {
            puEventData->uDecoder.sTuneScan.eEventType = TUNE_SCAN_EVENT_TYPE_STOP;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Unhandled command.\n", psObj->pacObjectName);
        }

        bPosted = SMSE_bPostEvent(hEvent);
        if (bPosted == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eIsTuneScanContentAvailable
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIsTuneScanContentAvailable (
    DECODER_OBJECT hDecoder,
    DECODER_TUNE_SCAN_STATUS_MASK *ptStatusMask
        )
{
    BOOLEAN bLocked = FALSE;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify input
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if ((ptStatusMask == NULL) || (bValid == FALSE))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    do
    {
        BOOLEAN bSupported;
        DECODER_OBJECT_STRUCT *psObj;

        // De-reference object
        psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Verify Advanced IR is supported
        bSupported = MODULE_bIsAdvancedIrSupported(psObj->hModule);
        if (bSupported == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Advanced IR is not supported by the Module.\n",
                psObj->pacObjectName);

            eReturnCode = SMSAPI_RETURN_CODE_UNSUPPORTED_API;
            break;
        }

        // Verify and lock object
        bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            break;
        }

        // Application should configure Tune Scan first
        if (psObj->sTuneScan.bConfigured == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Tune Scan is not configured.\n",
                psObj->pacObjectName);

            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        if (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES)
        {
            // Pass thru mask
            *ptStatusMask = psObj->sTuneScan.tStatusMask;
        }
        else if (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_ALL_CHANNELS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            break;
        }

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    if (bLocked == TRUE)
    {
        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eAudioPresence
 *
 *****************************************************************************/
static DECODER_AUDIO_PRESENCE_ENUM eAudioPresence (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bValid;
    DECODER_AUDIO_PRESENCE_ENUM eAudioPresence = 
        DECODER_AUDIO_PRESENCE_UNKNOWN;

    // Verify the object
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = 
            (DECODER_OBJECT_STRUCT*)hDecoder;

        eAudioPresence = psObj->eAudioPresence;
    }

    return eAudioPresence;
}

/*****************************************************************************
 FRIEND FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *   DECODER_DEBUG_vPrintCCache
 *
 *****************************************************************************/
void DECODER_DEBUG_vPrintCCache (
    DECODER_OBJECT hDecoder,
    BOOLEAN bChannels,
    BOOLEAN bCategories,
    BOOLEAN bVerbose
)
{
#if SMS_DEBUG == 1
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
        (DECODER_OBJECT_STRUCT *)hDecoder;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(
                        psObj->hEventHdlr, SMS_EVENT_CCACHE, &puEventData,
                        SMS_EVENT_OPTION_NONE
        );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Pass event off to the CCACHE function
            SMS_EVENT_CCACHE_STRUCT *psCCache =
            &puEventData->uDecoder.sCache;

            // Extract event information and populate it
            // Channel cache event
            psCCache->eType =
                CCACHE_EVENT_TYPE_PRINT;
            psCCache->uCache.sPrint.bChannels =
                bChannels;
            psCCache->uCache.sPrint.bCategories =
                bCategories;
            psCCache->uCache.sPrint.bVerbose =
                bVerbose;

            SMSE_bPostEvent(hEvent);
        }
    }
#endif
    return;
}

/*****************************************************************************
 *
 *   DECODER_DEBUG_vPrintSCache
 *
 *****************************************************************************/
void DECODER_DEBUG_vPrintSCache (
                DECODER_OBJECT hDecoder,
                BOOLEAN bVerbose
)
{
#if SMS_DEBUG == 1
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
        (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent( psObj->hEventHdlr,
                        SMS_EVENT_PRINT_SCACHE, &puEventData, SMS_EVENT_OPTION_NONE );
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sPrintSCache.hPlayback =
            psObj->hPlayback;
            puEventData->uDecoder.sPrintSCache.uSCache.
            sPrint.bVerbose = bVerbose;

            SMSE_bPostEvent(hEvent);
        }
    }
#endif
    return;
}

/*****************************************************************************
 *
 *   DECODER_vPrintTuneScan
 *
 *****************************************************************************/
void DECODER_DEBUG_vPrintTuneScan (
    DECODER_OBJECT hDecoder
        )
{
#if SMS_DEBUG == 1
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_TUNE_SCAN, &puEventData, SMS_EVENT_OPTION_NONE);

        if (hEvent != SMS_INVALID_EVENT_HDL)
        {
            puEventData->uDecoder.sTuneScan.eEventType =
                TUNE_SCAN_EVENT_TYPE_PRINT;

            SMSE_bPostEvent(hEvent);
        }
    }
#endif
    return;
}

/*****************************************************************************
 *
 *   DECODER_bRelease
 *
 *   Release SMS internal access to this DECODER
 *   This may cause object destruction
 *
 *****************************************************************************/
BOOLEAN DECODER_bRelease (
    DECODER_OBJECT hDecoder,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s(hDecoder: %p, eInitiator: %d)\n",
        __FUNCTION__, hDecoder, eInitiator);

    // Verify input
    if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Incorrect release request.");
        return FALSE;
    }

    // Verify SMS Object
    bResult = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (FALSE == bResult)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Invalid DECODER object.");
        return FALSE;
    }

    // Initiate object release
    bResult = bInitiateObjectRelease(psObj, eInitiator);
    if (FALSE == bResult)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   DECODER_eState
 *
 *****************************************************************************/
DECODER_STATE_ENUM DECODER_eState(
    DECODER_OBJECT hDecoder
        )
{
    DECODER_STATE_ENUM eState = DECODER_STATE_INVALID;
    BOOLEAN bLocked;

    // Verify and lock SMS Object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Pass thru state
        eState = psObj->eState;

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to lock DECODER.");
    }

    return eState;
}

/*****************************************************************************
 *
 *   DECODER_bAssociate
 *
 *****************************************************************************/
BOOLEAN DECODER_bAssociate(
    DECODER_OBJECT hDecoder,
    MODULE_OBJECT hModule,
    STI_HDL hSTI
        )
{
    BOOLEAN bResult = FALSE;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s(hDecoder: %p, hModule: %p, hSTI: %p)\n",
        __FUNCTION__, hDecoder, hModule, hSTI);

    do
    {
        // Verify SMS Object is valid
        bResult = SMSO_bValid((SMS_OBJECT)hDecoder);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Invalid DECODER object.");
            break;
        }

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_ASSOCIATE,
            &puEventData, SMS_EVENT_OPTION_URGENT);
        if (SMS_INVALID_EVENT_HDL == hEvent)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to allocate event.");
            bResult = FALSE;
            break;
        }

        // Populate event information
        puEventData->uDecoder.sAssociate.hModule = hModule;
        puEventData->uDecoder.sAssociate.eSubStatus =
            RADIO_eSubStatus(hModule, DECODER_INVALID_OBJECT);
        puEventData->uDecoder.sAssociate.tSubReasonCode =
            RADIO_tReasonCode(hModule, DECODER_INVALID_OBJECT);
        puEventData->uDecoder.sAssociate.un32SubSuspendDate =
            RADIO_un32SuspendDate(hModule, DECODER_INVALID_OBJECT);
        puEventData->uDecoder.sAssociate.tCapabilities =
            RADIO_tDecoderCapabilities(hModule, hDecoder);;
        puEventData->uDecoder.sAssociate.hSTI = hSTI;

        bResult = SMSE_bPostEvent(hEvent);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post ASSOCIATE event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: ASSOCIATE event is posted.\n", psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   DECODER_bUnassociateve
*
*****************************************************************************/
BOOLEAN DECODER_bUnassociate (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bResult = FALSE;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s(hDecoder: %p)\n", __FUNCTION__, hDecoder);

    do
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Verify SMS Object is valid
        bResult = SMSO_bValid((SMS_OBJECT)hDecoder);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Invalid DECODER object.");
            break;
        }

        bResult = SMSE_bPostSignal(psObj->hEventHdlr,
            SMS_EVENT_UNASSOCIATE, SMS_EVENT_OPTION_NONE);
        if (FALSE == bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post UNASSOCIATE event.");
            break;
        }

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: UNASSOCIATE event is posted.\n", psObj->pacObjectName);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   DECODER_bFeaturedFavorites
 *
 *  hDecoder - DECODER_OBJECT to operate on
 *  iEnable - 0 to disable, anything else to enable
 *
 *
 *****************************************************************************/
BOOLEAN DECODER_bFeaturedFavorites(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_FEATURED_FAVORITES_STRUCT const *psEvent
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_FEATURED_FAVORITES,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sFeaturedFavorites = *psEvent;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSmartFavorites
 *
 *****************************************************************************/
BOOLEAN DECODER_bSmartFavorites (
    DECODER_OBJECT hDecoder,
    SMS_EVENT_SMART_FAVORITES_STRUCT *psEventData
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (bValid == TRUE)
    {
        BOOLEAN bOwner;
        SMS_EVENT_SMART_FAVORITES_STRUCT sEventData;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        bOwner = SMSO_bIsOwner((SMS_OBJECT)hDecoder);
        if (bOwner == TRUE)
        {
            sEventData.eAction = SMART_FAVORITES_ACTION_UPDATE;

            // Perform synchronous update
            bPosted = SMART_FAVORITES_bEventHandler(
                hDecoder,
                psObj->hSmartFavorites,
                &sEventData);
        }
        else
        {
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Allocate an event
            hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_SMART_FAVORITES,
                &puEventData, SMS_EVENT_OPTION_NONE);

            if (hEvent != SMS_INVALID_EVENT_HDL)
            {
                // Extract event information and populate it
                puEventData->uDecoder.sSmartFavorites = *psEventData;
                bPosted = SMSE_bPostEvent(hEvent);
            }
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSetError
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetError(
    DECODER_OBJECT hDecoder, DECODER_ERROR_CODE_ENUM eErrorCode)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_ERROR,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sError.eErrorCode = eErrorCode;

            // TODO: BREAKPOINT
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_hModule
 *
 *****************************************************************************/
MODULE_OBJECT DECODER_hModule(DECODER_OBJECT hDecoder)
{
    MODULE_OBJECT hModule = MODULE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify input and make sure caller is already the owner
    // of the DECODER Object. If they are not, this should never
    // allow the caller to obtain an object member this way.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract object member they requested
        hModule = psObj->hModule;
    }

    return hModule;
}

/*****************************************************************************
 *
 *   DECODER_hCCache
 *
 *****************************************************************************/
CCACHE_OBJECT DECODER_hCCache(DECODER_OBJECT hDecoder)
{
    CCACHE_OBJECT hCCache = CCACHE_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify input and make sure caller is already the owner
    // of the DECODER Object. If they are not, this should never
    // allow the caller to obtain an object member this way.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract object member they requested
        hCCache = psObj->hCCache;
    }

    return hCCache;
}

/*****************************************************************************
 *
 *   DECODER_hGetSmartFavoritesHandle
 *
 *****************************************************************************/
SMART_FAVORITES_OBJECT DECODER_hGetSmartFavoritesHandle (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    SMART_FAVORITES_OBJECT hSmartFavorites = SMART_FAVORITES_INVALID_OBJECT;

    // Verify and make sure caller is already owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hSmartFavorites = psObj->hSmartFavorites;
    }

    return hSmartFavorites;
}

/*****************************************************************************
 *
 *   DECODER_hCME
 *
 *****************************************************************************/
CME_OBJECT DECODER_hCME(DECODER_OBJECT hDecoder)
{
    CME_OBJECT hCME = CME_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Verify input and make sure caller is already the owner
    // of the DECODER Object. If they are not, this should never
    // allow the caller to obtain an object member this way.
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Extract object member they requested
        hCME = psObj->hCME;
    }

    return hCME;
}

/*****************************************************************************
 *
 *   DECODER_hAllocateEvent
 *
 *   This API is called by someone who wants to allocate an event from the
 *   specific DECODER specified.
 *
 *****************************************************************************/
SMS_EVENT_HDL DECODER_hAllocateEvent(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    SMS_EVENT_DATA_UNION ** ppuEventData
        )
{
    BOOLEAN bValid;
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(
            psObj->hEventHdlr, eEvent, ppuEventData, tOptions);
    }

    return hEvent;
}

/*****************************************************************************
 *
 *   DECODER_vCreateSongList
 *
 *****************************************************************************/
void DECODER_vCreateSongList(DECODER_OBJECT hDecoder, SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_CREATE_SONGLIST, &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSonglist.hSonglist
                            = hSonglist;

            SMSE_bPostEvent(hEvent);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_vDestroySongList
 *
 *****************************************************************************/
void DECODER_vDestroySongList(DECODER_OBJECT hDecoder, SONGLIST_OBJECT hSonglist)
{
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_DESTROY_SONGLIST, &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSonglist.hSonglist
                            = hSonglist;

            SMSE_bPostEvent(hEvent);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_vPlaybackEvent
 *
 *****************************************************************************/
void DECODER_vPlaybackEvent(
    DECODER_OBJECT hDecoder, SMSAPI_EVENT_MASK tEventMask)
{
    // De-reference object
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    SMSU_tUpdate(&psObj->sEvent, tEventMask);

    return;
}

/*****************************************************************************
 *
 *   DECODER_vTuneMixEvent
 *
 *****************************************************************************/
void DECODER_vTuneMixEvent(
    DECODER_OBJECT hDecoder
        )
{
    // De-reference object
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    if (psObj != (DECODER_OBJECT_STRUCT*)NULL)
    {
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_TUNEMIX);
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_bPlayPause
 *
 *       Post the SMS_EVENT_PLAY or PAUSE event to the Decoder
 *
 *    input:
 *        hDecoder
 *        bPause
 *    output:
 *           TRUE if ok
 *           FALSE if not ok
 *****************************************************************************/
BOOLEAN DECODER_bPlayPause(DECODER_OBJECT hDecoder, BOOLEAN bPause)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        SMS_EVENT_TYPE_ENUM eType = SMS_EVENT_PLAY;

        if(bPause == TRUE)
        {
            eType = SMS_EVENT_PAUSE;
        }

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, eType, &puEventData,
            SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSeekTime
 *
 *       Post the SMS_EVENT_SEEK_TIME event to the Decoder
 *
 *    input:
 *        hDecoder -
 *        n32Seconds -
 *        bPauseAfterSeek -
 *    output:
 *           TRUE if ok
 *           FALSE if not ok
 *****************************************************************************/
BOOLEAN DECODER_bSeekTime(
    DECODER_OBJECT hDecoder, N32 n32Seconds, BOOLEAN bPauseAfterSeek)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_SEEK_TIME,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSeekTime.n32Offset = n32Seconds;
            puEventData->uDecoder.sSeekTime.bPauseAfterSeek
                            = bPauseAfterSeek;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSeekSong
 *
 *       Post the SMS_EVENT_SEEK_SONG event to the Decoder
 *
 *    input:
 *        hDecoder
 *        n16RelativeOffset
 *        bPauseAfterSeek
 *    output:
 *           TRUE if ok
 *           FALSE if not ok
 *****************************************************************************/
BOOLEAN DECODER_bSeekSong(
    DECODER_OBJECT hDecoder, N16 n16RelativeOffset, BOOLEAN bPauseAfterSeek)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_SEEK_SONG,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSeekSong.n16Offset
                            = n16RelativeOffset;
            puEventData->uDecoder.sSeekSong.bPauseAfterSeek
                            = bPauseAfterSeek;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSeekPrevious
 *
 *       Post the SMS_EVENT_SEEK_PREVIOUS event to the Decoder
 *
 *    input:
 *        hDecoder
 *        bPauseAfterSeek
 *    output:
 *           TRUE if ok
 *           FALSE if not ok
 *****************************************************************************/
BOOLEAN DECODER_bSeekPrevious(
    DECODER_OBJECT hDecoder, BOOLEAN bPauseAfterSeek)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_SEEK_PREVIOUS,
            &puEventData, SMS_EVENT_OPTION_NONE);

        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSeekProximal.bPauseAfterSeek
                            = bPauseAfterSeek;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSeekNext
 *
 *       Post the SMS_EVENT_SEEK_NEXT event to the Decoder
 *
 *    input:
 *        hDecoder
 *        bPauseAfterSeek
 *    output:
 *           TRUE if ok
 *           FALSE if not ok
 *****************************************************************************/
BOOLEAN DECODER_bSeekNext(
    DECODER_OBJECT hDecoder, BOOLEAN bPauseAfterSeek)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_SEEK_NEXT,
            &puEventData, SMS_EVENT_OPTION_NONE);

        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSeekProximal.bPauseAfterSeek
                            = bPauseAfterSeek;

            bPosted = SMSE_bPostEvent(hEvent);
        }
    }
    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_vSeekAlertEvent
 *
 *****************************************************************************/
void DECODER_vSeekAlertEvent(DECODER_OBJECT hDecoder)
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // A seek alert occurred. Set the mask...
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_SEEK_ALERT);

        // ... and update the application. Here we start the timer, which
        // will trigger notification event sending from Decoder context.
        CME_vStartTimeoutEventTimer(psObj->hCME);
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_bSetSeekServiceHandle
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetSeekServiceHandle(
    DECODER_OBJECT hDecoder, SEEK_SERVICE_ENUM eService,
    SEEK_SERVICE_OBJECT hSeekService)
{
    BOOLEAN bOwner, bOk = FALSE;

    // Verify
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        switch (eService)
        {
            case SEEK_SERVICE_ARTIST_TITLE:
            {
                psObj->hArtistTitleSeek = hSeekService;
                bOk = TRUE;
            }
            break;

            case SEEK_SERVICE_TRAFFIC_WEATHER:
            {
                psObj->hTrafficWeatherSeek = hSeekService;
                bOk = TRUE;
            }
            break;

            case SEEK_SERVICE_SPORTS:
            {
                psObj->hSportsSeek = hSeekService;
                bOk = TRUE;
            }
            break;

            case SEEK_SERVICE_INVALID:
            default:
                bOk = FALSE;
            break;
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *  DECODER_eSeekService
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eSeekService(
    DECODER_OBJECT hDecoder, SEEK_SERVICE_ENUM eService,
    SEEK_SERVICE_OBJECT *phSeekService)
{
    SEEK_SERVICE_OBJECT hService = SEEK_SERVICE_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult = SMSAPI_RETURN_CODE_SUCCESS;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        switch (eService)
        {
            case SEEK_SERVICE_ARTIST_TITLE:
            {
                hService = psObj->hArtistTitleSeek;
            }
            break;

            case SEEK_SERVICE_TRAFFIC_WEATHER:
            {
                hService = psObj->hTrafficWeatherSeek;
            }
            break;

            case SEEK_SERVICE_SPORTS:
            {
                hService = psObj->hSportsSeek;
            }
            break;

            case SEEK_SERVICE_INVALID:
            default:
            {
                eResult = SMSAPI_RETURN_CODE_INVALID_INPUT;
            }
            break;
        }

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // copy the handle to the provided location
                *phSeekService = hService;

            if (hService == SEEK_SERVICE_INVALID_OBJECT)
            {
                // if the handle is invalid it means the service
                // hasn't been started yet
                eResult = SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED;
            }
        }
    }
    else
    {
        eResult = SMSAPI_RETURN_CODE_ERROR;
    }

    return eResult;
}

/*****************************************************************************
 *
 *  DECODER_bSetPresetsHandle
 *
 *   This API is called by the presets service in order to inform the
 *   decoder that the service has started and to verify this decoder
 *   doesn't already have a presets service instance already running
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetPresetsHandle(
    DECODER_OBJECT hDecoder, PRESETS_OBJECT hPresets)
{
    BOOLEAN bLocked, bOk = FALSE;

    // Verify and lock SMS Object
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);

    if(bLocked == TRUE)
    {
        BOOLEAN bSyncronized;
        CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // If the handle is invalid, just
        // clear out the presets handle
        if(hPresets == PRESETS_INVALID_OBJECT)
        {
            psObj->hPresets = PRESETS_INVALID_OBJECT;
            bOk = TRUE;
        }
        // Otherwise, if this decoder doesn't have a presets
        // handle yet, then we may set it now
        else if(psObj->hPresets == PRESETS_INVALID_OBJECT)
        {
            psObj->hPresets = hPresets;
            bOk = TRUE;
        }

        bSyncronized = SMART_FAVORITES_bIsPresetsSync(hDecoder);
        if (bSyncronized == TRUE)
        {
            // Get Smart Favorites Category Id
            tCategoryId = SMART_FAVORITES.tCategoryId(hDecoder);
        }

        PRESETS_eSetBandSyncCategory(hPresets, tCategoryId);

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bOk;
}

/*****************************************************************************
 *
 *  DECODER_bSetSmartFavoritesHandle
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetSmartFavoritesHandle (
    DECODER_OBJECT hDecoder,
    SMART_FAVORITES_OBJECT hSmartFavorites,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bSuccess = FALSE, bLocked;

    // Verify and make sure caller is already owner
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if (hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            // If the handle is invalid, just clear out handle
            psObj->hSmartFavorites = SMART_FAVORITES_INVALID_OBJECT;
            bSuccess = TRUE;
        }
        else if (psObj->hSmartFavorites == SMART_FAVORITES_INVALID_OBJECT)
        {
            // Otherwise, if this decoder doesn't have a smart favorites
            // handle yet, then we may set it now
            psObj->hSmartFavorites = hSmartFavorites;
            bSuccess = TRUE;
        }

        PRESETS_eSetBandSyncCategory(psObj->hPresets, tCategoryId);

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   DECODER_vGenericUpdateEventMask
 *
 *****************************************************************************/
void DECODER_vGenericUpdateEventMask(
    DECODER_OBJECT hDecoder,
    SMSAPI_EVENT_MASK tEventMask
        )
{
    // De-reference object
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    SMSU_tUpdate(&psObj->sEvent, tEventMask);
    return;
}

/*****************************************************************************
 *
 *  DECODER_hPresets
 *
 *****************************************************************************/
PRESETS_OBJECT DECODER_hPresets(DECODER_OBJECT hDecoder)
{
    PRESETS_OBJECT hPresets = PRESETS_INVALID_OBJECT;
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        hPresets = psObj->hPresets;

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return hPresets;
}

/*****************************************************************************
 *
 *  DECODER_pacName
 *
 *****************************************************************************/
const char *DECODER_pacName(DECODER_OBJECT hDecoder)
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        return psObj->pacDecoderName;
    }

    return NULL;
}

/*****************************************************************************
 *
 *   DECODER_hGetTag
 *
 *****************************************************************************/
TAG_OBJECT DECODER_hGetTag(DECODER_OBJECT hDecoder)
{
    BOOLEAN bOwner;
    TAG_OBJECT hTag = TAG_INVALID_OBJECT;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // Verify the object
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        hTag = psObj->hTag;
    }

    return hTag;
}

#if SMS_LOGGING == 1
/*****************************************************************************
 *
 *  DECODER_vLog
 *
 *   This API provides an entry point into the DECODER's log so that objects
 *   and services utilizing the DECODER may add to that log.
 *
 *   Note:  You must already be in the context of this DECODER in order to
 *   utilize this!
 *
 *****************************************************************************/
void DECODER_vLog (
    DECODER_OBJECT hDecoder,
    const char *pcFormat,
    ...
        )
{
    BOOLEAN bOwner;

    // Verify input
    if (pcFormat == NULL)
    {
        return;
    }

    // Verify SMS Object and operating context
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
        (DECODER_OBJECT_STRUCT *)hDecoder;
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, pcFormat);

        // Log the requested information
        SMSE_vLogUsingList(
            psObj->hEventHdlr,
            pcFormat, &tList );

        // restore stack (push)
        va_end(tList);
    }

    return;
}
#elif __STDC_VERSION__ < 199901L
void DECODER_vLogNothing (
    DECODER_OBJECT hDecoder,
    const char *pcFormat,
    ...
        )
{
    return;
}
#endif


/*****************************************************************************
 *
 *  DECODER_vUpdateAudioDecoderBitRate
 *
 *****************************************************************************/
void DECODER_vUpdateAudioDecoderBitRate(
    DECODER_OBJECT hDecoder,
    AUDIO_DECODER_BITRATE tNewAudioDecoderBitrate
        )
{
    BOOLEAN bReturn;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    bReturn = ENGINEERING_DATA_bUpdateAudioDecoderBitrate(hDecoder,
                                            tNewAudioDecoderBitrate);
    if (bReturn == TRUE)
    {
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ENGINEERING_DATA);
    }

    return;
}


/*****************************************************************************
 *
 *  DECODER_vUpdateDetailedSignalQuality
 *
 *****************************************************************************/
void DECODER_vUpdateDetailedSignalQuality(
    DECODER_OBJECT hDecoder,
    DETAILED_SIGNAL_QUALITY_STRUCT const *psNewDetailedSignalQuality
        )
{
    BOOLEAN bReturn;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    bReturn =
        ENGINEERING_DATA_bUpdateNewDetailedSignalQuality(
            hDecoder,
            psNewDetailedSignalQuality
                );

    if (bReturn == TRUE)
    {
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ENGINEERING_DATA);
    }
    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateDetailedOverlaySignalQuality
 *
 *****************************************************************************/
void DECODER_vUpdateDetailedOverlaySignalQuality(
    DECODER_OBJECT hDecoder,
    DETAILED_OVERLAY_SIGNAL_QUALITY_STRUCT const *psNewDetailedOverlaySignalQuality
        )
{
    BOOLEAN bReturn;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    bReturn =
        ENGINEERING_DATA_bUpdateNewDetailedOverlaySignalQuality(
            hDecoder,
            psNewDetailedOverlaySignalQuality
                );

    if (bReturn == TRUE)
    {
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ENGINEERING_DATA);
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateLinkStatusInformation
 *
 *****************************************************************************/
void DECODER_vUpdateLinkStatusInformation(
    DECODER_OBJECT hDecoder,
    LINK_INFORMATION_STRUCT const *psNewLinkStatusInformation
        )
{
    BOOLEAN bReturn;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    bReturn =
        ENGINEERING_DATA_bUpdateLinkStatusInformation(
            hDecoder,
            psNewLinkStatusInformation
                );

    if (bReturn == TRUE)
    {
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ENGINEERING_DATA);
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateSignal
 *
 *****************************************************************************/
void DECODER_vUpdateSignal(
    DECODER_OBJECT hDecoder,
    const SIGNAL_QUALITY_STRUCT *psSignalQuality
    )
{
    // Verify input
    if(psSignalQuality != NULL)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Set new signal information (if different than current)
        if(psObj->sSignalQuality.eComposite != psSignalQuality->eComposite)
        {
            // Update signal
            psObj->sSignalQuality.eComposite = psSignalQuality->eComposite;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_SIGNAL);
        }

        // Set new state information (if different than current)
        if(psObj->sSignalQuality.eState != psSignalQuality->eState)
        {
            // Update signal state
            psObj->sSignalQuality.eState = psSignalQuality->eState;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_SIGNAL);
        }

        // Set new signal information (if different than current)
        if(psObj->sSignalQuality.eSatellite != psSignalQuality->eSatellite)
        {
            // Update signal
            psObj->sSignalQuality.eSatellite = psSignalQuality->eSatellite;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ANTENNA_AIMING);
        }

        // Set new signal information (if different than current)
        if(psObj->sSignalQuality.eTerrestrial != psSignalQuality->eTerrestrial)
        {
            // Update signal
            psObj->sSignalQuality.eTerrestrial = psSignalQuality->eTerrestrial;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ANTENNA_AIMING);
        }

        // Set new aiming information (if different than current)
        if(psObj->sSignalQuality.sAntennaAimingData.un8SatPercentSigLevel !=
            psSignalQuality->sAntennaAimingData.un8SatPercentSigLevel)
        {
            // Update signal state
            psObj->sSignalQuality.sAntennaAimingData.un8SatPercentSigLevel =
                psSignalQuality->sAntennaAimingData.un8SatPercentSigLevel;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ANTENNA_AIMING);
        }

        // Set new aiming information (if different than current)
        if(psObj->sSignalQuality.sAntennaAimingData.un8TerPercentSigLevel !=
            psSignalQuality->sAntennaAimingData.un8TerPercentSigLevel)
        {
            // Update signal state
            psObj->sSignalQuality.sAntennaAimingData.un8TerPercentSigLevel =
                psSignalQuality->sAntennaAimingData.un8TerPercentSigLevel;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ANTENNA_AIMING);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateAntennaState
 *
 *****************************************************************************/
void DECODER_vUpdateAntennaState(
    DECODER_OBJECT hDecoder,
    N8 n8Antenna,
    ANTENNA_STATE_ENUM eAntennaState
        )
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // Check if it's an antenna we have
    if((n8Antenna > -1) && (n8Antenna < psObj->n8NumAntennas))
    {
        // Set new state information (if different than current)
        if(psObj->apeAntennaState[n8Antenna] != eAntennaState)
        {
            // Update signal state
            psObj->apeAntennaState[n8Antenna] = eAntennaState;

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_ANTENNA);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateProgress
 *
 *****************************************************************************/
void DECODER_vUpdateProgress(
    DECODER_OBJECT hDecoder,
    UN8 un8UpdateProgress
        )
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    UN8 un8TotalUpdateProgress = psObj->un8TotalUpdateProgress;

    // set new status information
    psObj->un8ReceiverUpdateProgress = un8UpdateProgress;

    // condition the Reported Progress. Because we need to do some
    // remapping of the ccache we don't want to report 100% and then
    // make the app wait for several seconds. This might confuse the
    // user. Instead we will report a conditioned number such that 100%
    // coincides with the completion of the update
    // For now we'll just clip the reported progress at 99% until the
    // completion of the update. Later we can do a more sophisticated
    // conditioning to get a smoother progression if we feel it
    // necessary.
    if(psObj->un8ReceiverUpdateProgress == 100)
    {
        if(psObj->un8TotalUpdateProgress != 100)
        {
            psObj->un8TotalUpdateProgress = 99;
        }
        else
        {
            // do nothing. The reported progress will remain at 100%
        }
    }
    else
    {
        // We'll just report the received progress
        psObj->un8TotalUpdateProgress = psObj->un8ReceiverUpdateProgress;
    }

    // Update our state
    vUpdateState(psObj, DECODER_STATE_UPDATING);

    // Check if anything changed
    if(un8TotalUpdateProgress != psObj->un8TotalUpdateProgress)
    {
        // Set event mask to let app know new info
        // is available
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_UPDATE_PROGRESS);
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateProgress
 *
 *****************************************************************************/
void DECODER_vUpdateComplete(
    DECODER_OBJECT hDecoder
        )
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // the update is complete.

    // Check if we have a cache
    if(psObj->hCCache == CCACHE_INVALID_OBJECT)
    {
        // we don't have a valid ccache.
        // this means that the update started during
        // initialization so go back to initial state
        vUpdateState(psObj, DECODER_STATE_INITIAL);
    }
    else
    {
        // When this completes we will transition to READY.
        CCACHE_bRemap(psObj->hEventHdlr, vCCacheRemapCallback, psObj);
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateSubscription
 *
 *****************************************************************************/
void DECODER_vUpdateSubscription(DECODER_OBJECT hDecoder)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // only do this if we are ready.
    CCACHE_bUpdateSubscription(
        psObj->hEventHdlr, vCCacheSubscriptionUpdateCallback, psObj);

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateTunedServiceId
 *
 *****************************************************************************/
void DECODER_vUpdateTunedServiceId(
    DECODER_OBJECT hDecoder,
    SERVICE_ID tTunedServiceId,
    BOOLEAN bNeedToSave
        )
{
    CHANNEL_OBJECT hTunedChannel;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    BOOLEAN bInitialTune = FALSE;
    BOOLEAN bOwner;

    // Verify SMS Object and operating context
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // Did we actually tune anything?
        if(tTunedServiceId != SERVICE_INVALID_ID)
        {
            SERVICE_ID tCurrentServiceId;

            // Update the tuner state
            vUpdateTuneState(psObj, TUNE_STATE_TUNED);

            // If the new service id, doesn't change then
            // there is really nothing for us to do.
            tCurrentServiceId =
                CHANNEL.tServiceId(psObj->hTunedChannel);

            // go get the newly tuned channel from the cache
            hTunedChannel = CCACHE_hChannelFromIds(
                psObj->hCCache, tTunedServiceId, CHANNEL_INVALID_ID, FALSE);

            if(hTunedChannel != CHANNEL_INVALID_OBJECT)
            {
                if(tTunedServiceId != tCurrentServiceId)
                {
                    BROWSE_TYPE_ENUM eCurrentBrowseMode;
                    CATEGORY_OBJECT hTunedCategory;
                    CATEGORY_ID tTunedCategoryId;
                    CHANNEL_ID tTunedChannelId;
                    CHANNEL_ID tCurrentChannelId;

                    // Get the current browse mode
                    eCurrentBrowseMode = BROWSE_eGetMode(psObj->hBrowse);

                    // The currently tuned category
                    // is this channel's broadcast category
                    hTunedCategory = CHANNEL.hCategory(hTunedChannel, 0);

                    // Get current channel id
                    tCurrentChannelId = CHANNEL.tChannelId(psObj->hTunedChannel);

                    // update the last tuned channel.

                    // see if we have a value for currently tuned channel
                    if(tCurrentChannelId == CHANNEL_INVALID_ID)
                    {
                        // no we don't have a currently tuned channel.
                        // so in that case we'll set the last tuned channel
                        // and category to the newly tuned channel
                        // and category
                        vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
                            hTunedChannel, hTunedCategory);
                    }
                    else
                    {
                        // yes we have a currently tuned channel.
                        // so in that case we'll set the last tuned channel
                        // and category to the currently tuned channel
                        // and category
                        vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
                            psObj->hTunedChannel, psObj->hTunedCategory);
                    }

                    if(psObj->hBrowsedChannel != CHANNEL_INVALID_OBJECT)
                    {
                        SERVICE_ID tBrowsedServiceId;

                        // Get the browsed service Id
                        tBrowsedServiceId = CHANNEL.tServiceId(psObj->hBrowsedChannel);

                        if(tTunedServiceId == tBrowsedServiceId)
                        {
                            // If we just tuned the browsed
                            // channel, ensure we don't
                            // loose the browsed category
                            // state
                            hTunedCategory = psObj->hBrowsedCategory;
                        }
                    }

                    // Get the tuned category's Id
                    tTunedCategoryId = CATEGORY.tGetCategoryId(hTunedCategory);

                    // set browsed channel and category to
                    // the newly tuned channel and category
                    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED, hTunedChannel,
                        hTunedCategory);

                    // Update the browse state
                    // with the currently tuned
                    // channel & category
                    tTunedChannelId = CHANNEL.tChannelId(hTunedChannel);
                    BROWSE_bSetMode(psObj->hBrowse, eCurrentBrowseMode, tTunedChannelId,
                        tTunedCategoryId);

                    // If the channel being tuned is different than the
                    // channel currently tuned to, update it.
                    // otherwise there's no need to.
                    if(hTunedChannel != psObj->hTunedChannel)
                    {
                        if (psObj->hTunedChannel == CHANNEL_INVALID_OBJECT)
                        {
                            // if tuned channel changed, and the tuned chanel
                            // is invalid, then this must be the initial tune
                            bInitialTune = TRUE;
                        }

                        // Update tuned channel & category
                        vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_TUNED, hTunedChannel,
                            hTunedCategory);
                    }

                    SUB_NOTIFICATION_vChannelTuned(
                        psObj->hSubscriptionService,
                        psObj->hTunedChannel,
                        bInitialTune
                            );
                }

                // Save this channel's service id even if it has not changed
                // because it might not be saved before, e.g. when tune to
                // Flash Event.
                if (bNeedToSave == TRUE)
                {
                    vSaveTunedServiceId(psObj);
                }
            }
        }
        else // We could not tune (should never be sent)
        {
            // At this point, nobody knows what is going on

            // Update the tuner state
            vUpdateTuneState(psObj, TUNE_STATE_UNKNOWN);

            // Update handles
            vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
                CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);
            vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED,
                CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);
            vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_TUNED,
                CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateTuneScanContentState
 *
 *****************************************************************************/
void DECODER_vUpdateTuneScanContentState (
    DECODER_OBJECT hDecoder,
    BOOLEAN bIsAvailable
        )
{
    BOOLEAN bOwner;

    // Verify object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        BOOLEAN bUpdate = FALSE;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if (bIsAvailable == TRUE)
        {
            // Tune Scan Content is Not Available Now
            if ((psObj->sTuneScan.tStatusMask &
                    DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE) !=
                        DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE)
            {
                // Set new Content Availability State
                psObj->sTuneScan.tStatusMask |=
                    DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE;

                // Need to perform an update
                bUpdate = TRUE;
            }
        }
        else
        {
            // Tune Scan Content is Available Now
            if ((psObj->sTuneScan.tStatusMask &
                    DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE) ==
                        DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE)
            {
                psObj->sTuneScan.tStatusMask &=
                    ~DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE;

                // Need to perform an update
                bUpdate = TRUE;
            }
        }

        // Update requested
        if (bUpdate == TRUE)
        {
            // Update decoder's event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_TUNE_SCAN);

            // Notify all interested parties
            SMSU_bNotify(&psObj->sEvent);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateTuneScanTerminateState
 *
 *****************************************************************************/
void DECODER_vUpdateTuneScanTerminateState (
    DECODER_OBJECT hDecoder,
    BOOLEAN bTerminated
        )
{
    BOOLEAN bOwner;

    // Verify SMS object is owned
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        BOOLEAN bUpdate = FALSE;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if ((bTerminated == TRUE) &&
            (psObj->sTuneScan.bActive == TRUE))
        {
            if ((psObj->sTuneScan.tStatusMask &
                    DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED) !=
                        DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED)
            {
                // Set new Scan Aborted State
                psObj->sTuneScan.tStatusMask |=
                    DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED;

                // Need to perform an update
                bUpdate = TRUE;
            }

            // Module indicates that scan was aborted for some reason
            // So, set to non-active

            // Update scan state
            psObj->sTuneScan.bActive = FALSE;
        }
        else if (bTerminated == FALSE)
        {
            if ((psObj->sTuneScan.tStatusMask &
                    DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED) ==
                        DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED)
            {
                psObj->sTuneScan.tStatusMask &=
                    ~DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED;

                // Need to perform an update
                bUpdate = TRUE;
            }

            // State of the scan shall be reseted upon of the start
            // of the next cycle. So, we  can consider this state as active

            // Update scan state
            psObj->sTuneScan.bActive = TRUE;
        }

        // Update requested
        if (bUpdate == TRUE)
        {
            // Update decoder's event mask
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_TUNE_SCAN);

            // Notify all interested parties
            SMSU_bNotify(&psObj->sEvent);
        }
    }

    return;
}

/*****************************************************************************
 *
 *  DECODER_bTuneScanActive
 *
 *****************************************************************************/
BOOLEAN DECODER_bTuneScanActive(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner, bActive = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        bActive = psObj->sTuneScan.bActive;
    }

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
        "Tune Scan State: %s\n", (bActive == TRUE) ? "Active" : "Idle" );

    return bActive;
}

/*****************************************************************************
 *
 *  DECODER_tBrowsedServiceId
 *
 *****************************************************************************/
SERVICE_ID DECODER_tBrowsedServiceId(
    DECODER_OBJECT hDecoder
        )
{
    // Just call location function, someday this may be a public API
    return tBrowsedServiceId(hDecoder);
}

/*****************************************************************************
 *
 *  DECODER_tLastTunedServiceId
 *
 *****************************************************************************/
SERVICE_ID DECODER_tLastTunedServiceId(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    SERVICE_ID tLastTunedServiceId = SERVICE_INVALID_ID;

    // Verify SMS object is owned
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        tLastTunedServiceId =
            CHANNEL.tServiceId(psObj->hLastTunedChannel);
    }

    return tLastTunedServiceId;
}

/*****************************************************************************
 *
 *  DECODER_vTunedChanNoLongerAvail
 *
 *****************************************************************************/
void DECODER_vTunedChanNoLongerAvail(
    DECODER_OBJECT hDecoder
        )
{
    SERVICE_ID tServiceId;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    SMS_EVENT_TUNE_STRUCT sTune;

    tServiceId = tGetDefaultOrSafeServiceId(psObj);

    // Configure tune
    sTune.tChannelId = CHANNEL_INVALID_ID;
    sTune.tServiceId = tServiceId;
    sTune.bLockedOverride = FALSE;
    sTune.bMatureOverride = FALSE;
    sTune.bSkippedOverride = FALSE;
    sTune.bPlayUnrestricted = FALSE;

    // Cause RADIO to tune to the default or safe channel
    // with no overrides.
    bTuneServiceId(psObj, &sTune);

    return;
}

/*****************************************************************************
 *
 *  DECODER_vLastTunedChanNoLongerAvail
 *
 *****************************************************************************/
void DECODER_vLastTunedChanNoLongerAvail(
    DECODER_OBJECT hDecoder
        )
{
    SERVICE_ID tServiceId;
    CHANNEL_OBJECT hChannel;

    // De-reference object
    DECODER_OBJECT_STRUCT *psObj =
        (DECODER_OBJECT_STRUCT *)hDecoder;

    // Use the default or safe channel
    tServiceId = tGetDefaultOrSafeServiceId(psObj);

    // Get channel object
    hChannel = CCACHE_hChannelFromIds(
        psObj->hCCache, tServiceId, CHANNEL_INVALID_ID, FALSE);

    if (hChannel == CHANNEL_INVALID_OBJECT)
    {
        // Fail!
        vSetError(psObj, DECODER_ERROR_CODE_RADIO_ERROR);
    }

    // Set last tuned channel
    psObj->hLastTunedChannel = hChannel;

    return;
}

/*****************************************************************************
 *
 *   DECODER_bStartAllChanCatNotifications
 *
 *****************************************************************************/
BOOLEAN DECODER_bStartAllChanCatNotifications(
    DECODER_OBJECT hDecoder,
    BOOLEAN bNotifyChannels
        )
{
    BOOLEAN bPosted = FALSE;

    // De-reference object
    DECODER_OBJECT_STRUCT *psObj =
        (DECODER_OBJECT_STRUCT *)hDecoder;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
        SMS_EVENT_ALL_CHAN_CAT_NOTIFY, &puEventData, SMS_EVENT_OPTION_NONE);
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Extract event information and populate it
        puEventData->uDecoder.sAllChanCatNotify.bNotifyChannels =
            bNotifyChannels;

        bPosted = SMSE_bPostEvent(hEvent);
        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post All Chan/Cat event");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to allocate All Chan/Cat event");
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bRadioReady
 *
 *****************************************************************************/
BOOLEAN DECODER_bRadioReady (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bPosted = FALSE;

    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    if ( psObj != NULL )
    {
        vHandleRadioReadyEvent(psObj);
        bPosted = TRUE;
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bSetRadioSpecificData
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetRadioSpecificData(
    DECODER_OBJECT hDecoder,
    RADIO_PRIVATE_DATA_OBJECT hData
        )
{
    BOOLEAN bOwner;

    // Verify SMS Object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Set radio specific data object
        psObj->hRadioData = hData;
    }

    return bOwner;
}

/*****************************************************************************
 *
 *   DECODER_hGetRadioSpecificData
 *
 *****************************************************************************/
RADIO_PRIVATE_DATA_OBJECT DECODER_hGetRadioSpecificData(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    RADIO_PRIVATE_DATA_OBJECT hData =
        RADIO_PRIVATE_DATA_INVALID_OBJECT;

    // Verify SMS Object and pointer validity
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        if(psObj->bRadioInitialized == TRUE)
        {
            // Get radio specific data object
            hData = psObj->hRadioData;
        }
    }

    return hData;
}

/*****************************************************************************
 *
 *  DECODER_bSetSportZoneServiceHandle
 *
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetSportZoneServiceHandle(
    DECODER_OBJECT hDecoder,
    SPORT_ZONE_SERVICE_OBJECT hService)
{
    BOOLEAN bOwner;

    // Verify
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        psObj->hSportZoneService = hService;
    }

    return bOwner;
}

/*****************************************************************************
 *
 *   DECODER_hSportZoneService
 *
 *****************************************************************************/
SPORT_ZONE_SERVICE_OBJECT DECODER_hSportZoneService(
    DECODER_OBJECT hDecoder)
{
    SPORT_ZONE_SERVICE_OBJECT hService = SPORT_ZONE_SERVICE_INVALID_OBJECT;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        hService = psObj->hSportZoneService;
    }

    return hService;
}

/*****************************************************************************
 *
 *  DECODER_bSetSubNotificationServiceHandle
 *
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetSubNotificationServiceHandle(
    DECODER_OBJECT hDecoder,
    SUB_NOTIFICATION_OBJECT hService)
{
    BOOLEAN bOwner;

    // Verify
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        psObj->hSubscriptionService = hService;
    }

    return bOwner;
}

/*****************************************************************************
 *
 *   DECODER_hSubscriptionService
 *
 *****************************************************************************/
SUB_NOTIFICATION_OBJECT DECODER_hSubscriptionService(
    DECODER_OBJECT hDecoder)
{
    SUB_NOTIFICATION_OBJECT hService = SUB_NOTIFICATION_INVALID_OBJECT;
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        hService = psObj->hSubscriptionService;
    }

    return hService;
}

/*****************************************************************************
 *
 *   DECODER_eSetSubscriptionAlertText
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eSetSubscriptionAlertText(
    DECODER_OBJECT hDecoder, const char *pacSubAlertArtistText,
    const char *pacSubAlertTitleText)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // Lock object for exclusive access
    BOOLEAN bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);

    if(bOwner == TRUE)
    {
        CHANNEL_OBJECT hSubAlertChannel;

        // Extract the CCACHE sub alert channel
        hSubAlertChannel =
            CCACHE_hSubscriptionAlertChannel(psObj->hCCache);
        if(hSubAlertChannel != CHANNEL_INVALID_OBJECT)
        {
            // Set Sub Alert Channel's Artist Text
            CDO_vUpdate(hSubAlertChannel, pacSubAlertArtistText,
                CDO_FIELD_ARTIST);

            // Set Sub Alert Channel's Title Text
            CDO_vUpdate(hSubAlertChannel, pacSubAlertTitleText,
                CDO_FIELD_TITLE);
        }

        return SMSAPI_RETURN_CODE_SUCCESS;
    }

    return SMSAPI_RETURN_CODE_ERROR;
}

/*****************************************************************************
 *
 *   DECODER_vChannelEventMask
 *
 *****************************************************************************/
void DECODER_vChannelEventMask(
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK *ptChannelMask)
{
    if (ptChannelMask != NULL)
    {
        CCACHE_OBJECT hCCache;
        DECODER_OBJECT hDecoder;

        // Extract the channel's ccache handle
        hCCache = (CCACHE_OBJECT)SMSO_hParent((SMS_OBJECT)hChannel);
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)hCCache);

        if(hDecoder != DECODER_INVALID_OBJECT)
        {
            DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

            // is this the browsed channel?
            if (hChannel == psObj->hBrowsedChannel)
            {
                if (psObj->bInDecoderCallback == TRUE)
                {
                    // if just browsed
                    if (psObj->bBrowsed == TRUE)
                    {
                        // substitute the inter-channel mask
                        *ptChannelMask = psObj->tBrowsedEventMask;
                        // clear the flag
                        psObj->bBrowsed = FALSE;
                    }
                }
            }
        }
    }
}

/*****************************************************************************
 *
 *   DECODER_hGetEngineeringData
 *
 *****************************************************************************/
ENGINEERING_DATA_OBJECT DECODER_hGetEngineeringData (
    DECODER_OBJECT hDecoder
        )
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        return psObj->hEngineeringData;
    }
    else
    {
        return ENGINEERING_DATA_INVALID_OBJECT;
    }
}

/*****************************************************************************
*
*   DECODER_eModifyChanListRegisteredEventMask
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eModifyChanListRegisteredEventMask (
    DECODER_OBJECT hDecoder,
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        DECODER_OBJECT_STRUCT *psObj;

        psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                    SMS_EVENT_MODIFY_CHANLIST_EVENT_MASK,
                    &puEventData,
                    SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Extract event information and populate it
            puEventData->uDecoder.sChanlistEventStruct.eModification =
                    eModification;
            puEventData->uDecoder.sChanlistEventStruct.tChannelEventMask =
                    tEventMask;
            puEventData->uDecoder.sChanlistEventStruct.hChannelList =
                    hChannelList;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                //Error!!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME
                    ": Unable to post modify channellist event mask.");
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }
    return eReturnCode;
}

/*****************************************************************************
 *
 *  DECODER_eAlertTone
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eAlertTone(
    DECODER_OBJECT hDecoder,
    N8 n8Volume
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;

    eReturn = RADIO_eToneGenerationStart(hDecoder,
        0, // freq ignored for alert
        n8Volume,
        DECODER_TONE_GENERATION_BALANCE_BOTH, // ignored for alert
        TRUE); // alert

    return eReturn;
}

/*****************************************************************************
 *
 *   DECODER_bReset
 *
 *****************************************************************************/
BOOLEAN DECODER_bReset (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s(hDecoder: %p)\n", __FUNCTION__, hDecoder);

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        bPosted = SMSE_bPostSignal(psObj->hEventHdlr,
            SMS_EVENT_RESET, SMS_EVENT_OPTION_URGENT);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to reset this Decoder.");
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_hBrowsedChannel
 *
 *****************************************************************************/
CHANNEL_OBJECT DECODER_hBrowsedChannel(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hChannel = psObj->hBrowsedChannel;
    }

    return hChannel;
}

/*****************************************************************************
 *
 *  DECODER_vUpdateACO
 *
 *****************************************************************************/
void DECODER_vUpdateACO(
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    CHANNEL_ACO tACO
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        BOOLEAN bUpdated;

        bUpdated = CHANNEL_bUpdateACO(hChannel, tACO);
        if (bUpdated == TRUE)
        {
            // De-reference object
            DECODER_OBJECT_STRUCT *psObj =
                (DECODER_OBJECT_STRUCT *)hDecoder;
            OSAL_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = OSAL.eTimerStartRelative(
                psObj->hACOTimer,
                DECODER_CHANNEL_METADATA_COMMIT_TIMEOUT, 
                0); // One-shot

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Failed to Re-/Start Channel Metadata Commit Timer: %s (#%d)",
                        OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_tRequestedEventMask
 *
 *****************************************************************************/
DECODER_EVENT_MASK DECODER_tRequestedEventMask(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    DECODER_EVENT_MASK tRequestedMask = DECODER_OBJECT_EVENT_NONE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner  == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = 
            (DECODER_OBJECT_STRUCT *)hDecoder;

        tRequestedMask = psObj->tRequestMask;
    }

    return tRequestedMask;
}

/*****************************************************************************
 *
 *   DECODER_hBrowse
 *
 *****************************************************************************/
BROWSE_OBJECT DECODER_hBrowse(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    BROWSE_OBJECT hBrowse = BROWSE_INVALID_OBJECT;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hBrowse = psObj->hBrowse;
    }

    return hBrowse;
}

/*****************************************************************************
 *
 *   DECODER_hAllocateCMEUpdateEvent
 *
 *****************************************************************************/
SMS_EVENT_HDL DECODER_hAllocateCMEUpdateEvent(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    SMS_EVENT_HDL hUpdateEvent = SMS_INVALID_EVENT_HDL;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        SMS_EVENT_DATA_UNION *puData;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = 
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hUpdateEvent = 
            SMSE_hAllocateEvent(
                psObj->hEventHdlr, 
                SMS_EVENT_UPDATE_OBJECT, 
                &puData,
                SMS_EVENT_OPTION_STATIC
                    );

        if (hUpdateEvent != SMS_INVALID_EVENT_HDL)
        {
            // Populate event data
            puData->uDecoder.sUpdate.tEventMask = 
                DECODER_OBJECT_EVENT_SEEK_ALERT;
        }
    }

    return hUpdateEvent;
}

/*****************************************************************************
 *
 *   DECODER_bGetTuneMixIndexAvailable
 *
 *****************************************************************************/
BOOLEAN DECODER_bGetTuneMixIndexAvailable(
    DECODER_OBJECT hDecoder,
    UN8 *pun8Index,
    CHANNEL_ID *ptTuneMixId
        )
{
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        int iIndex;

        for( iIndex = 0; iIndex < psObj->un8MaxTuneMix; iIndex++ )
        {
            if (psObj->ahTuneMix[iIndex] == TUNEMIX_INVALID_OBJECT)
            {
                // Save TuneMix index
                if( pun8Index != NULL )
                {
                    *pun8Index = (UN8)iIndex;
                }
                // Save TuneMix ID
                if( ptTuneMixId != NULL )
                {
                    *ptTuneMixId = (CHANNEL_ID)(iIndex + TUNEMIX_CID_OFFSET);
                }

                return TRUE;
            }
        }
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   DECODER_vSetTuneMix
 *
 *****************************************************************************/
void DECODER_vSetTuneMix(
    DECODER_OBJECT hDecoder,
    UN8 un8TMIndex,
    TUNEMIX_OBJECT hTuneMix
        )
{
    BOOLEAN bLocked;

    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        if ((psObj->ahTuneMix[un8TMIndex] == TUNEMIX_INVALID_OBJECT) ||
            (hTuneMix == TUNEMIX_INVALID_OBJECT))
        {
            if (psObj->hTuneMixActive == psObj->ahTuneMix[un8TMIndex])
            {
                psObj->hTuneMixActive = TUNEMIX_INVALID_OBJECT;
            }
            psObj->ahTuneMix[un8TMIndex] = hTuneMix;
        }
        else
        {
            //Error!!This should never have happened...
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME
                ": An attempt to rewrite valid tunemix handle.");
        }

        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_hGetTuneMix
 *
 *****************************************************************************/
TUNEMIX_OBJECT DECODER_hGetTuneMix(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tTuneMixID
        )
{
    TUNEMIX_OBJECT hTuneMix = TUNEMIX_INVALID_OBJECT;

    if (tTuneMixID != CHANNEL_INVALID_ID)
    {
        BOOLEAN bOwner;

        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if(bOwner == TRUE)
        {
            DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
            UN16 un16Index = tTuneMixID - TUNEMIX_CID_OFFSET;

            if (un16Index < psObj->un8MaxTuneMix)
            {
                hTuneMix = psObj->ahTuneMix[un16Index];
            }
        }
    }

    return hTuneMix;
}

/*****************************************************************************
 *
 *   DECODER_hGetTuneMixActive
 *
 *****************************************************************************/
TUNEMIX_OBJECT DECODER_hGetTuneMixActive(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        return psObj->hTuneMixActive;
    }

    return TUNEMIX_INVALID_OBJECT;
}

/*****************************************************************************
 *
 *   DECODER_vSetTuneMixActive
 *
 *****************************************************************************/
void DECODER_vSetTuneMixActive(
    DECODER_OBJECT hDecoder,
    TUNEMIX_OBJECT hTuneMIx
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        psObj->hTuneMixActive = hTuneMIx;
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_un8GetMaxTuneMix
 *
 *****************************************************************************/
UN8 DECODER_un8GetMaxTuneMix(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        return psObj->un8MaxTuneMix;
    }

    return TUNEMIX_OBJECTS_NONE;
}

/*****************************************************************************
 *
 *   DECODER_bIsTuneMixSupported
 *
 *****************************************************************************/
BOOLEAN DECODER_bIsTuneMixSupported(
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bValid, bSupported = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Checking if recording capabilities status
        bSupported = MODULE_bIsIRSupported(psObj->hModule);
        if (psObj->un8MaxTuneMix == TUNEMIX_OBJECTS_NONE)
        {
            // SXI version 3.x is not supported
            bSupported = FALSE;
        }
    }

    return bSupported;
}

/*****************************************************************************
 *
 *   DECODER_eTuneMixTune
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eTuneMixTune(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tTuneMixId,
    BOOLEAN bOverride,
    BOOLEAN bPlayUnrestricted
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bValid;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        BOOLEAN bPosted;
        SMS_EVENT_TUNE_STRUCT sTune;
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Populate tune struct
        sTune.tChannelId = tTuneMixId;
        sTune.tServiceId = tTuneMixId;
        sTune.bLockedOverride = bOverride;
        sTune.bMatureOverride = bOverride;
        sTune.bSkippedOverride = bOverride;
        sTune.bPlayUnrestricted = bPlayUnrestricted;
        
        bPosted = bPostTune(psObj, &sTune);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post tune event");
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   DECODER_tTuneMixIdFromIndex
 *
 *****************************************************************************/
CHANNEL_ID DECODER_tTuneMixIdFromIndex(
    DECODER_OBJECT hDecoder,
    UN8 un8Index
        )
{
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    BOOLEAN bOk;

    // Checking the object
    bOk = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (TRUE == bOk)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT*)hDecoder;
        if (un8Index < psObj->un8MaxTuneMix)
        {
            tChannelId = un8Index + TUNEMIX_CID_OFFSET;
        }
    }

    return tChannelId;
}

/*****************************************************************************
 *
 *   DECODER_eTuneMixConfigure
 *
 *****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eTuneMixConfigure(
    DECODER_OBJECT hDecoder,
    UN8 un8Index
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
    BOOLEAN bValid;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);

    // Verify inputs
    if((bValid == TRUE) &&
       (un8Index < psObj->un8MaxTuneMix))
    {
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;
        BOOLEAN bPosted = FALSE;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_TUNEMIX_CONFIGURE,
            &puEventData,
            SMS_EVENT_OPTION_NONE);

        // Attempt to post the event
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            puEventData->uDecoder.sTuneMix.un8ChIndex =
                un8Index;
            puEventData->uDecoder.sTuneMix.hTuneMix =
                psObj->ahTuneMix[un8Index];

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to post tune event");
                eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to allocate tune event");
            eReturnCode = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }
    } while (FALSE);
    return eReturnCode;
}

/*****************************************************************************
*
*  DECODER_eIterateTuneMixList
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DECODER_eIterateTuneMixList(
    DECODER_OBJECT hDecoder,
    TUNEMIX_ITERATOR_CALLBACK pbIterator,
    void *pvArg
    )
{
    int iIndex;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
    BOOLEAN bOk;
    DECODER_OBJECT_STRUCT *psObj = 
        (DECODER_OBJECT_STRUCT*)hDecoder;

    // Decoder shall be locked
    bOk = SMSO_bOwner((SMS_OBJECT)hDecoder);
    
    // Validate input
    if ((pbIterator == NULL) || (bOk == FALSE))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    for (iIndex = 0; iIndex < psObj->un8MaxTuneMix; iIndex++)
    {
        // Only call function for existing TuneMixes
        if (psObj->ahTuneMix[iIndex] != TUNEMIX_INVALID_OBJECT)
        {
            // We found something
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

            bOk = pbIterator(psObj->ahTuneMix[iIndex], pvArg);
            if (bOk == FALSE)
            {
                break;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   DECODER_hGetSportsFlashHandle
 *
 *****************************************************************************/
SPORTS_FLASH_OBJECT DECODER_hSportsFlash (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    SPORTS_FLASH_OBJECT hSportsFlash = SPORTS_FLASH_INVALID_OBJECT;

    // Verify and make sure caller is already owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hSportsFlash = psObj->hSportsFlash;
    }

    return hSportsFlash;
}

/*****************************************************************************
 *
 *  DECODER_bSetSportsFlashHandle
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetSportsFlashHandle (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_OBJECT hSportsFlash
        )
{
    BOOLEAN bSuccess = FALSE, bOwner;

    // Verify and make sure caller is already owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if (hSportsFlash == SPORTS_FLASH_INVALID_OBJECT)
        {
            // If the handle is invalid, just clear out handle
            psObj->hSportsFlash = SPORTS_FLASH_INVALID_OBJECT;
            bSuccess = TRUE;
        }
        else if (psObj->hSportsFlash == SPORTS_FLASH_INVALID_OBJECT)
        {
            // Otherwise, if this decoder doesn't have a Sports Flash
            // handle yet, then we may set it now
            psObj->hSportsFlash = hSportsFlash;
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   DECODER_bSportsFlash
 *
 *  hDecoder - DECODER_OBJECT to operate on
 *  psEvent - Event data passing struct
 *
 *
 *****************************************************************************/
BOOLEAN DECODER_bSportsFlash(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_SPORTS_FLASH_STRUCT const *psEvent
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_SPORTS_FLASH,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sSportsFlash = *psEvent;
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bPlayFlashEvent
 *
 *****************************************************************************/
BOOLEAN DECODER_bPlayFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventID
        )
{
    BOOLEAN bOwner, bResult = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if ((bOwner == TRUE) && (tFlashEventID != SPORTS_FLASH_INVALID_EVENT_ID))
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        bResult = RADIO_bPlayFlashEvent(hDecoder, tFlashEventID);
        if (bResult == TRUE)
        {
            vUpdateTuneState(psObj, TUNE_STATE_TUNING_IN_PROGRESS);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   DECODER_bAbortFlashEvent
 *
 *****************************************************************************/
BOOLEAN DECODER_bAbortFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventID
        )
{
    BOOLEAN bOwner, bResult = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if ((bOwner == TRUE) && (tFlashEventID != SPORTS_FLASH_INVALID_EVENT_ID))
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        bResult = RADIO_bAbortFlashEvent(hDecoder, tFlashEventID);
        if (bResult == TRUE)
        {
            vUpdateTuneState(psObj, TUNE_STATE_TUNING_IN_PROGRESS);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   DECODER_bRemainFlashEvent
 *
 *****************************************************************************/
BOOLEAN DECODER_bRemainFlashEvent(
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_EVENT_ID tFlashEventID
        )
{
    BOOLEAN bOwner, bResult = FALSE;

    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if ((bOwner == TRUE) && (tFlashEventID != SPORTS_FLASH_INVALID_EVENT_ID))
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        bResult = RADIO_bRemainFlashEventChannel(hDecoder, tFlashEventID);
        if (bResult == TRUE)
        {
            vUpdateTuneState(psObj, TUNE_STATE_TUNING_IN_PROGRESS);
        }
    }

    return bResult;
}

/*****************************************************************************
 *
 *   DECODER_hGetTWNowHandle
 *
 *****************************************************************************/
TW_NOW_OBJECT DECODER_hGetTWNowHandle (
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner = FALSE;
    TW_NOW_OBJECT hTWNow = TW_NOW_INVALID_OBJECT;

    // Verify and make sure caller is already owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        hTWNow = psObj->hTWNow;
    }

    return hTWNow;
}

/*****************************************************************************
 *
 *  DECODER_bSetTWNowHandle
 *
 *****************************************************************************/
BOOLEAN DECODER_bSetTWNowHandle (
    DECODER_OBJECT hDecoder,
    TW_NOW_OBJECT hTWNow
        )
{
    BOOLEAN bSuccess = FALSE, bOwner = FALSE;

    // Verify and make sure caller is already owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if (bOwner == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if (hTWNow == TW_NOW_INVALID_OBJECT)
        {
            // If the handle is invalid, just clear out handle
            psObj->hTWNow = TW_NOW_INVALID_OBJECT;
            bSuccess = TRUE;
        }
        else if (psObj->hTWNow == TW_NOW_INVALID_OBJECT)
        {
            // Otherwise, if this decoder doesn't have a TW Now
            // handle yet, then we may set it now
            psObj->hTWNow = hTWNow;
            bSuccess = TRUE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   DECODER_bTWNow
 *
 *  hDecoder - DECODER_OBJECT to operate on
 *  psEvent - Event data passing struct
 *
 *
 *****************************************************************************/
BOOLEAN DECODER_bTWNow(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TW_NOW_STRUCT const *psEvent
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if((bValid == TRUE) && (psEvent != NULL))
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_TW_NOW,
            &puEventData, SMS_EVENT_OPTION_NONE);
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            // Extract event information and populate it
            puEventData->uDecoder.sTWNow = *psEvent;
            bPosted = SMSE_bPostEvent(hEvent);
        }
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   DECODER_bTWNowPlayAbort
 *
 *  hDecoder - DECODER_OBJECT to operate on
 *  tBulletinId - bulletin to play or abort
 *  bAbort - TRUE means that passed bulletin shall be aborted and
 *           FALSE - shall be played
 *
 *****************************************************************************/
BOOLEAN DECODER_bTWNowPlayAbort(
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId,
    BOOLEAN bAbort
        )
{
    BOOLEAN bResult;

    if (bAbort == TRUE)
    {
        bResult = RADIO_bAbortBulletin(
            hDecoder,
            tBulletinId
                );
    }
    else
    {
        bResult = RADIO_bPlayBulletin(
            hDecoder,
            tBulletinId
                );
    }

    if (bResult != FALSE)
    {
        // everything looks ok
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        // Update the tuner state to "In Progress"
        vUpdateTuneState(psObj, TUNE_STATE_TUNING_IN_PROGRESS);
    }

    return bResult;
}

/*****************************************************************************
 *
 *   DECODER_vUpdateTWNowStatus
 *
 *****************************************************************************/
void DECODER_vUpdateTWNowStatus (
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinId,
    BOOLEAN bTunningInProgress
        )
{
    TW_NOW_OBJECT hTWNow;

    hTWNow = DECODER_hGetTWNowHandle(hDecoder);
    if (hTWNow != TW_NOW_INVALID_OBJECT)
    {
        BOOLEAN bIsActive = FALSE,
                bChanged = FALSE;

        TW_NOW_vUpdateStatus(hTWNow, tBulletinId, &bIsActive, &bChanged);

        if ((bTunningInProgress == TRUE) &&
            (bChanged == TRUE) && (bIsActive == FALSE))
        {
            // Update the tuner state to "In Progress"
            vUpdateTuneState((DECODER_OBJECT_STRUCT*)hDecoder,
                TUNE_STATE_TUNING_IN_PROGRESS);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_vSelfTuneFallback
 *
 *****************************************************************************/
void DECODER_vSelfTuneFallback (
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId
        )
{
    BOOLEAN bSuccess = FALSE, bOwner = FALSE;
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    do
    {
        CHANNEL_OBJECT hChannel;
        CCACHE_OBJECT hCCache;
        SERVICE_ID tLastTunedServiceId, tNextServiceId;

        bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
        if (bOwner == FALSE)
        {
            // Error!
            break;
        }

        tLastTunedServiceId = DECODER_tLastTunedServiceId(hDecoder);
        if (tLastTunedServiceId != tServiceId)
        {
            // This is not for us.
            bSuccess = TRUE;
            break;
        }

        // This means that the Safe ServiceID is not available. Impossible...
        if (tServiceId == GsRadio.sChannel.tSafeServiceID)
        {
            // Error! 
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Safe Service Id is not available");
            break;
        }

        if (tServiceId != GsRadio.sChannel.tDefaultServiceID)
        {
            puts(DECODER_OBJECT_NAME": Trying default service id");
            tNextServiceId = GsRadio.sChannel.tDefaultServiceID;
        }
        else
        {
            puts(DECODER_OBJECT_NAME": Trying safe service id");
            tNextServiceId = GsRadio.sChannel.tSafeServiceID;
        }

        // get channnel cache handle
        hCCache = DECODER_hCCache(hDecoder);

        // Find a channel
        hChannel = CCACHE_hChannelFromIds(
            hCCache, 
            tNextServiceId,
            CHANNEL_INVALID_ID,
            TRUE);

        if (hChannel != CHANNEL_INVALID_OBJECT)
        {
            CATEGORY_OBJECT hCategory;

            hCategory = CHANNEL.hCategory(hChannel, 0);

            // Update the last tuned channel, 
            // to recursively continue the fallback.
            vUpdateHandles(
                psObj, 
                DECODER_UPDATE_HANDLE_LAST_TUNED, 
                hChannel, 
                hCategory
                    );

            bSuccess = RADIO_bRequestChannel(hDecoder, tNextServiceId);
            if (bSuccess == FALSE)
            {
                printf(DECODER_OBJECT_NAME
                    ": Failed to request Service ID: %d info\n",
                    tNextServiceId);

                // Not critical. 
                // Likely, this means a response delay.
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Failed to get the next candidate"
                " Service ID: %d for self tune",
                tNextServiceId);
            break;
        }

        bSuccess = TRUE;

    } while (FALSE);

    if ((bOwner == TRUE) && (bSuccess == FALSE))
    {
        // Error! Something went totally wrong...
        vSetError(psObj, DECODER_ERROR_CODE_GENERAL);
    }

    return;
}

/*****************************************************************************
 *
 *   DECODER_vUpdateAudioPresence
 *
 *****************************************************************************/
void DECODER_vUpdateAudioPresence (
    DECODER_OBJECT hDecoder,
    DECODER_AUDIO_PRESENCE_ENUM eAudioPresence
        )
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

    // Set new state information (if different than current)
    if(psObj->eAudioPresence != eAudioPresence)
    {
        // Update signal state
        psObj->eAudioPresence = eAudioPresence;

        // Set event mask
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_AUDIO_PRESENCE);
    }

    return;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*  psCreateObject
*
*****************************************************************************/
static DECODER_OBJECT_STRUCT *psCreateObject (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_ID tId,
    MODULE_OBJECT hModule,
    DECODER_EVENT_MASK tEventRequestMask,
    DECODER_OBJECT_EVENT_CALLBACK vEventCallback, 
    void *pvEventCallbackArg
        )
{
    DECODER_OBJECT_STRUCT *psObj = NULL;
    char *pacTempText;
    int iNum;

    // Compute length of the required name
    iNum = strlen(DECODER_OBJECT_NAME) + 1
                    + strlen(pacDecoderName) + 1;

    // Allocate enough memory for this name
    pacTempText = (char*) OSAL.pvMemoryAllocate(
        DECODER_OBJECT_NAME":Text", ++iNum, FALSE);
    if(pacTempText != NULL)
    {
        // Now actually construct the name
        snprintf( pacTempText, iNum,
            DECODER_OBJECT_NAME".%s", pacDecoderName );

        // Create an instance of this object
        psObj = (DECODER_OBJECT_STRUCT *)SMSO_hCreate(
            pacTempText,
            sizeof(DECODER_OBJECT_STRUCT) + iNum,
            SMS_INVALID_OBJECT, // Parent
            TRUE); // Lock
        if(psObj != NULL)
        {
            SMS_TASK_CONFIGURATION_STRUCT sConfig =
                gsDecoderTaskConfiguration;

            // Initialize object defaults
            *psObj = gsObjectDefaults;

            // Keep a copy of the object and decoder name.
            OSAL.bMemCpy((char *)(psObj + 1), pacTempText, iNum);

            // Point to the specific DECODER object and name
            psObj->pacObjectName = (char *)(psObj + 1);
            psObj->pacDecoderName = &psObj->pacObjectName[strcspn(
                &psObj->pacObjectName[0], ".") + 1];

            // Populate callback shim
            psObj->sCallbackShim.vEventCallback = vEventCallback;
            psObj->sCallbackShim.pvEventCallbackArg = pvEventCallbackArg;
            psObj->tRequestMask = tEventRequestMask;

            // Initialize asynchronous update configuration and
            // save inputs
            SMSU_vInitialize(
                &psObj->sEvent,
                psObj,
                DECODER_OBJECT_EVENT_ALL,
                tEventRequestMask,
                (SMSAPI_OBJECT_EVENT_CALLBACK)vEventCallbackShim,
                &psObj->sCallbackShim);

            // Configure decoder name
            sConfig.pacName = psObj->pacObjectName;

            // Install SMS Task for this decoder
            // Note, successful installation of this task will cause the
            // provided SMS object to be owned by the SMS task.
            psObj->hServicesTask = SMST_hInstall((SMS_OBJECT)psObj,
                &sConfig, (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)vEventHandler,
                &psObj->hEventHdlr);
            if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
            {
                // Success!
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                    "%s: DECODER task installed.\n", psObj->pacObjectName);
            }
            else
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME" Decoder task could not be installed.");

                vDestroyObject(psObj);
                psObj = NULL;
            }
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME" Could not create object.");
        }

        // Don't need this memory anymore
        OSAL.vMemoryFree(pacTempText);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME" Cannot allocate memory.");
    }

    return psObj;
}

/*****************************************************************************
*
*  vDestroyObject
*
*****************************************************************************/
static  void vDestroyObject (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOwner;

    // Check if this object is owned by the caller.
    bOwner = SMSO_bIsOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Destroy the async update configuration
        SMSU_vDestroy(&psObj->sEvent);

        // Destroy the  object itself
        SMSO_vDestroy((SMS_OBJECT)psObj);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Not owned object cannot be destroyed.");
    }

    return;
}

/*****************************************************************************
 *
 *   bInitializeDecoder
 *
 * This function is called each time the decoder enters the
 * READY state. Thus the code here must be prepared to initialize
 * or re-initialize everything necessary for the decoder's specific
 * capabilities.
 *
 *****************************************************************************/
static BOOLEAN bInitializeDecoder(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    // Initialize functionality common to all decoders...
    bOk = bInitializeGenericDecoder(psObj);
    if(bOk == FALSE)
    {
        // Error! We should not proceed
        return FALSE;
    }

    // Then initialize all specific decoder types that this
    // device supports

    /* Audio Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO) ==
        SRH_DEVICE_CAPABILITY_AUDIO)
    {
        bOk = bInitializeAudioDecoder(psObj);
        // Check for errors
        if(bOk == FALSE)
        {
            // Error! We should not proceed
            return FALSE;
        }
    }

    /* Video Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_VIDEO) ==
        SRH_DEVICE_CAPABILITY_VIDEO)
    {
        bOk = bInitializeVideoDecoder(psObj);
        // Check for errors
        if(bOk == FALSE)
        {
            // Error! We should not proceed
            return FALSE;
        }
    }

    /* Data Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_DATA) ==
        SRH_DEVICE_CAPABILITY_DATA)
    {
        bOk = bInitializeDataDecoder(psObj);
        // Check for errors
        if(bOk == FALSE)
        {
            // Error! We should not proceed
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitializeDecoder
 *
 * This function is called each time the decoder enters the
 * RELEASED state. Thus the code here must be prepared to uninitialize
 * everything necessary for the decoder's specific capabilities.
 *
 *****************************************************************************/
static void vUninitializeDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    // First uninitialize all specific decoder types that this
    // device supports

    /* Audio Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO) ==
        SRH_DEVICE_CAPABILITY_AUDIO)
    {
        vUninitializeAudioDecoder(psObj);
    }

    /* Video Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_VIDEO) ==
        SRH_DEVICE_CAPABILITY_VIDEO)
    {
        vUninitializeVideoDecoder(psObj);
    }

    /* Data Decoder */
    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_DATA) ==
        SRH_DEVICE_CAPABILITY_DATA)
    {
        vUninitializeDataDecoder(psObj);
    }

    // Then, uninitialize functionality common to all decoders...
    vUninitializeGenericDecoder(psObj);

    return;
}

/*****************************************************************************
 *
 *   bInitializeGenericDecoder
 *
 *   This function is called from the higher-level bInitializeDecoder function.
 *   All initialization performed here is necessary for all decoders
 *   regardless of their capabilities.
 *
 *****************************************************************************/
static BOOLEAN bInitializeGenericDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    if (psObj->apeAntennaState == NULL)
    {
        psObj->n8NumAntennas = RADIO_n8GetNumAntennas((DECODER_OBJECT)psObj);

        if (psObj->n8NumAntennas > 0)
        {
            // Allocate memory for the antenna state info
            psObj->apeAntennaState
                            = (ANTENNA_STATE_ENUM *)SMSO_hCreate(
                                DECODER_OBJECT_NAME":Antenna",
                                sizeof(ANTENNA_STATE_ENUM)
                                                * psObj->n8NumAntennas,
                                (SMS_OBJECT)psObj, // DECODER is parent,
                                FALSE // inherit lock-feature
                            );
            if(psObj->apeAntennaState == NULL)
            {
                // Error!
                return FALSE;
            }
            else
            {
                N8 n8Antenna;

                // Initialize
                for(n8Antenna = 0; n8Antenna < psObj->n8NumAntennas; n8Antenna++)
                {
                    ANTENNA_STATE_ENUM eAntennaState;

                    // Initialize antenna state
                    eAntennaState =
                        RADIO_eGetAntennaState((DECODER_OBJECT)psObj, n8Antenna);

                    // Update the antenna state
                    DECODER_vUpdateAntennaState(
                        (DECODER_OBJECT)psObj, n8Antenna, eAntennaState);
                }
            }
        }
    }

    // Check if we have a TAG for this object
    if(psObj->hTag == TAG_INVALID_OBJECT)
    {
        TAG_OBJECT hParentTag, hTag;
        SMSAPI_RETURN_CODE_ENUM eReturn;

        // get the module's config tag
        hParentTag = MODULE_hGetTag(psObj->hModule);

        // get this DECODER's config tag
        eReturn = TAG_eGet(DECODER_OBJECT_NAME, hParentTag, &hTag,
            psObj->pacDecoderName, TRUE); // create if it doesn't exist
        if((eReturn == SMSAPI_RETURN_CODE_SUCCESS) ||
                        (eReturn == SMSAPI_RETURN_CODE_CFG_NO_PARENT))
        {
            // got the tag, now store it for future use
            psObj->hTag = hTag;
        }
        else
        {
            // couldn't get or create this DECODER's tag
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitializeGenericDecoder
 *
 *   This function is called from the higher-level bInitializeDecoder function.
 *   All initialization performed here is necessary for all decoders
 *   regardless of their capabilities.
 *
 *****************************************************************************/
static void vUninitializeGenericDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    psObj->bBrowsed = FALSE;
    psObj->tBrowsedEventMask = CHANNEL_OBJECT_EVENT_NONE;
    psObj->bInDecoderCallback = FALSE;

    // Destroy Antenna State info
    if (psObj->apeAntennaState != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->apeAntennaState);
        psObj->apeAntennaState = NULL;
    }

    //Destroy the Engineering Data Object
    if (psObj->hEngineeringData != ENGINEERING_DATA_INVALID_OBJECT)
    {
        ENGINEERING_DATA_vDestroy(psObj->hEngineeringData);
        psObj->hEngineeringData = ENGINEERING_DATA_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
 *
 *   bInitializeAudioDecoder
 *
 *   This function is called from the higher-level bInitializeDecoder function.
 *   All initialization performed here is necessary for all decoders
 *   capable of providing audio services.
 *
 *****************************************************************************/
static BOOLEAN bInitializeAudioDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    BOOLEAN bSuccess;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bResult;

    // Initialize LEAGUE master lockable object
    bResult = LEAGUE_bSetMasterLockableObject((SMS_OBJECT)psObj);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
            DECODER_OBJECT_NAME
            ": failed to set DECODER as LEAGUE master lock object");
        // Error!
        return FALSE;
    }

    // Initialize Browse object
    if(psObj->hBrowse == BROWSE_INVALID_OBJECT)
    {
        psObj->hBrowse = BROWSE_hCreate((SMS_OBJECT)psObj,
            (DECODER_OBJECT)psObj);
        if(psObj->hBrowse == BROWSE_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
    }

    // Load the channel we were tuned to on the last power cycle.
    vLoadLastTunedServiceId(psObj);

    // Create a CME instance for this decoder,if it doesn't already exist
    // it would exist if this function is being called from reset.
    if(psObj->hCME == CME_INVALID_OBJECT)
    {
        psObj->hCME = CME_hCreate((DECODER_OBJECT)psObj);
        if(psObj->hCME == CME_INVALID_OBJECT)
        {
            // Error!
            return FALSE;
        }
    }

    // Retrieve locked channels which have been saved
    bSuccess = bRetrieveLockedChannels(psObj);
    if(bSuccess == FALSE)
    {
        return FALSE;
    }

    // Retrieve shipped channels which have been saved
    bSuccess = bRetrieveSkippedChannels(psObj);
    if(bSuccess == FALSE)
    {
        return FALSE;
    }

    // Retrieve unsubscribed channel text
    bSuccess = bRetrieveUnsubscribedText(psObj);
    if(bSuccess == FALSE)
    {
        return FALSE;
    }

    // Create a playback obj for this decoder, if it doesn't already exist
    // it would exist if this function is being called from reset
    if(psObj->hPlayback == PLAYBACK_INVALID_OBJECT)
    {
        BOOLEAN bIRSupported;

        // Check IR support by the Module
        bIRSupported = MODULE_bIsIRSupported(psObj->hModule);
        if (bIRSupported == TRUE)
        {
            // Create the object
            psObj->hPlayback = PLAYBACK_hCreate((DECODER_OBJECT)psObj);
            if(psObj->hPlayback != PLAYBACK_INVALID_OBJECT)
            {
                PLAYBACK_ERROR_CODE_ENUM eErrorCode;

                eErrorCode = RADIO_eSetPlaybackParams((DECODER_OBJECT)psObj,
                    GsRadio.sPlayback.un32DefaultWarningOffset);

                if (eErrorCode != PLAYBACK_ERROR_CODE_NONE)
                {
                    // inform playback of error
                    PLAYBACK_vSetError(psObj->hPlayback, eErrorCode);
                }
            }
        }
    }

    // Start subscription notification service, if it doesn't already exist
    // it would exist if this function is being called from reset
    if (psObj->hSubscriptionService == SUB_NOTIFICATION_INVALID_OBJECT)
    {
        eReturn = SUB_NOTIFICATION.eStart((DECODER_OBJECT)psObj);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Failed to start subscription service.");
            // we will still allow the decoder to start up even if the sub
            // notification service doesn't start since the service isn't
            // essential to operation of the decoder.
        }
        else
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: Started the subscription service.\n",
                psObj->pacObjectName);
        }
    }

    //Initialize the engineering data if we previously had one
    //otherwise create it
    if (psObj->hEngineeringData != ENGINEERING_DATA_INVALID_OBJECT)
    {
        ENGINEERING_DATA_vInitializeObject(psObj->hEngineeringData);
    }
    else
    {
        psObj->hEngineeringData =
            ENGINEERING_DATA_hCreate((DECODER_OBJECT)psObj);
    }

    if (psObj->hEngineeringData == ENGINEERING_DATA_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile,__LINE__,
                DECODER_OBJECT_NAME
                ": Failed to create Engineering Data Object");
    }
    else
    {
        if ( psObj->tRequestMask & DECODER_OBJECT_EVENT_ENGINEERING_DATA )
        {
            SMSAPI_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = MODULE_eModifyEngineeringData(
                                    psObj->hModule,
                                    (DECODER_OBJECT)psObj,
                                    SMSAPI_MODIFY_EVENT_MASK_ENABLE);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to modify engineering data");
            }
        }
    }

    // Request to subscribe to all of the data services available
    // If this post succeeds, we'll consider ourselves
    // an "assigned" subscriber
    psObj->bSubscribedToDS = DATASERVICE_MGR_bPostEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_DECODER_SUBSCRIBED,
        (void *)(DECODER_OBJECT)psObj);
    if (TRUE == psObj->bSubscribedToDS)
    {
        // Now we assume that Decoder is subscribed to the DSM.
        // Increment reference counter here. To decrement reference counter
        // DSM shall call MODULE_bRelease() to tell the Module
        // that the Module is not utilized anymore.

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: SUBSCRIBE request is posted to DSM.\n",
            psObj->pacObjectName);

        psObj->un16RefCounter++;
    }

    if (psObj->hACOTimer == OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eTimerCreate(
            &psObj->hACOTimer,
            DECODER_OBJECT_NAME":ACO:Timer",
            vFireACOEvent,
            (void *)psObj);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Failed to create ACO timer: %s (#%d)",
                    OSAL.pacGetReturnCodeName(eReturnCode), eReturnCode);
        }
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitializeAudioDecoder
 *
 *****************************************************************************/
static void vUninitializeAudioDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    // Stop minute ticker (if started)
    vStopMinuteTicker(psObj);

    // Get rid of channel/category references
    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED,
        CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);

    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_TUNED, CHANNEL_INVALID_OBJECT,
        CATEGORY_INVALID_OBJECT);

    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
        CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);

    // Destroy Playback
    if(psObj->hPlayback != PLAYBACK_INVALID_OBJECT)
    {
        PLAYBACK_vDestroy(psObj->hPlayback);
        psObj->hPlayback = PLAYBACK_INVALID_OBJECT;
    }

    // Destroy Content Monitoring Engine instance
    if(psObj->hCME != CME_INVALID_OBJECT)
    {
        CME_vDestroy(psObj->hCME);
        psObj->hCME = CME_INVALID_OBJECT;
    }

    // Destroy browse object
    if(psObj->hBrowse != BROWSE_INVALID_OBJECT)
    {
        BROWSE_vDestroy(psObj->hBrowse);
        psObj->hBrowse = BROWSE_INVALID_OBJECT;
    }

    if (psObj->hSubscriptionService != SUB_NOTIFICATION_INVALID_OBJECT)
    {
        SUB_NOTIFICATION.vStop((DECODER_OBJECT) psObj);
        psObj->hSubscriptionService = SUB_NOTIFICATION_INVALID_OBJECT;
    }

    if (psObj->hACOTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eTimerDelete(psObj->hACOTimer);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Failed to delete ACO timer");
        }
    }

    // Dereference DECODER as master lockable object for LEAGUES
    (void) LEAGUE_bSetMasterLockableObject(SMS_INVALID_OBJECT);

    return;
}

/*****************************************************************************
 *
 *   bInitializeVideoDecoder
 *
 *   This function is called from the higher-level bInitializeDecoder function.
 *   All initialization performed here is necessary for all decoders
 *   capable of providing video services.
 *
 *****************************************************************************/
static BOOLEAN bInitializeVideoDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    return FALSE;
}

/*****************************************************************************
 *
 *   vUninitializeVideoDecoder
 *
 *****************************************************************************/
static void vUninitializeVideoDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    return;
}

/*****************************************************************************
 *
 *   bInitializeDataDecoder
 *
 *   This function is called from the higher-level bInitializeDecoder function.
 *   All initialization performed here is necessary for all decoders
 *   capable of providing data services.
 *
 *****************************************************************************/
static BOOLEAN bInitializeDataDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitializeDataDecoder
 *
 *****************************************************************************/
static void vUninitializeDataDecoder(DECODER_OBJECT_STRUCT *psObj)
{
    return;
}

/*****************************************************************************
 *
 *   bInitializeObject
 *
 * This function gets called once to initialize anything the object
 * needs to initialize only once. This function runs to completion
 * before the decoder task/event handler leaves the initialize state.
 *
 *****************************************************************************/
static BOOLEAN bInitializeObject(DECODER_OBJECT_STRUCT *psObj)
{
    // Initialize object...

    // Initialize cached decoder data
    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED,
        CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);

    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_TUNED, CHANNEL_INVALID_OBJECT,
        CATEGORY_INVALID_OBJECT);

    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
        CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);

    psObj->bBrowsed = FALSE;
    psObj->tBrowsedEventMask = CHANNEL_OBJECT_EVENT_NONE;
    psObj->bInDecoderCallback = FALSE;

    // Initialize signal quality
    psObj->sSignalQuality.eComposite = SIGNAL_QUALITY_INVALID;
    psObj->sSignalQuality.eSatellite = SIGNAL_QUALITY_INVALID;
    psObj->sSignalQuality.eTerrestrial = SIGNAL_QUALITY_INVALID;
    psObj->sSignalQuality.eState = SIGNAL_STATE_UNKNOWN;

    // Initialize decoder state
    psObj->eState = DECODER_STATE_INITIAL;

    // Initialize decoder tune state
    psObj->eTuneState = TUNE_STATE_UNKNOWN;

    // Initialize decoder error code
    psObj->eErrorCode = DECODER_ERROR_CODE_NONE;

    // initialize update progress, map state and sub updates
    psObj->un8TotalUpdateProgress = 100;
    psObj->un8ReceiverUpdateProgress = 100;

    // Initialize Tune Scan
    psObj->sTuneScan = gsObjectDefaults.sTuneScan;

    // Initialize TuneMix
    psObj->un8MaxTuneMix = TUNEMIX_OBJECTS_NONE;
    psObj->hTuneMixActive = TUNEMIX_INVALID_OBJECT;
    OSAL.bMemSet(&psObj->ahTuneMix[0], 0, sizeof(psObj->ahTuneMix));

    // Initialize Audio Presence flag
    psObj->eAudioPresence = DECODER_AUDIO_PRESENCE_UNKNOWN;

    // Create a channel cache for this decoder (if one does not already exist_
    if(psObj->hCCache == CCACHE_INVALID_OBJECT)
    {
        psObj->hCCache = CCACHE_hCreate(
            (DECODER_OBJECT)psObj, psObj->pacObjectName);
        if(psObj->hCCache == CCACHE_INVALID_OBJECT)
        {
            return FALSE;
        }
    }

    // Read mask, thereby clearing (resetting) it
    SMSU_tMask(&psObj->sEvent);

    return TRUE;
}

/*****************************************************************************
 *
 *   vUninitObject
 *
 *****************************************************************************/
static void vUninitObject(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: Object uninitialization (%s)...\n",
        psObj->pacObjectName,
        psObj->bUninitialized == FALSE ? "initialized":
        "already uninitialized");

    // only do if we haven't already uninitialized
    if (psObj->bUninitialized == FALSE)
    {
        // don't call this function again, unless re-started
        psObj->bUninitialized = TRUE;

        // Perform DECODER uninitialization
        vUninitializeDecoder(psObj);

        /*
         * STOP User Services which were started
         */

        // Stop the Smart Favorites service if it is active
        if (psObj->hSmartFavorites != SMART_FAVORITES_INVALID_OBJECT)
        {
            SMART_FAVORITES.vStop((DECODER_OBJECT)psObj);
        }

        // Stop the presets service if it is active
        if(psObj->hPresets != PRESETS_INVALID_OBJECT)
        {
            PRESETS.eStop(psObj->hPresets);
            psObj->hPresets = PRESETS_INVALID_OBJECT;
        }

        // Stop the Artist/Title Seek service if it is active
        if(psObj->hArtistTitleSeek != SEEK_SERVICE_INVALID_OBJECT)
        {
            SEEK_vStop(psObj->hArtistTitleSeek);
            psObj->hArtistTitleSeek = SEEK_SERVICE_INVALID_OBJECT;
        }

        // Stop the Traffic/Weather Seek service if it is active
        if(psObj->hTrafficWeatherSeek != SEEK_SERVICE_INVALID_OBJECT)
        {
            SEEK_vStop(psObj->hTrafficWeatherSeek);
            psObj->hTrafficWeatherSeek = SEEK_SERVICE_INVALID_OBJECT;
        }

        // Stop the Sports Seek service if it is active
        if(psObj->hSportsSeek != SEEK_SERVICE_INVALID_OBJECT)
        {
            SEEK_vStop(psObj->hSportsSeek);
            psObj->hSportsSeek = SEEK_SERVICE_INVALID_OBJECT;
        }

        // Stop the Sports Zone service if it is active
        if(psObj->hSportZoneService != SPORT_ZONE_SERVICE_INVALID_OBJECT)
        {
            // stop the sport zone service (this will also set
            // psObj->hSportZoneService to SPORT_ZONE_SERVICE_INVALID_OBJECT)
            SPORT_ZONE_vStop(psObj->hSportZoneService);
        }

        // Destroy channel cache
        if(psObj->hCCache != CCACHE_INVALID_OBJECT)
        {
            CCACHE_vDestroy(psObj->hCCache);
            psObj->hCCache = CCACHE_INVALID_OBJECT;
        }

        // Destroy the SMS Update object
        SMSU_vDestroy(&psObj->sEvent);

        // Uninstall Satellite Module Services
        if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
        {
            SMS_TASK_HANDLE hServicesTask;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "Destroy DECODER Task...\n");

            // Task shutdown itself will destroy the object (psObj)
            // so there is no need to handle that here. This function
            // is called from the DECODER task context, so it will
            // request the SMS-Task to be deleted (shutdown). Once the
            // task is shutdown the last thing it does is destroy the
            // SMS-Object provided at the time of SMS-task creation.
            // This means you must not use psObj anymore after this call!
            hServicesTask = psObj->hServicesTask;
            psObj->hServicesTask = SMS_INVALID_TASK_HANDLE;
            SMST_vUninstall(hServicesTask);

            // Note: Since the thread which is calling this is the same
            // thread being shutdown, it is impossible for psObj to be
            // destroyed before the next event is handled (shutdown).
            // So we are free to return to the caller, handle more
            // events and then finally shutdown.
        }
    }

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "DECODER is destroyed.\n");

    return;
}

/*****************************************************************************
 *
 *   vUpdateState
 *
 *****************************************************************************/
static void vUpdateState(
    DECODER_OBJECT_STRUCT *psObj, DECODER_STATE_ENUM eNextState)
{
    DECODER_STATE_ENUM eCurrentState;

    // Populate with new state information...
    eCurrentState = psObj->eState;

    switch (eCurrentState)
    {
        case DECODER_STATE_INITIAL:
        {
            switch (eNextState)
            {
                case DECODER_STATE_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case DECODER_STATE_UPDATING:
                    eNextState = eTransitionToUpdatingState(psObj);
                break;

                case DECODER_STATE_READY:
                    eNextState = eTransitionToReadyState(psObj);
                break;

                case DECODER_STATE_RELEASED:
                    eNextState = eTransitionToReleasedState(psObj);
                break;

                case DECODER_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, psObj->eErrorCode);
                break;

                case DECODER_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case DECODER_STATE_UPDATING:
        {
            switch (eNextState)
            {
                case DECODER_STATE_INITIAL:
                    vTransitionFromUpdatingState(psObj);
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case DECODER_STATE_UPDATING:
                    // Do nothing.
                break;

                case DECODER_STATE_READY:
                    vTransitionFromUpdatingState(psObj);
                    eNextState = eTransitionToReadyState(psObj);
                break;

                case DECODER_STATE_RELEASED:
                    vTransitionFromUpdatingState(psObj);
                    eNextState = eTransitionToReleasedState(psObj);
                break;

                case DECODER_STATE_ERROR:
                    vTransitionFromUpdatingState(psObj);
                    eNextState =
                        eTransitionToErrorState(
                            psObj, psObj->eErrorCode);
                break;

                case DECODER_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case DECODER_STATE_READY:
        {
            switch (eNextState)
            {
                case DECODER_STATE_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case DECODER_STATE_UPDATING:
                    eNextState = eTransitionToUpdatingState(psObj);
                break;

                case DECODER_STATE_READY:
                    // Do nothing.
                break;

                case DECODER_STATE_RELEASED:
                    eNextState = eTransitionToReleasedState(psObj);
                break;

                case DECODER_STATE_ERROR:
                    eNextState =
                        eTransitionToErrorState(
                            psObj, psObj->eErrorCode);
                break;

                case DECODER_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case DECODER_STATE_RELEASED:
        {
            // Ignore. Can't go anywhere from here
            eNextState = eCurrentState;
        }
        break;

        case DECODER_STATE_ERROR:
        {
            switch (eNextState)
            {
                case DECODER_STATE_INITIAL:
                    eNextState = eTransitionToInitialState(psObj);
                break;

                case DECODER_STATE_RELEASED:
                    eNextState = eTransitionToReleasedState(psObj);
                break;

                case DECODER_STATE_ERROR:
                    // Do nothing.
                break;

                case DECODER_STATE_UPDATING:
                case DECODER_STATE_READY:
                case DECODER_STATE_INVALID:
                default:
                    // Ignore. Can't go there from here
                    eNextState = eCurrentState;
                break;
            }
        }
        break;

        case DECODER_STATE_INVALID:
        default:
            // Do nothing.
        break;
    }

    // Update internal state
    psObj->eState = eNextState;

    // Check if next state is different than the current state
    if(eCurrentState != eNextState)
    {
        // Set event mask to indicate state change
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_STATE);
    }

    return;
}

/*****************************************************************************
 *
 *   eTransitionToErrorState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eTransitionToErrorState(
    DECODER_OBJECT_STRUCT *psObj,
    DECODER_ERROR_CODE_ENUM eErrorCode
        )
{
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        DECODER_OBJECT_NAME": '%s' transitioned to %s due to error %s(%u).",
        psObj->pacDecoderName,
        MACRO_TO_STRING(DECODER_STATE_ERROR),
        SMSAPI_DEBUG_pacDecoderErrorCodeText(eErrorCode),
        eErrorCode
            );

    // Set local error code
    psObj->eErrorCode = eErrorCode;

    // Set current state as ERROR
    // TODO: BREAKPOINT
    return DECODER_STATE_ERROR;
}

/*****************************************************************************
 *
 *   eTransitionToInitialState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eTransitionToInitialState(
    DECODER_OBJECT_STRUCT *psObj)
{
    DECODER_STATE_ENUM eNextState = DECODER_STATE_INITIAL;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;
    BOOLEAN bPosted;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(DECODER_STATE_INITIAL));

    // Run specific RADIO uninitialization
    if (TRUE == psObj->bRadioInitialized)
    {
        RADIO_vUninitializeDecoder((DECODER_OBJECT)psObj);
        psObj->bRadioInitialized = FALSE;
    }

    // Re-Allocate our current event to post initialization
    hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_INITIALIZE,
        &puEventData, SMS_EVENT_OPTION_NONE);
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Post this event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            // Error!
            eNextState =
                eTransitionToErrorState(
                    psObj, DECODER_ERROR_CODE_EVENT_POST_ERROR);
        }
    }
    else
    {
        // Error!
        eNextState =
            eTransitionToErrorState(
                psObj, DECODER_ERROR_CODE_EVENT_ALLOCATION_ERROR);
    }

    return eNextState;
}

/*****************************************************************************
 *
 *   eTransitionToReadyState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eTransitionToReadyState(DECODER_OBJECT_STRUCT *psObj)
{
    DECODER_STATE_ENUM eNextState = DECODER_STATE_READY;
    BOOLEAN bSuccess;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName,  MACRO_TO_STRING(DECODER_STATE_READY));

    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO) ==
        SRH_DEVICE_CAPABILITY_AUDIO)
    {
        SUB_NOTIFICATION_vDecoderReady(psObj->hSubscriptionService);
    }

    // Init TuneMix
    bSuccess = bRetrieveTuneMixChannels(psObj);
    if(bSuccess == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Failed to retrieve TuneMix service.");
    }

    return eNextState;
}

/*****************************************************************************
 *
 *   eTransitionToReleasedState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eTransitionToReleasedState(
    DECODER_OBJECT_STRUCT *psObj)
{
    DECODER_STATE_ENUM eNextState = DECODER_STATE_RELEASED;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(DECODER_STATE_RELEASED));

    return eNextState;
}

/*****************************************************************************
 *
 *   eTransitionToUpdatingState
 *
 *****************************************************************************/
static DECODER_STATE_ENUM eTransitionToUpdatingState(
    DECODER_OBJECT_STRUCT *psObj)
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: Transition to %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(DECODER_STATE_UPDATING));

    // Clear browsed channel/category
    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED,
        CHANNEL_INVALID_OBJECT, CATEGORY_INVALID_OBJECT);

    // Set current state as UPDATING
    return DECODER_STATE_UPDATING;
}

/*****************************************************************************
 *
 *   vTransitionFromUpdatingState
 *
 *****************************************************************************/
static void vTransitionFromUpdatingState(DECODER_OBJECT_STRUCT *psObj)
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "%s: Transition from %s\n",
        psObj->pacObjectName, MACRO_TO_STRING(DECODER_STATE_UPDATING));

    if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO) ==
        SRH_DEVICE_CAPABILITY_AUDIO)
    {
        // Re-tune, but don't tune to any mature, locked, or skipped
        // channels.
        eTuneSelf(psObj, FALSE, FALSE, FALSE);
    }

    // Indicate update progress is 100%
    psObj->un8TotalUpdateProgress = 100;

    // tell app that the progress is 100%
    SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_UPDATE_PROGRESS);

    return;
}

/*******************************************************************************
 *
 *   vEventHandler
 *
 *   This function runs in the context of the Sirius Services Task
 *
 *******************************************************************************/
static void vEventHandler(DECODER_OBJECT hDecoder, SMS_EVENT_HDL hEvent)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    SMS_EVENT_TYPE_ENUM eEventType;
    SMS_EVENT_DATA_UNION const *puEventData;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        return;
    }

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    // Process event...
    switch (eEventType)
    {
        // The INITIALIZE event occurs one time only when the task is started
        // and completes before returning to the caller.
        case SMS_EVENT_INITIALIZE:
        {
            vHandleInitEvent(psObj);
        }
        break;

        // This event is received when the Module object itself becomes ready
        // or the Module object is already ready and successfully adds this
        // Decoder to it's management.
        case SMS_EVENT_ASSOCIATE:
        {
            vHandleAssociateEvent(psObj,
                &(puEventData->uDecoder.sAssociate));
        }
        break;

        // Event from Parent (MODULE) telling that MODULE
        // is not going to use this DECODER.
        case SMS_EVENT_UNASSOCIATE:
        {
            vHandleUnassociateEvent(psObj);
        }
        break;

        // This event is received when the physical Radio-DECODER itself
        // becomes ready.
        case SMS_EVENT_RADIO_READY:
        {
            vHandleRadioReadyEvent(psObj);
        }
        break;

        // Someone has requested this object release.
        // Only application or the upper level
        // object (a Module) can actually remove the Decoder.
        case SMS_EVENT_RELEASE:
        {
            vHandleReleaseEvent(psObj,
                puEventData->uDecoder.sRelease.eInitiator);
        }
        break;

        // Someone requested the DECODER to RESET
        case SMS_EVENT_RESET:
        {
            vHandleResetEvent(psObj);
        }
        break;

        // Final Stop event
        case SMS_EVENT_STOP:
        {
            vHandleStopEvent(psObj);
        }
        break;

        case SMS_EVENT_ERROR:
        {
            const DECODER_ERROR_CODE_ENUM eErrorCode =
                puEventData->uDecoder.sError.eErrorCode;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: SMS_EVENT_ERROR: eErrorCode: %d\n",
                psObj->pacObjectName, eErrorCode);

            // Error!
            vSetError(psObj, eErrorCode);
        }
        break;

        case SMS_EVENT_UPDATE_OBJECT:
        {
            const SMSAPI_EVENT_MASK tEventMask =
                puEventData->uDecoder.sUpdate.tEventMask;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: SMS_EVENT_UPDATE_OBJECT: tEventMask: 0x%X\n",
                psObj->pacObjectName, tEventMask);

            // Force an update of all that was requested
            SMSU_tUpdate(&psObj->sEvent, tEventMask);
        }
        break;

        case SMS_EVENT_UPDATE_TIME:
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_UPDATE_TIME\n", psObj->pacObjectName);

            // Set event mask
            // Force an update of all that was requested
            SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_TIME);
        }
        break;

        case SMS_EVENT_SYNCHRONIZE_TIME:
        {
            BOOLEAN bStarted;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SYNCHRONIZE_TIME\n", psObj->pacObjectName);

            // Start/Re-Start minute timer since clock was adjusted
            // in some significant way.
            bStarted = bStartMinuteTicker(psObj);
            if(bStarted == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to re-sync timer.");

                // Transition to error state
                vSetError(psObj, DECODER_ERROR_CODE_TIMER_ERROR);
            }
        }
        break;

        case SMS_EVENT_PLAY:
        {
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_PLAY\n", psObj->pacObjectName);

            // Playback commands are ignored during the scans
            if (psObj->sTuneScan.bActive == FALSE)
            {
                eResult = RADIO_ePlay(hDecoder);
                if(eResult == PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED)
                {
                    // inform playback of error
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Play is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_PAUSE:
        {
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_PAUSE\n", psObj->pacObjectName);

            // Playback commands are ignored during the scan
            if (psObj->sTuneScan.bActive == FALSE)
            {
                eResult = RADIO_ePause(hDecoder);
                if(eResult == PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED)
                {
                    // inform playback of error
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Pause is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_SEEK_TIME:
        {
            const SMS_EVENT_SEEK_TIME_STRUCT sSeekTime =
                puEventData->uDecoder.sSeekTime;
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SEEK_TIME: n32Offset: %d,"
                " bPauseAfterSeek: %d\n",
                psObj->pacObjectName, sSeekTime.n32Offset,
                sSeekTime.bPauseAfterSeek);

            // Playback commands are ignored during the scan
            if (psObj->sTuneScan.bActive == FALSE)
            {
                // seek time
                eResult = RADIO_eSeek(hDecoder, RADIO_SEEK_TYPE_TIME,
                    sSeekTime.n32Offset, sSeekTime.bPauseAfterSeek);
                if(eResult == PLAYBACK_ERROR_CODE_NONE)
                {
                    // see if the time shift took us back to "live"
                    // i.e. (not paused and timeoffset == 0)
                    PLAYBACK_vUpdatePause(psObj->hPlayback,
                        sSeekTime.bPauseAfterSeek);
                }
                else
                {
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Seek Time is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_SEEK_SONG:
        {
            const SMS_EVENT_SEEK_SONG_STRUCT sSeekSong =
                puEventData->uDecoder.sSeekSong;
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SEEK_SONG: n16Offset: %d,"
                " bPauseAfterSeek: %d\n",
                psObj->pacObjectName, sSeekSong.n16Offset,
                sSeekSong.bPauseAfterSeek);

            // Playback commands are ignored during the scan
            if (psObj->sTuneScan.bActive == FALSE)
            {
                // Seek song
                eResult = RADIO_eSeek(hDecoder, RADIO_SEEK_TYPE_SONG,
                    sSeekSong.n16Offset,
                    sSeekSong.bPauseAfterSeek);
                if(eResult == PLAYBACK_ERROR_CODE_NONE)
                {
                    // update the play/pause status
                    // see if the time shift took us back to "live"
                    // i.e. (not paused and timeoffset == 0)
                    PLAYBACK_vUpdatePause(psObj->hPlayback,
                        sSeekSong.bPauseAfterSeek);
                }
                else
                {
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Seek Song is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_SEEK_PREVIOUS:
        {
            SMS_EVENT_SEEK_PROXIMAL_STRUCT sSeekProximal =
                puEventData->uDecoder.sSeekProximal;
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SEEK_PREVIOUS: bPauseAfterSeek: %d\n",
                psObj->pacObjectName, sSeekProximal.bPauseAfterSeek);

            // Playback commands are ignored during the scan
            if (psObj->sTuneScan.bActive == FALSE)
            {
                // Seek Previous
                eResult = RADIO_eSeek(hDecoder, RADIO_SEEK_TYPE_PREVIOUS, 0,
                    sSeekProximal.bPauseAfterSeek);
                if(eResult == PLAYBACK_ERROR_CODE_NONE)
                {
                    // update the play/pause status
                    // see if the time shift took us back to "live"
                    // i.e. (not paused and timeoffset == 0)
                    PLAYBACK_vUpdatePause(psObj->hPlayback,
                        sSeekProximal.bPauseAfterSeek);
                }
                else
                {
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Seek Previous is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_SEEK_NEXT:
        {
            SMS_EVENT_SEEK_PROXIMAL_STRUCT sSeekProximal =
                puEventData->uDecoder.sSeekProximal;
            PLAYBACK_ERROR_CODE_ENUM eResult;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SEEK_NEXT: bPauseAfterSeek: %d\n",
                psObj->pacObjectName, sSeekProximal.bPauseAfterSeek);

            // Playback commands are ignored during the scan
            if (psObj->sTuneScan.bActive == FALSE)
            {
                // Seek Next
                eResult = RADIO_eSeek(hDecoder, RADIO_SEEK_TYPE_NEXT, 0,
                    sSeekProximal.bPauseAfterSeek);
                if(eResult == PLAYBACK_ERROR_CODE_NONE)
                {
                    // update the play/pause status
                    // see if the time shift took us back to "live"
                    // i.e. (not paused and timeoffset == 0)
                    PLAYBACK_vUpdatePause(psObj->hPlayback,
                        sSeekProximal.bPauseAfterSeek);
                }
                else
                {
                    PLAYBACK_vSetError(psObj->hPlayback, eResult);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "%s: Seek Next is ignored during the Scan.\n",
                    psObj->pacObjectName);
            }
        }
        break;

        case SMS_EVENT_FEATURED_FAVORITES:
        {
            const SMS_EVENT_FEATURED_FAVORITES_STRUCT sFeaturedFavorites =
                puEventData->uDecoder.sFeaturedFavorites;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_FEATURED_FAVORITES\n",
                psObj->pacObjectName);

            // Based on what we were told, call the right function
            if(sFeaturedFavorites.bEnable == TRUE)
            {
                PRESETS_vFavoritesStart(psObj->hPresets,
                    &sFeaturedFavorites.uEvent.sEnable);
            }
            else
            {
                PRESETS_vFavoritesStop(psObj->hPresets,
                    &sFeaturedFavorites.uEvent.sDisable);
            }
        }
        break;

        case SMS_EVENT_SMART_FAVORITES:
        {
            const SMS_EVENT_SMART_FAVORITES_STRUCT sSmartFavorites =
                puEventData->uDecoder.sSmartFavorites;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SMART_FAVORITES\n",
                psObj->pacObjectName);

            SMART_FAVORITES_bEventHandler(
                hDecoder,
                psObj->hSmartFavorites,
                &sSmartFavorites
                    );
        }
        break;

        case SMS_EVENT_TUNE_SCAN:
        {
            SMS_EVENT_TUNE_SCAN_STRUCT sTuneScan =
                puEventData->uDecoder.sTuneScan;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TUNE_SCAN\n",
                psObj->pacObjectName);

            vTuneScanEventHandler(
                psObj,
                &sTuneScan
                    );
        }
        break;

        case SMS_EVENT_ALL_CHAN_CAT_NOTIFY:
        {
            const BOOLEAN bNotifyChannels =
                puEventData->uDecoder.sAllChanCatNotify.bNotifyChannels;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_ALL_CHAN_CAT_NOTIFY\n",
                psObj->pacObjectName);

            vHandleAllChanCatNotificationEvent(hDecoder, bNotifyChannels);
        }
        break;

#if SMS_DEBUG == 1
        case SMS_EVENT_PRINT_SCACHE:
        {
            const SMS_EVENT_PRINT_SCACHE_STRUCT sPrintSCache =
                puEventData->uDecoder.sPrintSCache;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_PRINT_SCACHE\n",
                psObj->pacObjectName);

            // tell playback to print songs
            PLAYBACK_vPrintSongs(puEventData->uDecoder.sPrintSCache.hPlayback,
                sPrintSCache.uSCache.sPrint.bVerbose);
        }
        break;
#endif /* SMS_DEBUG == 1 */

        case SMS_EVENT_CREATE_SONGLIST:
        {
            const SONGLIST_OBJECT hSonglist =
                puEventData->uDecoder.sSonglist.hSonglist;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_CREATE_SONGLIST\n",
                psObj->pacObjectName);

            // tell playback to create the songlist
            PLAYBACK_vProcessCreateSongList(psObj->hPlayback, hSonglist);
        }
        break;

        case SMS_EVENT_DESTROY_SONGLIST:
        {
            const SONGLIST_OBJECT hSonglist =
                puEventData->uDecoder.sSonglist.hSonglist;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_DESTROY_SONGLIST\n",
                psObj->pacObjectName);

            // tell playabck to create the songlist
            PLAYBACK_vProcessDestroySongList(psObj->hPlayback, hSonglist);
        }
        break;

        case SMS_EVENT_CCACHE:
        {
            const SMS_EVENT_CCACHE_STRUCT sCache =
                puEventData->uDecoder.sCache;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_CACHE\n",
                psObj->pacObjectName);

            CCACHE_vEventHandler(psObj->hCCache, &sCache);
        }
        break;

        case SMS_EVENT_CHANNEL:
        {
            const SMS_EVENT_CHANNEL_STRUCT sChannel =
                puEventData->uDecoder.sChannel;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_CHANNEL\n",
                psObj->pacObjectName);

            vSetChannelAttrs(psObj, sChannel.bLocked, sChannel.bSkipped);
        }
        break;

        case SMS_EVENT_MODIFY_EVENT_MASK:
        {
            const SMS_EVENT_DECODER_EVENT_MASK_STRUCT sDecoderEventStruct =
                puEventData->uDecoder.sDecoderEventStruct;
            BOOLEAN bSuccess;

            bSuccess = bProcessDecoderEventMaskChange(psObj, &sDecoderEventStruct);
            if(bSuccess == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Could not process decoder"
                    " event mask change:");

                // Error!
                vSetError(psObj, DECODER_ERROR_CODE_GENERAL);
            }
        }
        break;

        case SMS_EVENT_MODIFY_CHANLIST_EVENT_MASK:
        {
            SMS_EVENT_CHANLIST_EVENT_MASK_STRUCT const *psChanlistEventStruct =
                &puEventData->uDecoder.sChanlistEventStruct;
            BOOLEAN bSuccess;

            bSuccess = CHANNELLIST_bProcessChannelListEventMaskChange(
                psChanlistEventStruct->hChannelList,
                psChanlistEventStruct->tChannelEventMask,
                psChanlistEventStruct->eModification
                    );
            if(bSuccess == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Could not process channellist"
                    " event mask change:");

                // Error!
                vSetError(psObj, DECODER_ERROR_CODE_GENERAL);
            }
        }
        break;

        case SMS_EVENT_TONE_GENERATION:
        {
            SMS_EVENT_TONE_GENERATION_STRUCT const *psToneGeneration =
                &puEventData->uDecoder.sToneGeneration;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TONE_GENERATION\n",
                psObj->pacObjectName);

            if (psToneGeneration->eAction == DECODER_TONE_GEN_START)
            {
                RADIO_eToneGenerationStart(hDecoder,
                    psToneGeneration->un32ToneFreqHz,
                    psToneGeneration->n8Volume,
                    psToneGeneration->eBalance,
                    FALSE); // not an alert
            }
            else if (psToneGeneration->eAction == DECODER_TONE_GEN_ALERT)
            {
                DECODER_eAlertTone(
                    hDecoder, psToneGeneration->n8Volume);
            }
            else if (psToneGeneration->eAction == DECODER_TONE_GEN_STOP)
            {
                RADIO_eToneGenerationStop(hDecoder);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unknown Tone Gen Action.");
            }
        }
        break;

        case SMS_EVENT_AUDIO_CONTROL:
        {
            SMS_EVENT_AUDIO_CONTROL_STRUCT const *psAudioControl =
                &puEventData->uDecoder.sAudioControl;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_AUDIO_CONTROL\n",
                psObj->pacObjectName);

            if (psAudioControl->eAction == DECODER_AUDIO_CONTROL_VOLUME)
            {
                // adjust mute
                RADIO_eAdjustVolume(hDecoder, psAudioControl->n16Level);
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME
                        ": Unknown or unhandled audio control action (%u).",
                    psAudioControl->eAction);
            }
        }
        break;

        case SMS_EVENT_METADATA_WAIT_TIMEOUT:
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_METADATA_WAIT_TIMEOUT\n",
                psObj->pacObjectName);

            SUB_NOTIFICATION_vMetadataTimeout(psObj->hSubscriptionService);
        }
        break;

        case SMS_EVENT_TEXT_REPLACEMENT_TIMEOUT:
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TEXT_REPLACEMENT_TIMEOUT\n",
                psObj->pacObjectName);

            SUB_NOTIFICATION_vTextReplacementTimeout(psObj->hSubscriptionService);
        }
        break;

        case SMS_EVENT_METADATA_CHANGED:
        {
             SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_METADATA_CHANGED\n",
                psObj->pacObjectName);

             SUB_NOTIFICATION_vMetaDataChanged(psObj->hSubscriptionService);
        }
        break;

        case SMS_EVENT_ART:
        {
            const SMS_EVENT_ART_STRUCT sArt =
                puEventData->uDecoder.sArt;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_ART\n", psObj->pacObjectName);

            // Pass event to channel art module
            CHANNEL_ART_vDecoderEventHandler(hDecoder, &sArt);
        }
        break;

        case SMS_EVENT_EPG:
        {
            const SMS_EVENT_EPG_STRUCT sEpg =
                puEventData->uDecoder.sEpg;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_EPG\n", psObj->pacObjectName);

            // Pass event to EPG Module
            EPG_MGR_vDecoderEventHandler(hDecoder, &sEpg);
        }
        break;

        case SMS_EVENT_BROWSE:
        {
            const SMS_EVENT_BROWSE_STRUCT sBrowse =
                puEventData->uDecoder.sBrowse;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_BROWSE\n", psObj->pacObjectName);

            vBrowseEventHandler(psObj, &sBrowse);
        }
        break;

        case SMS_EVENT_TUNE:
        {
            const SMS_EVENT_TUNE_STRUCT sTune =
                puEventData->uDecoder.sTune;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TUNE\n", psObj->pacObjectName);

            bTuneServiceId(psObj, &sTune);
        }
        break;

        case SMS_EVENT_TUNEMIX_CONFIGURE:
        {
            const SMS_EVENT_TUNEMIX_STRUCT sTune =
                puEventData->uDecoder.sTuneMix;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TUNEMIX_CONFIGURE\n", psObj->pacObjectName);

            bConfigureAttrList(hDecoder, &sTune);
        }
        break;

        case SMS_EVENT_SPORTS_FLASH:
        {
            const SMS_EVENT_SPORTS_FLASH_STRUCT sSportsFlash =
                puEventData->uDecoder.sSportsFlash;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_SPORTS_FLASH\n",
                psObj->pacObjectName);

            SPORTS_FLASH_bEventHandler(
                psObj->hSportsFlash,
                &sSportsFlash
                    );
        }
        break;

        case SMS_EVENT_TW_NOW:
        {
            const SMS_EVENT_TW_NOW_STRUCT sTWNow =
                puEventData->uDecoder.sTWNow;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: SMS_EVENT_TW_NOW\n",
                psObj->pacObjectName);

            TW_NOW_bDecoderEventHandler(
                hDecoder,
                psObj->hTWNow,
                &sTWNow
                    );
        }
        break;

        default:
        {
            // Un-handled event, pass on to RADIO specific handler
            RADIO_bDecoderEventHandler(
                hDecoder, eEventType, puEventData);
        }
        break;

    } // switch(eEventType)

    // Notify callback if registered and change occurred.
    SMSU_bNotify(&psObj->sEvent);

    // Trigger any CME events which have occurred.
    CME_vTriggerEvents(psObj->hCME, TRUE);

    // Notify callback if registered and SEEK alerts were
    // generated from CME events
    SMSU_bNotify(&psObj->sEvent);

    // Check if we need to destroy the object. This means not only
    // have we been told to STOP but also no existing modes of operation exist.
    if(psObj->eState == DECODER_STATE_RELEASED)
    {
        // Uninitialize Object
        vUninitObject(psObj);
    }

    // Unlock object, any modifications to be made to this object
    // have already been made.
    SMSO_vUnlock((SMS_OBJECT)hDecoder);

    // Event handled, simply return
    return;
}

/*****************************************************************************
 *
 *   eBrowseLocal
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eBrowseLocal (
    DECODER_OBJECT hDecoder,
    SMS_EVENT_BROWSE_STRUCT const *psBrowse
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bValid;

    // Check validity of object
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
        SMS_EVENT_HDL hEvent;
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
            SMS_EVENT_BROWSE, &puEventData, SMS_EVENT_OPTION_NONE);

        //Attempt to post the event
        if(hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Populate event
            puEventData->uDecoder.sBrowse = *psBrowse;

            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to post browse event.");
            }
            else
            {
                // All is well
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to allocate browse event.");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to lock DECODER.");
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   vBrowseEventHandler
 *
 *****************************************************************************/
static void vBrowseEventHandler (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_BROWSE_STRUCT const *psBrowse
        )
{
    BROWSE_TYPE_ENUM eBrowseType;
    BOOLEAN bOk = TRUE;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;
    BOOLEAN bBrowseChannel;

    // Get the current browse type
    eBrowseType = BROWSE_eGetMode(psObj->hBrowse);

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
        "Browse Request: [ Type: %d, Direction: %d, Id: %d ];"
        " Current Browse Type: %d\n",
        psBrowse->eBrowseType, psBrowse->eDirection,
        psBrowse->uId.tCategory, eBrowseType
            );

    // Request browse of channels?
    if(psBrowse->eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
    {
        // Configure what we are browsing
        bBrowseChannel = TRUE;

        // Are we using the correct browse type?
        if(eBrowseType != BROWSE_TYPE_ALL_CHANNELS)
        {
            // No, we need to switch

            // Get the current browsed Ids
            tChannelId = CHANNEL.tChannelId(psObj->hBrowsedChannel);
            tCategoryId = CATEGORY.tGetCategoryId(psObj->hBrowsedCategory);

            // Ensure we got a valid channel Id
            if(tChannelId != CHANNEL_INVALID_ID)
            {
                // Update the browsing mode, but keep
                // the same references
                bOk = BROWSE_bSetMode(
                    psObj->hBrowse, BROWSE_TYPE_ALL_CHANNELS,
                    tChannelId, tCategoryId);
                if(bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Cannot set browse mode.");
                }
            }
        }

        // Set-up browse
        if(psBrowse->uId.tChannel == CHANNEL_INVALID_ID)
        {
            // Use currently browsed channel
            tChannelId
                = CHANNEL.tChannelId(psObj->hBrowsedChannel);
        }
        else
        {
            // Use selected channel
            tChannelId = psBrowse->uId.tChannel;
        }
    }
    // Request browse of categories?
    else if(psBrowse->eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        // Configure what we are browsing
        bBrowseChannel = FALSE;

        // Are we using the correct browse type?
        if(eBrowseType != BROWSE_TYPE_CATEGORY)
        {
            // No, we need to switch

            // Get the current browsed Ids
            tChannelId = CHANNEL.tChannelId(psObj->hBrowsedChannel);
            tCategoryId = CATEGORY.tGetCategoryId(psObj->hBrowsedCategory);

            // We must at least have a valid channel
            if (tChannelId != CHANNEL_INVALID_ID)
            {
                // Update the browsing mode, but keep
                // the same references
                bOk = BROWSE_bSetMode(psObj->hBrowse, BROWSE_TYPE_CATEGORY,
                    tChannelId, tCategoryId);
                if(bOk == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Cannot set browse mode.");
                }
            }
        }

        // Set-up browse
        if(psBrowse->uId.tCategory == CATEGORY_INVALID_ID)
        {
            // Use current browse category
            tCategoryId
                = CATEGORY.tGetCategoryId(psObj->hBrowsedCategory);
        }
        else
        {
            // Use selected category
            tCategoryId = psBrowse->uId.tCategory;
        }

        if (tCategoryId != CATEGORY_INVALID_ID)
        {
            size_t tSize;

            tSize = CATEGORY.tSize((DECODER_OBJECT)psObj, tCategoryId);
            // If requested category is empty, we try to browse to
            // default channel's category
            if (tSize == 0)
            {
                CHANNEL_OBJECT hChannel;
                SERVICE_ID tServiceId;

                // Getting default of safe channel's object
                tServiceId = tGetDefaultOrSafeServiceId(psObj);
                hChannel = CCACHE_hChannelFromIds(
                    psObj->hCCache,
                    tServiceId,
                    CHANNEL_INVALID_ID,
                    FALSE);

                // Channel is found.
                if (hChannel != CHANNEL_INVALID_OBJECT)
                {
                    CATEGORY_OBJECT hCategory;

                    tChannelId = CHANNEL.tChannelId(hChannel);
                    hCategory = CHANNEL.hCategory(hChannel, 0);

                    tCategoryId = CATEGORY.tGetCategoryId(hCategory);
                    if (tCategoryId == CATEGORY_INVALID_ID)
                    {
                        // Browse by channel, since category does not exist
                        bBrowseChannel = TRUE;
                    }
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Failed to get handle of Default / Safe channel");
                    bOk = FALSE;
                }
            }
        }
    }
    else // Unknown browse request
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unknown browse request.");
        return;
    }

    // If mode was set, or previously ok
    if(bOk == TRUE)
    {
        // Perform the browse request
        bOk = BROWSE_bBrowse(psObj->hBrowse, psBrowse->eDirection,
            bBrowseChannel, &tChannelId, &tCategoryId, FALSE);
        if(bOk == TRUE)
        {
            CHANNEL_OBJECT hChannel;
            CATEGORY_OBJECT hCategory;

            // Get the category handle by Id
            hCategory = CCACHE_hCategory(
                psObj->hCCache, &tCategoryId, 0);

            // Get the channel handle by Id
            hChannel = CCACHE_hChannelFromIds(
                    psObj->hCCache, SERVICE_INVALID_ID,
                    tChannelId, FALSE);

            // Update the browsed object handles
            vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_BROWSED, hChannel,
                hCategory);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vUpdateTuneState
 *
 *****************************************************************************/
static void vUpdateTuneState(
    DECODER_OBJECT_STRUCT *psObj, TUNE_STATE_ENUM eNextState)
{
    // Populate with new state information...
    TUNE_STATE_ENUM eCurrentState = psObj->eTuneState;

    // Set new state information (if different than current)
    if(eCurrentState != eNextState)
    {
        // Update state
        psObj->eTuneState = eNextState;

        // Update the event
        SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_TUNE);
    }

    return;
}

/*****************************************************************************
 *
 *   eQualifyTuneRequest
 *
 *  This function will take as input a channel-id and/or service-id
 *  and based on what is possible return the service-id of a channel
 *  which is available.
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eQualifyTuneRequest(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT *psTune
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    CHANNEL_OBJECT hChannel;

    if((psTune->tChannelId == CHANNEL_INVALID_ID) &&
        (psTune->tServiceId == SERVICE_INVALID_ID))
    {
        // We have no idea what to tune, so let's
        // tune the currently browsed channel.
        hChannel = psObj->hBrowsedChannel;
        psTune->tServiceId = CHANNEL.tServiceId(hChannel);
    }
    else
    {
        // Use another channel based on provided input.
        // This function picks service-id over channel-id
        hChannel = CCACHE_hChannelFromIds(
            psObj->hCCache, psTune->tServiceId,
            psTune->tChannelId, FALSE);
    }

    // Check if we now have a valid CHANNEL handle
    if(hChannel == CHANNEL_INVALID_OBJECT)
    {
        eReturnCode = SMSAPI_RETURN_CODE_INVALID_CHANNEL;
    }
    else
    {
        BOOLEAN bChannelLocked, bChannelMature;
        TUNEMIX_OBJECT hTuneMix;

        // Extract service-id and channel-id for whatever channel
        // we have decided to work with.
        psTune->tServiceId = CHANNEL.tServiceId(hChannel);
        psTune->tChannelId = CHANNEL.tChannelId(hChannel);

        // is it a TuneMix channel?
        hTuneMix = DECODER_hGetTuneMix((DECODER_OBJECT)psObj, 
            psTune->tChannelId);

        if (TUNEMIX_INVALID_OBJECT != hTuneMix)
        {
            eReturnCode = TUNEMIX_eQualified(hTuneMix);
        }
        else
        {
            // is the channel locked?
            eReturnCode =
                CHANNEL.eIsLocked(hChannel, &bChannelLocked);
            if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // is the channel mature?
                eReturnCode =
                    CHANNEL.eIsMature(hChannel, &bChannelMature);

                if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    if((bChannelLocked == TRUE) &&
                        (psTune->bLockedOverride == FALSE))
                    {
                        // the channel is locked and caller did not 
                        // override it, so we can't tune
                        eReturnCode = SMSAPI_RETURN_CODE_LOCKED_CHANNEL;
                    }
                    else if((bChannelMature == TRUE) &&
                        (psTune->bMatureOverride == FALSE))
                    {
                        // the channel is mature and caller did not
                        // override it, so we can't tune
                        eReturnCode = SMSAPI_RETURN_CODE_MATURE_CHANNEL;
                    }
                }
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   bPostTune
 *
 *****************************************************************************/
static BOOLEAN bPostTune(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT const *psTune
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
        SMS_EVENT_TUNE, &puEventData, SMS_EVENT_OPTION_NONE);

    //Attempt to post the event
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        puEventData->uDecoder.sTune = *psTune;

        bPosted = SMSE_bPostEvent(hEvent);
        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to post tune event");
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to allocate tune event");
    }

    return bPosted;
}

/*****************************************************************************
 *
 *   bTuneServiceId
 *
 *****************************************************************************/
static BOOLEAN bTuneServiceId(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT const *psTune
        )
{
    BOOLEAN bTuneServiceId;

    bTuneServiceId =
        RADIO_bTuneServiceId(
            (DECODER_OBJECT)psObj, psTune->tServiceId,
            psTune->bMatureOverride, psTune->bLockedOverride,
            psTune->bSkippedOverride, psTune->bPlayUnrestricted
                );
    if(bTuneServiceId == TRUE)
    {
        // everything looks ok

        // Update the tuner state to "In Progress"
        vUpdateTuneState(psObj, TUNE_STATE_TUNING_IN_PROGRESS);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to tune service-id (%u).",
                psTune->tServiceId);

        // Tune did not complete.
        vSetError(psObj, DECODER_ERROR_CODE_RADIO_ERROR);
    }

    return bTuneServiceId;
}

/*****************************************************************************
 *
 *   bConfigureAttrList
 *
 *****************************************************************************/
static BOOLEAN bConfigureAttrList(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TUNEMIX_STRUCT const *psTuneMix
        )
{
    BOOLEAN bOk = TRUE;
    CHANNEL_ID tChannelId;
    TUNEMIX_STATUS_ENUM eStatus = TUNEMIX_STATUS_AVAILABLE;
    TUNEMIX_COMPONENT_ITERATOR_STRUCT sTuneMixComponents;

    tChannelId = psTuneMix->un8ChIndex + TUNEMIX_CID_OFFSET;

    // If object handle is set, it means we configure the list of 
    // components for existing Tune Mix object
    if (psTuneMix->hTuneMix != TUNEMIX_INVALID_OBJECT)
    {
        sTuneMixComponents.un8Count = 0;
        TUNEMIX.eIterateChannels(psTuneMix->hTuneMix, 
            bCountTuneMixComponents, &sTuneMixComponents.un8Count);
        if (sTuneMixComponents.un8Count < TUNEMIX_CHANNELS_MIN)
        {
            eStatus = TUNEMIX_STATUS_UNAVAILABLE;
            sTuneMixComponents.un8Count = 0;
        }
    }
    else
    {
        // Object handle is INVALID, means the object is removed and 
        // we need to clear the list by Channel ID.
        eStatus = TUNEMIX_STATUS_UNAVAILABLE;
        sTuneMixComponents.un8Count = 0;
    }

    bOk = RADIO_bTuneMixConfigure(
        hDecoder,
        &sTuneMixComponents.atServiceId[0],
        // the number of desired music channels
        sTuneMixComponents.un8Count,
        // AttributeChangeType
        psTuneMix->un8ChIndex);

    TUNEMIX_vUpdateTuneMixStatus(hDecoder,
        tChannelId, FALSE, eStatus);

    if(bOk == TRUE)
    {
        // Save configuration into tags here
        bOk = TUNEMIX_bSaveConfiguration(
            hDecoder,
            tChannelId,
            &sTuneMixComponents.atServiceId[0],
            sTuneMixComponents.un8Count);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to configure list for "
            "Tune Mix %u", tChannelId);
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bCountTuneMixComponents
 *
 *****************************************************************************/
static BOOLEAN bCountTuneMixComponents (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    TUNEMIX_COMPONENT_ITERATOR_STRUCT *psTuneMixComponents =
        (TUNEMIX_COMPONENT_ITERATOR_STRUCT *)pvArg;

    psTuneMixComponents->atServiceId[psTuneMixComponents->un8Count] =
                CHANNEL.tServiceId(hChannel);
    psTuneMixComponents->un8Count++;

    return TRUE;
}

/*****************************************************************************
 *
 *   vEventCallbackShim
 *
 *****************************************************************************/
static void vEventCallbackShim(
    DECODER_OBJECT hDecoder, SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;
    DECODER_CALLBACK_SHIM_STRUCT *psCallbackShim =
                    (DECODER_CALLBACK_SHIM_STRUCT *)pvEventCallbackArg;

    psObj->bInDecoderCallback = TRUE;
    psCallbackShim->vEventCallback(hDecoder, tEventMask,
        psObj->hBrowsedChannel, psCallbackShim->pvEventCallbackArg);
    psObj->bInDecoderCallback = FALSE;

    return;
}

/*****************************************************************************
 *
 *   vCCacheRemapCallback
 *
 *****************************************************************************/
static void vCCacheRemapCallback(
    CCACHE_OBJECT hCCache, BOOLEAN bComplete, void *pvCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvCallbackArg;
    BOOLEAN bOwner;

    // Caller must be the owner of a valid DECODER object
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Check if update is complete
        if(bComplete == TRUE)
        {
            // Update is complete.
            vUpdateState(psObj, DECODER_STATE_READY);
        }
        else
        {
            // Update did not complete.
            vSetError(psObj, DECODER_ERROR_CODE_CCACHE_ERROR);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vCCacheSubscriptionUpdateCallback
 *
 *****************************************************************************/
static void vCCacheSubscriptionUpdateCallback(
    CCACHE_OBJECT hCCache, BOOLEAN bComplete, void *pvCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvCallbackArg;
    BOOLEAN bOwner;

    // Caller must be the owner of a valid DECODER object
    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if(bOwner == TRUE)
    {
        // Check if update is complete
        if(bComplete == TRUE)
        {
            if((psObj->tCapabilities & SRH_DEVICE_CAPABILITY_AUDIO) ==
                SRH_DEVICE_CAPABILITY_AUDIO)
            {
                // You go and tune your bad self
                // If we were tuned to a skipped/locked/mature channel
                // before the sub update, it should be okay to retune
                // to it now.
                eTuneSelf(psObj, TRUE, TRUE, TRUE);
            }

            // Set event mask
            SMSU_tUpdate(&psObj->sEvent,
                DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED);
        }
        else
        {
            // Update did not complete.
            vSetError(psObj, DECODER_ERROR_CODE_CCACHE_ERROR);
        }
    }

    return;
}

/*****************************************************************************
*
*   vUpdateHandles
*
*****************************************************************************/
static void vUpdateHandles(
    DECODER_OBJECT_STRUCT *psObj, DECODER_UPDATE_HANDLE_ENUM eHandlesToUpdate,
    CHANNEL_OBJECT hChannel, CATEGORY_OBJECT hCategory)
{
    CHANNEL_OBJECT *phCurrentChannel;
    CATEGORY_OBJECT *phCurrentCategory;
    CATEGORY_EVENT_MASK tCatEventRequestMask;

    // Grab the handles we need to update
    switch (eHandlesToUpdate)
    {
        case DECODER_UPDATE_HANDLE_BROWSED:
        {
            phCurrentChannel = &psObj->hBrowsedChannel;
            phCurrentCategory = &psObj->hBrowsedCategory;

            // Just tell us about everything
            tCatEventRequestMask = CATEGORY_OBJECT_EVENT_ALL;
        }
        break;

        case DECODER_UPDATE_HANDLE_TUNED:
        {
            phCurrentChannel = &psObj->hTunedChannel;
            phCurrentCategory = &psObj->hTunedCategory;

            // Just tell us about everything
            tCatEventRequestMask = CATEGORY_OBJECT_EVENT_ALL;
        }
        break;

        case DECODER_UPDATE_HANDLE_LAST_TUNED:
        {
            phCurrentChannel = &psObj->hLastTunedChannel;
            phCurrentCategory = &psObj->hLastTunedCategory;

            // Just tell us about everything
            tCatEventRequestMask = CATEGORY_OBJECT_EVENT_ALL;
        }
        break;

        default:
        {
            // Unknown!
            return;
        }
    }

    /***************************************************
    * Handle Registration / Unregistration of objects *
    ***************************************************/

    ///////////////////////
    // Register category //
    ///////////////////////

    // Has the category changed?
    if(hCategory != *phCurrentCategory)
    {
        BOOLEAN bSetCategory = TRUE;

        // Determine if we're interested in
        // registering this category.  We want to
        // register if this category is valid and
        // doesn't match any of the tracked
        // category handles
        if((hCategory != CATEGORY_INVALID_OBJECT) && (hCategory
            != psObj->hTunedCategory) && (hCategory
            != psObj->hBrowsedCategory) && (hCategory
            != psObj->hLastTunedCategory))
        {
            // Register for updates regarding this
            // category.
            bSetCategory = CATEGORY_bRegisterNotification(hCategory,
                tCatEventRequestMask, vCategoryEventCallback,
                (void *)(size_t)psObj);
        }

        // Category set successfully?
        if(bSetCategory == TRUE)
        {
            // Only unregister this category if this
            // handle is not found in any of our
            // tracked category handles and it is valid
            if((*phCurrentCategory != CATEGORY_INVALID_OBJECT) &&
                (*phCurrentCategory
                != psObj->hTunedCategory) && (*phCurrentCategory
                != psObj->hBrowsedCategory) && (*phCurrentCategory
                != psObj->hLastTunedCategory))
            {
                // Unregister for category update notifications
                // regarding the previously set category
                CATEGORY_vUnregisterNotification(*phCurrentCategory,
                    vCategoryEventCallback, (void *)(size_t)psObj);
            }

            // replace the current category handle with new one
            *phCurrentCategory = hCategory;
        }
    }

    //////////////////////
    // Register channel //
    //////////////////////

    // Has the channel changed?
    if(hChannel != *phCurrentChannel)
    {
        BOOLEAN bSetChannel = TRUE;

        // Is the new channel valid? If so we need to indicate it's being used
        if(hChannel != CHANNEL_INVALID_OBJECT)
        {
            // are we setting the browsed channel?
            if(eHandlesToUpdate == DECODER_UPDATE_HANDLE_BROWSED)
            {
                // yes its the browsed channel, so register with the callback
                bSetChannel = CHANNEL_bRegisterNotification(
                    hChannel,
                    CHANNEL_OBJECT_EVENT_ALL, 
                    vChannelEventCallback, 
                    psObj, 
                    FALSE);
            }
            // are we setting the last-tuned channel?
            else if(eHandlesToUpdate == DECODER_UPDATE_HANDLE_LAST_TUNED)
            {
                // yes its the last tuned channel, so register with the callback
                bSetChannel = CHANNEL_bRegisterNotification(hChannel,
                    CHANNEL_OBJECT_EVENT_CHANNEL_ID |
                    CHANNEL_OBJECT_EVENT_ATTRIBUTES,
                     vLastTunedChannelEventCallback, psObj, FALSE);
            }
            else
            {
                // no its not the browsed channel, or last tuned,
                // so register with no callback
                // which indicates the CHANNEL is being used.
                bSetChannel = CHANNEL_bRegisterNotification(hChannel,
                    CHANNEL_OBJECT_EVENT_NONE);
            }
        }

        // New channel registration successful?
        if(bSetChannel == TRUE)
        {
            // Unregister for channel notifications regarding the previously
            // set channel (might be an invalid CHANNEL which is ok).

            // are we dealing with the browsed channel?
            if(eHandlesToUpdate == DECODER_UPDATE_HANDLE_BROWSED)
            {
                // yes, we're dealing with the browsed channel
                // Unregister for channel notifications regarding the previously
                // set channel (might be an invalid CHANNEL which is ok).
                CHANNEL_vUnregisterNotification(*phCurrentChannel,
                    vChannelEventCallback, psObj);

                psObj->bBrowsed = TRUE;
                psObj->tBrowsedEventMask =
                    CHANNEL_tCompareContent(hChannel, *phCurrentChannel);

                // Set event mask 'CHANNEL' to notify application that
                // the DECODER's browsed channel has changed.
                SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_CHANNEL);
            }
            else
            {
                // no, we're not dealing with the browsed channel
                // Unregister for channel notifications regarding the previously
                // set channel (might be an invalid CHANNEL which is ok).
                CHANNEL_vUnregisterNotification(*phCurrentChannel, NULL);
            }

            // replace the channel, with new channel.
            *phCurrentChannel = hChannel;
        }
    }

    return;
}

/*******************************************************************************
 *
 *   vChannelEventCallback
 *
 *   This callback function is used to register with CHANNEL objects which
 *   belong to a specific DECODER. The purpose of this function is to
 *   simply apply any CHANNEL object event to the entire DECODER object.
 *
 *   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
 *
 *****************************************************************************/
static void vChannelEventCallback(
    CHANNEL_OBJECT hChannel, CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvEventCallbackArg;

    psObj->tBrowsedEventMask = tEventMask;

    // Set event mask 'CHANNEL' to notify application that
    // the DECODER's browsed channel has changed.
    SMSU_tUpdate(&psObj->sEvent, DECODER_OBJECT_EVENT_CHANNEL);

    // Notify callback if registered and change occurred
    SMSU_bNotify(&psObj->sEvent);

    return;
}

/*******************************************************************************
 *
 *   vLastTunedChannelEventCallback
 *
 *   This callback function is used to register with CHANNEL objects which
 *   belong to a specific DECODER. The purpose of this function is to
 *   simply apply any CHANNEL object event to the entire DECODER object.
 *
 *   THIS FUNCTION IS EXCLUSIVELY CALLED IN THE CONTEXT OF A DECODER
 *
 *****************************************************************************/
static void vLastTunedChannelEventCallback(
    CHANNEL_OBJECT hChannel, CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvEventCallbackArg;
    BOOLEAN bNeedsUpdated;
    SMS_BEHAVIOR_VALUE_UNION uBehaviorValue;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Check first if we are tuned, or if tuning is in progress.
    if((TUNE_STATE_TUNED == psObj->eTuneState) ||
        (TUNE_STATE_TUNING_IN_PROGRESS == psObj->eTuneState))
    {
        // Someone beat us to it...forgedd-about-it
        return;
    }

    // Do I know everything I need to about this channel?
    bNeedsUpdated = CHANNEL_bNeedsUpdated(hChannel);
    if(bNeedsUpdated == TRUE)
    {
        // Never mind, we'll try again later
        return;
    }

    // We should try to tune this channel, but...

    // Determine if mature channels control how we
    // tune to the last tuned channel.
    SMS_bGetBehavior(SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE, &uBehaviorValue);

    // Now go tune...
    eReturnCode = eTuneSelf(psObj, uBehaviorValue.bValue, FALSE, FALSE);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        // Dude, if you can't tune yourself...who can you tune?
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to tune myself.");
    }

    return;
}

/*****************************************************************************
 *
 *   vHandleAllChanCatNotificationEvent
 *
 *****************************************************************************/
static void vHandleAllChanCatNotificationEvent(
    DECODER_OBJECT hDecoder,
    BOOLEAN bNotifyChannels
        )
{
    BOOLEAN bOwner;
    // Verify SMS Object and operating context
    bOwner = SMSO_bOwner((SMS_OBJECT)hDecoder);
    if(bOwner == TRUE)
    {
        BOOLEAN bMoreNotifications;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj =
            (DECODER_OBJECT_STRUCT *)hDecoder;

        if (bNotifyChannels == TRUE)
        {
            bMoreNotifications =
                CCACHE_bCheckNextChannelForAppNotification(psObj->hCCache);
        }
        else
        {
            bMoreNotifications =
                CCACHE_bCheckNextCategoryForAppNotification(psObj->hCCache);
        }

        if (bMoreNotifications == TRUE)
        {
            BOOLEAN bPosted;
            SMS_EVENT_HDL hEvent;
            SMS_EVENT_DATA_UNION *puEventData;

            // Allocate an event
            hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr,
                SMS_EVENT_ALL_CHAN_CAT_NOTIFY, &puEventData, SMS_EVENT_OPTION_NONE);

            //Attempt to post the event
            if(hEvent != SMS_INVALID_EVENT_HDL)
            {
                puEventData->uDecoder.sAllChanCatNotify.bNotifyChannels = bNotifyChannels;

                bPosted = SMSE_bPostEvent(hEvent);

                if (bPosted == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Unable to re-post All Chan/Cat event");
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to re-allocate All Chan/Cat event");
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vLoadLastTunedServiceId
 *
 *****************************************************************************/
static void vLoadLastTunedServiceId(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SERVICE_ID tServiceId = SERVICE_INVALID_ID;
    CHANNEL_OBJECT hChannel;
    CATEGORY_OBJECT hCategory = CATEGORY_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag;
    N32 n32TagValue = 0;
    BOOLEAN bOk;

    // retrieve one of components of active TuneMix channel if exist
    bOk = MODULE_bIsIRSupported(psObj->hModule);
    if (bOk == TRUE)
    {
        tServiceId = TUNEMIX_tGetSIDFromActiveTM(psObj->hTag);
    }

    if (tServiceId == SERVICE_INVALID_ID)
    {
        // retrieve the tag value associated with the last tuned channel
        // save the provided service id as the tag value associated with the
        // last tuned channel
        eResult = TAG_eGet(LAST_TUNED_SERVICE_ID_TAG_NAME, psObj->hTag, &hTag,
            NULL, TRUE); // create if it doesn't exist
        if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            N32 *pn32TagValue;
            size_t tSize;

            pn32TagValue = &n32TagValue;
            tSize = sizeof(N32);

            eResult = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
                (void **)&pn32TagValue, &tSize);

            // Use service id from config.
            tServiceId = (SERVICE_ID)n32TagValue;
        }

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // If the last tuned service id is not set yet,
            // or if an error occurred - use default channel
            tServiceId = GsRadio.sChannel.tDefaultServiceID;
        }
    }

    // Map service id to channel handle, and create it if needed
    hChannel = CCACHE_hChannelFromIds(
        psObj->hCCache, tServiceId, CHANNEL_INVALID_ID, TRUE);

    // Get this channel's broadcast category for now
    // IN THE FUTURE WE WILL GET THE LAST TUNED CATEGORY
    // ID FROM THE CONFIG MANAGER
    hCategory = CHANNEL.hCategory(hChannel, 0);

    // Update our last tuned handles
    vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
        hChannel, hCategory);

    // Request the latest and greatest info for this channel
    RADIO_bRequestChannel((DECODER_OBJECT)psObj, tServiceId);

    return;
}

/*****************************************************************************
 *
 *   tGetDefaultOrSafeServiceId
 *
 *   This function is used to get a default or safe channel when for some
 *   reason we don't have or couldn't map to a valid service id. This logic
 *   is based on the logic described in RX120
 *
 *****************************************************************************/
static SERVICE_ID tGetDefaultOrSafeServiceId(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SERVICE_ID tServiceId;
    CHANNEL_ID tChannelId;

    // use the default service id
    tServiceId = GsRadio.sChannel.tDefaultServiceID;

    // map it to a channel id
    tChannelId = CCACHE_tChannelIdFromServiceId(psObj->hCCache, tServiceId);

    // could we map it?
    if(tChannelId == CHANNEL_INVALID_ID)
    {
        // couldn't map to a channel, so use safe service id
        tServiceId = GsRadio.sChannel.tSafeServiceID;
    }

    return tServiceId;
}

/*****************************************************************************
 *
 *   vSaveTunedServiceId
 *
 *****************************************************************************/
static void vSaveTunedServiceId(DECODER_OBJECT_STRUCT *psObj)
{
    SERVICE_ID tServiceId;
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag;

    // get the currently tuned channel's service id
    tServiceId = CHANNEL.tServiceId(psObj->hTunedChannel);

    // save the provided service id as the tag value associated with the
    // last tuned channel
    eResult = TAG_eGet(LAST_TUNED_SERVICE_ID_TAG_NAME, psObj->hTag, &hTag,
        NULL, TRUE);// create if it doesn't exist
    if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 *pn32TagValue;
        N32 n32TagValue = SERVICE_INVALID_ID;
        size_t tSize;

        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);

        // don't look at return code form TAG_eGetTagValue
        // it won't return SUCCESS if tag has no value (like when
        // the config file is brand new). In error case, the tag value will
        // be unchanged from the intiial set value
        TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
            (void **)&pn32TagValue, &tSize);      


        if (tServiceId != (SERVICE_ID)n32TagValue)
        {
            n32TagValue = (N32)tServiceId;

            TAG_eSetTagValue(hTag, TAG_TYPE_INTEGER, (void *)&n32TagValue,
                sizeof(N32), TRUE);
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vCategoryEventCallback
 *
 *   This function is called whenever a category object currently in
 *   use by this decoder is updated or destroyed.
 *
 *****************************************************************************/
static void vCategoryEventCallback(
    CATEGORY_OBJECT hCategory, CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg)
{
    DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvEventCallbackArg;
    CATEGORY_TYPE_ENUM eCategoryType;
    CATEGORY_OBJECT hNewCategory = CATEGORY_INVALID_OBJECT;
    BOOLEAN bOk, bTuneUpdated = FALSE;

    // Verify ownership of this decoder
    bOk = SMSO_bOwner((SMS_OBJECT)psObj);
    if(bOk != TRUE)
    {
        // Error!  How did we get here?
        return;
    }

    // Are we interested in this category?
    if((psObj->hTunedCategory != hCategory) && (psObj->hBrowsedCategory
                    != hCategory) && (psObj->hLastTunedCategory != hCategory))
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
            "Unknown category update.\n");

        // Unregister this category
        CATEGORY_vUnregisterNotification(hCategory, vCategoryEventCallback,
            (void *)(size_t)psObj);
        return;
    }

    // Get the updated category's type
    eCategoryType = CATEGORY.eType(hCategory);

    // Has the last tuned category been updated?
    if(hCategory == psObj->hLastTunedCategory)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
            "Last Tuned Category: %d has been updated: %d\n",
            CATEGORY.tGetCategoryId(hCategory), tEventMask);

        // Has this category been destroyed?
        if((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                        == CATEGORY_OBJECT_EVENT_REMOVED)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Category: %d has been removed\n",
                CATEGORY.tGetCategoryId(hCategory));

            // If the last tuned category is not
            // a broadcast category, then switch the last
            // tuned category to the last tuned channel's
            // broadcast category
            if(eCategoryType != CATEGORY_TYPE_BROADCAST)
            {
                // Get the last tuned channel's broadcast category
                hNewCategory = CHANNEL.hCategory(psObj->hLastTunedChannel, 0);
            }
            else
            {
                // Well, the last tuned channel's category is being
                // deleted, and it is a broadcast category.  Just
                // invalidate the handle
                hNewCategory = CATEGORY_INVALID_OBJECT;
            }

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Updating Last Tuned Category: %d\n",
                CATEGORY.tGetCategoryId(hNewCategory));

            // Update the last tuned category handle
            vUpdateHandles(psObj, DECODER_UPDATE_HANDLE_LAST_TUNED,
                psObj->hLastTunedChannel, hNewCategory);
        }
    }

    // Has the tuned category been updated?
    if (hCategory == psObj->hTunedCategory)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
            "Tuned Category: %d has been updated: %d\n",
            CATEGORY.tGetCategoryId(hCategory), tEventMask);

        // Has this category been destroyed?
        if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                        == CATEGORY_OBJECT_EVENT_REMOVED)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Category: %d has been removed\n",
                CATEGORY.tGetCategoryId(hCategory));

            bTuneUpdated = bUpdateCategoryFallback(
                psObj,
                psObj->hTunedChannel,
                DECODER_UPDATE_HANDLE_TUNED
                    );

        }// CATEGORY_OBJECT_EVENT_DESTROYED

        else if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS)
                        == CATEGORY_OBJECT_EVENT_CONTENTS)
        {
            N16 n16ChanIndex;
            CHANNEL_ID tTunedChannelId;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Category: %d contents has been updated\n",
                CATEGORY.tGetCategoryId(hCategory));

            // Get the tuned channel id
            tTunedChannelId = CHANNEL.tChannelId( psObj->hTunedChannel );

            // Is this channel still in this category?
            n16ChanIndex = CATEGORY_n16GetIndexByChanId(
                            hCategory, tTunedChannelId );

            if (n16ChanIndex < 0)
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "Channel Id: %d doesn't exist in Category: %d\n",
                    CHANNEL.tChannelId(psObj->hTunedChannel), CATEGORY.tGetCategoryId(hCategory));

                bTuneUpdated = bUpdateCategoryFallback(
                    psObj,
                    psObj->hTunedChannel,
                    DECODER_UPDATE_HANDLE_TUNED
                        );

            }// n16ChanIndex == -1
        } // CATEGORY_OBJECT_EVENT_CONTENTS
    } // tuned category

    // Has the browsed category been updated?
    if (hCategory == psObj->hBrowsedCategory)
    {
        BROWSE_TYPE_ENUM eBrowseType;
        SMS_EVENT_BROWSE_STRUCT sBrowse;

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
            "Browsed Category: %d has been updated: %d\n",
            CATEGORY.tGetCategoryId(hCategory), tEventMask);

        // Get our current browsing mode
        eBrowseType = BROWSE_eGetMode( psObj->hBrowse );

        // The tuned category has been updated
        // and the browsed category must follow
        if (bTuneUpdated == TRUE)
        {
            CHANNEL_ID tTunedChannelId;
            CATEGORY_ID tTunedCategoryId;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Tuned Category was updated.\n",
                psObj->pacObjectName);

            // Get the tuned Ids
            tTunedChannelId =
                CHANNEL.tChannelId( psObj->hTunedChannel );

            tTunedCategoryId =
                CATEGORY.tGetCategoryId( psObj->hTunedCategory );

            // Set our mode using the currently
            // tuned Ids (channel / category)
            BROWSE_bSetMode(
                    psObj->hBrowse,
                    eBrowseType,
                    tTunedChannelId,
                    tTunedCategoryId );

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Updating Browsed Handles "
                "[ Category: %d, Channel Id: %d ]\n",
                CATEGORY.tGetCategoryId(hCategory),
                CHANNEL.tChannelId(psObj->hTunedChannel));

            // Update the browsed category handle
            // to use the tuned category handle
            vUpdateHandles(
                    psObj, DECODER_UPDATE_HANDLE_BROWSED,
                    psObj->hBrowsedChannel,
                    psObj->hTunedCategory);

            // Stop here
            return;
        }

        // Has this category been destroyed?
        if ((tEventMask & CATEGORY_OBJECT_EVENT_REMOVED)
                        == CATEGORY_OBJECT_EVENT_REMOVED)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Category: %d has been removed\n",
                CATEGORY.tGetCategoryId(hCategory));

            if (eBrowseType == BROWSE_TYPE_CATEGORY)
            {
                // Browsed category has been removed...
                bUpdateCategoryFallback(
                    psObj,
                    psObj->hBrowsedChannel,
                    DECODER_UPDATE_HANDLE_BROWSED
                        );
            }

        } // CATEGORY_OBJECT_EVENT_DESTROYED

        else if ((tEventMask & CATEGORY_OBJECT_EVENT_CONTENTS)
                        == CATEGORY_OBJECT_EVENT_CONTENTS)
        {
            CHANNEL_ID tBrowsedChannelId;
            N16 n16ChanIndex;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Category: %d contents has been updated\n",
                CATEGORY.tGetCategoryId(hCategory));

            // Get the browsed channel Id
            tBrowsedChannelId =
                CHANNEL.tChannelId( psObj->hBrowsedChannel );

            // Is this channel still in this category?
            n16ChanIndex = CATEGORY_n16GetIndexByChanId(
                            hCategory, tBrowsedChannelId );

            // If the channel has been removed and
            if (n16ChanIndex < 0)
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "Channel Id: %d doesn't exist in Category: %d\n",
                    CHANNEL.tChannelId(psObj->hBrowsedChannel), CATEGORY.tGetCategoryId(hCategory));

                sBrowse.eBrowseType = eBrowseType;

                // We're browsing channels...
                if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
                {
                    // Browse past the browsed channel
                    // to work this out (the browsed channel
                    // no longer belongs to any known category)
                    sBrowse.eDirection = SMSAPI_DIRECTION_NEXT;
                    sBrowse.uId.tChannel = CHANNEL_INVALID_ID;

                    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                        "%s: Browsing All Channels: Next\n",
                        psObj->pacObjectName);

                    vBrowseEventHandler(psObj, &sBrowse);
                }
                // We're browsing categories...

                else if (eBrowseType == BROWSE_TYPE_CATEGORY)
                {
                    // Browse to the browsed category
                    // again to work this out
                    sBrowse.eDirection = SMSAPI_DIRECTION_DIRECT;

                     // Get the browsed category Id
                    sBrowse.uId.tCategory = CATEGORY.tGetCategoryId(
                        psObj->hBrowsedCategory );

                    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                        "Browsing Category: %d\n", sBrowse.uId.tCategory);

                    vBrowseEventHandler(psObj, &sBrowse);
                }
            }
        } // CATEGORY_OBJECT_EVENT_CONTENTS
    } // browsed category updated

    return;
}

/*******************************************************************************
 *
 *   vSetError
 *
 *******************************************************************************/
static void vSetError(
    DECODER_OBJECT_STRUCT *psObj, DECODER_ERROR_CODE_ENUM eErrorCode)
{
    // Set local object error code
    psObj->eErrorCode = eErrorCode;

    // Transition to error state
    vUpdateState(psObj, DECODER_STATE_ERROR);

    return;
}

/*****************************************************************************
 *
 *   bChannelAttributeTagIterator
 *
 *****************************************************************************/
static BOOLEAN bChannelAttributeTagIterator(TAG_OBJECT hTag, void *pvArg)
{
    N32 n32TagValue, *pn32TagValue;
    SERVICE_ID tServiceId;
    size_t tSize;
    CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT *psIteratorStruct;
    STRING_OBJECT hTagName;

    psIteratorStruct = (CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT *)pvArg;

    // get this tag's name
    hTagName = TAG_hTagName(hTag);
    // is this a tag we're interested in?
    if(STRING.n16CompareCStr(SERVICE_ID_TAG_NAME, 0, hTagName) == 0)
    {
        // yes, we're interested in this tag

        SMSAPI_RETURN_CODE_ENUM eReturn;

        // get the tag value
        pn32TagValue = &n32TagValue;
        tSize = sizeof(N32);
        eReturn = TAG_eGetTagValue(hTag, TAG_TYPE_INTEGER,
            (void **)&pn32TagValue, &tSize);
        if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            tServiceId = (SERVICE_ID)n32TagValue;
            // is this the service id we're looking for
            // either specifically or if we're looking for all meaning
            // psIteratorStruct->tServiceId equals SERVICE_INVALID_ID
            if((tServiceId == psIteratorStruct->tServiceId)
            || (psIteratorStruct->tServiceId == SERVICE_INVALID_ID))
            {
                // found what we're looking for

                if(psIteratorStruct->bRemove == TRUE)
                {
                    // we need to remove this tag
                    TAG_eRemove(hTag, FALSE);
                }
                else if((psIteratorStruct->bLock == TRUE)
                     || (psIteratorStruct->bSkip == TRUE))
                {
                    CHANNEL_OBJECT hChannel;

                    // Search for the specified channel Id in the cache
                    hChannel = CCACHE_hChannelFromIds(
                        psIteratorStruct->psDecoderObj->hCCache,
                        tServiceId, CHANNEL_INVALID_ID,
                        TRUE);

                    if(hChannel != CHANNEL_INVALID_OBJECT)
                    {
                        if(psIteratorStruct->bLock == TRUE)
                        {
                            // lock the channel
                            CHANNEL_eSetAttribute(hChannel,
                                CHANNEL_OBJECT_ATTRIBUTE_LOCKED);

                            CHANNEL_eSetUnconfirmedAttribute(
                                hChannel, CHANNEL_OBJECT_ATTRIBUTE_LOCKED);

                            psIteratorStruct->sAttrs.ptLocked[
                                psIteratorStruct->sAttrs.un16NumLocked++
                                    ] = tServiceId;
                        }

                        if(psIteratorStruct->bSkip == TRUE)
                        {
                            // skip the channel
                            CHANNEL_eSetAttribute(hChannel,
                                CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);

                            CHANNEL_eSetUnconfirmedAttribute(
                                hChannel, CHANNEL_OBJECT_ATTRIBUTE_SKIPPED);

                            psIteratorStruct->sAttrs.ptSkipped[
                                psIteratorStruct->sAttrs.un16NumSkipped++
                                    ] = tServiceId;
                        }
                    }
                }
                else
                {
                    // nothing needs to be done
                }

                if(psIteratorStruct->tServiceId != SERVICE_INVALID_ID)
                {
                    // we were looking for a specific service id and
                    // we found it, so stop iterating
                    return FALSE;
                }
            }
        }
    }
    // keep iterating
    return TRUE;
}

/*****************************************************************************
 *
 *   bRetrieveLockedChannels
 *
 *****************************************************************************/
static BOOLEAN bRetrieveLockedChannels(DECODER_OBJECT_STRUCT *psObj)
{
    TAG_OBJECT hTag;
    BOOLEAN bReturn = FALSE;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)psObj;
    CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT sIteratorStruct;
    SMSAPI_RETURN_CODE_ENUM eResult;

    // populate the iterator structure
    sIteratorStruct.tServiceId = SERVICE_INVALID_ID; // all channels
    sIteratorStruct.bRemove = FALSE;
    sIteratorStruct.bLock = TRUE; // locked channels
    sIteratorStruct.bSkip = FALSE;
    sIteratorStruct.psDecoderObj = psObj;

    // Initialize channel attributes struct
    OSAL.bMemSet((void *)&sIteratorStruct.sAttrs, 0, sizeof(DECODER_CHANNEL_ATTRS_STRUCT));

    do
    {
        UN32 un32NumLockedChannels = 0, un32Index;

        // get the locked channels tag - create if it doesn't exist
        eResult = TAG_eGet(LOCKED_TAG_NAME, psObj->hTag, &hTag, NULL, TRUE);
        if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            bReturn = TRUE;
            break;
        }

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to get Locked channels tag.\n",
                psObj->pacObjectName);
            break;
        }

        // Get number of locked channel entries stored in sms configuration
        eResult = TAG_eNumberOfChildren(hTag, &un32NumLockedChannels);
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to get number of locked channels.\n",
                psObj->pacObjectName);
            break;
        }

        // No locked channel entries saved in the sms configuration
        if (un32NumLockedChannels == 0)
        {
            bReturn = TRUE;
            break;
        }

        // Allocate memory
        sIteratorStruct.sAttrs.ptLocked = (SERVICE_ID *)
            SMSO_hCreate(
                "ChanLockAttrs", sizeof(SERVICE_ID) * un32NumLockedChannels, SMS_INVALID_OBJECT, FALSE);

        if (sIteratorStruct.sAttrs.ptLocked == NULL)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to allocate Channel Locked Attribute array.\n",
                psObj->pacObjectName);
            break;
        }

        for (un32Index = 0; un32Index < un32NumLockedChannels; un32Index++)
        {
            sIteratorStruct.sAttrs.ptLocked[un32Index] = SERVICE_INVALID_ID;
        }

        // Iterate all the children of this tag, and lock each channel
        // that has a tag
        eResult = TAG_eIterateChildren(
            hTag,
            bChannelAttributeTagIterator,
            &sIteratorStruct
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Configure locked attributes
            bReturn = RADIO_bLockChannel(
                hDecoder,
                &sIteratorStruct.sAttrs.ptLocked[0],
                sIteratorStruct.sAttrs.un16NumLocked
                    );
        }

        // Free allocated memory
        SMSO_vDestroy((SMS_OBJECT)sIteratorStruct.sAttrs.ptLocked);

    } while (FALSE);

    return bReturn;
}

/******************************************************************************
 *
 *   bRetrieveTuneMixChannels
 *
 *****************************************************************************/
static BOOLEAN bRetrieveTuneMixChannels (DECODER_OBJECT_STRUCT *psObj)
{
    TAG_OBJECT hNextTMTag, hTag, hCurrentTMTag;
    BOOLEAN bOk = FALSE;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)psObj;
    SMSAPI_RETURN_CODE_ENUM eResultCode = SMSAPI_RETURN_CODE_ERROR;
    UN8 un8MaxTuneMix;
    UN32 un32TMTags = 0;

    do
    {
        // Check IR support by the Module
        bOk = MODULE_bIsIRSupported(psObj->hModule);
        if (bOk != TRUE)
        {
            puts(DECODER_OBJECT_NAME": IR isn't supported by module");

            // We allow this to be treated as valid case
            bOk = TRUE;
            break;
        }

        // Ask the Radio TuneMix maximum value
        bOk = MODULE_VERSION_bMaxTuneMix(psObj->hModule, &un8MaxTuneMix);
        if ((bOk != TRUE) || (un8MaxTuneMix == TUNEMIX_OBJECTS_NONE))
        {
            puts(DECODER_OBJECT_NAME": Failed to get TuneMix maximum "
                "number, or SXI 3.0 is not supported by module");

            // We allow this to be treated as valid case
            bOk = TRUE;
            break;
        }

        psObj->un8MaxTuneMix =
          (un8MaxTuneMix > TUNEMIX_OBJECTS_MAX) ? TUNEMIX_OBJECTS_MAX : un8MaxTuneMix;

        eResultCode = TAG_eGet(TUNEMIX_OBJECT_NAME, psObj->hTag, &hTag,
                               NULL, TRUE);

        if (eResultCode == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            break;
        }

        eResultCode = TAG_eNumberOfChildren(hTag, &un32TMTags);
        if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            bOk = FALSE;
            break;
        }

        if (un32TMTags == 0)
        {
            puts(DECODER_OBJECT_NAME": There are no TuneMix saved");
            break;
        }

        eResultCode = TAG_eFirstChild(hTag, &hCurrentTMTag);
        if (eResultCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            TUNEMIX_OBJECT hTuneMix;
            UN8 un8IndexRetrieved = 0, un8Index = 0;
            SMS_BEHAVIOR_VALUE_UNION uBehaviorValue;

            SMS_bGetBehavior(SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE,
                &uBehaviorValue);

            for( un8Index = 0; un8Index < un32TMTags; un8Index++)
            {
                hTuneMix = TUNEMIX_hCreateFromTags(
                     hDecoder, hCurrentTMTag, &un8IndexRetrieved,
                     uBehaviorValue.bValue);

                eResultCode = TAG_eNextSibling( hCurrentTMTag, &hNextTMTag);
                if (eResultCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }
                if (hTuneMix != TUNEMIX_INVALID_OBJECT)
                {
                    psObj->ahTuneMix[un8IndexRetrieved] = hTuneMix;
                }
                else
                {
                    TAG_eRemove(hCurrentTMTag, TRUE);
                }
                hCurrentTMTag = hNextTMTag;
            }
            bOk = TRUE;
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   bRetrieveSkippedChannels
 *
 *****************************************************************************/
static BOOLEAN bRetrieveSkippedChannels(DECODER_OBJECT_STRUCT *psObj)
{
    TAG_OBJECT hTag;
    BOOLEAN bReturn = FALSE;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)psObj;
    CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT sIteratorStruct;
    SMSAPI_RETURN_CODE_ENUM eResult;

    sIteratorStruct.tServiceId = SERVICE_INVALID_ID; // all channels
    sIteratorStruct.bRemove = FALSE;
    sIteratorStruct.bLock = FALSE;
    sIteratorStruct.bSkip = TRUE; // skipped channels
    sIteratorStruct.psDecoderObj = psObj;

    OSAL.bMemSet((void *)&sIteratorStruct.sAttrs, 0, sizeof(DECODER_CHANNEL_ATTRS_STRUCT));

    do
    {
        UN32 un32NumSkippedChannels = 0, un32Index;

        // get the skipped channels tag - create if it doesn't exist
        eResult = TAG_eGet(SKIPPED_TAG_NAME, psObj->hTag, &hTag, NULL, TRUE);
        if (eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
        {
            bReturn = TRUE;
            break;
        }

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to get Skipped channels tag.\n",
                psObj->pacObjectName);
            break;
        }

        // Get number of locked channel entries stored in sms configuration
        eResult = TAG_eNumberOfChildren(hTag, &un32NumSkippedChannels);
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to get Skipped channnels number.\n",
                psObj->pacObjectName);
            break;
        }

        // No skipped channel entries saved in the sms configuration
        if (un32NumSkippedChannels == 0)
        {
            bReturn = TRUE;
            break;
        }

        // Allocate memory
        sIteratorStruct.sAttrs.ptSkipped = (SERVICE_ID *)
            SMSO_hCreate(
                "ChanSkipedAttrs", sizeof(SERVICE_ID) * un32NumSkippedChannels, SMS_INVALID_OBJECT, FALSE);

        if (sIteratorStruct.sAttrs.ptSkipped == NULL)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to allocate Channel Skipped Attributes array.\n",
                psObj->pacObjectName);
            break;
        }

        for (un32Index = 0; un32Index < un32NumSkippedChannels; un32Index++)
        {
            sIteratorStruct.sAttrs.ptSkipped[un32Index] = SERVICE_INVALID_ID;
        }

        // Iterate all the children of this tag, and skip each channel
        // that has a tag
        eResult = TAG_eIterateChildren(
            hTag,
            bChannelAttributeTagIterator,
            &sIteratorStruct
                );

        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Configure skipped attributes
            bReturn = RADIO_bSkipChannel(
                hDecoder,
                &sIteratorStruct.sAttrs.ptSkipped[0],
                sIteratorStruct.sAttrs.un16NumSkipped
                    );
        }

        // Free allocated memory
        SMSO_vDestroy((SMS_OBJECT)sIteratorStruct.sAttrs.ptSkipped);

    } while (FALSE);

    return bReturn;
}

/*****************************************************************************
 *
 *   bRetrieveUnsubscribedText
 *
 *****************************************************************************/
static BOOLEAN bRetrieveUnsubscribedText(DECODER_OBJECT_STRUCT *psObj)
{
    BOOLEAN bSuccess = TRUE;

    do
    {
        // get the unsubscribed text for the artist field
        bSuccess = bRetrieveUnsubscribedTextTag(
                       psObj,
                       UNSUBSCRIBED_ARTIST_TEXT_TAG_NAME,
                       CDO_FIELD_ARTIST);
        if (bSuccess == FALSE)
        {
            break;
        }

        // get the unsubscribed text for the title field
        bSuccess = bRetrieveUnsubscribedTextTag(
                       psObj,
                       UNSUBSCRIBED_TITLE_TEXT_TAG_NAME,
                       CDO_FIELD_TITLE);
        if (bSuccess == FALSE)
        {
            break;
        }

        // get the unsubscribed text for the composer field
        bSuccess = bRetrieveUnsubscribedTextTag(
                       psObj,
                       UNSUBSCRIBED_COMPOSER_TEXT_TAG_NAME,
                       CDO_FIELD_COMPOSER);
        if (bSuccess == FALSE)
        {
            break;
        }

        // get the unsubscribed text for the album name field
        bSuccess = bRetrieveUnsubscribedTextTag(
                       psObj,
                       UNSUBSCRIBED_ALBUM_TEXT_TAG_NAME,
                       CDO_FIELD_ALBUM);
        if (bSuccess == FALSE)
        {
            break;
        }

        // get the unsubscribed text for the contentinfo field
        bSuccess = bRetrieveUnsubscribedTextTag(
                       psObj,
                       UNSUBSCRIBED_CONTENTINFO_TEXT_TAG_NAME,
                       CDO_FIELD_CONTENTINFO);
        if (bSuccess == FALSE)
        {
            break;
        }

    } while (0);

    return bSuccess;
}

/*****************************************************************************
 *
 *   bRetrieveUnsubscribedTextTag
 *
 *****************************************************************************/
static BOOLEAN bRetrieveUnsubscribedTextTag(
    DECODER_OBJECT_STRUCT *psObj, const char *pacTagName,
    CDO_FIELD_MASK tMask
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    STRING_OBJECT hString, *phString;
    TAG_OBJECT hTag;
    char *pacCStrBuffer = NULL;

    // Get the tag, but don't bother to create if it doesn't exist.
    eResult = TAG_eGet(pacTagName, psObj->hTag, &hTag, NULL, FALSE);
    if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        SMSAPI_RETURN_CODE_ENUM eReturn;
        size_t tSize;

        hString = STRING_INVALID_OBJECT;
        tSize = sizeof(STRING_OBJECT);
        phString = &hString;

        eReturn = TAG_eGetTagValue(hTag, TAG_TYPE_STRING, (void**)&phString, &tSize);
        if(eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            if (hString != STRING_INVALID_OBJECT)
            {
                size_t tStringSize;

                tStringSize = STRING.tSize(hString) +1;

                pacCStrBuffer =
                    (char *) SMSO_hCreate(
                            DECODER_OBJECT_NAME":CSTR",
                            tStringSize,
                            SMS_INVALID_OBJECT, FALSE);

                // Was the creation successful?
                if (pacCStrBuffer != NULL)
                {
                    // yes it was, so copy the tag value to it
                    STRING.tCopyToCStr(
                        hString,
                        pacCStrBuffer,
                        tStringSize
                            );

                    // Set the unsubscribed text.
                    eResult = eSetUnsubscribedText(
                                  (DECODER_OBJECT)psObj,
                                  pacCStrBuffer,
                                  tMask);
                    if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME
                        "Failed to set unsubscribed text: %s (#%d).",
                        SMSAPI_DEBUG_pacReturnCodeText(eResult), eResult
                            );
                    }

                    // We're done with this buffer.
                    SMSO_vDestroy((SMS_OBJECT)pacCStrBuffer);
                }

                // done with this
                STRING.vDestroy(hString);
            }
        }
        return TRUE;
    }
    else if(eResult == SMSAPI_RETURN_CODE_CFG_NO_PARENT)
    {
        // Set the unsubscribed text.
        eResult = eSetUnsubscribedText(
                      (DECODER_OBJECT)psObj,
                      GacUnSubscribedTextDefault,
                      tMask);
        if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            return TRUE;
        }
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   bStoreUnsubscribedText
 *
 *****************************************************************************/
static BOOLEAN bStoreUnsubscribedText(
    DECODER_OBJECT_STRUCT *psObj, const char *pacUnsubscribedText,
    CDO_FIELD_MASK tMask
        )
{
    BOOLEAN bSuccess = TRUE;
    STRING_OBJECT hNameString;

    // Create a String object for the unsubscribed text.
    // We'll use it to store the tag values.
    hNameString = STRING.hCreate(pacUnsubscribedText, 0);

    if (hNameString == STRING_INVALID_OBJECT)
    {
        // Failed to create the string. Can't store the the text.
        return FALSE;
    }

    do
    {
        if ((tMask & CDO_FIELD_ARTIST) == CDO_FIELD_ARTIST)
        {
            // we need to store the unsubscribed artist field text
            bSuccess = bStoreUnsubscribedTextTag(
                           psObj,
                           UNSUBSCRIBED_ARTIST_TEXT_TAG_NAME,
                           hNameString);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        if ((tMask & CDO_FIELD_TITLE) == CDO_FIELD_TITLE)
        {
            // we need to store the unsubscribed title field text
            bSuccess = bStoreUnsubscribedTextTag(
                           psObj,
                           UNSUBSCRIBED_TITLE_TEXT_TAG_NAME,
                           hNameString);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        if ((tMask & CDO_FIELD_COMPOSER) == CDO_FIELD_COMPOSER)
        {
            // we need to store the unsubscribed composer field text
            bSuccess = bStoreUnsubscribedTextTag(
                           psObj,
                           UNSUBSCRIBED_COMPOSER_TEXT_TAG_NAME,
                           hNameString);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        if ((tMask & CDO_FIELD_ALBUM) == CDO_FIELD_ALBUM)
        {
            // we need to store the unsubscribed album name field text
            bSuccess = bStoreUnsubscribedTextTag(
                psObj,
                UNSUBSCRIBED_ALBUM_TEXT_TAG_NAME,
                hNameString);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        if ((tMask & CDO_FIELD_CONTENTINFO) == CDO_FIELD_CONTENTINFO)
        {
            // we need to store the unsubscribed contentinfo field text
            bSuccess = bStoreUnsubscribedTextTag(
                           psObj,
                           UNSUBSCRIBED_CONTENTINFO_TEXT_TAG_NAME,
                           hNameString);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

    } while(0);

    if (bSuccess == TRUE)
    {
        // We were successful in setting Tag values. Now commit these changes.
        SMSAPI_RETURN_CODE_ENUM eResult;
        eResult = CM_eCommitChangesToFile();
        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            bSuccess = FALSE;
        }
    }

    // Done with the String
    STRING.vDestroy(hNameString);

    return bSuccess;
}

/*****************************************************************************
 *
 *   bStoreUnsubscribedTextTag
 *
 *****************************************************************************/
static BOOLEAN bStoreUnsubscribedTextTag(
    DECODER_OBJECT_STRUCT *psObj, const char *pacTagName,
    STRING_OBJECT hNameString
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hTag;

    // Get the tag, create if it doesn't exist.
    eResult = TAG_eGet(pacTagName, psObj->hTag, &hTag, NULL, TRUE);
    if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the tag value
        eResult = TAG_eSetTagValue(
                    hTag,
                    TAG_TYPE_STRING,
                    &hNameString,
                    sizeof(STRING_OBJECT),
                    FALSE); // not committing at this time
    }

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
 *
 *   bAddServiceIdTag
 *
 *****************************************************************************/
static BOOLEAN bAddServiceIdTag(TAG_OBJECT hParentTag, CHANNEL_OBJECT hChannel)
{
    TAG_OBJECT hNewTag;
    SERVICE_ID tServiceId;
    BOOLEAN bOk = FALSE;
    SMSAPI_RETURN_CODE_ENUM eResult;

    // get this channel's service id
    tServiceId = CHANNEL.tServiceId(hChannel);

    // add a tag
    eResult = TAG_eAdd(SERVICE_ID_TAG_NAME, hParentTag, &hNewTag, NULL);
    if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        N32 n32TagValue;

        n32TagValue = (N32)tServiceId;

        // set the tag's value
        eResult = TAG_eSetTagValue(hNewTag, TAG_TYPE_INTEGER,
            (void *)&n32TagValue, sizeof(N32), TRUE);
        if(eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            bOk = TRUE;
        }
        else
        {
            // we couldn't write a value so remove it
            TAG_eRemove(hNewTag, FALSE);
        }
    }

    return bOk;
}

/*****************************************************************************
 *
 *   bCCacheUnsubscribedChansIterator
 *
 *****************************************************************************/
static BOOLEAN bCCacheUnsubscribedChansIterator(
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    BOOLEAN bIsSubscribed = TRUE;

    eResult = CHANNEL.eIsSubscribed (
        hChannel,
        &bIsSubscribed
            );

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        if (bIsSubscribed == FALSE)
        {
            CHANNEL_EVENT_MASK tEventMask = (CHANNEL_OBJECT_EVENT_TITLE |
                                             CHANNEL_OBJECT_EVENT_ARTIST |
                                             CHANNEL_OBJECT_EVENT_COMPOSER);
            CHANNEL_vSetEvents (
                hChannel,
                tEventMask
                    );
        }
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
 *
 *   bProcessDecoderEventMaskChange
 *
 *****************************************************************************/
static BOOLEAN bProcessDecoderEventMaskChange(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_DECODER_EVENT_MASK_STRUCT const *psDecoderEventStruct
        )
{
    BOOLEAN bValid, bSuccess = FALSE;
    BOOLEAN bSignalStatusMask = FALSE,
            bAntennaAimingMask = FALSE,
            bAudioPresenceMask = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)psObj);

    if (bValid == TRUE)
    {
        DECODER_EVENT_MASK tNewRequestMask =
            psDecoderEventStruct->tDecoderEventMask;

        // Make the change to the event request mask
        // and see if it worked
        bSuccess =
            SMSU_bModifyRequestMask(&psObj->sEvent,
                NULL,
                NULL,
                psDecoderEventStruct->tDecoderEventMask,
                psDecoderEventStruct->eModification);

        if (bSuccess == TRUE)
        {
            // Have time events been requested?
            if(tNewRequestMask & DECODER_OBJECT_EVENT_TIME)
            {
                BOOLEAN bStarted;

                // Start/Re-Start minute timer since we have that event
                bStarted = bStartMinuteTicker(psObj);
                if(bStarted == FALSE)
                {
                    // Error!
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Unable to re-sync timer.");

                    // Transition to error state
                    vSetError(psObj, DECODER_ERROR_CODE_TIMER_ERROR);
                }
            }
            else // No time events
            {
                vStopMinuteTicker(psObj);
            }

            // Check if we had Signal Status monitor enabled
            if (psObj->tRequestMask & DECODER_OBJECT_EVENT_ANTENNA_SIGNAL)
            {
                bSignalStatusMask = TRUE;
            }

            // Check if we had Antenna Aiming monitor enabled
            if (psObj->tRequestMask & DECODER_OBJECT_EVENT_ANTENNA_AIMING)
            {
                bAntennaAimingMask = TRUE;
            }

            // Check if we had Audio Presence monitor enabled
            if (psObj->tRequestMask & DECODER_OBJECT_EVENT_AUDIO_PRESENCE)
            {
                bAudioPresenceMask = TRUE;
            }

            // Process signal event
            if (tNewRequestMask & DECODER_OBJECT_EVENT_ANTENNA_SIGNAL)
            {
                // New mask contains events, which require 
                // Signal Status monitor
                if ((psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_ENABLE) ||
                    (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE))
                {
                    // We're told to ENABLE or REPLACE
                    if (bSignalStatusMask == FALSE)
                    {
                        // Enabling, only if it was not enabled before
                        bSuccess = RADIO_bSignalMonitorSwitch(
                            (DECODER_OBJECT)psObj, TRUE);
                    }
                }
                else if ((psDecoderEventStruct->eModification ==
                                SMSAPI_MODIFY_EVENT_MASK_DISABLE))
                {
                    // We're told to DISABLE
                    if (bSignalStatusMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bSignalMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_ANTENNA_SIGNAL;
                    }
                }
            }
            else
            {
                // There is no Signal Monitor events in mask. Check
                // if it is REPLACE event
                if (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE)
                {
                    if (bSignalStatusMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bSignalMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_ANTENNA_SIGNAL;
                    }
                }
            }

            // Process antenna event
            if (tNewRequestMask & DECODER_OBJECT_EVENT_ANTENNA_AIMING)
            {
                // New mask contains events, which require 
                // Antenna Aiming monitor
                if ((psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_ENABLE) ||
                    (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE))
                {
                    // We're told to ENABLE or REPLACE
                    if (bAntennaAimingMask == FALSE)
                    {
                        // Enabling, only if it was not enabled before
                        bSuccess = RADIO_bAntennaAimingMonitorSwitch(
                            (DECODER_OBJECT)psObj, TRUE);
                    }
                }
                else if ((psDecoderEventStruct->eModification ==
                                SMSAPI_MODIFY_EVENT_MASK_DISABLE))
                {
                    // We're told to DISABLE
                    if (bAntennaAimingMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bAntennaAimingMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_ANTENNA_AIMING;
                    }
                }
            }
            else
            {
                // There is no Antenna Aiming Monitor events in mask. 
                // Check if it is REPLACE event
                if (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE)
                {
                    if (bAntennaAimingMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bAntennaAimingMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_ANTENNA_AIMING;
                    }
                }
            }

            // Process Audio Presence event
            if (tNewRequestMask & DECODER_OBJECT_EVENT_AUDIO_PRESENCE)
            {
                // New mask contains events, which require 
                // Audio Presence monitor
                if ((psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_ENABLE) ||
                    (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE))
                {
                    // We're told to ENABLE or REPLACE
                    if (bAudioPresenceMask == FALSE)
                    {
                        // Enabling, only if it was not enabled before
                        bSuccess = RADIO_bAudioPresenceMonitorSwitch(
                            (DECODER_OBJECT)psObj, TRUE);
                    }
                }
                else if ((psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_DISABLE))
                {
                    // We're told to DISABLE
                    if (bAudioPresenceMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bAudioPresenceMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_AUDIO_PRESENCE;

                        psObj->eAudioPresence = DECODER_AUDIO_PRESENCE_UNKNOWN;
                    }
                }
            }
            else
            {
                // There is no Audio Presence Monitor events in mask.
                // Check if it is REPLACE event
                if (psDecoderEventStruct->eModification ==
                    SMSAPI_MODIFY_EVENT_MASK_REPLACE)
                {
                    if (bAudioPresenceMask == TRUE)
                    {
                        // Disabling, only if it was enabled before
                        bSuccess = RADIO_bAudioPresenceMonitorSwitch(
                            (DECODER_OBJECT)psObj, FALSE);

                        tNewRequestMask &= ~DECODER_OBJECT_EVENT_AUDIO_PRESENCE;

                        psObj->eAudioPresence = DECODER_AUDIO_PRESENCE_UNKNOWN;
                    }
                }
            }

            psObj->tRequestMask = tNewRequestMask;
        }
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   vSetChannelAttrs
 *
 *****************************************************************************/
static void vSetChannelAttrs (
    DECODER_OBJECT_STRUCT *psObj,
    BOOLEAN bLocked,
    BOOLEAN bSkipped
        )
{
    UN32 un32Index;
    BOOLEAN bSuccess;
    DECODER_CHANNEL_ATTRS_STRUCT sChannels;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)psObj;

    // Initialize the iterator's argument
    OSAL.bMemSet(&sChannels, 0, sizeof(sChannels));

    // Get number of channels
    CCACHE_bIterateChannels(psObj->hCCache,
                FALSE,
                bCountChannelsIterator,
                (void *)&sChannels);

    do
    {
        // If the LOCKED asked
        if ((bLocked == TRUE) && (sChannels.un16NumLocked > 0))
        {
            sChannels.ptLocked = (SERVICE_ID *)
                SMSO_hCreate(
                    "ChanLockAttrs", sizeof(SERVICE_ID) * sChannels.un16NumLocked, SMS_INVALID_OBJECT, FALSE);

            if (sChannels.ptLocked == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Failed to allocate Channel Locked Attributes array");
                break;
            }

            // Initialize channels array
            for (un32Index = 0; un32Index < sChannels.un16NumLocked; un32Index++)
            {
                sChannels.ptLocked[un32Index] = SERVICE_INVALID_ID;
            }
        }

        // If the SKIPPED asked
        if ((bSkipped == TRUE) && (sChannels.un16NumSkipped > 0))
        {
            sChannels.ptSkipped = (SERVICE_ID *)
                SMSO_hCreate(
                    "ChanSkipAttrs", sizeof(SERVICE_ID) * sChannels.un16NumSkipped, SMS_INVALID_OBJECT, FALSE);

            if (sChannels.ptSkipped == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Failed to allocate Channel Skipped Attributes array");
                break;
            }

            // Initialize channels array
            for (un32Index = 0; un32Index < sChannels.un16NumSkipped; un32Index++)
            {
                sChannels.ptSkipped[un32Index] = SERVICE_INVALID_ID;
            }
        }

        // Reset number of channels
        sChannels.un16NumLocked = 0;
        sChannels.un16NumSkipped = 0;

        bSuccess = CCACHE_bIterateChannels(
            psObj->hCCache, FALSE,
            bSetChannelAttrsIter,
           (void *)&sChannels
                );

        if (bSuccess == TRUE)
        {
            if (bLocked == TRUE)
            {
                RADIO_bLockChannel(
                    hDecoder,
                    &sChannels.ptLocked[0],
                    sChannels.un16NumLocked);

                // Clear pending attribute flag
                psObj->tPendingAttrs &= ~CHANNEL_OBJECT_ATTRIBUTE_LOCKED;
            }

            if (bSkipped == TRUE)
            {
                RADIO_bSkipChannel(
                    hDecoder,
                    &sChannels.ptSkipped[0],
                    sChannels.un16NumSkipped);

                // Clear pending attribute flag
                psObj->tPendingAttrs &= ~CHANNEL_OBJECT_ATTRIBUTE_SKIPPED;
            }
        }

    } while (FALSE);

    if (sChannels.ptLocked != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)sChannels.ptLocked);
    }

    if (sChannels.ptSkipped != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)sChannels.ptSkipped);
    }

    return;
}

/*****************************************************************************
 *
 *   bSetChannelAttrsIter
 *
 *****************************************************************************/
static BOOLEAN bSetChannelAttrsIter (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    SERVICE_ID tServiceId;

    DECODER_CHANNEL_ATTRS_STRUCT *psChannels =
        (DECODER_CHANNEL_ATTRS_STRUCT *)pvArg;

    if ((hChannel == CHANNEL_INVALID_OBJECT) ||
        (pvArg == NULL))
    {
        return FALSE;
    }

    tServiceId = CHANNEL.tServiceId(hChannel);

    if (tServiceId != SERVICE_INVALID_ID)
    {
        // LOCKED requested?
        if (psChannels->ptLocked != NULL)
        {
            BOOLEAN bIsLocked = FALSE;

            CHANNEL_eIsLockedUnconfirmed(hChannel, &bIsLocked);

            if (bIsLocked == TRUE)
            {
                psChannels->ptLocked[psChannels->un16NumLocked++] = tServiceId;
            }
        }

        // SKIPPED requested?
        if (psChannels->ptSkipped != NULL)
        {
            BOOLEAN bIsSkipped = FALSE;

            CHANNEL_eIsSkippedUnconfirmed(hChannel, &bIsSkipped);

            if (bIsSkipped == TRUE)
            {
                psChannels->ptSkipped[psChannels->un16NumSkipped++] = tServiceId;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   bCountChannelsIterator
 *
 *****************************************************************************/
static BOOLEAN bCountChannelsIterator (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        )
{
    BOOLEAN bIsLocked = FALSE;
    BOOLEAN bIsSkipped = FALSE;

    SERVICE_ID tServiceId;

    DECODER_CHANNEL_ATTRS_STRUCT *psChannels =
        (DECODER_CHANNEL_ATTRS_STRUCT *)pvArg;

    if ((hChannel == CHANNEL_INVALID_OBJECT) ||
        (pvArg == NULL))
    {
        return FALSE;
    }

    tServiceId = CHANNEL.tServiceId(hChannel);

    if (tServiceId != SERVICE_INVALID_ID)
    {
        SMSAPI_RETURN_CODE_ENUM eRetrunCode;

        eRetrunCode = CHANNEL_eIsLockedUnconfirmed(hChannel, &bIsLocked);
        if ((eRetrunCode == SMSAPI_RETURN_CODE_SUCCESS) && 
            (bIsLocked == TRUE))
        {
            psChannels->un16NumLocked++;
        }

        eRetrunCode = CHANNEL_eIsSkippedUnconfirmed(hChannel, &bIsSkipped);
        if ((eRetrunCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (bIsSkipped == TRUE))
        {
            psChannels->un16NumSkipped++;
        }
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   vTuneScanEventHandler
 *
 *****************************************************************************/
static void vTuneScanEventHandler (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_SCAN_STRUCT const *psEventData
        )
{
    BOOLEAN bSuccess;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)psObj;

    if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_ABORT)
    {
        bSuccess = RADIO_bScanTerminate(hDecoder, TRUE);
        if (bSuccess == TRUE)
        {
            // Set scan status to non-active
            psObj->sTuneScan.bActive = FALSE;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to send Scan Abort request.\n",
                psObj->pacObjectName);
        }
    }
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_CFG)
    {
        bSuccess = RADIO_bScanSelectCfg(
            hDecoder,
            TRUE,
            psEventData->bScanLockedMature,
            psEventData->bScanLockedMature,
            psEventData->bScanSkipped
                );

        bSuccess &= RADIO_bSetPlaySeconds(
            hDecoder,
            psEventData->un8PlaySeconds
                );

        bSuccess &= RADIO_bScanItemsMon(hDecoder, TRUE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to send Tune Scan Cfg request.\n",
                psObj->pacObjectName);

            // Set Error
            vSetError(psObj, DECODER_ERROR_CODE_RADIO_ERROR);
        }

        // Update internal Radio Data configuration
        if (bSuccess == TRUE)
        {
            psObj->sTuneScan.bScanLockedMature = psEventData->bScanLockedMature;
            psObj->sTuneScan.bScanSkipped = psEventData->bScanSkipped;
            psObj->sTuneScan.un8PlaySeconds = psEventData->un8PlaySeconds;
        }
    }
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_FORWARD)
    {
        bSuccess = RADIO_bScanForward(hDecoder);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to send Scan Forward request.\n",
                psObj->pacObjectName);
        }
    }
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_BACKWARD)
    {
        bSuccess = RADIO_bScanBack(hDecoder, TRUE);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to send Scan Backward request.\n",
                psObj->pacObjectName);
        }
    }
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_START)
    {
        if (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES)
        {
            // Request content scan
            bSuccess = RADIO_bScanContent(hDecoder, TRUE);
            if (bSuccess == TRUE)
            {
                // Reset scan abort state
                DECODER_vUpdateTuneScanTerminateState(hDecoder, FALSE);
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                    "%s: Failed to send Scan Content request.\n",
                    psObj->pacObjectName);
            }
        }
        else if (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_ALL_CHANNELS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Channel Scan Requested - not supported now.\n",
                psObj->pacObjectName);
        }
    }
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_STOP)
    {
        bSuccess = RADIO_bScanTerminate(hDecoder, FALSE);
        if (bSuccess == TRUE)
        {
            // Set scan status to non-active
            psObj->sTuneScan.bActive = FALSE;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 1,
                "%s: Failed to send Scan Stop request.\n",
                psObj->pacObjectName);
        }
    }
#if SMS_DEBUG == 1
    else if (psEventData->eEventType == TUNE_SCAN_EVENT_TYPE_PRINT)
    {
        vPrintTuneScan(psObj);
    }
#endif // SMS_DEBUG == 1

    return;
}

/*****************************************************************************
 *
 *   eTuneSelf
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTuneSelf(
    DECODER_OBJECT_STRUCT *psObj,
    BOOLEAN bMatureOverride,
    BOOLEAN bLockedOverride,
    BOOLEAN bSkippedOverride
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    SMS_EVENT_TUNE_STRUCT sTune;

    do
    {
        // First determine which service id we can tune to
        if(psObj->hTunedChannel != CHANNEL_INVALID_OBJECT)
        {
            sTune.tServiceId = CHANNEL.tServiceId(psObj->hTunedChannel);
            sTune.tChannelId = CHANNEL.tChannelId(psObj->hTunedChannel);
            if ((sTune.tServiceId != CHANNEL_INVALID_ID) &&
                (sTune.tChannelId != SERVICE_INVALID_ID))
            {
                // we have valid tuned channel, so
                // it is not necessary to re-tune the self
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                break;
            }
        }

        sTune.tServiceId = CHANNEL.tServiceId(psObj->hLastTunedChannel);

        // Configure tune
        sTune.tChannelId = CHANNEL_INVALID_ID;
        sTune.bLockedOverride = bLockedOverride;
        sTune.bMatureOverride = bMatureOverride;
        sTune.bSkippedOverride = bSkippedOverride;
        sTune.bPlayUnrestricted = FALSE;

        // Request the tune and hope for the best
        eReturnCode = eQualifyTuneRequest(psObj, &sTune);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Try another channel
            sTune.tServiceId = GsRadio.sChannel.tDefaultServiceID;
            eReturnCode = eQualifyTuneRequest(psObj, &sTune);
            if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // Try another channel
                sTune.tServiceId = GsRadio.sChannel.tSafeServiceID;
                eReturnCode = eQualifyTuneRequest(psObj, &sTune);
                if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    // Fail!
                    vSetError(psObj, DECODER_ERROR_CODE_RADIO_ERROR);
                }
            }
        }

        // If the channel exists, tune it
        if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            BOOLEAN bTuned;

            bTuned = bTuneServiceId(psObj, &sTune);
            if(bTuned == FALSE)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Unable to tune service-id (%u).",
                    sTune.tServiceId);

                eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            }
        }
    } while(0);

    if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
    {
        printf(DECODER_OBJECT_NAME
            ": Self Tune completed with Service ID: %d\n",
            sTune.tServiceId);
        RADIO_vSelfTuneComplete((DECODER_OBJECT)psObj);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   vTimeAvailableNotification
*
*****************************************************************************/
static void vTimeAvailableNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        )
{
    BOOLEAN bValid;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)pvArg;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        BOOLEAN bPosted;

        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)pvArg;

        bPosted = SMSE_bPostSignal(psObj->hEventHdlr,
            SMS_EVENT_SYNCHRONIZE_TIME,
            SMS_EVENT_OPTION_NONE);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to send time sync event.");
        }
    }

    return;
}

/*****************************************************************************
*
*   vMinuteTickerCallback
*
*   Inputs:
*       hTimer - The timer which generated this callback.
*       pvArg - A pointer to an argument used when this function was registered
*           as a timer callback.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void vMinuteTickerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    BOOLEAN bValid;
    DECODER_OBJECT hDecoder = (DECODER_OBJECT)pvArg;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)hDecoder);
    if(bValid == TRUE)
    {
        DECODER_OBJECT_STRUCT *psObj = (DECODER_OBJECT_STRUCT *)hDecoder;

        SMSE_bPostSignal(psObj->hEventHdlr, SMS_EVENT_UPDATE_TIME,
            SMS_EVENT_OPTION_NONE);
    }

    return;
}

/*****************************************************************************
*
*   bStartMinuteTicker
*
*****************************************************************************/
static BOOLEAN bStartMinuteTicker (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    UN32 un32Seconds;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Create a 'minute-ticker' timer, if we haven't
    if(psObj->hMinuteTicker == OSAL_INVALID_OBJECT_HDL)
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        snprintf(&acName[0], sizeof(acName),
                 "%s:MinuteTicker", psObj->pacObjectName);

        eReturnCode =
            OSAL.eTimerCreate(
                &psObj->hMinuteTicker,
                &acName[0],
                vMinuteTickerCallback,
                (DECODER_OBJECT)psObj
                    );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            vStopMinuteTicker(psObj);
            return FALSE;
        }
    }

    // Let's register for time updates, if we havn't
    if(psObj->hTimeNotificationHandle ==
        OSAL_TIME_NOTIFICATION_INVALID_OBJECT)
    {
        eReturnCode = OSAL.eTimeSetRegisterNotification(
            &psObj->hTimeNotificationHandle,
            OSAL_TIME_UPDATE_MASK_ALL,
            (OSAL_TIME_UPDATE_HANDLER)vTimeAvailableNotification,
            psObj
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to register for time updates.");

            // Error!
            vStopMinuteTicker(psObj);
            return FALSE;
        }
    }

    // Get time from the system (seconds since epoch)
    // We don't really care about GMT/UTC time, we just need
    // the minutes anyway.
    eReturnCode = OSAL.eTimeGet(&un32Seconds);
    if((eReturnCode == OSAL_SUCCESS) ||
        (eReturnCode == OSAL_ERROR_INVALID_TIME))
    {
        UN8 un8SecondsElapsedInThisMinute;
        UN8 un8InitialOffsetSec = 0;

        // Start the 'minute-update' timer.

        // Configure it to fire one-minute from now on a minute interval
        // by adjusting the initial offset
        un8SecondsElapsedInThisMinute = un32Seconds % 60;
        if(un8SecondsElapsedInThisMinute != 0)
        {
            un8InitialOffsetSec = 59 - un8SecondsElapsedInThisMinute;
        }

        // Start timer at next occurring one minute interval from now
        eReturnCode =
            OSAL.eTimerStartRelative(
                    psObj->hMinuteTicker,
                    un8InitialOffsetSec * 1000,
                    60 * 1000 // rate = 60 seconds
                );
    }

    // Check success
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        vStopMinuteTicker(psObj);
        return FALSE;
    }
    else
    {
        return TRUE;
    }
}

/*****************************************************************************
*
*   vStopMinuteTicker
*
*****************************************************************************/
static void vStopMinuteTicker (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    // Unregister for any time updates
    if(OSAL_TIME_NOTIFICATION_INVALID_OBJECT !=
        psObj->hTimeNotificationHandle)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Unregister
        eReturnCode = OSAL.eTimeSetUnRegisterNotification(
            psObj->hTimeNotificationHandle);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hTimeNotificationHandle =
                OSAL_TIME_NOTIFICATION_INVALID_OBJECT;
        }
    }

    // Destroy 'minute-ticker'
    if(psObj->hMinuteTicker != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eTimerDelete(psObj->hMinuteTicker);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psObj->hMinuteTicker = OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/*****************************************************************************
*
*   bUpdateCategoryFallback
*
*****************************************************************************/
static BOOLEAN bUpdateCategoryFallback(
    DECODER_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    DECODER_UPDATE_HANDLE_ENUM eUpdate
        )
{
    // Let's take a look on a network category
    // of this channel.
    BOOLEAN bUpdated = FALSE;
    CATEGORY_OBJECT hCategory;

    // Extract network catgegory
    hCategory = CHANNEL.hCategory(hChannel, 0);
    if (hCategory != CATEGORY_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
            "Network Category: %d exist for Channel Id: %d\n",
            CATEGORY.tGetCategoryId(hCategory), CHANNEL.tChannelId(hChannel));

        // Network category exist.
        bUpdated = TRUE;
    }
    else
    {
        // Netwotk category does not exist. Try to find another one,
        // containing the browsed channel
        N16 n16Categories;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = CHANNEL.eNumCategories(hChannel, &n16Categories);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Network Category doesn't exist for Channel Id: %d\n",
                CHANNEL.tChannelId(hChannel));

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "Number of Categories for Channel Id: %d = %d\n",
                CHANNEL.tChannelId(hChannel), n16Categories);

            // If we here, then the network catgeory is invalid,
            // but it still will be in count. So, let's see behind it.
            if (n16Categories > 1)
            {
                // Extract the first category
                hCategory = CHANNEL.hCategory(hChannel, 1);
                if (hCategory != CATEGORY_INVALID_OBJECT)
                {
                    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                        "Using 1st Category: %d for Channel Id: %d\n",
                        CATEGORY.tGetCategoryId(hCategory), CHANNEL.tChannelId(hChannel));

                    // This channel added into some SMS category.
                    bUpdated = TRUE;
                }
                else
                {
                    // Since the number of categories > 1,
                    // this should not happen...
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DECODER_OBJECT_NAME": Failed to get SMS category."
                        " Channel ID: %d, Number of categories: %d",
                        CHANNEL.tChannelId(hChannel),
                        n16Categories);
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                    "No more categories left for Channel Id: %d\n",
                    CHANNEL.tChannelId(hChannel));

                // No more categories left, which containing
                // this channel. Bail...
                bUpdated = TRUE;
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Failed to get the number of categories"
                " on Channel ID: %d: %s (#%d)",
                CHANNEL.tChannelId(hChannel),
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode), eReturnCode);
        }
    }

    if (bUpdated == TRUE)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
             "Updating handles[ Mode: %d, Channel Id: %d, Category: %d ]\n",
            eUpdate, CHANNEL.tChannelId(hChannel), CATEGORY.tGetCategoryId(hCategory));

        vUpdateHandles(psObj, eUpdate, hChannel, hCategory);
    }

    return bUpdated;
}

/*****************************************************************************
 *
 *   vFireACOEvent
 *
 *****************************************************************************/
static void vFireACOEvent(
    OSAL_OBJECT_HDL hTimer, 
    void *pvArg
        )
{
    BOOLEAN bValid;

    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;
    
    bValid = SMSO_bValid((SMS_OBJECT)pvArg);
    if (bValid == TRUE)
    {
        // De-reference object
        DECODER_OBJECT_STRUCT *psObj = 
            (DECODER_OBJECT_STRUCT *)pvArg;

        // Allocate an event
        hEvent = SMSE_hAllocateEvent(
            psObj->hEventHdlr, 
            SMS_EVENT_CCACHE, 
            &puEventData, 
            SMS_EVENT_OPTION_NONE
                );

        if (hEvent != SMS_INVALID_EVENT_HDL)
        {
            BOOLEAN bPosted;

            // Set event data
            puEventData->uDecoder.sCache.eType = CCACHE_EVENT_TYPE_ACO;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 5,
                "%s: Posting ACO timer event.\n",
                psObj->pacObjectName);

            // Post event
            bPosted = SMSE_bPostEvent(hEvent);
            if (bPosted == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DECODER_OBJECT_NAME": Failed to post ACO event.");
            }
        }
    }

    return;
}

/*****************************************************************************
 *
 *   bReleaseParentModule
 *
 *****************************************************************************/
static BOOLEAN bReleaseParentModule (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult = TRUE;

    // If this DECODER is associated with the MODULE, release the MODULE.
    if (0 != (psObj->tCurrentModes & SMS_MODE_PARENT_ASSOCIATED))
    {
        bResult = MODULE_bRelease(psObj->hModule, SMS_OBJECT_RELEASE_BY_OTHERS);
        if (TRUE == bResult)
        {
            psObj->tCurrentModes &= ~SMS_MODE_PARENT_ASSOCIATED;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: RELEASE request is posted to MODULE.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to release MODULE.");
        }
    }

    return bResult;
}

/* Shutdown management */

/*****************************************************************************
 *
 *   bObjectBusy
 *
 *****************************************************************************/
static BOOLEAN bObjectBusy (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    // Check if this object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: BUSY: Not being released.\n", psObj->pacObjectName);
        return TRUE;
    }

    //  Check reference counter
    if (0 != psObj->un16RefCounter)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: BUSY: Reference Counter: %u\n", psObj->pacObjectName,
            psObj->un16RefCounter);
        return TRUE;
    }

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: DECODER is ready for shutdown.\n", psObj->pacObjectName);

    return FALSE;
}

/*****************************************************************************
 *
 *   bTryStop
 *
 *****************************************************************************/
static BOOLEAN bTryStop (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult;

    // Check if this object is currently utilized
    bResult = bObjectBusy(psObj);
    if (TRUE == bResult)
    {
        // Cannot stop now.
        // Waiting for shutdown completion.
        return FALSE;
    }

    if ((MODULE_INVALID_OBJECT != psObj->hModule) &&
        (0 != (psObj->tCurrentModes & SMS_MODE_PARENT_ADDED)))
    {
        // Tell owning MODULE that it should now remove this DECODER.
        // MODULE shall respond with UNASSOCIATE event to confirm
        // that this DECODER is removed.
        bResult = MODULE_bRemoveDecoder(psObj->hModule, (DECODER_OBJECT)psObj);
        if (TRUE == bResult)
        {
            // Clear flag
            psObj->tCurrentModes &= ~SMS_MODE_PARENT_ADDED;

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: REMOVE request is posted to MODULE. "
                "Waiting for UNASSOCIATE event.\n",
                psObj->pacObjectName);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Cannot remove DECODER (%s).",
                MACRO_TO_STRING(DECODER_ERROR_CODE_UNABLE_TO_REMOVE_DECODER));
        }

        // Now waiting for UNASSOCIATE event

        return bResult;
    }

    // Now this object is ready for shutdown.
    // But we should dispatch all remaining events.
    // by posting of special event which shall be dispatched
    // last from the queue.
    bResult = bPostFinalStop(psObj);
    if (FALSE == bResult)
    {
        // Transition to ERROR state
        vSetError(psObj, DECODER_ERROR_CODE_EVENT_POST_ERROR);
    }

    // Now waiting for FINAL STOP event.

    return bResult;
}

/*****************************************************************************
 *
 *   bInitiateRelease
 *
 *****************************************************************************/
static BOOLEAN bInitiateObjectRelease (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent = SMSE_hAllocateEvent(psObj->hEventHdlr, SMS_EVENT_RELEASE,
        &puEventData, SMS_EVENT_OPTION_NONE);
    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        // Populate event with caller's parameters
        puEventData->uDecoder.sRelease.eInitiator = eInitiator;

        // Post event
        bResult = SMSE_bPostEvent(hEvent);
        if (TRUE == bResult)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: RELEASE event is posted.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Cannot post event.");
        }
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Cannot allocate event.");
    }

    return bResult;
}

/*****************************************************************************
*
*   bPostFinalStop
*
*****************************************************************************/
static BOOLEAN bPostFinalStop(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult;

    bResult = SMSE_bPostSignal(psObj->hEventHdlr,
        SMS_EVENT_STOP, SMS_EVENT_OPTION_DEFERRED);
    if (TRUE == bResult)
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: FINAL STOP event is posted.\n", psObj->pacObjectName);
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unable to post FINAL STOP.");
    }

    return bResult;
}

/* Event handlers */

/*****************************************************************************
 *
 *   vHandleInitEvent
 *
 *****************************************************************************/
static void vHandleInitEvent(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_INITIALIZE\n", psObj->pacObjectName);
    do
    {
        // Check stopping flag indicating the object is currently being released
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: Object is currently being released. Ignore event.\n",
                psObj->pacObjectName);
            break;
        }

        // Initialize the DECODER object
        bOk = bInitializeObject(psObj);
        if (FALSE == bOk)
        {
            // Transition to error state
            vSetError(psObj, DECODER_ERROR_CODE_INITIALIZATION_ERROR);
            break;
        }

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: DECODER object is initialized.\n", psObj->pacObjectName);

        // Check if we were RESET
        if (0 == (psObj->tCurrentModes & SMS_MODE_RESET))
        {
            // Nothing else to do
            break;
        }

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: DECODER has been reset.\n", psObj->pacObjectName);

        // Tell owning MODULE that we are resetting. Note the above post
        // makes sure the INITIALIZE event gets put on the DECODER's queue first
        // while this post must go through the MODULE which will make the DECODER
        // ready again once the MODULE itself is ready.
        bOk = MODULE_bResetDecoder(psObj->hModule, (DECODER_OBJECT)psObj);
        if (FALSE == bOk)
        {
            // Transition to ERROR state
            vSetError(psObj, DECODER_ERROR_CODE_UNABLE_TO_RESET_MODULE);
            break;
        }

        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: RESET is posted to MODULE.\n", psObj->pacObjectName);

        // Clear mode
        psObj->tCurrentModes &= ~SMS_MODE_RESET;

    } while (FALSE);

    return;
}

/*****************************************************************************
 *
 *   vHandleRadioReadyEvent
 *
 *****************************************************************************/
static void vHandleRadioReadyEvent (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RADIO_READY\n", psObj->pacObjectName);

    do
    {
        // Try to stop if the object is currently
        // being released and ready for shutdown.
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: Object is currently being released."
                " Try to initiate object stop.\n",
                psObj->pacObjectName);

            bTryStop(psObj);
            break;
        }

        // Transition to the READY state
        vUpdateState(psObj, DECODER_STATE_READY);

    } while (FALSE);

    return;
}

/*****************************************************************************
 *
 *   vHandleAssociateEvent
 *
 *****************************************************************************/
static void vHandleAssociateEvent(
    DECODER_OBJECT_STRUCT *psObj,
    const SMS_EVENT_DECODER_ASSOCIATE_STRUCT *psAssociate
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_ASSOCIATE: eSubStatus: %d, tSubReasonCode: %d, "
        "un32SubSuspendDate: %u, hModule: %p, tCapabilities: 0x%X, hSTI: %p\n",
        psObj->pacObjectName, psAssociate->eSubStatus,
        psAssociate->tSubReasonCode, psAssociate->un32SubSuspendDate,
        psAssociate->hModule, psAssociate->tCapabilities, psAssociate->hSTI);

    do
    {
        // Only accept ASSOCIATE from the MODULE this DECODER is added to.
        // Otherwise tell the MODULE it can be released.
        if (psAssociate->hModule == psObj->hModule)
        {
            // Set approprite flag to not forget to post release to the MODULE.
            // If set, the DECODER shall release owning MODULE
            // by calling MODULE_bRelease().
            psObj->tCurrentModes |= SMS_MODE_PARENT_ASSOCIATED;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": ASSOCIATE from unknown MODULE.");

            MODULE_bRelease(psAssociate->hModule, SMS_OBJECT_RELEASE_BY_OTHERS);
            break;
        }

        // Check stopping flag indicating the object is currently being released
        if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: Object is currently being released. Ignore event.\n",
                psObj->pacObjectName);
            break;
        }

        // Copy in object parameters
        psObj->tCapabilities = psAssociate->tCapabilities;
        psObj->hSTI = psAssociate->hSTI;

        // Has the RADIO already been initialized?
        if (TRUE == psObj->bRadioInitialized)
        {
            // Done handling this event
            break;
        }

        // Perform RADIO specific initializations. Once the RADIO
        // is successfully initialized, a RADIO_READY event will
        // get posted.
        psObj->bRadioInitialized =
            RADIO_bInitializeDecoder((DECODER_OBJECT)psObj,
                psObj->tCapabilities, psObj->hSTI,
                psAssociate->eSubStatus,
                psAssociate->tSubReasonCode,
                psAssociate->un32SubSuspendDate,
                psObj->hEventHdlr);
        if (TRUE == psObj->bRadioInitialized)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: RADIO initialization is requested. "
                "Waiting for RADIO_READY event.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Transition to error state
            vSetError(psObj, DECODER_ERROR_CODE_RADIO_ERROR);
            break;
        }

        // Initialize DECODER configuration/setup using the RADIO
        psObj->bRadioInitialized = bInitializeDecoder(psObj);
        if (FALSE == psObj->bRadioInitialized)
        {
            // Transition to error state
            vSetError(psObj, DECODER_ERROR_CODE_INITIALIZATION_ERROR);
            break;
        }

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vHandleUnassociateEvent
*
*****************************************************************************/
static void vHandleUnassociateEvent(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bResult;

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_UNASSOCIATE\n", psObj->pacObjectName);

    // Now this object is ready for shutdown.
    // But we should dispatch all remaining events.
    // by posting of special event which shall be dispatched
    // last from the queue.
    bResult = bPostFinalStop(psObj);
    if (FALSE == bResult)
    {
        // Error!
        vSetError(psObj, DECODER_ERROR_CODE_EVENT_POST_ERROR);
    }

    // Now waiting for the FINAL STOP event.

    return;
}

/*****************************************************************************
*
*   vHandleReleaseEvent
*
*****************************************************************************/
static void vHandleReleaseEvent(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RELEASE: eInitiator: %d\n",
        psObj->pacObjectName, eInitiator);

    if ((SMS_OBJECT_RELEASE_BY_APP == eInitiator) ||
        (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator))
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: Handle APPLICATION initiated release...\n",
            psObj->pacObjectName);

        vHandleAppInitiatedRelease(psObj, eInitiator);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: Handle GENERAL release...\n",
            psObj->pacObjectName);

        vHandleGeneralRelease(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vHandleAppInitiatedRelease
*
*****************************************************************************/
static void vHandleAppInitiatedRelease(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        )
{
    BOOLEAN bResult;

    // This is release request initiated by the application
    // directly or by the parent object release.

    // Check stopping flag indicating the object is currently being released
    if (0 == (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        // This is the first release request initiated
        // by the Application or by the Parent object (MODULE)
        psObj->tCurrentModes |= SMS_MODE_STOPPING;
    }
    else
    {
        if (SMS_OBJECT_RELEASE_BY_APP == eInitiator)
        {
            // This never must happen!
            // This means incorrect and dangerous API usage!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": OBJECT IS ALREADY BEING RELEASED "
                "DIRECTLY BY THE APPLICATION OR BY THE PARENT OBJECT!");
        }
        else if (SMS_OBJECT_RELEASE_BY_PARENT == eInitiator)
        {
            // This is Parent initiated release occurred after
            // application initiated release, so just ignore it.

            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: Object is already being released. "
                "PARENT initiated release is ignored.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Incorrect release request.");
        }

        return;
    }

    if (TRUE == psObj->bSubscribedToDS)
    {
        // Tell the DSM we'd like to unsubscribe now
        bResult = DATASERVICE_MGR_bPostEvent(
            DATASERVICE_MGR_INVALID_OBJECT,
            DATASERVICE_FW_EVENT_DECODER_UNSUBSCRIBED,
            (void *)(DECODER_OBJECT)psObj);
        if (TRUE == bResult)
        {
            SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
                "%s: UNSUBSCRIBE request is posted to DSM.\n",
                psObj->pacObjectName);
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DECODER_OBJECT_NAME": Unable to unsubscribe from DSM.");

            // Transition to ERROR state
            vSetError(psObj, DECODER_ERROR_CODE_EVENT_POST_ERROR);
        }
    }

    // Perform common release routine
    vHandleGeneralRelease(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleGeneralRelease
*
*****************************************************************************/
static void vHandleGeneralRelease(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    // Check for unnecessary release request
    if (0 == psObj->un16RefCounter)
    {
        // Error! This means incorrect logic somewhere inside the SMS.
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DECODER_OBJECT_NAME": Unnecessary release!");
        return;
    }

    // Now reference counter can be decremented
    psObj->un16RefCounter--;

    // Try to stop if the object is currently
    // being released and ready for shutdown.
    bTryStop(psObj);

    return;
}

/*****************************************************************************
*
*   vHandleStopEvent
*
*****************************************************************************/
static void vHandleStopEvent(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_STOP\n", psObj->pacObjectName);

    // Now everything is uninitialized.
    // So the object can be safely destroyed.

    // If this DECODER is associated with MODULE,
    // we have to release MODULE now.
    bReleaseParentModule(psObj);

    // Remove tune mixes. Do not care much about return code
    (void)DECODER_eIterateTuneMixList((DECODER_OBJECT)psObj,
        bRemoveTuneMix, NULL);

    // Stop the Sports Flash service if it is active (check for NULL
    // happens inside). This will also set psObj->hSportsFlash 
    // to SPORTS_FLASH_INVALID_OBJECT
    SPORTS_FLASH_vStop(psObj->hSportsFlash);

    // Stop the TW Now service if it is active (check for NULL
    // happens inside). This will also set psObj->hTWNow
    // to TW_NOW_INVALID_OBJECT
    TW_NOW_vStop(psObj->hTWNow);

    // Run specific RADIO uninitialization
    if (TRUE == psObj->bRadioInitialized)
    {
        RADIO_vUninitializeDecoder((DECODER_OBJECT)psObj);
        psObj->bRadioInitialized = FALSE;
    }

    // Stop this DECODER now
    vUpdateState(psObj, DECODER_STATE_RELEASED);

    return;
}

/*****************************************************************************
 *
 *   vHandleResetEvent
 *
 *****************************************************************************/
static void vHandleResetEvent(
    DECODER_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
        "%s: SMS_EVENT_RESET\n", psObj->pacObjectName);

    // Check stopping flag indicating the object is currently being released
    if (0 != (psObj->tCurrentModes & SMS_MODE_STOPPING))
    {
        SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4,
            "%s: Object is currently being released. Ignore event.\n",
            psObj->pacObjectName);

        return;
    }

    // Indicate we have been instructed to RESET
    psObj->tCurrentModes |= SMS_MODE_RESET;

    // Transition to INITIAL now.
    vUpdateState(psObj, DECODER_STATE_INITIAL);

    // If this DECODER is associated with MODULE,
    // we have to release MODULE now.
    bReleaseParentModule(psObj);

    return;
}

/*****************************************************************************
 *
 *   bRemoveTuneMix
 *
 *****************************************************************************/
static BOOLEAN bRemoveTuneMix (
    TUNEMIX_OBJECT hTuneMix,
    void *pvArg
        )
{
    TUNEMIX_vDeallocate(hTuneMix);
    return TRUE;
}

#if SMS_DEBUG==1
/*****************************************************************************
 *
 *   vPrintTuneScan
 *
 *****************************************************************************/
static void vPrintTuneScan (
    DECODER_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bDebugEnabled;

    // Remember output enable state
    bDebugEnabled = OSAL.bOutputEnabledThisTask();

    // Enable debug output
    OSAL.vControlOutputThisTask(TRUE);

    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "********************************");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "           Tune Scan            ");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "********************************");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Active: %s\n",
        (psObj->sTuneScan.bActive == TRUE) ? "Yes" : "No");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Configured: %s\n",
        (psObj->sTuneScan.bConfigured == TRUE) ? "Yes" : "No");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Play Seconds: %d\n",
        psObj->sTuneScan.un8PlaySeconds);
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Scan Style: %s\n",
        (psObj->sTuneScan.eScanStyle == DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES) ?
            "Smarts" : "All Channels");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Locked / Mature Override: %s\n",
        (psObj->sTuneScan.bScanLockedMature == TRUE) ? "Enabled" : "Disabled");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Skipped Override: %s\n",
        (psObj->sTuneScan.bScanSkipped == TRUE) ? "Enabled" : "Disabled");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Content Available: %s\n",
        (psObj->sTuneScan.tStatusMask & DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE) ? "Yes" : "No");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, " Scan Aborted: %s\n",
        (psObj->sTuneScan.tStatusMask & DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED) ? "Yes" : "No");
    SMSAPI_DEBUG_vPrint(DECODER_OBJECT_NAME, 4, "********************************");

    // Disable output only if it was disabled when we started
    if(bDebugEnabled == FALSE)
    {
        // Disable debug output
        OSAL.vControlOutputThisTask(FALSE);
    }
    return;
}

#endif // SMS_DEBUG==1
