/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:DATASERVICE_MGR implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include <stdio.h>
#include <string.h>

#include "srh.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms.h"
#include "sms_obj.h"
#include "sms_task.h"
#include "decoder_obj.h"
#include "module_obj.h"
#include "radio.h"
#include "dataservice_base.h"
#include "dsrl_obj.h"
#include "device_obj.h"
#include "location_obj.h"
#include "dsrl_entry_obj.h"
#include "radio_data_service.h"

#include "dataservice_mgr_obj.h"
#include "_dataservice_mgr_obj.h"
#include "dataservice_mgr_impl.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hStart
*
*   This function will create all the basics needed for this service to
*   operate.  However, all initial processing to actually get this service
*   running is done at a later time.
*
*****************************************************************************/
static DATASERVICE_MGR_OBJECT hStart (
    const char *pacSRHDriverName,
    DSI tDSI,
    BOOLEAN bEnableAll,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    GENERIC_DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        )
{
    BOOLEAN bPosted = FALSE;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)NULL;

    if ( (tDSI >= GsRadio.sMisc.tDsiMin) &&
         (tDSI <= GsRadio.sMisc.tDsiMax) )
    {
        DATASERVICE_CREATE_STRUCT sCreateService;

        // Populate the creation structure
        DATASERVICE_IMPL_vInitCreateStruct(&sCreateService);

        sCreateService.pacSRHDriverName = pacSRHDriverName;
        sCreateService.bEnableAll = bEnableAll;
        sCreateService.tDataID = (DATASERVICE_ID)tDSI;
        sCreateService.tEventRequestMask = tEventRequestMask;
        sCreateService.vEventCallback = vEventCallback;

        // Create the manager object resources for this DSI
        psObj = psCreate( &sCreateService, pvEventCallbackArg );

        if (psObj != (DATASERVICE_MGR_OBJECT_STRUCT *)NULL)
        {
            // Kick off the process with the INITIAL event
            bPosted = DATASERVICE_MGR_bPostEvent(
                (DATASERVICE_MGR_OBJECT)psObj, DATASERVICE_FW_EVENT_INITIAL, NULL);
        }
    }

    if (bPosted == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post INITIAL event.");

        vUninitObject(psObj);
        psObj = NULL;
    }

    return (DATASERVICE_MGR_OBJECT)psObj;
}

/*****************************************************************************
*
*   vStop
*
*   This function stops the data service manager and causes resources to be
*   released.  The handle to the object should be treated as invalid.
*
*****************************************************************************/
static void vStop (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bOk;

    bOk = SMSO_bValid((SMS_OBJECT)hManager);

    if (bOk == TRUE)
    {
        // Post the event to stop this service
        bOk = DATASERVICE_MGR_bPostEvent(
            hManager, DATASERVICE_FW_EVENT_STOP, NULL
                );

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to post stop event.");
        }
    }
    return;
}

/*****************************************************************************
*
*   eState
*
*   This function returns the current state of the data service manager.
*
*****************************************************************************/
static DATASERVICE_STATE_ENUM eState (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bLocked;
    DATASERVICE_STATE_ENUM eState = DATASERVICE_STATE_INVALID;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hManager, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        eState = psObj->eState;

        SMSO_vUnlock((SMS_OBJECT)hManager);
    }

    return eState;
}

/*****************************************************************************
*
*   eErrorCode
*
*   This function returns the current error code of the data service manager.
*
*****************************************************************************/
static DATASERVICE_ERROR_CODE_ENUM eErrorCode (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bLocked;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_UNKNOWN;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hManager, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bLocked == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        eErrorCode = psObj->eErrorCode;

        SMSO_vUnlock((SMS_OBJECT)hManager);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   eManageDataStream
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eManageDataStream (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
    size_t tNumDMIsToConfigure
        )
{
    BOOLEAN bValid, bPosted = FALSE, bAllocate = FALSE;
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg =
        (SMS_EVENT_DATASERVICE_ARG_UNION *)NULL;
    DATASERVICE_DMI_CONFIG_STRUCT *psDMIs = (DATASERVICE_DMI_CONFIG_STRUCT *)NULL;

    // Validate object
    bValid = SMSO_bValid((SMS_OBJECT)hDataService);

    if (FALSE == bValid)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Validate the input as much as we can
    if ((NULL == psDMIsToConfigure) ||
        (0 == tNumDMIsToConfigure))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // We have to allocate memory for the event data
    // if the number of DMIs the caller wishes to configure is
    // too large for the event's built-in array
    if (tNumDMIsToConfigure > SMS_EVENT_DATASERVICE_DMI_ARRAY_SIZE)
    {
        // Allocate the event data memory now
        psDMIs = (DATASERVICE_DMI_CONFIG_STRUCT *)
            SMSO_hCreate(DATASERVICE_MGR_OBJECT_NAME":DMI List",
            tNumDMIsToConfigure * sizeof(DATASERVICE_DMI_CONFIG_STRUCT),
            SMS_INVALID_OBJECT, FALSE);

        bAllocate = TRUE;

        if ((DATASERVICE_DMI_CONFIG_STRUCT *)NULL == psDMIs)
        {
            return SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
        }
    }

    // Allocate this event now
    hEvent = hAllocateEvent(
        hDataService, DATASERVICE_FW_EVENT_MANAGE_DATA_STREAM, &puEventArg);

    // Verify we have an event to populate and post.
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Populate the argument for the event
        puEventArg->sStreamConfig.tNumDMIs = tNumDMIsToConfigure;

        // If we allocated then we need to free as well
        puEventArg->sStreamConfig.bFreeDMIs = bAllocate;

        // Setup the DMI pointer now
        if (FALSE == bAllocate)
        {
            // Point to the array in the event data
            puEventArg->sStreamConfig.pasDMIs = &puEventArg->sStreamConfig.asDMIs[0];
        }
        else
        {
            // Point to the newly allocated memory
            puEventArg->sStreamConfig.pasDMIs = psDMIs;
        }

        // Copy the DMIs over to the destination now
        OSAL.bMemCpy(
            &puEventArg->sStreamConfig.pasDMIs[0],
            &psDMIsToConfigure[0],
            tNumDMIsToConfigure * sizeof(DATASERVICE_DMI_CONFIG_STRUCT));

        // Post the data service event
        bPosted = SMSE_bPostEvent(hEvent);
    }

    if (FALSE == bPosted)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            "DATASERVICE_MGR_bPostEvent() Cannot post event.");

        if (TRUE == bAllocate)
        {
            SMSO_vDestroy((SMS_OBJECT)psDMIs);
        }

        return SMSAPI_RETURN_CODE_ERROR;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eEnableProduct
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnableProduct (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK tProductMask
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    DATASERVICE_IMPL_HDL hServiceImpl;
    BOOLEAN bValid;

    // The given hManager is assumed to be the IMPL handle
    // because this API is intended to be used with non-generic data
    // services which return IMPL handles (instead of MGR_OBJECT in generic case)
    hServiceImpl = (DATASERVICE_IMPL_HDL)hManager;

    do
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj;
        DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;
        SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT sEventArg;
        BOOLEAN bPosted;
        DATASERVICE_STATE_ENUM eState;

        // This function is a part of public API.
        // However, since this function operates constant data only,
        // just ensure that the provided handle is valid.
        bValid = DATASERVICE_IMPL_bIsValid(hServiceImpl);
        if (TRUE != bValid)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // restore pointers from impl
        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        hManager = (DATASERVICE_MGR_OBJECT)psObj;

        // Enabling of products is allowed in active states
        eState = DATASERVICE_IMPL_eState(hServiceImpl);
        if ((DATASERVICE_STATE_ERROR == eState) ||
            (DATASERVICE_STATE_STOPPED == eState) ||
            (DATASERVICE_STATE_INVALID == eState))
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

        // Data used by psFindProduct function is not changed
        // thru all of the service life. So, it safe to use it
        // from any thread as along as the service is alive.
        psProduct = psFindProduct(psObj, eProductType);
        if (NULL == psProduct)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
            break;
        }

        if (DATA_PRODUCT_MASK_NONE != (tProductMask & ~psProduct->tAllowedMask))
        {
            eReturnCode = SMSAPI_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        sEventArg.eProductType = eProductType;
        sEventArg.tMask = tProductMask;

        bPosted = DATASERVICE_MGR_bPostEvent(
            hManager,
            DATASERVICE_FW_EVENT_PRODUCT_ENABLE,
            &sEventArg);

        if (FALSE == bPosted)
        {
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
            break;
        }

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   eDisableProduct
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisableProduct (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    DATASERVICE_IMPL_HDL hServiceImpl;
    BOOLEAN bValid;

    // The given hManager is assumed to be the IMPL handle
    // because this API is intended to be used with non-generic data
    // services which return IMPL handles (instead of MGR_OBJECT in generic case)
    hServiceImpl = (DATASERVICE_IMPL_HDL)hManager;

    // This function is a part of public API.
    // However, since this function operates constant data only,
    // just ensure that the proviced handle is valid.
    bValid = DATASERVICE_IMPL_bIsValid(hServiceImpl);
    if (TRUE == bValid)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj;
        DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;

        // restore pointers from impl
        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        hManager = (DATASERVICE_MGR_OBJECT)psObj;

        // Data used by psFindProduct function is not changed
        // thru all of the service life. So, it safe to use it
        // from any thread as along as the service is alive.
        psProduct = psFindProduct(psObj, eProductType);
        if (NULL != psProduct)
        {
            BOOLEAN bPosted;

            bPosted = DATASERVICE_MGR_bPostEvent(
                hManager,
                DATASERVICE_FW_EVENT_PRODUCT_DISABLE,
                (void*)eProductType);

            if (TRUE == bPosted)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eProductState
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProductState (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK *ptProductMask,
    DATA_PRODUCT_STATE_ENUM *peProductState
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    DATASERVICE_IMPL_HDL hServiceImpl;
    BOOLEAN bLocked;

    if ((NULL == ptProductMask) || (NULL == peProductState))
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    // The given hManager is assumed to be the IMPL handle
    // because this API is intended to be used with non-generic data
    // services which return IMPL handles (instead of MGR_OBJECT in generic case)
    hServiceImpl = (DATASERVICE_IMPL_HDL)hManager;

    // Since this is a part of public API which can be called from
    // any thread and it calls thread-unsafe internal functions,
    // locking is necessary
    bLocked = DATASERVICE_IMPL_bLock(hServiceImpl);
    if (TRUE == bLocked)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj;
        DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;

        // restore pointers from impl
        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        psProduct = psFindProduct(psObj, eProductType);
        if (NULL != psProduct)
        {
            *ptProductMask = psProduct->tMask;
            *peProductState = psProduct->eState;
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_FOUND;
        }

        DATASERVICE_IMPL_vUnlock(hServiceImpl);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterateProducts
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateProducts (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    DATASERVICE_IMPL_HDL hServiceImpl;
    BOOLEAN bLocked;

    if (NULL == bCallback)
    {
        return SMSAPI_RETURN_CODE_BAD_ARGUMENT;
    }

    // The given hManager is assumed to be the IMPL handle
    // because this API is intended to be used with non-generic data
    // services which return IMPL handles (instead of MGR_OBJECT in generic case)
    hServiceImpl = (DATASERVICE_IMPL_HDL)hManager;

    // Since this is a part of public API which can be called from
    // any thread and it calls thread-unsafe internal functions,
    // locking is necessary
    bLocked = DATASERVICE_IMPL_bLock(hServiceImpl);
    if (TRUE == bLocked)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        DATASERVICE_MGR_PRODUCT_ITERATOR_ARG sArg;

        // restore pointers from impl
        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        sArg.hManager = hManager;
        sArg.bCallback = bCallback;
        sArg.pvCallbackArg = pvCallbackArg;

        eOsalReturnCode = OSAL.eLinkedListIterate(
            psObj->sProducts.hProducts,
            (OSAL_LL_ITERATOR_HANDLER)bProductIterator,
            &sArg);

        DATASERVICE_IMPL_vUnlock(hServiceImpl);

        if (OSAL_SUCCESS == eOsalReturnCode)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
        else if ((OSAL_NO_OBJECTS == eOsalReturnCode) ||
                 (OSAL_ERROR_INVALID_HANDLE == eOsalReturnCode))
        {
            eReturnCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eStartTimedEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eStartTimedEvent (
    DATASERVICE_MGR_OBJECT hManager,
    BOOLEAN bRepeat,
    UN32 un32IntervalInSeconds
        )
{
    BOOLEAN bPosted;

    SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT sEventArg;
    sEventArg.bRepeat = bRepeat;
    sEventArg.un32IntervalInSeconds = un32IntervalInSeconds;

    bPosted = DATASERVICE_MGR_bPostEvent(
        hManager,
        DATASERVICE_FW_EVENT_START_TIMED_EVENT,
        &sEventArg);

    if ( FALSE == bPosted )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post start timed event.");

        return SMSAPI_RETURN_CODE_ERROR;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   vStopTimedEvent
*
*****************************************************************************/
static void vStopTimedEvent (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bPosted;

    // Stop timed event takes no arguments.
    bPosted = DATASERVICE_MGR_bPostEvent(
        hManager,
        DATASERVICE_FW_EVENT_STOP_TIMED_EVENT,
        NULL);

    if ( FALSE == bPosted )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post stop timed event.");
    }

    return;
}

/*****************************************************************************
*
*   eSendServiceEvent
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSendServiceEvent (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bPosted;

    bPosted = DATASERVICE_MGR_bPostEvent(
        hManager,
        DATASERVICE_FW_EVENT_SERVICE_SPECIFIC,
        NULL);

    if ( FALSE == bPosted )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post service specific event.");

        return SMSAPI_RETURN_CODE_ERROR;
    }

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   DATASERVICE_MGR_hInitialize
*
*****************************************************************************/
SMS_OBJECT DATASERVICE_MGR_hInitialize(
    SMS_EVENT_HANDLER *phEventHandler,
    DEVICE_OBJECT *phDeviceObject
        )
{
    BOOLEAN bInitOk;
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl =
        (DATASERVICE_MGR_CTRL_STRUCT *)NULL;

    if (phEventHandler == NULL)
    {
        return SMS_INVALID_OBJECT;
    }

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "%s(phEventHandler: %p, phDeviceObject: %p)\n",
        __FUNCTION__, phEventHandler, phDeviceObject);

    // Create an instance of this object
    psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)
        SMSO_hCreate(
            DATASERVICE_MGR_OBJECT_NAME,
            sizeof(DATASERVICE_MGR_CTRL_STRUCT),
            SMS_INVALID_OBJECT, // Parent(root)
            TRUE ); // Lock
    if(psCtrl == NULL)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to allocate memory");
        return SMS_INVALID_OBJECT;
    }

    // Initialize data we will use for data services
    psCtrl->sObjectInfo.hSRHDriverName = STRING_INVALID_OBJECT;
    psCtrl->sObjectInfo.hModule = MODULE_INVALID_OBJECT;
    psCtrl->sObjectInfo.bModuleInError = FALSE;
    psCtrl->sObjectInfo.bModuleAssociated = FALSE;

    // We have not yet loaded the device group
    psCtrl->bDeviceGroupLoaded = FALSE;

    // Initialize time notification handle
    psCtrl->hTimeNotificationHandle = OSAL_TIME_NOTIFICATION_INVALID_OBJECT;

    do
    {
        // Set up data service table
        bInitOk = bCreateDataServiceTable(psCtrl);
        if (bInitOk == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to create Service Table.");
            break;
        }

        // Create the cached data service subscription list
        bInitOk = bInitCachedSubscriptionList(psCtrl);
        if (bInitOk == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to init Subscription List.");
            break;
        }

        // Create the timed event list
        bInitOk = bInitTimedEventList(psCtrl);
        if (bInitOk == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to init Timed Event List.");
            break;
        }

        // Create the device object
        psCtrl->sDevice.hDeviceObject = DEVICE_hInstall((SMS_OBJECT)psCtrl);
        if (psCtrl->sDevice.hDeviceObject == DEVICE_INVALID_OBJECT)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to install DEVICE.");
            break;
        }

        psCtrl->sDevice.sLastUpdateTime.un16Msec = 0;
        psCtrl->sDevice.sLastUpdateTime.un32Sec = 0;

        // Create the data service task
        bInitOk = bInitializeTask(psCtrl);
        if (bInitOk == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to initialize DSM Task.");
            break;
        }

        // Give these handles to the caller
        *phEventHandler = psCtrl->hEventHdlr;
        *phDeviceObject = psCtrl->sDevice.hDeviceObject;

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s(): DSM is initialized (%p).\n",
            __FUNCTION__, psCtrl);

        return (SMS_OBJECT)psCtrl;

    } while (FALSE);

    // Error!
    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
        DATASERVICE_MGR_OBJECT_NAME": DSM initialization failed.");

    // Something went wrong
    vDestroyDSMCtrl(psCtrl, TRUE);

    return SMS_INVALID_OBJECT;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vUnInitialize
*
*****************************************************************************/
void DATASERVICE_MGR_vUnInitialize(
    SMS_OBJECT hDSMCtrl
        )
{
    BOOLEAN bLocked;
    SMS_TASK_HANDLE hServicesTask = SMS_INVALID_TASK_HANDLE;

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "%s(hDSMCtrl: %p)\n", __FUNCTION__, hDSMCtrl);

    // Verify and lock the DS MGR control object
    bLocked = SMSO_bLock(
        hDSMCtrl, OSAL_OBJ_TIMEOUT_INFINITE );
    if(bLocked == TRUE)
    {
        DATASERVICE_MGR_CTRL_STRUCT *psCtrl =
            (DATASERVICE_MGR_CTRL_STRUCT *)hDSMCtrl;

        // Copy out task handle and invalidate
        hServicesTask = psCtrl->hServicesTask;
        psCtrl->hServicesTask = SMS_INVALID_TASK_HANDLE;

        // Destroy the control contents, but
        // not the object itself (the task will
        // do that when it shuts down)
        vDestroyDSMCtrl(psCtrl, FALSE);

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s(): DSM Ctrl object is un-initialized.\n", __FUNCTION__);

        // Release our hold on the control object
        SMSO_vUnlock(hDSMCtrl);
    }

    // Uninstall Task
    if(hServicesTask != SMS_INVALID_TASK_HANDLE)
    {
        // Task shutdown itself will destroy the object (DATASERVICE_MGR_CTRL_STRUCT)
        // so there is no need to handle that here. This function
        // is called from the application task context, so it will
        // request the SMS-Task to be deleted (shutdown). Once the
        // task is shutdown the last thing it does is destroy the
        // SMS-Object provided at the time of SMS-task creation.
        SMST_vUninstall(hServicesTask);

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s(): DSM Task is destroyed.\n", __FUNCTION__);
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bPostDevicePositionEvent
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bPostDevicePositionEvent (
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg =
        (SMS_EVENT_DATASERVICE_ARG_UNION *)NULL;
    SMS_EVENT_HDL hEvent;

    // Allocate this event now
    hEvent = hAllocateEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_DEVICE_POSITION,
        &puEventArg);

    // Verify we have an event to populate and post.
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        N32 n32Value;
        UN8 un8Bits;

        // Extract the attributes of the fixed object
        n32Value = OSAL_FIXED.n32Value(hLat);
        un8Bits = OSAL_FIXED.un8NumFractionalBits(hLat);

        // Place the fixed object into memory (won't fail)
        OSAL_FIXED.hCreateInMemory(
            n32Value, un8Bits,
            &puEventArg->sPosition.atLatData[0]);

        // Extract the attributes of the fixed object
        n32Value = OSAL_FIXED.n32Value(hLon);
        un8Bits = OSAL_FIXED.un8NumFractionalBits(hLon);

        // Place the fixed object into memory (won't fail)
        OSAL_FIXED.hCreateInMemory(
            n32Value, un8Bits,
            &puEventArg->sPosition.atLonData[0]);

        // Post the data service event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "DATASERVICE_MGR_bPostDevicePositionEvent() Cannot post event.");
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_hAllocateRadioEvent
*
*****************************************************************************/
SMS_EVENT_HDL DATASERVICE_MGR_hAllocateRadioEvent(
    SMS_EVENT_TYPE_ENUM eEvent,
    EVENT_OPTIONS_TYPE tOptions,
    void **ppvEventData
        )
{
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATA_UNION *puEventData;

    // Allocate an event
    hEvent = SMS_hAllocateDSMEvent(
        eEvent, (void **)&puEventData, tOptions);
    if (hEvent != SMS_INVALID_EVENT_HDL)
    {
        *ppvEventData = (void *)&puEventData->sRadio.apvData[0];
    }

    return hEvent;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_hAllocateEvent
*
*****************************************************************************/
SMS_EVENT_HDL DATASERVICE_IMPL_hAllocateEvent(
    DATASERVICE_IMPL_HDL hServiceImpl,
    void **ppvEventData
        )
{
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
    BOOLEAN bValid;

    // Validate event data pointer
    if (ppvEventData == NULL)
    {
        return FALSE;
    }

    // Validate
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg;

        // Allocate this event now
        hEvent = hAllocateEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_SERVICE_SPECIFIC, &puEventArg);
        if (hEvent != SMS_INVALID_EVENT_HDL)
        {
            *ppvEventData = (void *)&puEventArg->sImpl.apvData[0];
        }
    }
    return hEvent;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bPostEvent
*
*   Provides data service implementation objects the ability to post
*   events to the DSM
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bPostEvent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_FW_EVENT_ENUM eEvent,
    void *pvDataEventArg
        )
{
    BOOLEAN bPosted = FALSE, bValid;

    // Validate
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bPosted = DATASERVICE_MGR_bPostEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            eEvent, pvDataEventArg);
    }

    return bPosted;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vDecoderSubscribed
*
*****************************************************************************/
void DATASERVICE_IMPL_vDecoderSubscribed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bOwner;
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl =
        (DATASERVICE_MGR_CTRL_STRUCT *)NULL;

    // Validate ownership
    bOwner = DATASERVICE_IMPL_bOwner(hServiceImpl);

    if (TRUE == bOwner)
    {
        // Grab the DSM control object
        psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMS_hUseDSMCtrl();

        bOwner = SMSO_bOwner((SMS_OBJECT)psCtrl);
    }

    if (TRUE == bOwner)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;
            DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder;

            // Locate this decoder now
            psDecoder = psFindDecoderEntry(psCtrl, hDecoder);

            if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
            {
                break;
            }

            // Attempt to add to the service list
            // for this decoder
            eReturnCode = OSAL.eLinkedListAdd(
                psDecoder->hSubscriptions,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psObj);

            if (OSAL_ERROR_LIST_ITEM_NOT_UNIQUE == eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": DATASERVICE_IMPL_vUpdateDecoderSubscription()"
                    " service attempted to subscribe decoder multiple times");
            }
            else if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": DATASERVICE_IMPL_vUpdateDecoderSubscription()"
                    " unable to update decoder subscriptions");
            }

        } while (FALSE);
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bPostEvent
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bPostEvent (
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_FW_EVENT_ENUM eEvent,
    void *pvDataEventArg
        )
{
    BOOLEAN bPosted = FALSE;
    SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg =
        (SMS_EVENT_DATASERVICE_ARG_UNION *)NULL;
    SMS_EVENT_HDL hEvent;

    // Verify event type
    if (eEvent == DATASERVICE_FW_EVENT_DEVICE_POSITION)
    {
        // Don't use this function for these!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": DATASERVICE_MGR_bPostEvent() can't be used for this event (%u)",
            eEvent);
        return FALSE;
    }

    // Allocate this event now
    hEvent = hAllocateEvent(hManager, eEvent, &puEventArg);

    // Verify we have an event to populate and post.
    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Provide the argument for the event
        switch (eEvent)
        {
            case DATASERVICE_FW_EVENT_PRODUCT_ENABLE:
            {
                puEventArg->sProduct =
                    *(SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT*)pvDataEventArg;
            }
            break;

            case DATASERVICE_FW_EVENT_START_TIMED_EVENT:
            {
                puEventArg->sTimed =
                    *(SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT*)pvDataEventArg;
            }
            break;

            case DATASERVICE_FW_EVENT_SXI_MESSAGE:
            {
                puEventArg->uSxiMessage =
                    *(SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION*)pvDataEventArg;
            }
            break;

            default:
            {
                puEventArg->sStandard.pvArg = pvDataEventArg;
            }
            break;
        }

        // Post the data service event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "DATASERVICE_MGR_bPostEvent() Cannot post event.");
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vDSIStateChange
*
*****************************************************************************/
void DATASERVICE_MGR_vDSIStateChange (
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eState
        )
{
    do
    {
        BOOLEAN bOwner;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
        DATASERVICE_MGR_DSI_REF_STRUCT *psDSI;
        DATASERVICE_MGR_CTRL_STRUCT *psCtrl;

        // Are we the owner of this object?
        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);

        if (FALSE == bOwner)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": DATASERVICE_MGR_vDSIStateChange() not owner");
            break;
        }

        // Get the control object
        psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMS_hUseDSMCtrl();
        if ((DATASERVICE_MGR_CTRL_STRUCT *)NULL == psCtrl)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": DATASERVICE_MGR_vDSIStateChange() can't get control object");
            break;
        }

        // Find this DSI now
        psDSI = psFindDSI(psObj, tDSI);

        if ((DATASERVICE_MGR_DSI_REF_STRUCT *)NULL == psDSI)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": DATASERVICE_MGR_vDSIStateChange() invalid DSI");
            break;
        }

        vSetDSIState(psCtrl, psObj, psDSI, eState);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vReceivePayload
*
*****************************************************************************/
void DATASERVICE_MGR_vReceivePayload (
    DATASERVICE_MGR_OBJECT hManager,
    DSI tDSI,
    OSAL_BUFFER_HDL hPayload
        )
{
    do
    {
        BOOLEAN bOwner;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj;

        bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
        if (FALSE == bOwner)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": DATASERVICE_MGR_vReceivePayload() not owner");
            break;
        }

        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        if ((psObj->eState != DATASERVICE_STATE_READY) &&
            (psObj->eState != DATASERVICE_STATE_POI_UPDATES_ONLY))
        {
            // service is not ready to receive data
            break;
        }

        if (TRUE == psObj->bMultiDSIRequired)
        {
            // add DSI to the payload head
            size_t tBytesWritten;

            tBytesWritten = OSAL.tBufferWriteHead(
                hPayload, &tDSI, sizeof(tDSI));

            if (sizeof(tDSI) != tBytesWritten)
            {
                // unable to add DSI -- drop this data
                break;
            }
        }

        vCallEventHandler( psObj, DATASERVICE_EVENT_NEW_DATA, hPayload );
        return;

    } while (FALSE);

    // something wrong happened -- cleanup
    DATASERVICE_IMPL_bFreeDataPayload(hPayload);
    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_hRegisterTimedEvent
*
*   Data service implementations must use this function to create timed-based
*   events. This friend function will create timed events for data
*   service implementations; however, this function just delegates to the
*   private hRegisterTimedEvent after validating the service.
*
*****************************************************************************/
DATASERVICE_TIMED_EVENT_HDL DATASERVICE_IMPL_hRegisterTimedEvent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    void *pvTimerEventArg
        )
{
    BOOLEAN bValid;

    // Validate
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);

    if ( TRUE == bValid )
    {
        // Get our service object from the implementation
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            ((DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl) - 1;

        // As this function is just a shim, there's no need to
        // do error reporting WRT/the returned timed event.
        // That happens in the layer above and below ...
        return hRegisterTimedEvent((DATASERVICE_MGR_OBJECT)psObj,
                pvTimerEventArg );
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": Invalid service handle passed to "
            "DATASERVICE_IMPL_hRegisterTimedEvent");
    }

    return DATASERVICE_TIMED_EVENT_INVALID_HDL;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bSetTimedEvent
*
*   Tells the data service framework to set a timed event so that it
*   occurs in the designated time period.
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bSetTimedEvent (
    DATASERVICE_TIMED_EVENT_HDL hEvent,
    BOOLEAN bResetIfStarted,
    BOOLEAN bRepeatEvent,
    UN32 un32SecondsUntilEvent
        )
{
    BOOLEAN bSuccess = FALSE;

    // Prevent overflows
    if (un32SecondsUntilEvent > DSM_MAX_TIMEOUT_DURATION_IN_SECONDS)
    {
        return FALSE;
    }

    do
    {
        DATASERVICE_TIMED_EVENT_STRUCT *psTimed =
            (DATASERVICE_TIMED_EVENT_STRUCT *)hEvent;
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bStartTimer = TRUE;
        UN32 un32Interval = 0;
        UN32 un32MSUntilEvent = un32SecondsUntilEvent * DSM_MSECS_PER_SEC;

        // Verify we got a valid handle
        if (hEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
        {
            break;
        }

        if (bRepeatEvent == TRUE)
        {
            un32Interval = un32MSUntilEvent;
        }

        // Should we reset the timer if it
        // has already been started?
        if (bResetIfStarted == FALSE)
        {
            // Is the timer active?
            eReturnCode = OSAL.eTimerRemaining(
                psTimed->hTimer, (UN32 *)NULL);

            if (eReturnCode == OSAL_SUCCESS)
            {
                // We don't need to start this timer again
                bStartTimer = FALSE;
            }
            else if (eReturnCode != OSAL_TIMER_NOT_ACTIVE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": DATASERVICE_MGR_bRegisterTimedEvent() unable to query timer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        if (bStartTimer == TRUE)
        {
            // Start the timer
            eReturnCode =
                OSAL.eTimerStartRelative(
                psTimed->hTimer, un32MSUntilEvent, un32Interval);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": DATASERVICE_MGR_bRegisterTimedEvent() unable to start timer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }

        bSuccess = TRUE;
    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bStopTimedEvent
*
*   Tells the data service framework to stop a timed event.
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bStopTimedEvent (
    DATASERVICE_TIMED_EVENT_HDL hEvent
        )
{
    BOOLEAN bStopped = FALSE;
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed =
        (DATASERVICE_TIMED_EVENT_STRUCT *)hEvent;

    // Verify we got a valid handle
    if (hEvent == DATASERVICE_TIMED_EVENT_INVALID_HDL)
    {
        // Nothing to do -- just tell 'em it has stopped.
        return TRUE;
    }

    // We have a valid handle here, so try
    // to stop the event
    if (psTimed->hTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Stop this timer now
        eReturnCode = OSAL.eTimerStop(psTimed->hTimer);

        // Both of these return codes tell us the timer
        // is stopped
        if ((eReturnCode == OSAL_SUCCESS) ||
            (eReturnCode == OSAL_TIMER_NOT_ACTIVE))
        {
            bStopped = TRUE;
        }
    }

    return bStopped;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bAssociateModule
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bAssociateModule (
    const char *pacDriverName,
    MODULE_OBJECT hModule,
    STI_HDL hSTI
        )
{
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg =
        (SMS_EVENT_DATASERVICE_ARG_UNION *)NULL;
    BOOLEAN bPosted = FALSE;
    STRING_OBJECT hDriverName;

    // Duplicate the name of the driver
    hDriverName = STRING.hCreate(pacDriverName, 0);
    if (STRING_INVALID_OBJECT == hDriverName)
    {
        return FALSE;
    }

    // Allocate the event
    hEvent = hAllocateEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_MODULE_ASSOCIATE, &puEventArg);
    if (SMS_INVALID_EVENT_HDL != hEvent)
    {
        puEventArg->sModule.hDriverName = hDriverName;
        puEventArg->sModule.hModule = hModule;
        puEventArg->sModule.hSTI = hSTI;

        // Post the data service event
        bPosted = SMSE_bPostEvent(hEvent);
    }

    if (FALSE == bPosted)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "DATASERVICE_MGR_bAssociateModule() Cannot alloc/post event.");
        STRING.vDestroy(hDriverName);
    }

    return bPosted;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vUnassociateModule
*
*****************************************************************************/
void DATASERVICE_MGR_vUnassociateModule (
    MODULE_OBJECT hModule
        )
{
    SMS_EVENT_HDL hEvent;
    SMS_EVENT_DATASERVICE_ARG_UNION *puEventArg =
        (SMS_EVENT_DATASERVICE_ARG_UNION *)NULL;
    BOOLEAN bPosted = FALSE;

    // Allocate the event
    hEvent = hAllocateEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_MODULE_UNASSOCIATE, &puEventArg);
    if (hEvent != SMS_INVALID_EVENT_HDL)
    {
        puEventArg->sModule.hModule = hModule;

        // Post the data service event
        bPosted = SMSE_bPostEvent(hEvent);
        if(bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                "DATASERVICE_MGR_vUnassociateModule() Cannot post event.");
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vInitCreateStruct
*
*****************************************************************************/
void DATASERVICE_IMPL_vInitCreateStruct (
    DATASERVICE_CREATE_STRUCT *psCreate
        )
{
    if (psCreate != NULL)
    {
        psCreate->bEnableAll = TRUE;
        psCreate->bTimeRequired = TRUE;
        psCreate->pacSRHDriverName = NULL;
        psCreate->pacServiceObjectName = DEFAULT_DATA_SERVICE_OBJECT_NAME;
        psCreate->bRequiresDeviceGroup = FALSE;
        psCreate->tEventRequestMask = 0;
        psCreate->tServiceObjectSize = 0;
        psCreate->tSuggestedOTABufferByteSize = DSM_DEFAULT_OTA_MEMORY_SIZE;
        psCreate->vEventCallback = NULL;
        psCreate->tDataID = DATASERVICE_INVALID_ID;
        psCreate->bMultiDSIRequired = FALSE;
        psCreate->sProductsInfo.peProducts = NULL;
        psCreate->sProductsInfo.un8Count = 0;
        psCreate->sProductsInfo.bGetDSIForProduct = NULL;
        psCreate->sProductsInfo.eGetNextProductState = NULL;
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bProcessOptions
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bProcessOptions (
    DATASERVICE_OPTIONS_MASK tAllowedOptions,
    DATASERVICE_OPTIONS_STRUCT const *psOptions,
    DATASERVICE_OPTION_VALUES_STRUCT *psOptionValues
        )
{
    if (psOptionValues == NULL)
    {
        // No arguments?
        return FALSE;
    }

    // Set option default values
    psOptionValues->bUpdateRefDB = TRUE;
    psOptionValues->pcRefDBPath = NULL;
    psOptionValues->eImageOutputFormat = IMAGE_INVALID_FORMAT;

    if (psOptions == NULL)
    {
        // Let caller use default values, then
        return TRUE;
    }

    // Verify we have been provided only valid options
    // for this service
    if((psOptions->tMask & ~tAllowedOptions) != DATASERVICE_OPTION_NONE)
    {
        return FALSE;
    }

    // Process the arguments in order (by option value)

    // Reference Database Path & Update Flag
    if ((psOptions->tMask & DATASERVICE_OPTION_REFERENCE_DB_PATH)
            == DATASERVICE_OPTION_REFERENCE_DB_PATH)
    {
        BOOLEAN bSuccess = FALSE;

        do
        {
            size_t tPathLen;

            // Grab the provided path
            if (psOptions->pcRefDBPath == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": No argument provided for option "
                        "DATASERVICE_OPTION_REFERENCE_DB_PATH");
                break;
            }

            // Get the length, verify it
            tPathLen = strlen(psOptions->pcRefDBPath);
            if (tPathLen == 0)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": Invalid argument provided for option "
                        "DATASERVICE_OPTION_REFERENCE_DB_PATH");
                break;
            }

            // Allocate memory to store this path
            psOptionValues->pcRefDBPath =
                   (char *) SMSO_hCreate(
                        DATASERVICE_MGR_OBJECT_NAME":RefDBPath",
                        tPathLen + 1, SMS_INVALID_OBJECT, FALSE);
            if (psOptionValues->pcRefDBPath == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": unable to allocate memory");
                break;
            }

            // Copy the string
            snprintf(psOptionValues->pcRefDBPath,
                tPathLen + 1, "%s", psOptions->pcRefDBPath);

            bSuccess = TRUE;
        } while (FALSE);

        if (bSuccess == FALSE)
        {
            if (psOptionValues->pcRefDBPath != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psOptionValues->pcRefDBPath);
                psOptionValues->pcRefDBPath = (char *)NULL;
            }

            return FALSE;
        }
    }

    if ((psOptions->tMask & DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES)
            == DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES)
    {
        // Grab the update flag
        psOptionValues->bUpdateRefDB = FALSE;
    }

    if ((psOptions->tMask & DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT)
            == DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT)
    {
        // Grab the image format
        psOptionValues->eImageOutputFormat = psOptions->eImageOutputFormat;
    }


    return TRUE;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vFreeOptions
*
*****************************************************************************/
void DATASERVICE_IMPL_vFreeOptions (
    DATASERVICE_OPTION_VALUES_STRUCT *psOptions
        )
{
    BOOLEAN bValid;

    if ((DATASERVICE_OPTION_VALUES_STRUCT *)NULL == psOptions)
    {
        return;
    }

    // Validate the path option argument
    bValid = SMSO_bValid((SMS_OBJECT)psOptions->pcRefDBPath);
    if (TRUE == bValid)
    {
        // Need to free it
        SMSO_vDestroy((SMS_OBJECT)psOptions->pcRefDBPath);
        psOptions->pcRefDBPath = (char *)NULL;
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_hCreateNewService
*
*   This function is invoked whenever an SMS-internal data service wants
*   to start.  The caller provides identifying information regarding the
*   service it represents and this function creates a DATASERVICE_MGR_OBJECT
*   to handle that service and creates an SMS_OBJECT for the caller which
*   they can use as their service manager memory space.  The SMS_OBJECT
*   created is set as a child to the DATASERVICE_MGR_OBJECT in the call
*   to SMSO_hCreate.  This allows data services to get a handle to a valid
*   DATASERVICE_MGR_OBJECT which has been assigned to them and  create their
*   own object before the data service actually starts.
*
*****************************************************************************/
SMS_OBJECT DATASERVICE_IMPL_hCreateNewService (
    DATASERVICE_CREATE_STRUCT *psCreateService
        )
{
    SMS_OBJECT hServiceObject = SMS_INVALID_OBJECT;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj;

    // Verify inputs
    if (psCreateService == NULL)
    {
        return SMS_INVALID_OBJECT;
    }

    // Verify more inputs
    if (psCreateService->pacServiceObjectName == NULL)
    {
        return SMS_INVALID_OBJECT;
    }

    // Create the manager object resources for this DSI
    // without the event argument for now
    psObj = psCreate( psCreateService, NULL );
    if (psObj != NULL)
    {
        // The object that the implementation may use
        // as their service object is found immediately
        // after psObj
        hServiceObject = (SMS_OBJECT)(psObj + 1);

        // All internal services utilize this handle
        // as their event handler argument
        psObj->pvEventHandlerArg = (void *)hServiceObject;
    }

    return hServiceObject;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vDestroy
*
*   Data services which have failed the portion of their startup procedure
*   which occurs before invoking DATASERVICE_IMPL_bStart must call this
*   function to clean the data service manager's utilized memory.
*
*****************************************************************************/
void DATASERVICE_IMPL_vDestroy (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid;

    // Now validate the data service object
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        SMS_OBJECT hParent;
        BOOLEAN bLocked;

        // Lock the object
        bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);

        // Get the parent object
        hParent = SMSO_hParent((SMS_OBJECT)psObj);

        if (bLocked == TRUE)
        {
            // Remove this object from the list
            vRemoveManagerFromList((DATASERVICE_MGR_CTRL_STRUCT *)hParent, psObj);

            // Release this object
            vUninitObject(psObj);

            SMSO_vUnlock(hParent);
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vStop
*
*****************************************************************************/
void DATASERVICE_IMPL_vStop (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid;

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);

    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        // Stop this service now
        vStop((DATASERVICE_MGR_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vStopAutonomous
*
*   Starts the shutdown process for autonomous services which utilize
*   the DECODER subscription mechanism
*
*****************************************************************************/
void DATASERVICE_IMPL_vStopAutonomous (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid;

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        BOOLEAN bPosted;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        // Post the event to stop this service
        bPosted = DATASERVICE_MGR_bPostEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_AUTONOMOUS_STOP, NULL
                );

        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to post stop event.");
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vInitializeDSRLConfig
*
*****************************************************************************/
void DATASERVICE_IMPL_vInitializeDSRLConfig (
    DATASERVICE_DSRL_CONFIG_STRUCT *psDSRLConfig
        )
{
    if (psDSRLConfig != NULL)
    {
        psDSRLConfig->bDeviceEnabled = FALSE;
        psDSRLConfig->bFavoritesEnabled = FALSE;
        psDSRLConfig->eDeviceNotifyUnits = DISTANCE_UNIT_TYPE_UNKNOWN;
        psDSRLConfig->eServiceType = DATASERVICE_TYPE_UNKNOWN;
        psDSRLConfig->hCreateTargetFromTag = NULL;
        psDSRLConfig->hGetTagForTarget = NULL;
        psDSRLConfig->tParentObjectSize = 0;
        psDSRLConfig->tServiceDataSize = 0;
        psDSRLConfig->un32DeviceNotifyDistance = 0;
    }

    return;
}
/*****************************************************************************
*
*   DATASERVICE_IMPL_hConfigureDSRL
*
*   This function is invoked in order to specify DSRL parameters for data
*   service implementation objects which utilize the DSRL feature.  This API,
*   if successful, returns the top-level object for use with a service's
*   DSRL(s) and DSRL entries.
*
*****************************************************************************/
SMS_OBJECT DATASERVICE_IMPL_hConfigureDSRL (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_DSRL_CONFIG_STRUCT *psDSRLConfig
        )
{
    BOOLEAN bValid;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
    SMS_OBJECT hDSRLParent = SMS_INVALID_OBJECT;

    // Verify the service object
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if(bValid == FALSE)
    {
        return SMS_INVALID_OBJECT;
    }

    do
    {
        SMS_OBJECT hLockObject;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Verify type
        if (psDSRLConfig->eServiceType >= DATASERVICE_TYPE_UNKNOWN)
        {
            break;
        }

        // Verify favorites if enabled
        if (psDSRLConfig->bFavoritesEnabled == TRUE)
        {
            if ((psDSRLConfig->hGetTagForTarget == NULL) ||
                (psDSRLConfig->hCreateTargetFromTag == NULL))
            {
                // Error!
                break;
            }
        }

        // Verify device if enabled
        if (psDSRLConfig->bDeviceEnabled == TRUE)
        {
            if ((psDSRLConfig->un32DeviceNotifyDistance == 0) ||
                (psDSRLConfig->eDeviceNotifyUnits == DISTANCE_UNIT_TYPE_UNKNOWN))
            {
                // Error!
                break;
            }
        }

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            DATASERVICE_MGR_OBJECT_NAME":ImplDSRL (id: %u)",
            psObj->tDataID);

        // Create the top-level lock object
        hLockObject = SMSO_hCreate(
            &acName[0],
            psDSRLConfig->tParentObjectSize,
            SMS_INVALID_OBJECT, TRUE);

        if (hLockObject == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create the name
        snprintf( &acName[0],
            OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            DATASERVICE_MGR_OBJECT_NAME":MgrDSRL (id: %u)",
            psObj->tDataID);

        // Allocate a child object for us to track
        // DSRL attributes
        psObj->psDSRL = (DATASERVICE_DSRL_SUPPORT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(DATASERVICE_DSRL_SUPPORT_STRUCT),
                hLockObject, FALSE
                    );
        if(psObj->psDSRL == NULL)
        {
            // Error!
            SMSO_vDestroy(hLockObject);
            break;
        }

        // Initialize the object
        psObj->psDSRL->bDevDSRLStoppedByManager = FALSE;
        psObj->psDSRL->eServiceType = psDSRLConfig->eServiceType;
        psObj->psDSRL->sFavorites.hFavoritesTag = TAG_INVALID_OBJECT;
        psObj->psDSRL->tDSRLCreateSize = psDSRLConfig->tServiceDataSize;

        psObj->psDSRL->sFavorites.hGetTagForTarget =
            psDSRLConfig->hGetTagForTarget;
        psObj->psDSRL->sFavorites.hCreateTargetFromTag =
            psDSRLConfig->hCreateTargetFromTag;

        psObj->psDSRL->un32DeviceNotifyDistance =
            psDSRLConfig->un32DeviceNotifyDistance;
        psObj->psDSRL->eDeviceNotifyUnits =
            psDSRLConfig->eDeviceNotifyUnits;

        // Update the service information as well
        psObj->sInfo.bDeviceSupported = psDSRLConfig->bDeviceEnabled;
        psObj->sInfo.bFavoritesSupported = psDSRLConfig->bFavoritesEnabled;

        // Provide this object to the caller
        hDSRLParent = hLockObject;

    } while (FALSE);

    return hDSRLParent;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vDestroyDSRLParent
*
*****************************************************************************/
void DATASERVICE_IMPL_vDestroyDSRLParent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    SMS_OBJECT hDSRLParent
        )
{
    do
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        BOOLEAN bOwner, bValid;

        // Do we own the DSRL parent object?
        bOwner = SMSO_bOwner(hDSRLParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Is the service handle valid?
        bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
        if (bValid == FALSE)
        {
            break;
        }

        // Validate our child object
        bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
        if (bValid == TRUE)
        {
            // We need to clear / destroy it now
            OSAL.bMemSet(psObj->psDSRL, 0,
                sizeof(DATASERVICE_DSRL_SUPPORT_STRUCT));
            SMSO_vDestroy((SMS_OBJECT)psObj->psDSRL);

            psObj->psDSRL = NULL;
        }

        // Destroy the DSRL parent object now
        SMSO_vDestroy(hDSRLParent);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bIsValid
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bIsValid (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid = FALSE;

    if (hServiceImpl != DATASERVICE_IMPL_INVALID_HDL)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bValid = SMSO_bIsValid((SMS_OBJECT)psObj);
    }

    return bValid;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bValid
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bValid (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid = FALSE;

    if (hServiceImpl != DATASERVICE_IMPL_INVALID_HDL)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bValid = SMSO_bValid((SMS_OBJECT)psObj);
    }

    return bValid;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bOwner
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bOwner (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bOwner = FALSE;

    if (hServiceImpl != DATASERVICE_IMPL_INVALID_HDL)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    }

    return bOwner;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bLock
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bLock (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bLocked = FALSE;

    if (hServiceImpl != DATASERVICE_IMPL_INVALID_HDL)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        // Verify and lock the service object
        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE );
    }

    return bLocked;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vUnlock
*
*****************************************************************************/
void DATASERVICE_IMPL_vUnlock (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    if (hServiceImpl != DATASERVICE_IMPL_INVALID_HDL)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        SMSO_vUnlock((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_hSMSObj
*
*****************************************************************************/
SMS_OBJECT DATASERVICE_IMPL_hSMSObj (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    SMS_OBJECT hSMS = SMS_INVALID_OBJECT;
    BOOLEAN bValid;

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        hSMS = (SMS_OBJECT)psObj;
    }

    return hSMS;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_eState
*
*****************************************************************************/
DATASERVICE_STATE_ENUM DATASERVICE_IMPL_eState (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    DATASERVICE_STATE_ENUM eState = DATASERVICE_STATE_INVALID;
    BOOLEAN bLocked;

    bLocked = DATASERVICE_IMPL_bLock(hServiceImpl);
    if (bLocked == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        eState = DATA.eState((DATASERVICE_MGR_OBJECT)psObj);

        DATASERVICE_IMPL_vUnlock(hServiceImpl);
    }

    return eState;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_eErrorCode
*
*****************************************************************************/
DATASERVICE_ERROR_CODE_ENUM DATASERVICE_IMPL_eErrorCode (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    DATASERVICE_ERROR_CODE_ENUM eErrorCode = DATASERVICE_ERROR_UNKNOWN;
    BOOLEAN bLocked;

    bLocked = DATASERVICE_IMPL_bLock(hServiceImpl);
    if (bLocked == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        // Pull the error code from the DSM
        eErrorCode = DATA.eErrorCode((DATASERVICE_MGR_OBJECT)psObj);

        DATASERVICE_IMPL_vUnlock(hServiceImpl);
    }

    return eErrorCode;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bStart
*
*   This function is invoked in order to start a service previously created
*   with a call to DATASERVICE_IMPL_hCreateNewService.
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bStart (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify the service manager
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl );
    if(bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            ((DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl) - 1;

        // Kick off the process with the INITIAL event
        bPosted = DATASERVICE_MGR_bPostEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_INITIAL, NULL);

        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to post INITIAL event.");
        }
    }

    return bPosted;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bOTADataAvailable
*
*   This is used to query the data service manager to see
*   if OTA data is available for this service
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bOTADataAvailable (
    DATASERVICE_IMPL_HDL hServiceImpl
        )
{
    BOOLEAN bOwner, bOTADataAvailable = FALSE;

    // Verify and lock SMS Object
    // Verify caller owns the object
    bOwner =
        DATASERVICE_IMPL_bOwner(hServiceImpl);
    if (bOwner == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bOTADataAvailable = psObj->bOTADataAvailable;
    }

    return bOTADataAvailable;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_eManageDataStream
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM DATASERVICE_IMPL_eManageDataStream (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
    size_t tNumDMIsToConfigure
        )
{
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify the service manager
    bValid = DATASERVICE_IMPL_bValid(hServiceImpl );
    if(bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            ((DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl) - 1;

        // Pass this on
        eReturnCode = DATA.eManageDataStream(
            (DATASERVICE_MGR_OBJECT)psObj,
            psDMIsToConfigure, tNumDMIsToConfigure);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vError
*
*   The service specific managers call this function when they experience
*   an error.
*
*****************************************************************************/
void DATASERVICE_IMPL_vError (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    BOOLEAN bOwner;

    bOwner =
        DATASERVICE_IMPL_bOwner(hServiceImpl);
    if (bOwner == TRUE)
    {
        BOOLEAN bPosted;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bPosted = DATASERVICE_MGR_bPostEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_SERVICE_ERROR,
            (void *)(size_t)eErrorCode);

        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Failed to post data service error event.");
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vProductError
*
*   The service specific managers call this function when they experience
*   an error in one of their products.
*
*****************************************************************************/
void DATASERVICE_IMPL_vProductError (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eProductType
        )
{
    BOOLEAN bOwner;

    bOwner =
        DATASERVICE_IMPL_bOwner(hServiceImpl);
    if (bOwner == TRUE)
    {
        BOOLEAN bPosted;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

        bPosted = DATASERVICE_MGR_bPostEvent(
            (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_PRODUCT_ERROR,
            (void *)(size_t)eProductType);

        if (bPosted == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Failed to post data product error event.");
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bUpdateSubscribed
*
*   This function is called by the owner data service in order to provide
*   updates to the decoders(s) identified by hDecoder.  This function must only
*   be called when the service is already locked.  A return value of FALSE
*   indicates the message didn't go out because nobody is subscribed.
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bUpdateSubscribed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DECODER_OBJECT hDecoder,
    DATASERVICE_IMPL_UPDATE_SUBSCRIBED_ITERATOR bIterator,
    void *pvIterateArg
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_ITERATE_SHIM_STRUCT sShim;

    // Verify input
    if (bIterator == NULL)
    {
        return FALSE;
    }

    // Verify caller owns the object
    bOwner = DATASERVICE_IMPL_bOwner(hServiceImpl);
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // Get the control object
    psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)
        SMSO_hParent((SMS_OBJECT)psObj);
    if (psCtrl == (DATASERVICE_MGR_CTRL_STRUCT *)NULL)
    {
        return FALSE;
    }

    // We are looking for the decoder provided
    sShim.hDecoder = hDecoder;

    // We are iterating on behalf of this service
    sShim.psService = psObj;

    // Use the iterator arguments provided as well
    sShim.bIterator = bIterator;
    sShim.pvIteratorArg = pvIterateArg;

    DATASERVICE_MGR_vLog(
        DATASERVICE_MGR_OBJECT_NAME
        ": %s sending updates to subscribed decoders\n",
        psObj->acServiceName);

    // Notify all the subscribed DECODERs
    // of the update
    eReturnCode = OSAL.eLinkedListIterate(
        psCtrl->hSubscribedDecoders,
        (OSAL_LL_ITERATOR_HANDLER)bIterateSubscribedShim,
        (void *)&sShim );
    if ((eReturnCode == OSAL_SUCCESS) ||
        (eReturnCode == OSAL_NO_OBJECTS))
    {
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_hCreateDSRL
*
*****************************************************************************/
SMS_OBJECT DATASERVICE_IMPL_hCreateDSRL (
    DATASERVICE_IMPL_HDL hServiceImpl,
    size_t tDSRLObjectSize,
    const char *pacDSRLName,
    DSRL_TYPE_ENUM eType,
    void **ppvServiceData
        )
{
    SMS_OBJECT hDSRL = SMS_INVALID_OBJECT;
    BOOLEAN bValid;

    // Validate inputs
    if ((eType >= DSRL_MAX_TYPES) ||
        (ppvServiceData == NULL))
    {
        return SMS_INVALID_OBJECT;
    }

    // Clear the service data pointer
    *ppvServiceData = NULL;

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        BOOLEAN bMayCreate = FALSE, bLocked;
        DATASERVICE_STATE_ENUM eMgrState;

        // Only allow this if the data service is ready.
        eMgrState = DATASERVICE_IMPL_eState(hServiceImpl);
        if (eMgrState != DATASERVICE_STATE_READY)
        {
            return SMS_INVALID_OBJECT;
        }

        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj->psDSRL, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == TRUE)
        {
            DSRL_OBJECT hDummyDSRL = DSRL_INVALID_OBJECT;
            DSRL_OBJECT *phDstDSRL = &hDummyDSRL;

            switch (eType)
            {
                case DSRL_TYPE_FAVORITES:
                {
                    if ((psObj->psDSRL->sManagedDSRLs.hFavorites == DSRL_INVALID_OBJECT) &&
                        (psObj->sInfo.bFavoritesSupported == TRUE))
                    {
                        bMayCreate = TRUE;

                        phDstDSRL = &psObj->psDSRL->sManagedDSRLs.hFavorites;
                    }
                }
                break;

                case DSRL_TYPE_DEVICE:
                {
                    if ((psObj->psDSRL->sManagedDSRLs.hDeviceDSRL == DSRL_INVALID_OBJECT) &&
                        (psObj->sInfo.bDeviceSupported == TRUE))
                    {
                        bMayCreate = TRUE;

                        phDstDSRL = &psObj->psDSRL->sManagedDSRLs.hDeviceDSRL;
                    }
                }
                break;

                case DSRL_TYPE_STANDARD:
                {
                    bMayCreate = TRUE;
                }
                break;

                default:
                    // Nothing to do
                break;
            }

            if (bMayCreate == TRUE)
            {
                char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

                // Create the name for this object
                snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                    "%s:%s", &pacDSRLName[0], &psObj->acServiceName[0] );

                // Grab the DSRL owner from the manager
                // Create memory for the DSRL using that owner
                hDSRL =
                    SMSO_hCreate(
                        &acName[0],
                        tDSRLObjectSize + psObj->psDSRL->tDSRLCreateSize,
                        (SMS_OBJECT)psObj->psDSRL,
                        FALSE );

                if (hDSRL != SMS_INVALID_OBJECT)
                {
                    if (psObj->psDSRL->tDSRLCreateSize != 0)
                    {
                        // compute descriptor pointer
                        *ppvServiceData = (void*)((UN8*)hDSRL + tDSRLObjectSize);
                    }

                    // Put this new handle were we want it
                    *phDstDSRL = (DSRL_OBJECT)hDSRL;
                }
            }

            SMSO_vUnlock((SMS_OBJECT)psObj->psDSRL);
        }
    }

    return hDSRL;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vReportDSRLDestroyed
*
*****************************************************************************/
void DATASERVICE_IMPL_vReportDSRLDestroyed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DSRL_OBJECT hDSRL,
    DSRL_TYPE_ENUM eType
        )
{
    BOOLEAN bValid;

    // Validate input
    if (eType >= DSRL_MAX_TYPES)
    {
        return;
    }

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;
        BOOLEAN bLocked;

        bLocked = SMSO_bLock(
            (SMS_OBJECT)psObj->psDSRL, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == TRUE)
        {
            switch (eType)
            {
                case DSRL_TYPE_FAVORITES:
                {
                    if (psObj->psDSRL->sManagedDSRLs.hFavorites == hDSRL)
                    {
                        psObj->psDSRL->sManagedDSRLs.hFavorites = DSRL_INVALID_OBJECT;
                    }
                }
                break;

                case DSRL_TYPE_DEVICE:
                {
                    if (psObj->psDSRL->sManagedDSRLs.hDeviceDSRL == hDSRL)
                    {
                        psObj->psDSRL->sManagedDSRLs.hDeviceDSRL = DSRL_INVALID_OBJECT;
                    }
                }
                break;

                case DSRL_TYPE_STANDARD:
                default:
                    // Nothing to do
                break;
            }

            SMSO_vUnlock((SMS_OBJECT)psObj->psDSRL);
        }
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bDSRLUpdate
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bDSRLUpdate (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DSRL_ARG_STRUCT *psArg
        )
{
    BOOLEAN bSuccess = FALSE, bValid;

    // Validate input
    if ((psArg == NULL))
    {
        return FALSE;
    }

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            ((DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl) - 1;
        BOOLEAN bMayPost = TRUE;

        // Is this being called in order to interact
        // with a favorites DSRL?
        if (DSRL_TYPE_FAVORITES == psArg->eDSRLType)
        {
            // Extract the favorites supported flag from this
            // manager's read-only data to see if we can
            // go forth with this request.  We only need to worry
            // about favorites in this function because of
            // DSRL_ENTRY.eSetFavoriteByLocId, which doesn't
            // have any built-in protection from attempting
            // to use favorites with services that don't
            // support that feature.
            if (FALSE == psObj->sInfo.bFavoritesSupported)
            {
                bMayPost = FALSE;
            }
        }

        if (TRUE == bMayPost)
        {
            // Post the event now
            bSuccess = DATASERVICE_MGR_bPostEvent(
                (DATASERVICE_MGR_OBJECT)psObj, DATASERVICE_FW_EVENT_DSRL, psArg);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bDeviceRadiusUpdate
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bDeviceRadiusUpdate (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DISTANCE_OBJECT hRadius
        )
{
    BOOLEAN bValid, bSuccess = FALSE;
    DISTANCE_OBJECT hRadiusCopy;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        ((DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl) - 1;

    // Validate inputs
    if (hRadius == DISTANCE_INVALID_OBJECT)
    {
        return FALSE;
    }

    bValid = DATASERVICE_IMPL_bValid(hServiceImpl);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    // Duplicate the radius provided to us
    hRadiusCopy = DISTANCE.hDuplicate(hRadius);
    if (hRadiusCopy == DISTANCE_INVALID_OBJECT)
    {
        return FALSE;
    }

    // Tell the manager the radius was updated
    bSuccess = DATASERVICE_MGR_bPostEvent(
         (DATASERVICE_MGR_OBJECT)psObj,
            DATASERVICE_FW_EVENT_DEVICE_RADIUS,
            hRadiusCopy);

    if (bSuccess == FALSE)
    {
        // Destroy the object we created
        DISTANCE.vDestroy(hRadiusCopy);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bCreateFileBuffer
*
*   This function is called by the owner data service in order to create
*   an OSAL buffer with a FILE * backend.
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bCreateFileBuffer (
    OSAL_OBJECT_HDL *phBlockPool,
    OSAL_BUFFER_HDL *phBuffer,
    size_t tBlockSize
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    static UN32 un32Instance = 0;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Name the block pool
    snprintf( &acName[0], sizeof(acName),
        DATASERVICE_MGR_OBJECT_NAME":FileBuffer: %u", un32Instance++);

    // Create our file reader block pool (just two blocks)
    eReturnCode = OSAL.eBlockPoolCreate(
        phBlockPool,
        &acName[0],
        (UN16)tBlockSize, (UN16)2,
        OSAL_BLOCK_POOL_OPTION_NONE);
    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        return FALSE;
    }

    // Create our buffer
    *phBuffer =
        OSAL.hBufferAllocate(
            *phBlockPool,
            FALSE, // No read blocking
            FALSE, // No write blocking
            OSAL_BUFFER_ALLOCATE_OPTION_NONE );
    if(*phBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        // Error!

        OSAL.eBlockPoolDelete(*phBlockPool);
        *phBlockPool = OSAL_INVALID_OBJECT_HDL;

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vDestroyFileBuffer
*
*****************************************************************************/
void DATASERVICE_MGR_vDestroyFileBuffer (
    OSAL_OBJECT_HDL hBlockPool,
    OSAL_BUFFER_HDL hBuffer
        )
{
    if (hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        if (hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL.eBufferFree(hBuffer);
        }

        OSAL.eBlockPoolDelete(hBlockPool);
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bFillBufferBlock
*
*   This function is called in order to fill an open buffer block (if
*   one is available) with data from the input file
*
*   Returns TRUE if block was updated, FALSE otherwise
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bFillBufferBlock (
    FILE *psInputFile,
    OSAL_BUFFER_HDL hOutputBuffer
        )
{
    OSAL_BUFFER_BLOCK_HDL hBlock;
    UN8 *pun8Data = NULL;
    size_t tBlockSize = 0,
           tBytesRead;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify inputs
    if ((psInputFile == NULL) ||
        (hOutputBuffer == OSAL_INVALID_BUFFER_HDL))
    {
        return FALSE;
    }

    // Get an available block from the buffer
    hBlock = OSAL.hBufferGetBlock(
        hOutputBuffer, &pun8Data, &tBlockSize);
    if (hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
    {
        return FALSE;
    }

    // Read enough bytes to fill the block
    tBytesRead = fread(&pun8Data[0], sizeof(UN8), tBlockSize, psInputFile);

    // Write the block back into the buffer now
    eReturnCode = OSAL.eBufferWriteBlock(hBlock, tBytesRead);

    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bStateFSM
*
*   Finite state machine used for all data service implementation object
*   state transitions
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bStateFSM (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange,
    DATASERVICE_STATE_HANDLERS_STRUCT const *psHandlers,
    void *pvHandlerArg
        )
{
    BOOLEAN bOwner,
            bStateChanged = TRUE,
            bReportStateToCaller = FALSE;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hServiceImpl - 1;

    bOwner = DATASERVICE_IMPL_bOwner(hServiceImpl);
    if (bOwner == FALSE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": State transition outside of DSM task.");
        return FALSE;
    }

    // This first big switch statement helps us determine
    // what state(s) we can transition to from our current state,
    // and it determines if we need to inform the manager of the state transition
    switch (psStateChange->ePreviousState)
    {
        case DATASERVICE_STATE_INVALID:
        {
            switch(psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_INVALID:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_READY:
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_UNSUBSCRIBED:
                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_STOPPED:

                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_STOPPED:
        {
            switch(psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_STOPPED:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_READY:
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_UNSUBSCRIBED:
                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:

                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_INITIAL:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    // Report a change
                    bStateChanged = TRUE;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_ERROR:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INVALID:
                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_UNSUBSCRIBED:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_ERROR:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_INVALID:

                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_UNAVAILABLE:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_ERROR:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_INVALID:
                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_POI_UPDATES_ONLY:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_ERROR:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_INVALID:
                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_READY:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_READY:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_ERROR:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_INVALID:
                default:
                    break;
            }
        }
        break;

        case DATASERVICE_STATE_ERROR:
        {
            switch (psStateChange->eCurrentState)
            {
                case DATASERVICE_STATE_ERROR:
                {
                    // No change
                    bStateChanged = FALSE;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    // Report this change to the caller
                    bReportStateToCaller = TRUE;
                }
                break;

                // Transitions that don't matter
                case DATASERVICE_STATE_INITIAL:
                case DATASERVICE_STATE_INVALID:
                default:
                    break;
            }
        }
        break;

        default:
            break;
    }

    // Report the state transition to the manager if we
    // determined that we need to do so
    if (bReportStateToCaller == TRUE)
    {
        switch (psStateChange->eCurrentState)
        {
            case DATASERVICE_STATE_READY:
            case DATASERVICE_STATE_POI_UPDATES_ONLY:
            {
                if (psObj->bServiceResourcesReady == FALSE)
                {
                    bStateChanged =
                        psHandlers->bHandleServiceReady(pvHandlerArg);

                    psObj->bServiceResourcesReady = bStateChanged;
                }
            }
            break;

            case DATASERVICE_STATE_STOPPED:
            {
                bStateChanged =
                    psHandlers->bHandleServiceStopped(pvHandlerArg);
            }
            break;

            case DATASERVICE_STATE_ERROR:
            {
                bStateChanged =
                    psHandlers->bHandleServiceError(pvHandlerArg);
            }
            break;

            case DATASERVICE_STATE_UNSUBSCRIBED:
            case DATASERVICE_STATE_UNAVAILABLE:
            case DATASERVICE_STATE_INITIAL:
            case DATASERVICE_STATE_INVALID:
            default:
            {
                // Nothing to do -- no handlers defined for these
            }
            break;
        }
    }

    return bStateChanged;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bGenericStateHandler
*
*   Function provides standard procedure for dealing with a new state
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bGenericStateHandler (
    void *pvUnused
        )
{
    // Just say the state transition worked
    return TRUE;
}

#if SMS_LOGGING == 1

/*****************************************************************************
*
*   DATASERVICE_MGR_pacName
*
*   Only used for logging purposes
*
*****************************************************************************/
const char *DATASERVICE_MGR_pacName (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bOwner;
    const char *pacName = "Unknown";

    // Verify caller owns the object
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hManager);
    if (bOwner == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        // Extract name
        pacName = psObj->acServiceName;
    }

    return pacName;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_vLog
*
*****************************************************************************/
void DATASERVICE_IMPL_vLog (
    const char *pcFormat,
    ...
        )
{
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    BOOLEAN bOwner;

    psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMS_hUseDSMCtrl();

    // Verify caller owns the object
    bOwner = SMSO_bOwner((SMS_OBJECT)psCtrl);
    if (bOwner == TRUE)
    {
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, pcFormat);

        // Log the requested information
        SMSE_vLogUsingList(
            psCtrl->hEventHdlr,
            pcFormat, &tList );

        // restore stack (push)
        va_end(tList);
    }
    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vLog
*
*   This function is called by the owner data service in order to log
*   data to the owning data DECODER's log.
*
*****************************************************************************/
void DATASERVICE_MGR_vLog (
    const char *pcFormat,
    ...
        )
{
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    BOOLEAN bOwner;

    psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMS_hUseDSMCtrl();

    // Verify caller owns the object
    bOwner = SMSO_bOwner((SMS_OBJECT)psCtrl);
    if (bOwner == TRUE)
    {
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, pcFormat);

        // Log the requested information
        SMSE_vLogUsingList(
            psCtrl->hEventHdlr,
            pcFormat, &tList );

        // restore stack (push)
        va_end(tList);
    }
    return;
}
#elif __STDC_VERSION__ < 199901L

void DATASERVICE_IMPL_vLogNothing(
    const char *pcFormat,
    ...
        )
{
    return;
}

void DATASERVICE_MGR_vLogNothing(
    const char *pcFormat,
    ...
        )
{
    return;
}
#endif

/*****************************************************************************
*
*   DATASERVICE_MGR_hGetRadioSpecificData
*
*****************************************************************************/
RADIO_PRIVATE_DATA_OBJECT DATASERVICE_MGR_hGetRadioSpecificData (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bValid;
    RADIO_PRIVATE_DATA_OBJECT hData =
        RADIO_PRIVATE_DATA_INVALID_OBJECT;

    // Verify object
    bValid =
        SMSO_bValid((SMS_OBJECT)hManager);
    if (bValid == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        // Extract radio specific data object
        hData = psObj->hRadioSpecificData;
    }

    return hData;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_vSetRadioSpecificData
*
*****************************************************************************/
void DATASERVICE_MGR_vSetRadioSpecificData (
    DATASERVICE_MGR_OBJECT hManager,
    RADIO_PRIVATE_DATA_OBJECT hData
        )
{
    BOOLEAN bOwner;

    // Verify object ownership
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hManager);
    if (bOwner == TRUE)
    {
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        // Replace radio specific data pointer with one provided.
        psObj->hRadioSpecificData = hData;
    }

    return;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_bIterateRadioData
*
*   Utilized ONLY by the radio module(s) in order to iterate through
*   the private radio data for each service.
*
*****************************************************************************/
BOOLEAN DATASERVICE_MGR_bIterateRadioData (
    SMS_OBJECT hDataCtrl,
    DATASERVICE_RADIO_DATA_ITERATOR bIterator,
    void *pvIteratorArg
        )
{
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bOwner;
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl =
        (DATASERVICE_MGR_CTRL_STRUCT *)hDataCtrl;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_RADIO_ITERATOR_STRUCT sIterator;

    // Validate inputs
    if (bIterator == NULL)
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner(hDataCtrl);
    if (bOwner == FALSE)
    {
        return FALSE;
    }

    // Iterate the list of data services

    // Populate iterator structure
    sIterator.bIterator = bIterator;
    sIterator.pvIteratorArg = pvIteratorArg;

    // Search for the data service entry
    eReturnCode = OSAL.eLinkedListIterate(
        psCtrl->hMgrList,
        (OSAL_LL_ITERATOR_HANDLER)bDataServiceRadioDataIterator,
        (void *)&sIterator);
    if ((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_NO_OBJECTS))
    {
        // Services iterated
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   DATASERVICE_MGR_tGetDataId
*
*   Returns DataID of the given data service
*
*****************************************************************************/
DATASERVICE_ID DATASERVICE_MGR_tGetDataId (
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bOwner;

    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if (bOwner == FALSE)
    {
        return DATASERVICE_INVALID_ID;
    }

    return ((DATASERVICE_MGR_OBJECT_STRUCT *)hManager)->tDataID;
}

/*****************************************************************************
*
*   DATASERVICE_IMPL_bFreeDataPayload
*
*****************************************************************************/
BOOLEAN DATASERVICE_IMPL_bFreeDataPayload (
    OSAL_BUFFER_HDL hPayload
        )
{
    return RADIO_bFreeDataPayload(hPayload);
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vDestroyDSMCtrl
*
*****************************************************************************/
static void vDestroyDSMCtrl (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    BOOLEAN bDestroyObject
        )
{
    // Stop and free the contents of the subscription list
    vUninitCachedSubscriptionList(psCtrl);

    // Free the timed event list
    vUninitTimedEventList(psCtrl);

    // Stop and free the contents of the data service table
    vDestroyDataServiceTable(psCtrl);

    if (psCtrl->sDevice.hDeviceObject != DEVICE_INVALID_OBJECT)
    {
        DEVICE_vUninstall(psCtrl->sDevice.hDeviceObject);
        psCtrl->sDevice.hDeviceObject = DEVICE_INVALID_OBJECT;
    }

    // Unregister for any time updates
    if(OSAL_TIME_NOTIFICATION_INVALID_OBJECT !=
        psCtrl->hTimeNotificationHandle)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Unregister
        eReturnCode = OSAL.eTimeSetUnRegisterNotification(
            psCtrl->hTimeNotificationHandle);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psCtrl->hTimeNotificationHandle =
                OSAL_TIME_NOTIFICATION_INVALID_OBJECT;
        }
    }

    // We no longer consider the device group loaded
    psCtrl->bDeviceGroupLoaded = FALSE;

    if (bDestroyObject == TRUE)
    {
        SMSO_vDestroy((SMS_OBJECT)psCtrl);
    }
    return;
}

/*****************************************************************************
*
*   bInitializeTask
*
*****************************************************************************/
static BOOLEAN bInitializeTask (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    SMS_TASK_CONFIGURATION_STRUCT sConfig =
        gsDSMTaskConfiguration;

    // Install SMS Task now
    // Note, successful installation of this task will cause the
    // provided SMS object to be owned by the SMS task.
    psCtrl->hServicesTask = SMST_hInstall(
        (SMS_OBJECT)psCtrl, &sConfig,
        (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)vEventHandler,
        &psCtrl->hEventHdlr);
    if(psCtrl->hServicesTask == SMS_INVALID_TASK_HANDLE)
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": DSM task could not be installed.");
        return FALSE;
    }

    // Success!
    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "DSM task installed.\n");

    return TRUE;
}

/*****************************************************************************
*
*   vEventHandler
*
*****************************************************************************/
static void vEventHandler (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    SMS_EVENT_HDL hEvent
        )
{
    SMS_EVENT_TYPE_ENUM eEventType;
    SMS_EVENT_DATA_UNION const *puEventData;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)psCtrl, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
        return;
    }

    // Extract event information
    // Caution! Do not pass puEventData by reference to any other function
    // unless you are absolutely sure you do not end up re-allocating the
    // current event, modify the event data and continue using data from the
    // original event. If you need to do this, first copy what is needed
    // to a local variable and use that instead. This will prevent any
    // issues with potential re-allocation/overwriting of the current event.
    eEventType = SMSE_eGetEvent(hEvent, &puEventData);

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4, "Handling event %d", eEventType);

    switch (eEventType)
    {
        // The INITIALIZE event occurs one time only when the task is started
        // and completes before returning to the caller.
        case SMS_EVENT_INITIALIZE:
        {
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "SMS_EVENT_INITIALIZE\n");
        }
        break;

        // This is the final stop event
        // posted by the DSM itself after all services stop
        case SMS_EVENT_STOP:
        {
            vHandleFinalStopEvent(psCtrl);
        }
        break;

        case SMS_EVENT_DATASERVICE:
        {
            const DATASERVICE_MGR_OBJECT hManager = puEventData->sData.hManager;
            const DATASERVICE_FW_EVENT_ENUM eDataServiceEvent =
                puEventData->sData.eDataServiceEvent;

            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                                "Handling dataservice event %X", eDataServiceEvent);

            switch (eDataServiceEvent)
            {
                case DATASERVICE_FW_EVENT_MODULE_ASSOCIATE:
                {
                    vHandleModuleAssociateEvent(psCtrl,
                        puEventData->sData.uArg.sModule.hModule,
                        puEventData->sData.uArg.sModule.hSTI,
                        puEventData->sData.uArg.sModule.hDriverName);
                }
                break;

                case DATASERVICE_FW_EVENT_MODULE_UNASSOCIATE:
                {
                    vHandleModuleUnassociateEvent(psCtrl,
                        puEventData->sData.uArg.sModule.hModule);
                }
                break;

                case DATASERVICE_FW_EVENT_SERVICE_ERROR:
                {
                    BOOLEAN bValid;
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    const DATASERVICE_ERROR_CODE_ENUM eErrorCode =
                        (DATASERVICE_ERROR_CODE_ENUM)(size_t)
                            puEventData->sData.uArg.sStandard.pvArg;

                    bValid = SMSO_bValid((SMS_OBJECT)hManager);

                    if(bValid == TRUE)
                    {
                        SMSAPI_RETURN_CODE_ENUM eReturnCode;

                        // a particular service needs to transition to ERROR

                        // Stop the data flow for all DSIs this service manages
                        eReturnCode = RADIO_eDataServiceControlFlow(
                            psCtrl->hRadioDataCtrl, hManager, DSI_INVALID_ID, FALSE);
                        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                DATASERVICE_MGR_OBJECT_NAME": RADIO_eDataServiceControlFlow()"
                                    " failed: %u", eReturnCode);
                        }

                        // Update the error code & state
                        vProcessStateChange(psCtrl, psObj,
                            DATASERVICE_STATE_ERROR, eErrorCode);
                    }
                    else if (hManager == DATASERVICE_MGR_INVALID_OBJECT)
                    {
                        // All services need to transition to ERROR
                        vPlaceAllManagersIntoError(psCtrl, eErrorCode);
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_INITIAL:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

                    // Do we need to load the device group?
                    if ((TRUE == psObj->bRequiresDeviceGroup) &&
                        (FALSE == psCtrl->bDeviceGroupLoaded))
                    {
                        // Yes, load it now
                        vLoadDeviceGroup(psCtrl, psObj->pacSRHDriverName);
                    }

                    // Update the state to INITIAL
                    vProcessStateChange(
                        psCtrl, psObj,
                        DATASERVICE_STATE_INITIAL,
                        DATASERVICE_ERROR_CODE_NONE);

                    // try to start the service
                    bDataServiceStartIterator(psObj, psCtrl);
                }
                break;

                case DATASERVICE_FW_EVENT_STOP:
                {
                    BOOLEAN bValid;

                    bValid = SMSO_bValid((SMS_OBJECT)hManager);
                    if (bValid == TRUE)
                    {
                        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

                        // Handle this event
                        vHandleStopRequest(psObj, psCtrl);
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            DATASERVICE_MGR_OBJECT_NAME
                            ":Told to stop invalid data "
                            "service manager.");
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_MANAGE_DATA_STREAM:
                {
                    SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT sStream;

                    // Copy out event data
                    OSAL.bMemCpy(&sStream, &puEventData->sData.uArg.sStreamConfig,
                        sizeof(SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT));

                    vHandleManageDataStream(psCtrl, hManager, &sStream);
                }
                break;

                case DATASERVICE_FW_EVENT_DEVICE_RADIUS:
                {
                    const DISTANCE_OBJECT hRadius =
                        (DISTANCE_OBJECT)puEventData->sData.uArg.sStandard.pvArg;

                    vProcessDeviceRadiusUpdate(psCtrl, hManager, hRadius);
                }
                break;

                case DATASERVICE_FW_EVENT_DEVICE_POSITION:
                {
                    DATASERVICE_EVENT_TIMESTAMP_STRUCT *psLastUpdateTime =
                        &psCtrl->sDevice.sLastUpdateTime;
                    DATASERVICE_EVENT_TIMESTAMP_STRUCT const *psThisUpdateTime =
                        &puEventData->sData.uArg.sPosition.sTimestamp;

                    // Only process this data if it represents new information
                    if ( (psThisUpdateTime->un32Sec > psLastUpdateTime->un32Sec) ||
                         ( (psThisUpdateTime->un32Sec == psLastUpdateTime->un32Sec) &&
                           (psThisUpdateTime->un16Msec > psLastUpdateTime->un16Msec)))
                    {
                        OSAL_FIXED_OBJECT hLat, hLon;
                        BOOLEAN bPositionSet;

                        // Update to this timestamp now
                        *psLastUpdateTime = *psThisUpdateTime;

                        // Get the fixed objects from the flat memory space
                        hLat = (OSAL_FIXED_OBJECT)&puEventData->sData.uArg.sPosition.atLatData[0];
                        hLon = (OSAL_FIXED_OBJECT)&puEventData->sData.uArg.sPosition.atLonData[0];

                        // Provide that information to the device object
                        bPositionSet = DEVICE_bSetPosition(
                            psCtrl->sDevice.hDeviceObject, hLat, hLon);
                        if (bPositionSet == FALSE)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                DATASERVICE_MGR_OBJECT_NAME
                                ":Unable to set device position");
                        }
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_TIMEOUT:
                {
                    void *pvArg = puEventData->sData.uArg.sStandard.pvArg;

                    // A data service manager has received a timeout event

                    // Tell the data service manager to process the timeout now
                    vProcessTimeout(psCtrl, hManager, pvArg);
                }
                break;

                case DATASERVICE_FW_EVENT_DSRL:
                {
                    DSRL_ARG_STRUCT *psArg = (DSRL_ARG_STRUCT *)
                            puEventData->sData.uArg.sStandard.pvArg;

                    vHandleDSRLEvent(psCtrl, hManager, psArg);
                }
                break;

                case DATASERVICE_FW_EVENT_AUTONOMOUS_STOP:
                {
                    BOOLEAN bValid;

                    bValid = SMSO_bValid((SMS_OBJECT)hManager);
                    if (TRUE == bValid)
                    {
                        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                        DATASERVICE_DECODER_UNSUBSCRIBE_ITERATOR_STRUCT sIterator;

                        DATASERVICE_MGR_vLog(
                            DATASERVICE_MGR_OBJECT_NAME
                            ": Autonomous service %s beginning shutdown\n",
                        psObj->acServiceName);

                        // This service wants to be stopped
                        // when all decoders are unsubscribed from it
                        psObj->bStopWhenUnsubComplete = TRUE;

                        // This service no longer wants to receive
                        // subscription requests
                        psObj->tEventRequestMask &= ~DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED;

                        // Initialize our iterator to send unsubscribe
                        // requests for all decoders which are
                        // subscribed to this service
                        sIterator.hDecoder = DECODER_INVALID_OBJECT;
                        sIterator.psServiceToMatch = (DATASERVICE_MGR_OBJECT_STRUCT *)psObj;
                        sIterator.bUnsubscribeIssued = FALSE;

                        // Tell this service to unsubscribe all of its decoders
                        OSAL.eLinkedListIterate(
                            psCtrl->hSubscribedDecoders,
                            (OSAL_LL_ITERATOR_HANDLER)bIssueUnSubscribeRequestFromAllDecoders,
                            (void *)&sIterator);

                        // Did any events get issued?
                        if (FALSE == sIterator.bUnsubscribeIssued)
                        {
                            // No, stop now
                            DATA.vStop(hManager);
                        }
                    }
                }
                break;

                // A decoder has requested to be subscribed to data services
                case DATASERVICE_FW_EVENT_DECODER_SUBSCRIBED:
                {
                    const DECODER_OBJECT hDecoder =
                        (DECODER_OBJECT)puEventData->sData.uArg.sStandard.pvArg;

                    vHandleDecoderSubscribed(psCtrl, hDecoder);
                }
                break;

                // A decoder is now unsubscribed
                case DATASERVICE_FW_EVENT_DECODER_UNSUBSCRIBED:
                {
                    BOOLEAN bIsValid;
                    const DECODER_OBJECT hDecoder =
                        (DECODER_OBJECT)puEventData->sData.uArg.sStandard.pvArg;

                    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                        "DATASERVICE_FW_EVENT_DECODER_UNSUBSCRIBED: "
                        "hDecoder: %p\n", hDecoder);

                    // Were we provided a valid manager handle?
                    bIsValid = SMSO_bIsValid((SMS_OBJECT)hManager);

                    if (bIsValid == FALSE)
                    {
                        // This decoder doesn't want any more
                        // interaction with the DSM and has
                        // requested to be unsubscribed from
                        // all services
                        vHandleDecoderUnsubscribeReq(psCtrl, hDecoder);
                    }
                    else
                    {
                        // A decoder has provided us with a "directed
                        // unsubscribe".  This means that the unsubscribe
                        // handshake for a service has been completed.
                        vHandleDirectedDecoderUnsubscribe(psCtrl, hManager, hDecoder);
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_SERVICE_SPECIFIC:
                {
                    vHandleServiceSpecificEvent(
                        psCtrl, hManager,
                        (void *)&puEventData->sData.uArg.sImpl.apvData[0]);
                }
                break;

                case DATASERVICE_FW_EVENT_TIME_AVAILABLE:
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;

                    DATASERVICE_MGR_vLog(
                        DATASERVICE_MGR_OBJECT_NAME
                        ": Signal received that time is available.\n");

                    // Unregister for any further time updates
                    if(OSAL_TIME_NOTIFICATION_INVALID_OBJECT !=
                        psCtrl->hTimeNotificationHandle)
                    {
                        // Unregister
                        eReturnCode = OSAL.eTimeSetUnRegisterNotification(
                            psCtrl->hTimeNotificationHandle);
                        if(eReturnCode == OSAL_SUCCESS)
                        {
                            psCtrl->hTimeNotificationHandle =
                                OSAL_TIME_NOTIFICATION_INVALID_OBJECT;
                        }
                    }

                    // Rip through all current data services and assign them
                    // if we can.
                    if (psCtrl->hRadioDataCtrl != RADIO_PRIVATE_DATA_INVALID_OBJECT)
                    {
                        // Iterate the list if Data Services
                        eReturnCode = OSAL.eLinkedListIterate(
                            psCtrl->hMgrList,
                            (OSAL_LL_ITERATOR_HANDLER)bDataServiceStartIterator,
                            (void *)psCtrl);
                        if ((eReturnCode != OSAL_SUCCESS) &&
                            (eReturnCode != OSAL_NO_OBJECTS))
                        {
                            // Error!
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                DATASERVICE_MGR_OBJECT_NAME":Unable to assign "
                                "radio to data services");
                        }
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_PRODUCT_ENABLE:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    SMS_EVENT_DATASERVICE_PRODUCT_ARG_STRUCT sEventArg =
                        puEventData->sData.uArg.sProduct;
                    BOOLEAN bValid;

                    bValid = bValidateManager(psCtrl, hManager);
                    if (TRUE == bValid)
                    {
                        vHandleEnableProduct(psCtrl, psObj,
                            sEventArg.eProductType, sEventArg.tMask);
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_PRODUCT_DISABLE:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    DATA_PRODUCT_TYPE_ENUM eProductType = (DATA_PRODUCT_TYPE_ENUM)
                        puEventData->sData.uArg.sStandard.pvArg;
                    BOOLEAN bValid;

                    bValid = bValidateManager(psCtrl, hManager);
                    if (TRUE == bValid)
                    {
                        DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;

                        psProduct = psFindProduct(psObj, eProductType);
                        if (NULL != psProduct)
                        {
                            vHandleDisableProduct(psCtrl, psObj, psProduct);
                        }
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_START_TIMED_EVENT:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    SMS_EVENT_DATASERVICE_TIMED_EVENT_STRUCT sEventArg =
                        puEventData->sData.uArg.sTimed;
                    BOOLEAN bValid;

                    bValid = bValidateManager( psCtrl, hManager );
                    if ( TRUE == bValid )
                    {
                        vHandleStartTimedEvent ( psObj, sEventArg.bRepeat,
                                sEventArg.un32IntervalInSeconds );
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_STOP_TIMED_EVENT:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    BOOLEAN bValid;

                    bValid = bValidateManager( psCtrl, hManager );
                    if ( TRUE == bValid )
                    {
                        vHandleStopTimedEvent ( psObj );
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_SXI_MESSAGE:
                {
                    const SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION
                        *puSxiMessage = &puEventData->sData.uArg.uSxiMessage;

                    DATASERVICE_MGR_vLog(
                        DATASERVICE_MGR_OBJECT_NAME
                        ": SXI Messages distribution (CID=%d, SID=%d, PIDCnt=%d)",
                        (int)puSxiMessage->sContentBuffered.tChannleId,
                        (int)puSxiMessage->sContentBuffered.tServiceId,
                        (int)puSxiMessage->sContentBuffered.un8PIDCount);

                    OSAL.eLinkedListIterate(psCtrl->hMgrList,
                        (OSAL_LL_ITERATOR_HANDLER)bIssueSxiMessageToAllReadyServices,
                        (void*)puSxiMessage);

                    // Release memory if allocated for the message
                    if (puSxiMessage->sContentBuffered.atPIDList)
                    {
                        OSAL.vMemoryFree(puSxiMessage->sContentBuffered.atPIDList);
                    }
                }
                break;

                case DATASERVICE_FW_EVENT_PRODUCT_ERROR:
                {
                    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
                    DATA_PRODUCT_TYPE_ENUM eProductType = (DATA_PRODUCT_TYPE_ENUM)
                        puEventData->sData.uArg.sStandard.pvArg;
                    BOOLEAN bValid;

                    bValid = bValidateManager(psCtrl, hManager);
                    if (TRUE == bValid)
                    {
                        DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;

                        psProduct = psFindProduct(psObj, eProductType);
                        if ((NULL != psProduct) &&
                            (DATA_PRODUCT_STATE_DISABLED != psProduct->eState))
                        {
                            vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_ERROR);
                        }
                    }
                }
                break;

            }
        }
        break;

        default:
        {
            // Unhandled event, pass on to RADIO specific handler
            RADIO_bDataServiceEventHandler(
                psCtrl->hRadioDataCtrl, eEventType, puEventData);
        }
        break;
    }

    SMSO_vUnlock((SMS_OBJECT)psCtrl);
    return;
}

/*****************************************************************************
*
*   hCreateDSMChildCallback
*
*   Callback invoked by SMS when we attempt to create a DSM child object
*
*****************************************************************************/
static SMS_OBJECT hCreateDSMChildCallback (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_CREATE_CALLBACK_STRUCT *psCreate
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)NULL;
    do
    {
        BOOLEAN bOwner, bAdded;
        char acRootName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        char acTempName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        size_t tSize;
        DATASERVICE_CREATE_STRUCT *psCreateService;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Verify arguments
        bOwner = SMSO_bOwner((SMS_OBJECT)psCtrl);
        if (bOwner == FALSE)
        {
            break;
        }

        // Verify arguments
        if (psCreate == NULL)
        {
            // Error!
            break;
        }

        // Get a more convenient pointer
        psCreateService = psCreate->psCreateService;

        tSize = strlen(psCreateService->pacSRHDriverName)+1;

        // Construct a unique name for the data service manager object
        snprintf( &acRootName[0], sizeof(acRootName),
                 DATASERVICE_MGR_OBJECT_NAME":%s",
                 psCreateService->pacServiceObjectName);

        // Create an instance of this object
        psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acRootName[0],
                sizeof(DATASERVICE_MGR_OBJECT_STRUCT) + psCreateService->tServiceObjectSize,
                (SMS_OBJECT)psCtrl, FALSE
                    );

        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Store the SRH driver name now
        snprintf( &acTempName[0], sizeof(acTempName),"%s:SRH", &acRootName[0] );

        psObj->pacSRHDriverName =  (char *)
            SMSO_hCreate(
                &acTempName[0],
                tSize,
                (SMS_OBJECT)psObj, FALSE);
        if (psObj->pacSRHDriverName == (char *)NULL)
        {
            break;
        }

        snprintf((char *)psObj->pacSRHDriverName,
            tSize, "%s", psCreateService->pacSRHDriverName);

        // Store the caller's event parameters
        psObj->tEventRequestMask = psCreateService->tEventRequestMask;
        psObj->vEventHandler = psCreateService->vEventCallback;
        psObj->pvEventHandlerArg = psCreate->pvEventHandlerArg;

        // Set the initial state (unknown) and error code
        psObj->eErrorCode = DATASERVICE_ERROR_CODE_NONE;
        psObj->eState = DATASERVICE_STATE_INITIAL;

        // Keep track of the time requirement
        psObj->bRequiresTime = psCreateService->bTimeRequired;

        // Keep track of the device group requirement
        psObj->bRequiresDeviceGroup = psCreateService->bRequiresDeviceGroup;

        // Compose data service name
        if (psCreateService->tDataID < DSI_INVALID_ID)
        {
            // one-DSI service, add DSI to the service name
            snprintf(psObj->acServiceName, sizeof(psObj->acServiceName),
                "%s (dsi %u)",
                psCreateService->pacServiceObjectName,
                (DSI)psCreateService->tDataID);
        }
        else
        {
            // multi-DSI service, the given name is assumed to be unique
            snprintf(psObj->acServiceName, sizeof(psObj->acServiceName),
                "%s", psCreateService->pacServiceObjectName);
        }

        // Initlize the resource ready flag
        psObj->bServiceResourcesReady = FALSE;

        psObj->tDataID = psCreateService->tDataID;
        psObj->bMultiDSIRequired = psCreateService->bMultiDSIRequired;
        psObj->bStopWhenUnsubComplete = FALSE;


        // Initialize products
        if ((NULL != psCreateService->sProductsInfo.peProducts) &&
            (psCreateService->sProductsInfo.un8Count > 0) &&
            (NULL != psCreateService->sProductsInfo.bGetDSIForProduct) &&
            (NULL != psCreateService->sProductsInfo.eGetNextProductState))
        {
            UN8 un8Idx;
            BOOLEAN bOk = TRUE;

            psObj->vDSIStateHandler = vApplyDSIStateToProducts;

            psObj->sProducts.bGetDSIForProduct =
                psCreateService->sProductsInfo.bGetDSIForProduct;
            psObj->sProducts.eGetNextProductState =
                psCreateService->sProductsInfo.eGetNextProductState;

            snprintf( &acTempName[0], sizeof(acTempName),"%s:Products", &acRootName[0] );

            eReturnCode = OSAL.eLinkedListCreate(
                &psObj->sProducts.hProducts,
                acTempName,
                (OSAL_LL_COMPARE_HANDLER)n16CompareProducts,
                OSAL_LL_OPTION_UNIQUE |
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);

            if (OSAL_SUCCESS != eReturnCode)
            {
                break;
            }

            // Build products list
            for (un8Idx = 0; un8Idx < psCreateService->sProductsInfo.un8Count; un8Idx++)
            {
                DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;

                snprintf(&acTempName[0], sizeof(acTempName),"%s:Product %u",
                         &acRootName[0],
                         psCreateService->sProductsInfo.peProducts[un8Idx].eProductType);

                psProduct = (DATASERVICE_MGR_PRODUCT_INFO_STRUCT *)
                    OSAL.pvLinkedListMemoryAllocate(
                        acTempName, sizeof(*psProduct), FALSE);

                if (NULL == psProduct)
                {
                    bOk = FALSE;
                    break;
                }

                psProduct->eType = psCreateService->sProductsInfo.peProducts[un8Idx].eProductType;
                psProduct->tAllowedMask = psCreateService->sProductsInfo.peProducts[un8Idx].tAllowedMask;
                psProduct->sDSIInfo.tDSI = DSI_INVALID_ID;
                psProduct->eState = DATA_PRODUCT_STATE_DISABLED;
                psProduct->tMask = DATA_PRODUCT_MASK_NONE;

                eReturnCode = OSAL.eLinkedListAdd(
                    psObj->sProducts.hProducts,
                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                    psProduct);

                if (OSAL_SUCCESS != eReturnCode)
                {
                    bOk = FALSE;
                    break;
                }
            }

            if (FALSE == bOk)
            {
                break;
            }
        }
        else
        {
            psObj->vDSIStateHandler = vApplyDSIStateToService;
            psObj->sProducts.bGetDSIForProduct = NULL;
            psObj->sProducts.eGetNextProductState = NULL;
            psObj->sProducts.hProducts = OSAL_INVALID_OBJECT_HDL;
        }

        // Initialize DSI tracking list
        snprintf(&acTempName[0], sizeof(acTempName),"%s:DSIs",
                 &acRootName[0]);
        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hDSIs,
            acTempName,
            (OSAL_LL_COMPARE_HANDLER)n16CompareDSIs,
            OSAL_LL_OPTION_UNIQUE |
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS);

        if (OSAL_SUCCESS != eReturnCode)
        {
            break;
        }

        // Create DSI for one-DSI services
        if (psCreateService->tDataID < DSI_INVALID_ID)
        {
            DATASERVICE_MGR_DSI_REF_STRUCT *psDSI;

            psDSI = psCreateDSI(psObj,
                (DSI)psCreateService->tDataID,
                psCreateService->tSuggestedOTABufferByteSize,
                psCreateService->bEnableAll);

            if (NULL == psDSI)
            {
                break;
            }
        }

        psObj->sInfo.bDeviceSupported = FALSE;
        psObj->sInfo.bFavoritesSupported = FALSE;
        psObj->sInfo.tEventRequestMask = psObj->tEventRequestMask;

        psObj->hGenericTimedEvent = DATASERVICE_TIMED_EVENT_INVALID_HDL;

        // Attempt to add this manager to the list
        bAdded = bAddManagerToList(psCtrl, psObj);

        if (bAdded == FALSE)
        {
            break;
        }

        return (SMS_OBJECT)psObj;

    } while (FALSE);

    vUninitObject(psObj);

    return SMS_INVALID_OBJECT;
}

/*****************************************************************************
*
*   psCreate
*
*****************************************************************************/
static DATASERVICE_MGR_OBJECT_STRUCT *psCreate (
    DATASERVICE_CREATE_STRUCT *psCreateService,
    void *pvEventHandlerArg
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj;
    DATASERVICE_CREATE_CALLBACK_STRUCT sCreate;

    // Verify event handler & create structure
    if ((psCreateService == NULL) ||
        (psCreateService->pacSRHDriverName == NULL) ||
        (psCreateService->pacServiceObjectName == NULL))
    {
        // Error!
        return (DATASERVICE_MGR_OBJECT_STRUCT *)NULL;
    }

    // Populate the creation helper struct
    sCreate.psCreateService = psCreateService;
    sCreate.pvEventHandlerArg = pvEventHandlerArg;

    // Create the new manager object now
    psObj = (DATASERVICE_MGR_OBJECT_STRUCT *)
        SMS_hCreateDSMChildObject(
            (SMS_CREATE_DSM_CHILD_CALLBACK)hCreateDSMChildCallback,
            &sCreate);

    return psObj;
}

/*****************************************************************************
*
*   vRelease
*
*****************************************************************************/
static void vRelease (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bValid;

    // Do we have an object here?
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);

    if (bValid == TRUE)
    {
        // Unregister for device updates if
        // this manager is still registered
        vUnRegisterForDeviceUpdates(psObj);
    }

    // Iterate the list of timed events to clear any which
    // are associated with this service manager
    eReturnCode = OSAL.eLinkedListIterate(
        psCtrl->hTimedEvents,
        (OSAL_LL_ITERATOR_HANDLER)bRemoveAllTimedEventsForManager,
        (DATASERVICE_MGR_OBJECT)psObj);
    if ((eReturnCode != OSAL_SUCCESS) &&
        (eReturnCode != OSAL_NO_OBJECTS))
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to iterate timed event list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    // Destroy the object itself
    vUninitObject(psObj);

    return;
}

/*****************************************************************************
*
*   vUninitObject
*
*****************************************************************************/
static void vUninitObject (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bValid;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if (bValid == FALSE)
    {
        return;
    }

    // Clear attributes
    psObj->bServiceResourcesReady = FALSE;

    // Destroy SRH name
    if (psObj->pacSRHDriverName != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacSRHDriverName);
        psObj->pacSRHDriverName = NULL;
    }

    // Do we have an object here?
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
    if (bValid == TRUE)
    {
        BOOLEAN bLocked;
        SMS_OBJECT hParent;

        // Attempt to lock it
        bLocked = SMSO_bLock((SMS_OBJECT)psObj->psDSRL,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to lock DSRL parent");

            return;
        }

        // Get the parent object
        hParent = SMSO_hParent((SMS_OBJECT)psObj->psDSRL);

        // Clear the favorites function pointers
        psObj->psDSRL->sFavorites.hCreateTargetFromTag = NULL;
        psObj->psDSRL->sFavorites.hGetTagForTarget = NULL;

        // Free the DSRL memory
        SMSO_vDestroy((SMS_OBJECT)psObj->psDSRL);
        psObj->psDSRL = (DATASERVICE_DSRL_SUPPORT_STRUCT *)NULL;

        // Unlock the parent object now
        SMSO_vUnlock(hParent);
    }

    // Destroy products
    if (OSAL_INVALID_OBJECT_HDL != psObj->sProducts.hProducts)
    {
        // Destroy products data
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->sProducts.hProducts,
            (OSAL_LL_RELEASE_HANDLER)OSAL.vLinkedListMemoryFree);

        if (OSAL_SUCCESS == eReturnCode)
        {
            eReturnCode = OSAL.eLinkedListDelete(psObj->sProducts.hProducts);
            if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": Unable to destroy "
                    "products list in service %s, error %s",
                    psObj->acServiceName,
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }

            psObj->sProducts.hProducts = OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to destroy "
                "products data in service %s, error %s",
                psObj->acServiceName, OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    // Destroy DSI list
    if (OSAL_INVALID_OBJECT_HDL != psObj->hDSIs)
    {
        // Destroy DSI references
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->hDSIs,
            (OSAL_LL_RELEASE_HANDLER)OSAL.vLinkedListMemoryFree);

        if (OSAL_SUCCESS == eReturnCode)
        {
            eReturnCode = OSAL.eLinkedListDelete(psObj->hDSIs);
            if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": Unable to destroy "
                    "DSI list in service %s, error %s",
                    psObj->acServiceName,
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }

            psObj->hDSIs = OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to destroy "
                "DSI references in service %s, error %s",
                psObj->acServiceName, OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

    // Destroy the object itself
    SMSO_vDestroy((SMS_OBJECT)psObj);
    psObj = (DATASERVICE_MGR_OBJECT_STRUCT*)NULL;

    return;
}

/*****************************************************************************
*
*   bEnableDataFlow
*
*****************************************************************************/
static BOOLEAN bEnableDataFlow (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT const *psObj,
    DSI tDSI
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Start the data flow for the given DSI
    eReturnCode = RADIO_eDataServiceControlFlow(
            psCtrl->hRadioDataCtrl, 
            (DATASERVICE_MGR_OBJECT)psObj, tDSI, TRUE);
    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": RADIO_eDataServiceControlFlow()"
                " failed for DSI:%u (error %s)", tDSI,
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bServiceStart
*
*   This function is invoked when the radio utilized by this data service
*   reports the service is up & running.
*
*****************************************************************************/
static BOOLEAN bServiceStart (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_STATE_ENUM eReadyState
        )
{
    // Move to our ready state now
    vProcessStateChange(
        psCtrl, psObj,
        eReadyState, DATASERVICE_ERROR_CODE_NONE);

    // Does this service wish to receive subscription events?
    if (DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED ==
        (psObj->tEventRequestMask & DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED))
    {
        // Yes it does. Send all the decoders we know about to this manager
        OSAL.eLinkedListIterate(
            psCtrl->hSubscribedDecoders,
            (OSAL_LL_ITERATOR_HANDLER)bIterateDecodersForServiceStart,
            psObj);
    }

    // Let the caller know the service start
    // event was provided to the data service
    return TRUE;
}

/*****************************************************************************
*
*   vServiceHalted
*
*   This function is called in order to indicate that this data service's
*   support has been completely halted within SMS.
*
*****************************************************************************/
static void vServiceHalted (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // Call the specific manager's event handler
    // indicating the service has halted.
    vProcessStateChange(psCtrl, psObj,
        DATASERVICE_STATE_STOPPED, DATASERVICE_ERROR_CODE_NONE);

    // Free the memory in the radio for this service
    eReturnCode = RADIO_eDataServiceDestroy((DATASERVICE_MGR_OBJECT)psObj);
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": RADIO_eDataServiceDestroy()"
                " failed: %u", eReturnCode);
    }

    // Get rid of the data service manager now
    vRelease(psCtrl, psObj);

    return;
}

/*****************************************************************************
*
*   vLoadDeviceGroup
*
*****************************************************************************/
static void vLoadDeviceGroup (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    const char *pacSRHDriverName
        )
{
    FILE *psSRHDiscovery;

    // Open the SRH so we can interrogate it
    psSRHDiscovery = fopen( pacSRHDriverName, "discovery" );

    if ((FILE *)NULL == psSRHDiscovery)
    {
        // Could not determine our device group
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": unable to access SRH driver");
        return;
    }

    // This is a good time to have SMS
    // learn of its provisioned device group
    psCtrl->bDeviceGroupLoaded =
        SMS_bLoadDeviceGroup(psSRHDiscovery);

    if (FALSE == psCtrl->bDeviceGroupLoaded)
    {
        // Could not determine our device group
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": SRH driver not configured correctly --"
            " device group not found/loaded");
    }
    
    fclose(psSRHDiscovery);

    return;
}

/*****************************************************************************
*
*   vCallEventHandler
*
*****************************************************************************/
static void vCallEventHandler (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_EVENT_MASK tEvent,
    void *pvEventArg
        )
{
    if (psObj != NULL)
    {
        if ((psObj->tEventRequestMask & tEvent) == tEvent)
        {
            if(psObj->vEventHandler != NULL)
            {
                psObj->vEventHandler (
                    (DATASERVICE_MGR_OBJECT)psObj,
                    tEvent,
                    pvEventArg,
                    psObj->pvEventHandlerArg);
            }
        }
    }
    return;
}

/******************************************************************************
*
*   bInitCachedSubscriptionList
*
*   Create a linked list which performs caching of data service subscription
*   requests by DECODER objects.
*
******************************************************************************/
static BOOLEAN bInitCachedSubscriptionList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Verify the subscription list doesn't already exist...
    // we only want one instance of this list
    if (psCtrl->hSubscribedDecoders != OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Create subscription list
    eReturnCode =
        OSAL.eLinkedListCreate(
            &psCtrl->hSubscribedDecoders,
            DATASERVICE_MGR_OBJECT_NAME":DS Subs",
            (OSAL_LL_COMPARE_HANDLER)n16CompareDecoderEntries,
            OSAL_LL_OPTION_UNIQUE | OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/******************************************************************************
*
*   vUninitCachedSubscriptionList
*
******************************************************************************/
static void vUninitCachedSubscriptionList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Delete the subscribed decoders linked list
    if (psCtrl->hSubscribedDecoders == OSAL_INVALID_OBJECT_HDL)
    {
        return;
    }

    // This list MUST be empty by the time we call this function
    // So, only attempt a delete now
    eReturnCode = OSAL.eLinkedListDelete(psCtrl->hSubscribedDecoders );
    if (OSAL_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": Decoder list memory leak");
    }

    // Invalidate linked list
    psCtrl->hSubscribedDecoders = OSAL_INVALID_OBJECT_HDL;

    return;
}

/******************************************************************************
*
*   bInitTimedEventList
*
******************************************************************************/
static BOOLEAN bInitTimedEventList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // We only want one instance of this list
    if (psCtrl->hTimedEvents != OSAL_INVALID_OBJECT_HDL)
    {
        return FALSE;
    }

    // Create subscription list
    eReturnCode =
        OSAL.eLinkedListCreate(
            &psCtrl->hTimedEvents,
            DATASERVICE_MGR_OBJECT_NAME":TimedEvents",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS
                );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/******************************************************************************
*
*   vUninitTimedEventList
*
******************************************************************************/
static void vUninitTimedEventList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    // Delete the timed event linked list
    if (psCtrl->hTimedEvents != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListRemoveAll(
            psCtrl->hTimedEvents,
            (OSAL_LL_RELEASE_HANDLER)vDestroyTimedEventEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(
                psCtrl->hTimedEvents );

            // Invalidate linked list
            psCtrl->hTimedEvents =
                OSAL_INVALID_OBJECT_HDL;
        }
    }

    return;
}

/******************************************************************************
*
*   bCreateDataServiceTable
*
******************************************************************************/
static BOOLEAN bCreateDataServiceTable (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    // Only create if needed
    if (OSAL_INVALID_OBJECT_HDL == psCtrl->hMgrList)
    {
        eReturnCode =
            OSAL.eLinkedListCreate(
                &psCtrl->hMgrList,
                DATASERVICE_MGR_OBJECT_NAME":MgrList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareDataIds,
                OSAL_LL_OPTION_UNIQUE);
    }

    return (eReturnCode == OSAL_SUCCESS);
}

/******************************************************************************
*
*   vDestroyDataServiceTable
*
******************************************************************************/
static void vDestroyDataServiceTable (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eLinkedListDelete(psCtrl->hMgrList);
    if (eReturnCode == OSAL_SUCCESS)
    {
        psCtrl->hMgrList = OSAL_INVALID_OBJECT_HDL;
    }

    return;
}

/******************************************************************************
*
*   bGetDSMTag
*
******************************************************************************/
static BOOLEAN bGetDSMTag (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    TAG_OBJECT hSMSTag;

    // Get the top-level SMS tag
    hSMSTag = SMS_hGetTag();

    // Get the DSM tag
    eReturnCode = TAG_eGet(
        DSM_TAG_NAME, hSMSTag, &psCtrl->hTag, NULL, TRUE );
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // Error!
        return FALSE;
    }

    return TRUE;
}

/******************************************************************************
*
*   hAllocateEvent
*
******************************************************************************/
static SMS_EVENT_HDL hAllocateEvent (
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_FW_EVENT_ENUM eEvent,
    SMS_EVENT_DATASERVICE_ARG_UNION **ppuEventArg
        )
{
    SMS_EVENT_HDL hEvent = SMS_INVALID_EVENT_HDL;
    EVENT_OPTIONS_TYPE tOptions = SMS_EVENT_OPTION_NONE;
    SMS_EVENT_DATA_UNION *puEventData =
        (SMS_EVENT_DATA_UNION *)NULL;

    // If this event type is outside the range of
    // low-priority events, mark it as urgent
    if(eEvent > DATASERVICE_FW_UPDATE_LOW_PRIORITY_END)
    {
        tOptions |= SMS_EVENT_OPTION_URGENT;
    }

    // Allocate an event for the caller
    hEvent = SMS_hAllocateDSMEvent(
        SMS_EVENT_DATASERVICE,
        (void **)&puEventData, tOptions);

    if(hEvent != SMS_INVALID_EVENT_HDL)
    {
        // Copy the event type and manager handle
        puEventData->sData.eDataServiceEvent = eEvent;
        puEventData->sData.hManager = hManager;

        // Apply timestamp where needed
        if (eEvent == DATASERVICE_FW_EVENT_DEVICE_POSITION)
        {
            OSAL.vTimeUp(&puEventData->sData.uArg.sPosition.sTimestamp.un32Sec,
                            &puEventData->sData.uArg.sPosition.sTimestamp.un16Msec);
        }

        // Provide the caller with the argument union
        *ppuEventArg = &puEventData->sData.uArg;
    }

    return hEvent;
}

/*******************************************************************************
*
*   n16CompareDecoderEntries
*
*******************************************************************************/
static N16 n16CompareDecoderEntries (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psSub1,
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psSub2
        )
{
    if ((psSub1 == (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL) ||
        (psSub2 == (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL))
    {
        return N16_MIN;
    }

    if (psSub1->hDecoder != psSub2->hDecoder)
    {
        // Put all new entries at the end of the list
        return -1;
    }

    // Match
    return 0;
}

/*******************************************************************************
*
*   n16CompareEntryToHandle
*
*   Compares an existing data service manager entry to a given handle
*
* Outputs:
*   0   - Manager handles  match.
*   > 0 - Move on to the next manager.
*
*******************************************************************************/
static N16 n16CompareEntryToHandle (
    void *pvObj1,
    void *pvObj2
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)pvObj1;
    DATASERVICE_MGR_OBJECT hManager =
        (DATASERVICE_MGR_OBJECT)pvObj2;
    N16 n16Return = 1;

    if ((DATASERVICE_MGR_OBJECT)psObj == hManager)
    {
        n16Return = 0;
    }

    return n16Return;
}

/******************************************************************************
*
*   bValidateManager
*
******************************************************************************/
static BOOLEAN bValidateManager (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager
        )
{
    BOOLEAN bHandleValid, bManagerValidated = FALSE;

    bHandleValid = SMSO_bValid((SMS_OBJECT)hManager);
    if (bHandleValid == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;

        // Search for the manager in the list -- it won't
        // be here if it has stopped
        eReturnCode = OSAL.eLinkedListLinearSearch(
            psCtrl->hMgrList,
            &hEntry, n16CompareEntryToHandle,
            hManager);

        if (eReturnCode == OSAL_SUCCESS)
        {
            bManagerValidated = TRUE;
        }
    }

    return bManagerValidated;
}

/******************************************************************************
*
*   bDecoderSubscribedToService
*
******************************************************************************/
static BOOLEAN bDecoderSubscribedToService (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_SUBSCRIBER_DETECT_STRUCT sDetect;

    // Initialize structure
    sDetect.bHasSubscriptions = FALSE;
    sDetect.psObj = psObj;

    // Iterate the list to find out if there
    // are any decoders subscribed to this service
    eReturnCode = OSAL.eLinkedListIterate(
        psCtrl->hSubscribedDecoders,
        (OSAL_LL_ITERATOR_HANDLER)bIterateDecodersForServiceStop,
        (void *)&sDetect);

    if (OSAL_SUCCESS != eReturnCode)
    {
        // If we can't inspect this list then
        // we are forced to believe there are
        // subscribers for this service still
        sDetect.bHasSubscriptions = TRUE;
    }

    return sDetect.bHasSubscriptions;
}

/******************************************************************************
*
*   bIssueSubscribeRequestToAllReadyServices
*
******************************************************************************/
static BOOLEAN bIssueSubscribeRequestToAllReadyServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    if (psObj == NULL)
    {
        // Error! Get past this
        return TRUE;
    }

    // Does this service wish to receive
    // subscription events?
    if ((psObj->tEventRequestMask &
            DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED) !=
        DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED)
    {
        // The service does not need decoder subscription
        // Continue past this manager
        return TRUE;
    }

    // Only continue if this service is ready
    if (psObj->eState != DATASERVICE_STATE_READY)
    {
        // Continue past this manager
        return TRUE;
    }

    // Inform the manager of the subscription event
    vCallEventHandler(psObj,
        DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED,
        (void *)hDecoder);

    // Continue iterating
    return TRUE;
}

/******************************************************************************
*
*   bIssueUnsubscribeRequestToAllServices
*
******************************************************************************/
static BOOLEAN bIssueUnsubscribeRequestToAllServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        )
{
    if ((DATASERVICE_MGR_OBJECT_STRUCT *)NULL == psObj)
    {
        // Error! Get past this
        return TRUE;
    }

    // Inform the manager of the unsubscription event
    vCallEventHandler(psObj,
        DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED,
        (void *)hDecoder);

    // Keep going!
    return TRUE;
}

/******************************************************************************
*
*   bIssueUnSubscribeRequestFromAllDecoders
*
******************************************************************************/
static BOOLEAN bIssueUnSubscribeRequestFromAllDecoders (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_DECODER_UNSUBSCRIBE_ITERATOR_STRUCT *psUnsubscribe
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder) ||
        ((DATASERVICE_DECODER_UNSUBSCRIBE_ITERATOR_STRUCT *)NULL == psUnsubscribe))
    {
        // Error! Stop now
        return FALSE;
    }

    // Does this manager appear in this decoder's
    // subscription list?
    eReturnCode = OSAL.eLinkedListSearch(
        psDecoder->hSubscriptions,
        &hEntry,
        (void *)psUnsubscribe->psServiceToMatch);
    if (OSAL_SUCCESS != eReturnCode)
    {
        // Move past this entry
        return TRUE;
    }

    // Inform the manager of the unsubscription event
    vCallEventHandler(psUnsubscribe->psServiceToMatch,
        DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED,
        (void *)psDecoder->hDecoder
            );

    // We have issued an unsubscribe event
    psUnsubscribe->bUnsubscribeIssued = TRUE;

    return TRUE;
}

/******************************************************************************
*
*   bIssueSxiMessageToAllReadyServices
*
******************************************************************************/
static BOOLEAN bIssueSxiMessageToAllReadyServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    const SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION *puSxiMessage
        )
{
    if ((psObj != NULL) &&
        (psObj->tEventRequestMask & DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE) &&
        (psObj->eState == DATASERVICE_STATE_READY))
    {
        vCallEventHandler(psObj,
            DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE,
            (void *)puSxiMessage);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bIterateDecodersForServiceStart
*
*****************************************************************************/
static BOOLEAN bIterateDecodersForServiceStart (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    do
    {
        // Tell the manager of the subscribe event only if it is ready.
        if (DATASERVICE_STATE_READY != psObj->eState)
        {
            break;
        }

        // Check if this service is already subscribed to this decoder.
        eReturnCode = OSAL.eLinkedListSearch(
            psDecoder->hSubscriptions,
            &hEntry, (void *)psObj);
        if (OSAL_SUCCESS == eReturnCode)
        {
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 5,
                "%s service is already subscribed to decoder %p\n",
                psObj->acServiceName, psDecoder->hDecoder);
            break;
        }

        // The service is ready and not yet subscribed to this decoder.
        // So inform the manager of the subscription event.
        vCallEventHandler(psObj,
            DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED,
            (void *)psDecoder->hDecoder);

    } while (FALSE);

    // Keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bIterateDecodersForServiceStop
*
*****************************************************************************/
static BOOLEAN bIterateDecodersForServiceStop (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_SUBSCRIBER_DETECT_STRUCT *psDetect
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    if (((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder) ||
        ((DATASERVICE_SUBSCRIBER_DETECT_STRUCT *)NULL == psDetect))
    {
        // Error! Stop now
        return FALSE;
    }

    // Does this manager appear in this decoder's
    // subscription list?
    eReturnCode = OSAL.eLinkedListSearch(
        psDecoder->hSubscriptions,
        &hEntry,
        (void *)psDetect->psObj);
    if (OSAL_SUCCESS != eReturnCode)
    {
        // Move past this entry
        return TRUE;
    }

    // There are decoders which are subscribed to this service
    psDetect->bHasSubscriptions = TRUE;

    // Don't need to go any futher
    return FALSE;
}

/*****************************************************************************
*
*   bIterateSubscribedShim
*
*****************************************************************************/
static BOOLEAN bIterateSubscribedShim (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_ITERATE_SHIM_STRUCT *psShim
        )
{
    BOOLEAN bContinue = TRUE,
            bReportEntry = FALSE;

    if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": bIterateSubscribedShim()"
            " list corrupted");
        return FALSE;
    }

    // If a particular decoder object was specified then
    // make sure we only report that one decoder, otherwise
    // we're supposed to report to all decoders
    if ((psShim->hDecoder == DECODER_INVALID_OBJECT) ||
        (psShim->hDecoder == psDecoder->hDecoder))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Is this service in this decoder's subscribed services list?
        eReturnCode = OSAL.eLinkedListSearch(
            psDecoder->hSubscriptions,
            &hEntry,
            psShim->psService);

        if (eReturnCode == OSAL_SUCCESS)
        {
            bReportEntry = TRUE;
        }
    }

    if (bReportEntry == TRUE)
    {
        // Let the iterator work on this decoder
        bContinue = psShim->bIterator(psDecoder->hDecoder, psShim->pvIteratorArg);
    }

    return bContinue;
}

/******************************************************************************
*
*   psFindDecoderEntry
*
******************************************************************************/
static DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psFindDecoderEntry (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT sSearch,
        *psSubscribed = (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL;

    // We're looking for this decoder
    sSearch.hDecoder = hDecoder;

    // Search the list for this entry
    eReturnCode = OSAL.eLinkedListSearch(
        psCtrl->hSubscribedDecoders,
        &hEntry,
        (void *)&sSearch
            );

    if (OSAL_SUCCESS == eReturnCode)
    {
        // Extract the entry
        psSubscribed = (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }

    return psSubscribed;
}

/******************************************************************************
*
*   bAddDecoderToSubscribedList
*
******************************************************************************/
static BOOLEAN bAddDecoderToSubscribedList (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        )
{
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder =
        (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Create the new decoder entry
        psDecoder = (DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                DATASERVICE_MGR_OBJECT_NAME":Decoder",
                sizeof(DATASERVICE_SUBSCRIBED_DECODER_STRUCT), TRUE);

        if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": bAddDecoderToSubscribedList()"
                " out of memory");
            break;
        }

        // Store the decoder handle here
        psDecoder->hDecoder = hDecoder;

        // Initialize entry handle
        psDecoder->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // This decoder has not indicated it wants
        // to be released yet
        psDecoder->bReleaseWhenUnsubscribed = FALSE;

        // Initialize the LL handle
        psDecoder->hSubscriptions = OSAL_INVALID_OBJECT_HDL;

        // Create the list of services that this decoder
        // is subscribed to
        eReturnCode = OSAL.eLinkedListCreate(
            &psDecoder->hSubscriptions,
            DATASERVICE_MGR_OBJECT_NAME":DecoderSubs",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_UNIQUE);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": bAddDecoderToSubscribedList()"
                " unable to create subscription list for decoder");
            break;
        }

        // Add to the subscribed decoders list
        eReturnCode = OSAL.eLinkedListAdd(
            psCtrl->hSubscribedDecoders,
            &psDecoder->hEntry,
            (void *)psDecoder );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": bAddDecoderToSubscribedList()"
                " unable to add decoder to subscribed list");
            break;
        }

        // List has been updated
        return TRUE;

    } while (FALSE);

    // We failed to add this decoder to our list -- this
    // is a big problem.  The DSM can't support this decoder at all.
    if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL != psDecoder)
    {
        // Remove the decoder from the list and release it
        psDecoder->bReleaseWhenUnsubscribed = TRUE;

        bRemoveDecoderFromSubscribedList(psDecoder);
    }
    else
    {
        // We couldn't even allocate memory to support this decoder.
        // Just tell it that it's unsubscribed
        DECODER_bRelease(hDecoder, SMS_OBJECT_RELEASE_BY_OTHERS);
    }

    return FALSE;
}

/******************************************************************************
*
*   bRemoveDecoderFromSubscribedList
*
******************************************************************************/
static BOOLEAN bRemoveDecoderFromSubscribedList (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder
        )
{

    if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
    {
        return FALSE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bUnsubscribed;
        DECODER_OBJECT hDecoder = psDecoder->hDecoder;

        // An unsubscription just occurred.  Should we
        // be releasing the decoder now?
        if (TRUE != psDecoder->bReleaseWhenUnsubscribed)
        {
            // Nope.  Nothing to do.
            break;
        }

        if (OSAL_INVALID_OBJECT_HDL != psDecoder->hSubscriptions)
        {
            // Delete the services list in the entry
            eReturnCode = OSAL.eLinkedListDelete(psDecoder->hSubscriptions);
            if (OSAL_SUCCESS != eReturnCode)
            {
                if (OSAL_ERROR_LIST_NOT_EMPTY == eReturnCode)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": bRemoveDecoderFromSubscribedList()"
                        " Decoder being removed from subscription list prematurely");
                }
                else
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": bRemoveDecoderFromSubscribedList()"
                        " unable to update subscription list for decoder");
                }
                break;
            }
        }

        // Remove the entry now
        if (OSAL_INVALID_LINKED_LIST_ENTRY != psDecoder->hEntry)
        {
            eReturnCode = OSAL.eLinkedListRemove(psDecoder->hEntry);
            if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        DATASERVICE_MGR_OBJECT_NAME
                        ": bRemoveDecoderFromSubscribedList()"
                        " unable to remove decoder from subscription list");
                break;
            }
        }

        // Free the associated memory for the list entry
        OSAL.vLinkedListMemoryFree((void *)psDecoder);

        // Tell this decoder it is now unsubscribed from data services
        bUnsubscribed = DECODER_bRelease(hDecoder, SMS_OBJECT_RELEASE_BY_OTHERS);
        if (FALSE == bUnsubscribed)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": bRemoveDecoderFromSubscribedList()"
                    " unable to release DECODER.");
            break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bSetError
*
*****************************************************************************/
static BOOLEAN bSetError (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    vProcessStateChange(
        psCtrl, psObj,
        DATASERVICE_STATE_ERROR, eErrorCode);

    // keep on iteratin'
    return TRUE;
}

/*****************************************************************************
*
*   vProcessStateChange
*
*   This function is called in order to provide a data service manager a
*   context in which to process a state transition
*
*****************************************************************************/
static void vProcessStateChange (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_STATE_ENUM eState,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    DATASERVICE_STATE_CHANGE_STRUCT sStateChange;

    // Log this
    vLogStateChange(
        psCtrl,
        psObj->acServiceName,
        eState, eErrorCode);

    if ((DATASERVICE_STATE_ERROR == eState) &&
        (OSAL_INVALID_OBJECT_HDL != psObj->sProducts.hProducts))
    {
        // all products shall go to error state together with service
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListIterate(
            psObj->sProducts.hProducts,
            (OSAL_LL_ITERATOR_HANDLER)bProductForcedErrorIterator,
            psObj);

        if ((OSAL_SUCCESS != eReturnCode) &&
            (OSAL_NO_OBJECTS != eReturnCode))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Failed to "
                "set error for products in service %s",
                psObj->acServiceName);
        }
    }

    // Save off the previous state
    sStateChange.ePreviousState = psObj->eState;

    // Update the error code & state
    psObj->eErrorCode = eErrorCode;
    psObj->eState = eState;

    // Save the current state
    sStateChange.eCurrentState = psObj->eState;

    vCallEventHandler(psObj, DATASERVICE_EVENT_STATE,
        (void*)&sStateChange);

    return;
}

/*****************************************************************************
*
*   vHandleStopRequest
*
*****************************************************************************/
static void vHandleStopRequest (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    do
    {
        SMSAPI_RETURN_CODE_ENUM eSmsReturnCode;
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        BOOLEAN bResult;

        // Do we have products to stop now?
        if (OSAL_INVALID_OBJECT_HDL != psObj->sProducts.hProducts)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Detach all products from their DSIs and disable them.
            // DSIs will be released a bit later.
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->sProducts.hProducts,
                (OSAL_LL_ITERATOR_HANDLER)bProductForcedDisableIterator,
                psObj);

            if ((OSAL_SUCCESS != eReturnCode) &&
                (OSAL_NO_OBJECTS != eReturnCode))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": Failed to "
                    "disable products in service %s",
                    psObj->acServiceName);

                // Stop here
                break;
            }
        }

        // Iterate the list of timed events to clear any which
        // are associated with this service manager
        eOsalReturnCode = OSAL.eLinkedListIterate(
            psCtrl->hTimedEvents,
            (OSAL_LL_ITERATOR_HANDLER)bStopAllTimedEventsForManager,
            (DATASERVICE_MGR_OBJECT)psObj);
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to iterate timed event list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));

            // Stop here
            break;
        }

        // Do we need to tell the radio to stop?
        if (psObj->bOTADataAvailable == TRUE)
        {
            // Stop the radio
            eSmsReturnCode = RADIO_eDataServiceStop(
                psCtrl->hRadioDataCtrl,
                (DATASERVICE_MGR_OBJECT)psObj);
            if (eSmsReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": Radio request to stop data "
                    "service failed: %u", eSmsReturnCode);
                break;
            }
        }

        // Stop this service!
        bResult = bStopService(psCtrl, psObj);
        if (TRUE != bResult)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to stop service");
            break;
        }

        // Try to finalize DSM stopping
        bTryStopDSM(psCtrl);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   vProcessDeviceRadiusUpdate
*
*****************************************************************************/
static void vProcessDeviceRadiusUpdate (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DISTANCE_OBJECT hRadius
        )
{
    LOCATION_OBJECT hOldLocation = LOCATION_INVALID_OBJECT,
                    hNewLocation = LOCATION_INVALID_OBJECT;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

    do
    {
        BOOLEAN bSuccess, bValid;
        DSRL_OBJECT hDSRL;

        // Is this manager valid?
        bValid = bValidateManager(psCtrl, hManager);

        if (bValid != TRUE)
        {
            // Not a problem
            break;
        }

        // Extract the device dsrl handle
        hDSRL = psObj->psDSRL->sManagedDSRLs.hDeviceDSRL;

        // Grab the old target we used as the registration argument
        hOldLocation = (LOCATION_OBJECT)
            DEVICE.pvRegistrationArgument(psObj->psDSRL->hDeviceReg);
        if (hOldLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Make a copy so we can modify safely
        hNewLocation = LOCATION.hDuplicate(hOldLocation);
        if (hNewLocation == LOCATION_INVALID_OBJECT)
        {
            break;
        }

        // Update the radius in the new location object
        bSuccess = LOCATION_bUpdateRadius(
            hNewLocation, hRadius, FALSE);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Tell the DSRL about the new device location
        bSuccess = DSRL_bSetDeviceLocation(hDSRL, hNewLocation);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Update the registration argument now
        bSuccess = DEVICE.bReplaceRegistrationArgument(
            psObj->psDSRL->hDeviceReg, TRUE, (void *)hNewLocation);

        if (bSuccess == FALSE)
        {
            break;
        }

        // We don't need this anymore
        LOCATION.vDestroy(hOldLocation);

        // Clear this handle so we don't free it
        hNewLocation = LOCATION_INVALID_OBJECT;
    } while (FALSE);

    LOCATION.vDestroy(hNewLocation);

    return;
}

/*****************************************************************************
*
*   vProcessDeviceLocationUpdate
*
*****************************************************************************/
static void vProcessDevicePositionUpdate (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DEVICE_EVENT_MASK tEventMask,
    LOCATION_OBJECT hDevLocation
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
    BOOLEAN bValid;

    // Is this manager valid?
    bValid = bValidateManager(psCtrl, hManager);

    if (bValid != TRUE)
    {
        // Not a problem
        return;
    }

    // Only provide this message to the service
    // if it is ready
    if (psObj->eState == DATASERVICE_STATE_READY)
    {
        BOOLEAN bLocationUpdated;

        // Update this service's device location in a safe manner
        bLocationUpdated =
            bSetServiceDeviceLocation(psCtrl, psObj, hDevLocation);
        if (bLocationUpdated == TRUE)
        {
            DSRL_ARG_STRUCT *psArg;

            // Create a new DSRL arg for this event -
            // it always has just one target (the device location)
            psArg = DSRL_psCreateArg( 1 );

            if (psArg == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                        ": vProcessDevicePositionUpdate()"
                        " Creation a new DSRL arg failure");
                return;
            }

            psArg->eDSRLType = DSRL_TYPE_DEVICE;
            psArg->hDSRL = psObj->psDSRL->sManagedDSRLs.hDeviceDSRL;
            psArg->tNumTargets = 1;

            // Initial position configured - time to send the DSRL
            // in to the service for processing (kinda sorta a delayed create)
            if ((tEventMask & DEVICE_EVENT_INITIAL_POSITION_CONFIGURED) ==
                    DEVICE_EVENT_INITIAL_POSITION_CONFIGURED)
            {
                psArg->eAction = DSRL_ACTION_ADD;
                psArg->ahTargetList[0] = hDevLocation;
            }

            // Position/radius updated -- tell the manager the target
            // location has been replaced
            if (((tEventMask & DEVICE_EVENT_POSITION_UPDATE) == DEVICE_EVENT_POSITION_UPDATE) ||
                ((tEventMask & DEVICE_EVENT_REGISTRATION_UPDATE) == DEVICE_EVENT_REGISTRATION_UPDATE))
            {
                psArg->eAction = DSRL_ACTION_MODIFY;
                psArg->uAction.sModify.eModifyType = DSRL_MODIFY_OPERATION_REPLACE;
                psArg->uAction.sModify.bForceDSRLStateChange = FALSE;
                psArg->ahTargetList[0] = LOCATION.hDuplicate(hDevLocation);
            }

            DATASERVICE_MGR_vLog(
                DATASERVICE_MGR_OBJECT_NAME
                ": Device position update event for %s\n",
                psObj->acServiceName);

            // Call the DSRL update handler
            vHandleDSRLEvent(psCtrl, hManager, psArg);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                    ": Unable to update device location for %s",
                    psObj->acServiceName);
        }
    }

    return;
}

/*****************************************************************************
*
*   vProcessTimeout
*
*   This function is called in order to provide a data service manager a
*   context in which to process a timeout event.  However, we must first
*   validate that this is still a manager which can receive updates.
*
*****************************************************************************/
static void vProcessTimeout (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvArg
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
    BOOLEAN bValid;

    // Is this manager valid?
    bValid = bValidateManager(psCtrl, hManager);

    if (bValid != TRUE)
    {
        // Not a problem
        return;
    }

    // Only provide this message to the service
    // if it is ready
    if (psObj->eState == DATASERVICE_STATE_READY)
    {
        DATASERVICE_MGR_vLog(
            DATASERVICE_MGR_OBJECT_NAME
            ": Timeout occurred for %s\n",
            psObj->acServiceName);

        // Inform the manager of data reception
        vCallEventHandler( psObj, DATASERVICE_EVENT_TIMEOUT, pvArg );
    }

    return;
}

/*****************************************************************************
*
*   vHandleDSRLEvent
*
*****************************************************************************/
static void vHandleDSRLEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_ARG_STRUCT *psArg
        )
{
    do
    {
        BOOLEAN bDSRLAvailable = TRUE, bValid;
        DATASERVICE_MGR_OBJECT_STRUCT *psObj =
            (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

        // Is this manager valid?
        bValid = bValidateManager(psCtrl, hManager);

        if (bValid != TRUE)
        {
            // Not a problem
            break;
        }

        // Do we have an object here?
        bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
        if (bValid != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": vHandleDSRLEvent()"
                    " %s doesn't support DSRL feature",
                    psObj->acServiceName);
            break;
        }

        // We have a special trick for favorites configuration updates.
        // When the application wishes to add a favorite, they don't
        // supply the favorites DSRL handle.  So, whenever we encounter
        // a DSRL update with a handle equal to DSRL_INVALID_OBJECT,
        // it means we need to attempt to use the favorites DSRL
        if ((psArg->eDSRLType == DSRL_TYPE_FAVORITES) &&
            (psArg->hDSRL == DSRL_INVALID_OBJECT))
        {
            // Use the favorites handle.  If it's invalid,
            // well that is just fine
            psArg->hDSRL = psObj->psDSRL->sManagedDSRLs.hFavorites;
        }

        // Don't send the DSRL event if the DSRL is in the ERROR state.
        // But let the Remove event through since we want the
        // ability to destroy DSRLs regardless of their state
        if (psArg->eAction != DSRL_ACTION_REMOVE)
        {
            DSRL_STATE_ENUM eDSRLState;

            // Get the state of the DSRL
            eDSRLState = DSRL.eState(psArg->hDSRL);
            if (eDSRLState == DSRL_STATE_ERROR)
            {
                // Don't allow changes to DSRLs that are in error
                bDSRLAvailable = FALSE;
            }

            // Validate the state of the service
            if (psObj->eState != DATASERVICE_STATE_READY)
            {
                // Don't allow changes when the service isn't ready
                bDSRLAvailable = FALSE;
            }
        }

        // Only provide this message to the service if it is ready
        if (bDSRLAvailable == TRUE)
        {
            BOOLEAN bHandleEvent;

            // Perform any necessary pre-processing of the DSRL event now
            bHandleEvent = bPreProcessDSRLEvent(psCtrl, psObj, psArg);
            if (bHandleEvent == FALSE)
            {
                break;
            }

            if (psArg->eDSRLType == DSRL_TYPE_FAVORITES)
            {
                // Perform any necessary pre-processing of this event
                // and find out if the service's event handler needs to
                // be invoked
                bHandleEvent = bPreProcessFavoriteEvent(psCtrl, psObj, &psArg);
            }
            else if (psArg->eDSRLType == DSRL_TYPE_DEVICE)
            {
                // Perform any necessary pre-processing of this event
                // and find out if the service's event handler needs to
                // be invoked
                bHandleEvent = bPreProcessDeviceEvent(psCtrl, psObj, psArg);
            }

            DATASERVICE_MGR_vLog(
                DATASERVICE_MGR_OBJECT_NAME
                ": %s event %s %s\n",
                (psArg->eDSRLType == DSRL_TYPE_FAVORITES) ?
                                "Favorite" :
                (psArg->eDSRLType == DSRL_TYPE_DEVICE) ?
                                "Device" :
                                "DSRL",
                (bHandleEvent == TRUE) ? "sent to" : "ignored by",
                psObj->acServiceName);

            if (bHandleEvent == TRUE)
            {
                // Inform the owner data service that a
                // DSRL needs some work done on it
                vCallEventHandler(psObj,
                    DATASERVICE_INTERNAL_EVENT_DSRL, (void *)psArg);
            }

            if (psArg->eDSRLType == DSRL_TYPE_FAVORITES)
            {
                // Perform any necessary post-processing of this event
                vPostProcessFavoriteEvent( psObj, psArg );
            }
            else if (psArg->eDSRLType == DSRL_TYPE_DEVICE)
            {
                // Perform any necessary post-processing of this event
                vPostProcessDeviceEvent( psObj, psArg );
            }
        }

    } while (FALSE);

    DSRL_vDestroyArg(psArg);

    return;
}

/*****************************************************************************
*
*   vHandleServiceSpecificEvent
*
*****************************************************************************/
static void vHandleServiceSpecificEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvArg
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
                    (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;
    BOOLEAN bValid;

    // Is this manager valid?
    bValid = bValidateManager(psCtrl, hManager);

    if (bValid == TRUE)
    {
        // Inform the owner data service that a
        // service specific event is ready to be processed
        vCallEventHandler(psObj,
            DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC,
            pvArg);

        // And inform application, if it is subscribed for
        // SERVICE event
        vCallEventHandler(psObj,
            DATASERVICE_EVENT_SERVICE,
            pvArg);
    }

    return;
}

/*****************************************************************************
*
*   bPreProcessDSRLEvent
*
*****************************************************************************/
static BOOLEAN bPreProcessDSRLEvent (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        )
{
    BOOLEAN bLocked, bSuccess = FALSE;

    // We're only interested in a subset of actions
    // at this time
    if ((DSRL_ACTION_ADD != psArg->eAction) &&
        (DSRL_ACTION_MODIFY != psArg->eAction))
    {
        return TRUE;
    }

    // Lock the DSRL now
    bLocked = SMSO_bLock(
        (SMS_OBJECT)psArg->hDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (TRUE != bLocked)
    {
        return FALSE;
    }

    switch (psArg->eAction)
    {
        case DSRL_ACTION_ADD:
        {
            // Set the type for this DSRL
            DSRL_vSetType(psArg->hDSRL, psObj->psDSRL->eServiceType);

            // Set this DSRL to Initial
            DSRL_vSetState(psArg->hDSRL, DSRL_STATE_INITIAL);

            // All is well
            bSuccess = TRUE;
        }
        break;

        case DSRL_ACTION_MODIFY:
        {
            // Inform the DSRL that its targets have been updated
            DSRL_vTargetsUpdated(psArg->hDSRL);

            // All is well
            bSuccess = TRUE;
        }
        break;

        case DSRL_ACTION_REMOVE:
        case DSRL_ACTION_REFRESH:
        case DSRL_ACTION_INVALID:
        default:
        {
        }
        break;
    }

    SMSO_vUnlock((SMS_OBJECT)psArg->hDSRL);

    return bSuccess;
}

/*****************************************************************************
*
*   bPreProcessFavoriteEvent
*
*   Returns a boolean indicating if the service's event handler
*   needs to be called
*
*****************************************************************************/
static BOOLEAN bPreProcessFavoriteEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT **ppsArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bProcessEvent = TRUE;
    DSRL_ARG_STRUCT *psArg = *ppsArg;

    // Do we have the top-level data tag in the config file?
    if (psCtrl->hTag == TAG_INVALID_OBJECT)
    {
        BOOLEAN bTagsInitialized;

        DATASERVICE_MGR_vLog(
            DATASERVICE_MGR_OBJECT_NAME": Acquiring top-level Data Tag\n");

        // Get the DSM tag
        bTagsInitialized = bGetDSMTag(psCtrl);
        if (bTagsInitialized == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": vPreProcessFavoriteEvent()"
                    " Unable to initialize favorites tag");
            return TRUE;
        }
    }

    // Verify presence of tag for this service
    if (psObj->psDSRL->sFavorites.hFavoritesTag == TAG_INVALID_OBJECT)
    {
        DATASERVICE_MGR_vLog(
            DATASERVICE_MGR_OBJECT_NAME
            ": Acquiring tag for %s\n",
            psObj->acServiceName);

        // Get this service's tag from within the favorites tag
        eReturnCode = TAG_eGet(
            FAVORITES_TAG_NAME,
            psCtrl->hTag,
            &psObj->psDSRL->sFavorites.hFavoritesTag,
            psObj->acServiceName, TRUE);

        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Stop here -- error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": vPreProcessFavoriteEvent()"
                    " Unable to service specific favorites tag");
            return TRUE;
        }
    }

    // Handle the creation of a favorites list
    if (psArg->eAction == DSRL_ACTION_ADD)
    {
        DSRL_ARG_STRUCT *psFavoriteArgs;
        DATASERVICE_MGR_TAG_ITERATOR_STRUCT sIterator;
        size_t tChildCount = 0;

        do
        {
            // Determine how large the DSRL_ARG_STRUCT's target list needs to be
            eReturnCode = TAG_eIterateChildren(
                psObj->psDSRL->sFavorites.hFavoritesTag,
                (TAG_ITERATION_HANDLER)bChildCountIterator,
                &tChildCount);
            if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) || (tChildCount == 0))
            {
                // Stop here -- nothing to do
                break;
            }

            // We have a list of favorites which need to get
            // added to the event args

            // Create a new args structure for the favorite targets
            psFavoriteArgs = DSRL_psCreateArg( tChildCount );
            if (psFavoriteArgs == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": vPreProcessFavoriteEvent()"
                        " allocation failure");

                // Stop here -- error
                break;
            }

            // Set arguments
            psFavoriteArgs->eAction = DSRL_ACTION_ADD;
            psFavoriteArgs->eDSRLType = DSRL_TYPE_FAVORITES;
            psFavoriteArgs->hDSRL = psArg->hDSRL;
            psFavoriteArgs->tNumTargets = 0;

            // Destroy the old argument
            DSRL_vDestroyArg(*ppsArg);

            // Replace with new argument
            *ppsArg = psFavoriteArgs;
            psArg = *ppsArg;

            // Configure the iterator structure
            sIterator.tMaxEntries = tChildCount;
            sIterator.psArg = psArg;
            sIterator.hCreateTargetFromTag = psObj->psDSRL->sFavorites.hCreateTargetFromTag;

            // Now, populate the target list with the entries we find in the config file
            TAG_eIterateChildren(
                psObj->psDSRL->sFavorites.hFavoritesTag,
                (TAG_ITERATION_HANDLER)bTargetCreationIterator,
                &sIterator);

        } while (FALSE);
    }
    // Handle the modification of a favorite entry
    else if (psArg->eAction == DSRL_ACTION_MODIFY)
    {
        if (psArg->hDSRL == DSRL_INVALID_OBJECT)
        {
            // Don't have the service handle an event
            // for a non-existent DSRL
            bProcessEvent = FALSE;
        }

        if (psArg->uAction.sModify.eModifyType != DSRL_MODIFY_OPERATION_REMOVEALL)
        {
            if (psObj->psDSRL->sFavorites.hGetTagForTarget != NULL)
            {
                TAG_OBJECT hEntryTag;
                size_t tIndex;

                for (tIndex = 0; tIndex < psArg->tNumTargets; tIndex++)
                {
                    // Call the tag retrieval callback for this service
                    hEntryTag = psObj->psDSRL->sFavorites.hGetTagForTarget(
                        psObj->psDSRL->sFavorites.hFavoritesTag,
                        psArg->ahTargetList[tIndex]);

                    // If we were told to remove this tag, and the tag is valid
                    if ((hEntryTag != TAG_INVALID_OBJECT) &&
                        (psArg->uAction.sModify.eModifyType == DSRL_MODIFY_OPERATION_REMOVE))
                    {
                        // Remove it (and commit now)
                        TAG_eRemove(hEntryTag, TRUE);
                    }
                }
            }
        }
        else // DSRL_MODIFY_OPERATION_REMOVEALL
        {
            // Remove the service's favorite tag...NOW
            TAG_eRemove(psObj->psDSRL->sFavorites.hFavoritesTag, TRUE);
            psObj->psDSRL->sFavorites.hFavoritesTag = TAG_INVALID_OBJECT;
        }
    }

    return bProcessEvent;
}

/*****************************************************************************
*
*   vPostProcessFavoriteEvent
*
*****************************************************************************/
static void vPostProcessFavoriteEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        )
{
    // Clear the favorites DSRL if we just destroyed it
    if (psArg->eAction == DSRL_ACTION_REMOVE)
    {
        psObj->psDSRL->sManagedDSRLs.hFavorites =  DSRL_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bPreProcessDeviceEvent
*
*****************************************************************************/
static BOOLEAN bPreProcessDeviceEvent (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        )
{
    BOOLEAN bProcessEvent = TRUE;

    if (psArg->eAction == DSRL_ACTION_ADD)
    {
        BOOLEAN bLocationUpdated;
        LOCATION_OBJECT hTargetLocation =
            (LOCATION_OBJECT)psArg->ahTargetList[0];

        // Attempt to update this location using the
        // current device location
        bLocationUpdated = DEVICE_bUpdateLocWithCurrentDevLoc(
            psCtrl->sDevice.hDeviceObject,
            hTargetLocation);

        // Register this service for device updates now
        // using this location
        bProcessEvent = bRegisterForDeviceUpdates(
            psCtrl, psObj, hTargetLocation);

        if (bProcessEvent == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": unable to register %s for device updates",
                psObj->acServiceName);
        }
        else
        {
            LOCATION_OBJECT hServiceLocation;

            // Replace the LOCATION object with something the service
            // can take ownership of
            hServiceLocation = LOCATION.hDuplicate(hTargetLocation);
            if (hServiceLocation == LOCATION_INVALID_OBJECT)
            {
                // Can't let the manager do this
                bProcessEvent = FALSE;
            }
            else
            {
                // Provide the DSRL with the device location
                // instance for this service (not the duplicated
                // location target entry)
                bProcessEvent = DSRL_bSetDeviceLocation(
                    psArg->hDSRL, hTargetLocation);

                // We're managing the original location with
                // the device registration...put the copy
                // in the target list now
                psArg->ahTargetList[0] = hServiceLocation;
            }
        }

        // Were we able to update our location using
        // the device position?
        if (bLocationUpdated == FALSE)
        {
            // No, we'll have to try again later, so
            // don't let the manager process this event
            bProcessEvent = FALSE;
        }

        if (bProcessEvent == TRUE)
        {
            // The manager will be given this DSRL, so
            // we're hands off now
            psObj->psDSRL->bDevDSRLStoppedByManager = TRUE;
        }
    }
    else if (psArg->eAction == DSRL_ACTION_REMOVE)
    {
        bProcessEvent = bPreProcessRemoveDeviceDSRLEvent(psObj);
    }

    return bProcessEvent;
}

/*****************************************************************************
*
*   bPreProcessRemoveDeviceDSRLEvent
*
*****************************************************************************/
static BOOLEAN bPreProcessRemoveDeviceDSRLEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bProcessEvent = TRUE;

    // Is this DSRL going to be stopped by the manager?
    if (psObj->psDSRL->bDevDSRLStoppedByManager == FALSE)
    {
        // No, that's our job now!
        DSRL_vDestroy(psObj->psDSRL->sManagedDSRLs.hDeviceDSRL);

        // FALSE indicates to the caller
        bProcessEvent = FALSE;
    }

    return bProcessEvent;
}

/*****************************************************************************
*
*   vPostProcessDeviceEvent
*
*****************************************************************************/
static void vPostProcessDeviceEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        )
{
    // Clear the device DSRL if we just destroyed it
    if (psArg->eAction == DSRL_ACTION_REMOVE)
    {
        vPostProcessRemoveDeviceDSRLEvent(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vPostProcessRemoveDeviceDSRLEvent
*
*****************************************************************************/
static void vPostProcessRemoveDeviceDSRLEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    // Unregister for device notification and
    // free associated memory
    vUnRegisterForDeviceUpdates(psObj);

    // Clear device handles for the info struct
    psObj->psDSRL->sManagedDSRLs.hDeviceDSRL = DSRL_INVALID_OBJECT;

    return;
}

/*****************************************************************************
*
*   bChildCountIterator
*
*****************************************************************************/
static BOOLEAN bChildCountIterator(
    TAG_OBJECT hTag,
    size_t *ptChildCount
        )
{
    (*ptChildCount)++;

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   bTargetCreationIterator
*
*****************************************************************************/
static BOOLEAN bTargetCreationIterator(
    TAG_OBJECT hTag,
    DATASERVICE_MGR_TAG_ITERATOR_STRUCT *psIterator
        )
{
    DSRL_TARGET_OBJECT hTarget;

    if (psIterator == NULL)
    {
        // Stop now
        return FALSE;
    }

    if (psIterator->psArg->tNumTargets >= psIterator->tMaxEntries)
    {
        // We can't go any further since we would exceed the
        // array bounds. This shouldn't happen
        return FALSE;
    }

    // Get the target created
    hTarget = psIterator->hCreateTargetFromTag(hTag);
    if (hTarget != DSRL_TARGET_INVALID_OBJECT)
    {
        // Add it to the list
        psIterator->psArg->ahTargetList[
            psIterator->psArg->tNumTargets++] = hTarget;
    }

    // Keep iterating
    return TRUE;
}

/******************************************************************************
*
*   vHandleDecoderSubscribed
*
******************************************************************************/
static void vHandleDecoderSubscribed (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        )
{
    BOOLEAN bAdded;

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "DATASERVICE_FW_EVENT_DECODER_SUBSCRIBED: hDecoder: %p\n", hDecoder);

    // Add this to the list of subscribed decoders
    bAdded = bAddDecoderToSubscribedList(psCtrl, hDecoder);
    if (TRUE == bAdded)
    {
        DATASERVICE_MGR_vLog(
            DATASERVICE_MGR_OBJECT_NAME
            ": DECODER %p subscribed to data services\n",
            hDecoder);

        // Send sub event to all ready services
        OSAL.eLinkedListIterate(
            psCtrl->hMgrList,
            (OSAL_LL_ITERATOR_HANDLER)
                bIssueSubscribeRequestToAllReadyServices,
            hDecoder);
    }

    return;
}

/******************************************************************************
*
*   vHandleDecoderUnsubscribeReq
*
******************************************************************************/
static void vHandleDecoderUnsubscribeReq (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        )
{
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bRemoved;

        // Find this decoder entry now
        psDecoder = psFindDecoderEntry(psCtrl, hDecoder);

        if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": vHandleDecoderUnsubscribeReq()"
                " unable to find decoder entry");
            break;
        }

        // This decoder wants to be released when
        // all of its subscriptions have been cleared
        psDecoder->bReleaseWhenUnsubscribed = TRUE;

        // Issue this request to all of the decoder's
        // subscribed data services
        eReturnCode = OSAL.eLinkedListIterate(
            psDecoder->hSubscriptions,
            (OSAL_LL_ITERATOR_HANDLER)bIssueUnsubscribeRequestToAllServices,
            (void *)hDecoder);

        // We can only service the unsubscribe request now
        // if there are no subscriptions for this
        // decoder at this time.  If the list is empty,
        // we can just go ahead and unsubscribe this decoder
        // completely.  If not, we'll have to wait for the
        // services to become unsubscribed
        if (OSAL_NO_OBJECTS != eReturnCode)
        {
            if (OSAL_SUCCESS != eReturnCode)
            {
                // print error
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": vHandleDecoderUnsubscribeReq()"
                    " unable to iterate decoder subscription list");
            }

            // Always break
            break;
        }

        // Remove this decoder from the list and the DSM at large
        bRemoved = bRemoveDecoderFromSubscribedList(psDecoder);
        if (FALSE == bRemoved)
        {
            break;
        }

        DATASERVICE_MGR_vLog(
            DATASERVICE_MGR_OBJECT_NAME
            ": DECODER %p unsubscribed from data services\n",
            hDecoder);

    } while (FALSE);

    return;
}

/******************************************************************************
*
*   vHandleDirectedDecoderUnsubscribe
*
******************************************************************************/
static void vHandleDirectedDecoderUnsubscribe (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DECODER_OBJECT hDecoder
        )
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder;
        BOOLEAN bRemoved;
        UN32 un32ServicesRemaining = 0;

        // Locate this decoder now
        psDecoder = psFindDecoderEntry(psCtrl, hDecoder);

        if ((DATASERVICE_SUBSCRIBED_DECODER_STRUCT *)NULL == psDecoder)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": vHandleDirectedDecoderUnsubscribe()"
                    " unable to locate decoder");
            break;
        }

        // Remove this entry from the service list

        // Search for it first
        eReturnCode = OSAL.eLinkedListSearch(
            psDecoder->hSubscriptions,
            &hEntry,
            (void *)psObj);

        // Did we find it?
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": vHandleDirectedDecoderUnsubscribe()"
                " unable to search decoder subscriptions (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));

            break;
        }

        // Yes!  Remove it now
        eReturnCode = OSAL.eLinkedListRemove(hEntry);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": vHandleDirectedDecoderUnsubscribe()"
                " unable to modify decoder subscriptions");
            break;
        }

        // Does this service wish to stop when all
        // subscriptions have been removed?
        if (TRUE == psObj->bStopWhenUnsubComplete)
        {
            BOOLEAN bServiceHasSubscribers;

            // Find out if any other decoders are subscribed to this service
            bServiceHasSubscribers = bDecoderSubscribedToService(psCtrl, psObj);

            // We can shutdown only if there are no more subscribers
            if (FALSE == bServiceHasSubscribers)
            {
                // Issue the stop now
                DATA.vStop((DATASERVICE_MGR_OBJECT)psObj);
            }
        }

        // How many services do we still have in
        // the subscribed list?
        eReturnCode = OSAL.eLinkedListItems(
            psDecoder->hSubscriptions, &un32ServicesRemaining);
        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": vHandleDirectedDecoderUnsubscribe()"
                " unable to query list");
            break;
        }

        if (un32ServicesRemaining > 0)
        {
            // Okay, stop here because we have more services
            // that must be unsubscribed before we even
            // start thinking of removing this decoder
            // from the DSM list
            break;
        }

        // Remove this decoder from the list and the DSM at large
        bRemoved = bRemoveDecoderFromSubscribedList(psDecoder);
        if (FALSE == bRemoved)
        {
            break;
        }

    } while (FALSE);

    return;
}

/******************************************************************************
*
*   vHandleManageDataStream
*
******************************************************************************/
static void vHandleManageDataStream (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT *psArg
        )
{
    BOOLEAN bValid;

    // Verify Manager is valid
    bValid = SMSO_bValid((SMS_OBJECT)hManager);
    if(bValid == TRUE)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = RADIO_eDataServiceManageStream(
            psCtrl->hRadioDataCtrl, hManager,
            (DATASERVICE_DMI_CONFIG_STRUCT const *)&psArg->pasDMIs[0],
            psArg->tNumDMIs);
        if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": RADIO_eDataServiceManageStream()"
                    " failed: %u", eReturnCode);
        }
    }

    // Do we need to free the DMI list memory?
    if (TRUE == psArg->bFreeDMIs)
    {
        // Yeah, do it now
        SMSO_vDestroy((SMS_OBJECT)psArg->pasDMIs);
    }

    return;
}

/*****************************************************************************
*
*   vPlaceAllManagersIntoError
*
*****************************************************************************/
static void vPlaceAllManagersIntoError (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
    // All services need to transition to ERROR
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_MGR_ERROR_ITERATOR_STRUCT sIterator;

    // Is our module reporting an error?
    if (eErrorCode == DATASERVICE_ERROR_CODE_DEVICE_FAILURE)
    {
        // Yeah, remember that
        psCtrl->sObjectInfo.bModuleInError = TRUE;
    }

    // Populate iterator structure
    sIterator.bIterator = bSetError;
    sIterator.eErrorCode = eErrorCode;
    sIterator.psCtrl = psCtrl;

    // Iterate the list of services to send them into error
    eReturnCode = OSAL.eLinkedListIterate(
        psCtrl->hMgrList,
        (OSAL_LL_ITERATOR_HANDLER)bDataServiceErrorIterator,
        (void *)&sIterator);
    if ((eReturnCode != OSAL_SUCCESS) &&
        (eReturnCode != OSAL_NO_OBJECTS))
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME":Unable to set dataservice error");
    }
    return;
}

/*****************************************************************************
*
*   bAddManagerToList
*
*****************************************************************************/
static BOOLEAN bAddManagerToList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Add to the data service table
    eReturnCode = OSAL.eLinkedListAdd(
        psCtrl->hMgrList, &psObj->hMasterEntry, psObj);

    if (OSAL_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": Cannot add data service %s, error %s\n",
            psObj->acServiceName, OSAL.pacGetReturnCodeName(eReturnCode));
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   vRemoveManagerFromList
*
*****************************************************************************/
static void vRemoveManagerFromList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    if (OSAL_INVALID_LINKED_LIST_ENTRY != psObj->hMasterEntry)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // remove service from the list
        eReturnCode = OSAL.eLinkedListRemove(psObj->hMasterEntry);
        psObj->hMasterEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Failed to remove "
                "service %s from services list",
                psObj->acServiceName);
        }
    }

    return;
}

/*****************************************************************************
*
*   bDataServiceRadioDataIterator
*
*****************************************************************************/
static BOOLEAN bDataServiceRadioDataIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_RADIO_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bIterate = TRUE;

    if(psIterator->bIterator != NULL)
    {
        RADIO_PRIVATE_DATA_OBJECT hRadioData;

        // Get the radio data for this service
        hRadioData = DATASERVICE_MGR_hGetRadioSpecificData(
            (DATASERVICE_MGR_OBJECT)psObj);

        // Call iterator
        bIterate = psIterator->bIterator(
            hRadioData, psIterator->pvIteratorArg);
    }

    // Go through the whole list
    return bIterate;
}

/*****************************************************************************
*
*   bDataServiceStartIterator
*
*****************************************************************************/
static BOOLEAN bDataServiceStartIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    BOOLEAN bWaiting;

    // Check whether the service is waiting for other objects.
    bWaiting = bIsWaitingForOthers(psCtrl, psObj);

    if (FALSE == bWaiting)
    {
        BOOLEAN bSuccess;

        // The service is not waiting for other objects.
        // It's time to start the service's associated DSIs now
        bSuccess = bStartAllDSIs(psCtrl, psObj);

        if (FALSE == bSuccess)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": Ready not accepted in service %s.",
                    psObj->acServiceName);
        }
    }

    // Go through the whole list
    return TRUE;
}

/*****************************************************************************
*
*   bDataServiceErrorIterator
*
*****************************************************************************/
static BOOLEAN bDataServiceErrorIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_ERROR_ITERATOR_STRUCT *psIterator
        )
{
    BOOLEAN bIterate;

    // Call iterator
    bIterate = psIterator->bIterator(
        psIterator->psCtrl, psObj,
        psIterator->eErrorCode);

    // Go through the whole list
    return bIterate;
}

/*****************************************************************************
*
*   hRegisterTimedEvent
*
*   Data service implementations must use this function to create timed-based
*   events.  The data service framework implements the correct usage of the
*   OSAL timer, SMS event handle management, and timer callback
*   implementation.  Events which occur due to these timed events are
*   provided to the data service implementation via the event
*   DATASERVICE_EVENT_TIMEOUT.  All timed events are stopped
*   for a service when that service is stopped.
*
*****************************************************************************/
static DATASERVICE_TIMED_EVENT_HDL hRegisterTimedEvent (
    DATASERVICE_MGR_OBJECT hManager,
    void *pvTimerEventArg
        )
{
    BOOLEAN bOwner;
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed =
        (DATASERVICE_TIMED_EVENT_STRUCT *)NULL;
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj =
        (DATASERVICE_MGR_OBJECT_STRUCT *)hManager;

    bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
    if ( bOwner == FALSE )
    {
        return DATASERVICE_TIMED_EVENT_INVALID_HDL;
    }

    // Get the Ctrl object
    psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMSO_hParent((SMS_OBJECT)psObj);
    if ( psCtrl == (DATASERVICE_MGR_CTRL_STRUCT *)NULL )
    {
        return DATASERVICE_TIMED_EVENT_INVALID_HDL;
    }

    // Allocate memory for this registration
    psTimed = (DATASERVICE_TIMED_EVENT_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
            DATASERVICE_MGR_OBJECT_NAME":TimedEvent",
            sizeof(DATASERVICE_TIMED_EVENT_STRUCT),
            TRUE);

    if ( psTimed == (DATASERVICE_TIMED_EVENT_STRUCT *)NULL )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": bRegisterTimedEvent() unable to create timer entry");

        return DATASERVICE_TIMED_EVENT_INVALID_HDL;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        static UN32 un32Instance = 0;

        // Create the name for this timer
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            DATASERVICE_MGR_OBJECT_NAME":Timer: %u (service %u)",
            un32Instance++, psObj->tDataID);

        // Attempt to allocate an event from the DSM task
        psTimed->hTimedEvent = hAllocateTimerEvent(
            psCtrl, (DATASERVICE_MGR_OBJECT)psObj, pvTimerEventArg);
        if ( psTimed->hTimedEvent == SMS_INVALID_EVENT_HDL )
        {
            break;
        }

        // Create the timer for this event now
        eReturnCode = OSAL.eTimerCreate(
            &psTimed->hTimer,
            &acName[0],
            vGenericTimerHandler,
            (void *)psTimed->hTimedEvent
               );

        if( eReturnCode != OSAL_SUCCESS )
        {
            // Error!
            break;
        }

        // Add this entry to the list now
        psTimed->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        eReturnCode = OSAL.eLinkedListAdd(
            psCtrl->hTimedEvents,
            &psTimed->hEntry, (void *)psTimed);
        if( eReturnCode != OSAL_SUCCESS )
        {
            // Error!
            break;
        }

        // Assign the manager handle now
        psTimed->hManager = (DATASERVICE_MGR_OBJECT)psObj;

        // Provide this as a handle to the caller
        return (DATASERVICE_TIMED_EVENT_HDL)psTimed;

    } while ( FALSE );

    // Destroy this entry
    vDestroyTimedEventEntry(psTimed);

    return DATASERVICE_TIMED_EVENT_INVALID_HDL;
}

/*****************************************************************************
*
*   hAllocateTimerEvent
*
*****************************************************************************/
static SMS_EVENT_HDL hAllocateTimerEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvTimerEventArg
        )
{

    SMS_EVENT_HDL hTimerEvent = SMS_INVALID_EVENT_HDL;
    BOOLEAN bOwner;

    // Verify context
    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if(bOwner == TRUE)
    {
        SMS_EVENT_DATA_UNION *puEventData;

        // Allocate a data service event
        hTimerEvent = SMSE_hAllocateEvent(
            psCtrl->hEventHdlr,
            SMS_EVENT_DATASERVICE,
            &puEventData, SMS_EVENT_OPTION_STATIC);
        if (hTimerEvent != SMS_INVALID_EVENT_HDL)
        {
            // For this manager
            puEventData->sData.hManager = hManager;

            // Timeout event
            puEventData->sData.eDataServiceEvent = DATASERVICE_FW_EVENT_TIMEOUT;

            // Use provided argument
            puEventData->sData.uArg.sStandard.pvArg = pvTimerEventArg;
        }
    }

    return hTimerEvent;
}

/*****************************************************************************
*
*   bStopAllTimedEventsForManager
*
*****************************************************************************/
static BOOLEAN bStopAllTimedEventsForManager (
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed,
    DATASERVICE_MGR_OBJECT hManager
        )
{
    if (psTimed != NULL)
    {
        if (psTimed->hManager == hManager)
        {
            // Stop this event now
            DATASERVICE_IMPL_bStopTimedEvent(
                (DATASERVICE_TIMED_EVENT_HDL)psTimed);
        }
    }

    // Iterate entire list
    return TRUE;
}

/*****************************************************************************
*
*   bRemoveAllTimedEventsForManager
*
*****************************************************************************/
static BOOLEAN bRemoveAllTimedEventsForManager (
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed,
    DATASERVICE_MGR_OBJECT hManager
        )
{
    if (psTimed != NULL)
    {
        if (psTimed->hManager == hManager)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Remove this entry from the list
            eReturnCode = OSAL.eLinkedListRemove(psTimed->hEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                // Remove & Destroy this entry
                vDestroyTimedEventEntry(psTimed);
            }
        }
    }

    // Iterate entire list
    return TRUE;
}

/*****************************************************************************
*
*   vDestroyTimedEventEntry
*
*****************************************************************************/
static void vDestroyTimedEventEntry(
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed
        )
{
    BOOLEAN bFreeAll = TRUE;

    if (psTimed == NULL)
    {
        return;
    }

    // Clear entry handle
    psTimed->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // First, deal with the OSAL timer
    if (psTimed->hTimer != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Stop this timer now
        eReturnCode = OSAL.eTimerStop(psTimed->hTimer);

        // Both of these return codes tell us the timer
        // is stopped.
        if ((eReturnCode == OSAL_SUCCESS) ||
            (eReturnCode == OSAL_TIMER_NOT_ACTIVE))
        {
            // Delete the timer
            OSAL.eTimerDelete(psTimed->hTimer);
            psTimed->hTimer = OSAL_INVALID_OBJECT_HDL;

            // Clear the manager handle
            psTimed->hManager = DATASERVICE_MGR_INVALID_OBJECT;
        }
        else
        {
            // We were unable to stop the timer,
            // so don't free psTimed object and
            // don't free the event handle
            bFreeAll = FALSE;
        }
    }

    if (bFreeAll == TRUE)
    {
        // Now free the SMS event handle associated with this event
        if (psTimed->hTimedEvent != SMS_INVALID_EVENT_HDL)
        {
            SMSE_vReleaseStaticEvent(psTimed->hTimedEvent);
            psTimed->hTimedEvent = SMS_INVALID_EVENT_HDL;
        }

        // Finally, free the associated memory
        OSAL.vLinkedListMemoryFree((void *)psTimed);
    }
    return;
}

/*****************************************************************************
*
*   bStopService
*
*****************************************************************************/
static BOOLEAN bStopService (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bValid;

    // Check to see if we may have some additional work
    // to do in shutting down DSRLs
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
    if (bValid == TRUE)
    {
        if (psObj->psDSRL->sManagedDSRLs.hDeviceDSRL != DSRL_INVALID_OBJECT)
        {
            BOOLEAN bDSRLStoppedByManager;

            // Perform the steps that occur when we're shutting down
            // the device DSRL, and learn if this DSRL will actually
            // be stopped by the manager or by the framework here
            bDSRLStoppedByManager = bPreProcessRemoveDeviceDSRLEvent(psObj);

            if (bDSRLStoppedByManager == FALSE)
            {
                // This DSRL will not be stopped by the manager, so
                // we need to finish up our stop processing now
                vPostProcessRemoveDeviceDSRLEvent(psObj);
            }
            else
            {
                // This DSRL will be stopped by the manager, however,
                // the DEVICE notification function shall be unregistered from
                // DSM any way
                vUnRegisterForDeviceUpdates(psObj);
            }
        }
    }

    // Remove this manager from the
    // global list of managers
    vRemoveManagerFromList(psCtrl, psObj);

    vServiceHalted(psCtrl, psObj);

    // keep on iteratin'
    return TRUE;
}

/*****************************************************************************
*
*   bRegisterForDeviceUpdates
*
*****************************************************************************/
static BOOLEAN bRegisterForDeviceUpdates (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hDeviceLocationInstance
        )
{
    BOOLEAN bRegistered = FALSE, bValid;

    // Verify this is a DSRL-based service
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
    if (bValid == TRUE)
    {
        if (psObj->psDSRL->hDeviceReg == DEVICE_REGISTRATION_INVALID_OBJECT)
        {
            // Start device update notifications for this service
            psObj->psDSRL->hDeviceReg =
                DEVICE_hRegisterForUpdates(
                    psCtrl->sDevice.hDeviceObject,
                    psObj->psDSRL->un32DeviceNotifyDistance,
                    psObj->psDSRL->eDeviceNotifyUnits,
                    (void *)hDeviceLocationInstance,
                    vDeviceUpdateCallback,
                    (DATASERVICE_MGR_OBJECT)psObj );
        }

        if (psObj->psDSRL->hDeviceReg != DEVICE_REGISTRATION_INVALID_OBJECT)
        {
            bRegistered = TRUE;
        }
    }

    return bRegistered;
}

/*****************************************************************************
*
*   vUnRegisterForDeviceUpdates
*
*****************************************************************************/
static void vUnRegisterForDeviceUpdates (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bValid;

    // Do we have an object here?
    bValid = SMSO_bIsValid((SMS_OBJECT)psObj->psDSRL);
    if (bValid == TRUE)
    {
        if (psObj->psDSRL->hDeviceReg != DEVICE_REGISTRATION_INVALID_OBJECT)
        {
            LOCATION_OBJECT hDevLocation;

            // Free the location we use as a registration argument
            hDevLocation = (LOCATION_OBJECT)
                DEVICE.pvRegistrationArgument(psObj->psDSRL->hDeviceReg);
            if (hDevLocation != LOCATION_INVALID_OBJECT)
            {
                LOCATION.vDestroy(hDevLocation);
            }

            DEVICE.vUnregisterForUpdates(psObj->psDSRL->hDeviceReg);
            psObj->psDSRL->hDeviceReg = DEVICE_REGISTRATION_INVALID_OBJECT;
        }
    }

    return;
}

/*****************************************************************************
*
*   bSetServiceDeviceLocation
*
*****************************************************************************/
static BOOLEAN bSetServiceDeviceLocation (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hServiceDevLocation
        )
{
    BOOLEAN bDSRLsLocked,
            bUpdated = FALSE;

    // Lock this service's DSRLs
    bDSRLsLocked =
        SMSO_bLock((SMS_OBJECT)psObj->psDSRL, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bDSRLsLocked == TRUE)
    {
        // Ask the device object to keep us updated
        bUpdated = DEVICE_bUpdateLocWithCurrentDevLoc(
            psCtrl->sDevice.hDeviceObject, hServiceDevLocation);

        SMSO_vUnlock((SMS_OBJECT)psObj->psDSRL);
    }

    return bUpdated;
}

/*****************************************************************************
*
*   vDeviceUpdateCallback
*
*****************************************************************************/
static void vDeviceUpdateCallback (
    void *pvRegistrationArg,
    DEVICE_EVENT_MASK tEventMask,
    DISTANCE_OBJECT hDistance,
    void *pvCallbackArg
        )
{
    DATASERVICE_MGR_OBJECT hManager =
        (DATASERVICE_MGR_OBJECT)pvCallbackArg;
    LOCATION_OBJECT hDevLocation =
        (LOCATION_OBJECT)pvRegistrationArg;
    BOOLEAN bOwner;

    if( (tEventMask & ~DEVICE_EVENT_ALL) != DEVICE_EVENT_NONE )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME
            ": unknown values in device event mask: %u\n",
            tEventMask);
    }

    // Verify we're in the right context
    bOwner = SMSO_bOwner((SMS_OBJECT)hManager);
    if (bOwner == TRUE)
    {
        DATASERVICE_MGR_CTRL_STRUCT *psCtrl;

        // Get the DSM control from this manager
        psCtrl = (DATASERVICE_MGR_CTRL_STRUCT *)SMS_hUseDSMCtrl();

        if (psCtrl != (DATASERVICE_MGR_CTRL_STRUCT *)NULL)
        {
            // Tell the manager the device location was updated
            vProcessDevicePositionUpdate(psCtrl, hManager, tEventMask, hDevLocation);
        }
    }

    return;
}

/*****************************************************************************
*
*   vGenericTimerHandler
*
*   This is the timer handler used for all data services wishing to
*   receive timeout-based events.  The argument is always an SMS_EVENT_HDL.
*   The event used here must be a pre-allocated event.
*
*****************************************************************************/
static void vGenericTimerHandler (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        )
{
    SMS_EVENT_HDL hEvent = (SMS_EVENT_HDL)pvArg;

    // Post the event
    SMSE_bPostEvent(hEvent);

    return;
}

/*****************************************************************************
*
*   vLogStateChange
*
*****************************************************************************/
static void vLogStateChange (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    const char *pacServiceName,
    DATASERVICE_STATE_ENUM eState,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        )
{
#if (OSAL_LOG == 1) && (SMS_LOGGING == 1)
    const char *pacState;
    const char *pacErrorCode = "";

    switch (eState)
    {
        case DATASERVICE_STATE_INITIAL:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_INITIAL);
        }
        break;

        case DATASERVICE_STATE_UNSUBSCRIBED:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_UNSUBSCRIBED);
        }
        break;

        case DATASERVICE_STATE_UNAVAILABLE:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_UNAVAILABLE);
        }
        break;

        case DATASERVICE_STATE_POI_UPDATES_ONLY:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_POI_UPDATES_ONLY);
        }
        break;

        case DATASERVICE_STATE_READY:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_READY);
        }
        break;

        case DATASERVICE_STATE_STOPPED:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_STOPPED);
        }
        break;

        case DATASERVICE_STATE_ERROR:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_ERROR);
        }
        break;

        case DATASERVICE_STATE_INVALID:
        {
            pacState =
                MACRO_TO_STRING(DATASERVICE_STATE_INVALID);
        }
        break;

        default:
        {
            pacState = "Unknown";
        }
    }

    if (eState == DATASERVICE_STATE_ERROR)
    {
        switch (eErrorCode)
        {
            case DATASERVICE_ERROR_CODE_NONE:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_NONE);
            }
            break;

            case DATASERVICE_ERROR_CODE_GENERAL:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_GENERAL);
            }
            break;

            case DATASERVICE_ERROR_CODE_DEVICE_FAILURE:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DEVICE_FAILURE);
            }
            break;

            case DATASERVICE_ERROR_CODE_INVALID_DEVICE:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_INVALID_DEVICE);
            }
            break;

            case DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND);
            }
            break;

            case DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH);
            }
            break;

            case DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE);
            }
            break;

            case DATASERVICE_ERROR_CODE_DATABASE_CORRUPT:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_CORRUPT);
            }
            break;
            case DATASERVICE_ERROR_CODE_DATABASE_READONLY:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_READONLY);
            }
            break;

            case DATASERVICE_ERROR_CODE_DATABASE_TOO_OLD:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_CODE_DATABASE_TOO_OLD);
            }
            break;

            case DATASERVICE_ERROR_UNKNOWN:
            {
                pacErrorCode =
                    MACRO_TO_STRING(DATASERVICE_ERROR_UNKNOWN);
            }
            break;

            default:
            {
                pacErrorCode = "Unknown";
            }
        }

        SMSE_vLog(
            psCtrl->hEventHdlr,
            "%s experienced error: %s",
            pacServiceName, pacErrorCode);
    }
    else
    {
        SMSE_vLog(
            psCtrl->hEventHdlr,
            "%s transitioned to %s",
            pacServiceName, pacState);
    }
#endif

    return;
}

/*****************************************************************************
*
*   vTimeAvailableNotification
*
*****************************************************************************/
static void vTimeAvailableNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        )
{
    BOOLEAN bOk;

    // Post the event to send time available notification
    bOk = DATASERVICE_MGR_bPostEvent(
        DATASERVICE_MGR_INVALID_OBJECT,
        DATASERVICE_FW_EVENT_TIME_AVAILABLE, NULL
            );

    if (bOk == FALSE)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post time"
            " available event.");
    }

    return;
}

/*******************************************************************************
*
*   n16CompareDataIds
*
*   Compares data service ids
*
*******************************************************************************/
static N16 n16CompareDataIds (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj1,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj2
        )
{
    return COMPARE(psObj1->tDataID, psObj2->tDataID);
}

/*****************************************************************************
*
*   bCheckModule
*
*****************************************************************************/
static BOOLEAN bCheckModule (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    do
    {
        N16 n16DriverNameCompare;

        // Do we have an associated module yet?
        if ((MODULE_INVALID_OBJECT == psCtrl->sObjectInfo.hModule) ||
            (FALSE == psCtrl->sObjectInfo.bModuleAssociated))
        {
            // Stop here in all cases
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "MODULE is not associated with DSM.\n");
            break;
        }

        // We do have a module, but are we configured to use this driver?
        n16DriverNameCompare = STRING.n16CompareCStr(
            psObj->pacSRHDriverName, 0, 
            psCtrl->sObjectInfo.hSRHDriverName);
        if (n16DriverNameCompare != 0)
        {
            // Nope!  We have a module, but not for the SRH
            // identified with the request to start this service.
            // This service will be waiting forever for a MODULE
            // attached to the requested SRH to come online.

            // This will never occur in a production system,
            // so this should be not in any SMS test
            break;
        }

        // Is this module in the error state?
        if (TRUE == psCtrl->sObjectInfo.bModuleInError)
        {
            break;
        }

        return TRUE;

    } while (FALSE);

    // Now check if the module is in the ERROR state.
    // If so, the service should go to the error state too.
    if (TRUE == psCtrl->sObjectInfo.bModuleInError)
    {
        vProcessStateChange(psCtrl, psObj,
            DATASERVICE_STATE_ERROR,
            DATASERVICE_ERROR_CODE_DEVICE_FAILURE);
    }

    return FALSE;
}

/*****************************************************************************
*
*   bIsWaitingForTime
*
*****************************************************************************/
static BOOLEAN bIsWaitingForTime (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bIsWaiting = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eOSALReturnCode;

        if (FALSE == psObj->bRequiresTime)
        {
            // time is not needed for this service
            break;
        }

        // Check if time is valid
        eOSALReturnCode = OSAL.eTimeGet((UN32 *)NULL);
        if (OSAL_SUCCESS == eOSALReturnCode)
        {
            // time is ready
            break;
        }

        // Whatever happens below we are waiting on time
        bIsWaiting = TRUE;

        if (OSAL_TIME_NOTIFICATION_INVALID_OBJECT !=
            psCtrl->hTimeNotificationHandle)
        {
            // already registered for time, don't need to do it again
            break;
        }

        // Let's register for time updates
        eOSALReturnCode = OSAL.eTimeSetRegisterNotification(
            &psCtrl->hTimeNotificationHandle,
            OSAL_TIME_UPDATE_MASK_ALL,
            vTimeAvailableNotification,
            NULL);

        if (OSAL_SUCCESS == eOSALReturnCode)
        {
            // registered, will be waiting for time
            break;
        }

        // don't have time, can't get time...
        // move to the error state and we're done
        vProcessStateChange(psCtrl, psObj,
            DATASERVICE_STATE_UNAVAILABLE,
            DATASERVICE_ERROR_CODE_NONE);

    } while(FALSE);

    return bIsWaiting;
}

/*****************************************************************************
*
*   bIsWaitingForOthers
*
*****************************************************************************/
static BOOLEAN bIsWaitingForOthers (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bIsWaiting = FALSE;
    BOOLEAN bWaitingForTime;

    // Is this service waiting for time?
    bWaitingForTime = bIsWaitingForTime(psCtrl, psObj);
    if (TRUE == bWaitingForTime)
    {
        // The service is waiting for time
        // to be initialized. Keep waiting.
        return TRUE;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;
        UN32 un32Items = 0;
        SMSAPI_RETURN_CODE_ENUM eReturnCode;
        BOOLEAN bModuleOk;

        // Check whether this service is waiting for module to be assigned.
        // Service can be in READY state if started with no associated DSIs.
        if ((DATASERVICE_STATE_INITIAL != psObj->eState) &&
            ((DATASERVICE_STATE_READY != psObj->eState) ||
             (OSAL_INVALID_OBJECT_HDL == psObj->sProducts.hProducts)))
        {
            break;
        }

        if (RADIO_PRIVATE_DATA_INVALID_OBJECT != psObj->hRadioSpecificData)
        {
            // radio already initialized
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListItems(psObj->hDSIs, &un32Items);
        if ((OSAL_SUCCESS != eOsalReturnCode) || (0 == un32Items))
        {
            // This service has no associated DSIs.
            // Radio layer is not needed at the moment.
            break;
        }

        // Service has associated DSIs, but its radio layer
        // is not initialized. Check whether it can be done
        // right now.
        bModuleOk = bCheckModule(psCtrl, psObj);
        if (FALSE == bModuleOk)
        {
            // We're waiting on the module for this service
            bIsWaiting = TRUE;
            break;
        }

        // Service is not waiting for module.
        // It's time to initialize its radio layer.
        eReturnCode = RADIO_eDataServiceInit(
            psCtrl->hRadioDataCtrl,
            (DATASERVICE_MGR_OBJECT)psObj,
            psObj->tDataID);

        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Successfully initialized
            psObj->bOTADataAvailable = TRUE;

            // We are no longer waiting
            bIsWaiting = FALSE;
        }
        else
        {
            // Radio layer is not initialized because of an error.
            // Just report this error and continue. All attempts
            // to use the radio will fail and produce subsequent
            // errors.
            // Put this service into the error state
            vProcessStateChange(psCtrl, psObj,
                DATASERVICE_STATE_ERROR,
                DATASERVICE_ERROR_CODE_DEVICE_FAILURE);

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": failed to "
                "initialize radio layer for service %s: %s",
                psObj->acServiceName,
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));
        }

    } while (FALSE);

    return bIsWaiting;
}

/*****************************************************************************
*
*   bStartDSIIterator
*
*****************************************************************************/
static BOOLEAN bStartDSIIterator(
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_START_DSI_ITERATOR_ARG *psArg
        )
{
    // TODO: FSM for this
    if (DATASERVICE_STATE_INVALID == psDSI->eState)
    {
        // Don't worry about return code from this function
        vStartDSI(psArg->psCtrl, psArg->psObj, psDSI);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bStartAllDSIs
*
*****************************************************************************/
static BOOLEAN bStartAllDSIs (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    DATASERVICE_START_DSI_ITERATOR_ARG sArg;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    sArg.psCtrl = psCtrl;
    sArg.psObj = psObj;

    // Iterate all the DSIs for this service to get them started
    eReturnCode = OSAL.eLinkedListIterate(
        psObj->hDSIs,
        (OSAL_LL_ITERATOR_HANDLER)bStartDSIIterator,
        &sArg);

    // Did we just try to start a multi-dsi service?
    if ( OSAL_NO_OBJECTS == eReturnCode )
    {
        // Yes, we can just send it to the READY state now if:
        // 1. It is in the INITIAL state thus it is in the 
        //    process of being started, and
        // 2. We have the radio data setup for the DSM (which
        //    means that the DSM is ready to go)
        if ( (psObj->eState == DATASERVICE_STATE_INITIAL) &&
             (psCtrl->hRadioDataCtrl != RADIO_PRIVATE_DATA_INVALID_OBJECT) )
        {
            bServiceStart(psCtrl, psObj, DATASERVICE_STATE_READY);
        }
    }
    else if ( OSAL_SUCCESS != eReturnCode )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": failed to "
            "start DSIs of service %s: %s",
            psObj->acServiceName,
            OSAL.pacGetReturnCodeName(eReturnCode));

        vProcessStateChange(psCtrl, psObj,
            DATASERVICE_STATE_ERROR,
            DATASERVICE_ERROR_CODE_GENERAL);
        
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bProductStateUpdateIterator
*
*****************************************************************************/
static BOOLEAN bProductStateUpdateIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_PRODUCT_STATE_UPDATE_ITERATOR_ARG *psArg
            )
{
    // Match DSI and check that the product is in valid state
    if ((psArg->tDSI == psProduct->sDSIInfo.tDSI) &&
        (DATA_PRODUCT_STATE_DISABLED != psProduct->eState) &&
        (DATA_PRODUCT_STATE_ERROR != psProduct->eState))
    {
        DATA_PRODUCT_STATE_ENUM eProductState;

        // What does the interface say the product state should be now?
        eProductState = psArg->psObj->sProducts.eGetNextProductState(
            (DATASERVICE_IMPL_HDL)(psArg->psObj + 1),
            psProduct->eType,
            psArg->tDSI,
            psArg->eDSIState,
            psArg->ptDMIs,
            psArg->un8DMICount);

        // Ok, now set that state
        vSetProductState(psArg->psObj, psProduct, eProductState);
    }
    return TRUE;
}

/*****************************************************************************
*
*   bProductForcedErrorIterator
*
*****************************************************************************/
static BOOLEAN bProductForcedErrorIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
            )
{
    if ((DATA_PRODUCT_STATE_DISABLED != psProduct->eState) &&
        (DATA_PRODUCT_STATE_ERROR != psProduct->eState))
    {
        vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_ERROR);
    }
    return TRUE;
}

/*****************************************************************************
*
*   bProductForcedDisableIterator
*
*****************************************************************************/
static BOOLEAN bProductForcedDisableIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
            )
{
    if (DATA_PRODUCT_STATE_DISABLED != psProduct->eState)
    {
        // Just mark product as disabled without managing its DSI
        vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_DISABLED);
        psProduct->sDSIInfo.tDSI = DSI_INVALID_ID;
    }
    return TRUE;
}

/*****************************************************************************
*
*   bProductIterator
*
*****************************************************************************/
static BOOLEAN bProductIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_PRODUCT_ITERATOR_ARG *psArg
        )
{
    return psArg->bCallback(
        psArg->hManager,
        psProduct->eType, psProduct->tMask, psProduct->eState,
        psArg->pvCallbackArg);
}

/*****************************************************************************
*
*   un8GetDMIs
*
* The output dynamic array pptDMIs must be freed by caller
*
*****************************************************************************/
static UN8 un8GetDMIs (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    SXM_DMI **pptDMIs
        )
{
    UN8 un8DMIsCount = 0;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    *pptDMIs = NULL;

    if ((DATASERVICE_STATE_READY != psDSI->eState) &&
        (DATASERVICE_STATE_POI_UPDATES_ONLY != psDSI->eState))
    {
        return 0;
    }

    // retrieve the list of associated DMIs

    eReturnCode = RADIO_eDataServiceGetDMIsCount(
        (DATASERVICE_MGR_OBJECT)psObj,
        psDSI->tDSI,
        &un8DMIsCount);

    if ((SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) &&
        (un8DMIsCount > 0))
    {
        *pptDMIs = (SXM_DMI *)OSAL.pvMemoryAllocate(
            "DMIs", sizeof(SXM_DMI) * un8DMIsCount, FALSE);

        if (NULL != *pptDMIs)
        {
            eReturnCode = RADIO_eDataServiceGetDMIs(
                (DATASERVICE_MGR_OBJECT)psObj,
                psDSI->tDSI, *pptDMIs, un8DMIsCount);
        }
        else
        {
            // simulate error
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }
    }

    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
    {
        un8DMIsCount = 0;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to retrieve DMIs for DSI %u in service %s.",
                psDSI->tDSI, psObj->acServiceName);
    }

    return un8DMIsCount;
}

/*****************************************************************************
*
*   vApplyDSIStateToProducts
*
*   DSI state handler for multi-DSI services
*
*   TODO: Write a real FSM to deal with stuff like this
*
*****************************************************************************/
static void vApplyDSIStateToProducts (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM ePreviousState
        )
{
    // First of all ensure that service is not kept in initial state
    if (DATASERVICE_STATE_INITIAL == psObj->eState)
    {
        // Assume everything is ok if the service
        // is still in INITIAL state. Advance to ready state.
        BOOLEAN bOk;

        // Start this service now
        bOk = bServiceStart(psCtrl, psObj, DATASERVICE_STATE_READY);
        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": Ready not accepted in service %s.",
                    psObj->acServiceName);

            return;
        }
    }

    // If this DSI is transitioning out of initial/unsubscribed
    // then we should enable the data flow
    if ((DATASERVICE_STATE_READY == psDSI->eState) ||
        (DATASERVICE_STATE_POI_UPDATES_ONLY == psDSI->eState))
    {
        BOOLEAN bDataFlowStarted;

        bDataFlowStarted = bEnableDataFlow(psCtrl, psObj, psDSI->tDSI);
        if (FALSE == bDataFlowStarted)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to start data flow for service %s (DSI: %u)", 
                psObj->acServiceName, psDSI->tDSI);
        }
    }

    if (OSAL_INVALID_OBJECT_HDL != psObj->sProducts.hProducts)
    {
        // The products for this DSI must now be updated
        DATASERVICE_MGR_PRODUCT_STATE_UPDATE_ITERATOR_ARG sArg;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        sArg.psObj = psObj;
        sArg.tDSI = psDSI->tDSI;
        sArg.eDSIState = psDSI->eState;
        sArg.un8DMICount = un8GetDMIs(psObj, psDSI, &sArg.ptDMIs);

        // Iterate all products now
        eReturnCode = OSAL.eLinkedListIterate(
            psObj->sProducts.hProducts,
            (OSAL_LL_ITERATOR_HANDLER)bProductStateUpdateIterator,
            &sArg);

        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Failed to update "
                "product states of service %s, DSI %u -- error %s",
                psObj->acServiceName, psDSI->tDSI,
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        if (NULL != sArg.ptDMIs)
        {
            OSAL.vMemoryFree(sArg.ptDMIs);
        }
    }

    return;
}

/*****************************************************************************
*
*   vApplyDSIStateToService
*
*   DSI state handler for single DSI services
*
*   TODO: Write a real FSM to deal with stuff like this
*
*****************************************************************************/
static void vApplyDSIStateToService (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM ePreviousState
        )
{
    DATASERVICE_STATE_ENUM eState = psDSI->eState;

    // No DSI state can pull a service out of following states
    if ((DATASERVICE_STATE_STOPPED == psObj->eState) ||
        (DATASERVICE_STATE_ERROR == psObj->eState) ||
        (DATASERVICE_STATE_INVALID == psObj->eState))
    {
        return;
    }

    // If this service is transitioning out of initial/unsubscribed
    // then we should enable the data flow and start the service
    if ((DATASERVICE_STATE_READY == eState) ||
        (DATASERVICE_STATE_POI_UPDATES_ONLY == eState))
    {
        BOOLEAN bOk;

        bOk = bEnableDataFlow(psCtrl, psObj, psDSI->tDSI);
        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to start data flow for service %s (DSI: %u)", 
                psObj->acServiceName, psDSI->tDSI);
            return;
        }

        bOk = bServiceStart(psCtrl, psObj, eState);

        if (FALSE == bOk)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME
                    ": Ready not accepted in service %s.",
                    psObj->acServiceName);
        }

        return;
    }

    // Error and special cases for OTA-required services
    if (DATASERVICE_STATE_INVALID == eState)
    {
        // This DSI state cannot be directly reflected
        // onto the service state. Assume a error for this case.
        eState = DATASERVICE_STATE_ERROR;
    }

    vProcessStateChange(psCtrl, psObj, eState,
        (DATASERVICE_STATE_ERROR == eState)
            ? DATASERVICE_ERROR_CODE_GENERAL
            : DATASERVICE_ERROR_CODE_NONE);

    return;
}

/*****************************************************************************
*
*   vSetDSIState
*
*   DSI states meaning:
*
*   INVALID - has never been provided to radio layer (should not appear here)
*   INITIAL - provided to radio layer, waiting for response
*   POI_UPDATES - monitored by radio layer, partially subscribed
*   READY - monitored by radio layer, fully subscribed
*   UNSUBSCRIBED - monitored by radio layer, not subscribed
*   UNAVAILABLE - may be handled by radio layer, but not monitoring
*   STOPPED - no longer handled by radio layer, not monitoring
*   ERROR - internal error, not known by radio layer
*
*****************************************************************************/
static void vSetDSIState (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM eState
        )
{
    DATASERVICE_STATE_ENUM ePreviousState = psDSI->eState;
    psDSI->eState = eState;

    // TODO: this is the kinda thing we need to remove (create an FSM to deal 
    // with proper state transitions)
    if (DATASERVICE_STATE_INITIAL == eState)
    {
        // INITIAL DSI state is a temporary state which means that
        // DSI is being started for monitoring. It is used for informational
        // and internal purposes only and needs to be ignored here.
        return;
    }

    psObj->vDSIStateHandler(psCtrl, psObj, psDSI, ePreviousState);

    return;
}

/*****************************************************************************
*
*   n16CompareProducts
*
*****************************************************************************/
static N16 n16CompareProducts (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct1,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct2
        )
{
    return COMPARE(psProduct1->eType, psProduct2->eType);
}

/*****************************************************************************
*
*   psFindProduct
*
*****************************************************************************/
static DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psFindProduct (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATA_PRODUCT_TYPE_ENUM eProductType
        )
{
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT sSearchCriteria;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    sSearchCriteria.eType = eProductType;

    eReturnCode = OSAL.eLinkedListSearch(
        psObj->sProducts.hProducts,
        &hEntry, &sSearchCriteria);

    if (OSAL_SUCCESS == eReturnCode)
    {
        psProduct = (DATASERVICE_MGR_PRODUCT_INFO_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }

    return psProduct;
}

/*****************************************************************************
*
*   vHandleEnableProduct
*
*****************************************************************************/
static void vHandleEnableProduct (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK tProductMask
        )
{
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct;
    BOOLEAN bOk;

    psProduct = psFindProduct(psObj, eProductType);

    if (NULL == psProduct)
    {
        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s: Product %d is not found.\n",
            psObj->acServiceName, eProductType);

        return;
    }

    if (DATA_PRODUCT_STATE_DISABLED != psProduct->eState)
    {
        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s: Product %d is in state %d and cannot be enabled.\n",
            psObj->acServiceName, psProduct->eType, psProduct->eState);

        return;
    }

    // Get information about DSI used by the product with specified mask
    bOk = psObj->sProducts.bGetDSIForProduct(
        eProductType, tProductMask, &psProduct->sDSIInfo);

    if (FALSE == bOk)
    {
        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s: Cannot enable product %d with mask 0x%X.\n",
            psObj->acServiceName,  eProductType, tProductMask);

        return;
    }

    // Start with INITIAL state
    psProduct->tMask = tProductMask;
    vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_INITIAL);

    // There are some cases when a product can work without monitoring a DSI.
    // In those cases it gives the invalid DSI id.
    if (DSI_INVALID_ID != psProduct->sDSIInfo.tDSI)
    {
        DATASERVICE_MGR_DSI_REF_STRUCT *psDSI;

        // Check whether this DSI is also used by another product
        psDSI = psFindDSI(psObj, psProduct->sDSIInfo.tDSI);
        if (NULL == psDSI)
        {
            // This DSI is used for the first time
            psDSI = psCreateDSI(
                psObj, psProduct->sDSIInfo.tDSI,
                psProduct->sDSIInfo.tSuggestedOTABufferByteSize,
                psProduct->sDSIInfo.bEnableAllDMIs);

            if (NULL != psDSI)
            {
                BOOLEAN bWaiting;

                // Check whether the service is waiting for other objects.
                // This also guarantees that radio layer is initialized if
                // everything is ok.
                bWaiting = bIsWaitingForOthers(psCtrl, psObj);
                if (FALSE == bWaiting)
                {
                    // Start this DSI.
                    // In a error case, the DSI changes its state appropriately,
                    // which will cause the product to change its state as well.
                    // So, the result can be ignored here.
                    vStartDSI(psCtrl, psObj, psDSI);
                }
            }
            else
            {
                // Failed to create DSI... this is an error for sure
                vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_ERROR);
            }
        }
        else
        {
            // This DSI is already used by another product.
            // Take it for this product as well.
            vDSIRef(psObj, psDSI);

            // DSI has INVALID or INITIAL state during its initialization.
            // Keep the service in INITIAL state in these cases.
            if ((DATASERVICE_STATE_INVALID != psDSI->eState) &&
                (DATASERVICE_STATE_INITIAL != psDSI->eState))
            {
                DATASERVICE_MGR_PRODUCT_STATE_UPDATE_ITERATOR_ARG sArg;

                sArg.psObj = psObj;
                sArg.tDSI = psDSI->tDSI;
                sArg.eDSIState = psDSI->eState;
                sArg.un8DMICount = un8GetDMIs(psObj, psDSI, &sArg.ptDMIs);

                // Advance product to appropriate state
                bProductStateUpdateIterator(psProduct, &sArg);

                if (NULL != sArg.ptDMIs)
                {
                    OSAL.vMemoryFree(sArg.ptDMIs);
                }
            }
        }
    }
    else
    {
        // This product doesn't use any DSI, advance to READY state
        vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_READY);
    }

    return;
}

/*****************************************************************************
*
*   vHandleDisableProduct
*
*****************************************************************************/
static void vHandleDisableProduct (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct
        )
{
    if (DATA_PRODUCT_STATE_DISABLED != psProduct->eState)
    {
        DATASERVICE_MGR_DSI_REF_STRUCT *psDSI;

        vSetProductState(psObj, psProduct, DATA_PRODUCT_STATE_DISABLED);

        // The associated DSI is no longer needed
        // for this product, detach it

        psDSI = psFindDSI(psObj, psProduct->sDSIInfo.tDSI);
        psProduct->sDSIInfo.tDSI = DSI_INVALID_ID;

        if (NULL != psDSI)
        {
            vDSIUnref(psCtrl, psObj, psDSI);
        }
    }

    return;
}

/*****************************************************************************
*
*   vHandleStartTimedEvent
*
*****************************************************************************/
static void vHandleStartTimedEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bRepeat,
    UN32 un32IntervalInSeconds
        )
{
    BOOLEAN bOk;

    // Generic data services don't have implementation handles - just the base
    // dataservice object structure; DATASERVICE_IMPL_hRegisterTimedEvent
    // expects an implementation handle, so we call the private
    // hRegisterTimedEvent instead.

    // First, let's see if we've created a timed event for this yet
    if ( DATASERVICE_TIMED_EVENT_INVALID_HDL == psObj->hGenericTimedEvent )
    {
        // Note that we pass NULL, as the generic timed event API
        // does not support callback arguments.
        psObj->hGenericTimedEvent = hRegisterTimedEvent(
                (DATASERVICE_MGR_OBJECT)psObj, NULL );
    }

    // Make sure our timed event is actually valid now.
    if ( DATASERVICE_TIMED_EVENT_INVALID_HDL == psObj->hGenericTimedEvent )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to create timed event.");
        return;
    }

    // Now that we know we have a valid timed event handle, we can kick off
    // the event; we reset the event (if it's already running) and apply
    // whatever repeat & timing settings the caller gave us.
    bOk = DATASERVICE_IMPL_bSetTimedEvent ( psObj->hGenericTimedEvent, TRUE,
            bRepeat, un32IntervalInSeconds );

    // Make sure that went okay
    if ( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": DATASERVICE_IMPL_bSetTimedEvent"
            " failed.");
    }

    return;
}

/*****************************************************************************
*
*   vHandleStopTimedEvent
*
*****************************************************************************/
static void vHandleStopTimedEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bOk;

    // Make sure our timed event is valid before attemping a stop.
    if ( DATASERVICE_TIMED_EVENT_INVALID_HDL == psObj->hGenericTimedEvent )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": timed event handle is invalid.");
        return;
    }

    bOk = DATASERVICE_IMPL_bStopTimedEvent ( psObj->hGenericTimedEvent );

    // Make sure the stop timed event call went okay
    if ( FALSE == bOk )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": failed call to"
            "DATASERVICE_IMPL_bStopTimedEvent.");
    }

    return;
}

/*****************************************************************************
*
*   vSetProductState
*
*****************************************************************************/
static void vSetProductState (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATA_PRODUCT_STATE_ENUM eState
        )
{
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT sEventArg;
    DATA_PRODUCT_STATE_ENUM ePreviousState = psProduct->eState;

    // Set product state
    psProduct->eState = eState;

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "%s: product %d is moved to state %d.\n",
        psObj->acServiceName, psProduct->eType, psProduct->eState);

    // Inform service about product state change
    sEventArg.eType = psProduct->eType;
    sEventArg.ePreviousState = ePreviousState;
    sEventArg.eState = eState;

    vCallEventHandler(
        psObj,
        DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE,
        &sEventArg);

    return;
}

/*****************************************************************************
*
*   n16CompareDSIs
*
*****************************************************************************/
static N16 n16CompareDSIs (
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI1,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI2
        )
{
    return COMPARE(psDSI1->tDSI, psDSI2->tDSI);
}

/*****************************************************************************
*
*   psCreateDSI
*
*****************************************************************************/
static DATASERVICE_MGR_DSI_REF_STRUCT *psCreateDSI (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSI tDSI,
    size_t tSuggestedOTABufferByteSize,
    BOOLEAN bEnableAllDMIs
        )
{
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI = NULL;

    psDSI = (DATASERVICE_MGR_DSI_REF_STRUCT *)
        OSAL.pvLinkedListMemoryAllocate(
            DATASERVICE_MGR_OBJECT_NAME":DSIRef",
            sizeof(*psDSI), FALSE);

    if (NULL != psDSI)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        psDSI->tDSI = tDSI;
        psDSI->tSuggestedOTABufferByteSize = tSuggestedOTABufferByteSize;
        psDSI->bEnableAllDMIs = bEnableAllDMIs;
        psDSI->eState = DATASERVICE_STATE_INVALID;
        psDSI->tRefCount = 1;
        psDSI->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Add the DSI to the tracking list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hDSIs, &psDSI->hEntry, psDSI);

        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Failed to add DSI %u to the list in service %s",
                tDSI, psObj->acServiceName);

            OSAL.vLinkedListMemoryFree(psDSI);
            psDSI = NULL;
        }
    }

    return psDSI;
}

/*****************************************************************************
*
*   psFindDSI
*
*****************************************************************************/
static DATASERVICE_MGR_DSI_REF_STRUCT *psFindDSI(
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSI tDSI
        )
{
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    DATASERVICE_MGR_DSI_REF_STRUCT sSearchCriteria;

    sSearchCriteria.tDSI = tDSI;

    eReturnCode = OSAL.eLinkedListSearch(
        psObj->hDSIs, &hEntry, &sSearchCriteria);

    if ((OSAL_SUCCESS == eReturnCode) &&
        (OSAL_INVALID_LINKED_LIST_ENTRY != hEntry))
    {
        psDSI = (DATASERVICE_MGR_DSI_REF_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }

    return psDSI;
}

/*****************************************************************************
*
*   vStartDSI
*
*****************************************************************************/
static void vStartDSI (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    // The first state for a DSI must be reported as INITIAL
    vSetDSIState(psCtrl, psObj, psDSI, DATASERVICE_STATE_INITIAL);

    // Attempt to start DSI.
    eReturnCode = RADIO_eDataServiceAddDSI(
        psCtrl->hRadioDataCtrl,
        (DATASERVICE_MGR_OBJECT)psObj,
        psDSI->tDSI,
        psDSI->tSuggestedOTABufferByteSize,
        psDSI->bEnableAllDMIs);

    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": failed to add DSI %u "
            "in service %s: %s",
            psDSI->tDSI, psObj->acServiceName,
            SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));

        vSetDSIState(psCtrl, psObj, psDSI, DATASERVICE_STATE_ERROR);
    }

    return;
}

/*****************************************************************************
*
*   vStopDSI
*
*****************************************************************************/
static void vStopDSI (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        )
{
    if ((DATASERVICE_STATE_INVALID != psDSI->eState) &&
        (DATASERVICE_STATE_STOPPED != psDSI->eState) &&
        (DATASERVICE_STATE_ERROR != psDSI->eState))
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = RADIO_eDataServiceRemoveDSI(
            psCtrl->hRadioDataCtrl,
            (DATASERVICE_MGR_OBJECT)psObj, psDSI->tDSI);

        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode)
        {
            vSetDSIState(psCtrl, psObj, psDSI, DATASERVICE_STATE_STOPPED);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": RADIO_eDataServiceRemoveDSI failed"
                "in service %s (DSI %u) (error %s)",
                psObj->acServiceName, 
                psDSI->tDSI, 
                SMSAPI_DEBUG_pacReturnCodeText(eReturnCode));

            vSetDSIState(psCtrl, psObj, psDSI, DATASERVICE_STATE_ERROR);
        }
    }

    return;
}

/*****************************************************************************
*
*   vDSIRef
*
*****************************************************************************/
static void vDSIRef (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        )
{
    psDSI->tRefCount++;

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "%s: DSI %u is reused, state = %u, refcount = %u.\n",
        psObj->acServiceName, psDSI->tDSI,
        psDSI->eState, psDSI->tRefCount);

    return;
}

/*****************************************************************************
*
*   vDSIUnref
*
* The specified DSI pointer can become invalid after this function.
*
*****************************************************************************/
static void vDSIUnref (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        )
{
    if (psDSI->tRefCount > 1)
    {
        // This DSI is still used by other products
        // Decrease reference count only
        psDSI->tRefCount--;

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s: DSI %u is still used, "
            "state = %u, refcount = %u.\n",
            psObj->acServiceName, psDSI->tDSI,
            psDSI->eState, psDSI->tRefCount);
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "%s: DSI %u is no longer used.\n",
            psObj->acServiceName, psDSI->tDSI);

        vStopDSI(psCtrl, psObj, psDSI);

        eReturnCode = OSAL.eLinkedListRemove(psDSI->hEntry);

        if (OSAL_SUCCESS == eReturnCode)
        {
            OSAL.vLinkedListMemoryFree(psDSI);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Failed to destroy reference count entry "
                "for DSI %u in service %s",
                psDSI->tDSI, psObj->acServiceName);
        }
    }

    return;
}

/*****************************************************************************
*
*   vHandleModuleAssociateEvent
*
*****************************************************************************/
static void vHandleModuleAssociateEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    MODULE_OBJECT hModule,
    STI_HDL hSTI,
    STRING_OBJECT hSRHDriverName
        )
{
    BOOLEAN bModuleValid;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "DATASERVICE_FW_EVENT_MODULE_ASSOCIATE: "
        "hModule: %p, hSTI: %p, hSRHDriverName: %s\n",
        hModule, hSTI, STRING_INVALID_OBJECT != hSRHDriverName ?
            STRING.pacCStr(hSRHDriverName) : "STRING_INVALID_OBJECT");
    do
    {
        // Validate the module handle
        bModuleValid = SMSO_bValid((SMS_OBJECT)hModule);
        if (FALSE == bModuleValid)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to assign invalid MODULE to DSM.");
            break;
        }

        // Validate Driver Name
        if (STRING_INVALID_OBJECT == hSRHDriverName)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": hSRHDriverName is not specified.");
            break;
        }

        // Do we already have a module?
        if (MODULE_INVALID_OBJECT != psCtrl->sObjectInfo.hModule)
        {
            // We don't need multiple modules...
            // so just skip the rest of this
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "MODULE (%p) is already assigned. "
                "Skip MODULE (%p) association.",
                psCtrl->sObjectInfo.hModule, hModule);
            break;
        }

        if (RADIO_PRIVATE_DATA_INVALID_OBJECT == psCtrl->hRadioDataCtrl)
        {
            // Initialize the radio data
            psCtrl->hRadioDataCtrl =
                RADIO_hDataServiceInitialize((SMS_OBJECT)psCtrl, hSTI);
            if (RADIO_PRIVATE_DATA_INVALID_OBJECT == psCtrl->hRadioDataCtrl)
            {
                // Error!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    DATASERVICE_MGR_OBJECT_NAME": Unable to initialize radio");
                break;
            }
        }

        // Store these handles now
        psCtrl->sObjectInfo.hSRHDriverName = hSRHDriverName;
        psCtrl->sObjectInfo.hModule = hModule;
        psCtrl->sObjectInfo.bModuleAssociated = TRUE;

        // Iterate the list if Data Services
        eReturnCode = OSAL.eLinkedListIterate(
            psCtrl->hMgrList,
            (OSAL_LL_ITERATOR_HANDLER)bDataServiceStartIterator,
            (void *)psCtrl);
        if ((eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS))
        {
            // Just output the error message
            // if some Data Services are not started.
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Unable to assign radio to all data services.");
        }

        // MODULE is now associated with DSM
        return;

    } while (FALSE);

    // Error! Cleanup.

    if (STRING_INVALID_OBJECT != hSRHDriverName)
    {
        STRING.vDestroy(hSRHDriverName);
    }

    if (TRUE == bModuleValid)
    {
        MODULE_bRelease(hModule, SMS_OBJECT_RELEASE_BY_OTHERS);
    }

    return;
}

/*****************************************************************************
*
*   vHandleModuleUnassociateEvent
*
*****************************************************************************/
static void vHandleModuleUnassociateEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    MODULE_OBJECT hModule
        )
{
    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
        "DATASERVICE_FW_EVENT_MODULE_UNASSOCIATE: hModule: %p\n", hModule);

    do
    {
        // Do we have an associated module?
        if ((MODULE_INVALID_OBJECT == psCtrl->sObjectInfo.hModule) ||
            (FALSE == psCtrl->sObjectInfo.bModuleAssociated))
        {
            // Ignore the event
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "MODULE is not associated with DSM.\n");
            break;
        }

        // Only accept this event from the module we're currently using
        if (hModule != psCtrl->sObjectInfo.hModule)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": UNASSOCIATE from inappropriate MODULE (%p).",
                hModule);
            break;
        }

        // Reset flag
        psCtrl->sObjectInfo.bModuleAssociated = FALSE;

        // Try to finalize DSM stopping
        bTryStopDSM(psCtrl);

    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   bTryStopDSM
*
*****************************************************************************/
static BOOLEAN bTryStopDSM (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN32 un32Services;
    BOOLEAN bResult = FALSE;

    // DSM shall be completely stopped only if:
    // 1) All Data Services are stopped.
    // 2) Assigned MODULE is unassociated from DSM.
    //    It means that the MODULE is not going to send any events to DSM
    //    (the MODULE is currently being released).

    do
    {
        // How many services do we still have running?
        eReturnCode = OSAL.eLinkedListItems(psCtrl->hMgrList, &un32Services);
        if (OSAL_SUCCESS != eReturnCode)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME
                ": Could not get number of Data Services: %d (%s).",
                eReturnCode, OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }
        else if (0 != un32Services)
        {
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "Not all Data Services are stopped yet (%u remained).\n",
                un32Services);
            break;
        }

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "All Data Services are now STOPPED.\n");

        // Check if the MODULE is still associated with DSM
        if (TRUE == psCtrl->sObjectInfo.bModuleAssociated)
        {
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "MODULE is still associated wuth DSM.\n");
            break;
        }

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "MODULE is not associated wuth DSM.\n");

        // Now DSM stopping shall be completed.
        // Before releasing the MODULE we should dispatch
        // all remaining events by posting of special event
        // which shall be dispatched last from the queue.
        bResult = bPostFinalStopEvent(psCtrl);

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   bPostFinalStopEvent
*
*****************************************************************************/
static BOOLEAN bPostFinalStopEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    BOOLEAN bResult;

    bResult = SMSE_bPostSignal(psCtrl->hEventHdlr,
        SMS_EVENT_STOP, SMS_EVENT_OPTION_DEFERRED);
    if (TRUE == bResult)
    {
        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "FINAL STOP event is posted.\n");
    }
    else
    {
        // Error!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            DATASERVICE_MGR_OBJECT_NAME": Unable to post FINAL STOP event.");
    }

    return bResult;
}

/*****************************************************************************
*
*   vHandleFinalStopEvent
*
*****************************************************************************/
static void vHandleFinalStopEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        )
{
    SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4, "SMS_EVENT_STOP\n");

    // This is the final event in the DSM stopping process.

    // Uninit radio data if necessary
    if (RADIO_PRIVATE_DATA_INVALID_OBJECT != psCtrl->hRadioDataCtrl)
    {
        RADIO_vDataServiceUninitialize(psCtrl->hRadioDataCtrl);
        psCtrl->hRadioDataCtrl = RADIO_PRIVATE_DATA_INVALID_OBJECT;

        SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
            "DSM RADIO is un-initialized.\n");
    }

    // We should release the MODULE now
    if (MODULE_INVALID_OBJECT != psCtrl->sObjectInfo.hModule)
    {
        BOOLEAN bResult;

        bResult = MODULE_bRelease(psCtrl->sObjectInfo.hModule,
            SMS_OBJECT_RELEASE_BY_OTHERS);
        if (TRUE == bResult)
        {
            SMSAPI_DEBUG_vPrint(DATASERVICE_MGR_OBJECT_NAME, 4,
                "RELEASE is posted to MODULE.\n");

            // Clear all of the module information
            psCtrl->sObjectInfo.hModule = MODULE_INVALID_OBJECT;
            psCtrl->sObjectInfo.bModuleInError = FALSE;
            STRING.vDestroy(psCtrl->sObjectInfo.hSRHDriverName);
            psCtrl->sObjectInfo.hSRHDriverName = STRING_INVALID_OBJECT;
        }
        else
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                DATASERVICE_MGR_OBJECT_NAME": Unable to release MODULE.");
        }
    }

    return;
}

#ifdef SUPPORT_CUNIT
#include <dataservice_mgr_obj.cunit>
#endif
