/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the dataservice mgr implementation header for the sms implemented
 * data services.
 *
 ******************************************************************************/

#ifndef DATASERVICE_MGR_IMPL_H_
#define DATASERVICE_MGR_IMPL_H_

#include "standard.h"

#include "osal.h"
#include "sms_api.h"
#include "dataservice_base.h"
#include "device_obj.h"
#include "dsrl_obj.h"
#include "tag_obj.h"
#include "sti_api.h"
#include "radio.h"

// If logging hasn't been configured already
// then only enable when we're in debug
#ifndef SMS_LOGGING
#if SMS_DEBUG == 1 && OSAL_LOG == 1
#define SMS_LOGGING 1
#else
#define SMS_LOGGING 0
#endif
#endif

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Internal manager event types
// TODO: Ensure these don't collide with data service
// events found in sms_api.h
#define DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED       (0x00400)
#define DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED     (0x00800)
#define DATASERVICE_INTERNAL_EVENT_DSRL                     (0x02000)
#define DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC         (0x04000)
#define DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE            (0x08000)
#define DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE              (0x10000)

#define DATASERVICE_INTERNAL_EVENT_ALL (                  \
    (DATASERVICE_EVENT_MASK)                              \
    DATASERVICE_EVENT_ALL                               | \
    DATASERVICE_INTERNAL_EVENT_DECODER_SUBSCRIBED       | \
    DATASERVICE_INTERNAL_EVENT_DECODER_UNSUBSCRIBED     | \
    DATASERVICE_INTERNAL_EVENT_DSRL                     | \
    DATASERVICE_INTERNAL_EVENT_SERVICE_SPECIFIC         | \
    DATASERVICE_INTERNAL_EVENT_PRODUCT_STATE            | \
    DATASERVICE_INTERNAL_EVENT_SXI_MESSAGE                \
    )

typedef struct dataservice_impl_struct * DATASERVICE_IMPL_HDL;
#define DATASERVICE_IMPL_INVALID_HDL (DATASERVICE_IMPL_HDL)0

typedef struct dataservice_timed_event_struct * DATASERVICE_TIMED_EVENT_HDL;
#define DATASERVICE_TIMED_EVENT_INVALID_HDL (DATASERVICE_TIMED_EVENT_HDL)0

// Used to iterate through subscribed decoders
// for autonomous services
typedef BOOLEAN (*DATASERVICE_IMPL_UPDATE_SUBSCRIBED_ITERATOR) (
    DECODER_OBJECT hDecoder,
    void *pvArg
        );

/*****  Favorites Manipulation *****/
typedef DSRL_TARGET_OBJECT (*DATASERVICE_IMPL_CREATE_TARGET_FROM_TAG) (
    TAG_OBJECT hTag
        );

typedef TAG_OBJECT (*DATASERVICE_IMPL_GET_TAG_FOR_TARGET) (
    TAG_OBJECT hParentTag,
    DSRL_TARGET_OBJECT hTarget
        );

typedef struct dataservice_dsi_info_struct
{
    DSI tDSI;
    size_t tSuggestedOTABufferByteSize;
    BOOLEAN bEnableAllDMIs;

} DATASERVICE_DSI_INFO_STRUCT;

typedef BOOLEAN (*DATASERVICE_GET_DSI_FOR_PRODUCT) (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        );

typedef DATA_PRODUCT_STATE_ENUM (*DATASERVICE_GET_NEXT_PRODUCT_STATE) (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        );

typedef struct dataservice_product_descriptor_struct
{
    DATA_PRODUCT_TYPE_ENUM eProductType;
    DATA_PRODUCT_MASK tAllowedMask;

} DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT;

typedef struct dataservice_products_info_struct
{
    const DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT *peProducts;
    UN8 un8Count;
    DATASERVICE_GET_DSI_FOR_PRODUCT bGetDSIForProduct;
    DATASERVICE_GET_NEXT_PRODUCT_STATE eGetNextProductState;

} DATASERVICE_PRODUCTS_INFO_STRUCT;

typedef struct dataservice_product_state_event_arg_struct
{
    DATA_PRODUCT_TYPE_ENUM eType;
    DATA_PRODUCT_STATE_ENUM ePreviousState;
    DATA_PRODUCT_STATE_ENUM eState;

} DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT;

// Structure groups data service attributes
// for a more convenient service start function
typedef struct dataservice_create_struct
{
    // General manager attributes
    const char *pacSRHDriverName;
    const char *pacServiceObjectName;
    size_t tServiceObjectSize;
    BOOLEAN bRequiresDeviceGroup;

    // OTA-related attributes
    DATASERVICE_ID tDataID; // = associated DSI for one-DSI services
    BOOLEAN bEnableAll;
    BOOLEAN bTimeRequired;
    size_t tSuggestedOTABufferByteSize;

    // Application callback info
    DATASERVICE_EVENT_MASK tEventRequestMask;
    GENERIC_DATASERVICE_EVENT_CALLBACK vEventCallback;

    // Enhanced behavior attributes
    BOOLEAN bMultiDSIRequired;
    DATASERVICE_PRODUCTS_INFO_STRUCT sProductsInfo;

} DATASERVICE_CREATE_STRUCT;

// Structure groups DSRL attributes
// for a more convenient service start function
typedef struct dataservice_dsrl_config_struct
{
    // General attributes
    DATASERVICE_TYPE_ENUM eServiceType;
    size_t tParentObjectSize;
    size_t tServiceDataSize;

    // Favorites manipulation
    BOOLEAN bFavoritesEnabled;
    DATASERVICE_IMPL_CREATE_TARGET_FROM_TAG hCreateTargetFromTag;
    DATASERVICE_IMPL_GET_TAG_FOR_TARGET hGetTagForTarget;

    // Device-based attributes
    BOOLEAN bDeviceEnabled;
    UN32 un32DeviceNotifyDistance;
    DISTANCE_UNIT_TYPE_ENUM eDeviceNotifyUnits;

} DATASERVICE_DSRL_CONFIG_STRUCT;

typedef struct dataservice_option_values_struct
{
    // Argument for DATASERVICE_OPTION_REFERENCE_DB_PATH
    char *pcRefDBPath;

    // Argument for DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES
    BOOLEAN bUpdateRefDB;

    // Argument for DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT
    IMAGE_FORMAT_ENUM eImageOutputFormat;

    // Argument(s) for something else goes here...

} DATASERVICE_OPTION_VALUES_STRUCT;

/*
    General State Handling
*/

typedef BOOLEAN (*DATASERVICE_STATE_HANDLER) (
    void *pvArg
        );

typedef struct dataservice_state_handlers_struct
{
    DATASERVICE_STATE_HANDLER bHandleServiceReady;
    DATASERVICE_STATE_HANDLER bHandleServiceStopped;
    DATASERVICE_STATE_HANDLER bHandleServiceError;

} DATASERVICE_STATE_HANDLERS_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

// Verify that the size of the structure provided is less than
// SMS_EVENT_DATASERVICE_UNION. If it is not, then a compiler
// error will occur...something like "size of array is negative",
// which means the structure is too large. Any module that
// generates this error needs to have this structure reduced in size
#define DSM_IMPL_EVENT_COMPILE_TIME_ASSERT(structure) \
 typedef char structure##_is_too_big_you_need_to_reduce_the_size \
     [(sizeof(structure) <= sizeof(SMS_EVENT_DATASERVICE_UNION)) ? 1 : -1]

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

void DATASERVICE_IMPL_vInitCreateStruct (
    DATASERVICE_CREATE_STRUCT *psCreate
        );

BOOLEAN DATASERVICE_IMPL_bProcessOptions (
    DATASERVICE_OPTIONS_MASK tAllowedOptions,
    DATASERVICE_OPTIONS_STRUCT const *psOptions,
    DATASERVICE_OPTION_VALUES_STRUCT *psOptionValues
        );

void DATASERVICE_IMPL_vFreeOptions (
    DATASERVICE_OPTION_VALUES_STRUCT *psOptions
        );

SMS_OBJECT DATASERVICE_IMPL_hCreateNewService (
    DATASERVICE_CREATE_STRUCT *psCreateService
        );

void DATASERVICE_IMPL_vDestroy (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

BOOLEAN DATASERVICE_IMPL_bIsValid (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

BOOLEAN DATASERVICE_IMPL_bValid (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

BOOLEAN DATASERVICE_IMPL_bOwner (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

BOOLEAN DATASERVICE_IMPL_bLock (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

void DATASERVICE_IMPL_vUnlock (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

SMS_OBJECT DATASERVICE_IMPL_hSMSObj (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

DATASERVICE_STATE_ENUM DATASERVICE_IMPL_eState (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

DATASERVICE_ERROR_CODE_ENUM DATASERVICE_IMPL_eErrorCode (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

void DATASERVICE_IMPL_vStop (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

void DATASERVICE_IMPL_vStopAutonomous (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

SMS_EVENT_HDL DATASERVICE_IMPL_hAllocateEvent(
    DATASERVICE_IMPL_HDL hServiceImpl,
    void **ppvEventData
        );

BOOLEAN DATASERVICE_IMPL_bPostEvent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_FW_EVENT_ENUM eEvent,
    void *pvDataEventArg
        );

void DATASERVICE_IMPL_vDecoderSubscribed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DECODER_OBJECT hDecoder
        );

BOOLEAN DATASERVICE_IMPL_bStateFSM (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_STATE_CHANGE_STRUCT const *psStateChange,
    DATASERVICE_STATE_HANDLERS_STRUCT const *psHandlers,
    void *pvHandlerArg
        );

BOOLEAN DATASERVICE_IMPL_bGenericStateHandler (
    void *pvUnused
        );

BOOLEAN DATASERVICE_IMPL_bStart (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

DATASERVICE_TIMED_EVENT_HDL DATASERVICE_IMPL_hRegisterTimedEvent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    void *pvTimerEventArg
        );

BOOLEAN DATASERVICE_IMPL_bSetTimedEvent (
    DATASERVICE_TIMED_EVENT_HDL hEvent,
    BOOLEAN bResetIfStarted,
    BOOLEAN bRepeatEvent,
    UN32 un32TimeUntilEvent
        );

BOOLEAN DATASERVICE_IMPL_bStopTimedEvent (
    DATASERVICE_TIMED_EVENT_HDL hEvent
        );

BOOLEAN DATASERVICE_IMPL_bOTADataAvailable (
    DATASERVICE_IMPL_HDL hServiceImpl
        );

void DATASERVICE_IMPL_vError (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

void DATASERVICE_IMPL_vProductError (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eProductType
        );

BOOLEAN DATASERVICE_IMPL_bUpdateSubscribed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DECODER_OBJECT hDecoder,
    DATASERVICE_IMPL_UPDATE_SUBSCRIBED_ITERATOR bIterator,
    void *pvIterateArg
        );

void DATASERVICE_IMPL_vInitializeDSRLConfig (
    DATASERVICE_DSRL_CONFIG_STRUCT *psDSRLConfig
        );

SMS_OBJECT DATASERVICE_IMPL_hConfigureDSRL (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_DSRL_CONFIG_STRUCT *psDSRLConfig
        );

void DATASERVICE_IMPL_vDestroyDSRLParent (
    DATASERVICE_IMPL_HDL hServiceImpl,
    SMS_OBJECT hDSRLParent
        );

SMS_OBJECT DATASERVICE_IMPL_hCreateDSRL (
    DATASERVICE_IMPL_HDL hServiceImpl,
    size_t tDSRLObjectSize,
    const char *pacDSRLName,
    DSRL_TYPE_ENUM eType,
    void **ppvServiceData
        );

void DATASERVICE_IMPL_vReportDSRLDestroyed (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DSRL_OBJECT hDSRL,
    DSRL_TYPE_ENUM eType
        );

BOOLEAN DATASERVICE_IMPL_bDSRLUpdate (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DSRL_ARG_STRUCT *psArg
        );

BOOLEAN DATASERVICE_IMPL_bDeviceRadiusUpdate (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DISTANCE_OBJECT hRadius
        );

SMSAPI_RETURN_CODE_ENUM DATASERVICE_IMPL_eManageDataStream (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
    size_t tNumDMIsToConfigure
        );

BOOLEAN DATASERVICE_IMPL_bFreeDataPayload (
    OSAL_BUFFER_HDL hPayload
        );

#if SMS_LOGGING == 1

void DATASERVICE_IMPL_vLog (
    const char *pcFormat,
    ...
        );
#else
#if __STDC_VERSION__ >= 199901L
#define DATASERVICE_IMPL_vLog(...)
#else
void DATASERVICE_IMPL_vLogNothing(
    const char *pcFormat,
    ...
        );
#define DATASERVICE_IMPL_vLog if(1);else DATASERVICE_IMPL_vLogNothing
#endif
#endif

#endif /* DATASERVICE_MGR_IMPL_H_ */
