/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CM implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "sms_task.h"
#include "string_obj.h"
#include "cm_tag_interface.h"
#include "tag_cm_interface.h"
#include "tag_obj.h"
#include "cm.h"
#include "_cm.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/******************************************************************************
*
*   CM_eInitialize
*
* Initializes the CM, by validating the config file, reading it and creating
* the tree of TAG_OBJECTs
*
* Inputs:
*   pacConfigFile - A valid pointer to a character array which specifies the
*                   name of the configuration file.
*   pacFilePath - 	A valid pointer to a character array which specifies the
*                   path to the configuration file.
*   pacConfigInitializerFile - A valid pointer to a character array which
*                   specifies the name of the initializer / default config
*                   file
*   pacConfigInitializerPath - A valid pointer to a character array which
*                   specifies the name of the initializer / default config
*                   path
*   pbConfigFileReset - A pointer to a boolean which, if non-NULL, will
*                   be populated with a TRUE / FALSE value indicating if the
*                   configuration file was reset.
*
*
* Outputs:
*   SMSAPI_RESULT_CODE_ENUM - The result of the initialization request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM CM_eInitialize(
    const char *pacConfigFile,
    const char *pacConfigPath,
    const char *pacConfigInitializerFile,
    const char *pacConfigInitialzerPath,
    BOOLEAN *pbConfigFileReset
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    CM_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid, bOk = FALSE;

    // validate inputs; note: the initializer path
    // and file *can* be NULL
    if ( (pacConfigFile == NULL) || (pacConfigPath == NULL) )
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // has the object already been created?
    bValid = SMSO_bIsValid((SMS_OBJECT)ghCM);
    if (bValid == TRUE)
    {
        return SMSAPI_RETURN_CODE_ALREADY_INITIALIZED;
    }

    // create it
    psObj = (CM_OBJECT_STRUCT *)
        SMSO_hCreate(
            CM_OBJECT_NAME,
            sizeof(CM_OBJECT_STRUCT),
            SMS_INVALID_OBJECT,
            TRUE
                );

    // successful?
    if (psObj != NULL)
    {
        // initialize
        psObj->pacConfigFileName = NULL;
        psObj->pacSecondaryConfigFileName = NULL;
        psObj->pacConfigInitializerFileName = NULL;

        bOk = bSetFilenames( psObj,
                             pacConfigFile,
                             pacConfigPath,
                             pacConfigInitializerFile,
                             pacConfigInitialzerPath
                           );
        if (bOk == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturn;

            psObj->hTopTag = TAG_INVALID_OBJECT;
            psObj->bPendingChanges = FALSE;

            // Create the removed tag list
            eReturn =
                OSAL.eLinkedListCreate(
                    &psObj->hRemovedTagList,
                    CM_OBJECT_NAME":ChildList",
                    NULL,
                    OSAL_LL_OPTION_LINEAR
                        );
            if(eReturn != OSAL_SUCCESS)
            {
                // Error!
                bOk = FALSE;
            }
        }
    }

    if (bOk == TRUE)
    {
        SMS_TASK_CONFIGURATION_STRUCT sConfig =
            gsCMTaskConfiguration;

        // Install SMS Task for the CM
        psObj->eInitResult = SMSAPI_RETURN_CODE_ERROR;

        // Configure name
        sConfig.pacName = CM_OBJECT_NAME;

        psObj->hServicesTask = SMST_hInstall((SMS_OBJECT)psObj,
            &sConfig, (SMS_OBJECT_EVENT_HANDLER_PROTOTYPE)vEventHandler,
            &psObj->hEventHdlr);
        if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
        {
            // Success!
            puts(CM_OBJECT_NAME":SMS task installed.");

            // If the caller cares about whether or not our
            // config file was reconstituted, let them know.
            if ( NULL != pbConfigFileReset )
            {
                if ( ( CM_CONFIG_SOURCE_NEW_MINIMAL_SMS_CFG
                        == psObj->eConfigSource ) ||
                     ( CM_CONFIG_SOURCE_CONFIG_INITIALIZER
                        == psObj->eConfigSource ) )
                {
                    *pbConfigFileReset = TRUE;

                    // We want to make sure that we write these changes
                    // to disk right away, such that we're not waiting
                    // on other events for our new config file to
                    // be written.
                    eReturn = CM_eCommitChangesToFile();

                    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            CM_OBJECT_NAME": could not commit changes to file!");
                    }
                }
                else
                {
                    *pbConfigFileReset = FALSE;
                }
            }
        }
        else
        {
            // Error!
            vUninitialize();

            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CM_OBJECT_NAME": task could not be installed.");
        }

        eReturn = psObj->eInitResult;
    }
    else
    {
        // Error!
        vUninitialize();
        eReturn = SMSAPI_RETURN_CODE_ERROR;

        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CM_OBJECT_NAME": could not be initialized.");
    }

    return eReturn;
}

/******************************************************************************
*
*   CM_vUninitialize
*
* Uninitializes the CM, destroying all TAG_OBJECTs which belonged to it and
* freeing all resources used by the CM
*
* Inputs:
*  none
*
* Outputs:
*  none
*
******************************************************************************/
void CM_vUninitialize(void)
{
    BOOLEAN bValid, bPosted = FALSE;

    // Verify SMS Object is valid
    bValid = SMSO_bValid((SMS_OBJECT)ghCM);
    if(bValid == TRUE)
    {
        // De-reference object
        CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

        bPosted = SMSE_bPostSignal(psObj->hEventHdlr, SMS_EVENT_STOP,
            SMS_EVENT_OPTION_SYNCHRONOUS);
        if(bPosted == FALSE)
        {
            // Error!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CM_OBJECT_NAME": Unable to stop the CM.");
            return;
        }

        // Uninstall Sirius Module Services
        if(psObj->hServicesTask != SMS_INVALID_TASK_HANDLE)
        {
            SMS_TASK_HANDLE hServicesTask = psObj->hServicesTask;

            // Task shutdown itself will destroy the object (psObj)
            // so there is no need to handle that here. This function
            // is called from the application task context, so it will
            // request the SMS-Task to be deleted (shutdown). Once the
            // task is shutdown the last thing it does is destroy the
            // SMS-Object provided at the time of SMS-task creation.
            // This means you must not use psObj anymore after this call!
            psObj->hServicesTask = SMS_INVALID_TASK_HANDLE;
            SMST_vUninstall(hServicesTask);
        }

        // the object itself will destroyed in the sms task handler
        // but we need to clear out the global handle.
        ghCM = CM_INVALID_OBJECT;
    }

    return;
}

/******************************************************************************
*
*   CM_eCommitChangesToFile
*
* Write the CM's TAGs and their current values to the config file
*
* Inputs:
*  none
*
* Outputs:
*   SMSAPI_RESULT_CODE_ENUM - The result of the commit request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM CM_eCommitChangesToFile( void )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturn;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;
    BOOLEAN bLocked;
    UN32 un32TimeRemaining = 0, un32Timeout = CM_COMMIT_TIMEOUT;

    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    eOsalReturn = OSAL.eTimerRemaining(
        psObj->hTimerObj,
        &un32TimeRemaining
            );
    if (eOsalReturn == OSAL_SUCCESS)
    {
        printf(CM_OBJECT_NAME":Timer has %u remaining time\n", un32TimeRemaining);
        // sum the total amount of time the longest pending commit by adding the amount of time the timer
        // had been running before we reset it.
        psObj->un32TimeCommitHeldOff = psObj->un32TimeCommitHeldOff + (CM_COMMIT_TIMEOUT - un32TimeRemaining);
        if (psObj->un32TimeCommitHeldOff > CM_COMMIT_THRESHOLD)
        {
            printf(CM_OBJECT_NAME":Commit held off for %u ms. (SHOULD) Force a commit.\n", psObj->un32TimeCommitHeldOff);
            un32Timeout = 0;
        }

    }
    else if (eOsalReturn == OSAL_TIMER_NOT_ACTIVE)
    {
        puts(CM_OBJECT_NAME":Timer not active");
    }
    else
    {
        puts(CM_OBJECT_NAME": Error. Remaining time could not be retreived");
    }

    // start timer relative (one shot)
    eOsalReturn = OSAL.eTimerStartRelative(psObj->hTimerObj,
            un32Timeout, 0);

    SMSO_vUnlock((SMS_OBJECT)ghCM);

    if (eOsalReturn == OSAL_SUCCESS)
    {
        eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturn;
}

/******************************************************************************
*
*   CM_bPendingChanges
*
* This function indicates whether or not the CM has any pending changes waiting
* for a commit.
*
* Inputs:
*   none
*
* Outputs:
*   TRUE if there are any pending changes in wait, FALSE if not.
*
******************************************************************************/
BOOLEAN CM_bPendingChanges( void )
{
    BOOLEAN bLocked, bPending = FALSE;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        bPending = psObj->bPendingChanges;

        SMSO_vUnlock((SMS_OBJECT)ghCM);
    }

    return bPending;
}

/******************************************************************************
*
*   CM_ePrintTags
*
* This function will print out the Tags, starting at the CM's TopTag and
* iterating through the child tags
*
* Inputs
*  psFile - A valid pointer to a file to which the TAGs will be written.
*
* Outputs
*  SMSAPI_RESULT_CODE_ENUM - The result of the print request.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM CM_ePrintTags(
    FILE *psFile
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    if (psFile == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        N32 n32NumWritten;

        n32NumWritten = TAGCM_n32WriteTagToFile(psObj->hTopTag, psFile, 0);
        if (n32NumWritten != EOF)
        {
            eReturn = SMSAPI_RETURN_CODE_SUCCESS;
        }

        SMSO_vUnlock((SMS_OBJECT)ghCM);
    }

    return eReturn;
}

/******************************************************************************
*
*   CM_hGetTopTag
*
* This function is used to retrieve the CM's TopTag.
*
* Inputs
*  none
*
* Outputs
*  A handle to the CM TopTag. TAG_INVALID_OBJECT will be returned on error.
*
******************************************************************************/
TAG_OBJECT CM_hGetTopTag( void)
{
    TAG_OBJECT hTopTag = TAG_INVALID_OBJECT;
    BOOLEAN bLocked;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        hTopTag = psObj->hTopTag;

        SMSO_vUnlock((SMS_OBJECT)ghCM);
    }

    return hTopTag;
}

/******************************************************************************
*
*   CM_hTrueString
*
* This function is used to retrieve the CM's True String.
*
* Inputs
*  none
*
* Outputs
*  A handle to the CM True String.
*  STRING_INVALID_OBJECT will be returned on error.
*
******************************************************************************/
STRING_OBJECT CM_hTrueString( void )
{
    BOOLEAN bValid;
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // validate
    bValid =
        SMSO_bValid((SMS_OBJECT)ghCM);
    if(bValid == TRUE)
    {
        hString = psObj->hTrueString;
    }

    return hString;
}

/******************************************************************************
*
*   CM_hFalseString
*
* This function is used to retrieve the CM's False String.
*
* Inputs
*  none
*
* Outputs
*  A handle to the CM False String.
*  STRING_INVALID_OBJECT will be returned on error.
*
******************************************************************************/
STRING_OBJECT CM_hFalseString( void )
{
    BOOLEAN bValid;
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // validate
    bValid =
        SMSO_bValid((SMS_OBJECT)ghCM);
    if(bValid == TRUE)
    {
        hString = psObj->hFalseString;
    }

    return hString;
}

/******************************************************************************
*
*   CM_eStringToBoolean
*
* This function is used to determine if the input string matches the
* TRUE or FALSE string.
*
* Inputs
*  hString - the STRING_OBJECT to evaluate
*  pbConvertedValue - location to where the TRUE or FALSE mapped value is
*                     copied
*
* Outputs
*  SMSAPI_RETURN_CODE_SUCCESS if the inputs supplied are valid and the
*  STRING evaluated to either TRUE or FALSE.
*
******************************************************************************/
SMSAPI_RETURN_CODE_ENUM	CM_eStringToBoolean(
    STRING_OBJECT hString,
    BOOLEAN *pbConvertedValue
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (pbConvertedValue != NULL)
    {
        CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;
        BOOLEAN bValid;

        // validate
        bValid =
            SMSO_bValid((SMS_OBJECT)ghCM);
        if(bValid == TRUE)
        {
            if (STRING.n16Compare(psObj->hTrueString, hString, TRUE) == 0)
            {
                *pbConvertedValue = TRUE;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else if (STRING.n16Compare(psObj->hFalseString, hString, TRUE) == 0)
            {
                *pbConvertedValue = FALSE;
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
            else
            {
                eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
                      CM-TAG INTERFACE FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       CMTAG_pacBuffer
*
*       This function verifies that the shared buffer is large enough.
*       If it isn't, a new, larger buffer is created and the old one destroyed.
*
*       Inputs:
*               tRequiredSize - the size in bytes we need the buffer to be
*               bRetain - if TRUE, the portion of the buffer being used is
*                         copied to the new buffer
*
*       Outputs:
*               The pointer to the Buffer on success, NULL on error
*
*****************************************************************************/
char *CMTAG_pacBuffer(
    size_t tRequiredSize,
    BOOLEAN bRetain
        )
{
    BOOLEAN bOwner;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // verify ownership
    bOwner =
        SMSO_bOwner((SMS_OBJECT)ghCM);
    if(bOwner == FALSE)
    {
        // not the owner
        return NULL;
    }

    // is the existing shared buffer large enough?
    if (psObj->tCMBufferSize < tRequiredSize)
    {
        // no, need to resize

        char *pacNewBuffer;
        size_t tNewSize;

        // we'll request more memory than the minimum required
        // This will hopefully reduce the number of times we need to allocate.
        tNewSize =  tRequiredSize+SHARED_BUFFER_SIZE_EXTRA_INCREMENT;

        pacNewBuffer =
            (char *)SMSO_hCreate(
                        CM_OBJECT_NAME":SB",
                        tNewSize,
                        (SMS_OBJECT)psObj,
                        FALSE );

        if (pacNewBuffer != NULL)
        {
            if (bRetain == TRUE)
            {
                size_t tNumToCopy;
                tNumToCopy =
                    psObj->pacCMBufferPtr - psObj->pacCMBuffer +1;

                // resize, but retain contents
                OSAL.bMemCpy(
                    pacNewBuffer,
                    psObj->pacCMBuffer,
                    tNumToCopy
                        );

                psObj->pacCMBufferPtr =
                    pacNewBuffer + tNumToCopy -1;
            }
            else
            {
                psObj->pacCMBufferPtr = pacNewBuffer;
            }

            // get rid of the old buffer
            SMSO_vDestroy((SMS_OBJECT)psObj->pacCMBuffer);

            // use the newly created buffer
            psObj->pacCMBuffer = pacNewBuffer;
            psObj->tCMBufferSize = tNewSize;
        }
        else
        {
            // we couldn't resize the buffer to the requested size
            return NULL;
        }
    }

    // buffer is large enough
    return psObj->pacCMBuffer;
}

/******************************************************************************
*
*   CMTAG_bPendingChanges
*
* This function is used to set the value of the config managers pending flag
* TRUE means a change is pending (not yet committed to the file)
*
******************************************************************************/
BOOLEAN CMTAG_bSetPendingChanges(
    BOOLEAN bPending
        )
{
    BOOLEAN bOwner, bReturn = FALSE;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // verify
    bOwner =
        SMSO_bOwner((SMS_OBJECT)ghCM);
    if(bOwner == TRUE)
    {
        psObj->bPendingChanges = bPending;

        bReturn = TRUE;
    }

    return bReturn;
}

/******************************************************************************
*
*   CMTAG_bTagRemoved
*
* This function causes a TAG to be placed into the Removed Tag list, but not
* destroyed. This way if some object has the TAG Handle, but doesn't know the
* TAG is removed, the attempt to use the TAG will fail, but not cause a crash.
*
******************************************************************************/
BOOLEAN CMTAG_bTagRemoved(
    TAG_OBJECT hTag,
    BOOLEAN bCommit
        )
{
    BOOLEAN bOwner, bReturn = FALSE;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    // verify
    bOwner =
        SMSO_bOwner((SMS_OBJECT)ghCM);
    if(bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // add the child to the end of the removed tag list
        eOsalReturnCode = OSAL.eLinkedListAdd(
                              psObj->hRemovedTagList,
                              OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                              hTag
                                  );

        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            // did the caller want to commit now?
            if (bCommit == TRUE)
            {
                // yes. commit changes to file now
                SMSAPI_RETURN_CODE_ENUM eReturn;

                eReturn = CM_eCommitChangesToFile();
                if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
                {
                    bReturn = TRUE;
                }
            }
            else
            {
                // no we won't commit now, so we have pending changes.
                psObj->bPendingChanges = TRUE;
                bReturn = TRUE;
            }
        }
    }

    return bReturn;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       bSetFilenames
*
*       Called during initialization, this function creates buffers
*       which store character strings representing names and locations
*       of configuration files (active and secondary).
*
*
*       Inputs:
*               psObj                 object whose file names are being set
*               pacCfgFileNameInput   File name provided to the
*                                     Configuration Manager on startup
*               pacCfgFilePathInput   Path provided to the Configuration
*                                     Manager on startup
*               pacCfgInitializerFileNameInput
*                                     Initializer file name provided
*                                     to the Configuration Manager on startup
*               pacCfgInitialierFilePathInput
*                                     Initializer path provided to the
*                                     Configuration Manager on startup
*
*       Outputs:
*               TRUE if buffers were created successfully, FALSE on error.
*
*       Notes:
*
*****************************************************************************/
static BOOLEAN bSetFilenames(
    CM_OBJECT_STRUCT *psObj,
    const char *pacCfgFileNameInput,
    const char *pacCfgFilePathInput,
    const char *pacCfgInitializerFileNameInput,
    const char *pacCfgInitializerFilePathInput
        )
{
    N16     n16StringPrintResult;
    size_t  tFileNameLength, tPathLength, tTotalLength;

    do
    {
        // The total length is the length of the path plus the
        // length of the name, plus 1 extra byte for our null terminator,
        // and another byte for the '/' that we'll be applying to the
        // end of the name (see below.)
        tFileNameLength = strlen( pacCfgFileNameInput );
        tPathLength = strlen( pacCfgFilePathInput );
        tTotalLength = tFileNameLength + tPathLength + 2;

        // Allocate memory for the file name
        psObj->pacConfigFileName =
            (char *) SMSO_hCreate(
                         CM_OBJECT_NAME":"CM_FILENAME_NAME,
                         tTotalLength,
                         (SMS_OBJECT)psObj,
                         FALSE );

        if (psObj->pacConfigFileName == NULL)
        {
            break;
        }

        // Populate the file name; not the we we insert an extra, unix-style path
        // separator between the path and the name, as a) we'll need one if it's not
        // there, and b) both unix and windows are oddly happy with an extra '/'
        // sandwiched in the name if we already have a path separator.
        n16StringPrintResult = snprintf( psObj->pacConfigFileName,
                                         tTotalLength,
                                         "%s/%s",
                                         pacCfgFilePathInput,
                                         pacCfgFileNameInput
                                       );

        if ((unsigned)n16StringPrintResult != (tTotalLength-1))
        {
            break;
        }

        tTotalLength =  tTotalLength + strlen(SECONDARY_CONFIG_FILE_TOKEN);

        // Allocate memory for the secondary file name
        psObj->pacSecondaryConfigFileName =
            (char *) SMSO_hCreate(
                         CM_OBJECT_NAME":"CM_SECOND_FILENAME_NAME,
                         tTotalLength,
                         (SMS_OBJECT)psObj,
                         FALSE );

        if (psObj->pacSecondaryConfigFileName == NULL)
        {
            break;
        }

        // Populate the secondary file name
        n16StringPrintResult = snprintf( psObj->pacSecondaryConfigFileName,
                                         tTotalLength,
                                         "%s/%s%s",
                                         pacCfgFilePathInput,
                                         pacCfgFileNameInput,
                                         SECONDARY_CONFIG_FILE_TOKEN
                                       );

        if ((unsigned)n16StringPrintResult != (tTotalLength-1))
        {
            break;
        }

        // Unlike the "normal" config path, the initializer path and
        // initialier file are optional. If they're NULL, just set the
        // pacDefaultConfigFileName to NULL as well and return "success".
        if ( ( NULL == pacCfgInitializerFilePathInput ) ||
             ( NULL == pacCfgInitializerFileNameInput )  )
        {
            psObj->pacConfigInitializerFileName = NULL;
            return TRUE;
        }

        // The total length is the length of the path plus the
        // length of the name, plus 1 extra byte for our null terminator,
        // and another byte for the '/' that we'll be applying to the
        // end of the name (see below.)
        tFileNameLength = strlen( pacCfgInitializerFileNameInput );
        tPathLength = strlen( pacCfgInitializerFilePathInput );
        tTotalLength = tFileNameLength + tPathLength + 2;

        // Allocate memory for the file name
        psObj->pacConfigInitializerFileName =
            (char *) SMSO_hCreate(
                         CM_OBJECT_NAME":"CM_INITIALIZER_FILENAME_NAME,
                         tTotalLength,
                         (SMS_OBJECT)psObj,
                         FALSE
                         );

         if (psObj->pacConfigInitializerFileName == NULL)
         {
             break;
         }

         // Populate the file name
         n16StringPrintResult = snprintf( psObj->pacConfigInitializerFileName,
                                          tTotalLength,
                                          "%s/%s",
                                          pacCfgInitializerFilePathInput,
                                          pacCfgInitializerFileNameInput
                                         );

         if ( n16StringPrintResult != (N16)(tTotalLength - 1) )
         {
             break;
         }

        return TRUE;

    } while (0);

    // something went wrong
    vUnsetFilenames(psObj);

    return FALSE;
}

/*****************************************************************************
*
*       vUnsetFilenames
*
*       Called on shutdown to release memory associated with configuration
*       file names and directories.
*
*       Inputs:
*               psObj  object whose file names are being unset
*
*       Outputs:
*               None
*
*       Notes:
*
*****************************************************************************/
static void vUnsetFilenames(
    CM_OBJECT_STRUCT *psObj
        )
{
    if (psObj->pacConfigFileName != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacConfigFileName);
        psObj->pacConfigFileName = NULL;
    }

    if (psObj->pacSecondaryConfigFileName != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacSecondaryConfigFileName);
        psObj->pacSecondaryConfigFileName = NULL;
    }

    if ( psObj->pacConfigInitializerFileName != NULL )
    {
        SMSO_vDestroy( (SMS_OBJECT)psObj->pacConfigInitializerFileName );
        psObj->pacConfigInitializerFileName = NULL;
    }

    return;
}

/*****************************************************************************
*
*   vCreateDefaultTopTag
*
* This function will create top tag in certain cases when one is not found.
*
*****************************************************************************/
static void vCreateDefaultTopTag(
    CM_OBJECT_STRUCT *psObj
        )
{
    psObj->hTopTag = TAGCM_hCreate(
                         (SMS_OBJECT)psObj,
                         CM_DEFAULT_TOP_TAG_NAME,
                         NULL,
                         STRING_INVALID_OBJECT
                             );

    return;
}

/*******************************************************************************
*
*   eReadCfgFile
*
* This function does the read of the config file
*
******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eReadCfgFile(
    const char *pacConfigFileName,
    CM_OBJECT_STRUCT *psObj
        )
{
    FILE *psFile;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bSuccess;

    psFile = fopen(pacConfigFileName, "rb");
    if (psFile != NULL)
    {
        BOOLEAN bOk;
        UN16 un16ChksumComputed = 0;
        UN16 un16ChksumRead = 0;
        size_t tContentLength = 0;
        BOOLEAN bChecksumExists = FALSE;

        // Ensure the configuration file is valid

        bOk = bReadChecksum(
                psFile, psObj->pacCMBuffer, psObj->tCMBufferSize,
                &tContentLength, &un16ChksumRead, &bChecksumExists);

        if ((TRUE == bOk) && (TRUE == bChecksumExists))
        {
            bOk = bComputeChecksum(
                    psFile, psObj->pacCMBuffer, psObj->tCMBufferSize,
                    tContentLength, &un16ChksumComputed);
        }

        // If checksum tag is blank (bNonBlankChecksum == FALSE), then
        // un16ChksumRead remains unchanged (0 as initialized) and 
        // bComputeChecksum() call is skipped. In this case un16ChksumComputed
        // is also untouched so it is also 0 and equal to un16ChksumRead
        if ((FALSE == bOk) || (un16ChksumComputed != un16ChksumRead))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CM_OBJECT_NAME": Checksum error detected.");

            eReturn = SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM;
        }
        else
        {
            N32 n32Return;

            // rewind to the beginning of the file
            n32Return = fseek(psFile, 0, SEEK_SET);

            if (0 == n32Return)
            {
                psObj->eReadState = CM_STATE_SEARCHING_FOR_BEGIN_DELIMITER;

                do
                {
                    size_t tNumRead;

                    tNumRead = fread(psObj->pacCMReadBuffer,
                                sizeof(char), READ_BUFF_SIZE - 1, psFile);

                    if (tNumRead == 0)
                    {
                        break;
                    }

                    // Null terminate input...guaranteed!
                    psObj->pacCMReadBuffer[tNumRead] = '\0';

                    // process a chunk
                    eReturn = eProcessChunk(psObj, tNumRead);

                } while (eReturn == SMSAPI_RETURN_CODE_SUCCESS);
            }
        }

        fclose(psFile);
    }
    else
    {
        printf(CM_OBJECT_NAME": Unable to open configuration file %s.\n",
                psObj->pacConfigFileName);
    }

    // we're done reading the file; null out the buffer in case we have to
    // use it again.
    bSuccess = OSAL.bMemSet(psObj->pacCMReadBuffer, (UN8)0, READ_BUFF_SIZE);

    if ( TRUE != bSuccess )
    {
        eReturn = SMSAPI_RETURN_CODE_ERROR;
    }

    return eReturn;
}

/*****************************************************************************
*
*    eProcessChunk
*
* This function will take a chunk of the config file and process it by looking
* for tags, tag values and so on
*
* Inputs:
*
* psObj    the config object whose line is being processed
* tChunkSize  the size of the chunk to process
*
* Outputs:
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessChunk(
    CM_OBJECT_STRUCT *psObj,
    size_t tChunkSize
        )
{
    static BOOLEAN bClose = FALSE;
    static TAG_OBJECT hCurrentTag = TAG_INVALID_OBJECT;
    CM_PROCESS_STRUCT sProcess;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;

    sProcess.pacCurrentLocation = psObj->pacCMReadBuffer;
    sProcess.pacReadBufferEnd = psObj->pacCMReadBuffer+tChunkSize-1;
    sProcess.pacStart = NULL;
    sProcess.pacEnd = NULL;

    while( (sProcess.pacCurrentLocation <= sProcess.pacReadBufferEnd) &&
           ( eReturn == SMSAPI_RETURN_CODE_SUCCESS) )
    {
        switch(psObj->eReadState)
        {
            case CM_STATE_SEARCHING_FOR_BEGIN_DELIMITER:
            {
                eReturn = eProcessStateSearchingForBeginDelimiter(
                              psObj,
                              &sProcess);
            }
            break;

            case CM_STATE_SEARCHING_FOR_CLOSE_DELIMITER:
            {
                eReturn = eProcessStateSearchingForCloseDelimiter(
                              psObj,
                              &sProcess,
                              hCurrentTag,
                              &bClose);
            }
            break;

            case CM_STATE_SEARCHING_FOR_END_DELIMITER:
            {
                eReturn = eProcessStateSearchingForEndDelimiter(
                              psObj,
                              &sProcess,
                              &hCurrentTag,
                              bClose);
            }
            break;

            case CM_STATE_READ_COMPLETE:
            {
                // we're done
                return eReturn;
            }
        }
    }

    return eReturn;
}

/******************************************************************************
*
*   bTrimWhitespace
*
* This function is used to trim whitespace from the beginning and end. It will
* write a NULL at the end, after the last non-whitespace character.
*
******************************************************************************/
static BOOLEAN bTrimWhitespace(
    char **ppacStart,
    char **ppacEnd
        )
{
    if (*ppacStart == NULL)
    {
        return FALSE;
    }

    // skip leading whitespace
    while( (WHITESPACE(**ppacStart) == TRUE) &&
           ( *ppacStart < *ppacEnd ) )
    {
        *ppacStart = *ppacStart+1;
    }

    // skip trailing whitespace
    while( (WHITESPACE(**ppacEnd) == TRUE) &&
           ( *ppacStart < *ppacEnd ) )
    {
        *ppacEnd = *ppacEnd-1;
    }

    if (WHITESPACE(**ppacStart) == TRUE)
    {
        // there are no non-whitespace characters.
        return FALSE;
    }

    // NULL terminate
    *(*ppacEnd+1) = '\0';
    // there is non-whitespace characters present
    return TRUE;
}

/******************************************************************************
*
*   bReadChecksum
*
* Reads checksum from the given configuration file.
* Provides configuration file content location.
*
* psFile - the configuration file, must be opened for reading
* pacBuffer - the buffer which can be used as a read buffer
* tBufferSize - the size of buffer in bytes
* ptContentLength - the location where the length of
*    configuration file content will be placed
* pun16Checksum - the location where the read checksum will be placed
*
* Returns TRUE if the checksum is found and read, FALSE otherwise.
*
******************************************************************************/
static BOOLEAN bReadChecksum(
    FILE *psFile,
    char *pacBuffer,
    size_t tBufferSize,
    size_t *ptContentLength,
    UN16 *pun16Checksum,
    BOOLEAN *pbChecksumExists
        )
{
    size_t tFileSize;
    ptrdiff_t tCheckSumOffset;
    size_t tNumCharsRead;
    N32 n32Return;
    char *pacChecksumStart;
    char *pacChecksumEnd;
    BOOLEAN bChecksumExists;
    BOOLEAN bOk;

    if ((NULL == psFile) || (NULL == pacBuffer) || (0 == tBufferSize))
    {
        // bad arguments
        return FALSE;
    }

    // make sure we have valid pbChecksumExists pointer
    if (NULL == pbChecksumExists)
    {
        pbChecksumExists = &bChecksumExists;
    }

    // get file size (assume the given file is not updated)
    bOk = OSAL.bFileSystemGetFileSize(psFile, &tFileSize);
    if (FALSE == bOk)
    {
        return FALSE;
    }

    // we need to seek from the back of the file.
    // if the file is smaller than our buffer, just seek to the beginning.
    // otherwise seek from the back of the file an offset equal to the
    // size of our buffer
    if (tFileSize < tBufferSize)
    {
        n32Return = fseek(psFile, 0, SEEK_SET);
    }
    else
    {
        // seek to the beginning of the chunk we are going to read
        n32Return = fseek(psFile, -(N32)tBufferSize, SEEK_END);
    }

    if (n32Return != 0)
    {
        // seek failed
        return FALSE;
    }

    // get the file position for the beginning of this chunk
    // we'll use it later to compute the file position of the checksum tag
    tCheckSumOffset = ftell(psFile);

    // read from the file
    tNumCharsRead = fread(pacBuffer, sizeof(char), tBufferSize, psFile);

    if (0 == tNumCharsRead)
    {
        // read failure
        return FALSE;
    }

    // is the checksum tag present?
    pacChecksumStart = strstr(pacBuffer, CHECKSUM_TAG_OPEN);
    if (NULL == pacChecksumStart)
    {
        // checksum tag not found
        return FALSE;
    }

    // we found the checksum tag, so now figure out the corresponding
    // file position.
    tCheckSumOffset += (size_t)(pacChecksumStart - pacBuffer);

    // now let's read the checksum

    // get the position of the start of the tag close
    pacChecksumEnd = strstr(pacChecksumStart, CHECKSUM_TAG_CLOSE);
    if (NULL == pacChecksumEnd)
    {
        // checksum tag broken
        return FALSE;
    }

    // the beginning of the checksum is just after the tag opening
    pacChecksumStart += sizeof(CHECKSUM_TAG_OPEN) - 1;
    // the end of the checksum is just before the tag close
    pacChecksumEnd--;

    // see if our "end" ptr is before our "begin" ptr
    // this happens when the open and close tag butt up against
    // each other with no blank space
    if (pacChecksumEnd < pacChecksumStart)
    {
        // empty checksum
        return FALSE;
    }

    *pbChecksumExists = bTrimWhitespace(&pacChecksumStart, &pacChecksumEnd);
    
    // blank checksum is a valid case, meaning the checksum shall
    // be calculated anew. But if it is not blank, then we
    // need to read and return it
    if (TRUE == *pbChecksumExists)
    {
        // read the checksum
        if (NULL != pun16Checksum)
        {
            *pun16Checksum = (UN16)atoi(pacChecksumStart);
        }

        // content parameters
        if (NULL != ptContentLength)
        {
            *ptContentLength = (size_t)tCheckSumOffset;
        }
    }

    return TRUE;
}

/******************************************************************************
*
*   bComputeChecksum
*
* Computes the checksum of (a part of) a file
*
* psFile - the configuration file, must be opened for reading
* pacBuffer - the buffer which can be used as a read buffer
* tBufferSize - the size of buffer in bytes
* tContentLength - the length of configuration file content
* pun16Checksum - the location where the computed checksum will be placed
*
* Returns TRUE if the checksum is computed well, FALSE otherwise.
*
******************************************************************************/
static BOOLEAN bComputeChecksum(
    FILE *psFile,
    char *pacBuffer,
    size_t tBufferSize,
    size_t tContentLength,
    UN16 *pun16Checksum
        )
{
    UN16 un16ComputedChecksum = 0;
    N32 n32Return;

    if ((NULL == psFile) || (NULL == pacBuffer) || (0 == tBufferSize))
    {
        // bad arguments
        return FALSE;
    }

    n32Return = fseek(psFile, 0, SEEK_SET);
    if (0 != n32Return)
    {
        // seek failed
        return FALSE;
    }

    while (tContentLength > 0)
    {
        size_t tBlockLength;
        size_t tNumCharsRead;
        char *pacChar;
        char *pacEnd;

        tBlockLength = (tBufferSize < tContentLength)
            ? tBufferSize : tContentLength;

        tNumCharsRead = fread(pacBuffer, sizeof(char), tBlockLength, psFile);
        if (tNumCharsRead != tBlockLength)
        {
            // error reading from file
            return FALSE;
        }

        pacEnd = pacBuffer + tNumCharsRead;
        for (pacChar = pacBuffer; pacChar != pacEnd; pacChar++)
        {
            // Only use non-whitespace characters to compute checksum
            if (!WHITESPACE(*pacChar))
            {
                un16ComputedChecksum += *pacChar;
            }
        }

        tContentLength -= tNumCharsRead;
    }

    // Invert the bits
    un16ComputedChecksum = ~(un16ComputedChecksum);

    if (NULL != pun16Checksum)
    {
        *pun16Checksum = un16ComputedChecksum;
    }

    return TRUE;
}

/******************************************************************************
*
*   bStampChecksum
*
* This function writes the checksum to the file. The given file must be opened
* for writing and must contain the only configuration contents. The checksum is
* appended to the end of file.
*
* psFile - the configuration file, must be opened for writing
* un16Checksum - the checksum to stamp
*
******************************************************************************/
static BOOLEAN bStampChecksum(
    FILE *psFile,
    UN16 un16Checksum
        )
{
    N32 n32Return;
    size_t tWritten;
    char acBuf[CM_CHECKSUM_BUF_LENGTH];

    if (NULL == psFile)
    {
        // bad arguments
        return FALSE;
    }

    n32Return = fseek(psFile, 0, SEEK_END);
    if (0 != n32Return)
    {
        // seek failure
        return FALSE;
    }

    // write the checksum
    n32Return = snprintf(acBuf, sizeof(acBuf), "%u", un16Checksum);
    if ((n32Return <= 0) || (n32Return >= sizeof(acBuf)))
    {
        // this happens when the stack buffer is too short
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CM_OBJECT_NAME": error writing checksum");
        return FALSE;
    }

    WRITE_LITERAL(psFile, "\n" CHECKSUM_TAG_OPEN " ");
    tWritten = fwrite(acBuf, sizeof(char), (size_t)n32Return, psFile);
    WRITE_LITERAL(psFile, " " CHECKSUM_TAG_CLOSE "\n");

    return ((size_t)n32Return == tWritten) ? TRUE : FALSE;
}

/******************************************************************************
*
*   bAppendBuffer
*
* This function is used to add characters to the CMBuffer. It will add them
* after the current working point. If necessary the buffer will be resized if
* appending more characters than it can currently hold. If resized, the
* previous characters will be copied to the new buffer
*
******************************************************************************/
static BOOLEAN bAppendBuffer(
    CM_OBJECT_STRUCT *psObj,
    char *pacStart,
    char *pacEnd
        )
{
    size_t tNumToCopy, tNumInBuffer;
    BOOLEAN bReturn = FALSE;

    if (pacEnd >= pacStart)
    {
        char *pacBuffer;

        tNumToCopy = pacEnd-pacStart+1;
        tNumInBuffer =
            psObj->pacCMBufferPtr - psObj->pacCMBuffer +1;

        // verify the buffer size, if necessary resize it and copy the contents
        pacBuffer = CMTAG_pacBuffer(tNumInBuffer+tNumToCopy, TRUE);

        if (pacBuffer != NULL)
        {
            // copy in the new chars
            bReturn = OSAL.bMemCpy(
                psObj->pacCMBufferPtr,
                pacStart,
                tNumToCopy
                    );
            // update the pointer to the end of the text
            psObj->pacCMBufferPtr =
                psObj->pacCMBufferPtr+tNumToCopy;
            // NULL terminate
            *psObj->pacCMBufferPtr = '\0';
        }
    }
    return bReturn;
}

/******************************************************************************
*
*   vResetBuffer
*
* This NULL terminates the beginning of the CMBuffer and resets the CMBufferPtr
* to point to the beginning
*
******************************************************************************/
static void vResetBuffer( void )
{
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;

    if (psObj->pacCMBuffer != NULL)
    {
        // reset the buffer ptr back to the beginning of the buffer
        psObj->pacCMBufferPtr = psObj->pacCMBuffer;
        // NULL terminate.
        *psObj->pacCMBufferPtr = '\0';
    }

    return;
}

/*****************************************************************************
*
*   eProcessStateSearchingForBeginDelimiter
*
* This function handles CM_STATE_SEARCHING_FOR_BEGIN_DELIMITER
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForBeginDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess
      )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;

    // Scan for beginning delimiter, when buffer was read
    // we guaranteed it is NULL terminated already.
    psProcess->pacStart = strpbrk(
        psProcess->pacCurrentLocation, TAG_BEGIN_DELIMITER );
    do{
        if (psProcess->pacStart == NULL)
        {
            // didn't find TAG_BEGIN_DELIMITER
            // add characters to the value buffer
            BOOLEAN bOk;

            bOk = bAppendBuffer(
                      psObj,
                      psProcess->pacCurrentLocation,
                      psProcess->pacReadBufferEnd);
            if (bOk == FALSE)
            {
                // ERROR
                eReturn = SMSAPI_RETURN_CODE_ERROR;
                break;
            }

            psProcess->pacCurrentLocation = psProcess->pacReadBufferEnd+1;
        }
        else
        {
            // found the TAG_BEGIN_DELIMITER
            BOOLEAN bOk;
            if (psProcess->pacStart > psProcess->pacCurrentLocation)
            {
                // add characters to our buffer
                bOk = bAppendBuffer(
                          psObj,
                          psProcess->pacCurrentLocation,
                          psProcess->pacStart-1);
                if (bOk == FALSE)
                {
                    // ERROR
                    eReturn = SMSAPI_RETURN_CODE_ERROR;
                    break;
                }
            }

            // now we need to look for CM_STATE_SEARCHING_FOR_CLOSE_DELIMITER
            psProcess->pacCurrentLocation = psProcess->pacStart+1;
            psObj->eReadState = CM_STATE_SEARCHING_FOR_CLOSE_DELIMITER;
        }
    } while(0);

    return eReturn;
}

/*****************************************************************************
*
*   eProcessStateSearchingForCloseDelimiter
*
* This function handles CM_STATE_SEARCHING_FOR_CLOSE_DELIMITER
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForCloseDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT hCurrentTag,
    BOOLEAN *pbClose
      )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;

    // look at characters for TAG_DEFINITION_CLOSE_DELIMITER
    while(psProcess->pacCurrentLocation <= psProcess->pacReadBufferEnd)
    {
        if (*(psProcess->pacCurrentLocation) == TAG_DEFINITION_CLOSE_DELIMITER)
        {
            // found
            BOOLEAN bTextPresent;
            char *pacStartValue, *pacEndValue;
            pacStartValue = psObj->pacCMBuffer;
            pacEndValue = psObj->pacCMBufferPtr;
            // add characters to value buffer up until the
            // TAG_BEGIN_DELIMITER
            bTextPresent =
                bTrimWhitespace(&pacStartValue, &pacEndValue);
            if (bTextPresent == TRUE)
            {
                // non-whitesapce characters are present

                // put them into the value string
                STRING_bDecodeAndModifyCStr(
                    psObj->hValueString,
                    pacStartValue
                        );

                // set the tag value
                eReturn = TAG_eSetTagValue(
                    hCurrentTag,
                    TAG_TYPE_STRING,
                    &(psObj->hValueString),
                    sizeof(STRING_OBJECT),
                    FALSE
                        );

                if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                {
                    break;
                }
            }

            // this is a close tag
            *pbClose = TRUE;
            // now we need to look for the end delimiter
            psObj->eReadState = CM_STATE_SEARCHING_FOR_END_DELIMITER;
            psProcess->pacCurrentLocation++;

            // reset our buffer so characters will be added at the
            // beginning
            vResetBuffer();
            break;
        }
        else
        {
            // the close delimter wasn't found
            if (!WHITESPACE(*(psProcess->pacCurrentLocation)))
            {
                // non-whitespace found

                // this isn't a close tag
                *pbClose = FALSE;
                // now we need to look for the end delimiter
                psObj->eReadState = CM_STATE_SEARCHING_FOR_END_DELIMITER;
                break;
            }
            psProcess->pacCurrentLocation++;
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eProcessStateSearchingForEndDelimiter
*
* This function handles CM_STATE_SEARCHING_FOR_END_DELIMITER
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessStateSearchingForEndDelimiter(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT *phCurrentTag,
    BOOLEAN bClose
      )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                         // really static??

    // Scan for TAG_END_DELIMITER, when buffer was read
    // we guaranteed it is NULL terminated already.
    psProcess->pacEnd = strpbrk(
        psProcess->pacCurrentLocation, TAG_END_DELIMITER );

    do
    {
        if (psProcess->pacEnd == NULL)
        {
            eReturn = eTagEndDelimiterNotFound(psObj, psProcess);
        }
        else
        {
            eReturn = eTagEndDelimiterFound(
                          psObj,
                          psProcess,
                          phCurrentTag,
                          bClose
                              );
        }
    } while(0);

    return eReturn;
}

/*****************************************************************************
*
*   eTagEndDelimiterNotFound
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTagEndDelimiterNotFound(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess
        )
{
    // didn't find TAG_END_DELIMITER
    // add characters to the buffer (for tag name)
    BOOLEAN bOk;
    bOk = bAppendBuffer(
              psObj,
              psProcess->pacCurrentLocation,
              psProcess->pacReadBufferEnd);
    if (bOk == FALSE)
    {
        // ERROR
        return SMSAPI_RETURN_CODE_ERROR;
    }
    psProcess->pacCurrentLocation = psProcess->pacReadBufferEnd+1;

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eTagEndDelimiterFound
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTagEndDelimiterFound(
    CM_OBJECT_STRUCT *psObj,
    CM_PROCESS_STRUCT *psProcess,
    TAG_OBJECT *phCurrentTag,
    BOOLEAN bClose

        )
{
    // found TAG_END_DELIMITER
    BOOLEAN bTextPresent;
    char *pacTagNameStart, *pacTagNameEnd;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    static BOOLEAN bDoneWithThisTag = FALSE;
    static TAG_OBJECT hParentTag = TAG_INVALID_OBJECT;

    do
    {
        // add characters to name up until the TAG_END_DELIMITER
        if (psProcess->pacEnd > psProcess->pacCurrentLocation)
        {
            BOOLEAN bOk;
            bOk = bAppendBuffer(
                      psObj,
                      psProcess->pacCurrentLocation,
                      psProcess->pacEnd-1);
            if (bOk == FALSE)
            {
                // ERROR
                eReturn = SMSAPI_RETURN_CODE_ERROR;
                break;
            }
        }
        pacTagNameStart = psObj->pacCMBuffer;
        pacTagNameEnd = psObj->pacCMBufferPtr;

        // add characters to value buffer up until the
        // TAG_START_DELIMITER
        bTextPresent =
            bTrimWhitespace(&pacTagNameStart, &pacTagNameEnd);
        if (bTextPresent == TRUE)
        {
            // non-whitespace text is present
            if (bClose == FALSE)
            {
                // this isn't a close tag.
                char *pacTagInstNameStart = NULL,
                     *pacTagInstNameEnd = NULL;

                // check to see if there is an instance name
                pacTagInstNameStart =
                    strstr(pacTagNameStart, TAG_INSTANCE_DELIMITER);
                if (pacTagInstNameStart != NULL)
                {
                    // there is an instance name in this tag
                    pacTagNameEnd = pacTagInstNameStart-1;

                    // starting with pacInstname, scan for quote.
                    // Note: we have already guaranteed the string
                    // we scan is NULL terminated.
                    pacTagInstNameStart =
                        strpbrk(pacTagInstNameStart, TAG_QUOTE);
                    if (pacTagInstNameStart != NULL)
                    {
                        pacTagInstNameEnd =
                            strpbrk(pacTagInstNameStart+1, TAG_QUOTE);
                    }

                    // verify that an opening quote and close
                    // quote are found
                    if ((pacTagInstNameStart == NULL) ||
                        (pacTagInstNameEnd == NULL))
                    {
                        // ERROR
                        eReturn =
                            SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE;
                        break;
                    }

                    // advance past the opening quote
                    pacTagInstNameStart = pacTagInstNameStart+1;

                    // advance before the close quote
                    pacTagInstNameEnd = pacTagInstNameEnd-1;

                    bTextPresent =
                        bTrimWhitespace(
                            &pacTagInstNameStart,
                            &pacTagInstNameEnd);
                    if (bTextPresent == FALSE)
                    {
                        // ERROR
                        eReturn =
                            SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE;
                        break;
                    }

                    bTrimWhitespace(
                        &pacTagNameStart,
                        &pacTagNameEnd);
                }

                if (psObj->hTopTag == TAG_INVALID_OBJECT)
                {
                    // if there isn't a top tag, we need to
                    // create it

                    // create the top tag
                    // the CM object is the parent, so all tags are
                    // child objects of the CM
                    psObj->hTopTag = TAGCM_hCreate(
                        (SMS_OBJECT)psObj,
                        pacTagNameStart,
                        pacTagInstNameStart,
                        STRING_INVALID_OBJECT
                            );


                    if (psObj->hTopTag == TAG_INVALID_OBJECT)
                    {
                        // something went wrong.
                        // get rid of STRINGs we might have created
                        eReturn = SMSAPI_RETURN_CODE_ERROR;
                        break;
                    }

                    // everything went ok

                    // the parent tag is the top tag
                    hParentTag = psObj->hTopTag;
                    // the current tag is the top tag
                    *phCurrentTag = hParentTag;
                }
                else
                {
                    if ( bDoneWithThisTag == FALSE)
                    {
                        // this means the current tag is a group
                        // tag, so it becomes the parent tag
                        hParentTag = *phCurrentTag;
                    }

                    if (strcmp(pacTagNameStart, CHECKSUM_TAG_NAME) == 0)
                    {
                        // we reached the checksum. We don't want to
                        // create a tag object for this.
                        // the checksum has already been validated,
                        // and it has been checked that there is no
                        // other text. so we're done.
                        psObj->eReadState = CM_STATE_READ_COMPLETE;
                        break;
                    }

                    // add a new tag under our parent tag
                    eReturn = TAG_eAdd(
                        pacTagNameStart,
                        hParentTag,
                        phCurrentTag,
                        pacTagInstNameStart
                            );
                    if ( eReturn != SMSAPI_RETURN_CODE_SUCCESS)
                    {
                        break;
                    }
                }
                // we aren't done with this tag because we need to
                // see if it has a value
                bDoneWithThisTag = FALSE;
            }
            else
            {
                // deal with close...
                STRING_OBJECT hString;
                hString = TAG_hTagName(hParentTag);

                if (STRING.n16CompareCStr(pacTagNameStart, 0, hString)==0)
                {
                    // this is the close of a group tag since
                    // there is a child present
                    hParentTag = TAGCM_hGetParentTag(hParentTag);
                }
                else
                {
                    // double check value of text (make sure the
                    // close tag name matches the open tag name)
                    hString = TAG_hTagName(*phCurrentTag);
                    if (STRING.n16CompareCStr(pacTagNameStart, 0, hString)==0)
                    {
                        // we found the close of this tag,
                        // so we're done with it
                        bDoneWithThisTag = TRUE;
                    }
                    else
                    {
                        // ERROR
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            CM_OBJECT_NAME":%s does not match "
                            "the opening tag\n", pacTagNameStart);

                        eReturn =
                            SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE;
                        break;
                    }
                }
            }
        }

        // we need to search for the begin DELIMITER
        psProcess->pacCurrentLocation = psProcess->pacEnd+1;
        psObj->eReadState = CM_STATE_SEARCHING_FOR_BEGIN_DELIMITER;
        // reset our buffer so characters will be added at the
        // beginning
        vResetBuffer();

    } while(0);

    return eReturn;
}

/******************************************************************************
*
*   eCheckConfigFile
*
******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCheckConfigFile (
    const char *pacFileName,
    CM_OBJECT_STRUCT *psObj,
    CM_CONFIG_SOURCE_ENUM eConfigSource
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOk;
    UN8 un8Attrs;

    bOk = OSAL.bFileSystemGetFileAttributes(
        pacFileName, &un8Attrs );

    if ( TRUE == bOk )
    {
        // The initializer exists, eh? Try reading it ...
        eReturn = eReadCfgFile( pacFileName, psObj );

        if ( SMSAPI_RETURN_CODE_SUCCESS == eReturn )
        {
            // See if we got a top tag out of this ...
            if ( TAG_INVALID_OBJECT != psObj->hTopTag )
            {
                psObj->eConfigSource = eConfigSource;
            }
        }
    }

    return eReturn;
}

/******************************************************************************
*
*   vInitializeObject
*
******************************************************************************/
static void vInitializeObject(CM_OBJECT_STRUCT *psObj)
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bOk;

    bOk =
        SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bOk == TRUE)
    {
        BOOLEAN bCfgExists = FALSE, bInitExists = FALSE;

        // store into the global handle
        ghCM = (CM_OBJECT)psObj;

        do
        {
            // set the default config source to not applicable
            // (in case of an error)
            psObj->eConfigSource = CM_CONFIG_SOURCE_NOT_APPLICABLE;

            // allocate memory for the read buffer
            psObj->pacCMReadBuffer =
                (char *) SMSO_hCreate(
                             CM_OBJECT_NAME":RB",
                             READ_BUFF_SIZE,
                             (SMS_OBJECT)psObj,
                             FALSE );

            if (psObj->pacCMReadBuffer == NULL)
            {
                // Error
                break;
            }

            // allocate memory for the shared buffer
            psObj->pacCMBuffer =
                (char *) SMSO_hCreate(
                             CM_OBJECT_NAME":SB",
                             INITIAL_SHARED_BUFF_SIZE,
                             (SMS_OBJECT)psObj,
                             FALSE );


            if (psObj->pacCMBuffer == NULL)
            {
                // Error
                break;
            }

            psObj->tCMBufferSize = INITIAL_SHARED_BUFF_SIZE;
            psObj->pacCMBufferPtr = psObj->pacCMBuffer;

            // while getting and setting values a STRING will be used
            // It'll initially be created with some generic text since
            // otherwise no STRING will be created
            psObj->hValueString = STRING_hCreate(
                                       (SMS_OBJECT)ghCM,
                                       "Value String",
                                       strlen("Value String"),
                                       BUFF_SIZE
                                           );
            if (psObj->hValueString == STRING_INVALID_OBJECT)
            {
                // Error
                break;
            }

            psObj->hTrueString = STRING_hCreateConst(CM_TRUE_TEXT, strlen(CM_TRUE_TEXT));

            if (psObj->hTrueString == STRING_INVALID_OBJECT)
            {
                // Error
                break;
            }

            psObj->hFalseString = STRING_hCreateConst(CM_FALSE_TEXT, strlen(CM_FALSE_TEXT));

            if (psObj->hFalseString == STRING_INVALID_OBJECT)
            {
                // Error
                break;
            }

            // See if the config file exists.
            // If it doesn't, try temporary "sms.cfg.new" file first, which can
            // be there in case the system was aborted while renaming temp file
            // into config.
            // If temp file is not present, try to repopulate config using
            // default. Note that if the app developer can opt not to provide a
            // "defaults" path, in which case our default config file name will
            // just be NULL.
            eReturn = eCheckConfigFile(
                psObj->pacConfigFileName,
                psObj,
                CM_CONFIG_SOURCE_EXISTING_SMS_CFG );

            if (SMSAPI_RETURN_CODE_SUCCESS == eReturn)
            {
                bCfgExists = TRUE;
            }

            // Try "temporary" file
            if ( FALSE == bCfgExists )
            {
                eReturn = eCheckConfigFile(
                    psObj->pacSecondaryConfigFileName,
                    psObj,
                    CM_CONFIG_SOURCE_EXISTING_SMS_CFG );

                if (SMSAPI_RETURN_CODE_SUCCESS == eReturn)
                {
                    bCfgExists = TRUE;
                }
            }

            // Only try to load the backup file if a) we need to, and b) we have
            // a backup filename.
            if (( FALSE == bCfgExists ) &&
                ( NULL != psObj->pacConfigInitializerFileName ))
            {
                eReturn = eCheckConfigFile(
                    psObj->pacConfigInitializerFileName,
                    psObj,
                    CM_CONFIG_SOURCE_CONFIG_INITIALIZER );

                if (SMSAPI_RETURN_CODE_SUCCESS == eReturn)
                {
                    bInitExists = TRUE;
                }
            }

            if ( TAG_INVALID_OBJECT == psObj->hTopTag )
            {
                // We didn't get a top tag. In debug mode, we'll only create
                // a default top tag if there were no files on the filesystem,
                // as we want to alert the developer / user that his config
                // is corrupt.
                if ( ( (FALSE == bCfgExists) && (FALSE == bInitExists) ) ||
                        (1 != SMS_DEBUG) )
                {
                    // the file didn't exist, or we are in release mode
                    // so create a default top tag
                    vCreateDefaultTopTag(psObj);

                    // this also means we're starting from scratch, so modify
                    // the config source
                    psObj->eConfigSource = CM_CONFIG_SOURCE_NEW_MINIMAL_SMS_CFG;

                    if ( psObj->hTopTag == TAG_INVALID_OBJECT )
                    {
                        // Error
                        break;
                    }

                    eReturn = SMSAPI_RETURN_CODE_SUCCESS;
                }
            }

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
            else
            {
                OSAL_RETURN_CODE_ENUM eOsalReturn;

                // Initialize
                psObj->un32TimeCommitHeldOff = 0;

                // PRE-ALLOCATE EVENT (SMS_EVENT_TIMEOUT)
                if(psObj->hTimeOutEvent == SMS_INVALID_EVENT_HDL)
                {
                    psObj->hTimeOutEvent =
                        SMSE_hAllocateEvent( psObj->hEventHdlr,
                            SMS_EVENT_TIMEOUT, NULL,
                            SMS_EVENT_OPTION_STATIC
                                );
                    if(psObj->hTimeOutEvent == SMS_INVALID_EVENT_HDL)
                    {
                        // Error!
                        break;
                    }
                }

                // create timer
                eOsalReturn = OSAL.eTimerCreate (
                    &psObj->hTimerObj,
                    CM_OBJECT_NAME":Timer",
                    vTimerHandler,
                    psObj->hTimeOutEvent);
                if (eOsalReturn != OSAL_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        CM_OBJECT_NAME": Unable to create Timer .");
                    break;
                }
            }

            psObj->eInitResult = eReturn;

            // unlock the object
            SMSO_vUnlock((SMS_OBJECT)psObj);

            return;

        } while (0);
    }

    psObj->eInitResult = SMSAPI_RETURN_CODE_ERROR;

    return;
}

/******************************************************************************
*
*   vUninitialize
*
******************************************************************************/
static void vUninitialize(void)
{
    BOOLEAN bLocked;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;
    OSAL_RETURN_CODE_ENUM eOsalReturn;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return;
    }

    if (psObj->hTimerObj != OSAL_INVALID_OBJECT_HDL)
    {
         // see if the timer is set. if it is, then that means there is a pending
         // commit.
         eOsalReturn = OSAL.eTimerRemaining(
            psObj->hTimerObj,
            NULL
                );
         printf(CM_OBJECT_NAME":Timer is %sactive\n", eOsalReturn != OSAL_TIMER_NOT_ACTIVE ? "not " : "");
         if (eOsalReturn != OSAL_TIMER_NOT_ACTIVE)
         {
             OSAL.eTimerStop(psObj->hTimerObj);

             // there is a commit pending (or we can't tell), so commit now befere shutting down
             puts(CM_OBJECT_NAME":Committing pending commit before shutdown");
             eCommitChangesToFile();
         }

         // no longer need this
         OSAL.eTimerDelete(psObj->hTimerObj);
         psObj->hTimerObj = OSAL_INVALID_OBJECT_HDL;
    }

    // free memory for the shared buffer
    if (psObj->pacCMBuffer != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacCMBuffer);
        psObj->pacCMBuffer = NULL;
    }
    psObj->tCMBufferSize = 0;
    psObj->pacCMBufferPtr = NULL;

    // free memory for the read buffer
    if (psObj->pacCMReadBuffer != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psObj->pacCMReadBuffer);
        psObj->pacCMReadBuffer = NULL;
    }

    if (psObj->hRemovedTagList != OSAL_INVALID_OBJECT_HDL)
    {
        // Remove all entries from the list and destroy each content entry
        OSAL.eLinkedListRemoveAll(
            psObj->hRemovedTagList,
            (OSAL_LL_RELEASE_HANDLER)TAGCM_vDestroy
                );

        // Destroy the list itself
        OSAL.eLinkedListDelete(psObj->hRemovedTagList);
        psObj->hRemovedTagList = OSAL_INVALID_OBJECT_HDL;
    }

    if (psObj->hTopTag != TAG_INVALID_OBJECT)
    {
        // destroy the top tag.
        // this will also destroy all of its children, and their children etc.
        TAGCM_vDestroy(psObj->hTopTag);
        psObj->hTopTag = TAG_INVALID_OBJECT;
    }
    if (psObj->hValueString != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hValueString);
        psObj->hValueString = STRING_INVALID_OBJECT;
    }

    if (psObj->hTrueString != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTrueString);
        psObj->hTrueString = STRING_INVALID_OBJECT;
    }

    if (psObj->hFalseString != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hFalseString);
        psObj->hFalseString = STRING_INVALID_OBJECT;
    }

    psObj->bPendingChanges = FALSE;

    vUnsetFilenames(psObj);

    // unlock the object
    SMSO_vUnlock((SMS_OBJECT)psObj);

    return;
}

/*******************************************************************************
 *
 *   vEventHandler
 *
 *   This function runs in the context of the Sirius Services Task
 *
 *******************************************************************************/
static void vEventHandler(CM_OBJECT_STRUCT *psObj, SMS_EVENT_HDL hEvent)
{
    SMS_EVENT_TYPE_ENUM eEventType;
    BOOLEAN bLocked;

    // Lock object for exclusive access
    bLocked = SMSO_bLock((SMS_OBJECT)psObj, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Error!
       return;
    }

    // Extract event information (only event type)
    eEventType = SMSE_eGetEvent(hEvent, NULL);

    // Process event...
    switch (eEventType)
    {
        // The INITIALIZE event occurs one time only when the task is started
        // and completes before returning to the caller.
        case SMS_EVENT_INITIALIZE:
        {
            puts(CM_OBJECT_NAME": SMS_EVENT_INITIALIZE");

            // Initialize the CM object
            vInitializeObject(psObj);
        }
        break;

        case SMS_EVENT_TIMEOUT:
            puts(CM_OBJECT_NAME":SMS_EVENT_TIMEOUT");
            eCommitChangesToFile();
        break;


        case SMS_EVENT_STOP:
            puts(CM_OBJECT_NAME": SMS_EVENT_STOP");
            vUninitialize();
        break;

        default:
        break;

    } // switch(eEventType)

    // Unlock object, any modifications to be made to this object
    // have already been made.
    SMSO_vUnlock((SMS_OBJECT)psObj);

    // Event handled, simply return
    return;
}

/******************************************************************************
*
*   eCommitChangesToFile
*
******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eCommitChangesToFile( void )
{
    BOOLEAN bLocked, bOk;
    CM_OBJECT_STRUCT *psObj = (CM_OBJECT_STRUCT *)ghCM;
    UN16 un16Checksum;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;

    // lock
    bLocked =
        SMSO_bLock((SMS_OBJECT)ghCM, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        FILE *psFile;
        N32 n32NumWritten;
        N32 n32Return;

        psFile = fopen(psObj->pacSecondaryConfigFileName, "wb+");
        if (NULL == psFile)
        {
            break;
        }

        // write the tags to the secondary file
        n32NumWritten = TAGCM_n32WriteTagToFile(psObj->hTopTag, psFile, 0);

        if (n32NumWritten == EOF)
        {
            fclose(psFile);
            break;
        }

        // compute the checksum of the secondary file
        bOk = bComputeChecksum(
            psFile, psObj->pacCMBuffer, psObj->tCMBufferSize,
            (size_t)n32NumWritten, &un16Checksum);

        if (bOk == FALSE)
        {
            fclose(psFile);
            break;
        }

        // stamp the checksum into the secondary file
        bOk = bStampChecksum(psFile, un16Checksum);

        fclose(psFile);

        if (bOk == FALSE)
        {
            break;
        }

        // the last step is to rename secondary config file. 
        // After the rename it will be the actual config file
        n32Return = rename( psObj->pacSecondaryConfigFileName,
                    psObj->pacConfigFileName );
        if ( n32Return == 0)
        {
            // flush the new file (necessary for some platforms)
            psFile = fopen(psObj->pacConfigFileName, "rb+");
            if (psFile != NULL)
            {
                fflush(psFile);
                fclose(psFile);

                psObj->bPendingChanges = FALSE;
                eReturn = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    } while (FALSE);

    // zero out the time since the last commit, since we just committed.
    psObj->un32TimeCommitHeldOff = 0;

    SMSO_vUnlock((SMS_OBJECT)ghCM);

    return eReturn;
}

/*****************************************************************************
 *
 *   vTimerHandler
 *
 *****************************************************************************/
static void vTimerHandler(
    OSAL_OBJECT_HDL hTimer, void *pvArg)
{
    SMSE_bPostEvent((SMS_EVENT_HDL) pvArg);
    return;
}

#ifdef SUPPORT_CUNIT
#include <cm.cunit>
#endif
