/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:CID header file.
 *
 ******************************************************************************/

#ifndef CID_OBJ_H_
#define CID_OBJ_H_

#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

/*
     CID Types Enumeration
     A collection of 'known' or supported CID types.
     As new types are invented, they get added here and treated as all other
     CIDs are treated.

     NOTE: also add any new types to union cid_type_text
*/
typedef enum cid_enum
{
    CID_FIRST = 0,

    CID_PID = CID_FIRST,
    CID_AID,
    CID_SXI_ID,
    CID_SXI_ARTIST_ID,
    CID_SXI_SONG_ID,
    CID_SXI_MARKET_ID,
    CID_SXI_SPORTS_ID,
    CID_SONG_TAG_ID,
    CID_ARTIST_TEXT,
    CID_TITLE_TEXT,
    CID_COMPOSER_TEXT,
    CID_CONTENTINFO_TEXT,
    CID_LEAGUE_TEAM_ID,
    CID_ALBUM_NAME_TEXT,

    CID_LAST = CID_ALBUM_NAME_TEXT,
    NUM_CID_TYPES,

    CID_INVALID

} CID_ENUM;

typedef struct cid_pool * CID_POOL;
#define CID_POOL_INVALID_OBJECT (CID_POOL)NULL

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

/*
    CID Type Object Data Interface
    This is how we interface with any CID object data type
*/
typedef struct cid_type_obj_data_interface_struct
{
    // Minimum Expected Size (if known, otherwise 0)
    size_t tMinSize;

    // Object Data Create/Destroy Methods

    const void *(*pvCreate) (
        const void *pvSrcObjectData,
        size_t tMinimumSize,
        SMS_OBJECT hParent,
        BOOLEAN bConstant
            );

    void (*vDestroy) (
        const void *pvObject
            );

    // Object Data Manipulation Methods

    const void *(*pvDuplicate) (
        const void *pvObject
            );

    BOOLEAN (*bModify) (
        const void **ppvObject,
        const void *pvSrcObjectData
            );

    BOOLEAN (*bCopy) (
        void **ppvDstObject,
        const void *pvSrcObject
            );

    size_t (*tSize) (
        const void *pvObject
            );

    N16 (*n16Compare) (
        const void *pvObject1,
        const void *pvObject2,
        BOOLEAN bBinary
            );

    // Object Data I/O Methods

    N32 (*n32FWrite) (
        const void *pvObject,
        FILE *psFile
            );

    N32 (*n32FWriteToMemory) (
        const void *pvObject,
        void **ppvMemory
            );

    const void *(*pvRead) (
        FILE *psFile
            );

    BOOLEAN (*bReadFromMemory) (
        const void **ppvObjectData,
        void **ppvMemory,
        SMS_OBJECT hParent
            );

    N32 (*n32FPrintf) (
        const void *pvObject,
        FILE *psFile
            );

    N32 (*n32GetValue) (
        const void *pvObject,
        void **ppvValue
            );

} CID_TYPE_OBJ_DATA_INTERFACE_STRUCT;

/*
    CID Type Interface
    This is how we interface with any CID type
*/
typedef struct cid_type_interface_struct
{
    // CID type
    CID_ENUM eType;

    // Type Init/Uninit methods

    BOOLEAN (*bInitialize) ( void );

    void (*vUnInitialize) ( void );

    // CID Input Data Parse Method

    BOOLEAN (*bParse) (
        CD_OBJECT hCDO,
        const void *pvData
            );

    // Object Data Methods
    CID_TYPE_OBJ_DATA_INTERFACE_STRUCT sObjData;

} CID_TYPE_INTERFACE_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

BOOLEAN CID_bInstall (
    SMS_OBJECT hSMS
        );

void CID_vUninstall (
    void
        );

BOOLEAN CID_bRegister (
    const CID_TYPE_INTERFACE_STRUCT *psIntf
        );

CID_OBJECT CID_hCreate (
	CID_POOL hCidPool,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        );

CID_OBJECT CID_hCreateConst (
    CID_ENUM eType,
    const void *pvObjectDataPtr
        );

void CID_vDestroy (
    CID_OBJECT hCID
        );

const void *CID_pvObjectData (
    CID_OBJECT hCID
        );

CID_ENUM CID_eType (
    CID_OBJECT hCID
        );

size_t CID_tSize (
    CID_OBJECT hCID
        );

BOOLEAN CID_bModify (
    CID_OBJECT *phCID,
    const void *pvObjectDataPtr
        );

BOOLEAN CID_bCopy (
    CID_OBJECT hDst,
    CID_OBJECT hSrc
        );

N32 CID_n32FWriteToMemory (
    CID_OBJECT hCID,
    void **ppvMemory
        );

CID_OBJECT CID_hReadFromMemory (
    const void **ppvMemory
        );

N32 CID_n32GetValue (
    CID_OBJECT hCID,
    void **ppvValue
        );

CID_POOL CID_hCreatePool (
    SMS_OBJECT hSMS
        );

void CID_vDestroyPool (
	CID_POOL hCidPool
        );

#endif /* CID_OBJ_H */
