/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:Channel header file.
 *
 ******************************************************************************/

#ifndef CHANNEL_OBJ_H_
#define CHANNEL_OBJ_H_

#include "standard.h"
#include "sms_api.h"
#include "ccache.h"
#include "presets_obj.h"
#include "cid_obj.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

typedef enum channel_type_enum
{
    CHANNEL_TYPE_AUDIO,
    CHANNEL_TYPE_DATA,
    CHANNEL_TYPE_VIDEO,
    CHANNEL_TYPE_UNKNOWN

} CHANNEL_TYPE_ENUM;

// Content Type
typedef enum content_type_enum
{
    CONTENT_TYPE_MUSIC_PID,
    CONTENT_TYPE_MUSIC_AT,
    CONTENT_TYPE_TALK_PID,
    CONTENT_TYPE_TALK_AT,
    CONTENT_TYPE_LIVE_PID,
    CONTENT_TYPE_LIVE_AT,
    CONTENT_TYPE_UNKNOWN

} CONTENT_TYPE_ENUM;

// IR Navigation Class
typedef enum ir_navigation_class_enum
{
    IR_NAVIGATION_CLASS_UNRESTRICTED_TYPE_A,
    IR_NAVIGATION_CLASS_UNRESTRICTED_TYPE_B,
    IR_NAVIGATION_CLASS_RESTRICTED_TYPE_A,
    IR_NAVIGATION_CLASS_RESTRICTED_TYPE_B,
    IR_NAVIGATION_CLASS_DISALLOWED_TYPE_A,
    IR_NAVIGATION_CLASS_DISALLOWED_TYPE_B

} IR_NAVIGATION_CLASS_ENUM;

// Type which defines the channel attribute mask
typedef UN32 CHANNEL_OBJECT_ATTRIBUTE_MASK;

#define CHANNEL_OBJECT_ATTRIBUTE_NONE           (0x0000)
#define CHANNEL_OBJECT_ATTRIBUTE_LOCKED         (0x0001)
#define CHANNEL_OBJECT_ATTRIBUTE_SKIPPED        (0x0002)
#define CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED     (0x0004)
#define CHANNEL_OBJECT_ATTRIBUTE_UPDATED        (0x0008)
#define CHANNEL_OBJECT_ATTRIBUTE_MATURE         (0x0010)
#define CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR    (0x0020)
#define CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT      (0x0040)

#define CHANNEL_OBJECT_ATTRIBUTE_ALL ( \
    (CHANNEL_OBJECT_ATTRIBUTE_MASK) \
    CHANNEL_OBJECT_ATTRIBUTE_LOCKED | \
    CHANNEL_OBJECT_ATTRIBUTE_SKIPPED | \
    CHANNEL_OBJECT_ATTRIBUTE_SUBSCRIBED | \
    CHANNEL_OBJECT_ATTRIBUTE_FREE_TO_AIR | \
    CHANNEL_OBJECT_ATTRIBUTE_MATURE | \
    CHANNEL_OBJECT_ATTRIBUTE_SUB_ALERT \
    )

typedef enum channel_attribute_status_enum
{
    CHANNEL_ATTRIBUTE_STATUS_ERROR,
    CHANNEL_ATTRIBUTE_STATUS_NO_CHANGE,
    CHANNEL_ATTRIBUTE_STATUS_CHANGED
}CHANNEL_ATTRIBUTE_STATUS_ENUM;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

// All events except CHANNEL_OBJECT_EVENT_REMOVED
// Used for initial configuration of even notification
// for CHANNEL and CHANNELLIST objects.
#define CHANNEL_OBJECT_EVENT_INITIAL \
    (CHANNEL_OBJECT_EVENT_ALL & ~(CHANNEL_OBJECT_EVENT_REMOVED))

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

CHANNEL_OBJECT CHANNEL_hCreate (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        );

void CHANNEL_vDestroy (
    CHANNEL_OBJECT hChannel
        );

CHANNEL_TYPE_ENUM CHANNEL_eType (
    CHANNEL_OBJECT hChannel
        );

BOOLEAN CHANNEL_bRegisterNotification (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventRequestMask,
    ...
        );

BOOLEAN CHANNEL_bNotifyIfPending (
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvCallbackArg
        );

void CHANNEL_vUnregisterNotification (
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    ...
        );

BOOLEAN CHANNEL_bAddCategory (
    CHANNEL_OBJECT hChannel,
    CATEGORY_OBJECT hCategory
        );

void CHANNEL_vRemoveCategory (
    CHANNEL_OBJECT hChannel,
    CATEGORY_OBJECT hCategory,
    BOOLEAN bSuppressCatEvents
        );

void CHANNEL_vUpdateArtService (
    CHANNEL_OBJECT hChannel,
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        );

CHANNEL_ART_SERVICE_OBJECT CHANNEL_hGetArtService (
    CHANNEL_OBJECT hChannel
        );

void CHANNEL_vUpdateArt (
    CHANNEL_OBJECT hChannel,
    BOOLEAN bNotify
        );

void CHANNEL_vSetEvents (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask
        );

N16 CHANNEL_n16CompareChannelIds(
    void *pvArg1,
    void *pvArg2
        );

N16 CHANNEL_n16CompareServiceIds(
    void *pvArg1,
    void *pvArg2
        );

N16 CHANNEL_n16CompareAlternateIds(
    CHANNEL_OBJECT hChannel1,
    CHANNEL_OBJECT hChannel2
        );

CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eSetAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        );

CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eClearAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        );

BOOLEAN CHANNEL_bCompareAttributes(
    CHANNEL_OBJECT hChan1,
    CHANNEL_OBJECT hChan2
        );

BOOLEAN CHANNEL_bUpdateChannelId(
    CHANNEL_OBJECT hChannel,
    CHANNEL_ID tChannelId
        );

BOOLEAN CHANNEL_bUpdateIds(
    CHANNEL_OBJECT hChannel,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId
        );

BOOLEAN CHANNEL_bUpdateType(
    CHANNEL_OBJECT hChannel,
    CHANNEL_TYPE_ENUM eType
        );

BOOLEAN CHANNEL_bUpdateShortName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        );

BOOLEAN CHANNEL_bUpdateMediumName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        );

BOOLEAN CHANNEL_bUpdateLongName(
    CHANNEL_OBJECT hChannel,
    const char *pacName
        );

BOOLEAN CHANNEL_bUpdateLongChanDesc(
    CHANNEL_OBJECT hChannel,
    const char *pacLongDesc
        );

BOOLEAN CHANNEL_bUpdateShortChanDesc(
    CHANNEL_OBJECT hChannel,
    const char *pacLongDesc
        );

BOOLEAN CHANNEL_bUpdateSimilarChannels(
    CHANNEL_OBJECT hChannel,
    UN32 un32NumSimilarChannels,
    SERVICE_ID *patSimilarChannels
        );

BOOLEAN CHANNEL_bUpdatePlayOnSelectMethod(
    CHANNEL_OBJECT hChannel,
    CHANNEL_PLAY_ON_SELECT_METHOD_ENUM ePlayOnSelectMethod
        );

BOOLEAN CHANNEL_bUpdateContentType(
    CHANNEL_OBJECT hChannel,
    CONTENT_TYPE_ENUM eContentType
        );

BOOLEAN CHANNEL_bUpdateIRNavigationClass(
    CHANNEL_OBJECT hChannel,
    IR_NAVIGATION_CLASS_ENUM eIRNavigationClass
        );

BOOLEAN CHANNEL_bUpdateACO(
    CHANNEL_OBJECT hChannel,
    CHANNEL_ACO tACO
        );

BOOLEAN CHANNEL_bUpdateCategoryId(
    CHANNEL_OBJECT hChannel,
    CATEGORY_ID tCategoryId
        );

void CHANNEL_vClearBroadcast (
    CHANNEL_OBJECT hChannel
        );

BOOLEAN CHANNEL_bInUse(
    CHANNEL_OBJECT hChannel
        );

BOOLEAN CHANNEL_bNeedsUpdated (
    CHANNEL_OBJECT hChannel
        );

BOOLEAN CHANNEL_bBlockNotifications (
    CHANNEL_OBJECT hChannel
        );

void CHANNEL_vReleaseNotifications (
    CHANNEL_OBJECT hChannel
        );

CD_OBJECT CHANNEL_hCDO (
    CHANNEL_OBJECT hChannel,
    BOOLEAN bOverrideSubscription
        );

void CHANNEL_vSetBasePresetHdl (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl
        );

CHANNEL_PRESET_HDL CHANNEL_hGetBasePresetHdl (
    CHANNEL_OBJECT hChannel
        );

void CHANNEL_vSetCurrentPresetHdl (
    CHANNEL_OBJECT hChannel,
    CHANNEL_PRESET_HDL hPresetHdl
        );

UN32 CHANNEL_un32NumSimilarChannels (
    CHANNEL_OBJECT hChannel
        );

void CHANNEL_vSetProgramsList (
    CHANNEL_OBJECT hChannel,
    EPG_CHANNEL_OBJECT hEpgChannel
        );

CID_OBJECT CHANNEL_hCidCreate (
    CHANNEL_OBJECT hChannel,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        );

const char *CHANNEL_pacPlayOnSelectMethod(
    CHANNEL_OBJECT hChannel
        );

const char *CHANNEL_pacContentType(
    CHANNEL_OBJECT hChannel
        );

CHANNEL_PLAY_ON_SELECT_METHOD_ENUM CHANNEL_ePlayOnSelectMethod(
    CHANNEL_OBJECT hChannel
        );

CONTENT_TYPE_ENUM CHANNEL_eContentType(
    CHANNEL_OBJECT hChannel
        );

IR_NAVIGATION_CLASS_ENUM CHANNEL_eIRNavigationClass(
    CHANNEL_OBJECT hChannel
        );

CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eSetUnconfirmedAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        );

CHANNEL_ATTRIBUTE_STATUS_ENUM CHANNEL_eClearUnconfirmedAttribute(
    CHANNEL_OBJECT hChannel,
    CHANNEL_OBJECT_ATTRIBUTE_MASK tAttribute
        );

SMSAPI_RETURN_CODE_ENUM CHANNEL_eIsLockedUnconfirmed (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbLocked
        );

SMSAPI_RETURN_CODE_ENUM CHANNEL_eIsSkippedUnconfirmed (
    CHANNEL_OBJECT hChannel,
    BOOLEAN *pbSkipped
        );

CHANNEL_EVENT_MASK CHANNEL_tCompareContent(
    CHANNEL_OBJECT hOldChannel,
    CHANNEL_OBJECT hNewChannel
        );

#endif /* CHANNEL_OBJ_H */
