/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CDO implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <stddef.h>

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_api_debug.h"
#include "sms_obj.h"
#include "radio.h"
#include "channel_obj.h"
#include "channel_art_obj.h"
#include "channel_art_mgr_obj.h"
#include "string_obj.h"
#include "cid_obj.h"
#include "cme.h"

#include "cdo_obj.h"
#include "_cdo_obj.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eType
*
* This object interface method is used to retrieve the Content Type
* which is part of the provided Content Description Object.
*
*****************************************************************************/
static CDO_TYPE_ENUM eType (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CDO_INVALID;
    }

    // Extract CDO type enumeration
    return psObj->sCDO.psMap->psInfo->eType;
}

/*****************************************************************************
*
*   hId
*
* This object interface method is used to retrieve the CID
* for the general content represented in this object.
*
*****************************************************************************/
static CID_OBJECT hId (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Return the CID_OBJECT requested
    return psObj->sCDO.psMap->hId;
}

/*****************************************************************************
*
*   pacDescription
*
* This object interface method is used to retrieve the content's
* description text for this object.
*
*****************************************************************************/
static const char *pacDescription (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return NULL;
    }

    // Return the description requested
    return psObj->sCDO.psMap->psInfo->pacDescription;
}

/*****************************************************************************
*
*   hArtist
*
* This object interface method is used to retrieve the Artist STRING
* for the content represented in this object.
*
*****************************************************************************/
static STRING_OBJECT hArtist (
    CD_OBJECT hCDO
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == TRUE)
    {
        // Dig into the CID which contains the STRING object we need
        if(psObj->hArtist != CID_INVALID_OBJECT)
        {
            hString = (STRING_OBJECT)CID_pvObjectData(psObj->hArtist);
        }
    }

    return hString;
}

/*****************************************************************************
*
*   hArtistId
*
* This object interface method is used to retrieve the Artist CID
* for the content represented in this object.
*
*****************************************************************************/
static CID_OBJECT hArtistId (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Return the CID_OBJECT requested
    return psObj->hArtist;
}

/*****************************************************************************
*
*   hTitle
*
* This object interface method is used to retrieve the Title STRING
* for the content represented in this object.
*
*****************************************************************************/
static STRING_OBJECT hTitle (
    CD_OBJECT hCDO
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == TRUE)
    {
        // Dig into the CID which contains the STRING object we need
        if(psObj->hTitle != CID_INVALID_OBJECT)
        {
            hString = (STRING_OBJECT)CID_pvObjectData(psObj->hTitle);
        }
    }

    return hString;
}

/*****************************************************************************
*
*   hTitleId
*
* This object interface method is used to retrieve the Title CID
* for the content represented in this object.
*
*****************************************************************************/
static CID_OBJECT hTitleId (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Return the CID_OBJECT requested
    return psObj->hTitle;
}

/*****************************************************************************
*
*   hComposer
*
* This object interface method is used to retrieve the Composer STRING
* for the content represented in this object.
*
*****************************************************************************/
static STRING_OBJECT hComposer (
    CD_OBJECT hCDO
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == TRUE)
    {
        // Dig into the CID which contains the STRING object we need
        if(psObj->hComposer != CID_INVALID_OBJECT)
        {
            hString = (STRING_OBJECT)CID_pvObjectData(psObj->hComposer);
        }
    }

    return hString;
}

/*****************************************************************************
*
*   hComposerId
*
* This object interface method is used to retrieve the Composer CID
* for the content represented in this object.
*
*****************************************************************************/
static CID_OBJECT hComposerId (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Return the CID_OBJECT requested
    return psObj->hComposer;
}

/*****************************************************************************
*
*   hAlbum
*
* This object interface method is used to retrieve the Album Name STRING
* for the content represented in this object.
*
*****************************************************************************/
static STRING_OBJECT hAlbum (
    CD_OBJECT hCDO
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == TRUE)
    {
        // Dig into the CID which contains the STRING object we need
        if(psObj->hAlbum != CID_INVALID_OBJECT)
        {
            hString = (STRING_OBJECT)CID_pvObjectData(psObj->hAlbum);
        }
    }

    return hString;
}

/*****************************************************************************
*
*   hContentInfo
*
* This object interface method is used to retrieve the ContentInfo STRING
* for the content represented in this object.
*
*****************************************************************************/
static STRING_OBJECT hContentInfo (
    CD_OBJECT hCDO
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == TRUE)
    {
        // Dig into the CID which contains the STRING object we need
        if(psObj->hContentInfo != CID_INVALID_OBJECT)
        {
            hString = (STRING_OBJECT)CID_pvObjectData(psObj->hContentInfo);
        }
    }

    return hString;
}

/*****************************************************************************
*
*   hContentInfoId
*
* This object interface method is used to retrieve the ContentInfo CID
* for the content represented in this object.
*
*****************************************************************************/
static CID_OBJECT hContentInfoId (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CID_INVALID_OBJECT;
    }

    // Return the CID_OBJECT requested
    return psObj->hContentInfo;
}


/*****************************************************************************
*
*   hArt
*
*  This object interface method is used to retrieve the art associated with
*  this CDO object.
*
*****************************************************************************/
static CHANNEL_ART_OBJECT hArt (
          CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj = (CD_OBJECT_STRUCT *)hCDO;
    BOOLEAN bValid;

    // Verify inputs
    bValid = SMSO_bValid((SMS_OBJECT)hCDO);
    if( bValid == FALSE )
    {
        // Error!
        return CHANNEL_ART_INVALID_OBJECT;
    }

    return psObj->hArt;
}

/*****************************************************************************
*
*   n16Compare
*
* This object interface method is used to perform a relative compare
* of two CDO's. The caller must have ownership of both CDO's for this
* to work. A CDO's relationship to one another is determined by the
* Content Type is represents. Thus the CDOs assoicated with each CDO is
* compared.
*
*   Outputs:
*       0   - CDOs have the same value (equal)
*       > 0 - CDO1 is greater than (after) CDO2
*       < 0 - CDO1 is less than (before) CDO2 or error
*
*****************************************************************************/
static N16 n16Compare (
    CD_OBJECT hCDO1,
    CD_OBJECT hCDO2
    )
{
    N16 n16Result = N16_MIN;
    CD_OBJECT_STRUCT *psObj1 = (CD_OBJECT_STRUCT *)hCDO1,
                      *psObj2 = (CD_OBJECT_STRUCT *)hCDO2;

    // Verify inputs
    if( (SMSO_bOwner((SMS_OBJECT)hCDO1) == FALSE) ||
        (SMSO_bOwner((SMS_OBJECT)hCDO2) == FALSE) )
    {
        // Error!
        return n16Result;
    }

    // Both CDOs must be of the same type
    if(psObj1->sCDO.psMap->psInfo->eType !=
        psObj2->sCDO.psMap->psInfo->eType)
    {
        // Not equal (or maybe error?)
        return n16Result;
    }

    // Check if a CDO specific compare method exists
    if(psObj1->sCDO.psMap->psInfo->psInterface->n16Compare != NULL)
    {
        // Call this specific CDO's compare method
        n16Result = psObj1->sCDO.psMap->psInfo->psInterface->n16Compare(
            (void*)&psObj1->sCDO.uCDO, (void*)&psObj2->sCDO.uCDO);
    }
    else
    {
        // Compare method doesn't exist, so all we can do is compare
        // the CDO's CIDs
        n16Result =
            CID.n16Compare(psObj1->sCDO.psMap->hId, psObj2->sCDO.psMap->hId);
    }

    return n16Result;
}

/*****************************************************************************
*
*   bIterateTypes
*
* Iterates the static CDO content for all available CDO's known
* at the time this API is called. The caller provides a callback function
* which will be executed for each CDO in our list. This list is static
* and available to any caller's context.
*
*****************************************************************************/
static BOOLEAN bIterateTypes (
    CDO_ITERATOR_CALLBACK bIteratorCallback,
    void *pvIteratorCallbackArg
        )
{
    BOOLEAN bRetval = FALSE;
    UN32 un32Index;

    // Check if a valid callback was provided
    if(bIteratorCallback == NULL)
    {
        // Error! Can't iterate without a callback
        return FALSE;
    }

    // Iterate each of the known and initialized CDO's
    for(un32Index = 0; un32Index < CDO_NUM_TYPES; un32Index++)
    {
        // Only need to iterate those entries which have
        // a CID entry already.
        if(gasTypeMap[un32Index].hId != CID_INVALID_OBJECT)
        {
            bRetval = bIteratorCallback(
                gasTypeMap[un32Index].hId,
                gasTypeMap[un32Index].psInfo->eType,
                gasTypeMap[un32Index].psInfo->pacDescription,
                pvIteratorCallbackArg
                    );
            if(bRetval == FALSE)
            {
                // Callback terminated iteration
                bRetval = TRUE;
                break;
            }
        }
    }

    // Check if iteration completed
    if(un32Index == CDO_NUM_TYPES)
    {
        // Iteration was complete
        bRetval = TRUE;
    }

    return bRetval;
}

/*****************************************************************************
*
*   n32FPrintf
*
* This object interface method is used by the caller to send formatted
* output of a CDO's contents to a specified file or device.
* This is mainly helpful during debugging of CDO's but could be used by
* an application for any reason. This API is different than the n32FWrite()
* method which instead writes the contents of a CDO to a file for the
* purposes of later re-generating the CDO (for storage of the object).
* This API instead sends the CDO as a verbose formatted output version.
*
* Inputs:
*
*   hCDO - The CDO handle the caller wishes to write.
*   psFile - The device to write the CDO contents to.
*
* Outputs:
*
*   The number of characters written or EOF on error.
*
*****************************************************************************/
static N32 n32FPrintf (
    CD_OBJECT hCDO,
    FILE *psFile
        )
{
    const CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;
    N32 n32Return = 0;
    N32 n32Temp   = 0;

    const char *pacBorder =
        "++++++++++++++++++++++++++++++++++++++++++++++++++\n";
    const char *pacDivider =
        "--------------------------------------------------\n";

    // Verify inputs
    if((SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE) || (psFile == NULL))
    {
        // Error!
        return EOF;
    }

    // Send formatted output to the specified device

    /**************************************************************************/
    n32Return += fprintf(psFile, "\n");
    n32Return += fprintf(psFile, pacBorder);

    n32Return += fprintf(psFile, "Content Description Object (CDO)\n");
    n32Return += fprintf(psFile, pacDivider);
    n32Return += fprintf(psFile, "hCDO = 0x%p\n", psObj);

    /*******************************Common Content Description*****************/

    n32Return += fprintf(psFile, "%s\n", pacDivider);

    // Print common CDO info...

    // Type
    n32Return += fprintf(psFile, "eType: %s\n\n",
        psObj->sCDO.psMap->psInfo->pacTypeText);

    // pacDescription
    n32Return += fprintf(psFile, "pacDescription: %s\n\n",
        psObj->sCDO.psMap->psInfo->pacDescription);

    // hId
    n32Return += fprintf(psFile, "Content Description Id (hId = 0x%p)\n",
        psObj->sCDO.psMap->hId);
    n32Temp = CID.n32FPrintf(psObj->sCDO.psMap->hId, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return += fprintf(psFile, "%s\n", pacDivider);

    n32Return += fprintf(psFile, "Artist (hCID = 0x%p)\n", psObj->hArtist);
    n32Temp = CID.n32FPrintf(psObj->hArtist, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return += fprintf(psFile, "Title (hCID = 0x%p)\n", psObj->hTitle);
    n32Temp = CID.n32FPrintf(psObj->hTitle, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return +=
        fprintf(psFile, "Composer (hCID = 0x%p)\n", psObj->hComposer);
    n32Temp = CID.n32FPrintf(psObj->hComposer, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return +=
        fprintf(psFile, "Album Name (hCID = 0x%p)\n", psObj->hAlbum);
    n32Temp = CID.n32FPrintf(psObj->hAlbum, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return +=
        fprintf(psFile, "ContentInfo (hCID = 0x%p)\n", psObj->hContentInfo);
    n32Temp = CID.n32FPrintf(psObj->hContentInfo, psFile);
    if (n32Temp > 0)
    {
        n32Return += n32Temp;
    }
    n32Return += fprintf(psFile, "\n");

    n32Return += fprintf(psFile, "%s\n", pacDivider);

    // Check if a formatted print method exists
    if(psObj->sCDO.psMap->psInfo->psInterface->n32FPrintf != NULL)
    {
        // Call this CDO's formatted print method
        n32Temp = psObj->sCDO.psMap->psInfo->psInterface->n32FPrintf(
            (void*)&psObj->sCDO.uCDO, psFile);
        if (n32Temp > 0)
        {
            n32Return += n32Temp;
        }
    }

    /**************************************************************************/
    n32Return += fprintf(psFile, "%s\n", pacBorder);

    return n32Return;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CDO_bInitialize
*
* Initialize all CDO static content. This needs to be called once when SMS
* is started. It initializes all the CDO static content such as;
* Report Markets, Sports Types, Leagues and Teams.
*
*****************************************************************************/
BOOLEAN CDO_bInitialize ( void )
{
    BOOLEAN bRetval = FALSE;
    UN32 un32Index;

    // Call each known CDO's initialization routine
    for(un32Index = 0; un32Index < CDO_NUM_TYPES; un32Index++)
    {
        // Does an initializer exist for this type?
        if(gasTypeMap[un32Index].psInfo->psInterface->bInitialize != NULL)
        {
            // Call SMS Initializer (one time)
            bRetval = gasTypeMap[un32Index].psInfo->psInterface->bInitialize();
            if(bRetval == FALSE)
            {
                // Error! Something went wrong.
                break;
            }
        }

        // Assign default map entry based on CDO type
        if(gasTypeMap[un32Index].psInfo->eType == CDO_UNKNOWN)
        {
            // Use CDO_UNKNOWN as the default entry
            gpsDefaultMapEntry = &gasTypeMap[un32Index];
        }
    }

    // Check if everything was successful, if not un-roll
    if(un32Index < CDO_NUM_TYPES)
    {
        // Call previously successful uninitializers
        while(un32Index)
        {
            // Decrement indexer
            un32Index--;

            // Does an uninitializer exist?
            if(gasTypeMap[un32Index].psInfo->psInterface->
                vUnInitialize != NULL)
            {
                // Call SMS Un-Initializer
                gasTypeMap[un32Index].psInfo->psInterface->
                    vUnInitialize();
            }
        }

        // Uninitialize default map entry
        gpsDefaultMapEntry = NULL;
        return FALSE;
    }

    return bRetval;
}

/*****************************************************************************
*
*   CDO_vUnInitialize
*
* Un-initialize all CDO static content. Needs to be called when SMS is
* stopped.
*
*****************************************************************************/
void CDO_vUnInitialize ( void )
{
    UN32 un32Index;

    // Call each CDOs uninitialization routine. This cleans up
    // and removes any CDO static content initialized at start-up
    // Call each known CDO's initialization routine
    for(un32Index = 0; un32Index < CDO_NUM_TYPES; un32Index++)
    {
        // Does an uninitializer exist?
        if(gasTypeMap[un32Index].psInfo->psInterface->vUnInitialize != NULL)
        {
            // Call SMS Un-Initializer (one time)
            gasTypeMap[un32Index].psInfo->psInterface->vUnInitialize();
        }
    }

    // Uninitialize default map entry
    gpsDefaultMapEntry = NULL;

    return;
}

/*****************************************************************************
*
*   CDO_vAssignParser
*
* This function is used to directly assign a CDO parser to a particular CDO
* type, while also providing it's CID representing that type.
*
*****************************************************************************/
void CDO_vAssignParser (
    CDO_TYPE_ENUM eType,
    CID_OBJECT hId,
    CDO_ID_PARSER bParser
        )
{
    size_t tIndex;

    // Determine which type of CDO this will be based on type provided.
    for(tIndex = 0; tIndex < CDO_NUM_TYPES; tIndex++)
    {
        // Check if content matches this entry in the map table
        if(gasTypeMap[tIndex].psInfo->eType == eType)
        {
            // Match! Apply provided id and parser
            gasTypeMap[tIndex].hId = hId;
            gasTypeMap[tIndex].bParser = bParser;

            // We're done
            break;
        }
    }

    return;
}

/*****************************************************************************
*
*   CDO_vUpdate
*
* Create/Update a CDO which belongs to the channel object provided
* using the passed in string for the fields indicated by tMask.
*
* hChannel - A channel object for which to associate this CDO object with.
* This is only required if this is a new object to be created. Otherwise
* providing CHANNEL_INVALID_OBJECT maybe provided.
*
* pacText - A test string which will be used for each CDO text field
*
* tMask - A mask indicating which fields in the CDO should be updated.
*
*****************************************************************************/
void CDO_vUpdate (
    CHANNEL_OBJECT hChannel,
    const char *pacText,
    CDO_FIELD_MASK tMask
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)CD_INVALID_OBJECT;
    CD_OBJECT hCDO = CD_INVALID_OBJECT;
    CHANNEL_EVENT_MASK tEventMask = CHANNEL_OBJECT_EVENT_NONE;
    BOOLEAN bChanged, bValid, bSubscribed = FALSE;

    // Check input. At the very least valid text must be provided
    if(pacText == NULL)
    {
        // Error!
        return;
    }

    // Check if they provided hChannel is valid and we have
    // ownership of the object.
    bValid = SMSO_bOwner((SMS_OBJECT)hChannel);
    if(bValid == TRUE)
    {
        // Extract CDO from channel. If any. Override subscription state
        // to extract the actual CHANNEL's CDO.
        hCDO = CHANNEL_hCDO(hChannel, TRUE);
        psObj = (CD_OBJECT_STRUCT *)hCDO;
        if(psObj == NULL)
        {
            // Error!
            return;
        }
    }
    else
    {
        return;
    }

    // are we supposed to modify the Artist field?
    if ((tMask & CDO_FIELD_ARTIST) == CDO_FIELD_ARTIST)
    {
        // Update content if it exists, otherwise if not...create it
        bChanged =
            CME_bUpdateContent(
                hCDO, CID_ARTIST_TEXT, (void *)pacText,
                &psObj->hArtist
                    );
        if(bChanged == TRUE)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_ARTIST;
        }
    }

    // are we supposed to modify the Title field?
    if ((tMask & CDO_FIELD_TITLE) == CDO_FIELD_TITLE)
    {
        // Update content if it exists, otherwise if not...create it
        bChanged =
            CME_bUpdateContent(
                hCDO, CID_TITLE_TEXT, (void *)pacText,
                &psObj->hTitle
                    );
        if(bChanged == TRUE)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_TITLE;
        }
    }

    // are we supposed to modify the Composer field?
    if ((tMask & CDO_FIELD_COMPOSER) == CDO_FIELD_COMPOSER)
    {
        // Update content if it exists, otherwise if not...create it
        bChanged =
            CME_bUpdateContent(
                hCDO, CID_COMPOSER_TEXT, (void *)pacText,
                &psObj->hComposer
                    );
        if(bChanged == TRUE)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_COMPOSER;
        }
    }

    // are we supposed to modify the Album Name field?
    if ((tMask & CDO_FIELD_ALBUM) == CDO_FIELD_ALBUM)
    {
        // Update content if it exists, otherwise if not...create it
        bChanged =
            CME_bUpdateContent(
            hCDO, CID_ALBUM_NAME_TEXT, (void *)pacText,
            &psObj->hAlbum
            );
        if(bChanged == TRUE)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_ALBUM;
        }
    }

    // are we supposed to modify the ContentInfo field?
    if ((tMask & CDO_FIELD_CONTENTINFO) == CDO_FIELD_CONTENTINFO)
    {
        // Update content if it exists, otherwise if not...create it
        bChanged =
            CME_bUpdateContent(
                hCDO, CID_CONTENTINFO_TEXT, (void *)pacText,
                &psObj->hContentInfo
                    );
        if(bChanged == TRUE)
        {
            tEventMask |= CHANNEL_OBJECT_EVENT_CONTENTINFO;
        }
    }

    // check if this channel is subscribed
    CHANNEL.eIsSubscribed(hChannel, &bSubscribed);
    // we only set events for content if the channel is subscribed
    // because if it isn't subscribed, the unsubscribed content will be used.
    if (bSubscribed == TRUE)
    {
        // At this point all the common information has been processed.
        // Update event mask with any relevant events which have occurred.
        CHANNEL_vSetEvents(hChannel, tEventMask);
    }

    return;
}

/*****************************************************************************
*
*   CDO_hCreate
*
* This function is used to directly create a CDO. It can only be called
* from within SMS.
*
*****************************************************************************/
CD_OBJECT CDO_hCreate (
    CHANNEL_OBJECT hChannel
        )
{
    CD_OBJECT_STRUCT *psObj;

    // Create an instance of this object
    psObj = (CD_OBJECT_STRUCT *)
        SMSO_hCreate(
            CD_OBJECT_NAME,
            sizeof(CD_OBJECT_STRUCT),
            (SMS_OBJECT)hChannel, // Child of CHANNEL
            FALSE );
    if(psObj != NULL)
    {
        // Initialize CDO object elements
        psObj->hArtist = CID_INVALID_OBJECT;
        psObj->hTitle = CID_INVALID_OBJECT;
        psObj->hComposer = CID_INVALID_OBJECT;
        psObj->hAlbum = CID_INVALID_OBJECT;
        psObj->hContentInfo = CID_INVALID_OBJECT;
        psObj->hArt = CHANNEL_ART_INVALID_OBJECT;

        // Initialize CDO structure with the default map entry
        psObj->sCDO.psMap = gpsDefaultMapEntry;
        psObj->sCDO.uCDO = *(CDO_UNION *)(
            gpsDefaultMapEntry->psInfo->pvDefaultObjData);
    }

    return (CD_OBJECT)psObj;
}

/*****************************************************************************
*
*   CDO_bAssignType
*
* This function is used to directly assign a CDO type to a particular CDO.
* Any applicable parser for this type is returned.
*
*****************************************************************************/
CDO_ID_PARSER CDO_bAssignType (
    CD_OBJECT hCDO,
    CDO_TYPE_ENUM eType
        )
{
    CDO_ID_PARSER bParser = CDO_INVALID_ID_PARSER;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return CDO_INVALID_ID_PARSER;
    }

    // Check if new type matches existing type. If not, we must
    // uninitialize the old one.
    if(eType != psObj->sCDO.psMap->psInfo->eType)
    {
        size_t tIndex;

        // Uninitialize existing CDO. This forces any previous information
        // regarding this CDO (which was another type) to be released/freed
        // as necessary by calling the specific CDO uninitialize method and
        // then re-initializing the generic CDO itself.
        vUnInit(&psObj->sCDO);

        // Determine which type of CDO this will be based on type provided.
        for(tIndex = 0; tIndex < CDO_NUM_TYPES; tIndex++)
        {
            // Check if content matches this entry in the map table
            if(gasTypeMap[tIndex].psInfo->eType == eType)
            {
                // Inform CME of the content change
                CME_bChangeContent(hCDO,
                    psObj->sCDO.psMap->hId, gasTypeMap[tIndex].hId);

                // Match! Apply mapping to this CDO's info
                psObj->sCDO.psMap = &gasTypeMap[tIndex];

                // Initialize the CDO union with defaults
                psObj->sCDO.uCDO =
                    *(CDO_UNION *)(gasTypeMap[tIndex].psInfo->pvDefaultObjData);

                break;
            }
        }
    }

    // Use this parser (if one exists)
    bParser = psObj->sCDO.psMap->bParser;

    return bParser;
}

/*****************************************************************************
*
*   CDO_vAssignPID
*
* This function is used to directly assign a program ID to a particular
* CDO.
*
*****************************************************************************/
void CDO_vAssignPID (
    CD_OBJECT hCDO,
    PROGRAM_ID tProgramId
        )
{
    BOOLEAN bOwner;
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);

    // Verify inputs
    if( TRUE == bOwner )
    {
        psObj->tProgramId =tProgramId;
    }

    return;
}

/*****************************************************************************
*
*   CDO_tGetPID
*
* This function is used to retrieve a program ID from a particular
* CDO.
*
*****************************************************************************/
PROGRAM_ID CDO_tGetPID (
    CD_OBJECT hCDO
        )
{
    BOOLEAN bOwner;
    PROGRAM_ID tProgramId = PROGRAM_INVALID_ID;

    bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);

    // Verify inputs
    if( TRUE == bOwner )
    {
        CD_OBJECT_STRUCT *psObj =
            (CD_OBJECT_STRUCT *)hCDO;

        tProgramId = psObj->tProgramId;
    }

    return tProgramId;
}

/*****************************************************************************
*
*   CDO_vDestroy
*
* This function is used to directly destroy any CDO. It can only be called
* from within SMS.
*
*****************************************************************************/
void CDO_vDestroy (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;

    // Verify inputs
    if(SMSO_bOwner((SMS_OBJECT)hCDO) == FALSE)
    {
        // Error!
        return;
    }

    // Free objects as necessary
    CME_eDestroyContent(hCDO, &psObj->hArtist);
    CME_eDestroyContent(hCDO, &psObj->hTitle);
    CME_eDestroyContent(hCDO, &psObj->hComposer);
    CME_eDestroyContent(hCDO, &psObj->hAlbum);
    CME_eDestroyContent(hCDO, &psObj->hContentInfo);

    // Destroy CHANNEL art handles
    psObj->hArt = CHANNEL_ART_INVALID_OBJECT;

    // Uninitialize CDO structure as the structure always
    // remains with the CDO.
    vUnInit(&psObj->sCDO);

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hCDO);

    return;
}

/*****************************************************************************
*
*   CDO_pvContentData
*
* This is a function which retrieves the CDO structure from inside a provided
* CDO. This is necessary for CDOs to retrieve their own structures for
* parsing.
*
*****************************************************************************/
void *CDO_pvContentData (
    CD_OBJECT hCDO
        )
{
    CD_OBJECT_STRUCT *psObj =
        (CD_OBJECT_STRUCT *)hCDO;
    BOOLEAN bOwner;

    // Verify CDO provided is valid and the caller is the owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);
    if(bOwner == FALSE)
    {
        // Error!
        return NULL;
    }

    return (void*)&psObj->sCDO.uCDO;
}

/*****************************************************************************
*
*   CDO_hCDO
*
* Extract CDO handle from a specific CDO content data belonging to
* one of the sub-classed CDO's.
*
*****************************************************************************/
CD_OBJECT CDO_hCDO (
    const void *pvContentData
        )
{
    if(pvContentData != NULL)
    {
        return (CD_OBJECT)(
            (char *)pvContentData - offsetof(CD_OBJECT_STRUCT, sCDO.uCDO)
                );
    }
    else
    {
        return CD_INVALID_OBJECT;
    }
}

/*****************************************************************************
*
*   CDO_bHasId
*
* This function is used to examine a CDO for a specific Content-ID (CID)
* provided by the caller. If this CDO has this id, then TRUE is returned
* otherwise, FALSE is returned if it does not contain it. Keep in mind
* this functions checks the entire CDO meaning it's sub-types as well for
* any content id which matches the one provided.
*
* Inputs:
*   hCDO - A valid CDO handle for which to examine and determine if
*       the provided CID is contained within it.
*   hId - A valid CID to look for within the CDO.
*
* Returns:
*   TRUE of the CID was found anywhere within the CDO, otherwise FALSE
*   is returned if it could not be found.
*
*****************************************************************************/
BOOLEAN CDO_bHasId (
    CD_OBJECT hCDO,
    CID_OBJECT hId
        )
{
    CD_OBJECT_STRUCT *psObj = (CD_OBJECT_STRUCT *)hCDO;
    BOOLEAN bOwner, bHasId = FALSE;

    // Verify CDO provided is valid and the caller is the owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hCDO);
    if(bOwner == TRUE)
    {
        BOOLEAN bValid;

        // We must check and be sure the input CID is valid
        bValid = SMSO_bValid((SMS_OBJECT)hId);
        if(bValid == TRUE)
        {
            UN16 n16Equal;

            // OK, now that that is over with, we can look to see if this
            // CID matches one we have in the CDO. Any hit is enough
            // to cause us to fall out and return TRUE.
            do
            {
                // hId == hId? (general content type)
                n16Equal = CID.n16Equal(hId, psObj->sCDO.psMap->hId);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }


                // hId == hArtist?
                n16Equal = CID.n16Equal(hId, psObj->hArtist);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }

                // hId == hTitle?
                n16Equal = CID.n16Equal(hId, psObj->hTitle);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }

                // hId == hComposer?
                n16Equal = CID.n16Equal(hId, psObj->hComposer);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }

                // hId == hAlbum?
                n16Equal = CID.n16Equal(hId, psObj->hAlbum);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }

                // hId == hContentInfo?
                n16Equal = CID.n16Equal(hId, psObj->hContentInfo);
                if(n16Equal == 0)
                {
                    // Equal, so they match
                    bHasId = TRUE;
                    break;
                }

                // Nothing there, so what about within the content itself?
                // To learn this we must use the method based on content type.
                if(psObj->sCDO.psMap->psInfo->psInterface->bHasId != NULL)
                {
                    // Use CDO type specific method to look for this CID.
                    // Each CDO type knows how to navigate its own information
                    // about what CIDs it has and what they are used for.
                    bHasId = psObj->sCDO.psMap->psInfo->psInterface->bHasId(
                        (void*)&psObj->sCDO.uCDO,
                        hId
                            );
                }

            } while(FALSE);
        }
    }

    return bHasId;
}

/*****************************************************************************
*
*   CDO_pacEnumText
*
* This function translates a provided CDO_TYPE_ENUM into a text
* string representation.
*
*****************************************************************************/
const char *CDO_pacEnumText(
    CDO_TYPE_ENUM eType
        )
{
#if SMS_DEBUG == 1
    UN32 un32Index;
    const char *pacReturnString = "error";

    // Run through all the CDO types
    for(un32Index = 0; un32Index < CDO_NUM_TYPES; un32Index++)
    {
        // Does this type match?
        if(gasTypeMap[un32Index].psInfo->eType == eType)
        {
            pacReturnString = gasTypeMap[un32Index].psInfo->pacTypeText;
            break;
        }
    }

    return pacReturnString;
#else
    return "Unknown";
#endif
}

/*****************************************************************************
*
*   CDO_hCidCreate
*
* Create a CID which is modifiable, meaning it may be created, modified and
* later destroyed. Recycling of non-constant CIDs will also be handled
* here as well. This is done via the CDO object.
*
* Inputs:
*   hCDO - CDO associated with this CID
*   eType - The type of CID object to create.
*   pvObjectDataPtr - The raw source data from which to create this object.
*
* Outputs:
*   CID_OBJECT - a new or recycled CID which represents the content provided.
*
*****************************************************************************/
CID_OBJECT CDO_hCidCreate (
	CD_OBJECT hCDO,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        )
{
	CID_OBJECT hCid = CID_INVALID_OBJECT;
	CHANNEL_OBJECT hChannel;

	// The parent of a CDO is a CHANNEL
	hChannel = (CHANNEL_OBJECT)SMSO_hParent((SMS_OBJECT)hCDO);
	if(hChannel != CHANNEL_INVALID_OBJECT)
	{
		hCid = CHANNEL_hCidCreate(hChannel, eType, pvObjectDataPtr);
	}

	return hCid;
}

/*****************************************************************************
*
*   CDO_vUpdateArt
*
*   Update this CHANNEL_OBJECT's channel art handle.  If changed, indicate
*   this update to the application.
*
*****************************************************************************/
void CDO_vUpdateArt (
    CD_OBJECT hCDO,
    CHANNEL_OBJECT hChannel
        )
{
    CD_OBJECT_STRUCT *psObj;
    CHANNEL_ART_SERVICE_OBJECT hService;
    CHANNEL_ART_OBJECT hOldArt;
    SERVICE_ID tServiceId;
    UN32 un32ProgramId = 0;
    CDO_TYPE_ENUM eType;
    BOOLEAN bIsSubscribed;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Verify input
    if ( ( CD_INVALID_OBJECT == hCDO ) ||
         ( CHANNEL_INVALID_OBJECT == hChannel ) )
    {
        // Error!
        return;
    }

    // Get the art service from our channel
    hService = CHANNEL_hGetArtService( hChannel );

    // Cast our handle to a CD object
    psObj = (CD_OBJECT_STRUCT *)hCDO;

    // Remember what our current art is; we'll this later to see
    // if the art actually changed
    hOldArt = psObj->hArt;

    // We'll also need the program ID for the currently playing
    // track
    un32ProgramId = CDO_tGetPID( hCDO );

    // Get the eType for our Channel
    eType = CDO.eType(hCDO);

    // Check the subscription state of this CDO's channel

    eReturn = CHANNEL.eIsSubscribed( hChannel, &bIsSubscribed );

    // Make sure that worked ...
    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            CD_OBJECT_NAME": Couldn't get channel subscription status!"
            );
    }

    // Channels that are unsubscribed all share the same CDO.
    // If this channel is unsubscribed, just give it the
    // service default image (which is on SID 0).

    if ( ( SMSAPI_RETURN_CODE_SUCCESS != eReturn ) ||
         ( FALSE == bIsSubscribed ) )
    {
        tServiceId = 0;
    }
    else
    {
        tServiceId = CHANNEL.tServiceId( hChannel );
    }

    // Update the channel art object handle for this channel
    psObj->hArt = CHANNEL_ART_MGR_hGetArtForCDO( hService,
        tServiceId, un32ProgramId, eType );

    // Check to see that the handle actually changed
    if ( psObj->hArt != hOldArt )
    {
        // Update the channel's event mask with the art event
        CHANNEL_vSetEvents(hChannel, CHANNEL_OBJECT_EVENT_ART);
    }

    return;
}
/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   vInit
*
* Initialize the CDO structure. This must be called once when a CDO
* is created to properly initialize it's CDO.
*
*****************************************************************************/
static void vInit (
    CDO_STRUCT *psObj
        )
{
    CD_OBJECT hCDO;

    // Check input
    if(psObj == NULL)
    {
        return;
    }

    // Grab the CDO...
    hCDO = CDO_hCDO(&psObj->uCDO);

    // Inform CME of the content change. This content is no
    // longer part of this CDO. In this case we only notify the
    // CME that the content has changed. We don't destroy anything since
    // these CIDs are constant (unlike other types).
    if(psObj->psMap != NULL)
    {
        CME_bChangeContent(hCDO, psObj->psMap->hId, gpsDefaultMapEntry->hId);
    }

    // Initialize CDO structure with the default map entry
    psObj->psMap = gpsDefaultMapEntry;
    psObj->uCDO = *(CDO_UNION *)(
        gpsDefaultMapEntry->psInfo->pvDefaultObjData);

    return;
}

/*****************************************************************************
*
*   vUnInit
*
* Un-Initialize the CDO structure. This must be called once when the associated
* CDO is no longer needed (destroyed).
*
*****************************************************************************/
void vUnInit (
    CDO_STRUCT *psObj
        )
{
    // Check input
    if(psObj != NULL)
    {
        // Check if an uninit method exists for a specific CDO.
        if(psObj->psMap->psInfo->psInterface->vUnInit != NULL)
        {
            // Call this CDO's uninitialization method.
            psObj->psMap->psInfo->psInterface->vUnInit((void*)&psObj->uCDO);
        }

        // Re-Initialize CDO now that it is available.
        vInit(psObj);
    }

    return;
}

/*****************************************************************************
*
*   vUnInitDefault
*
* Default Un-initialize method
*
*****************************************************************************/
static void vUnInitDefault (
    const CDO_UNION *puObj
        )
{
    return;
}

/*****************************************************************************
*
*   n16EqualDefault
*
* Default compare method. Does nothing but always report unequal
*
*****************************************************************************/
static N16 n16EqualDefault (
    const CDO_UNION *puObj1,
    const CDO_UNION *puObj2
        )
{
    // Not equal
    return -1;
}

/*****************************************************************************
*
*   n16CompareDefault
*
* Default sort method, Does nothing but always reports first object is
* less than the second object.
*
*****************************************************************************/
static N16 n16CompareDefault (
    const CDO_UNION *puObj1,
    const CDO_UNION *puObj2
        )
{
    if (puObj1 == puObj2)
    {
        return 0;
    }

    // Not equal
    return -1;
}

/*****************************************************************************
*
*   n32FPrintfDefault
*
* Default formatted print method. Does nothing.
*
*****************************************************************************/
static N32 n32FPrintfDefault (
    const CDO_UNION *puObj,
    FILE *psFile
        )
{
    return 0;
}
