/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:CCache header file.
 *
 ******************************************************************************/

#ifndef CCACHE_H_
#define CCACHE_H_

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "cid_obj.h"
#include "sms_event_types.h"

typedef struct ccache_object * CCACHE_OBJECT;
#define CCACHE_INVALID_OBJECT (CCACHE_OBJECT)NULL

// Remap callback
typedef void (*CCACHE_REMAP_CALLBACK)(
    CCACHE_OBJECT hCCache,
    BOOLEAN bComplete,
    void *pvCallbackArg
        );

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// CCache channel iterator function prototype
typedef BOOLEAN (*CCACHE_CHANNEL_ITERATOR_HANDLER) (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

// CCache category iterator function prototype
typedef BOOLEAN (*CCACHE_CATEGORY_ITERATOR_HANDLER) (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        );

typedef SMSAPI_EVENT_MASK CCACHE_EVENT_MASK;

// Update callback
typedef void (*CCACHE_OBJECT_EVENT_CALLBACK)(
    CCACHE_OBJECT hCCache,
    CCACHE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// CCACHE Object Event Mask
#define CCACHE_OBJECT_EVENT_NONE            (0x0000)
#define CCACHE_OBJECT_EVENT_UPDATE          (0x0001)
#define CCACHE_OBJECT_EVENT_ALL           (    \
    (CCACHE_EVENT_MASK)                        \
    CCACHE_OBJECT_EVENT_UPDATE                 \
        )

// ccache event types
typedef enum ccache_event_type_enum
{
    CCACHE_EVENT_TYPE_PRINT,
    CCACHE_EVENT_TYPE_REMAP,
    CCACHE_EVENT_TYPE_ACO

} CCACHE_EVENT_TYPE_ENUM;

typedef enum ccache_remap_state_enum
{
    CCACHE_REMAP_INIT,
    CCACHE_REMAP_UPDATE_CATEGORY_VECTOR,
    CCACHE_REMAP_UPDATE_CHANNEL_INFO,
    CCACHE_REMAP_SUBSCRIPTION,
    CCACHE_REMAP_FINAL,
    CCACHE_REMAP_ERROR

} CCACHE_REMAP_STATE_ENUM;

typedef enum
{
    CCACHE_UPDATE_TYPE_ART_SERVICE,
    CCACHE_UPDATE_TYPE_ALBUM_ART,
    CCACHE_UPDATE_TYPE_CHAN_ART,
} CCACHE_UPDATE_TYPE_ENUM;

typedef struct sms_event_ccache_remap_struct
{
    // Initial transition?
    BOOLEAN bInitial;

    // Notifications blocked?
    BOOLEAN bBlocked;

    // CEML Blocked?
    SMSAPI_RETURN_CODE_ENUM eCemsDisableReturnCode;

    // Remap state
    CCACHE_REMAP_STATE_ENUM eState;

    // Remap list entry handles
    OSAL_LINKED_LIST_ENTRY hThisEntry;
    OSAL_LINKED_LIST_ENTRY hLastEntry;

    // Final reportback info
    CCACHE_REMAP_CALLBACK vCallback;
    void *pvCallbackArg;

} SMS_EVENT_CCACHE_REMAP_STRUCT;

typedef struct sms_event_ccache_print_struct
{
    BOOLEAN bChannels;
    BOOLEAN bCategories;
    BOOLEAN bVerbose;

} SMS_EVENT_CCACHE_PRINT_STRUCT;

typedef union sms_event_ccache_union
{
    SMS_EVENT_CCACHE_PRINT_STRUCT sPrint;
    SMS_EVENT_CCACHE_REMAP_STRUCT sRemap;

} SMS_EVENT_CCACHE_UNION;

typedef struct sms_event_ccache_struct
{
    CCACHE_EVENT_TYPE_ENUM eType;
    SMS_EVENT_CCACHE_UNION uCache;

} SMS_EVENT_CCACHE_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */

CCACHE_OBJECT CCACHE_hCreate (
    DECODER_OBJECT hDecoder,
    const char *pacName
        );

void CCACHE_vDestroy (
    CCACHE_OBJECT hCCache
        );

void CCACHE_vUpdateChannelArt (
    CCACHE_OBJECT hCCache,
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CCACHE_UPDATE_TYPE_ENUM eUpdateType
        );

void CCACHE_vCleanChannelsEpgData (
    CCACHE_OBJECT hCCache
        );

CHANNEL_OBJECT CCACHE_hChannel (
    CCACHE_OBJECT hCCache,
    CHANNEL_ID *ptChannelId,
    N16 *pn16Offset
        );

CHANNEL_ID CCACHE_tChannelIdFromServiceId (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId
        );

CHANNEL_OBJECT CCACHE_hChannelFromIds (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bCreateIfNotFound
        );

CATEGORY_OBJECT CCACHE_hCategory (
    CCACHE_OBJECT hCCache,
    CATEGORY_ID *ptCategoryId,
    N16 n16Offset
        );

CATEGORY_ID CCACHE_tAddCategory (
    CCACHE_OBJECT hCCache,
    CATEGORY_TYPE_ENUM eCategoryType,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        );

void CCACHE_vRemoveCategory(
    CCACHE_OBJECT hCCache,
    CATEGORY_ID tCategoryId
        );

BOOLEAN CCACHE_bRemoveChannel(
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT hChannel
        );

void CCACHE_vStats (
    CCACHE_OBJECT hCCache,
    N16 *pn16NumValidChannels,
    N16 *pn16NumValidCategories
        );

BOOLEAN CCACHE_bSortChannelLists (
    CCACHE_OBJECT hCCache
        );

BOOLEAN CCACHE_bSortCategoryList (
    CCACHE_OBJECT hCCache
        );

BOOLEAN CCACHE_bRegisterNotification (
    CCACHE_OBJECT hCCache,
    CCACHE_EVENT_MASK tEventRequestMask,
    CCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

BOOLEAN CCACHE_bUnRegisterNotification (
    CCACHE_OBJECT hCCache,
    CCACHE_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

BOOLEAN CCACHE_bRegisterAllChannelsNotification (
    CCACHE_OBJECT hCCache,
    CHANNEL_EVENT_MASK tEventRequestMask,
    BOOLEAN bImmediateNotify,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

BOOLEAN CCACHE_bUnRegisterAllChannelsNotification (
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

SMSAPI_RETURN_CODE_ENUM CCACHE_eHandleAppChannelNotifyOnChange (
    CCACHE_OBJECT hCCache,
    CHANNEL_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        );

BOOLEAN CCACHE_bCheckNextChannelForAppNotification (
    CCACHE_OBJECT hCCache
        );

BOOLEAN CCACHE_bRegisterAllCategoriesNotification (
    CCACHE_OBJECT hCCache,
    CATEGORY_EVENT_MASK tEventRequestMask,
    BOOLEAN bImmediateNotify,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

BOOLEAN CCACHE_bUnRegisterAllCategoriesNotification (
    CCACHE_OBJECT hCCache,
    CATEGORY_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

SMSAPI_RETURN_CODE_ENUM CCACHE_eHandleAppCategoryNotifyOnChange (
    CCACHE_OBJECT hCCache,
    CATEGORY_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        );

BOOLEAN CCACHE_bCheckNextCategoryForAppNotification (
    CCACHE_OBJECT hCCache
        );

void CCACHE_vEventHandler (
    CCACHE_OBJECT hCCache,
    SMS_EVENT_CCACHE_STRUCT const *psCCache
        );

BOOLEAN CCACHE_bUpdateChannel (
    CHANNEL_OBJECT hChannel
        );

BOOLEAN CCACHE_bUpdateChannelId(
    CHANNEL_OBJECT hChannelToUpdate,
    CHANNEL_ID tNewChannelId
        );

CHANNEL_OBJECT CCACHE_hUnsubscribedChannel (
    CCACHE_OBJECT hCCache
        );

CHANNEL_OBJECT CCACHE_hSubscriptionAlertChannel (
    CCACHE_OBJECT hCCache
        );

BOOLEAN CCACHE_bUpdateSubscription (
    SMS_EVENT_HANDLER hEventHdlr,
    CCACHE_REMAP_CALLBACK vCallback,
    void *pvCallbackArg
        );

BOOLEAN CCACHE_bRemap (
    SMS_EVENT_HANDLER hEventHdlr,
    CCACHE_REMAP_CALLBACK vCallback,
    void *pvCallbackArg
        );

BOOLEAN CCACHE_bIterateChannels (
    CCACHE_OBJECT hCCache,
    BOOLEAN bIterateByServiceId,
    CCACHE_CHANNEL_ITERATOR_HANDLER bIterator,
    void *pvArg
        );

BOOLEAN CCACHE_bIterateCategories (
    CCACHE_OBJECT hCCache,
    CCACHE_CATEGORY_ITERATOR_HANDLER bIterator,
    void *pvArg
        );

CID_OBJECT CCACHE_hCidCreate (
    CCACHE_OBJECT hCCache,
    CID_ENUM eType,
    const void *pvObjectDataPtr
        );

CHANNEL_OBJECT CCACHE_hCreateChannel (
    CCACHE_OBJECT hCCache,
    SERVICE_ID tServiceId,
    CHANNEL_ID tChannelId,
    BOOLEAN bNotify
        );

#endif /* CCACHE_H_ */
