/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CAS implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "cid_obj.h"
#include "cel.h"
#include "string_obj.h"
#include "category_obj.h"
#include "cal_content.h"
#include "cal_alert.h"
#include "cas.h"
#include "_cas.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hCreateList
*
* Creates a content alert list (CAL). This list may be used to add,
* remove, replace, enable and disable content to be monitored. It also is
* used to maintain and acknowledge active events when they occur.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object for which to attach
*               or associate this content alert list with.
*   vCALCallback
*               A function which is called when an alert concerning
*               any content registered to this list occurs.
*   pvCALCallbackArg
*               An argument which is anonymous and application specific. It
*               is provided to the caller when the provided vCALCallback
*               function is called.
*
* Outputs:
*
*   CAL_OBJECT  A valid CAL_OBJECT handle on success or CAL_INVALID_OBJECT
*                   if an error or failure occurred.
*
*****************************************************************************/
static CAL_OBJECT hCreateList (
    DECODER_OBJECT hDecoder,
    CAL_CALLBACK vCALCallback,
    void *pvCALCallbackArg
        )
{
    // call the friend function to actually create the list.
    // NOTE: ALL CAL's created vis this API are configured to auto-acknowledge
    // the CEML Events received by the CAL. THis is because the application
    // doesn't need to know about the CEML and can't access the active CEML
    // events anyway.
    return CAS_hCreateList (
               hDecoder,
               vCALCallback,
               pvCALCallbackArg,
               TRUE // auto ack
                    );
}

/*****************************************************************************
*
*   vDestroyList
*
* Destroys a Content Alert List (CAL).
*
* Inputs:
*
*   hCAL   A handle to a valid Content Alert List for which to
*           destroy and release all it's resources.
*
* Outputs:
*
*   None.
*
*****************************************************************************/
static void vDestroyList (
    CAL_OBJECT hCAL
        )
{
    BOOLEAN bLocked;
    SMS_OBJECT hParent;
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;

    // Verify and lock SMS Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // Unable to lock the provided object
        return;
    }

    hParent = SMSO_hParent((SMS_OBJECT)hCAL);

    // Destroy CEML
    if (psObj->hCEML != CEML_INVALID_OBJECT)
    {
        BOOLEAN bBlocked = FALSE;
        CATEGORY_ID tCategoryId;

        // For CAL, category could be attached / detached via
        // public APIs. So, make sure that we have one.
        tCategoryId = CAL.tCategoryId(hCAL);
        if (tCategoryId != CATEGORY_INVALID_ID)
        {
            // Block all further notifications for all channels
            bBlocked = CATEGORY_bBlockNotifications(
                psObj->hDecoder, tCategoryId);
        }

        CEML.vDestroyList(psObj->hCEML);
        psObj->hCEML = CEML_INVALID_OBJECT;

        // Releasing category notifications
        if (bBlocked == TRUE)
        {
            CATEGORY_vReleaseNotifications(
                psObj->hDecoder, tCategoryId);
        }
    }

    // destroy alert
    if (psObj->hAlert != CAL_ALERT_INVALID_OBJECT)
    {
        CAL_ALERT_vDestroy(psObj->hAlert);
        psObj->hAlert = CAL_ALERT_INVALID_OBJECT;
    }

    // uninitialize
    psObj->pvCALCallbackArg = NULL;
    psObj->vCALCallback = (CAL_CALLBACK)NULL;

    CAL.eDetachCategory(hCAL);

    // intitally enabled
    psObj->bEnabled = FALSE;

    // Destroy the object itself
    SMSO_vDestroy((SMS_OBJECT)psObj);

    puts(CAL_OBJECT_NAME": Destroyed CAL");

    SMSO_vUnlock((SMS_OBJECT)hParent);

    return;
}

/*****************************************************************************
*
*   eRegister
*
* Register content for Alerts
*
* Inputs:
*
*   hCAL       A handle to a valid Content Alert List for which to
*               register this content with.
*   hCID        A content identifier representing content to be monitored.
*   pvContentArg A pointer to an anonymous type (defined by the caller)
*               which will be associated with this registered content and
*               provided to the caller when the content event occurs or when
*               this content is iterated.
*   un32Options Options provided by the caller about this registration
*               such as initial, end, etc.
*   hArtistText A handle to a STRING obect containing artist text to be
*               associated with this registered content.
*   hTitleText  A handle to a STRING obect containing title text to be
*               associated with this registered content.
*   phCalContent  A pointer to store the registered content object.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM  SMSAPI_RETURN_CODE_SUCCESS on success.
*                         SMSAPI_RETURN_CODE_DUPLICATE_CONTENT if this content is
*                         already registered.
*                         SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegister (
    CAL_OBJECT hCAL,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    STRING_OBJECT hArtistText,
    STRING_OBJECT hTitleText,
    CAL_CONTENT_OBJECT *phCalContent
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;
    CAL_CONTENT_OBJECT hContent;

    // Verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        // create a CAL_CONTENT object
        hContent = CAL_CONTENT_hCreate (
                       pvContentArg,
                       un32Options,
                       hArtistText,
                       hTitleText,
                       hCAL,
                       psObj->hCEML);

        if(hContent != CAL_CONTENT_INVALID_OBJECT)
        {
            // CAL_CONTENT was created successfully

            // even if the caller didn't want to be notified of the
            // END, this serivce wants to know
            // (so we can get rid of the alert)
            un32Options = (un32Options | SMS_EVENT_REGISTRATION_OPTION_END);

            // Register content with our CEML
            eReturnCode =
                CEML.eRegister(
                    psObj->hCEML,
                    hCID,
                    (void *)hContent,
                    un32Options
                        );

            if(eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
            {
                if(phCalContent != NULL)
                {
                    *phCalContent = hContent;
                }
                printf("%s: hContent = 0x%p\n", CAL_OBJECT_NAME, hContent);
            }
            else
            {
                // since we didn't register the content, destroy the object
                // we created.
                CAL_CONTENT_vDestroy(hContent);
            }
        }
        else
        {
            // Error!
            eReturnCode = SMSAPI_RETURN_CODE_ERROR;
        }

        // Unlock CAL object
        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eIterate
*
* This method allows iteration of the CAL.
*
* Inputs:
*
*   hCAL   A handle to a valid Content Alert List for which to
*           iterate over.
*   bContentIterator
*           A function to call for each content entry in the list.
*   pvContentIteratorArg
*           An anonymous pointer which is caller specific and provided
*           each time bContentIterator is called.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterate (
    CAL_OBJECT hCAL,
    CAL_CONTENT_ITERATOR_CALLBACK bContentIterator,
    void *pvContentIteratorArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Verify callback pointer
    if(bContentIterator == NULL)
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Verify and lock CAL Object
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        CAL_OBJECT_STRUCT *psObj = (CAL_OBJECT_STRUCT *)hCAL;
        SMSAPI_RETURN_CODE_ENUM eCEMLReturn;
        CAL_TO_CEML_ITERATOR_STRUCT sCEMLIteratorStruct;

        sCEMLIteratorStruct.pvCALIteratorArg = pvContentIteratorArg;
        sCEMLIteratorStruct.bCALCallback = bContentIterator;

        // iterate contentlist
        eCEMLReturn = CEML.eIterateContentList (
            psObj->hCEML,
            bCALCEMLIterator,
            &sCEMLIteratorStruct
                );

        if (eCEMLReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bEnabled
*
*   This API is used to retrieve the enabled status of this CAL obj.
*
*   Inputs:
*
*   hContent - A handle to a valid CAL.
*
*   Outputs:
*
*   TRUE if Enabled. FALSE if Disabled
*
*****************************************************************************/
static BOOLEAN bEnabled (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked, bEnabled = FALSE;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        bEnabled = psObj->bEnabled;

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return bEnabled;
}

/*****************************************************************************
*
*   eEnable
*
*   This API is used to globally enable alerts for the entire CAL.
*
*   Inputs:
*
*   hCAL       A handle to a valid Content Alert List for which to
*               enable this content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eEnable(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // Enable content
    psObj->bEnabled = TRUE;

    SMSO_vUnlock((SMS_OBJECT)hCAL);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   eDisable
*
*   This API is used to globally disable alerts for the entire CAL.
*
*   Inputs:
*
*   hCAL       A handle to a valid Content Alert List for which to
*              disable this content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDisable (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == FALSE)
    {
        // couldn't lock it
        return SMSAPI_RETURN_CODE_ERROR;
    }

    psObj->bEnabled = FALSE;

    SMSO_vUnlock((SMS_OBJECT)hCAL);

    return SMSAPI_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
*
*   tNumItems
*
*   This API is used to report the number of registered CAL_CONTENT objects in
*   a Content Alert List to the caller.
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object for which the caller wishes to
*              retrieve the number items it contains.
*
*   Outputs:
*
*   The number of registered items in this CAL object.
*
*****************************************************************************/
static 	size_t tNumItems (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked;
    size_t tItems = 0;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // Iterate all registered content...
        tItems = CEML.tContentListItems(psObj->hCEML);

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return tItems;
}

/*****************************************************************************
*
*   bExists
*
*   This API is used to determine if content is already in the CAL's
*   Contentlist
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object for which the caller wishes to
*              retrieve the number items it contains.
*
*   Outputs:
*
*   The number of registered items in this CAL object.
*
*****************************************************************************/
static BOOLEAN bExists (
    CAL_OBJECT hCAL,
    CID_OBJECT hCID
        )
{
    CAL_CEML_ITERATOR_STRUCT sIterator;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;

    BOOLEAN bLocked;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // Search our list and determine
        // if content with this CID already is registered.
        sIterator.hId = hCID;
        sIterator.bFound = FALSE;

        // Iterate all registered content...
        eReturn = CEML.eIterateContentList(
                      psObj->hCEML,
                      bEqualCIDs,
                      &sIterator
                          );

        SMSO_vUnlock((SMS_OBJECT)hCAL);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            return sIterator.bFound;
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   eRemoveAll
*
*   This API is used to remove all registered content from the cal
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object for which the caller wishes to
*              remove all registered content.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM     SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRemoveAll (
    CAL_OBJECT hCAL
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;

    BOOLEAN bLocked;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        eReturnCode = CEML.eRemoveAll(psObj->hCEML);

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   tCategoryId
*
*   This API is used to retrieve the id of the CATEGORY attached to this CAL.
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object which the caller wishes to
*              retrieve the attached CATEGORY id
*
*   Outputs:
*
*   A valid CATEGORY_ID ion success if a CATEGORY is currently attached.
*   CATEGORY_INVALID_ID if no CATEGORY is attached, or on error.
*
*****************************************************************************/
static CATEGORY_ID tCategoryId (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        // At this point we have exclusive access to the CAL
        tCategoryId = psObj->tCategoryId;

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return tCategoryId;
}

/*****************************************************************************
*
*   eAttachCategory
*
*   This API is used to attach a CATGEORY to this Content Alert List object.
*
*   Inputs:
*
*   hCAL                A handle to a valid CAL object which the caller wishes
*                       to attach a category
*   tCategoryId         Id of category to attach
*   tCatAddOptions      options for how channels will be added to the attched
*                       category
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success
*   SMSAPI_RETURN_CODE_INVALID_OPTIONS if invalid options were supplied
*   SMSAPI_RETURN_CODE_ERROR on error
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAttachCategory (
    CAL_OBJECT hCAL,
    CATEGORY_ID tCategoryId,
    CAL_AUTO_ADD_OPTIONS tCatAddOptions,
    ...
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // Check if options are valid first and that there are no other
    // unknown options selected.
    // Then make sure they have provided either BOTTOM or TOP
    // as a trigger option, but not both or neither. One or the other
    // must be provided.
    if (  ( ( tCatAddOptions & ~CAL_AUTO_ADD_OPTION_ALL ) !=
              CAL_AUTO_ADD_OPTION_NONE )
       || ( ( ( (tCatAddOptions & CAL_AUTO_ADD_OPTION_TOP) == CAL_AUTO_ADD_OPTION_TOP) ^
              ( (tCatAddOptions &  CAL_AUTO_ADD_OPTION_BOTTOM) == CAL_AUTO_ADD_OPTION_BOTTOM) ) == FALSE )
        )
    {
        // Error! Invalid options provided
        return SMSAPI_RETURN_CODE_INVALID_OPTIONS;
    }

    // verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        BOOLEAN bCategoryValid = FALSE;

        // we can only attach a cat if there is not one currently attached
        // so let's check if one is attached
        if (psObj->tCategoryId == CATEGORY_INVALID_ID)
        {
            // no there is not a cat attached. ok to proceed

            if (tCategoryId == CATEGORY_INVALID_ID)
            {
                // the user wants us to create the category
                va_list tNewCatArg;
                const char *pacLongName, *pacShortName;

                va_start( tNewCatArg, tCatAddOptions );
                pacLongName = va_arg( tNewCatArg, const char * );
                pacShortName = va_arg( tNewCatArg, const char * );
                va_end( tNewCatArg );

                tCategoryId = CATEGORY_tCreateVirtualCategory (
                                  psObj->hDecoder,
                                  pacLongName,
                                  pacShortName,
                                  0, FALSE, FALSE
                                      );

                // was the creation successful?
                if (tCategoryId == CATEGORY_INVALID_ID)
                {
                    // failed to create the category!
                    SMSO_vUnlock((SMS_OBJECT)hCAL);
                    return SMSAPI_RETURN_CODE_ERROR;
                }

                // we had to create a cat
                psObj->bCreatedCategory = TRUE;
                bCategoryValid = TRUE;
            }
            else
            {
                BOOLEAN bDecoderLocked;
                // we didn't have to create a cat
                psObj->bCreatedCategory = FALSE;

                // we need to see if the provided Category ID
                // is assocaited with a real CATEGORY
                bDecoderLocked = SMSO_bLock((SMS_OBJECT)psObj->hDecoder,
                                            OSAL_OBJ_TIMEOUT_INFINITE);
                if (bDecoderLocked == TRUE)
                {
                    CCACHE_OBJECT hCCache;
                    CATEGORY_OBJECT hCategory;

                    hCCache = DECODER_hCCache(psObj->hDecoder);

                    hCategory = CCACHE_hCategory(hCCache, &tCategoryId, 0);
                    if(hCategory != CATEGORY_INVALID_OBJECT)
                    {
                        // category exists

                        // alert channels wouldn't be able to be added to a
                        // broadcast cat - let's reject an attempt to attach one
                        CATEGORY_TYPE_ENUM eType;

                        eType = CATEGORY.eType(hCategory);

                        if (eType != CATEGORY_TYPE_BROADCAST)
                        {
                            bCategoryValid = TRUE;
                        }
                    }

                    SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
                }
            }

            if (bCategoryValid == TRUE)
            {
                psObj->tCategoryId = tCategoryId;
                psObj->tCatAddOptions = tCatAddOptions;

                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eDetachCategory
*
*   This API is used to detach a CATGEORY from this Content Alert List object.
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object which the caller wishes to
*              detatch the attached CATEGORY
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_ENUM
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDetachCategory (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    BOOLEAN bLocked;

    // verify and lock SMS Object (CAL)
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);

    if (bLocked == TRUE)
    {
        if (psObj->bCreatedCategory == TRUE)
        {
            CATEGORY.vDestroy (
                psObj->hDecoder,
                psObj->tCategoryId
                    );

            psObj->bCreatedCategory = FALSE;
        }

        // no category is attached, so update these parameters
        psObj->tCategoryId = CATEGORY_INVALID_ID;
        psObj->tCatAddOptions = CAL_AUTO_ADD_OPTION_NONE;

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return eReturnCode;
}



/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CAS_hCreateList
*
* Creates a content alert list (CAL). This list may be used to add,
* remove, replace, enable and disable content to be monitored. It also is
* used to maintain and acknowledge active events when they occur. NOTE: This
* friend functions takes one more parameter than the global API. This allows
* SMS objects to create a CAL that does not automatically acknowledge CEML
* events that result in a call to the CAL Callback. All CEML events that do not
* result in a call to the CAL callback will still be auto ack'd.
*
* Inputs:
*
*   hDecoder    A handle to a valid DECODER object for which to attach
*               or associate this content alert list with.
*   vCALCallback
*               A function which is called when an alert concerning
*               any content registered to this list occurs.
*   pvCALCallbackArg
*               An argument which is anonymous and application specific. It
*               is provided to the caller when the provided vCALCallback
*               function is called.
*   bAutoAck - TRUE if the caller wants the CAL to auto-acknowledge CEML events.
*              FALSE if it does not want the CAL to auto-acknowledge CEML events
*
* Outputs:
*
*   CAL_OBJECT  A valid CAL_OBJECT handle on success or CAL_INVALID_OBJECT
*                   if an error or failure occurred.
*
*****************************************************************************/
CAL_OBJECT CAS_hCreateList (
    DECODER_OBJECT hDecoder,
    CAL_CALLBACK vCALCallback,
    void *pvCALCallbackArg,
    BOOLEAN bAutoAck
        )
{
    CAL_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bValid;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static UN32 un32Instance = 0;

    do
    {
        // Verify inputs are correct
        bValid = SMSO_bValid((SMS_OBJECT)hDecoder);

        if (bValid == FALSE) // Valid Decoder?
        {
            // Error!
            break;
        }

        // Construct a unique name for this list.
        snprintf(&acName[0],
            sizeof(acName),
            CAL_OBJECT_NAME":Obj:%u",
            un32Instance++);

        // Create a CAL object
        psObj = (CAL_OBJECT_STRUCT *)
            SMSO_hCreate(
                &acName[0],
                sizeof(CAL_OBJECT_STRUCT),
                (SMS_OBJECT)hDecoder,
                FALSE
                    );
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // Create the CEML for this alert list
        psObj->hCEML =
            CEML.hCreateList(
                hDecoder,
                (CONTENT_COMPARE_HANDLER)NULL,
                vCALEventCallback,
                pvCALCallbackArg
                    );
        if (psObj->hCEML == CEML_INVALID_OBJECT)
        {
            // Error!
            break;
        }

        // remember the decoder we're associated with
        psObj->hDecoder = hDecoder;

        // Initialize callback and arg
        psObj->vCALCallback = vCALCallback;
        psObj->pvCALCallbackArg = pvCALCallbackArg;

        // no category attached at creation
        psObj->tCategoryId = CATEGORY_INVALID_ID;
        psObj->tCatAddOptions = CAL_AUTO_ADD_OPTION_NONE;
        psObj->bCreatedCategory = FALSE;

        // intitally enabled
        psObj->bEnabled = TRUE;

        psObj->bAutoAck = bAutoAck;

        puts(CAL_OBJECT_NAME": Created CAL");
        // Return CAL object to caller
        return (CAL_OBJECT)psObj;

    } while(0);


    // Something went wrong
    vDestroyList((CAL_OBJECT)psObj);

    return CAL_INVALID_OBJECT;
}

/*****************************************************************************
*
*   CAL_hDecoder
*
* A fxn to get the DECODER this CAL is operating on
*
*****************************************************************************/
DECODER_OBJECT CAL_hDecoder(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;

    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        hDecoder = psObj->hDecoder;
    }

    return hDecoder;
}

/*****************************************************************************
*
*   CAL_hGetNextActiveEvent
*
* A fxn to get the first available active (non-Acknowledged) event from the
* CEML's Active Event List. The function will then get the corresponding
* CAL_ALERT and return it.
*
*****************************************************************************/
CAL_ALERT_OBJECT CAL_hGetNextActiveEvent(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj = (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    CAL_ALERT_OBJECT hAlert = CAL_ALERT_INVALID_OBJECT;

    // validate input
    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        // Iterate the CEML's Active Event List. Disregard return value,
        // because in case of error hAlert will just remain NULL
        (void) CEML.eIterateActiveList(
                      psObj->hCEML,
                      bActiveEventIterator,
                      (void*)&hAlert
                          );
    }

    return hAlert;
}

/*****************************************************************************
*
*   CAL_bGetAlertsTunedStatus
*
* A fxn to retrieve the enabled status of alerts
* on the currently tuned channel.
*
*****************************************************************************/
BOOLEAN CAL_bGetAlertsTunedStatus(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked, bEnabled = FALSE;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        bEnabled = psObj->bAlertOnTunedChannel;

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return bEnabled;
}

/*****************************************************************************
*
*   CAL_eEnableAlertsTuned
*
* A fxn to enable/disable alerts on the currently tuned channel.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_eEnableAlertsTuned(
    CAL_OBJECT hCAL,
    BOOLEAN bEnable
        )
{
    CAL_OBJECT_STRUCT *psObj = (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // validate input
    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        psObj->bAlertOnTunedChannel = bEnable;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CAL_bGetAlertsFinishedStatus
*
* A fxn to retrieve the enabled status of alerts at content finish.
*
*****************************************************************************/
BOOLEAN CAL_bGetAlertsFinishedStatus(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bLocked, bEnabled = FALSE;

    // lock the CAL
    bLocked =
        SMSO_bLock((SMS_OBJECT)hCAL, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        bEnabled = psObj->bAlertOnFinishedContent;

        SMSO_vUnlock((SMS_OBJECT)hCAL);
    }

    return bEnabled;
}

/*****************************************************************************
*
*   CAL_eEnableAlertsFinished
*
* A fxn to enable/disable alerts at content finish.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_eEnableAlertsFinished(
    CAL_OBJECT hCAL,
    BOOLEAN bEnable
        )
{
    CAL_OBJECT_STRUCT *psObj = (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // validate input
    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        psObj->bAlertOnFinishedContent = bEnable;
        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CAL_hGetNextEndedEventAttributes
*
* A fxn to get the attributes (Channel Id, Artist name and Title) of the
* first available event from the CEML's List of ended events.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_hGetNextEndedEventAttributes(
    CAL_OBJECT hCAL,
    CHANNEL_ID *ptChannelId,
    STRING_OBJECT *phArtist,
    STRING_OBJECT *phTitle
        )
{
    CAL_OBJECT_STRUCT *psObj = (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // validate input
    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        // iterate the CEML's Active Event List
        eReturnCode = CEML.eGetEndedEventAttributes(
                      psObj->hCEML,
                      ptChannelId,
                      phArtist,
                      phTitle
                          );
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CAL_tCatAddOptions
*
*   This API is used to retrieve the category add options of this CAL.
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object which the caller wishes to
*              retrieve the category add options
*
*   Outputs:
*
*   The cat add options.
*
*****************************************************************************/
CAL_AUTO_ADD_OPTIONS CAL_tCatAddOptions (
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bValid;
    CAL_AUTO_ADD_OPTIONS tCategoryAddOptions = CAL_AUTO_ADD_OPTION_NONE;

    // Verify SMS Object (CAL)
    bValid =
        SMSO_bValid((SMS_OBJECT)hCAL);

    if (bValid == TRUE)
    {
        tCategoryAddOptions = psObj->tCatAddOptions;
    }

    return tCategoryAddOptions;
}

/*****************************************************************************
*
*   CAL_eAcknowledge
*
*   This API is used to acknowledge that the application received an alert
*   from this CAL. It should only be called from within the CAL Callback.
*
*   Inputs:
*
*   hCAL       A handle to a valid CAL object for which the caller wishes to
*              acknowledge that it received an alert.
*
*   Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_eAcknowledge(
    CAL_OBJECT hCAL
        )
{
    CAL_OBJECT_STRUCT *psObj =
        (CAL_OBJECT_STRUCT *)hCAL;
    BOOLEAN bOwner;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_API_NOT_ALLOWED;

    // verify
    bOwner =
        SMSO_bOwner((SMS_OBJECT)hCAL);

    if (bOwner == TRUE)
    {
        // ack
        eReturn = CEML.eAcknowledge(psObj->hCEML);
    }

    return eReturn;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/



/*****************************************************************************
*
*   hGetAlert
*
*****************************************************************************/
static CAL_ALERT_OBJECT hGetAlert(
    CAL_OBJECT_STRUCT *psObj,
    UN32 un32Flags,
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent
        )
{
    UN32 un32Options;
    CAL_ALERT_OBJECT hAlert;
    BOOLEAN bContentEnabled;

    un32Options = CAL_CONTENT.un32Options(hContent);


    // see if there is already an alert we can reuse
    hAlert = CAL_CONTENT_hGetAlert(hContent, hChannel);
    if (hAlert != CAL_ALERT_INVALID_OBJECT)
    {
        // there is already an alert for this content.
        return hAlert;
    }

    // if the cal is not enabled, or the content isn't enabled, don't
    // create an alert or use the CAL's Alert. no alert is required.
    // if there was an alert present on a disabled content or disabled cal,
    // it would have been found earlier and returned.
    // the final event must go through, though, even if disabled so that the
    // user can clean up any resources they created which are contained in the
    // CAL_CONTENT object's content argument
    bContentEnabled = CAL_CONTENT.bEnabled(hContent);
    if ( ( (bContentEnabled == TRUE) && (psObj->bEnabled == TRUE) ) ||
         ( un32Flags & ACTIVE_EVENT_FLAG_FINAL ) )
    {
        // see if we should use our alert
        if  ( ( (un32Flags & ACTIVE_EVENT_FLAG_END) &&
                (un32Options & SMS_EVENT_REGISTRATION_OPTION_END) )
           || (un32Flags & ACTIVE_EVENT_FLAG_FINAL) )
        {
            if (psObj->hAlert == CAL_ALERT_INVALID_OBJECT)
            {
                // if we haven't created the alert create it now.
                psObj->hAlert = CAL_ALERT_hCreate(CHANNEL_INVALID_OBJECT,
                                                  CAL_CONTENT_INVALID_OBJECT);
                SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "Created Alert (END or FINAL): un32Flags:%d, un32Options: %d", un32Flags, un32Options);
            }
            CAL_ALERT_bSet(psObj->hAlert, hChannel, hContent);
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "Alert on Channel %d", CHANNEL.tChannelId(hChannel));
            return psObj->hAlert;
        }

        // see if we should create a new alert
        if ( ( (un32Flags & ACTIVE_EVENT_FLAG_END) &&
               (un32Options & SMS_EVENT_REGISTRATION_OPTION_END) ) ||
             ( (un32Flags & ACTIVE_EVENT_FLAG_INITIAL) &&
               (un32Options & SMS_EVENT_REGISTRATION_OPTION_INITIAL) ) )
        {
            hAlert = CAL_ALERT_hCreate (hChannel, hContent);
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "Created Alert: un32Flags:%d, un32Options: %d", un32Flags, un32Options);
            return hAlert;
        }
    }
    // no alert required
    return CAL_ALERT_INVALID_OBJECT;
}

/*****************************************************************************
*
*   bAutoAdd
*
*****************************************************************************/
static BOOLEAN bAutoAdd(
    CAL_OBJECT_STRUCT *psObj,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags
        )
{
    CHANNEL_ID tAlertChanId, tTunedChanId;
    CHANNEL_OBJECT hChannel;
    BOOLEAN bContentEnabled, bReturn = FALSE;
    SMSAPI_RETURN_CODE_ENUM eChannelReturn;
    N16 n16Offset;
    CAL_CONTENT_OBJECT hContent;

    hContent = CAL_ALERT.hContent(hAlert);
    bContentEnabled = CAL_CONTENT.bEnabled(hContent);

    if ( (bContentEnabled == FALSE) ||
         (psObj->bEnabled == FALSE) ||
         (psObj->tCategoryId == CATEGORY_INVALID_ID) ||
         (psObj->tCatAddOptions & CAL_AUTO_ADD_OPTION_OFF) ||
         (un32Flags & ACTIVE_EVENT_FLAG_END) ||
         (un32Flags & ACTIVE_EVENT_FLAG_FINAL) )
    {
        // The cal_content is not enabled OR
        // the cal is not enabled OR
        // no valid category attached OR
        // the auto add feature is not desired OR
        // the content is ending
        SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: FALSE for %s content and flags %d", bContentEnabled?"enabled":"disabled", un32Flags);
        return FALSE;
    }

    hChannel = CAL_ALERT.hChannel(hAlert);

    if (hChannel == CHANNEL_INVALID_OBJECT)
    {
        SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: FALSE for invalid channel");
        return FALSE;
    }

    tAlertChanId = CHANNEL.tChannelId(hChannel);

    // see if this channel is already in the category by
    // searching the channel's category list
    eChannelReturn = CHANNEL.eCategoryOffset(hChannel,
                                             psObj->tCategoryId,
                                             &n16Offset);

    if ( eChannelReturn == SMSAPI_RETURN_CODE_SUCCESS )
    {
        // the channel is already in the category
        if (psObj->tCatAddOptions & CAL_AUTO_ADD_OPTION_UNIQUE)
        {
            // not unique, so don't add
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: FALSE for non-unique item in list");
            return FALSE;
        }
    }

    tTunedChanId = DECODER.tCurrentChannelId(psObj->hDecoder);

    if (tTunedChanId == tAlertChanId)
    {
        // content is occuring on the tuned channel
        if (psObj->tCatAddOptions & CAL_AUTO_ADD_OPTION_IF_TUNED)
        {
            // content is on the tuned channel
            // and we are configured to add in this case
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: TRUE for tuned channel");
            bReturn = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: FALSE for tuned channel");
        }
    }
    else
    {
        if (psObj->tCatAddOptions & CAL_AUTO_ADD_OPTION_IF_NOT_TUNED)
        {
            // content is not on the tuned channel
            // and we are configured to add in this case
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: TRUE for non-tuned channel");
            bReturn = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bAutoAdd: FALSE for non-tuned channel");
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   bCallCallback
*
*****************************************************************************/
static BOOLEAN bCallCallback(
    CAL_OBJECT_STRUCT *psObj,
    CAL_CONTENT_OBJECT hContent,
    UN32 un32Flags
        )
{
    BOOLEAN bContentEnabled, bCallCallback = FALSE;
    UN32 un32Options;

    // is there a callback provided
    if (psObj->vCALCallback == NULL)
    {
         // no callback to call
         return FALSE;
    }

    un32Options = CAL_CONTENT.un32Options(hContent);
    bContentEnabled = CAL_CONTENT.bEnabled(hContent);

    if ( un32Flags & ACTIVE_EVENT_FLAG_FINAL )
    {
        // the event is a final in which case we must call the callback
        // to give the caller a chance to clean up
        bCallCallback = TRUE;
        SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bCallCallback: TRUE for FINAL flag");
    }
    else if ( (bContentEnabled == TRUE) && (psObj->bEnabled == TRUE) )
    {
        // both the CAL and CAL_CONTENT are enabled
        if ( ( (un32Flags & ACTIVE_EVENT_FLAG_END) &&
               (un32Options & SMS_EVENT_REGISTRATION_OPTION_END) ) ||
             ( (un32Flags & ACTIVE_EVENT_FLAG_INITIAL) &&
               (un32Options & SMS_EVENT_REGISTRATION_OPTION_INITIAL) ) )
        {
            // at least one requested option is set
            bCallCallback = TRUE;
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bCallCallback: TRUE for flags %d and options %d", un32Flags, un32Options);
        }
        else
        {
            SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bCallCallback: FASE for flags %d and options %d", un32Flags, un32Options);
        }
    }
    else
    {
        // don't call callback
        bCallCallback = FALSE;
        SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 2, "bCallCallback: FASE for disabled content");
    }

    return bCallCallback;
}

/*****************************************************************************
*
*   vProcessAlert
*
*    Process the Alert after sending it to the apps callback
*
*****************************************************************************/
static void vProcessAlert(
    CAL_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags
        )
{
    if ( ( hAlert == psObj->hAlert) ||
         ( un32Flags & ACTIVE_EVENT_FLAG_END ) )
    {
        BOOLEAN bEnabled;

        // this alert is no longer needed.

        // remove alert from the CAL_CONTENT's list
        CAL_CONTENT_eRemoveFromAlertList(hContent, hChannel);

        // Collect info about ended event
        bEnabled = SEEK_CONTENT.bEnabled(hContent);

        if ( (un32Flags & ACTIVE_EVENT_FLAG_END) &&
             (psObj->bAlertOnFinishedContent == TRUE) &&
             (bEnabled == TRUE))
        {
            CHANNEL_ID tChannelId;
            STRING_OBJECT hArtist, hTitle;

            tChannelId = CHANNEL.tChannelId(hChannel);
            hArtist = CAL_CONTENT.hArtistText(hContent);
            hTitle = CAL_CONTENT.hTitleText(hContent);

            CEL_eAddEndedEvent(psObj->hCEML, tChannelId, hArtist, hTitle);
        }

        if (hAlert == psObj->hAlert)
        {
            // clear alert (but don't destroy because we'll reuse it later)
            CAL_ALERT_vClear(hAlert);
        }
        else
        {
            // destory alert
            CAL_ALERT_vDestroy(hAlert);
        }
    }
    else
    {
        // leave alert as is.
    }

    return;
}

/*******************************************************************************
*
*   bEqualCIDs
*
* An iterator callback function which determines if two CIDs are equal or not.
* If not FALSE is returned. Otherwise if they are TRUE is returned.
*
*****************************************************************************/
static BOOLEAN bEqualCIDs (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        )
{
    CAL_CEML_ITERATOR_STRUCT *psIterator =
        (CAL_CEML_ITERATOR_STRUCT *)pvEventIteratorArg;
    N16 n16Result;

    // Check if id's match. Use CID's equality compare method.
    n16Result = CID.n16Equal(hCID, psIterator->hId);
    if(n16Result == 0)
    {
        // Match!
        psIterator->bFound = TRUE;
        // Stop looking.
        return FALSE;
    }

    // Keep looking
    return TRUE;
}

/*******************************************************************************
*
*   bCALCEMLIterator
*
* The CAL iterator maps to the CEML list. This CEML callback translates to a
* CAL callback
*
*****************************************************************************/
static BOOLEAN bCALCEMLIterator (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        )
{
    CAL_TO_CEML_ITERATOR_STRUCT *psIterator =
        (CAL_TO_CEML_ITERATOR_STRUCT *)pvEventIteratorArg;
    CAL_CONTENT_OBJECT hContent = (CAL_CONTENT_OBJECT)pvContentArg;
    BOOLEAN bReturn;

    bReturn = psIterator->bCALCallback(hContent, psIterator->pvCALIteratorArg);

    return bReturn;
}

/*****************************************************************************
*
*   vCALEventCallback
*
*   This is the callback CEML calls.
*   In this fxn the CAL decides what to do and if necessary will do stuff
*   like add chan to attached cat, remove from attached cat, call CAL
*   Callback etc.
*
*****************************************************************************/
static void vCALEventCallback (
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventCallbackArg
        )
{
    // first thing we need to do is get our CONTENT_OBJECT
    // so we know what we're dealing with
    CAL_CONTENT_OBJECT hContent = (CAL_CONTENT_OBJECT)pvContentArg;
    BOOLEAN bLocked, bTakeAction = FALSE;
    CAL_ALERT_OBJECT hAlert = CAL_ALERT_INVALID_OBJECT;
    CAL_OBJECT hCAL = CAL_INVALID_OBJECT;
    CAL_OBJECT_STRUCT *psObj = NULL;

    do
    {
        // lock the CONTENT (which also locks the parent CAL)
        bLocked =
            SMSO_bLock((SMS_OBJECT)hContent, OSAL_OBJ_TIMEOUT_INFINITE);

        if (bLocked == FALSE)
        {
            // ERROR!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CAL_OBJECT_NAME": callback - invalid hContent.");
            return;
        }

        hCAL = (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)hContent);
        if (hCAL == CAL_INVALID_OBJECT)
        {
            // ERROR!
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                CAL_OBJECT_NAME": callback - invalid cal.");
            break;
        }

        psObj = (CAL_OBJECT_STRUCT *)hCAL;

        // get an alert handle
        hAlert = hGetAlert(psObj, un32Flags, hChannel, hContent);

        if (hAlert == CAL_ALERT_INVALID_OBJECT)
        {
            // did not get an alert
            break;
        }

        // auto add to attached category?
        bTakeAction = bAutoAdd(psObj, hAlert, un32Flags);
        printf(CAL_OBJECT_NAME": AutoAdd: %s\n",
               (bTakeAction==TRUE ? "Yes" : "No"));
        if (bTakeAction == TRUE)
        {
            // add channel to the category
            CAL_ALERT.eAddToCategory(hAlert, TRUE);
        }

        bTakeAction = bCallCallback(psObj, hContent, un32Flags);
        printf(CAL_OBJECT_NAME":Call Callback: %s\n",
               (bTakeAction==TRUE ? "Yes" : "No"));
        if ( bTakeAction == TRUE )
        {
            // yes they did. Call it.
            psObj->vCALCallback(
                hCAL,
                hAlert,
                un32Flags,
                psObj->pvCALCallbackArg
                    );
        }

        // process the alert, seeing if it should be destroyed, cleared,
        // or neither
        vProcessAlert( psObj, hChannel, hContent, hAlert, un32Flags );

    } while(0);

    if( ( un32Flags & ACTIVE_EVENT_FLAG_FINAL) == ACTIVE_EVENT_FLAG_FINAL &&
        ( hCAL != CAL_INVALID_OBJECT ) )
    {
            CAL_CONTENT_vDestroy(hContent);

            SMSO_vUnlock((SMS_OBJECT)hCAL);

        // It isn't necesary to ack on the final so we can return.
        return;
    }

    // the psObj != NULL check is not technically required because
    // psObj will never be null as bTakeAction == TRUE
    // but it is added here to make static error checkers happy.
    if ( ( bTakeAction == FALSE )
            || ( (psObj != NULL) && (psObj->bAutoAck == TRUE) )
        )
    {
        // Since we didn't call the callback, the user of this cal did not
        // receive an alert and therefore could not ack it. so we need to
        // -OR-
        // The CAL is configured to auto ack

        CEML.eAcknowledge(hCEML);
    }

    // unlock the CAL if we have a valid handle
	if ( hCAL != CAL_INVALID_OBJECT )
	{
		SMSO_vUnlock((SMS_OBJECT)hCAL);
	}

    return;
}

/*****************************************************************************
*
*   bActiveEventIterator
*
*****************************************************************************/
static BOOLEAN bActiveEventIterator(
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventIteratorArg
        )
{
    CAL_ALERT_OBJECT *phAlert =
        (CAL_ALERT_OBJECT *)pvEventIteratorArg;
    CAL_CONTENT_OBJECT hContent = 
        (CAL_CONTENT_OBJECT)pvContentArg;

    SMSAPI_DEBUG_vPrint(CAL_OBJECT_NAME, 5, "Iterating Active event on channel %d, flags %x",
        CHANNEL.tChannelId(hChannel), un32Flags);

    // acknowledge the event (removes it from the active list)
    CEML.eAcknowledge(hCEML);

    // No need to search for matching alert for END
    if (((un32Flags & ACTIVE_EVENT_FLAG_END) != ACTIVE_EVENT_FLAG_END))
    {
        // Get alert for this content / channel
        *phAlert = CAL_CONTENT_hGetAlert(
            hContent,
            hChannel );
    }

    if (*phAlert == CAL_ALERT_INVALID_OBJECT)
    {
        // There is no matching content event found or it is END event
        // which we do not report. Continue iteration
        return TRUE;
    }
    else
    {
        // only want one valid event, so stop iterating
        return FALSE;
    }
}
