/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:CAL_ALERT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "ccache.h"
#include "channel_obj.h"
#include "cas.h"
#include "cal_content.h"
#include "cal_alert.h"
#include "_cal_alert.h"

#include "sms_api_debug.h"
/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   eAddToCategory
*
* Adds this ALERT's CHANNEL to the CATEGORY.
*
* Inputs:
*
*   hAlert - A handle to a valid CAL_ALERT_OBJECT object whose associated
*            CHANNEL is to be added  to the attached CATEGORY of the owner CAL.
*   bAddDuplicate - Boolean value to force a duplicate channel to be added.
*            If True a channel will always be added a category regardless of
*            whether or not it was previously in the category.  If it is false
*            no duplicate channels will be added
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddToCategory (
    CAL_ALERT_OBJECT hAlert,
    BOOLEAN bAddDuplicate
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    DECODER_OBJECT hDecoder;
    CAL_OBJECT hCAL = CAL_INVALID_OBJECT;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        BOOLEAN bLocked;

        // lock the content object to get CAL parent handle
        bLocked = SMSO_bLock((SMS_OBJECT)psObj->hContent,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if (bLocked == TRUE)
        {
            hCAL = (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)psObj->hContent);
            SMSO_vUnlock((SMS_OBJECT)psObj->hContent);
        }

        // get the decoder handle
        hDecoder = CAL_hDecoder(hCAL);

        // if hDecoder is valid, that implies hCAL was valid
        if ( (hDecoder != DECODER_INVALID_OBJECT) &&
             (psObj->hContent != CAL_CONTENT_INVALID_OBJECT) &&
             (psObj->hChannel != CHANNEL_INVALID_OBJECT))
        {
            CATEGORY_ID tCategoryId;
            CHANNEL_ID tChannelId;
            size_t tNumChannelsInCat;
            CAL_AUTO_ADD_OPTIONS tCategoryAddOptions;

            tCategoryId = CAL.tCategoryId(hCAL);
            if (tCategoryId == CATEGORY_INVALID_ID)
            {
                // category isn't attached to CAL
                return SMSAPI_RETURN_CODE_ERROR;
            }

            tCategoryAddOptions = CAL_tCatAddOptions(hCAL);
            tNumChannelsInCat = CATEGORY.tSize(hDecoder, tCategoryId);

            tChannelId = CHANNEL.tChannelId(psObj->hChannel);

            // If no duplicate should be added search the category.
            if (FALSE == bAddDuplicate)
            {
                N16 n16Offset;
                SMSAPI_RETURN_CODE_ENUM eChannelReturn;

                eChannelReturn = CHANNEL.eCategoryOffset(psObj->hChannel,
                                                         tCategoryId,
                                                         &n16Offset);

                if ( eChannelReturn == SMSAPI_RETURN_CODE_SUCCESS )
                {
                    // the channel is already in the category
                    return SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
                }
            }

            if ( (tCategoryAddOptions & CAL_AUTO_ADD_OPTION_BOTTOM) ||
                 ( tNumChannelsInCat == 0 ) )
            {
                // if we are adding to the bottom, or there are no channels
                // currently in the category
                eReturn = CATEGORY.eInsertNewChannel(
                            hDecoder,
                            tCategoryId,
                            tChannelId
                                );
            }
            else if (tCategoryAddOptions & CAL_AUTO_ADD_OPTION_TOP)
            {
                // if we are adding to the top
                CAL_ALERT_ITERATE_CATEGORY_STRUCT sIterateStruct;

                sIterateStruct.tChannelId = tChannelId;
                // tNumIterations won't be used in the iterator.
                // we're just reusing a iterator structure
                // but we can initialize it to 1, so nothing is uninitialized
                sIterateStruct.tNumIterations = 1;

                // adding to the top of the category has to be done via
                // the category iterator.
                eReturn = CATEGORY.eIterate(
                              hDecoder,
                              tCategoryId,
                              n16AddToTopIterator,
                              (void *)&sIterateStruct
                                  );
            }
            else
            {
                // inavlid options. should never happen, but....
                return SMSAPI_RETURN_CODE_ERROR;
            }

            // successfully added?
            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {
                // yes
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   eTune
*
* This API is used to tune to the channel on which the alert is active.
*
* Inputs:
*
*   hAlert - handle to a valid CAL_ALERT_OBJECT object whose associated
*       CHANNEL is to be tuned to.
*   bLockOverride - This argument is used to signify if the caller wishes to
*       override the lock attribute of the channel associated with this alert.
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success, SMSAPI_RETURN_CODE_ERROR on error.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eTune (
    CAL_ALERT_OBJECT hAlert,
    BOOLEAN bLockOverride
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        do
        {
            DECODER_OBJECT hDecoder = DECODER_INVALID_OBJECT;
            CAL_OBJECT hCAL = CAL_INVALID_OBJECT;
            BOOLEAN bLocked;

            if(psObj->hChannel == CHANNEL_INVALID_OBJECT)
            {
                // Error!
                break;
            }

            // get the decoder handle

            // first we get the CAL's handle
            bLocked = SMSO_bLock((SMS_OBJECT)psObj->hContent, 
                OSAL_OBJ_TIMEOUT_INFINITE);
            if (bLocked == TRUE)
            {
                hCAL = (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)psObj->hContent);
                SMSO_vUnlock((SMS_OBJECT)psObj->hContent);
            }

            // now we can get the decoder handle
            hDecoder = CAL_hDecoder(hCAL);

            if (hDecoder != DECODER_INVALID_OBJECT)
            {
                SERVICE_ID tChannelId;

                // get the channel id
                tChannelId = CHANNEL.tChannelId(psObj->hChannel);

                // tune
                eReturnCode = DECODER.eTuneDirect (
                            hDecoder,
                            tChannelId,
                            bLockOverride
                                );
            }
        } while (FALSE);
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   hChannel
*
* This API is used to find the CHANNEL object handle which caused this
* CAL_ALERT_OBJECT to be generated.
*
* Inputs:
*
*   hAlert - A handle to a valid CAL_ALERT_OBJECT for which the caller wishes
*            to learn which CHANNEL object it is associated with.
*
* Outputs:
*
*   a handle to a valid CHANNEL_OBJECT or CHANNEL_INVALID_OBJECT on error
*
*****************************************************************************/
static CHANNEL_OBJECT hChannel (
    CAL_ALERT_OBJECT hAlert
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        hChannel = psObj->hChannel;
    }

    return hChannel;
}

/*****************************************************************************
*
*   hContent
*
* This API is used to find the CAL_CONTENT object handle which caused this
* CAL_ALERT_OBJECT to be generated.
*
* Inputs:
*
*   hAlert - A handle to a valid CAL_ALERT_OBJECT for which the caller wishes
*            to learn which CAL_CONTENT object it is associated with.
*
* Outputs:
*
*   A handle to a valid CAL_CONTENT_OBJECT or
*   CAL_CONTENT_INVALID_OBJECT on error
*
*****************************************************************************/
static CAL_CONTENT_OBJECT hContent (
    CAL_ALERT_OBJECT hAlert
        )
{
    BOOLEAN bValid;
    CAL_CONTENT_OBJECT hContent = CAL_CONTENT_INVALID_OBJECT;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        CAL_ALERT_OBJECT_STRUCT *psObj =
            (CAL_ALERT_OBJECT_STRUCT *)hAlert;
        hContent = psObj->hContent;
    }

    return hContent;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   CAL_ALERT_hCreate
*
*****************************************************************************/
CAL_ALERT_OBJECT CAL_ALERT_hCreate (
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;

    do
    {
        // Create a CAL_ALERT object
        psObj = (CAL_ALERT_OBJECT_STRUCT *)
                    SMSO_hCreate(
                        CAL_ALERT_OBJECT_NAME,
                        sizeof(CAL_ALERT_OBJECT_STRUCT),
                        (SMS_OBJECT)SMS_INVALID_OBJECT,
                        FALSE
                            );
        if(psObj == NULL)
        {
            // Error!
            break;
        }

        // set the alert's channel and content handles and flags
        bOk = CAL_ALERT_bSet((CAL_ALERT_OBJECT)psObj, hChannel, hContent);
        if(bOk == FALSE)
        {
            // set to invalid to make sure no channel gets unregistered.
            psObj->hChannel = CHANNEL_INVALID_OBJECT;

            // destroy the Alert we just created
            CAL_ALERT_vDestroy((CAL_ALERT_OBJECT)psObj);

            break;
        }

        // a dummy alert object can be created that has an invalid content
        // handle. in this case, don't try to add to the content object's
        // alert list
        if (hContent != CAL_CONTENT_INVALID_OBJECT)
        {
            eReturnCode = CAL_CONTENT_eAddToAlertList(hContent,
                                                      (CAL_ALERT_OBJECT)psObj);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {

                // destroy the Alert we just created
                CAL_ALERT_vDestroy((CAL_ALERT_OBJECT)psObj);
                break;
            }
        }
        printf(CAL_ALERT_OBJECT_NAME": created Alert %p\n", psObj);
        // Return CAL_ALERT object to caller
        return (CAL_ALERT_OBJECT)psObj;

    } while(0);

    return CAL_ALERT_INVALID_OBJECT;
}

/*****************************************************************************
*
*   CAL_ALERT_vDestroy
*
*****************************************************************************/
void CAL_ALERT_vDestroy (
    CAL_ALERT_OBJECT hAlert
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;

    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        // remove the alert from the category
        CAL_ALERT_eRemoveFromCategory(hAlert);

        // we're done with the channel, so unregister
        CHANNEL_vUnregisterNotification(psObj->hChannel, NULL);
        // destroy the Alert
        SMSO_vDestroy((SMS_OBJECT)psObj);

        printf(CAL_ALERT_OBJECT_NAME": destroyed Alert %p\n", psObj);
    }

    return;
}

/*****************************************************************************
*
*   CAL_ALERT_eRemoveFromCategory
*
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM CAL_ALERT_eRemoveFromCategory(
    CAL_ALERT_OBJECT hAlert
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        // use category iterator to look for channel
        DECODER_OBJECT hDecoder;
        CAL_OBJECT hCAL;
        CATEGORY_ID tCategoryId;
        CAL_ALERT_ITERATE_CATEGORY_STRUCT sIterateStruct;
        CAL_ITERATOR_STRUCT sCALIteratorStruct;

        // get the cal handle
        hCAL = (CAL_OBJECT)SMSO_hParent((SMS_OBJECT)psObj->hContent);

        // get the category id from the CAL
        tCategoryId = CAL.tCategoryId(hCAL);
        if (tCategoryId == CATEGORY_INVALID_ID)
        {
            // category isn't attached to CAL
            return SMSAPI_RETURN_CODE_ERROR;
        }

        hDecoder = CAL_hDecoder(hCAL);


        // get the id of the channel this alert is on
        sIterateStruct.tServiceId = CHANNEL.tServiceId(psObj->hChannel);
        sIterateStruct.tNumIterations =
            CATEGORY.tSize( hDecoder, tCategoryId);

        if ( (sIterateStruct.tNumIterations == 0) ||
             (sIterateStruct.tServiceId == SERVICE_INVALID_ID) )
        {
            // there aren't any channels in the category,
            // so don't bother to search it.

            // since this channel obviously isn't in the category,
            // don't need to remove
            return SMSAPI_RETURN_CODE_SUCCESS;
        }

        // before we can remove this channel from the category, we need to
        // make sure there aren't OTHER matches of registered content occuring
        // on it. For example if LEAGUE was registered and a team in the LEAGUE
        // was registered, a game is playing with tha registered team in it
        // and then the team is unregistered. In that case we can't remove the
        // channel when unregistering the team, because there is still a league match

        // set up for iteration
        sCALIteratorStruct.bFound = FALSE;
        sCALIteratorStruct.hChannel = psObj->hChannel;

        eReturnCode = CAL.eIterate(hCAL, bCALIterator, &sCALIteratorStruct);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) && (sCALIteratorStruct.bFound == FALSE))
        {
            SMSAPI_RETURN_CODE_ENUM eResult;

            // No additional alerts on this channel, removing from cat
            puts("No additional alerts on this channel, removing from cat");

            // iterate the decoder's category list,
            // looking for the attached category id.
            eResult = CATEGORY.eIterate(
                        hDecoder,
                        tCategoryId,
                        n16RemoveChanByIdIterateFxn,
                        (void *)&sIterateStruct
                            );

            if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
            {
                eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
            }
        }
        else
        {
            puts("Additional alerts on this channel, NOT removing from cat");
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   CAL_ALERT_vClear
*
*   Used to clear the Alert Object (for later re-use)
*
*****************************************************************************/
void CAL_ALERT_vClear(
    CAL_ALERT_OBJECT hAlert
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid == TRUE)
    {
        // remove the alert from the category
        CAL_ALERT_eRemoveFromCategory(hAlert);

        // we're done with the channel, so unregister
        CHANNEL_vUnregisterNotification(psObj->hChannel, NULL);

        // clear the handles
        psObj->hContent = CAL_CONTENT_INVALID_OBJECT;
        psObj->hChannel = CHANNEL_INVALID_OBJECT;

    }

    return;
}

/*****************************************************************************
*
*   CAL_ALERT_bSet
*
*****************************************************************************/
BOOLEAN CAL_ALERT_bSet(
    CAL_ALERT_OBJECT hAlert,
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent
        )
{
    CAL_ALERT_OBJECT_STRUCT *psObj =
        (CAL_ALERT_OBJECT_STRUCT *)hAlert;
    BOOLEAN bValid, bOk;

    // Verify SMS Object (CAL_ALERT_OBJECT)
    bValid =
        SMSO_bValid((SMS_OBJECT)hAlert);

    if (bValid != TRUE)
    {
        SMSAPI_DEBUG_vPrint(CAL_ALERT_OBJECT_NAME, 2, "CAL_ALERT_bSet: Invalid hAlert handle (%p)", hAlert);
        return FALSE;
    }
    else
    {
        // set the handles and flags
        psObj->hContent = hContent;
        psObj->hChannel = hChannel;

        SMSAPI_DEBUG_vPrint(CAL_ALERT_OBJECT_NAME, 2, "CAL_ALERT_bSet: Setting hContent %p and Channel %d", 
            hContent, CHANNEL.tChannelId(hChannel));
        // only register if valid channel handle is given
        // (an invalid channel handle is used when content
        // is being removed via FINAL event)
        if (hChannel != CHANNEL_INVALID_OBJECT)
        {
            bOk = CHANNEL_bRegisterNotification(psObj->hChannel,
                                                CHANNEL_OBJECT_EVENT_NONE);
            SMSAPI_DEBUG_vPrint(CAL_ALERT_OBJECT_NAME, 2, "CAL_ALERT_bSet: CHANNEL_bRegisterNotification returned %d", bOk);
        }
        else
        {
            bOk = TRUE;
            SMSAPI_DEBUG_vPrint(CAL_ALERT_OBJECT_NAME, 2, "CAL_ALERT_bSet: returning %d", bOk);
        }
    }

    return bOk;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   n16AddToTopIterator
*
*****************************************************************************/
static N16 n16AddToTopIterator(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    CAL_ALERT_ITERATE_CATEGORY_STRUCT *psObj;

    // Verify Inputs. we need a valid category, a valid pointer to a struct in
    // pvIterateArg, hChannel could possibly by CHANNEL_INVALID_OBJECT so don't
    // check for that. This function won't use the channel object passed in.
    if ( ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pvIterateArg == NULL ) )
    {
        return 0;
    }

    psObj = (CAL_ALERT_ITERATE_CATEGORY_STRUCT *)pvIterateArg;

    CATEGORY.eInsertBeforeChannel(hCategory, psObj->tChannelId);

    // we're done (we only need to iterate the first channel)
    return 0;
}

/*****************************************************************************
*
*   n16RemoveChanByIdIterateFxn
*
*****************************************************************************/
static N16 n16RemoveChanByIdIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        )
{
    CAL_ALERT_ITERATE_CATEGORY_STRUCT *psObj;
    SERVICE_ID tServiceId;

    // Verify Inputs, we need a valid category, a valid pointer to a struct in
    // pvIterateArg, hChannel could possibly by CHANNEL_INVALID_OBJECT so don't
    // check for that
    if ( ( hCategory == CATEGORY_INVALID_OBJECT ) ||
         ( pvIterateArg == NULL ) )
    {
        return 0;
    }

    psObj = (CAL_ALERT_ITERATE_CATEGORY_STRUCT *)pvIterateArg;
    tServiceId = CHANNEL.tServiceId(hChannel);

    // is this is the item we want to remove?
    if ( psObj->tServiceId == tServiceId )
    {
        // yes, remove it

        CATEGORY.eRemoveChannel( hCategory );

        // Regardless of the return, we tried

        return 0;
    }

    psObj->tNumIterations--;
    if (psObj->tNumIterations == 0)
    {
        // we've checked everything
        return 0;
    }

    // Keep moving in a forward direction
    return 1;
}

/*****************************************************************************
*
*   bCALIterator
*
*****************************************************************************/
static BOOLEAN bCALIterator(
    CAL_CONTENT_OBJECT hContent,
    void *pvContentIteratorArg
        )
{
    CAL_ITERATOR_STRUCT *psStruct = (CAL_ITERATOR_STRUCT *)pvContentIteratorArg;
    BOOLEAN bFound;

    bFound =
        CAL_CONTENT_bLookForChannelInAlertList(hContent, psStruct->hChannel);

    // did
    if (bFound == TRUE)
    {
        psStruct->bFound = TRUE;
        // we can stop iterating. We found one.
        return FALSE;
    }

    // keep on iteratin'
    return TRUE;

}
