/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the browse object implementation for the
 *  Satellite Module Services (SMS).
 *
 ******************************************************************************/

#include "standard.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "decoder_obj.h"
#include "ccache.h"
#include "category_obj.h"
#include "browse_obj.h"
#include "_browse_obj.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       BROWSE_hCreate
*
*       This API creates a BROWSE object.
*
*       Inputs:
*           hOwner - A handle to the object which is requesting the creation
*               a BROWSE object.
*           hDecoder - The DECODER upon which to perform browse operations.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
BROWSE_OBJECT BROWSE_hCreate (
    SMS_OBJECT hOwner,
    DECODER_OBJECT hDecoder
        )
{
    BROWSE_OBJECT hBrowse =
        BROWSE_INVALID_OBJECT;
    BROWSE_OBJECT_STRUCT *psObj = NULL;
    BOOLEAN bOk;

    // Validate the owner object
    bOk = SMSO_bValid(hOwner);
    if (bOk == FALSE)
    {
        return BROWSE_INVALID_OBJECT;
    }

    // Validate the decoder
    bOk = SMSO_bValid((SMS_OBJECT)hDecoder);
    if (bOk == FALSE)
    {
        return BROWSE_INVALID_OBJECT;
    }
    else // Build the object now
    {
        BOOLEAN bLocked;
        static UN32 un32InitInstance = 0;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        do
        {
            // Construct a unique name for the browse object
            snprintf( &acName[0], sizeof(acName),
                      BROWSE_OBJECT_NAME":%u",
                      un32InitInstance++ );

            // Create an instance of this object
            psObj = (BROWSE_OBJECT_STRUCT *)
                SMSO_hCreate(
                    &acName[0],
                    sizeof(BROWSE_OBJECT_STRUCT),
                    hOwner,  // Child
                    FALSE );
            if(psObj == NULL)
            {
                // Error!
                break;
            }

            // Initialize our attributes
            psObj->hOwner = hOwner;
            psObj->hDecoder = hDecoder;

            // Lock the decoder, to get the ccache handle
            bLocked =
                SMSO_bLock(
                    (SMS_OBJECT)psObj->hDecoder,
                    OSAL_OBJ_TIMEOUT_INFINITE
                        );
            if (bLocked == TRUE)
            {
                psObj->hCCache = DECODER_hCCache( hDecoder);
                SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
            }

            // Check the ccache handle
            if (psObj->hCCache == CCACHE_INVALID_OBJECT)
            {
                break;
            }

            // Initialize browse type
            psObj->eBrowseType = BROWSE_TYPE_INVALID;

            // Set the channel browse style
            bOk = BROWSE_bSetChannelBrowseStyle (
                (BROWSE_OBJECT)psObj,
                (BROWSE_CHANNEL_COMPARE_HANDLER)NULL,
                NULL );

            if (bOk != TRUE)
            {
                break;
            }

            // Set the category browse style
            bOk = BROWSE_bSetCategoryBrowseStyle (
                (BROWSE_OBJECT)psObj,
                (BROWSE_CATEGORY_COMPARE_HANDLER)NULL,
                NULL );

            if (bOk != TRUE)
            {
                break;
            }

            hBrowse = (BROWSE_OBJECT)psObj;
            return hBrowse;
        }
        while (0);
    }

    BROWSE_vDestroy((BROWSE_OBJECT)psObj);
    return BROWSE_INVALID_OBJECT;
}

/*****************************************************************************
*
*       BROWSE_vDestroy
*
*****************************************************************************/
void BROWSE_vDestroy (
    BROWSE_OBJECT hBrowse
        )
{
    BOOLEAN bOwner;

    // Is there anything to do?
    if (hBrowse == BROWSE_INVALID_OBJECT)
    {
        // Nothing to do here
        return;
    }

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOwner == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Reset browse type
        psObj->eBrowseType = BROWSE_TYPE_INVALID;

        // Clear handlers
        psObj->bBrowseCategoryCompareHandler = NULL;
        psObj->pvBrowseCategoryCompareArg = NULL;
        psObj->bBrowseChannelCompareHandler = NULL;
        psObj->pvBrowseChannelCompareArg = NULL;

        // Clear object handles
        psObj->hOwner = SMS_INVALID_OBJECT;
        psObj->hDecoder = DECODER_INVALID_OBJECT;
        psObj->hCCache = CCACHE_INVALID_OBJECT;

        // Clear reference information
        psObj->tReferenceChannelId = CHANNEL_INVALID_ID;
        psObj->tReferenceCategoryId = CATEGORY_INVALID_ID;
        psObj->n16ChannelCategoryOffset = 0;

        // Destroy object
        SMSO_vDestroy((SMS_OBJECT)hBrowse);
    }

    return;
}

/*****************************************************************************
*
*       BROWSE_eGetMode
*
*       Queries the current operating mode of this browse object.
*
*       Inputs:
*           hBrowse - A valid handle to a BROWSE object from which
*               to extract the current operating mode.
*
*       Outputs:
*           An appropriate value of type BROWSE_TYPE_ENUM
*
*****************************************************************************/
BROWSE_TYPE_ENUM BROWSE_eGetMode (
    BROWSE_OBJECT hBrowse
        )
{
    BOOLEAN bOwner;
    BROWSE_TYPE_ENUM eBrowseType = BROWSE_TYPE_INVALID;

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOwner == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Extract the current browse mode
        eBrowseType = psObj->eBrowseType;
    }

    return eBrowseType;
}

/*****************************************************************************
*
*       BROWSE_bSetMode
*
*       Sets the current operating mode of this browse object by providing
*       the explicit mode value and starting reference Ids.
*
*       Inputs:
*           hBrowse - A valid handle to a BROWSE object to which the
*               new mode is to be applied.
*           eBrowseType - The browse mode in which to operate
*           tReferenceChanndlId - The initial channel Id to use
*               when browsing ALL_CHANNELS mode
*           tReferenceCategoryId - The initial category Id to use
*               when browsing CATEGORY mode
*
*       Outputs:
*           TRUE on success, FALSE on error
*
*****************************************************************************/
BOOLEAN BROWSE_bSetMode (
    BROWSE_OBJECT hBrowse,
    BROWSE_TYPE_ENUM eBrowseType,
    CHANNEL_ID tReferenceChannelId,
    CATEGORY_ID tReferenceCategoryId
        )
{
    BOOLEAN bOk;

    // Validate object ownership
    bOk = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOk == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;
        N16 n16InitialChannelOffset = -1;

        // Lock the decoder
        bOk =
            SMSO_bLock((SMS_OBJECT)psObj->hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
        if (bOk == TRUE)
        {
            CATEGORY_OBJECT hReferenceCategory;

            // Determine the offset of the provided
            // channel within the category or cache,
            // based on the browse type requested
            if (eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
            {
                CHANNEL_OBJECT hReferenceChannel;

                n16InitialChannelOffset = 0;

                // Get the reference channel handle and it's
                // offset within the ccache.
                hReferenceChannel =  CCACHE_hChannel(
                    psObj->hCCache, &tReferenceChannelId,
                    &n16InitialChannelOffset );

                // Get the reference category handle --
                // use the channel's broadcast category
                hReferenceCategory = CHANNEL.hCategory(
                    hReferenceChannel, 0);

                // Verify we have a valid channel handle
                if (hReferenceChannel != CHANNEL_INVALID_OBJECT)
                {
                    if (hReferenceCategory != CATEGORY_INVALID_OBJECT)
                    {
                        // Get the reference category Id
                        tReferenceCategoryId = CATEGORY.tGetCategoryId(
                            hReferenceCategory );
                    }
                    else
                    {
                        // A valid category is not required here
                        tReferenceCategoryId = CATEGORY_INVALID_ID;
                    }
                }
            }
            else if (eBrowseType == BROWSE_TYPE_CATEGORY)
            {
                // Does the caller have a category Id which
                // looks like it may be valid?
                if (tReferenceCategoryId != CATEGORY_INVALID_ID)
                {
                    // Get the handle for the category Id provided
                    hReferenceCategory = CCACHE_hCategory(
                        psObj->hCCache, &tReferenceCategoryId, 0);

                    if (hReferenceCategory != CATEGORY_INVALID_OBJECT)
                    {
                        // Get this channel's offset from within
                        // the provided category
                        n16InitialChannelOffset = CATEGORY_n16GetIndexByChanId(
                            hReferenceCategory, tReferenceChannelId );
                    }
                }
                else
                {
                    // The caller is telling us the
                    // reference channel has no category
                    // Just let it pass
                    n16InitialChannelOffset = 0;
                }
            }
            SMSO_vUnlock((SMS_OBJECT)psObj->hDecoder);
        }

        // Did we get a valid offset?
        if ( (n16InitialChannelOffset >= 0) ||
             (eBrowseType == BROWSE_TYPE_CATEGORY) )
        {
            // Set the browse type
            psObj->eBrowseType = eBrowseType;

            // Set the reference Ids
            psObj->tReferenceChannelId = tReferenceChannelId;
            psObj->tReferenceCategoryId = tReferenceCategoryId;

            // Determine channel offset
            psObj->n16ChannelCategoryOffset = n16InitialChannelOffset;

            // Success
            return TRUE;
        }
    }

    // Error
    return FALSE;
}

/*****************************************************************************
*
*       BROWSE_bRefresh
*
*       Refreshes internal offsets of the BROWSE object
*
*       Inputs:
*           hBrowse - A valid handle to a BROWSE object from which the
*               Ids are to be retrieved.
*
*       Outputs:
*           TRUE on success, FALSE on error
*
*****************************************************************************/
BOOLEAN BROWSE_bRefresh (
    BROWSE_OBJECT hBrowse
        )
{
    BOOLEAN bOk;

    // Validate object ownership
    bOk = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOk == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        bOk = BROWSE_bSetMode(
            hBrowse,
            psObj->eBrowseType,
            psObj->tReferenceChannelId,
            psObj->tReferenceCategoryId);
    }

    return bOk;
}

/*****************************************************************************
*
*       BROWSE_bGetBrowsed
*
*       Gets the current browsed channel/category Ids.
*
*       Inputs:
*           hBrowse - A valid handle to a BROWSE object from which the
*               Ids are to be retrieved.
*           *ptBrowsedChannelId - A valid pointer to a CHANNEL_ID
*               which will store the currently browsed channel Id
*           *ptBrowsedCategoryId - A valid pointer to a CATEGORY_ID
*               which will store the currently browsed category Id
*
*       Outputs:
*           TRUE on success, FALSE on error
*
*****************************************************************************/
BOOLEAN BROWSE_bGetBrowsed (
    BROWSE_OBJECT hBrowse,
    CHANNEL_ID *ptBrowsedChannelId,
    CATEGORY_ID *ptBrowsedCategoryId
        )
{
    BOOLEAN bOwner;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    CATEGORY_ID tCategoryId = CATEGORY_INVALID_ID;

    // Verify inputs (all pointers can't be NULL)
    if ((ptBrowsedChannelId == NULL) &&
        (ptBrowsedCategoryId == NULL))
    {
        return FALSE;
    }

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOwner == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Extract the reference Ids
        tChannelId = psObj->tReferenceChannelId;
        tCategoryId = psObj->tReferenceCategoryId;
    }

    // Copy the values out
    if (ptBrowsedChannelId != NULL)
    {
        *ptBrowsedChannelId = tChannelId;
    }

    if (ptBrowsedCategoryId != NULL)
    {
        *ptBrowsedCategoryId = tCategoryId;
    }

    return bOwner;
}

/*****************************************************************************
*
*       BROWSE_n16GetBrowsedChannelOffset
*
*       Gets the offset of the current browsed channel within the
*       category.
*
*       Inputs:
*           hBrowse - A valid handle to a BROWSE object from which the
*               offset is to be retrieved.
*
*       Outputs:
*           An offset expressed as an N16, with N16_MIN returned on error.
*
*****************************************************************************/
N16 BROWSE_n16GetBrowsedChannelOffset (
    BROWSE_OBJECT hBrowse
        )
{
    N16 n16Offset = N16_MIN;
    BOOLEAN bOwner;

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOwner == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        n16Offset = psObj->n16ChannelCategoryOffset;
    }

    return n16Offset;
}

/*****************************************************************************
*
*       BROWSE_bSetChannelBrowseStyle
*
*       This API is used set the current channel browsing style for this
*       browse object. If a handler is not provided by the caller, the default
*       functionality is to include all channels in the browse.  If a handler is
*       provided, that will be used while browsing channels to determine if that
*       channel is to be provided or not.
*
*       Inputs:
*           hBrowse - A handle to a valid BROWSE object for which the caller
*               wishes to set the current channel browse style.
*           bBrowseChannelCompareHandler - The optional compare handler that
*               will be used during the browsing process to determine if a
*               channel should or should not be included.
*           *pvBrowseChannelCompareArg - The pointer to an optional argument
*               provided to the compare handler to help make the determination
*               of whether or not the channel is to be included in the browse.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
BOOLEAN BROWSE_bSetChannelBrowseStyle (
    BROWSE_OBJECT hBrowse,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        )
{
    BOOLEAN bOwner;

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Set the channel browse style
        psObj->bBrowseChannelCompareHandler =
            bBrowseChannelCompareHandler;
        psObj->pvBrowseChannelCompareArg =
            pvBrowseChannelCompareArg;
    }

    return bOwner;
}

/*****************************************************************************
*
*       BROWSE_bSetCategoryBrowseStyle
*
*       This API is used set the current category browsing style for this
*       browse object. If a handler is not provided by the caller, the default
*       functionality is to include all categories in the browse.  If a handler
*       is provided, that will be used while browsing categories to determine
*       if that category is to be provided or not.
*
*       Inputs:
*           hBrowse - A handle to a valid BROWSE object for which the caller
*               wishes to set the current category browse style.
*           bBrowseCategoryCompareHandler - The optional compare handler that
*               will be used during the browsing process to determine if a
*               category should or should not be included.
*           *pvBrowseCategoryCompareArg - The pointer to an optional argument
*               provided to the compare handler to help make the determination
*               of whether or not the category is to be included in the browse.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
BOOLEAN BROWSE_bSetCategoryBrowseStyle (
    BROWSE_OBJECT hBrowse,
    BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
    void *pvBrowseCategoryCompareArg
        )
{
    BOOLEAN bOwner;

    // Validate object ownership
    bOwner = SMSO_bOwner((SMS_OBJECT)hBrowse);
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Set the category browse style
        psObj->bBrowseCategoryCompareHandler =
            bBrowseCategoryCompareHandler;
        psObj->pvBrowseCategoryCompareArg =
            pvBrowseCategoryCompareArg;
    }

    return bOwner;
}

/*****************************************************************************
*
*       BROWSE_bBrowse
*
*       This API is used to perform a browse operation based upon the
*       current state of the browse object and a provided direction.
*
*       Inputs:
*           hBrowse - A handle to a valid BROWSE object which the caller
*               wishes to utilize.
*           eDirection - The direction in which to perform the browse.
*           bBrowseChannel - A flag indicating if the direction provided
*               refers to a channel or category browse.  This argument is
*               ignored for browse objects operating as BROWSE_TYPE_ALL_CHANNELS.
*           *ptChannelId - A valid pointer to a CHANNEL_ID which is to store
*               the resultant channel Id
*           *ptCategoryId - A valid pointer to a CATEGORY_ID which is to
*               store the resultant category Id
*           bBrowsingChanlist - flag indicating if the browse is on a channellist
*               (as opposed to decoder). channellist can browse to empty
*               category, a decoder cannot.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
BOOLEAN BROWSE_bBrowse (
    BROWSE_OBJECT hBrowse,
    SMSAPI_DIRECTION_ENUM eDirection,
    BOOLEAN bBrowseChannel,
    CHANNEL_ID *ptChannelId,
    CATEGORY_ID *ptCategoryId,
    BOOLEAN bBrowsingChanlist
        )
{
    BOOLEAN bOk;

    // Verify inputs
    if ( (ptChannelId == NULL) ||
         (ptCategoryId == NULL))
    {
        return FALSE;
    }

    // Validate object ownership
    bOk = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOk == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Validate decoder ownership
        bOk = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if (bOk == TRUE)
        {
            N16 n16ChannelCategoryOffset =
                psObj->n16ChannelCategoryOffset;

            // Reset success flag
            bOk = FALSE;

            if ((psObj->eBrowseType == BROWSE_TYPE_ALL_CHANNELS) ||
                (bBrowseChannel == TRUE))
            {
                // Service the browse channels request
                bOk = bBrowseChannels(
                    psObj,
                    eDirection,
                    ptChannelId,
                    &n16ChannelCategoryOffset,
                    ptCategoryId
                        );
            }
            else if (psObj->eBrowseType == BROWSE_TYPE_CATEGORY)
            {
                // Service the browse categories request
                bOk = bBrowseCategories (
                    psObj,
                    eDirection,
                    ptChannelId,
                    &n16ChannelCategoryOffset,
                    ptCategoryId,
                    bBrowsingChanlist
                        );
            }

            // Update the browse object

            // Reference category
            psObj->tReferenceCategoryId = *ptCategoryId;
            psObj->n16ChannelCategoryOffset = n16ChannelCategoryOffset;

            // Reference channel
            psObj->tReferenceChannelId = *ptChannelId;
        }
    }

    return bOk;
}

/*****************************************************************************
*
*       BROWSE_bTryBrowse
*
*****************************************************************************/
BOOLEAN BROWSE_bTryBrowse (
    BROWSE_OBJECT hBrowse,
    CHANNEL_ID tChannelId,
    CATEGORY_ID tCategoryId
        )
{
    BOOLEAN bOk;

    // Verify inputs
    if ( (tChannelId == CHANNEL_INVALID_ID) &&
         (tCategoryId == CATEGORY_INVALID_ID))
    {
        return FALSE;
    }

    // Validate object ownership
    bOk = SMSO_bOwner((SMS_OBJECT)hBrowse);
    if (bOk == TRUE)
    {
        BROWSE_OBJECT_STRUCT *psObj =
            (BROWSE_OBJECT_STRUCT *)hBrowse;

        // Validate decoder ownership
        bOk = SMSO_bOwner((SMS_OBJECT)psObj->hDecoder);
        if (bOk == TRUE)
        {
            CATEGORY_OBJECT hCategory;
            CHANNEL_OBJECT hChannel;
            N16 n16Offset;

            // Reset success flag
            bOk = FALSE;

            // Get the channel handle by providing
            // its offset within the cache
            hChannel = CCACHE_hChannel(
                psObj->hCCache, &tChannelId, NULL);

            // are we dealing with a category chanlist?
            if (psObj->eBrowseType == BROWSE_TYPE_CATEGORY)
            {
                // this is a category chanlist.

                // get reference category based on id
                hCategory = CCACHE_hCategory(psObj->hCCache,
                    &psObj->tReferenceCategoryId, 0 );

                // ask reference category if this channel is a member
                n16Offset = CATEGORY_n16GetIndexByChanId(
                    hCategory, tChannelId );

                // Is channel is the browsed category?
                if (n16Offset < 0)
                {
                    // chan isn't in the category.
                    // it doesn't satisfy browse criteria
                    return FALSE;
                }

                // the channel is in the category.

            }

            // If there is no browse compare handler
            //     then the channel meets the browse criteria.

            // is there a browse compare handler?
            if (psObj->bBrowseChannelCompareHandler != NULL)
            {
                // see if the channel satisfies the browse compare handler
                bOk = psObj->bBrowseChannelCompareHandler(
                    psObj->hDecoder,
                    hChannel,
                    psObj->pvBrowseChannelCompareArg );
            }
        }
    }

    return bOk;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*       bBrowseChannels
*
*       This function performs a channel browse.
*
*       Inputs:
*           psObj - A valid handle to a BROWSE_OBJECT_STRUCT.
*           eDirection - A enumerated value indicating how to perform
*               this browse operation.
*           *ptChannelId - A valid pointer to a CHANNEL_ID which is to store
*               the resultant channel Id
*           *pn16ChannelOffset - The offset of this channel within the
*               category.
*           *ptCategoryId - A valid pointer to a CATEGORY_ID which is to
*               store the resultant category Id
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
static BOOLEAN bBrowseChannels (
    const BROWSE_OBJECT_STRUCT *psObj,
    SMSAPI_DIRECTION_ENUM eDirection,
    CHANNEL_ID *ptChannelId,
    N16 *pn16ChannelOffset,
    CATEGORY_ID *ptCategoryId
        )
{
    BOOLEAN bSuccess = FALSE;

    if (psObj->eBrowseType == BROWSE_TYPE_CATEGORY)
    {
        bSuccess = bBrowseChannelsByCategory(
            psObj, eDirection, ptChannelId, pn16ChannelOffset, ptCategoryId);
    }
    else if (psObj->eBrowseType == BROWSE_TYPE_ALL_CHANNELS)
    {
        bSuccess = bBrowseAllChannels(
            psObj, eDirection, ptChannelId, pn16ChannelOffset, ptCategoryId);
    }

    return bSuccess;
}

/*****************************************************************************
*
*       bBrowseChannelsByCategory
*
*****************************************************************************/
static BOOLEAN bBrowseChannelsByCategory (
    const BROWSE_OBJECT_STRUCT *psObj,
    SMSAPI_DIRECTION_ENUM eDirection,
    CHANNEL_ID *ptChannelId,
    N16 *pn16ChannelOffset,
    CATEGORY_ID *ptCategoryId
        )
{
    CATEGORY_OBJECT hCategory;
    BOOLEAN bSuccess = FALSE;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CHANNEL_ID tCandidateChanId;
    N16 n16Offset = 0;
    N32 n32NumChannels;
    size_t tIterationsAvailable = 1;
    CATEGORY_ID tCategoryId = *ptCategoryId;

    // Grab the category handle from the reference
    // category Id (useful later)
    hCategory = CCACHE_hCategory(psObj->hCCache, &tCategoryId, 0 );

    // Get the size of this category
    n32NumChannels = (N32)CATEGORY.tSize(
        psObj->hDecoder, tCategoryId );
    if (n32NumChannels == 0)
    {
        // No need to look at the channels. There aren't any!
        return TRUE;
    }

    // See if this is a direct browse
    if (eDirection == SMSAPI_DIRECTION_DIRECT)
    {
        // The channel Id we're looking
        // for was provided to us

        if (*ptChannelId != CHANNEL_INVALID_ID)
        {
            n16Offset = CATEGORY_n16GetIndexByChanId(
                hCategory, *ptChannelId );

            // Ensure something is there
            if (n16Offset < 0)
            {
                return FALSE;
            }
        }
    }
    else if ((eDirection == SMSAPI_DIRECTION_PREVIOUS) ||
             (eDirection == SMSAPI_DIRECTION_NEXT))
    {
        // Calculate our absolute offset based on our last
        // channel offset and the direction in which the
        // caller would like to go
        n16Offset = (N16)eDirection +
            psObj->n16ChannelCategoryOffset;

        // Ensure we stay within our bounds
        if (n16Offset == n32NumChannels)
        {
            n16Offset = 0;
        }
        else if (n16Offset == -1)
        {
            n16Offset = (N16)n32NumChannels - 1;
        }

        // We may iterate through the number of
        // channels available
        tIterationsAvailable = n32NumChannels;
    }
    else
    {
        // Error, incorrect direction operation
        return FALSE;
    }

    // Prepare offset for pre-increment below
    n16Offset -= (N16)eDirection;

    // Search until we find a channel or or run
    // through all the channels necessary
    while ( ( bSuccess == FALSE ) &&
            ( tIterationsAvailable-- != 0 ) )
    {
        // Keep browsing in the specified direction until a
        // channel is found that the caller wants to browse.

        // Update the offset
        n16Offset += (N16)eDirection;

        // Ensure we stay within our bounds
        if (n16Offset == n32NumChannels)
        {
            n16Offset = 0;
        }
        else if (n16Offset == -1)
        {
            n16Offset = (N16)n32NumChannels - 1;
        }

        // Get the channel handle by providing
        // its offset within the category
        hChannel = CATEGORY_hGetChanHdlByOffset(
            hCategory,
            n16Offset );

        tCandidateChanId = CHANNEL.tChannelId(hChannel);
        // Does the caller wish to handle the
        // channel browse filtering?
        if (psObj->bBrowseChannelCompareHandler != NULL)
        {
            // See if the caller wants to include this
            // channel in the browse if the channel is valid
            if (tCandidateChanId == CHANNEL_INVALID_ID)
            {
                bSuccess = FALSE;
            }
            else
            {
                bSuccess = psObj->bBrowseChannelCompareHandler(
                        psObj->hDecoder,
                        hChannel,
                        psObj->pvBrowseChannelCompareArg );
            }
        }
        // Otherwise, just validate the id
        else if (tCandidateChanId != CHANNEL_INVALID_ID)
        {
            bSuccess = TRUE;
        }
    }

    // Determine if we found a suitable channel
    if ( bSuccess == TRUE )
    {
        // Copy out the new values
        *ptCategoryId = tCategoryId;

        // Get this channel's Id
        *ptChannelId = CHANNEL.tChannelId( hChannel );

        // This channel's offset
        *pn16ChannelOffset = (N16)n16Offset;
    }

    return bSuccess;
}

/*****************************************************************************
*
*       bBrowseAllChannels
*
*****************************************************************************/
static BOOLEAN bBrowseAllChannels (
    const BROWSE_OBJECT_STRUCT *psObj,
    SMSAPI_DIRECTION_ENUM eDirection,
    CHANNEL_ID *ptChannelId,
    N16 *pn16ChannelOffset,
    CATEGORY_ID *ptCategoryId
        )
{
    CATEGORY_OBJECT hCategory =
        CATEGORY_INVALID_OBJECT;
    BOOLEAN bInclude = FALSE;
    CHANNEL_OBJECT hChannel = CHANNEL_INVALID_OBJECT;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID, tCandidateChanId;
    N16 n16Offset = 0;
    N16 n16IterationsAvailable = 0;
    CATEGORY_ID tReferenceCategoryId;

    // Grab the number of channels available
    CCACHE_vStats( psObj->hCCache, &n16IterationsAvailable, NULL );

    // We can browse only +/- 1, or 0
    n16Offset = (N16)eDirection;

    // See if this is a direct browse
    if (eDirection == SMSAPI_DIRECTION_DIRECT)
    {
        // The channel id we're looking
        // for was provided to us
        tChannelId = *ptChannelId;
    }
    else if ((eDirection == SMSAPI_DIRECTION_PREVIOUS) ||
             (eDirection == SMSAPI_DIRECTION_NEXT))
    {
        // The channel id we're looking
        // for is the reference channel
        tChannelId = psObj->tReferenceChannelId;
    }
    else
    {
        // Browsed to nothing
        return FALSE;
    }

    // Search until we find a channel or run
    // through all the channels necessary
    while ( ( bInclude == FALSE ) &&
            ( n16IterationsAvailable > 0 ) )
    {
        // Keep browsing in the specified direction until a
        // channel is found that the caller wants to browse.

        // Get the channel handle by providing
        // its offset within the cache
        hChannel = CCACHE_hChannel(
            psObj->hCCache, &tChannelId, &n16Offset);

        tCandidateChanId = CHANNEL.tChannelId(hChannel);
        if (tCandidateChanId != CHANNEL_INVALID_ID)
        {
            // Include this channel unless compare handler
            // below decides otherwise.
            bInclude = TRUE;

            // Does the caller wish to handle the
            // channel browse filtering?
            if (psObj->bBrowseChannelCompareHandler != NULL)
            {
                // Determine if this channel can be included
                bInclude =
                    psObj->bBrowseChannelCompareHandler(
                        psObj->hDecoder,
                        hChannel,
                        psObj->pvBrowseChannelCompareArg );
            }
        }

        // At this point we have either decided the channel
        // will or will not be included. If it will not, we
        // must move on to the next channel.
        if(bInclude == FALSE)
        {
            // Decrement our iterations through the ccache
            n16IterationsAvailable--;

            // If we have no offset (direct browse), we must
            // move on.
            if(n16Offset == 0)
            {
                n16Offset = (N16)SMSAPI_DIRECTION_NEXT;
            }
        }
    }

    // Determine if we found a suitable channel
    if ( bInclude == TRUE )
    {
        // Get the broadcast category for this channel
        hCategory = CHANNEL.hCategory( hChannel, 0);
        if (hCategory != CATEGORY_INVALID_OBJECT)
        {
            // Update reference information
            tReferenceCategoryId =
                CATEGORY.tGetCategoryId( hCategory );
        }
        else
        {
            tReferenceCategoryId =
                CATEGORY_INVALID_ID;
        }

        // Copy out the new values
        *ptCategoryId = tReferenceCategoryId;

        // Get this channel's Id
        *ptChannelId = CHANNEL.tChannelId( hChannel );

        // This channel's offset
        *pn16ChannelOffset = n16Offset;
    }

    return bInclude;
}

/*****************************************************************************
*
*       bBrowseCategories
*
*       This function performs a category browse.
*
*       Inputs:
*           psObj - A valid handle to a BROWSE_OBJECT_STRUCT.
*           eDirection - A enumerated value indicating how to perform
*               this browse operation.
*           *ptChannelId - A valid pointer to a CHANNEL_ID which is to store
*               the resultant channel Id
*           *pn16ChannelOffset - The offset of this channel within the
*               category.
*           *ptCategoryId - A valid pointer to a CATEGORY_ID which is to
*               store the resultant category Id
*           bBrowsingChanlist - flag indicating if the browse is on a channellist
*               (as opposed to decoder). channellist can browse to empty
*               category while a decoder cannot.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
static BOOLEAN bBrowseCategories (
    const BROWSE_OBJECT_STRUCT *psObj,
    SMSAPI_DIRECTION_ENUM eDirection,
    CHANNEL_ID *ptChannelId,
    N16 *pn16ChannelOffset,
    CATEGORY_ID *ptCategoryId,
    BOOLEAN bBrowsingChanlist
        )
{
    BOOLEAN bSuccess = FALSE;
    BOOLEAN bBrowsedCategory = FALSE;
    CATEGORY_OBJECT hCategory;
    CHANNEL_ID tChannelId = CHANNEL_INVALID_ID;
    CATEGORY_ID tCategoryId = *ptCategoryId;
    N16 n16CategoryOffset = (N16)eDirection;
    N16 n16FilteredChannelOffset = 0;
    N16 n16NumCategories = 1;

    // If we're looking for a relative offset, get the
    // number of categories in the system and set
    // our starting category id
    if (eDirection != SMSAPI_DIRECTION_DIRECT)
    {
        // Get the number of categories in the cache
        CCACHE_vStats( psObj->hCCache, NULL, &n16NumCategories );

        if (n16NumCategories == 0)
        {
            return FALSE;
        }

        // Always start from the reference category Id
        tCategoryId = psObj->tReferenceCategoryId;
    }

    // our reference cat might not exist anymore. let's check
    hCategory = CCACHE_hCategory(
                psObj->hCCache,
                &tCategoryId,
                0 );
    if (hCategory == CATEGORY_INVALID_OBJECT)
    {
        // the reference category is gone, so let's use the invalid cat id
        tCategoryId = CATEGORY_INVALID_ID;
    }

    // Search until we find a category or run
    // through all the categories necessary
    while ( ( bBrowsedCategory == FALSE ) &&
            ( n16NumCategories-- != 0 ) )
    {
        // Keep browsing in the specified direction until you find
        // a category that the caller wants to browse
        hCategory = CCACHE_hCategory(
                        psObj->hCCache,
                        &tCategoryId,
                        n16CategoryOffset );

        // Does the caller wish to handle the
        // category browse filtering?
        if (psObj->bBrowseCategoryCompareHandler != NULL)
        {
            if (hCategory == CATEGORY_INVALID_OBJECT)
            {
                bBrowsedCategory = FALSE;
            }
            else
            {
                // See if the application wants to include this category
                // in the browse if the category is valid
                bBrowsedCategory = psObj->bBrowseCategoryCompareHandler(
                                    psObj->hDecoder,
                                    hCategory,
                                    psObj->pvBrowseCategoryCompareArg,
                                    &n16FilteredChannelOffset );
            }
        }
        // Otherwise, just validate the handle
        else if (hCategory != CATEGORY_INVALID_OBJECT)
        {
            bBrowsedCategory = TRUE;
        }

        // Did we successfully browse to that category?
        if (bBrowsedCategory == TRUE)
        {
            // Now, attempt to navigate within
            // this category to the desired filtered offset
            bBrowsedCategory = bNavigateToRequestedOffset(
                psObj,
                n16FilteredChannelOffset,
                &tChannelId,
                pn16ChannelOffset,
                &tCategoryId,
                bBrowsingChanlist);
        }
    }

    // If we were successful we need to compute an
    // absolute offset for the reference channel
    // within the category

    if (bBrowsedCategory == TRUE)
    {
        N16 n16ChannelOffset;
        CHANNEL_OBJECT hChannel;

        // Determine the absolute
        // offset for this channel
        n16ChannelOffset = CATEGORY_n16GetIndexByChanId(
            hCategory, tChannelId );
        if(N16_MIN == n16ChannelOffset)
        {
            n16ChannelOffset = 0;
        }

        // Get the channel object by its location
        // in the category
        hChannel = CATEGORY_hGetChanHdlByOffset(
            hCategory, n16ChannelOffset);

        if (hChannel != CHANNEL_INVALID_OBJECT)
        {
            // Mark as a success
            bSuccess = TRUE;
        }

        // Copy out the new values to the caller
        *ptChannelId = tChannelId;
        *ptCategoryId = tCategoryId;
        *pn16ChannelOffset = n16ChannelOffset;
    }

    return bSuccess;
}

/*****************************************************************************
*
*       bNavigateToRequestedOffset
*
*       This function navigates within a browsed category to a given
*       offset within that category.
*
*       Inputs:
*           psObj - A valid handle to a BROWSE_OBJECT_STRUCT.
*           n16RequestedOffset - The offset within the category
*               to which we must navigate.  This is the "filtered"
*               offset.
*           *ptChannelId - A valid pointer to a CHANNEL_ID which is to store
*               the resultant channel Id
*           *pn16ChannelOffset - The offset of this channel within the
*               category.
*           tCategoryId - The CATEGORY_ID of the category in which
*               to navigate.
*           bBrowsingChanlist - flag indicating if the browse is on a channellist
*               (as opposed to decoder). channellist can browse to empty
*               category while a decoder cannot.
*
*       Outputs:
*           TRUE on success or FALSE on error.
*
*****************************************************************************/
static BOOLEAN bNavigateToRequestedOffset (
    const BROWSE_OBJECT_STRUCT *psObj,
    N16 n16RequestedOffset,
    CHANNEL_ID *ptChannelId,
    N16 *pn16ChannelOffset,
    CATEGORY_ID *ptCategoryId,
    BOOLEAN bBrowsingChanlist
        )
{
    CATEGORY_OBJECT hCategory;
    CHANNEL_OBJECT hFirstChannel = CHANNEL_INVALID_OBJECT;
    BOOLEAN bSuccess = FALSE;
    size_t tNumChannels;
    N16 n16NumAcceptedChannels = 0;
    N16 n16Direction = 1;

    // Get the number of channels in
    // this category
    tNumChannels = CATEGORY.tSize(
        psObj->hDecoder, *ptCategoryId );

    // Get this category's handle and first channel info
    hCategory = CCACHE_hCategory(
        psObj->hCCache, ptCategoryId, 0);

    if (tNumChannels > 0)
    {
        // this cat has channels in it
        // cycle through to get the first valid channel

        N16 n16Index;
        for (n16Index = 0; n16Index < (N16)tNumChannels; n16Index++)
        {
            hFirstChannel = CATEGORY_hGetChanHdlByOffset( hCategory, n16Index);
            if (hFirstChannel != CHANNEL_INVALID_OBJECT)
            {
                break;
            }
        }
    }

    *ptChannelId = CHANNEL.tChannelId( hFirstChannel );

    // Verify the category handle
    if (hCategory == CATEGORY_INVALID_OBJECT)
    {
        return FALSE;
    }

    if ((hFirstChannel != CHANNEL_INVALID_OBJECT) &&
        (*ptChannelId != CHANNEL_INVALID_ID) &&
        (tNumChannels != 0))
    {
        // Always perform a browse to the first channel
        // in the category
        bSuccess = bBrowseChannels(
            psObj, SMSAPI_DIRECTION_DIRECT,
            ptChannelId, pn16ChannelOffset, ptCategoryId );
        if (bSuccess == TRUE)
        {
            // We have accepted one channel
            n16NumAcceptedChannels = 1;

            // We have one less channel to check
            tNumChannels--;
        }

        // Continue browsing only if we
        // have not yet accepted all the requested channels
        if (n16NumAcceptedChannels < (n16RequestedOffset + 1))
        {
            BOOLEAN bBrowseOk;

            // Get this category's first channel
            // Verify the direction in which
            // we must search
            if (n16RequestedOffset < 0)
            {
                // Update direction if offset is negative
                n16Direction *= -1;
            }

            // Iterate through this category enough times
            // to cover all channels within it (if necessary)
            while (tNumChannels-- > 0)
            {
                // Browse this channel
                bBrowseOk = bBrowseChannels(
                    psObj, (SMSAPI_DIRECTION_ENUM)n16Direction,
                    ptChannelId, pn16ChannelOffset, ptCategoryId );

                if (bBrowseOk == TRUE)
                {
                    // Increment the number of channels
                    // accepted by the browse
                    n16NumAcceptedChannels += n16Direction;

                    // If we have reached the desired offset
                    // then we're all done
                    if (n16NumAcceptedChannels == (n16RequestedOffset + 1))
                    {
                        bSuccess = TRUE;
                        break;
                    }
                }
            }
        }
    }
    else if (bBrowsingChanlist == TRUE)
    {
        // no channels in the reference category
        bSuccess = TRUE;
    }

    return bSuccess;
}

