/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AT_SEEK implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_version.h"
#include "sms_api.h"
#include "sms_obj.h"

#include "ccache.h"
#include "cas.h"
#include "cal_alert.h"
#include "decoder_obj.h"
#include "cm.h"

#include "seek.h"
#include "seek_content.h"
#include "at_seek_content.h"
#include "at_seek.h"
#include "_at_seek.h"

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/



/*****************************************************************************
*   eAvailableToRegister
*
* This API is used to query if content playing on a channel is available for
* registration for seek alerts.
*
* Inputs:
*
*   hDecoder   A handle to a valid Decoder whose Artist/Title Seek Service the
*              caller wishes to know if content is available to be registered
*              with.
*   tChannelId  The id of the channel whose content the caller wished to know
*               the availability to register
*   peTitleAvailablility - destination pointer where the availability of title
*                          registration will be copied
*   peArtistAvailablility - destination pointer where the availability of artist
*                           registration will be copied
*
* Outputs:
*
*   SMSAPI_RETURN_CODE_SUCCESS on success or not on error.
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAvailableToRegister (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    SEEK_AVAILABILTY_ENUM *peTitleAvailablility,
    SEEK_AVAILABILTY_ENUM *peArtistAvailablility
        )
{
    BOOLEAN bLocked;
    CD_OBJECT hCDO;
    CID_OBJECT hArtistCID, hTitleCID;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;

    // did the caller provide us with at least one valid pointer?
    if ( (peTitleAvailablility == NULL) &&
           (peArtistAvailablility == NULL) )
    {
        // Error!
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }

    // Lock the object
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        eReturnCode = DECODER_eSeekService(hDecoder,
                          SEEK_SERVICE_ARTIST_TITLE, &hSeekService);
        if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
        {
            SEEK_SERVICE_OBJECT_STRUCT *psObj =
                (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            hCDO = hGetCDOAndCids(
                       hDecoder,
                       tChannelId,
                       &hArtistCID,
                       &hTitleCID
                           );

            if (hCDO == CD_INVALID_OBJECT)
            {
                // we got an invalid cdo so we cannot identify content
                // so seeks are currently unavailable for this channel.
                if (peArtistAvailablility != NULL)
                {
                    *peArtistAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                }
                if (peTitleAvailablility != NULL)
                {
                    *peTitleAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                }
            }
            else
            {
                STRING_OBJECT hArtistTextString, hTitleTextString;
                size_t tArtistStringSize, tTitleStringSize;

                // we were able to retrieve cids

                // get the artist and title text strings
                hArtistTextString = CDO.hArtist(hCDO);
                hTitleTextString = CDO.hTitle(hCDO);

                // get the strings' size
                tArtistStringSize = STRING.tSize(hArtistTextString);
                tTitleStringSize = STRING.tSize(hTitleTextString);

                // is the caller interested in artist availability?
                if (peArtistAvailablility != NULL)
                {
                    // yes they are
                    do
                    {
                        BOOLEAN bRegistered = FALSE;

                        // did we retrieve an artist cid and do we have the
                        // artist text req'd for registration?
                        if ( (hArtistCID == CID_INVALID_OBJECT) ||
                             (tArtistStringSize == 0) )
                        {
                            // no, so we cannot identify the artist
                            *peArtistAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                            break;
                        }

                        // we can identify the artist, now see if the artist is
                        // registered already
                        bRegistered = CAL.bExists(
                                          psObj->hSeekList,
                                          hArtistCID);

                        if (bRegistered == TRUE)
                        {
                            // already registered
                            *peArtistAvailablility =
                                SEEK_AVAILIBILITY_ALREADY_REGISTERED;
                        }
                        else
                        {
                            // not already registered
                            *peArtistAvailablility = SEEK_AVAILABILTY_AVAILABLE;
                        }
                    }
                    while(0);
                }

                // is the caller interested in title availability?
                if (peTitleAvailablility != NULL)
                {
                    // yes they are
                    do
                    {
                        BOOLEAN bRegistered = FALSE;

                        // did we retrieve a title cid
                        // and do we have the title text
                        // req'd to register?
                        if ( (hTitleCID == CID_INVALID_OBJECT) ||
                             (tTitleStringSize == 0) )
                        {
                            // no, so we cannot identify the title
                            *peTitleAvailablility = SEEK_AVAILIBILITY_UNAVAILABLE;
                            break;
                        }

                        // we can identify the title, now see if the title is
                        // registered already
                        bRegistered = CAL.bExists(
                                          psObj->hSeekList,
                                          hTitleCID);

                        if (bRegistered == TRUE)
                        {
                            // already registered
                            *peTitleAvailablility =
                                SEEK_AVAILIBILITY_ALREADY_REGISTERED;
                        }
                        else
                        {
                            // not already registered
                            *peTitleAvailablility = SEEK_AVAILABILTY_AVAILABLE;
                        }
                    }
                    while(0);
                }
            }
        }
        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // couldn't lock object
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }

    return eReturnCode;
}

/*****************************************************************************
*   eRegister
*
* This API is used to register content for Artist/Title Seek Alerts
*
* Inputs:
*
*   hDecoder   A handle to a valid Decoder whose Artist/Title Seek Service the
*              caller wishes to register content with
*              with.
*   eType           The artist/title seek type the caller wants to register
*                   content for
*   tChannelId      The id of the channel on which content to be registered is
*                   playing on
*
* Outputs:
*   phSeekContent  A pointer to store the registered content object.
*   Returns a error code.
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eRegister (
    DECODER_OBJECT hDecoder,
    AT_SEEK_ENUM eType,
    CHANNEL_ID tChannelId,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    BOOLEAN bLocked;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;
    STRING_OBJECT hArtistTextString, hTitleTextString;
    SEEK_SERVICE_OBJECT hSeekService = SEEK_SERVICE_INVALID_OBJECT;
    size_t tArtistStringSize, tTitleStringSize;

    // Lock the service
    bLocked = SMSO_bLock(
        (SMS_OBJECT)hDecoder, OSAL_OBJ_TIMEOUT_INFINITE);
    if(bLocked == TRUE)
    {
        do
        {
            CD_OBJECT hCDO;
            CID_OBJECT hCID, hArtistCID, hTitleCID;
            SEEK_SERVICE_OBJECT_STRUCT *psObj;
            BOOLEAN bRegistered;

            eReturnCode = DECODER_eSeekService(hDecoder,
                              SEEK_SERVICE_ARTIST_TITLE, &hSeekService);
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }

            psObj = (SEEK_SERVICE_OBJECT_STRUCT *)hSeekService;

            // get the channels cdo and its cids
            hCDO = hGetCDOAndCids(
                       hDecoder,
                       tChannelId,
                       &hArtistCID,
                       &hTitleCID
                           );

            if (hCDO == CD_INVALID_OBJECT)
            {
                // failure to get cids
                eReturnCode =  SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                break;
            }

            // get the artist and title text strings
            hArtistTextString = CDO.hArtist(hCDO);
            hTitleTextString = CDO.hTitle(hCDO);

            // get the string's size
            tArtistStringSize = STRING.tSize(hArtistTextString);
            tTitleStringSize = STRING.tSize(hTitleTextString);

            // based on type of seek requested,
            // determine the cid we'll register with the cal.
            // Also determine whether we have sufficient text information
            if (eType == AT_SEEK_ARTIST)
            {
                // we need to have valid artist text to allow registration for
                // an ARTIST_SEEK. the string has length > 0
                if (tArtistStringSize == 0)
                {
                    eReturnCode = SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                    break;
                }

                // the CID to be registered with the CAL is the ArtistCID
                hCID = hArtistCID;
            }
            else if (eType == AT_SEEK_TITLE)
            {
                // valid title text to
                // allow registration for a TITLE_SEEK
                if ( tTitleStringSize == 0 )
                {
                    eReturnCode = SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                    break;
                }

                // the CID to be registered with the CAL is the TitleCID
                hCID = hTitleCID;
            }
            else
            {
                // not a valid seek type!
                eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
                break;
            }

            // do we have a valid CID to register?
            if (hCID == CID_INVALID_OBJECT)
            {
                eReturnCode = SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST;
                break;
            }

            // see if this is already registered.
            bRegistered = CAL.bExists(
                              psObj->hSeekList,
                              hCID);
            if (bRegistered == TRUE)
            {
                eReturnCode = SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
                break;
            }

            psItem = psRegisterContent(
                psObj,
                hArtistCID,
                hTitleCID,
                eType,
                hCID,
                hArtistTextString,
                hTitleTextString,
                TRUE, // when newly registering, the content is enabled
                phSeekContent
                    );

            if (psItem != NULL)
            {
                BOOLEAN bOk;
                bOk = bAddToConfigFile(
                          psObj,
                          psItem,
                          hArtistTextString,
                          hTitleTextString
                              );
                if (bOk == TRUE)
                {
                    eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
                }
            }

        } while (0);

        // Unlock object
        SMSO_vUnlock((SMS_OBJECT)hDecoder);
    }
    else
    {
        // couldn't lock object
        eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    }


    return eReturnCode;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*   AT_SEEK_eAddToConfig
*****************************************************************************/
SMSAPI_RETURN_CODE_ENUM AT_SEEK_eAddToConfig(
   SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem,
   STRING_OBJECT hArtistTextString,
   STRING_OBJECT hTitleTextString
       )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    if (psItem != NULL)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj;
        BOOLEAN bOk;

        bOk = SMSO_bOwner((SMS_OBJECT)psItem->hSeekService);
        if (bOk == FALSE)
        {
            return SMSAPI_RETURN_CODE_API_NOT_ALLOWED;
        }

        psObj = (SEEK_SERVICE_OBJECT_STRUCT*)psItem->hSeekService;

        bOk = bAddToConfigFile(
                  psObj,
                  psItem,
                  hArtistTextString,
                  hTitleTextString
                      );
        if (bOk == TRUE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
        }
    }
    return eReturnCode;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*   bHandleSeekAlert
*
* This is the specific seek service interface function used to
* handle seek alerts
*
* Returns TRUE if a Seek Alert Event should be generated. FALSE if not
*****************************************************************************/
static BOOLEAN bHandleSeekAlert(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_ALERT_OBJECT hAlert
        )
{
    BOOLEAN bSetDecoderSeekAlertEvent = FALSE;
    CAL_CONTENT_OBJECT hContent;
    CHANNEL_OBJECT hChannel;

    // get the content and channel that caused the alert
    hContent = CAL_ALERT.hContent(hAlert);
    hChannel = CAL_ALERT.hChannel(hAlert);

    // should we allow the seek alert
    bSetDecoderSeekAlertEvent = bSeekAlertAllowed(
                                 hSeek,
                                 hContent,
                                 hChannel
                                     );

    return bSetDecoderSeekAlertEvent;
}

/*****************************************************************************
*   bSeekAlertAllowed
*
* This function implements the rules to determine if the application should be
* informed of a content match via a seek alert
*
* Returns TRUE is a Seek Alert is allowed, FALSE if it is not
*****************************************************************************/
static BOOLEAN bSeekAlertAllowed(
    SEEK_SERVICE_OBJECT hSeek,
    CAL_CONTENT_OBJECT hContent,
    CHANNEL_OBJECT hChannel
        )
{
    // we start out assuming the alert is allowed, then we'll check the rules
    // to see if it is not allowed
    BOOLEAN bValid, bEnabled, bAlertAllowed = TRUE;

    CHANNEL_ID tCurrentlyTunedChannelId, tChannelId;
    SEEK_SERVICE_OBJECT_STRUCT *psObj;
    AT_SEEK_ENUM eType;
    DECODER_OBJECT hDecoder;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

    do
    {
        // 1. The service must be enabled for a seek alert to be allowed
        printf(AT_SEEK_OBJECT_NAME": Service is :%s\n",
               psObj->eState == SEEK_STATE_ENABLED ? "ENABLED" : "NOT_ENABLED");
        // is service enabled?
        if (psObj->eState != SEEK_STATE_ENABLED)
        {
            // service is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

        // 2. The registered item must be enabled for a seek alert to be allowed
        bEnabled = SEEK_CONTENT.bEnabled(hContent);
        printf(AT_SEEK_OBJECT_NAME":Item is :%s\n",
               bEnabled == TRUE ? "ENABLED" : "DISABLED");
        // is item enabled?
        if (bEnabled == FALSE)
        {
            // item is disabled, so seek alert is not allowed
            bAlertAllowed = FALSE;
            break;
        }

        // 3. If it's not indicated otherwise the matched content should not 
        //    be occurring on the tuned channel for a seek alert to be allowed

        // All SEEK_SERVICE objects belong to a DECODER object (their parent)
        hDecoder = (DECODER_OBJECT)SMSO_hParent((SMS_OBJECT)psObj);

        // Are alerts allowed on the tuned channel?
        bEnabled = SEEK.bGetAlertsTunedStatus(hDecoder, SEEK_SERVICE_ARTIST_TITLE);
        if (bEnabled == FALSE)
        {
            tCurrentlyTunedChannelId = DECODER.tCurrentChannelId(hDecoder);
            tChannelId = CHANNEL.tChannelId(hChannel);
            printf(AT_SEEK_OBJECT_NAME": Content %s currently tuned channel\n",
                tCurrentlyTunedChannelId == tChannelId ? "ON" : "NOT ON");
            // is the content on the currently tuned channel?
            if (tCurrentlyTunedChannelId == tChannelId)
            {
                // content occurred on the currently tuned channel, so seek alert
                // is not allowed
                bAlertAllowed = FALSE;
                break;
            }
        }

        // 4. If both Artist and Title seeks are registered, we notify only Title
        //    match for one content alert

        // is this an artist seek?
        eType = AT_SEEK_CONTENT.eType(hContent);
        if (eType == AT_SEEK_ARTIST)
        {
            AT_SEEK_ITERATOR_STRUCT sIteratorStruct;
            CD_OBJECT hCDO;
            CID_OBJECT hArtistCID, hTitleCID;

            // getting Title CID from reported channel
            tChannelId = CHANNEL.tChannelId(hChannel);
            hCDO = hGetCDOAndCids(hDecoder, tChannelId, &hArtistCID, &hTitleCID);
            if (hCDO == CD_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrint(AT_SEEK_OBJECT_NAME, 3,
                    "Cannot get CDO for channel %d", tChannelId);

                bAlertAllowed = FALSE;
                break;
            }

            sIteratorStruct.bMatch = FALSE;
            sIteratorStruct.hId = hTitleCID;

            // iterate this service searching for a match that is
            // registered AND enabled
            SEEK.eIterate(hDecoder,
                SEEK_SERVICE_ARTIST_TITLE,
                bATSeekContentIterator,
                &sIteratorStruct);

            if (sIteratorStruct.bMatch == TRUE)
            {
                // yes the title is also registered AND enabled AND 
                // playing on reported channel. We will not allow this 
                // alert to be passed on to the app.
                puts(AT_SEEK_OBJECT_NAME
                    ": Suppressing artist seek - "
                    "title is also registered AND enabled.");

                bAlertAllowed = FALSE;
                break;
            }
            else
            {
                puts(AT_SEEK_OBJECT_NAME
                     ": Allowing artist seek "
                     "- title is not also registered AND enabled.");
            }
        }
    } while (0);

    printf(AT_SEEK_OBJECT_NAME": Seek Alert %s \n",
           bAlertAllowed == TRUE ? "ALLOWED" : "NOT ALLOWED");

    return bAlertAllowed;
}

/*****************************************************************************
*   bInit
*
* This function implements the service specific initialization
*
* Returns TRUE if successful, FALSE if not
*****************************************************************************/
static BOOLEAN bInit(
    SEEK_SERVICE_OBJECT hSeek
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    BOOLEAN bValid, bReturn = FALSE;

    bValid = SMSO_bValid((SMS_OBJECT)hSeek);
    if (bValid == TRUE)
    {
        SEEK_SERVICE_OBJECT_STRUCT *psObj;

        psObj = (SEEK_SERVICE_OBJECT_STRUCT*)hSeek;

        // Attach a category to the CAL
        eResult = CAL.eAttachCategory (
            psObj->hSeekList,
            CATEGORY_INVALID_ID,
            AT_SEEK_CAL_AUTO_ADD_OPTIONS,
            "My Seeks",
            "Seeks"
                );

        if (eResult != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // Error!
            bReturn = FALSE;
        }
        else
        {
            bReturn = bRegisterContentFromConfigFile(psObj);
        }
    }

    return bReturn;
}

/*****************************************************************************
*
*   hGetCDOAndCids
*
*****************************************************************************/
static CD_OBJECT hGetCDOAndCids(
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    CID_OBJECT *phArtistCID,
    CID_OBJECT *phTitleCID
        )
{
    BOOLEAN bReturn = FALSE;
    CCACHE_OBJECT hCCache;
    CHANNEL_OBJECT hChannel;
    CD_OBJECT hCDO;

    // did caller give us a valid id?
    if (tChannelId == CHANNEL_INVALID_ID)
    {
        // no, so instead we use the currently tuned channel id
        tChannelId = DECODER.tCurrentChannelId(hDecoder);
    }

    // From the DECODER handle, determine the channel cache handle
    hCCache = DECODER_hCCache(hDecoder);

    // get the channel obj for this requested chan id
    hChannel = CCACHE_hChannelFromIds (
        hCCache,
        SERVICE_INVALID_ID,
        tChannelId,
        FALSE
            );

    // get the channel's cdo
    hCDO = CHANNEL.hCDO(hChannel);
    if (hCDO != CD_INVALID_OBJECT)
    {
        // get the cids from the cdo
        bReturn = bRetrieveCidsForSeeks(
                         hCDO,
                         phArtistCID,
                         phTitleCID
                             );
        if (bReturn == FALSE)
        {
            // couldn't get cids, so this content is not valid for us
            hCDO = CD_INVALID_OBJECT;
        }
    }

    return hCDO;
}

/*****************************************************************************
*
*    bRetrieveCidsForSeeks
*
*    This is a local function which looks at a CDO and extracts the various
*    CIDs that can be used for registering.
*
*****************************************************************************/
static BOOLEAN bRetrieveCidsForSeeks(
    CD_OBJECT hCDO,
    CID_OBJECT *phArtistCID,
    CID_OBJECT *phTitleCID
        )
{
    CDO_TYPE_ENUM eCDOType;
    CID_OBJECT hArtistCID, hTitleCID;

    // Load result pointers as needed/requested
    if(phArtistCID == NULL) // Do they want it?
    {
        // No, use local
        phArtistCID = &hArtistCID;
    }
    if(phTitleCID == NULL) // Do they want it?
    {
        // No, use local
        phTitleCID = &hTitleCID;
    }

    // clear the values
    *phArtistCID = CID_INVALID_OBJECT;
    *phTitleCID = CID_INVALID_OBJECT;

    // get the type of cdo
    eCDOType = CDO.eType(hCDO);

    // extract cids based on cdo type
    switch (eCDOType)
    {
        case CDO_MUSIC:
        {
            *phTitleCID = MUSIC.hSongId(hCDO);
            *phArtistCID = MUSIC.hArtistId(hCDO);
        }
        break;

        case CDO_NEWS:
        case CDO_ENTERTAINMENT:
        case CDO_REPORT:
        case CDO_SPORTS:
        case CDO_NON_PROGRAM:
        case CDO_UNKNOWN:
        {
            // at least for the time being we'll use text cids
            //  for seeking artist or title on these content types.
            // Sirius has Id's for News, Entertainment that could possibly be
            // considered as Title Id CIDs, but pdt logs show that
            // isn't consistently true
        }
        break;

        case CDO_INVALID:
        default:
        {
            // can't seek on invalid cdo
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*    bAddToConfigFile
*
*****************************************************************************/
static BOOLEAN bAddToConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem,
    STRING_OBJECT hArtistTextString,
    STRING_OBJECT hTitleTextString
        )
{
    TAG_OBJECT hTag;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    // add a new Registered Content Tag
    // this will be the parent tag of all tags needed to persist an
    // instance of content registered with this service
    eReturn = TAG_eAdd(
          SEEK_REGISTERED_CONTENT,
          psObj->hTag,
          &hTag,
          NULL
              );

    // successful?
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // no
        return FALSE;
    }

    // add all the tags needed to persist an
    // instance of at seek content
    eReturn = eAddToConfigFile (
                  hTag,
                  psItem,
                  hArtistTextString,
                  hTitleTextString
                      );

    // successful?
    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // no
        return FALSE;
    }

    // yes
    return TRUE;
}

/*****************************************************************************
*
*    psRegisterContent
*
*****************************************************************************/
static SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psRegisterContent(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    CID_OBJECT hArtistCID,
    CID_OBJECT hTitleCID,
    AT_SEEK_ENUM eType,
    CID_OBJECT hCID,
    STRING_OBJECT hArtistTextString,
    STRING_OBJECT hTitleTextString,
    BOOLEAN bEnabled,
    SEEK_CONTENT_OBJECT *phSeekContent
        )
{
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem = NULL;

    // create a seek content item of type that is applicable to the
    // artist/title service
    psItem = SEEK_CONTENT_psCreateRegisteredItem((SEEK_SERVICE_OBJECT)psObj, bEnabled);
    if (psItem != NULL)
    {
        SMSAPI_RETURN_CODE_ENUM eReturn;

        // initialize the service specific content portion
        AT_SEEK_CONTENT_vSetServiceSpecificInfo(
            (void *)psItem,
            hArtistCID,
            hTitleCID,
            eType,
            (SEEK_SERVICE_OBJECT)psObj
                );

        // we have all the information we need, now we can
        // register with the CAL
        eReturn = CAL.eRegister(
            psObj->hSeekList,
            hCID,
            (void *)psItem,
            AT_SEEK_CAL_REGISTRATION_OPTIONS,
            hArtistTextString,
            hTitleTextString,
            (CAL_CONTENT_OBJECT *)phSeekContent
                );

        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // we couldn't successfully register for some reason, destroy the
            // item we created
            SEEK_CONTENT_vDestroyRegisteredItem(psItem);
            psItem = NULL;
        }
    }

    return psItem;
}

/*****************************************************************************
*
*   bRegisterContentFromConfigFile
*
*****************************************************************************/
static BOOLEAN bRegisterContentFromConfigFile(
    SEEK_SERVICE_OBJECT_STRUCT *psObj
        )
{
    REGISTERED_CONTENT_TAG_ITERATOR_STRUCT sIteratorStruct;
    SMSAPI_RETURN_CODE_ENUM eResult;

    if (psObj->hTag == TAG_INVALID_OBJECT)
    {
        // must not be a config file
        return TRUE;
    }

    sIteratorStruct.psObj = psObj;

    // the seek service object holds the parent tag of all registered content
    // tags. Iterate through all the child tags, and for each registered
    // content tag extract the information and register and configure.
    eResult = TAG_eIterateChildren(
        psObj->hTag,
        bRegisteredContentTagIterator,
        &sIteratorStruct
            );

    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bRegisteredContentTagIterator
*
*****************************************************************************/
static BOOLEAN bRegisteredContentTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        )
{
    REGISTERED_CONTENT_TAG_ITERATOR_STRUCT *psIteratorStruct;
    STRING_OBJECT hTagName;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psItem;
    BOOLEAN bEnabled;
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    CID_OBJECT hArtistCID = CID_INVALID_OBJECT, hTitleCID = CID_INVALID_OBJECT;
    STRING_OBJECT hArtistTextString = STRING_INVALID_OBJECT,
                  hTitleTextString = STRING_INVALID_OBJECT;

    psIteratorStruct = (REGISTERED_CONTENT_TAG_ITERATOR_STRUCT *)pvArg;

    // get the tag name
    hTagName = TAG_hTagName(hTag);
    // see if the tag name is one that we're looking for
    if (STRING.n16CompareCStr(SEEK_REGISTERED_CONTENT, 0, hTagName) == 0)
    {
        // we found a Registered Content Tag
        CID_OBJECT hCID = CID_INVALID_OBJECT;
        AT_SEEK_ENUM eType;

        do
        {
            // get type
            eType = eGetATSeekTypeFromTags(hTag);

            if (eType == AT_SEEK_UNKNOWN)
            {
                // can't do anything more with this entry
                break;
            }

            eReturn =
                eGetEnabledFromTag(psIteratorStruct->psObj, hTag, &bEnabled);

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            // get the CIDS
            eReturn = eGetCidsFromTags(
                          hTag,
                          eType,
                          &hCID,
                          &hArtistCID,
                          &hTitleCID
                              );

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            eReturn = eGetArtistTitleTextFromTags(
                hTag,
                &hArtistTextString,
                &hTitleTextString
                    );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                // can't do anything more with this entry
                break;
            }

            psItem = psRegisterContent(
                         psIteratorStruct->psObj,
                         hArtistCID,
                         hTitleCID,
                         eType,
                         hCID,
                         hArtistTextString,
                         hTitleTextString,
                         bEnabled,
                         NULL
                             );

            if (psItem != NULL)
            {
                psItem->hTag = hTag;
            }
        } while (0);
    }

    // destroy anything allocated
    if (hArtistCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hArtistCID);
    }
    if (hTitleCID != CID_INVALID_OBJECT)
    {
        CID_vDestroy(hTitleCID);
    }

    if (hArtistTextString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hArtistTextString);
    }
    if (hTitleTextString != STRING_INVALID_OBJECT)
    {
        STRING.vDestroy(hTitleTextString);
    }

    // keep iterating
    return TRUE;
}

/*****************************************************************************
*
*   eGetATSeekTypeFromTags
*
*****************************************************************************/
static AT_SEEK_ENUM eGetATSeekTypeFromTags(
    TAG_OBJECT hTag
        )
{
    SMSAPI_RETURN_CODE_ENUM eResult;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;
    AT_SEEK_ENUM eType = AT_SEEK_UNKNOWN;

    // get the type tag
    eResult = TAG_eGet(
        AT_SEEK_TYPE,
        hTag,
        &hChildTag,
        NULL,
        FALSE // don't create if it isn't found
            );
    if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // got the tag, now get its value
        STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
        size_t tSize;

        tSize = sizeof(STRING_OBJECT);
        phString = &hString;
        eResult = TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);
        if (eResult == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // got a value, now compare it to the type enums to see which one
            // it is
            do
            {
                // try AT_SEEK_ARTIST
                eType = AT_SEEK_ARTIST;
                // compare value to the type text
                if (STRING.n16CompareCStr(
                        AT_SEEK_CONTENT_pacATSeekTypeText(eType),
                        0,
                        hString) == 0)
                {
                    // matched, so we now know it is AT_SEEK_ARTIST
                    break;
                }

                // try AT_SEEK_TITLE
                eType = AT_SEEK_TITLE;
                // Initialize CID entry as empty
                if (STRING.n16CompareCStr(
                        AT_SEEK_CONTENT_pacATSeekTypeText(eType),
                        0,
                        hString) == 0)
                {
                    // matched, so we now know it is AT_SEEK_TITLE
                    break;
                }

                // couldn't find a match, so the type will
                // remain AT_SEEK_UNKNOWN
            } while (0);

            STRING.vDestroy(hString);
        }
    }

    return eType;
}

/*****************************************************************************
*
*   eGetEnabledFromTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetEnabledFromTag(
    SEEK_SERVICE_OBJECT_STRUCT *psObj,
    TAG_OBJECT hTag,
    BOOLEAN *pbEnabled
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn = SMSAPI_RETURN_CODE_ERROR;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;

    if (pbEnabled != NULL)
    {
        // get enabled tag
        eReturn = TAG_eGet(
            AT_SEEK_ENABLED,
            hTag,
            &hChildTag,
            NULL,
            FALSE
                );
        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            // get enabled tag value
            STRING_OBJECT hString = STRING_INVALID_OBJECT, *phString;
            size_t tSize;

            tSize = sizeof(STRING_OBJECT);
            phString = &hString;
            eReturn = TAG_eGetTagValue(
                hChildTag,
                TAG_TYPE_STRING,
                (void **)&phString,
                &tSize);

            if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
            {

                eReturn = CM_eStringToBoolean(
                    hString,
                    pbEnabled
                        );

                // we're done with this
                STRING.vDestroy(hString);
            }
        }
    }
    return eReturn;
}

/*****************************************************************************
*
*   eGetCidsFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetCidsFromTags(
    TAG_OBJECT hTag,
    AT_SEEK_ENUM eType,
    CID_OBJECT *phCID,
    CID_OBJECT *phArtistCID,
    CID_OBJECT *phTitleCID
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hChildTag;
    STRING_OBJECT hArtistCIDString = STRING_INVALID_OBJECT,
                  hTitleCIDString = STRING_INVALID_OBJECT,
                  *phString;
    size_t tSize, tArtistStringSize = 0, tTitleStringSize = 0;
    char *pacCStrBuffer = NULL;
    void *pvVoid;

    *phCID = CID_INVALID_OBJECT;
    *phArtistCID = CID_INVALID_OBJECT;
    *phTitleCID = CID_INVALID_OBJECT;

    // get artist cid tag
    eReturn = TAG_eGet(
        AT_SEEK_ARTIST_CID,
        hTag,
        &hChildTag,
        NULL,
        FALSE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // get the artist cid
        phString = &hArtistCIDString;

        // don't need to look at the return code, because we will instead
        // base our decisions on whether or not hArtistCIDString is
        // valid or not. If error, or no value hArtistCIDString is INVALID
        // but we can continue in either case.
        TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);

        tArtistStringSize = STRING.tSize(hArtistCIDString);
    }

    // get title cid tag
    eReturn = TAG_eGet(
        AT_SEEK_TITLE_CID,
        hTag,
        &hChildTag,
        NULL,
        FALSE
            );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // get the title cid tag
        phString = &hTitleCIDString;

        // don't need to look at the return code, because we will instead
        // base our decisions on whether or not hTitleCIDString is
        // valid or not. If error, or no value hTitleCIDString is INVALID
        // but we can continue in either case.
        TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phString,
            &tSize);

        tTitleStringSize = STRING.tSize(hTitleCIDString);
    }

    // size for the bigger of the two
    tSize = (tArtistStringSize > tTitleStringSize ?
            tArtistStringSize : tTitleStringSize);

    tSize++; // add one for NULL

    // allocate memory for the read buffer
    pacCStrBuffer =
        (char *) SMSO_hCreate(
                    AT_SEEK_OBJECT_NAME":CSTR",
                    tSize,
                    SMS_INVALID_OBJECT, FALSE // No lock necessary
                    );
    if (pacCStrBuffer != NULL)
    {
        if (hArtistCIDString != STRING_INVALID_OBJECT)
        {
            // copy to a c-string so that we can create a CID from it
            STRING.tCopyToCStr(hArtistCIDString, pacCStrBuffer, tSize);
            pvVoid = (void *)pacCStrBuffer;
            // Read from memory assumes memory provided as an input is
            // NULL terminated. It is in fact since we just did a
            // string copy above into a buffer we know is at least one
            // larger than the source (thus it will be NULL terminated).
            *phArtistCID = CID_hReadFromMemory((void const**)&pvVoid);
            STRING.vDestroy(hArtistCIDString);
        }

        if (hTitleCIDString != STRING_INVALID_OBJECT)
        {
            // copy to a c-string so that we can create a CID from it
            STRING.tCopyToCStr(hTitleCIDString, pacCStrBuffer, tSize);
            pvVoid = (void *)pacCStrBuffer;
            // Read from memory assumes memory provided as an input is
            // NULL terminated. It is in fact since we just did a
            // string copy above into a buffer we know is at least one
            // larger than the source (thus it will be NULL terminated).
            *phTitleCID = CID_hReadFromMemory((void const**)&pvVoid);
            STRING.vDestroy(hTitleCIDString);
        }

        // we're done with the buffer
        SMSO_vDestroy((SMS_OBJECT)pacCStrBuffer);

        // based on the at seek type, set the cid to register for
        switch (eType)
        {
            case AT_SEEK_ARTIST:
                *phCID = *phArtistCID;
            break;
            case AT_SEEK_TITLE:
                *phCID = *phTitleCID;
            break;
            default:
                eReturn = SMSAPI_RETURN_CODE_ERROR;
            break;
        }
    }
    else
    {
        // couldn't allocate memory
        eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
    }

    return eReturn;
}

/*****************************************************************************
*
*   eGetArtistTitleTextFromTags
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGetArtistTitleTextFromTags(
    TAG_OBJECT hTag,
    STRING_OBJECT *phArtistText,
    STRING_OBJECT *phTitleText
        )
{
    SMSAPI_RETURN_CODE_ENUM eLocalReturn, eReturn = SMSAPI_RETURN_CODE_SUCCESS;
    TAG_OBJECT hChildTag = TAG_INVALID_OBJECT;
    size_t tSize;

    if ((phArtistText == NULL) || (phTitleText == NULL))
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    // get title text
    eLocalReturn = TAG_eGet(
        AT_SEEK_TITLE_TEXT,
        hTag,
        &hChildTag,
        NULL,
        TRUE
            );
    if (eLocalReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        tSize = sizeof(STRING_OBJECT);

        // get title text
        TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phTitleText,
            &tSize);
    }
    else if (eLocalReturn != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET)
    {
        // it's ok to have no value, but other return codes indicate error
        eReturn = SMSAPI_RETURN_CODE_ERROR;
    }

    // get artist text tag
    eLocalReturn = TAG_eGet(
                  AT_SEEK_ARTIST_TEXT,
                  hTag,
                  &hChildTag,
                  NULL,
                  TRUE
                      );
    if (eLocalReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        tSize = sizeof(STRING_OBJECT);

        // get artist text
        TAG_eGetTagValue(
            hChildTag,
            TAG_TYPE_STRING,
            (void **)&phArtistText,
            &tSize);
    }
    else if (eLocalReturn != SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET)
    {
        // it's ok to have no value, but other return codes indicate error
        eReturn = SMSAPI_RETURN_CODE_ERROR;
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteTypeToTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteTypeToTag(
    TAG_OBJECT hParentTag,
    AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSpecificInfo
        )
{
    STRING_OBJECT hValueString;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTypeTag = TAG_INVALID_OBJECT;

    // add the type tag
    eReturn = TAG_eAdd(
                  AT_SEEK_TYPE,
                  hParentTag,
                  &hTypeTag,
                  NULL
                      );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // extract the type
        AT_SEEK_ENUM eType;

        // create a string containing the type text
        eType = psATSpecificInfo->eType;
        hValueString =
            STRING.hCreate(AT_SEEK_CONTENT_pacATSeekTypeText(eType), 0);

        eReturn = TAG_eSetTagValue(
                      hTypeTag,
                      TAG_TYPE_STRING,
                      &hValueString,
                      sizeof(STRING_OBJECT),
                      FALSE
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // couldn't set the value, so remove the tag we just added
            TAG_eRemove(hTypeTag, FALSE);
        }

        // done with this string
        STRING.vDestroy(hValueString);
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteCIDsToTag
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteCIDsToTag(
    TAG_OBJECT hParentTag,
    AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSpecificInfo
        )
{
    STRING_OBJECT hValueString = STRING_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hArtistCidTag = TAG_INVALID_OBJECT,
               hTitleCidTag = TAG_INVALID_OBJECT;
    size_t tArtistCIDSize, tTitleCIDSize, tCIDSize;
    char *pacCIDBuffer;
    void *pvVoid;

    do
    {
        // get the cid sizes
        tArtistCIDSize = CID_tSize(psATSpecificInfo->hArtistCID);
        tTitleCIDSize = CID_tSize(psATSpecificInfo->hTitleCID);
        // get the largest cid size
        tCIDSize = (tArtistCIDSize > tTitleCIDSize ?
                    tArtistCIDSize : tTitleCIDSize);

        // allocate memory for the read buffer
        pacCIDBuffer =
            (char *) SMSO_hCreate(
                         AT_SEEK_OBJECT_NAME":CB",
                         tCIDSize,
                         SMS_INVALID_OBJECT, FALSE // No lock necessary
                         );
        if (pacCIDBuffer == NULL)
        {
            eReturn = SMSAPI_RETURN_CODE_OUT_OF_MEMORY;
            break;
        }

        pvVoid = pacCIDBuffer;

        // add a tag for the artist cid
        eReturn = TAG_eAdd(
                      AT_SEEK_ARTIST_CID,
                      hParentTag,
                      &hArtistCidTag,
                      NULL
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // create a string containing the artist cid
        if (psATSpecificInfo->hArtistCID != CID_INVALID_OBJECT)
        {
            CID_n32FWriteToMemory(psATSpecificInfo->hArtistCID, (void **)&pvVoid);
            hValueString = STRING.hCreate(pacCIDBuffer, tCIDSize);
            // set the tag value
            eReturn = TAG_eSetTagValue(
                          hArtistCidTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          FALSE
                              );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }

        // add tag for the title cid
        eReturn = TAG_eAdd(
                      AT_SEEK_TITLE_CID,
                      hParentTag,
                      &hTitleCidTag,
                      NULL
                          );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // create a string containing the title cid
        if (psATSpecificInfo->hTitleCID != CID_INVALID_OBJECT)
        {
            pvVoid = pacCIDBuffer;
            CID_n32FWriteToMemory(psATSpecificInfo->hTitleCID, (void **)&pvVoid);

            // see if this was already created when handling the artist CID
            if (hValueString == STRING_INVALID_OBJECT)
            {
                // wasn't already created, so creatge it now
                hValueString = STRING.hCreate(pacCIDBuffer, tCIDSize);
            }
            else
            {
                // already created, just modify it
                STRING.bModifyCStr(hValueString, pacCIDBuffer);
            }

            // write the tag vale
            eReturn = TAG_eSetTagValue(
                          hTitleCidTag,
                          TAG_TYPE_STRING,
                          &hValueString,
                          sizeof(STRING_OBJECT),
                          FALSE
                              );
            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                break;
            }
        }
    } while(0);

    if(pacCIDBuffer != NULL)
    {
        // if we allocated this buffer, delete it
        SMSO_vDestroy((SMS_OBJECT)pacCIDBuffer);
    }

    if (hValueString != STRING_INVALID_OBJECT)
    {
        // if we created the string, delete it
        STRING.vDestroy(hValueString);
    }

    if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
    {
        // something went wrong, so remove the tags we added
        if (TAG_INVALID_OBJECT != hArtistCidTag)
        {
            TAG_eRemove(hArtistCidTag, FALSE);
        }
        if (TAG_INVALID_OBJECT != hTitleCidTag)
        {
            TAG_eRemove(hTitleCidTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eWriteTextToTag
*
* common fxn that can be used for ArtistText, TitleText
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWriteTextToTag(
    TAG_OBJECT hParentTag,
    const char *pacTagName,
    STRING_OBJECT hText
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    TAG_OBJECT hTextTag = TAG_INVALID_OBJECT;

    // add the text tag
    eReturn = TAG_eAdd(
          pacTagName,
          hParentTag,
          &hTextTag,
          NULL
              );
    if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
    {
        // set the artist text tag value
        eReturn = TAG_eSetTagValue(
            hTextTag,
            TAG_TYPE_STRING,
            &hText,
            sizeof(STRING_OBJECT),
            FALSE
                );
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            // something went wrong, so remove the tag we added
            TAG_eRemove(hTextTag, FALSE);
        }
    }

    return eReturn;
}

/*****************************************************************************
*
*   eAddToConfigFile
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddToConfigFile (
    TAG_OBJECT hParentTag,
    void *pvItem,
    STRING_OBJECT hArtistTextString,
    STRING_OBJECT hTitleTextString
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturn;
    AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psATSpecificInfo;
    SEEK_CONTENT_REGISTERED_ITEM_STRUCT *psGenericItem;

    psGenericItem = (SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)pvItem;
    psATSpecificInfo = (AT_SEEK_CONTENT_REGISTERED_ITEM_STRUCT *)
                           &psGenericItem->uServiceSpecific;

    // verify that the pointer to the data isn't NULL
    if ( psATSpecificInfo == NULL )
    {
        return SMSAPI_RETURN_CODE_ERROR;
    }

    do
    {
        // set the parent tag
        psGenericItem->hTag = hParentTag;

        // update the enable tag (the tag gets created in this case since
        // it doesn't already exist)
        eReturn = SEEK_CONTENT_eUpdateEnabledTag(psGenericItem, FALSE);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write the type tag
        eReturn = eWriteTypeToTag(hParentTag, psATSpecificInfo);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write the cid tags
        eReturn = eWriteCIDsToTag(hParentTag, psATSpecificInfo);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write artist text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      AT_SEEK_ARTIST_TEXT,
                      hArtistTextString);
        if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
        {
            break;
        }

        // write title text
        eReturn = eWriteTextToTag(
                      hParentTag,
                      AT_SEEK_TITLE_TEXT,
                      hTitleTextString);

        if (eReturn == SMSAPI_RETURN_CODE_SUCCESS)
        {
            eReturn = CM_eCommitChangesToFile();
        }

    } while (0);

    return eReturn;
}

/*****************************************************************************
*
*   bATSeekContentIterator
*
*****************************************************************************/
static BOOLEAN bATSeekContentIterator(
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    SEEK_CONTENT_OBJECT hContent,
    void *pvSeekContentIteratorArg)
{

    AT_SEEK_ENUM eType;
    AT_SEEK_ITERATOR_STRUCT *psStruct =
      (AT_SEEK_ITERATOR_STRUCT *)pvSeekContentIteratorArg;

    if (psStruct == NULL)
    {
        // stop iterating
        return FALSE;
    }

    // we are only looking for content registered for artist alert
    eType = AT_SEEK_CONTENT.eType(hContent);
    if (eType == AT_SEEK_TITLE)
    {
        CID_OBJECT hId;

        // does this match the cid we're looking for?
        hId = AT_SEEK_CONTENT_hTitleCID(hContent);
        if (CID.n16Equal(hId, psStruct->hId) == 0)
        {
            // is the artist alert enabled?
            BOOLEAN bEnabled;
            bEnabled = SEEK_CONTENT.bEnabled(hContent);
            if (bEnabled == TRUE)
            {
                // artist registered AND enabled
                psStruct->bMatch = TRUE;
                // stop iterating
                return FALSE;
            }
        }
    }
    // continue iterating
    return TRUE;
}
