/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:ALERTC_EVENT implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"
#include "alertc_event_obj.h"
#include "_alertc_event_obj.h"


/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   ALERTC_EVENT_hCreate
*
* This object interface method is a friend function that is used to create a
* an AlertC Event object
*
* Inputs:
*
*   hTrafficMgr - The SMS object that owns this object
*   tCode - An osal payload that points to the first byte for this Alert-c
*   hTrafficLocID - The market/bsa information for this AlertC message
*
* Outputs:
*
*   A valid ALERTC_EVENT on success, a ALERTC_EVENT_INVALID_OBJECT on error.
*
*****************************************************************************/
ALERTC_EVENT_OBJECT ALERTC_EVENT_hCreate (
    SMS_OBJECT hOwner,
    TRAFFIC_EVENT_CODE tCode,
    ALERTC_EVENT_TYPE_ENUM eType,
    TRAFFIC_MSG_CLASS_ENUM eClass,
    ALERTC_QUANTIFIER_TYPE_ENUM eQuantType,
    STRING_OBJECT hText,
    STRING_OBJECT hTextMod
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)ALERTC_EVENT_INVALID_OBJECT;
    BOOLEAN bOk;

    // Verify that owner object is valid and owned
    bOk = SMSO_bOwner(hOwner);
    if (bOk == FALSE)
    {
        // Error!
        return ALERTC_EVENT_INVALID_OBJECT;
    }

    // Create an instance of the ALERTC_EVENT object
    psObj = (ALERTC_EVENT_OBJECT_STRUCT *)
        SMSO_hCreate(
            ALERTC_EVENT_OBJECT_NAME,
            sizeof(ALERTC_EVENT_OBJECT_STRUCT),
            (SMS_OBJECT)hOwner, FALSE);

    if(psObj == NULL)
    {
        // Error!
        return ALERTC_EVENT_INVALID_OBJECT;
    }

    // Init the object members
    psObj->tCode = tCode;
    psObj->eType = eType;
    psObj->eClass = eClass;
    psObj->eQuantType = eQuantType;
    psObj->hText = hText;
    psObj->hTextMod = hTextMod;

    return (ALERTC_EVENT_OBJECT)psObj;
}

/*****************************************************************************
*
*       ALERTC_EVENT_tEventCode
*
*****************************************************************************/
TRAFFIC_EVENT_CODE ALERTC_EVENT_tEventCode (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    TRAFFIC_EVENT_CODE tCode = TRAFFIC_INVALID_EVENT_CODE;
    BOOLEAN bOwner;


    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        tCode = psObj->tCode;
    }

    return tCode;
}

/*****************************************************************************
*
*       ALERTC_EVENT_eType
*
*****************************************************************************/
ALERTC_EVENT_TYPE_ENUM ALERTC_EVENT_eType (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    ALERTC_EVENT_TYPE_ENUM eType = ALERTC_EVENT_TYPE_UNKNOWN;
    BOOLEAN bOwner;


    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        eType = psObj->eType;
    }

    return eType;
}

/*****************************************************************************
*
*       ALERTC_EVENT_eClass
*
*****************************************************************************/
TRAFFIC_MSG_CLASS_ENUM ALERTC_EVENT_eClass (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    TRAFFIC_MSG_CLASS_ENUM eClass = TRAFFIC_MSG_CLASS_UNKNOWN;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        eClass = psObj->eClass;
    }

    return eClass;
}

/*****************************************************************************
*
*       ALERTC_EVENT_eQuantType
*
*****************************************************************************/
ALERTC_QUANTIFIER_TYPE_ENUM ALERTC_EVENT_eQuantType (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    ALERTC_QUANTIFIER_TYPE_ENUM eQuantType = ALERTC_QUANTIFIER_TYPE_UNKNOWN;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        eQuantType = psObj->eQuantType;
    }

    return eQuantType;
}

/*****************************************************************************
*
*       ALERTC_EVENT_hText
*
*****************************************************************************/
STRING_OBJECT ALERTC_EVENT_hText (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    STRING_OBJECT hText = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        hText = psObj->hText;
    }

    return hText;
}

/*****************************************************************************
*
*       ALERTC_EVENT_hTextMod
*
*****************************************************************************/
STRING_OBJECT ALERTC_EVENT_hTextMod (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    STRING_OBJECT hTextMod = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        hTextMod = psObj->hTextMod;
    }

    return hTextMod;
}

/*****************************************************************************
*
*       ALERTC_EVENT_bSetEventCode
*
*****************************************************************************/
BOOLEAN ALERTC_EVENT_bSetEventCode (
    ALERTC_EVENT_OBJECT hEventInfo,
    TRAFFIC_EVENT_CODE tCode
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    BOOLEAN bOwner;

    // Determine if the caller owns this resource
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);

    // Verify inputs.
    if(bOwner == TRUE)
    {
        psObj->tCode = tCode;
    }

    return bOwner;
}

/*****************************************************************************
*
*       ALERTC_EVENT_n16Compare
*
*       This is the function used to compare two ALERTC_EVENT objects
*
*       Inputs:
*               None
*
*       Outputs:
*               0   - Objects have the same value (equal, error)
*               > 0 - Object1 is greater than (after) Object2
*               < 0 - Object1 is less than (before) Object2
*
****************************************************************************/
N16 ALERTC_EVENT_n16Compare(
    ALERTC_EVENT_OBJECT hEventInfo1,
    ALERTC_EVENT_OBJECT hEventInfo2
        )
{
    ALERTC_EVENT_OBJECT_STRUCT
        *psObj1 = (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo1,
        *psObj2 = (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo2;
    BOOLEAN bOwner = FALSE;

    // Verify inputs. Object handles must be valid and caller own both.
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo1);

    if(bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo2);

        if(bOwner == FALSE)
        {
            // Error!
            return N16_MIN;
        }
    }
    else
    {
        // Error!
        return N16_MIN;
    }

    if (psObj1->tCode < psObj2->tCode)
    {
        return -1;
    }
    else if (psObj1->tCode > psObj2->tCode)
    {
        return 1;
    }

    return 0;
}

/*****************************************************************************
*
*   ALERTC_EVENT_vDestroy
*
* This object interface method is used by the caller to destroy the specified
* ALERTC_EVENT object and all members of the object
*

* Inputs:
*
*   hEventInfo - A handle to a valid ALERTC_EVENT object that the caller wants
*                 to get rid of
*
* Outputs:
*
*   Nada
*
*****************************************************************************/
void ALERTC_EVENT_vDestroy (
    ALERTC_EVENT_OBJECT hEventInfo
        )
{
    ALERTC_EVENT_OBJECT_STRUCT *psObj =
        (ALERTC_EVENT_OBJECT_STRUCT *)hEventInfo;
    BOOLEAN bOwner;

    // Verify inputs. Object handle must be valid.
    bOwner = SMSO_bOwner((SMS_OBJECT)hEventInfo);
    if(bOwner == FALSE)
    {
        // Error!
        return;
    }

    // Clear memeber variables
    psObj->tCode = TRAFFIC_INVALID_EVENT_CODE;
    psObj->eType = ALERTC_EVENT_TYPE_UNKNOWN;
    psObj->eClass = TRAFFIC_MSG_CLASS_UNKNOWN;
    psObj->eQuantType = ALERTC_QUANTIFIER_TYPE_UNKNOWN;

    if (psObj->hText != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hText);
        psObj->hText = STRING_INVALID_OBJECT;
    }

    if (psObj->hTextMod != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psObj->hTextMod);
        psObj->hTextMod = STRING_INVALID_OBJECT;
    }

    // Free object instance
    SMSO_vDestroy((SMS_OBJECT)hEventInfo);

    return;
}


/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

