/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_WIND_RADII_AREA implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "agw_shape_obj.h"
#include "agw_storm_position_obj.h"
#include "agw_wind_radii_area_obj.h"
#include "_agw_wind_radii_area_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   eWindSpeed
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eWindSpeed (
    AGW_WIND_RADII_AREA_OBJECT hAgwWindRd,
    OSAL_FIXED_OBJECT hWindSpeed
        )
{
    AGW_WIND_RADII_AREA_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwWindRd,
                        AGW_SHAPE_TYPE_WIND_RADII_FIELD, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sWindSpeed,
                            hWindSpeed, TRUE);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}

/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_WIND_RADII_AREA_OBJECT hAgwWindRd,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    AGW_WIND_RADII_AREA_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwWindRd,
        AGW_SHAPE_TYPE_WIND_RADII_FIELD, (void**)&psObj);

    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    switch (eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return += fprintf(psFile, "%2u | %p\n",
                    psObj->sData.un16SequenceID,
                    psObj->sData.sWindSpeed.hValue
                        );
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            n32Return += fprintf(psFile,
                    "SHAPE TYPE: %s\n",
                    AGW_WIND_RADII_AREA_OBJECT_NAME);
            n32Return += fprintf(psFile,", Sequence ID: %u\n",
                    psObj->sData.un16SequenceID
                        );
            n32Return += fprintf(psFile,"\tWind speed:");
            n32Return += OSAL_FIXED.n32FPrintf(
                    psObj->sData.sWindSpeed.hValue, psFile, FALSE
                        );
            n32Return += fprintf(psFile,"\n");
        }
        break;
    }

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_WIND_RADII_AREA_hCreate
 *
 *****************************************************************************/
AGW_WIND_RADII_AREA_OBJECT AGW_WIND_RADII_AREA_hCreate(
    SMS_OBJECT hParent,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_WIND_RADII_AREA_OBJECT hResult = AGW_WIND_RADII_AREA_INVALID_OBJECT;
    AGW_WIND_RADII_AREA_OBJECT_STRUCT *psObj = 
        (AGW_WIND_RADII_AREA_OBJECT_STRUCT *)NULL;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        hResult = (AGW_WIND_RADII_AREA_OBJECT)
            AGW_SHAPE_hCreate(
                (void**)&psObj,
                AGW_WIND_RADII_AREA_OBJECT_NAME": Obj",
                psHeader,
                AGW_SHAPE_TYPE_WIND_RADII_FIELD,
                sizeof(AGW_WIND_RADII_AREA_OBJECT_STRUCT),
                0, /* Won't be a separate DSRL entry */
                hParent
                    );

        if (AGW_WIND_RADII_AREA_INVALID_OBJECT == hResult)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_WIND_RADII_AREA_OBJECT_NAME": failed to create\n"
                        );
            break;
        }

        // Init
        OSAL.bMemSet(&psObj->sData, 0, sizeof(psObj->sData));

    } while (FALSE);

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_WIND_RADII_AREA_bUpdate
 *
 *   Update object data and prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN AGW_WIND_RADII_AREA_bUpdate(
    AGW_WIND_RADII_AREA_OBJECT hAgwWindRd,
    AGW_WIND_RADII_AREA_DATA_STRUCT *psWindRdData
        )
{
    BOOLEAN bOk = FALSE;
    AGW_WIND_RADII_AREA_OBJECT_STRUCT *psObj = 
        (AGW_WIND_RADII_AREA_OBJECT_STRUCT *)NULL;

    do
    {
        // Check input
        if (psWindRdData == NULL)
        {
            break;
        }

        AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwWindRd,
                           AGW_SHAPE_TYPE_WIND_RADII_FIELD, (void**)&psObj);
        if (psObj == NULL)
        {
            break;
        }

        // Copy data at once
        psObj->sData = *psWindRdData;

        // Perform extra actions to copy FIXED objects
        bOk = AGW_MGR_bCopyFixedTo(&psWindRdData->sWindSpeed, &psObj->sData.sWindSpeed);
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != NULL))
    {
        // Clean up fields in case of error
        OSAL.bMemSet(&psObj->sData, 0, sizeof(psObj->sData));
    }

    return bOk;
}

/*****************************************************************************
 *
 *   AGW_WIND_RADII_AREA_vDestroy
 *
 *****************************************************************************/
void AGW_WIND_RADII_AREA_vDestroy(
    AGW_WIND_RADII_AREA_OBJECT hAgwWindRd
        )
{
    printf(AGW_WIND_RADII_AREA_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwWindRd);

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT)hAgwWindRd);

    return;
}

/*****************************************************************************
 *                              PRIVATE FUNCTIONS
 *****************************************************************************/
