/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_STORM_POSITION implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "agw_shape_obj.h"
#include "agw_storm_position_obj.h"
#include "_agw_storm_position_obj.h"
#include "agw_storm_obj.h"
#include "agw_wind_radii_area_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   hStormID
*
*****************************************************************************/
static STRING_OBJECT hStormID(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        hResult = AGW_STORM_hID(psObj->hStorm);
    }

    return hResult;
}


/*****************************************************************************
*
*   hStormName
*
*****************************************************************************/
static STRING_OBJECT hStormName(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        hResult = AGW_STORM_hName(psObj->hStorm);
    }

    return hResult;
}


/*****************************************************************************
*
*   ePositionType
*
*****************************************************************************/
static AGW_STORM_POSITION_TYPE_ENUM ePositionType(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    AGW_STORM_POSITION_TYPE_ENUM eResult = AGW_STORM_POSITION_TYPE_UNKNOWN;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        eResult = psObj->sData.ePositionType;
    }

    return eResult;
}


/*****************************************************************************
*
*   eDirection
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eDirection (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    OSAL_FIXED_OBJECT hDirection
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                            AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sDirection,
                                hDirection, psObj->sData.bDirectionAvailable);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   eSpeed
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eSpeed (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    OSAL_FIXED_OBJECT hSpeed
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(
                            &psObj->sData.sSpeed,
                            hSpeed,
                            psObj->sData.bSpeedAvailable
                                );
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   eMaxSustainedWinds
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eMaxSustainedWinds (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    OSAL_FIXED_OBJECT hMaxWinds
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(
                            &psObj->sData.sMaxWinds, hMaxWinds, TRUE
                                );
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   eGusts
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eGusts (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    OSAL_FIXED_OBJECT hGusts
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sData.sGusts, hGusts, TRUE);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   ePressure
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePressure (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    OSAL_FIXED_OBJECT hPressure
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj
                            );
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(
                            &psObj->sData.sPressure, hPressure, TRUE
                                );
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   tObservationTime
*
*****************************************************************************/
static TIME_T tObservationTime(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    TIME_T tResult = 0;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        tResult = psObj->sData.tTimeStamp;
    }

    return tResult;
}


/*****************************************************************************
*
*   eStormType
*
*****************************************************************************/
static AGW_STORM_TYPE_ENUM eStormType(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    AGW_STORM_TYPE_ENUM eResult = AGW_STORM_TYPE_UNKNOWN;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        eResult = psObj->sData.eStormType;
    }

    return eResult;
}

/*****************************************************************************
*
*   un16SequenceID
*
*****************************************************************************/
static UN16 un16SequenceID(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    UN16 un16Result = UN16_MAX;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        un16Result = psObj->sData.un16SequenceID;
    }

    return un16Result;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0, n32Temp;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    switch (eOutputOption)
    {
        case SMS_OUTPUT_OPTION_TERSE:
        {
            n32Return += fprintf(psFile,
                    AGW_STORM_POSITION_OBJECT_NAME" | ");
            n32Temp = STRING.n32FWrite(
                            hStormName(hAgwStormPos), psFile
                                );
            if(n32Temp>0)
            {
                n32Return += n32Temp;
            }
            n32Return += fprintf(psFile, " | %2u | %1u\n",
                    psObj->sData.un16SequenceID,
                    psObj->sData.ePositionType
                        );
        }
        break;
        case SMS_OUTPUT_OPTION_VERBOSE:
        case SMS_OUTPUT_OPTION_GROSS:
        {
            n32Return += fprintf(psFile,
                    "SHAPE TYPE: %s\n",
                    AGW_STORM_POSITION_OBJECT_NAME);
            n32Return += fprintf(psFile, "\tStorm ID: ");
            n32Temp = STRING.n32FWrite(hStormID(hAgwStormPos), psFile);
            if(n32Temp > 0)
            {
                n32Return += n32Temp;
            }
            n32Return += fprintf(psFile, ", Storm Name: ");
            n32Temp = STRING.n32FWrite(hStormName(hAgwStormPos), psFile);
            if(n32Temp > 0)
            {
                n32Return += n32Temp;
            }
            n32Return += fprintf(psFile,", Sequence ID: %u\n",
                    psObj->sData.un16SequenceID
                        );
            n32Return += fprintf(psFile,", Position type: %u\n",
                    psObj->sData.ePositionType
                        );

            if (psObj->sData.bDirectionAvailable)
            {
                n32Return += fprintf(psFile,"\tDirection:");
                n32Return += OSAL_FIXED.n32FPrintf(
                        psObj->sData.sDirection.hValue, psFile, FALSE
                            );
            }
            if (psObj->sData.bSpeedAvailable)
            {
                n32Return += fprintf(psFile,"\tSpeed:");
                n32Return += OSAL_FIXED.n32FPrintf(
                        psObj->sData.sSpeed.hValue, psFile, FALSE
                            );
            }
            n32Return += fprintf(psFile,"\tMax Sustained Winds:");
            n32Return += OSAL_FIXED.n32FPrintf(
                    psObj->sData.sMaxWinds.hValue, psFile, FALSE
                        );
            n32Return += fprintf(psFile,"\tGusts:");
            n32Return += OSAL_FIXED.n32FPrintf(
                    psObj->sData.sGusts.hValue, psFile, FALSE
                        );
            n32Return += fprintf(psFile,"\tPressure:");
            n32Return += OSAL_FIXED.n32FPrintf(
                    psObj->sData.sPressure.hValue, psFile, FALSE
                        );
            n32Return += fprintf(psFile, "\tTimeStamp: %d\n",
                    psObj->sData.tTimeStamp
                        );
            n32Return += fprintf(psFile, "\tStorm Type: %d\n",
                    psObj->sData.eStormType
                        );
        }
        break;
    }

    return n32Return;
}

/*******************************************************************************
 *
 *   eIterateWindRadii
 *
 *******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateWindRadii(
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    AGW_SHAPE_CALLBACK bCallback,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode = AGW_RETURN_CODE_ERROR;
    AGW_RETURN_CODE_ENUM eDataAgwReturnCode;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = 
        (AGW_STORM_POSITION_OBJECT_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    AGW_WIND_RADII_ITERATE_SHIM_ARG_STRUCT sArg;

    do
    {
        if (bCallback == NULL)
        {
            eAgwReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        eDataAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                                AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj
                                    );
        if (psObj == NULL)
        {
            eAgwReturnCode = eDataAgwReturnCode;
            break;
        }

        // Iterate thru all entries
        sArg.bIteratorCallback = bCallback;
        sArg.pvIteratorCallback = pvArg;

        eOsalReturnCode = OSAL.eLinkedListIterate(psObj->hWindRadiiList,
                                    bIterateWindRadiiShimCallback,
                                    (void*) &sArg);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to iterate through wind radii objects (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eAgwReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}

/*******************************************************************************
 *
 *   eIterateStormPosition
 *
 *******************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eIterateStormPosition(
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    AGW_SHAPE_CALLBACK bCallback,
    void *pvArg
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    // Get the real data
    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwStormPos,
                                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);
    if (psObj != NULL)
    {
        // Iterate through all SP entries
        eAgwReturnCode = AGW_STORM_eIteratePositions(psObj->hStorm, bCallback, pvArg);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}

/*******************************************************************************
 *
 *   eStormPositionHasIntersection
 *
 *******************************************************************************/
static AGW_RETURN_CODE_ENUM eStormPositionHasIntersection (
    AGW_SHAPE_OBJECT hShape,
    const AGW_LOCATION_DESC_STRUCT *psLocDesc
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Check intersection with the shape directly
        eReturnCode = AGW_SHAPE_eHasDirectIntersectionWith(hShape, psLocDesc);
        if (eReturnCode == AGW_RETURN_CODE_NO_INTERSECTION)
        {
            AGW_WIND_RADII_INTERSECTION_CHECK_ARG_STRUCT sIterator;
            SMSAPI_RETURN_CODE_ENUM eReturn;

            // Fill in iterator structure
            sIterator.eResult = AGW_RETURN_CODE_NO_INTERSECTION;
            sIterator.psLocDesc = psLocDesc;

            // Try to check intersection with wind radii if we have them
            eReturn = eIterateWindRadii((AGW_STORM_POSITION_OBJECT)hShape,
                                bStormPositionHasIntersectionWindRadiiIterator,
                                (void*)&sIterator
                                    );

            // Populate the iteration result
            eReturnCode = sIterator.eResult;

            if (eReturn != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_POSITION_OBJECT_NAME": failed to iterate wind radii to check intersection (%s)",
                    SMSAPI_DEBUG_pacReturnCodeText(eReturn)
                        );
                break;
            }
        }
        else if (eReturnCode != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to check direct intersection (%s)",
                AGW_MGR_pacGetAgwReturnCodeName(eReturnCode)
                    );
            break;
        }
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_STORM_POSITION_hCreate
 *
 *****************************************************************************/
AGW_STORM_POSITION_OBJECT AGW_STORM_POSITION_hCreate(
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_STORM_POSITION_OBJECT hResult = AGW_STORM_POSITION_INVALID_OBJECT;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        hResult = (AGW_STORM_POSITION_OBJECT)
            AGW_SHAPE_hCreate(
                (void**) &psObj,
                AGW_STORM_POSITION_OBJECT_NAME": Obj",
                psHeader,
                AGW_SHAPE_TYPE_STORM_POSITION,
                sizeof(AGW_STORM_POSITION_OBJECT_STRUCT),
                tDescriptorSize,
                hParent);

        if (AGW_STORM_POSITION_INVALID_OBJECT == hResult)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_POSITION_OBJECT_NAME": failed to create."
                        );
            break;
        }

        // Init
        psObj->hStorm = AGW_STORM_INVALID_OBJECT;
        psObj->hWindRadiiList = OSAL_INVALID_OBJECT_HDL;

        // Create list to accommodate Wind RADII objects
        eOsalReturnCode = OSAL.eLinkedListCreate(
                                &psObj->hWindRadiiList,
                                AGW_STORM_POSITION_OBJECT_NAME":WRLst",
                                NULL,
                                OSAL_LL_OPTION_NONE
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to cteate wind radii list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        // Set interface
        bOk = AGW_SHAPE_bSetInterface((AGW_SHAPE_OBJECT)hResult, &GsStormPositionCallback);
        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to set interface"
                    );
        }
    } while (FALSE);

    if ((bOk == FALSE) && (hResult != AGW_STORM_POSITION_INVALID_OBJECT))
    {
        AGW_STORM_POSITION_vDestroy(hResult);
        hResult = AGW_STORM_POSITION_INVALID_OBJECT;
    }

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_STORM_POSITION_bUpdate
 *
 *   Update object data and prepares object for collecting points
 *
 *****************************************************************************/
BOOLEAN AGW_STORM_POSITION_bUpdate(
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    AGW_STORM_POSITION_DATA_STRUCT *psStormPosData
        )

{
    BOOLEAN bOk = FALSE;
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = 
        (AGW_STORM_POSITION_OBJECT_STRUCT *)NULL;

    do
    {
        if (psStormPosData == NULL)
        {
            break;
        }

        AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
            AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);
        if (psObj == NULL)
        {
            break;
        }

        // Copy data at once
        psObj->sData = *psStormPosData;

        // Perform extra actions to update FIXED objects
        if (psStormPosData->bDirectionAvailable == TRUE)
        {
            bOk = AGW_MGR_bCopyFixedTo(&psStormPosData->sDirection,
                     &psObj->sData.sDirection);
            if (bOk == FALSE)
            {
                break;
            }
        }

        if (psStormPosData->bSpeedAvailable == TRUE)
        {
            bOk = AGW_MGR_bCopyFixedTo(&psStormPosData->sSpeed,
                     &psObj->sData.sSpeed);
            if (bOk == FALSE)
            {
                break;
            }
        }

        bOk = AGW_MGR_bCopyFixedTo(&psStormPosData->sMaxWinds,
                 &psObj->sData.sMaxWinds);
        if (bOk == FALSE)
        {
            break;
        }

        bOk = AGW_MGR_bCopyFixedTo(&psStormPosData->sGusts,
                 &psObj->sData.sGusts);
        if (bOk == FALSE)
        {
            break;
        }

        bOk = AGW_MGR_bCopyFixedTo(&psStormPosData->sPressure,
                 &psObj->sData.sPressure);
        if (bOk == FALSE)
        {
            break;
        }

        bOk = TRUE;
    } while (FALSE);

    if ((bOk == FALSE) && (psObj != NULL))
    {
        OSAL.bMemSet(&psObj->sData, 0, sizeof(psObj->sData));
    }

    return bOk;
}

/*****************************************************************************
 *
 *   AGW_STORM_POSITION_vDestroy
 *
 *****************************************************************************/
void AGW_STORM_POSITION_vDestroy(
    AGW_STORM_POSITION_OBJECT hAgwStormPos
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;

    printf(AGW_STORM_POSITION_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwStormPos);

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);

    if (psObj != NULL)
    {
        if (psObj->hStorm != AGW_STORM_INVALID_OBJECT)
        {
            // Detach the storm since we expect that the storm
            // will not be used in such cases
            psObj->hStorm = AGW_STORM_INVALID_OBJECT;
        }

        // remove all wind radii objects
        if (psObj->hWindRadiiList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;

            eOsalReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->hWindRadiiList,
                (OSAL_LL_RELEASE_HANDLER) AGW_WIND_RADII_AREA_vDestroy
                    );
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_POSITION_OBJECT_NAME": failed to clean up wind radii list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hWindRadiiList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_POSITION_OBJECT_NAME": failed to delete wind radii list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            psObj->hWindRadiiList = OSAL_INVALID_OBJECT_HDL;
        }
    }

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT)hAgwStormPos);

    return;
}

/*****************************************************************************
 *
 *   AGW_STORM_POSITION_eAttachStorm
 *
 *   This function sets associated storm with that particular storm position
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_STORM_POSITION_eAttachStorm(
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    AGW_STORM_OBJECT hStorm
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_ERROR;
    AGW_RETURN_CODE_ENUM eDataResult;

    do
    {
        if (hStorm == AGW_STORM_INVALID_OBJECT)
        {
            eResult = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        eDataResult = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);
        if (psObj == NULL)
        {
            eResult = eDataResult;
            break;
        }

        // We need to attach the storm
        eResult = AGW_STORM_eAddPosition(hStorm, hAgwStormPos);
        if (eResult != AGW_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to remove position from the storm (%s)",
                AGW_MGR_pacGetAgwReturnCodeName(eResult)
                    );
            break;
        }

        psObj->hStorm = hStorm;

    } while (FALSE);

    return eResult;
}

/*****************************************************************************
 *
 *   AGW_STORM_POSITION_eAppendWindRadii
 *
 *****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_STORM_POSITION_eAppendWindRadii(
    AGW_STORM_POSITION_OBJECT hAgwStormPos,
    AGW_WIND_RADII_AREA_OBJECT hWindRadii
        )
{
    AGW_STORM_POSITION_OBJECT_STRUCT *psObj = NULL;
    AGW_RETURN_CODE_ENUM eResult = AGW_RETURN_CODE_ERROR;
    AGW_RETURN_CODE_ENUM eDataResult;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;

    do
    {
        eDataResult = AGW_SHAPE_eData((AGW_SHAPE_OBJECT) hAgwStormPos,
                        AGW_SHAPE_TYPE_STORM_POSITION, (void**)&psObj);
        if (psObj == NULL)
        {
            eResult = eDataResult;
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListAdd(
                                    psObj->hWindRadiiList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR, hWindRadii
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_POSITION_OBJECT_NAME": failed to add entry (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eResult = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*****************************************************************************
 *                              PRIVATE FUNCTIONS
 *****************************************************************************/

/*******************************************************************************
 *
 *   bStormPositionHasIntersectionWindRadiiIterator
 *
 *******************************************************************************/
static BOOLEAN bStormPositionHasIntersectionWindRadiiIterator(
    AGW_SHAPE_OBJECT hWindRadiiShape,
    void *pvArg
        )
{
    BOOLEAN bContinue = TRUE;
    AGW_WIND_RADII_INTERSECTION_CHECK_ARG_STRUCT *psArg =
        (AGW_WIND_RADII_INTERSECTION_CHECK_ARG_STRUCT*) pvArg;

    psArg->eResult = AGW_SHAPE_eHasIntersectionWith(hWindRadiiShape, psArg->psLocDesc);
    if (psArg->eResult == AGW_RETURN_CODE_SUCCESS)
    {
        // We are ok and can stop iteration
        bContinue = FALSE;
    }
    else if (psArg->eResult != AGW_RETURN_CODE_NO_INTERSECTION)
    {
        // Bad thing happened
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            AGW_STORM_POSITION_OBJECT_NAME": failed to check wind radii intersection (%s)",
            AGW_MGR_pacGetAgwReturnCodeName(psArg->eResult)
                );
        bContinue = FALSE;
    }

    return bContinue;
}

/*******************************************************************************
 *
 *   bIterateWindRadiiShimCallback
 *
 * This function is used to make protection against AGW_SHAPE_CALLBACK prototype.
 * Since in general case the AGW_SHAPE_CALLBACK might have different arguments
 * in comparison with OSAL_LL_ITERATOR_HANDLER.
 *
 *******************************************************************************/
static BOOLEAN bIterateWindRadiiShimCallback(
    void *pvData,
    void *pvArg
        )
{
    BOOLEAN bContinue;
    AGW_WIND_RADII_AREA_OBJECT hWindRadii =
        (AGW_WIND_RADII_AREA_OBJECT)pvData;
    AGW_WIND_RADII_ITERATE_SHIM_ARG_STRUCT *psArg =
        (AGW_WIND_RADII_ITERATE_SHIM_ARG_STRUCT*) pvArg;

    bContinue = psArg->bIteratorCallback((AGW_SHAPE_OBJECT) hWindRadii,
                                         psArg->pvIteratorCallback
                                             );

    return bContinue;
}
