/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_STORM implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "_agw_storm_obj.h"
#include "agw_storm_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                           PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   AGW_STORM_hCreate
*
*   This function created the new Storm objects. Please note, in case of success
*   the STRING object passed through the arguments will be tracked by new object
*   and application must not release them. Otherwise, if function failed to create
*   such a new object the caller must take care these STRING objects.
*
*   Input:
*       hParent - handle of the parent object
*       phID - address of the STRING object which keeps the storm identifier.
*       phName - address of the STRING object which keeps the storm name.
*
*   Outputs:
*       Created storm object or AGW_STORM_INVALID_OBJECT in case of error.
*
*
*****************************************************************************/
AGW_STORM_OBJECT AGW_STORM_hCreate(
    SMS_OBJECT hParent,
    STRING_OBJECT *phID,
    STRING_OBJECT *phName
        )
{
    AGW_STORM_OBJECT hResult = AGW_STORM_INVALID_OBJECT;
    AGW_STORM_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;

    do
    {
        // Check input
        if ((phID == NULL) || (phName == NULL))
        {
            break;
        }

        // Create the instance
        psObj = (AGW_STORM_OBJECT_STRUCT*)SMSO_hCreate(
                            AGW_STORM_OBJECT_NAME":Obj",
                            sizeof(AGW_STORM_OBJECT_STRUCT), hParent, FALSE);
        if (psObj == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_OBJECT_NAME": failed to create the object");
            break;
        }

        // Create the list
        eOsalReturnCode = OSAL.eLinkedListCreate(&psObj->hPositionList,
                                AGW_STORM_OBJECT_NAME":PosList",
                                (OSAL_LL_COMPARE_HANDLER)n16StormPositionSort,
                                OSAL_LL_OPTION_NONE
                                    );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_OBJECT_NAME": failed to create Storm Position List (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        // Get strings ownership
        psObj->hStormID = *phID;
        psObj->hStormName = *phName;

        // Invalidate the string references
        *phID = STRING_INVALID_OBJECT;
        *phName = STRING_INVALID_OBJECT;

        // Populate the object
        hResult = (AGW_STORM_OBJECT) psObj;
    } while (FALSE);

    if (hResult == AGW_STORM_INVALID_OBJECT)
    {
        // Destroy in case of error
        vDestroyObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   AGW_STORM_vDestroy
*
*****************************************************************************/
void AGW_STORM_vDestroy(
    AGW_STORM_OBJECT hObj
        )
{
    BOOLEAN bOwner;

    // Check input
    bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
    if (bOwner == TRUE)
    {
        // Make real destroy
        vDestroyObject((AGW_STORM_OBJECT_STRUCT*)hObj);
    }

    return;
}

/*****************************************************************************
*
*   AGW_STORM_bCompareByID
*
*****************************************************************************/
BOOLEAN AGW_STORM_bCompareByID(
    AGW_STORM_OBJECT hObj,
    STRING_OBJECT hID
        )
{
    BOOLEAN bOK = FALSE;
    BOOLEAN bOwner;

    // Check input
    bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
    if ((bOwner == TRUE) && (hID != STRING_INVALID_OBJECT))
    {
        AGW_STORM_OBJECT_STRUCT *psObj = (AGW_STORM_OBJECT_STRUCT*)hObj;
        N16 n16Compare;

        // Compare the string
        n16Compare = STRING.n16Compare(psObj->hStormID, hID, TRUE);
        bOK = (n16Compare == 0) ? TRUE : FALSE;
    }

    return bOK;
}

/*****************************************************************************
*
*   AGW_STORM_eAddPosition
*
*****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_STORM_eAddPosition(
    AGW_STORM_OBJECT hObj,
    AGW_STORM_POSITION_OBJECT hStormPositionObject
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    AGW_STORM_OBJECT_STRUCT *psObj = (AGW_STORM_OBJECT_STRUCT*)hObj;

    BOOLEAN bOwner;

    do
    {
        // Check input
        bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
        if ((bOwner == FALSE) ||
            (hStormPositionObject == AGW_STORM_POSITION_INVALID_OBJECT))
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Add the entry
        eOsalReturnCode = OSAL.eLinkedListAdd(psObj->hPositionList,
                                    OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                                    (void*)hStormPositionObject
                                        );
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_OBJECT_NAME
                ": failed to add the position desc object to the list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   AGW_STORM_eIteratePositions
*
*****************************************************************************/
AGW_RETURN_CODE_ENUM AGW_STORM_eIteratePositions(
    AGW_STORM_OBJECT hObj,
    AGW_SHAPE_CALLBACK bCallback,
    void *pvCallbackArgv
        )
{
    AGW_RETURN_CODE_ENUM eReturnCode = AGW_RETURN_CODE_ERROR;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    AGW_STORM_OBJECT_STRUCT *psObj = (AGW_STORM_OBJECT_STRUCT*)hObj;
    AGW_STORM_POSITION_ITERATE_SHIM_ARG_STRUCT sArgs;
    BOOLEAN bOwner;

    do
    {
        // Check input
        bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
        if (bOwner == FALSE)
        {
            eReturnCode = AGW_RETURN_CODE_NOT_OWNER;
            break;
        }

        if (bCallback == NULL)
        {
            eReturnCode = AGW_RETURN_CODE_BAD_ARGUMENT;
            break;
        }

        // Fill in the args
        sArgs.bIteratorCallback = bCallback;
        sArgs.pvIteratorCallback = pvCallbackArgv;

        // Iterate through all known positions
        eOsalReturnCode = OSAL.eLinkedListIterate(
                                psObj->hPositionList, bIteratePositionShim,
                                (void*)&sArgs);
        if ((eOsalReturnCode != OSAL_SUCCESS) && (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                AGW_STORM_OBJECT_NAME": failed to iterate positions list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            break;
        }

        eReturnCode = AGW_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
*
*   AGW_STORM_hID
*
*****************************************************************************/
STRING_OBJECT AGW_STORM_hID(
    AGW_STORM_OBJECT hObj
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Check input
    bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
    if (bOwner == TRUE)
    {
        AGW_STORM_OBJECT_STRUCT *psObj = (AGW_STORM_OBJECT_STRUCT*) hObj;

        // Populate the result
        hResult = psObj->hStormID;
    }

    return hResult;
}

/*****************************************************************************
*
*   AGW_STORM_hName
*
*****************************************************************************/
STRING_OBJECT AGW_STORM_hName(
    AGW_STORM_OBJECT hObj
        )
{
    STRING_OBJECT hResult = STRING_INVALID_OBJECT;
    BOOLEAN bOwner;

    // Check input
    bOwner = SMSO_bOwner((SMS_OBJECT) hObj);
    if (bOwner == TRUE)
    {
        AGW_STORM_OBJECT_STRUCT *psObj = (AGW_STORM_OBJECT_STRUCT*) hObj;

        // Populate the result
        hResult = psObj->hStormName;
    }

    return hResult;
}

/*****************************************************************************
 *                           FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *                           PRIVATE FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   n16StormPositionSort
*
*****************************************************************************/
static N16 n16StormPositionSort(
    AGW_STORM_POSITION_OBJECT hStPos1,
    AGW_STORM_POSITION_OBJECT hStPos2
        )
{
    N16 n16Result;

    UN16 un16SeqId1 = AGW_STORM_POSITION.un16SequenceID(hStPos1);
    UN16 un16SeqId2 = AGW_STORM_POSITION.un16SequenceID(hStPos2);

    n16Result = COMPARE(un16SeqId1, un16SeqId2);

    return n16Result;
}

/*****************************************************************************
*
*   bIteratePositionShim
*
* This function is used to make protection against AGW_SHAPE_CALLBACK prototype.
* Since in general case the AGW_SHAPE_CALLBACK might have different arguments
* in comparison with OSAL_LL_ITERATOR_HANDLER.
*
*****************************************************************************/
static BOOLEAN bIteratePositionShim(
    void *pvData,
    void *pvArg
        )
{
    AGW_STORM_POSITION_OBJECT hStPos =
        (AGW_STORM_POSITION_OBJECT) pvData;
    AGW_STORM_POSITION_ITERATE_SHIM_ARG_STRUCT *psArg =
        (AGW_STORM_POSITION_ITERATE_SHIM_ARG_STRUCT*) pvArg;
    BOOLEAN bContinue;

    bContinue = psArg->bIteratorCallback(
                    (AGW_SHAPE_OBJECT)hStPos,
                    psArg->pvIteratorCallback);

    return bContinue;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject(
    AGW_STORM_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        if (psObj->hStormID != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hStormID);
            psObj->hStormID = STRING_INVALID_OBJECT;
        }

        if (psObj->hStormName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psObj->hStormName);
            psObj->hStormName = STRING_INVALID_OBJECT;
        }

        // Clean up storm positions list
        if (psObj->hPositionList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eOsalReturnCode;
            eOsalReturnCode = OSAL.eLinkedListRemoveAll(psObj->hPositionList, NULL);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_OBJECT_NAME
                    ": failed to remove all Storm Position Descs (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            eOsalReturnCode = OSAL.eLinkedListDelete(psObj->hPositionList);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_STORM_OBJECT_NAME
                    ": failed to delete Storm Position Descs list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode)
                        );
            }

            psObj->hPositionList = OSAL_INVALID_OBJECT_HDL;
        }

        // Release the data
        SMSO_vDestroy((SMS_OBJECT) psObj);
    }

    return;
}
