/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:AGW_PRESSURE_CENTER implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include <string.h>
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"

#include "agw_shape_obj.h"
#include "agw_pressure_center_obj.h"
#include "_agw_pressure_center_obj.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
 *                             PUBLIC FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
*
*   eType
*
*****************************************************************************/
static AGW_PRESSURE_TYPE_ENUM eType (
    AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr
        )
{
    AGW_PRESSURE_TYPE_ENUM eResult = AGW_PRESSURE_TYPE_UNKNOWN;
    AGW_PRESSURE_CENTER_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwPressCntr,
                        AGW_SHAPE_TYPE_PRESSURE_CENTER, (void**)&psObj);
    if (psObj != NULL)
    {
        eResult = psObj->eType;
    }

    return eResult;
}

/*****************************************************************************
*
*   un16Pressure
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM ePressure (
    AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr,
    OSAL_FIXED_OBJECT hPressure
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    AGW_RETURN_CODE_ENUM eAgwReturnCode;
    AGW_PRESSURE_CENTER_OBJECT_STRUCT *psObj = NULL;

    eAgwReturnCode = AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwPressCntr,
                        AGW_SHAPE_TYPE_PRESSURE_CENTER, (void**)&psObj);
    if (psObj != NULL)
    {
        eAgwReturnCode = AGW_SHAPE_ePopulateFixedValue(&psObj->sPressure,
                            hPressure, TRUE);
    }

    eReturnCode = AGW_MGR_eMapToSmsApiReturnCode(eAgwReturnCode);

    return eReturnCode;
}


/*****************************************************************************
*
*   n32FPrintf
*
*****************************************************************************/
static N32 n32FPrintf (
    AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr,
    FILE *psFile,
    SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
        )
{
    N32 n32Return = 0;
    AGW_PRESSURE_CENTER_OBJECT_STRUCT *psObj = NULL;

    AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwPressCntr,
        AGW_SHAPE_TYPE_PRESSURE_CENTER, (void**)&psObj);

    if ((psFile == NULL) || (psObj == NULL))
    {
        return EOF;
    }

    n32Return +=
            fprintf(psFile, AGW_PRESSURE_CENTER_OBJECT_NAME
                ": PRESSURE CENTER TYPE: "
                    );

    switch(psObj->eType)
    {
        case AGW_PRESSURE_TYPE_HIGH:
        {
            n32Return += fprintf(psFile, "AGW_PRESSURE_HIGH\n");
        }
        break;
        case AGW_PRESSURE_TYPE_LOW:
        {
            n32Return += fprintf(psFile, "AGW_PRESSURE_LOW\n");
        }
        break;
        case AGW_PRESSURE_TYPE_UNKNOWN:
        {
            n32Return += fprintf(psFile, "Unknown\n");
        }
        break;
    }

    n32Return +=
            OSAL_FIXED.n32FPrintf(psObj->sPressure.hValue, psFile, FALSE);

    return n32Return;
}

/*****************************************************************************
 *                              FRIEND FUNCTIONS
 *****************************************************************************/
/*****************************************************************************
 *
 *   AGW_PRESSURE_CENTER_hCreate
 *
 *****************************************************************************/
AGW_PRESSURE_CENTER_OBJECT AGW_PRESSURE_CENTER_hCreate(
    SMS_OBJECT hParent,
    size_t tDescriptorSize,
    AGW_PRODUCT_HEADER_STRUCT *psHeader
        )
{
    AGW_PRESSURE_CENTER_OBJECT_STRUCT *psObj = NULL;
    AGW_PRESSURE_CENTER_OBJECT hResult = AGW_PRESSURE_CENTER_INVALID_OBJECT;

    do
    {
        // Check inputs
        if (hParent == SMS_INVALID_OBJECT)
        {
            break;
        }

        // Create an instance of the AGW Msg object
        hResult = (AGW_PRESSURE_CENTER_OBJECT)
            AGW_SHAPE_hCreate(
                (void**) &psObj,
                AGW_PRESSURE_CENTER_OBJECT_NAME": Obj",
                psHeader,
                AGW_SHAPE_TYPE_PRESSURE_CENTER,
                sizeof(AGW_PRESSURE_CENTER_OBJECT_STRUCT),
                tDescriptorSize,
                hParent);

        if (AGW_PRESSURE_CENTER_INVALID_OBJECT == hResult)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PRESSURE_CENTER_OBJECT_NAME": failed to create\n"
                        );
            break;
        }

        // Init
        psObj->eType = AGW_PRESSURE_TYPE_UNKNOWN;

    } while (FALSE);

    return hResult;
}

/*****************************************************************************
 *
 *   AGW_PRESSURE_CENTER_bUpdate
 *
 *****************************************************************************/
BOOLEAN AGW_PRESSURE_CENTER_bUpdate(
    AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr,
    AGW_INTERNAL_PRESSURE_TYPE_ENUM ePressureType,
    UN16 un16Pressure
        )
{
    BOOLEAN bOk = FALSE;
    AGW_PRESSURE_CENTER_OBJECT_STRUCT *psObj = NULL;

    do
    {
        AGW_SHAPE_eData((AGW_SHAPE_OBJECT)hAgwPressCntr,
                            AGW_SHAPE_TYPE_PRESSURE_CENTER, (void**)&psObj);
        if (psObj == NULL)
        {
            break;
        }

        // Get public type based on internal
        psObj->eType = AGW_MGR_eGetPublicPressureType(ePressureType);
        if (psObj->eType == AGW_PRESSURE_TYPE_UNKNOWN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    AGW_PRESSURE_CENTER_OBJECT_NAME
                    ": incorrect pressure type (%d)",
                    ePressureType
                        );
            break;
        }

        psObj->sPressure.hValue =
            AGW_MGR_hCreateFixedFromIntInMem(un16Pressure, 0,
                psObj->sPressure.aValueData);

        bOk = TRUE;
    } while (FALSE);

    return bOk;
}


/*****************************************************************************
 *
 *   AGW_PRESSURE_CENTER_vDestroy
 *
 *****************************************************************************/
void AGW_PRESSURE_CENTER_vDestroy(
   AGW_PRESSURE_CENTER_OBJECT hAgwPressCtrl
        )
{
    printf(AGW_PRESSURE_CENTER_OBJECT_NAME
            ": attempting to destroy object 0x%p\n",
            hAgwPressCtrl);

    AGW_SHAPE_vDestroy((AGW_SHAPE_OBJECT)hAgwPressCtrl);

    return;
}
