/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _WS_ALERTS_MGR_OBJ_H_
#define _WS_ALERTS_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "ws_alerts_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define WS_ALERTS_MGR_OBJECT_NAME "WSAlertsService"

#define WS_ALERTS_CREATE_DB_IN_MEMORY 1

// Constant which defines the size of the buffer used by
// ALERTS service for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define WS_ALERTS_SHARED_BUFFER_LEN ( 100 )

//Actual maximum size (in bytes) of the ISO 8859-1 Coded string that we can receive in the update.
//Since the Baudot string is smaller than ISO coded string we can use this size for both.
//Size is computed from the following formula: (LBSIZE +1)*8 where LBSIZE is between 0 and 4095.
#define WS_ALERTS_MAX_SIZE_OF_DATA_STRING ( 4096 )

//Constant which defines maximum LUT name length
#define WS_ALERTS_LUT_NAME_BUFFER_LEN ( 13 )

// WS Alerts update threshold for device-based DSRL
#define WS_ALERTS_DEVICE_DISTANCE_THRESHOLD     (10)

// WS Alerts clean up timer max value
#define WS_ALERTS_CLEAN_UP_TIMER_UTC_SEC_MAX     N32_MAX

// Supported service startup options
#define WS_ALERTS_SUPPORTED_OPTIONS (      \
    (DATASERVICE_OPTIONS_MASK)           \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      | \
    DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES   \
    )

// decimal 0.0003 corresponds to ~ 2 meters
#define WS_ALERTS_LOCATION_OFFSET_FIXED_VALUE (1258)

//Location ID: 1st byte - Type of location
//             2nd byte - Tabindex
//             last 2 bytes - Lcode
#define WS_ALERTS_GET_TYPE_FROM_LOCATION_ID(locid) \
    (UN8)((locid) >> 24)
#define WS_ALERTS_GET_TABINDEX_FROM_LOCATION_ID(locid) \
    (UN8)(((locid) >> 16) & 0x00FF)
#define WS_ALERTS_GET_LCODE_FROM_LOCATION_ID(locid) \
    (UN16)((locid) & 0x0000FFFF)

#define WS_ALERTS_GENERATE_LOCATION_ID(tableid, tabindex, lcode) \
    (LOC_ID)( (tableid << 24) | (tabindex << 16) | lcode );

#define WS_ALERTS_CONSTRUCT_LOCATION_ID_FROM_POLYGON_ID(polid) \
    ((UN32)(1<<31) | polid)

//WS Alert decoder

//WS_alerts Look Up tables class
#define WS_ALERTS_LUT_CLASS_ALERT_TYPE 0
#define WS_ALERTS_LUT_CLASS_ALERT_LOCATION 1
#define WS_ALERTS_LUT_CLASS_COMMON_PHRASE_CODE 2

#define WS_ALERTS_STRING_ELEMENT_QUESTION_MARK '?'
#define WS_ALERTS_STRING_ELEMENT_TERMINATOR '\0'
#define WS_ALERTS_STRING_ELEMENT_COMMA_SPACE ", "
#define WS_ALERTS_STRING_ELEMENT_COMMA ','
#define WS_ALERTS_STRING_ELEMENT_SPACE ' '
#define WS_ALERTS_STRING_ELEMENT_BRACKETS_VALUES ") values ("
#define WS_ALERTS_STRING_ELEMENT_BRACKET_SEMICOLON ");"

//SQLite data types
#define WS_ALERTS_TYPE_INT "int"
#define WS_ALERTS_TYPE_BLOB "blob"
#define WS_ALERTS_TYPE_TEXT "text"
#define WS_ALERTS_TYPE_UNKNOWN "unknown"

// Max number of parameters among all used prepared statements
#define MAX_SQL_BIND_PARAMS 4


  /**************/
 /** TYPEDEFS **/
/**************/

// Alerts Facing object
typedef struct ws_alerts_app_object_struct
{
    // Argument for DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES
    BOOLEAN bUpdateRefDB;

    // Path to the database file
    char *pacFullyQualifiedDatabaseFilePath;

    // Shared buffer
    char acBuffer[WS_ALERTS_MAX_SQL_STRING_LENGTH];

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // DB Connection Handle & File Path for ref database
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    char *pacRefDatabaseFilePath;

    // SQL prepared statement for selecting IDs from West RTree table
    SQL_PREPARED_STATEMENT_HANDLE hSelectIdsFromWestRTreeStmt;

    // SQL prepared statement for selecting IDs from East RTree table
    SQL_PREPARED_STATEMENT_HANDLE hSelectIdsFromEastRTreeStmt;

    // DB Connection Handle & File Path for persist database
    SQL_INTERFACE_OBJECT hSQLPersistConnection;
    char *pacRefDatabaseDirPath;
    char *pacPersistDatabaseFilePath;

    // SQL prepared statement for selecting alert messages by location
    SQL_PREPARED_STATEMENT_HANDLE hSelectAlertMsgsByLocStmt;

    // SQL prepared statement for selecting polygons
    SQL_PREPARED_STATEMENT_HANDLE hSelectPolygonsStmt;

    WS_ALERT_MSG_ID  tCurrentMsgId;

    // A list of active targets (a list of DSRLs for the application)
    OSAL_OBJECT_HDL hDSRLList;

    // A list of DSRL entries
    OSAL_OBJECT_HDL hEntriesList;

    // A list of DSRL entries candidates for removal
    OSAL_OBJECT_HDL hRemovedEntriesList;

    // A list of Locations WS_ALERTS_LOCATION_OBJECT
    OSAL_OBJECT_HDL hLocationPool;

    // A list of Look Up tables and its actual versions
    // Each entry contains a WS_ALERTS_LUT_VERSION_STRUCT
    OSAL_OBJECT_HDL hLUTVersionsList;

    // A list of IDs of locations (LOC_ID)
    // next to a specified target
    OSAL_OBJECT_HDL hLocIdList;

    // A list of locations WS_ALERTS_LOCATION_OBJECT
    // with IDs from hLocIdList
    OSAL_OBJECT_HDL hLocationList;

    // A list of IDs of messages associated with
    // locations from hLocationList
    OSAL_OBJECT_HDL hMsgIdList;

    SMS_LANGUAGE_ENUM eCurrentLanguage;

    UN32 un32TimerUTCSec;

    // Transaction ID used for DSRL updates
    UN32 un32TransactionID;

    // Dummy message used for search
    WS_ALERT_MSG_OBJECT hDummyMsg;

    // Dummy location object used for searching purposes only
    WS_ALERTS_LOCATION_OBJECT hDummyLoc;

} WS_ALERTS_APP_OBJECT_STRUCT;

// Alerts Manager object Structure
typedef struct ws_alerts_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The version of the DB contents
    UN16 un16DBContentVersion;

    // Path to reference database(s) location
    char *pacRefDatabaseDirPath;

    // Path to current reference database
    const char *pacCurRefDatabaseFilePath;

    // Path to next reference database
    const char *pacNextRefDatabaseFilePath;

    //Alerts protocol plugin handle
    WS_ALERTS_INTERFACE_OBJECT hInterface;

    // Flag to enable OTA ref DB updates
    BOOLEAN bRefDBUpdatesEnabled;

    // For periodic check of old messages
    DATASERVICE_TIMED_EVENT_HDL hCheckDbEvent;

} WS_ALERTS_MGR_OBJECT_STRUCT;

// Look Up table version structure
typedef struct ws_alerts_lut_version_struct
{
    UN32 un32LutId;
    UN8 un8Version;
} WS_ALERTS_LUT_VERSION_STRUCT;

// DSRL descriptor structure
typedef struct ws_alerts_dsrl_desc_struct
{
    OSAL_OBJECT_HDL hTargetList;
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
    OSAL_OBJECT_HDL hEntriesList;
} WS_ALERTS_DSRL_DESC_STRUCT;

// DSRL entry descriptor structure
typedef struct ws_alerts_dsrl_entry_desc_struct {
    OSAL_OBJECT_HDL hDSRLList;
    OSAL_LINKED_LIST_ENTRY hPoolEntry;
    // Indicates that entry is candidate to remove
    // if not equal to the same field in App object struct
    UN32 un32TransactionID;
} WS_ALERTS_DSRL_ENTRY_DESC_STRUCT;

// This struct is used as iterator argument at
// removing expired messages from DSRL
typedef struct ws_alerts_expired_msgs_iterator_struct
{
    UN32 un32CurUTCsec;
    UN32 un32NextExpirUTCsec;
} WS_ALERTS_EXPIRED_MSGS_ITERATOR_STRUCT;

// This struct is used as iterator argument at
// DSRL list modification
typedef struct ws_alerts_dsrl_entries_list_iterator_struct
{
    DSRL_OBJECT hDSRL;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;
} WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT;

typedef struct ws_alerts_msg_loc_struct
{
    WS_ALERT_MSG_ID tMsgId;
    OSAL_OBJECT_HDL hLocationsList;
} WS_ALERTS_MSG_LOC_STRUCT;

typedef struct ws_alerts_msg_loc_cont_struct
{
    WS_ALERTS_LOCATION_OBJECT hCurrentLocation;

    //linked list of WS_ALERTS_MSG_LOC_STRUCT
    OSAL_OBJECT_HDL hMsgIdList;
} WS_ALERTS_MSG_LOC_CONT_STRUCT;


// Structure used to perform ws alerts database queries
typedef struct ws_alerts_db_query_result_struct
{
    STRING_OBJECT hText;
    STRING_OBJECT hExtRef;
    STRING_OBJECT hExtType;
    UN32 *pun32LonLat;
    size_t tLonLatArraySize;
    UN8 un8Priority;
    BOOLEAN bResultantRows;
    WS_ALERTS_DB_ROW_UNION uDbRow;
} WS_ALERTS_DB_QUERY_RESULT_STRUCT;

// This structure is used to retrieve/store information from/to DB
typedef struct ws_alerts_db_result_struct
{
    BOOLEAN bResult;
    WS_ALERTS_DB_ROW_UNION uDbRow;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;
} WS_ALERTS_DB_RESULT_STRUCT;

typedef struct ws_alerts_db_query_objects_struct
{
    BOOLEAN bResult;
    OSAL_OBJECT_HDL hObjectsList;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;
} WS_ALERTS_DB_QUERY_OBJECTS_STRUCT;

typedef struct ws_alerts_location_key_struct
{
    SQL_INTERFACE_OBJECT hSQLPersistConnection;
    WS_ALERT_MSG_ID  tMsgId;
    LOC_ID  tLocationId;
} WS_ALERTS_LOCATION_KEY_STRUCT;

typedef struct ws_alert_msg_location_cross_struct
{
    LOCATION_OBJECT hCurrentLocation;
    BOOLEAN bCross;
    UN8 un8NumOfRayCrossings;
} WS_ALERT_MSG_LOCATION_CROSS_STRUCT;

typedef struct ws_alert_msg_loc_lists_struct
{
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc;
    WS_ALERTS_APP_OBJECT_STRUCT *psObj;
} WS_ALERT_MSG_LOC_LISTS_STRUCT;

typedef struct ws_alerts_location_box_struct
{
    N32 n32FixedBottomLeftLat;
    N32 n32FixedTopRightLat;
    N32 n32FixedhBottomLeftLon;
    N32 n32FixedTopRightLon;
} WS_ALERTS_LOCATION_BOX_STRUCT;

typedef struct ws_alerts_msg_elements_iterator_struct
{
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj;
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj;
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc;
    WS_ALERT_MSG_DESC_STRUCT sMsgDesc;
    BOOLEAN bCreateNewMsg;
    BOOLEAN bFastForward;
    BOOLEAN bResult;
} WS_ALERTS_MSG_ELEMENTS_ITERATOR_STRUCT;

typedef struct ws_alerts_labels_iterator_struct
{
    SQL_INTERFACE_OBJECT hConnection;
    char *pacBuffer;
    size_t tBufferSize;
    OSAL_OBJECT_HDL hLUTDescList;
    BOOLEAN bResult;
} WS_ALERTS_LABELS_ITERATOR_STRUCT;

typedef struct ws_alerts_message_ids_iterator_struct
{
    WS_ALERTS_APP_OBJECT_STRUCT *psObj;
    DSRL_OBJECT hDSRL;
    UN32 un32CurUTCsec;
    UN32 un32NearestEndTime;
    BOOLEAN bResult;
} WS_ALERTS_MESSAGE_IDS_ITERATOR_STRUCT;

typedef struct ws_alerts_location_ids_iterator_struct
{
    WS_ALERTS_APP_OBJECT_STRUCT *psObj;
    WS_ALERTS_LOCATION_OBJECT hWSAlertsLocation;
} WS_ALERTS_LOCATION_IDS_ITERATOR_STRUCT;

typedef struct ws_alerts_rtree_values_struct
{
    N32 n32LatMin;
    N32 n32LatMax;
    N32 n32LonMin;
    N32 n32LonMax;
    BOOLEAN bResult;
} WS_ALERTS_RTREE_DESC_STRUCT;

/****************/
/** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static WS_ALERTS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

static WS_ALERTS_APP_OBJECT_STRUCT *psGetAppFacingObject(
    WS_ALERTS_SERVICE_OBJECT hWSAlertsService
        );

static void vReleaseAppFacingObject(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static void vLocationDescRelease(
    WS_ALERTS_LOCATION_OBJECT hObj
        );

static void vReleaseMsgLocDesc(
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc
        );

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

/* Object Private Prototypes */

static BOOLEAN bHandleServiceReady (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

#if WS_ALERTS_CREATE_DB_IN_MEMORY == 0

static void vInitDbCleanUp(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static void vRemoveExpiredMsgs(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32CurrentTime
        );

#endif

static BOOLEAN bInitAppFacingObject (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static BOOLEAN bHandleServiceReady (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static void vSetError (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bLoadDBContentVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    N16 *pn16DBVer
        );

static BOOLEAN bProcessSelectUpdateVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psResult
        );

static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN32 un32SchemaVer
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static void vStartCleanUpTimer(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    UN32 un32NewUTCsec
        );

static void vStopCleanUpTimer (
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bLoadLUTVersions(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bProcessSelectTableVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bDBProcessSelectMsgIds (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_MSG_LOC_CONT_STRUCT *psMsgLocCont
        );

static BOOLEAN bDBProcessSelectLocations(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bDBProcessSelectPolygons(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bDBProcessSelectLocIds (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    OSAL_OBJECT_HDL hLocIdList
        );

static BOOLEAN bDBProcessSelectMessages (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_MSG_ROW_STRUCT *psResult
        );

static BOOLEAN bCreateMsgIdList(
    WS_ALERTS_MSG_LOC_CONT_STRUCT *psMsgLocCont,
    WS_ALERT_MSG_ID tMsgId
        );

static N16 n16CompareEntriesByLocID(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation1,
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation2
        );

static N16 n16CompareLocationsByMsgId (
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLoc1,
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLoc2
        );

//DSRL prototypes
static BOOLEAN bCreateList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bModifyList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDeleteList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRefreshList(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bRefreshDSRLs(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static WS_ALERT_MSG_OBJECT hSearchEntry(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERT_MSG_ID tMsgID
        );

static BOOLEAN bEntryAddToDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL,
    UN32 un32TransactionID
        );

static BOOLEAN bDSRLRemoveEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static void vDSRLRelease(
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLDescInit(
    DSRL_ARG_STRUCT *psDSRLArg,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static void vDSRLFinalizeEntry(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bDSRLAddTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static void vChangeAllDSRLsState(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_STATE_ENUM eState
        );

static BOOLEAN bDSRLChangeState (
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static void vChangeDSRLState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static BOOLEAN bRemoveUnusedEntries(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static void vRemoveExpiredDSRLEntries(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bDSRLRemoveExpiredEntry(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_EXPIRED_MSGS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bDSRLReplaceTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRemoveTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bDSRLRemoveAllTargets(
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bEntryRemoveDSRLLink(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    DSRL_OBJECT hDSRL
        );

static BOOLEAN bDSRLRemoveEntryLink(
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static BOOLEAN bDSRLRefresh(
    DSRL_OBJECT hDSRL,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    WS_ALERTS_DSRL_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bHandleCreateList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static N16 n16DSRLSortByPriority (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    void *pvSortArg
        );

static BOOLEAN bHandleModifyList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static BOOLEAN bHandleRefreshList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vHandleDeleteList(
    WS_ALERTS_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vRemoveAlertLocationRow(
    WS_ALERTS_LOCATIONS_ROW_STRUCT *psAlertLocationRow
        );

/* Manager Interface APIs */
static BOOLEAN bConstructMessages(
    WS_ALERTS_SERVICE_OBJECT hAlertsService,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc,
    BOOLEAN bRemoveMsgsFromDB
        );

static BOOLEAN bIterateMsgElements(
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem,
    WS_ALERTS_MSG_ELEMENTS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bCheckVerMin(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32LutId,
    UN8 un8MinVersion
        );

static BOOLEAN bInitMsgDesc(
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        );

static void vDestroyMsgDesc(
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        );

static void vDestroySArg(
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg
        );

static BOOLEAN bIsMsgDelimiter(
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem
        );

static BOOLEAN bProcessMessage(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc,
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElem,
    WS_ALERTS_DB_QUERY_RESULT_STRUCT *psArg,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        );

static BOOLEAN bProcessAlertLocationElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessAlertTypeElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bProcessAlertCommonPhraseElement(
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bPrepareMessageColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc
        );

static BOOLEAN bIterateLocationsToFill(
    LOC_ID tLocationId,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPrepareLocationsColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_LOCATION_KEY_STRUCT *psLocationKey
        );

static BOOLEAN bIterateMsgTypesToFill(
    WS_ALERTS_MSG_TYPES_ENTRY_STRUCT *psMsgTypesEntry,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPrepareMsgTypesColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_DB_RESULT_STRUCT *psResult
        );

static BOOLEAN bIteratePolygonsToFill(
    WS_ALERTS_POLYGON_DESC_STRUCT *psPolygon,
    WS_ALERTS_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPreparePolygonColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    WS_ALERTS_POLYGON_DESC_STRUCT *psPolygon
        );

static BOOLEAN bPrepareUpdateColumn(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    OSAL_LINKED_LIST_ENTRY *phLabelEntry
        );

static void vStateUpdateCompleted(
    WS_ALERTS_SERVICE_OBJECT hAlertsService
        );

static void vRemoveOldMsgs(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc
        );

static BOOLEAN bStoreMsgInDB(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    WS_ALERT_MSG_DESC_STRUCT *psMsgDesc,
    UN16 un16MsgSig
        );

static BOOLEAN bDBUpdateBegin(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        );

static BOOLEAN bDBUpdateEnd(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static BOOLEAN bDeleteRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bUpdateRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bGetCurrentDbVersion(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    STRING_OBJECT hLutName,
    N32 *pn32DBVer
        );

static BOOLEAN bCheckLutTable(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bUpdateRtreeTables(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc,
    UN16 un16Tabindex,
    UN16 un16Lcode
        );

static void vRemoveRtreeEntryById(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    LOC_ID tId
        );

static N16 n16CompareByLutId(
    WS_ALERTS_LUT_VERSION_STRUCT* psLUTVersion1,
    WS_ALERTS_LUT_VERSION_STRUCT* psLUTVersion2
        );

static BOOLEAN bLUTTableExist(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bProcessCheckTableOnExistance (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    BOOLEAN *pbTableExist
        );

static BOOLEAN bCheckLabelsList(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bIterateLabelDescs (
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc,
    void *pvArg
        );

static BOOLEAN bCreateLUTTable(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bProcessSelectLutDescription (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    OSAL_OBJECT_HDL psResult
        );

static void vReleaseDescriptionRow(
    WS_ALERTS_LUT_DESC_ROW *psLUTDesc
        );

static BOOLEAN bShapeIteratorCallbackPoint(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        );

static BOOLEAN bShapeIteratorCallbackSquare(
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        );

static BOOLEAN bRefDBBank (
    WS_ALERTS_SERVICE_OBJECT hWsAlertsService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        );

static void vTooOldDBVer (
    WS_ALERTS_SERVICE_OBJECT hWSAlertsService
        );

static BOOLEAN bDBProcessSelectLocIdsFromMsgId(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bDBProcessSelectMsgTypes(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    WS_ALERTS_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bAddTypesToMsg(
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hMsgTypeList,
    UN32 un32MsgId
        );

static BOOLEAN bAddRestLocationsToMsg (
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hLocationsList,
    UN32 un32MsgId
        );

static WS_ALERTS_LOCATION_OBJECT hGetLocationById (
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    UN32 un32LocationId
        );

static BOOLEAN bSelectLocationsByRTree (
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        );

static BOOLEAN bRemoveExtraLocations (
    WS_ALERTS_APP_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hLocation
        );

static BOOLEAN bGetAlertMsgsList(
    DSRL_TARGET_OBJECT hTarget,
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    DSRL_OBJECT hDSRL
        );

static STRING_OBJECT hGetDateTimeString (
    TIME_T ptSecs
        );

static BOOLEAN bIterateLocationIds (
    LOC_ID tLocId,
    WS_ALERTS_LOCATION_IDS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bIterateMsgIds(
    WS_ALERTS_MSG_LOC_STRUCT *psMsgLocDesc,
    WS_ALERTS_MESSAGE_IDS_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bChangeLUTVersion(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static N16 n16LUTDescComparator (
    void *pvObj1,
    void *pvObj2
        );

static BOOLEAN bCheckRowExist(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc,
    BOOLEAN *bExist
        );

static BOOLEAN bDBProcessSelectRowFromKey(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *bResult
        );

static BOOLEAN bInsertRow(
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psPartDesc
        );

static BOOLEAN bPrepareColumnForUpdate(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    OSAL_LINKED_LIST_ENTRY *phLabelEntry
        );

static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        );

static N16 n16CompareMsgByID(
    WS_ALERT_MSG_OBJECT hMsg1,
    WS_ALERT_MSG_OBJECT hMsg2
        );

static WS_ALERT_MSG_OBJECT hCreateMessage(
    WS_ALERTS_APP_OBJECT_STRUCT *psAppObj,
    OSAL_OBJECT_HDL *phLocationsList,
    WS_ALERT_MSG_ID tMsgID,
    UN32 un32CurrentTime
        );

/***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const WS_ALERTS_INTERFACE_STRUCT WS_ALERTS =
{
    /*.hStart = */       hStart,
    /*.vStop = */        (void (*) (WS_ALERTS_SERVICE_OBJECT hWSAlertsService))DATASERVICE_IMPL_vStop,
    /*.eState = */       (DATASERVICE_STATE_ENUM (*) (WS_ALERTS_SERVICE_OBJECT hWSAlertsService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */   (DATASERVICE_ERROR_CODE_ENUM (*) (WS_ALERTS_SERVICE_OBJECT hWSAlertsService))DATASERVICE_IMPL_eErrorCode,
    /*.eGetReferenceDataVersion = */eGetReferenceDataVersion
};

// Manager interface for plugins
const WS_ALERTS_MGR_INTERFACE_STRUCT GsWSAlertsMgrIntf =
{
    /*.bConstructMessages = */bConstructMessages,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.bRefDBBank = */bRefDBBank,
    /*.vTooOldDBVer = */vTooOldDBVer,
    /*.vStateUpdateCompleted = */vStateUpdateCompleted,
    /*.bDeleteRow = */bDeleteRow,
    /*.bUpdateRow = */bUpdateRow,
    /*.bGetCurrentDbVersion = */bGetCurrentDbVersion,
    /*.bCheckLutTable = */bCheckLutTable,
    /*.bCheckRowExist = */bCheckRowExist,
    /*.bInsertRow = */bInsertRow
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsWSAlertsStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif /* _WS_ALERTS_MGR_OBJ_H_ */
