/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio, Inc.         */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _STOCK_TICKER_MGR_OBJ_H_
#define _STOCK_TICKER_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "string_obj.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "db_util.h"
#include "stock_ticker_interface.h"
#include "stock_ticker_db_constants.h"
#include "stock_msg_obj.h"
#include "stock_symbol_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define STOCK_TICKER_MGR_OBJECT_NAME "StocksService"

// Constant which defines the size of the buffer used by
// weather for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define STOCK_TICKER_SHARED_BUFFER_LEN ( 300 )

// Defines number of hash items kept per message type
#define STOCK_TICKER_MSG_HASH_COUNT_MAX (50)

#if ((SMS_DEBUG==1) && (DEBUG_OBJECT == 1))
#define STOCK_TICKER_EXTRA_DEBUG 0 // make this in 1 if needed more debug output
#endif

// Stock Ticker symbol values timeout
#define STOCK_TICKER_VALUES_TIMEOUT ((UN32)(15 * 60)) /* sec */

// Stock Ticker symbol values timeout
#define STOCK_TICKER_TIMER_PERIOD ((UN32)(1 * 60)) /* sec */

// Supported service startup options
#define STOCK_TICKER_SUPPORTED_OPTIONS ( \
    (DATASERVICE_OPTIONS_MASK) DATASERVICE_OPTION_REFERENCE_DB_PATH \
        )

// Max number of parameters among all used prepared statements
#define MAX_SQL_BIND_PARAMS 2

  /**************/
 /** TYPEDEFS **/
/**************/

typedef UN8 STOCK_TICKER_MSG_FIELD_MASK;
#define STOCK_TICKER_MSG_FIELD_MASK_NONE   (STOCK_TICKER_MSG_FIELD_MASK)(0x00)
#define STOCK_TICKER_MSG_FIELD_MASK_NAME   (STOCK_TICKER_MSG_FIELD_MASK)(0x01)
#define STOCK_TICKER_MSG_FIELD_MASK_VALUES (STOCK_TICKER_MSG_FIELD_MASK)(0x02)
#define STOCK_TICKER_MSG_FIELD_MASK_ALL \
    (STOCK_TICKER_MSG_FIELD_MASK) \
        (STOCK_TICKER_MSG_FIELD_MASK_NAME | \
         STOCK_TICKER_MSG_FIELD_MASK_VALUES )

// Defines flags to control DSRL build
typedef UN8 STOCK_DSRL_BUILD_FLAGS;
#define STOCK_DSRL_BUILD_FLAG_NONE \
    ((STOCK_DSRL_BUILD_FLAGS) 0x00)
#define STOCK_DSRL_BUILD_FLAG_CHECK_EXISTENCE \
    ((STOCK_DSRL_BUILD_FLAGS) 0x01)
#define STOCK_DSRL_BUILD_FLAG_TRY_TO_REMOVE \
    ((STOCK_DSRL_BUILD_FLAGS) 0x02)
#define STOCK_DSRL_BUILD_FLAG_PROCESS_DSRL_LIST \
    ((STOCK_DSRL_BUILD_FLAGS) 0x04)

// Stock data items states
typedef enum stock_data_state_enum
{
    STOCK_DATA_STATE_INITIAL = 0,
    STOCK_DATA_STATE_UPDATED,
    STOCK_DATA_STATE_CREATED,
    STOCK_DATA_STATE_STABLE,
    STOCK_DATA_STATE_REMOVED,
    STOCK_DATA_STATE_ERROR
} STOCK_DATA_STATE_ENUM;

// Stock message descriptor
typedef struct stock_msg_desc_struct
{
    // Message state
    STOCK_DATA_STATE_ENUM eState;

    // Handled object
    STOCK_MSG_OBJECT hMessage;

    // Updated fields mask for UPDATED and CREATED state
    STOCK_TICKER_MSG_FIELD_MASK tFieldMask;

    // Last price update up-time
    UN32 un32LastUpdateTimeStamp;

    // Message desc handle in the manager's list
    OSAL_LINKED_LIST_ENTRY hEntry;

    // Message desc handle in the manager's pool list
    OSAL_LINKED_LIST_ENTRY hPoolEntry;

    // Keeps reference counter from the DSRLs
    UN32 un32Ref;

    // Keeps sign that this data is allocated for some reason
    BOOLEAN bIsAllocated;
} STOCK_MSG_DESC_STRUCT;

typedef struct stock_ticker_dsrl_target_desc_struct
{
    // Controlled target
    STOCK_SYMBOL_OBJECT hTarget;

    // Controlled target stock symbol index if exists
    // of STOCK_INVALID_ID otherwise
    STOCK_ID tStockId;

    // Keeps handler of the target in the monitored items list
    OSAL_LINKED_LIST_ENTRY hMonitorListEntry;

    // Keeps handler of the target in the owner DSRL list
    OSAL_LINKED_LIST_ENTRY hEntry;
} STOCK_TICKER_DSRL_TARGET_DESC_STRUCT;

// DSRL descriptor
typedef struct stock_ticker_dsrl_desc_struct
{
    // DSRL Entry
    DSRL_OBJECT hDSRL;

    // List of DSRL Targets
    OSAL_OBJECT_HDL hTargetList;

    // List of entries by StockMsgDesc
    OSAL_OBJECT_HDL hEntriesList;

    // Handle of the object in the DSRL list
    OSAL_LINKED_LIST_ENTRY hEntry;
} STOCK_TICKER_DSRL_DESC_STRUCT;

// Stock message Hash
typedef struct stocks_msg_hash_struct
{
    // Identifier in frame if DB
    UN32 un32Id;
    // Old identifier in frame if DB if state is UPDATED
    UN32 un32OldId;
    // Message type
    STOCK_TICKER_MSG_TYPE_ENUM eType;
    // Calculated CRC32 for certain message payload
    STOCK_TICKER_HASH tHash;
    // List handle
    OSAL_LINKED_LIST_ENTRY hEntry;
    // Message state
    STOCK_DATA_STATE_ENUM eState;
} STOCK_TICKER_MSG_HASH_STRUCT;

// Stock message Hash search struct
typedef struct stock_ticker_msg_hash_iterator_data_struct
{
    struct
    {
        // Calculated CRC32 for certain message payload
        OSAL_CRC_RESULT tHash;
    } sInput;

    struct
    {
        // Keeps found item or NULL in case of absence
        STOCK_TICKER_MSG_HASH_STRUCT *psFoundEntry;
        // Keeps item with lowest id (i.e. the oldest hash)
        // for this type
        STOCK_TICKER_MSG_HASH_STRUCT *psOldestEntry;
    } sOutput;
} STOCK_TICKER_MSG_HASH_ITERATOR_DATA_STRUCT;

// Stock Message Hash Control Structure
typedef struct stock_ticker_msg_hash_control_struct
{
    // Next hash record index
    UN32 un32NextIndex;
    // Keeps list of hashes per type
    OSAL_OBJECT_HDL hHashList[STOCK_TICKER_MSG_TYPE_MAX];
    // Keeps list of changed hashes
    OSAL_OBJECT_HDL hHashCacheList;
} STOCK_TICKER_MSG_HASH_CONTROL_STRUCT;

// Stocks Provider info control struct
typedef struct stock_ticker_provider_info_struct
{
    // Keeps data provider information text
    STRING_OBJECT hDataProviderInfoText;

    // Data state
    STOCK_DATA_STATE_ENUM eState;

    // Last update data sequence number
    UN8 un8SequenceNumber;

    // Last update data sequence number
    UN8 un8PrevSequenceNumber;
} STOCK_TICKER_PROVIDER_INFO_CONTROL_STRUCT;

typedef struct stock_ticker_mgr_ver_ctrl_struct
{
    // The version of the DB contents
    N16 n16DBContentVersion;
} STOCK_TICKER_MGR_VER_CTRL_STRUCT;

typedef struct stock_ticker_db_control_struct
{
    // SQL Connection
    SQL_INTERFACE_OBJECT hSQLConnection;

    // DB file name
    char *pacDBFilePath;

    // Keeps sign of the transaction state.
    BOOLEAN bInTransaction;

    // Keeps sign of updated symbols in DB
    BOOLEAN bSymbolsUpdated;

    // Version counter
    UN32 un32ContentVersion;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // Prepared SQL statement for selecting symbol by ID
    SQL_PREPARED_STATEMENT_HANDLE hSelectSymbolByIdStmt;

    // Prepared SQL statement for selecting data version
    SQL_PREPARED_STATEMENT_HANDLE hSelectDataVersionStmt;

    // Prepared SQL statement for updating symbol
    SQL_PREPARED_STATEMENT_HANDLE hUpdateSymbolStmt;

    // Prepared SQL statement for inserting symbol
    SQL_PREPARED_STATEMENT_HANDLE hInsertSymbolStmt;

} STOCK_TICKER_DB_CONTROL_STRUCT;

// Timer control structure
typedef struct stock_ticker_timer_control_struct
{
    DATASERVICE_TIMED_EVENT_HDL hExpirationEvent;

} STOCK_TICKER_TIMER_CONTROL_STRUCT;

// Stocks Facing object
typedef struct stock_ticker_app_object_struct
{
    // Keeps service initialized flag
    BOOLEAN bInitialized;

    // Keeps DSRL as DSRL_OBJECTs
    OSAL_OBJECT_HDL hDSRLList;

    // Keeps all known messages
    OSAL_OBJECT_HDL hSymbolMsgList;

    // Keeps all known messages
    OSAL_OBJECT_HDL hSymbolMsgPoolList;

    // Keeps all monitoring targets
    OSAL_OBJECT_HDL hTargetsMonitorList;

    // Keeps messages between start and stop requests
    // from the Protocol for fast processing
    OSAL_OBJECT_HDL hSymbolMsgCacheList;

    // Message hash cache control structure instance
    STOCK_TICKER_MSG_HASH_CONTROL_STRUCT sHashCtrl;

    // Stock provider info control structure
    STOCK_TICKER_PROVIDER_INFO_CONTROL_STRUCT sProviderInfoCtrl;

    // DB Connection Handle control struct
    STOCK_TICKER_DB_CONTROL_STRUCT sDB;

    //timer for periodical check of old messages
    STOCK_TICKER_TIMER_CONTROL_STRUCT sTimer;

    // Data service options
    DATASERVICE_OPTION_VALUES_STRUCT sOptionValues;

    // Service share buffer
    char acBuffer[STOCK_TICKER_SHARED_BUFFER_LEN];

    // Dummy msg used for search
    STOCK_MSG_OBJECT hDummyMsg;

} STOCK_TICKER_APP_OBJECT_STRUCT;

// Service manager data structure
typedef struct stock_ticker_mgr_object_struct
{
    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Stocks OTA data parser
    STOCK_TICKER_INTERFACE_OBJECT hParser;

    // SMS Object used to control the app-facing objects
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;

} STOCK_TICKER_MGR_OBJECT_STRUCT;

// Structure used to perform stocks database queries
typedef struct stock_ticker_db_query_result_struct
{
    struct
    {
        // Service handle
        STOCK_TICKER_SERVICE_OBJECT hService;
        // Reference to the App Facing object
        STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
        // User callback function
        STOCK_TICKER_SYMBOL_ITERATOR bIterator;
        // Aux data from the API user
        void *pvIteratorArg;
    } sInput;

    struct
    {
        BOOLEAN bResultantRows;
        UN32 un32HashId;
#if SMS_DEBUG == 1
        size_t tRowCount;
#endif
        STOCK_TICKER_DB_ROW_UNION uDbRow;
    } sOutput;
} STOCK_TICKER_DB_QUERY_RESULT_STRUCT;

// Structure to perform iterations in stocks
typedef struct stock_ticker_msg_descs_iterator_struct
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc;
    STOCK_DSRL_BUILD_FLAGS tFlags;
} STOCK_TICKER_MSG_DESCS_ITERATOR_STRUCT;

typedef struct stock_ticker_msg_for_dsrl_iterator_struct
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    STOCK_MSG_DESC_STRUCT *psMsgDesc;
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
} STOCK_TICKER_MSG_FOR_DSRL_ITERATOR_STRUCT;

typedef struct stock_ticker_target_desc_iterator_struct
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    STOCK_MSG_DESC_STRUCT *psMsgDesc;
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc;
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
} STOCK_TICKER_TARGET_DESC_ITERATOR_STRUCT;

typedef struct stock_ticker_dsrl_state_change_iterator_struct
{
    DSRL_STATE_ENUM eOldState;
    DSRL_STATE_ENUM eNewState;
} STOCK_TICKER_DSRL_STATE_CHANGE_ITERATOR_STRUCT;

typedef struct stock_ticker_msg_expiration_iterator_struct
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    UN32 un32ExpiredMessages;
    UN32 un32CurrentTimeStamp;
    STOCK_TICKER_PLUGIN_VALUES_ARG_STRUCT sValues;
} STOCK_TICKIER_MSG_EXPIRATION_ITERATOR_STRUCT;

typedef struct stock_ticker_target_monitor_list_iterator_struct
{
    STOCK_MSG_DESC_STRUCT *psMsgDesc;
    STRING_OBJECT hSymbolNameFromMessage;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;
} STOCK_TICKER_TARGET_MONITOR_LIST_ITERATOR_STRUCT;

typedef struct stock_ticker_msg_cache_list_process_iterator_struct
{
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj;
    BOOLEAN bIsDBUpdated;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;
} STOCK_TICKER_MSG_CACHE_LIST_PROCESS_ITERATOR_STRUCT;

typedef struct stock_msg_index_range_check_iterator_arg_struct
{
    STOCK_ID tFirstIndex;
    STOCK_ID tLastIndex;
    BOOLEAN bFound;
} STOCK_MSG_INDEX_RANGE_CHECK_ITERATOR_ARG_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/
/* Object Public Prototypes */

static STOCK_TICKER_SERVICE_OBJECT hStart(
   const char *pacSRHDriverName,
   DATASERVICE_EVENT_MASK tEventRequestMask,
   DATASERVICE_EVENT_CALLBACK vEventCallback,
   void *pvEventCallbackArg,
   DATASERVICE_OPTIONS_STRUCT const *psOptions
      );

static BOOLEAN bStockSymbolsUpdated(
    STOCK_TICKER_SERVICE_OBJECT hStocksService,
    UN32* pun32ContentVersion
        );

static SMSAPI_RETURN_CODE_ENUM eIterateStockSymbols (
    STOCK_TICKER_SERVICE_OBJECT hStocksService,
    STOCK_TICKER_SYMBOL_ITERATOR vCallback,
    void *pvvCallbackArg
       );

static STRING_OBJECT hDataProviderInfo (
    STOCK_TICKER_SERVICE_OBJECT hStocksService
       );

static STOCK_TICKER_RETURN_CODE_ENUM ePayloadProcessStarted (
    STOCK_TICKER_SERVICE_OBJECT hService
       );

static STOCK_TICKER_RETURN_CODE_ENUM ePayloadProcessCommited (
    STOCK_TICKER_SERVICE_OBJECT hService
       );

static STOCK_TICKER_RETURN_CODE_ENUM eUpdate (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT *psArg
       );

static STOCK_TICKER_RETURN_CODE_ENUM eIsPayloadProcessNeeded (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash
        );

static STOCK_TICKER_RETURN_CODE_ENUM eIsMessageNeeded (
    STOCK_TICKER_SERVICE_OBJECT hService,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT const *puData
        );

static const char *pacGetReturnCodeName (
    STOCK_TICKER_RETURN_CODE_ENUM eCode
        );

static const char *pacGetMessageTypeName(
    STOCK_TICKER_MSG_TYPE_ENUM eType
        );

/*** -- Object Private Prototypes -- ***/

static void vEventHandler(
   DATASERVICE_MGR_OBJECT hDataService,
   DATASERVICE_EVENT_MASK tCurrentEvent,
   void *pvEventArg,
   void *pvEventCallbackArg
      );

static void vSetError (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static void vDestroyObject (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

/* -- APP FACING OBJECT -- */
static STOCK_TICKER_RETURN_CODE_ENUM eInitAppFacingObject(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_APP_OBJECT_STRUCT *psGetAppFacingObject(
    STOCK_TICKER_SERVICE_OBJECT hStocksService
        );

static STOCK_TICKER_APP_OBJECT_STRUCT *psOwnedAppFacingObject(
    STOCK_TICKER_SERVICE_OBJECT hStocksService
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessPayload (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static STOCK_TICKER_RETURN_CODE_ENUM eUpdateProviderData (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_PROVIDER_INFO_ARG_STRUCT *psProviderInfo
        );

static STOCK_TICKER_RETURN_CODE_ENUM eUpdateSymbolValue (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_VALUES_ARG_STRUCT *psValues,
    STOCK_MSG_DESC_STRUCT *psKnownMsgDesc,
    BOOLEAN bUpdateTimeStamp,
    BOOLEAN bDoTouch
        );

static STOCK_TICKER_RETURN_CODE_ENUM eUpdateSymbol (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_PLUGIN_SYMBOL_ARG_STRUCT *psSymbol
        );

/* -- DB -- */
static STOCK_TICKER_RETURN_CODE_ENUM eInitDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDBStartTransaction(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDBEndTransaction(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eFlushDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bUseCache
        );

static STOCK_TICKER_RETURN_CODE_ENUM eLoadDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static UN32 un32DBExtractDataVersion (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDBUpdateDataVersion(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bDBProcessSelectDataVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    STOCK_TICKER_DB_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bDBVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static void vUninitDB(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceReady (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bHandleServiceError (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessServiceData(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

/* -- TIMER -- */
static STOCK_TICKER_RETURN_CODE_ENUM eExpirationTimerInit(
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eExpirationTimerUnInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static void vExpirationTimerStart (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        );

static void vExpirationTimerStop (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static void vExpirationTimerUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32PeriodInSec
        );

/* -- HASH -- */
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashItemCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    UN32 un32Id,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash,
    STOCK_TICKER_MSG_HASH_STRUCT **ppsResult
        );

static void vStockMsgHashItemDestroy (
    STOCK_TICKER_MSG_HASH_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vStockMsgHashItemDestroyNotFromList(
    STOCK_TICKER_MSG_HASH_STRUCT *psDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashLoadDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashIsPersistentStorageRequired (
    STOCK_TICKER_MSG_TYPE_ENUM eType
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashValuesClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheFlushDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashCacheUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    OSAL_CRC_RESULT tHash
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgHashUninit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static N16 n16StockMsgHashCompare (
    STOCK_TICKER_MSG_HASH_STRUCT *psObj1,
    STOCK_TICKER_MSG_HASH_STRUCT *psObj2
        );

static BOOLEAN bStockMsgHashIterator (
    void *pvData,
    void *pvArg
        );

/* -- MESSAGES -- */
static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bStockMsgDescIterateDBProcessResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescFlushDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN bFullData
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUnInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescCreate (
    STOCK_MSG_DESC_STRUCT **ppsDesc,
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DATA_STRUCT *psMsgData
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescGetStateInDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex,
    STRING_OBJECT hName
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescGetNameFromDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex,
    STRING_OBJECT *phName
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUpdateNameFromDB(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static void vStockMsgDescSetState(
    STOCK_MSG_DESC_STRUCT *psDesc,
    STOCK_DATA_STATE_ENUM eState
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescUpdateInPool(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psDesc,
    BOOLEAN bIsAllocated
        );

static STOCK_MSG_DESC_STRUCT *psStockMsgDescGet(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tIndex
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcess (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    BOOLEAN *pbIsDBUpdated
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescIsApplicableForTarget(
    STOCK_MSG_DESC_STRUCT *psDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcessForDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescProcessForDSRLs (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgDescInvalidateExpired (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN *pbExpired
        );

static void vStockMsgDescDestroy(
    STOCK_MSG_DESC_STRUCT *psDesc
        );

static void vStockMsgDescDestroyFull(
    STOCK_MSG_DESC_STRUCT *psDesc
        );

static void vStockMsgDescRelease(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheProcess (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheClean (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheAdd (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockMsgCacheUninit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

/* -- Data Provider Info */
static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoLoadDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoFlushDB (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eStockDataProviderInfoUnInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

/* -- DSRL -- */
static STOCK_TICKER_RETURN_CODE_ENUM eDSRLInit (
    STOCK_TICKER_MGR_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasToUpdate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasMessagesToUpdateByValues (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_ID tFirstIndex,
    STOCK_ID tLastIndex
        );

static STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDescCreate (
    DSRL_ARG_STRUCT *psDSRLArg
        );

static void vDSRLDescSetState (
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasEntryForTarget (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        );

static STOCK_MSG_DESC_STRUCT *psDSRLDescGetEntryByHandle (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    OSAL_LINKED_LIST_ENTRY *phEntry
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescStateTransition (
     STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
     DSRL_STATE_ENUM eOldState,
     DSRL_STATE_ENUM State
        );

static void vDSRLDescEntryFinalizeFunction (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hDSRLEntry,
    void *pvFinalizeArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescAddEntry (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescRemoveEntry (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    OSAL_LINKED_LIST_ENTRY hEntry,
    BOOLEAN bProcessAsDSRL
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescRemoveAllEntries (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    BOOLEAN bDoDSRLUpdate
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescHasEntry (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    OSAL_LINKED_LIST_ENTRY *phEntry
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescApplicableForMessage (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescBuild (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_DSRL_BUILD_FLAGS tFlags
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLDescBuildBasedOnTargets (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        );

static void vDSRLDescDestroy (
    DSRL_OBJECT hDSRL
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLUnInit (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static N16 n16DSRLDescGetEntryByHandleSearcher (
    STOCK_MSG_DESC_STRUCT *psMsgDesc,
    DSRL_ENTRY_OBJECT hDSRLEntry
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyAdd (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyReplace (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyRemove (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModifyRemoveAll (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListDestroy (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListModify (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLListRefresh (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    DSRL_ARG_STRUCT *psDSRLArg
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorUnInit(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorRegister(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorUnregister(
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetMonitorProcess(
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_MSG_DESC_STRUCT *psMsgDesc
        );

static STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psDSRLTargetCreate (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_SYMBOL_OBJECT *phTarget
        );

static STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psDSRLTargetGet (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_SYMBOL_OBJECT hTarget
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetRemove (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTarget
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetBuildAll (
    STOCK_TICKER_APP_OBJECT_STRUCT *psAppObj,
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc,
    DSRL_TARGET_OBJECT ahTargetList[],
    size_t tNumTargets
        );

static STOCK_TICKER_RETURN_CODE_ENUM eDSRLTargetRemoveAll (
    STOCK_TICKER_DSRL_DESC_STRUCT *psDSRLDesc
        );

static void vDSRLTargetRelease (
    STOCK_TICKER_DSRL_TARGET_DESC_STRUCT *psTargetDesc
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const STOCK_TICKER_SERVICE_INTERFACE_STRUCT STOCK_TICKER = {
   /*.hStart = */                   hStart,
   /*.vStop = */                    (void (*) (STOCK_TICKER_SERVICE_OBJECT hService))DATASERVICE_IMPL_vStop,
   /*.eState = */                   (DATASERVICE_STATE_ENUM (*) (STOCK_TICKER_SERVICE_OBJECT hService))DATASERVICE_IMPL_eState,
   /*.eErrorCode = */               (DATASERVICE_ERROR_CODE_ENUM (*) (STOCK_TICKER_SERVICE_OBJECT hService))DATASERVICE_IMPL_eErrorCode,
   /*.bStockSymbolsUpdated = */     bStockSymbolsUpdated,
   /*.eIterateStockSymbols = */     eIterateStockSymbols,
   /*.hDataProviderInfo = */        hDataProviderInfo
};

const STOCK_TICKER_SERVICE_MGR_INTERFACE_STRUCT GsStockTickerMgrIntf = {
    /*.ePayloadProcessStarted = */  ePayloadProcessStarted,
    /*.ePayloadProcessCommited = */ ePayloadProcessCommited,
    /*.eUpdate = */                 eUpdate,
    /*.eIsPayloadProcessNeeded = */ eIsPayloadProcessNeeded,
    /*.eIsMessageNeeded = */        eIsMessageNeeded,
    /*.pacGetReturnCodeName = */    pacGetReturnCodeName,
    /*.pacGetMessageTypeName = */   pacGetMessageTypeName
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsStocksStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)bHandleServiceError
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif /* _STOCK_TICKER_MGR_OBJ_H_ */
