/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SRM_OBJ_H_
#define _SRM_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include "srm_obj.h"
#include "sms_task.h"
#include "sms_update.h"
#include "tag_obj.h"
#include "radio.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define SRM_OBJECT_NAME "SRM"

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct srm_object_struct
{
    /* THINGS PROVIDED BY APPLICATION */

    // SRH Driver Name
    const char *pacDriverName;

    // SRM's SRH name
    const char *pacSrmName;

    /* THINGS TO INITIALIZE ONCE */

    // Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    // SRM object name
    const char *pacObjectName;

    // SRM Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // Services task for this SRM
    SMS_TASK_HANDLE hServicesTask;

    // Linked list of Modules associated with this SRM
    OSAL_OBJECT_HDL hModuleList;

    // Object State attributes

    // The actual state of the SRM object
    SRM_STATE_ENUM eState;

    // Radio Initialized
    BOOLEAN bRadioInitialized;

    // Device Handle
    FILE *psDevice;

    /* THINGS TO RE-INITIALIZE ON DEMAND */

    // Object's operational mode mask
    SMS_MODE_MASK tCurrentModes;

    // Object Error Codes
    SRM_ERROR_CODE_ENUM eErrorCode;

    // config manager tag
    TAG_OBJECT hTag;

    // Object which contains radio specific data
    RADIO_PRIVATE_DATA_OBJECT hRadioData;

    // flag to indicate that vUninitObject will need to be called when handling
    // an event in the STOPPED state
    BOOLEAN bUninitialized;

    // Object's reference counter
    UN16 un16RefCounter;

} SRM_OBJECT_STRUCT;

typedef struct srm_module_entry_struct
{
    MODULE_OBJECT hModule;
    BOOLEAN bValid;
    MODULE_ID tId;

} SRM_MODULE_ENTRY_STRUCT, *PSRM_MODULE_ENTRY_STRUCT;

typedef struct srm_module_in_reset_struct
{
    MODULE_OBJECT hModule;
    BOOLEAN bAllInReset;

} SRM_MODULE_IN_RESET_STRUCT, *PSRM_MODULE_IN_RESET_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SRM_OBJECT hGet (
    const char *pacDriverName,
    const char *pacSRMName,
    SRM_EVENT_MASK tEventRequestMask,
    SRM_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vRelease (
    SRM_OBJECT hSRM
        );

static SRM_STATE_ENUM eState (
    SRM_OBJECT hSRM
        );

static SRM_ERROR_CODE_ENUM eErrorCode (
    SRM_OBJECT hSRM
        );

static SRM_EVENT_MASK tEventMask (
    SRM_OBJECT hSRM
        );

static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        );

static BOOLEAN bUpdate (
    SRM_OBJECT hSRM,
    SRM_EVENT_MASK tMask
        );

/* Object Private Prototypes */

static SRM_OBJECT_STRUCT *psCreateObject (
    const char *pacDriverName,
    const char *pacName,
    SRM_EVENT_MASK tEventRequestMask,
    SRM_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vDestroyObject (
    SRM_OBJECT_STRUCT *psObj
        );

static N16 n16CompareModules (
    void *pvObj1,
    void *pvObj2
        );

static N16 n16CompareModuleId (
    void *pvArg1,
    void *pvArg2
        );

static BOOLEAN bResetModuleIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bAllInResetIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bReleaseModuleIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bInitializeObject (
    SRM_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    SRM_OBJECT_STRUCT *psObj
        );

static void vUpdateState (
    SRM_OBJECT_STRUCT *psObj,
    SRM_STATE_ENUM eNewState
        );

static SRM_STATE_ENUM eTransitionToErrorState (
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher,
    SRM_ERROR_CODE_ENUM eErrorCode
        );

static SRM_STATE_ENUM eTransitionToInitialState (
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static SRM_STATE_ENUM eTransitionToReadyState (
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static SRM_STATE_ENUM eTransitionToStoppedState (
    SRM_OBJECT_STRUCT *psObj,
    SMS_DESCRIPTOR_DISPATCH_STRUCT *psDispatcher
        );

static void vEventHandler (
    SRM_OBJECT hSRM,
    SMS_EVENT_HDL hEvent
        );

static BOOLEAN bModuleIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bDispatchToAll (
    void *pvDescriptor,
    SMS_EVENT_TYPE_ENUM eType,
    void *pvEventArg
        );

static void vSetError (
    SRM_OBJECT_STRUCT *psObj,
    SRM_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bProcessSrmEventMaskChange (
    SRM_OBJECT_STRUCT *psObj,
    SMS_EVENT_SRM_EVENT_MASK_STRUCT const *psSrmEventMask
        );

static BOOLEAN bAssociateWithModule (
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        );

/* Shutdown management */

static BOOLEAN bObjectBusy (
    SRM_OBJECT_STRUCT *psObj
        );

static BOOLEAN bTryStop (
    SRM_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitiateObjectRelease (
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static BOOLEAN bReleaseModules (
    SRM_OBJECT_STRUCT *psObj
        );

static BOOLEAN bPostFinalStop (
    SRM_OBJECT_STRUCT *psObj
        );

/* Event handlers */

static void vHandleInitEvent (
    SRM_OBJECT_STRUCT *psObj
        );

static void vHandleRadioReadyEvent(
    SRM_OBJECT_STRUCT *psObj,
    FILE *psDevice
        );

static void vHandleReleaseEvent (
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleAppInitiatedRelease (
    SRM_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleGeneralRelease (
    SRM_OBJECT_STRUCT *psObj
        );

static void vHandleStopEvent (
    SRM_OBJECT_STRUCT *psObj
        );

static void vHandleRemoveModuleEvent (
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        );

static void vHandleResetEvent (
    SRM_OBJECT_STRUCT *psObj,
    MODULE_OBJECT hModule
        );

  /***************/
 /** VARIABLES **/
/***************/

// SRM Object Defaults
static const SRM_OBJECT_STRUCT gsObjectDefaults =
{
    // SRH Driver Name
    NULL,

    // SRM's SRH name
    NULL,

    // Capabilities
    SRH_DEVICE_CAPABILITY_NONE,

    // SRM object name
    NULL,

    // SRM Update Control Structure
    {
        // Object update masks
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,

        // Object update callback list
        OSAL_INVALID_OBJECT_HDL,
        OSAL_INVALID_LINKED_LIST_ENTRY,

        // Object being updated
        NULL,

        // In-active callbacks
        FALSE
    },

    // Event Handler
    SMS_INVALID_EVENT_HANDLER,

    // Services task for this SRM
    SMS_INVALID_TASK_HANDLE,

    // Linked list of module descriptors associated with this SRM
    OSAL_INVALID_OBJECT_HDL,

    // The actual state of the SRM object
    SRM_STATE_INITIAL,

    // Radio Initialized
    FALSE,

    // Device Handle
    NULL,

    // Object's operational mode mask
    SMS_MODE_NONE,

    // Object Error Codes
    SRM_ERROR_CODE_NONE,

    // config manager tag
    TAG_INVALID_OBJECT,

    // Pointer to contain radio specific data
    RADIO_PRIVATE_DATA_INVALID_OBJECT,

    // bUninitialized
    FALSE,

    // un16RefCounter
    0

};

// SRM Object configuration for SMS
static const SMS_TASK_CONFIGURATION_STRUCT sSrmTaskConfiguration =
{
    /*.pacName = */"SRM", // gets changed later
    /*.un32StackSize = */8192, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_MEDIUM,
    /*.un32Options = */OSAL_TASK_OPTION_NONE,
    /*.un16ReportingInterval = */30, // seconds
    /*.un32EventQueueSize = */8, // events
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_NONE
};

// Global (re-usable) instance of an interface for this object
const SRM_OBJECT_INTERFACE_STRUCT SRM =
{
    /*.hGet = */hGet,
    /*.vRelease = */vRelease,
    /*.eState = */eState,
    /*.eErrorCode = */eErrorCode,
    /*.tEventMask = */tEventMask,
    /*.eModifyRegisteredEventMask = */eModifyRegisteredEventMask,
    /*.bUpdate = */bUpdate
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _SRM_OBJ_H_
