/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SAFECAM_MGR_OBJ_H_
#define _SAFECAM_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "srm_obj.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"
#include "safecam_interface.h"
#include "safecam_db_constants.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define SAFECAM_MGR_OBJECT_NAME "SafecamService"

// Constant which defines the size of the buffer used by
// SAFETY_CAMERAS service for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define SAFECAM_SHARED_BUFFER_LEN (SAFECAM_MAX_SQL_STRING_LENGTH)

// Constant defines the minimum device location
// delta which weather wants to know about
#define SAFECAM_DEVICE_DISTANCE_THRESHOLD (2)

// Supported service startup options
#define SAFECAM_SUPPORTED_OPTIONS (              \
    (DATASERVICE_OPTIONS_MASK)                  \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      | \
    DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES   \
    )

// Max number of parameters among all used prepared statements
#define MAX_SQL_BIND_PARAMS 4

  /**************/
 /** TYPEDEFS **/
/**************/

// SafeCam App Facing object
typedef struct safecam_app_object_struct
{
    // Handle SQL Connection
    SQL_INTERFACE_OBJECT hSQLRefConnection;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // Prepared statement for selecting sublocations by uid
    SQL_PREPARED_STATEMENT_HANDLE hSelectSubLocationsByUIDStmt;

    // Prepared statement for selecting alert locations by location
    SQL_PREPARED_STATEMENT_HANDLE hSelectAlertLocationsByLocStmt;

    // Shared buffer
    char acBuffer[SAFECAM_SHARED_BUFFER_LEN];

    // A list of SAFECAM LOCATION items
    OSAL_OBJECT_HDL hDSRLEntryPool;

    // A list of active targets (a list of DSRLs for the application)
    // Each entry contains a SAFECAM_DSRL_DESC_STRUCT
    OSAL_OBJECT_HDL hDSRLList;

    // List of all SAFETY_CAMERAS alert type objects
    OSAL_OBJECT_HDL hAlertTypesList;

    // Flag indicating status of alert type update
    // If alert types were updated in DB but not synchronized with objects
    // then flag is TRUE, otherwise FALSE
    BOOLEAN bAlertTypesUpdated;

    // Flag used to show alert types update state for application
    // Flag will be set to TRUE on any alert types update and skipped to FALSE
    // on eIterateAlertTypes call
    BOOLEAN bAppAlertTypesUpdateNotify;

    // Dummy location object used for search
    SAFETY_CAMERAS_LOCATION_OBJECT hDummySafeCamLoc;

    // List of DSRL entries which are candidates for destroying
    OSAL_OBJECT_HDL hRemovedEntriesList;

    // List of DSRL entries generated for current DSRL operation
    OSAL_OBJECT_HDL hNewDSRLEntriesList;

    // Transaction ID used for DSRL updates
    UN32 un32TransactionID;

} SAFECAM_APP_OBJECT_STRUCT;

// SafeCam Manager object Structure
typedef struct safecam_mgr_object_struct
{
    // SMS Object used to control the app-facing objects
    SAFECAM_APP_OBJECT_STRUCT *psAppObj;

    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // SafeCam protocol plugin handle
    SAFECAM_INTERFACE_OBJECT hInterface;

    // Path to reference database(s) location
    char *pacRefDatabaseDirPath;

    // Path to current reference database
    const char *pacCurRefDatabaseFilePath;

    // Path to next reference database
    const char *pacNextRefDatabaseFilePath;

    // Flag to enable OTA ref DB updates
    BOOLEAN bRefDBUpdatesEnabled;

} SAFECAM_MGR_OBJECT_STRUCT;

// DSRL description structure
typedef struct safecam_dsrl_desc_struct
{
    OSAL_OBJECT_HDL hTargetList;
    OSAL_LINKED_LIST_ENTRY hDSRLListEntry;
    OSAL_OBJECT_HDL hEntryList;
} SAFECAM_DSRL_DESC_STRUCT;

typedef struct safecam_dsrl_entry_desc_struct {
    OSAL_OBJECT_HDL hDSRLList;
    OSAL_LINKED_LIST_ENTRY hPoolEntry;
    UN32 un32TransactionID;
} SAFECAM_DSRL_ENTRY_DESC_STRUCT;

// This struct is used as iterator argument to iterate entries list
typedef struct safecam_entries_list_iterator_struct {
    SAFECAM_APP_OBJECT_STRUCT *psAppObj;
    DSRL_OBJECT hDSRL;
} SAFECAM_ENTRIES_LIST_ITERATOR_STRUCT;

// This structure is used to retrieve/store information from/to DB
typedef struct safecam_db_result_struct
{
    SAFECAM_DB_ROW_UNION uDbRow;
    BOOLEAN bResult;
} SAFECAM_DB_RESULT_STRUCT;

// This structure is used to store multiple objects from DB
typedef struct safecam_db_query_objects_struct {
    OSAL_OBJECT_HDL hObjectsList;
    BOOLEAN bResult;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj;
    // This flag is used to skip new state for alert types objects
    // right after creation
    BOOLEAN bSkipNew;
} SAFECAM_DB_QUERY_OBJECTS_STRUCT;

// This structure is used to fill location with sublocations from DB
typedef struct safecam_db_query_sublocations_struct {
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc;
    BOOLEAN bSpeedInMiles;
    SAFECAM_APP_OBJECT_STRUCT *psAppObj;
    BOOLEAN bResult;
} SAFECAM_DB_QUERY_SUBLOCATIONS_STRUCT;

// This structure is used to store lat and lon fixed values
// for geographical point
typedef struct safecam_geo_point_fixed_struct {
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT hLon;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE * 2];
} SAFECAM_GEO_POINT_FIXED_STRUCT;

// This structure represents min/max coordinates of alert location
// Rtree rectangle
typedef struct safecam_location_box_struct {
    OSAL_FIXED_OBJECT hLatMin;
    OSAL_FIXED_OBJECT hLatMax;
    OSAL_FIXED_OBJECT hLonMin;
    OSAL_FIXED_OBJECT hLonMax;
} SAFECAM_LOCATION_BOX_STRUCT;

// This type is used for sublocations iterator
typedef struct safecam_sublocations_iterator_struct {
    SAFETY_CAMERAS_ALERT_ID tAlertType;
    BOOLEAN bUpdateRequired;
    BOOLEAN bClear;
} SAFECAM_SUBLOCATIONS_ITERATOR_STRUCT;


  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static SAFETY_CAMERAS_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvAppEventCallbackArg,
    DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

static SMSAPI_RETURN_CODE_ENUM eIterateAlertTypes (
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    SAFETY_CAMERAS_ALERT_TYPES_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        );

static BOOLEAN bAddAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        );

static BOOLEAN bAddSubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        );

static BOOLEAN bModifyAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        );

static BOOLEAN bModifySubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        );

static BOOLEAN bDeleteAlertLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        );

static BOOLEAN bGetAlertLocationData (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        );

static BOOLEAN bGetSubLocationData (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        );

static BOOLEAN bUpdateAlertLocationRtree (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_LOCATIONS_ROW_STRUCT *psAlertLocation
        );

static BOOLEAN bAddAlertType (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        );

static BOOLEAN bDeleteAlertType (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        );

static BOOLEAN bIsAlertTypeExist (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertType
        );

static BOOLEAN bDeleteSubLocation (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    SAFECAM_SUB_LOCATIONS_ROW_STRUCT *psSubLocation
        );

static void vTooOldUpdate (
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
        );

static BOOLEAN bRefDBBank(
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    STRING_OBJECT *phInUseDB,
    STRING_OBJECT *phNextDB
        );

static BOOLEAN bDBUpdateBegin (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize
        );

static BOOLEAN bDBUpdateEnd (
    SQL_INTERFACE_OBJECT hConnection,
    char *pacBuffer,
    size_t tBufferSize,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static DATASERVICE_ERROR_CODE_ENUM eGetReferenceDataVersion (
    const char *pcContainingDirectoryPath,
    DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
    DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

/* Object Private Prototypes */
static BOOLEAN bInitAppFacingObject(
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        );

static SAFECAM_APP_OBJECT_STRUCT *psGetAppFacingObject(
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
        );

static void vReleaseAppFacingObject(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        );

static void vUninitAppFacingObject(
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static void vSetError (
    SAFECAM_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bDSRLUpdateContent(
    DSRL_OBJECT hDSRL,
    SAFECAM_APP_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCreateNewAlertTypes(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN bSkipNew
        );

static BOOLEAN bRemoveUnusedSafecamLocs(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj
        );

static BOOLEAN bEntryRemoveFromDSRL(
    DSRL_ENTRY_OBJECT hDSRLEntry,
    SAFECAM_ENTRIES_LIST_ITERATOR_STRUCT *psIterator
        );

static void vChangeDSRLState(
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState
        );

static N32 n32ExtractUpdateNumber(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBSchemaVersion(
    SQL_INTERFACE_OBJECT hSQLConnection,
    UN8 un8SchemaVer
        );

static N32 n32ExtractDataVersion(
    SQL_INTERFACE_OBJECT hConnection,
    void *pvArg
        );

static BOOLEAN bHandleServiceReady (
    SAFECAM_MGR_OBJECT_STRUCT *psObj
        );

static N16 n16CompareAlertTypeById (
     SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType1,
     SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType2
        );

static BOOLEAN bDBFillAlertTypeRowData(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_RESULT_STRUCT *psResult
        );

static void vAlertTypeRelease (
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType
        );

static SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psCreateAlertType(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    SAFECAM_ALERT_TYPES_ROW_STRUCT *psAlertTypeRow
        );

static SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psGetAlertTypeByID (
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    SAFETY_CAMERAS_ALERT_ID tId
        );

static BOOLEAN bDBCreateNewAlertTypes (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SAFECAM_DB_QUERY_OBJECTS_STRUCT *psResult
        );

static BOOLEAN bCreateNewAlertTypes(
    SAFECAM_APP_OBJECT_STRUCT *psAppObj,
    BOOLEAN bSkipNew
        );

static void vEntryRelease(
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
        );

static void vDSRLRelease (
    DSRL_OBJECT hDSRL
        );

static N16 n16CompareEntriesByUID (
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc1,
    SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc2
        );

static N16 n16CompareHandles(
    void *pvArg1,
    void *pvArg2
        );

  /***************/
 /** VARIABLES **/
/***************/

// Internal interface to Safety Camera Manager from plugin
const SAFECAM_MGR_INTERFACE_STRUCT GsSafeCamMgr =
{
    /*.bAddAlertLocation = */bAddAlertLocation,
    /*.bAddSubLocation = */bAddSubLocation,
    /*.bModifyAlertLocation = */bModifyAlertLocation,
    /*.bModifySubLocation = */bModifySubLocation,
    /*.bDeleteAlertLocation = */bDeleteAlertLocation,
    /*.bGetAlertLocationData = */bGetAlertLocationData,
    /*.bGetSubLocationData = */bGetSubLocationData,
    /*.bUpdateAlertLocationRtree = */bUpdateAlertLocationRtree,
    /*.bAddAlertType = */bAddAlertType,
    /*.bDeleteAlertType = */bDeleteAlertType,
    /*.bIsAlertTypeExist = */bIsAlertTypeExist,
    /*.bDeleteSubLocation = */bDeleteSubLocation,
    /*.bRefDBBank = */bRefDBBank,
    /*.bDBUpdateBegin = */bDBUpdateBegin,
    /*.bDBUpdateEnd = */bDBUpdateEnd,
    /*.vTooOldUpdate = */vTooOldUpdate
};

// Global (re-usable) instance of an interface for this object
const SAFETY_CAMERAS_INTERFACE_STRUCT SAFETY_CAMERAS = {
    /*.hStart = */              hStart,
    /*.vStop = */               (void (*) (SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService))DATASERVICE_IMPL_vStop,
    /*.eState = */              (DATASERVICE_STATE_ENUM (*) (SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */          (DATASERVICE_ERROR_CODE_ENUM (*) (SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService))DATASERVICE_IMPL_eErrorCode,
    /*.eIterateAlertTypes = */  eIterateAlertTypes,
    /*.eGetReferenceDataVersion = */ eGetReferenceDataVersion
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsSafeCamStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */DATASERVICE_IMPL_bGenericStateHandler
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif /* _SAFECAM_MGR_OBJ_H_ */
