/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _RFD_INTERFACE_OBJ_H_
#define _RFD_INTERFACE_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"

#include "sms_api.h"
#include "sms_task.h"
#include "sms_event_types.h"

#include "sql_interface_obj.h"
#include "rfd_receiver.h"
#include "rfd_platform_port.h"
#include "rfd_receiver_config.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

#define RFD_INTERFACE_OBJECT_NAME "RFDInt"

// These paths are built off of the SMS path
#define RFD_BASE_DIR          "rfd"
#define RFD_WORKING_DIR       "rfdr_w"
#define RFD_CONSUMER_DIR      "consum_w"
#define RFD_INTERFACE_DB_NAME "rfdint.db"

// This should be the largest RFDR_CLIENT_X_MAX_BLOCK_LEN_BYTES
// that is defined in rfd_receiver_config.h
#define RFD_MAX_PAYLOAD_SIZE_IN_BYTES (8192)

#define RFD_CONSUMER_TASK_STACK_SIZE (12 * 1024)    // 12K
#define RFD_CONSUMER_TASK_QUEUE_SIZE (16)
#define RFD_CONSUMER_TASK_REPORTING_INTERVAL (0)

// The number of chareters that should be present in the file
// name if the RFD file is an update file
#define RFD_UPDATE_FILE_NAME_LEN (5)

// The charecter used to indicate that the file name is an update file
#define RFD_UPDATE_FILE_START_CHAR 'U'

// The number of charecters used to indicate the version
// in the update file field
#define RFD_UPDATE_VER_STRING_LEN (2)

#define RFD_UPDATE_FILE_MAX_VERSION (99)
#define RFD_INTERFACE_DB_VERSION (2)
#define RFD_INTERFACE_MAX_FILE_ATTEMPTS (3)

// Mode in which all RFD files are opened
#define RFD_FILE_MODE "rb"

// Poll the RFD lib every 5 minutes
#define RFD_INTERFACE_FILE_POLL_RATE (5*60*1000)

#define RFD_CLIENT_TABLE_NAME "Clients"
#define RFD_PROGRESS_TABLE_NAME "Progress"
#define RFD_VERSION_TABLE_NAME "Version"

#define RFD_INTERFACE_CREATE_CLIENT_TABLE \
    "create table "RFD_CLIENT_TABLE_NAME"( " \
        "clientId int, "\
        "RFDRetry int, " \
        "RetryFileVer int, " \
        "primary key (clientId)" \
        ");"

#define RFD_INTERFACE_CLIENT_ID_FORMAT "%2u"

#define RFD_INTERFACE_CREATE_PROGRESS_TABLE \
    "create table "RFD_PROGRESS_TABLE_NAME"( " \
        "clientId int, "\
        "RFDFile varchar, " \
        "RFDFileVersion int, " \
        "progressIndex int, " \
        "numIntAttempts int, " \
        "primary key (clientId)" \
        ");"

#define RFD_INTERFACE_CREATE_VERSION_TABLE \
    "create table "RFD_VERSION_TABLE_NAME" ( " \
        "version int  " \
        " ); "

#define RFD_INTERFACE_INSERT_VERSION_ROW \
    "insert into "RFD_VERSION_TABLE_NAME" values(%d);"

#define RFD_INTERFACE_INSERT_VERSION_LEN (32)

#define RFD_INTERFACE_INSERT_CLIENT_ROW \
    "insert or ignore into "RFD_CLIENT_TABLE_NAME \
    " values( "RFD_INTERFACE_CLIENT_ID_FORMAT", 0, 0);"

#define RFD_INTERFACE_INSERT_CLIENT_LEN (64)

#define RFD_INTERFACE_SELECT_ALL_DB_VERSION \
    "select * from "RFD_VERSION_TABLE_NAME";"

#define RFD_INTERFACE_INSERT_PROGRESS_ROW \
    "insert or replace into "RFD_PROGRESS_TABLE_NAME" " \
    "values(?, ?, ?, ?, ?);"

#define RFD_INTERFACE_SET_RETRY_CLIENT_ROW \
    "update "RFD_CLIENT_TABLE_NAME" set " \
    "RFDRetry=%1u " \
    "where clientId="RFD_INTERFACE_CLIENT_ID_FORMAT";"

#define RFD_INTERFACE_SET_CLIENT_ROW \
    "update "RFD_CLIENT_TABLE_NAME" set " \
    "RFDRetry=0, RetryFileVer=%5u " \
    "where clientId="RFD_INTERFACE_CLIENT_ID_FORMAT";"

#define RFD_INTERFACE_CLIENT_ROW_LEN (72)

#define RFD_INTERFACE_UPDATE_IND_PROGRESS_ROW \
    "update "RFD_PROGRESS_TABLE_NAME" set " \
    "progressIndex=%10u " \
    "where clientId="RFD_INTERFACE_CLIENT_ID_FORMAT";"

#define RFD_INTERFACE_UPDATE_ATTEMPTS_PROGRESS_ROW \
    "update "RFD_PROGRESS_TABLE_NAME" set " \
    "numIntAttempts = %1u, progressIndex=0 " \
    "where clientId=%1u;"

#define RFD_INTERFACE_UPDATE_PROGRESS_LEN (80)

#define RFD_INTERFACE_CLEAR_PROGRESS_ROW \
    "delete from "RFD_PROGRESS_TABLE_NAME" " \
    "where clientId=%1u;"

#define RFD_INTERFACE_CLEAR_PROGRESS_LEN (48)

#define RFD_INTERFACE_SELECT_CLIENT_ROW \
    "select * from "RFD_CLIENT_TABLE_NAME" " \
    "where clientId="RFD_INTERFACE_CLIENT_ID_FORMAT";"

#define RFD_INTERFACE_SELECT_PROGRESS_ROW \
    "select * from "RFD_PROGRESS_TABLE_NAME" " \
    "where clientId="RFD_INTERFACE_CLIENT_ID_FORMAT";"

#define RFD_INTERFACE_SELECT_PROGRESS_LEN (48)

  /**************/
 /** TYPEDEFS **/
/**************/

/* RFD Ctrl object elements */
typedef struct rfd_interface_ctrl_object_struct
{
    // The RFD Receiver Handler
    RFDR_HANDLE hRFDReceiver;

    // The number of clients which have
    // made a connection to the RFD lib
    UN32 un32NumActiveConnections;

    // Progress tracking DB connection
    SQL_INTERFACE_OBJECT hSQL;

    // List tracks the number of outstanding
    // MFM connections we have
    OSAL_OBJECT_HDL hMFMConnections;

} RFD_INTERFACE_CTRL_OBJECT_STRUCT;

// Structure which specifies all attributes
// of a single database progress row that is
// available within the database
typedef struct rfd_interface_progress_row_struct
{
    // As per the interface object using this structure
    UN8 un8ClientId;

    // The RFD update file path
    STRING_OBJECT hFilePath;

    // The version associated with that file
    RFD_UPDATE_VERSION tFileVersion;

    // How far the client is in processing an RFD file
    RFD_PROGRESS_INDEX tProgressIndex;

    // The number of attempts made to process this file
    // by the RFD Interface
    UN8 un8NumIntAttempts;

} RFD_INTERFACE_PROGRESS_ROW_STRUCT;

typedef struct rfd_interface_client_row_struct
{
    // Flag indicates if we're currently retrying a file
    BOOLEAN bRFDRetry;

    // The version of the file under retry
    RFD_UPDATE_VERSION tRetryVersion;

} RFD_INTERFACE_CLIENT_ROW_STRUCT;

// Structure used for file processing event(s)
typedef struct rfd_interface_process_file_desc
{
    // The RFD connection we're working for
    RFD_INTERFACE_OBJECT hRFDConnection;

    // The callback that will be made when we have a new file
    RFD_FILE_PROCESSOR_CALLBACK eFileProcessorCallback;
    void *pvFileProcessorCallbackArg;

    // The progress info which is mirrored in the DB
    RFD_INTERFACE_PROGRESS_ROW_STRUCT sProgressRow;

    // The directory in which files are placed
    // for this client by the RFD Lib
    char *pcWorkingDirectory;

    // The file to process
    FILE *psRFDFile;

    // The status of file processing
    RFD_PROCESS_STATUS_ENUM eStatus;

    // The bit-width of the version numbers
    // used by this interface
    size_t tVersionBitWidth;

} RFD_INTERFACE_PROCESS_FILE_DESC;

/* Private object elements */
typedef struct rfd_interface_object_struct
{
    ///////////// RFD Lib Components /////////////
    BOOLEAN bRFDLibPresent;

    // Buffer Required to provide the RFD Lib with
    // input payloads
    UN8 aun8PayloadBuffer[RFD_MAX_PAYLOAD_SIZE_IN_BYTES];

    // Handles for manipulating the RFD SDK for this client
    RFD_COLLECTOR_THREAD_DATA_HANDLE hCollector;
    RFD_CONSUMER_THREAD_DATA_HANDLE  hConsumer;

    // We need the default message config info
    // so we can reference it during the lifetime
    // of this connection in order to
    // parse some of the data from RFD messages
    const RFD_MESSAGE_CONFIG_INFO *psMsgCfgInfo;

    ///////////// General Components /////////////

    // RFD Lib's Unique ID for this client.
    // Must map to the definition
    // in rfd_receiver_config.h
    UN8 un8ClientId;

    // The current version of the caller's DB
    RFD_UPDATE_VERSION tCurVersion;

    // The collecting file version
    RFD_UPDATE_VERSION tFileVersion;

    // Data stored in the DB for this client
    RFD_INTERFACE_CLIENT_ROW_STRUCT sClientRow;

    // The timer handler used to initiate
    // RFD Lib file polling and the associated event
    OSAL_OBJECT_HDL hPollTimer;
    SMS_EVENT_HDL hPollEvent;

    // Consumer task for this connection
    SMS_TASK_HANDLE hConsumerTask;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // This is a pointer to the structure used
    // to aid in events related to file processing.
    // This is only here to ensure we destroy it
    // when the client closes the connection
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess;

    // Flag indicates if this RFD client
    // has activated payload processing
    BOOLEAN bMsgProcessingActive;

    // TRUE if the client indicated that
    // the RFD update currently found
    // over the air is needed, and
    // FALSE otherwise.
    BOOLEAN bClientWantsThisRFDUpdate;

    // Flag indicates if the RFD connection
    // is in the process of shutting down,
    // which prevents it from processing events
    // queued after the shutdown event
    BOOLEAN bShuttingDown;

    // Optional callback struct for new RFD features
    RFD_OPTIONAL_CALLBACKS_STRUCT sOptCallbacks;

} RFD_INTERFACE_OBJECT_STRUCT;

// Enumeration specifying all the available fields
// in the db_version table
typedef enum rfd_interface_db_fields_enum {
    RFD_INTERFACE_DB_CLIENT_ID = 0,
    RFD_INTERFACE_DB_FILENAME,
    RFD_INTERFACE_DB_FILEVER,
    RFD_INTERFACE_DB_PROGRESS_INDEX,
    RFD_INTERFACE_DB_NUM_INT_ATTEMPTS,
    RFD_INTERFACE_DB_MAX_FIELDS

} RFD_INTERFACE_DB_FIELDS_ENUM;

typedef enum rfd_interface_db_client_fields_enum {
    RFD_INTERFACE_DB_CLIENT_CLIENT_ID = 0,
    RFD_INTERFACE_DB_CLIENT_RFD_RETRY,
    RFD_INTERFACE_DB_CLIENT_RETRY_VER,
    RFD_INTERFACE_DB_CLIENT_MAX_FIELDS

} RFD_INTERFACE_DB_CLIENT_FIELDS_ENUM;

// Structure which aids in selecting process
// data from the DB
typedef struct rfd_interface_progress_result_struct
{
    BOOLEAN bDataPresent;
    BOOLEAN bSuccess;
    RFD_INTERFACE_PROGRESS_ROW_STRUCT sProgressRow;

} RFD_INTERFACE_PROGRESS_RESULT_STRUCT;

typedef struct rfd_interface_client_result_struct
{
    RFD_INTERFACE_OBJECT_STRUCT *psObj;
    BOOLEAN bDataPresent;
    BOOLEAN bSuccess;

} RFD_INTERFACE_CLIENT_RESULT_STRUCT;

typedef enum rfd_lib_status_enum
{
    RFD_LIB_STATUS_ENABLED,
    RFD_LIB_STATUS_DISABLED,
    RFD_LIB_STATUS_ERROR

} RFD_LIB_STATUS_ENUM;

typedef struct rfd_interface_file_transfer_struct
{
    RFD_INTERFACE_OBJECT_STRUCT *psObj;
    BOOLEAN bFileReady;
    BOOLEAN bSuccess;

} RFD_INTERFACE_FILE_TRANSFER_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

/* Object Private Prototypes */

static RFD_LIB_STATUS_ENUM eInitRFD(void);

static BOOLEAN bStartRFD(
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCtrl
        );

static void vUninitRFD(void);

static RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCreateCtrlObject( void );

static void vUninitCtrlObject (
    RFD_INTERFACE_CTRL_OBJECT_STRUCT *psCtrl
        );

static void vUninitObject (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static void vEventHandler (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    SMS_EVENT_HDL hEvent
        );

static BOOLEAN bHandleNewFileEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static void vRFDFileTransferCallback (
    const TCHAR acFileName[],
    RFD_CONSUMER_FILE_INFO_STRUCT *psFileInfo,
    RFD_INTERFACE_FILE_TRANSFER_STRUCT *psTransfer
    );

static void vHandlePayloadEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bThisUpdateNeeded (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    size_t tPayloadSize
        );

static void vHandlePollEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static void vHandleProcessEvent (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static const char *pacRFDClientName (
    UN8 un8ClientId
        );

static BOOLEAN bLoadClientStatus (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bSetWorkingDir (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        );

static void vResetAll (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    BOOLEAN bRetryRFD
        );

static void vFinalizeFileProcessing (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    BOOLEAN bRetryNeeded
        );

static void vClearDirectory (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        );

static BOOLEAN bFileQualifier (
    const char *pcItemName,
    const void *pvArg
        );

static void vDeleteDBProgressRow (
    UN8 un8ClientId
        );

static void vClearMemoryProgressRow (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        );

static void vSetMemoryProgressRow (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    RFD_INTERFACE_PROGRESS_ROW_STRUCT *psProgressRow
        );

static BOOLEAN bStartProgressTracking (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    RFD_CONSUMER_FILE_INFO_STRUCT *psInfo,
    RFD_UPDATE_VERSION tUpdateFileVersion,
    const char *pcRFDFilePath
        );

static BOOLEAN bLoadCurrentProgress (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRewindProgress(
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        );

static BOOLEAN bCreateDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectVersion (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    BOOLEAN *pbVersionMatched
        );

static BOOLEAN bProcessSelectClient (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    RFD_INTERFACE_CLIENT_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectProgress (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    RFD_INTERFACE_PROGRESS_RESULT_STRUCT *psResult
        );

static BOOLEAN bPrepareProgressColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    RFD_INTERFACE_PROGRESS_ROW_STRUCT *psProgressRow
        );

static BOOLEAN bStartFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    SMS_EVENT_HANDLER hEventHdlr
        );

static void vResumeFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess,
    SMS_EVENT_HANDLER hEventHdlr
        );

static void vEndFileProcessing (
    RFD_INTERFACE_PROCESS_FILE_DESC *psProcess
        );

static BOOLEAN bInitFileProcessingAttrs (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    UN8 un8ClientId,
    RFD_FILE_PROCESSOR_CALLBACK eProcessFile,
    void *pvProcessFileArg,
    size_t tVersionBitWidth
        );

static void vUnInitFileProcessingAttrs (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static void vStartPollTimer (
    RFD_INTERFACE_OBJECT_STRUCT *psObj
        );

static void vPollTimerExpire (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static const char *pacRFDStatusText (
    RFD_STATUS eStatus
        );

static BOOLEAN bExtractVersionDefault (
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptBaseVersion,
    RFD_UPDATE_VERSION *ptNewVersion
        );

static void vReportExpiredDBDefault (
    RFD_INTERFACE_OBJECT hConnection,
    void *pvCallbackArg
        );

static RFD_FILE_STATUS_ENUM eRFDUpdateNeededDefault (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        );

static void vInitOptionalCallbacks (
    RFD_INTERFACE_OBJECT_STRUCT *psObj,
    RFD_OPTIONAL_CALLBACKS_STRUCT *psCallbacks
        );

  /***************/
 /** VARIABLES **/
/***************/

// Path variables that are used in rfd_receiver_config.h
// If we get some dynamic RFD SDK features, hopefully we will have an
// API to do this instead.
/*
extern char GacBasePathName[RFD_MAX_PATH_LEN];
extern char GacConsumerBasePathName[RFD_MAX_PATH_LEN];
*/

static RFD_INTERFACE_CTRL_OBJECT_STRUCT *gpsRFDCtrlObj =
    (RFD_INTERFACE_CTRL_OBJECT_STRUCT *)NULL;

// SRM Object configuration for SMS
static const SMS_TASK_CONFIGURATION_STRUCT sRFDIntTaskConfiguration =
{
    /*.pacName = */RFD_INTERFACE_OBJECT_NAME,
    /*.un32StackSize = */RFD_CONSUMER_TASK_STACK_SIZE, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_LOW,
    /*.un32Options = */OSAL_TASK_OPTION_DEBUG_OUTPUT,
    /*.un16ReportingInterval = */RFD_CONSUMER_TASK_REPORTING_INTERVAL, // seconds
    /*.un32EventQueueSize = */RFD_CONSUMER_TASK_QUEUE_SIZE, // events
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_DUAL_Q
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _RFD_INTERFACE_OBJ_H_
