/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio, Inc.         */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _MAPS_MGR_OBJ_H_
#define _MAPS_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "maps_interface.h"
#include "maps_db_constants.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define MAPS_MGR_OBJECT_NAME "MapsService"

/* Object name prefix for objects */
#define MAPS_APP_FACING_OBJECT_NAME MAPS_MGR_OBJECT_NAME":AppFacing"

// Supported service startup options
#define MAPS_SUPPORTED_OPTIONS (              \
    (DATASERVICE_OPTIONS_MASK)                \
    DATASERVICE_OPTION_REFERENCE_DB_PATH      \
    )

// Used to create full path to map image file
#define MAPS_PATH_BUILD_FMT "%s%s"

#define MAPS_MGR_BUFFER_SIZE (5120)

#define MAPS_MARKETS_COUNT_MAX (96)

#define MAPS_PARSE_SUBNAME_FMT "%[^0-9]%d"

// Map image files defines
#define MAPS_IMAGE_FILE_NAME_LENGHT_MAX (25) //WashingtonDC3200_V00.png
#define MAPS_IMAGE_FILE_EXT ".png"
#define MAPS_IMAGE_FILE_NAME_FMT "%s%d_V%02d"MAPS_IMAGE_FILE_EXT

// Horizontal resolution constants
#define MAPS_ZOOM_X_RESOLUTION_LOW        (400)
#define MAPS_ZOOM_X_RESOLUTION_MEDIUM     (800)
#define MAPS_ZOOM_X_RESOLUTION_HIGH       (1600)
#define MAPS_ZOOM_X_RESOLUTION_HIGHEST    (3200)

  // Max number of parameters among all used prepared statements
#define MAX_SQL_BIND_PARAMS 4

  /**************/
 /** TYPEDEFS **/
/**************/

// Markets descriptors structure (MFM File Identity Subname)
typedef struct maps_market_descriptor_struct
{
    char acMarketName[MAPS_SUBNAME_LENGHT_MAX];

    char acUpdateName[MAPS_SUBNAME_LENGHT_MAX];

    TRAFFIC_MARKET tMarketID;

    UN32 un32Version;

} MAPS_MARKET_DESCRIPTOR_STRUCT;

// Maps App Facing object struct
typedef struct maps_app_object_struct
{
    // DB Connection Handles
    SQL_INTERFACE_OBJECT hSQLRefConnection;
    SQL_INTERFACE_OBJECT hSQLPersistConnection;
    SQL_INTERFACE_OBJECT hSQLMemoryConnection;

    // Prepared Statements handles
    SQL_PREPARED_STATEMENT_HANDLE hFindByNameStmt;
    SQL_PREPARED_STATEMENT_HANDLE hFindNearestStmt;
    SQL_PREPARED_STATEMENT_HANDLE hFindMapDetailsStmt;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // Prioritized market ID
    TRAFFIC_MARKET tPrioritizedMarketId;
    // Subname descriptors
    UN16 un16MarketsCount;
    MAPS_MARKET_DESCRIPTOR_STRUCT asDescriptors[MAPS_MARKETS_COUNT_MAX];

} MAPS_APP_OBJECT_STRUCT;

// Service manager data structure
typedef struct maps_mgr_object_struct
{
    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The handle to the generic data service manager object
    DATASERVICE_MGR_OBJECT hDataServiceManager;

    // Maps OTA data parser
    MAPS_INTERFACE_OBJECT hParser;

    // Application facing object
    MAPS_APP_OBJECT_STRUCT *psAppObj;

    // DB File Path and Shared Buffer
    char *pcReferenceDBDirPath;
    char *pcPersistentDBDirPath;
    char acBuffer[MAPS_SHARED_BUFFER_LENGTH];

    UN8 aun8Buffer[MAPS_MGR_BUFFER_SIZE];
} MAPS_MGR_OBJECT_STRUCT;

// Structure used to perform database queries
typedef struct maps_db_query_result_struct
{
    BOOLEAN bResultantRows;
    MAPS_DB_ROW_UNION uDbRow;

} MAPS_DB_QUERY_RESULT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/
/* Object Public Prototypes */

static MAPS_SERVICE_OBJECT hStart(
   const char *pacSRHDriverName,
   const char *pacOutputDir,
   DATASERVICE_EVENT_MASK tEventRequestMask,
   DATASERVICE_EVENT_CALLBACK vEventCallback,
   void *pvEventCallbackArg,
   DATASERVICE_OPTIONS_STRUCT const *psOptions
      );

static SMSAPI_RETURN_CODE_ENUM ePersistentDataCleanup (
    MAPS_SERVICE_OBJECT hService,
    const char *pcReferenceDBPath,
    const char *pcPesistentDBPath
        );

static SMSAPI_RETURN_CODE_ENUM eFindNearestMap (
    MAPS_SERVICE_OBJECT hService,
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        );

static SMSAPI_RETURN_CODE_ENUM eLookupMapByName (
    MAPS_SERVICE_OBJECT hService,
    const char *pacMapName,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        );

static SMSAPI_RETURN_CODE_ENUM ePrioritizeMarket (
    MAPS_SERVICE_OBJECT hService,
    TRAFFIC_MARKET tMarketId
        );

/* Object Private Prototypes */

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bProcessMessage(
    MAPS_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bHandleServiceReady (
    MAPS_MGR_OBJECT_STRUCT *psObj
        );

static void vSetError (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static void vUninitObject (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bFullDelete
        );

static BOOLEAN bCreatePersistentDB (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bCreateMemoryDB(
    MAPS_MGR_OBJECT_STRUCT *psObj,
    SQL_INTERFACE_OBJECT hReferenceConection,
    SQL_INTERFACE_OBJECT hPersistentConection,
    SQL_INTERFACE_OBJECT *phMemoryConection
        );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
        );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
       );

static BOOLEAN bPrepareUpdatedMapInsert (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    void *pvCallbackArg
        );

static BOOLEAN bPrepareUpdatedImageInsert (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    void *pvCallbackArg
        );

static BOOLEAN bSelectMapByNameMemoryCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psBuild
        );

static BOOLEAN bSelectMapByNamePersistCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psQueryResult
        );

static BOOLEAN bSelectImagesVersionMemoryCallback (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    UN32 *pun32Version
        );

static BOOLEAN bProcessNearestAreaQuery (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_NEAREST_STRUCT *psBuild
        );

static BOOLEAN bProcessNearestMapQuery (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    MAPS_DB_QUERY_NEAREST_STRUCT *psBuild
        );

static BOOLEAN bUpdateMemoryDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvData
        );

static BOOLEAN bUpdateMemoryMapSetTable (
    MAPS_SERVICE_OBJECT hMapsService,
    MAPS_UPDATES_ROW_STRUCT *psUpdateRow
        );

static BOOLEAN bRemoveUnusedImageFiles (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvData
        );

static BOOLEAN bFillDescriptorsArray (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvData
        );

static SMSAPI_RETURN_CODE_ENUM eBuildFullMapPath (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    const char *pacDirPath,
    char *pacImageFilePath,
    size_t tImageFilePathSize
        );

static BOOLEAN bQueryMapByName (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
    MAPS_DB_QUERY_MAP_BY_NAME_STRUCT *psBuild
        );

static BOOLEAN bQueryMarketImagesVersion (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    const char *pacMapName,
    UN32 *pun32LatestVersion
        );

static BOOLEAN bInitAppFacingObject (
    MAPS_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitAppFacingObject(
    MAPS_APP_OBJECT_STRUCT *psAppObj
        );

static MAPS_APP_OBJECT_STRUCT *psGetAppFacingObject(
    MAPS_SERVICE_OBJECT hMapsService
        );

static N16 n16GetDescriptor (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacName,
    MAPS_MARKET_DESCRIPTOR_STRUCT *psDescriptor
        );

static BOOLEAN bSetVersion (
    MAPS_SERVICE_OBJECT hMapsService,
    UN16 un16Index,
    UN32 un32Version
        );

static MAPS_SERVICE_DETAIL_LEVEL_ENUM eResolutionToEnum (
    UN16 un16HorResolution
        );

static BOOLEAN bClearPersistentDirectory(
    MAPS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRemovePersistentFile(
    const char *pcDirItem,
    void *pvArg
        );

static BOOLEAN bCreateDatabaseConnectionsLocked (
    MAPS_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        );

static BOOLEAN bBuildDescriptorsArrayLocked (
    MAPS_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bCreatePreparedStatements (
    MAPS_APP_OBJECT_STRUCT *psAppObj
        );

static void vReleaseDatabaseConnectionsLocked (
    MAPS_APP_OBJECT_STRUCT *psAppObj
        );

static SMSAPI_RETURN_CODE_ENUM eCreateUpdateImageFile(
    MAPS_SERVICE_OBJECT hMapsService,
    FILE *psRFDFile,
    size_t tFileSize,
    const char *pacMarketName,
    UN16 un16HorResolution,
    UN32 un32Version
        );

/* GsMapsMgrIntf functions definition */
static BOOLEAN bAddUpdate (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubname,
    UN32 un32NewVersion,
    UN16 un16HorResolution,
    FILE *psRFDFile,
    size_t tFileSize
        );

static UN32 un32GetVersionForSubname (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubname
        );

static N8 n8ComparePriority (
    MAPS_SERVICE_OBJECT hMapsService,
    const char *pacSubnameA,
    const char *pacSubnameB
        );

  /***************/
 /** VARIABLES **/
/***************/

const MAPS_INTERFACE_STRUCT MAPS =
{
    /* hStart */ hStart,
    /* vStop */ (void (*) (MAPS_SERVICE_OBJECT hService))DATASERVICE_IMPL_vStop,
    /* eState */ (DATASERVICE_STATE_ENUM (*) (MAPS_SERVICE_OBJECT hService))DATASERVICE_IMPL_eState,
    /* eErrorCode */ (DATASERVICE_ERROR_CODE_ENUM (*) (MAPS_SERVICE_OBJECT hService))DATASERVICE_IMPL_eErrorCode,
    /*.ePersistentDataCleanup = */ePersistentDataCleanup,
    /*.eFindNearestMap = */eFindNearestMap,
    /*.eLookupMapByName = */eLookupMapByName,
    /*.ePrioritizeMarket = */ePrioritizeMarket
};

// Manager interface for plugins
const MAPS_MGR_INTERFACE_STRUCT GsMapsMgrIntf =
{
    /*.bAddUpdate = */ bAddUpdate,
    /*.un32GetVersionForSubname = */ un32GetVersionForSubname,
    /* n8ComparePriority = */ n8ComparePriority
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsMapsStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)bHandleServiceReady,
    /*.bHandleServiceStopped = */DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */DATASERVICE_IMPL_bGenericStateHandler
};

#endif /* _MAPS_MGR_OBJ_H_ */
