/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
* DESCRIPTION
*
* PRIVATE HEADER
*
******************************************************************************/

/*********************************/
/** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _LEAGUE_OBJ_H_
#define _LEAGUE_OBJ_H_

/**************/
/** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "team_obj.h"
#include "sql_interface_obj.h"

/***************/
/** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define LEAGUE_OBJECT_NAME "LEAGUE"

#define CREATE_VERSION_TABLE \
    "create table LeagueVersion ( " \
    "TableVersion int,  " \
    "TableComplete int  " \
    "); "

#define INSERT_VERSION_ROW \
    "INSERT INTO LeagueVersion VALUES(-1,0);"

#define SELECT_VERSION "select * from LeagueVersion;"

#define UPDATE_VERSION \
    "Update LeagueVersion set " \
    "TableVersion = %d, "\
    "TableComplete = %d " \
    ";"

// get all leagues
#define QUERY_FOR_ALL_LEAGUES "select * from leagues order by LEAGUEID"

// retrieve league version and rowid
#define QUERY_LEAGUE_VERSION "select Version,ROWID from leagues where (LEAGUEID = %u)"

// add a league to the db
#define INSERT_LEAGUE \
    "insert into Leagues (LEAGUEID, Version, Abbr, Name, Sport, SFEnabled, SFTiers) " \
    "VALUES(?, ?, ?, ?, ?, ?, ?);"

// update a league entry
#define UPDATE_LEAGUE \
    "Update Leagues set " \
    "LEAGUEID = ?, "\
    "Version = ?, " \
    "Abbr = ?, " \
    "Name = ?, " \
    "Sport = ? " \
    "SFEnabled = ? " \
    "SFTiers = ? " \
    "where ROWID = ?;"

// remove obsolete leagues from db
#define PRUNE_LEAGUES "delete from Leagues where (Version != %u)"

#define CREATE_LEAGUE_TABLE \
    "create table Leagues (  "\
    "LEAGUEID int, " \
    "Version tinyint, " \
    "Abbr text, " \
    "Name text, " \
    "Sport int, " \
    "SFEnabled int, " \
    "SFTiers int, " \
    "primary key (LEAGUEID) " \
    " ); "

#define LEAGUE_MAX_QUERY_LENGTH 500

/**************/
/** TYPEDEFS **/
/**************/

/* Private object elements */

typedef enum league_fields_enum
{
    LEAGUE_FIELD_ID = 0,
    LEAGUE_FIELD_VERSION,
    LEAGUE_FIELD_ABBREV,
    LEAGUE_FIELD_NAME,
    LEAGUE_FIELD_SPORT,
    LEAGUE_FIELD_SF_ENABLED,
    LEAGUE_FIELD_SF_TIERS,
    LEAGUE_MAX_FIELDS

}LEAGUE_FIELDS_ENUM;

// Basic league list object structure
typedef struct league_object_struct
{
    // Team list
    OSAL_OBJECT_HDL hTeamList;

    // CID for this league type
    CID_OBJECT hId;

    // League information
    LEAGUE_STRUCT *psLeague;

    // Save flag
    BOOLEAN bSaved;

} LEAGUE_OBJECT_STRUCT;

typedef struct league_ctrl_struct
{
    // A handle to a linked list of available leagues which maps CIDs to
    // more verbose and abstracted league content to populate
    // an associated CDO with. Note that this list is dynamic and shared
    // amongst all SMS Decoders. Thus access to it must be protected.
    // To enforce this protection this structure is declared as an
    // SMS Object and the built-in locking mechanism is used to
    // limit exclusive access to this element.
    OSAL_OBJECT_HDL hLeagueList;
    OSAL_OBJECT_HDL hLeagueTrashBin;

    // Sports DB elements
    char *pacFileName;
    SQL_INTERFACE_OBJECT hSQLConnection;
    char acQueryBuffer[LEAGUE_MAX_QUERY_LENGTH];

    // Current version info
    N32 n32Version;
    BOOLEAN bComplete;

    // CID methods
    CID_ENUM eType;
    CID_OBJECT hWorkingLeagueId;

    // Keeps reference to the object which shall be locked
    // prior LEAGUE object lock where it's needed.
    SMS_OBJECT hMasterLockableObject;

} LEAGUE_CTRL_STRUCT;

typedef struct league_version_match_iterator_struct
{
    N32 n32Version;
    UN32 un32Matching;

} LEAGUE_VERSION_MATCH_ITERATOR_STRUCT;

typedef struct league_iterator_struct
{
    LEAGUE_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback;
    void *pvContentIteratorCallbackArg;

} LEAGUE_ITERATOR_STRUCT;

typedef struct league_team_iterator_struct
{
    TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback;
    void *pvContentIteratorCallbackArg;

} LEAGUE_TEAM_ITERATOR_STRUCT;

typedef struct league_print_iterator_struct
{
    FILE *psFile;
    size_t tNumLeagues;

} LEAGUE_PRINT_ITERATOR_STRUCT;

typedef struct league_save_iterator_struct
{
    N32 n32Version;
    size_t tInsertCount;
    size_t tReplacedCount;
    size_t tAlreadyUpToDateCount;

} LEAGUE_SAVE_ITERATOR_STRUCT;

typedef struct league_row_struct
{
    UN32 un32LeagueId;
    N32 n32Version;
    const char *pacAbbrev;
    const char *pacName;
    UN32 un32SportId;
    UN32 un32SportsFlashEnabled;
    UN32 un32SportsFlashTiers;
    N64 n64RowId;

} LEAGUE_ROW_STRUCT;

typedef struct league_version_query_struct
{
    N32 n32Version;
    N64 n64RowId;
    BOOLEAN bFound;

} LEAGUE_VERSION_QUERY_STRUCT;

typedef struct league_team_entry_struct
{
    TEAM_OBJECT hTeam;
    CID_OBJECT hLeagueTeamId;

} LEAGUE_TEAM_ENTRY_STRUCT;

typedef struct league_search_struct
{
    CID_OBJECT hLeagueId;
    BOOLEAN bFound;

} LEAGUE_SEARCH_STRUCT;

/************/
/** MACROS **/
/************/

/****************/
/** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static STRING_OBJECT hName (
    LEAGUE_OBJECT hLeague
    );

static STRING_OBJECT hAbbreviation (
    LEAGUE_OBJECT hLeague
    );

static LEAGUE_ENUM eLeague (
    LEAGUE_OBJECT hLeague
    );

static BOOLEAN bIterateContent (
    LEAGUE_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
    void *pvContentIteratorCallbackArg
    );

static CID_OBJECT hId (
    LEAGUE_OBJECT hLeague
    );

static SPORTS_ENUM eSport (
    LEAGUE_OBJECT hLeague
    );

static N32 n32Version (
    LEAGUE_OBJECT hLeague
    );

static BOOLEAN bGetTableVersion (
    N32 *pn32Version,
    BOOLEAN *pbComplete
    );

static CID_OBJECT hTeamId (
    LEAGUE_OBJECT hLeague,
    TEAM_OBJECT hTeam
        );

static UN8 un8Id (
    LEAGUE_OBJECT hLeague
    );

/* Object Private Prototypes */

static void vDestroyLeagueObject (
    LEAGUE_OBJECT_STRUCT *psLeagueObject
    );

static BOOLEAN bIterator (
    LEAGUE_OBJECT_STRUCT const *psLeagueObject,
    LEAGUE_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bTeamIterator (
    LEAGUE_TEAM_ENTRY_STRUCT *psTeamEntry,
    LEAGUE_TEAM_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bCountMatchingVersion (
    LEAGUE_OBJECT_STRUCT const *psLeagueObject,
    LEAGUE_VERSION_MATCH_ITERATOR_STRUCT *psMatching
    );

static BOOLEAN bSaveLeague (
    LEAGUE_OBJECT_STRUCT *psLeagueObject,
    LEAGUE_SAVE_ITERATOR_STRUCT *psLeagueSaveIterator
    );

static LEAGUE_OBJECT_STRUCT *psCreateLeagueObject (
    SPORTS_ENUM eSport,
    LEAGUE_ENUM eLeague,
    CID_OBJECT hId,
    N32 n32Version,
    const char *pacName,
    const char *pacAbbreviation,
    BOOLEAN bSportsFlashEnabled,
    UN8 un8SportsFlashTiers
    );

static BOOLEAN bQueryLeagueVersion (
    UN32 un32LeagueId,
    N32 *pn32Version,
    N64 *pn64RowId
    );

static BOOLEAN bLeagueVersionHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bLeagueTableHandler (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bPrepareLeagueColumn (
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    LEAGUE_ROW_STRUCT *psLeagueRow
    );

static BOOLEAN bCreateDBTables (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    );

static BOOLEAN bVerifyDBVersion (
    SQL_INTERFACE_OBJECT hSQLConnection,
    void *pvArg
    );

static BOOLEAN bProcessSelectDBVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bProcessSelectTableVersionResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
    );

static BOOLEAN bPrune (
    N32 n32Version
    );

static N16 n16CompareIds (
    LEAGUE_OBJECT hLeague1,
    LEAGUE_OBJECT hLeague2
        );

static N16 n16CompareTeams(
    void *pvObj1, 
    void *pvObj2
        );

static void vReleaseTeamEntry(
    void *pvData
        );

static BOOLEAN bSearchForLeague(
    LEAGUE_OBJECT hLeague,
    void *pvArg
        );

static BOOLEAN bIsSportsFlashEnabled(
    LEAGUE_OBJECT hLeague
        );

static BOOLEAN bMasterLock( void );

static void vMasterUnlock( void );

/***************/
/** VARIABLES **/
/***************/

// League object control structure initialization
static LEAGUE_CTRL_STRUCT *gpsLeagueCtrl = NULL;

// Default league info
static const LEAGUE_STRUCT gsDefaultLeague =
{
    SPORTS_UNKNOWN,
    LEAGUE_UNKNOWN,
    STRING_INVALID_OBJECT,
    STRING_INVALID_OBJECT,
    FALSE,
    0,
    0
};

// Global (re-usable) instance of an interface for this object
const LEAGUE_OBJECT_INTERFACE_STRUCT LEAGUE =
{
    /*.hName = */hName,
    /*.hAbbreviation = */hAbbreviation,
    /*.eLeague = */eLeague,
    /*.bIterateContent = */bIterateContent,
    /*.hId = */hId,
    /*.hTeamId = */hTeamId,
    /*.eSport = */eSport,
    /*.n32Version = */n32Version,
    /*.bGetTableVersion = */bGetTableVersion,
    /*.un8Id =*/un8Id,
    /*.bIsSportsFlashEnabled = */bIsSportsFlashEnabled
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif	// _LEAGUE_OBJ_H_
