/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _FUEL_STATION_OBJ_H_
#define _FUEL_STATION_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "fuel_db_constants.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"
#include "fuel_station_obj.h"

/* Object name prefix for objects */
#define FUEL_STATION_OBJECT_NAME "FuelStation"

#define FUEL_STATION_BASE_AMENITY (FUEL_STATION_AMENITY_OPEN_24HR)
#define FUEL_STATION_MAX_AMENITY (FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL)
#define FUEL_STATION_AMENITY_BITMASK (0x3)
#define FUEL_AMENITY_AVAILABLE (1)
#define FUEL_AMENITY_SECONDARY_AVAILABLE (2)
#define FUEL_AMENITY_NOT_AVAILABLE (3)
#define FUEL_24HR_AMENITY_OFFSET (0)
#define FUEL_EMERGENCY_AMENITY_OFFSET (2)
#define FUEL_SERVICE_AMENITY_OFFSET (4)
#define FUEL_OILCHANGE_AMENITY_OFFSET (4)
#define FUEL_UNDERHALF_AMENITY_OFFSET (6)
#define FUEL_OVERHALF_AMENITY_OFFSET (6)
#define FUEL_DISCOUNT_AMENITY_OFFSET (8)
#define FUEL_STORE_AMENITY_OFFSET (10)
#define FUEL_SUPERMKT_AMENITY_OFFSET (10)
#define FUEL_SNACKS_AMENITY_OFFSET (12)
#define FUEL_RESTAURANT_AMENITY_OFFSET (12)
#define FUEL_TRUCKSTOP_AMENITY_OFFSET (14)
#define FUEL_HOTEL_AMENITY_OFFSET (14)

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */
typedef struct fuel_station_object_struct
{
    STRING_OBJECT hBrand;
    STRING_OBJECT hAdditionalInfo;
    LOCATION_OBJECT hLocation;

    // The text version for
    // all price entries
    N16 n16PriceTextVersion;

    // Amenities info
    UN32 un32Amenities;

    // Logo info
    N16 n16LogoId;

    STATE_ID tStateId;

    // Raw phone data
    N64 n64RawPhone;

    // A linked list of fuel price data
    OSAL_OBJECT_HDL hFuelPrices;

    // Readily-available attribute used to compare
    // stations without digging into LOCATION & LOCID objects
    LOC_ID tIDForCompare;

    // Station owner handle
    SMS_OBJECT hOwner;

} FUEL_STATION_OBJECT_STRUCT;

// Price description
typedef struct fuel_station_price_desc_struct
{
    // The type of fuel
    UN8 un8FuelType;

    // Mask specifies which fields
    // are in use below
    FUEL_INFO_MASK tInfoMask;
    
    // Enum indicates availability of fuel
    FUEL_AVAILABILITY_ENUM eAvailability;

    // The price of this fuel in the
    // lowest denomination
    UN32 un32FuelPrice;

    // The time of the report
    UN32 un32ReportTime;

    // The descriptive texts for this fuel type
    STRING_OBJECT hShortFuelName;
    STRING_OBJECT hLongFuelName;

    // An enumerated representation of hFuelString
    FUEL_TYPE_ENUM eFuelType;

    // The number of refueling positions available
    // at this station, if known
    UN8 un8NumPositions;

    // Dynamic position data
    FUEL_INFO_DYNAMIC_POSITION_STRUCT sDynamic;

    // Entry handle
    OSAL_LINKED_LIST_ENTRY hEntry;

} FUEL_STATION_PRICE_DESC_STRUCT;

// Structure used to aid in price flush operations
typedef struct fuel_station_price_flush_struct
{
    // The age at which prices must be flushed
    UN32 un32PriceExpireAge;

    // Flag indicating if any prices have been
    // flushed from this object
    BOOLEAN bPricesFlushed;

    // Tracks the oldest price data
    // found in the list after the flush
    UN32 *pun32OldestPriceAfterFlush;

} FUEL_STATION_PRICE_FLUSH_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static STRING_OBJECT hBrand (
    FUEL_STATION_OBJECT hFuelStation
        );

static IMAGE_OBJECT hBrandLogo (
    FUEL_STATION_OBJECT hFuelStation,
    FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType
        );

static STRING_OBJECT hAdditionalInfo (
    FUEL_STATION_OBJECT hFuelStation
        );

static LOCATION_OBJECT hLocation (
    FUEL_STATION_OBJECT hFuelStation
        );

static UN8 un8NumAvailableFuelTypes (
    FUEL_STATION_OBJECT hFuelStation
        );

static SMSAPI_RETURN_CODE_ENUM eFuelInfo (
    FUEL_STATION_OBJECT hFuelStation,
    UN8 un8FuelIndex,
    FUEL_INFO_STRUCT *psFuelInfo
        );

static SMSAPI_RETURN_CODE_ENUM eAmenities (
    FUEL_STATION_OBJECT hFuelStation,
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities
        );

static N32 n32FPrintf (
    FUEL_STATION_OBJECT hFuelStation,
    FILE *psFile
        );

/* Object Private Prototypes */

static FUEL_STATION_PRICE_DESC_STRUCT *psGetPriceDesc (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    UN8 un8FuelType,
    BOOLEAN *pbNewEntry,
    FUEL_INFO_MASK tInitialzerMask
        );

static BOOLEAN bUpdatePriceEntry (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
    FUEL_TYPE_TEXT_CALLBACK hFuelTextCallback,
    FUEL_TYPE_ENUM_CALLBACK eFuelTypeCallback,
    FUEL_PRICE_ENTRY_STRUCT *psPriceEntry,
    void *pvCallbackArg
        );

static BOOLEAN bAddRefuelingPositions (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    FUEL_STATION_ROW_STRUCT *psStationRow
        );

static BOOLEAN bUpdateDynamicPositions (
    FUEL_STATION_OBJECT_STRUCT *psObj,
    UN16 un16ReportTime,
    FUEL_POSITION_UPDATE_STRUCT *psPosition
        );

static void vReleasePriceData (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData
        );

static BOOLEAN bIteratePricesForFlush (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    FUEL_STATION_PRICE_FLUSH_STRUCT *psFlush
        );

static BOOLEAN bIterateToRemoveDynamicPos (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    BOOLEAN *pbStationUpdated
        );

static N16 n16ComparePriceEntriesByPrice (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData1,
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData2
        );

static N16 n16ComparePriceEntriesByType (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData1,
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData2
        );

static N16 n16ComparePriceEntryToType (
    FUEL_STATION_PRICE_DESC_STRUCT *psPriceData,
    UN8 *pun8FuelType
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const FUEL_STATION_OBJECT_INTERFACE_STRUCT FUEL_STATION =
{
    /*.hBrand = */hBrand,
    /*.hBrandLogo = */hBrandLogo,
    /*.hAdditionalInfo = */hAdditionalInfo,
    /*.hLocation = */hLocation,
    /*.un8NumAvailableFuelTypes = */un8NumAvailableFuelTypes,
    /*.eFuelInfo = */eFuelInfo,
    /*.eAmenities = */eAmenities,
    /*.n32FPrintf = */n32FPrintf
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _FUEL_STATION_OBJ_H_
