/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _DECODER_OBJ_H_
#define _DECODER_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "srh.h"
#include "sms_task.h"
#include "browse_obj.h"
#include "cme.h"
#include "radio.h"
#include "tag_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define DECODER_OBJECT_NAME "DECODER"

// The number of times we'll allow the Decoder to reset
#define DECODER_RESET_LIMIT (3)

#define LOCKED_TAG_NAME "Locked"
#define SKIPPED_TAG_NAME "Skipped"
#define SERVICE_ID_TAG_NAME "ServiceId"
#define LAST_TUNED_SERVICE_ID_TAG_NAME "LastTunedServId"
#define UNSUBSCRIBED_ARTIST_TEXT_TAG_NAME "UnsubscribedArtistText"
#define UNSUBSCRIBED_TITLE_TEXT_TAG_NAME "UnsubscribedTitleText"
#define UNSUBSCRIBED_COMPOSER_TEXT_TAG_NAME "UnsubscribedComposerText"
#define UNSUBSCRIBED_ALBUM_TEXT_TAG_NAME "UnsubscribedAlbumNameText"
#define UNSUBSCRIBED_CONTENTINFO_TEXT_TAG_NAME "UnsubscribedContentInfoText"

#define DECODER_CHANNEL_METADATA_COMMIT_TIMEOUT (60 * 1000) // 1 minute

// First TuneMix service ID (SXI 3.2)
#define TUNEMIX_CID_OFFSET (0x1001)

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct decoder_callback_shim_struct
{
    // Original callback
    DECODER_OBJECT_EVENT_CALLBACK vEventCallback;

    // Original event callback argument
    void *pvEventCallbackArg;

} DECODER_CALLBACK_SHIM_STRUCT;

typedef struct decoder_tune_scan_struct
{
    UN8 un8PlaySeconds;
    BOOLEAN bScanLockedMature;
    BOOLEAN bScanSkipped;
    DECODER_TUNE_SCAN_STYLE_ENUM eScanStyle;
    DECODER_TUNE_SCAN_STATUS_MASK tStatusMask;
    BOOLEAN bConfigured;
    BOOLEAN bActive;

} DECODER_TUNE_SCAN_STRUCT;

typedef struct decoder_object_struct
{
    /* THINGS PROVIDED BY APPLICATION */

    // DECODER's name
    const char *pacDecoderName;

    // Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tCapabilities;

    // STI Connection Handle
    STI_HDL hSTI;

    /* THINGS TO INITIALIZE ONCE */

    // Module this object is attached to
    MODULE_OBJECT hModule;

    // DECODER object name
    const char *pacObjectName;

    // Decoder Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // Services task for this decoder
    SMS_TASK_HANDLE hServicesTask;

    /*
     * Sub-Object Handles
     */

    // Browse object
    BROWSE_OBJECT hBrowse;

    // Channel Cache Handle
    CCACHE_OBJECT hCCache;

    // Content Monitoring Engine Object (CME)
    CME_OBJECT hCME;

    // Playback Resource
    PLAYBACK_OBJECT hPlayback;

    // Smart favorites service
    SMART_FAVORITES_OBJECT hSmartFavorites;

    // Presets service object
    PRESETS_OBJECT hPresets;

    // Service Handles
    SEEK_SERVICE_OBJECT hArtistTitleSeek;
    SEEK_SERVICE_OBJECT hTrafficWeatherSeek;
    SEEK_SERVICE_OBJECT hSportsSeek;

    /* THINGS TO RE-INITIALIZE ON DEMAND or RESET */

    // Radio Initialized
    BOOLEAN bRadioInitialized;

    // Decoder browsed channel / category handles
    CHANNEL_OBJECT hBrowsedChannel;
    CATEGORY_OBJECT hBrowsedCategory;

    // Decoder tuned channel / category handles
    CHANNEL_OBJECT hTunedChannel;
    CATEGORY_OBJECT hTunedCategory;

    // Decoder last tuned channel / category handles
    CHANNEL_OBJECT hLastTunedChannel;
    CATEGORY_OBJECT hLastTunedCategory;

    // channel map update progress (percent)
    UN8 un8TotalUpdateProgress;
    UN8 un8ReceiverUpdateProgress;

    // Antenna
    N8 n8NumAntennas;
    ANTENNA_STATE_ENUM *apeAntennaState;

    // Signal Quality
    SIGNAL_QUALITY_STRUCT sSignalQuality;

    // Object's operational mode mask
    SMS_MODE_MASK tCurrentModes;

    // Decoder State
    DECODER_STATE_ENUM eState;

    // Tune State
    TUNE_STATE_ENUM eTuneState;

    // Object Error Codes
    DECODER_ERROR_CODE_ENUM eErrorCode;

    // SMS Callback shim
    DECODER_CALLBACK_SHIM_STRUCT sCallbackShim;

    // config manager tag
    TAG_OBJECT hTag;

    // sport zone service
    SPORT_ZONE_SERVICE_OBJECT hSportZoneService;

    SUB_NOTIFICATION_OBJECT hSubscriptionService;

    // Object which contains radio specific data
    RADIO_PRIVATE_DATA_OBJECT hRadioData;

    // just browsed flag
    BOOLEAN bBrowsed;
    // browsed channel channel event
    CHANNEL_EVENT_MASK tBrowsedEventMask;
    // In decoder context flag
    BOOLEAN bInDecoderCallback;

    // Flag indicates if this decoder is subscribed to data
    // services such as channel art
    BOOLEAN bSubscribedToDS;

    // flag to indicate that vUninitObject will need to be called when handling
    // an event in the RELEASED state
    BOOLEAN bUninitialized;

    ENGINEERING_DATA_OBJECT hEngineeringData;
    DECODER_EVENT_MASK tRequestMask;

    // Tune Scan Configuration Struct
    DECODER_TUNE_SCAN_STRUCT sTuneScan;

    // Time Update Notification Handle
    OSAL_TIME_NOTIFICATION_OBJECT hTimeNotificationHandle;

    // Minute-ticker
    OSAL_OBJECT_HDL hMinuteTicker;

    // Alternate Channel Order commit timer
    OSAL_OBJECT_HDL hACOTimer;

    // Pending unconfirmed attributes currently in progress of setting
    CHANNEL_OBJECT_ATTRIBUTE_MASK tPendingAttrs;

    // Object reference counter
    UN16 un16RefCounter;

    // TuneMix handles array
    TUNEMIX_OBJECT ahTuneMix[TUNEMIX_OBJECTS_MAX];

    // maximum number of TuneMix services specified
    // by module
    UN8 un8MaxTuneMix;

    // currently tuned Tune Mix object hdl
    TUNEMIX_OBJECT hTuneMixActive;

    // Sports Flash service
    SPORTS_FLASH_OBJECT hSportsFlash;

    // TW Now object handle
    TW_NOW_OBJECT hTWNow;

    // Audio Presence status
    DECODER_AUDIO_PRESENCE_ENUM eAudioPresence;

} DECODER_OBJECT_STRUCT;

typedef enum decoder_update_handle_enum
{
    DECODER_UPDATE_HANDLE_LAST_TUNED,
    DECODER_UPDATE_HANDLE_BROWSED,
    DECODER_UPDATE_HANDLE_TUNED

} DECODER_UPDATE_HANDLE_ENUM;

typedef struct decoder_channel_attrs_struct
{
    SERVICE_ID *ptLocked;
    SERVICE_ID *ptSkipped;
    UN16 un16NumLocked;
    UN16 un16NumSkipped;

} DECODER_CHANNEL_ATTRS_STRUCT;

typedef struct channel_attribute_tag_iterator_struct
{
    SERVICE_ID tServiceId;
    DECODER_OBJECT_STRUCT *psDecoderObj;
    BOOLEAN bRemove;
    BOOLEAN bLock;
    BOOLEAN bSkip;
    DECODER_CHANNEL_ATTRS_STRUCT sAttrs;

} CHANNEL_ATTRIBUTE_TAG_ITERATOR_STRUCT;

typedef struct tunemix_component_iterator_struct
{
    UN8 un8Count;
    SERVICE_ID atServiceId[TUNEMIX_CHANNELS_MAX];

} TUNEMIX_COMPONENT_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static DECODER_OBJECT hGet (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    DECODER_EVENT_MASK tEventRequestMask,
    DECODER_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vRelease (
    DECODER_OBJECT hDecoder
        );

static DECODER_STATE_ENUM eState (
    DECODER_OBJECT hDecoder
        );

static DECODER_ERROR_CODE_ENUM eErrorCode (
    DECODER_OBJECT hDecoder
        );

static DECODER_EVENT_MASK tEventMask (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        );

static BOOLEAN bUpdate (
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tMask
        );

static SMSAPI_RETURN_CODE_ENUM eTuneDirect (
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId,
    BOOLEAN bOverride
        );

static TUNE_STATE_ENUM eTuneState (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eSignalQuality (
    DECODER_OBJECT hDecoder,
    SIGNAL_QUALITY_STRUCT *psSignalQuality
        );

static N8 n8NumAntennas (
    DECODER_OBJECT hDecoder
        );

static ANTENNA_STATE_ENUM eAntennaState (
    DECODER_OBJECT hDecoder,
    N8 n8Antenna
        );

static SMSAPI_RETURN_CODE_ENUM eUpdateProgress  (
    DECODER_OBJECT hDecoder,
    UN8 *pun8UpdateProgress
        );

static CHANNEL_ID tGetChannelId (
    DECODER_OBJECT hDecoder,
    SERVICE_ID tServiceId
        );

static CHANNEL_ID tCurrentChannelId (
    DECODER_OBJECT hDecoder
        );

static SERVICE_ID tCurrentServiceId (
    DECODER_OBJECT hDecoder
        );

static CATEGORY_ID tCurrentCategoryId (
    DECODER_OBJECT hDecoder
        );

static CHANNEL_ID tLastTunedChannelId (
    DECODER_OBJECT hDecoder
        );

static CATEGORY_ID tLastTunedCategoryId (
    DECODER_OBJECT hDecoder
        );

static CHANNEL_ID tBrowsedChannelId (
    DECODER_OBJECT hDecoder
        );

static SERVICE_ID tBrowsedServiceId(
    DECODER_OBJECT hDecoder
        );

static CATEGORY_ID tBrowsedCategoryId (
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseChannelStyleSet (
    DECODER_OBJECT hDecoder,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseChannel (
    DECODER_OBJECT hDecoder,
    SMSAPI_DIRECTION_ENUM eDirection
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseToChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseCategoryStyleSet(
    DECODER_OBJECT hDecoder,
    BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
    void *pvBrowseCategoryCompareArg
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseCategory (
    DECODER_OBJECT hDecoder,
    SMSAPI_DIRECTION_ENUM eDirection
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseToCategory (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

static SMSAPI_RETURN_CODE_ENUM eUseChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    DECODER_CHANNEL_ACCESS_CALLBACK vChannelAccessCallback,
    void *pvChannelAccessCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eUseCategory (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    DECODER_CATEGORY_ACCESS_CALLBACK vCategoryAccessCallback,
    void *pvCategoryAccessCallbackArg
        );

static BOOLEAN bLockChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        );

static BOOLEAN bUnlockChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        );

static BOOLEAN bSkipChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        );

static BOOLEAN bUnskipChannel (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eIsChannelLocked (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    BOOLEAN *pbLocked
        );

static SMSAPI_RETURN_CODE_ENUM eIsChannelSkipped (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    BOOLEAN *pbSkipped
        );

static SMSAPI_RETURN_CODE_ENUM eIsChannelSubscribed (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    BOOLEAN *pbSubscribed
        );

static SMSAPI_RETURN_CODE_ENUM eIsChannelMature (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    BOOLEAN *pbMature
        );

static SMSAPI_RETURN_CODE_ENUM eIsChannelFreeToAir (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelId,
    BOOLEAN *pbFreeToAir
        );

static PLAYBACK_OBJECT hPlayback(
    DECODER_OBJECT hDecoder
        );

static SMSAPI_RETURN_CODE_ENUM eSetUnsubscribedText(
    DECODER_OBJECT hDecoder,
    const char *pacUnsubscribedText,
    CDO_FIELD_MASK tMask
        );

static SMSAPI_RETURN_CODE_ENUM eStartToneGeneration (
        DECODER_OBJECT hDecoder,
        UN32 un32ToneFreqHz,
        N8 n8Volume,
        DECODER_TONE_GENERATION_BALANCE_ENUM eBalance
            );

static SMSAPI_RETURN_CODE_ENUM eStopToneGeneration (
        DECODER_OBJECT hDecoder
            );

static SMSAPI_RETURN_CODE_ENUM eAlertToneGeneration(
             DECODER_OBJECT hDecoder,
              N8 n8Volume,
              int iCmd,
              ...);

static SMSAPI_RETURN_CODE_ENUM eAudio(
    DECODER_OBJECT hDecoder,
    N16 n16Level
        );

static SMSAPI_RETURN_CODE_ENUM eTuneScanConfigure (
    DECODER_OBJECT hDecoder,
    UN8 un8PlaySeconds,
    BOOLEAN bScanLockedMature,
    BOOLEAN bScanSkipped,
    DECODER_TUNE_SCAN_STYLE_ENUM eScanStyle
        );

static SMSAPI_RETURN_CODE_ENUM eTuneScan (
    DECODER_OBJECT hDecoder,
    DECODER_TUNE_SCAN_CMD_ENUM eScanCmd
        );

static SMSAPI_RETURN_CODE_ENUM eIsTuneScanContentAvailable (
    DECODER_OBJECT hDecoder,
    DECODER_TUNE_SCAN_STATUS_MASK *ptStatusMask
        );

static DECODER_AUDIO_PRESENCE_ENUM eAudioPresence (
    DECODER_OBJECT hDecoder
        );

/* Object Private Prototypes */

static DECODER_OBJECT_STRUCT *psCreateObject (
    SRM_OBJECT hSRM,
    const char *pacDecoderName,
    MODULE_ID tId,
    MODULE_OBJECT hModule,
    DECODER_EVENT_MASK tEventRequestMask,
    DECODER_OBJECT_EVENT_CALLBACK vEventCallback, 
    void *pvEventCallbackArg
        );

static  void vDestroyObject (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitializeDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeGenericDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitializeGenericDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeAudioDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitializeAudioDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeVideoDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitializeVideoDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeDataDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitializeDataDecoder (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeObject (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vUpdateState (
    DECODER_OBJECT_STRUCT *psObj,
    DECODER_STATE_ENUM eNewState
        );

static DECODER_STATE_ENUM eTransitionToErrorState(
    DECODER_OBJECT_STRUCT *psObj,
    DECODER_ERROR_CODE_ENUM eErrorCode
        );

static DECODER_STATE_ENUM eTransitionToInitialState(
    DECODER_OBJECT_STRUCT *psObj
        );

static DECODER_STATE_ENUM eTransitionToReadyState(
    DECODER_OBJECT_STRUCT *psObj
        );

static DECODER_STATE_ENUM eTransitionToReleasedState(
    DECODER_OBJECT_STRUCT *psObj
        );

static DECODER_STATE_ENUM eTransitionToUpdatingState(
    DECODER_OBJECT_STRUCT *psObj
        );

static void vTransitionFromUpdatingState(
    DECODER_OBJECT_STRUCT *psObj
        );

static void vEventHandler (
    DECODER_OBJECT hDecoder,
    SMS_EVENT_HDL hEvent
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseLocal (
    DECODER_OBJECT hDecoder,
    SMS_EVENT_BROWSE_STRUCT const *psBrowse
        );

static void vBrowseEventHandler (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_BROWSE_STRUCT const *psBrowse
        );

static void vUpdateTuneState (
    DECODER_OBJECT_STRUCT *psObj,
    TUNE_STATE_ENUM eNewState
        );

static SMSAPI_RETURN_CODE_ENUM eQualifyTuneRequest(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT *psTune
        );

static BOOLEAN bPostTune(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT const *psTune
        );

static BOOLEAN bTuneServiceId(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_STRUCT const *psTune
        );

static BOOLEAN bConfigureAttrList(
    DECODER_OBJECT hDecoder,
    SMS_EVENT_TUNEMIX_STRUCT const *psTune
        );

static void vEventCallbackShim(
    DECODER_OBJECT hDecoder,
    SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vCCacheRemapCallback (
    CCACHE_OBJECT hCCache,
    BOOLEAN bComplete,
    void *pvCallbackArg
        );

static void vCCacheSubscriptionUpdateCallback (
    CCACHE_OBJECT hCCache,
    BOOLEAN bComplete,
    void *pvCallbackArg
        );

static void vUpdateHandles (
    DECODER_OBJECT_STRUCT *psObj,
    DECODER_UPDATE_HANDLE_ENUM eHandlesToUpdate,
    CHANNEL_OBJECT hChannel,
    CATEGORY_OBJECT hCategory
        );

static void vChannelEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vLastTunedChannelEventCallback(
    CHANNEL_OBJECT hChannel, CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg);

static void vHandleAllChanCatNotificationEvent(
    DECODER_OBJECT hDecoder,
    BOOLEAN bNotifyChannels
        );

static void vLoadLastTunedServiceId(
    DECODER_OBJECT_STRUCT *psObj
        );

static SERVICE_ID tGetDefaultOrSafeServiceId(
    DECODER_OBJECT_STRUCT *psObj
        );

static void vSaveTunedServiceId(
    DECODER_OBJECT_STRUCT *psObj
        );

static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vSetError(
    DECODER_OBJECT_STRUCT *psObj,
    DECODER_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bChannelAttributeTagIterator(
    TAG_OBJECT hTag,
    void *pvArg
        );

static BOOLEAN bRetrieveLockedChannels(
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRetrieveSkippedChannels(
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRetrieveTuneMixChannels(
    DECODER_OBJECT_STRUCT *psObj
        );


static BOOLEAN bAddServiceIdTag(
    TAG_OBJECT hParentTag,
    CHANNEL_OBJECT hChannel
        );

static BOOLEAN bRetrieveUnsubscribedText(
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRetrieveUnsubscribedTextTag(
    DECODER_OBJECT_STRUCT *psObj,
    const char *pacTagName,
    CDO_FIELD_MASK tMask
        );

static BOOLEAN bStoreUnsubscribedText(
    DECODER_OBJECT_STRUCT *psObj,
    const char *pacUnsubscribedText,
    CDO_FIELD_MASK tMask
        );

static BOOLEAN bStoreUnsubscribedTextTag(
    DECODER_OBJECT_STRUCT *psObj,
    const char *pacTagName,
    STRING_OBJECT hNameString
        );

static BOOLEAN bCCacheUnsubscribedChansIterator(
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

static void vTuneScanEventHandler (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_TUNE_SCAN_STRUCT const *psEventData
        );

static SMSAPI_RETURN_CODE_ENUM eTuneSelf(
    DECODER_OBJECT_STRUCT *psObj,
    BOOLEAN bMatureOverride,
    BOOLEAN bLockedOverride,
    BOOLEAN bSkippedOverride
        );

static void vTimeAvailableNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        );

static void vMinuteTickerCallback (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static BOOLEAN bStartMinuteTicker (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vStopMinuteTicker (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bUpdateCategoryFallback(
    DECODER_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    DECODER_UPDATE_HANDLE_ENUM eUpdate
        );

static void vFireACOEvent(
    OSAL_OBJECT_HDL hTimer, 
    void *pvArg
        );

#if SMS_DEBUG==1

static void vPrintTuneScan (
    DECODER_OBJECT_STRUCT *psObj
        );

#endif // SMS_DEBUG==1

// Channel Art Specific iterators

static BOOLEAN bProcessDecoderEventMaskChange(
    DECODER_OBJECT_STRUCT *psObj,
    SMS_EVENT_DECODER_EVENT_MASK_STRUCT const *psDecoderEventStruct
        );

static void vSetChannelAttrs (
    DECODER_OBJECT_STRUCT *psObj,
    BOOLEAN bLocked,
    BOOLEAN bSkipped
        );

static BOOLEAN bSetChannelAttrsIter (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

static BOOLEAN bCountChannelsIterator (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

static BOOLEAN bReleaseParentModule (
    DECODER_OBJECT_STRUCT *psObj
        );

/* Shutdown management */

static BOOLEAN bObjectBusy (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bTryStop (
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitiateObjectRelease (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static BOOLEAN bPostFinalStop (
    DECODER_OBJECT_STRUCT *psObj
        );

/* Event handlers */

static void vHandleInitEvent (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vHandleRadioReadyEvent (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vHandleAssociateEvent (
    DECODER_OBJECT_STRUCT *psObj,
    const SMS_EVENT_DECODER_ASSOCIATE_STRUCT *psAssociate
        );

static void vHandleUnassociateEvent (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vHandleReleaseEvent (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleAppInitiatedRelease (
    DECODER_OBJECT_STRUCT *psObj,
    SMS_OBJECT_RELEASE_INITIATOR_ENUM eInitiator
        );

static void vHandleGeneralRelease (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vHandleStopEvent (
    DECODER_OBJECT_STRUCT *psObj
        );

static void vHandleResetEvent(
    DECODER_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRemoveTuneMix (
    TUNEMIX_OBJECT hTuneMix,
    void *pvArg
        );

static BOOLEAN bCountTuneMixComponents (
    CHANNEL_OBJECT hChannel,
    void *pvArg
        );

  /***************/
 /** VARIABLES **/
/***************/

// DECODER Object Defaults
static const DECODER_OBJECT_STRUCT gsObjectDefaults =
{
    // DECODER's name
    NULL,

    // Capabilities
    SRH_DEVICE_CAPABILITY_NONE,

    // STI Connection Handle
    STI_INVALID_HDL,

    // Module this object is attached to
    MODULE_INVALID_OBJECT,

    // DECODER object name
    NULL,

    // Decoder Update Control Structure
    {
        // Object update masks
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,
        SMS_OBJECT_EVENT_NONE,

        // Object update callback list
        OSAL_INVALID_OBJECT_HDL,
        OSAL_INVALID_LINKED_LIST_ENTRY,

        // Object being updated
        NULL,

        // In-active callbacks
        FALSE
    },

    // Event Handler
    SMS_INVALID_EVENT_HANDLER,

    // Services task for this decoder
    SMS_INVALID_TASK_HANDLE,

    // Browse object
    BROWSE_INVALID_OBJECT,

    // Channel Cache Handle
    CCACHE_INVALID_OBJECT,

    // Content Monitoring Engine Object (CME)
    CME_INVALID_OBJECT,

    // Playback Resource
    PLAYBACK_INVALID_OBJECT,

    // Smart Favorites service
    SMART_FAVORITES_INVALID_OBJECT,

    // Presets service object
    PRESETS_INVALID_OBJECT,

    // Service Handles
    SEEK_SERVICE_INVALID_OBJECT,
    SEEK_SERVICE_INVALID_OBJECT,
    SEEK_SERVICE_INVALID_OBJECT,

    // Radio Initialized
    FALSE,

    // Decoder browsed channel / category handles
    CHANNEL_INVALID_OBJECT,
    CATEGORY_INVALID_OBJECT,

    // Decoder tuned channel / category handles
    CHANNEL_INVALID_OBJECT,
    CATEGORY_INVALID_OBJECT,

    // Decoder last tuned channel / category handles
    CHANNEL_INVALID_OBJECT,
    CATEGORY_INVALID_OBJECT,

    // channel map update progress (percent)
    100,
    100,

    // Antenna
    0,
    NULL,

    // Signal Quality
    {
        SIGNAL_STATE_UNKNOWN,
        SIGNAL_QUALITY_INVALID,
        SIGNAL_QUALITY_INVALID,
        SIGNAL_QUALITY_INVALID,
        { 0, 0 }
    },

    // Object's operational mode mask
    SMS_MODE_NONE,

    // Decoder State
    DECODER_STATE_INITIAL,

    // Tune State
    TUNE_STATE_UNKNOWN,

    // Object Error Codes
    DECODER_ERROR_CODE_NONE,

    // Callback shim
    {
        (DECODER_OBJECT_EVENT_CALLBACK)NULL,
        NULL
    },

    // Tag object
    TAG_INVALID_OBJECT,

    // sport zone service
    SPORT_ZONE_SERVICE_INVALID_OBJECT,
    // subscription service
    SUB_NOTIFICATION_INVALID_OBJECT,

    // Pointer to contain radio specific data
    RADIO_PRIVATE_DATA_INVALID_OBJECT,

    // just browsed flag
    FALSE,
    // browsed channel channel event
    CHANNEL_OBJECT_EVENT_NONE,
    // In decoder context flag
    FALSE,

    // bSubscribedToDS
    FALSE,

    // bUninitialized
    FALSE,

    //hEngineeringData;
    ENGINEERING_DATA_INVALID_OBJECT,

    // bRequestMask
    DECODER_OBJECT_EVENT_NONE,

    // Tune Scan Configuration Struct
    {
        6,
        FALSE,
        FALSE,
        DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES,
        DECODER_TUNE_SCAN_STATUS_NONE,
        FALSE,
        FALSE
    },
    OSAL_TIME_NOTIFICATION_INVALID_OBJECT,
    OSAL_INVALID_OBJECT_HDL,
    OSAL_INVALID_OBJECT_HDL,

    // Pending unconfirmed attributes currently in progress of setting
    CHANNEL_OBJECT_ATTRIBUTE_NONE,

    // un16RefCounter
    0,

    // TuneMix handles array
    {TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT,
     TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT,
     TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT, TUNEMIX_INVALID_OBJECT,
     TUNEMIX_INVALID_OBJECT},

    // maximum number of TuneMix services specified
    // by module
    0,

    // currently tuned Tune Mix object
    TUNEMIX_INVALID_OBJECT,

    // Sports Flash object handle
    SPORTS_FLASH_INVALID_OBJECT,

    // TW Now object handle
    TW_NOW_INVALID_OBJECT,

    // eAudioPresence
    DECODER_AUDIO_PRESENCE_UNKNOWN
};

// DECODER Object configuration for SMS
static const SMS_TASK_CONFIGURATION_STRUCT gsDecoderTaskConfiguration =
{
    /*.pacName = */"DECODER",
    /*.un32StackSize = */16384, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_MEDIUM,
    /*.un32Options = */OSAL_TASK_OPTION_DEBUG_OUTPUT,
    /*.un16ReportingInterval = */30, // seconds
    /*.un32EventQueueSize = */288, // Bosch ID#8: Increase the queue size as discussed with SXM
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_NONE
};

// Global (re-usable) instance of an interface for this object
const DECODER_OBJECT_INTERFACE_STRUCT DECODER =
{
    // Public
    /*.hGet = */hGet,
    /*.vRelease = */vRelease,
    /*.eState = */eState,
    /*.eErrorCode = */eErrorCode,
    /*.tEventMask = */tEventMask,
    /*.eModifyRegisteredEventMask = */eModifyRegisteredEventMask,
    /*.bUpdate = */bUpdate,
    /*.eTuneDirect = */eTuneDirect,
    /*.eTuneState = */eTuneState,
    /*.eSignalQuality = */eSignalQuality,
    /*.n8NumAntennas = */n8NumAntennas,
    /*.eAntennaState = */eAntennaState,
    /*.eUpdateProgress = */eUpdateProgress,
    /*.tGetChannelId = */tGetChannelId,
    /*.tCurrentChannelId = */tCurrentChannelId,
    /*.tCurrentServiceId = */tCurrentServiceId,
    /*.tCurrentCategoryId = */tCurrentCategoryId,
    /*.tLastTunedChannelId = */tLastTunedChannelId,
    /*.tLastTunedCategoryId = */tLastTunedCategoryId,
    /*.tBrowsedChannelId = */tBrowsedChannelId,
    /*.tBrowsedCategoryId = */tBrowsedCategoryId,
    /*.eBrowseChannelStyleSet = */eBrowseChannelStyleSet,
    /*.eBrowseChannel = */eBrowseChannel,
    /*.eBrowseToChannel = */eBrowseToChannel,
    /*.eBrowseCategoryStyleSet = */eBrowseCategoryStyleSet,
    /*.eBrowseCategory = */eBrowseCategory,
    /*.eBrowseToCategory = */eBrowseToCategory,
    /*.eUseChannel = */eUseChannel,
    /*.eUseCategory = */eUseCategory,
    /*.bLockChannel = */bLockChannel,
    /*.bUnlockChannel = */bUnlockChannel,
    /*.bSkipChannel = */bSkipChannel,
    /*.bUnskipChannel = */bUnskipChannel,
    /*.eIsChannelLocked = */eIsChannelLocked,
    /*.eIsChannelSkipped = */eIsChannelSkipped,
    /*.eIsChannelSubscribed = */eIsChannelSubscribed,
    /*.eIsChannelMature = */eIsChannelMature,
    /*.eIsChannelFreeToAir = */eIsChannelFreeToAir,
    /*.hPlayback = */hPlayback,
    /*.eSetUnsubscribedText = */eSetUnsubscribedText,
    /*.eStartToneGeneration = */eStartToneGeneration,
    /*.eStopToneGeneration = */eStopToneGeneration,
    /*.eAlertToneGeneration = */eAlertToneGeneration,
    /*.eAudio = */eAudio,
    /*.eTuneScanConfigure = */eTuneScanConfigure,
    /*.eTuneScan = */eTuneScan,
    /*.eIsTunescanContentAvailable = */eIsTuneScanContentAvailable,
    /*.eAudioPresence = */eAudioPresence
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _DECODER_OBJ_H_
