/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _DATASERVICE_MGR_OBJ_H_
#define _DATASERVICE_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "radio.h"

#include "dataservice_mgr_obj.h"
#include "dataservice_mgr_impl.h"
#include "dsrl_obj.h"
#include "device_obj.h"
#include "tag_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define DATASERVICE_MGR_OBJECT_NAME "DATA"

// Tag names
#define DSM_TAG_NAME "DataServices"
#define FAVORITES_TAG_NAME "Favorites"

// Payload queue size
#define DSM_PAYLOAD_QUEUE_SIZE (48)

// Timeouts are provided to OSAL in MS.  Any number greater
// than what is below results in an overflow when multiplied by 1000
#define DSM_MAX_TIMEOUT_DURATION_IN_SECONDS (4294967)

#define DSM_MSECS_PER_SEC (1000)

#define DEFAULT_DATA_SERVICE_OBJECT_NAME "GenericData"

// This is used when the data service connection
// doesn't specify an OTA memory size.  That happens
// when the generic data service is used.  The generic
// service may be used for any of our services, which
// means we need to have enough memory to handle
// our most memory-intensive service, which is AGW
// That service needs about 200k, so that's what
// we will set this to.
#define DSM_DEFAULT_OTA_MEMORY_SIZE (1024 * 200)

  /************/
 /** MACROS **/
/************/

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct dataservice_mgr_fav_attrs_struct
{
    // Config file tag
    TAG_OBJECT hFavoritesTag;

    // Create target callback
    DATASERVICE_IMPL_CREATE_TARGET_FROM_TAG hCreateTargetFromTag;

    // Tag locator callback
    DATASERVICE_IMPL_GET_TAG_FOR_TARGET hGetTagForTarget;

} DATASERVICE_MGR_FAV_ATTRS_STRUCT;

typedef struct dataservice_mgr_managed_dsrl_struct
{
    DSRL_OBJECT hDeviceDSRL;
    DSRL_OBJECT hFavorites;

} DATASERVICE_MGR_MANAGED_DSRL_STRUCT;

typedef struct dataservice_mgr_dsrl_support_struct
{
    // Service type for these DSRLs
    DATASERVICE_TYPE_ENUM eServiceType;

    // Favorites service attributes
    DATASERVICE_MGR_FAV_ATTRS_STRUCT sFavorites;

    // Device location attributes
    BOOLEAN bDevDSRLStoppedByManager;
    UN32 un32DeviceNotifyDistance;
    DISTANCE_UNIT_TYPE_ENUM eDeviceNotifyUnits;
    DEVICE_REGISTRATION_OBJECT hDeviceReg;

    // DSRL descriptor size
    size_t tDSRLCreateSize;

    DATASERVICE_MGR_MANAGED_DSRL_STRUCT sManagedDSRLs;

} DATASERVICE_DSRL_SUPPORT_STRUCT;

typedef struct dataservice_mgr_info_struct
{
    // General attributes
    DATASERVICE_EVENT_MASK tEventRequestMask;
    BOOLEAN bFavoritesSupported;
    BOOLEAN bDeviceSupported;

} DATASERVICE_MGR_INFO_STRUCT;

typedef struct dataservice_mgr_product_info_struct
{
    DATA_PRODUCT_TYPE_ENUM eType;
    DATA_PRODUCT_MASK tAllowedMask;
    DATA_PRODUCT_STATE_ENUM eState;
    DATA_PRODUCT_MASK tMask;
    DATASERVICE_DSI_INFO_STRUCT sDSIInfo;

} DATASERVICE_MGR_PRODUCT_INFO_STRUCT;

typedef struct dataservice_mgr_products_struct
{
    DATASERVICE_GET_DSI_FOR_PRODUCT bGetDSIForProduct;
    DATASERVICE_GET_NEXT_PRODUCT_STATE eGetNextProductState;
    OSAL_OBJECT_HDL hProducts;

} DATASERVICE_MGR_PRODUCTS_STRUCT;

typedef struct dataservice_mgr_dsi_ref_struct
{
    // The DSI
    DSI tDSI;

    // The Manager's suggested buffer size
    // for OTA messages and data type
    size_t tSuggestedOTABufferByteSize;

    // A flag that determines if a service should have
    // all sub-streams (DMIs) enabled at initialization
    // (not used for multi-dsi services)
    BOOLEAN bEnableAllDMIs;

    // DSI state
    DATASERVICE_STATE_ENUM eState;

    // Reference counter
    size_t tRefCount;

    // The entry of this item
    OSAL_LINKED_LIST_ENTRY hEntry;

} DATASERVICE_MGR_DSI_REF_STRUCT;

struct dataservice_mgr_ctrl_struct;
struct dataservice_mgr_object_struct;

typedef void (*DSI_STATE_HANDLER) (
    struct dataservice_mgr_ctrl_struct const *psCtrl,
    struct dataservice_mgr_object_struct *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM ePreviousState
        );

typedef struct dataservice_mgr_object_struct
{
    // Event handler (callback)
    DATASERVICE_EVENT_MASK tEventRequestMask;
    GENERIC_DATASERVICE_EVENT_CALLBACK vEventHandler;
    void *pvEventHandlerArg;

    // Static information which is threadsafe
    // for this service
    DATASERVICE_MGR_INFO_STRUCT sInfo;

    // Pointer to structure which groups support for
    // DSRL-based attributes
    DATASERVICE_DSRL_SUPPORT_STRUCT *psDSRL;

    // State and current Error Code for the
    // Data Service Manager
    DATASERVICE_STATE_ENUM eState;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode;

    // A flag indicating if this object requires
    // time to be known for this service
    BOOLEAN bRequiresTime;

    // A flag indicating if the underlying framework is providing
    // OTA data
    BOOLEAN bOTADataAvailable;

    // Flag indicating if the service has made
    // its resources ready yet
    BOOLEAN bServiceResourcesReady;

    // Flag indicating if an autonomous service
    // will automatically stop when all
    // decoders are unsubscribed
    BOOLEAN bStopWhenUnsubComplete;

    // SRH Name we are starting data services on
    const char *pacSRHDriverName;

    // Name of the top-level service (if any)
    char acServiceName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Radio specific data object
    RADIO_PRIVATE_DATA_OBJECT hRadioSpecificData;

    // The entry containing this object in master list
    OSAL_LINKED_LIST_ENTRY hMasterEntry;

    DATASERVICE_ID tDataID;
    BOOLEAN bMultiDSIRequired;
    DATASERVICE_MGR_PRODUCTS_STRUCT sProducts;
    DSI_STATE_HANDLER vDSIStateHandler;

    // Associated Data service universal identifiers
    OSAL_OBJECT_HDL hDSIs;

    // Flag indicates if service needs the device group to be loaded
    BOOLEAN bRequiresDeviceGroup;

    // Associated timed event (for generic data services)
    DATASERVICE_TIMED_EVENT_HDL hGenericTimedEvent;

} DATASERVICE_MGR_OBJECT_STRUCT;

// Manager event structure
typedef struct dataservice_mgr_event_struct
{
    // Data Service this event is coming from
    DATASERVICE_MGR_OBJECT hManager;

    // The event info to dispatch.
    DATASERVICE_FW_EVENT_ENUM eDataEvent;

    // Event arg
    void *pvArg;

} DATASERVICE_MGR_EVENT_STRUCT;

typedef struct dataservice_mgr_sms_objects_struct
{
    // SRH Name
    STRING_OBJECT hSRHDriverName;

    // Handle to our MODULE and
    // variables which store status
    MODULE_OBJECT hModule;
    BOOLEAN bModuleInError;
    BOOLEAN bModuleAssociated;

} DATASERVICE_MGR_SMS_OBJECTS_STRUCT;

typedef struct dataservice_mgr_device_struct
{
    DEVICE_OBJECT hDeviceObject;
    DATASERVICE_EVENT_TIMESTAMP_STRUCT sLastUpdateTime;

} DATASERVICE_MGR_DEVICE_STRUCT;

typedef struct dataservice_mgr_ctrl_struct
{
    // List of data service managers current in existence
    OSAL_OBJECT_HDL hMgrList;

    // Structure for all DSM-wide device-related attributes
    DATASERVICE_MGR_DEVICE_STRUCT sDevice;

    // List of all cached decoder subscription requests
    OSAL_OBJECT_HDL hSubscribedDecoders;

    // List of all timed events created for
    // data services
    OSAL_OBJECT_HDL hTimedEvents;

    // Event Handler
    SMS_EVENT_HANDLER hEventHdlr;

    // Services task for the DSM
    SMS_TASK_HANDLE hServicesTask;

    // Handle to the top-level radio data control object
    RADIO_PRIVATE_DATA_OBJECT hRadioDataCtrl;

    // DSM top-level tag
    TAG_OBJECT hTag;

    // SRM and other object's in use by the DSM
    DATASERVICE_MGR_SMS_OBJECTS_STRUCT sObjectInfo;

    // Time Update Notification Handle
    OSAL_TIME_NOTIFICATION_OBJECT hTimeNotificationHandle;

    // Flag indicates if we've loaded the system-wide device group
    BOOLEAN bDeviceGroupLoaded;

} DATASERVICE_MGR_CTRL_STRUCT;

typedef struct dataservice_timed_event_struct
{
    // The OSAL timer object feeding this event
    // to the manager
    OSAL_OBJECT_HDL hTimer;

    // The event itself
    SMS_EVENT_HDL hTimedEvent;

    // The manager associated with this event
    DATASERVICE_MGR_OBJECT hManager;

    // Linked list entry handle
    OSAL_LINKED_LIST_ENTRY hEntry;

} DATASERVICE_TIMED_EVENT_STRUCT;

typedef struct dataservice_subscribed_decoder_struct
{
    // Flag indicates if this decoder is
    // to be released when it has become
    // fully unsubscribed from services
    BOOLEAN bReleaseWhenUnsubscribed;

    // The decoder that is available for subscriptions
    DECODER_OBJECT hDecoder;

    // The list of services the decoder is subscribed to
    OSAL_OBJECT_HDL hSubscriptions;

    // List entry handle
    OSAL_LINKED_LIST_ENTRY hEntry;

} DATASERVICE_SUBSCRIBED_DECODER_STRUCT;

typedef struct dataservice_radio_iterator_struct
{
    BOOLEAN (*bIterator)(RADIO_PRIVATE_DATA_OBJECT hRadioData, void *pvIteratorArg);
    void *pvIteratorArg;

} DATASERVICE_RADIO_ITERATOR_STRUCT;

typedef struct dataservice_mgr_error_iterator_struct
{
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    DATASERVICE_ERROR_CODE_ENUM eErrorCode;
    BOOLEAN (*bIterator)
        (DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
         DATASERVICE_MGR_OBJECT_STRUCT *psObj,
         DATASERVICE_ERROR_CODE_ENUM eErrorCode);

} DATASERVICE_MGR_ERROR_ITERATOR_STRUCT;

typedef struct dataservice_mgr_tag_iterator_struct
{
    size_t tMaxEntries;
    DSRL_ARG_STRUCT *psArg;
    DATASERVICE_IMPL_CREATE_TARGET_FROM_TAG hCreateTargetFromTag;

} DATASERVICE_MGR_TAG_ITERATOR_STRUCT;

typedef struct dataservice_decoder_unsubscribe_iterator_struct
{
    DECODER_OBJECT hDecoder;
    DATASERVICE_MGR_OBJECT_STRUCT *psServiceToMatch;
    BOOLEAN bUnsubscribeIssued;

} DATASERVICE_DECODER_UNSUBSCRIBE_ITERATOR_STRUCT;

typedef struct dataservice_subscriber_detect_struct
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bHasSubscriptions;

} DATASERVICE_SUBSCRIBER_DETECT_STRUCT;

typedef struct dataservice_iterate_shim_struct
{
    DECODER_OBJECT hDecoder;
    DATASERVICE_MGR_OBJECT_STRUCT *psService;
    DATASERVICE_IMPL_UPDATE_SUBSCRIBED_ITERATOR bIterator;
    void *pvIteratorArg;

} DATASERVICE_ITERATE_SHIM_STRUCT;

typedef struct dataservice_start_dsi_iterator_arg
{
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl;
    DATASERVICE_MGR_OBJECT_STRUCT *psObj;

} DATASERVICE_START_DSI_ITERATOR_ARG;

typedef struct dataservice_mgr_product_state_update_iterator_arg
{
    DATASERVICE_MGR_OBJECT_STRUCT *psObj;
    DSI tDSI;
    DATASERVICE_STATE_ENUM eDSIState;
    SXM_DMI *ptDMIs;
    UN8 un8DMICount;

} DATASERVICE_MGR_PRODUCT_STATE_UPDATE_ITERATOR_ARG;

typedef struct dataservice_mgr_product_iterator_arg
{
    DATASERVICE_MGR_OBJECT hManager;
    DATA_PRODUCT_ITERATOR_CALLBACK bCallback;
    void *pvCallbackArg;

} DATASERVICE_MGR_PRODUCT_ITERATOR_ARG;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static DATASERVICE_MGR_OBJECT hStart (
    const char *pacSRHDriverName,
    DSI tDSI,
    BOOLEAN bEnableAll,
    DATASERVICE_EVENT_MASK tEventRequestMask,
    GENERIC_DATASERVICE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vStop (
    DATASERVICE_MGR_OBJECT hManager
        );

static DATASERVICE_STATE_ENUM eState (
    DATASERVICE_MGR_OBJECT hManager
        );

static DATASERVICE_ERROR_CODE_ENUM eErrorCode (
    DATASERVICE_MGR_OBJECT hManager
        );

static SMSAPI_RETURN_CODE_ENUM eManageDataStream (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
    size_t tNumDMIsToConfigure
        );

static SMSAPI_RETURN_CODE_ENUM eEnableProduct (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK tProductMask
        );

static SMSAPI_RETURN_CODE_ENUM eDisableProduct (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType
        );

static SMSAPI_RETURN_CODE_ENUM eProductState (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK *ptProductMask,
    DATA_PRODUCT_STATE_ENUM *peProductState
        );

static SMSAPI_RETURN_CODE_ENUM eIterateProducts (
    DATASERVICE_MGR_OBJECT hManager,
    DATA_PRODUCT_ITERATOR_CALLBACK bCallback,
    void *pvCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eStartTimedEvent (
    DATASERVICE_MGR_OBJECT hManager,
    BOOLEAN bRepeat,
    UN32 un32IntervalInSeconds
        );

static void vStopTimedEvent (
    DATASERVICE_MGR_OBJECT hManager
        );

static SMSAPI_RETURN_CODE_ENUM eSendServiceEvent (
    DATASERVICE_MGR_OBJECT hManager
        );

/* Object Private Prototypes */

static void vDestroyDSMCtrl (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    BOOLEAN bDestroyObject
        );

static BOOLEAN bInitializeTask (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vEventHandler (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    SMS_EVENT_HDL hEvent
        );

typedef struct dataservice_create_callback_struct
{
    DATASERVICE_CREATE_STRUCT *psCreateService;
    void *pvEventHandlerArg;

} DATASERVICE_CREATE_CALLBACK_STRUCT;

static SMS_OBJECT hCreateDSMChildCallback (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_CREATE_CALLBACK_STRUCT *psCreate
        );

static DATASERVICE_MGR_OBJECT_STRUCT *psCreate (
    DATASERVICE_CREATE_STRUCT *psCreateService,
    void *pvEventHandlerArg
        );

static void vRelease (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bServiceStart (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_STATE_ENUM eReadyState
        );

static BOOLEAN bEnableDataFlow (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT const *psObj,
    DSI tDSI
        );

static void vServiceHalted (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vLoadDeviceGroup (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    const char *pacSRHDriverName
        );

static void vCallEventHandler (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_EVENT_MASK tEvent,
    void *pvEventArg
        );

static BOOLEAN bInitCachedSubscriptionList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vUninitCachedSubscriptionList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static BOOLEAN bInitTimedEventList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vUninitTimedEventList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static BOOLEAN bCreateDataServiceTable (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vDestroyDataServiceTable (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static BOOLEAN bGetDSMTag (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static SMS_EVENT_HDL hAllocateEvent (
    DATASERVICE_MGR_OBJECT hManager,
    DATASERVICE_FW_EVENT_ENUM eEvent,
    SMS_EVENT_DATASERVICE_ARG_UNION **ppuEventArg
        );

static N16 n16CompareDecoderEntries (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psSub1,
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psSub2
        );

static N16 n16CompareEntryToHandle (
    void *pvObj1,
    void *pvObj2
        );

static BOOLEAN bValidateManager (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager
        );

static BOOLEAN bDecoderSubscribedToService (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bIssueSubscribeRequestToAllReadyServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bIssueUnsubscribeRequestToAllServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bIssueUnSubscribeRequestFromAllDecoders (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_DECODER_UNSUBSCRIBE_ITERATOR_STRUCT *psUnsubscribe
        );

static BOOLEAN bIssueSxiMessageToAllReadyServices (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    const SMS_EVENT_DATASERVICE_SXI_MESSAGE_EVENT_UNION *puSxiMessage
        );

static BOOLEAN bIterateDecodersForServiceStart (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bIterateDecodersForServiceStop (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_SUBSCRIBER_DETECT_STRUCT *psDetect
        );

static BOOLEAN bIterateSubscribedShim (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder,
    DATASERVICE_ITERATE_SHIM_STRUCT *psShim
        );

static DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psFindDecoderEntry (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bAddDecoderToSubscribedList (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bRemoveDecoderFromSubscribedList (
    DATASERVICE_SUBSCRIBED_DECODER_STRUCT *psDecoder
        );

static BOOLEAN bSetError (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static void vProcessStateChange (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_STATE_ENUM eState,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static void vHandleStopRequest (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vProcessDeviceRadiusUpdate (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DISTANCE_OBJECT hRadius
        );

static void vProcessDevicePositionUpdate (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DEVICE_EVENT_MASK tEventMask,
    LOCATION_OBJECT hDevLocation
        );

static void vProcessTimeout (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvArg
        );

static void vHandleDSRLEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DSRL_ARG_STRUCT *psArg
        );

static void vHandleServiceSpecificEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvArg
        );

static BOOLEAN bPreProcessDSRLEvent (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        );

static BOOLEAN bPreProcessFavoriteEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT **ppsArg
        );

static void vPostProcessFavoriteEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        );

static BOOLEAN bPreProcessDeviceEvent (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        );

static BOOLEAN bPreProcessRemoveDeviceDSRLEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vPostProcessDeviceEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSRL_ARG_STRUCT *psArg
        );

static void vPostProcessRemoveDeviceDSRLEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bChildCountIterator(
    TAG_OBJECT hTag, size_t *ptChildCount
        );

static BOOLEAN bTargetCreationIterator(
    TAG_OBJECT hTag,
    DATASERVICE_MGR_TAG_ITERATOR_STRUCT *psIterator
        );

static void vHandleDecoderSubscribed (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        );

static void vHandleDecoderUnsubscribeReq (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DECODER_OBJECT hDecoder
        );

static void vHandleDirectedDecoderUnsubscribe (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    DECODER_OBJECT hDecoder
        );

static void vHandleManageDataStream (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    SMS_EVENT_DATASERVICE_STREAM_CONFIG_ARG_STRUCT *psArg
        );

static void vPlaceAllManagersIntoError (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static BOOLEAN bAddManagerToList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vRemoveManagerFromList (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bDataServiceRadioDataIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_RADIO_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bDataServiceStartIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static BOOLEAN bDataServiceErrorIterator (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_ERROR_ITERATOR_STRUCT *psIterator
        );

static DATASERVICE_TIMED_EVENT_HDL hRegisterTimedEvent (
    DATASERVICE_MGR_OBJECT hManager,
    void *pvTimerEventArg
        );

static SMS_EVENT_HDL hAllocateTimerEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT hManager,
    void *pvTimerEventArg
        );

static BOOLEAN bStopAllTimedEventsForManager (
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed,
    DATASERVICE_MGR_OBJECT hManager
        );

static BOOLEAN bRemoveAllTimedEventsForManager (
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed,
    DATASERVICE_MGR_OBJECT hManager
        );

static void vDestroyTimedEventEntry(
    DATASERVICE_TIMED_EVENT_STRUCT *psTimed
        );

static BOOLEAN bStopService (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bRegisterForDeviceUpdates (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hDeviceLocationInstance
        );

static void vUnRegisterForDeviceUpdates (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bSetServiceDeviceLocation (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    LOCATION_OBJECT hServiceDevLocation
        );

static void vDeviceUpdateCallback (
    void *pvRegistrationArg,
    DEVICE_EVENT_MASK tEventMask,
    DISTANCE_OBJECT hDistance,
    void *pvCallbackArg
        );

static void vGenericTimerHandler (
    OSAL_OBJECT_HDL hTimer,
    void *pvArg
        );

static void vLogStateChange (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    const char *pacServiceName,
    DATASERVICE_STATE_ENUM eState,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static void vTimeAvailableNotification (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        );

static N16 n16CompareDataIds (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj1,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj2
        );

static BOOLEAN bCheckModule (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bIsWaitingForTime (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bIsWaitingForOthers (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bStartDSIIterator(
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_START_DSI_ITERATOR_ARG *psArg
        );

static BOOLEAN bStartAllDSIs (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bProductStateUpdateIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_PRODUCT_STATE_UPDATE_ITERATOR_ARG *psArg
            );

static BOOLEAN bProductForcedErrorIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
            );

static BOOLEAN bProductForcedDisableIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
            );

static BOOLEAN bProductIterator (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATASERVICE_MGR_PRODUCT_ITERATOR_ARG *psArg
            );

static void vApplyDSIStateToProducts (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM ePreviousState
        );

static void vApplyDSIStateToService (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM ePreviousState
        );

static void vSetDSIState (
    DATASERVICE_MGR_CTRL_STRUCT const *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI,
    DATASERVICE_STATE_ENUM eState
        );

static N16 n16CompareProducts (
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct1,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct2
        );

static DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psFindProduct (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATA_PRODUCT_TYPE_ENUM eProductType
        );

static void vHandleEnableProduct (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK tProductMask
        );

static void vHandleDisableProduct (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct
        );

static void vHandleStartTimedEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bRepeat,
    UN32 un32IntervalInSeconds
        );

static void vHandleStopTimedEvent (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj
        );

static void vSetProductState (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_PRODUCT_INFO_STRUCT *psProduct,
    DATA_PRODUCT_STATE_ENUM eState
        );

static N16 n16CompareDSIs (
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI1,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI2
        );

static DATASERVICE_MGR_DSI_REF_STRUCT *psCreateDSI (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSI tDSI,
    size_t tSuggestedOTABufferByteSize,
    BOOLEAN bEnableAllDMIs
        );

static DATASERVICE_MGR_DSI_REF_STRUCT *psFindDSI (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DSI tDSI
        );

static void vStartDSI (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        );

static void vStopDSI (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        );

static void vDSIRef (
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        );

static void vDSIUnref (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    DATASERVICE_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_MGR_DSI_REF_STRUCT *psDSI
        );

static void vHandleModuleAssociateEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    MODULE_OBJECT hModule,
    STI_HDL hSTI,
    STRING_OBJECT hSRHDriverName
        );

static void vHandleModuleUnassociateEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl,
    MODULE_OBJECT hModule
        );

static BOOLEAN bTryStopDSM (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static BOOLEAN bPostFinalStopEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

static void vHandleFinalStopEvent (
    DATASERVICE_MGR_CTRL_STRUCT *psCtrl
        );

/***************/
/** VARIABLES **/
/***************/

// DSM task config struct
static const SMS_TASK_CONFIGURATION_STRUCT gsDSMTaskConfiguration =
{
    /*.pacName = */"DSM",
    /*.un32StackSize = */32768, // bytes
    /*.ePriority = */OSAL_TASK_PRIORITY_LOW,
    /*.un32Options = */OSAL_TASK_OPTION_DEBUG_OUTPUT,
    /*.un16ReportingInterval = */0, // seconds -- no set interval
    /*.un32EventQueueSize = */4096, // Bosch ID#7: Increase the queue size as discussed with SXM 
    /*.un32EventHandlerOptions = */SMS_EVENT_HANDLER_OPTION_DUAL_Q
};

// Global (re-usable) instance of an interface for this object
const GENERIC_DATASERVICE_INTERFACE_STRUCT DATA =
{
  /*.hStart = */hStart,
  /*.vStop =  */vStop,
  /*.eState = */eState,
  /*.eErrorCode = */eErrorCode,
  /*.eManageDataStream = */eManageDataStream,
  /*.eEnableProduct = */eEnableProduct,
  /*.eDisableProduct = */eDisableProduct,
  /*.eProductState = */eProductState,
  /*.eIterateProducts = */eIterateProducts,
  /*.eStartTimedEvent = */eStartTimedEvent,
  /*.vStopTimedEvent = */vStopTimedEvent,
  /*.eSendServiceEvent = */eSendServiceEvent
};

#endif	// _DATASERVICE_MGR_OBJ_H_
