/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CHANNELLIST_OBJ_H_
#define _CHANNELLIST_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "browse_obj.h"
#include "category_obj.h"
#include "sms_update.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CHANNELLIST_OBJECT_NAME "ChannelList"


// The following are events for which the application may be notified
// of events that occur via the CHANNELLIST object they create. These
// specific events however do not impact the make-up or construction of the
// list, i.e. members, so these cannot be used to "filter" CHANNELs
// which may be members of the CHANNELLIST.
#define CHANNEL_OBJECT_EVENT_METADATA      (      \
    (CHANNEL_EVENT_MASK)                          \
    CHANNEL_OBJECT_EVENT_TITLE |                  \
    CHANNEL_OBJECT_EVENT_ARTIST |                 \
    CHANNEL_OBJECT_EVENT_COMPOSER |               \
    CHANNEL_OBJECT_EVENT_ALBUM |                  \
    CHANNEL_OBJECT_EVENT_CONTENTINFO |            \
    CHANNEL_OBJECT_EVENT_DESCRIPTION |            \
    CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS |       \
    CHANNEL_OBJECT_EVENT_TYPE_INFO |              \
    CHANNEL_OBJECT_EVENT_EPG |                    \
    CHANNEL_OBJECT_EVENT_IR_NAV_CLASS |           \
    CHANNEL_OBJECT_EVENT_ACO                      \
        )

// The following are events for which the application may be notified
// of events that occur via the CHANNELLIST object they create. These
// specific events however do may impact the make-up or construction of the
// list, i.e. members, so these may be used to "filter" CHANNELs
// which may be members of the CHANNELLIST.
#define CHANNEL_OBJECT_EVENT_CHANNEL_INFO  (      \
    (CHANNEL_EVENT_MASK)                          \
    CHANNEL_OBJECT_EVENT_SERVICE_ID |             \
    CHANNEL_OBJECT_EVENT_CHANNEL_ID |             \
    CHANNEL_OBJECT_EVENT_ATTRIBUTES |             \
    CHANNEL_OBJECT_EVENT_NAME |                   \
    CHANNEL_OBJECT_EVENT_CATEGORY |               \
    CHANNEL_OBJECT_EVENT_ART |                    \
    CHANNEL_OBJECT_EVENT_REMOVED                  \
        )


// Note: A compiler error will occur here if the defined channel list
// metadata and channel info masks are not a complete union of all
// available channel masks. Not having this so exposes the risk to miss
// needed channel events.
COMPILE_TIME_ASSERT(
    (CHANNEL_OBJECT_EVENT_METADATA | CHANNEL_OBJECT_EVENT_CHANNEL_INFO)
        == CHANNEL_OBJECT_EVENT_ALL,
    CHECK_CHANNEL_OBJECT_EVENT_masks_for_channellist_are_not_complete);

// Note: A compiler error will occur here if you have redundant channel list
// masks in the metadata and channel info masks defined above. If there
// were redundant events set the CHANNELLIST would result in duplicate
// events being sent.
COMPILE_TIME_ASSERT(
    (CHANNEL_OBJECT_EVENT_METADATA & CHANNEL_OBJECT_EVENT_CHANNEL_INFO)
        == CHANNEL_OBJECT_EVENT_NONE,
    CHECK_CHANNEL_OBJECT_EVENT_masks_for_channellist_have_redundant_events);

  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

typedef struct channellist_object_struct
{
    // ChannelList Object Name
    char acName[OSAL_DEV_MAX_NAME_LENGTH_WITH_NULL];

    // Decoder object this channel belongs to
    DECODER_OBJECT hDecoder;

    // Browse object which performs all
    // basic browsing operations
    BROWSE_OBJECT hBrowse;

    // Reference channel entry
    OSAL_LINKED_LIST_ENTRY hReferenceChannelEntry;

    // Reference category object
    CATEGORY_OBJECT hReferenceCategory;

    // Top (first) channel entry
    OSAL_LINKED_LIST_ENTRY hTopChannelEntry;

    // Bottom (last) channel entry
    OSAL_LINKED_LIST_ENTRY hBottomChannelEntry;

    // Number of channels requested to be in the list
    // before / after the reference
    UN16 un16RequestedBefore;
    UN16 un16RequestedAfter;

    // Number of channels actually in this list
    // before / after the reference
    UN16 un16BeforeActual;
    UN16 un16AfterActual;

    // Channel List containing CHANNELLIST_ENTRY_STRUCTs
    OSAL_OBJECT_HDL hChannelList;

    OSAL_OBJECT_HDL hChannelListOld;
    OSAL_LINKED_LIST_ENTRY hCursor;

    CHANNEL_EVENT_MASK tEventRequestMask;

    // Channellist Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    BOOLEAN bReady;

} CHANNELLIST_OBJECT_STRUCT;

typedef struct channellist_entry_struct
{
    CHANNEL_OBJECT hChannel;
    CHANNEL_EVENT_MASK tMask;

} CHANNELLIST_ENTRY_STRUCT;

typedef struct channellist_iterator
{
    CHANNEL_OBJECT hChannel;
    UN16 un16Count;

} CHANNELLIST_ITERATOR_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static CHANNELLIST_OBJECT hCreate (
    DECODER_OBJECT hDecoder,
    BROWSE_TYPE_ENUM eBrowseType,
    CHANNEL_ID tReferenceChannelId,
    CATEGORY_ID tReferenceCategoryId,
    UN16 un16RequestedBefore,
    UN16 un16RequestedAfter,
    CHANNEL_EVENT_MASK tEventRequestMask,
    CHANNELLIST_OBJECT_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        );

static void vDestroy (
    CHANNELLIST_OBJECT hChannelList
        );

static BOOLEAN bUpdate (
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tMask
        );

static SMSAPI_RETURN_CODE_ENUM eModifyRegisteredEventMask (
    CHANNELLIST_OBJECT hChannelList,
    CHANNEL_EVENT_MASK tEventMask,
    SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
        );

static CHANNEL_OBJECT hChannel (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        );

static CHANNEL_ID tChannelId (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        );

static CHANNEL_EVENT_MASK tChannelEventMask (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        );

static CATEGORY_OBJECT hCategory (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        );

static CATEGORY_ID tCategoryId (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16Offset
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseList (
    CHANNELLIST_OBJECT hChannelList,
    N16 n16BrowseOffset
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseStyleSet(
    CHANNELLIST_OBJECT hChannelList,
    BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
    void *pvBrowseCategoryCompareArg,
    BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
    void *pvBrowseChannelCompareArg
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseCategory (
    CHANNELLIST_OBJECT hChannelList,
    SMSAPI_DIRECTION_ENUM eDirection
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseToCategory (
    CHANNELLIST_OBJECT hChannelList,
    CATEGORY_ID tCategoryId
        );

static size_t tSize (
    CHANNELLIST_OBJECT hChannelList,
    UN16 *pun16Before,
    UN16 *pun16After
        );

/* Object Private Prototypes */

static CHANNELLIST_OBJECT_STRUCT *psLockSystem (
    CHANNELLIST_OBJECT hChannelList
        );

static void vUnlockSystem (
    CHANNELLIST_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitChannelList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    BROWSE_TYPE_ENUM eBrowseType,
    CHANNEL_ID tReferenceChannelId,
    CATEGORY_ID tReferenceCategoryId
        );

static void vChannelInfoEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vChannelMetadataEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vCategoryEventCallback (
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static void vCCacheEventCallback (
    CCACHE_OBJECT hCCache,
    CCACHE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static BOOLEAN bCreateList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CCACHE_OBJECT hCCache
        );

static BOOLEAN bPopulateChannelList (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CCACHE_OBJECT hCCache,
    size_t tAvailable
        );

#if DEBUG_OBJECT==1

static BOOLEAN bPrintList(
    void *pvData,
    void *pvArg
        );

#endif // DEBUG_OBJECT==1

static BOOLEAN bUnregisterChannelNotificationIterator(
    void *pvData,
    void *pvArg
        );

static BOOLEAN bChannelCountIterator(
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdateChannelMask(
    CHANNELLIST_ENTRY_STRUCT *psEntry,
    void *pvArg
        );

static SMSAPI_RETURN_CODE_ENUM eBrowseCategories (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CATEGORY_ID tCategoryId,
    SMSAPI_DIRECTION_ENUM eDirection );

static void vClearChannellist(
    CHANNELLIST_OBJECT_STRUCT *psObj
        );

static void vUnregisterChannelNotifications(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hChannelList
        );

static void vClearChannelSet(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hChannelList
        );

static void vUpdateChannelPresetHandle(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    N16 n16ChannelOffset,
    BOOLEAN bResetHandle
        );

static BOOLEAN bBrowseAllChannelsCompareHandler(
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static BOOLEAN bBrowseAllCategoriesCompareHandler(
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT hCategory,
    void *pvIterateArg,
    N16 *pn16Offset
        );

static N16 n16CompareChannelIDs (
    CHANNELLIST_ENTRY_STRUCT *psEntry1,
    CHANNELLIST_ENTRY_STRUCT *psEntry2
        );

static N16 n16CompareChannelHandles (
    CHANNELLIST_ENTRY_STRUCT *psEntry1,
    CHANNELLIST_ENTRY_STRUCT *psEntry2
        );

static CHANNELLIST_ENTRY_STRUCT *psGetChannelEntry(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    N16 n16Offset
        );

static CHANNELLIST_ENTRY_STRUCT *psGetChannelEntryByHandle(
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        );

static OSAL_RETURN_CODE_ENUM eAddChannel (
    CHANNELLIST_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    OSAL_LINKED_LIST_ENTRY *phChannelEntry,
    BOOLEAN bBefore
        );

static OSAL_RETURN_CODE_ENUM eRemoveChannelEntry (
    OSAL_LINKED_LIST_ENTRY hChannelEntry
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const CHANNELLIST_OBJECT_INTERFACE_STRUCT CHANNELLIST =
{
    /*.hCreate = */hCreate,
    /*.vDestroy = */vDestroy,
    /*.bUpdate = */bUpdate,
    /*.eModifyRegisteredEventMask = */eModifyRegisteredEventMask,
    /*.hChannel = */hChannel,
    /*.tChannelId = */tChannelId,
    /*.tChannelEventMask = */tChannelEventMask,
    /*.hCategory = */hCategory,
    /*.tCategoryId = */tCategoryId,
    /*.eBrowseList = */eBrowseList,
    /*.eBrowseStyleSet = */eBrowseStyleSet,
    /*.eBrowseCategory = */eBrowseCategory,
    /*.eBrowseToCategory = */eBrowseToCategory,
    /*.tSize = */tSize
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif  // _CHANNELLIST_OBJ_H_
