/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CHANNEL_ART_MGR_OBJ_H_
#define _CHANNEL_ART_MGR_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "dataservice_mgr_impl.h"
#include "sql_interface_obj.h"

#include "channel_art_mgr_obj.h"
#include "channel_art_obj.h"
#include "channel_art_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CHANNEL_ART_MGR_OBJECT_NAME "Art Service"

/* Channel Art Directory Name */
#define CHANNEL_ART_DIRECTORY_NAME "channelart"

/* Album Art Directory Name */
#define ALBUM_ART_DIRECTORY_NAME "albumart"

#define CHANNEL_ART_DEFAULT_IMAGE_ID (0)

#define CHANNEL_ART_OPTION_ALL ( \
    CHANNEL_ART_OPTION_FILE_PATH | \
    CHANNEL_ART_OPTION_EPHEMERAL_FILE_PATH  \
        )

// Constant which defines the size of the buffer used by
// channel art for many purposes.  The longest strings that will
// be used by this service are DB queries / commands, so make that
// the size of the buffer
#define CHANNEL_ART_SHARED_BUFFER_LEN ( CHANNEL_ART_MAX_SQL_STRING_LENGTH )

/* Image File name format constants */

// Image files have names of the format IxxxxxTyy,
// where Ixxx indicates the file id
// and Tyy indicates the file type

// These constants are used to create filenames via snprintf
#define CHANNEL_ART_FILE_NAME_ID_LABEL "I%05uT%02u"
#define CHANNEL_ART_FILE_NAME_ID_LABEL_LEN (9)

#define CHANNEL_ART_JPEG_FILE_EXTENSION ".jpg"
#define CHANNEL_ART_PNG_FILE_EXTENSION ".png"

#define ALBUM_ART_PER_DB_ATTACH_NAME "ref"

#define ALBUM_ART_FILE_NAME_ID_LABEL "A%04uV%03u.jpg"
#define ALBUM_ART_FILE_NAME_ID_LABEL_LEN (13)

// Album art uses two special associations. The default image for
// service 0 is defined by the Album Art spec as being the whole-
// service default.
#define ALBUM_ART_SERVICE_DEFAULT_ID ((SERVICE_ID)(0))

// Album art will delete any expired timed associations, plus
// any that are *about* to expire for efficiency reasons.
// This window defines how far out AA is allowed to delete
// any timed associations.
#define ALBUM_ART_CLEANUP_WINDOW_MS (100)

// Album art keep this amount of recently received images untouched
// during scrub procedure.
#define ALBUM_ART_PERSISTENT_IMAGES_COUNT (2)
#define ALBUM_ART_PERSISTENT_IMAGES_COUNT_MAX (255)

/* Channel Art Event & Processing Constants */

// This macro compares two association rows in order to
// determine if they are of the same source type, source id
// and image type (and neither is a remove).  If so,
// the assoc rows are considered "equal"
#define CHANNEL_ART_ASSOC_TYPE_EQUAL(psAssoc1, psAssoc2)       \
    (                                                          \
        ((psAssoc1)->bCategory == (psAssoc2)->bCategory)       \
     && ((psAssoc1)->un16Source == (psAssoc2)->un16Source)     \
     && ((psAssoc1)->un8ImageType == (psAssoc2)->un8ImageType) \
     && ((psAssoc1)->bRemove == FALSE)                         \
     && ((psAssoc2)->bRemove == FALSE)                         \
    )

// This macro compares two association rows in order to
// determine if they carry the same data (except for version
// data, since that is covered in other places). If so,
// the assoc rows are considered "equal"
#define CHANNEL_ART_ASSOC_EQUAL(psAssoc1, psAssoc2)       \
    (                                                          \
        ((psAssoc1)->bCategory == (psAssoc2)->bCategory)       \
     && ((psAssoc1)->un16Source == (psAssoc2)->un16Source)     \
     && ((psAssoc1)->un8ImageType == (psAssoc2)->un8ImageType) \
     && ((psAssoc1)->bContent == (psAssoc2)->bContent) \
     && ((psAssoc1)->un16ImageId == (psAssoc2)->un16ImageId) \
     && ((psAssoc1)->bHighPriority == (psAssoc2)->bHighPriority) \
    )

// The selected images may not indicate secondary images
// without also indicating primary images
#define CHANNEL_ART_SELECTED_IS_VALID(selected) \
    ( ((selected) != CHANNEL_ART_AVAILABLE_IMAGE_NONE) && \
      (((selected) & \
      (CHANNEL_ART_AVAILABLE_IMAGE_LOGO | CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO)) \
        != CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO) \
    )

/* Defines mask values for each product for filtration purposes */
#define CHANNEL_ART_PRODUCT_MASK_ALBUM  (1 << 0)
#define CHANNEL_ART_PRODUCT_MASK_CHAN   (1 << 1)

// Max number of parameters among all used prepared statements
#define MAX_SQL_BIND_PARAMS 8


  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

// Structure which associates a channel art
// object with a specific service id
// and other association attributes
// that are available within the
// channel art database
typedef struct channel_art_assoc_struct
{
    BOOLEAN bCategory;             // The type of service (cat/chan)
    UN16 un16Source;               // The source identifier
    CHANNEL_ART_OBJECT hChannelArt;// Accompanying channel art object
    BOOLEAN bDefault;              // The default flag
    OSAL_LINKED_LIST_ENTRY hEntry; // The linked list entry handle
    BOOLEAN bUpdated;              // Flag indicating if this association has changed
    BOOLEAN bReferenced;           // Flag indicating if this association has been referenced
                                   // the current association message
} CHANNEL_ART_ASSOC_STRUCT;


typedef struct album_art_pid_to_art_map
{
    PROGRAM_ID tPID;                  // The program ID
    CHANNEL_ART_OBJECT hArt;          // Accompanying channel art object
    BOOLEAN bInUse;                   // If TRUE, this association is in use
} ALBUM_ART_PID_TO_ART_MAP;

// The following struct is contains all the possible album art objects for a given
// service ID. One of the goals of album art was to preserve the same interface
// as channel art: the Channel Art Object. However, album art could return one
// of many different images depending on the current program ID and eType, which
// would require changing the interface of of the Channel Art Object if all this
// information was stored there. Instead, we store all the possibilities in the
// album_art_assoc_struct and give each CDO a Channel Art Object based off its
// PID/eType.
//
// Note that until the forthcoming "safe to delete" functionality is completed
// (in the near but indefinite future), this struct just stores two PIDs worth
// of art: one for the currently playing song, as well as one "on deck."
// Eventually, this will be replaced with a list of art that will be cleaned
// up using the forthcoming "safe to delete" metric.

typedef struct album_art_assoc_struct
{
    SERVICE_ID tSID;                     // The service ID

    OSAL_OBJECT_HDL hPidArtList;         // List of ALBUM_ART_PID_TO_ART_MAP
    UN32 un32PidListCapacity;            // Defines Pid list capacity

    CHANNEL_ART_OBJECT hSidArtTimed;     // Accompanying channel art object
    UN32 un32ExpirationDelta;            // If non-zero, the Unix epoch (seconds) when the
                                         // association expires

    CHANNEL_ART_OBJECT hDefaultArt;      // SID-specific default art
    CDO_TYPE_ENUM eType;                 // The type of content the default association applies to

    OSAL_LINKED_LIST_ENTRY hEntry;       // The linked list entry handle

    BOOLEAN bDefault;                    // The service default flag
    BOOLEAN bBlank;                      // The blank flag
    BOOLEAN bUpdated;                    // The updated flag

} ALBUM_ART_ASSOC_STRUCT;

typedef struct channel_art_mgr_ota_ctrl_struct
{
    // Flag indicating if the current message being processed
    // has generated events which need to be provided
    // to the subscribed decoders
    BOOLEAN bEventsPending;

    // Flag indicating if the secondary image available
    // with a primary image is needed
    BOOLEAN bSecondaryImageNeeded;

    // Flag indicating if the primary image in a set
    // of images is new or if it is an update
    BOOLEAN bPrimaryImageUpdated;

    // Stores the primary image type for later
    // use when processing secondary images
    CHANNEL_ART_ATTRIB_ROW_STRUCT sPrimaryImageAttrs;

} CHANNEL_ART_MGR_OTA_CTRL_STRUCT;

// Structure used to group attributes needed to
// process association messages
typedef struct channel_art_mgr_assoc_ctrl_struct
{
    // Attributes used for processing association
    // messages
    BOOLEAN bLastAssocSucceeded;
    CHANNEL_ART_ASSOC_ROW_STRUCT sChanLastAssocRow;
    CHANNEL_ART_ASSOC_STRUCT *psChanLastArtAssoc;

    // Other association message processing attributes

    // Flag indicating if the current association
    // message still has pending association entries
    // (this tells us if we can mark the current
    // association message as completed
    BOOLEAN bCurrentAssociationHasPendingEntries;

    // Flag indicating if this is the first time
    // the interface is providing us with a given
    // group of associations
    BOOLEAN bVerifyLineupAtCompletion;

    // Flag indicating if the current association
    // message applies to channels or categories
    BOOLEAN bCategory;

    // Flag indicating if the current association
    // message applies to overlay channels
    BOOLEAN bOverlay;

} CHANNEL_ART_MGR_ASSOC_CTRL_STRUCT;

typedef struct channel_art_owner_object_struct
{
    OSAL_OBJECT_HDL hChanAssociations;
    OSAL_OBJECT_HDL hAlbumAssociations;
    OSAL_OBJECT_HDL hLineBitmaps;

    // Default channel art associations
    CHANNEL_ART_ASSOC_STRUCT sDefaultChannelAssoc;
    CHANNEL_ART_ASSOC_STRUCT sDefaultCategoryAssoc;

    BOOLEAN bAlbumArtInitialized;
    BOOLEAN bChannelArtInitialized;

    // Following structure is introduced
    // to support deferred products deinitialization.
    // When product is disabled, all references are copied to sBackup
    // to be deinitialized next time when product is enabled
    // or on the service stop.
    struct
    {
        OSAL_OBJECT_HDL hChanAssociations;
        OSAL_OBJECT_HDL hAlbumAssociations;
        OSAL_OBJECT_HDL hLineBitmaps;

        // Default channel art associations
        CHANNEL_ART_ASSOC_STRUCT sDefaultChannelAssoc;
        CHANNEL_ART_ASSOC_STRUCT sDefaultCategoryAssoc;
    } sBackup;
} CHANNEL_ART_OWNER_OBJECT_STRUCT;

typedef struct channel_art_mgr_object_struct
{
    // Application Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // The handle to the channel art interface
    CHANNEL_ART_INTERFACE_OBJECT hChanInterface;

    // The handle to the channel art interface
    ALBUM_ART_INTERFACE_OBJECT hAlbInterface;

    // The object which owns the art list & art objects
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner;

    // DBMS connection handles
    SQL_INTERFACE_OBJECT hChanSQLConnection;
    SQL_INTERFACE_OBJECT hAlbumPerSQLConnection;

    // SQL bind parameters holder
    SQL_BIND_PARAMETER_STRUCT asBindParams[MAX_SQL_BIND_PARAMS];

    // Channel Art File Paths
    char *pacBasePath;
    char *pacChannelArtServiceFilePath;
    char *pacEphemeralPath;
    char *pacFullyQualifiedChanDatabaseFilePath;
    char *pacFullyQualifiedAlbumPerDatabaseFilePath;
    char *pacFullyQualifiedAlbumRefDatabaseFilePath;

    // Buffer used for string processing
    char acBuffer[CHANNEL_ART_SHARED_BUFFER_LEN+1];

    // Flag used to indicate if the art
    // objects managed here need to
    // be reset before being used
    BOOLEAN bResetArtObjects;

    // Flag indicating if the manager
    // is currently in an SQL transaction
    BOOLEAN bChanInTransaction;
    BOOLEAN bAlbumInTransaction;

    // Flag indicating if secondary
    // images are supported
    BOOLEAN bSecondarySupported;

    // Control structure for processing
    // association messages
    CHANNEL_ART_MGR_ASSOC_CTRL_STRUCT sAssocCtrl;

    // Control structure for processing OTA messaging
    CHANNEL_ART_MGR_OTA_CTRL_STRUCT sOTACtrl;

    // Image types that should be processed
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tSelectedImageTypes;

    // SQL prepared statement for selecting image attributes
    SQL_PREPARED_STATEMENT_HANDLE hSelectImageAttrsStmt;

    // SQL prepared statement for updating associated image attributes
    SQL_PREPARED_STATEMENT_HANDLE hUpdateAssocRowStmt;

    // SQL prepared statement for inserting associated image attributes
    SQL_PREPARED_STATEMENT_HANDLE hInsertAssocRowStmt;

    // SQL prepared statement for selecting image attributes for album art
    SQL_PREPARED_STATEMENT_HANDLE hAlbumSelectImageAttrsStmt;

    // Handle to expire event for removing
    // stale timed-association album art.
    DATASERVICE_TIMED_EVENT_HDL hCleanupEvent;

    // We can't get the time until next expiration from a
    // dataservice timed event (unlike OSAL Timers), so we
    // have to save the absolute time of the next cleanup
    // ourselves.
    UN32 un32NextCleanupSecs;

    // We have to assign ephemeral images an ID; we use
    // this to track the next ephemeral ID that we'll
    // hand out.
    UN32 un32NextEphemeralID;

} CHANNEL_ART_MGR_OBJECT_STRUCT;

// Structure used to perform channel art database queries
typedef struct channel_art_db_query_result_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bResultantRows;
    CHANNEL_ART_DB_ROW_UNION uDbRow;

} CHANNEL_ART_DB_QUERY_RESULT_STRUCT;

// Structure used to perform channel art database queries
typedef struct album_art_db_query_result_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bResultantRows;
    ALBUM_ART_DB_ROW_UNION uDbRow;

} ALBUM_ART_DB_QUERY_RESULT_STRUCT;

// Structure used to perform a channel art database join
typedef struct channel_art_db_join_result_struct
{
    BOOLEAN bSuccess;
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;

} CHANNEL_ART_DB_JOIN_RESULT_STRUCT;

/* General typedefs */

// Typedefs to aid in linked list operations

// Structure which aids in the process of updating
// all instances of an image in memory
typedef struct channel_art_mgr_image_update_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    CHANNEL_ART_ASSOC_STRUCT *psAssoc;
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psImageAttrs;
    BOOLEAN bEventAdded;

} CHANNEL_ART_MGR_IMAGE_UPDATE_STRUCT;

// Structure which aids in the "defaulting" of
// a particular channel art object
typedef struct channel_art_mgr_set_default_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    CHANNEL_ART_ATTRIB_ROW_STRUCT sImageAttrs;
    BOOLEAN bEventAdded;

} CHANNEL_ART_MGR_SET_DEFAULT_STRUCT;

typedef struct channel_art_mgr_default_by_type_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bCategory;
    BOOLEAN bOverlay;
    BOOLEAN bEventAdded;

} CHANNEL_ART_MGR_DEFAULT_BY_TYPE_STRUCT;

typedef struct channel_art_mgr_reset_assoc_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    BOOLEAN bCategory;
    BOOLEAN bOverlay;
    BOOLEAN bAssocUpdated;

} CHANNEL_ART_MGR_RESET_ASSOC_STRUCT;

typedef struct channel_art_default_leftover_struct
{
    CHANNEL_ART_OBJECT hDefaultArt;
    BOOLEAN bCategory;
    BOOLEAN bOverlay;
    BOOLEAN *pbEventGenerated;

} CHANNEL_ART_DEFAULT_LEFTOVER_STRUCT;

typedef enum channel_art_update_type_enum
{
    CHANNEL_ART_UPDATE_TYPE_START,
    CHANNEL_ART_UPDATE_TYPE_STOP,
    CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_START,
    CHANNEL_ART_UPDATE_TYPE_ALBUM_ART_STOP,
    CHANNEL_ART_UPDATE_TYPE_CHAN_ART_START,
    CHANNEL_ART_UPDATE_TYPE_CHAN_ART_STOP,
    CHANNEL_ART_UPDATE_TYPE_CHAN_ASSOC_CHANGE,
    CHANNEL_ART_UPDATE_TYPE_ALBUM_ASSOC_CHANGE,
    CHANNEL_ART_INVALID_UPDATE_TYPE

} CHANNEL_ART_UPDATE_TYPE_ENUM;

typedef struct channel_art_sub_update_struct
{
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj;
    CHANNEL_ART_UPDATE_TYPE_ENUM eUpdateType;
    BOOLEAN bSuccess;
    UN32 un32ActiveProductMask;

} CHANNEL_ART_SUB_UPDATE_STRUCT;

typedef enum channel_art_assoc_process_result_enum
{
    // The association was updated
    CHANNEL_ART_ASSOC_PROCESS_RESULT_UPDATED,

    // Obvious
    CHANNEL_ART_ASSOC_PROCESS_RESULT_NOT_UPDATED,
    CHANNEL_ART_ASSOC_PROCESS_RESULT_PENDING,

    // The association wasn't met, and it's time
    // to force the default on this entry
    CHANNEL_ART_ASSOC_PROCESS_RESULT_PENDING_FORCE_DEFAULT,

    CHANNEL_ART_ASSOC_PROCESS_RESULT_ERROR

} CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM;

typedef struct channel_art_translate_struct
{
    BOOLEAN bSuccess;
    OSAL_OBJECT_HDL hList;

} CHANNEL_ART_TRANSLATE_STRUCT;

/*
    Structure used to provide updates to
    subscribed decoders
*/

typedef union art_update_source_type_union
{
    SERVICE_ID tServiceId;
    CATEGORY_ID tCategoryId;

} ART_UPDATE_SOURCE_TYPE_UNION;

typedef struct art_update_struct
{
    CHANNEL_ART_OBJECT hArt;
    ART_UPDATE_SOURCE_TYPE_UNION uId;
    OSAL_LINKED_LIST_ENTRY hEntry;

} ART_UPDATE_STRUCT;

typedef struct album_art_cleanup_struct
{
    UN32 un32NextTimeout;
    UN32 un32Now;
} ALBUM_ART_CLEANUP_STRUCT;

/* 
    Structure used by CHANNEL_ART_vDecoderEventHandler 
    to process updates for that decoder
*/
typedef struct decoder_art_update_process_struct
{
    ART_UPDATE_STRUCT const *psCurUpdate;

} DECODER_ART_UPDATE_PROCESS_STRUCT;

// Structure used by the FSM to handle states on a
// per-product basis.
typedef BOOLEAN (*PRODUCT_STATE_HANDLER)
        ( CHANNEL_ART_MGR_OBJECT_STRUCT *psObj );

// Structure to hold a handler for each state for a product
typedef struct art_product_state_handlers
{
    PRODUCT_STATE_HANDLER bInitHandler;
    PRODUCT_STATE_HANDLER bReadyHandler;
    PRODUCT_STATE_HANDLER bDisabledHandler;
    PRODUCT_STATE_HANDLER bErrorHandler;
} ART_PRODUCT_STATE_HANDLERS;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static CHANNEL_ART_SERVICE_OBJECT hStart (
    const char *pacSRHDriverName,
    CHANNEL_ART_SERVICE_EVENT_MASK tEventRequestMask,
    CHANNEL_ART_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tSelectedImageTypes,
    CHANNEL_ART_OPTIONS_MASK tOptions,
    ...
        );

/* Public prototypes which utilize the channel art object */

static SMSAPI_RETURN_CODE_ENUM eUseArt (
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_ACCESS_CALLBACK vChannelArtAccessCallback,
    void *pvCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eImage (
    CHANNEL_ART_OBJECT hArt,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType,
    IMAGE_OBJECT *phImage,
    BOOLEAN *pbIsDefault
        );

/* Object Private Prototypes */

/* General */

static BOOLEAN bProcessOptions (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_OPTIONS_MASK tOptions,
    va_list *ptArgList
        );

static void vEventHandler (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

static BOOLEAN bProductFSM (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_STATE_ENUM eOldState,
    DATA_PRODUCT_STATE_ENUM eNewState,
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitializeArtOwner (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static void vUninitObject (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitDefaultAssociations (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static void vHandleDecoderSubscribed (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    DECODER_OBJECT hDecoder
        );

static DATA_PRODUCT_STATE_ENUM eGetNextState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        );

static BOOLEAN bGetDSIInfo (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        );

static BOOLEAN bHandleMessageReception (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bBuildCommonFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bBuildChannelArtFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bBuildAlbumArtFilePaths (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static CHANNEL_ART_OBJECT hChanGetArtForSourceAndType (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16SourceId,
    BOOLEAN bCategory
        );

static BOOLEAN bChanProcessArtDatabase (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bChanAssocsAvailable,
    BOOLEAN bOverlayChanAssocsAvailable,
    BOOLEAN bCatAssocsAvailable
        );

static BOOLEAN bAlbumProcessDatabase (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psOb
        );

static BOOLEAN bChanRegenerateArtAssocTable (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN bChanAssocsAvailable,
    BOOLEAN bOverlayChanAssocsAvailable,
    BOOLEAN bCatAssocsAvailable
        );

static BOOLEAN bImageTypeSelected (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        );

static CHANNEL_ART_AVAILABLE_IMAGE_MASK tImageTypeToMask (
    CHANNEL_ART_IMAGETYPE_ENUM eImageType
        );

/* Message Processors */

static BOOLEAN bProcessMessage (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        );

static BOOLEAN bChanProcessImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t tImageDataByteLen,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bAlbumProcessImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hPayload,
    STRING_OBJECT *phCaption,
    CHANNEL_ART_OBJECT *phArtToUpdate
        );

static BOOLEAN bWriteAlbumArtToFile(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16Id, UN16 un16Ver,
    char const * const pacFilePath,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bUpdateArtObjectsForImageUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    BOOLEAN *pbEventGenerated
        );

static CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eChanProcessAssocUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow
        );

static CHANNEL_ART_ASSOC_PROCESS_RESULT_ENUM eAlbumProcessAssocUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_OBJECT *phArtToUpdate,
    BOOLEAN bFromDB
        );

static BOOLEAN bChanFinalizeAssocProcessing (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN *pbEventGenerated
        );

/* Manager Interface functions */

static CHANNEL_ART_MGR_PROCESS_ASSOC_ENUM eChanAssocBegin (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_IMAGETYPE_ENUM eBaseImageType,
    BOOLEAN bVerifyLineup,
    BOOLEAN bCategory,
    BOOLEAN bOverlay
        );

static BOOLEAN bChanAssocEnd (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        );

static BOOLEAN bAlbumAssocBegin (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        );

static BOOLEAN bAlbumAssocEnd (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService
        );

static BOOLEAN bChanAssociationUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psNewAssoc,
    BOOLEAN *pbAssociationPending
        );

static BOOLEAN bAlbumAssociationUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_OBJECT *phArtToUpdate
        );

static BOOLEAN bChanResetAssociations (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    BOOLEAN bContentOnly,
    BOOLEAN bOverlay,
    BOOLEAN bCategory
        );

static BOOLEAN bChannelImageUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    size_t tImageDataByteLen,
    OSAL_BUFFER_HDL hImageData
        );

static BOOLEAN bAlbumImageUpdate (
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    ALBUM_ART_ASSOC_ROW_STRUCT *psAssocRow,
    OSAL_BUFFER_HDL hImageData,
    STRING_OBJECT *phCaption,
    CHANNEL_ART_OBJECT *phArtToUpdate
        );

/* Linked List Iterators / Searchers */

static BOOLEAN bResetContentArtWithMatchingType (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bUpdateChannelArtWithMatchingImageAttrs (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bIterateAssocTypesForImage (
    CHANNEL_ART_IMAGE_ASSOC_TYPE_ENUM eAssocType,
    void *pvCallbackArg
        );

static BOOLEAN bDefaultAssociationsByType (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bClearReferencedFlags (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    void *pvUnused
        );

static BOOLEAN bDefaultLeftoverAssociations (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_DEFAULT_LEFTOVER_STRUCT *psLeftover
        );

static BOOLEAN bCopyUpdatedAssocsToDecoderList (
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate,
    CHANNEL_ART_OBJECT hArt,
    UN16 un16Source,
    BOOLEAN bUpdated,
    BOOLEAN bCategory
        );

static BOOLEAN bChanCopyUpdatedAssocsToDecoderList (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate
        );

static BOOLEAN bAlbumCopyUpdatedAssocsToDecoderList (
    ALBUM_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_TRANSLATE_STRUCT *psTranslate
        );

static N16 n16ChanCompareAssociations (
    CHANNEL_ART_ASSOC_STRUCT *psAssoc1,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc2
        );

static N16 n16AlbumCompareAssociations (
    ALBUM_ART_ASSOC_STRUCT *psAssoc1,
    ALBUM_ART_ASSOC_STRUCT *psAssoc2
        );

static N16 n16CompareLineBitmaps (
    LINE_BITMAP_INFO_STRUCT *psLineBitmapInfo1,
    LINE_BITMAP_INFO_STRUCT *psLineBitmapInfo2
        );

static N16 n16FindAssociationWithImageAttrs (
    void *pvArg1,
    void *pvArg2
        );

static BOOLEAN bChanInspectDBInfo (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbChanAssocsAvailable,
    BOOLEAN *pbOverlayChanAssocsAvailable,
    BOOLEAN *pbCatAssocsAvailable
        );

static BOOLEAN bAlbumInspectDBInfo (
    SQL_INTERFACE_OBJECT hSQLConnection
        );

static BOOLEAN bChanUpdateAssociation (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psNewAssocRow,
    BOOLEAN *pbNewerVersion
        );

static BOOLEAN bChanUpdateAssociationEntry (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT **ppsAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow,
    BOOLEAN *pbAssociationEntryUpdated
        );

static BOOLEAN bChanRemoveAssociation (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow
        );

static BOOLEAN bChanDefaultAssoc (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    BOOLEAN bUpdateDB,
    BOOLEAN bOnlyReplaceExisting
        );

static void vSetUpdateAssocRowStmtParams (
    CHANNEL_ART_ASSOC_ROW_STRUCT const *psAssocRow,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        );

static void vSetInsertAssocRowStmtParams (
    CHANNEL_ART_ASSOC_ROW_STRUCT const *psAssocRow,
    SQL_BIND_PARAMETER_STRUCT *psBindParams
        );

/* Database query processors */

static BOOLEAN bProcessSelectDBInfoResult (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        );

#if 0
static BOOLEAN bProcessSelectImageAssoc (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        );
#endif

static BOOLEAN bProcessSelectImageAttrs (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bProcessSelectLineBitmap (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_QUERY_RESULT_STRUCT *psResult
        );

static BOOLEAN bChanProcessJoinDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT *psResult
        );

static BOOLEAN bAlbumProcessJoinDB (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    CHANNEL_ART_DB_JOIN_RESULT_STRUCT *psResult
        );

/* General Purpose */

static CHANNEL_ART_ASSOC_STRUCT * psChanGetArtAssoc (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    BOOLEAN bDefault,
    BOOLEAN bCategory,
    UN16 un16Source
        );

static ALBUM_ART_ASSOC_STRUCT* psAlbumGetAssocByService (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    SERVICE_ID tSID
        );

static CHANNEL_ART_OBJECT hAlbumGetArt (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    SERVICE_ID tSID,
    PROGRAM_ID tPID,
    CDO_TYPE_ENUM eType
        );

static SMSAPI_RETURN_CODE_ENUM eGetAlbumArtProperties (
    CHANNEL_ART_OBJECT hArt,
    UN32 *pun32ImageId,
    UN32 *pun32ImageVer
        );

static BOOLEAN bGetAssocRow (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psSearchParameters
        );

static BOOLEAN bGetAssocRowFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_STRUCT *psAssoc,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psAssocRow,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psSearchParameters
        );

static BOOLEAN bGetAttribRowFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psRelatedAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        );

static BOOLEAN bGetAttribRowFromDB (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        );

static BOOLEAN bGetAttribRow (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_ASSOC_ROW_STRUCT *psRelatedAssocRow,
    CHANNEL_ART_ATTRIB_ROW_STRUCT *psAttribRow
        );

static BOOLEAN bGetLineBitmapFromList (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        );

static BOOLEAN bGetLineBitmapFromDB (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        );

static BOOLEAN bGetLineBitmap (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        );

static void vAddNewLineBitmapEntry (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    LINE_BITMAP_INFO_STRUCT *psLineInfo
        );

static CHANNEL_ART_ASSOC_STRUCT * psChanAddNewAssocEntry (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    BOOLEAN bCategory,
    UN16 un16Source
        );

static SMSAPI_RETURN_CODE_ENUM eAlbumProbeDefaultImage(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    UN16 un16ImageId,
    UN16 *pun16ImageVersion
        );

static ALBUM_ART_ASSOC_STRUCT * psAlbumAddNewAssocEntry (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    SERVICE_ID tSID
        );

static ALBUM_ART_PID_TO_ART_MAP *psAlbumPidAssocFindArt (
    ALBUM_ART_ASSOC_STRUCT *psEntry,
    PROGRAM_ID tPID
        );

static ALBUM_ART_PID_TO_ART_MAP *psAlbumPidAssocAllocate (
    ALBUM_ART_ASSOC_STRUCT *psEntry,
    PROGRAM_ID tPID
        );

static void vAlbumPidAssocDestroy (
    ALBUM_ART_PID_TO_ART_MAP *psObj
        );

static void vAlbumPidAssocListClean(
    ALBUM_ART_ASSOC_STRUCT *psEntry
        );

static void vAlbumPidAssocListScrub (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    SERVICE_ID tSID,
    CHANNEL_ID tChannelID,
    PROGRAM_ID *atPIDList,
    UN8 un8PIDListSize
        );

static void vAlbumAssocDestroy (
    ALBUM_ART_ASSOC_STRUCT *psEntry
        );

// Reserved for future safe-to-delete functionality
#if 0
static void vAlbumRemoveAssocEntry (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psObj,
    UN16 un16ServiceId,
    UN32 un32ProgramId
        );
#endif

static BOOLEAN bChanClearUpdatedFlag (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bAlbumClearUpdatedFlag (
    void *pvData,
    void *pvArg
        );

static void vChanReleaseAssociations (
    void *pvData
        );

static void vAlbumReleaseAssociations (
    void *pvData
        );

static void vReleaseLineBitmaps (
    void *pvData
        );

static BOOLEAN bIssueUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    BOOLEAN bAlbum
        );

static BOOLEAN bChanClearAllUpdatedFlags (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bAlbumClearAllUpdatedFlags (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static void vStartTransaction (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbInTransaction
        );

static void vEndTransaction (
    SQL_INTERFACE_OBJECT hSQLConnection,
    BOOLEAN *pbInTransaction
        );

static BOOLEAN bSendUpdate (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    CHANNEL_ART_UPDATE_TYPE_ENUM eType,
    DECODER_OBJECT hDecoder
        );

static BOOLEAN bIterateSubscribedForEvent (
    DECODER_OBJECT hDecoder,
    CHANNEL_ART_SUB_UPDATE_STRUCT *psUpdate
        );

/*
    Functions used to manage output for decoders and
    for processing by those decoders
*/

static OSAL_OBJECT_HDL hCreateUpdateListForDecoder (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType,
    OSAL_OBJECT_HDL hAssociations,
    OSAL_LL_ITERATOR_HANDLER pfIteratorHandler
        );

static OSAL_OBJECT_HDL hChanCreateUpdateListForDecoder(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType
        );

static OSAL_OBJECT_HDL hAlbumCreateUpdateListForDecoder(
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    SMS_EVENT_ART_TYPE_ENUM eType
        );

static void vDestroyDecoderUpdateList (
    OSAL_OBJECT_HDL hUpdateList
        );

static N16 n16CompareEntriesForDecodersBySvcId (
    ART_UPDATE_STRUCT *psUpdate1,
    ART_UPDATE_STRUCT *psUpdate2
        );

static N16 n16CompareEntriesForDecodersByCatId (
    ART_UPDATE_STRUCT *psUpdate1,
    ART_UPDATE_STRUCT *psUpdate2
        );

static BOOLEAN bDestroyEntryForDecoder (
    ART_UPDATE_STRUCT *psUpdate,
    void *pvUnused
        );

static void vUpdateDecoderArt (
    CCACHE_OBJECT hCCache,
    CHANNEL_ART_SERVICE_OBJECT hArtService,
    SMS_EVENT_ART_TYPE_ENUM eType,
    OSAL_OBJECT_HDL hArtList
        );

static BOOLEAN bIterateChannelsForArt (
    CHANNEL_OBJECT hChannel,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        );

static BOOLEAN bIterateCategoriesForArt (
    CATEGORY_OBJECT hCategory,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        );

// The CCache doens't have a method to get all CDOs,
// so we just iterate over each channel and get at the
// CDOs that way.
static BOOLEAN bIterateCDOsForArt (
    CHANNEL_OBJECT hChannel,
    DECODER_ART_UPDATE_PROCESS_STRUCT *psArtUpdate
        );

static void vSetError (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj,
    DATASERVICE_ERROR_CODE_ENUM eErrorCode
        );

static size_t tImageFilenameLen (
    IMAGE_FORMAT_ENUM eFormat,
    UN16 un16ServiceSpecificImageType,
    const char *pacFilePath,
    size_t tLabelLength
        );

static void vAlbumAssocCleanup(
    CHANNEL_ART_SERVICE_OBJECT hArtService
        );

static BOOLEAN bDeleteExpiredTimedAssociations (
    ALBUM_ART_ASSOC_STRUCT *psAssoc,
    ALBUM_ART_CLEANUP_STRUCT *psCleanup
        );

static BOOLEAN bChanHandleErrorState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bChanHandleDisabledState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static void vChanCleanUp (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bFullCleanUp
        );

static BOOLEAN bChanHandleInitState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bChanHandleReadyState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bAlbumHandleErrorState ( 
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj 
        );

static BOOLEAN bAlbumHandleDisabledState ( 
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj 
        );

static void vAlbumCleanUp (
    CHANNEL_ART_OWNER_OBJECT_STRUCT *psArtOwner,
    BOOLEAN bFullCleanUp
        );

static BOOLEAN bAlbumHandleInitState (
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj
        );

static BOOLEAN bAlbumHandleReadyState ( 
    CHANNEL_ART_MGR_OBJECT_STRUCT *psObj 
        );

static BOOLEAN bAlbumCreatePersistDBTables (
    SQL_INTERFACE_OBJECT hPerSQLConnection,
    void *pvArg
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const CHANNEL_ART_INTERFACE_STRUCT CHANNEL_ART =
{
    /*.hStart = */hStart,
    /*.vStop = */(void (*) (CHANNEL_ART_SERVICE_OBJECT hChannelArtService))DATASERVICE_IMPL_vStopAutonomous,
    /*.eState = */(DATASERVICE_STATE_ENUM (*) (CHANNEL_ART_SERVICE_OBJECT hChannelArtService))DATASERVICE_IMPL_eState,
    /*.eErrorCode = */(DATASERVICE_ERROR_CODE_ENUM (*) (CHANNEL_ART_SERVICE_OBJECT hChannelArtService))DATASERVICE_IMPL_eErrorCode,
    /*.eUseArt = */eUseArt,
    /*.eImage = */eImage
};

// Manager interface for plugins
const CHANNEL_ART_MGR_INTERFACE_STRUCT GsArtMgrIntf =
{
    /*.eChanArtAssocBegin =*/eChanAssocBegin,
    /*.bChanArtAssocEnd =*/bChanAssocEnd,
    /*.bChanArtAssociationUpdate = */bChanAssociationUpdate,
    /*.bChanArtResetAssociations = */bChanResetAssociations,
    /*.bChanArtImageUpdate = */bChannelImageUpdate,
    /*.bAlbumArtImageUpdate = */bAlbumImageUpdate,
    /*.bAlbumArtAssociationUpdate = */bAlbumAssociationUpdate,
    /*.bAlbumArtAssocBegin = */ bAlbumAssocBegin,
    /*.bAlbumArtAssocEnd = */bAlbumAssocEnd
};

const DATASERVICE_STATE_HANDLERS_STRUCT GsChannelArtStateHandlers =
{
    /*.bHandleServiceReady = */(DATASERVICE_STATE_HANDLER)DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceStopped = */(DATASERVICE_STATE_HANDLER)DATASERVICE_IMPL_bGenericStateHandler,
    /*.bHandleServiceError = */(DATASERVICE_STATE_HANDLER)DATASERVICE_IMPL_bGenericStateHandler
};

const DATASERVICE_PRODUCT_DESCRIPTOR_STRUCT GasArtProducts[] =
{
    { DATA_PRODUCT_TYPE_ALBUM_ART,   CHANNEL_ART_PRODUCT_MASK_ALBUM },
    { DATA_PRODUCT_TYPE_CHANNEL_ART, CHANNEL_ART_PRODUCT_MASK_CHAN }
};

const ART_PRODUCT_STATE_HANDLERS GsChanStateHandlers =
{
    /*.bInitHandler = */bChanHandleInitState,
    /*.bReadyHandler = */bChanHandleReadyState,
    /*.bDisabledHandler = */bChanHandleDisabledState,
    /*.bErrorHandler = */bChanHandleErrorState
};

const ART_PRODUCT_STATE_HANDLERS GsAlbumStateHandlers =
{
    /*.bInitHandler = */bAlbumHandleInitState,
    /*.bReadyHandler = */bAlbumHandleReadyState,
    /*.bDisabledHandler = */bAlbumHandleDisabledState,
    /*.bErrorHandler = */bAlbumHandleErrorState
};

#endif    // _CHANNEL_ART_MGR_OBJ_H_
