/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CEL_H_
#define _CEL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "cid_obj.h"
#include "cme.h"
#include "cel.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CEML_OBJECT_NAME "CEML"

  /**************/
 /** TYPEDEFS **/
/**************/

// Registered Content structure used to maintain the
// registered content list and list's enable/disable flag as well
// as the currently iterated registered content.
typedef struct registered_content_struct
{
    // Caller's compare/sort handler. This is provided by the caller
    // to keep registered content in a sorted list according to how the
    // caller desires.
    CONTENT_COMPARE_HANDLER n16CompareHandler;

    // A linked list of registered content.
    OSAL_OBJECT_HDL hContentList;

    // Currently iterated registered content. Use to determine which
    // content entry is being iterated when using eReplace, eRemove, eEnable
    // and/or eDisable.
    OSAL_LINKED_LIST_ENTRY hCurrentContent;

    // Flag to indicate if events are blocked or unblocked for this list
    BOOLEAN bBlocked;

} REGISTERED_CONTENT_STRUCT;

// Active content structure used to maintain the
// active content list and current active event being iterated.
typedef struct active_events_struct
{
    // A linked list of active events.
    OSAL_OBJECT_HDL hEventList;

    // Caller's callback and argument for active events
    ACTIVE_EVENT_CALLBACK vEventCallback;
    void *pvEventCallbackArg;

    // Currently iterated active event. Used to determine which event
    // entry is being iterated when using eAcknowledge.
    OSAL_LINKED_LIST_ENTRY hCurrentEvent;

} ACTIVE_EVENTS_STRUCT;

//  The main Content Event Monitoring List (CEML) structure used to keep
//  both registered and active lists.
typedef struct ceml_object_struct
{
    // The DECODER this event list is associated with
    DECODER_OBJECT hDecoder;

    // The registrered content list
    REGISTERED_CONTENT_STRUCT sRegistered;

    // The active list object structure
    ACTIVE_EVENTS_STRUCT sActive;

    // A linked list of ended events.
    OSAL_OBJECT_HDL hEndedEventList;

} CEML_OBJECT_STRUCT;

// One of these exist for each content entry
typedef struct registered_content_entry_struct
{
    // A registered content entry handle which points to itself
    OSAL_LINKED_LIST_ENTRY hEntry;

    // CEML object associated with this content
    CEML_OBJECT_STRUCT *psObj;

    // CME content entry handle which references this content. Returned
    // when content is registered with the CME.
    CME_REGISTERED_ENTRY hCMEContent;

    // Caller's content argument associated with this registered content.
    // Returned to the caller when an event is active or content iterated.
    void *pvContentArg;

} REGISTERED_CONTENT_ENTRY_STRUCT;

// A structure used to iterate the registered content list
typedef struct registered_content_list_iterator_struct
{
    // The content event monitoring list object performing the iteration
    CEML_OBJECT_STRUCT *psObj;

    // Caller's content iterator and argument
    CONTENT_ITERATOR_CALLBACK bContentIterator;
    void *pvContentIteratorArg;

    // A pointer to the current entry being iterated.
    OSAL_LINKED_LIST_ENTRY *phCurrentEntry;

    // Continue iteration flag
    BOOLEAN bContinue;

} REGISTERED_CONTENT_LIST_ITERATOR_STRUCT;

// A structure to search the registered content list
typedef struct registered_content_list_search_iterator_struct
{
    // Event found indicator
    BOOLEAN bFound;

    // CID to look for
    CID_OBJECT hId;

} REGISTERED_CONTENT_LIST_SEARCH_ITERATOR_STRUCT;

// One of these exist for each active event
typedef struct active_event_entry_struct
{
    // An event entry which points to itself.
    OSAL_LINKED_LIST_ENTRY hEntry;

    // Event information (provided by the CME) when an event occurs.

    // The CHANNEL object the event occured on
    CHANNEL_OBJECT hChannel;

    // The event flags to indicate if initial, end and/or final
    UN32 un32Flags;

    // The event callback argument registered with this entry
    void *pvEventCallbackArg;

    // Caller's content specific argument registered with this content.
    void *pvContentArg;

} ACTIVE_EVENT_ENTRY_STRUCT;

typedef struct active_event_list_iterator_struct
{
    // The event list object performing the iteration
    CEML_OBJECT_STRUCT *psObj;

    // Caller's event iterator and argument
    ACTIVE_EVENT_ITERATOR_CALLBACK bEventIterator;
    void *pvEventIteratorArg;

    // A pointer to the current entry being iterated.
    OSAL_LINKED_LIST_ENTRY *phCurrentEntry;

} ACTIVE_EVENT_LIST_ITERATOR_STRUCT;

//  Ended event structure 
typedef struct ended_event_struct
{
    // the channel the registered content was found on
    CHANNEL_ID tChannelId;

    // artist text
    STRING_OBJECT hArtist;

    // title text
    STRING_OBJECT hTitle;

} ENDED_EVENT_ENTRY_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static CEML_OBJECT hCreateList (
    DECODER_OBJECT hDecoder,
    CONTENT_COMPARE_HANDLER n16CompareHandler,
    ACTIVE_EVENT_CALLBACK vEventCallback,
    void *pvEventCallbackArg
        );

static void vDestroyList (
    CEML_OBJECT hCEML
        );

static size_t tContentListItems (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eIterateContentList (
    CEML_OBJECT hCEML,
    CONTENT_ITERATOR_CALLBACK bContentIterator,
    void *pvContentIteratorArg
        );

static SMSAPI_RETURN_CODE_ENUM eIterateActiveList (
    CEML_OBJECT hCEML,
    ACTIVE_EVENT_ITERATOR_CALLBACK bEventIterator,
    void *pvEventIteratorArg
        );

static SMSAPI_RETURN_CODE_ENUM eRegister (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options
        );

static CID_OBJECT hCID (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eReplace (
    CEML_OBJECT hCEML,
    UN32 un32Options
        );

static SMSAPI_RETURN_CODE_ENUM eRemove(
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveAll(
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eEnable(
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eDisable (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eContentEnabled (
    CEML_OBJECT hCEML,
    BOOLEAN *pbEnabled
        );

static SMSAPI_RETURN_CODE_ENUM eAcknowledge (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eBlockAllEvents (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eUnBlockAllEvents (
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eEventsBlocked (
    CEML_OBJECT hCEML,
    BOOLEAN *pbBlocked
        );

static SMSAPI_RETURN_CODE_ENUM eSearchForCurrent(
    CEML_OBJECT hCEML
        );

static SMSAPI_RETURN_CODE_ENUM eGetEndedEventAttributes(
    CEML_OBJECT hCEML,
    CHANNEL_ID *ptChannelId,
    STRING_OBJECT *phArtist,
    STRING_OBJECT *phTitle
        );

/* Object Private Prototypes */

static BOOLEAN bRegisteredContentIterator (
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry,
    void *pvIteratorCallbackArg
        );

static BOOLEAN bContentIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void *pvIteratorCallbackArg
        );

static BOOLEAN bContentRemoveIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void *pvIteratorCallbackArg
        );

static BOOLEAN bContentArgIterator (
    DECODER_OBJECT hDecoder,
    CONTENT_REGISTRATION_STRUCT *psContent,
    void **ppvContentArg
        );

static void vActiveEventCallback (
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    UN32 un32Flags,
    void *pvEventCallbackArg
        );

static BOOLEAN bActiveEventIterator (
    ACTIVE_EVENT_ENTRY_STRUCT *psEvent,
    void *pvIteratorArg
        );

static void vDestroyContent (
    REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry
        );

static void vDestroyEvent (
    ACTIVE_EVENT_ENTRY_STRUCT *psEvent
        );

static void vDestroyEndedEvent (
    ENDED_EVENT_ENTRY_STRUCT *psEvent
        );

static BOOLEAN bIdRegistered (
    CEML_OBJECT hCEML,
    CID_OBJECT hId
        );

static BOOLEAN bEqualCIDs (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvContentIteratorArg
        );

static N16 n16ContentCompareHandler (
    const REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry1,
    const REGISTERED_CONTENT_ENTRY_STRUCT *psContentEntry2
        );

static BOOLEAN bRemoveAllContentIterator (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global instance of an interface for this object
const CEML_OBJECT_INTERFACE_STRUCT CEML =
{
    /*.hCreateList = */hCreateList,
    /*.vDestroyList = */vDestroyList,
    /*.tContentListItems = */tContentListItems,
    /*.eIterateContentList = */eIterateContentList,
    /*.eIterateActiveList = */eIterateActiveList,
    /*.eRegister = */eRegister,
    /*.hCID = */hCID,
    /*.eReplace = */eReplace,
    /*.eRemove = */eRemove,
    /*.eRemoveAll = */eRemoveAll,
    /*.eEnable = */eEnable,
    /*.eDisable = */eDisable,
    /*.eContentEnabled = */eContentEnabled,
    /*.eAcknowledge = */eAcknowledge,
    /*.eBlockAllEvents = */eBlockAllEvents,
    /*.eUnBlockAllEvents = */eUnBlockAllEvents,
    /*.eEventsBlocked = */eEventsBlocked,
    /*.eSearchForCurrent = */eSearchForCurrent,
    /*.eGetEndedEventAttributes = */eGetEndedEventAttributes
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _CEL_H_
