/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

/*********************************/
/** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CATEGORY_OBJ_H_
#define _CATEGORY_OBJ_H_

/**************/
/** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_update.h"
#include "dataservice_mgr_obj.h"

/***************/
/** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CATEGORY_OBJECT_NAME "CATEGORY"

/* Serialization tag names */
#define CATEGORY_OBJECT_SHORT_NAME_TAG "ShortName"
#define CATEGORY_OBJECT_MED_NAME_TAG "MedName"
#define CATEGORY_OBJECT_LONG_NAME_TAG "LongName"
#define CATEGORY_OBJECT_TYPE_TAG "Type"
#define CATEGORY_OBJECT_UNIQUE_TAG "Unique"
#define CATEGORY_OBJECT_REPLACE_TAG "Replace"
#define CATEGORY_OBJECT_SERVICE_ID_TAG "ServiceId"
#define CATEGORY_OBJECT_CHANNELS_TAG "Channels"

/**************/
/** TYPEDEFS **/
/**************/
/* Private object elements */

typedef struct category_get_handle_by_offset_struct
{
    CHANNEL_OBJECT hChannel;	// This will return the channel handle when the item is found
    N16 n16OffsetCntr;		// This will be modified inside the iterator

} CATEGORY_GET_HANDLE_BY_OFFSET_STRUCT;

typedef struct category_get_index_by_id_struct
{
    CHANNEL_ID tChannelId;
    SERVICE_ID tServiceId;
    N16 n16Index;
    UN32 un32CategorySize;

} CATEGORY_GET_INDEX_BY_ID_STRUCT;

typedef struct category_del_chan_by_id_struct
{
    CHANNEL_ID tChannelId;
} CATEGORY_DEL_CHAN_BY_ID_STRUCT;

typedef struct category_del_chan_by_index_struct
{
    UN16 un16Index;
} CATEGORY_DEL_CHAN_BY_INDEX_STRUCT;

typedef struct category_chan_occurrence_struct
{
    // The ID to look for
    CHANNEL_ID tChannelId;

    // The number of times this id is encountered
    N16 n16Occurrence;

    // The current iteration index
    N16 n16Index;

    // The index limit (stop condition)
    N16 n16Limit;

} CATEGORY_CHAN_OCCURRENCE_STRUCT;

typedef struct category_iteration_ctrl_struct
{
    // The current entry being iterated
    OSAL_LINKED_LIST_ENTRY hCurrentEntry;

    // A flag indicating if the current
    // channel has been removed via
    // CATEGORY.eRemoveChannel
    BOOLEAN bChannelRemoved;

    // A counter indicating if channels
    // have been added before
    // the current entry via
    // CATEGORY.eInsertBeforeChannel
    UN8 un8ChannelsAddedBefore;

} CATEGORY_ITERATION_CTRL_STRUCT;

typedef struct category_iterator_struct
{
    CATEGORY_OBJECT_ITERATOR bIterator;
    CATEGORY_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude;
    void *pvArg;

} CATEGORY_ITERATOR_STRUCT;

typedef struct category_serialization_struct
{
    // Channels Parent Tag
    TAG_OBJECT hChannelsTag;

    // Next Sibling Tag
    TAG_OBJECT hNextSiblingTag;

    // Channels Array
    CHANNEL_OBJECT *phChannel;

    // Channel Cache Handle
    CCACHE_OBJECT hCCache;

    // Operation Result
    BOOLEAN bSuccess;

    // Commit Request
    BOOLEAN bCommitMe;

    // Iteration Index
    size_t tIndex;

    // Number of child channel tags
    UN32 un32Channels;

} CATEGORY_SERIALIZATION_STRUCT;

typedef struct category_object_struct
{
    DECODER_OBJECT hDecoder;

    CATEGORY_TYPE_ENUM eCategoryType;

    // Category Id
    CATEGORY_ID tId;

    // User Id
    USER_ID tUserId;

    // String object this category belongs to
    STRING_OBJECT hShortName;
    STRING_OBJECT hMediumName;
    STRING_OBJECT hLongName;

    // Channel list of channels associated with this category
    OSAL_OBJECT_HDL hChannelList;

    // Structure which groups attributes
    // used during iteration
    CATEGORY_ITERATION_CTRL_STRUCT sIterationCtrl;

    // Category Update Control Structure
    SMSU_EVENT_STRUCT sEvent;

    // Currently filtered notifications for this category
    SMSAPI_EVENT_MASK tFilter;

    // Channel Art Handles
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService;
    CHANNEL_ART_OBJECT hArt;

    // Unique Items
    BOOLEAN bUniqueItems;
    BOOLEAN bReplace;

} CATEGORY_OBJECT_STRUCT;

typedef struct category_destroy_callback_shim_struct
{
    // Original callback
    SMSAPI_OBJECT_EVENT_CALLBACK vEventCallback;

    // Original event callback argument
    void *pvEventCallbackArg;

} CATEGORY_DESTROY_CALLBACK_SHIM_STRUCT;


/************/
/** MACROS **/
/************/

/****************/
/** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static CATEGORY_ID tGetCategoryId (
    CATEGORY_OBJECT hCategory
        );

static USER_ID tGetUserId(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

static SMSAPI_RETURN_CODE_ENUM eSetUserId(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    USER_ID tUserId
        );

static size_t tSize(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

static CATEGORY_EVENT_MASK tEventMask (
    CATEGORY_OBJECT hCategory
        );

static STRING_OBJECT hShortName (
    CATEGORY_OBJECT hCategory
        );

static STRING_OBJECT hMediumName (
    CATEGORY_OBJECT hCategory
        );

static STRING_OBJECT hLongName (
    CATEGORY_OBJECT hCategory
        );

static SMSAPI_RETURN_CODE_ENUM eRename(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
	const char *pacNewLongName,
	const char *pacNewShortName
		);

static CATEGORY_TYPE_ENUM eType (
    CATEGORY_OBJECT hCategory
        );

static CHANNEL_ART_OBJECT hArt (
    CATEGORY_OBJECT hCategory
        );

static CATEGORY_ID tCreateUserCategory(
    DECODER_OBJECT hDecoder,
    const char *pacLongName,
    const char *pacShortName,
    CATEGORY_CHANNEL_INDEX tInitialNumChannels
        );

static void vDestroyDerivedCategory (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

static SMSAPI_RETURN_CODE_ENUM eIterate(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CATEGORY_ITERATION_HANDLER n16IterateFxn,
    void *pvIterateArg
        );

static SMSAPI_RETURN_CODE_ENUM eInsertNewChannel(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eInsertAfterChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eInsertBeforeChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eReplaceChannel(
    CATEGORY_OBJECT hCategory,
    CHANNEL_ID tChannelId
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveChannel(
    CATEGORY_OBJECT hCategory
        );

static SMSAPI_RETURN_CODE_ENUM eSort(
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId,
    CATEGORY_SORT_HANDLER n16SortFxn
        );

static SMSAPI_RETURN_CODE_ENUM eNotifyOnChange (
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT_EVENT_CALLBACK vNotifierCallback,
    void *pvNotifierCallbackArg
        );

static SMSAPI_RETURN_CODE_ENUM eIterateAll (
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT_ITERATOR bIterator,
    CATEGORY_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude,
    void *pvArg
        );

/* Object Private Prototypes */

static N16 n16GetHdlByOffsetIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static N16 n16GetIndexByChanIdIterateFxn (
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static N16 n16GetIndexByServiceIdIterateFxn(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

static CATEGORY_OBJECT hGetHandle (
    DECODER_OBJECT hDecoder,
    CATEGORY_ID tCategoryId
        );

static BOOLEAN bRemoveCategoryFromChannel (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bNotifyCategoryUpdated (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bDetermineChanOccurrence (
    void *pvObj1,
    void *pvObj2
        );

static CATEGORY_OBJECT_STRUCT *psCreateCategory (
    CATEGORY_TYPE_ENUM eCategoryType,
    DECODER_OBJECT hDecoder,
    SMS_OBJECT hParent,
    CATEGORY_ID tId,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName,
    BOOLEAN bUniqueItems,
    BOOLEAN bReplace
        );

static BOOLEAN bProcessCategoryNames(
    CATEGORY_TYPE_ENUM eType,
    const char **ppacLongName,
    const char **ppacMediumName,
    const char **ppacShortName
        );

static BOOLEAN bUpdateNames(
    CATEGORY_OBJECT_STRUCT *psObj,
    const char *pacLongName,
    const char *pacMediumName,
    const char *pacShortName
        );

static void vNotifyRegisteredAndProtectIterator (
    CATEGORY_OBJECT_STRUCT *psObj,
    SMSAPI_EVENT_MASK tEventMask
        );

static SMSAPI_RETURN_CODE_ENUM eRenameLocal(
    CATEGORY_OBJECT_STRUCT *psObj,
    const char *pacNewLongName,
    const char *pacNewMedName,
    const char *pacNewShortName
        );

static SMSAPI_RETURN_CODE_ENUM eInsertAfterChannelLocal (
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        );

static SMSAPI_RETURN_CODE_ENUM eInsertBeforeChannelLocal (
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        );

static SMSAPI_RETURN_CODE_ENUM eReplaceChannelLocal (
    CATEGORY_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel
        );

static BOOLEAN bAlwaysInclude (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        );

static BOOLEAN bIteratorShim (
    CATEGORY_OBJECT hCategory,
    void *pvArg
        );

static N16 n16ChannelCompare(void *pvObj1, void *pvObj2);

static void vChannelRemappedEventCallback (
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

static BOOLEAN bSerializationIterator (
    void *pvData,
    void *pvArg
        );

static BOOLEAN bRestoreIterator (
    TAG_OBJECT hTag,
    void *pvArg
        );

static SMSAPI_RETURN_CODE_ENUM eSortDirect(
    CATEGORY_OBJECT_STRUCT *psObj,
    CATEGORY_SORT_HANDLER n16SortFxn
        );

static BOOLEAN bBlockNotifications(
    CATEGORY_OBJECT hCategory
        );

static void vReleaseNotifications(
    CATEGORY_OBJECT hCategory
        );

/***************/
/** VARIABLES **/
/***************/

// Global (re-usable) instance of an interface for this object
const CATEGORY_OBJECT_INTERFACE_STRUCT CATEGORY =
{
    // Public
    /*.tGetCategoryId = */tGetCategoryId,
    /*.tGetUserId = */tGetUserId,
    /*.eSetUserId = */eSetUserId,
    /*.tSize = */tSize,
    /*.tEventMask = */tEventMask,
    /*.hShortName = */hShortName,
    /*.hMediumName = */hMediumName,
    /*.hLongName = */hLongName,
    /*.eRename = */eRename,
    /*.eType = */eType,
    /*.hArt = */hArt,
    /*.tCreate = */tCreateUserCategory,
    /*.vDestroy = */vDestroyDerivedCategory,
    /*.eIterate = */eIterate,
    /*.eInsertNewChannel = */eInsertNewChannel,
    /*.eInsertAfterChannel = */eInsertAfterChannel,
    /*.eInsertBeforeChannel = */eInsertBeforeChannel,
    /*.eReplaceChannel = */eReplaceChannel,
    /*.eRemoveChannel = */eRemoveChannel,
    /*.eSort = */eSort,
    /*.eNotifyOnChange = */eNotifyOnChange,
    /*.eIterateAll = */eIterateAll
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif	// _CATEGORY_OBJ_H_
