/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _CAS_H
#define _CAS_H

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "cas.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

/* Object name prefix for objects */
#define CAL_OBJECT_NAME "CAL"

// this is only used to check for undefined options. It contains mutually
// exclusive options, therefore it should never be used as the actual auto
// add options.
#define CAL_AUTO_ADD_OPTION_ALL ( \
    CAL_AUTO_ADD_OPTION_OFF | \
    CAL_AUTO_ADD_OPTION_UNIQUE | \
    CAL_AUTO_ADD_OPTION_IF_TUNED | \
    CAL_AUTO_ADD_OPTION_IF_NOT_TUNED |\
    CAL_AUTO_ADD_OPTION_TOP |\
    CAL_AUTO_ADD_OPTION_BOTTOM\
        )

  /**************/
 /** TYPEDEFS **/
/**************/

// A structure used to search the ceml list
typedef struct cal_ceml_iterator_struct
{
    // content found indicator
    BOOLEAN bFound;

    // CID to look for
    CID_OBJECT hId;

} CAL_CEML_ITERATOR_STRUCT;

// A structure used to search the ceml content list
typedef struct cal_to_ceml_iterator_struct
{
    void *pvCALIteratorArg;

    CAL_CONTENT_ITERATOR_CALLBACK bCALCallback;

} CAL_TO_CEML_ITERATOR_STRUCT;

//  The main Content Alert List (CAL) structure used to keep
//  registered content.
typedef struct cal_object_struct
{
    // the decoder this CAL is associated with
    DECODER_OBJECT hDecoder;

    // The CEML this alert list is associated with
    CEML_OBJECT hCEML;

    // The callback registered to receive ALERTS
    CAL_CALLBACK vCALCallback;
    // The app provided argument to be provided to the CALCallback
    void *pvCALCallbackArg;

    // attached category information
    CATEGORY_ID tCategoryId;
    // the options used to add to the attached category
    CAL_AUTO_ADD_OPTIONS tCatAddOptions;
    // flag indicating if the attached category was created
    // by the cas (TRUE) or not (FALSE)
    BOOLEAN bCreatedCategory;

    // an alert that can be reused by the CAL
    CAL_ALERT_OBJECT hAlert;

    // indication of whether the CAL is enabled or not
    BOOLEAN bEnabled;

    // indication if CAL should auto scknowledge CEML
    BOOLEAN bAutoAck;

    // generate alerts on tuned channel
    BOOLEAN bAlertOnTunedChannel;

    // generate alerts if content has finished playing
    BOOLEAN bAlertOnFinishedContent;

} CAL_OBJECT_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static CAL_OBJECT hCreateList (
    DECODER_OBJECT hDecoder,
    CAL_CALLBACK vCALCallback,
    void *pvCALCallbackArg
        );

static void vDestroyList (
    CAL_OBJECT hCAL
    	);

static SMSAPI_RETURN_CODE_ENUM eRegister (
    CAL_OBJECT hCAL,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    STRING_OBJECT hArtistText,
    STRING_OBJECT hTitleText,
    CAL_CONTENT_OBJECT *phCalContent
        );

static SMSAPI_RETURN_CODE_ENUM eIterate (
    CAL_OBJECT hCAL,
    CAL_CONTENT_ITERATOR_CALLBACK bContentIterator,
    void *pvContentIteratorArg
        );

static BOOLEAN bEnabled (
    CAL_OBJECT hCAL
        );

static SMSAPI_RETURN_CODE_ENUM eEnable (
    CAL_OBJECT hCAL
        );

static SMSAPI_RETURN_CODE_ENUM eDisable (
    CAL_OBJECT hCAL
        );

static 	size_t tNumItems (
    CAL_OBJECT hCAL
        );

static BOOLEAN bExists (
    CAL_OBJECT hCAL,
    CID_OBJECT hCID
        );

static SMSAPI_RETURN_CODE_ENUM eRemoveAll (
    CAL_OBJECT hCAL
        );

static CATEGORY_ID tCategoryId (
    CAL_OBJECT hCAL
        );

static SMSAPI_RETURN_CODE_ENUM eAttachCategory (
    CAL_OBJECT hCAL,
    CATEGORY_ID tCategoryId,
    CAL_AUTO_ADD_OPTIONS tCatAddOptions,
    ...
        );

static SMSAPI_RETURN_CODE_ENUM eDetachCategory (
    CAL_OBJECT hCAL
        );

/* Object Private Prototypes */

static CAL_ALERT_OBJECT hGetAlert(
    CAL_OBJECT_STRUCT *psObj,
    UN32 un32Flags,
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent
        );

static void vCALEventCallback (
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventCallbackArg
        );

static BOOLEAN bEqualCIDs (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        );

static BOOLEAN bAutoAdd(
    CAL_OBJECT_STRUCT *psObj,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags
        );

static BOOLEAN bCallCallback(
    CAL_OBJECT_STRUCT *psObj,
    CAL_CONTENT_OBJECT hContent,
    UN32 un32Flags
        );

static void vProcessAlert(
    CAL_OBJECT_STRUCT *psObj,
    CHANNEL_OBJECT hChannel,
    CAL_CONTENT_OBJECT hContent,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags
        );

static BOOLEAN bCALCEMLIterator (
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvEventIteratorArg
        );

static BOOLEAN bActiveEventIterator(
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventIteratorArg
        );

  /***************/
 /** VARIABLES **/
/***************/

// Global instance of an interface for this object
const CONTENT_ALERT_SERVICE_INTERFACE_STRUCT CAS =
{
    /*.hCreateList = */hCreateList,
    /*.vDestroyList = */vDestroyList
};

// Global instance of an interface for this object
const CONTENT_ALERT_LIST_INTERFACE_STRUCT CAL =
{
    /*.eRegister = */eRegister,
    /*.eIterate = */eIterate,
    /*.bEnabled = */bEnabled,
    /*.eEnable = */eEnable,
    /*.eDisable = */eDisable,
    /*.tNumItems = */tNumItems,
    /*.bExists = */bExists,
    /*.eRemoveAll = */eRemoveAll,
    /*.tCategoryId = */tCategoryId,
    /*.eAttachCategory = */eAttachCategory,
    /*.eDetachCategory = */eDetachCategory
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _CAS_H
