/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the PVN 1 specific alerts data service
 *  for the Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include <string.h>
#include <stdio.h>

#include "sms_api.h"
#include "sms_obj.h"
#include "sms.h"
#include "baudot.h"
#include "string_obj.h"
#include "location_obj.h"
#include "rfd_interface_obj.h"
#include "dataservice_mgr_obj.h"
#include "ws_alerts_db_constants.h"
#include "ws_alerts_interface.h"
#include "_ws_alerts_pvn1.h"
#include "sms_api_debug.h"
#include "db_util.h"
#include "ds_util.h"

static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
*
*   tMinimumOTABufferByteSize
*
*   Calculate the minimum number of bytes needed for this service
*   based upon how it's going to be used
*
*****************************************************************************/
static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        )
{
    // We'll always need this as a starting point
    size_t tByteSize = WS_ALERTS1_HIGH_RATE_PAYLOAD_MAX_SIZE
        + WS_ALERTS1_NORM_RATE_PAYLOAD_MAX_SIZE;

    if (TRUE == bDBUpdatesEnabled)
    {
        tByteSize += WS_ALERTS1_RFD_META_PAYLOAD_MAX_SIZE
            + WS_ALERTS1_RFD_BLOCK_PAYLOAD_MAX_SIZE;
    }

    return tByteSize;
}

/*****************************************************************************
 *
 *   hInit
 *
 *   Creates and initializes the PVN0 alerts interface object.
 *
 *   Inputs:
 *       hAlertsService - A valid handle to a ALERTS_SERVICE_OBJECT representing
 *           the central alerts service manager.
 *
 *   Output:
 *       An object handle representing this instantiation of the
 *       alerts1 interface (representing high-band support)
 *
 *****************************************************************************/
static WS_ALERTS_INTERFACE_OBJECT hInit(
    WS_ALERTS_SERVICE_OBJECT hAlertsService,
    SMS_OBJECT hParent,
    SQL_INTERFACE_OBJECT hSQLRefConnection,
    RFD_UPDATE_VERSION tCurrentVersion,
    BOOLEAN bDBUpdatesEnabled
        )
{
    BOOLEAN bOwner = FALSE;
    WS_ALERTS_INTERFACE_OBJECT hResult = WS_ALERTS_INTERFACE_INVALID_OBJECT;
    WS_ALERTS1_OBJECT_STRUCT *psObj = (WS_ALERTS1_OBJECT_STRUCT*)NULL;
    BOOLEAN bTemp = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bOk;

    do
    {
        // Verify we own service handle
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (WS_ALERTS1_OBJECT_STRUCT*)SMSO_hCreate(WS_ALERTS1_OBJECT_NAME,
            sizeof(*psObj), hParent, FALSE);

        if (psObj == NULL)
        {
            break;
        }

        psObj->hWSAlertsService = hAlertsService;

        psObj->bProcessCurAU = TRUE;

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hStateMsgHighList,
            WS_ALERTS1_OBJECT_NAME":StateMsgHighList",
            (OSAL_LL_COMPARE_HANDLER)n16WSAlertsFindByStateId,
            OSAL_LL_OPTION_LINEAR );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create list StateMsgList" );
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->hStateMsgNormalList,
            WS_ALERTS1_OBJECT_NAME":StateMsgNormalList",
            (OSAL_LL_COMPARE_HANDLER)n16WSAlertsFindByStateId,
            OSAL_LL_OPTION_LINEAR );

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create list StateMsgList" );
            break;
        }

        // Request access to ISO 3309 CRC32
        eReturnCode = OSAL.eGetCRC(
            &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! destroy object
            break;
        }

        // Create our file reader buffer
        bOk = DATASERVICE_MGR_bCreateFileBuffer(
            &psObj->hBlockPool,
            &psObj->hBuffer,
            WS_ALERTS1_BUFFER_SIZE );

        if (bOk == FALSE)
        {
            // Error! destroy object
            break;
        }

        psObj->eCurrentLanguage = LOCALIZATION.eGetLanguage();

        bOk = bBuildServiceFilePath(psObj);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Can't create Service File Path" );
            break;
        }

        bOk = bCleanDbDir( psObj );

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Can't clean up Service File Directory" );
        }

        if (bDBUpdatesEnabled == TRUE)
        {
            bTemp = bInitRFD(psObj, tCurrentVersion);
            if (bTemp == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to init RFD" );
                break;
            }
        }
        hResult = (WS_ALERTS_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    if (hResult == WS_ALERTS_INTERFACE_INVALID_OBJECT)
    {
        vUnInit(psObj);
    }

    return hResult;
}

/*****************************************************************************
 *
 *   vUnInit
 *
 *   Uninitializes and destroys the PVN 1 alerts interface object.
 *   This API must only be called when already in the context of the
 *   alerts manager.
 *
 *   Inputs:
 *       hInterface - The interface object handle created with a
 *           previous call to hInit.
 *
 *****************************************************************************/
static void vUnInit (
    WS_ALERTS_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        WS_ALERTS1_OBJECT_STRUCT *psObj =
                        (WS_ALERTS1_OBJECT_STRUCT *)hInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Disconnect RFD interface if it is connected
        if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
        {
            RFD_INTERFACE_vDisconnect(psObj->hRFD);
            psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;
        }

        // Destroy the file buffer
        DATASERVICE_MGR_vDestroyFileBuffer(psObj->hBlockPool, psObj->hBuffer);
        psObj->hBlockPool = OSAL_INVALID_OBJECT_HDL;
        psObj->hBuffer = OSAL_INVALID_BUFFER_HDL;

        // Destroy the CRC object if it exists
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            // Release our hold of the CRC computation handle
            eReturnCode = OSAL.eReleaseCRC( psObj->hCRC );

            // Only finish clean up if that succeeded.  This allows
            // us to more easily track this kind of issue
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
            }

        }

        vReleaseStateMsgList(psObj, &psObj->hStateMsgHighList);
        vReleaseStateMsgList(psObj, &psObj->hStateMsgNormalList);

        if (psObj->pacServiceFilePath != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->pacServiceFilePath);
            psObj->pacServiceFilePath = NULL;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
 *
 *   bProcessMessage
 *
 *   The entry point for the interface's OTA message processing.  All messages
 *   received by the Alerts manager come through this API.
 *
 *   Inputs:
 *       hInterface - The interface object handle created with a
 *           previous call to hInit.
 *       hPayload - A valid OSAL_BUFFER_HDL containing the OTA alerts data
 *
 *   Output:
 *       TRUE on success, FALSE on error
 *
 *****************************************************************************/
static BOOLEAN bProcessMessage(
    WS_ALERTS_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bResult = FALSE, bOwner = FALSE;
    WS_ALERTS1_OBJECT_STRUCT *psObj = (WS_ALERTS1_OBJECT_STRUCT*)hInterface;
    WS_ALERTS1_MSG_CAROUSEL_ENUM eMessageType = WS_ALERTS1_INVALID_MSG;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    SMS_LANGUAGE_ENUM eLang = SMS_INVALID_LANGUAGE;
    size_t tBitsRead;

    do
    {
        if ((hInterface == WS_ALERTS_INTERFACE_INVALID_OBJECT) ||
            (phPayload == NULL)
        )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Invalid input parameters");
            break;
        }

        if (*phPayload == OSAL_INVALID_BUFFER_HDL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Invalid payload");
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        eMessageType = eGetMessageType(psObj, *phPayload);

        switch (eMessageType)
        {
            case WS_ALERTS1_HIGH_RATE_MSG:
            case WS_ALERTS1_NORMAL_RATE_MSG:
            {
                // To see if LANG is current
                tBitsRead = OSAL.tBufferPeekBits(*phPayload, (void*)&eLang, 0,
                    WS_ALERTS1_LANGUAGE_CODE_BITLEN, WS_ALERTS1_LANGUAGE_OFFSET);
                if (tBitsRead != WS_ALERTS1_LANGUAGE_CODE_BITLEN)
                {
                    puts(WS_ALERTS1_OBJECT_NAME" LANG read Failed\n");
                    break;
                }
                if (eLang != psObj->eCurrentLanguage)
                {
                    puts(WS_ALERTS1_OBJECT_NAME" Inappropriate language\n");
                    break;
                }
                // Validate the message.
                bResult = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload,
                    (OSAL_CRC_RESULT*)&psObj->un32CurrentMsgCRC);
                if (bResult == FALSE)
                {
                    puts(WS_ALERTS1_OBJECT_NAME" Packet Invalid\n");
                    break;
                }
                // First, we need to parse the message header
                bResult = bParseMessageHeader(psObj, *phPayload, eMessageType);
                if (bResult != TRUE)
                {
                    printf(WS_ALERTS1_OBJECT_NAME": skip AU\n");
                    break;
                }

                eReturnCode = eAddAccessUnitToStorage(psObj);

                if ((eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) &&
                    (eReturnCode != SMSAPI_RETURN_CODE_DUPLICATE_CONTENT))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME
                        ": failed to add AU to storage");
                    break;
                }
                bResult = TRUE;
            }
            break;

            case WS_ALERTS1_DB_UPDATE_MSG:
            case WS_ALERTS1_METADATA_UPDATE_MSG:
            {
                if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
                {
                    BOOLEAN bValid = FALSE;

                    // Pass the RFD AU to the RFD Interface
                    bValid = RFD_INTERFACE_bProcessPayload(psObj->hRFD, *phPayload);

                    if (bValid != TRUE)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WS_ALERTS1_OBJECT_NAME": RFD_INTERFACE_bProcessPayload() failed");
                        bResult = FALSE;
                        break;
                    }
                    *phPayload = OSAL_INVALID_BUFFER_HDL;
                    bResult = TRUE;
                }
                else
                {
                    // skip if RFD interface object was not created
                    bResult = TRUE;
                }
            }
            break;

            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": Unknown message type: %u", eMessageType);
                bResult = FALSE;
            }
        }


    } while (FALSE);

    return bResult;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
 *****************************************************************************/

/****************************************************************************
*
*   bInitRFD
*
*****************************************************************************/
static BOOLEAN bInitRFD(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    RFD_UPDATE_VERSION tCurrentVersion
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        RFD_OPTIONAL_CALLBACKS_STRUCT sCallbacks;
        WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl =
            (WS_ALERTS1_RFD_CTRL_STRUCT *)NULL;

        OSAL.bMemSet(&sCallbacks, 0, sizeof(sCallbacks));

        // Populate callbacks
        sCallbacks.vReportExpiredDB = vNotifyDBExpired;
        sCallbacks.eUpdateNeeded = eRFDUpdateNeeded;

        // Create the RFD processor object
        psRFDCtrl = psCreateRFDCtrl(psObj);
        if (psRFDCtrl == NULL)
        {
            break;
        }

        // Remember what version we're on
        psRFDCtrl->tCurrentVersion = tCurrentVersion;

        // Connect to RFD now
        psObj->hRFD = RFD_INTERFACE_hConnect (
            WS_ALERTS1_RFD_CLIENT_ID,
            tCurrentVersion,
            WS_ALERTS1_MAX_VERSION_BITLEN,
            eRFDFileProcessor,
            &sCallbacks,
            psRFDCtrl);

        if (psObj->hRFD == RFD_INTERFACE_INVALID_OBJECT)
        {
            SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/****************************************************************************
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl != NULL)
    {
        vUninitRFDCtrl(psRFDCtrl);
        SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);
    }

    return;
}

/*****************************************************************************
 *
 *   eParseMessageHeader
 *
 *****************************************************************************/
static BOOLEAN bParseMessageHeader (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    WS_ALERTS1_MSG_CAROUSEL_ENUM eMessageType
        )
{
    size_t tBitsRead;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bResult = FALSE;

    do
    {
        // We can now get rid of the CRC
        bResult = DS_UTIL_bCutCRC(hPayload);
        if (bResult == FALSE)
        {
            break;
        }

        // We can also now seek past the PVN & Car ID
        tBitsRead = OSAL.tBufferSeekHeadBits(
                        hPayload, WS_ALERTS1_TRIM_BITLEN);
        if (tBitsRead == WS_ALERTS1_TRIM_BITLEN)
        {
            OSAL.bMemSet(&(psObj->sAUData), 0,
                sizeof(WS_ALERTS1_AU_DATA_STRUCT));

            psObj->sAUData.eCarouselID = eMessageType;

            eReturnCode = eParseWSAlertsHeader (psObj, hPayload);
            if (eReturnCode == SMSAPI_RETURN_CODE_DUPLICATE_CONTENT)
            {
                puts(WS_ALERTS1_OBJECT_NAME": skipping previously parsed AU");
                break;
            }
            else if (eReturnCode == SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT)
            {
                puts(WS_ALERTS1_OBJECT_NAME": skipping AU");
                break;
            }
            else if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": Invalid WS_Alerts Header");
                break;
            }
            bResult = TRUE;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": Unable to seek past Message Header / Footer");
            bResult = FALSE;
        }
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eGetMessageType
*
*****************************************************************************/
static WS_ALERTS1_MSG_CAROUSEL_ENUM eGetMessageType (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    PVN tPVN = (PVN)0;
    size_t tBitsRead;
    WS_ALERTS1_MSG_CAROUSEL_ENUM eMessageType = WS_ALERTS1_INVALID_MSG;
    UN8 un8CarouselID = (UN8)eMessageType;

    do
    {
        // Peek at the first bits to see if this is the right PVN and Car ID
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &tPVN, 0,
            WS_ALERTS1_PVN_BITLEN, 0);

        if (tBitsRead != WS_ALERTS1_PVN_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" PVN Peek Failed\n");
            break;
        }

        // Verify the PVN
        if (tPVN != WS_ALERTS1_PVN)
        {
            printf(WS_ALERTS1_OBJECT_NAME" Incorrect PVN - got %u, expected %u\n",
                tPVN, WS_ALERTS1_PVN);
            break;
        }

        // Peek at the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8CarouselID, 0,
            WS_ALERTS1_CAROUSEL_BITLEN, WS_ALERTS1_PVN_BITLEN);

        if (tBitsRead != WS_ALERTS1_CAROUSEL_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" CarID Peek Failed\n");
            break;
        }

        // We read all the fields, so set the return type
        eMessageType = (WS_ALERTS1_MSG_CAROUSEL_ENUM)un8CarouselID;
    }while (FALSE);
    return eMessageType;
}

/*****************************************************************************
 *
 *   eParseWSAlertsHeader
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eParseWSAlertsHeader (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOwner, bSuccess = TRUE, bFieldPresent = FALSE, bSTATEFLAG = FALSE;
    size_t tBitsRead;
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

    UN16 un16AUCT = 0, un16AUTOT = 0, un16EPOCH = 0;
    UN8 un8STATE = 0, un8STVER = 0, un8CTSIZE = 0;
    SMS_LANGUAGE_ENUM eLANG = SMS_INVALID_LANGUAGE;

    do
    {
        // Verify inputs
        if ((psObj == NULL) ||
            (hPayload == OSAL_INVALID_BUFFER_HDL))
        {
            puts(WS_ALERTS1_OBJECT_NAME": eParseWSAlertsHeader: invalid params\n");
            eReturnCode = SMSAPI_RETURN_CODE_INVALID_INPUT;
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            eReturnCode = SMSAPI_RETURN_CODE_NOT_OWNER;
            break;
        }

        // Read STATEFLAG
        tBitsRead = OSAL.tBufferReadHeadBits (
            hPayload, &bSTATEFLAG, 0,
            WS_ALERTS1_STATE_FLAG_BITLEN );

        if (tBitsRead != WS_ALERTS1_STATE_FLAG_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" STATEFLAG read Failed\n");
            break;
        }

        // Read STATE ID
        tBitsRead = OSAL.tBufferReadHeadBits (
            hPayload, &un8STATE, 0,
            WS_ALERTS1_STATE_ID_BITLEN );

        if (tBitsRead != WS_ALERTS1_STATE_ID_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" STATE read Failed\n");
            break;
        }

        // Read LANG
        tBitsRead = OSAL.tBufferReadHeadBits (
            hPayload, (void*)&eLANG, 0,
            WS_ALERTS1_LANGUAGE_CODE_BITLEN );

        if (tBitsRead != WS_ALERTS1_LANGUAGE_CODE_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" LANG read Failed\n");
            break;
        }

        // Read STVER
        tBitsRead = OSAL.tBufferReadHeadBits (
            hPayload, &un8STVER, 0,
            WS_ALERTS1_STATE_VERSION_BITLEN );

        if (tBitsRead != WS_ALERTS1_STATE_VERSION_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" STVER read Failed\n");
            break;
        }

        // Is CTSIZE There?
        bFieldPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
            hPayload, &bFieldPresent, 0,
            WS_ALERTS1_PRESENCE_FLAG_BITLEN );

        if (tBitsRead != WS_ALERTS1_PRESENCE_FLAG_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" CTSIZE Field Present flag read failed\n");
            break;
        }

        if (bFieldPresent == TRUE)
        {

            // CTSIZE is present
            tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8CTSIZE, 0,
                WS_ALERTS1_CTSIZE_BITLEN );

            if (tBitsRead != WS_ALERTS1_CTSIZE_BITLEN)
            {
                puts(WS_ALERTS1_OBJECT_NAME" CTSIZE read failed\n");
                break;
            }

            // SX-9845-0039, Sec 5.1.7 states that the size
            // of AUTOT and AUCT are CTSIZE+1, so
            // increment CTSIZE
            un8CTSIZE++;

            // Now parse AUTOT and AUCT using CTSIZE
            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUTOT, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                puts(WS_ALERTS1_OBJECT_NAME" AUTOT read failed\n");
                break;
            }

            // SX-9845-0039, Sec 5.1.8 states that there are
            // AUTOT+1 AUs in this carousel.  So increment AUTOT
            un16AUTOT++;

            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUCT, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                puts(WS_ALERTS1_OBJECT_NAME" AUCT read failed\n");
                break;
            }
        }
        else
        {

            // No CTSIZE field, so only one AU for this state carousel
            // according to RX190, Sec 6.1.5
            un16AUCT = 0;
            un16AUTOT = 1;
        }

        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &un16EPOCH, WS_ALERTS1_EPOCH_BITLEN);

        if (bSuccess == FALSE)
        {
            puts(WS_ALERTS1_OBJECT_NAME" EPOCH read failed\n");
            break;
        }

        psObj->sAUData.bSTATEFLAG = bSTATEFLAG;
        psObj->sAUData.un8STATE = un8STATE;
        psObj->sAUData.eLANG = eLANG;
        psObj->sAUData.un8STVER = un8STVER;
        psObj->sAUData.un16AUTOT = un16AUTOT;
        psObj->sAUData.un16AUCT = un16AUCT;
        psObj->sAUData.un16EPOCH = un16EPOCH;
        psObj->sAUData.un32CRC = psObj->un32CurrentMsgCRC;
        psObj->sAUData.hPayload = hPayload;

        eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

    } while (FALSE);

    return eReturnCode;
}

/*****************************************************************************
 *
 *   eAddAccessUnitToStorage
 *
 *****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eAddAccessUnitToStorage(
    WS_ALERTS1_OBJECT_STRUCT *psObj
        )
{
    SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc = NULL;
    WS_ALERTS1_UNPARSED_AU_STRUCT *psUnParsedAUDesc = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    BOOLEAN bStateMsgDescChanged = FALSE;
    BOOLEAN bSkipAU = TRUE;
    UN32 un32Items = 0;

    do
    {
        psStateMsgDesc = psWSAlertsGetStateMsgDesc(psObj, &bStateMsgDescChanged);

        if ( psStateMsgDesc == NULL )
        {
            eReturnCode = SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT;
            break;
        }

        if ( bStateMsgDescChanged != TRUE )
        {
            if ( (psObj->sAUData.un16EPOCH > psStateMsgDesc->un16EPOCH) ||
                 (psObj->sAUData.un8STVER > psStateMsgDesc->un8Version) )
            {
                vRenewStateMsg( psObj, psStateMsgDesc );
                bSkipAU = FALSE;
            }
            else
            {
                // this structure will be used for search
                WS_ALERTS1_UNPARSED_AU_STRUCT sSearchCriteria;

                //initializing the data in structure
                sSearchCriteria.un16AUCT = psObj->sAUData.un16AUCT;

                // Look up the message descriptor by state id
                eOsalReturnCode = OSAL.eLinkedListSearch(
                    psStateMsgDesc->hUnParsedAUsList,
                    &hEntry,
                    &sSearchCriteria);

                // Check error codes
                if ((eOsalReturnCode == OSAL_SUCCESS) &&
                    (hEntry != OSAL_INVALID_LINKED_LIST_ENTRY))
                {
                    WS_ALERTS1_UNPARSED_AU_STRUCT *psExistingUnParsedAUDesc = NULL;

                    psExistingUnParsedAUDesc = (WS_ALERTS1_UNPARSED_AU_STRUCT*)OSAL.pvLinkedListThis(hEntry);
                    if ( psObj->sAUData.un32CRC != psExistingUnParsedAUDesc->un32CRC )
                    {
                        vRenewStateMsg( psObj, psStateMsgDesc );
                        bSkipAU = FALSE;
                    }
                }
                else if ((eOsalReturnCode == OSAL_OBJECT_NOT_FOUND) ||
                         (eOsalReturnCode == OSAL_NO_OBJECTS))
                {
                    bSkipAU = FALSE;
                }
            }
        }
        else
        {
            bSkipAU = FALSE;
        }

        if ( bSkipAU == FALSE )
        {
            psUnParsedAUDesc = (WS_ALERTS1_UNPARSED_AU_STRUCT *)
                SMSO_hCreate(
                    WS_ALERTS1_OBJECT_NAME":UnParsedAU",
                    sizeof(WS_ALERTS1_UNPARSED_AU_STRUCT),
                    (SMS_OBJECT)psObj, FALSE);

            if (psUnParsedAUDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME
                    ": failed to create unparsed AU desc");
                break;
            }

            psUnParsedAUDesc->un16AUCT = psObj->sAUData.un16AUCT;
            psUnParsedAUDesc->un32CRC = psObj->sAUData.un32CRC;
            psUnParsedAUDesc->hPayload = psObj->sAUData.hPayload;

            if ( psStateMsgDesc->un16AUTotal > 1 )
            {
                bPlacePayloadToFile(psObj, psUnParsedAUDesc);
            }

            // Add this to our list of message elements for this service
            eOsalReturnCode = OSAL.eLinkedListAdd(
                psStateMsgDesc->hUnParsedAUsList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                psUnParsedAUDesc);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to add AU to the list (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
                break;
            }

            // Extract and check number of items in the list
            eOsalReturnCode = OSAL.eLinkedListItems(
                psStateMsgDesc->hUnParsedAUsList,
                &un32Items);
            if ( eOsalReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to get AUs list size (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
                break;
            }

            if (un32Items == psStateMsgDesc->un16AUTotal)
            {
                WS_ALERTS1_ITERATOR_DATA_STRUCT sIteratorData;

                printf(WS_ALERTS1_OBJECT_NAME": Collected new set of AUs for a state.\n");
                psStateMsgDesc->eLANG = psObj->sAUData.eLANG;

                sIteratorData.bRemoveMsgsFromDB = TRUE;
                sIteratorData.psObj = psObj;
                sIteratorData.psStateMsgDesc = psStateMsgDesc;

                eOsalReturnCode = OSAL.eLinkedListIterate(
                    psStateMsgDesc->hUnParsedAUsList,
                    (OSAL_LL_ITERATOR_HANDLER)bProcessAuIterator,
                    (void *)&sIteratorData);

                if ( (eOsalReturnCode != OSAL_SUCCESS) )
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to iterate Parsed AUs list (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode));
                    break;
                }

                GsWSAlertsMgrIntf.vStateUpdateCompleted( psObj->hWSAlertsService );
            }

            // AU saved but set is not full. Continue collecting
            eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;

        }
        else
        {
            eReturnCode = SMSAPI_RETURN_CODE_DUPLICATE_CONTENT;
        }

    } while (FALSE);

    if ( (eReturnCode !=  SMSAPI_RETURN_CODE_SUCCESS) &&
         (psUnParsedAUDesc != NULL) )
    {
        vReleaseUnParsedAU( psObj, psUnParsedAUDesc );
    }

    return eReturnCode;
}

/*****************************************************************************
 *
 *   bProcessAuIterator
 *
 *****************************************************************************/

static BOOLEAN bProcessAuIterator (
    WS_ALERTS1_UNPARSED_AU_STRUCT *psUnParsedAUDesc,
    WS_ALERTS1_ITERATOR_DATA_STRUCT *psIteratorData
        )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    WS_ALERTS1_PARSED_AU_STRUCT *psParsedAUDesc = NULL;

    do
    {
        if ( psIteratorData->psStateMsgDesc->un16AUTotal > 1 )
        {
            bOk = bGetPayloadFromFile(psIteratorData->psObj, psUnParsedAUDesc);

            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to get payload from file");
                break;
            }
        }

        psParsedAUDesc = (WS_ALERTS1_PARSED_AU_STRUCT *)
            SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":ParsedAU",
                sizeof(WS_ALERTS1_PARSED_AU_STRUCT),
                (SMS_OBJECT)psIteratorData->psObj, FALSE);

        if (psParsedAUDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Failed to create psParsedAUDesc");
            break;
        }

        eOsalReturnCode = OSAL.eLinkedListCreate(
            &psParsedAUDesc->hMsgElementsList,
            WS_ALERTS1_OBJECT_NAME":MsgElementsList",
            NULL,
            OSAL_LL_OPTION_NONE);

        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create list MsgElementsList (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        psIteratorData->psObj->psParsedAUDesc = psParsedAUDesc;

        bOk = bCompleteParseAUHeader( psIteratorData->psObj, psUnParsedAUDesc->hPayload );

        if ( bOk != TRUE )
        {
            printf("bAddAUToStorage: AU header is not parsed properly\n");
            break;
        }

        bOk = bParseAUPayload( psIteratorData->psObj, psUnParsedAUDesc->hPayload );

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to parse AU payload");
            break;
        }

        bOk= bProcessAU(
            psIteratorData->psObj,
            psIteratorData->psStateMsgDesc,
            psIteratorData->bRemoveMsgsFromDB);

        if (bOk != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to process parsed AU");
            break;
        }
        psIteratorData->bRemoveMsgsFromDB = FALSE;

        //Renew the buffer
        DATASERVICE_MGR_vDestroyFileBuffer(psIteratorData->psObj->hBlockPool,
            psIteratorData->psObj->hBuffer);

        bOk = DATASERVICE_MGR_bCreateFileBuffer(
            &psIteratorData->psObj->hBlockPool,
            &psIteratorData->psObj->hBuffer,
            WS_ALERTS1_BUFFER_SIZE);

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create buffer");
            break;
        }

    }while(FALSE);

    vReleaseParsedAU( psIteratorData->psObj->psParsedAUDesc );


    return bOk;
}

/*****************************************************************************
 *
 *   vRenewStateMsg
 *
 *****************************************************************************/
static void vRenewStateMsg(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc
        )
{
    vReleaseUnParsedAUsList(psObj, psStateMsgDesc->hUnParsedAUsList);

    psStateMsgDesc->un8Version = psObj->sAUData.un8STVER;
    psStateMsgDesc->un16EPOCH = psObj->sAUData.un16EPOCH;
    psStateMsgDesc->un16AUTotal = psObj->sAUData.un16AUTOT;

    return;
}

/*****************************************************************************
 *
 *   bCompleteParseAUHeader
 *
 *****************************************************************************/
static BOOLEAN bCompleteParseAUHeader(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bResult = FALSE, bSuccess= FALSE;

    UN16 un16INDEX = 0;
    UN8 un8CLASS = 0, un8TABID = 0, un8TPSIZE = 0,
        un8TVMIN = 0, un8TABINDEX = 0;
    BOOLEAN bFieldPresent = FALSE;
    size_t tBitsRead;
    UN16 un16CurTabIdx, un16TABNO = 0;

    do
    {
        tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
            &(psObj->psParsedAUDesc->un8ETIME),0, WS_ALERTS1_ETIME_BITLEN);

        if (tBitsRead != WS_ALERTS1_ETIME_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" ETIME read failed\n");
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
            &(psObj->psParsedAUDesc->un8ISIZE), 0, WS_ALERTS1_ISIZE_BITLEN);

        if (tBitsRead != WS_ALERTS1_ISIZE_BITLEN)
        {
            puts(WS_ALERTS1_OBJECT_NAME" ISIZE read failed\n");
            break;
        }

        // SX-9845-0039, Sec 5.1.12 states that the size
        // of INDEX is ISIZE+1, so increment ISIZE
        psObj->psParsedAUDesc->un8ISIZE++;

        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &un16TABNO, psObj->psParsedAUDesc->un8ISIZE);

        if (bSuccess == FALSE)
        {
            puts(WS_ALERTS1_OBJECT_NAME" TABNO read failed\n");
            break;
        }

        psObj->psParsedAUDesc->un16TABNO = 0;
        un16TABNO++;

        if (un16TABNO > WS_ALERTS1_TABLE_DESCS_MAX_NUM)
        {
            printf(WS_ALERTS1_OBJECT_NAME" TABNO %d is too big", (int)un16TABNO);
            break;
        }

        for (un16CurTabIdx = 0;
             un16CurTabIdx < un16TABNO;
             un16CurTabIdx++)
        {
            WS_ALERTS1_TABLE_DESC_STRUCT *pDesc;

            bResult = FALSE;

            // Now parse INDEX using ISIZE
            un16INDEX = 0;
            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16INDEX, psObj->psParsedAUDesc->un8ISIZE);

            if (bSuccess == FALSE)
            {
                break;
            }

            un8CLASS = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8CLASS, 0, WS_ALERTS1_TAB_CLASS_BITLEN);

            if (tBitsRead != WS_ALERTS1_TAB_CLASS_BITLEN)
            {
                break;
            }

            un8TABID = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8TABID, 0, WS_ALERTS1_TAB_TABID_BITLEN);

            if (tBitsRead != WS_ALERTS1_TAB_TABID_BITLEN)
            {
                break;
            }

            un8TPSIZE = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8TPSIZE, 0, WS_ALERTS1_TAB_TPSIZE_BITLEN);

            if (tBitsRead != WS_ALERTS1_TAB_TPSIZE_BITLEN)
            {
                break;
            }

            un8TVMIN = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8TVMIN, 0, WS_ALERTS1_TAB_TVMIN_BITLEN);

            if (tBitsRead != WS_ALERTS1_TAB_TVMIN_BITLEN)
            {
                break;
            }

            // Is TABINDEX there?
            bFieldPresent = FALSE;
            tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                WS_ALERTS1_PRESENCE_FLAG_BITLEN );

            if (tBitsRead != WS_ALERTS1_PRESENCE_FLAG_BITLEN)
            {
                break;
            }

            if (bFieldPresent == TRUE)
            {
                un8TABINDEX = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                    &un8TABINDEX, 0, WS_ALERTS1_TAB_TABINDEX_BITLEN);

                if (tBitsRead != WS_ALERTS1_TAB_TABINDEX_BITLEN)
                {
                    break;
                }
            }
            else
            {
                un8TABINDEX = psObj->sAUData.un8STATE;
            }

            // Try to see if the table already exists. If it does rewrite
            // the place with new data; othwerwise, put as a new one.
            pDesc = psGetTable(psObj, un16INDEX);
            if (pDesc == NULL)
            {
                pDesc = &psObj->psParsedAUDesc->asTabDesc[psObj->psParsedAUDesc->un16TABNO];
                ++psObj->psParsedAUDesc->un16TABNO;
            }
            pDesc->tINDEX = un16INDEX;
            pDesc->tCLASS = un8CLASS;
            pDesc->tTABID = un8TABID;
            pDesc->tTPSIZE = un8TPSIZE;
            pDesc->tTVMIN = un8TVMIN;
            pDesc->tTABINDEX = un8TABINDEX;

            bResult = TRUE;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *   bParseAUPayload
 *
 *****************************************************************************/
static BOOLEAN bParseAUPayload(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *psMsgElem = NULL;
    BOOLEAN bSkipElement = FALSE;
    UN16 un16Index = 0;
    UN16 un16PAR = 0;
    UN8 un8ParLength = 0;
    UN8 un8FMT = 0;
    size_t tBitsRemaining = 0;
    size_t tBitsRead = 0;
    size_t tIndex = 0;
    size_t tLength = 0;
    BOOLEAN bSuccess = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    int iReturn;

    size_t tOSIZE = 0;

    do
    {
        un16Index = 0;
        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &un16Index, psObj->psParsedAUDesc->un8ISIZE);

        if (bSuccess == FALSE)
        {
            printf("error in index parse\n");
            break;
        }

        un8ParLength = u8GetPARLength(psObj, un16Index);

        un16PAR = 0;
        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &un16PAR, un8ParLength);

        // Create the Message Element descriptor
        psMsgElem =
            (WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *) SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":StateMsg:MsgElement",
                sizeof(WS_ALERTS1_MESSAGE_ELEMENT_STRUCT),
                (SMS_OBJECT)psObj, FALSE);

        if (psMsgElem == NULL)
        {
            printf("Cannot create the Message Element descriptor\n");
            break;
        }

        un8FMT = 0;
        //FMT field is absent in Message delimiter element (PAR=0).
        //Skip it in this case.
        if ( !((un16Index == 0) && (un16PAR == 0)) )
        {
            tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                &un8FMT, 0, WS_ALERTS1_MSG_FMT_BITLEN);

            if (tBitsRead != WS_ALERTS1_MSG_FMT_BITLEN)
            {
                break;
            }
        }

        psMsgElem->tINDEX = un16Index;
        psMsgElem->tPAR = un16PAR;
        psMsgElem->tFMT = un8FMT;

        if (un16Index == 0)
        {
            switch (un16PAR)
            {
                case WS_ALERTS1_MESSAGE_DELIMITER_ELEM:
                {
                    psMsgElem->uData.un16MSGSIG = 0;
                    bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                        &psMsgElem->uData.un16MSGSIG,
                        WS_ALERTS1_MESSAGE_DELIMITER_ELEM_BITLEN);

                    break;
                }
                case WS_ALERTS1_START_TIME_ELEM:
                case WS_ALERTS1_END_TIME_ELEM:
                case WS_ALERTS1_GENERIC_TIME_ELEM:
                {
                    tOSIZE = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &tOSIZE, 0,
                        WS_ALERTS1_START_TIME_ELEM_OSIZE_BITLEN);

                    if (tBitsRead != WS_ALERTS1_START_TIME_ELEM_OSIZE_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    tOSIZE = tOSIZE*4+4;

                    psMsgElem->uData.un16TimeOffset = 0;

                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &(psMsgElem->uData.un16TimeOffset),
                        tOSIZE);

                    break;
                }

                case WS_ALERTS1_POLYGON_ELEM:
                {
                    psMsgElem->uData.sPolygon.un8LOCNO = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &(psMsgElem->uData.sPolygon.un8LOCNO), 0,
                        WS_ALERTS1_POLYGON_ELEM_LOCNO_BITLEN);

                    if (tBitsRead != WS_ALERTS1_POLYGON_ELEM_LOCNO_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    if ( (psMsgElem->uData.sPolygon.un8LOCNO <
                         WS_ALERTS1_POLYGON_DELTA_MIN_NUM) ||
                         (psMsgElem->uData.sPolygon.un8LOCNO >
                         WS_ALERTS1_POLYGON_DELTA_MAX_NUM) )
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    psMsgElem->uData.sPolygon.un16ANCHORLat = 0;
                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &(psMsgElem->uData.sPolygon.un16ANCHORLat),
                        WS_ALERTS1_POLYGON_ELEM_ANCHOR_LAT_BITLEN);

                    if (bSuccess != TRUE)
                    {
                        break;
                    }

                    psMsgElem->uData.sPolygon.un16ANCHORLon = 0;
                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &(psMsgElem->uData.sPolygon.un16ANCHORLon),
                        WS_ALERTS1_POLYGON_ELEM_ANCHOR_LON_BITLEN);

                    if (bSuccess != TRUE)
                    {
                        break;
                    }

                    for ( tIndex = 0;
                          tIndex <= psMsgElem->uData.sPolygon.un8LOCNO;
                          tIndex++ )
                    {
                        psMsgElem->uData.sPolygon.n16DELTAPLat[tIndex] = 0;
                        bSuccess = OSAL.bBufferReadBitsToN16(
                            hPayload, &(psMsgElem->uData.sPolygon.n16DELTAPLat[tIndex]),
                            WS_ALERTS1_POLYGON_ELEM_DELTAP_LAT_BITLEN);

                        if (bSuccess != TRUE)
                        {
                            break;
                        }

                        psMsgElem->uData.sPolygon.n16DELTAPLon[tIndex] = 0;
                        bSuccess = OSAL.bBufferReadBitsToN16(
                            hPayload, &(psMsgElem->uData.sPolygon.n16DELTAPLon[tIndex]),
                            WS_ALERTS1_POLYGON_ELEM_DELTAP_LON_BITLEN);

                        if (bSuccess != TRUE)
                        {
                            break;
                        }
                    }
                    break;
                }

                case WS_ALERTS1_TEXT_LTR_ELEM:
                case WS_ALERTS1_TEXT_LTR_LC_ELEM:
                case WS_ALERTS1_TEXT_FIG_ELEM:
                {
                    STRING_OBJECT hText;
                    BOOLEAN bCaps = TRUE;
                    BOOLEAN bLetters = TRUE;

                    if (un16PAR == WS_ALERTS1_TEXT_LTR_LC_ELEM)
                    {
                        bCaps = FALSE;
                    }

                    if (un16PAR == WS_ALERTS1_TEXT_FIG_ELEM)
                    {
                        bLetters = FALSE;
                        bCaps = FALSE;
                    }

                    bSuccess = bGetFreeText((SMS_OBJECT)psObj,
                        hPayload,
                        &hText,
                        bLetters,
                        bCaps);
                    if (bSuccess == FALSE)
                    {
                        break;
                    }

                    psMsgElem->uData.hFreeText = hText;
                    break;
                }

                case WS_ALERTS1_PHONEME_ELEM:
                {
                    //Read and ignore a phoneme message
                    tLength = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &tLength, 0,
                        WS_ALERTS1_PHONEME_ELEM_LENGTH_BITLEN);

                    if (tBitsRead != WS_ALERTS1_PHONEME_ELEM_LENGTH_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    tOSIZE = 8*(tLength+1);

                    // This field is not implemented in current protocol version
                    tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, tOSIZE);

                    if (tBitsRead != tOSIZE)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    bSkipElement = TRUE;
                    break;
                }
                case WS_ALERTS1_LARGE_CONTAINER_ELEM:
                {
                    tLength = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &tLength, 0,
                        WS_ALERTS1_LARGE_CONTAINER_ELEM_LENGTH_BITLEN);

                    if (tBitsRead != WS_ALERTS1_LARGE_CONTAINER_ELEM_LENGTH_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    tOSIZE = 8*(tLength+1);

                    // This field is not implemented in current protocol version
                    tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, tOSIZE);

                    if (tBitsRead != tOSIZE)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    bSkipElement = TRUE;
                    break;
                }
                case WS_ALERTS1_SMALL_CONTAINER_ELEM:
                {
                    tLength = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &tLength, 0,
                        WS_ALERTS1_SMALL_CONTAINER_ELEM_LENGTH_BITLEN);

                    if (tBitsRead != WS_ALERTS1_SMALL_CONTAINER_ELEM_LENGTH_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    tOSIZE = tLength+1;

                    // This field is not implemented in current protocol version
                    tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, tOSIZE);

                    if (tBitsRead != tOSIZE)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    bSkipElement = TRUE;
                    break;
                }
                case WS_ALERTS1_FILE_LINK_ELEM:
                {
                    bSuccess = bGetFreeText((SMS_OBJECT)psObj, hPayload,
                        &psMsgElem->uData.hFileLink, TRUE, TRUE);

                    if ( bSuccess == TRUE )
                    {
                        STRING_vDestroy(psMsgElem->uData.hFileLink);
                        psMsgElem->uData.hFileLink = STRING_INVALID_OBJECT;
                    }
                    else
                    {
                        break;
                    }

                    bSkipElement = TRUE;
                    break;
                }
                case WS_ALERTS1_PHONE_NUMBER_ELEM:
                {
                    WS_ALERTS1_PHONE_NUMBER_STRUCT sPhoneNumber;
                    char acBuf[WS_ALERTS1_SHARED_BUFFER_LEN];
                    size_t tLength = 0;

                    OSAL.bMemSet(&sPhoneNumber, 0, sizeof(sPhoneNumber));

                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &sPhoneNumber.un8INT, 0,
                        WS_ALERTS1_PHONE_NUMBER_ELEM_INT_BITLEN);

                    if (tBitsRead != WS_ALERTS1_PHONE_NUMBER_ELEM_INT_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &sPhoneNumber.un16AREA,
                        WS_ALERTS1_PHONE_NUMBER_ELEM_AREA_BITLEN);

                    if (bSuccess != TRUE)
                    {
                        break;
                    }

                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &sPhoneNumber.un16EXCH,
                        WS_ALERTS1_PHONE_NUMBER_ELEM_EXCH_BITLEN);

                    if (bSuccess != TRUE)
                    {
                        break;
                    }

                    bSuccess = OSAL.bBufferReadBitsToUN16(
                        hPayload, &sPhoneNumber.un16NUM,
                        WS_ALERTS1_PHONE_NUMBER_ELEM_NUM_BITLEN);

                    if (bSuccess != TRUE)
                    {
                        break;
                    }

                    if ( (sPhoneNumber.un8INT == WS_ALERTS1_PHONE_COUNTRY_ID_US) ||
                         (sPhoneNumber.un8INT == WS_ALERTS1_PHONE_COUNTRY_ID_CANADA) )
                    {
                        iReturn = snprintf( acBuf,
                            WS_ALERTS1_SHARED_BUFFER_LEN,
                            WS_ALERTS1_PHONE_NUMBER_STRING,
                            WS_ALERTS1_PHONE_COUNTRY_ID_STRING_US,
                            sPhoneNumber.un16AREA,
                            sPhoneNumber.un16EXCH,
                            sPhoneNumber.un16NUM );

                        if (iReturn <= 0)
                        {
                            bSuccess = FALSE;
                            break;
                        }
                    }
                    else if ( sPhoneNumber.un8INT == WS_ALERTS1_PHONE_COUNTRY_ID_MEXICO )
                    {
                        iReturn = snprintf( acBuf,
                            WS_ALERTS1_SHARED_BUFFER_LEN,
                            WS_ALERTS1_PHONE_NUMBER_STRING,
                            WS_ALERTS1_PHONE_COUNTRY_ID_STRING_MEXICO,
                            sPhoneNumber.un16AREA,
                            sPhoneNumber.un16EXCH,
                            sPhoneNumber.un16NUM );

                        if (iReturn <= 0)
                        {
                            bSuccess = FALSE;
                            break;
                        }
                    }
                    else
                    {
                        bSuccess = FALSE;
                        break;
                    }

                    tLength = strlen(acBuf);
                    psMsgElem->uData.hPhoneNum = STRING.hCreate(acBuf, tLength);

                    break;
                }
                case WS_ALERTS1_SYSTEM_PRIORITY_ELEM:
                {
                    psMsgElem->uData.un8SPRI = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
                        &psMsgElem->uData.un8SPRI, 0,
                        WS_ALERTS1_SYSTEM_PRIORITY_ELEM_SPRI_BITLEN);

                    if (tBitsRead != WS_ALERTS1_SYSTEM_PRIORITY_ELEM_SPRI_BITLEN)
                    {
                        bSuccess = FALSE;
                    }

                    break;
                }
                default:
                {
                    printf("Unknown parameter PAR=%d\n", un16PAR);
                    bSuccess = FALSE;
                    break;
                }
            }
        }

        if (bSuccess != TRUE)
        {
            printf("Parser error\n");
            break;
        }

        if (bSkipElement == FALSE)
        {
            // Add this to our list of message elements for this service
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->psParsedAUDesc->hMsgElementsList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psMsgElem
                    );
            if (eReturnCode == OSAL_SUCCESS)
            {
                psMsgElem = NULL;
            }
            else
            {
                printf(WS_ALERTS1_OBJECT_NAME": failed to add message element to list(%s)\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                break;
            }
        }
        else
        {
            // Since this message is skipped, destroy it
            SMSO_vDestroy((SMS_OBJECT) psMsgElem);
            psMsgElem = NULL;

            // Continue reading
            bSkipElement = FALSE;
        }

        // How many bits are left?
        tBitsRemaining = OSAL.tBufferGetSizeInBits(hPayload);

        // If there are less than 8 bits left, then we
        // only have bit stuffing left in this message
        if (tBitsRemaining <= 7)
        {
            break;
        }
    } while (TRUE);

    if (psMsgElem != NULL)
    {
        vReleaseMsgElement(psMsgElem);
        bSuccess = FALSE;
    }

    if (bSuccess == TRUE)
    {
        bSuccess = bCheckAU( psObj, psObj->psParsedAUDesc->hMsgElementsList);
    }

    return bSuccess;
}

/*****************************************************************************
 *
 *   bCheckAU
 *
 *****************************************************************************/
static BOOLEAN bCheckAU (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hMsgElementsList
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS1_AU_CHECK_ITERATOR_STRUCT sIterator;
    BOOLEAN bSuccess = FALSE;

    sIterator.psObj = psObj;
    sIterator.bIsAlertTypePresent = FALSE;
    sIterator.bIsAlertLocationPresent = FALSE;
    sIterator.bSuccess = TRUE;

    eReturnCode = OSAL.eLinkedListIterate(
        hMsgElementsList,
        (OSAL_LL_ITERATOR_HANDLER)bAuCheckIterator,
        (void *)&sIterator);

    if (eReturnCode == OSAL_SUCCESS)
    {
        if (sIterator.bSuccess == TRUE)
        {
            bSuccess = TRUE;
        }
    }
    else
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": failed to iterate message elements list (%s)",
            OSAL.pacGetReturnCodeName(eReturnCode));
    }

    return bSuccess;
}

/*******************************************************************************
*
*   bAuCheckIterator
*
*******************************************************************************/
static BOOLEAN bAuCheckIterator (
    WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *psMsgElem,
    WS_ALERTS1_AU_CHECK_ITERATOR_STRUCT *psIterator
        )
{
    const WS_ALERTS1_TABLE_DESC_STRUCT *psLookUpTable;

    psLookUpTable = psGetTable(psIterator->psObj, psMsgElem->tINDEX);
    if (psLookUpTable == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": there is no table for indedx %d",
            psMsgElem->tINDEX);
        psIterator->bSuccess = FALSE;
        return FALSE;
    }

    if ( (psMsgElem->tINDEX != 0) &&
         (psLookUpTable->tCLASS == WS_ALERTS1_LUT_CLASS_ALERT_TYPE) )
    {
        psIterator->bIsAlertTypePresent = TRUE;
    }
    else if ( psLookUpTable->tCLASS == WS_ALERTS1_LUT_CLASS_ALERT_LOCATION )
    {
        psIterator->bIsAlertLocationPresent = TRUE;
    }

    if ( (psMsgElem->tINDEX == 0) &&
         (psMsgElem->tPAR == WS_ALERTS1_POLYGON_ELEM) )
    {
        psIterator->bIsAlertLocationPresent = TRUE;
    }

    if ( (psMsgElem->tINDEX == 0) &&
         (psMsgElem->tPAR == 0) )
    {
        if ( (psIterator->bIsAlertTypePresent == FALSE) ||
             (psIterator->bIsAlertLocationPresent == FALSE) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": message does not contain type or location element");
            psIterator->bSuccess = FALSE;
            return FALSE;
        }

        psIterator->bIsAlertTypePresent = FALSE;
        psIterator->bIsAlertLocationPresent = FALSE;
    }
    return TRUE;
}

/*****************************************************************************
 *
 *   psGetTable
 *
 *****************************************************************************/
static WS_ALERTS1_TABLE_DESC_STRUCT* psGetTable(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    UN16 un16Index
        )
{
    UN16 un16ForIndex;
    WS_ALERTS1_TABLE_DESC_STRUCT *psResult;
    if (un16Index == 0 /* Special case to provide fake table*/)
    {
        psResult = &psObj->psParsedAUDesc->asTabDesc[WS_ALERTS1_TABLE_DESCS_MAX_NUM];
    }
    else 
    {
        for (un16ForIndex = 0, psResult = NULL;
            un16ForIndex < psObj->psParsedAUDesc->un16TABNO;
            un16ForIndex++ )
        {
            if (psObj->psParsedAUDesc->asTabDesc[un16ForIndex].tINDEX == un16Index)
            {
                psResult = &psObj->psParsedAUDesc->asTabDesc[un16ForIndex];
                break;
            }
        }
    }

    return psResult;
}

/*****************************************************************************
 *
 *   u8GetPARLength
 *
 *****************************************************************************/
static UN8 u8GetPARLength (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    UN16 un16Index
        )
{
    const WS_ALERTS1_TABLE_DESC_STRUCT *psTable;

    if (un16Index == 0)
    {
        return WS_ALERTS1_MSG_PAR_FIXED_BITLEN;
    }

    //From SX-9845-0039_Weather_And_Security_Alerts_Protocol_v1.3, page 37:
    //The INDEX value 0x0 is reserved for use as an indicator of Special Case
    //Message Elements.
    psTable = psGetTable(psObj, un16Index);
    if (psTable != NULL)
    {
        return psTable->tTPSIZE;
    }

    printf("%s: index %d not found in the header\n", __FUNCTION__, (int)un16Index);
    return 0;
}

/*****************************************************************************
 *
 *   bPlacePayloadToFile
 *
 *****************************************************************************/
static BOOLEAN bPlacePayloadToFile(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_UNPARSED_AU_STRUCT *psUnParsedAUDesc
)
{
    BOOLEAN bOk = TRUE;
    FILE *pFile = NULL;
    char acPath[WS_ALERTS1_SHARED_BUFFER_LEN];
    size_t tNumBytesRead,
           tNumBytesWritten,
           tSizeRemaining,
           tNumBytesToRead;
    int iReturn;

    do
    {
        if ((psObj == NULL) || (psUnParsedAUDesc == NULL) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": %s : invalid input",
                __FUNCTION__);
            bOk = FALSE;
            break;
        }

        iReturn = snprintf(&psUnParsedAUDesc->acFilename[0],
            WS_ALERTS1_PAYLOAD_FILE_NAME_LENGTH,
            WS_ALERTS1_FILE_NAME_ID_LABEL,
            (UN32)psUnParsedAUDesc->hPayload);

        if (iReturn <= 0)
        {
            bOk = FALSE;
            break;
        }

        iReturn = snprintf(&acPath[0],
            WS_ALERTS1_SHARED_BUFFER_LEN,
            "%s/%s",
            psObj->pacServiceFilePath,
            psUnParsedAUDesc->acFilename);

        if (iReturn <= 0)
        {
            bOk = FALSE;
            break;
        }

        // Open the file, creating it if necessary
        pFile = fopen( acPath, "wb");

        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": cannot open file %s\n", acPath);
            bOk = FALSE;
            break;
        }

        tSizeRemaining = OSAL.tBufferGetSize ( psUnParsedAUDesc->hPayload );
        while (tSizeRemaining > 0)
        {
            if (tSizeRemaining > sizeof(psObj->acBuffer))
            {
                tNumBytesToRead = sizeof(psObj->acBuffer);
            }
            else
            {
                tNumBytesToRead = tSizeRemaining;
            }

            // Extract the data a piece at a time
            tNumBytesRead = OSAL.tBufferReadHead(psUnParsedAUDesc->hPayload,
                &psObj->acBuffer[0], tNumBytesToRead);

            if (tNumBytesRead > 0)
            {
                // Write the data to the file system
                tNumBytesWritten = fwrite(
                    &psObj->acBuffer[0], sizeof(UN8),
                    tNumBytesRead, pFile );

                if (tNumBytesRead != tNumBytesWritten)
                {
                    // File accesses shouldn't fail -- we have to
                    // have access to storage for this service to be
                    // useful.  Consider this an error
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME
                        ": failed to write %d bytes (written %d) "
                        "to file",
                        tNumBytesRead, tNumBytesWritten);
                    bOk = FALSE;
                    break;
                }

                // Reduce the number of bytes we have left to read
                tSizeRemaining -= tNumBytesRead;
            }
            else
            {
                // We are unable to read from the buffer
                // for some reason
                bOk = FALSE;
                break;
            }
        }

        if (bOk == FALSE)
        {
            break;
        }

        //write tail of bits if it exists
        tSizeRemaining = OSAL.tBufferGetSizeInBits(psUnParsedAUDesc->hPayload);
        if (tSizeRemaining == 0)
        {
            bOk = FALSE;
            break;
        }

        OSAL.tBufferReadHead(psUnParsedAUDesc->hPayload,
            &psObj->acBuffer[0], 1);

        tNumBytesWritten = fwrite(
            &psObj->acBuffer[0], sizeof(UN8),
            1, pFile );

        if (tNumBytesWritten != 1)
        {
            // File accesses shouldn't fail -- we have to
            // have access to storage for this service to be
            // useful.  Consider this an error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to write last byte to file");
            bOk = FALSE;
            break;
        }

        psUnParsedAUDesc->un8NumTailBits = (UN8)tSizeRemaining;
        psUnParsedAUDesc->hPayload = OSAL_INVALID_BUFFER_HDL;

    } while (FALSE);

    // Close the file
    if (pFile != NULL)
    {
        fclose(pFile);
    }

    return bOk;
}

/*******************************************************************************
 *
 *      bGetPayloadFromFile
 *
 *******************************************************************************/
static BOOLEAN bGetPayloadFromFile(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_UNPARSED_AU_STRUCT *psUnParsedAUDesc
        )
{
    BOOLEAN bSuccess = TRUE;
    FILE *pFile = NULL;
    size_t tReadBytes;
    size_t tRemainBytes;
    size_t tWrittenBytes;
    size_t tWriteBits;
    size_t tWrittenBits;
    size_t tReadSize;
    size_t tNumBytesToRead;
    size_t tOsalBufferSize = 0;
    char acPath[WS_ALERTS1_SHARED_BUFFER_LEN];
    int iReturn;

    do
    {
        tOsalBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tOsalBufferSize > 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": provided payload buffer cannot be reused since "
                "is has not been fully read out");
            bSuccess = FALSE;
            break;
        }

        iReturn = snprintf(&acPath[0],
            WS_ALERTS1_SHARED_BUFFER_LEN,
            "%s/%s",
            psObj->pacServiceFilePath,
            psUnParsedAUDesc->acFilename);

        if (iReturn <= 0)
        {
            bSuccess = FALSE;
            break;
        }

        printf("acPath = %s\n", acPath);

        pFile = fopen(acPath, "rb");
        if (pFile == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to open file '%s' for binary reading",
                psObj->acBuffer);
            bSuccess = FALSE;
            break;
        }

        bSuccess = OSAL.bFileSystemGetFileSize(pFile,
            &tReadSize);
        if (bSuccess != TRUE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": cannot get data file size %s",
                psObj->acBuffer);
            break;
        }

        //It is possible we need not all bits of last byte in the file
        tRemainBytes = tReadSize - 1;

        // Read only requested no. of bytes.
        while (tRemainBytes > 0)
        {
            if (tRemainBytes > sizeof(psObj->acBuffer))
            {
                tNumBytesToRead = sizeof(psObj->acBuffer);
            }
            else
            {
                tNumBytesToRead = tRemainBytes;
            }

            tReadBytes = fread(&psObj->acBuffer[0], 1, tNumBytesToRead, pFile);
            if (tReadBytes > 0)
            {
                tWrittenBytes = OSAL.tBufferWriteTail(psObj->hBuffer,
                    &psObj->acBuffer[0], tReadBytes
                                              );

                if (tWrittenBytes != tReadBytes)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME
                        ": failed to write %d bytes (written %d) "
                        "to OSAL buffer",
                        tReadBytes, tWrittenBytes);
                    bSuccess = FALSE;
                    break;
                }
            }
            tRemainBytes -= tReadBytes;
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        //read tail
        tReadBytes = fread(&psObj->acBuffer[0], 1, 1, pFile);
        if (tReadBytes > 0)
        {
            if ( psUnParsedAUDesc->un8NumTailBits == 0 )
            {
                tWriteBits = 8;
            }
            else
            {
                tWriteBits = psUnParsedAUDesc->un8NumTailBits;
            }

            tWrittenBits = OSAL.tBufferWriteTailBits(psObj->hBuffer,
                &psObj->acBuffer[0], 0, tWriteBits);

            if (tWrittenBits != tWriteBits)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME
                    ": failed to write tail bits "
                    "into OSAL buffer");
                bSuccess = FALSE;
                break;
            }
        }
        else
        {
            bSuccess = FALSE;
            break;
        }

        psUnParsedAUDesc->hPayload = psObj->hBuffer;

    } while ( FALSE );

    if (pFile != NULL)
    {
        fclose(pFile);
    }

    remove( acPath );

    return bSuccess;
}

/*******************************************************************************
 *
 *      bGetFreeText
 *
 *******************************************************************************/
static BOOLEAN bGetFreeText(
    SMS_OBJECT hStringOwner,
    OSAL_BUFFER_HDL hPayload,
    STRING_OBJECT *phText,
    BOOLEAN bStringStartsWithLetters,
    BOOLEAN bStringStartsWithCaps
        )
{
    size_t tNumSymbolsFound = 0;
    BOOLEAN bResult = FALSE;

    if ( (hStringOwner == SMS_INVALID_OBJECT) ||
         (phText == NULL) )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": invalid input parameter");
        return bResult;
    }

    *phText = BAUDOT_hToString(
        hStringOwner,
        hPayload,
        BAUDOT_BEHAVIOR_PROCESS_TO_END,
        bStringStartsWithLetters,
        bStringStartsWithCaps,
        WS_ALERTS1_FREE_TEXT_ELEM_MAX_SYMBOLS,
        &tNumSymbolsFound );

    if (*phText != STRING_INVALID_OBJECT)
    {
        bResult = TRUE;
    }
    else
    {
        //something went wrong
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": failed to get text string");
    }

    return bResult;
}

/*****************************************************************************
 *
 *   psWSAlertsGetStateMsgDesc
 *
 *****************************************************************************/
static WS_ALERTS1_STATE_MSG_DESC_STRUCT *psWSAlertsGetStateMsgDesc(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    BOOLEAN *pbStateMsgDescChanged
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_OBJECT_HDL hStateMsgList = OSAL_INVALID_OBJECT_HDL;
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8StateId;
    WS_ALERTS1_AU_DATA_STRUCT *psAUData = NULL;
    WS_ALERTS1_STATE_MSG_DESC_STRUCT sDummyStateMsg;

    do
    {
        if (psObj == NULL)
        {
            printf("psWSAlertsGetStateMsgDesc: psObj is undefined\n");
            break;
        }

        psAUData = &(psObj->sAUData);

        if ( psAUData->eCarouselID == WS_ALERTS1_HIGH_RATE_MSG )
        {
            hStateMsgList = psObj->hStateMsgHighList;
        }
        else if ( psAUData->eCarouselID == WS_ALERTS1_NORMAL_RATE_MSG )
        {
            hStateMsgList = psObj->hStateMsgNormalList;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": unknown CarId. Failed to create state message desc");
            break;
        }

        un8StateId = 
            WS_ALERTS1_GENERATE_STATE_ID(psAUData->bSTATEFLAG, psAUData->un8STATE);
        sDummyStateMsg.un8StateId = un8StateId;

        // Look up the message descriptor by state id
        eOsalReturnCode = OSAL.eLinkedListSearch(
            hStateMsgList,
            &hEntry,
            (void*)&sDummyStateMsg);
        // Check error codes
        if (eOsalReturnCode == OSAL_SUCCESS)
        {
            psStateMsgDesc = (WS_ALERTS1_STATE_MSG_DESC_STRUCT*)OSAL.pvLinkedListThis(hEntry);

            if ( psStateMsgDesc == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME
                    ": failed to create state message desc with state_id %d",
                    un8StateId);
                break;
            }
        }
        else if ( (eOsalReturnCode == OSAL_OBJECT_NOT_FOUND) ||
                  (eOsalReturnCode == OSAL_NO_OBJECTS) )
        {
            psStateMsgDesc = (WS_ALERTS1_STATE_MSG_DESC_STRUCT *)
                SMSO_hCreate(
                    WS_ALERTS1_OBJECT_NAME":StateMsg",
                    sizeof(WS_ALERTS1_STATE_MSG_DESC_STRUCT),
                    (SMS_OBJECT)psObj, FALSE);

            if (psStateMsgDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME
                    ": failed to create state message desc with state_id %d",
                    un8StateId);
                break;
            }
            psStateMsgDesc->un8StateId = un8StateId;
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to get state message desc with state_id %d",
                un8StateId);
            break;
        }

        if (psStateMsgDesc->hUnParsedAUsList == OSAL_INVALID_OBJECT_HDL)
        {
            psStateMsgDesc->un8Version = psAUData->un8STVER;
            psStateMsgDesc->un16AUTotal = psAUData->un16AUTOT;
            psStateMsgDesc->un16EPOCH = psAUData->un16EPOCH;
            psStateMsgDesc->bProcessed = FALSE;

            eReturnCode = OSAL.eLinkedListCreate(
                &psStateMsgDesc->hUnParsedAUsList,
                WS_ALERTS1_OBJECT_NAME":UnParsedAUsList",
                (OSAL_LL_COMPARE_HANDLER)n16CompareAUsByAUCT,
                OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE);

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to create list StateMsgList (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
                vReleaseStateMsg( psObj, psStateMsgDesc );
                psStateMsgDesc = NULL;
                break;
            }

            eReturnCode = OSAL.eLinkedListAdd(hStateMsgList,
                NULL, psStateMsgDesc);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME
                    ": failed to add StateMsgDesc 0x%p to the list (%s)",
                    psStateMsgDesc, OSAL.pacGetReturnCodeName(eReturnCode));

                vReleaseStateMsg( psObj, psStateMsgDesc );
                psStateMsgDesc = NULL;
                break;
            }

            *pbStateMsgDescChanged = TRUE;
        }

    } while (FALSE);

    return psStateMsgDesc;
}

/*****************************************************************************
 *
 *   vReleaseStateMsgList
 *
 *****************************************************************************/
static void vReleaseStateMsgList (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL *phStateMsgList
        )
{
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hNextEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psEntry = NULL;

    if( (psObj != NULL) && (phStateMsgList != NULL) &&
        (*phStateMsgList != OSAL_INVALID_OBJECT_HDL) )
    {
        hEntry = OSAL.hLinkedListFirst( *phStateMsgList, NULL );
        while( hEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            hNextEntry = OSAL.hLinkedListNext( hEntry, NULL );
            psEntry = ( WS_ALERTS1_STATE_MSG_DESC_STRUCT* )
                                            OSAL.pvLinkedListThis( hEntry );

            vReleaseStateMsg( psObj, psEntry );
            hEntry = hNextEntry;
        }

        eReturnCode = OSAL.eLinkedListRemoveAll(*phStateMsgList, NULL);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME":failed to clean up list (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode = OSAL.eLinkedListDelete(*phStateMsgList);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to delete "
                "list. (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }

        *phStateMsgList = OSAL_INVALID_OBJECT_HDL;
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseStateMsg
 *
 *****************************************************************************/
static void vReleaseStateMsg (
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc
    )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if (psStateMsgDesc != NULL)
    {
        if (psStateMsgDesc->hUnParsedAUsList != OSAL_INVALID_OBJECT_HDL)
        {
            vReleaseUnParsedAUsList( psObj, psStateMsgDesc->hUnParsedAUsList );

            eReturnCode = OSAL.eLinkedListDelete(psStateMsgDesc->hUnParsedAUsList);

            if (eReturnCode == OSAL_SUCCESS)
            {
                psStateMsgDesc->hUnParsedAUsList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psStateMsgDesc);

    }
    return;
}

/*****************************************************************************
 *
 *   vReleaseUnParsedAUsList
 *
 *****************************************************************************/
static void vReleaseUnParsedAUsList(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    OSAL_OBJECT_HDL hAUList
        )
{
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hNextEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_ERROR;
    WS_ALERTS1_UNPARSED_AU_STRUCT *psEntry = NULL;

    if( (psObj != NULL) && (hAUList != OSAL_INVALID_OBJECT_HDL) )
    {
        hEntry = OSAL.hLinkedListFirst( hAUList, NULL );
        while( hEntry != OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            hNextEntry = OSAL.hLinkedListNext( hEntry, NULL );
            psEntry = ( WS_ALERTS1_UNPARSED_AU_STRUCT* )
                OSAL.pvLinkedListThis( hEntry );

            vReleaseUnParsedAU( psObj, psEntry );
            hEntry = hNextEntry;
        }

        eOsalReturnCode = OSAL.eLinkedListRemoveAll(hAUList, NULL);
        if ((eOsalReturnCode != OSAL_SUCCESS) &&
            (eOsalReturnCode != OSAL_NO_OBJECTS))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME":failed to clean up list (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode));
        }
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseUnParsedAU
 *
 *****************************************************************************/
static void  vReleaseUnParsedAU(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_UNPARSED_AU_STRUCT *psAuDesc
        )
{
    BOOLEAN bValid;
    char acPath[WS_ALERTS1_SHARED_BUFFER_LEN];
    int iReturn;

    bValid = SMSO_bValid((SMS_OBJECT)psAuDesc);
    if ( bValid == TRUE )
    {
        //remove payload file
        if ( psAuDesc->acFilename[0] != '\0' )
        {
            iReturn = snprintf(&acPath[0],
                WS_ALERTS1_SHARED_BUFFER_LEN,
                "%s/%s",
                psObj->pacServiceFilePath,
                psAuDesc->acFilename);

            if (iReturn <= 0)
            {
                return;
            }

            remove( acPath );
        }

        psAuDesc->hPayload = OSAL_INVALID_BUFFER_HDL;

        // Free memory
        SMSO_vDestroy((SMS_OBJECT)psAuDesc);
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseParsedAU
 *
 *****************************************************************************/
static void vReleaseParsedAU(
    WS_ALERTS1_PARSED_AU_STRUCT *psAuDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bValid;

    bValid = SMSO_bValid((SMS_OBJECT)psAuDesc);
    if (bValid == TRUE)
    {
        printf(WS_ALERTS1_OBJECT_NAME": attempting to destroy Au Desc %p\n",
            psAuDesc);

        if (psAuDesc->hMsgElementsList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psAuDesc->hMsgElementsList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseMsgElement);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListDelete(psAuDesc->hMsgElementsList);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    psAuDesc->hMsgElementsList = OSAL_INVALID_OBJECT_HDL;
                }
                else
                {
                    puts(WS_ALERTS1_OBJECT_NAME": Error deleting hMsgElementsList");
                }
            }
            else
            {
                puts(WS_ALERTS1_OBJECT_NAME": Error removing entries from hMsgElementsList\n");
            }
        }
        // Free memory
        SMSO_vDestroy((SMS_OBJECT)psAuDesc);
    }

    return;
}

/*****************************************************************************
 *
 *   vReleaseMsgElement
 *
 *****************************************************************************/
static void vReleaseMsgElement (
    WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *psMsgElement
        )
{
    if (psMsgElement != NULL)
    {
        if (psMsgElement->tINDEX == 0)
        {
            switch (psMsgElement->tPAR)
            {
                case WS_ALERTS1_TEXT_LTR_ELEM:
                case WS_ALERTS1_TEXT_LTR_LC_ELEM:
                case WS_ALERTS1_TEXT_FIG_ELEM:
                {
                    if (psMsgElement->uData.hFreeText != STRING_INVALID_OBJECT)
                    {
                        STRING_vDestroy(psMsgElement->uData.hFreeText);
                        psMsgElement->uData.hFreeText = STRING_INVALID_OBJECT;
                    }
                }
                break;

                case WS_ALERTS1_PHONEME_ELEM:
                {
                    if (psMsgElement->uData.hPhoneme != STRING_INVALID_OBJECT)
                    {
                        STRING_vDestroy(psMsgElement->uData.hPhoneme);
                        psMsgElement->uData.hPhoneme = STRING_INVALID_OBJECT;
                    }
                }
                break;

                case WS_ALERTS1_FILE_LINK_ELEM:
                {
                    if (psMsgElement->uData.hFileLink != STRING_INVALID_OBJECT)
                    {
                        STRING_vDestroy(psMsgElement->uData.hFileLink);
                        psMsgElement->uData.hFileLink = STRING_INVALID_OBJECT;
                    }
                }
                break;

                case WS_ALERTS1_PHONE_NUMBER_ELEM:
                {
                    if (psMsgElement->uData.hPhoneNum != STRING_INVALID_OBJECT)
                    {
                        STRING_vDestroy(psMsgElement->uData.hPhoneNum);
                        psMsgElement->uData.hPhoneNum = STRING_INVALID_OBJECT;
                    }
                }
                break;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psMsgElement);
    }

    return;
}

/*****************************************************************************
*
*   bBuildServiceFilePath
*
*****************************************************************************/
static BOOLEAN bBuildServiceFilePath(
    WS_ALERTS1_OBJECT_STRUCT *psObj
        )
{
    size_t tSMSPathLen = 0;
    const char *pacSMSPath = NULL;
    size_t tPathLen = 0;
    BOOLEAN bOk = FALSE;
    int iReturn;

    do
    {
        pacSMSPath = SMS_pacGetPath();
        if (pacSMSPath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to get SMS Path");
            break;
        }

        tSMSPathLen = strlen(pacSMSPath);

        tPathLen = tSMSPathLen +
            strlen(WS_ALERTS_SERVICE_FOLDER) + 2;

        // Now we know the size of the path, so we can now
        // allocate the proper amount of memory
        psObj->pacServiceFilePath =
            (char *)SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":ServiceFilePath",
                tPathLen + 1,
                (SMS_OBJECT)psObj,
                FALSE);

        // Ensure allocation succeeded
        if (psObj->pacServiceFilePath == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to allocate "
                "service path memory");
            break;
        }

        // Construct the full location database filename
        iReturn = snprintf(psObj->pacServiceFilePath,
            tPathLen,
            "%s/%s",
            pacSMSPath,
            WS_ALERTS_SERVICE_FOLDER);

        if (iReturn <= 0)
        {
            break;
        }

        printf(WS_ALERTS1_OBJECT_NAME": service path created: %s\n",
            psObj->pacServiceFilePath );

        bOk = TRUE;

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
 *
 *   n16WSAlertsFindByStateId
 *
 *****************************************************************************/
static N16 n16WSAlertsFindByStateId (
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsg1,
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsg2
        )
{
    N16 n16Result = N16_MIN;

    if((psStateMsg1 != NULL) && (psStateMsg2 != NULL))
    {
        n16Result = 
            COMPARE(psStateMsg1->un8StateId, psStateMsg2->un8StateId);
    }

    return n16Result;
}

/*******************************************************************************
 *
 *   n16CompareAUsByAUCT
 *
 *******************************************************************************/
static N16 n16CompareAUsByAUCT(
    WS_ALERTS1_UNPARSED_AU_STRUCT *psObj1,
    WS_ALERTS1_UNPARSED_AU_STRUCT *psObj2
        )
{
    N16 n16Result = N16_MIN;

    if ((psObj1 != NULL) && (psObj2 != NULL))
    {
        n16Result = COMPARE(psObj1->un16AUCT, psObj2->un16AUCT);
    }

    return n16Result;
}

/*******************************************************************************
 *
 *   bProcessAU
 *
 *******************************************************************************/
static BOOLEAN bProcessAU(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc,
    BOOLEAN bRemoveMsgsFromDB
        )
{
    WS_ALERTS1_MSG_ELEM_ITERATOR_STRUCT sIterator;
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = FALSE;

    do
    {
        psMsgSetDesc = (WS_ALERTS_MSG_SET_DESC_STRUCT*)
            SMSO_hCreate(WS_ALERTS1_OBJECT_NAME":MsgSetDesc",
            sizeof(*psMsgSetDesc), (SMS_OBJECT)psObj, FALSE
                );
        if (psMsgSetDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot create Message Set desc struct");
            bSuccess = FALSE;
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&(psMsgSetDesc->hMsgElementsList),
            WS_ALERTS1_OBJECT_NAME":MsgElementsList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_NONE );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create MsgElementsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
            break;
        }

        psMsgSetDesc->un8StateId = psStateMsgDesc->un8StateId;
        psMsgSetDesc->eLanguage = psStateMsgDesc->eLANG;
        psMsgSetDesc->un8Version = psStateMsgDesc->un8Version;

        if ( psObj->sAUData.eCarouselID == WS_ALERTS1_HIGH_RATE_MSG )
        {
            psMsgSetDesc->bIsHighRate = TRUE;
        }
        else
        {
            psMsgSetDesc->bIsHighRate = FALSE;
        }

        sIterator.psObj = psObj;
        sIterator.psStateMsgDesc = psStateMsgDesc;
        sIterator.psMsgSetDesc = psMsgSetDesc;
        sIterator.bResult = FALSE;

        eReturnCode = OSAL.eLinkedListIterate(
            psObj->psParsedAUDesc->hMsgElementsList,
            ( OSAL_LL_ITERATOR_HANDLER )bIterateMsgElements, &sIterator );
        if( (eReturnCode != OSAL_SUCCESS) ||
            (sIterator.bResult != TRUE) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot iterate Message Elements ist");
            bSuccess = FALSE;
            break;
        }

        bSuccess = GsWSAlertsMgrIntf.bConstructMessages(
            psObj->hWSAlertsService,
            psMsgSetDesc,
            bRemoveMsgsFromDB);

    }
    while (FALSE);

    vReleaseMsgSet(psMsgSetDesc);

    return bSuccess;
}

/*****************************************************************************
 *
 *   bIterateMsgElements
 *
 *****************************************************************************/
static BOOLEAN bIterateMsgElements(
    WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *psPvnMsgElem,
    WS_ALERTS1_MSG_ELEM_ITERATOR_STRUCT *psIterator
        )
{
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMgrMsgElem = NULL;
    WS_ALERTS1_TABLE_DESC_STRUCT *psLookUpTable = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    UN32 un32LutId = 0;
    BOOLEAN bSuccess = FALSE;
    int iReturn;

    if ((psPvnMsgElem == NULL) ||
        (psIterator == NULL))
    {
        return FALSE;
    }

    do
    {
        psMgrMsgElem = (WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT*)
            SMSO_hCreate(WS_ALERTS1_OBJECT_NAME":MsgElemDesc",
                sizeof(*psMgrMsgElem), (SMS_OBJECT)psIterator->psObj, FALSE);
        if (psMgrMsgElem == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot create Message Element desc");
            break;
        }

        psLookUpTable = psGetTable(psIterator->psObj, psPvnMsgElem->tINDEX);
        if( psLookUpTable == NULL )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Invalid Look Up Table");
            break;
        }

        if ( psPvnMsgElem->tINDEX > 0 )
        {
            char acBuf[WS_ALERTS1_SHARED_BUFFER_LEN];
            size_t tLength = 0;

            psMgrMsgElem->bIsRegularElement = TRUE;

            iReturn = snprintf( acBuf,
                WS_ALERTS1_SHARED_BUFFER_LEN,
                WS_ALERTS_LUT_NAME,
                psLookUpTable->tCLASS,
                psLookUpTable->tTABID,
                psIterator->psStateMsgDesc->eLANG );

            if (iReturn <= 0)
            {
                break;
            }

            tLength = strlen(acBuf);
            psMgrMsgElem->uType.sLutDesc.hDecodeTableName = STRING.hCreate(acBuf, tLength);

            un32LutId = WS_ALERTS_GET_LUT_ID(
                (UN8)psLookUpTable->tCLASS,
                (UN8)psLookUpTable->tTABID,
                (UN8)psIterator->psStateMsgDesc->eLANG );

            psMgrMsgElem->uType.sLutDesc.un32DecodeTableId = un32LutId;
            psMgrMsgElem->uType.sLutDesc.un8MinVersion = (UN8)psLookUpTable->tTVMIN;

            switch( psLookUpTable->tCLASS )
            {
                case WS_ALERTS1_LUT_CLASS_ALERT_TYPE:
                case WS_ALERTS1_LUT_CLASS_COMMON_PHRASE_CODE:
                {
                    psMgrMsgElem->uType.sLutDesc.un16EntryId = psPvnMsgElem->tPAR;
                }
                break;

                case WS_ALERTS1_LUT_CLASS_ALERT_LOCATION:
                {
                    psMgrMsgElem->uType.sLutDesc.un16EntryId = psLookUpTable->tTABINDEX;
                    psMgrMsgElem->uType.sLutDesc.un16EntryId2 = psPvnMsgElem->tPAR;
                }
                break;

                default:
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": Unknown LUT Class");
                }
                break;
            }
        }
        else if( psPvnMsgElem->tINDEX == 0 )
        {
            psMgrMsgElem->bIsRegularElement = FALSE;

            bSuccess = bFillMsgData(psIterator->psObj,
                psIterator->psStateMsgDesc,
                psPvnMsgElem,
                psMgrMsgElem,
                psLookUpTable);

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to fill message data");
                break;
            }
        }

        if( psPvnMsgElem->tFMT & WS_ALERTS1_PARSER_FORMAT_CAP_MASK )
        {
            psMgrMsgElem->sTextFormat.bIsCapital = TRUE;
        }

        switch( psPvnMsgElem->tFMT & WS_ALERTS1_PARSER_FORMAT_TAIL_MASK )
        {
            case WS_ALERTS1_PARSER_FORMAT_TAIL_SPACE:
            {
                psMgrMsgElem->sTextFormat.eTail = WS_ALERTS_MSG_ELEM_TAIL_SPACE;
            }
            break;

            case WS_ALERTS1_PARSER_FORMAT_TAIL_COMMA_SPACE:
            {
                psMgrMsgElem->sTextFormat.eTail = WS_ALERTS_MSG_ELEM_TAIL_COMMA_SPACE;
            }
            break;

            case WS_ALERTS1_PARSER_FORMAT_TAIL_PERIOD_SPACE:
            {
                psMgrMsgElem->sTextFormat.eTail = WS_ALERTS_MSG_ELEM_TAIL_PERIOD_SPACE;
            }
            break;

            case WS_ALERTS1_PARSER_FORMAT_TAIL_NO_SUFFIX:
            {
                psMgrMsgElem->sTextFormat.eTail = WS_ALERTS_MSG_ELEM_TAIL_NO_SUFFIX;
            }
            break;

            default:
                break;
        }

        if( psPvnMsgElem->tFMT & WS_ALERTS1_PARSER_FORMAT_BREAK_MASK )
        {
            psMgrMsgElem->sTextFormat.bIsBrake = TRUE;
        }

        eReturnCode = OSAL.eLinkedListAdd( psIterator->psMsgSetDesc->hMsgElementsList,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR, ( void * )psMgrMsgElem );

        if( eReturnCode != OSAL_SUCCESS )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot add entry to list (%s)",
                OSAL.pacGetReturnCodeName( eReturnCode ));
            break;
        }
        bSuccess = TRUE;

    }
    while (FALSE);

    if ( (bSuccess == FALSE) &&
         (psMgrMsgElem != NULL) )
    {
        vDestroyMessageElement(psMgrMsgElem);
    }

    psIterator->bResult = bSuccess;

    return bSuccess;
}

/*****************************************************************************
 *
 *   bFillMsgData
 *
 *****************************************************************************/
static BOOLEAN bFillMsgData(
    WS_ALERTS1_OBJECT_STRUCT *psObj,
    WS_ALERTS1_STATE_MSG_DESC_STRUCT *psStateMsgDesc,
    WS_ALERTS1_MESSAGE_ELEMENT_STRUCT *psPvnMsgElem,
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMgrMsgElem,
    WS_ALERTS1_TABLE_DESC_STRUCT *psLookUpTable
        )
{
    if ( (psObj == NULL) ||
         (psStateMsgDesc == NULL) ||
         (psPvnMsgElem == NULL) ||
         (psLookUpTable == NULL) )
    {
        return FALSE;
    }

    switch( psPvnMsgElem->tPAR )
    {
        case WS_ALERTS1_MESSAGE_DELIMITER_ELEM:
        {
            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_MSG_DELIMITER;
            psMgrMsgElem->uData.un16MsgSignature = psPvnMsgElem->uData.un16MSGSIG;
        }
        break;

        case WS_ALERTS1_START_TIME_ELEM:
        {
            psMgrMsgElem->uData.tEventTime = tGetDateTime(
                psStateMsgDesc->un16EPOCH,
                psObj->psParsedAUDesc->un8ETIME,
                psPvnMsgElem->uData.un16TimeOffset);

            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_START_TIME_ELEM;
        }
        break;

        case WS_ALERTS1_END_TIME_ELEM:
        {
            psMgrMsgElem->uData.tEventTime = tGetDateTime(
                psStateMsgDesc->un16EPOCH,
                psObj->psParsedAUDesc->un8ETIME,
                psPvnMsgElem->uData.un16TimeOffset);

            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_END_TIME_ELEM;
        }
        break;

        case WS_ALERTS1_GENERIC_TIME_ELEM:
        {
            psMgrMsgElem->uData.tEventTime = tGetDateTime(
                psStateMsgDesc->un16EPOCH,
                psObj->psParsedAUDesc->un8ETIME,
                psPvnMsgElem->uData.un16TimeOffset);

            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_GENERIC_TIME_ELEM;
        }
        break;

        case WS_ALERTS1_POLYGON_ELEM:
        {
            WS_ALERTS_POLYGON_DESC_STRUCT *psPolygon = NULL;
            UN32 un32Index = 0;

            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_POLYGON_ELEM;

            OSAL.bMemSet(&psMgrMsgElem->uData.sPolygonDesc, 0,
                sizeof(psMgrMsgElem->uData.sPolygonDesc));

            psPolygon = &psMgrMsgElem->uData.sPolygonDesc;

            // A polygon is described by an Anchor point, LOCNO+1 Delta points
            // and duplicated Anchor (the same start and end points are needed
            // for algorithms working with locations).
            // Each point is described by 2 values: lon and lat
            // So we need an array of (1+LOCNO+1+1)*2 elements to store those values
            psPolygon->tLonLatArraySize =
                2 * ( psPvnMsgElem->uData.sPolygon.un8LOCNO + 3 );
            psPolygon->pn32PolygonLonLat =
                ( N32 * )
                SMSO_hCreate(
                    WS_ALERTS1_OBJECT_NAME":MsgElement:Polygon:LonLat",
                    sizeof(UN32)
                    * ( psPolygon->tLonLatArraySize ),
                    ( SMS_OBJECT )psObj, FALSE );

            if( psPolygon->pn32PolygonLonLat == NULL )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot allocate memory for"
                    " polygon coordinates");
                SMSO_vDestroy( ( SMS_OBJECT )psPolygon );
                break;
            }

            //LONGITUDE is -178..-50 degrees
            //LATITUDE is 0..90 degrees
            //see SX-9845-0039 section 5.2.3.5.2 for details
            psPolygon->pn32PolygonLonLat[0] =
                WS_ALERTS1_POLYGON_ANCHOR_LON_INT(
                    psPvnMsgElem->uData.sPolygon.un16ANCHORLon );
            psPolygon->pn32PolygonLonLat[1] =
                WS_ALERTS1_POLYGON_ANCHOR_LAT_INT(
                    psPvnMsgElem->uData.sPolygon.un16ANCHORLat );
            psPolygon->n32MinLon = psPolygon->pn32PolygonLonLat[0];
            psPolygon->n32MaxLon = psPolygon->pn32PolygonLonLat[0];
            psPolygon->n32MinLat = psPolygon->pn32PolygonLonLat[1];
            psPolygon->n32MaxLat = psPolygon->pn32PolygonLonLat[1];

            for( un32Index = 1;
                 un32Index <= (UN32)psPvnMsgElem->uData.sPolygon.un8LOCNO + 1;
                 un32Index++ )
            {
                psPolygon->pn32PolygonLonLat[2 * un32Index] =
                    psPolygon->pn32PolygonLonLat[2 * un32Index - 2] +
                    WS_ALERTS1_POLYGON_DELTAP_LON_INT(
                        psPvnMsgElem->uData.sPolygon.n16DELTAPLon[un32Index - 1] );
                psPolygon->pn32PolygonLonLat[2 * un32Index + 1] =
                    psPolygon->pn32PolygonLonLat[2 * un32Index - 1] +
                    WS_ALERTS1_POLYGON_DELTAP_LAT_INT(
                        psPvnMsgElem->uData.sPolygon.n16DELTAPLat[un32Index - 1] );

                //choose min and max coords
                if(psPolygon->n32MinLon > psPolygon->pn32PolygonLonLat[2 * un32Index])
                {
                    psPolygon->n32MinLon = psPolygon->pn32PolygonLonLat[2 * un32Index];
                }

                if(psPolygon->n32MaxLon < psPolygon->pn32PolygonLonLat[2 * un32Index])
                {
                    psPolygon->n32MaxLon = psPolygon->pn32PolygonLonLat[2 * un32Index];
                }

                if(psPolygon->n32MinLat > psPolygon->pn32PolygonLonLat[2 * un32Index + 1])
                {
                    psPolygon->n32MinLat = psPolygon->pn32PolygonLonLat[2 * un32Index + 1];
                }

                if(psPolygon->n32MaxLat < psPolygon->pn32PolygonLonLat[2 * un32Index + 1])
                {
                    psPolygon->n32MaxLat = psPolygon->pn32PolygonLonLat[2 * un32Index + 1];
                }
            }

            // duplicate the first point in the last element
            //for correct work of location filtration algorithm
            psPolygon->pn32PolygonLonLat[ 2 * ( psPvnMsgElem->uData.sPolygon.un8LOCNO + 2 ) ] =
                psPolygon->pn32PolygonLonLat[ 0 ];
            psPolygon->pn32PolygonLonLat[ 2 * ( psPvnMsgElem->uData.sPolygon.un8LOCNO + 2 ) + 1 ] =
                psPolygon->pn32PolygonLonLat[ 1 ];

        }
        break;

        case WS_ALERTS1_TEXT_LTR_ELEM:
        case WS_ALERTS1_TEXT_LTR_LC_ELEM:
        case WS_ALERTS1_TEXT_FIG_ELEM:
        {
            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_TEXT_ELEM;
            psMgrMsgElem->uData.hText = psPvnMsgElem->uData.hFreeText;
        }
        break;

        case WS_ALERTS1_PHONE_NUMBER_ELEM:
        {
            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_PHONE_NUMBER_ELEM;
            psMgrMsgElem->uData.hText = psPvnMsgElem->uData.hPhoneNum;
        }
        break;

        case WS_ALERTS1_SYSTEM_PRIORITY_ELEM:
        {
            psMgrMsgElem->uType.eType = WS_ALERTS_MSG_ELEM_TYPE_SYSTEM_PRIORITY_ELEM;
            psMgrMsgElem->uData.un8Priority = psPvnMsgElem->uData.un8SPRI;
        }
        break;

        case WS_ALERTS1_PHONEME_ELEM:
        case WS_ALERTS1_LARGE_CONTAINER_ELEM:
        case WS_ALERTS1_SMALL_CONTAINER_ELEM:
        default:
            break;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *   tGetDateTimeString
 *
 *****************************************************************************/
static TIME_T tGetDateTime(
    UN16 un16Epoch,
    UN8 un8ETime,
    UN16 un16Offset
        )
{
    TIME_T tSecs;

    tSecs = WS_ALERTS1_EPOCH_TO_SECONDS_COEFF * un16Epoch +
            WS_ALERTS1_ETIME_TO_SECONDS_COEFF * un8ETime +
            WS_ALERTS1_TIME_OFFSET_TO_SECONDS_COEFF * un16Offset;

    return tSecs;
}

/*****************************************************************************
*
*   vReleaseMsgSet
*
*****************************************************************************/
static void vReleaseMsgSet(
    WS_ALERTS_MSG_SET_DESC_STRUCT *psMsgSetDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psMsgSetDesc != NULL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psMsgSetDesc->hMsgElementsList,
            (OSAL_LL_RELEASE_HANDLER) vDestroyMessageElement);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to cleanup"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        }

        eReturnCode = OSAL.eLinkedListDelete(psMsgSetDesc->hMsgElementsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to delete"
                " list (%s)", OSAL.pacGetReturnCodeName(eReturnCode));
        }

        psMsgSetDesc->hMsgElementsList = OSAL_INVALID_OBJECT_HDL;

        SMSO_vDestroy((SMS_OBJECT)psMsgSetDesc);
    }

    return;
}

/*****************************************************************************
 *
 *   vDestroyMessageElement
 *
 *****************************************************************************/
static void vDestroyMessageElement (
    WS_ALERTS_MGR_MESSAGE_ELEMENT_STRUCT *psMsgElement
        )
{
    if (psMsgElement->bIsRegularElement == TRUE)
    {
        if (psMsgElement->uType.sLutDesc.hDecodeTableName != STRING_INVALID_OBJECT)
        {
            STRING.vDestroy(psMsgElement->uType.sLutDesc.hDecodeTableName);
        }
    }
    else
    {
        switch (psMsgElement->uType.eType)
        {
            case WS_ALERTS_MSG_ELEM_TYPE_TEXT_ELEM:
            {
                if(psMsgElement->uData.hText != STRING_INVALID_OBJECT)
                {
                    STRING.vDestroy(psMsgElement->uData.hText);
                }
            }
            break;

            case WS_ALERTS_MSG_ELEM_TYPE_POLYGON_ELEM:
            {
                if(psMsgElement->uData.sPolygonDesc.pn32PolygonLonLat)
                {
                    SMSO_vDestroy(
                        (SMS_OBJECT)psMsgElement->uData.sPolygonDesc.pn32PolygonLonLat);
                }
            }
            break;

            default:
                break;
        }
    }

    SMSO_vDestroy((SMS_OBJECT)psMsgElement);

    return;
}

/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static WS_ALERTS1_RFD_CTRL_STRUCT *psCreateRFDCtrl(
    WS_ALERTS1_OBJECT_STRUCT *psObj
        )
{
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (WS_ALERTS1_RFD_CTRL_STRUCT*)
        SMSO_hCreate(
        WS_ALERTS1_OBJECT_NAME":RFDCtrl",
        sizeof(*psRFDCtrl),
        SMS_INVALID_OBJECT, FALSE );
    if (psRFDCtrl == NULL)
    {
        return (WS_ALERTS1_RFD_CTRL_STRUCT*)NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hWSAlertsService = psObj->hWSAlertsService;

    return psRFDCtrl;
}

/*****************************************************************************
*
*   eRFDUpdateNeeded
*
*****************************************************************************/
static RFD_FILE_STATUS_ENUM eRFDUpdateNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        )
{
    RFD_FILE_STATUS_ENUM eFileStatus = RFD_FILE_STATUS_INVALID;

    if ( pvCallbackArg != NULL )
    {
        WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl =
            (WS_ALERTS1_RFD_CTRL_STRUCT *)pvCallbackArg;
        RFD_UPDATE_VERSION tBaseVersionUnused = 0;
        BOOLEAN bOk = FALSE;

        // Get the version info from the file name using
        // version extractor function
        bOk = bGetFileVersionFromName(pcFileName, ptFileVersion);

        // Did we extract that information without error?
        if ( bOk == TRUE )
        {
            // WS Alerts doesn't compare against a base version,
            // so we call RFD_INTERFACE_eCompareFileVersions with
            // RFD_INTERFACE_BASELINE_CHECK_SKIP as the last argument
            // to disable the baseline check.

            // Determine if this update is needed
            eFileStatus = RFD_INTERFACE_eCompareFileVersions(
                tBaseVersionUnused, psRFDCtrl->tCurrentVersion,
                *ptFileVersion, tVersionBitWidth,
                RFD_INTERFACE_BASELINE_CHECK_SKIP );
        }
    }

    return eFileStatus;

}

/*****************************************************************************
 *
 *       bGetFileVersionFromName
 *
 *       This API is used to parse a file name from an RFD file to determine
 *       file version.  Filenames parsed by this function must be
 *       in the form UYYY where YYY is the version the file will be
 *       upgraded to.
 *
 *****************************************************************************/
static BOOLEAN bGetFileVersionFromName(
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptNewVersion
        )
{
    size_t tFileNameStrLen = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Verify that the length of pcFileName is 4
        tFileNameStrLen = strlen(pcFileName);

        if (tFileNameStrLen != WS_ALERTS1_RFD_UPDATE_NAME_LEN)
        {
            break;
        }

        // Verify that the first character is 'U'
        if (pcFileName[0] != WS_ALERTS1_RFD_UPDATE_TOKEN)
        {
            break;
        }

        *ptNewVersion = (RFD_UPDATE_VERSION)atoi(&pcFileName[1]);
        bResult = TRUE;

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eRFDFileProcessor
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl = (WS_ALERTS1_RFD_CTRL_STRUCT*)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk;

        // Initialize the RFD Ctrl object and lock it
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex);

        if (bOk == TRUE)
        {
            // Process the update header now
            bOk = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Unable to process RFD file header");

            return RFD_PROCESS_RESULT_ERROR;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        // Free associated memory if we're stopping
        vDestroyRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion);
    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult >= RFD_PROCESS_RESULT_ERROR)
        {
            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   vUninitRFDCtrl
*
*****************************************************************************/
static void vUninitRFDCtrl (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    if (psRFDCtrl->hPartsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psRFDCtrl->hPartsList,
            (OSAL_LL_RELEASE_HANDLER)vReleasePartDesc);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to clean up PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
        eReturnCode = OSAL.eLinkedListDelete(psRFDCtrl->hPartsList);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to delete PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
        psRFDCtrl->hPartsList = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    return;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        )
{
    BOOLEAN bResult = FALSE;
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCurDB = STRING_INVALID_OBJECT;

    do
    {
        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = 0;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psRFDCtrl->hBlockPool,
            &psRFDCtrl->hBuffer,
            WS_ALERTS1_RFD_READ_BLOCK_SIZE);

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create file reader buffer");
            break;
        }

        // Find out where to place our working copy
        bSuccess = GsWSAlertsMgrIntf.bRefDBBank(
            psRFDCtrl->hWSAlertsService, &hCurDB, &psRFDCtrl->hDBPath);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting update from the beginning?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB));
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        // if update starts from the beginning, need to prepare DB
        if (tStartingIndex == 0)
        {
            bSuccess = GsWSAlertsMgrIntf.bDBUpdateBegin(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->acSQLCommandBuffer,
                WS_ALERTS_MAX_SQL_STRING_LENGTH);
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        bResult = TRUE;
    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    return bResult;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psRFDFile
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        size_t tBitsRead;

        // Fill the buffer with file data
        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
            psRFDFile, psRFDCtrl->hBuffer);

        if (bSuccess == FALSE)
        {
            // Couldn't access the file
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": Unable to read file data to buffer");
            break;
        }

        // Read flags
        tBitsRead = OSAL.tBufferReadHeadBits (
            psRFDCtrl->hBuffer, NULL, 0,
            WS_ALERTS1_RFD_FLAGS_BITLEN );

        if (tBitsRead != WS_ALERTS1_RFD_FLAGS_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot read flags from buffer");
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(&(psRFDCtrl->hPartsList),
            WS_ALERTS1_OBJECT_NAME":PartsList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR|OSAL_LL_OPTION_UNIQUE );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create PartsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
            break;
        }

        bSuccess = bGetPartsList(psRFDCtrl, psRFDCtrl->hPartsList);
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to get partitions list");
            break;
        }

        // Time to clear out our buffer -- just seek past everything
        OSAL.tBufferSeekHeadBits(psRFDCtrl->hBuffer, N32_MAX);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult;
    BOOLEAN bSuccess = TRUE;

    // Process the next group
    eResult = eProcessUpdateGroup(psRFDCtrl, psFile);

    if (eResult != RFD_PROCESS_RESULT_ERROR)
    {
        // Increment our progress index
        psRFDCtrl->tCurProgressIndex++;

        if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
        {
            // Report our progress
            bSuccess = RFD_INTERFACE_bReportProgress(
                psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex);
            if (bSuccess == FALSE)
            {
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }
    }

    // Are we done?
    if (eResult == RFD_PROCESS_RESULT_COMPLETE)
    {
        bSuccess = GsWSAlertsMgrIntf.bDBUpdateEnd(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->acSQLCommandBuffer,
            WS_ALERTS_MAX_SQL_STRING_LENGTH,
            tUpdateVersion);
        if (bSuccess == FALSE)
        {
            eResult = RFD_PROCESS_RESULT_ERROR;
        }
        else
        {
            psRFDCtrl->tCurrentVersion = tUpdateVersion;
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection = SQL_INTERFACE.hConnect(
            pcDBPath,
            tOptions,
            (DATASERVICE_ERROR_CODE_ENUM *)NULL );

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*******************************************************************************
*
*       bGetPartsList
*
*******************************************************************************/
static BOOLEAN bGetPartsList(
    WS_ALERTS1_RFD_CTRL_STRUCT *psObj,
    OSAL_OBJECT_HDL hPartsList
        )
{
    BOOLEAN bResult = FALSE;

    do
    {
        BOOLEAN bSuccess = FALSE;
        UN16 un16PartsNum = 0;
        WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc = NULL;
        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
        size_t tBitsRead = 0;

        tBitsRead = OSAL.tBufferReadHeadBits(
            psObj->hBuffer, &un16PartsNum, 0,
            WS_ALERTS1_PARTITIONS_BITLEN);

        if (tBitsRead != WS_ALERTS1_PARTITIONS_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot read num of partitions from buffer");
            break;
        }

        un16PartsNum++;

        printf(WS_ALERTS1_OBJECT_NAME": Number of parts is %u\n", un16PartsNum);

        while (un16PartsNum-- > 0)
        {
            psPartDesc = (WS_ALERTS1_PARTITION_DESC_STRUCT*)
                SMSO_hCreate(WS_ALERTS1_OBJECT_NAME":PartDesc",
                sizeof(*psPartDesc), (SMS_OBJECT)psObj, FALSE
                    );
            if (psPartDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot create part desc struct"
                        );
                bSuccess = FALSE;
                break;
            }

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                &(psPartDesc->un8Class), 0,
                WS_ALERTS1_PARTITION_CLASS_BITLEN
                    );
            if (tBitsRead != WS_ALERTS1_PARTITION_CLASS_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot read partition class from buffer"
                        );
                bSuccess = FALSE;
                break;
            }

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                &(psPartDesc->un8Tab), 0,
                WS_ALERTS1_PARTITION_TAB_BITLEN
                    );
            if (tBitsRead != WS_ALERTS1_PARTITION_TAB_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot read partition table ID"
                        );
                bSuccess = FALSE;
                break;
            }

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                &(psPartDesc->un8Lang), 0,
                WS_ALERTS1_PARTITION_LANG_BITLEN
                    );
            if (tBitsRead != WS_ALERTS1_PARTITION_LANG_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot read partition language ID"
                        );
                bSuccess = FALSE;
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
                &(psPartDesc->un16VerMin), WS_ALERTS1_PARTITION_VER_MIN_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get min version"
                        );
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
                &(psPartDesc->un16UpdateVer), WS_ALERTS1_PARTITION_UPDATE_VER_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get update version"
                        );
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN32(psObj->hBuffer,
                &(psPartDesc->un32ByteSeekOffset), WS_ALERTS1_PARTITION_PAR_INDEX_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get par index"
                        );
                break;
            }

#if (DEBUG_OBJECT == 1)
            vPrintPartitionInfo(psPartDesc);
#endif

            eReturnCode = OSAL.eLinkedListAdd(hPartsList,
                                OSAL_INVALID_LINKED_LIST_ENTRY_PTR, psPartDesc);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot add part desc to list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                bSuccess = FALSE;
                break;
            }
            psPartDesc = NULL;
        }

        if (psPartDesc != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psPartDesc);
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
*
*   eProcessUpdateGroup
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessUpdateGroup (
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc =
        (WS_ALERTS1_PARTITION_DESC_STRUCT*)NULL;
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc = NULL;
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hLutName = STRING_INVALID_OBJECT;
    N32 n32LUTVer;
    char acBuf[WS_ALERTS1_SHARED_BUFFER_LEN];
    size_t tLength = 0;
    int iReturn;

    // need to init current partition if empty
    if (psRFDCtrl->hCurrentPartEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        psRFDCtrl->hCurrentPartEntry =
            OSAL.hLinkedListFirst(psRFDCtrl->hPartsList,
            (void**)&psPartDesc
                );
    }
    else
    {
        psRFDCtrl->hCurrentPartEntry =
            OSAL.hLinkedListNext(psRFDCtrl->hCurrentPartEntry,
            (void**)&psPartDesc
                );
    }

    // if current entry is NULL, we've completed all parts
    if (psRFDCtrl->hCurrentPartEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        return RFD_PROCESS_RESULT_COMPLETE;
    }

    do
    {
        //reinit buffer for every partition to avoid block issues
        if (psRFDCtrl->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = OSAL.eBufferFree(psRFDCtrl->hBuffer);
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to free buffer (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            psRFDCtrl->hBuffer = OSAL.hBufferAllocate(
                psRFDCtrl->hBlockPool, FALSE, FALSE,
                OSAL_BUFFER_ALLOCATE_OPTION_NONE
                    );
            if (psRFDCtrl->hBuffer == OSAL_INVALID_BUFFER_HDL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to allocate buffer",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }
        }

        iReturn = snprintf( acBuf,
            WS_ALERTS1_SHARED_BUFFER_LEN,
            WS_ALERTS_LUT_NAME,
            psPartDesc->un8Class,
            psPartDesc->un8Tab,
            psPartDesc->un8Lang );

        if (iReturn <= 0)
        {
            break;
        }

        tLength = strlen(acBuf);

        hLutName = STRING.hCreate(acBuf,tLength);
        if ( hLutName == STRING_INVALID_OBJECT )
        {
            break;
        }

        // Check if we need to apply an update from this partition
        bSuccess = GsWSAlertsMgrIntf.bGetCurrentDbVersion(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->acSQLCommandBuffer,
            WS_ALERTS_MAX_SQL_STRING_LENGTH,
            hLutName,
            &n32LUTVer);

        if ( bSuccess != TRUE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to apply alert"
                " type update"
                    );
            break;
        }

        if( n32LUTVer != N32_MIN )
        {
            if(psPartDesc->un16UpdateVer == n32LUTVer)
            {
                //Ignore this partition
                eResult = RFD_PROCESS_RESULT_INCOMPLETE;
                break;
            }
            if(n32LUTVer < psPartDesc->un16VerMin)
            {
                //The receiver is not able to apply the update in the Partition.
                eResult = RFD_PROCESS_RESULT_ERROR;
                break;
            }
        }
        else
        {
            if( psPartDesc->un16VerMin !=0 )
            {
                //The receiver is not able to apply the update in the Partition.
                eResult = RFD_PROCESS_RESULT_ERROR;
                break;
            }
            //We have new table! Proceed the update.
        }

        eReturnCode = OSAL.eLinkedListCreate(&(psPartDesc->hLabelsList),
            WS_ALERTS1_OBJECT_NAME":LabelsList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create LabelsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
            break;
        }

        // Read new table description
        bSuccess = bGetLabelsList( psRFDCtrl, psPartDesc, psFile );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to apply alert"
                " type update"
                    );
            break;
        }

        // Create the LUT update object - protocol independent
        // description of the update
        psLutUpdateDesc = (WS_ALERTS_LUT_UPDATE_DESC_STRUCT*)
            SMSO_hCreate(
            WS_ALERTS1_OBJECT_NAME":LutUpdateDesc",
            sizeof(WS_ALERTS_LUT_UPDATE_DESC_STRUCT),
            (SMS_OBJECT)psRFDCtrl, FALSE
                );
        if (psLutUpdateDesc == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": can't create Lut update object");
            break;
        }

        psLutUpdateDesc->hLutName = hLutName;

        bSuccess = bInitLutUpdateDesc(psRFDCtrl, psPartDesc, psLutUpdateDesc);

        if ( bSuccess == FALSE )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to init LutUpdate object"
                    );
            break;
        }

        bSuccess = GsWSAlertsMgrIntf.bCheckLutTable(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->acSQLCommandBuffer,
            WS_ALERTS_MAX_SQL_STRING_LENGTH,
            psLutUpdateDesc
                );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to check LUT table"
                    );
            break;
        }

        bSuccess = bPerformUpdate( psRFDCtrl, psPartDesc, psLutUpdateDesc, psFile );

        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to apply alert"
                " type update"
                    );
            break;
        }

        eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    } while (FALSE);

    vReleaseLutUpdateDesc(psLutUpdateDesc);
    vReleaseLabelsList(psPartDesc->hLabelsList);

    return eResult;
}

/*******************************************************************************
*
*       bInitLutUpdateDesc
*
*******************************************************************************/
static BOOLEAN bInitLutUpdateDesc(
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl,
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    OSAL_LINKED_LIST_ENTRY hPartLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS1_LABEL_DESC_STRUCT *psPartLabelDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    BOOLEAN bSuccess = TRUE;

    if ( (psPartDesc == NULL) ||
         (psLutUpdateDesc == NULL) )
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": invalid input");
        return FALSE;
    }

    do
    {
        psLutUpdateDesc->un8ClassId = psPartDesc->un8Class;
        psLutUpdateDesc->un8TableId = psPartDesc->un8Tab;
        psLutUpdateDesc->un8NumOfLabels = psPartDesc->un8NumOfLabels;
        psLutUpdateDesc->un16UpdateVersion = psPartDesc->un16UpdateVer;

        eReturnCode = OSAL.eLinkedListCreate(&(psLutUpdateDesc->hLabelsList),
            WS_ALERTS1_OBJECT_NAME":LabelsList",
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_LINEAR);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to create LabelsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode) );
            bSuccess = FALSE;
            break;
        }

        hPartLabelEntry = OSAL.hLinkedListFirst(psPartDesc->hLabelsList,
            ( void ** )&psPartLabelDesc);

        while (hPartLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
        {
            // Create the Label object
            psLabelDesc = (WS_ALERTS_LABEL_DESC_STRUCT*)
                SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":LabelDesc",
                sizeof(WS_ALERTS_LABEL_DESC_STRUCT),
                (SMS_OBJECT)psRFDCtrl, FALSE
                    );
            if (psLabelDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot create label object"
                        );
                bSuccess = FALSE;
                break;
            }

            psLabelDesc->un8LabelId = psPartLabelDesc->un8LabelId;
            psLabelDesc->hLbText = psPartLabelDesc->hLbText;

            switch (psPartLabelDesc->un8DataType)
            {
                case WS_ALERTS1_PARTITION_DTYPE_INT:
                {
                    psLabelDesc->eDataType = WS_ALERTS_PARTITION_DTYPE_INT;
                }
                break;

                case WS_ALERTS1_PARTITION_DTYPE_STRING:
                {
                    psLabelDesc->eDataType = WS_ALERTS_PARTITION_DTYPE_STRING;
                }
                break;

                case WS_ALERTS1_PARTITION_DTYPE_BAUDOT_STRING:
                {
                    psLabelDesc->eDataType = WS_ALERTS_PARTITION_DTYPE_BAUDOT_STRING;
                }
                break;

                case WS_ALERTS1_PARTITION_DTYPE_ARRAY_OF_INT:
                {
                    psLabelDesc->eDataType = WS_ALERTS_PARTITION_DTYPE_ARRAY_OF_INT;
                }
                break;

                default:
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": unknown data type" );
                    bSuccess = FALSE;
                    break;
            }

            // Add this to the list of labels
            eReturnCode = OSAL.eLinkedListAdd(
                psLutUpdateDesc->hLabelsList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psLabelDesc );

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to add LabelDesc to the list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode) );
                bSuccess = FALSE;
                break;
            }

            hPartLabelEntry = OSAL.hLinkedListNext(hPartLabelEntry, ( void ** )&psPartLabelDesc);
        }

    } while ( FALSE );

    if ( bSuccess == FALSE )
    {
        vReleaseLutUpdateDesc(psLutUpdateDesc);
    }

    return bSuccess;
}

/*******************************************************************************
*
*       vReleaseLutUpdateDesc
*
*******************************************************************************/
static void vReleaseLutUpdateDesc(
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if ( psLutUpdateDesc != NULL )
    {
        if ( psLutUpdateDesc->hLutName != STRING_INVALID_OBJECT )
        {
            STRING.vDestroy( psLutUpdateDesc->hLutName );
        }

        if ( psLutUpdateDesc->hLabelsList != OSAL_INVALID_OBJECT_HDL )
        {
            eReturnCode = OSAL.eLinkedListRemoveAll( psLutUpdateDesc->hLabelsList,
                (OSAL_LL_RELEASE_HANDLER)SMSO_vDestroy
                    );
            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListDelete( psLutUpdateDesc->hLabelsList );

                if( eReturnCode != OSAL_SUCCESS )
                {
                    SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to delete"
                        " list (%s)", OSAL.pacGetReturnCodeName( eReturnCode )
                            );
                }
                psLutUpdateDesc->hLabelsList = OSAL_INVALID_OBJECT_HDL;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to clean up LabelsList (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psLutUpdateDesc);
    }

    return;
}

/*******************************************************************************
*
*       vReleasePartDesc
*
*******************************************************************************/
static void vReleasePartDesc(
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc
        )
{
    SMSO_vDestroy((SMS_OBJECT)psPartDesc);

    return;
}

/*******************************************************************************
*
*       vReleaseLabelsList
*
*******************************************************************************/
static void vReleaseLabelsList(
    OSAL_OBJECT_HDL hLabelsList
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    if ( hLabelsList != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eLinkedListRemoveAll( hLabelsList,
            (OSAL_LL_RELEASE_HANDLER)vReleaseLabel
                );
        if (eReturnCode == OSAL_SUCCESS)
        {
            eReturnCode = OSAL.eLinkedListDelete( hLabelsList );

            if( eReturnCode != OSAL_SUCCESS )
            {
                SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to delete"
                    " list (%s)", OSAL.pacGetReturnCodeName( eReturnCode )
                        );
            }
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to clean up LabelsList (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
        }
    }
    return;
}

/*******************************************************************************
*
*       bGetLabelsList
*
*******************************************************************************/
static BOOLEAN bGetLabelsList(
    WS_ALERTS1_RFD_CTRL_STRUCT *psObj,
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc,
    FILE *psFile
        )
{
    BOOLEAN bResult = FALSE;
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    int iResult = 0;
    BOOLEAN bSuccess = TRUE;
    UN8 un8LCount = 0;
    size_t tBitsRead = 0;

    do
    {
        iResult = fseek(psFile, psPartDesc->un32ByteSeekOffset, SEEK_SET);
        if (iResult != 0)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to set file position"
                    );
            break;
        }

        bSuccess = DATASERVICE_MGR_bFillBufferBlock(
            psFile, psObj->hBuffer
                );
        if (bSuccess == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": failed to copy data to block"
                    );
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(
            psObj->hBuffer, &un8LCount, 0,
            WS_ALERTS1_PARTITION_LBNO_BITLEN
                );
        if (tBitsRead != WS_ALERTS1_PARTITION_LBNO_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot read num of labels from buffer"
                    );
            bResult = FALSE;
        }
        un8LCount++; // number of labels is LBNO + 1
        psPartDesc->un8NumOfLabels =  un8LCount;
        printf(WS_ALERTS1_OBJECT_NAME": Number of labels is %d\n", un8LCount);

        while(un8LCount-- > 0)
        {
            psLabelDesc = (WS_ALERTS1_LABEL_DESC_STRUCT*)
                SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":LabelDesc",
                sizeof(WS_ALERTS1_LABEL_DESC_STRUCT),
                (SMS_OBJECT)psObj, FALSE
                    );
            if (psLabelDesc == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get aregion field"
                        );
                bSuccess = FALSE;
                break;
            }

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                &(psLabelDesc->un8LabelId),
                0, WS_ALERTS1_PARTITION_LBNID_BITLEN
                    );
            if (tBitsRead != WS_ALERTS1_PARTITION_LBNID_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get idsize field"
                        );
                bSuccess = FALSE;
                break;
            }

            bSuccess = bGetFreeText( (SMS_OBJECT)psObj,
                psObj->hBuffer, &(psLabelDesc->hLbText), TRUE, TRUE );
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get LBTXT field"
                        );
                break;
            }

            tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                &(psLabelDesc->un8DataType),
                0, WS_ALERTS1_PARTITION_DTYPE_BITLEN
                    );
            if (tBitsRead != WS_ALERTS1_PARTITION_DTYPE_BITLEN)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get DTYPE field"
                        );
                bSuccess = FALSE;
                break;
            }

            bSuccess = OSAL.bBufferReadBitsToUN16(psObj->hBuffer,
                &(psLabelDesc->un16LbSize), WS_ALERTS1_PARTITION_LBSIZE_BITLEN
                    );
            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot get LBSIZE field"
                        );
                break;
            }

            psLabelDesc->un16LbSize++;

            if ( ((psLabelDesc->un8DataType == WS_ALERTS1_PARTITION_DTYPE_INT) ||
                  (psLabelDesc->un8DataType == WS_ALERTS1_PARTITION_DTYPE_ARRAY_OF_INT) ) &&
                  (psLabelDesc->un16LbSize > WS_ALERTS1_DTYPE_INT_MAX_BITLEN) )
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": invalid size of integer field"
                        );
                bSuccess = FALSE;
                break;
            }

            if ( psLabelDesc->un8DataType == WS_ALERTS1_PARTITION_DTYPE_ARRAY_OF_INT )
            {
                psLabelDesc->un8ArrayNum = 0;

                tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                    &(psLabelDesc->un8ArrayNum),
                    0, WS_ALERTS1_PARTITION_ARRAYNO_BITLEN
                        );
                if (tBitsRead != WS_ALERTS1_PARTITION_ARRAYNO_BITLEN)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": cannot get ARRAYNO field"
                            );
                    bSuccess = FALSE;
                    break;
                }
            }

#if (DEBUG_OBJECT == 1)
            vPrintLabelInfo(psLabelDesc);
#endif

            // Add this to the list of labels
            eReturnCode = OSAL.eLinkedListAdd(
                psPartDesc->hLabelsList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psLabelDesc
            );

            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to add LabelDesc to the list (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
        }

        if (bSuccess == FALSE)
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(psPartDesc->hLabelsList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseLabel
                    );
            if (eReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": failed to clean up LabelsList (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode)
                        );
            }
            break;
        }

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*******************************************************************************
*
*       bPerformUpdate
*
*******************************************************************************/
static BOOLEAN bPerformUpdate(
    WS_ALERTS1_RFD_CTRL_STRUCT *psObj,
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc,
    FILE *psFile
        )
{
    UN8 un8Utype = 0;
    size_t tBitsRead;
    size_t tIndex;
    BOOLEAN bSuccess = FALSE, bDataPresence = FALSE, bContunue = TRUE;
    BOOLEAN bEntryExist = FALSE;
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDescKey1 = NULL;
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDescKey2 = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;

    do
    {
        tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
            &un8Utype,
            0, WS_ALERTS1_UPDATE_UTYPE_BITLEN
                );
        if (tBitsRead != WS_ALERTS1_UPDATE_UTYPE_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot get utype field"
                    );
            break;
        }

        if ( un8Utype != WS_ALERTS1_UPDATE_UTYPE_END )
        {
            hLabelEntry = OSAL.hLinkedListFirst(psPartDesc->hLabelsList,
                ( void ** )&psLabelDescKey1);

            if(hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": Invalid LL element"
                        );
                break;
            }

            bDataPresence = bReadFlag( psObj );

            psLabelDescKey1->bPresence = bDataPresence;

            //read the first label value which is a key in update operations
            if (bDataPresence == TRUE)
            {
                bSuccess = bGetLabelValue(psObj, psLabelDescKey1,
                    psFile);

                if ( bSuccess == FALSE )
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": cannot get label value"
                            );
                    break;
                }
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": First field is absent"
                        );
                break;
            }
        }

        switch (un8Utype)
        {
            case WS_ALERTS1_UPDATE_UTYPE_DELETE:
            {
                if ( psPartDesc->un8Class == WS_ALERTS_LUT_LOCATION_CLASS_ID )
                {
                    //the key of Location entry in LUT consists of two labels.
                    //read the second one.
                    hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, ( void ** )&psLabelDescKey2);

                    if(hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WS_ALERTS1_OBJECT_NAME": Invalid LL element"
                                );
                        break;
                    }

                    OSAL.bMemSet(&psLabelDescKey2->uCurLbValue, 0, sizeof(psLabelDescKey2->uCurLbValue));

                    bDataPresence = bReadFlag( psObj );

                    if (bDataPresence == TRUE)
                    {
                        bSuccess = bGetLabelValue(psObj, psLabelDescKey2,
                            psFile);

                        if ( bSuccess == FALSE )
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                WS_ALERTS1_OBJECT_NAME": cannot get label value"
                                    );
                            break;
                        }
                    }
                    else
                    {
                        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            WS_ALERTS1_OBJECT_NAME": the label - Second part of the key is absent"
                                );
                        break;
                    }
                }

                bSuccess = bCopyLabelValues(psPartDesc, psLutUpdateDesc);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to fill LutUpdateDesc"
                            );
                    break;
                }

                bSuccess = GsWSAlertsMgrIntf.bDeleteRow(
                    psObj->hDBConnection,
                    psObj->acSQLCommandBuffer,
                    WS_ALERTS_MAX_SQL_STRING_LENGTH,
                    psLutUpdateDesc );

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to delete row"
                            );
                    break;
                }
            }
            break;

            case WS_ALERTS1_UPDATE_UTYPE_MODIFY:
            case WS_ALERTS1_UPDATE_UTYPE_NEW:
            {
                hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, ( void ** )&psLabelDescKey1);
                if (psPartDesc->un8NumOfLabels > 0)
                {
                    for (tIndex = 0; tIndex < (size_t)(psPartDesc->un8NumOfLabels - 1); tIndex++)
                    {
                        OSAL.bMemSet(&psLabelDescKey1->uCurLbValue, 0, sizeof(psLabelDescKey1->uCurLbValue));

                        if(hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY)
                        {
                            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                WS_ALERTS1_OBJECT_NAME": Invalid LL element"
                                    );
                            break;
                        }

                        bDataPresence = bReadFlag( psObj );

                        psLabelDescKey1->bPresence = bDataPresence;

                        if (bDataPresence == TRUE)
                        {
                            bSuccess = bGetLabelValue(psObj, psLabelDescKey1,
                                psFile);

                            if ( bSuccess == FALSE )
                            {
                                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                                    WS_ALERTS1_OBJECT_NAME": cannot get label value"
                                        );
                                break;
                            }

                            // Grab more data
                            DATASERVICE_MGR_bFillBufferBlock(
                                psFile, psObj->hBuffer);

                        }

                        hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, ( void ** )&psLabelDescKey1);
                    }
                }

                bSuccess = bCopyLabelValues(psPartDesc, psLutUpdateDesc);
                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to fill LutUpdateDesc"
                            );
                    break;
                }

                bSuccess = GsWSAlertsMgrIntf.bCheckRowExist(
                    psObj->hDBConnection,
                    psObj->acSQLCommandBuffer,
                    WS_ALERTS_MAX_SQL_STRING_LENGTH,
                    psLutUpdateDesc,
                    &bEntryExist);

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to check the row exist"
                            );
                    break;
                }

                if( bEntryExist == TRUE )
                {
                    bSuccess = GsWSAlertsMgrIntf.bUpdateRow(
                        psObj->hDBConnection,
                        psObj->acSQLCommandBuffer,
                        WS_ALERTS_MAX_SQL_STRING_LENGTH,
                        psLutUpdateDesc);
                }
                else
                {
                    bSuccess = GsWSAlertsMgrIntf.bInsertRow(
                        psObj->hDBConnection,
                        psObj->acSQLCommandBuffer,
                        WS_ALERTS_MAX_SQL_STRING_LENGTH,
                        psLutUpdateDesc);
                }

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": failed to update row"
                            );
                    break;
                }

            }
            break;

            case WS_ALERTS1_UPDATE_UTYPE_END:
            {
                bContunue = FALSE;
            }
            break;

        }

        eReturnCode = OSAL.eLinkedListIterate(psPartDesc->hLabelsList,
            (OSAL_LL_ITERATOR_HANDLER)bReleaseLabelData, NULL
                );
        if( (eReturnCode != OSAL_SUCCESS) &&
            (eReturnCode != OSAL_NO_OBJECTS) )
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot iterate labels list"
                " (%s)",
                OSAL.pacGetReturnCodeName(eReturnCode)
                    );
            bSuccess = FALSE;
            break;
        }

        if ( eReturnCode != OSAL_SUCCESS )
        {
            bSuccess = FALSE;
            break;
        }

        if ( (bContunue == FALSE) ||
             (bSuccess == FALSE) )
        {
            break;
        }

    }
    while ( TRUE );

    return bSuccess;
}

/*******************************************************************************
*
*       bCopyLabelValues
*
*******************************************************************************/
static BOOLEAN bCopyLabelValues(
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc,
    WS_ALERTS_LUT_UPDATE_DESC_STRUCT *psLutUpdateDesc
        )
{
    OSAL_LINKED_LIST_ENTRY hPartLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_LINKED_LIST_ENTRY hLabelEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    WS_ALERTS1_LABEL_DESC_STRUCT *psPartLabelDesc = NULL;
    WS_ALERTS_LABEL_DESC_STRUCT *psLabelDesc = NULL;
    BOOLEAN bSuccess = TRUE;

    if ( (psPartDesc == NULL) ||
         (psLutUpdateDesc == NULL) )
   {
        return FALSE;
   }

    hPartLabelEntry = OSAL.hLinkedListFirst(psPartDesc->hLabelsList,
        ( void ** )&psPartLabelDesc);

    hLabelEntry = OSAL.hLinkedListFirst(psLutUpdateDesc->hLabelsList,
        ( void ** )&psLabelDesc);

    while (hPartLabelEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        if ( hLabelEntry == OSAL_INVALID_LINKED_LIST_ENTRY )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": invalid LutUpdate Label entry" );
            bSuccess = FALSE;
            break;
        }

        if ( (psPartLabelDesc == NULL) ||
             (psLabelDesc == NULL) )
        {
            SMSAPI_DEBUG_vPrintErrorFull( gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": invalid Label desc" );
            bSuccess = FALSE;
            break;
        }

        psLabelDesc->bPresence = psPartLabelDesc->bPresence;

        if ( psPartLabelDesc->bPresence == TRUE )
        {
            psLabelDesc->tCurrentArrayNum = psPartLabelDesc->tCurrentArrayNum;
            psLabelDesc->uCurLbValue = psPartLabelDesc->uCurLbValue;
        }

        hPartLabelEntry = OSAL.hLinkedListNext(hPartLabelEntry, ( void ** )&psPartLabelDesc);
        hLabelEntry = OSAL.hLinkedListNext(hLabelEntry, ( void ** )&psLabelDesc);
    }

    return bSuccess;
}

/*******************************************************************************
*
*       bReadFlag
*
*******************************************************************************/
static BOOLEAN bReadFlag(
    WS_ALERTS1_RFD_CTRL_STRUCT *psObj
        )
{
    size_t tBitsRead = 0;
    UN8 un8Flag = 0;

    tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
        &un8Flag,
        0, WS_ALERTS1_UPDATE_FLAG_BITLEN
            );
    if (tBitsRead != WS_ALERTS1_UPDATE_FLAG_BITLEN)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            WS_ALERTS1_OBJECT_NAME": cannot get flag field"
                );
    }

    if ( un8Flag == 0 )
    {
        return FALSE;
    }
    else
    {
        return TRUE;
    }
}

/*******************************************************************************
*
*       bGetLabelValue
*
*******************************************************************************/
static BOOLEAN bGetLabelValue(
    WS_ALERTS1_RFD_CTRL_STRUCT *psObj,
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDesc,
    FILE *psFile
        )
{
    size_t tByteCount = 0;
    size_t tBitsRead = 0;
    size_t tIndex = 0;
    BOOLEAN bSuccess = FALSE;
    UN16 un16Temp;
    N32 n32Temp;
    UN16 un16Size;
    WS_ALERTS_LABEL_VALUE_UNION *puLabelValue = NULL;

    puLabelValue = &(psLabelDesc->uCurLbValue);

    switch (psLabelDesc->un8DataType)
    {
        case WS_ALERTS1_PARTITION_DTYPE_INT:
        {
            puLabelValue->un16Value = 0;

            bSuccess = OSAL.bBufferReadBitsToUN16(
                psObj->hBuffer, &(puLabelValue->un16Value),
                psLabelDesc->un16LbSize);

            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot read N16 Label value"
                        );
                break;
            }
        }
        break;

        case WS_ALERTS1_PARTITION_DTYPE_STRING:
        {
            puLabelValue->pcValue = (char*)
                SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":stringLabelValue",
                psLabelDesc->un16LbSize,
                (SMS_OBJECT)psObj, FALSE
                    );

            if ( puLabelValue->pcValue == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot create stringLabelValue object"
                        );
                break;
            }

            tByteCount = 0;
            do
            {
                tBitsRead = OSAL.tBufferReadHeadBits(psObj->hBuffer,
                    &(puLabelValue->pcValue[tByteCount]), 0,
                    WS_ALERTS1_DTYPE_STRING_SIZE);
                if (tBitsRead != WS_ALERTS1_DTYPE_STRING_SIZE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": cannot get string Label value"
                            );
                    break;
                }

                tByteCount++;

                if ( tByteCount > psLabelDesc->un16LbSize )
                {
                    break;
                }
            }
            while (puLabelValue->pcValue[tByteCount - 1] != '\0');

            bSuccess = TRUE;
        }
        break;

        case WS_ALERTS1_PARTITION_DTYPE_BAUDOT_STRING:
        {
            bSuccess = bGetFreeText((SMS_OBJECT)psObj,
                psObj->hBuffer,
                &(puLabelValue->hValue), TRUE, TRUE);

            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot Get Free Text"
                        );
                break;
            }

#if (DEBUG_OBJECT == 1)
            STRING.n32FPrintf(puLabelValue->hValue, stdout);
#endif
        }
        break;

        case WS_ALERTS1_PARTITION_DTYPE_ARRAY_OF_INT:
        {
            un16Size = 0;
            bSuccess = OSAL.bBufferReadBitsToUN16(
                psObj->hBuffer, &un16Size,
                psLabelDesc->un16LbSize);

            if (bSuccess != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot read UN16 Label value"
                        );
                break;
            }

            printf("un16Size = %d\n", un16Size);

            puLabelValue->pn32Data = (N32*)
                SMSO_hCreate(
                WS_ALERTS1_OBJECT_NAME":LabelValue",
                sizeof(N32) * (un16Size + 2),
                (SMS_OBJECT)psObj, FALSE
                    );

            if ( puLabelValue->pn32Data == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    WS_ALERTS1_OBJECT_NAME": cannot create LabelValue object"
                        );
                bSuccess = FALSE;
                break;
            }

            for(tIndex = 0; tIndex <= un16Size; tIndex++)
            {
                un16Temp = 0;

                bSuccess = OSAL.bBufferReadBitsToUN16(
                    psObj->hBuffer, &un16Temp,
                    psLabelDesc->un16LbSize);

                if (bSuccess != TRUE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        WS_ALERTS1_OBJECT_NAME": cannot read UN16 Label value"
                            );
                    break;
                }

                if ( psLabelDesc->un8LabelId == WS_ALERTS_CLASS1_LABID_LON )
                {
                    n32Temp = WS_ALERTS1_LON_INT(un16Temp);
                }
                else if ( psLabelDesc->un8LabelId == WS_ALERTS_CLASS1_LABID_LAT )
                {
                    n32Temp = WS_ALERTS1_LAT_INT(un16Temp);
                }
                else
                {
                    n32Temp = (N32)un16Temp;
                }

                puLabelValue->pn32Data[tIndex] = n32Temp;
            }

#if (DEBUG_OBJECT == 1)
            for(tIndex = 0; tIndex <= un16Size; tIndex++)
            {

                printf("int#%d = %d\n", tIndex, puLabelValue->pn32Data[tIndex]);

            }
#endif

            puLabelValue->pn32Data[un16Size + 1] = puLabelValue->pn32Data[0];
            psLabelDesc->tCurrentArrayNum = un16Size + 2;

        }
        break;

    }

    if ( bSuccess == FALSE )
    {
        bReleaseLabelData( psLabelDesc );
    }

    return bSuccess;

}

/*******************************************************************************
*
*       vReleaseLabel
*
*******************************************************************************/
static void vReleaseLabel(
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDesc
        )
{
    if ( psLabelDesc != NULL )
    {
        if (psLabelDesc->hLbText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psLabelDesc->hLbText);
            psLabelDesc->hLbText = STRING_INVALID_OBJECT;
        }

        bReleaseLabelData( psLabelDesc );

        SMSO_vDestroy((SMS_OBJECT)psLabelDesc);
    }

    return;
}

/*******************************************************************************
*
*       bReleaseLabelData
*
*******************************************************************************/
static BOOLEAN bReleaseLabelData(
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDesc
        )
{
    switch (psLabelDesc->un8DataType)
    {
        case WS_ALERTS1_PARTITION_DTYPE_STRING:
        {
            if (psLabelDesc->uCurLbValue.pcValue != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psLabelDesc->uCurLbValue.pcValue);
                psLabelDesc->uCurLbValue.pcValue = NULL;
            }
        }
        break;

        case WS_ALERTS1_PARTITION_DTYPE_BAUDOT_STRING:
        {
            if (psLabelDesc->uCurLbValue.hValue != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psLabelDesc->uCurLbValue.hValue);
                psLabelDesc->uCurLbValue.hValue = STRING_INVALID_OBJECT;
            }
        }
        break;

        case WS_ALERTS1_PARTITION_DTYPE_ARRAY_OF_INT:
        {
            if (psLabelDesc->uCurLbValue.pn32Data != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psLabelDesc->uCurLbValue.pn32Data);
                psLabelDesc->uCurLbValue.pn32Data = NULL;
            }
        }
        break;
    }

    return TRUE;
}


#if (DEBUG_OBJECT == 1)
/*******************************************************************************
*
*       vPrintPartitionInfo
*
*******************************************************************************/
static void vPrintPartitionInfo(
    WS_ALERTS1_PARTITION_DESC_STRUCT *psPartDesc
        )
{
    printf(WS_ALERTS1_OBJECT_NAME"------------------------\n");
    printf(WS_ALERTS1_OBJECT_NAME": Class -"
            " %u\n", psPartDesc->un8Class);
    printf(WS_ALERTS1_OBJECT_NAME": Tab - "
            " %u\n", psPartDesc->un8Tab);
    printf(WS_ALERTS1_OBJECT_NAME": Lang - "
            " %u\n", psPartDesc->un8Lang);
    printf(WS_ALERTS1_OBJECT_NAME": VerMin - "
            " %u\n", psPartDesc->un16VerMin);
    printf(WS_ALERTS1_OBJECT_NAME": UpdateVer - "
            " %u\n", psPartDesc->un16UpdateVer);
    printf(WS_ALERTS1_OBJECT_NAME": seek offset - "
            " %u\n", psPartDesc->un32ByteSeekOffset);
    printf(WS_ALERTS1_OBJECT_NAME"------------------------\n");
}

/*******************************************************************************
*
*       vPrintLabelInfo
*
*******************************************************************************/
static void vPrintLabelInfo(
    WS_ALERTS1_LABEL_DESC_STRUCT *psLabelDesc
        )
{
    const char *pacLbtext;

    pacLbtext = STRING.pacCStr(psLabelDesc->hLbText);

    printf(WS_ALERTS1_OBJECT_NAME"------------------------\n");
    printf(WS_ALERTS1_OBJECT_NAME": LBID -"
            " %u\n", psLabelDesc->un8LabelId);
    printf(WS_ALERTS1_OBJECT_NAME": LbText - "
            " %s\n", pacLbtext);
    printf(WS_ALERTS1_OBJECT_NAME": DTYPE - "
            " %u\n", psLabelDesc->un8DataType);
    printf(WS_ALERTS1_OBJECT_NAME": LBSIZE - "
            " %u\n", psLabelDesc->un16LbSize);
    printf(WS_ALERTS1_OBJECT_NAME": ARRAYNO - "
            " %u\n", psLabelDesc->un8ArrayNum);
    printf(WS_ALERTS1_OBJECT_NAME"------------------------\n");
}

/*****************************************************************************
 *
 *   vPrintBuffer
 *
 *****************************************************************************/
static void vPrintBuffer(
    OSAL_BUFFER_HDL hPayload
)
{
    size_t tOffsetInBits = 0,
           tSizeRemaining = 0,
           tNumBitsToRead = 0,
           tBitsRead = 0;
    UN8 un8Tmp = 0;

    // Calculate the number of bytes
    tSizeRemaining = OSAL.tBufferGetSizeInBits(hPayload);

    printf("vPrintBuffer: bits in buffer - %d \n",tSizeRemaining);


    while (tSizeRemaining > 0)
    {
        if (tSizeRemaining > 8)
        {
            tNumBitsToRead = 8;
        }
        else
        {
            tNumBitsToRead = tSizeRemaining;
        }

        un8Tmp = 0;

//        tOffsetInBits = tBytesRead * 8;
        tBitsRead = OSAL.tBufferPeekBits(
            hPayload, &un8Tmp, 0, tNumBitsToRead, tOffsetInBits);
        printf("%x ",un8Tmp);

        if (tBitsRead != tNumBitsToRead)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot read bits from buffer"
                    );
            break;
        }

        tOffsetInBits += tBitsRead;
        tSizeRemaining -= tBitsRead;
    }
}

/*****************************************************************************
 *
 *   vPrintBufferToFile
 *
 *****************************************************************************/
static void vPrintBufferToFile(
    OSAL_BUFFER_HDL hPayload
)
{
    size_t tOffsetInBits = 0,
           tSizeRemaining = 0,
           tNumBitsToRead = 0,
           tBitsRead = 0,
           tNumBytesWritten = 0;

    UN8 un8Tmp = 0;
    FILE *psFileWrite = (FILE*)NULL;


    psFileWrite = fopen("/home/ruinadud/02", "wb");
    if (psFileWrite == NULL)
    {
    printf("error while open psFileWrite\n");
}

    // Calculate the number of bytes
    tSizeRemaining = OSAL.tBufferGetSizeInBits(hPayload);

    while (tSizeRemaining > 0)
    {
        if (tSizeRemaining > 8)
        {
            tNumBitsToRead = 8;
        }
        else
        {
            tNumBitsToRead = tSizeRemaining;
        }

        un8Tmp = 0;

//        tOffsetInBits = tBytesRead * 8;
        tBitsRead = OSAL.tBufferPeekBits(
            hPayload, &un8Tmp, 0, tNumBitsToRead, tOffsetInBits);

        tNumBytesWritten = fwrite(
            &un8Tmp, sizeof(UN8),
            1, psFileWrite );

        if (tNumBytesWritten != 1)
        {
            // File accesses shouldn't fail -- we have to
            // have access to storage for this service to be
            // useful.  Consider this an error
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME
                ": failed to write %d bytes (written %d) "
                "to file",
                1, tNumBytesWritten);
            break;
        }

        if (tBitsRead != tNumBitsToRead)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                WS_ALERTS1_OBJECT_NAME": cannot read bits from buffer"
                    );
            break;
        }

        tOffsetInBits += tBitsRead;
        tSizeRemaining -= tBitsRead;
    }
    fclose (psFileWrite);
}
#endif

/*****************************************************************************
*
*   vNotifyDBExpired
*
*****************************************************************************/
static void vNotifyDBExpired(
    RFD_INTERFACE_OBJECT hConnection,
    void *pvCallbackArg
        )
{
    WS_ALERTS1_RFD_CTRL_STRUCT *psRFDCtrl =
        (WS_ALERTS1_RFD_CTRL_STRUCT*)pvCallbackArg;

    if (psRFDCtrl != NULL)
    {
        GsWSAlertsMgrIntf.vTooOldDBVer(psRFDCtrl->hWSAlertsService);
    }

    return;
}

/*****************************************************************************
*
*   bCleanDbDir
*
*****************************************************************************/
static BOOLEAN bCleanDbDir (
    WS_ALERTS1_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_HDL hDirItems = OSAL_INVALID_OBJECT_HDL;
    char *pcFileName;
    size_t tPathLen;
    int iReturn;

    // Find out how big the path is,
    // then add WS_ALERTS1_MAX_FILE_NAME_LEN
    tPathLen = strlen(psObj->pacServiceFilePath);

    tPathLen += WS_ALERTS1_MAX_FILE_NAME_LEN;

    // Create a buffer for this operation
    pcFileName = (char *)SMSO_hCreate(
        WS_ALERTS1_OBJECT_NAME":DelBuf",
        tPathLen, (SMS_OBJECT)psObj, FALSE);
    if (pcFileName == NULL)
    {
        return FALSE;
    }

    // Get the list of all the items in the directory
    eReturnCode = OSAL.eFileSystemGetDirItems(
        &hDirItems, psObj->pacServiceFilePath,
        (OSAL_FS_FILE_QUALIFIER)bFileQualifier, NULL);
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL_LINKED_LIST_ENTRY hEntry;
        const char *pcEntry;

        // Now, iterate all items to remove them
        hEntry = OSAL.hLinkedListFirst(hDirItems, (void **)&pcEntry);

        while ((hEntry != OSAL_INVALID_LINKED_LIST_ENTRY) &&
               (pcEntry != NULL))
        {
            // Prepend the working directory to the file name
            iReturn = snprintf( &pcFileName[0], tPathLen,
                "%s/%s", &psObj->pacServiceFilePath[0],
                &pcEntry[0] );

            if (iReturn <= 0)
            {
                break;
            }

            // Remove this file
            remove(&pcFileName[0]);

            // Move to the next entry
            hEntry = OSAL.hLinkedListNext(hEntry, (void **)&pcEntry);
        }

        // Release our dir items list
        OSAL.eFileSystemReleaseDirItems(hDirItems);
    }

    SMSO_vDestroy((SMS_OBJECT)pcFileName);

    if ((eReturnCode == OSAL_SUCCESS) ||
        (eReturnCode == OSAL_NO_OBJECTS))
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bFileQualifier
*
*****************************************************************************/
static BOOLEAN bFileQualifier (
    const char *pcItemName,
    const void *pvArg
        )
{
    const char *pcResult;

    // if this file ends with the extension for
    // the DB files we skip it
    pcResult = strstr(pcItemName, WS_ALERTS1_DB_FILE_EXT);
    if (pcResult != NULL)
    {
        return FALSE;
    }

    // Not a match
    return TRUE;
}
