/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Table Data Service utility functions for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "sms_obj.h"
#include "string_obj.h"
#include "baudot.h"
#include "radio.h"

#include "tds_util.h"
#include "_tds_util.h"
#include "ds_util.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/


/*****************************************************************************
*
*   TDS_UTIL_hConnect
*
*****************************************************************************/
TDS_UTIL_OBJECT TDS_UTIL_hConnect (
    SMS_OBJECT hTDSParent,
    SMS_OBJECT hRowDataParent,
    TDS_CALLBACKS_STRUCT *psCallbacks,
    BOOLEAN bRestrictTables
        )
{
    BOOLEAN bOwner;

    // Verify inputs (callback args may be NULL)
    if ((psCallbacks == NULL) || (
        ((psCallbacks->hCreateTable == NULL) ||
        (psCallbacks->vDestroyTable == NULL) ||
        (psCallbacks->bAddColumn == NULL) ||
        (psCallbacks->bRowStatus == NULL) ||
        (psCallbacks->bAddRowData == NULL) ||
        (psCallbacks->bNewTable == NULL)))
            )
    {
        return TDS_UTIL_INVALID_OBJECT;
    }

    if (bRestrictTables == FALSE)
    {
        // TODO: support this when needed
        return TDS_UTIL_INVALID_OBJECT;
    }

    // Verify we own service handles
    bOwner = SMSO_bOwner((SMS_OBJECT)hTDSParent);
    if (bOwner == TRUE)
    {
        bOwner = SMSO_bOwner((SMS_OBJECT)hRowDataParent);
    }

    if (bOwner == TRUE)
    {
        TDS_OBJECT_STRUCT *psObj;

        // Create an instance of this object
        psObj = (TDS_OBJECT_STRUCT *)
            SMSO_hCreate(
                TDS_OBJECT_NAME,
                sizeof(TDS_OBJECT_STRUCT),
                hTDSParent, FALSE );

        if (psObj == NULL)
        {
            return TDS_UTIL_INVALID_OBJECT;
        }

        // Save table manipulation callbacks & args
        psObj->sCallbacks = *psCallbacks;

        // Save the row data parent object
        psObj->hRowDataParent = hRowDataParent;

        // Save the table restriction flag as well
        psObj->bRestrictTables = bRestrictTables;

        // Scode lookup not provided yet
        psObj->sScodeLUT.ppacSCodeLUT = NULL;
        psObj->sScodeLUT.tNumEntriesInLUT = 0;

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Create the list of table trackers -- use pre-allocated elements
            eReturnCode = OSAL.eLinkedListCreate(
                &psObj->hTables,
                TDS_OBJECT_NAME":Table",
                (OSAL_LL_COMPARE_HANDLER)n16CompareTables,
                OSAL_LL_OPTION_UNIQUE |
                OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS );
            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error! Destroy object
                break;
            }

            // Request access to ISO 3309 CRC32
            eReturnCode = OSAL.eGetCRC(
                &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error! destroy object
                break;
            }

            return (TDS_UTIL_OBJECT)psObj;

        } while (0);

        TDS_UTIL_vDisconnect((TDS_UTIL_OBJECT)psObj);
    }

    return TDS_UTIL_INVALID_OBJECT;
}

/*****************************************************************************
*
*   TDS_UTIL_bRegisterTable
*
*****************************************************************************/
BOOLEAN TDS_UTIL_bRegisterTable (
    TDS_UTIL_OBJECT hTDS,
    TDS_TABLE_DESC_STRUCT *psTable
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify input
    if ((psTable == NULL) || (
        ((psTable->tId == TDS_TABLE_INVALID_ID) ||
        (psTable->tDefVer == TDS_TABLE_INVALID_DEF_VERSION) ||
        (psTable->tContentVer == TDS_TABLE_INVALID_CONTENT_VERSION) ||
        (psTable->tDefCarouselId == TDS_INVALID_CAROUSEL_ID) ||
        (psTable->tContentCarouselId == TDS_INVALID_CAROUSEL_ID)))
            )
    {
        return FALSE;
    }

    // Verify we own service handle
    bOwner = SMSO_bOwner((SMS_OBJECT)hTDS);
    if (bOwner == TRUE)
    {
        TDS_OBJECT_STRUCT *psObj = (TDS_OBJECT_STRUCT *)hTDS;

        // Add this table to our tracking list
        bSuccess = bAddTableToTrackingList(psObj, psTable);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   TDS_UTIL_bSetScodeLUT
*
*   Used to provide a lookup table that this object may reference when
*   processing SCode data
*
*****************************************************************************/
BOOLEAN TDS_UTIL_bSetScodeLUT (
    TDS_UTIL_OBJECT hTDS,
    const char **ppacSCodeLUT,
    size_t tNumEntriesInLUT
        )
{
    BOOLEAN  bOwner;

    // Verify input
    if (ppacSCodeLUT == NULL)
    {
        return FALSE;
    }

    // Verify we own service handle
    bOwner = SMSO_bOwner((SMS_OBJECT)hTDS);
    if (bOwner == TRUE)
    {
        TDS_OBJECT_STRUCT *psObj = (TDS_OBJECT_STRUCT *)hTDS;

        // Save the table pointer and the table size
        psObj->sScodeLUT.ppacSCodeLUT = ppacSCodeLUT;
        psObj->sScodeLUT.tNumEntriesInLUT = tNumEntriesInLUT;
    }

    return bOwner;
}

/*****************************************************************************
*
*   TDS_UTIL_vDisconnect
*
*****************************************************************************/
void TDS_UTIL_vDisconnect(
    TDS_UTIL_OBJECT hTDS
        )
{
    BOOLEAN bOwner;

    // Verify we own service handle
    bOwner = SMSO_bOwner((SMS_OBJECT)hTDS);
    if (bOwner == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        TDS_OBJECT_STRUCT *psObj = (TDS_OBJECT_STRUCT *)hTDS;

        if (psObj->hTables != OSAL_INVALID_OBJECT_HDL)
        {
            // Iterate entries to clear them all
            eReturnCode = OSAL.eLinkedListIterate(
                psObj->hTables,
                (OSAL_LL_ITERATOR_HANDLER)bRemoveTableEntry,
                (void *)psObj);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Destroy the table and clear the handle
                OSAL.eLinkedListDelete(psObj->hTables);
                psObj->hTables = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy the CRC object if it exists
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            // Release our hold of the CRC computation handle
            eReturnCode = OSAL.eReleaseCRC( psObj->hCRC );

            // Only finish clean up if that succeeded.  This allows
            // us to more easily track this kind of issue
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Reset table manipulation callbacks
        psObj->sCallbacks.hCreateTable = NULL;
        psObj->sCallbacks.vDestroyTable = NULL;
        psObj->sCallbacks.bAddColumn = NULL;
        psObj->sCallbacks.bRowStatus = NULL;
        psObj->sCallbacks.bAddRowData = NULL;
        psObj->sCallbacks.bNewTable = NULL;
        psObj->sCallbacks.pvTableArg = NULL;

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }
    return;
}

/*****************************************************************************
*
*   TDS_UTIL_bProcessPayload
*
*****************************************************************************/
BOOLEAN TDS_UTIL_bProcessPayload (
    TDS_UTIL_OBJECT hTDS,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify payload
    if (hPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hTDS);
    if (bOwner == TRUE)
    {
        TDS_OBJECT_STRUCT *psObj =
            (TDS_OBJECT_STRUCT *)hTDS;
        BOOLEAN bMessageValid, bRead, bDefCar = FALSE;
        PVN tPVN = 0;
        size_t tBitsRead;
        UN8 un8CarouselID = 0;
        TDS_TABLE_ID tTabID = 0;
        TDS_TABLE_DEF_VERSION tTabDefVer = 0;
        TDS_TABLE_TRACKING_STRUCT *psTable;

        // Validate the message
        bMessageValid = DS_UTIL_bIsCRCValid(psObj->hCRC, hPayload, NULL);
        if (bMessageValid == FALSE)
        {
            puts(TDS_OBJECT_NAME" Packet Invalid");

            return TRUE;
        }

        // Cut off CRC
        bMessageValid = DS_UTIL_bCutCRC(hPayload);
        if (bMessageValid == FALSE)
        {
            puts(TDS_OBJECT_NAME" Failed to cut off CRC");

            return TRUE;
        }

        // Read the PVN
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &tPVN, 0, TDS_PVN_BITLEN);

        if (tBitsRead != TDS_PVN_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(TDS_OBJECT_NAME" Unable to read PVN\n");

            return TRUE;
        }

        // Verify the PVN
        if (tPVN != TDS_PVN1)
        {
            printf(TDS_OBJECT_NAME" Incorrect PVN - got %u, expected %u\n",
                tPVN, TDS_PVN1);

            return TRUE;
        }

        // Read the Carousel Id
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8CarouselID, 0, TDS_CAR_BITLEN);

        if (tBitsRead != TDS_CAR_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(TDS_OBJECT_NAME" Unable to read Carousel Id");

            return TRUE;
        }

        // Read the table ID
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, (UN16 *)&tTabID, TDS_TABID_BITLEN);
        if (bRead == FALSE)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(TDS_OBJECT_NAME": Unable to read table ID");

            return TRUE;
        }

        // Read table Version
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &tTabDefVer, 0, TDS_TABVER_BITLEN);

        if (tBitsRead != TDS_TABVER_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(TDS_OBJECT_NAME" Unable to read Table Version");

            return TRUE;
        }

        // Find this table
        psTable = psGetTableById(psObj, tTabID);
        if (psTable == NULL)
        {
            // We don't care about this table id
            return TRUE;
        }

        // TODO: Support this when needed:
        // This doesn't work with unrestricted tables -- we're gonna
        // have to add something separate which gives the global
        // definition for carousel ids but still allows individual known
        // table carousel ids to be configured.  Maybe calling
        // TDS_UTIL_bRegisterTable with a table desc that has the carousels
        // configured but the table id invalid?
        if (psTable->sDesc.tDefCarouselId == (TDS_CAROUSEL_ID)un8CarouselID)
        {
            bDefCar = TRUE;
        }
        else if (psTable->sDesc.tContentCarouselId == (TDS_CAROUSEL_ID)un8CarouselID)
        {
            bDefCar = FALSE;
        }
        else
        {
            printf(TDS_OBJECT_NAME" Unknown carousel ID: %u used for table ID: %d\n", un8CarouselID, tTabID);
            return TRUE;
        }

        if (bDefCar == TRUE)
        {
            // Process definition
            bSuccess = bProcessDefinitionMsg(psObj, tTabDefVer, psTable, hPayload);
        }
        else
        {
            // Process content data
            bSuccess = bProcessContentMsg(psObj, tTabDefVer, psTable, hPayload);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   TDS_UTIL_pacDataType
*
*****************************************************************************/
char const *TDS_UTIL_pacDataType (
    TDS_DATA_TYPE_ENUM eType
        )
{
    char const *pacType = "Unknown";

    switch (eType)
    {
        case TDS_DATA_TYPE_INT:
        {
            pacType = "Integer";
        }
        break;

        case TDS_DATA_TYPE_INT_ARRAY:
        {
            pacType = "Integer Array";
        }
        break;

        case TDS_DATA_TYPE_STRING:
        {
            pacType = "String";
        }
        break;

        case TDS_DATA_INVALID_TYPE:
        {
            pacType = "Invalid";
        }
        break;

        // default left out to force a warning if we're missing something
    }

    return pacType;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bAddTableToTrackingList
*
*****************************************************************************/
static BOOLEAN bAddTableToTrackingList (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DESC_STRUCT *psTableDesc
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    TDS_TABLE_TRACKING_STRUCT *psNewTable;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    do
    {
        // Name this entry
        snprintf( &acName[0], sizeof(acName),
            TDS_OBJECT_NAME":TableTracking: %u",
            psTableDesc->tId);

        // Allocate memory for the new table tracking info
        psNewTable = (TDS_TABLE_TRACKING_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                &acName[0],
                sizeof(TDS_TABLE_TRACKING_STRUCT), TRUE);
        if (psNewTable == NULL)
        {
            break;
        }

        // Initialize the entry handle
        psNewTable->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Initialize the label list handle
        psNewTable->sInProgress.hLabels = OSAL_INVALID_OBJECT_HDL;

        // Populate the descriptor
        psNewTable->sDesc = *psTableDesc;

        // Initialize the in-progress table info
        vResetInProgressTable(psObj, psNewTable);

        // Add this table to the list
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->hTables,
            &psNewTable->hEntry, psNewTable);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME
                ": unable to add table to tracking list: "
                "\n\tError: %s \n\tTable Id: %u",
                OSAL.pacGetReturnCodeName(eReturnCode),
                psNewTable->sDesc.tId);
            break;
        }

        return TRUE;

    } while (FALSE);

    if (psNewTable != NULL)
    {
        BOOLEAN bRemoved;

        // Remove this table entry
        bRemoved = bRemoveTableEntry(psNewTable, psObj);
        if (bRemoved == TRUE)
        {
            // We have to manually delete this entry since
            // it was not added to the list
            OSAL.vLinkedListMemoryFree(psNewTable);
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   bCreateTable
*
*****************************************************************************/
static BOOLEAN bCreateTable (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_TABLE_DEF_VERSION tTabDefVer,
    TDS_TABLE_TIME tTabTime
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Name this list
        snprintf( &acName[0], sizeof(acName),
            TDS_OBJECT_NAME":LabelList: %u",
            psTable->sDesc.tId);

        // Create the unsorted label list for this table
        // and use pre-allocated elements.  This list is
        // unsorted because we never search for a particular
        // label id, we only traverse this list in the order
        // in which labels were added to it.  Kinda convenient.
        eReturnCode = OSAL.eLinkedListCreate(
            &psTable->sInProgress.hLabels,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME": Unable to create label list:\n"
                "\t: ID: %d DefVer: %d TableTime: %d\n",
                psTable->sDesc.tId, tTabDefVer, tTabTime);
            break;
        }

        // Create the payload list for this table
        eReturnCode = OSAL.eLinkedListCreate(
            &psTable->sInProgress.hContentMsgs,
            &acName[0],
            (OSAL_LL_COMPARE_HANDLER)NULL,
            OSAL_LL_OPTION_NONE );
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME": Unable to create payload list:\n"
                "\t: ID: %d\n",
                psTable->sDesc.tId);
            break;
        }

        // Create the table instance now
        psTable->sInProgress.hTable =
            psObj->sCallbacks.hCreateTable(
                psTable->sDesc.tId,
                tTabDefVer, tTabTime,
                psObj->sCallbacks.pvTableArg);

        if (psTable->sInProgress.hTable == TDS_UTIL_INVALID_TABLE_INSTANCE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME": Create table callback failed:\n"
                "\t: ID: %d DefVer: %d TableTime: %d\n",
                psTable->sDesc.tId, tTabDefVer, tTabTime);
            break;
        }

        // Save this table definition version
        psTable->sInProgress.tDefVer = tTabDefVer;

        // Initialize the smallest symbol size used by this table
        psTable->sInProgress.tSmallestSymbolSizeInBits = SIZE_T_MAX;

        // Smallest size isn't known yet
        psTable->sInProgress.bSmallestSizeKnown = FALSE;

        // Note: we don't need to track the table time
        // attribute -- we don't need to worry about
        // that here since we just build the tables,
        // we don't manage them
        bSuccess = TRUE;

    } while (FALSE);

    return bSuccess;
}

/*****************************************************************************
*
*   vResetInProgressTable
*
*****************************************************************************/
static void vResetInProgressTable (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable
        )
{
    if (psTable == NULL)
    {
        return;
    }

    // Delete the label list if necessary
    if (psTable->sInProgress.hLabels != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Iterate the list to empty it
        eReturnCode = OSAL.eLinkedListIterate(
            psTable->sInProgress.hLabels,
            (OSAL_LL_ITERATOR_HANDLER)bRemoveLabel, NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psTable->sInProgress.hLabels);
            psTable->sInProgress.hLabels = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete the message list if necessary
    if (psTable->sInProgress.hContentMsgs != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        eReturnCode = OSAL.eLinkedListRemoveAll(
            psTable->sInProgress.hContentMsgs,
            (OSAL_LL_RELEASE_HANDLER)NULL);

        if (eReturnCode == OSAL_SUCCESS)
        {
            OSAL.eLinkedListDelete(psTable->sInProgress.hContentMsgs);
            psTable->sInProgress.hContentMsgs = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Clear in progress table data
    psTable->sInProgress.tDefVer = TDS_TABLE_INVALID_DEF_VERSION;
    psTable->sInProgress.tContentVer = TDS_TABLE_INVALID_CONTENT_VERSION;
    psTable->sInProgress.tNumEntryMessagesLeft = 0;

    // Reset these values
    psTable->sInProgress.tSmallestSymbolSizeInBits = SIZE_T_MAX;
    psTable->sInProgress.bSmallestSizeKnown = FALSE;

    // Clear table instance if necessary
    if (psTable->sInProgress.hTable != TDS_UTIL_INVALID_TABLE_INSTANCE)
    {
        psObj->sCallbacks.vDestroyTable(psTable->sInProgress.hTable);
        psTable->sInProgress.hTable = TDS_UTIL_INVALID_TABLE_INSTANCE;
    }

    return;
}

/*****************************************************************************
*
*   bTableChangeover
*
*****************************************************************************/
static BOOLEAN bTableChangeover (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable
        )
{
    BOOLEAN bSuccess;

    // Send table to the TDS client via callback
    bSuccess = psObj->sCallbacks.bNewTable(
        psTable->sInProgress.hTable,
        psTable->sInProgress.tContentVer,
        psObj->sCallbacks.pvTableArg);
    if (bSuccess == TRUE)
    {
        // Give up the table instance object handle now
        psTable->sInProgress.hTable = TDS_UTIL_INVALID_TABLE_INSTANCE;
    }

    // Update the table versions no matter what just happened --
    // If this table caused this client issues, don't give it to
    // them again.  If this table changeover worked don't give it
    // to them again.
    psTable->sDesc.tDefVer = psTable->sInProgress.tDefVer;
    psTable->sDesc.tContentVer = psTable->sInProgress.tContentVer;

    // Reset in progress info
    vResetInProgressTable(psObj, psTable);

    return bSuccess;
}

/*****************************************************************************
*
*   psGetTableById
*
*****************************************************************************/
static TDS_TABLE_TRACKING_STRUCT *psGetTableById (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_ID tId
        )
{
    TDS_TABLE_TRACKING_STRUCT *psResult = NULL, sSearchCriteria;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Searching for this table
    sSearchCriteria.sDesc.tId = tId;

    // Try to find this table
    eReturnCode = OSAL.eLinkedListSearch(
        psObj->hTables, &hEntry, &sSearchCriteria);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // Grab this entry
        psResult = (TDS_TABLE_TRACKING_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        if (psObj->bRestrictTables == FALSE)
        {
            // TODO: Add an entry if tables aren't restricted
        }
    }

    return psResult;
}

/*****************************************************************************
*
*   bProcessDefinitionMsg
*
*****************************************************************************/
static BOOLEAN bProcessDefinitionMsg (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DEF_VERSION tNewDefVer,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tBitsRead;
    BOOLEAN bRead, bSuccess = FALSE;
    UN8 un8Data, un8NumLabels = 0, un8Index, un8RawDataType;
    UN16 un16Data;
    TDS_TABLE_TIME tTableTime = 0;
    UN8 un8RawLabelSize = 0;
    TDS_LABEL_STRUCT *psLabel = (TDS_LABEL_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Has this version of this table def been
    // marked as something we need to collect?
    if (tNewDefVer == psTable->sInProgress.tDefVer)
    {
        // Yes, now has this table been generated already?
        if (psTable->sInProgress.hTable != TDS_UTIL_INVALID_TABLE_INSTANCE)
        {
            // Yes! Nothing to do here
            return TRUE;
        }
    }
    else if (psTable->sInProgress.tDefVer != TDS_TABLE_INVALID_DEF_VERSION)
    {
        // We don't want this version of this table
        return TRUE;
    }

    // Read the table time
    un16Data = 0;
    bRead = OSAL.bBufferReadBitsToUN16(
        hPayload, &un16Data, TDS_TTIME_BITLEN);
    if (bRead == FALSE)
    {
        // Can't read this payload for some reason
        return TRUE;
    }

    tTableTime = un16Data;

    // Create the table now
    bSuccess = bCreateTable(psObj, psTable, tNewDefVer, tTableTime);
    if (bSuccess == FALSE)
    {
        // Error!  Table creation failed!
        return FALSE;
    }

    // Skip unused bits
    tBitsRead = OSAL.tBufferSeekHeadBits(
        hPayload, TDS_RFU_BITLEN);

    // Read the number of labels
    tBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &un8NumLabels, 0, TDS_LBNO_BITLEN);

    if (tBitsRead != TDS_DEF_HDR_BITLEN)
    {
        // Problem reading this payload, just move on
        return TRUE;
    }

    // SX-9845-0022, Section 4.2
    un8NumLabels++;

    // Iterate over all the labels here
    for (un8Index = 0; un8Index < un8NumLabels; un8Index++)
    {
        // Create a label entry for this info
        psLabel = (TDS_LABEL_STRUCT *)
            OSAL.pvLinkedListMemoryAllocate(
                TDS_OBJECT_NAME":Label",
                sizeof(TDS_LABEL_STRUCT),
                TRUE);
        if (psLabel == NULL)
        {
            return TRUE;
        }

        // Initialize label entry
        psLabel->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        psLabel->bVariableData = FALSE;
        psLabel->eRawStringType = TDS_STRING_RAW_TYPE_ASCII;
        psLabel->un16ArrayLenSize = 0;
        psLabel->un32SymbolCount = 0;
        psLabel->un8SymbolSize = 0;
        psLabel->un8RowOrderId = un8Index;
        psLabel->sEntry.sColumn.tLabelId = TDS_LABEL_INVALID_ID;
        psLabel->sEntry.sColumn.eType = TDS_DATA_INVALID_TYPE;

        do
        {
            // Read the label ID
            un8Data = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8Data, 0, TDS_LBID_BITLEN);
            psLabel->sEntry.sColumn.tLabelId = (TDS_LABEL_ID)un8Data;

            // Read the label type
            tBitsRead += OSAL.tBufferReadHeadBits(
                hPayload, &psLabel->bVariableData,
                0, TDS_LBTYPE_BITLEN);

            // Read the data type
            un8RawDataType = 0;
            tBitsRead += OSAL.tBufferReadHeadBits(
                hPayload, &un8RawDataType, 0, TDS_DTYPE_BITLEN);

            un16Data = 0;
            bRead = OSAL.bBufferReadBitsToUN16(
                hPayload, &un16Data, TDS_LBSIZE_BITLEN);
            if (bRead == TRUE)
            {
                un8RawLabelSize = (UN8)un16Data;
                tBitsRead += TDS_LBSIZE_BITLEN;
            }

            if (tBitsRead != TDS_LABEL_ENTRY_HDR_BITLEN)
            {
                // Stop here
                bSuccess = FALSE;
                break;
            }

            // Refer to SX-9845-0022 Table 11 for all these cases
            if (un8RawDataType == TDS_DTYPE_INT)
            {
                size_t tNumBits;

                // This is an integer
                psLabel->sEntry.sColumn.eType = TDS_DATA_TYPE_INT;
                psLabel->un8SymbolSize = (un8RawLabelSize + 1);
                psLabel->un32SymbolCount = TDS_INT_SYM_COUNT;

                // Calculate how many bits this entry data uses
                tNumBits = (size_t)(psLabel->un8SymbolSize * psLabel->un32SymbolCount);

                // This is the only label type right now that
                // details the actual symbol size of each entry,
                // and as such that means this is the only field
                // that we can use to calculate the smallest symbol
                // size that is ever going to show up in the table data
                if (tNumBits < psTable->sInProgress.tSmallestSymbolSizeInBits)
                {
                    psTable->sInProgress.tSmallestSymbolSizeInBits = tNumBits;
                    psTable->sInProgress.bSmallestSizeKnown = TRUE;
                }
            }
            else if (un8RawDataType == TDS_DTYPE_STRING)
            {
                // This is a c-string
                psLabel->sEntry.sColumn.eType = TDS_DATA_TYPE_STRING;
                psLabel->eRawStringType = TDS_STRING_RAW_TYPE_ASCII;
                psLabel->un8SymbolSize = TDS_CSTRING_SYM_BITLEN;
                psLabel->un32SymbolCount = (un8RawLabelSize + 1);
                psLabel->sEntry.uData.sString.hString = STRING_INVALID_OBJECT;
            }
            else if (un8RawDataType == TDS_DTYPE_BAUDOT)
            {
                // This is a baudot coded string
                psLabel->sEntry.sColumn.eType = TDS_DATA_TYPE_STRING;
                psLabel->eRawStringType = TDS_STRING_RAW_TYPE_BAUDOT;
                psLabel->un8SymbolSize = TDS_BAUDOT_SYM_BITLEN;
                psLabel->un32SymbolCount = (un8RawLabelSize + 1);
                psLabel->sEntry.uData.sString.hString = STRING_INVALID_OBJECT;
            }
            else if (un8RawDataType == TDS_DTYPE_INT_ARRAY)
            {
                UN16 un16NumArrayEntries = 0;

                // This is an int array
                psLabel->sEntry.sColumn.eType = TDS_DATA_TYPE_INT_ARRAY;

                // Read the len size
                un8Data = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload, &un8Data, 0, TDS_LENSIZE_BITLEN);
                if (tBitsRead != TDS_LENSIZE_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

                // SX-9845-0022 Section 4.2.8
                psLabel->un16ArrayLenSize = un8Data + 1;

                // Read the array max field
                bRead = OSAL.bBufferReadBitsToUN16(
                    hPayload, &un16NumArrayEntries,
                    psLabel->un16ArrayLenSize);
                if (bRead == FALSE)
                {
                    bSuccess = FALSE;
                    break;
                }

                // TDS Protocol SX-9845-0022 version 1.6 (November 08, 2016)
                // says actual array max value is ARRAYMAX+1
                ++un16NumArrayEntries;

                // Now, allocate the memory required for this array

                // Clean up any old memory
                if (psLabel->sEntry.uData.sIntArray.pun32Integers != NULL)
                {
                    SMSO_vDestroy((SMS_OBJECT)psLabel->sEntry.uData.sIntArray.pun32Integers);
                }

                // Allocate the required storage space
                psLabel->sEntry.uData.sIntArray.pun32Integers = (UN32 *)
                    SMSO_hCreate(
                        TDS_OBJECT_NAME":IntArray",
                        sizeof(UN32) * un16NumArrayEntries,
                        psObj->hRowDataParent, FALSE);
                if (psLabel->sEntry.uData.sIntArray.pun32Integers == NULL)
                {
                    bSuccess = FALSE;
                    break;
                }

                // Size of each array entry
                psLabel->un8SymbolSize = (un8RawLabelSize + 1);

                // Save the allocated size of the array as the symbol count
                psLabel->un32SymbolCount = (UN32)un16NumArrayEntries;
            }
            else if (un8RawDataType == TDS_DTYPE_SCODE)
            {
                // This is an scode entry that we're gonna
                // translate into a string
                psLabel->sEntry.sColumn.eType = TDS_DATA_TYPE_STRING;
                psLabel->eRawStringType = TDS_STRING_RAW_TYPE_SCODE;
                psLabel->un8SymbolSize = TDS_SCODE_SYM_BITLEN;
                psLabel->un32SymbolCount = un8RawLabelSize + 1;
                psLabel->sEntry.uData.sString.hString = STRING_INVALID_OBJECT;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TDS_OBJECT_NAME": unknown data type: %u",
                    un8RawDataType);

                bSuccess = FALSE;
                break;
            }

            if (psLabel->bVariableData == FALSE)
            {
                // This is a label with constant data
                // Populate the value now using bProcessLabelData
                bSuccess = bProcessLabelData(psObj, hPayload, psTable, psLabel);

                if (bSuccess == FALSE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        TDS_OBJECT_NAME": Unable to process constant label data\n");
                    break;
                }
            }

            // Is the EXTCNT field present?
            if (bFieldPresent(&psTable->sInProgress, hPayload, NULL) == TRUE)
            {
                // Read it
                un8Data = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload, &un8Data, 0, TDS_EXTCNT_BITLEN);

                // SX-9845-0022, Table 7
                un8Data = (un8Data + 1) * 8;

                // Seek past the extra data
                tBitsRead += OSAL.tBufferSeekHeadBits(hPayload, un8Data);


                if (tBitsRead != (size_t)(TDS_EXTCNT_BITLEN + un8Data))
                {
                    bSuccess = FALSE;
                    break;
                }
            }
            // Don't read any extension info since we don't support
            // any of it -- it'll just get freed later
        } while (FALSE);

        if (bSuccess == TRUE)
        {
            // Tell the client about this label
            bSuccess = psObj->sCallbacks.bAddColumn(
                psTable->sInProgress.hTable,
                (BOOLEAN)(un8Index == (un8NumLabels - 1)), // bLastColumn
                &psLabel->sEntry.sColumn,
                psObj->sCallbacks.pvTableArg);

            if (bSuccess == TRUE)
            {
                // Add this label to the list now
                eReturnCode = OSAL.eLinkedListAdd(
                    psTable->sInProgress.hLabels,
                    &psLabel->hEntry,
                    psLabel);
                if (eReturnCode != OSAL_SUCCESS)
                {
                    bSuccess = FALSE;
                }

                // Clear this pointer now
                psLabel = NULL;
            }
        }

        if (bSuccess == FALSE)
        {
            // Stop processing this message
            break;
        }
    }

    if (bSuccess == FALSE)
    {
        // Clear the progress we've made with this table
        vResetInProgressTable(psObj, psTable);

        if (psLabel != NULL)
        {
            bRemoveLabel(psLabel, NULL);
            OSAL.vLinkedListMemoryFree((void *)psLabel);
        }
    }

    // Don't report any errors once we get past
    // the initial validation of inputs
    return TRUE;
}

/*****************************************************************************
*
*   bProcessContentMsg
*
*****************************************************************************/
static BOOLEAN bProcessContentMsg (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_DEF_VERSION tNewDefVer,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tBitsRead;
    BOOLEAN bFirst = FALSE,
            bRead,
            bTableVersionUpdated,
            bProcessedMsg,
            bSuccess = TRUE;
    TDS_TABLE_CONTENT_VERSION tNewContentVer;
    UN16 un16Data;
    UN8 un8Data;

    // Is the Table version field present?
    if (bFieldPresent(&psTable->sInProgress, hPayload, NULL) == FALSE)
    {
        // TODO: Support this when needed (tds_util
        // only supports atomic table updates at this time)
        return TRUE;
    }

    // Read the table version
    un16Data = 0;
    bRead = OSAL.bBufferReadBitsToUN16(hPayload, &un16Data, TDS_TDVER_BITLEN);
    if (bRead == FALSE)
    {
        return TRUE;
    }

    // Use this value
    tNewContentVer = (TDS_TABLE_CONTENT_VERSION)un16Data;

    // Is this an update from the table currently in use?
    bTableVersionUpdated = ((tNewDefVer != psTable->sDesc.tDefVer) ||
                            (tNewContentVer != psTable->sDesc.tContentVer));
    if (bTableVersionUpdated == FALSE)
    {
        // Nothing to do now
        return TRUE;
    }

    // The received table version(s) don't match the
    // versions for the current table -- now
    // determine if we can process this message

    // Is there an in-progress version configured yet?
    if (psTable->sInProgress.tDefVer == TDS_TABLE_INVALID_DEF_VERSION)
    {
        // No - reset the in-progress table to
        // ensure a good clean start
        vResetInProgressTable(psObj, psTable);

        // The in-progress table version is now this
        psTable->sInProgress.tDefVer = tNewDefVer;
        psTable->sInProgress.tContentVer = tNewContentVer;

        // All done -- we need to wait for this table
        // definition to get processed
        return TRUE;
    }

    // Do the in-progress table version(s) match
    // the versions we have here?
    if ((psTable->sInProgress.tDefVer != tNewDefVer) ||
        ((psTable->sInProgress.tContentVer != tNewContentVer) &&
         (psTable->sInProgress.tContentVer != TDS_TABLE_INVALID_CONTENT_VERSION)))
    {
        // No -- this means broadcast is sending out
        // multiple table versions at the same time
        // and we're in the middle of processing another
        // version right now

        // All done!
        return TRUE;
    }

    psTable->sInProgress.tContentVer = tNewContentVer;

    // Now we know we have matching versions --
    // Do we have a table instance to work with?
    if (psTable->sInProgress.hTable == TDS_UTIL_INVALID_TABLE_INSTANCE)
    {
        // Nope -- keep waiting
        return TRUE;
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32NumProcessedMsgs = 0;

        // Is this the first message we're processing for this table update?
        eReturnCode = OSAL.eLinkedListItems(
            psTable->sInProgress.hContentMsgs,
            &un32NumProcessedMsgs);

        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME": Unable to query msg list: %s]\n",
                OSAL.pacGetReturnCodeName(eReturnCode));

            // Clear the progress we've made with this table
            vResetInProgressTable(psObj, psTable);

            return FALSE;
        }

        if (un32NumProcessedMsgs == 0)
        {
            // This is the first message for this table update
            bFirst = TRUE;
        }
    }

    // Seek past the entry count (not used)
    tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, TDS_ENTCT_BITLEN);

    // Read the "count size"
    un8Data = 0;
    tBitsRead += OSAL.tBufferReadHeadBits(
        hPayload, &un8Data, 0, TDS_CTSIZE_BITLEN);

    // Verify we read those without issue
    if (tBitsRead != (TDS_ENTCT_BITLEN + TDS_CTSIZE_BITLEN))
    {
        return TRUE;
    }

    // SX-9845-0022 Section 5.1.7
    un8Data++;

    // Read the total number of access units which
    // comprise this update
    un16Data = 0;
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un16Data, 0, un8Data);
    if (tBitsRead != un8Data)
    {
        return TRUE;
    }

    // Only do this once
    if (bFirst == TRUE)
    {
        // SX-9845-0022 Section 1.8
        // This field is crazy big for some reason -- 65k messages! really?
        psTable->sInProgress.tNumEntryMessagesLeft = (size_t)(un16Data);
        psTable->sInProgress.tNumEntryMessagesLeft++;
    }

    // Read this access unit's sequence number
    un16Data = 0;
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &un16Data, 0, un8Data);
    if (tBitsRead != un8Data)
    {
        return TRUE;
    }

    // Is this a message we've already processed?
    bProcessedMsg = bIsProcessedMsg(psTable, (size_t)un16Data);
    if (bProcessedMsg == FALSE)
    {
        // Nope -- process it now
        size_t tMsgId;

        // Save the message id now
        tMsgId = (size_t)un16Data;

        // Process all the labels present in this message
        bSuccess = bProcessLabels(psObj, psTable, hPayload);

        if (bSuccess == TRUE)
        {
            // Mark this message as processed
            bSuccess = bMarkMsgProcessed(psTable, tMsgId);
        }

        if (bSuccess == TRUE)
        {
            // Reduce the number of entry messages left for this update
            psTable->sInProgress.tNumEntryMessagesLeft--;

            if (psTable->sInProgress.tNumEntryMessagesLeft == 0)
            {
                // Table is complete
                bSuccess = bTableChangeover(psObj, psTable);
            }
        }
    }

    if (bSuccess == FALSE)
    {
        // Clear the progress we've made with this table
        vResetInProgressTable(psObj, psTable);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessLabels
*
*   We have no idea how many entries are in this message, so just process
*   every single bit until there are no more.  Any padding bits will be set
*   to zero, so at least there's that.
*
*****************************************************************************/
static BOOLEAN bProcessLabels (
    TDS_OBJECT_STRUCT *psObj,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    OSAL_BUFFER_HDL hPayload
        )
{
    OSAL_LINKED_LIST_ENTRY hEntry;
    TDS_LABEL_STRUCT *psLabel = (TDS_LABEL_STRUCT *)NULL;
    BOOLEAN bBufferEmpty,
            bSuccess = FALSE,
            bStartNewRow = TRUE,
            bRowsReported = FALSE,
            bInformClient = FALSE;
    UN8 un8PrevRowOrderId;

    // Grab the first label descriptor
    hEntry = OSAL.hLinkedListFirst(
        psTable->sInProgress.hLabels,
        (void **)&psLabel);
    if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
        (psLabel == NULL))
    {
        return FALSE;
    }


    do
    {
        // Is this label present in the message?
        if (bFieldPresent(&psTable->sInProgress, hPayload, &bBufferEmpty))
        {
            // Yes, process the label
            bSuccess = bProcessLabelData(psObj, hPayload, psTable, psLabel);
            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    TDS_OBJECT_NAME": Unable to process label %d for table %d\n",
                    psLabel->sEntry.sColumn.tLabelId, psTable->sDesc.tId);
                break;
            }

            bInformClient = TRUE;
        }
        // No, but does this label have constant data?
        else if (psLabel->bVariableData == FALSE)
        {
            // Yes - Inform client of all constant data
            bInformClient = TRUE;
        }

        if (bInformClient == TRUE)
        {
            // Are we starting a new row?
            if (bStartNewRow == TRUE)
            {
                // Do we need to end the previous row?
                if (bRowsReported == TRUE)
                {
                    // Yes -- end it now
                    bSuccess = psObj->sCallbacks.bRowStatus(
                        psTable->sInProgress.hTable,
                        TDS_ROW_OPERATION_END_ROW,
                        psObj->sCallbacks.pvTableArg);
                    if (bSuccess == FALSE)
                    {
                        break;
                    }
                }

                // Start the new row
                bSuccess = psObj->sCallbacks.bRowStatus(
                    psTable->sInProgress.hTable,
                    TDS_ROW_OPERATION_START_ROW,
                    psObj->sCallbacks.pvTableArg);
                if (bSuccess == FALSE)
                {
                    break;
                }
            }

            // Tell the client to update their table contents
            bSuccess = psObj->sCallbacks.bAddRowData(
                psTable->sInProgress.hTable,
                &psLabel->sEntry,
                psObj->sCallbacks.pvTableArg );
            if (bSuccess == FALSE)
            {
                break;
            }

            // From now on we've reported one or
            // more rows
            bRowsReported = TRUE;

            // Reset flag
            bInformClient = FALSE;
        }

        // Save this label's row order id
        un8PrevRowOrderId = psLabel->un8RowOrderId;

        // Grab the next entry (wraps around for convenience)
        psLabel = (TDS_LABEL_STRUCT *)NULL;
        hEntry = OSAL.hLinkedListNext(hEntry, (void **)&psLabel);

        // Ensure the entry is valid (it should always be since
        // this is a circular list)
        if ((hEntry == OSAL_INVALID_LINKED_LIST_ENTRY) ||
            (psLabel == NULL))
        {
            bSuccess = FALSE;
            break;
        }

        // This next label starts a new row if its
        // order id is <= the last row id
        bStartNewRow = (psLabel->un8RowOrderId <= un8PrevRowOrderId);

    } while (bBufferEmpty == FALSE);

    if ((bSuccess == TRUE) && (bRowsReported == TRUE))
    {
        // End this last row
        bSuccess = psObj->sCallbacks.bRowStatus(
            psTable->sInProgress.hTable,
            TDS_ROW_OPERATION_END_ROW,
            psObj->sCallbacks.pvTableArg);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessLabelData
*
*****************************************************************************/
static BOOLEAN bProcessLabelData (
    TDS_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_LABEL_STRUCT *psLabel
        )
{
    BOOLEAN bSuccess = FALSE;

    switch (psLabel->sEntry.sColumn.eType)
    {
        case TDS_DATA_TYPE_INT:
        {
            // Read the integer
            psLabel->sEntry.uData.sInt.un32Data = 0;

            bSuccess = OSAL.bBufferReadBitsToUN32(
                hPayload,
                &psLabel->sEntry.uData.sInt.un32Data,
                psLabel->un8SymbolSize);
        }
        break;

        case TDS_DATA_TYPE_INT_ARRAY:
        {
            UN32 un32Index, un32ArrayLen = 0;

            // Verify the array is present
            if (psLabel->sEntry.uData.sIntArray.pun32Integers == NULL)
            {
                break;
            }

            // Read the array size
            bSuccess = OSAL.bBufferReadBitsToUN32(
                hPayload,
                &un32ArrayLen,
                psLabel->un16ArrayLenSize);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Verify we won't overstep our bounds
            if (un32ArrayLen > psLabel->un32SymbolCount)
            {
                break;
            }

            // Store this array length
            psLabel->sEntry.uData.sIntArray.tArraySize = (size_t)un32ArrayLen;

            for (un32Index = 0; un32Index < un32ArrayLen; un32Index++)
            {
                // Read the array entry data
                psLabel->sEntry.uData.sIntArray.pun32Integers[un32Index] = 0;

                bSuccess = OSAL.bBufferReadBitsToUN32(
                    hPayload,
                    &psLabel->sEntry.uData.sIntArray.pun32Integers[un32Index],
                    psLabel->un8SymbolSize);

                if (bSuccess == FALSE)
                {
                    break;
                }
            }
        }
        break;

        case TDS_DATA_TYPE_STRING:
        {
            bSuccess = bProcessStringData(psObj, hPayload, psLabel);
        }
        break;

        case TDS_DATA_INVALID_TYPE:
        default:
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                TDS_OBJECT_NAME": Unknown label type: %u\n",
                psLabel->sEntry.sColumn.eType);
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessStringData
*
*****************************************************************************/
static BOOLEAN bProcessStringData (
    TDS_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    TDS_LABEL_STRUCT *psLabel
        )
{
    BOOLEAN bSuccess = TRUE;

    // Ensure the string handle is invalid first
    if (psLabel->sEntry.uData.sString.hString != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psLabel->sEntry.uData.sString.hString);
        psLabel->sEntry.uData.sString.hString = STRING_INVALID_OBJECT;
    }

    switch (psLabel->eRawStringType)
    {
        case TDS_STRING_RAW_TYPE_ASCII:
        {
            // Create the string directly from the buffer
            psLabel->sEntry.uData.sString.hString =
                STRING_hCreateFromBuffer(
                    psObj->hRowDataParent, hPayload,
                    (UN8)psLabel->un8SymbolSize,
                    psLabel->un32SymbolCount);
        }
        break;

        case TDS_STRING_RAW_TYPE_BAUDOT:
        {
            // Read Baudot data
            psLabel->sEntry.uData.sString.hString =
                BAUDOT_hToString(
                    psObj->hRowDataParent,
                    hPayload,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, psLabel->un32SymbolCount,
                    (size_t *)NULL);
        }
        break;

        case TDS_STRING_RAW_TYPE_SCODE:
        {
            UN32 un32Index;
            UN8 un8SCode;
            const char *pacTranslated;
            size_t tAppended, tNumBitsRead;

            // We must have a lookup table for scodes!
            if (psObj->sScodeLUT.ppacSCodeLUT == NULL)
            {
                bSuccess = FALSE;
                break;
            }

            // Create an empty string and tell it that it needs
            // to be big enough to hold the largest scode-based string
            // possible for this label
            psLabel->sEntry.uData.sString.hString = STRING_hCreate(
                psObj->hRowDataParent, "", 0, psLabel->un32SymbolCount);

            if (psLabel->sEntry.uData.sString.hString == STRING_INVALID_OBJECT)
            {
                break;
            }

            // Iterate through all the codes now
            for (un32Index = 0; un32Index < psLabel->un32SymbolCount; un32Index++)
            {
                // Read the next scode
                un8SCode = 0;
                tNumBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload, &un8SCode, 0, psLabel->un8SymbolSize);
                if (tNumBitsRead != psLabel->un8SymbolSize)
                {
                    break;
                }

                // Bounds-check!
                if (un8SCode >= psObj->sScodeLUT.tNumEntriesInLUT)
                {
                    bSuccess = FALSE;
                    break;
                }

                if (un8SCode == 0)
                {
                    // SX-9845-0022, Section 8.3.3: SCode has been terminated

                    break;
                }

                // Get the translated value for this scode
                pacTranslated = psObj->sScodeLUT.ppacSCodeLUT[un8SCode];

                // Append the string with this new data now
                tAppended = STRING.tAppendCStr(psLabel->sEntry.uData.sString.hString, pacTranslated);

                if (tAppended == 0)
                {
                    bSuccess = FALSE;
                    break;
                }
            }
        }
        break;

        default:
        {
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bFieldPresent
*
*   A simple function which reads 1 bit and reports TRUE or FALSE
*   based on that bit's value.
*
*****************************************************************************/
static BOOLEAN bFieldPresent(
    TDS_IN_PROGRESS_STRUCT *psInProgress,
    OSAL_BUFFER_HDL hBuffer,
    BOOLEAN *pbBufferEmpty
        )
{
    size_t tBufferSizeInBits;
    BOOLEAN bPresent = FALSE;
    size_t tBitsRead, tSmallestSymbolInBits = 0;

    // Initialize if necessary
    if (pbBufferEmpty != NULL)
    {
        *pbBufferEmpty = FALSE;
    }

    // Do we have a known smallest symbol size?
    if (TRUE == psInProgress->bSmallestSizeKnown)
    {
        // Yes, use it
        tSmallestSymbolInBits = psInProgress->tSmallestSymbolSizeInBits;
    }

    // Get buffer length
    tBufferSizeInBits = OSAL.tBufferGetSizeInBits(hBuffer);

    // Is the buffer smaller than our smallest symbol size
    // with the presence flag?
    if (tBufferSizeInBits < (tSmallestSymbolInBits + TDS_PRESENCE_FLAG_BITLEN))
    {
        if (pbBufferEmpty != NULL)
        {
            *pbBufferEmpty = TRUE;
        }

        return FALSE;
    }

    // We probably have enough data in this buffer to
    // extract meaninful data, so continue on that assumption

    // Read the presence flag
    tBitsRead = OSAL.tBufferReadHeadBits(
        hBuffer, &bPresent, 0, TDS_PRESENCE_FLAG_BITLEN);

    if (tBitsRead != TDS_PRESENCE_FLAG_BITLEN)
    {
        // Indicate presence if we can't read the buffer
        // to force the caller to fail as well
        return TRUE;
    }

    return bPresent;
}

/*****************************************************************************
*
*   bIsProcessedMsg
*
*****************************************************************************/
static BOOLEAN bIsProcessedMsg (
    TDS_TABLE_TRACKING_STRUCT *psTable,
    size_t tMsgId
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;

    eReturnCode = OSAL.eLinkedListSearch(
        psTable->sInProgress.hContentMsgs, &hEntry, (void *)tMsgId);
    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }
    else if (eReturnCode == OSAL_OBJECT_NOT_FOUND)
    {
        return FALSE;
    }

    // Just tell the caller that this message
    // has been processed since we have no idea
    // what the deal is here
    return TRUE;
}

/*****************************************************************************
*
*   bMarkMsgProcessed
*
*****************************************************************************/
static BOOLEAN bMarkMsgProcessed (
    TDS_TABLE_TRACKING_STRUCT *psTable,
    size_t tMsgId
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Add this message ID to the list
    eReturnCode = OSAL.eLinkedListAdd(
        psTable->sInProgress.hContentMsgs, OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
        (void *)tMsgId);

    return (eReturnCode == OSAL_SUCCESS);
}

/*******************************************************************************
*
*   n16CompareTables
*
*   This function compares two TDS_TABLE_TRACKING_STRUCT and orders
*   them by table id.
*
*   Inputs:
*       psUpdate1 (in list), psUpdate2 (compare against)
*
*   Outputs:
*       0   - Objects have the same value (equal, error)
*       > 0 - Object1(in list) is greater than (after) Object2
*       < 0 - Object1(in list) is less than (before) Object2
*
*******************************************************************************/
static N16 n16CompareTables(
    TDS_TABLE_TRACKING_STRUCT *psTable1,
    TDS_TABLE_TRACKING_STRUCT *psTable2
        )
{
    N16 n16Result;

    if ((psTable1 == NULL) || (psTable2 == NULL))
    {
        return N16_MIN;
    }

    // Compare these two
    n16Result = (N16)(psTable1->sDesc.tId - psTable2->sDesc.tId);

    return n16Result;
}

/*****************************************************************************
*
*   bRemoveTableEntry
*
*****************************************************************************/
static BOOLEAN bRemoveTableEntry(
    TDS_TABLE_TRACKING_STRUCT *psTable,
    TDS_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    if (psTable == NULL)
    {
        // Weird!
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            TDS_OBJECT_NAME": Table list corrupted!\n");

        return FALSE;
    }

    // Clear the table descriptor
    psTable->sDesc.tId = TDS_TABLE_INVALID_ID;
    psTable->sDesc.tDefVer = TDS_TABLE_INVALID_DEF_VERSION;
    psTable->sDesc.tContentVer = TDS_TABLE_INVALID_CONTENT_VERSION;
    psTable->sDesc.tDefCarouselId = TDS_INVALID_CAROUSEL_ID;
    psTable->sDesc.tContentCarouselId = TDS_INVALID_CAROUSEL_ID;

    // Clear any data for the in-progress table
    vResetInProgressTable(psObj, psTable);

    // Remove the entry if necessary -- if we have a valid entry
    // handle it means this was added to the list at some point and
    // we want to free the memory associated with this entry now
    if (psTable->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Remove this entry (this will invalidate the psTable pointer)
        eReturnCode = OSAL.eLinkedListRemove(psTable->hEntry);

        // Entry has been removed, free the associated memory now
        if (eReturnCode == OSAL_SUCCESS)
        {
            psTable->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            OSAL.vLinkedListMemoryFree(psTable);
        }
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bRemoveLabel
*
*****************************************************************************/
static BOOLEAN bRemoveLabel (
    TDS_LABEL_STRUCT *psLabel,
    void *pvUnused
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    switch (psLabel->sEntry.sColumn.eType)
    {
        case TDS_DATA_TYPE_INT:
        {
            psLabel->sEntry.uData.sInt.un32Data = 0;
        }
        break;

        case TDS_DATA_TYPE_INT_ARRAY:
        {
            if (psLabel->sEntry.uData.sIntArray.pun32Integers != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psLabel->sEntry.uData.sIntArray.pun32Integers);
                psLabel->sEntry.uData.sIntArray.pun32Integers = NULL;
            }

            psLabel->sEntry.uData.sIntArray.tArraySize = 0;
        }
        break;

        case TDS_DATA_TYPE_STRING:
        {
            if (psLabel->sEntry.uData.sString.hString != STRING_INVALID_OBJECT)
            {
                STRING_vDestroy(psLabel->sEntry.uData.sString.hString);
                psLabel->sEntry.uData.sString.hString = STRING_INVALID_OBJECT;
            }
        }
        break;

        case TDS_DATA_INVALID_TYPE:
        {
            break;
        }

        // No default to force warning
    }

    // Reset attributes
    psLabel->sEntry.sColumn.eType = TDS_DATA_INVALID_TYPE;
    psLabel->bVariableData = FALSE;
    psLabel->eRawStringType = TDS_STRING_RAW_TYPE_ASCII;
    psLabel->un32SymbolCount = 0;
    psLabel->un8SymbolSize = 0;

    // Remove the entry if necessary -- if we have a valid entry
    // handle it means this was added to the list at some point and
    // we want to free the memory associated with this entry now
    if (psLabel->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        // Remove this entry (this will invalidate the psTable pointer)
        eReturnCode = OSAL.eLinkedListRemove(psLabel->hEntry);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psLabel->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            OSAL.vLinkedListMemoryFree(psLabel);
        }
    }

    if (eReturnCode == OSAL_SUCCESS)
    {
        return TRUE;
    }

    return FALSE;
}

