/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Stock Ticker protocol implementation for the
 *  Simple Module Services (SMS)
 *
 ******************************************************************************/

#include "_stock_ticker_protocol.h"
#include "stock_ticker_interface.h"
#include "string_obj.h"
#include "string.h"
#include "sms.h"
#include "sms_obj.h"
#include "dataservice_mgr_impl.h"
#include "xmappid_packet_util.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/
/*****************************************************************************
*
*   hInit
*
*****************************************************************************/
static STOCK_TICKER_INTERFACE_OBJECT hInit(
    STOCK_TICKER_SERVICE_OBJECT hService,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bOwner;
    STOCK_TICKER_INTERFACE_OBJECT hResult = STOCK_TICKER_INTERFACE_INVALID_OBJECT;
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj = NULL;
    OSAL_RETURN_CODE_ENUM eOsalReturnCode;
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode;

    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": Initializing for service 0x%p\n",
           hService);

    do
    {
        // Verify we own service handle
        bOwner = SMSO_bOwner(hParent);
        if (bOwner == FALSE)
        {
            break;
        }

        // Create an instance of this object
        psObj = (STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *)
            SMSO_hCreate(
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME": Obj",
                    sizeof(STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT),
                    hParent, FALSE);
        if (psObj == NULL)
        {
            break;
        }

        OSAL.bMemSet(psObj, 0, sizeof(*psObj));
        psObj->hStockTickerService = hService;
        psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
        psObj->pacBuffer = NULL;
        psObj->tBufferSize = 0;

        // Request access to ISO 3309 CRC32
        eOsalReturnCode = OSAL.eGetCRC(
                &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to create CRC handler (%s)",
                    OSAL.pacGetReturnCodeName(eOsalReturnCode));
            break;
        }

        // Initialize default buffer
        eReturnCode = eEnsureBufferEnough(psObj, STOCK_TICKER_DATA_BUFFER_SIZE);
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to initialize buffer (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode) );
            break;
        }

        hResult = (STOCK_TICKER_INTERFACE_OBJECT)psObj;
    } while (FALSE);

    if (hResult == STOCK_TICKER_INTERFACE_INVALID_OBJECT)
    {
        vUninitObject(psObj);
    }

    return hResult;
}

/*****************************************************************************
*
*   vUnInit
*
*****************************************************************************/
static void vUnInit(
    STOCK_TICKER_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner = FALSE;
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj = (STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT*)hInterface;

    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": Uninitializing\n");

    do
    {
        // Check input
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        vUninitObject(psObj);
    } while (FALSE);

    return;
}

/*****************************************************************************
*
*   eProcessMessage
*
*****************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessage(
    STOCK_TICKER_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
            )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    BOOLEAN bOwner = FALSE;
    BOOLEAN bDoTouch = FALSE;
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj = (STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT*)hInterface;
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psMsgData = NULL;
    STOCKS_PAYLOAD_HEADER_STRUCT sHeader;
#if SMS_DEBUG == 1
    size_t tBufferSize = 0;
#endif

    puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME": Processing payload");

    do
    {
        // Check input
        if ((hInterface == STOCK_TICKER_INTERFACE_INVALID_OBJECT) ||
            (phPayload == NULL))
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": invalid input (hInterface = %p, phPayload = %p)",
                    hInterface, phPayload
                        );
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Check ownership
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Process message header
        eResult = eProcessMessageHeader(psObj, *phPayload, &sHeader);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            if (eResult == STOCK_TICKER_RETURN_CODE_BAD_XMAPP_HDR)
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to parse payload header (%s)\n",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
                eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            }
            else
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to parse payload header (%s)",
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
            }
            break;
        }

        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": found message %s\n",
                GsStockTickerMgrIntf.pacGetMessageTypeName(sHeader.eMsgType)
                    );

        // Get message parser data
        psMsgData = psGetMsgParserData(psObj, sHeader.eMsgType);
        if (psMsgData == NULL)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to get message parse data for %s",
                    GsStockTickerMgrIntf.pacGetMessageTypeName(sHeader.eMsgType));
            break;
        }

        // Update message parser data by new block
        eResult = eUpdateMsgParserData(psObj, psMsgData, phPayload, &sHeader);
        if (eResult == STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER)
        {
            puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": not enough data to do something. Just skip...");
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }
        else if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to update message parse data for %s (%s)",
                    GsStockTickerMgrIntf.pacGetMessageTypeName(sHeader.eMsgType),
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        // In case if the message data block is ready to be processed at once
        // the handler as READY state. If not - just wait another payload
        if (psMsgData->eState != STOCKS_MSG_PARSER_STATE_READY)
        {
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": nothing to parse for message %s. "
                "Waiting for the next payload...\n",
                GsStockTickerMgrIntf.pacGetMessageTypeName(sHeader.eMsgType)
                    );
            break;
        }

#if SMS_DEBUG == 1
        tBufferSize = OSAL.tBufferGetSize(psMsgData->hBuffer);
        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
            ": Payload to process is %d byte(s)\n", tBufferSize
                    );
#endif
        // Check the payload for processing necessity
        eResult = eIsProcessingRequired(psObj, psMsgData);
        if ((eResult == STOCK_TICKER_RETURN_CODE_FALSE) ||
            (eResult == STOCK_TICKER_RETURN_CODE_IO_ERROR))
        {
            // Just skip this package
            puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": payload is not required. Skip it.");
            psMsgData->eState = STOCKS_MSG_PARSER_STATE_SKIPPED;
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }
        else if (eResult == STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH)
        {
            // The package has the same datas as we already processed, but
            // the messages expiration timer shall be reset in order to
            // prevent premature invalidation.
            puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": payload is not unique, "
                    "but related messages shall be touched.");
            bDoTouch = TRUE;
        }
        else if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            // Well, the package shall be parsed and populated to the
            // service
            puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME": payload is needed");
        }
        else
        {
            // Ooops, wrong error code
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to check payload (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        eResult =
            GsStockTickerMgrIntf.ePayloadProcessStarted(
                psObj->hStockTickerService);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to START processing via Manager");
            break;
        }

        switch (sHeader.eMsgType)
        {
            case STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL:
            {
                eResult =
                    eProcessMessageStockSymbol(psObj, psMsgData, bDoTouch);
            }
            break;
            case STOCK_TICKER_MSG_TYPE_STOCK_VALUE:
            {
                eResult =
                    eProcessMessageStockValue(psObj, psMsgData, bDoTouch);
            }
            break;
            case STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX:
            {
                eResult =
                    eProcessMessageStockValueEx(psObj, psMsgData, bDoTouch);
            }
            break;
            case STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG:
            {
                eResult =
                    eProcessMessageDataProviderMsg(psObj, psMsgData, bDoTouch);
            }
            break;
            case STOCK_TICKER_MSG_TYPE_INVALID:
            case STOCK_TICKER_MSG_TYPE_MAX:
            {
                // Just to avoid compiler warning for this type and
                // let compiler make a warning if some enum value is not
                // handled here.
            }
            break;
        }

#if SMS_DEBUG == 1
        // In case of any error just report the log about this
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": payload processing finished by error code %s",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
        }
#endif
        // Regardless of current error code try to commit data passed
        // to the Manager.
        eResult =
            GsStockTickerMgrIntf.ePayloadProcessCommited(
                psObj->hStockTickerService);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to COMMIT via Manager");
        }
    } while (FALSE);

    if ((psMsgData != NULL) &&
        ((psMsgData->eState == STOCKS_MSG_PARSER_STATE_READY) ||
         (psMsgData->eState == STOCKS_MSG_PARSER_STATE_SKIPPED)))
    {
        // Complete processing and wait for the next paylaod to continue
        vCompleteMsgParserData(psMsgData);
    }

    return eResult;
}

/*******************************************************************************
*
*        eState
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eState (
    STOCK_TICKER_INTERFACE_OBJECT hInterface
        )
{
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj = (STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT*)hInterface;
    STOCK_TICKER_MSG_TYPE_ENUM eIndex;
    BOOLEAN bOwner = FALSE;

    do
    {
        // Check ownership
        bOwner = SMSO_bOwner((SMS_OBJECT)psObj);
        if (bOwner == FALSE)
        {
            break;
        }

        puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME":");

        // Print out the state of the parser data

        for (eIndex = STOCK_TICKER_MSG_TYPE_START;
             eIndex < STOCK_TICKER_MSG_TYPE_MAX;
             ++eIndex)
        {
            STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData = psObj->apsMsgParserDataList[eIndex];
            if (psData != NULL)
            {
                size_t tBufferSize = 0;
                if (psData->hBuffer != OSAL_INVALID_BUFFER_HDL)
                {
                    tBufferSize = OSAL.tBufferGetSize(psData->hBuffer);
                }

                printf("\tData[#%d] => %s\n\t\tState = %d, seqId = %4d, msgSeqId = %4d, hBuf = %p [%d byte(s)], cnt = %d\n",
                            eIndex,
                            GsStockTickerMgrIntf.pacGetMessageTypeName(psData->eMsgType),
                            psData->eState,
                            psData->un32SeqNumber,
                            psData->un32MsgSeqNumber,
                            psData->hBuffer,
                            tBufferSize,
                            psData->un32PayloadPorcessedCounter
                                );
            }
            else
            {
                printf("\tData[#%d] is empty\n", eIndex);
            }
        }
    } while (FALSE);

    return STOCK_TICKER_RETURN_CODE_SUCCESS;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/
/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/*******************************************************************************
*
*        eProcessMessageHeader
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageHeader(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    STOCKS_PAYLOAD_HEADER_STRUCT *psHeader
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    XM_APPID tAppID;
    XM_SEQID tSeqID;
    BOOLEAN bOk = FALSE;

    do
    {
        // Check input
        if ((psObj == NULL) || (hPayload == OSAL_INVALID_BUFFER_HDL) ||
            (psHeader == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Extract XMAPP data
        bOk = XMAPPID_PACKET_UTIL_bParsePayloadFromPacket(hPayload, &tAppID, &tSeqID);
        if (bOk == FALSE)
        {
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": [%d] failed to process header to extract XMAPP data\n",
                __LINE__
                    );
            eResult = STOCK_TICKER_RETURN_CODE_BAD_XMAPP_HDR;
            break;
        }

        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": App %d, Seq %d\n", tAppID, tSeqID);

        // Detect message type defined as AppID
        switch (tAppID)
        {
            case STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_SYMBOL:
            case STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_VALUE:
            case STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_VALUE_EX:
            case STOCK_TICKER_MSG_PROTOCOL_TYPE_DATA_PROVIDER_MSG:
            {
                psHeader->eMsgType =
                    (STOCK_TICKER_MSG_TYPE_ENUM)
                        (tAppID - STOCK_TICKER_MSG_PROTOCOL_TYPE_START);
                psHeader->un32SeqNumber = tSeqID;
                eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
            }
            break;
            default:
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": unknown message type %d",
                        tAppID
                            );
                psHeader->eMsgType = STOCK_TICKER_MSG_TYPE_INVALID;
                psHeader->un32SeqNumber = STOCK_TICKER_INVALID_SEQID;
                eResult = STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE;
            }
            break;
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eCheckMessageStockSymbol
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockSymbol (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tBufferSize;

    do
    {
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize < STOCK_TICKER_MGS_SYMBOL_HDR_BYTELEN)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
            break;
        }

        if (psObj->bDataIsCollected == FALSE)
        {
            size_t tPeekSize;

            // Peek several message header fields to calculate required buffer size
            tPeekSize = OSAL.tBufferPeek(psObj->hBuffer,
                            &psObj->uData.sSymbol.un8SymbolCount,
                            STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE,
                            STOCK_TICKER_MGS_SYMBOL_COUNT_OFFSET_BYTESIZE
                                );
            tPeekSize += OSAL.tBufferPeek(psObj->hBuffer,
                            &psObj->uData.sSymbol.un8SymbolSize,
                            STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE,
                            STOCK_TICKER_MGS_SYMBOL_SIZE_OFFSET_BYTESIZE
                                );
            if (tPeekSize != (STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE +
                              STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE))
            {
                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }

            // Check data correctness
            if (psObj->uData.sSymbol.un8SymbolSize == 0)
            {
                eResult = STOCK_TICKER_RETURN_CODE_BAD_DATA;
                break;
            }

            // The data is ready-to-use
            psObj->bDataIsCollected = TRUE;
        }

        // Check message size left for data
        if (tBufferSize <
                (size_t)((psObj->uData.sSymbol.un8SymbolCount *
                          psObj->uData.sSymbol.un8SymbolSize) +
                          STOCK_TICKER_MGS_SYMBOL_HDR_BYTELEN))
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eCheckMessageStockValue
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockValue (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tBufferSize;

    do
    {
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize < STOCK_TICKER_MGS_VALUE_HDR_BYTELEN)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
            break;
        }

        if (psObj->bDataIsCollected == FALSE)
        {
            size_t tPeekSize;
            BOOLEAN bOk;

            // Peek number of symbols in the message
            tPeekSize = OSAL.tBufferPeek(psObj->hBuffer,
                            &psObj->uData.sValues.un8Count,
                            STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE,
                            STOCK_TICKER_MGS_VALUE_COUNT_OFFSET_BYTESIZE
                                );
            if (tPeekSize != STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE)
            {
                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }

            // Peek index of the first addressed stock symbol
            bOk = OSAL.bBufferPeekBitsToUN16(psObj->hBuffer,
                            &psObj->uData.sValues.un16StartIndex,
                            STOCK_TICKER_MGS_VALUE_INDEX_BITSIZE,
                            STOCK_TICKER_MGS_VALUE_INDEX_OFFSET_BITSIZE);
            if (bOk == FALSE)
            {
                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }

            // Data cached and ready-to-use
            psObj->bDataIsCollected = TRUE;
        }

        // Check total size
        if (tBufferSize <
                (size_t)((psObj->uData.sValues.un8Count *
                          STOCK_TICKER_MGS_VALUE_SYMBOL_BYTESIZE) +
                          STOCK_TICKER_MGS_VALUE_HDR_BYTELEN))
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eCheckMessageStockValueEx
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockValueEx (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tBufferSize;

    do
    {
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize < STOCK_TICKER_MGS_VALUE_EX_HDR_BYTELEN)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
            break;
        }

        if (psObj->bDataIsCollected == FALSE)
        {
            size_t tPeekSize;
            // Peek number of symbols in the message
            tPeekSize = OSAL.tBufferPeek(psObj->hBuffer,
                                    &psObj->uData.sValuesEx.un8Count,
                                    STOCK_TICKER_MGS_VALUE_EX_COUNT_BYTESIZE,
                                    STOCK_TICKER_MGS_VALUE_EX_COUNT_OFFSET_BYTESIZE
                                        );
            if (tPeekSize != STOCK_TICKER_MGS_VALUE_EX_COUNT_BYTESIZE)
            {
                eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
                break;
            }

            // Data cached and ready-to-use
            psObj->bDataIsCollected = TRUE;
        }

        // Check total size
        if (tBufferSize <
                (size_t)((psObj->uData.sValuesEx.un8Count *
                              STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE) +
                          STOCK_TICKER_MGS_VALUE_EX_HDR_BYTELEN))
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eCheckMessageDataProviderMsg
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageDataProviderMsg (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tBufferSize;


    do
    {
        tBufferSize = OSAL.tBufferGetSize(psObj->hBuffer);
        if (tBufferSize < STOCK_TICKER_MGS_DATA_PROVIDER_HDR_BYTESIZE)
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
            break;
        }

        if (psObj->bDataIsCollected == FALSE)
        {
            size_t tPeekSize;

            // Peek size of the provider info string
            tPeekSize =
                OSAL.tBufferPeek(psObj->hBuffer,
                    &psObj->uData.sProviderInfo.un8TextSize,
                    STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_BYTESIZE,
                    STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_OFFSET_BYTESIZE
                        );
            if (tPeekSize != STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_BYTESIZE)
            {
                eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
                break;
            }

            // Data cached and ready-to-use
            psObj->bDataIsCollected = TRUE;
        }

        // Check total size
        if (tBufferSize <
                (size_t)(psObj->uData.sProviderInfo.un8TextSize +
                                STOCK_TICKER_MGS_DATA_PROVIDER_HDR_BYTESIZE))
        {
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eCheckMessage
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessage (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;

    switch (psObj->eMsgType)
    {
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE:
        {
            eResult = eCheckMessageStockValue(psObj);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX:
        {
            eResult = eCheckMessageStockValueEx(psObj);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL:
        {
            eResult = eCheckMessageStockSymbol(psObj);
        }
        break;
        case STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG:
        {
            eResult = eCheckMessageDataProviderMsg(psObj);
        }
        break;
        default:
        {
            eResult = STOCK_TICKER_RETURN_CODE_UNKNOWN_MSG_TYPE;
        }
        break;
    }

    return eResult;
}

/*******************************************************************************
*
*        eProcessMessageStockSymbol
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockSymbol(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hPayload = psData->hBuffer;
    BOOLEAN bReadOk = FALSE;
    size_t tReadBytes = 0;
    size_t tBufferSize = 0;
    STOCK_ID tSymbolIter = 0;
    UN8 un8SeqNumber = 0;
    UN8 un8SymbolCount = 0;
    UN8 un8SymbolSize = 0;
    UN16 un16SymbolStartIndex = 0;

    puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
        ": Processing Stock Symbol message");

    do
    {
        // Read message header
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8SeqNumber,
                        STOCK_TICKER_MGS_SYMBOL_SEQ_NUMBER_BYTESIZE);
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8SymbolCount,
                        STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE);
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8SymbolSize,
                        STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE);
        bReadOk = OSAL.bBufferReadBitsToUN16(hPayload, &un16SymbolStartIndex,
                        STOCK_TICKER_MGS_SYMBOL_INDEX_BITSIZE);
        if (bReadOk == TRUE)
        {
            tReadBytes += STOCK_TICKER_MGS_SYMBOL_INDEX_BYTESIZE;
        }

        if (tReadBytes != STOCK_TICKER_MGS_SYMBOL_HDR_BYTELEN)
        {
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                       STOCK_TICKER_PROTOCOL_OBJECT_NAME
                       ": failed to read header (%d of %d byte(s)",
                       tReadBytes, STOCK_TICKER_MGS_SYMBOL_HDR_BYTELEN
                            );
             eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
             break;
        }

        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
            ": Seq %2d, Count %2d, Size %2d, Start Index %d\n",
            un8SeqNumber, un8SymbolCount, un8SymbolSize, un16SymbolStartIndex);

        // Check message size left for data
        tBufferSize = OSAL.tBufferGetSize(hPayload);
        tReadBytes = un8SymbolCount * un8SymbolSize;
        if (tBufferSize < tReadBytes)
        {
             printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                 ": insufficient buffer size to read %d symbol(s) "
                 "(%d instead of %d byte(s))",
                 un8SymbolCount, tBufferSize, tReadBytes);
             eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
             break;
        }

        eResult = eEnsureBufferEnough(psObj, un8SymbolSize + 1);
        if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to check buffer (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            break;
        }

        // Process all symbols one-by-one
        for (tSymbolIter = un16SymbolStartIndex;
             tSymbolIter < (un8SymbolCount + un16SymbolStartIndex);
             ++tSymbolIter)
        {
            tReadBytes =
                OSAL.tBufferReadHead(hPayload, psObj->pacBuffer, un8SymbolSize);
            if (tReadBytes != un8SymbolSize)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to read symbol #%d with size %d byte(s)",
                        tSymbolIter, un8SymbolSize);
                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }
            psObj->pacBuffer[tReadBytes] = '\0';

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                   ": index %04d -> %s\n", un16SymbolIndex, psObj->acBuffer);
#endif

            // Update value
            eResult = eNotifyManagerByNewName(psObj, tSymbolIter,
                                    psObj->pacBuffer, un8SymbolSize,
                                    bDoTouch
                                        );
            if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to update name for #%d (%s)",
                        tSymbolIter, GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                break;
            }
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eProcessMessageStockValue
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockValue(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hPayload = psData->hBuffer;
    BOOLEAN bReadOk = FALSE;
    size_t tReadBytes = 0;
    size_t tBufferSize = 0;
    STOCK_ID tSymbolIter = 0;
    UN16 un16StockStartIndex = 0;
    UN8 un8Count = 0;

    puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
       ": Processing Stock Value message");

    do
    {
        // Read number of symbols in the message
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8Count,
                            STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE);

        // Read the index of the first message in the sequence
        bReadOk = OSAL.bBufferReadBitsToUN16(hPayload, &un16StockStartIndex,
                            STOCK_TICKER_MGS_VALUE_INDEX_BITSIZE);
        if (bReadOk == TRUE)
        {
            tReadBytes += STOCK_TICKER_MGS_VALUE_INDEX_BYTESIZE;
        }

        // Check read bytes for header size
        if (tReadBytes != STOCK_TICKER_MGS_VALUE_HDR_BYTELEN)
        {
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                       STOCK_TICKER_PROTOCOL_OBJECT_NAME
                       ": failed to read header (%d of %d byte(s)",
                       tReadBytes, STOCK_TICKER_MGS_VALUE_HDR_BYTELEN
                            );
             eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
             break;
        }

        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
            ": contains %d symbols(s) staring from index %d\n",
            un8Count, un16StockStartIndex);

        // Check message size left for data
        tBufferSize = OSAL.tBufferGetSize(hPayload);
        tReadBytes = un8Count * STOCK_TICKER_MGS_VALUE_SYMBOL_BYTESIZE;
        if (tBufferSize < tReadBytes)
        {
             printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                 ": insufficient buffer size to read %d symbol(s) "
                 "value (%d instead of %d byte(s))",
                 un8Count, tBufferSize, tReadBytes);
             eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
             break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        if (bDoTouch == FALSE)
        {
            UN8 un8IntValue;
            UN8 un8FracValue;
            UN8 un8IntDeltaValue;
            UN8 un8FracDeltaValue;
            BOOLEAN bDoesDown;

            for (tSymbolIter = un16StockStartIndex;
                 tSymbolIter < (un8Count + un16StockStartIndex);
                 ++tSymbolIter)
            {
                un8IntValue = un8FracValue = 0;
                un8IntDeltaValue = un8FracDeltaValue = 0;
                bDoesDown = FALSE;

                tReadBytes =
                    OSAL.tBufferReadHead(hPayload, &un8IntValue,
                        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_BYTESIZE
                            );
                tReadBytes +=
                    OSAL.tBufferReadHead(hPayload, &un8FracValue,
                        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE
                            );
                tReadBytes +=
                    OSAL.tBufferReadHead(hPayload, &un8IntDeltaValue,
                        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE
                            );
                tReadBytes +=
                    OSAL.tBufferReadHead(hPayload, &un8FracDeltaValue,
                        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE
                            );

                if (tReadBytes != STOCK_TICKER_MGS_VALUE_SYMBOL_BYTESIZE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": failed to record #%d (read %d of %u bytes(s)). "
                            "Buffer has %d byte(s)",
                            tSymbolIter, tReadBytes,
                            STOCK_TICKER_MGS_VALUE_SYMBOL_BYTESIZE,
                            OSAL.tBufferGetSize(hPayload));
                    eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                    break;
                }

                // Extract the direction bit
                bDoesDown =
                    (un8IntDeltaValue & STOCK_TICKER_MGS_VALUE_DELTA_DIRECTION_MASK) ?
                            TRUE : FALSE;
                // Eliminate the influence of the sign bit
                un8IntDeltaValue =
                    un8IntDeltaValue & (STOCK_TICKER_MGS_VALUE_DELTA_DIRECTION_MASK - 1);

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": #%04d -> %05d.%02d (%c%05d.%02d)\n",
                    un16StockIndex, un8IntValue, un8FracValue,
                    (bDoesDown == TRUE ? '-' : '+'),
                    un8FracDeltaValue, un8IntDeltaValue
                        );
#endif
                // Update value
                eResult = eNotifyManagerByNewValue(psObj,
                                        STOCK_TICKER_MSG_TYPE_STOCK_VALUE,
                                        tSymbolIter,
                                        un8IntValue, un8FracValue,
                                        un8IntDeltaValue, un8FracDeltaValue,
                                        bDoesDown
                                            );
                if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": failed to update value for #%d (%s)",
                            tSymbolIter,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }
        }
        else
        {
            // Iterate through all stock indexes to simply touch them
            for (tSymbolIter = un16StockStartIndex;
                 tSymbolIter < (un8Count + un16StockStartIndex);
                 ++tSymbolIter)
            {
                // Update value
                eResult = eNotifyManagerByNewValueViaTouch(psObj,
                                        STOCK_TICKER_MSG_TYPE_STOCK_VALUE,
                                        tSymbolIter
                                            );
                if (eResult != STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": failed to touch value for #%d (%s)",
                            tSymbolIter,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                    break;
                }
            }
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eProcessMessageStockValueEx
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockValueEx(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hPayload = psData->hBuffer;
    BOOLEAN bReadOk = FALSE;
    size_t tReadBytes = 0;
    size_t tBufferSize = 0;
    UN8 un8Count = 0;
    STOCK_ID tSymbolIter = 0;

    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
        ": Processing Extended Stock Value  message");

    do
    {
        // Read number of symbols in the message
        tReadBytes = OSAL.tBufferReadHead(hPayload, &un8Count,
            STOCK_TICKER_MGS_VALUE_EX_COUNT_BYTESIZE);

        // Check read bytes for header size
        if (tReadBytes != STOCK_TICKER_MGS_VALUE_EX_HDR_BYTELEN)
        {
             SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                       STOCK_TICKER_PROTOCOL_OBJECT_NAME
                       ": failed to read header (%d of %d byte(s)",
                       tReadBytes, STOCK_TICKER_MGS_VALUE_HDR_BYTELEN
                            );
             eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
             break;
        }

        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
            ": contains %d symbols(s)\n", un8Count);

        // Check message size left for data
        tBufferSize = OSAL.tBufferGetSize(hPayload);
        tReadBytes = un8Count * STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE;
        if (tBufferSize < tReadBytes)
        {
             printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                 ": insufficient buffer size to read %d symbol(s) "
                 "value (%d instead of %d byte(s))",
                 un8Count, tBufferSize, tReadBytes);
             eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
             break;
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        for (tSymbolIter = 0;
             tSymbolIter < un8Count;
             ++tSymbolIter)
        {
            UN32 un32IntValue = 0;
            UN8 un8FracValue = 0;
            UN32 un32IntDeltaValue = 0;
            UN8 un8FracDeltaValue = 0;
            UN16 un16StockIndex = 0;
            BOOLEAN bDoesDown = FALSE;

            tReadBytes = 0;

            // Read values
            bReadOk = OSAL.bBufferReadBitsToUN16(hPayload, &un16StockIndex,
                                    STOCK_TICKER_MGS_VALUE_EX_INDEX_BITSIZE
                                        );
            if (bReadOk == TRUE)
            {
                tReadBytes += STOCK_TICKER_MGS_VALUE_EX_INDEX_BYTESIZE;
            }

            if (bDoTouch == FALSE)
            {
                bReadOk =
                    OSAL.bBufferReadBitsToUN32(hPayload, &un32IntValue,
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BITSIZE
                            );
                if (bReadOk == TRUE)
                {
                    tReadBytes +=
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BYTESIZE;
                }
                tReadBytes +=
                    OSAL.tBufferReadHead(hPayload, &un8FracValue,
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE
                            );
                bReadOk =
                    OSAL.bBufferReadBitsToUN32(hPayload, &un32IntDeltaValue,
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BITSIZE
                            );
                if (bReadOk == TRUE)
                {
                    tReadBytes +=
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE;
                }
                tReadBytes +=
                    OSAL.tBufferReadHead(hPayload, &un8FracDeltaValue,
                        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE
                            );

                if (tReadBytes != STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": failed to record #%d (read %d of %d bytes(s))",
                            tSymbolIter, tReadBytes,
                            STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE);
                    eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                    break;
                }

                // Extract the direction bit
                bDoesDown =
                    (un32IntDeltaValue & STOCK_TICKER_MGS_VALUE_EX_DELTA_DIRECTION_MASK) ? TRUE : FALSE;
                // Eliminate the influence of the sign bit
                un32IntDeltaValue =
                    un32IntDeltaValue & (STOCK_TICKER_MGS_VALUE_EX_DELTA_DIRECTION_MASK - 1);

#if ((SMS_DEBUG == 1) && (STOCK_TICKER_EXTRA_DEBUG == 1))
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": #%04d -> %05d.%02d (%c%05d.%02d)\n",
                    tSymbolIter, un32IntValue, un8FracValue,
                    (bDoesDown == TRUE ? '-' : '+'),
                    un32IntDeltaValue, un8FracDeltaValue
                            );
#endif

                eResult = eNotifyManagerByNewValue(psObj, STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX,
                                        un16StockIndex,
                                        un32IntValue, un8FracValue,
                                        un32IntDeltaValue, un8FracDeltaValue,
                                        bDoesDown
                                            );

                // In case of bad return code notify the log
                if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                    (eResult != STOCK_TICKER_RETURN_CODE_BAD_STATE))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to update value for #%d (%s)",
                        un16StockIndex,
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                    break;
                }
            }
            else
            {
                tReadBytes += OSAL.tBufferSeekHead(hPayload,
                                STOCK_TICKER_MGS_VALUE_EX_SYMBOL_VALUE_BYTESIZE);
                if (tReadBytes != STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE)
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                            STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": failed to record #%d (read %d of %d bytes(s))",
                            tSymbolIter, tReadBytes,
                            STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE);
                    eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                    break;
                }

                eResult =
                    eNotifyManagerByNewValueViaTouch(psObj,
                        STOCK_TICKER_MSG_TYPE_STOCK_VALUE_EX,
                        (STOCK_ID)un16StockIndex);

                // In case of bad return code notify the log
                if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                    (eResult != STOCK_TICKER_RETURN_CODE_BAD_STATE))
                {
                    SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to touch value for #%d (%s)",
                        un16StockIndex,
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );
                    break;
                }
            }
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eProcessMessageDataProviderMsg
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageDataProviderMsg(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    OSAL_BUFFER_HDL hPayload = psData->hBuffer;
    size_t tReadBytes = 0;
    UN8 un8RefNumber = 0;
    UN8 un8TextSize = 0;
    STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT sArg;

    puts(STOCK_TICKER_PROTOCOL_OBJECT_NAME
        ": Processing Data Provider message");

    do
    {
        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.eType = STOCK_TICKER_MSG_TYPE_DATA_PROVIDER_MSG;
        sArg.bDoTouch = bDoTouch;
        sArg.uData.sProviderInfo.hInfo = STRING_INVALID_OBJECT;
        sArg.uData.sProviderInfo.un8SeqNumber = 0;

        // Read header
        tReadBytes += OSAL.tBufferReadHead(hPayload,
                            &sArg.uData.sProviderInfo.un8SeqNumber,
                            STOCK_TICKER_MGS_DATA_PROVIDER_SEQ_NUMBER_BYTESIZE
                                );
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8RefNumber,
                            STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_BYTESIZE
                                );
        tReadBytes += OSAL.tBufferReadHead(hPayload, &un8TextSize,
                            STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_BYTESIZE
                                );
        if (tReadBytes != STOCK_TICKER_MGS_DATA_PROVIDER_HDR_BYTESIZE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to read Data Provider Message header (%d of %d byte(s))",
                    tReadBytes, STOCK_TICKER_MGS_DATA_PROVIDER_HDR_BYTESIZE);
            eReturnCode = STOCK_TICKER_RETURN_CODE_IO_ERROR;
            break;
        }

        // In current implementation we are interested in the information
        // for the stocks provider only, so check the ref number.
        if (un8RefNumber != STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_STOCK_FEED)
        {
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": skip. Since stocks provider required only\n");
            eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
            break;
        }

        // Check read size
        if (un8TextSize > 0)
        {
            eReturnCode = eEnsureBufferEnough(psObj, un8TextSize + 1 /* null */);
            if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to allocate buffer (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
                break;
            }

            // Read the data
            tReadBytes = OSAL.tBufferReadHead(hPayload, psObj->pacBuffer, un8TextSize);
            if (tReadBytes != un8TextSize)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to read Data Provider Message text (%d of %d byte(s))",
                        tReadBytes, un8TextSize);
                eReturnCode = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }
            // Ensure the string has \0 at the end
            psObj->pacBuffer[un8TextSize] = '\0';

            // Create string representation
            sArg.uData.sProviderInfo.hInfo = STRING.hCreate( psObj->pacBuffer, un8TextSize );
            if (sArg.uData.sProviderInfo.hInfo == STRING_INVALID_OBJECT)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to create STRING_OBJECT for the Data Provider Message text"
                            );
                eReturnCode = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
                break;
            }
        }

        // Update information in the manager
        eReturnCode = GsStockTickerMgrIntf.eUpdate(
                                    psObj->hStockTickerService, &sArg
                                        );
        if (eReturnCode != STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to update Provider data in the Manager (%s)",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eReturnCode)
                        );
            break;
        }

    } while (FALSE);

    // Destroy the string temp object if it still exists
    if (sArg.uData.sProviderInfo.hInfo != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(sArg.uData.sProviderInfo.hInfo);
        sArg.uData.sProviderInfo.hInfo = STRING_INVALID_OBJECT;
    }

    return eReturnCode;
}

/*******************************************************************************
*
*        eEnsureBufferEnough
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eEnsureBufferEnough(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    size_t tBufferSize
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eReturnCode = STOCK_TICKER_RETURN_CODE_ERROR;
    do
    {
        if ((psObj->pacBuffer != NULL) && (psObj->tBufferSize < tBufferSize))
        {
            // Release the memory since the buffer is not
            // enough
            SMSO_vDestroy((SMS_OBJECT) psObj->pacBuffer);
            psObj->pacBuffer = NULL;
            psObj->tBufferSize = 0;
        }

        // Allocate memory if needed
        if ((psObj->pacBuffer == NULL) && (tBufferSize > 0))
        {
            psObj->pacBuffer = (char*)SMSO_hCreate(
                STOCK_TICKER_PROTOCOL_OBJECT_NAME":TmpBuf",
                tBufferSize,
                (SMS_OBJECT) psObj->hStockTickerService,
                FALSE);
            if (psObj->pacBuffer == NULL)
            {
                eReturnCode = STOCK_TICKER_RETURN_CODE_MEMORY_ERROR;
                break;
            }

            psObj->tBufferSize = tBufferSize;
        }

        eReturnCode = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eReturnCode;
}

/*******************************************************************************
*
*        psCreateMsgParserData
*
*******************************************************************************/
static STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psCreateMsgParserData(
    SMS_OBJECT hParent,
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType
        )
{
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psResult = NULL;

    psResult = (STOCK_TICKER_MSG_PARSER_DATA_STRUCT*)SMSO_hCreate(
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME":MsgPrsData",
                    sizeof(STOCK_TICKER_MSG_PARSER_DATA_STRUCT),
                    hParent,
                    FALSE
                        );

    if (psResult != NULL)
    {
        OSAL.bMemSet(psResult, 0, sizeof(*psResult));
        psResult->eMsgType = eMsgType;
        psResult->eState = STOCKS_MSG_PARSER_STATE_EMPTY;
        psResult->hBuffer = OSAL_INVALID_BUFFER_HDL;
        psResult->hCurrentBuffer = OSAL_INVALID_BUFFER_HDL;
        psResult->un32MsgSeqNumber = STOCK_TICKER_INVALID_SEQID;
        psResult->un32SeqNumber = STOCK_TICKER_INVALID_SEQID;
        psResult->bDataIsCollected = FALSE;
#if SMS_DEBUG == 1
        psResult->un32PayloadPorcessedCounter = 0;
#endif
    }

    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": created message parser data 0x%p for %s\n",
            psResult, GsStockTickerMgrIntf.pacGetMessageTypeName(eMsgType)
                );

    return psResult;
}

/*******************************************************************************
*
*        eAppendPayloadToMsgParserData
*
*  Appends all data from the provided payload to the message parser data object
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eAppendPayloadToMsgParserData (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psProtoObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;

    // Check input
    if ((psObj != NULL) && (hPayload != OSAL_INVALID_BUFFER_HDL))
    {
        OSAL_RETURN_CODE_ENUM eOsalReturnCode;

        // Merge buffers
        eOsalReturnCode = OSAL.eBufferAppend(psObj->hBuffer, hPayload);
        if (eOsalReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": failed to merge buffers (%s)",
                OSAL.pacGetReturnCodeName(eOsalReturnCode)
                    );
            eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
        }
        else
        {
            eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
        }
    }

    return eResult;
}

/*******************************************************************************
*
*        eUpdateMsgParserData
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eUpdateMsgParserData(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psProtoObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    STOCKS_PAYLOAD_HEADER_STRUCT *psHeader
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    size_t tBufferSize = 0;
    UN32 un32NextSeqNumber = 0;
    BOOLEAN bSyncSeqFound = FALSE;
    BOOLEAN bGoodSequenceNumber = FALSE;
    BOOLEAN bRecheck;

    do
    {
        // Input check
        if ((psObj == NULL) || (phPayload == NULL) ||
            (*phPayload == OSAL_INVALID_BUFFER_HDL) || (psHeader == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Get payload size
        tBufferSize = OSAL.tBufferGetSize(*phPayload);

        ///////////////////////////////////////////////////////////////////////
        // The dirty check for the payload
        if ((tBufferSize < STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN) &&
            (psObj->eState == STOCKS_MSG_PARSER_STATE_EMPTY))
        {
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": [%d] skip the payload since we don't known how to handle "
                    "it due to low size to check sync seq (%d of %d byte(s))\n",
                    __LINE__,
                    tBufferSize, STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN
                        );
            eResult = STOCK_TICKER_RETURN_CODE_NOT_ENOUGH_BUFFER;
            break;
        }

        ///////////////////////////////////////////////////////////////////////
        // Read and checks the sync sequence existence
        if (tBufferSize >= STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN)
        {
            UN8 aun8Buffer[STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN];
            N32 n32Result = 0;
            size_t tReadBytes = 0;

            tReadBytes = OSAL.tBufferPeek(*phPayload, aun8Buffer,
                                            STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN, 0
                                                );
            if (tReadBytes != STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN)
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to read sync seq %d of %d (%s)\n",
                        tReadBytes, STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN,
                        GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                            );

                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }

            // Compare buffer with pattern
            n32Result = memcmp(aun8Buffer, STOCK_TICKER_MSG_SYNC_SEQUENCE,
                                    STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN);

            bSyncSeqFound = (n32Result == 0) ? TRUE : FALSE;

            // Skip the sync sequence then
            if (bSyncSeqFound == TRUE)
            {
                size_t tSkippedBytes;

                // Skip sync sequence
                tSkippedBytes = OSAL.tBufferSeekHead(*phPayload,
                                            STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN);
                if (tSkippedBytes != STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN)
                {
                    eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                    break;
                }
            }
        }

        ///////////////////////////////////////////////////////////////////////
        // Calculate next and check the received sequence numbers
        // to validate payloads sequence
        un32NextSeqNumber =
            (psObj->un32SeqNumber + 1) % STOCK_TICKER_SEQ_NUMBER_MOD;
        bGoodSequenceNumber =
            (psHeader->un32SeqNumber == un32NextSeqNumber) ? TRUE : FALSE;

        ///////////////////////////////////////////////////////////////////////
        // Perform buffer operations
        if (bSyncSeqFound == TRUE)
        {
            if (psObj->eState == STOCKS_MSG_PARSER_STATE_EMPTY)
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": [%d] start collecting payload\n",
                       __LINE__);

                // Since the buffer just started keep it to use
                // it for composing complete message payload
                psObj->hBuffer = *phPayload;
                psObj->un32SeqNumber = psHeader->un32SeqNumber;
                psObj->eState = STOCKS_MSG_PARSER_STATE_COLLECTING;
                *phPayload = OSAL_INVALID_BUFFER_HDL;
            }
            else if (psObj->eState == STOCKS_MSG_PARSER_STATE_COLLECTING)
            {
                // Since we are received new payload which said that
                // we have new sync sequence just store the new payload buffer
                // in the hCurrentBuffer member and let caller process the
                // data from the hBuffer. After processing the caller
                // must call vCompleteMsgParserData() to complete the processing
                // of the fully composed buffer, release it and moved the buffer
                // we just received as another current one.
                psObj->hCurrentBuffer = *phPayload;
                psObj->un32SeqNumber = psHeader->un32SeqNumber;
                psObj->eState = STOCKS_MSG_PARSER_STATE_CHECKING;
                psObj->un32PayloadPorcessedCounter++;
                *phPayload = OSAL_INVALID_BUFFER_HDL;
            }
            else
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": [%d] incorrect state (%d)\n", __LINE__, psObj->eState
                            );
                eResult = STOCK_TICKER_RETURN_CODE_BAD_STATE;
                break;
            }
        }
        else /* We don't have sync sequence in this payload */
        {
            if (psObj->eState == STOCKS_MSG_PARSER_STATE_EMPTY)
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": [%d] skip the payload since we have't "
                        "had sync sequence\n", __LINE__);
            }
            else if (psObj->eState == STOCKS_MSG_PARSER_STATE_COLLECTING)
            {
                if (bGoodSequenceNumber == TRUE)
                {
                    // So we detected next payload thus just copy data from it
                    // to the previously reserved buffer
                    eResult =
                        eAppendPayloadToMsgParserData(psProtoObj, psObj,
                                                      *phPayload);
                    if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                    {
                        psObj->un32SeqNumber = psHeader->un32SeqNumber;
                    }
                    else
                    {
                        // Skip all buffer since some missed block detected
                        printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": [%d] reset the collected payload because "
                            "of failed appending operation (%s)\n",
                            __LINE__,
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                                );
                        vResetMsgParserData(psObj);
                    }
                }
                else
                {
                    // Skip all buffer since some missed block detected
                    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": [%d] skip the collected payload due to broken payloads sequence\n",
                            __LINE__
                                );

                    // Drop the current buffers due to broken sequence of payloads
                    vResetMsgParserData(psObj);
                }
            }
            else
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME": [%d] incorrect state (%d)\n",
                        __LINE__, psObj->eState
                            );
                eResult = STOCK_TICKER_RETURN_CODE_BAD_STATE;
                break;
            }
        }

        // Since the parser data object has something to process - let's check
        bRecheck = TRUE;
        while (bRecheck == TRUE)
        {
            // We are not expecting re-check here. Let see...
            bRecheck = FALSE;

            // Since at this moment we might have fully composed payload let's
            // try to check this fact
            if ((psObj->eState == STOCKS_MSG_PARSER_STATE_COLLECTING) ||
                (psObj->eState == STOCKS_MSG_PARSER_STATE_CHECKING))
            {
                STOCK_TICKER_RETURN_CODE_ENUM eCheckResult;

                eCheckResult = eCheckMessage(psObj);
                if (eCheckResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
                {
                    psObj->eState = STOCKS_MSG_PARSER_STATE_READY;
                }
                else if (psObj->eState == STOCKS_MSG_PARSER_STATE_CHECKING)
                {
                    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": [%d] skip the payload since it is not ready\n",
                            __LINE__
                                );

                    // Let say that we are not interested in the composed payload
                    // and let try to check just received one in the same statement.
                    vCompleteMsgParserData(psObj);
                    bRecheck = TRUE;
                }
#if SMS_DEBUG==1
                else
                {
                    printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                            ": checking finished with %s\n",
                            GsStockTickerMgrIntf.pacGetReturnCodeName(eCheckResult)
                                );
                }
#endif
            }
        }

        eResult = STOCK_TICKER_RETURN_CODE_SUCCESS;
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        vCompleteMsgParserData
*
*******************************************************************************/
static void vCompleteMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    // Check input
    if (psObj != NULL)
    {
        if (psObj->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            DATASERVICE_IMPL_bFreeDataPayload(psObj->hBuffer);
            psObj->hBuffer = OSAL_INVALID_BUFFER_HDL;
            psObj->eState = STOCKS_MSG_PARSER_STATE_EMPTY;
        }

        // Do we have data for further collecting?
        if (psObj->hCurrentBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            psObj->hBuffer = psObj->hCurrentBuffer;
            psObj->hCurrentBuffer = OSAL_INVALID_BUFFER_HDL;
            psObj->eState = STOCKS_MSG_PARSER_STATE_COLLECTING;
        }

        // No data cached
        psObj->bDataIsCollected = FALSE;
        OSAL.bMemSet(&psObj->uData, 0, sizeof(psObj->uData));
    }

    return;
}

/*******************************************************************************
*
*        vResetMsgParserData
*
*******************************************************************************/
static void vResetMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    // Check input
    if (psObj != NULL)
    {
        if (psObj->hBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            DATASERVICE_IMPL_bFreeDataPayload(psObj->hBuffer);
            psObj->hBuffer = OSAL_INVALID_BUFFER_HDL;
        }

        if (psObj->hCurrentBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            DATASERVICE_IMPL_bFreeDataPayload(psObj->hCurrentBuffer);
            psObj->hCurrentBuffer = OSAL_INVALID_BUFFER_HDL;
        }

        psObj->bDataIsCollected = FALSE;
        psObj->un32SeqNumber = STOCK_TICKER_INVALID_SEQID;
        psObj->un32MsgSeqNumber = STOCK_TICKER_INVALID_SEQID;
        psObj->eState = STOCKS_MSG_PARSER_STATE_EMPTY;
    }

    return;
}

/*******************************************************************************
*
*        vDestroyMsgParserData
*
*******************************************************************************/
static void vDestroyMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        vResetMsgParserData(psObj);
        SMSO_vDestroy((SMS_OBJECT) psObj);
    }

    return;
}

/*******************************************************************************
*
*        psGetMsgParserData
*
*  Provides caller by the parser data handler. It tries to find it and
*  in case of absence creates a new one corresponded to requested message
*  type.
*
*******************************************************************************/
static STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psGetMsgParserData (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType
        )
{
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psResult = 
        (STOCK_TICKER_MSG_PARSER_DATA_STRUCT *) NULL;

    do
    {
        // Check input
        if (psObj == NULL)
        {
            break;
        }

        if (psObj->apsMsgParserDataList[eMsgType] != NULL)
        {
            // So, we have data handler
            psResult = psObj->apsMsgParserDataList[eMsgType];
        }
        else
        {
            STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psCreated;

            // Create new instance
            psCreated = psCreateMsgParserData((SMS_OBJECT)psObj, eMsgType);
            if (psCreated == NULL)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME
                        ": failed to create message parser data"
                            );
                break;
            }

            // Add item to the list
            psObj->apsMsgParserDataList[eMsgType] = psCreated;

            // Make the object as returned value
            psResult = psCreated;
        }

    } while (FALSE);

    return psResult;
}

/*******************************************************************************
*
*        eIsProcessingRequired
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eIsProcessingRequired (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psMsgData
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT const *puData;

    do
    {
        // Check input
        if ((psObj == NULL) || (psMsgData == NULL))
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        ////////////////////////////
        // Check in general - do we need this type of messages or not.
        puData =
            (psMsgData->bDataIsCollected == TRUE) ? &psMsgData->uData : NULL;
        eResult = GsStockTickerMgrIntf.eIsMessageNeeded(
                        psObj->hStockTickerService,
                        psMsgData->eMsgType, puData);

        // So, at least this type of messages is needed by the Service
        if (eResult == STOCK_TICKER_RETURN_CODE_SUCCESS)
        {
            OSAL_CRC_RESULT tCRC = 0;
            size_t tNumBytesToCRC;
            size_t tBytesProcessed = 0;
            BOOLEAN bOk;

            ////////////////////////////
            // Initialize the CRC
            bOk = OSAL.bInitializeCRC(psObj->hCRC, &tCRC);
            if (bOk == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to initialize OSAL.CRC");
                eResult = STOCK_TICKER_RETURN_CODE_ERROR;
                break;
            }

            ////////////////////////////
            // Compute the CRC of the entire payload
            tNumBytesToCRC = OSAL.tBufferGetSize(psMsgData->hBuffer);

            tCRC = OSAL.tComputeCRC(
                psObj->hCRC, tCRC, psMsgData->hBuffer,
                0, tNumBytesToCRC, &tBytesProcessed
                    );

            ////////////////////////////
            // Did we process the entire payload?
            if (tBytesProcessed != tNumBytesToCRC)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": CRC processed len mismatch payload %p len! "
                    "(%d against required %d)",
                    psMsgData->hBuffer, tBytesProcessed, tNumBytesToCRC
                        );
                // Let's pretend that everything is fine just bad CRC
                eResult = STOCK_TICKER_RETURN_CODE_IO_ERROR;
                break;
            }

            // Invert the CRC bits (RX 185, Section 4.3.3)
            tCRC ^= STOCK_TICKER_CRC_INVERSION_VALUE;

            ////////////////////////////
            // Check the message necessity using Manager
            eResult = GsStockTickerMgrIntf.eIsPayloadProcessNeeded(
                            psObj->hStockTickerService,
                            psMsgData->eMsgType, tCRC
                                );
            if ((eResult != STOCK_TICKER_RETURN_CODE_SUCCESS) &&
                (eResult != STOCK_TICKER_RETURN_CODE_FALSE) &&
                (eResult != STOCK_TICKER_RETURN_CODE_NEED_TO_TOUCH))
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": failed to check payload with unexpected result %s",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                        );
            }
            else
            {
                printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                    ": eIsPayloadProcessNeeded told %s\n",
                    GsStockTickerMgrIntf.pacGetReturnCodeName(eResult));
            }
        }
        else if (eResult != STOCK_TICKER_RETURN_CODE_FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": failed to check payload message type for necessity (%s)",
                GsStockTickerMgrIntf.pacGetReturnCodeName(eResult)
                    );
        }
    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eNotifyManagerByNewValue
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewValue(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_ID tIndex,
    UN32 un32Price, UN8 un8PriceFrac,
    UN32 un32PriceDelta, UN8 un8PriceDeltaFrac,
    BOOLEAN bDoesDown
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult = STOCK_TICKER_RETURN_CODE_ERROR;
    STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT sArg;
    STOCK_TICKER_QUOTE_STRUCT *psQuote;
    BOOLEAN bValue;

    do
    {
        // Init argument
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.eType = eType;
        sArg.uData.sValueMsg.tIndex = tIndex;
        sArg.uData.sValueMsg.eState = STOCK_MSG_STATE_INVALID;

        psQuote = &sArg.uData.sValueMsg.sQuote;

        // Check input
        if (psObj == NULL)
        {
            eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
            break;
        }

        // Check the value
        bValue = STOCK_VALUE_IS_VALID(un8PriceFrac);
        if (bValue == TRUE)
        {
            psQuote->un32Price = un32Price;
            psQuote->un8PriceFrac = un8PriceFrac;
            psQuote->un32PriceDelta = un32PriceDelta;
            psQuote->un8PriceDeltaFrac = un8PriceDeltaFrac;
            psQuote->eDir = STOCK_MSG_PRICE_DIRECTION_UNCHANGED;

            if ((un32PriceDelta > 0) || (un8PriceDeltaFrac > 0))
            {
                // Since the delta value is not zero
                // need to define direction of the data
                psQuote->eDir =
                    (bDoesDown == TRUE) ? STOCK_MSG_PRICE_DIRECTION_DOWN :
                                          STOCK_MSG_PRICE_DIRECTION_UP;
            }

            // Good price we have
            sArg.uData.sValueMsg.eState = STOCK_MSG_STATE_PRICE_AVAILABLE;
        }
        else
        {
            bValue = STOCK_VALUE_IS_NA(un8PriceFrac);
            if (bValue == TRUE)
            {
                sArg.uData.sValueMsg.eState = STOCK_MSG_STATE_PRICE_NOT_AVAILABLE;
            }
            else
            {
                bValue = STOCK_VALUE_IS_DELISTED(un8PriceFrac);
                if (bValue == TRUE)
                {
                    sArg.uData.sValueMsg.eState = STOCK_MSG_STATE_DELISTED;
                }
                else
                {
                    // Some value between 100 and 253 received
                    sArg.uData.sValueMsg.eState = STOCK_MSG_STATE_INVALID;
                }
            }
        }

        // Check the value state
        if (sArg.uData.sValueMsg.eState != STOCK_MSG_STATE_INVALID)
        {
            // Update data via manager interface
            eResult =
                GsStockTickerMgrIntf.eUpdate(psObj->hStockTickerService,
                    &sArg);
        }
        else
        {
            // If we are here it means that we received something bad
            // Just print out it, then
            printf(STOCK_TICKER_PROTOCOL_OBJECT_NAME
                ": [%d] for Stock #%d (Price = %u.%u, PriceDelta = %c%d.%u)\n",
                __LINE__, tIndex, un32Price, un8PriceFrac,
                (bDoesDown == TRUE ? '-' : '+'), un32PriceDelta, un8PriceDeltaFrac
                    );
            eResult = STOCK_TICKER_RETURN_CODE_BAD_STATE;
        }

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
*
*        eNotifyManagerByNewValueViaTouch
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewValueViaTouch(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_ID tIndex
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;
    STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT sArg;

    // Init argument
    OSAL.bMemSet(&sArg, 0, sizeof(sArg));
    sArg.eType = eType;
    sArg.bDoTouch = TRUE;
    sArg.uData.sValueMsg.tIndex = tIndex;

    // Update data via manager interface
    eResult =
        GsStockTickerMgrIntf.eUpdate(psObj->hStockTickerService, &sArg);

    return eResult;
}

/*******************************************************************************
*
*        eNotifyManagerByNewName
*
*******************************************************************************/
static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewName(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_ID tIndex,
    const char *pacName,
    size_t tNameLength,
    BOOLEAN bDoTouch
        )
{
    STOCK_TICKER_RETURN_CODE_ENUM eResult;

    // Check input
    if ((psObj == NULL) || (pacName == NULL) || (tNameLength <= 0))
    {
        eResult = STOCK_TICKER_RETURN_CODE_BAD_INPUT;
    }
    else
    {
        STOCK_TICKER_PLUGIN_INTERFACE_ARG_STRUCT sArg;

        // Init the variable
        OSAL.bMemSet(&sArg, 0, sizeof(sArg));
        sArg.eType = STOCK_TICKER_MSG_TYPE_STOCK_SYMBOL;
        sArg.bDoTouch = bDoTouch;
        sArg.uData.sSymbolMsg.tIndex = tIndex;
        sArg.uData.sSymbolMsg.pacName = pacName;
        sArg.uData.sSymbolMsg.tNameLength = tNameLength;

        // Update data via manager interface
        eResult =
            GsStockTickerMgrIntf.eUpdate(psObj->hStockTickerService, &sArg);
    }

    return eResult;
}

/*******************************************************************************
*
*        vUninitObject
*
*******************************************************************************/
static void vUninitObject(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eOsalReturnCode = OSAL_SUCCESS;
    STOCK_TICKER_MSG_TYPE_ENUM eIndex;

    do
    {
        if (psObj == NULL)
        {
            break;
        }

        // Remove CRC calculator
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            eOsalReturnCode = OSAL.eReleaseCRC(psObj->hCRC);
            if (eOsalReturnCode != OSAL_SUCCESS)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                        STOCK_TICKER_PROTOCOL_OBJECT_NAME": failed to release CRC (%s)",
                        OSAL.pacGetReturnCodeName(eOsalReturnCode)
                            );
            }
            psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
        }

        for (eIndex = STOCK_TICKER_MSG_TYPE_START;
             eIndex < STOCK_TICKER_MSG_TYPE_MAX;
             ++eIndex)
        {
            if (psObj->apsMsgParserDataList[eIndex] != NULL)
            {
                vDestroyMsgParserData(psObj->apsMsgParserDataList[eIndex]);
                psObj->apsMsgParserDataList[eIndex] = NULL;
            }
        }

        // Release memory if exists
        if (psObj->pacBuffer != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT) psObj->pacBuffer);
            psObj->pacBuffer = NULL;
            psObj->tBufferSize = 0;
        }

        // Release object
        SMSO_vDestroy((SMS_OBJECT)psObj);
    } while (FALSE);

    return;
}
