/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the high band Phonetics functions for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "sms_obj.h"
#include "string_obj.h"

#include "phonetics_interface.h"
#include "_phonetics_highband.h"
#include "scode_table.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   Creates and initializes the highband phonetics interface object.
*
*****************************************************************************/
static PHONETICS_INTERFACE_OBJECT hInit(
    PHONETICS_SERVICE_OBJECT hPhoneticsService,
    SMS_OBJECT hParent
        )
{
    BOOLEAN bOwner;

    // Verify we own service handle
    bOwner = SMSO_bOwner(hParent);
    if (bOwner == TRUE)
    {
        PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj;

        // Create an instance of this object
        psObj = (PHONETICS_HIGHBAND_OBJECT_STRUCT *)
            SMSO_hCreate(
                PHONETICS_HIGHBAND_OBJECT_NAME,
                sizeof(PHONETICS_HIGHBAND_OBJECT_STRUCT),
                hParent, FALSE );

        if (psObj == NULL)
        {
            return PHONETICS_INTERFACE_INVALID_OBJECT;
        }

        do
        {
            TDS_CALLBACKS_STRUCT sCallbacks;
            BOOLEAN bSuccess;

            // Clear the structure first (in case this definition
            // changes / grows to include things we don't know about)
            OSAL.bMemSet(&sCallbacks, 0, sizeof(TDS_CALLBACKS_STRUCT));

            // Configure our callbacks for TDS
            sCallbacks.hCreateTable =
                (TDS_CREATE_TABLE_CALLBACK)hCreateTable;
            sCallbacks.vDestroyTable = vDestroyTable;
            sCallbacks.bAddColumn =
                (TDS_ADD_COLUMN_CALLBACK)bAddColumnToTable;
            sCallbacks.bRowStatus =
                (TDS_ROW_STATUS_CALLBACK)bRowStatus;
            sCallbacks.bAddRowData =
                (TDS_ADD_ROW_DATA_CALLBACK)bAddRowDataToTable;
            sCallbacks.bNewTable =
                (TDS_NEW_TABLE_READY_CALLBACK)bNewTableReady;
            sCallbacks.pvTableArg = (void *)psObj;

            // Connect to TDS now
            psObj->hTDS = TDS_UTIL_hConnect(
                (SMS_OBJECT)psObj,
                (SMS_OBJECT)hParent,
                &sCallbacks, TRUE );
            if (psObj->hTDS == TDS_UTIL_INVALID_OBJECT)
            {
                break;
            }

            // Provide TDS with the SCode lookup table info now
            bSuccess = TDS_UTIL_bSetScodeLUT(
                psObj->hTDS,
                GasSCodeLUT,
                SCODE_TABLE_NUM_ENTRIES);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Save the service handle
            psObj->hPhoneticsService = hPhoneticsService;

            return (PHONETICS_INTERFACE_OBJECT)psObj;

        } while (0);

        vUninit((PHONETICS_INTERFACE_OBJECT)psObj);
    }

    return PHONETICS_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vUninit
*
*   Unnitializes the highband phonetics interface object and preforms all
*   necessary memory management
*
*****************************************************************************/
static void vUninit (
    PHONETICS_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj =
            (PHONETICS_HIGHBAND_OBJECT_STRUCT *)hInterface;

        // Disconnect from TDS if necessary
        if (psObj->hTDS != TDS_UTIL_INVALID_OBJECT)
        {
            TDS_UTIL_vDisconnect(psObj->hTDS);
            psObj->hTDS = TDS_UTIL_INVALID_OBJECT;
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bAddTable
*
*   This function allows the manager to specify which tables it is interested
*   in and allows this object to properly configure the TDS_UTIL object.
*
*****************************************************************************/
static BOOLEAN bAddTable (
    PHONETICS_INTERFACE_OBJECT hInterface,
    PHONETICS_FILE_TYPE_ENUM eType,
    UN8 un8TableId,
    UN8 un8TableDefVer,
    UN16 un16TableContentVer
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj =
            (PHONETICS_HIGHBAND_OBJECT_STRUCT *)hInterface;
        TDS_TABLE_DESC_STRUCT sTableDesc;

        // Configure the table descriptor with
        // the provided attributes
        sTableDesc.tId = (TDS_TABLE_ID)un8TableId;
        sTableDesc.tDefVer = (TDS_TABLE_DEF_VERSION)un8TableDefVer;
        sTableDesc.tContentVer = (TDS_TABLE_CONTENT_VERSION)un16TableContentVer;
        sTableDesc.tDefCarouselId = PHONETICS_HIGHBAND_DEF_CARID;

        // Now, add the interface-specific carousel id info
        switch (eType)
        {
            case PHONETICS_FILE_TYPE_CHANNELS:
            {
                sTableDesc.tContentCarouselId = PHONETICS_HIGHBAND_CHAN_CONT_CARID;
            }
            break;

            case PHONETICS_FILE_TYPE_CATEGORIES:
            {
                sTableDesc.tContentCarouselId = PHONETICS_HIGHBAND_CAT_CONT_CARID;
            }
            break;

            case PHONETICS_FILE_TYPE_LEAGUES:
            {
                sTableDesc.tContentCarouselId = PHONETICS_HIGHBAND_LEAGUE_CONT_CARID;
            }
            break;

            case PHONETICS_FILE_TYPE_TEAMS:
            {
                sTableDesc.tContentCarouselId = PHONETICS_HIGHBAND_TEAM_CONT_CARID;
            }
            break;

            case PHONETICS_FILE_TYPE_MARKETS:
            {
                sTableDesc.tContentCarouselId = PHONETICS_HIGHBAND_MARKET_CONT_CARID;
            }
            break;

            case PHONETICS_FILE_INVALID_TYPE:
            default:
            {
                // Just stop here
                return FALSE;
            }
        }

        // Register this table with TDS
        bSuccess = TDS_UTIL_bRegisterTable(
            psObj->hTDS, &sTableDesc);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessMessage
*
*****************************************************************************/
static BOOLEAN bProcessMessage (
    PHONETICS_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj =
            (PHONETICS_HIGHBAND_OBJECT_STRUCT *)hInterface;

        bSuccess = TDS_UTIL_bProcessPayload(psObj->hTDS, hPayload);
    }

    return bSuccess;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bValidateColumn
*
*****************************************************************************/
static BOOLEAN bValidateColumn (
    TDS_COLUMN_STRUCT *psColumn,
    PHONETICS_HIGHBAND_LABEL_ENUM *peLabelType
        )
{
    BOOLEAN bValid = FALSE;
    PHONETICS_HIGHBAND_LABEL_ENUM eLabelType;

    if (peLabelType == NULL)
    {
        // Use local variable
        peLabelType = &eLabelType;
    }

    switch (psColumn->tLabelId)
    {
        case PHONETICS_COL_ID1:
        {
            *peLabelType = PHONETICS_HIGHBAND_LABEL_ID1;

            // This has to be an integer
            if (psColumn->eType == TDS_DATA_TYPE_INT)
            {
                bValid = TRUE;
            }
        }
        break;

        case PHONETICS_COL_ID2:
        {
            *peLabelType = PHONETICS_HIGHBAND_LABEL_ID2;

            // This has to be an integer array
            if (psColumn->eType == TDS_DATA_TYPE_INT_ARRAY)
            {
                bValid = TRUE;
            }
        }
        break;

        case PHONETICS_COL_VER:
        {
            *peLabelType = PHONETICS_HIGHBAND_LABEL_EVER;

            // This has to be an integer
            if (psColumn->eType == TDS_DATA_TYPE_INT)
            {
                bValid = TRUE;
            }
        }
        break;

        case PHONETICS_COL_PHONE0:
        case PHONETICS_COL_PHONE1:
        case PHONETICS_COL_PHONE2:
        case PHONETICS_COL_PHONE3:
        case PHONETICS_COL_PHONE4:
        case PHONETICS_COL_PHONE5:
        case PHONETICS_COL_PHONE6:
        case PHONETICS_COL_PHONE7:
        {
            *peLabelType = PHONETICS_HIGHBAND_LABEL_PHONETICS;

            // This has to be a string
            if (psColumn->eType == TDS_DATA_TYPE_STRING)
            {
                bValid = TRUE;
            }
        }
        break;

        case PHONETICS_COL_LANG0:
        case PHONETICS_COL_LANG1:
        case PHONETICS_COL_LANG2:
        case PHONETICS_COL_LANG3:
        case PHONETICS_COL_LANG4:
        case PHONETICS_COL_LANG5:
        case PHONETICS_COL_LANG6:
        case PHONETICS_COL_LANG7:
        {
            *peLabelType = PHONETICS_HIGHBAND_LABEL_LANGUAGE;

            // This has to be an integer
            if (psColumn->eType == TDS_DATA_TYPE_INT)
            {
                bValid = TRUE;
            }
        }
        break;

        default:
        {
            *peLabelType = PHONETICS_HIGHBAND_UNKNOWN_LABEL;

            // Not a problem because we really don't care
            bValid = TRUE;
        }
        break;
    }

    return bValid;
}

/*****************************************************************************
*
*   hCreateTable
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to start working on a new instance of a table for phonetics.
*
*****************************************************************************/
static TDS_UTIL_TABLE_INSTANCE hCreateTable (
    TDS_TABLE_ID tId,
    TDS_TABLE_DEF_VERSION tTabDefVer,
    TDS_TABLE_TIME tTimeInMinutes,
    PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj
        )
{
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)NULL;
    PHONETICS_FILE_TYPE_ENUM eType;
    BOOLEAN bValid;

    // Validate SMS object
    bValid = SMSO_bValid((SMS_OBJECT)psObj);

    // Verify inputs
    if ((bValid == FALSE) ||
        (tId == TDS_TABLE_INVALID_ID) ||
        (tTabDefVer == TDS_TABLE_INVALID_DEF_VERSION))
    {
        return TDS_UTIL_INVALID_TABLE_INSTANCE;
    }

    // Does phonetics know about this table id?
    eType = GsPhoneticsMgrIntf.eTypeForId((UN8)tId);
    if (eType >= PHONETICS_FILE_INVALID_TYPE)
    {
        // Nope!
        return TDS_UTIL_INVALID_TABLE_INSTANCE;
    }

    do
    {

        // Allocate memory for this table
        psTable = (PHONETICS_HIGHBAND_TABLE_STRUCT *)
            SMSO_hCreate(
                PHONETICS_HIGHBAND_OBJECT_NAME":Table",
                sizeof(PHONETICS_HIGHBAND_TABLE_STRUCT),
                (SMS_OBJECT)psObj, FALSE);
        if (psTable == NULL)
        {
            break;
        }

        // Create the phonetic list for this table
        psTable->hList = GsPhoneticsMgrIntf.hCreatePhoneticsList(eType);
        if (psTable->hList == OSAL_INVALID_OBJECT_HDL)
        {
            break;
        }

        // Store the table type & definition version
        psTable->tDefVer = tTabDefVer;
        psTable->eType = eType;
        psTable->bPhoneticsNodePopulated = FALSE;

        // Clear the node pointer
        psTable->psPhonetics = (PHONETICS_LIST_NODE_STRUCT *)NULL;

        // Clear the phonetics array
        OSAL.bMemSet(
            &psTable->ahPhonetics[0], 0,
            sizeof(STRING_OBJECT) * PHONETICS_MAX_PHONETIC_ENTRIES);

        return (TDS_UTIL_TABLE_INSTANCE)psTable;

    } while (FALSE);

    if (psTable != NULL)
    {
        vDestroyTable((TDS_UTIL_TABLE_INSTANCE)psTable);
    }

    return TDS_UTIL_INVALID_TABLE_INSTANCE;
}

/*****************************************************************************
*
*   bAddColumnToTable
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to add a column to a phonetics table.
*
*   However: Even though the phonetics service may update its tables from
*   time to time, we really don't care about anything other than the base
*   definition which is guaranteed to always be there and never gets changed.
*   So, we'll just use this to verify structure and for debug output.
*
*****************************************************************************/
static BOOLEAN bAddColumnToTable (
    TDS_UTIL_TABLE_INSTANCE hTable,
    BOOLEAN bLastColumn,
    TDS_COLUMN_STRUCT *psColumn,
    PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj
        )
{
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)hTable;
    BOOLEAN bValid;

    // Validate SMS objects
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if (bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)psTable);
    }

    if (bValid == FALSE)
    {
        return FALSE;
    }

    // Validate this column definition now
    bValid = bValidateColumn(psColumn, (PHONETICS_HIGHBAND_LABEL_ENUM *)NULL);
    if (bValid == FALSE)
    {
        return FALSE;
    }

    printf(PHONETICS_HIGHBAND_OBJECT_NAME
        ": Column added to table %s:\n"
        "    Label Id: %d\n"
        "    Type: %s\n"
        "    %s\n",
        GsPhoneticsMgrIntf.pacFileType(psTable->eType),
        psColumn->tLabelId,
        TDS_UTIL_pacDataType(psColumn->eType),
        (bLastColumn == TRUE)?"Table Complete":" ");

    return TRUE;
}

/*****************************************************************************
*
*   bRowStatus
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to inform us that a new row is starting or if we're
*   ending a row we previously worked on.
*
*****************************************************************************/
static BOOLEAN bRowStatus (
    TDS_UTIL_TABLE_INSTANCE hTable,
    TDS_ROW_OPERATION_ENUM eRowOperation,
    PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj
        )
{
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)hTable;
    BOOLEAN bValid, bSuccess = FALSE;

    // Validate SMS objects
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if (bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)psTable);
    }

    if (bValid == FALSE)
    {
        return FALSE;
    }

    // Are we starting a new row?
    if (eRowOperation == TDS_ROW_OPERATION_START_ROW)
    {
        // Clear the phonetics array
        OSAL.bMemSet(
            &psTable->ahPhonetics[0], 0,
            sizeof(STRING_OBJECT) * PHONETICS_MAX_PHONETIC_ENTRIES);

        // If TDS only reported columns that the manager didn't
        // accept we'll have an empty phonetic entry here
        if (psTable->psPhonetics != NULL)
        {
            if (psTable->bPhoneticsNodePopulated == TRUE)
            {
                // This is a memory leak!
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	                PHONETICS_HIGHBAND_OBJECT_NAME
                    ": Table memory leak in %s)\n",
                    GsPhoneticsMgrIntf.pacFileType(psTable->eType));
            }
            else
            {
                // Not a problem, just use this node for this
                // row instead of allocating another
                bSuccess = TRUE;
            }
        }
        else
        {
            psTable->psPhonetics = (PHONETICS_LIST_NODE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    PHONETICS_HIGHBAND_OBJECT_NAME":ListEntry",
                    sizeof(PHONETICS_LIST_NODE_STRUCT),
                    TRUE );
            if (psTable->psPhonetics != (PHONETICS_LIST_NODE_STRUCT *)NULL)
            {
                // Phonetics node allocated, but no data in it yet
                psTable->bPhoneticsNodePopulated = FALSE;

                // This entry is initially without any phonetic data
                psTable->psPhonetics->bPhoneticsPresent = FALSE;

                psTable->psPhonetics->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

                bSuccess = TRUE;
            }
        }
    }
    else if (eRowOperation == TDS_ROW_OPERATION_END_ROW)
    {
        // Verify the phonetics array has been cleared
        UN8 un8Index;

        for (un8Index = 0; un8Index < PHONETICS_MAX_PHONETIC_ENTRIES; un8Index++)
        {
            if (psTable->ahPhonetics[un8Index] != STRING_INVALID_OBJECT)
            {
                // This is a memory leak due to bad OTA data
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
	                PHONETICS_HIGHBAND_OBJECT_NAME
                    ": Phonetic entry memory leak in %s\n",
                    GsPhoneticsMgrIntf.pacFileType(psTable->eType));

                STRING_vDestroy(psTable->ahPhonetics[un8Index]);
                psTable->ahPhonetics[un8Index] = STRING_INVALID_OBJECT;
            }
        }

        if (psTable->bPhoneticsNodePopulated == TRUE)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Add entry to list (should be easy since this was
            // pre-allocated)
            eReturnCode = OSAL.eLinkedListAdd(
                psTable->hList,
                &psTable->psPhonetics->hEntry,
                (void *)psTable->psPhonetics);
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Forget about this entry
                psTable->psPhonetics = (PHONETICS_LIST_NODE_STRUCT *)NULL;
                psTable->bPhoneticsNodePopulated = FALSE;
                bSuccess = TRUE;
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bAddRowDataToTable
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to add data to the current row
*
*****************************************************************************/
static BOOLEAN bAddRowDataToTable (
    TDS_UTIL_TABLE_INSTANCE hTable,
    TDS_COLUMN_ENTRY_STRUCT *psColumnEntry,
    PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj
        )
{
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)hTable;
    BOOLEAN bValid, bSuccess = TRUE;
    PHONETICS_HIGHBAND_LABEL_ENUM eLabel =
        PHONETICS_HIGHBAND_UNKNOWN_LABEL;

    // Validate SMS objects
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if (bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)psTable);
    }

    if (bValid == FALSE)
    {
        return FALSE;
    }

    if (psTable->psPhonetics == NULL)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            PHONETICS_HIGHBAND_OBJECT_NAME": Table %s corrupted)\n",
            GsPhoneticsMgrIntf.pacFileType(psTable->eType));
        return FALSE;
    }

    // Did we get the expected data type for this column?
    bValid = bValidateColumn(&psColumnEntry->sColumn, &eLabel);
    if (bValid == FALSE)
    {
        // We have bad data here
        return FALSE;
    }

    switch (eLabel)
    {
        case PHONETICS_HIGHBAND_LABEL_ID1:
        {
            // Label Entry ID
            psTable->psPhonetics->sID.un32Id =
                psColumnEntry->uData.sInt.un32Data;
            psTable->psPhonetics->sID.un8NumLeagues = 0;

            // All we need is the primary id to call this
            // node populated
            psTable->bPhoneticsNodePopulated = TRUE;
        }
        break;

        case PHONETICS_HIGHBAND_LABEL_ID2:
        {
            // League List
            size_t tNumLeagues = psColumnEntry->uData.sIntArray.tArraySize,
                   tIndex = 0;

            // Cap the number of leagues we'll take in (SX-9845-0246, Section No.One.Cares)
            if (tNumLeagues > PHONETICS_MAX_LEAGUES)
            {
                tNumLeagues = PHONETICS_MAX_LEAGUES;
            }

            psTable->psPhonetics->sID.un8NumLeagues = 0;

            for (tIndex = 0; tIndex < tNumLeagues; tIndex++)
            {
                // Copy the league
                psTable->psPhonetics->sID.aun16Leagues[tIndex] =
                    (UN16)psColumnEntry->uData.sIntArray.pun32Integers[tIndex];

                // We have another league now
                psTable->psPhonetics->sID.un8NumLeagues++;
            }
        }
        break;

        case PHONETICS_HIGHBAND_LABEL_EVER:
        {
            // Entry version
            // We don't need this
        }
        break;

        case PHONETICS_HIGHBAND_LABEL_PHONETICS:
        {
            // Phonetic Entry

            // Get the phonetics array index for this ID
            UN8 un8ArrayIndex = PHONETICS_COL_ID_TO_INDEX(psColumnEntry->sColumn.tLabelId);

            // Verify the index we calculated
            if (un8ArrayIndex >= PHONETICS_MAX_PHONETIC_ENTRIES)
            {
                bSuccess = FALSE;
                break;
            }

            // Just take this string from TDS (it's okay for this
            // string to be STRING_INVALID_OBJECT)
            psTable->ahPhonetics[un8ArrayIndex] = psColumnEntry->uData.sString.hString;

            if (psColumnEntry->uData.sString.hString != STRING_INVALID_OBJECT)
            {
                // We have new phonetics for this entry
                psTable->psPhonetics->bPhoneticsPresent = TRUE;

                // Clear this handle now
                psColumnEntry->uData.sString.hString = STRING_INVALID_OBJECT;
            }

            break;
        }

        case PHONETICS_HIGHBAND_LABEL_LANGUAGE:
        {
            // Get the phonetics array index for this ID
            UN8 un8ArrayIndex = LANGUAGE_COL_ID_TO_INDEX(psColumnEntry->sColumn.tLabelId);
            UN8 un8Language = psColumnEntry->uData.sInt.un32Data;

            // Verify the language provided and the index we calculated
            if ((un8Language >= PHONETICS_MAX_LANGUAGES) ||
                (un8ArrayIndex >= PHONETICS_MAX_PHONETIC_ENTRIES))
            {
                bSuccess = FALSE;
                break;
            }

            // Language indicator for one of the phonetic entries
            if (psTable->psPhonetics->ahPhonetics[un8Language] == STRING_INVALID_OBJECT)
            {
                // Just take this string from TDS
                psTable->psPhonetics->ahPhonetics[un8Language] = psTable->ahPhonetics[un8ArrayIndex];
            }
            else
            {
                // Get the manager to concatenate these two
                bSuccess = GsPhoneticsMgrIntf.bConcatPhoneticData(
                    psTable->psPhonetics->ahPhonetics[un8Language],
                    psTable->ahPhonetics[un8ArrayIndex]);

                if (bSuccess == TRUE)
                {
                    // Free the source string now
                    STRING_vDestroy(psTable->ahPhonetics[un8ArrayIndex]);
                }
            }

            if (bSuccess == TRUE)
            {
                // Clear the handle from the array
                psTable->ahPhonetics[un8ArrayIndex] = STRING_INVALID_OBJECT;
            }
        }
        break;

        case PHONETICS_HIGHBAND_UNKNOWN_LABEL:
        {
            // Not a problem -- we just don't care
        }
        break;

        default:
        {
            // This should never happen
            bSuccess = FALSE;
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vDestroyTable
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to destroy an in-progress table for one of three reasons:
*   1) An error occured
*   2) We disconnected while TDS was building a table
*
*****************************************************************************/
static void vDestroyTable (
    TDS_UTIL_TABLE_INSTANCE hTable
        )
{
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)hTable;

    if (psTable != NULL)
    {
        UN8 un8Index;

        // Clear simple attributes
        psTable->eType = PHONETICS_FILE_INVALID_TYPE;
        psTable->tDefVer = TDS_TABLE_INVALID_DEF_VERSION;

        // Clear in-progress phonetic node, if it exists
        if (psTable->psPhonetics != NULL)
        {
            GsPhoneticsMgrIntf.bRemoveNode(psTable->psPhonetics, NULL);
            OSAL.vLinkedListMemoryFree(psTable->psPhonetics);
            psTable->psPhonetics = NULL;
        }

        // Clear phonetics array
        for (un8Index = 0;
             un8Index < PHONETICS_MAX_PHONETIC_ENTRIES;
             un8Index++)
        {
            STRING.vDestroy(psTable->ahPhonetics[un8Index]);
            psTable->ahPhonetics[un8Index] = STRING_INVALID_OBJECT;
        }

        // Destroy phonetic table, if it exists
        if (psTable->hList != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            eReturnCode = OSAL.eLinkedListIterate(
                psTable->hList,
                (OSAL_LL_ITERATOR_HANDLER)GsPhoneticsMgrIntf.bRemoveNode,
                NULL);
            if (eReturnCode == OSAL_SUCCESS)
            {
                OSAL.eLinkedListDelete(psTable->hList);
                psTable->hList = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)hTable);
    }

    return;
}

/*****************************************************************************
*
*   bNewTableReady
*
*   This is a callback defined for use with TDS_UTIL and is invoked whenver
*   TDS wishes to inform us that a table has been completed
*
*****************************************************************************/
static BOOLEAN bNewTableReady (
    TDS_UTIL_TABLE_INSTANCE hTable,
    TDS_TABLE_CONTENT_VERSION tTabContentVer,
    PHONETICS_HIGHBAND_OBJECT_STRUCT *psObj
        )
{
    BOOLEAN bValid, bSuccess = TRUE;
    PHONETICS_HIGHBAND_TABLE_STRUCT *psTable =
        (PHONETICS_HIGHBAND_TABLE_STRUCT *)hTable;

    // Validate SMS objects
    bValid = SMSO_bValid((SMS_OBJECT)psObj);
    if (bValid == TRUE)
    {
        bValid = SMSO_bValid((SMS_OBJECT)psTable);
    }

    if (bValid == FALSE)
    {
        return FALSE;
    }

    // Report the phonetics list to the mgr,
    // clear all stuff related to this table
    bSuccess = GsPhoneticsMgrIntf.bNewList(
        psObj->hPhoneticsService,
        psTable->eType,
        (UN8)psTable->tDefVer,
        (UN16)tTabContentVer,
        psTable->hList );
    if (bSuccess == TRUE)
    {
        // Clear the linked list handle
        psTable->hList = OSAL_INVALID_OBJECT_HDL;
    }

    // We're done with this table
    vDestroyTable(hTable);

    return bSuccess;
}
