/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the PVN1 Movie (currently on high-band)
 *  functions for the Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include "sms_api.h"
#include "dataservice_base.h"
#include "dataservice_mgr_obj.h"
#include "baudot.h"
#include "string_obj.h"

#include "movie_obj.h"
#include "rfd_interface_obj.h"
#include "db_util.h"
#include "ds_util.h"
#include "movies_interface.h"
#include "theater_times_obj.h"
#include "_movies_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   tMinimumOTABufferByteSize
*
*   Calculate the minimum number of bytes needed for this service
*   based upon how it's going to be used
*
*****************************************************************************/
static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        )
{
    // We'll always need this as a starting point
    size_t tByteSize = MOVIES1_TIMES_PAYLOAD_MAX_SIZE
        + MOVIES1_DESC_PAYLOAD_MAX_SIZE;

    if (TRUE == bDBUpdatesEnabled)
    {
        tByteSize += MOVIES1_RFD_META_PAYLOAD_MAX_SIZE
            + MOVIES1_RFD_BLOCK_PAYLOAD_MAX_SIZE;
    }

    return tByteSize;
}

/*****************************************************************************
*
*   hInit
*
*****************************************************************************/
static MOVIES_INTERFACE_OBJECT hInit (
    MOVIES_SERVICE_OBJECT hMoviesService,
    SMS_OBJECT hParent,
    BOOLEAN bDBUpdatesEnabled,
    UN8 un8CurDBVersion
        )
{
    MOVIES1_OBJECT_STRUCT *psObj;
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl = NULL;

    // Create an instance of this object
    psObj = (MOVIES1_OBJECT_STRUCT *)
        SMSO_hCreate(
            MOVIES1_OBJECT_NAME,
            sizeof(MOVIES1_OBJECT_STRUCT),
            hParent, FALSE );

    if (psObj == NULL)
    {
        return MOVIES_INTERFACE_INVALID_OBJECT;
    }

    do
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Save the service handle
        psObj->hMoviesService = hMoviesService;

        // Request access to ISO 3309 CRC32
        eReturnCode = OSAL.eGetCRC(
            &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! destroy object
            break;
        }

        eReturnCode = OSAL.eLinkedListCreate(
            &psObj->sTimesCtrl.hStateCarouselList,
            MOVIES1_OBJECT_NAME":StateCarList",
            (OSAL_LL_COMPARE_HANDLER)n16CompareStateCarEntry,
            (OSAL_LL_OPTION_LINEAR | OSAL_LL_OPTION_UNIQUE));

        if (eReturnCode != OSAL_SUCCESS)
        {
            // Error! destroy object
            break;
        }

        psObj->sDescCtrl.tCurVersion = MOVIE_INVALID_VERSION;
        psObj->sTimesCtrl.tCurVersion = MOVIE_INVALID_VERSION;

        // Ensure this is cleared
        psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;

        if (bDBUpdatesEnabled == TRUE)
        {
            // Create RFD processor object
            psRFDCtrl = psCreateRFDCtrl(psObj);
            if (psRFDCtrl == NULL)
            {
                break;
            }

            psObj->hRFD = RFD_INTERFACE_hConnect(
                MOVIES1_RFD_CLIENT_ID,
                (RFD_UPDATE_VERSION)un8CurDBVersion,
                MOVIES1_MAX_VERSION_BITLEN,
                eRFDFileProcessor,
                (RFD_OPTIONAL_CALLBACKS_STRUCT *)NULL,
                (void *)psRFDCtrl);

            if (psObj->hRFD == RFD_INTERFACE_INVALID_OBJECT)
            {
                // Error! destroy object
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES1_OBJECT_NAME": Can't create RFD connection");
                break;
            }
        }

        return (MOVIES_INTERFACE_OBJECT)psObj;

    } while (0);

    vDestroyRFDCtrl(psRFDCtrl);
    vUninit((MOVIES_INTERFACE_OBJECT)psObj);

    return MOVIES_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   eParseMessageHeader
*
*****************************************************************************/
static MOVIE_MESSAGE_TYPE_ENUM eParseMessageHeader (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    OSAL_BUFFER_HDL hPayload,
    MOVIE_VERSION *ptVersion
        )
{
    MOVIE_MESSAGE_TYPE_ENUM eMessageType = MOVIE_MESSAGE_TYPE_ERROR;

    // Verify inputs
    if ((hInterfaceObj == MOVIES_INTERFACE_INVALID_OBJECT) ||
        (hPayload == OSAL_INVALID_BUFFER_HDL) ||
        (ptVersion == NULL)
       )
    {
        return MOVIE_MESSAGE_TYPE_ERROR;
    }

    // Initialize version
    *ptVersion = 0;

    do
    {
        BOOLEAN bOwner;
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;
        BOOLEAN bMessageValid;
        UN8 un8CarouselID = 0;
        PVN tPVN = (PVN)0;
        size_t tBitsRead;

        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Peek at the first bits to see if this is the right PVN and Car ID
        // and if this should go to RFD
        tBitsRead = OSAL.tBufferPeekBits(hPayload, &tPVN, 0,
            MOVIES1_PVN_BITLEN, 0);

        if (tBitsRead != MOVIES1_PVN_BITLEN)
        {
            puts(MOVIES1_OBJECT_NAME" PVN Peek Failed\n");
            break;
        }

        tBitsRead = OSAL.tBufferPeekBits(hPayload, &un8CarouselID, 0,
            MOVIES1_CAROUSEL_BITLEN, MOVIES1_PVN_BITLEN);

        if (tBitsRead != MOVIES1_CAROUSEL_BITLEN)
        {
            puts(MOVIES1_OBJECT_NAME" CarID Peek Failed\n");
            break;
        }

        if (tPVN != MOVIES1_PVN)
        {
            printf(MOVIES1_OBJECT_NAME" Unknown PVN: %u\n", tPVN);

            // Ignore this message
            eMessageType = MOVIE_MESSAGE_TYPE_IGNORE;

            break;
        }

        printf(MOVIES1_OBJECT_NAME": CarID %d found\n", un8CarouselID);
        if ((un8CarouselID == MOVIES1_RFD_UPDATE_CAR_ID) ||
            (un8CarouselID == MOVIES1_RFD_METADATA_CAR_ID))
        {
            BOOLEAN bRFDProcessed;

            // Pass the RFD AU to the RFD Interface
            bRFDProcessed =
                RFD_INTERFACE_bProcessPayload(psObj->hRFD, hPayload);

            if (bRFDProcessed != TRUE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    MOVIES1_OBJECT_NAME
                    ": RFD_INTERFACE_eProcessPayload() failed (%u)");
            }

            return MOVIE_MESSAGE_TYPE_RFD;
        }

        // Are we processing a times message?
        if ((MOVIE_MESSAGE_TYPE_ENUM)un8CarouselID == MOVIE_MESSAGE_TYPE_TIMES)
        {
            BOOLEAN bShouldProcess;

            // Clear the current state id field
            psObj->sTimesCtrl.tCurAccessUnitStateID = 0;

            // Peek at the state ID field in the payload now
            tBitsRead = OSAL.tBufferPeekBits(
                hPayload, 
                &psObj->sTimesCtrl.tCurAccessUnitStateID,
                0, MOVIES1_STATEID_BITLEN, 
                MOVIES_STATEID_PEEK_INDEX);
            if (tBitsRead != MOVIES1_STATEID_BITLEN)
            {
                // Read failed -- the message is probably
                // garbled, although that is unlikely since
                // the CRC checked out.
                break;
            }

            // Do we need to process this message?
            bShouldProcess = MOVIES_MGR_bShouldProcessTimes(
                psObj->hMoviesService, psObj->sTimesCtrl.tCurAccessUnitStateID);
            if (bShouldProcess == FALSE)
            {
                printf(MOVIES1_OBJECT_NAME
                    ": Filtering Times Payload: State ID: %d\n",
                    psObj->sTimesCtrl.tCurAccessUnitStateID);

                // Ignore this message
                eMessageType = MOVIE_MESSAGE_TYPE_IGNORE;

                // Nope!
                break;
            }
        }

        // Validate the message
        bMessageValid = DS_UTIL_bIsCRCValid(psObj->hCRC, hPayload, NULL);
        if (bMessageValid == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" Packet Invalid");
            break;
        }

        // Validate the message
        bMessageValid = DS_UTIL_bCutCRC(hPayload);
        if (bMessageValid == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" Failed to cut off CRC");
            break;
        }

        // Seek past the PVN/Carousel Id now
        tBitsRead = OSAL.tBufferSeekHeadBits(
            hPayload, 
            MOVIES1_PVN_BITLEN + MOVIES1_CAROUSEL_BITLEN );

        if (tBitsRead != (MOVIES1_PVN_BITLEN + MOVIES1_CAROUSEL_BITLEN))
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out.
            break;
        }

        // Now grab the version field and state fields
        if ( (((MOVIE_MESSAGE_TYPE_ENUM)un8CarouselID) ==
                  MOVIE_MESSAGE_TYPE_DESCRIPTION) ||
             (((MOVIE_MESSAGE_TYPE_ENUM)un8CarouselID) ==
                 MOVIE_MESSAGE_TYPE_TIMES)
           )
        {
            // Read the Version Field
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, ptVersion, 0,
                    MOVIES1_VER_BITLEN );

            if (tBitsRead != MOVIES1_VER_BITLEN)
            {
                // Read failed
                puts(MOVIES1_OBJECT_NAME" Unable to read Version\n");
                break;
            }

            // Read the state field for the times carousel
            if (((MOVIE_MESSAGE_TYPE_ENUM)un8CarouselID) ==
                    MOVIE_MESSAGE_TYPE_TIMES)
            {
                // Seek past the State Field since we already have it
                tBitsRead = OSAL.tBufferSeekHeadBits (
                        hPayload, MOVIES1_STATEID_BITLEN );

                if (tBitsRead != MOVIES1_STATEID_BITLEN)
                {
                    // Read failed
                    puts(MOVIES1_OBJECT_NAME" Unable to read State ID for Listings\n");
                    break;
                }

                printf(MOVIES1_OBJECT_NAME
                    ": Processing Times Payload: ver: %d, State ID: %d\n",
                    *ptVersion, psObj->sTimesCtrl.tCurAccessUnitStateID);
            }
        }
        else
        {
            printf(MOVIES1_OBJECT_NAME
                " Carousel ID %d not handled at this time\n", un8CarouselID);
            break;
        }

        // We read all the fields, so set the return type
        eMessageType = (MOVIE_MESSAGE_TYPE_ENUM)un8CarouselID;

        switch (eMessageType)
        {
            case MOVIE_MESSAGE_TYPE_DESCRIPTION:
            {
                BOOLEAN bShouldProcess = FALSE, bParsed;

                // Save the version - we need it for the call below
                psObj->sDescCtrl.tCurAccessUnitVersion = *ptVersion;

                bParsed = bParseDescriptionHeader(psObj, hPayload, &bShouldProcess);

                // Should we continue?
                if ((FALSE == bParsed) || (FALSE == bShouldProcess))
                {
                    // Nope
                    eMessageType = MOVIE_MESSAGE_TYPE_IGNORE;
                    break;
                }
            }
            break;

            case MOVIE_MESSAGE_TYPE_TIMES:
            {
                BOOLEAN bShouldProcess = FALSE, bParsed;

                // Save the version - we need it for the call below
                psObj->sTimesCtrl.tCurAccessUnitVersion = *ptVersion;

                // See if we need to process this AU
                bParsed = bParseTimesHeader(psObj, hPayload, &bShouldProcess);

                // Should we continue?
                if ((FALSE == bParsed) || (FALSE == bShouldProcess))
                {
                    // Nope
                    eMessageType = MOVIE_MESSAGE_TYPE_IGNORE;
                    break;
                }
            }
            break;

            default:
                break;
        }

    } while (FALSE);

    return eMessageType;
}

/*****************************************************************************
*
*   hParseDescription
*
*****************************************************************************/
static MOVIE_OBJECT hParseDescription (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    SMS_OBJECT hMovieParent,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbMoreData
        )
{
    STRING_OBJECT hActors = STRING_INVALID_OBJECT;
    MOVIE_MULTI_LANG_FIELD_STRUCT sName, sSynopsis;

    OSAL.bMemSet(&sName, 0, sizeof(MOVIE_MULTI_LANG_FIELD_STRUCT));
    OSAL.bMemSet(&sSynopsis, 0, sizeof(MOVIE_MULTI_LANG_FIELD_STRUCT));

    do
    {
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;
        BOOLEAN bOwner, bFieldPresent, bSuccess;
        size_t tBitsRead, tBitsLeft;
        MOVIE_OBJECT hMovie = MOVIE_INVALID_OBJECT;
        MOVIE_ID tID = 0;
        UN8 un8CurLangIdx, un8CurLang;
        UN8 un8RunTime = 0, un8MapIndex, un8MORESIZE = 0;
        STRING_OBJECT hScratchString = STRING_INVALID_OBJECT;
        MOVIE_RATINGS_MAPPING_STRUCT *psCurMapping;
        MOVIE_RATING_STRUCT *psCurRating;
        size_t tNumRatings = 0;
        MOVIE_RATING_CODE tRating;

        // Verify inputs
        if ((hInterfaceObj == MOVIES_INTERFACE_INVALID_OBJECT) ||
            (hPayload == OSAL_INVALID_BUFFER_HDL) ||
            (pbMoreData == NULL)
           )
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &tID, 0,
                MOVIES1_MVID_BITLEN );

        if (tBitsRead != MOVIES1_MVID_BITLEN)
        {
            // If we can't get the ID we can't do anything
            // with this entry
            break;
        }

        // Read a name for each language in the language map
        // they go by order of MSB
        for (un8CurLangIdx = 0, bSuccess = TRUE;
             un8CurLangIdx < MOVIES1_NUM_LANGUAGES;
             un8CurLangIdx++)
        {
            // Build our cur Lang mask
            un8CurLang = 1 << (MOVIES1_NUM_LANGUAGES - un8CurLangIdx - 1);

            // Is this language present in the map?
            if ( (psObj->sDescCtrl.un8LangMap & un8CurLang) != un8CurLang)
            {
                // This language isn't a part of this message
                continue;
            }

            // Read the presence bit
                bFieldPresent = FALSE;
                tBitsRead = OSAL.tBufferReadHeadBits (
                        hPayload, &bFieldPresent, 0,
                        MOVIES1_PRESENT_FLAG_BITLEN );

                if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

            if (bFieldPresent != TRUE)
                {
                // This language isn't a part of this field
                continue;
            }

            // Parse the name for this language
            hScratchString = BAUDOT_hToString(
                hMovieParent, hPayload,
                        BAUDOT_BEHAVIOR_PROCESS_TO_END,
                        TRUE, TRUE, 0, NULL);

                    if (hScratchString == STRING_INVALID_OBJECT)
                    {
                        // We have to fail here because we may have just lost sync
                        // with the data in this message
                puts(MOVIES1_OBJECT_NAME": Failed to create hName string");
                        bSuccess = FALSE;
                        break;
                    }

            switch (un8CurLang)
                    {
                case MOVIES1_LANG_ENGLISH_MASK:
                {
                    sName.hEnglish = hScratchString;
                    }
                break;

                case MOVIES1_LANG_FRENCH_MASK:
                    {
                    sName.hFrench = hScratchString;
                }
                break;

                case MOVIES1_LANG_SPANISH_MASK:
                {
                    sName.hSpanish = hScratchString;
                }
                break;

                case MOVIES1_LANG_RESERVED1_MASK:
                case MOVIES1_LANG_RESERVED2_MASK:
                case MOVIES1_LANG_RESERVED3_MASK:
                default:
                {
                        STRING_vDestroy(hScratchString);
                    }
                }
            }

        if (bSuccess == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" error reading movie names");
            break;
        }

        // Loop through the available rating codes
        for (un8MapIndex = 0; 
             un8MapIndex < psObj->sDescCtrl.un8NumRatingsMappings; 
             un8MapIndex++)
        {
            // Grab the current mapping structure
            psCurMapping = &psObj->sDescCtrl.asMappings[un8MapIndex];

            bFieldPresent = FALSE;
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &bFieldPresent, 0,
                    MOVIES1_PRESENT_FLAG_BITLEN );

            if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }

            if (bFieldPresent != TRUE)
            {
                continue;
            }

            // We have a rating for this movie here
            tNumRatings++;

            psCurRating = &psObj->sDescCtrl.asRatings[un8MapIndex];

            // Read the rating code for this mapping
            tRating = 0;
                tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &tRating, 0,
                        MOVIES1_RCODE_BITLEN );

                if (tBitsRead != MOVIES1_RCODE_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

            // Get the info for this rating now
            bSuccess = MOVIES_MGR_bGetRatingInfo(
                psObj->hMoviesService,
                psCurMapping->tSCode,
                tRating, 
                psCurRating);
                }

        if (bSuccess == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" error reading rating codes");
            break;
        }

        // Get the run time for the movie
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8RunTime, 0,
                MOVIES1_MVRUN_BITLEN );

        if (tBitsRead != MOVIES1_MVRUN_BITLEN)
        {
            break;
        }

        // According to RX190, MVRUN is in 5-minute increments
        // from a base of 60 minutes.  Adjust our time accordingly
        un8RunTime = MOVIES1_MVRUN_BASE_MINUTES +
            (un8RunTime * MOVIES1_MVRUN_INCR_MINUTES);

        // See if the actors field is present
        bFieldPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            break;
        }

        if (bFieldPresent == TRUE)
        {
            hActors = BAUDOT_hToString(hMovieParent, hPayload,
                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                TRUE, TRUE, 0, NULL);

            if (hActors == STRING_INVALID_OBJECT)
            {
                puts(MOVIES1_OBJECT_NAME": Failed to create hActors string");
                break;
            }
        }

        // Read a movie synopsis for each language in the language map
        // they go by order of MSB
        // We read them all out, but we only keep the english
        for (un8CurLangIdx = 0; un8CurLangIdx < MOVIES1_NUM_LANGUAGES; un8CurLangIdx++)
        {
            // Build our cur Lang mask
            un8CurLang = 1 << (MOVIES1_NUM_LANGUAGES - un8CurLangIdx - 1);

            // Is this language present in the map?
            if ( (psObj->sDescCtrl.un8LangMap & un8CurLang) != un8CurLang)
            {
                // This language isn't a part of this message
                continue;
            }
            
                bFieldPresent = FALSE;
                tBitsRead = OSAL.tBufferReadHeadBits (
                        hPayload, &bFieldPresent, 0,
                        MOVIES1_PRESENT_FLAG_BITLEN );

                if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

            if (bFieldPresent != TRUE)
                {
                // This language isn't a part of this field
                continue;
            }

                    hScratchString = BAUDOT_hToString(hMovieParent, hPayload,
                        BAUDOT_BEHAVIOR_PROCESS_TO_END,
                        TRUE, TRUE, 0, (size_t *)NULL);

                    if (hScratchString == STRING_INVALID_OBJECT)
                    {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__, 
                    MOVIES1_OBJECT_NAME": Failed to create hSynopsis string");
                        bSuccess = FALSE;
                        break;
                    }

            switch (un8CurLang)
                    {
                case MOVIES1_LANG_ENGLISH_MASK:
                {
                    sSynopsis.hEnglish = hScratchString;
                    }
                break;

                case MOVIES1_LANG_FRENCH_MASK:
                    {
                    sSynopsis.hFrench = hScratchString;
                }
                break;

                case MOVIES1_LANG_SPANISH_MASK:
                {
                    sSynopsis.hSpanish = hScratchString;
                }
                break;

                case MOVIES1_LANG_RESERVED1_MASK:
                case MOVIES1_LANG_RESERVED2_MASK:
                case MOVIES1_LANG_RESERVED3_MASK:
                default:
                {
                        STRING_vDestroy(hScratchString);
                    }
                break;
                }
            }

        if (bSuccess == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" error reading movie synopsis");
            break;
        }

        // See if the MORESIZE field is present
        bFieldPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            break;
        }

        if (bFieldPresent == TRUE)
        {
            tBitsRead = OSAL.tBufferReadHeadBits (
                            hPayload, &un8MORESIZE, 0,
                            MOVIES1_MORESIZE_BITLEN );

            if (tBitsRead != MOVIES1_MORESIZE_BITLEN)
            {
                break;
            }

            // We aren't supposed to do anything with the MORE
            // field, so just seek the number of bits specified
            // by MORESIZE
            tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, un8MORESIZE);

            if (tBitsRead != un8MORESIZE)
            {
                break;
            }
        }

        tBitsLeft = OSAL.tBufferGetSizeInBits(hPayload);

        // 8 is bits in a byte.  We will have at most 7 bits of
        // padding bits in the packet
        if (tBitsLeft < 8)
        {
            *pbMoreData = FALSE;
        }
        else
        {
            *pbMoreData = TRUE;
        }

        // Create the movie now
        hMovie = MOVIE_hCreate(
            hMovieParent, tID, &sName, un8RunTime,
              hActors, &sSynopsis, tNumRatings, 
              &psObj->sDescCtrl.asRatings[0]);

        if (hMovie == MOVIE_INVALID_OBJECT)
        {
            puts(MOVIES1_OBJECT_NAME": Error creating movie object\n");
            break;
        }

        // Everything all worked out
        return hMovie;

    } while (FALSE);

    STRING_vDestroy(hActors);
    STRING_vDestroy(sName.hEnglish);
    STRING_vDestroy(sName.hFrench);
    STRING_vDestroy(sName.hSpanish);
    STRING_vDestroy(sSynopsis.hEnglish);
    STRING_vDestroy(sSynopsis.hFrench);
    STRING_vDestroy(sSynopsis.hSpanish);

    return MOVIE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   bMarkDescriptionPayloadAsParsed
*
*****************************************************************************/
static BOOLEAN bMarkDescriptionPayloadAsParsed (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    BOOLEAN *pbCarouselComplete
        )
{
    do
    {
        BOOLEAN bOwner;
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        // Only process this request if we have an active AU
        if (psObj->sDescCtrl.un16CurAccessUnitAUTOT > 0)
        {
            BOOLEAN bSuccess;

            // Mark this AU in our entry
            bSuccess = bSetAUNumAsParsed(
                &psObj->sDescCtrl.pun8AUVector[0],
                psObj->sDescCtrl.un16CurAccessUnitAUTOT,
                psObj->sDescCtrl.un16CurAccessUnitAUNUM,
                pbCarouselComplete);

            if (bSuccess == FALSE)
            {
                break;
            }

            // Reset our Current AU variables
            psObj->sDescCtrl.un16CurAccessUnitAUTOT = 0;
            psObj->sDescCtrl.un16CurAccessUnitAUNUM = 0;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bResetDescriptionPayloadTracking
*
*****************************************************************************/
static BOOLEAN bResetDescriptionPayloadTracking (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        )
{
    puts(MOVIES1_OBJECT_NAME" Attempting to reset desc payload tracking");

    do
    {
        BOOLEAN bOwner;
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        psObj->sDescCtrl.tCurVersion = MOVIE_INVALID_VERSION;

        puts(MOVIES1_OBJECT_NAME" Desc payload tracking successfully reset");

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bParseTimes
*
*****************************************************************************/
static BOOLEAN bParseTimes (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    SMS_OBJECT hTimesParent,
    OSAL_BUFFER_HDL hPayload,
    THEATER_TIMES_OBJECT *phTheaterTimes,
    THEATER_ID *ptTHID,
    BOOLEAN *pbMoreData
        )
{
    do
    {
    BOOLEAN bOwner = FALSE, bSuccess = FALSE;
    MOVIES1_OBJECT_STRUCT *psObj =
        (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;
    size_t tBitsRead = 0, tBitsLeft = 0;
    UN8 un8MVIdx = 0, un8SHOWIdx = 0, 
        un8MVNO = 0, un8SHOWNO = 0, 
        un8ESHOWNO = 0, un8TimeData = 0;
    UN16 un16THID = 0, un16ShowTime = 0;
    THEATER_ID tTHID = 0;
    MOVIE_ID tMVID = 0;
    THEATER_TIMES_MOVIE_ENTRY_OBJECT hMovieEntry = 
        THEATER_TIMES_MOVIE_ENTRY_INVALID_OBJECT;
        MOVIE_RATING_CODE tRatingException;
        UN16 un16Index;

        // Verify inputs
        if ((hInterfaceObj == MOVIES_INTERFACE_INVALID_OBJECT) ||
            (hPayload == OSAL_INVALID_BUFFER_HDL) ||
            (phTheaterTimes == NULL) ||
            (ptTHID == NULL) ||
            (pbMoreData == NULL)
           )
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        bSuccess = OSAL.bBufferReadBitsToUN16 (
                hPayload, &un16THID,
                psObj->sTimesCtrl.un8CurAccessUnitTHIDSIZE );

        if (bSuccess == FALSE)
        {
            break;
        }

        tTHID = MAKE_UNIQUE_THID(
            psObj->sTimesCtrl.tCurAccessUnitStateID, un16THID);

        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8MVNO, 0,
                MOVIES1_MVNO_BITLEN );

        if (tBitsRead != MOVIES1_MVNO_BITLEN)
        {
            break;
        }

        // According to RX190, Sec 6.2.2,
        // MVNO is really MVNO+1, so increment accordingly
        un8MVNO++;

        printf(MOVIES1_OBJECT_NAME": Processing Times for %u(%u:%u) with %d movies\n",
            tTHID, psObj->sTimesCtrl.tCurAccessUnitStateID, un16THID, un8MVNO);

        // Do we need to create a new times object?
        if (*phTheaterTimes == THEATER_TIMES_INVALID_OBJECT)
        {
            // Yes, create it now
            *phTheaterTimes = THEATER_TIMES_hCreate(
                psObj->hMoviesService, hTimesParent, tTHID,
                EPOCH_TO_UNIXTIME(psObj->sTimesCtrl.un16CurAccessUnitEPOCH,
                psObj->sTimesCtrl.un8CurAccessUnitETIME),
                un8MVNO);

            if (*phTheaterTimes == THEATER_TIMES_INVALID_OBJECT)
            {
                break;
            }
        }
        else
        {
            // No, re-use the one we have here
            bSuccess = THEATER_TIMES_bReuse(
                *phTheaterTimes, tTHID,
                EPOCH_TO_UNIXTIME(psObj->sTimesCtrl.un16CurAccessUnitEPOCH,
                psObj->sTimesCtrl.un8CurAccessUnitETIME),
                un8MVNO);

            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // We have a usable THID now
        *ptTHID = tTHID;

        // Loop through the MVNO movies that have times
        for (un8MVIdx = 0;
             (un8MVIdx < un8MVNO) && (bSuccess == TRUE);
             un8MVIdx++)
        {
            // Zero out tMVID since OSAL.tBufferReadHeadBits won't do it for us
            tMVID = 0;
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &tMVID, 0,
                    MOVIES1_MVID_BITLEN );

            if (tBitsRead != MOVIES1_MVID_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }

            un8SHOWNO = 0;
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &un8SHOWNO, 0,
                    MOVIES1_SHOWNO_BITLEN );

            if (tBitsRead != MOVIES1_SHOWNO_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }

            // According to RX190, Sec 6.2.5
            // the ESHOWNO field is only present if SHOWNO
            // is equal to 0xF

            if (un8SHOWNO == MOVIES1_ESHOWNO_PRESENT_VALUE)
            {
                un8ESHOWNO = 0;
                tBitsRead = OSAL.tBufferReadHeadBits (
                        hPayload, &un8ESHOWNO, 0,
                        MOVIES1_ESHOWNO_BITLEN );

                if (tBitsRead != MOVIES1_ESHOWNO_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }
            }

            // According to the table found in RX190, Sec 6.2.5,
            // adjust or un8SHOWNO variable accordingly
            if (un8SHOWNO < MOVIES1_ESHOWNO_PRESENT_VALUE)
            {
                // the SHOWNO field represents SHOWNO+1 fields
                // so increment to handle this
                un8SHOWNO++;
            }
            else if ( (un8SHOWNO == MOVIES1_ESHOWNO_PRESENT_VALUE) &&
                      (un8ESHOWNO < MOVIES1_NO_SHOWTIMES_VALUE)
                    )
            {
                // According to RX190, Sec 6.2.5,
                // the number of showtimes is now ESHOWNO+16
                un8SHOWNO = un8ESHOWNO + MOVIES1_MAX_NUM_TIMES_FOR_SHOWNO;
            }
            else if ( (un8SHOWNO == MOVIES1_NO_SHOWTIMES_VALUE) &&
                      (un8ESHOWNO == MOVIES1_NO_SHOWTIMES_VALUE)
                    )
            {
                // If both fields are equal to 0xF, then this means that
                // there are no showtimes for this movie at this theater
                // The App would display something like "Call Theater for Time(s)"
                // in this case.  We don't care about that noise though, we just
                // know that we don't have any times to parse for this MVID
                un8SHOWNO = 0;
            }
            else
            {
                // Shouldn't happen, so log an error and break out
                printf(MOVIES1_OBJECT_NAME
                    ": Invalid values found in SHOWNO(%d) and/or ESHOWNO(%d)\n",
                    un8SHOWNO, un8ESHOWNO);
                break;
            }

            printf(MOVIES1_OBJECT_NAME": Found MVID:%u with %d showtimes\n",tMVID, un8SHOWNO);

            // No exception yet for this movie
            tRatingException = MOVIE_RATING_INVALID_CODE;

            // Search for ratings exceptions now for this movie
            for( un16Index = 0; 
                 un16Index < psObj->sTimesCtrl.un16NumExceptions; 
                 un16Index++)
            {
                // Can we match the id?
                if (psObj->sTimesCtrl.asExceptions[un16Index].tMovieID == tMVID)
                {
                    // Yes, use this exception now
                    tRatingException = 
                        psObj->sTimesCtrl.asExceptions[un16Index].tRating;

                    // Stop looking now
                break;
            }
            }

            // Add this movie and any exception we may have for it
            hMovieEntry = THEATER_TIMES_hAddMovie(
                *phTheaterTimes, tMVID, 
                psObj->sTimesCtrl.psRatings->tSCode, 
                tRatingException);

            if (hMovieEntry == THEATER_TIMES_MOVIE_ENTRY_INVALID_OBJECT)
            {
                puts(MOVIES1_OBJECT_NAME
                    ": Failed adding movie to theater times object");
                break;
            }

            // Loop through the SHOWNO showings for this movie
            // We increment the current un16ShowTime value as we
            // move through the showings since each one is an offset
            // from the previous one
            for (un8SHOWIdx = 0, un16ShowTime = 0;
                 un8SHOWIdx < un8SHOWNO; un8SHOWIdx++)
            {
                un8TimeData = 0;
                tBitsRead = OSAL.tBufferReadHeadBits (
                        hPayload, &un8TimeData, 0,
                        MOVIES1_TIME_BITLEN );

                if (tBitsRead != MOVIES1_TIME_BITLEN)
                {
                    bSuccess = FALSE;
                    break;
                }

                while (un8TimeData == MOVIES1_EXT_TIME_VALUE)
                {
                    // This is the largest possible value, but it
                    // is always followed by at least one more
                    // time value(called TIME2 in PVN1 spec).
                    // The value might be zero, might be
                    // MOVIES1_EXT_TIME_VALUE, and must be
                    // concated before we add the show time.
                    un16ShowTime += MOVIES1_EXT_TIME_VALUE;

                    tBitsRead = OSAL.tBufferReadHeadBits (
                            hPayload, &un8TimeData, 0,
                            MOVIES1_TIME_BITLEN );

                    if (tBitsRead != MOVIES1_TIME_BITLEN)
                    {
                        bSuccess = FALSE;
                        break;
                    }
                }

                if (bSuccess == FALSE)
                {
                    // We had a problem while reading
                    // the extended start time values, so exit out
                    puts(MOVIES1_OBJECT_NAME
                        ": Failed reading extended showtime values\n");
                    break;
                }

                // Update our start time value
                un16ShowTime += un8TimeData;

                // Now we have a complete showtime, so add it to
                // our THEATER_TIMES object
                bSuccess = THEATER_TIMES_bAddShowTimeOffset(
                    *phTheaterTimes, hMovieEntry, un16ShowTime);

                if (bSuccess == FALSE)
                {
                    puts(MOVIES1_OBJECT_NAME
                        ": Failed adding showtime to theater times object\n");
                    break;
                }
            }
        }

        if (bSuccess == FALSE)
        {
            // We failed to read a value correctly at some point
            // in the for-loops above, exit out
            break;
        }

        tBitsLeft = OSAL.tBufferGetSizeInBits(hPayload);

        // 8 is bits in a byte.
        // We will have at most 7 bits of padding bits in the packet
        if (tBitsLeft < 8)
        {
            *pbMoreData = FALSE;
        }
        else
        {
            *pbMoreData = TRUE;
        }

        // Everything worked out
        return TRUE;

    } while (FALSE);

    // Some sort of error
    return FALSE;
}

/*****************************************************************************
*
*   bMarkTimesPayloadAsParsed
*
*****************************************************************************/
static BOOLEAN bMarkTimesPayloadAsParsed (
    MOVIES_INTERFACE_OBJECT hInterfaceObj,
    BOOLEAN *pbCarouselComplete
        )
{
    BOOLEAN bOwner, bFound, bSuccess;
    MOVIES1_OBJECT_STRUCT *psObj =
        (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry;

    do
    {
        // Verify inputs
        if (hInterfaceObj == MOVIES_INTERFACE_INVALID_OBJECT)
        {
            break;
        }

        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        bFound = bFindStateAUEntry(psObj,
            psObj->sTimesCtrl.tCurAccessUnitStateID,
            0,
            FALSE,  // Just match the state entry, not the AUNUM
            &psEntry);

        if (bFound == FALSE)
        {
            // We don't have an entry for this state
            // so create it
            psEntry = (MOVIE_TIMES_STATE_CAROUSEL_STRUCT *)
                SMSO_hCreate(MOVIES1_OBJECT_NAME":StCarEntry",
                    sizeof(MOVIE_TIMES_STATE_CAROUSEL_STRUCT),
                    (SMS_OBJECT)psObj, FALSE);

            if (psEntry == NULL)
            {
                break;
            }

            // Initilize
            psEntry->tID = psObj->sTimesCtrl.tCurAccessUnitStateID;
            psEntry->bActive = FALSE;

            // Save our entry to our linked-list
            eReturnCode = OSAL.eLinkedListAdd(
                psObj->sTimesCtrl.hStateCarouselList,
                OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
                (void *)psEntry);

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Since it isn't in our list we
                // have no other refrence to it
                // so get rid of it before we error out
                vReleaseStateCarEntry(psEntry);
                break;
            }
        }

        // Mark this AU in our entry
        bSuccess = bMarkTimesAUNumAsParsed(psEntry,
            psObj->sTimesCtrl.un16CurAccessUnitAUTOT,
            psObj->sTimesCtrl.un16CurAccessUnitAUNUM,
            pbCarouselComplete);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Reset our Current AU variables
        psObj->sTimesCtrl.un16CurAccessUnitAUTOT = 0;
        psObj->sTimesCtrl.un16CurAccessUnitAUNUM = 0;
        psObj->sTimesCtrl.un16CurAccessUnitEPOCH = 0;
        psObj->sTimesCtrl.un8CurAccessUnitETIME = 0;
        psObj->sTimesCtrl.un8CurAccessUnitTHIDSIZE = 0;

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bResetTimesPayloadTracking
*
*****************************************************************************/
static BOOLEAN bResetTimesPayloadTracking (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        )
{
    puts(MOVIES1_OBJECT_NAME" Attempting to reset times payload tracking");

    do
    {
        BOOLEAN bOwner;
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Verify inputs
        bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
        if (bOwner == FALSE)
        {
            break;
        }

        eReturnCode = OSAL.eLinkedListIterate(
            psObj->sTimesCtrl.hStateCarouselList,
            (OSAL_LL_ITERATOR_HANDLER)bResetStateAUEntryIterator,
            NULL);

        if ( (eReturnCode != OSAL_SUCCESS) &&
             (eReturnCode != OSAL_NO_OBJECTS)
           )
        {
            break;
        }

        psObj->sTimesCtrl.tCurVersion = MOVIE_INVALID_VERSION;

        puts(MOVIES1_OBJECT_NAME" Times payload tracking successfully reset");

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vUninit
*
*****************************************************************************/
static void vUninit (
    MOVIES_INTERFACE_OBJECT hInterfaceObj
        )
{
    BOOLEAN bOwner;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterfaceObj);
    if (bOwner == TRUE)
    {
        MOVIES1_OBJECT_STRUCT *psObj =
            (MOVIES1_OBJECT_STRUCT *)hInterfaceObj;

        if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
        {
            RFD_INTERFACE_vDisconnect(psObj->hRFD);
            psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;
        }

        // Destroy the CRC object if it exists
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            // Release our hold of the CRC computation handle
            eReturnCode = OSAL.eReleaseCRC( psObj->hCRC );

            // Only finish clean up if that succeeded.  This allows
            // us to more easily track this kind of issue
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
            }
        }

        // Destroy Times State Carousel Tracking List if it exists
        if (psObj->sTimesCtrl.hStateCarouselList != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eLinkedListRemoveAll(
                psObj->sTimesCtrl.hStateCarouselList,
                (OSAL_LL_RELEASE_HANDLER)vReleaseStateCarEntry);

            if (eReturnCode == OSAL_SUCCESS)
            {
                eReturnCode = OSAL.eLinkedListDelete(
                    psObj->sTimesCtrl.hStateCarouselList);

                if (eReturnCode == OSAL_SUCCESS)
                {
                    psObj->sTimesCtrl.hStateCarouselList = OSAL_INVALID_OBJECT_HDL;
                }
            }
        }

        if (psObj->sDescCtrl.pun8AUVector != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psObj->sDescCtrl.pun8AUVector );
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static MOVIES1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    MOVIES1_OBJECT_STRUCT *psObj
        )
{
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (MOVIES1_RFD_CTRL_STRUCT *)
        SMSO_hCreate(
            MOVIES1_OBJECT_NAME":RFDCtrl",
            sizeof(MOVIES1_RFD_CTRL_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (psRFDCtrl == NULL)
    {
        return (MOVIES1_RFD_CTRL_STRUCT *)NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hMoviesService = psObj->hMoviesService;

    return psRFDCtrl;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCurDB = STRING_INVALID_OBJECT;

    do
    {
        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = tStartingIndex;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create FIXED constants that are used to work
        // with Theater Updates
        psRFDCtrl->hDegDivsor =
            OSAL_FIXED.hCreateFromFixed(
                MOVIES1_LAT_LON_DIVSOR_VALUE, 0);

        if (psRFDCtrl->hDegDivsor == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        psRFDCtrl->hLonBase =
            OSAL_FIXED.hCreateFromFixed(
                MOVIES1_LON_BASE_VALUE, 0);

        if (psRFDCtrl->hLonBase == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psRFDCtrl->hBlockPool,
            &psRFDCtrl->hBuffer,
            MOVIES1_RFD_READ_BLOCK_SIZE
                );
        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = MOVIES_MGR_bPrepareForRefDBUpdate(
            psRFDCtrl->hMoviesService, &hCurDB, &psRFDCtrl->hDBPath);

        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting a new update?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB));
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        if (tStartingIndex == 0)
        {
            bSuccess = MOVIES_MGR_bStartDBUpdate(
                psRFDCtrl->hDBConnection,
                &psRFDCtrl->acSQLCommandBuffer[0],
                sizeof(psRFDCtrl->acSQLCommandBuffer)
                    );
            if (bSuccess == FALSE)
            {
                break;
            }
        }
    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hMoviesService = MOVIES_SERVICE_INVALID_OBJECT;
    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    psRFDCtrl->tStartProgressIndex = 0;
    psRFDCtrl->tCurProgressIndex = 0;

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    if (psRFDCtrl->hDegDivsor != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psRFDCtrl->hDegDivsor);
        psRFDCtrl->hDegDivsor = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psRFDCtrl->hLonBase != OSAL_FIXED_INVALID_OBJECT)
    {
        OSAL_FIXED.vDestroy(psRFDCtrl->hLonBase);
        psRFDCtrl->hLonBase = OSAL_FIXED_INVALID_OBJECT;
    }

    SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);

    return;
}

/*****************************************************************************
*
*   bParseDescriptionHeader
*
*****************************************************************************/
static BOOLEAN bParseDescriptionHeader (
    MOVIES1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbShouldProcess
        )
{
    do
    {
        BOOLEAN bSuccess = TRUE, bFieldPresent = FALSE, bAUFound;
        size_t tBitsRead;
        UN16 un16AUNUM = 0, un16AUTOT = 0;
        UN8 un8MAPNO = 0, un8LangMap = 0, un8Index;
        MOVIE_RATINGS_MAPPING_STRUCT *psCurMapping;

        // Read the RSYSVER Field
        tBitsRead = OSAL.tBufferSeekHeadBits (
                hPayload, MOVIES1_RSYSVER_BITLEN );

        if (tBitsRead != MOVIES1_RSYSVER_BITLEN)
        {
            break;
        }

        // Is CTSIZE There?
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            break;
        }

        if (bFieldPresent == TRUE)
        {
            UN8 un8CTSIZE = 0;

            // CTSIZE is present
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &un8CTSIZE, 0,
                    MOVIES1_CTSIZE_BITLEN );

            if (tBitsRead != MOVIES1_CTSIZE_BITLEN)
            {
                break;
            }

            // SX-9845-0027, Sec 5.1.6 states that the size
            // of AUTOT and AUNUM are CTSIZE+1, so
            // increment CTSIZE
            un8CTSIZE++;

            // Now parse AUTOT and AUNUM using CTSIZE
            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUTOT, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                break;
            }

            // SX-9845-0027, Sec 5.1.7 states that there are
            // AUTOT+1 AUs in this carousel.  So
            // increment AUTOT
            un16AUTOT++;

            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUNUM, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                break;
            }
        }
        else
        {
            // No CTSIZE field, so only one AU for this state carousel
            // according to SX-9845-0027, Sec 5.1.5
            un16AUTOT = 1;
        }

        // Read the language
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8LangMap, 0,
                MOVIES1_LANG_BITLEN );

        if (tBitsRead != MOVIES1_LANG_BITLEN)
        {
            break;
        }

        // This payload is supposed to include english language fields, as it is 
        // used as the "base" language in this service. If english is not found,
        // we merely print a warning (for logging purposes) and keep decoding.
        if ( (un8LangMap & MOVIES1_LANG_ENGLISH_MASK) != MOVIES1_LANG_ENGLISH_MASK)
        {
            puts(MOVIES1_OBJECT_NAME" Movie Description AU doesn't contain English");
        }

        // Read the number of rating maps available
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8MAPNO, 0,
                MOVIES1_MAPNO_BITLEN );

        if (tBitsRead != MOVIES1_MAPNO_BITLEN)
        {
            break;
        }

        // SX 9845-0027, Section 5.1.9
        un8MAPNO++;

        // Validate this since we use it to 
        // index into an array
        if (un8MAPNO > MOVIES1_MAX_RATING_MAPPINGS)
        {
            break;
        }

        // Iterate through all of the ratings mappings
        for (un8Index = 0; un8Index < un8MAPNO; un8Index++)
        {
            // Grab the current mapping structure
            psCurMapping = &psObj->sDescCtrl.asMappings[un8Index];

            // Read the RMAP ID
            psCurMapping->un8RMap = 0;
            tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &psCurMapping->un8RMap, 0,
                MOVIES1_RMAP_BITLEN );

            if (tBitsRead != MOVIES1_RMAP_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }

            // Read the SCODE
            psCurMapping->tSCode = 0;
            tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &psCurMapping->tSCode, 0,
                    MOVIES1_SCODE_BITLEN );

            if (tBitsRead != MOVIES1_SCODE_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }
            }

        if (bSuccess == FALSE)
        {
            puts(MOVIES1_OBJECT_NAME" error reading rating mappings or we didn't find our SCODE");
            break;
        }

        // Is this version different than the version we're currently processing?
        if (psObj->sDescCtrl.tCurAccessUnitVersion !=
            psObj->sDescCtrl.tCurVersion)
        {
            // Yeah, reset tracking now
            bSuccess = bResetDescriptionPayloadTracking((MOVIES_INTERFACE_OBJECT)psObj);

            if (bSuccess == FALSE)
            {
                break;
            }

            // Initialize the AU vector
            bSuccess = bInitDescAUVector(psObj, un16AUNUM);

            if (bSuccess == FALSE)
            {
                break;
            }

            // Set our new version
            psObj->sDescCtrl.tCurVersion =
                psObj->sDescCtrl.tCurAccessUnitVersion;
        }

        // See if we already processed this access unit
        bAUFound = bGetAUNumParsed(&psObj->sDescCtrl.pun8AUVector[0],
            un16AUTOT, un16AUNUM);

        if (bAUFound == TRUE )
        {
            *pbShouldProcess = FALSE;
        }
        else
        {
            *pbShouldProcess = TRUE;

            psObj->sDescCtrl.un16CurAccessUnitAUTOT = un16AUTOT;
            psObj->sDescCtrl.un16CurAccessUnitAUNUM = un16AUNUM;
            psObj->sDescCtrl.un8LangMap = un8LangMap;
            psObj->sDescCtrl.un8NumRatingsMappings = un8MAPNO;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bParseTimesHeader
*
*****************************************************************************/
static BOOLEAN bParseTimesHeader (
    MOVIES1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN *pbShouldProcess
        )
{
    do
    {
        BOOLEAN bSuccess = TRUE, bFound, bFieldPresent;
        size_t tBitsRead;
        UN16 un16EPOCH = 0, un16AUNUM = 0, un16AUTOT = 0,
             un16Index;
        UN8 un8THIDSIZE = 0, un8ETIME = 0,
            un8RMAP = 0, un8Index;

        // Is CTSIZE There?
        bFieldPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            break;
        }

        if (bFieldPresent == TRUE)
        {
            UN8 un8CTSIZE = 0;

            // CTSIZE is present
            tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &un8CTSIZE, 0,
                    MOVIES1_CTSIZE_BITLEN );

            if (tBitsRead != MOVIES1_CTSIZE_BITLEN)
            {
                break;
            }

            // RX190, Sec6.1.5 states that the size
            // of AUTOT and AUNUM are CTSIZE+1, so
            // increment CTSIZE
            un8CTSIZE++;

            // Now parse AUTOT and AUNUM using CTSIZE
            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUTOT, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                break;
            }

            // RX190, Sec6.1.6 states that there are
            // AUTOT+1 AUs in this carousel.  So
            // increment AUTOT
            un16AUTOT++;

            bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
                &un16AUNUM, un8CTSIZE);

            if (bSuccess == FALSE)
            {
                break;
            }
        }
        else
        {
            // No CTSIZE field, so only one AU for this state carousel
            // according to RX190, Sec 6.1.5
            un16AUTOT = 1;
        }

        // Read RMAP Field
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &un8RMAP, 0,
                MOVIES1_RMAP_BITLEN );

        if (tBitsRead != MOVIES1_RMAP_BITLEN)
        {
            break;
        }

        // We don't have a mapping yet
        psObj->sTimesCtrl.psRatings = (MOVIE_RATINGS_MAPPING_STRUCT *)NULL;

        // Look for the appropriate mapping
        for (un8Index = 0; 
             un8Index < psObj->sDescCtrl.un8NumRatingsMappings; 
             un8Index++)
        {
            if (psObj->sDescCtrl.asMappings[un8Index].un8RMap == un8RMAP)
            {
                // We have our mapping now
                psObj->sTimesCtrl.psRatings = &psObj->sDescCtrl.asMappings[un8Index];
                break;
            }
        }

        if (psObj->sTimesCtrl.psRatings == (MOVIE_RATINGS_MAPPING_STRUCT *)NULL)
        {
            // No Match!
            break;
        }

        // We don't have any exceptions yet
        psObj->sTimesCtrl.un16NumExceptions = 0;

        // Is REXCEPT present?
        bFieldPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
                hPayload, &bFieldPresent, 0,
                MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            break;
        }

        if (bFieldPresent == TRUE)
        {
            BOOLEAN bRead;
            MOVIE_TIMES_RATINGS_EXCEPTION_STRUCT *psCurException;

            // Read number of rating exceptions from REXCEPT
            bRead = OSAL.bBufferReadBitsToUN16(
                hPayload, 
                &psObj->sTimesCtrl.un16NumExceptions, 
                            MOVIES1_REXCEPT_BITLEN );

            if (bRead != TRUE)
            {
                break;
            }

            // SX-9845-0027 says the rating exception
            // count is actually REXCEPT + 1
            psObj->sTimesCtrl.un16NumExceptions++;

            // Make sure we don't overrun our exceptions array
            if ( MOVIES1_MAX_RATING_EXCEPTIONS < psObj->sTimesCtrl.un16NumExceptions )
            {
                puts(MOVIES1_OBJECT_NAME": un16NumExceptions greater than MAX_RATING_EXCEPTIONS");
                bSuccess = FALSE;
                break;
            }

            // Read all exceptions now
            for (un16Index = 0; 
                 un16Index < psObj->sTimesCtrl.un16NumExceptions; 
                 un16Index++)
            {
                // Grab the current exception data
                psCurException = &psObj->sTimesCtrl.asExceptions[un16Index];

                // Read the movie ID
                psCurException->tMovieID = (MOVIE_ID)0;
                tBitsRead = OSAL.tBufferReadHeadBits (
                    hPayload, &psCurException->tMovieID, 0,
                    MOVIES1_MVID_BITLEN );

                if (tBitsRead != MOVIES1_MVID_BITLEN)
            {
                    bSuccess = FALSE;
                    break;
                }

                // Read the new rating
                psCurException->tRating = (MOVIE_RATING_CODE)0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload, &psCurException->tRating, 0, 
                    MOVIES1_RCODE_BITLEN);

                if ( tBitsRead != MOVIES1_RCODE_BITLEN )
                {
                puts(MOVIES1_OBJECT_NAME": Error while processing REXCEPT fields");
                    bSuccess = FALSE;
                break;
            }
        }
        }

        if (bSuccess == FALSE)
        {
            break;
        }

        bSuccess = OSAL.bBufferReadBitsToUN16(hPayload,
            &un16EPOCH, MOVIES1_EPOCH_BITLEN);

        if (bSuccess == FALSE)
        {
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
            &un8ETIME,0, MOVIES1_ETIME_BITLEN);

        if (tBitsRead != MOVIES1_ETIME_BITLEN)
        {
            break;
        }

        tBitsRead = OSAL.tBufferReadHeadBits(hPayload,
            &un8THIDSIZE, 0, MOVIES1_THIDSIZE_BITLEN);

        if (tBitsRead != MOVIES1_THIDSIZE_BITLEN)
        {
            break;
        }

        // If this is a new version, clear our State Car list
        if (psObj->sTimesCtrl.tCurAccessUnitVersion !=
            psObj->sTimesCtrl.tCurVersion)
        {
            bSuccess = bResetTimesPayloadTracking((MOVIES_INTERFACE_OBJECT)psObj);

            if (bSuccess == FALSE)
            {
                break;
            }

            psObj->sTimesCtrl.tCurVersion =
                psObj->sTimesCtrl.tCurAccessUnitVersion;
        }

        // Search our State Carousel list to see if we have an entry
        // for this AU.
        bFound = bFindStateAUEntry(psObj,
            psObj->sTimesCtrl.tCurAccessUnitStateID, un16AUNUM, TRUE, NULL);

        if (bFound == TRUE)
        {
            //  indicate we already parsed this
            *pbShouldProcess = FALSE;
        }
        else
        {
            *pbShouldProcess = TRUE;
            psObj->sTimesCtrl.un16CurAccessUnitAUTOT = un16AUTOT;
            psObj->sTimesCtrl.un16CurAccessUnitAUNUM = un16AUNUM;
            psObj->sTimesCtrl.un16CurAccessUnitEPOCH = un16EPOCH;
            psObj->sTimesCtrl.un8CurAccessUnitETIME = un8ETIME;
            psObj->sTimesCtrl.un8CurAccessUnitTHIDSIZE = un8THIDSIZE;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bInitDescAUVector
*
*****************************************************************************/
static BOOLEAN bInitDescAUVector (
    MOVIES1_OBJECT_STRUCT *psObj,
    UN16 un16NumAUs
        )
{
    BOOLEAN bSuccess;
    size_t tAUVecSizeNeeded = (un16NumAUs / 8) + 1;

    // Setup the AU vector
    do
    {

        // See if our AU vector will
        // handle this AUTOT value
        if ( tAUVecSizeNeeded > psObj->sDescCtrl.tAUVectorSize)
        {
            // Free our current vector (if it is there)
            if (psObj->sDescCtrl.pun8AUVector != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psObj->sDescCtrl.pun8AUVector);
            }

            psObj->sDescCtrl.pun8AUVector = (UN8 *)
                SMSO_hCreate(MOVIES1_OBJECT_NAME":DescAUVec",
                    tAUVecSizeNeeded,
                    (SMS_OBJECT)psObj, FALSE);

            if (psObj->sDescCtrl.pun8AUVector == NULL)
            {
                break;
            }

            // Save our vector size
            psObj->sDescCtrl.tAUVectorSize = tAUVecSizeNeeded;
        }
        else
        {
            // We have a vector of adequate size or we
            // don't need a vector.
            // Clear our vector if it is there
            if (psObj->sDescCtrl.pun8AUVector != NULL)
            {
                bSuccess =
                    OSAL.bMemSet(psObj->sDescCtrl.pun8AUVector,
                        0, psObj->sDescCtrl.tAUVectorSize);

                if (bSuccess == FALSE)
                {
                    break;
                }
            }
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bFindStateAUEntry
*
*****************************************************************************/
static BOOLEAN bFindStateAUEntry (
    MOVIES1_OBJECT_STRUCT *psObj,
    MOVIE_STATE_ID tStateID,
    UN16 un16AUSeqNum,
    BOOLEAN bMatchAUSeqNum,
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT **ppsEntry
        )
{
    BOOLEAN bFound = FALSE;
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT sEntryToFind;
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;


    sEntryToFind.tID = tStateID;

    eReturnCode = OSAL.eLinkedListSearch(
        psObj->sTimesCtrl.hStateCarouselList,
        &hEntry, &sEntryToFind);

    if (eReturnCode == OSAL_SUCCESS)
    {
        psEntry = (MOVIE_TIMES_STATE_CAROUSEL_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        if (bMatchAUSeqNum == FALSE)
        {
            // We just wanted to find the state entry,
            // so we are done
            bFound = TRUE;
        }
        else
        {
            if ( (un16AUSeqNum < psEntry->un16NumAUs) &&
                 (psEntry->bActive == TRUE)
               )
            {
                bFound = bGetAUNumParsed(&psEntry->pun8AUVector[0],
                    psEntry->un16NumAUs, un16AUSeqNum);
            }
        }
    }

    if ( (bFound == TRUE) && (ppsEntry != NULL) )
    {
        *ppsEntry = psEntry;
    }

    return bFound;
}


/*****************************************************************************
*
*   bMarkTimesAUNumAsParsed
*
*****************************************************************************/
static BOOLEAN bMarkTimesAUNumAsParsed(
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum,
    BOOLEAN *pbCarouselComplete
        )
{
    BOOLEAN bSuccess = FALSE;

    do
    {
        if (psEntry == NULL)
        {
            break;
        }

        if (pbCarouselComplete == NULL)
        {
            break;
        }

        if (psEntry->bActive == FALSE)
        {
            size_t tSizeNeeded = (un16NumAUs / 8) + 1;

            // See if our AU vector will
            // handle this AUTOT value
            if ( tSizeNeeded > psEntry->tAUVectorSize)
            {
                // Free our current vector (if it is there)
                if (psEntry->pun8AUVector != NULL)
                {
                    SMSO_vDestroy((SMS_OBJECT)psEntry->pun8AUVector);
                }

                psEntry->pun8AUVector = (UN8 *)
                    SMSO_hCreate(MOVIES1_OBJECT_NAME":StCarEntry:Vec",
                        tSizeNeeded,
                        (SMS_OBJECT)psEntry, FALSE);

                if (psEntry->pun8AUVector == NULL)
                {
                    break;
                }

                // Save our vector size
                psEntry->tAUVectorSize = tSizeNeeded;
            }
            else
            {
                // We have a vector of adequate size or we
                // don't need a vector.
                // Clear our vector if it is there
                if (psEntry->pun8AUVector != NULL)
                {
                    bSuccess =
                        OSAL.bMemSet(psEntry->pun8AUVector,
                            0, psEntry->tAUVectorSize);

                    if (bSuccess == FALSE)
                    {
                        break;
                    }
                }
            }

            // Now that our entry is ready to be active,
            // make it so
            psEntry->un16NumAUs = un16NumAUs;
            psEntry->bActive = TRUE;
        }

        // Now that we have an active entry, mark the AU Num as parsed
        bSuccess = bSetAUNumAsParsed(&psEntry->pun8AUVector[0],
            psEntry->un16NumAUs, un16AUSeqNum, pbCarouselComplete);

    } while (FALSE);

    return bSuccess;

}

/*****************************************************************************
*
*   bGetAUNumParsed
*
*****************************************************************************/
static BOOLEAN bGetAUNumParsed(
    UN8 *pun8AUList,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum
        )
{
    BOOLEAN bFound = FALSE;

    if (pun8AUList != NULL)
    {
        // We have a vector, so check to see if the
        // bit for this AUNUM is set in the vector
        UN8 un8Byte = (UN8)(un16AUSeqNum / 8);
        UN8 un8Bit = un16AUSeqNum % 8;
        UN8 un8Mask = (1 << un8Bit);
        UN8 *pun8Byte = &pun8AUList[un8Byte];

        if ((*pun8Byte & un8Mask) == un8Mask)
        {
            bFound = TRUE;
        }
    }


    return bFound;
}

/*****************************************************************************
*
*   bSetAUNumAsParsed
*
*****************************************************************************/
static BOOLEAN bSetAUNumAsParsed(
    UN8 *pun8AUList,
    UN16 un16NumAUs,
    UN16 un16AUSeqNum,
    BOOLEAN *pbCarouselComplete
        )
{
    UN8 un8Byte = (UN8)(un16AUSeqNum / 8);
    UN8 un8Bit = un16AUSeqNum % 8;
    UN8 un8Mask = (1 << un8Bit);
    UN16 un16AUIdx;
    BOOLEAN bComplete = TRUE;

    // Set the flag as not complete, just in case the caller
    // didn't initial the value
    *pbCarouselComplete = FALSE;

    // Mark the entry in our vector if we have more than
    // a single AU in this carousel
    pun8AUList[un8Byte] |= un8Mask;

    // Look to see if we have a complete mask
    for (un16AUIdx = 0;
         un16AUIdx < un16NumAUs;
         un16AUIdx++)
    {
        un8Byte = (UN8)(un16AUIdx / 8);
        un8Bit = un16AUIdx % 8;
        un8Mask = (1 << un8Bit);

        if ( (pun8AUList[un8Byte] & un8Mask) != un8Mask)
        {
            bComplete = FALSE;
            break;
        }
    }

    if (bComplete == TRUE)
    {
        // We've seen all the AUs, so the carousel is complete
        *pbCarouselComplete = TRUE;
    }

    return TRUE;
}


/*****************************************************************************
*
*   bResetStateAUEntryIterator
*
*****************************************************************************/
static BOOLEAN bResetStateAUEntryIterator (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry,
    void *pvArg
        )
{
    if (psEntry != NULL)
    {
        psEntry->bActive = FALSE;
        psEntry->un16NumAUs = 0;
    }

    // Continue with the iteration
    return TRUE;
}

/*****************************************************************************
*
*   vReleaseStateCarEntry
*
*****************************************************************************/
static void vReleaseStateCarEntry (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry
        )
{
    if (psEntry != NULL)
    {
        // Reset variables
        psEntry->tID = MOVIE_INVALID_STATE_ID;
        psEntry->un16NumAUs = UN16_MIN;

        // Get rid of the vector if it is present
        if (psEntry->pun8AUVector != NULL)
        {
            SMSO_vDestroy((SMS_OBJECT)psEntry->pun8AUVector);
            psEntry->pun8AUVector = NULL;
            psEntry->tAUVectorSize = 0;
        }

        // Free the whole entry
        SMSO_vDestroy((SMS_OBJECT)psEntry);
    }

    return;
}

/*****************************************************************************
*
*   n16CompareStateCarEntry
*
*****************************************************************************/
static N16 n16CompareStateCarEntry (
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry1,
    MOVIE_TIMES_STATE_CAROUSEL_STRUCT *psEntry2
        )
{
    N16 n16Return = N16_MIN;

    if ( (psEntry1 != NULL) && (psEntry2 != NULL) )
    {
        n16Return = COMPARE(psEntry1->tID, psEntry2->tID);
    }

    return n16Return;
}

/*****************************************************************************
*
*   eRFDFileProcessor
*
*   The callback invoked by RFD when an update file has been fully recieved
*   and is ready for processing
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl = (MOVIES1_RFD_CTRL_STRUCT *)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk, bProcessThisFile = FALSE;

        // Initialize the RFD Ctrl object now
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex);
        if (bOk == TRUE)
        {
            // Process the update header now
            bProcessThisFile = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }
        else
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                MOVIES1_OBJECT_NAME": Unable to process RFD file header");

            return RFD_PROCESS_RESULT_ERROR;
        }

        if (bProcessThisFile == FALSE)
        {
            // Done with this file
            return RFD_PROCESS_RESULT_COMPLETE;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        // Free associated memory if we're stopping
        vDestroyRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion);

    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult  >= RFD_PROCESS_RESULT_ERROR)
        {
            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    BOOLEAN bDataTransferred, bShouldProcess = FALSE;
    UN8 un8UFILE = 0;
    PVN tPVN = (PVN)0;
    size_t tBitsRead;

    do
    {
        // No entries yet for this update
        psRFDCtrl->un8NumEntriesInUpdate = 0;

        // Fill the buffer with file data
        bDataTransferred = DATASERVICE_MGR_bFillBufferBlock(
            psFile, psRFDCtrl->hBuffer);
        if (bDataTransferred == FALSE)
        {
            // Couldn't access the file
            break;
        }

        // Parse File Header, read PVN
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &tPVN, 0,
            MOVIES1_PVN_BITLEN);
        if ((tBitsRead != MOVIES1_PVN_BITLEN) || 
            (tPVN != MOVIES1_PVN))
        {
            SMSAPI_DEBUG_vPrint(MOVIES1_OBJECT_NAME, 2, 
                "Unavailable or incorrect PVN number (bits:%d, PVN=%d)", tBitsRead, tPVN);
            break;
        }

        // Read Update File Type
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8UFILE, 0,
            MOVIES1_UFILE_BITLEN);
        if (tBitsRead != MOVIES1_UFILE_BITLEN)
        {
            SMSAPI_DEBUG_vPrint(MOVIES1_OBJECT_NAME, 2,
                "Unavailable update file type (bits:%d)", tBitsRead);
            break;
        }

        // Save the file type
        psRFDCtrl->eFileType = (MOVIE1_UFILE_TYPE_ENUM)un8UFILE;

        // Do we need to process the ratings table?
        if ((psRFDCtrl->eFileType == MOVIE1_UFILE_TYPE_ONLY_RATINGS) ||
            (psRFDCtrl->eFileType == MOVIE1_UFILE_TYPE_THEATERS_AND_RATINGS))
        {
            BOOLEAN bOk;

            // Yep, process them now
            bOk = bProcessRatings(psRFDCtrl, psFile);

            if (bOk == FALSE)
            {
                break;
            }
        }

        // Do we need to process theaters?
        if ((psRFDCtrl->eFileType == MOVIE1_UFILE_TYPE_ONLY_THEATERS) ||
            (psRFDCtrl->eFileType == MOVIE1_UFILE_TYPE_THEATERS_AND_RATINGS))
        {
            // Seek past the theater version (we don't use it)
            tBitsRead = OSAL.tBufferSeekHeadBits(
                psRFDCtrl->hBuffer, MOVIES1_THVER_BITLEN);
            if (tBitsRead != MOVIES1_THVER_BITLEN)
            {
                SMSAPI_DEBUG_vPrint(MOVIES1_OBJECT_NAME, 2,
                    "Unavailable theaters version (bits:%d)", tBitsRead);
                break;
            }

            // Read the number of states, store it
            // as the number of entries in this update
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer,
                &psRFDCtrl->un8NumEntriesInUpdate, 0,
                MOVIES1_NSTATE_BITLEN);
            if (tBitsRead != MOVIES1_NSTATE_BITLEN)
            {
                SMSAPI_DEBUG_vPrint(MOVIES1_OBJECT_NAME, 2,
                    "Unavailable number of states (bits:%d)", tBitsRead);
                break;
            }

            // Section 7.4.2 states that NSTATE is NSTATE+1
            psRFDCtrl->un8NumEntriesInUpdate++;
        }

        bShouldProcess = TRUE;
    } while (FALSE);

    return bShouldProcess;
}

/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    BOOLEAN bSuccess = TRUE;

    // Do we have any work left?
    if (psRFDCtrl->un8NumEntriesInUpdate != psRFDCtrl->tCurProgressIndex)
    {
        bSuccess = bProcessState(psRFDCtrl, psFile);

        if (bSuccess == TRUE)
        {
            // Increment our progress index
            psRFDCtrl->tCurProgressIndex++;

            if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
            {
                // Report our progress
                bSuccess = RFD_INTERFACE_bReportProgress(
                    psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex);
            }
        }
    }

    if (bSuccess == TRUE)
    {
        // Are we done?
        if (psRFDCtrl->un8NumEntriesInUpdate == psRFDCtrl->tCurProgressIndex)
        {
            // Have the manager finalize the DB
            bSuccess = MOVIES_MGR_bEndDBUpdate(
                psRFDCtrl->hDBConnection,
                &psRFDCtrl->acSQLCommandBuffer[0],
                sizeof(psRFDCtrl->acSQLCommandBuffer),
                (MOVIE_VERSION)tUpdateVersion);

            if (bSuccess == TRUE)
            {
                // Yes -- tell the caller
                eResult = RFD_PROCESS_RESULT_COMPLETE;
            }
        }
    }

    if (bSuccess == FALSE)
    {
        eResult = RFD_PROCESS_RESULT_ERROR;
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessRatings
*
*****************************************************************************/
static BOOLEAN bProcessRatings (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    BOOLEAN bSuccess = FALSE,
            bInTransaction = FALSE;

    // We only need to process the ratings table
    // if our progress index is 0.  Anything
    // higher than that means we've already processed it
    // and we've moved on to the theater list
    if (psRFDCtrl->tCurProgressIndex > 0)
    {
        return TRUE;
    }

    do
    {
        size_t tBitsRead;
        BOOLEAN bOk, bProcessRatings, bDone = FALSE;
        MOVIE_RATINGS_ROW_STRUCT sRatings;
        MOVIE1_UTYPE_ENUM eType;

        // Seek past the RSYSVER now
        tBitsRead =  OSAL.tBufferSeekHeadBits(
            psRFDCtrl->hBuffer, MOVIES1_RSYSVER_BITLEN);
        if (tBitsRead != MOVIES1_RSYSVER_BITLEN)
        {
            break;
        }

        // Begin a transaction now
        bInTransaction = SQL_INTERFACE.bStartTransaction(psRFDCtrl->hDBConnection);
        if (bInTransaction == FALSE)
        {
            break;
        }

        // Clear the ratings table now
        bOk = MOVIES_MGR_bClearRatings(psRFDCtrl->hDBConnection);

        if (bOk == FALSE)
        {
            break;
        }

        while (bDone == FALSE)
        {
            bProcessRatings = FALSE;

            // Read the UTYPE to find out what we have to do
            eType = (MOVIE1_UTYPE_ENUM)0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &eType, 0,
                MOVIES1_UTYPE_BITLEN);
            if (tBitsRead != MOVIES1_UTYPE_BITLEN)
            {
                break;
            }

            switch (eType)
            {
                case MOVIE1_UTYPE_NEW:
                case MOVIE1_UTYPE_MODIFY:
                {
                    bProcessRatings = TRUE;
                }
                break;

                case MOVIE1_UTYPE_DELETE:
                {
                    // Don't do anything - we have nothing to
                    // delete since we're starting with an empty
                    // ratings table
                }
                break;

                case MOVIE1_UTYPE_END_OF_LIST:
                {
                    bDone = TRUE;
                }
                break;

                default:
                {
                    bOk = FALSE;
                    bDone = TRUE;
                }
            }

            if (bDone == FALSE)
            {
                // Read SCODE (present for every utype except EOL)
                sRatings.tSysCode = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer, &sRatings.tSysCode, 0,
                    MOVIES1_SCODE_BITLEN);
                if (tBitsRead != MOVIES1_SCODE_BITLEN)
                {
                    bOk = FALSE;
                    break;
                }

                if (bProcessRatings == TRUE)
                {
                    bOk = bProcessRatingsSystem(psRFDCtrl, &sRatings, psFile);
                }
            }

            // Keep the buffer block populated
            DATASERVICE_MGR_bFillBufferBlock(
                psFile, psRFDCtrl->hBuffer);
        }

        if (bOk == FALSE)
        {
            break;
        }

        bSuccess = TRUE;

    } while (FALSE);

    if (bInTransaction == TRUE)
    {
        SQL_INTERFACE.bEndTransaction(psRFDCtrl->hDBConnection);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessRatingsSystem
*
*****************************************************************************/
static BOOLEAN bProcessRatingsSystem (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    MOVIE_RATINGS_ROW_STRUCT *psRatingsRow,
    FILE *psFile
        )
{
    BOOLEAN bSuccess = TRUE;
    size_t tBitsRead;
    BOOLEAN bRatingsPresent;

    // Read the ratings sytem name now
    psRatingsRow->hSysName = BAUDOT_hToString(
        SMS_INVALID_OBJECT, psRFDCtrl->hBuffer,
        BAUDOT_BEHAVIOR_PROCESS_TO_END, TRUE, TRUE,
        MOVIES1_MAX_STEXT_SYMBOLS, (size_t *)NULL);
    if (psRatingsRow->hSysName == STRING_INVALID_OBJECT)
    {
        return FALSE;
    }

    do
    {
        // Are there any more ratings present?
        bRatingsPresent = FALSE;
        tBitsRead = OSAL.tBufferReadHeadBits (
            psRFDCtrl->hBuffer, &bRatingsPresent, 0,
            MOVIES1_PRESENT_FLAG_BITLEN );

        if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
        {
            bSuccess = FALSE;
            break;
        }

        if (bRatingsPresent == TRUE)
        {
            psRatingsRow->tRatingCode = 0;
                tBitsRead = OSAL.tBufferReadHeadBits (
                psRFDCtrl->hBuffer,
                &psRatingsRow->tRatingCode, 0,
                MOVIES1_RCODE_BITLEN );

            if (tBitsRead != MOVIES1_RCODE_BITLEN)
            {
                bSuccess = FALSE;
                break;
            }

            // Read the ratings sytem name now
            psRatingsRow->hRatingText = BAUDOT_hToString(
                SMS_INVALID_OBJECT, psRFDCtrl->hBuffer,
                BAUDOT_BEHAVIOR_PROCESS_TO_END, TRUE, TRUE,
                MOVIES1_MAX_RTEXT_SYMBOLS, (size_t *)NULL);
            if (psRatingsRow->hRatingText == STRING_INVALID_OBJECT)
            {
                bSuccess = FALSE;
                break;
            }

            // Tell the manager to insert this line into the database
            bSuccess = MOVIES_MGR_bInsertRatings(
                psRFDCtrl->hDBConnection,
                psRatingsRow);

            // Whatever happened there we're done with this string
            STRING_vDestroy(psRatingsRow->hRatingText);
            psRatingsRow->hRatingText = STRING_INVALID_OBJECT;

            if (bSuccess == FALSE)
            {
                break;
            }
        }
    } while (bRatingsPresent == TRUE);

    // Clean up any memory left behind
    if (psRatingsRow->hSysName != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRatingsRow->hSysName);
        psRatingsRow->hSysName = STRING_INVALID_OBJECT;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessState
*
*****************************************************************************/
static BOOLEAN bProcessState (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    BOOLEAN bOk = FALSE, bInformManager = TRUE;
    UN8 un8State = 0, un8THSIZE = 0;
    size_t tBitsRead;

    // Did we already process this region?
    if (psRFDCtrl->tCurProgressIndex < psRFDCtrl->tStartProgressIndex)
    {
        bInformManager = FALSE;
    }

    // Fill the buffer with file data
    DATASERVICE_MGR_bFillBufferBlock(psFile, psRFDCtrl->hBuffer);

    do
    {
        // Read State Code
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8State, 0,
            MOVIES1_STATE_BITLEN);
        if (tBitsRead != MOVIES1_STATE_BITLEN)
        {
            break;
        }

        // Read THSIZE
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8THSIZE, 0,
            MOVIES1_THSIZE_BITLEN);
        if (tBitsRead != MOVIES1_THSIZE_BITLEN)
        {
            break;
        }

        // Read the theaters for this state
        bOk = bProcessTheatersForState(
            psRFDCtrl, bInformManager, psFile,
            un8State, un8THSIZE, psRFDCtrl->tNewVersion );

    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bProcessTheatersForState
*
*****************************************************************************/
static BOOLEAN bProcessTheatersForState (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FILE *psFile,
    UN8 un8State,
    UN8 un8THSIZE,
    RFD_UPDATE_VERSION tVersion
        )
{
    UN8 un8UTYPE = 0, un8ZTYPE;
    UN16 un16THID;
    THEATER_ID tTHID;
    BOOLEAN bOk = TRUE;
    BOOLEAN bAmenitiesUpdated = FALSE;
    THEATER_ROW_STRUCT sTheaterRow;
    UN32 un32THLAT, un32THLON;
    size_t tBitsRead;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    OSAL.bMemSet(&sTheaterRow, 0, sizeof(sTheaterRow));
    sTheaterRow.tStateID = (STATE_ID)un8State;

    if (bInformManager == TRUE)
    {
        bOk = SQL_INTERFACE.bStartTransaction(psRFDCtrl->hDBConnection);
    }

    do
    {
        DATASERVICE_MGR_bFillBufferBlock(
            psFile, psRFDCtrl->hBuffer);

        // Read the UTYPE
        un8UTYPE = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8UTYPE, 0,
            MOVIES1_UTYPE_BITLEN);
        if (tBitsRead != MOVIES1_UTYPE_BITLEN)
        {
            bOk = FALSE;
            break;
        }

        if ((MOVIE1_UTYPE_ENUM)un8UTYPE != MOVIE1_UTYPE_END_OF_LIST)
        {
            un16THID = 0;
            bOk = OSAL.bBufferReadBitsToUN16 (
                psRFDCtrl->hBuffer, &un16THID, un8THSIZE );

            if (bOk == FALSE)
            {
                break;
            }

            tTHID = MAKE_UNIQUE_THID(un8State, un16THID);

            if ((MOVIE1_UTYPE_ENUM)un8UTYPE == MOVIE1_UTYPE_DELETE)
            {
                if (bInformManager == TRUE)
                {
                    // Remove the Theater
                    bOk = MOVIES_MGR_bRemoveTheater(
                        psRFDCtrl->hDBConnection,
                        &psRFDCtrl->acSQLCommandBuffer[0],
                        sizeof(psRFDCtrl->acSQLCommandBuffer),
                        tTHID );
                }
            }
            else
            {
                // Set the theater ID
                sTheaterRow.tID = tTHID;

                // Read the Theater Name
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    sTheaterRow.hName =
                        BAUDOT_hToString(
                            SMS_INVALID_OBJECT,
                            psRFDCtrl->hBuffer,
                            BAUDOT_BEHAVIOR_PROCESS_TO_END,
                            TRUE, TRUE, 0, NULL);

                    if (sTheaterRow.hName == STRING_INVALID_OBJECT)
                    {
                        bOk = FALSE;
                        break;
                    }
                }
                else
                {
                    sTheaterRow.hName = STRING_INVALID_OBJECT;
                }

                // Read the Theater Address
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    sTheaterRow.hAddr =
                        BAUDOT_hToString(
                            SMS_INVALID_OBJECT,
                            psRFDCtrl->hBuffer,
                            BAUDOT_BEHAVIOR_PROCESS_TO_END,
                            FALSE, TRUE, 0, NULL);

                    if (sTheaterRow.hAddr == STRING_INVALID_OBJECT)
                    {
                        bOk = FALSE;
                        break;
                    }
                }
                else
                {
                    sTheaterRow.hAddr = STRING_INVALID_OBJECT;
                }

                // Read the City
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    sTheaterRow.hCity =
                        BAUDOT_hToString(
                            SMS_INVALID_OBJECT,
                            psRFDCtrl->hBuffer,
                            BAUDOT_BEHAVIOR_PROCESS_TO_END,
                            TRUE, TRUE, 0, NULL);

                    if (sTheaterRow.hCity == STRING_INVALID_OBJECT)
                    {
                        bOk = FALSE;
                        break;
                    }
                }
                else
                {
                    sTheaterRow.hCity = STRING_INVALID_OBJECT;
                }

                // Read Lattitude
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    un32THLAT = 0;
                    bOk = OSAL.bBufferReadBitsToUN32 (
                        psRFDCtrl->hBuffer, &un32THLAT, MOVIES1_THLAT_BITLEN );

                    if (bOk == FALSE)
                    {
                        break;
                    }

                    sTheaterRow.hLat =
                        OSAL_FIXED.hCreateInMemory((N32)un32THLAT, 0,
                            &sTheaterRow.atLatFixedData[0]);

                    eReturnCode = OSAL_FIXED.eDivide(
                        sTheaterRow.hLat, psRFDCtrl->hDegDivsor, sTheaterRow.hLat);

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        bOk = FALSE;
                        break;
                    }
                }
                else
                {
                    sTheaterRow.hLat = OSAL_FIXED_INVALID_OBJECT;
                }

                // Read Longitude
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    un32THLON = 0;
                    bOk = OSAL.bBufferReadBitsToUN32 (
                        psRFDCtrl->hBuffer, &un32THLON, MOVIES1_THLON_BITLEN );

                    if (bOk == FALSE)
                    {
                        break;
                    }

                    sTheaterRow.hLon =
                        OSAL_FIXED.hCreateInMemory((N32)un32THLON, 0,
                            &sTheaterRow.atLonFixedData[0]);

                    eReturnCode = OSAL_FIXED.eDivide(
                        sTheaterRow.hLon, psRFDCtrl->hDegDivsor, sTheaterRow.hLon );

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        bOk = FALSE;
                        break;
                    }

                    eReturnCode = OSAL_FIXED.eSubtract(
                        psRFDCtrl->hLonBase, sTheaterRow.hLon, sTheaterRow.hLon );

                    if (eReturnCode != OSAL_SUCCESS)
                    {
                        bOk = FALSE;
                        break;
                    }
                }
                else
                {
                    sTheaterRow.hLon = OSAL_FIXED_INVALID_OBJECT;
                }

                // Read ZIP Code
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    // First read the zip type
                    un8ZTYPE = 0;
                    tBitsRead = OSAL.tBufferReadHeadBits(
                        psRFDCtrl->hBuffer, &un8ZTYPE, 0,
                        MOVIES1_ZTYPE_BITLEN);

                    if (tBitsRead != MOVIES1_ZTYPE_BITLEN)
                    {
                        bOk = FALSE;
                        break;
                    }

                    if (un8ZTYPE == MOVIES1_ZTYPE_INTEGER)
                    {
                        UN32 un32Zip = 0;
                        char acZip[MOVIES1_MAX_ZIP_INT_CHARS + 1];

                        // Read the zip code as an integer
                        bOk = OSAL.bBufferReadBitsToUN32(
                            psRFDCtrl->hBuffer, &un32Zip,
                            MOVIES1_ZIP_INTEGER_BITLEN);
                        if (bOk == FALSE)
                        {
                            break;
                        }

                        // Create a string for the zip code
                        snprintf(&acZip[0], sizeof(acZip), "%u", un32Zip);
                        sTheaterRow.hZIP = STRING.hCreate(&acZip[0], 0);
                    }
                    else // Read ZIP as a string (baudot)
                    {
                        sTheaterRow.hZIP =
                            BAUDOT_hToString(
                                SMS_INVALID_OBJECT,
                                psRFDCtrl->hBuffer,
                                BAUDOT_BEHAVIOR_PROCESS_TO_END,
                                TRUE, TRUE, 0, NULL);
                    }
                }
                else
                {
                    sTheaterRow.hZIP = STRING_INVALID_OBJECT;
                }

                // Read Phone
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    UN16 un16AreaCode = 0,
                         un16Exchange = 0,
                         un16Number = 0;
                    char acPhone[MOVIES1_MAX_PHONE_CHARS + 1];
                    OSAL.bBufferReadBitsToUN16(
                        psRFDCtrl->hBuffer, &un16AreaCode,
                        MOVIES1_PHONE_AREA_BITLEN);
                    OSAL.bBufferReadBitsToUN16(
                        psRFDCtrl->hBuffer, &un16Exchange,
                        MOVIES1_PHONE_EXCH_BITLEN);
                    OSAL.bBufferReadBitsToUN16(
                        psRFDCtrl->hBuffer, &un16Number,
                        MOVIES1_PHONE_NUM_BITLEN);

                    // Create a string for the phone number
                    snprintf(&acPhone[0], sizeof(acPhone), "(%03u)%03u-%04u",
                        un16AreaCode, un16Exchange, un16Number );
                    sTheaterRow.hPhone = STRING.hCreate(&acPhone[0], 0);
                }
                else
                {
                    sTheaterRow.hPhone = STRING_INVALID_OBJECT;
                }

                // Read Amenities 1
                sTheaterRow.un16Amenities = 0;
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    bOk = bProcessAmenities(
                        psRFDCtrl, TRUE, &sTheaterRow.un16Amenities);
                    if (bOk == FALSE)
                    {
                        return FALSE;
                    }

                    bAmenitiesUpdated = TRUE;
                }

                // Read Amenities 2
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    bOk = bProcessAmenities(
                        psRFDCtrl, FALSE, &sTheaterRow.un16Amenities);
                    if (bOk == FALSE)
                    {
                        return FALSE;
                    }

                    bAmenitiesUpdated = TRUE;
                }

                // Read Extension
                if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
                {
                    UN8 un8ExtensionSize = 0;

                    // Read the extension size
                    tBitsRead = OSAL.tBufferReadHeadBits(
                        psRFDCtrl->hBuffer, &un8ExtensionSize, 0,
                        MOVIES1_EXTENSION_SIZE_BITLEN);
                    if (tBitsRead != MOVIES1_EXTENSION_SIZE_BITLEN)
                    {
                        bOk = FALSE;
                        break;
                    }

                    // Seek past the extension now
                    tBitsRead = OSAL.tBufferSeekHead(
                        psRFDCtrl->hBuffer, un8ExtensionSize);

                    if (tBitsRead != un8ExtensionSize)
                    {
                        bOk = FALSE;
                        break;
                    }
                }

                if (bInformManager == TRUE)
                {
                    // Add or update the theater
                    if ((MOVIE1_UTYPE_ENUM)un8UTYPE == MOVIE1_UTYPE_NEW)
                    {
                        bOk = MOVIES_MGR_bInsertTheater(
                            psRFDCtrl->hDBConnection,
                            &psRFDCtrl->acSQLCommandBuffer[0],
                            sizeof(psRFDCtrl->acSQLCommandBuffer),
                            &sTheaterRow );
                    }
                    else
                    {
                        bOk = MOVIES_MGR_bUpdateTheater(
                            psRFDCtrl->hDBConnection,
                            &psRFDCtrl->acSQLCommandBuffer[0],
                            sizeof(psRFDCtrl->acSQLCommandBuffer),
                            &sTheaterRow, bAmenitiesUpdated );
                    }
                }
                else
                {
                    bOk = TRUE;
                }
            }
        }

    } while ((MOVIE1_UTYPE_ENUM)un8UTYPE != MOVIE1_UTYPE_END_OF_LIST && bOk == TRUE);

    if (bInformManager == TRUE)
    {
        SQL_INTERFACE.bEndTransaction(psRFDCtrl->hDBConnection);
    }

    if (bOk == FALSE)
    {
        // Free any strings created
        if (sTheaterRow.hName != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sTheaterRow.hName);
        }

        if (sTheaterRow.hAddr != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sTheaterRow.hAddr);
        }

        if (sTheaterRow.hCity != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sTheaterRow.hCity);
        }

        if (sTheaterRow.hZIP != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sTheaterRow.hZIP);
        }

        if (sTheaterRow.hPhone != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(sTheaterRow.hPhone);
        }
    }

    return bOk;
}

/*****************************************************************************
*
*   bProcessAmenities
*
*   Process the amenities entry for a theater
*
*****************************************************************************/
static BOOLEAN bProcessAmenities (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN16 *pun16Amenities
        )
{
    size_t tBitsRead = 0,
           tBitOffset = 0;
    UN8 un8CurrentAmen,
        un8RawAmenValue;
    UN16 un16Temp;

    // Are we working on the first amenities field?
    if (bAmenities1 == FALSE)
    {
        // No, make sure we don't overwrite amen1 with
        // these values from amen2
        tBitOffset = MOVIES_AMENITIES_ARRAY_SIZE * MOVIES1_AMEN_FIELD_BITLEN;
    }

    for (un8CurrentAmen = 0;
         un8CurrentAmen < MOVIES_AMENITIES_ARRAY_SIZE;
         un8CurrentAmen++)
    {
        // Read two bits per field
        un8RawAmenValue = 0;
        tBitsRead += OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8RawAmenValue, 0,
            MOVIES1_AMEN_FIELD_BITLEN);

        // Take that value
        un16Temp = (UN16)un8RawAmenValue;

        // Place it at the correct bit position
        un16Temp <<= tBitOffset;

        // Save the result
        *pun16Amenities |= un16Temp;

        // Update the bit position
        tBitOffset += MOVIES1_AMEN_FIELD_BITLEN;
    }

    // Make sure we read it all
    if (tBitsRead != (MOVIES_AMENITIES_ARRAY_SIZE * MOVIES1_AMEN_FIELD_BITLEN))
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bFieldPresent
*
*   A simple function which reads 1 bits and reports TRUE or FALSE
*   based on that bit's value.
*
*****************************************************************************/
static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        )
{
    BOOLEAN bPresent = FALSE;
    size_t tBitsRead;

    // Read the presence flag
    tBitsRead = OSAL.tBufferReadHeadBits(
        hBuffer, &bPresent, 0, MOVIES1_PRESENT_FLAG_BITLEN);

    if (tBitsRead != MOVIES1_PRESENT_FLAG_BITLEN)
    {
        // Indicate presence if we can't read the buffer
        // to force the caller to fail as well
        return TRUE;
    }

    return bPresent;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection =
            SQL_INTERFACE.hConnect(
                pcDBPath, tOptions,
                (DATASERVICE_ERROR_CODE_ENUM *)NULL);

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
   MOVIES1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}
