/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the PVN 1 specific EV charging data service
 *  for the Sirius Module Services (SMS)
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"

#include <string.h>
#include <stdio.h>

#include "sms_api.h"
#include "sms_obj.h"
#include "baudot.h"
#include "string_obj.h"
#include "location_obj.h"
#include "rfd_interface_obj.h"
#include "dataservice_mgr_obj.h"
#include "fuel_db_constants.h"
#include "sql_interface_obj.h"
#include "db_util.h"
#include "ds_util.h"
#include "fuel_interface.h"
#include "fuel_mgr_obj.h"
#include "_evcharging_pvn1.h"

#include "sms_api_debug.h"
static const char *gpacThisFile = __FILE__;

/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   hInit
*
*   Creates and initializes the PVN0 fuel interface object.
*
*   Inputs:
*       hFuelService - A valid handle to a FUEL_SERVICE_OBJECT representing
*           the central fuel service manager.
*
*   Output:
*       An object handle representing this instantiation of the
*       fuel1 interface (representing high-band support)
*
*****************************************************************************/
static FUEL_OTA_INTERFACE_OBJECT hInit(
    FUEL_SERVICE_OBJECT hFuelService,
    SMS_OBJECT hParent,
    FUEL_DB_INTERFACE_OBJECT hDBInterface,
    const FUEL_DB_INTERFACE_STRUCT *psDBInterface,
    const FUEL_OTA_INTERFACE_STRUCT *psOTAInterface
        )
{
    BOOLEAN bOwner;

    // Verify we own service handle
    bOwner = SMSO_bOwner((SMS_OBJECT)hParent);
    if (bOwner == TRUE)
    {
        EV1_OBJECT_STRUCT *psObj;

        // Create an instance of this object
        psObj = (EV1_OBJECT_STRUCT *)
            SMSO_hCreate(
                EV1_OBJECT_NAME,
                sizeof(EV1_OBJECT_STRUCT),
                hParent, FALSE );

        if (psObj == NULL)
        {
            return FUEL_OTA_INTERFACE_INVALID_OBJECT;
        }

        // Initialize logo tracking
        vInitLogoTracking(&psObj->sLogoCtrl, FALSE);

        do
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Save the service handles
            psObj->hFuelService = hFuelService;
            psObj->hDBInterface = hDBInterface;

            // Request access to ISO 3309 CRC32
            eReturnCode = OSAL.eGetCRC(
                &psObj->hCRC, OSAL_CRC_TYPE_ISO3309_CRC32 );

            if (eReturnCode != OSAL_SUCCESS)
            {
                // Error! destroy object
                break;
            }

            // Ensure this is cleared
            psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;

            // Logo updates aren't enabled yet
            psObj->sLogoCtrl.bEnabled = FALSE;

            return (FUEL_OTA_INTERFACE_OBJECT)psObj;

        } while (0);

        vUnInit((FUEL_OTA_INTERFACE_OBJECT)psObj);
    }

    return FUEL_OTA_INTERFACE_INVALID_OBJECT;
}

/*****************************************************************************
*
*   vUnInit
*
*   Uninitializes and destroys the PVN 1 fuel price interface object.
*   This API must only be called when already in the context of the
*   fuel price manager.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*
*****************************************************************************/
static void vUnInit (
    FUEL_OTA_INTERFACE_OBJECT hInterface
        )
{
    BOOLEAN bOwner;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        EV1_OBJECT_STRUCT *psObj =
            (EV1_OBJECT_STRUCT *)hInterface;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Destroy the CRC object if it exists
        if (psObj->hCRC != OSAL_INVALID_OBJECT_HDL)
        {
            // Release our hold of the CRC computation handle
            eReturnCode = OSAL.eReleaseCRC( psObj->hCRC );

            // Only finish clean up if that succeeded.  This allows
            // us to more easily track this kind of issue
            if (eReturnCode == OSAL_SUCCESS)
            {
                psObj->hCRC = OSAL_INVALID_OBJECT_HDL;
            }
        }

        SMSO_vDestroy((SMS_OBJECT)psObj);
    }

    return;
}

/*****************************************************************************
*
*   bProductStateChange
*
*****************************************************************************/
static BOOLEAN bProductStateChange (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bOwner, bSuccess = TRUE;
    EV1_OBJECT_STRUCT *psObj =
        (EV1_OBJECT_STRUCT *)hInterface;

    // Ensure we are the interface owner
    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if ((FALSE == bOwner) ||
        (DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *)NULL == psState)
    {
        return FALSE;
    }

    switch (psState->eType)
    {
        case DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES:
        {
            // Let the logo FSM deal with it
            bSuccess = bLogoProductStateChange(psObj, psState);
        }
        break;

        case DATA_PRODUCT_TYPE_DB_UPDATES:
        {
            // Let the RFD FSM deal with it
            bSuccess = bRFDProductStateChange(psObj, psState);
        }
        break;

        case DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES:
        {
            // Let the Availability FSM deal with it
            bSuccess = bAvailabilityProductStateChange(psObj, psState);
        }
        break;

        default:
        {
            // Don't know how to handle this
            bSuccess = FALSE;
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bLogoProductStateChange
*
*   Implements state FSM for the logo product
*
*****************************************************************************/
static BOOLEAN bLogoProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bActionNeeded = FALSE, bEnabled = FALSE;

    switch (psState->ePreviousState)
    {
        case DATA_PRODUCT_STATE_INITIAL:
        {
            // Only perform an action when moving to READY
            if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
                bEnabled =  TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
            else if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
                bEnabled =  TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_DISABLED:
        default:
        {
            // Nothing to do
        }
        break;
    }

    if (TRUE == bActionNeeded)
    {
        // Initialize logo tracking under the right circumstances
        vInitLogoTracking(&psObj->sLogoCtrl, bEnabled);
    }

    return TRUE;
}

/*****************************************************************************
*
*   bRFDProductStateChange
*
*   Implements state FSM for the RFD product
*
*****************************************************************************/
static BOOLEAN bRFDProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bSuccess = TRUE, bActionNeeded = FALSE;

    switch (psState->ePreviousState)
    {
        case DATA_PRODUCT_STATE_INITIAL:
        {
            // Only perform an action when moving to READY
            if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            if (DATA_PRODUCT_STATE_READY == psState->eState ||
                DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_DISABLED:
        default:
        {
            // Nothing to do
        }
        break;
    }

    if (TRUE == bActionNeeded)
    {
        if (DATA_PRODUCT_STATE_READY == psState->eState)
        {
            // Enable RFD updates when we enter the READY
            // state under the right circumstances
            bSuccess = bEnableRFDUpdates(psObj);
        }
        else if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
        {
            // Disable RFD updates under the right circumstances
            vDisableRFDUpdates(psObj);
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bAvailabilityProductStateChange
*
*   Implements state FSM for the availability product
*
*****************************************************************************/
static BOOLEAN bAvailabilityProductStateChange (
    EV1_OBJECT_STRUCT *psObj,
    DATASERVICE_PRODUCT_STATE_EVENT_ARG_STRUCT *psState
        )
{
    BOOLEAN bSuccess = TRUE, bActionNeeded = FALSE;

    switch (psState->ePreviousState)
    {
        case DATA_PRODUCT_STATE_INITIAL:
        {
            // Only perform an action when moving to READY
            if (DATA_PRODUCT_STATE_READY == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_ERROR:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_READY:
        {
            // Only perform an action when moving to DISABLED
            if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_UNSUBSCRIBED:
        {
            if (DATA_PRODUCT_STATE_READY == psState->eState ||
                DATA_PRODUCT_STATE_DISABLED == psState->eState)
            {
                bActionNeeded = TRUE;
            }
        }
        break;

        case DATA_PRODUCT_STATE_DISABLED:
        default:
        {
            // Nothing to do
        }
        break;
    }

    if (TRUE == bActionNeeded)
    {
        if (DATA_PRODUCT_STATE_READY == psState->eState)
        {
            // Enable RFD updates when we enter the READY
            // state under the right circumstances
            bSuccess = bEnableAvailabilityUpdates(psObj);
        }
        else if (DATA_PRODUCT_STATE_DISABLED == psState->eState)
        {
            // Disable RFD updates under the right circumstances
            vDisableAvailabilityUpdates(psObj);
        }
    }
  
    return bSuccess;
}

/*****************************************************************************
*
*   bGetDSIForProduct
*
*   Provides the caller with the mapping of products/DSIs used by this 
*   interface
*
*****************************************************************************/
static BOOLEAN bGetDSIForProduct (
    DATA_PRODUCT_TYPE_ENUM eType,
    DATA_PRODUCT_MASK tMask,
    DATASERVICE_DSI_INFO_STRUCT *psDSIInfo
        )
{
    BOOLEAN bSuccess = TRUE;

    switch (eType)
    {
        // Both of these products are on the same DSI
        case DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES:
        case DATA_PRODUCT_TYPE_DB_UPDATES:
        {
            psDSIInfo->bEnableAllDMIs = TRUE;
            psDSIInfo->tDSI = EV1_BASE_DSI;

            psDSIInfo->tSuggestedOTABufferByteSize = 
                EV1_OTA_BUFFER_SIZE_BASE_DSI;
        }
        break;

        // Availability product is all that is 
        // on the "update" DSI
        case DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES:
        {
            psDSIInfo->bEnableAllDMIs = TRUE;
            psDSIInfo->tDSI = EV1_UPDATE_DSI;
            psDSIInfo->tSuggestedOTABufferByteSize = 
                EV1_OTA_BUFFER_SIZE_UPDATE_DSI;
        }
        break;

        default:
        {
            bSuccess = FALSE;
        }
        break;
    }

    return bSuccess;
}

/*****************************************************************************
*
*   eNextProductState
*
*****************************************************************************/
static DATA_PRODUCT_STATE_ENUM eNextProductState (
    DATASERVICE_IMPL_HDL hServiceImpl,
    DATA_PRODUCT_TYPE_ENUM eType,
    DSI tDSI,
    DATASERVICE_STATE_ENUM eDSIState,
    SXM_DMI *pDMIs,
    UN8 un8DMICount
        )
{
    DATA_PRODUCT_STATE_ENUM eNextState = 
        DATA_PRODUCT_STATE_ERROR;

    switch (eType)
    {
        // Logo product
        case DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES:
        {
            switch (eDSIState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    eNextState = DATA_PRODUCT_STATE_INITIAL;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    // Both of these mean no logo updates!
                    eNextState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    eNextState = DATA_PRODUCT_STATE_READY;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    eNextState = DATA_PRODUCT_STATE_DISABLED;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:
                {
                    eNextState = DATA_PRODUCT_STATE_ERROR;
                }
                break;
            }
        }
        break;

        // RFD Product
        case DATA_PRODUCT_TYPE_DB_UPDATES:
        {
            switch (eDSIState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    eNextState = DATA_PRODUCT_STATE_INITIAL;
                }
                break;

                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    eNextState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_READY:
                {
                    // Both of these mean RFD updates are active
                    eNextState = DATA_PRODUCT_STATE_READY;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    eNextState = DATA_PRODUCT_STATE_DISABLED;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:
                {
                    eNextState = DATA_PRODUCT_STATE_ERROR;
                }
                break;
            }
        }
        break;

        // Availability product
        case DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES:
        {
            switch (eDSIState)
            {
                case DATASERVICE_STATE_INITIAL:
                {
                    eNextState = DATA_PRODUCT_STATE_INITIAL;
                }
                break;

                case DATASERVICE_STATE_POI_UPDATES_ONLY:
                case DATASERVICE_STATE_UNSUBSCRIBED:
                {
                    eNextState = DATA_PRODUCT_STATE_UNSUBSCRIBED;
                }
                break;

                case DATASERVICE_STATE_READY:
                {
                    // Both of these mean RFD updates are active
                    eNextState = DATA_PRODUCT_STATE_READY;
                }
                break;

                case DATASERVICE_STATE_STOPPED:
                {
                    eNextState = DATA_PRODUCT_STATE_DISABLED;
                }
                break;

                case DATASERVICE_STATE_UNAVAILABLE:
                case DATASERVICE_STATE_ERROR:
                case DATASERVICE_STATE_INVALID:
                {
                    eNextState = DATA_PRODUCT_STATE_ERROR;
                }
                break;
            }
        }
        break;

        default:
        {
            // Nothing to do
        }
        break;
    }

    return eNextState;
}

/*****************************************************************************
*
*   bProcessMessage
*
*   The entry point for the interface's OTA message processing.  All messages
*   received by the Fuel manager come through this API.
*
*   Inputs:
*       hInterface - The interface object handle created with a
*           previous call to hInit.
*       hPayload - A valid OSAL_BUFFER_HDL containing the OTA fuel data
*
*   Output:
*       TRUE on success, FALSE on error
*
*****************************************************************************/
static BOOLEAN bProcessMessage (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bOwner, bSuccess = FALSE;

    // Verify inputs
    if ((hInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT) ||
        (phPayload == NULL)
       )
    {
        return FALSE;
    }

    // Verify buffer
    if (*phPayload == OSAL_INVALID_BUFFER_HDL)
    {
        return FALSE;
    }

    bOwner = SMSO_bOwner((SMS_OBJECT)hInterface);
    if (bOwner == TRUE)
    {
        EV1_OBJECT_STRUCT *psObj =
            (EV1_OBJECT_STRUCT *)hInterface;
        BOOLEAN bMessageValid;
        UN8 un8CarouselID = 0;
        PVN tPVN = (PVN)0;
        size_t tBitsRead;
        size_t tBytesRead;
        DSI tDSI = (DSI)0;

        // The DSI is written to the head of the buffer in the DSM for 
        // multi-product services.  So read & validate it.
        tBytesRead = OSAL.tBufferReadHead (
                *phPayload, &tDSI, sizeof(tDSI));

        if (tBytesRead != sizeof(tDSI))
        {
            // Unable to read DSI from payload
            return FALSE;
        }

        if ((tDSI != EV1_BASE_DSI) &&
            (tDSI != EV1_UPDATE_DSI))
        {
            // I don't know who this is for - 
            // why did it get routed to EV?
            return FALSE;
        }

        // Peek at the PVN
        tBitsRead = OSAL.tBufferPeekBits(
            *phPayload, &tPVN, 0,
            EV1_PVN_BITLEN, 0);

        if (tBitsRead != EV1_PVN_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(EV1_OBJECT_NAME" Unable to read PVN");

            return TRUE;
        }

        // Verify the PVN
        if (tPVN != EV1_PVN)
        {
            printf(EV1_OBJECT_NAME" Incorrect PVN - got %u, expected %u\n",
                tPVN, EV1_PVN);

            return TRUE;
        }

        // Peek at the Carousel Id
        tBitsRead = OSAL.tBufferPeekBits(*phPayload, &un8CarouselID, 0,
            EV1_CAROUSEL_BITLEN, EV1_PVN_BITLEN);

        if (tBitsRead != EV1_CAROUSEL_BITLEN)
        {
            // Read failed -- the message is probably
            // garbled, although that is unlikely since
            // the CRC checked out. Don't do anything rash here.
            puts(EV1_OBJECT_NAME" Unable to read Carousel Id\n");

            return TRUE;
        }

        // Is this an availability message?
        if ((EV1_CHARGER_AVAILABILITY_1_CAROUSEL_ID == un8CarouselID) ||
            (EV1_CHARGER_AVAILABILITY_2_CAROUSEL_ID == un8CarouselID))
        {
            BOOLEAN bRegionNeeded;
            
            // Initialize the region attribute
            psObj->sAvailCtrl.tRegion = (FUEL_REGION)0;

            // Peek the region id field
            bRegionNeeded = OSAL.bBufferPeekBitsToUN16(
                *phPayload, (UN16 *)&psObj->sAvailCtrl.tRegion,
                EV1_REGION_BITLEN,
                EV1_PVN_BITLEN + EV1_CAROUSEL_BITLEN
                );

            if (TRUE == bRegionNeeded)
            {
                // Check to see if we want this region
                // before we validate the message (we don't know the text 
                // version just yet)
                bRegionNeeded = GsFuelMgrIntf.bIsRegionNeeded(
                    psObj->hFuelService,
                    psObj->sAvailCtrl.tRegion, 0);
            }

            // Do we need this region?
            if (FALSE == bRegionNeeded)
            {
                // Nope!
                return TRUE;
            }
        }

        // Validate the message (don't consume the CRC bytes though)
        bMessageValid = DS_UTIL_bIsCRCValid(psObj->hCRC, *phPayload, NULL);
        if (FALSE == bMessageValid)
        {
            puts(EV1_OBJECT_NAME" Packet Invalid");

            return TRUE;
        }

        // Is this an RFD message?
        if ((EV1_STATION_CAROUSEL_ID == un8CarouselID) ||
            (EV1_METADATA_CAROUSEL_ID == un8CarouselID))
        {
            // Pass the RFD AU to the RFD Interface
            bSuccess = RFD_INTERFACE_bProcessPayload(
                psObj->hRFD, *phPayload);

            // Regardless of the error code, we don't
            // own this payload anymore
            *phPayload = OSAL_INVALID_BUFFER_HDL;

            if (bSuccess == FALSE)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV1_OBJECT_NAME
                    ": RFD_INTERFACE_bProcessPayload() failed");
            }

            // Just stop here
            return bSuccess;
        }

        // We can now get rid of the CRC
        tBitsRead = OSAL.tBufferSeekTail(*phPayload, EV1_CRC_BYTELEN);
        tBitsRead *= 8; // make this a bit count

        // We can also now seek past the PVN & Car ID
        tBitsRead += OSAL.tBufferSeekHeadBits(
            *phPayload, EV1_PVN_BITLEN + EV1_CAROUSEL_BITLEN);

        if (tBitsRead != EV1_TRIM_BITLEN)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV1_OBJECT_NAME": Unable to seek past Message Header / Footer");

            // Stop here
            return FALSE;
        }

        // Now send the message on it's way
        switch (un8CarouselID)
        {
            case EV1_STATION_CAROUSEL_ID:
            case EV1_METADATA_CAROUSEL_ID:
            {
                // Should never happen   
            }
            break;

            case EV1_LOGO_UPDATE_CAROUSEL_ID:
            {
                // Process the logo message
                bSuccess = bProcessLogoMessage(psObj, phPayload);
            }
            break;

            case EV1_CHARGER_AVAILABILITY_1_CAROUSEL_ID:
            case EV1_CHARGER_AVAILABILITY_2_CAROUSEL_ID:
            {
                bSuccess = bProcessAvailabilityMessage(psObj, *phPayload);
            }
            break;

            case EV1_CHARGER_TYPE_CAROUSEL_ID:
            {
                bSuccess = bProcessChargerTypeMessage(psObj, *phPayload);
            }
            break;

            default:
            {
                // Don't make this a big deal
                bSuccess = TRUE;

                printf(EV1_OBJECT_NAME": Unknown Car ID: %u\n",
                    un8CarouselID);
            }
        }
    }

    return bSuccess;
}

/*****************************************************************************
*
*   bAddRegionsForLocation
*
*   This function provides the fuel manager with the ability to map N fuel
*   regions to a central lat/lon with a given radius.
*
*****************************************************************************/
static BOOLEAN bAddRegionsForLocation (
    FUEL_OTA_INTERFACE_OBJECT hInterface,
    LOCATION_OBJECT hCenterLocation,
    void *pvArg
        )
{
    FUEL_REGION tRegion,
                tCenterRegion,
                tStartRegion,
                tSWRegion,
                tNERegion,
                tDiff;
    UN8 un8Lat;
    N8  n8Lon;
    UN8 un8CurRow, un8CurCol;
    UN8 un8NumColsBefore,
        un8NumColsAfter,
        un8NumRowsBefore,
        un8NumRowsAfter;
    UN8 un8NumColsToProcess, un8NumRowsToProcess;
    UN8 un8Offset;
    BOOLEAN bIsPoint = FALSE;
    OSAL_FIXED_OBJECT hCenterLat,
                      hNELat,
                      hNELon,
                      hLat,
                      hLatDiff,
                      hCenterLon,
                      hLon,
                      hLonDiff;
    OSAL_FIXED_OBJECT_DATA atFixedData[OSAL_FIXED_OBJECT_SIZE*2];
    OSAL_FIXED_OBJECT_DATA atNEFixedData[OSAL_FIXED_OBJECT_SIZE*2];
    OSAL_RETURN_CODE_ENUM eReturnCode;
    SMSAPI_RETURN_CODE_ENUM eReturn;
    BOOLEAN bSuccess;
    EV1_OBJECT_STRUCT *psObj =
        (EV1_OBJECT_STRUCT *)hInterface;

    // Verify inputs
    if ((hCenterLocation == LOCATION_INVALID_OBJECT) ||
        (hInterface == FUEL_OTA_INTERFACE_INVALID_OBJECT))
    {
        return FALSE;
    }

    // Init memory
    OSAL.bMemSet(&atNEFixedData[0], 0, sizeof(atNEFixedData));
    OSAL.bMemSet(&atFixedData[0], 0, sizeof(atNEFixedData));

    // Get the center location's attributes
    hCenterLat = LOCATION.hLat(hCenterLocation);
    hCenterLon = LOCATION.hLon(hCenterLocation);
    eReturn = LOCATION.eIsPoint(hCenterLocation, &bIsPoint);

    if (((eReturn == SMSAPI_RETURN_CODE_SUCCESS) &&
         (bIsPoint == TRUE)) ||  // the location is empty area
        (hCenterLat == OSAL_FIXED_INVALID_OBJECT) ||
        (hCenterLon == OSAL_FIXED_INVALID_OBJECT)) // .. or does not have a base point
    {
        // Just move on
        return TRUE;
    }

    /*****************************************************
     * Calculate the region for the center of the location
     *****************************************************/

    // Just grab the lat / lon values as whole numbers
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hCenterLat);
    n8Lon = (N8)OSAL_FIXED.n32Floor(hCenterLon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un8Offset = (un8Lat - EV1_INITIAL_LAT) / EV1_REG_SIZE;
    tCenterRegion = EV1_REG_FROM_LAT_OFFSET(un8Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un8Offset = (n8Lon - EV1_INITIAL_LON) / EV1_REG_SIZE;
    tCenterRegion |= EV1_REG_FROM_LON_OFFSET(un8Offset);

    /*****************************************
     * Calculate the region for the NE corner
     *****************************************/
    // Get NE corner coordinates via LOCATION_bTopRight()
    hLat = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 0]);
    hLon = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE * 1]);

    LOCATION_bTopRight(hCenterLocation, hLat, hLon);

    // Keep a copy of the Lat
    OSAL_FIXED.bCopyToMemory(hLat, &atNEFixedData[0]);
    hNELat = (OSAL_FIXED_OBJECT)&atNEFixedData[0];

    // Keep a copy of the Lon
    OSAL_FIXED.bCopyToMemory(hLon, &atNEFixedData[OSAL_FIXED_OBJECT_SIZE]);
    hNELon = (OSAL_FIXED_OBJECT)&atNEFixedData[OSAL_FIXED_OBJECT_SIZE];

    // Just grab the lat / lon values as whole numbers
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hNELat);
    n8Lon = (N8)OSAL_FIXED.n32Floor(hNELon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un8Offset = (un8Lat - EV1_INITIAL_LAT) / EV1_REG_SIZE;
    tNERegion = EV1_REG_FROM_LAT_OFFSET(un8Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un8Offset = (n8Lon - EV1_INITIAL_LON) / EV1_REG_SIZE;
    tNERegion |= EV1_REG_FROM_LON_OFFSET(un8Offset);

    /*****************************************
     * Calculate the region for the SW corner
     *****************************************/

    // Calculate the latitude difference from the center to the NE
    hLatDiff = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
    eReturnCode = OSAL_FIXED.eSubtract(hNELat, hCenterLat, hLatDiff);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // So, the latitude is centerlat - latdiff
    hLat = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    eReturnCode = OSAL_FIXED.eSubtract(hCenterLat, hLatDiff, hLat);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Extract the latitude as a whole number
    un8Lat = (UN8)OSAL_FIXED.n32Floor(hLat);

    // Calculate the longitude difference from the center to the NE
    hLonDiff = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[0]);
    eReturnCode = OSAL_FIXED.eSubtract(hNELon, hCenterLon, hLonDiff);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // So, the longitude is centerlon - londiff
    hLon = OSAL_FIXED.hCreateInMemory(0, 0, &atFixedData[OSAL_FIXED_OBJECT_SIZE]);
    eReturnCode = OSAL_FIXED.eSubtract(hCenterLon, hLonDiff, hLon);

    if (eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Extract the longitude as a whole number
    n8Lon = (N8)OSAL_FIXED.n32Floor(hLon);

    // Using the latitude, calculate the offset from
    // the service's initial latitude to start computing
    // the region
    un8Offset = (un8Lat - EV1_INITIAL_LAT) / EV1_REG_SIZE;
    tSWRegion = EV1_REG_FROM_LAT_OFFSET(un8Offset);

    // Using the longitude, calculate the offset from
    // the service's initial longitude to complete computing
    // the region
    un8Offset = (n8Lon - EV1_INITIAL_LON) / EV1_REG_SIZE;
    tSWRegion |= EV1_REG_FROM_LON_OFFSET(un8Offset);

    /***************************************
     * All Regions have now been calculated
     ***************************************/

    // We now have the center region and the SW region and the NE region

    // Determine the number of rows & columns
    // before (south and west of) the center region that
    // must be processed
    tDiff = tCenterRegion - tSWRegion;
    un8NumColsBefore = tDiff / EV1_REG_GRID_COL_HEIGHT;
    un8NumRowsBefore = tDiff % EV1_REG_GRID_COL_HEIGHT;

    // Determine the number of rows & columns
    // after (north and east of) the center region that
    // must be processed
    tDiff = tNERegion - tCenterRegion;
    un8NumColsAfter = tDiff / EV1_REG_GRID_COL_HEIGHT;
    un8NumRowsAfter = tDiff % EV1_REG_GRID_COL_HEIGHT;

    // Start at the region in the SW
    tStartRegion = tCenterRegion - (EV1_REG_GRID_COL_HEIGHT * un8NumColsBefore);
    if (tStartRegion > 0)
    {
        tStartRegion -= un8NumRowsBefore;
    }

    if (tStartRegion < 0)
    {
        tStartRegion = 0;
    }

    // Calculate the total number of columns to process
    un8NumColsToProcess = un8NumColsBefore + un8NumColsAfter + 1; // + 1 for the center

    // Calculate the total number of rows to process
    un8NumRowsToProcess = un8NumRowsBefore + un8NumRowsAfter + 1; // + 1 for the center

    // Iterate through all the columns & rows
    for (un8CurCol = 0; un8CurCol < un8NumColsToProcess; un8CurCol++)
    {
        // Calculate the "base" region for this column
        tRegion = tStartRegion + (EV1_REG_GRID_COL_HEIGHT * un8CurCol);

        for (un8CurRow = 0; un8CurRow < un8NumRowsToProcess; un8CurRow++)
        {
            // Tell the manager to add this region
            bSuccess = GsFuelMgrIntf.bAddRegionOfInterest(
                psObj->hFuelService, (FUEL_REGION)(tRegion + un8CurRow), pvArg);

            // Ensure it succeeded
            if (bSuccess == FALSE)
            {
                return FALSE;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   eProcessAmenities
*
*   Process protocol-specific amenities for the caller
*
*****************************************************************************/
static SMSAPI_RETURN_CODE_ENUM eProcessAmenities (
    size_t tNumAmenities,
    AMENITY_STRUCT *pasFuelAmenities,
    UN32 un32RawAmenitiesData
        )
{
    // Validate inputs
    if ((tNumAmenities < FUEL_STATION_NUM_AMENITIES) ||
        (pasFuelAmenities == NULL))
    {
        return SMSAPI_RETURN_CODE_INVALID_INPUT;
    }
    else
    {
        UN8 un8AmenityOffset = 0, un8RawAmen;
        FUEL_STATION_AMENITY_ENUM eCurAmenity = EV1_STATION_BASE_AMENITY;
        SMSAPI_RETURN_CODE_ENUM eReturnCode =
            SMSAPI_RETURN_CODE_SUCCESS;

        do
        {
            switch (eCurAmenity)
            {
                case FUEL_STATION_AMENITY_OPEN_24HR: // index 0
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_24HR_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_24HR_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_OPEN_24HR;

                        if (un8RawAmen == EV1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }

                        un8AmenityOffset++;
                    }

                    // Move to the next appropriate amenity type
                    eCurAmenity = FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL;
                }
                break;

                case FUEL_STATION_AMENITY_AFFILIATED_NETWORK_CARDS_ACCEPTED: // index 1
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_AFFIL_CARDS_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_AMENITY_SECONDARY_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_AFFILIATED_NETWORK_CARDS_ACCEPTED;

                        pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;

                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_ALL_NETWORK_CARDS_ACCEPTED: // index 1
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_ALL_CARDS_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == EV1_AMENITY_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_ALL_NETWORK_CARDS_ACCEPTED;
                        if (un8RawAmen == EV1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_CREDIT_DEBIT: // index 2
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_CREDITDEBIT_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_CREDITDEBUG_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_CREDIT_DEBIT;
                        if (un8RawAmen == EV1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_PUBLIC_ACCESS: // index 3
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_PUBLIC_ACCESS_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_PUBLIC_ACCESS;
                        if (un8RawAmen == EV1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_PATRONS_ONLY: // index 3
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_PATRON_ACCESS_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_SECONDARY_AVAILABLE) ||
                        (un8RawAmen == EV1_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_PATRONS_ONLY;
                        if (un8RawAmen == EV1_AMENITY_SECONDARY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_CASH_ACCEPTED: // index 4
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_CASH_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_CASH_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_CASH_ACCEPTED;
                        if (un8RawAmen == EV1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                case FUEL_STATION_AMENITY_RESERVATIONS: // index 5
                {
                    un8RawAmen =
                        ((un32RawAmenitiesData >> EV1_RESERVATIONS_AMENITY_OFFSET)
                            & EV1_STATION_AMENITY_BITMASK);

                    if ((un8RawAmen == EV1_AMENITY_AVAILABLE) ||
                        (un8RawAmen == EV1_RESERVATIONS_AMENITY_NOT_AVAILABLE))
                    {
                        pasFuelAmenities[un8AmenityOffset].uAmenity.eFuelStationAmenity =
                            FUEL_STATION_AMENITY_RESERVATIONS;
                        if (un8RawAmen == EV1_AMENITY_AVAILABLE)
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_AVAILABLE;
                        }
                        else
                        {
                            pasFuelAmenities[un8AmenityOffset].eStatus = AMENITY_STATUS_UNAVAILABLE;
                        }
                        un8AmenityOffset++;
                    }
                }
                break;

                default:
                {
                    // We can't identify these amenities
                    eReturnCode = SMSAPI_RETURN_CODE_ERROR;
                    break;
                }
            }
        } while (++eCurAmenity <= EV1_STATION_MAX_AMENITY);

        if ((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS) &&
            (un8AmenityOffset == 0))
        {
            // We didn't find any amenities, let the caller know
            eReturnCode = SMSAPI_RETURN_CODE_NO_OBJECTS;
        }

        return eReturnCode;
    }
}

/*****************************************************************************
*
*   psCreateRFDCtrl
*
*****************************************************************************/
static EV1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    EV1_OBJECT_STRUCT *psObj,
    FUEL_DB_INTERFACE_OBJECT hDBInterface
        )
{
    EV1_RFD_CTRL_STRUCT *psRFDCtrl;

    // Create the RFD processor object
    psRFDCtrl = (EV1_RFD_CTRL_STRUCT *)
        SMSO_hCreate(
            EV1_OBJECT_NAME":RFDCtrl",
            sizeof(EV1_RFD_CTRL_STRUCT),
            SMS_INVALID_OBJECT, FALSE);
    if (psRFDCtrl == NULL)
    {
        return (EV1_RFD_CTRL_STRUCT *)NULL;
    }

    // Populate the structure with the minimum required
    psRFDCtrl->hDBInterface = hDBInterface;

    // Get the database content version
    psRFDCtrl->tCurrentVersion = (RFD_UPDATE_VERSION)
        GsEVChargingDBIntf.un16DBContentVer(hDBInterface);

    return psRFDCtrl;
}

/*****************************************************************************
*
*   bInitRFDCtrl
*
*****************************************************************************/
static BOOLEAN bInitRFDCtrl (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        )
{
    BOOLEAN bSuccess = FALSE;
    STRING_OBJECT hCurDB = STRING_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    do
    {
        // Store the RFD connection handle
        psRFDCtrl->hRFD = hRFD;

        // Intialize the sql command buffer
        psRFDCtrl->pcSQLCommandBuffer = (char *)
            SMSO_hCreate(
            EV1_OBJECT_NAME": SQLBuffer",
                FUEL_MAX_SQL_STRING_LENGTH,
                SMS_INVALID_OBJECT, FALSE);
        if (psRFDCtrl->pcSQLCommandBuffer == NULL)
        {
            break;
        }

        psRFDCtrl->tBufferSize = FUEL_MAX_SQL_STRING_LENGTH;

        // Initialize the structure
        psRFDCtrl->tStartProgressIndex = tStartingIndex;
        psRFDCtrl->tCurProgressIndex = 0;

        // Create the brand text list
        eReturnCode = OSAL.eLinkedListCreate(
            &psRFDCtrl->hBrandTexts,
            EV1_OBJECT_NAME":BrandTexts",
            (OSAL_LL_COMPARE_HANDLER)n16CompareTextEntries,
            OSAL_LL_OPTION_UNIQUE
                );
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error!
            break;
        }

        // Create our file reader buffer
        bSuccess = DATASERVICE_MGR_bCreateFileBuffer(
            &psRFDCtrl->hBlockPool,
            &psRFDCtrl->hBuffer,
            EV1_RFD_READ_BLOCK_SIZE
                );
        if (bSuccess == FALSE)
        {
            // Error! destroy object
            break;
        }

        // Create all of our fixed objects used
        // for constant values
        bSuccess = bCreateFixedConstants(&psRFDCtrl->sFixed);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Find out where to place our working copy
        bSuccess = GsEVChargingDBIntf.bRefDBBank(
            psRFDCtrl->hDBInterface, &hCurDB, &psRFDCtrl->hDBPath);
        if (bSuccess == FALSE)
        {
            break;
        }

        // Are we starting a new update?
        if (tStartingIndex == 0)
        {
            // Yes!  Get a copy of the db
            bSuccess = DB_UTIL_bCopyDB(
                STRING.pacCStr(psRFDCtrl->hDBPath),
                STRING.pacCStr(hCurDB));
            if (bSuccess == FALSE)
            {
                break;
            }
        }

        // Connect to our working database file now
        bSuccess = bConnectToDB(psRFDCtrl, FALSE);
        if (bSuccess == FALSE)
        {
            break;
        }

        if (tStartingIndex == 0)
        {
            bSuccess = GsEVChargingDBIntf.bDBUpdateBegin(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->pcSQLCommandBuffer,
                psRFDCtrl->tBufferSize);

            if (bSuccess == FALSE)
            {
                break;
            }
        }

    } while (FALSE);

    if (hCurDB != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(hCurDB);
    }

    return bSuccess;
}

/*****************************************************************************
*
*   vDestroyRFDCtrl
*
*****************************************************************************/
static void vDestroyRFDCtrl (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (psRFDCtrl == NULL)
    {
        return;
    }

    psRFDCtrl->hRFD = RFD_INTERFACE_INVALID_OBJECT;

    // Disconnect from the database
    if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        STRING_vDestroy(psRFDCtrl->hDBPath);
        psRFDCtrl->hDBPath = STRING_INVALID_OBJECT;
    }

    psRFDCtrl->tStartProgressIndex = 0;
    psRFDCtrl->tCurProgressIndex = 0;

    if (psRFDCtrl->pcSQLCommandBuffer != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT)psRFDCtrl->pcSQLCommandBuffer);
        psRFDCtrl->pcSQLCommandBuffer = NULL;
    }

    // Empty out the text list
    eReturnCode = OSAL.eLinkedListRemoveAll(
        psRFDCtrl->hBrandTexts,
        (OSAL_LL_RELEASE_HANDLER)vRemoveTextEntry );
    if (eReturnCode == OSAL_SUCCESS)
    {
        OSAL.eLinkedListDelete(psRFDCtrl->hBrandTexts);
        psRFDCtrl->hBrandTexts = OSAL_INVALID_OBJECT_HDL;
    }

    // Destroy the file buffer
    DATASERVICE_MGR_vDestroyFileBuffer(
        psRFDCtrl->hBlockPool, psRFDCtrl->hBuffer);
    psRFDCtrl->hBlockPool = OSAL_INVALID_OBJECT_HDL;
    psRFDCtrl->hBuffer = OSAL_INVALID_BUFFER_HDL;

    // Destroy all of our fixed objects
    vDestroyFixedConstants(&psRFDCtrl->sFixed);

    // Destroy this object
    SMSO_vDestroy((SMS_OBJECT)psRFDCtrl);

    return;
}

/*****************************************************************************
*
*   vDeleteDB
*
*****************************************************************************/
static void vDeleteDB (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    if (psRFDCtrl->hDBPath != STRING_INVALID_OBJECT)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);
        remove(pcDBPath);
    }
    return;
}

/*****************************************************************************
*
*   bConnectToDB
*
*****************************************************************************/
static BOOLEAN bConnectToDB (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        )
{
    SQL_INTERFACE_OPTIONS_MASK tOptions = SQL_INTERFACE_OPTIONS_NONE;
    BOOLEAN bConnected = FALSE;

    if (psRFDCtrl->hDBPath != NULL)
    {
        const char *pcDBPath;

        pcDBPath = STRING.pacCStr(psRFDCtrl->hDBPath);

        if (bPerformIntegrityCheck == FALSE)
        {
            tOptions |= SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK;
        }

        // Connect to the database
        psRFDCtrl->hDBConnection =
            SQL_INTERFACE.hConnect(
                pcDBPath, tOptions,
                (DATASERVICE_ERROR_CODE_ENUM *)NULL);

        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            bConnected = TRUE;
        }
    }

    return bConnected;
}

/*****************************************************************************
*
*   bFinalizeDB
*
*****************************************************************************/
static BOOLEAN bFinalizeDB (
   EV1_RFD_CTRL_STRUCT *psRFDCtrl
        )
{
    BOOLEAN bDBReady;

    bDBReady = bConnectToDB(psRFDCtrl, TRUE);
    if (bDBReady == TRUE)
    {
        SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
        psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
    }

    return bDBReady;
}

/*****************************************************************************
*
*   bEnableAvailabilityUpdates
*
*****************************************************************************/
static BOOLEAN bEnableAvailabilityUpdates (
    EV1_OBJECT_STRUCT *psObj
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (OSAL_INVALID_OBJECT_HDL != psObj->sChgTypeCtrl.hChargerTypeTables)
    {
        // Already initialized
        return TRUE;
    }

    // Create the list of charger type tables that
    // we're going to need in order to process
    // availability messages
    eReturnCode = OSAL.eLinkedListCreate(
        &psObj->sChgTypeCtrl.hChargerTypeTables,
        EV1_OBJECT_NAME":ChargerTables",
        (OSAL_LL_COMPARE_HANDLER)n16CompareChargerMappings,
        OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS |
        OSAL_LL_OPTION_UNIQUE);

    return (eReturnCode == OSAL_SUCCESS);
}

/*****************************************************************************
*
*   vDisableAvailabilityUpdates
*
*****************************************************************************/
static void vDisableAvailabilityUpdates (
    EV1_OBJECT_STRUCT *psObj
        )
{
    if (OSAL_INVALID_OBJECT_HDL != psObj->sChgTypeCtrl.hChargerTypeTables)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Remove and free all entries
        eReturnCode = OSAL.eLinkedListRemoveAll(
            psObj->sChgTypeCtrl.hChargerTypeTables,
            vReleaseChargerTypeTable);

        if (OSAL_SUCCESS == eReturnCode)
        {
            eReturnCode = OSAL.eLinkedListDelete(psObj->sChgTypeCtrl.hChargerTypeTables);
            if (OSAL_SUCCESS != eReturnCode)
            {
                SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                    EV1_OBJECT_NAME": Unable to destroy charger tables (%s)",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                psObj->sChgTypeCtrl.hChargerTypeTables = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    return;
}

/*****************************************************************************
*
*   bProcessAvailabilityMessage
*
*****************************************************************************/
static BOOLEAN bProcessAvailabilityMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bProcessMessage, bMore;
    EV1_AVAILABILITY_WORK_STRUCT sWork;

    // Initialize the work unit
    OSAL.bMemSet((void *)&sWork, 0, sizeof(EV1_AVAILABILITY_WORK_STRUCT));

    // First, process the availability message header and
    // populate the work unit
    bProcessMessage = bProcessAvailabilityMessageHeader(
        psObj, &sWork, hPayload );
    if (bProcessMessage == FALSE)
    {
        return TRUE;
    }

    do
    {
        // Continue to process the message until we're done
        bMore = bProcessAvailabilityMessageData(psObj, &sWork, hPayload);
    } while (TRUE == bMore);

    return TRUE;
}

/*****************************************************************************
*
*   bProcessAvailabilityMessageHeader
*
*****************************************************************************/
static BOOLEAN bProcessAvailabilityMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_AVAILABILITY_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bRegionUpdateNeeded = FALSE;

    do
    {
        BOOLEAN bRead;
        size_t tBitsRead;

        // Seek past the region id field
        tBitsRead = OSAL.tBufferSeekHeadBits(
            hPayload,
            EV1_REGION_BITLEN);

        if (EV1_REGION_BITLEN != tBitsRead)
        {
            break;
        }

        // Read the time field
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, &psWork->un16Time, 
            EV1_AVAIL_TIME_BITLEN);
        if (FALSE == bRead)
        {
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "TIME=%u", psWork->un16Time);

        // Read the CTAUVERSION field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8Version, 0, 
            EV1_AVAIL_CT_AUVER_BITLEN);
        if (EV1_AVAIL_CT_AUVER_BITLEN != tBitsRead)
        {
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CTAUVER=%u", psWork->un8Version);

        // Does the manager want to process updates for this region?
        bRegionUpdateNeeded = GsFuelMgrIntf.bRegionUpdateBegin( 
            psObj->hFuelService, 
            psWork->un16Time, 
            psObj->sAvailCtrl.tRegion,
            psWork->un8Version);

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "Region update needed = %u", bRegionUpdateNeeded);

        if (FALSE == bRegionUpdateNeeded)
        {
            // Manager doesn't want this
            break;
        }

        // Get the table that matches this version
        psWork->psChargerTable = psChargerTable(psObj, psWork->un8Version);
        if ((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL == psWork->psChargerTable)
        {
            // We don't have a table for this message!
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "Got Charger table of version %u", psWork->un8Version);

        // Read the CSSIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8CSSize, 0, 
            EV1_AVAIL_CSSIZE_BITLEN);
        if (EV1_AVAIL_CSSIZE_BITLEN != tBitsRead)
        {
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CSSIZE=%u", psWork->un8CSSize+1);

        // Section 7.1.6 SX-9845-0219
        psWork->un8CSSize++;

        // Read the ATYPESIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8ATypeSize, 0, 
            EV1_AVAIL_ATYPESIZE_BITLEN);
        if (EV1_AVAIL_ATYPESIZE_BITLEN != tBitsRead)
        {
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ATYPESIZE=%u", psWork->un8ATypeSize+1);

        // Section 7.1.7 SX-9845-0219
        psWork->un8ATypeSize++;

        // Read the NUMSIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8NumSize, 0, 
            EV1_AVAIL_NUMSIZE_BITLEN);
        if (EV1_AVAIL_NUMSIZE_BITLEN != tBitsRead)
        {
            break;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "NUMSIZE=%u", psWork->un8NumSize+1);

        // Section 7.1.8 SX-9845-0219
        psWork->un8NumSize++;

        // Read the CTSIZE Presence flag
        if (TRUE == bFieldPresent(hPayload))
        {
            UN8 un8CTSize = 0;

            // Read the CTSIZE field
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8CTSize, 0, 
                EV1_AVAIL_CTSIZE_BITLEN);

            if (EV1_AVAIL_CTSIZE_BITLEN != tBitsRead)
            {
                break;
            }

            // SX-9845-0218, Section 7.1
            un8CTSize++;

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CTSIZE=%u", un8CTSize);

            // We're going to seek past both of these fields,
            // so calculate the size of the seek
            un8CTSize *= 2;

            // Seek past the AU fields
            tBitsRead = OSAL.tBufferSeekHeadBits(
                hPayload, un8CTSize);

            if (un8CTSize != tBitsRead)
            {
                break;
            }
        }

        // We want this message
        return TRUE;
    } while (FALSE);

    // Did we start an update?
    if (TRUE == bRegionUpdateNeeded)
    {
        // Yes - close this update now
        GsFuelMgrIntf.vRegionUpdateComplete(psObj->hFuelService);
    }

    return FALSE;
}

/*****************************************************************************
*
*   bProcessAvailabilityMessageData
*
*****************************************************************************/
static BOOLEAN bProcessAvailabilityMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_AVAILABILITY_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bMoreToProcess = TRUE;

    do
    {
        BOOLEAN bOk, bDataPresent = FALSE;
        FUEL_STATION_ID tStation = (FUEL_STATION_ID)0;
        size_t tBitsRead;
        UN8 un8Index, un8VLCBit = 0, un8FuelTypeMapping,
            un8TableLength = 0,
            un8ExtensionSize;
        FUEL_POSITION_UPDATE_STRUCT *psCurUpdate;

        for (un8Index = 0; un8Index < EV1_AVAIL_MAX_VLC_CODE_LEN; un8Index++)
        {
            un8VLCBit = 0;
            tBitsRead =  OSAL.tBufferReadHeadBits(
                    hPayload, &un8VLCBit, 0, EV1_AVAIL_VLC_BITLEN);

            if (tBitsRead != EV1_AVAIL_VLC_BITLEN)
            {
                bMoreToProcess = FALSE;
                break;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "VL=%u", un8VLCBit);

            if (un8VLCBit != 0)
            {
                break;
            }
        }

        if (un8VLCBit != 0)
        {
            tStation = psWork->tPrevStation + un8Index + 1;
        }
        else
        {
            // No, just read the station's ID
            tStation = 0;
            bOk = OSAL.bBufferReadBitsToUN32(
                hPayload, (UN32 *)&tStation,
                psWork->un8CSSize);
            if (bOk == FALSE)
            {
                bMoreToProcess = FALSE;
                break;
            }
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CSUID=%d", tStation);

        // Save this station ID as our previous
        psWork->tPrevStation = tStation;

        // Is the availability data present?
        bDataPresent = bFieldPresent(hPayload);

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "AVAIL=%u", bDataPresent);

        if (TRUE == bDataPresent)
        {
            // Read ENO field
            un8TableLength = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload,
                &un8TableLength, 0,
                EV1_AVAIL_ENO_BITLEN);

            if (EV1_AVAIL_ENO_BITLEN != tBitsRead)
            {
                bMoreToProcess = FALSE;
                break;
            }

            // Section 7.2.4 SX-9845-0219
            un8TableLength++;

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ENO=%u", un8TableLength);

            if (EV1_AVAIL_MAX_TYPES < un8TableLength)
            {
                // Somehow this number is larger than should be possible
                // stop now to avoid lots of stuff
                bMoreToProcess = FALSE;
                break;
            }

            // Loop over all entries in this table instance
            for (un8Index = 0; un8Index < un8TableLength; un8Index++)
            {
                // Read the data for this table row
                psCurUpdate = &psObj->sAvailCtrl.asPositions[un8Index];

                // ATYPE
                un8FuelTypeMapping = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload,
                    &un8FuelTypeMapping,
                    0, psWork->un8ATypeSize);

                if (psWork->un8ATypeSize != tBitsRead)
                {
                    bMoreToProcess = FALSE;
                    break;
                }

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ATYPE=%u", un8FuelTypeMapping);

                // Get the fuel type and strings from the mapping
                psCurUpdate->un8FuelType =
                    psWork->psChargerTable->asMappings[un8FuelTypeMapping].un8ChargerType;
                psCurUpdate->eFuelType =
                    GsEVChargingDBIntf.eMatchFuelType(psCurUpdate->un8FuelType);
                psCurUpdate->hLongText =
                    psWork->psChargerTable->asMappings[un8FuelTypeMapping].hLongText;
                psCurUpdate->hShortText =
                    psWork->psChargerTable->asMappings[un8FuelTypeMapping].hShortText;

                // NUMAVAIL
                psCurUpdate->sPosition.un16NumberAvailableRefuelingPositions = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload,
                    &psCurUpdate->sPosition.un16NumberAvailableRefuelingPositions,
                    0, psWork->un8NumSize);

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "NUMAVAIL=%u", psCurUpdate->sPosition.un16NumberAvailableRefuelingPositions);

                if (psWork->un8NumSize != tBitsRead)
                {
                    bMoreToProcess = FALSE;
                    break;
                }

                // NUMINUSE
                psCurUpdate->sPosition.un16NumberInUseRefuelingPositions = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    hPayload,
                    &psCurUpdate->sPosition.un16NumberInUseRefuelingPositions,
                    0, psWork->un8NumSize);

                if (psWork->un8NumSize != tBitsRead)
                {
                    bMoreToProcess = FALSE;
                    break;
                }

                // NUMOFF
                psCurUpdate->sPosition.un16NumberOfflineRefuelingPositions = 0;
                if (TRUE == bFieldPresent(hPayload))
                {
                    tBitsRead = OSAL.tBufferReadHeadBits(
                        hPayload,
                        &psCurUpdate->sPosition.un16NumberOfflineRefuelingPositions,
                        0, psWork->un8NumSize);

                    if (psWork->un8NumSize != tBitsRead)
                    {
                        bMoreToProcess = FALSE;
                        break;
                    }
                }
            }
        }

        // Is extension field present?
        if (TRUE == bFieldPresent(hPayload))
        {
            // Read extension size
            tBitsRead = OSAL.tBufferReadHeadBits(
                hPayload, &un8ExtensionSize, 0,
                EV1_AVAIL_EXT_SIZE_BITLEN);
            if (EV1_AVAIL_EXT_SIZE_BITLEN != tBitsRead)
            {
                bMoreToProcess = FALSE;
                break;
            }

            // Section 7.2.10 & 7.2.11 SX-9845-0219
            un8ExtensionSize++;

            // Seek past extension
            tBitsRead = OSAL.tBufferSeekHeadBits(
                hPayload, un8ExtensionSize);

            if (tBitsRead != un8ExtensionSize)
            {
                bMoreToProcess = FALSE;
                break;
            }
        }

        // Are we done?
        if (FALSE == bMoreToProcess)
        {
            break;
        }

        // Report this update to the manager.  This report is done for 
        // every station listed in the message - even for those
        // with no accompanying data
        bMoreToProcess = GsFuelMgrIntf.bPositionUpdate(
            psObj->hFuelService, psWork->un16Time, tStation,
            &psObj->sAvailCtrl.asPositions[0], un8TableLength);

    } while (FALSE);

    if (FALSE == bMoreToProcess)
    {
        // Tell the manager we're all done
        GsFuelMgrIntf.vRegionUpdateComplete(psObj->hFuelService);
    }

    return bMoreToProcess;
}

/*****************************************************************************
*
*   bProcessChargerTypeMessage
*
*****************************************************************************/
static BOOLEAN bProcessChargerTypeMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        )
{
    BOOLEAN bProcessMessage, bSuccess;
    EV1_CHARGER_TYPE_WORK_STRUCT sWork;

    // Initialize the work unit
    OSAL.bMemSet((void *)&sWork, 0, sizeof(EV1_CHARGER_TYPE_WORK_STRUCT));

    // First, process the message header and populate the work unit
    bProcessMessage = bProcessChargerTypeMessageHeader(
        psObj, &sWork, hPayload );
    if (bProcessMessage == FALSE)
    {
        return TRUE;
    }

    // Process the message body now
    bSuccess = bProcessChargerTypeMessageData(psObj, &sWork, hPayload);

    return bSuccess;
}

/*****************************************************************************
*
*   bProcessChargerTypeMessageHeader
*
*****************************************************************************/
static BOOLEAN bProcessChargerTypeMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_CHARGER_TYPE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable = 
        (EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL;
    do
    {
        size_t tBitsRead;
        EV1_CHARGER_TYPE_TABLE_STRUCT *psExistingTable;
        UN8 un8Version = 0;

        // Read the version field
        un8Version = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8Version,0, 
            EV1_CHGTYPE_CT_AUVER_BITLEN);

        if (EV1_CHGTYPE_CT_AUVER_BITLEN != tBitsRead)
        {
            break;
        }

        // Does this table already exist?
        psExistingTable = psChargerTable(psObj, un8Version);
    
        if ((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL != psExistingTable)
        {
            // Yep...stop here
            break;
        }

        // Create a new instance of the table
        psTable = (EV1_CHARGER_TYPE_TABLE_STRUCT *)
                OSAL.pvLinkedListMemoryAllocate(
                    EV1_OBJECT_NAME":ChgTable", 
                    sizeof (EV1_CHARGER_TYPE_TABLE_STRUCT), 
                    TRUE);
    
        if ((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL == psTable)
        {
            // Can't process this message without the necessary memory!
            break;
        }

        // The table is this version!
        psTable->un8Version = un8Version;

        // Read the type size field
        psWork->un8ATypeSize = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8ATypeSize,0, 
            EV1_CHGTYPE_ATYPESIZE_BITLEN);

        if (EV1_CHGTYPE_ATYPESIZE_BITLEN != tBitsRead)
        {
            break;
        }

        // Section 7.1.7 SX-9845-0219
        psWork->un8ATypeSize++;

        // Read the num field
        psWork->un8ChargeTypeCount = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psWork->un8ChargeTypeCount,0, 
            EV1_CHGTYPE_NUM_BITLEN);

        if (EV1_CHGTYPE_NUM_BITLEN != tBitsRead)
        {
            break;
        }

        // Section 8.2.2 SX-9845-0219
        psWork->un8ChargeTypeCount++;

        // Use the table we allocated
        psWork->psTable = psTable;

        return TRUE;

    } while (FALSE);

    if ((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL != psWork->psTable)
    {
        vReleaseChargerTypeTable(psWork->psTable);
        psWork->psTable = (EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bProcessChargerTypeMessageData
*
*****************************************************************************/
static BOOLEAN bProcessChargerTypeMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_CHARGER_TYPE_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    UN8 un8Index;
    size_t tBitsRead;
    UN8 un8AType;
    EV1_CHARGER_MAPPING_ENTRY_STRUCT *psCurMapping;
    BOOLEAN bSuccess = TRUE;

    for (un8Index = 0;
         un8Index < psWork->un8ChargeTypeCount; 
         un8Index++)
    {
        // Read the "ATYPE" field
        un8AType = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8AType, 0, 
            psWork->un8ATypeSize);

        if (EV1_CHGTYPE_MAX_MAPPINGS <= un8AType)
        {
            // Value out of range
            bSuccess = FALSE;
            break;
        }

        // We want this mapping slot
        psCurMapping = &psWork->psTable->asMappings[un8AType];

        psCurMapping->un8ChargerType = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &psCurMapping->un8ChargerType, 0, 
            EV1_CHGTYPE_CHTYPE_BITLEN);

        if (EV1_CHGTYPE_CHTYPE_BITLEN != tBitsRead)
        {
            bSuccess = FALSE;
            break;
        }

        // Is the short text present?
        if (bFieldPresent(hPayload))
        {
            psCurMapping->hShortText = 
                BAUDOT_hToString(
                    (SMS_OBJECT)psObj->hFuelService,
                    hPayload,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 36, NULL);

            if (STRING_INVALID_OBJECT == psCurMapping->hShortText)
            {
                bSuccess = FALSE;
                break;
            }
        }

        // Is the long text present?
        if (bFieldPresent(hPayload))
        {
            psCurMapping->hLongText = 
                BAUDOT_hToString(
                    (SMS_OBJECT)psObj->hFuelService,
                    hPayload,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 36, NULL);

            if (STRING_INVALID_OBJECT == psCurMapping->hLongText)
            {
                bSuccess = FALSE;
                break;
            }
        }
    }

    if (TRUE == bSuccess)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // All went well, try to add this table to our list now
        eReturnCode = OSAL.eLinkedListAdd(
            psObj->sChgTypeCtrl.hChargerTypeTables,
            OSAL_INVALID_LINKED_LIST_ENTRY_PTR,
            (void *)psWork->psTable);

        if (OSAL_SUCCESS != eReturnCode)
        {
            bSuccess = FALSE;
        }
    }
    
    // Did we run across any issues?
    if (FALSE == bSuccess)
    {
        // Yeah, give up on this table
        vReleaseChargerTypeTable(psWork->psTable);
        psWork->psTable = (EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL;
    }
    
    return bSuccess;
}

/*****************************************************************************
*
*   n16CompareChargerMappings
*
*****************************************************************************/
static N16 n16CompareChargerMappings (
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable1,
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable2
        )
{
    if (((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL != psTable1) &&
        ((EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL != psTable2))
    {
        return COMPARE(psTable1->un8Version, psTable2->un8Version);
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   psChargerTable
*
*****************************************************************************/
static EV1_CHARGER_TYPE_TABLE_STRUCT *psChargerTable (
    EV1_OBJECT_STRUCT *psObj,
    UN8 un8Version
        )
{
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable = 
        (EV1_CHARGER_TYPE_TABLE_STRUCT *)NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Configure our search structure
    psObj->sChgTypeCtrl.sSearchStruct.un8Version = un8Version;

    // Search for this table version
    eReturnCode = OSAL.eLinkedListSearch(
        psObj->sChgTypeCtrl.hChargerTypeTables,
        &hEntry, (void *)&psObj->sChgTypeCtrl.sSearchStruct );
    if (OSAL_SUCCESS == eReturnCode)
    {
        // Extract this table
        psTable = (EV1_CHARGER_TYPE_TABLE_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);
    }

    return psTable;
}

/*****************************************************************************
*
*   bProcessLogoMessage
*
*****************************************************************************/
static BOOLEAN bProcessLogoMessage (
    EV1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bProcessMessage, bOk;
    EV1_LOGO_WORK_STRUCT sWork;

    // Do we want to process this?
    if (psObj->sLogoCtrl.bEnabled == FALSE)
    {
        // Nope!
        return TRUE;
    }

    // Initialize the work unit
    sWork.un16SeqNo = EV1_FIRST_MULTI_AU_SEQ_NO;
    sWork.un16MessagesNeeded = 1;
    sWork.bUpdateLogo = FALSE;

    // Clear the logo row data
    OSAL.bMemSet(&sWork.sLogo, 0, sizeof(FUEL_LOGO_ROW_STRUCT));

    // All logos from this interface are NOT baseline logos
    sWork.sLogo.bIsBaselineLogo = FALSE;

    // First, process the logo message header and
    // populate the work unit
    bProcessMessage = bProcessLogoMessageHeader(
        psObj, &sWork, *phPayload );
    if (bProcessMessage == FALSE)
    {
        return TRUE;
    }

    // Process the logo message data now
    bOk = bProcessLogoMessageData(psObj, &sWork, phPayload);

    return bOk;
}

/*****************************************************************************
*
*   bProcessLogoMessageHeader
*
*****************************************************************************/
static BOOLEAN bProcessLogoMessageHeader (
    EV1_OBJECT_STRUCT *psObj,
    EV1_LOGO_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL hPayload
        )
{
    size_t tBitsRead = 0;
    BOOLEAN bProcessThisMessage, bRead;

    // Read the update version field
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->sLogo.un8LogoTableVer, 0,
        EV1_LOGO_UVER_BITLEN);
    if (tBitsRead != EV1_LOGO_UVER_BITLEN)
    {
        return FALSE;
    }

    // Read the logo id
    bRead = OSAL.bBufferReadBitsToUN16(
        hPayload, &psWork->sLogo.un16LogoId,
        EV1_LOGOID_BITLEN);
    if (bRead == FALSE)
    {
        return FALSE;
    }

    // Read the logo type
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->sLogo.eLogoType, 0,
        EV1_LOGO_TYPE_BITLEN);
    if (tBitsRead != EV1_LOGO_TYPE_BITLEN)
    {
        return FALSE;
    }

    // Read the operation type
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->bUpdateLogo, 0,
        EV1_LOGO_OPER_BITLEN);
    if (tBitsRead != EV1_LOGO_OPER_BITLEN)
    {
        return FALSE;
    }

    // Read the file version
    tBitsRead = OSAL.tBufferReadHeadBits(
        hPayload, &psWork->sLogo.un8LogoVer, 0,
        EV1_LOGO_FVER_BITLEN);
    if (tBitsRead != EV1_LOGO_FVER_BITLEN)
    {
        return FALSE;
    }

    // Are the AU fields present?
    if (bFieldPresent(hPayload))
    {
        UN8 un8CTSize = 0;

        // Read the CTSIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            hPayload, &un8CTSize, 0,
            EV1_LOGO_CTSIZE_BITLEN);
        if (tBitsRead != EV1_LOGO_CTSIZE_BITLEN)
        {
            return FALSE;
        }

        // SX9845-0216, Section 5.1.7
        un8CTSize++;

        // Read the total number of access units
        // for this message
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, &psWork->un16MessagesNeeded,
            un8CTSize);
        if (bRead == FALSE)
        {
            return FALSE;
        }

        // Read the sequence number for this
        // message
        bRead = OSAL.bBufferReadBitsToUN16(
            hPayload, &psWork->un16SeqNo,
            un8CTSize);
        if (bRead == FALSE)
        {
            return FALSE;
        }
    }

    // Can we process this logo message?
    bProcessThisMessage = bCanProcessLogoMessage(
        &psObj->sLogoCtrl, psWork);
    if (bProcessThisMessage == FALSE)
    {
        // No, we can't use this message
        return FALSE;
    }

    // Only need to worry about the following when
    // we are dealing with logo updates, not when
    // we're deleting a logo
    if (psWork->bUpdateLogo == TRUE)
    {
        size_t tNumBits, tNumBitsFromByteSize;

        // The header may end with pad bits -- this is because
        // the payload of this message is a image data that is
        // byte-aligned.  Find out how many bits are in this buffer
        // and how many bytes are in the buffer.
        tNumBits = OSAL.tBufferGetSizeInBits(hPayload);
        tNumBitsFromByteSize = OSAL.tBufferGetSize(hPayload) * 8;

        // Find the bit-size difference reported here (if any)
        tNumBits -= tNumBitsFromByteSize;

        // Seek past this difference, which is all pad bits
        tBitsRead = OSAL.tBufferSeekHeadBits(hPayload, tNumBits);
        if (tBitsRead != tNumBits)
        {
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bProcessLogoMessageData
*
*   This function is utilized when processing a work unit based on text
*   message data.  Each text entry is provided to the manager via the
*   manager interface.
*
*****************************************************************************/
static BOOLEAN bProcessLogoMessageData (
    EV1_OBJECT_STRUCT *psObj,
    EV1_LOGO_WORK_STRUCT *psWork,
    OSAL_BUFFER_HDL *phPayload
        )
{
    BOOLEAN bOk = TRUE,
            bResetProgress = FALSE;

    // Set all in-progress logo attributes if we're
    // starting a new logo message sequence
    if (psWork->un16SeqNo == EV1_FIRST_MULTI_AU_SEQ_NO)
    {
        // Save the logo row data
        psObj->sLogoCtrl.sLogoInProgress = psWork->sLogo;

        // Initialize the number of messages processed
        psObj->sLogoCtrl.un16MessagesProcessed = 0;

        // Initialize the logo data handle
        psObj->sLogoCtrl.hLogoData = *phPayload;
    }
    else if (psObj->sLogoCtrl.hLogoData == OSAL_INVALID_BUFFER_HDL)
    {
        // Start a new series
        psObj->sLogoCtrl.hLogoData = *phPayload;
    }
    else if (psWork->bUpdateLogo == TRUE)// Append data if not deleting
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Append the contents of the payload
        // into our logo data
        eReturnCode = OSAL.eBufferAppend(
            psObj->sLogoCtrl.hLogoData,
            *phPayload );

        bOk =  (eReturnCode == OSAL_SUCCESS);
    }

    // Did that work?
    if (bOk == TRUE)
    {
        // Clear the payload pointer given to us
        // since we now own the payload
        *phPayload = OSAL_INVALID_BUFFER_HDL;

        // Increment the number of messages we've processed
        // for this logo
        psObj->sLogoCtrl.un16MessagesProcessed++;

        // We know this logo has been fully constructed when
        // we're received the right number of messages
        if (psObj->sLogoCtrl.un16MessagesProcessed == psWork->un16MessagesNeeded)
        {
            // Tell the manager about this logo now
            bOk = GsFuelMgrIntf.bLogoUpdate(
                psObj->hFuelService, &psWork->sLogo,
                (BOOLEAN)!psWork->bUpdateLogo,
                psObj->sLogoCtrl.hLogoData);

            // The manager now owns the logo data
            psObj->sLogoCtrl.hLogoData = OSAL_INVALID_BUFFER_HDL;

            // done with this logo now, need to reset
            bResetProgress = TRUE;
        }
    }
    else
    {
        bResetProgress = TRUE;
    }

    if (bResetProgress == TRUE)
    {
        vInitLogoTracking(&psObj->sLogoCtrl, TRUE);
    }

    return bOk;
}

/*****************************************************************************
*
*   vInitLogoTracking
*
*****************************************************************************/
static void vInitLogoTracking (
    EV1_LOGO_CTRL_STRUCT *psLogoCtrl,
    BOOLEAN bEnabled
        )
{
    if (psLogoCtrl->bEnabled != bEnabled)
    {
        // Save the enabled flag
        psLogoCtrl->bEnabled = bEnabled;

        // Destroy any in-progress logo buffer data
        if (psLogoCtrl->hLogoData != OSAL_INVALID_BUFFER_HDL)
        {
            OSAL.eBufferFree(psLogoCtrl->hLogoData);
            psLogoCtrl->hLogoData = OSAL_INVALID_BUFFER_HDL;
        }

        // Clear the last sequence number attribute
        // (this is the key value in this structure)
        psLogoCtrl->n32LastSeqNo = EV1_LAST_SEQ_NO_INITIAL_VAL;
    }

    return;
}

/*****************************************************************************
*
*   bCanProcessLogoMessage
*
*****************************************************************************/
static BOOLEAN bCanProcessLogoMessage (
    EV1_LOGO_CTRL_STRUCT *psLogoCtrl,
    EV1_LOGO_WORK_STRUCT *psWork
        )
{
    // First see if this work unit is as expected.
    // If it isn't clear any in-progress logo data.
    // Are we currently processing a multi-AU
    // logo?  If so, validate this payload against
    // what we're currently tracking
    if (psLogoCtrl->n32LastSeqNo != EV1_LAST_SEQ_NO_INITIAL_VAL)
    {
        // Verify Logo Id, Logo Version, Update version, and sequence number.
        // If any of this values don't match what we expect,
        // we'll have to trash our current progress.
        if ((psWork->sLogo.un16LogoId != psLogoCtrl->sLogoInProgress.un16LogoId) ||
            (psWork->sLogo.un8LogoTableVer != psLogoCtrl->sLogoInProgress.un8LogoTableVer) ||
            (psWork->sLogo.un8LogoVer != psLogoCtrl->sLogoInProgress.un8LogoVer) ||
            (psWork->sLogo.eLogoType != psLogoCtrl->sLogoInProgress.eLogoType) ||
            (psWork->un16SeqNo != (UN16)psLogoCtrl->n32LastSeqNo + 1))
        {
            // Well, we lost sync.  So, clear out any in-progress
            // logo data and we'll start again
            vInitLogoTracking(psLogoCtrl, TRUE);
        }
    }

    // Okay, now see if this sequence number is as expected
    return (psWork->un16SeqNo == (UN16)(psLogoCtrl->n32LastSeqNo + 1));
}

/*****************************************************************************
*
*   bEnableRFDUpdates
*
*****************************************************************************/
static BOOLEAN bEnableRFDUpdates (
    EV1_OBJECT_STRUCT *psObj
        )
{
    EV1_RFD_CTRL_STRUCT *psRFDCtrl = (EV1_RFD_CTRL_STRUCT *)NULL;

    if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
    {
        // Already initialized
        return TRUE;
    }

    do
    {
        RFD_OPTIONAL_CALLBACKS_STRUCT sCallbacks;

        // Create the RFD processor object
        psRFDCtrl = psCreateRFDCtrl(psObj, psObj->hDBInterface);
        if (psRFDCtrl == NULL)
        {
            break;
        }

        // Clear the callback structure
        OSAL.bMemSet((void *)&sCallbacks, 0, sizeof(RFD_OPTIONAL_CALLBACKS_STRUCT));

        // Populate the optional callback structure
        sCallbacks.eUpdateNeeded = eRFDUpdateNeeded;

        // Connect to RFD
        psObj->hRFD = RFD_INTERFACE_hConnect(
            EV1_RFD_CLIENT_ID,
            psRFDCtrl->tCurrentVersion,
            EV1_MAX_VERSION_BITLEN,
            eRFDFileProcessor,
            &sCallbacks,
            (void *)psRFDCtrl);

        if (psObj->hRFD == RFD_INTERFACE_INVALID_OBJECT)
        {
            // Error! destroy object
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV1_OBJECT_NAME": Can't create RFD connection");
            break;
        }

        return TRUE;

    } while (FALSE);

    vDestroyRFDCtrl(psRFDCtrl);

    return FALSE;
}

/*****************************************************************************
*
*   vDisableRFDUpdates
*
*****************************************************************************/
static void vDisableRFDUpdates (
    EV1_OBJECT_STRUCT *psObj
        )
{
    // Disconnect RFD interface if it is connected
    if (psObj->hRFD != RFD_INTERFACE_INVALID_OBJECT)
    {
        RFD_INTERFACE_vDisconnect(psObj->hRFD);
        psObj->hRFD = RFD_INTERFACE_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bProcessRFDHeader
*
*****************************************************************************/
static BOOLEAN bProcessRFDHeader (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    BOOLEAN bDataTransferred, bEntryAdded;
    size_t tBitsRead;
    UN8 un8BRNum = 0,
        un8BRCode;
    STRING_OBJECT hBrand;
    BOOLEAN bRead, bOk;

    // Fill the buffer with file data
    bDataTransferred = DATASERVICE_MGR_bFillBufferBlock(
        psFile, psRFDCtrl->hBuffer);
    if (bDataTransferred == FALSE)
    {
        // Couldn't access the file
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 2, "Processing File Header");

    // Process the charge types in the header now
    bOk = bProcessChargeTypes(psRFDCtrl, psFile);
    if (bOk == FALSE)
    {
        // Something went wrong
        return FALSE;
    }

    // Read the BRNUM field
    tBitsRead = OSAL.tBufferReadHeadBits(
        psRFDCtrl->hBuffer, &un8BRNum, 0,
        EV1_BRNUM_BITLEN);
    if (tBitsRead != EV1_BRNUM_BITLEN)
    {
        return FALSE;
    }

    // RX 185, Section 7.2.1
    un8BRNum++;

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 3, "BRNUM = %d", un8BRNum);

    do
    {
        // Read the BRCODE field
        un8BRCode = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8BRCode, 0,
            EV1_BRCODE_BITLEN);
        if (tBitsRead != EV1_BRCODE_BITLEN)
        {
            return FALSE;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "BRCODE = %d", un8BRCode);

        // Read the BAUDOT string for this brand code
        hBrand = BAUDOT_hToString(
            SMS_INVALID_OBJECT,
            psRFDCtrl->hBuffer,
            BAUDOT_BEHAVIOR_PROCESS_TO_END,
            TRUE, TRUE, 0, NULL);
        if (hBrand == STRING_INVALID_OBJECT)
        {
            return FALSE;
        }

        // Add a new brand text entry now
        bEntryAdded = bAddBrandText(
            psRFDCtrl, un8BRCode, hBrand);
        if (bEntryAdded == FALSE)
        {
            return FALSE;
        }

    } while (--un8BRNum > 0);

    // Now, make sure the default "unbranded"
    // brand is there
    // Create the string object
    hBrand = STRING.hCreate(EV1_UNBRANDED_TEXT,0);

    // Attempt to add it -- we pass ownership
    // of the string object to this function,
    // so if it fails it'll be freed there
    bAddBrandText(
        psRFDCtrl, EV1_UNBRANDED_ID, hBrand);

    // Read the region count field
    psRFDCtrl->un16NumRegionsInUpdate = 0;
    bRead = OSAL.bBufferReadBitsToUN16(
        psRFDCtrl->hBuffer,
        &psRFDCtrl->un16NumRegionsInUpdate,
        EV1_DESC_REG_COUNT_BITLEN);
    if (bRead == FALSE)
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 3, "Num Regions To Update = %d", psRFDCtrl->un16NumRegionsInUpdate);
    
    if (psRFDCtrl->un16NumRegionsInUpdate == 0)
    {
        SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
            EV1_OBJECT_NAME": Invalid number of regions in file header");

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bProcessChargeTypes
*
*****************************************************************************/
static BOOLEAN bProcessChargeTypes (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    size_t tBitsRead;
    UN8 un8ChTypeNum = 0;
    BOOLEAN bSuccess;
    FUEL_TEXT_ROW_STRUCT sTextRow;

    // This will always be false here
    sTextRow.bBrandText = FALSE;

    // Read the CHTYPE_NUM field
    tBitsRead = OSAL.tBufferReadHeadBits(
        psRFDCtrl->hBuffer, &un8ChTypeNum, 0,
        EV1_CHTYPE_NUM_BITLEN);
    if (tBitsRead != EV1_CHTYPE_NUM_BITLEN)
    {
        return FALSE;
    }

    // SX9845-0219, Section 6.2.1.1
    un8ChTypeNum++;

    do
    {
        // Read the CHTYPE field
        sTextRow.un8TextId = 0;
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &sTextRow.un8TextId, 0,
            EV1_CHTYPE_BITLEN);
        if (tBitsRead != EV1_CHTYPE_BITLEN)
        {
            return FALSE;
        }

        // Skip past the CVER field
        tBitsRead = OSAL.tBufferSeekHeadBits(
            psRFDCtrl->hBuffer, EV1_CVER_BITLEN);
        if (tBitsRead != EV1_CVER_BITLEN)
        {
            return FALSE;
        }

        // Is the Short Text Description Present?
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            // Read the short text desc
            sTextRow.hText =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (sTextRow.hText == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }
        }
        else
        {
            sTextRow.hText = STRING_INVALID_OBJECT;
        }

        // Is the Long Text Description Present?
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            // Read the long text desc
            sTextRow.hLongText =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (sTextRow.hLongText == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }
        }
        else
        {
            sTextRow.hLongText = STRING_INVALID_OBJECT;
        }

        // Tell the DB to update the text entry now
        bSuccess = GsEVChargingDBIntf.bUpdateTextEntry(
            FUEL_DB_INTERFACE_INVALID_OBJECT,
            psRFDCtrl->hDBConnection,
            &sTextRow);

        if (bSuccess == FALSE)
        {
            break;
        }

    } while (--un8ChTypeNum > 0);

    return bSuccess;
}

/*****************************************************************************
*
*   eProcessNextRFDEntry
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        )
{
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_INCOMPLETE;
    BOOLEAN bSuccess = TRUE;

    // Do we have any work left?
    if (psRFDCtrl->un16NumRegionsInUpdate != psRFDCtrl->tCurProgressIndex)
    {
        // Process the next region
        bSuccess = bProcessRegion(psRFDCtrl, psFile);

        if (bSuccess == TRUE)
        {
            // Increment our progress index
            psRFDCtrl->tCurProgressIndex++;

            if (psRFDCtrl->tCurProgressIndex > psRFDCtrl->tStartProgressIndex)
            {
                // Report our progress
                bSuccess = RFD_INTERFACE_bReportProgress(
                    psRFDCtrl->hRFD, psRFDCtrl->tCurProgressIndex);
            }
        }
    }

    if (bSuccess == TRUE)
    {
        // Are we done?
        if (psRFDCtrl->un16NumRegionsInUpdate == psRFDCtrl->tCurProgressIndex)
        {
            // Finalize the DB
            bSuccess = GsEVChargingDBIntf.bDBUpdateEnd(
                psRFDCtrl->hDBConnection,
                psRFDCtrl->pcSQLCommandBuffer,
                psRFDCtrl->tBufferSize, (UN16)tUpdateVersion);

            if (bSuccess == TRUE)
            {
                // Yes -- tell the caller
                eResult = RFD_PROCESS_RESULT_COMPLETE;
            }
        }
    }

    if (bSuccess == FALSE)
    {
        eResult = RFD_PROCESS_RESULT_ERROR;
    }

    return eResult;
}

/*****************************************************************************
*
*   bProcessRegion
*
*   Process a region entry in the DB update file
*
*****************************************************************************/
static BOOLEAN bProcessRegion(
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        )
{
    UN8 un8UType,
        un8CSSize = 0,
        un8Div = 0,
        un8SubReg = 0;
    size_t tBitsRead;
    BOOLEAN bOk;
    UN16 un16SubDiv = 0;
    FUEL_STATION_ROW_STRUCT sStationRow;
    BOOLEAN bInformManager = TRUE;
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType =
        FUEL_STATION_UPDATE_MAX_TYPES;
    N32 n32PrevLat = -1, n32PrevLon = -1;

    // Did we already process this region?
    if (psRFDCtrl->tCurProgressIndex < psRFDCtrl->tStartProgressIndex)
    {
        bInformManager = FALSE;
    }

    // Read the region id field
    psRFDCtrl->tRegion = 0;
    bOk = OSAL.bBufferReadBitsToUN16(
        psRFDCtrl->hBuffer,
        (UN16 *)&psRFDCtrl->tRegion,
        EV1_REGION_BITLEN);

    if (bOk == FALSE)
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 2, "Parsing tRegion = %d", psRFDCtrl->tRegion);

    if (bInformManager == TRUE)
    {
        // We now have the region ID, and version.
        // See if the manager wants the update
        bOk = GsEVChargingDBIntf.bRegionUpdateBegin(
            psRFDCtrl->hDBConnection,
            psRFDCtrl->pcSQLCommandBuffer,
            psRFDCtrl->tBufferSize,
            psRFDCtrl->tRegion);
        if (bOk == FALSE)
        {
            return FALSE;
        }
    }

    do
    {
        BOOLEAN bBreak = FALSE;

        // Read the CSSIZE field
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8CSSize, 0,
            EV1_CSSIZE_BITLEN);
        if (tBitsRead != EV1_CSSIZE_BITLEN)
        {
            break;
        }

        // SX9845-0128, Section 6.2.2.2
        un8CSSize++;

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 3, "CSSIZE = %d", un8CSSize);

        do
        {
            // Fill the buffer with file data
            DATASERVICE_MGR_bFillBufferBlock(
                psFile, psRFDCtrl->hBuffer);

            // Read the UTYPE field
            un8UType = 0;
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8UType, 0,
                EV1_DESC_UPDATE_TYPE_BITLEN);
            if (tBitsRead != EV1_DESC_UPDATE_TYPE_BITLEN)
            {
                break;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 3, "UTYPE = %d", un8UType);

            switch (un8UType)
            {
                case EV1_UPDATE_TYPE_DELETE:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_DELETE;
                }
                break;

                case EV1_UPDATE_TYPE_NEW:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_NEW;
                }
                break;

                case EV1_UPDATE_TYPE_MODIFY:
                {
                    eUpdateType = FUEL_STATION_UPDATE_TYPE_MODIFY_GIVEN_ATTRIBS;
                }
                break;

                case EV1_UPDATE_TYPE_END:
                {
                    // All done with this region
                    bBreak = TRUE;
                }
                break;

                default:
                {
                    // Don't know what we've read,
                    // and we don't know how to continue either
                    bBreak = TRUE;
                }
            }

            // Break if we experienced a problem or if
            // we don't have a valid update type
            if ((bBreak == TRUE) ||
                (eUpdateType >= FUEL_STATION_UPDATE_MAX_TYPES))
            {
                break;
            }

            // Clear the row
            OSAL.bMemSet(&sStationRow, 0, sizeof(FUEL_STATION_ROW_STRUCT));

            // Clear State ID
            sStationRow.tStateID = STATE_INVALID_ID;

            // All station data is for this region
            // and this version
            sStationRow.tRegion = psRFDCtrl->tRegion;

            // Put back coordinates of previous station
            sStationRow.n32PrevLat = n32PrevLat;
            sStationRow.n32PrevLon = n32PrevLon;

            // Process the station entry and provide
            // it to the manager if necessary
            bOk = bProcessStation(
                psRFDCtrl, bInformManager, &sStationRow, eUpdateType,
                un8CSSize, &un8SubReg, &un8Div, &un16SubDiv);

            // Saving coordinates for next time
            n32PrevLat = sStationRow.n32PrevLat;
            n32PrevLon = sStationRow.n32PrevLon;

        } while (bOk == TRUE);

        if (bOk == FALSE)
        {
            break;
        }

        if (bInformManager == TRUE)
        {
            // Inform the manager that the update
            // for this region is done
            bOk = GsEVChargingDBIntf.bRegionUpdateEnd(psRFDCtrl->hDBConnection);
        }
    } while (FALSE);

    return bOk;
}

/*****************************************************************************
*
*   bProcessStation
*
*   Process a station entry in the DB update file
*
*****************************************************************************/
static BOOLEAN bProcessStation(
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    FUEL_STATION_UPDATE_TYPE_ENUM eUpdateType,
    UN8 un8CSSize,
    UN8 *pun8LastSubRegion,
    UN8 *pun8LastDivision,
    UN16 *pun16LastSubDivision
        )
{
    BOOLEAN bOk,
            bLocationProvided = FALSE,
            bAmenitiesUpdated = FALSE,
            bTypeCountKnown = FALSE;
    size_t tBitsRead;
    UN8 un8LocType = 0,
        un8BRCode = FUEL_INVALID_BRAND_ID;
    UN16 un16FieldVersion = 0;

    // Use our types array
    psStationRow->psTypes = &psRFDCtrl->asTypes[0];
    OSAL.bMemSet(psRFDCtrl->asTypes, 0, sizeof(psRFDCtrl->asTypes));
    psStationRow->bUpdateTypes = FALSE;
    psStationRow->bUpdateNumbers = FALSE;

    SMSAPI_DEBUG_vPrint("", 2, "CSUID");

    // Get the station ID
    psStationRow->tStationId = 0;
    bOk = OSAL.bBufferReadBitsToUN32(
        psRFDCtrl->hBuffer,
        (UN32 *)&psStationRow->tStationId,
        un8CSSize);
    if (bOk == FALSE)
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CSUID = %d", psStationRow->tStationId);

    SMSAPI_DEBUG_vPrint("", 2, "EVER");

    // Read the EVER
    bOk = OSAL.bBufferReadBitsToUN16(
        psRFDCtrl->hBuffer, &un16FieldVersion,
        EV1_UPDATE_ENTRY_VER_BITLEN);

    if (bOk == FALSE)
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "EVER = %d", un16FieldVersion);

    if (un16FieldVersion == (UN16)psRFDCtrl->tCurrentVersion)
    {
        // Skip this entry - we already know about it
        bInformManager = FALSE;
    }

    // If we're deleting an entry, we don't
    // need to read anything else.
    if (eUpdateType != FUEL_STATION_UPDATE_TYPE_DELETE)
    {

        SMSAPI_DEBUG_vPrint("", 2, "LOC");

        // Extract this station's LOC information
        // Read the LOC type
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8LocType, 0,
            EV1_LOC_BITLEN);

        // Ensure that was read correctly
        if (tBitsRead != EV1_LOC_BITLEN)
        {
            return FALSE;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "LOC type = %d", un8LocType);

        switch (un8LocType)
        {
            case EV1_LOC_SUBREG:
            {
                SMSAPI_DEBUG_vPrint("", 2, "LOC_SUBREG");

                *pun8LastSubRegion = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer, pun8LastSubRegion, 0,
                    EV1_LOC_SUBREG_BITLEN);
                if (tBitsRead != EV1_LOC_SUBREG_BITLEN)
                {
                    return FALSE;
                }
            }
            /* no break */

            case EV1_LOC_DIV:
            {
                SMSAPI_DEBUG_vPrint("", 2, "LOC_DIV");

                *pun8LastDivision = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer, pun8LastDivision, 0,
                    EV1_LOC_DIV_BITLEN);
                if (tBitsRead != EV1_LOC_SUBREG_BITLEN)
                {
                    return FALSE;
                }
            }
            /* no break */

            case EV1_LOC_SUBDIV:
            {
                SMSAPI_DEBUG_vPrint("", 2, "LOC_SUBDIV");

                // Broadcast gave us a new location
                bLocationProvided = TRUE;
                *pun16LastSubDivision = 0;
                bOk = OSAL.bBufferReadBitsToUN16(
                    psRFDCtrl->hBuffer, pun16LastSubDivision,
                    EV1_LOC_SUBDIV_BITLEN);
                if (bOk == FALSE)
                {
                    return FALSE;
                }
            }
            /* no break */
        }

        if (bLocationProvided == TRUE)
        {
            // Compute the lat / lon for this station
            bOk = bComputeLatLonFromGrid(
                psRFDCtrl, psStationRow,
                *pun8LastSubRegion,
                *pun8LastDivision,
                *pun16LastSubDivision);
            if (bOk == FALSE)
            {
                return FALSE;
            }
            
            // Saving calculated values for future use
            psStationRow->n32PrevLat = psStationRow->n32Lat;
            psStationRow->n32PrevLon = psStationRow->n32Lon;
        }
        else // Location info not provided
        {
            if (un8LocType == EV1_LOC_PREV)
            {
                // Use previous values
                psStationRow->n32Lat = psStationRow->n32PrevLat;
                psStationRow->n32Lon = psStationRow->n32PrevLon;
            }
            else
            {
                // Invalidate these fields
                psStationRow->n32Lat = -1;
                psStationRow->n32Lon = -1;
            }
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+BRCODE");

        // "Unknown Brand" by default
        un8BRCode = 0;

        // Read the brand code
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "BRCODE");

            // Read the BRCode field
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8BRCode, 0,
                EV1_BRCODE_BITLEN);

            if (tBitsRead != EV1_BRCODE_BITLEN)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "BRCODE = %d", un8BRCode);
        }

        // Get the brand string from the brand code
        psStationRow->hBrand = hGetBrandString(psRFDCtrl, un8BRCode );

        SMSAPI_DEBUG_vPrint("", 2, "F+CSNAME");

        // Read the Charge Station name (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "CSNAME");

            psStationRow->hName =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (psStationRow->hName == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CSNAME = %s", STRING.pacCStr(psStationRow->hName));
        }
        else
        {
            psStationRow->hName = STRING_INVALID_OBJECT;
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+LOGOID");

        // Read the Logo ID
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "LOGOID");

            // Read the LogoID field
            psStationRow->n16LogoId = 0;
            bOk = OSAL.bBufferReadBitsToN16(
                    psRFDCtrl->hBuffer, &psStationRow->n16LogoId,
                    EV1_LOGOID_BITLEN);

            if (bOk == FALSE)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "LODOID = %d", psStationRow->n16LogoId);
        }
        else
        {
            psStationRow->n16LogoId = -1;
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+DESC");

        // Read the description (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "DESC");

            psStationRow->hDesc =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (psStationRow->hDesc == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "DESC = %s", STRING.pacCStr(psStationRow->hDesc));
        }
        else
        {
            psStationRow->hDesc = STRING_INVALID_OBJECT;
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+ADDR");

        // Read the address (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "ADDR");

            psStationRow->hAddr =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    FALSE, TRUE, 0, NULL);
            if (psStationRow->hAddr == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ADDR = %s", STRING.pacCStr(psStationRow->hAddr));
        }
        else
        {
            psStationRow->hAddr = STRING_INVALID_OBJECT;
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+CITY");

        // Read the city (Baudot)
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {

            SMSAPI_DEBUG_vPrint("", 2, "CITY");

            psStationRow->hCity =
                BAUDOT_hToString(
                    SMS_INVALID_OBJECT,
                    psRFDCtrl->hBuffer,
                    BAUDOT_BEHAVIOR_PROCESS_TO_END,
                    TRUE, TRUE, 0, NULL);
            if (psStationRow->hCity == STRING_INVALID_OBJECT)
            {
                return FALSE;
            }

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CITY = %s", STRING.pacCStr(psStationRow->hCity));
        }
        else
        {
            psStationRow->hCity = STRING_INVALID_OBJECT;
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+STATE");

        // Read state code
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8State = 0;
            
            SMSAPI_DEBUG_vPrint("", 2, "STATE");

            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8State, 0,
                EV1_DESC_STATE_BITLEN);

            if (tBitsRead != EV1_DESC_STATE_BITLEN)
            {
                return FALSE;
            }

            // Save the state id
            psStationRow->tStateID = (STATE_ID)un8State;

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "STATE = %d", psStationRow->tStateID);
        }
        else
        {
            psStationRow->tStateID = STATE_INVALID_ID;
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+ZIP");

        // Read ZIP
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8ZipType = 0;


            SMSAPI_DEBUG_vPrint("", 2, "ZIP Type");

            // First read the zip type
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer, &un8ZipType, 0,
                EV1_ZIP_TYPE_BITLEN);

            if (tBitsRead != EV1_ZIP_TYPE_BITLEN)
            {
                return FALSE;
            }

            if (un8ZipType == EV1_ZIP_TYPE_INTEGER)
            {
                UN32 un32Zip = 0;
                char acZip[EV1_MAX_ZIP_INT_CHARS + 1];


                SMSAPI_DEBUG_vPrint("", 2, "ZIP int");

                // Read the zip code as an integer
                bOk = OSAL.bBufferReadBitsToUN32(
                    psRFDCtrl->hBuffer, &un32Zip,
                    EV1_ZIP_BITLEN);
                if (bOk == FALSE)
                {
                    return FALSE;
                }

                // Create a string for the zip code
                snprintf(&acZip[0], sizeof(acZip), "%05u", un32Zip);
                psStationRow->hZIP = STRING.hCreate(&acZip[0], 0);

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ZIP = %s", acZip);
            }
            else // Read ZIP as a string (baudot)
            {

                SMSAPI_DEBUG_vPrint("", 2, "ZIP string");

                psStationRow->hZIP =
                    BAUDOT_hToString(
                        SMS_INVALID_OBJECT,
                        psRFDCtrl->hBuffer,
                        BAUDOT_BEHAVIOR_PROCESS_TO_END,
                        TRUE, TRUE, 0, NULL);

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "ZIP = %s", STRING.pacCStr(psStationRow->hZIP));
            }
        }
        else
        {
            psStationRow->hZIP = STRING_INVALID_OBJECT;
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+PHONE");

        // Read Phone
        psStationRow->n64Phone = 0;
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN16 un16AreaCode = 0,
                 un16Exchange = 0,
                 un16Number = 0;

            SMSAPI_DEBUG_vPrint("", 2, "PHONE area");

            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16AreaCode, 10);

            SMSAPI_DEBUG_vPrint("", 2, "PHONE exch");

            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16Exchange, 10);

            SMSAPI_DEBUG_vPrint("", 2, "PHONE number");

            OSAL.bBufferReadBitsToUN16(
                psRFDCtrl->hBuffer, &un16Number, 14);

            // Convert to single int
            psStationRow->n64Phone =
                FUEL_MGR_n64GeneratePhoneFromFields(
                    un16AreaCode, un16Exchange, un16Number);

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "PHONE = (%03d) %03d %04d", un16AreaCode, un16Exchange, un16Number);
        }


        SMSAPI_DEBUG_vPrint("", 2, "F+CHTYPEARR");

        // Read the charge type array
        psStationRow->un8NumPermanentTypes = 0;
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8Index;

            SMSAPI_DEBUG_vPrint("", 2, "CHTYPE ENO");

            // Setting presence flag for Charge Types array
            psStationRow->bUpdateTypes = TRUE;

            // First read the number of entries in the array
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer,
                &psStationRow->un8NumPermanentTypes, 0,
                EV1_ENO_BITLEN);

            if (tBitsRead != EV1_ENO_BITLEN)
            {
                return FALSE;
            }

            // SX9845-0219, Section 6.2.3.14
            psStationRow->un8NumPermanentTypes++;
            bTypeCountKnown = TRUE;

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "CHType ENO = %d", psStationRow->un8NumPermanentTypes);

            // Bounds check
            if (psStationRow->un8NumPermanentTypes > FUEL_REFUELING_TYPES_ARRAY_SIZE)
            {
                return FALSE;
            }

            // Now, read all the entries
            for (un8Index = 0;
                un8Index < psStationRow->un8NumPermanentTypes; un8Index++)
            {
                SMSAPI_DEBUG_vPrint("", 2, "CHTYPE");

                psRFDCtrl->asTypes[un8Index].un8FuelType = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer,
                    &psRFDCtrl->asTypes[un8Index].un8FuelType, 0,
                    EV1_CHTYPE_BITLEN);

                if (tBitsRead != EV1_CHTYPE_BITLEN)
                {
                    return FALSE;
                }

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 5, "CHType = %d", psRFDCtrl->asTypes[un8Index].un8FuelType);
            }
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+PNUMARR");

        // Read the charge position number array
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            UN8 un8Index, un8NumChargeTypes = 0;

            SMSAPI_DEBUG_vPrint("", 2, "PNUM ENO");

            // Setting presence flag for Positions Number array
            psStationRow->bUpdateNumbers = TRUE;

            // First read the number of entries in the array
            tBitsRead = OSAL.tBufferReadHeadBits(
                psRFDCtrl->hBuffer,
                &un8NumChargeTypes, 0,
                EV1_ENO_BITLEN);

            if (tBitsRead != EV1_ENO_BITLEN)
            {
                return FALSE;
            }

            // SX9845-0219, Section 6.2.3.14
            un8NumChargeTypes++;

            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "PNUM ENO = %d", un8NumChargeTypes);

            // We got the number of charge types
            // already -- so make sure this number matches it!
            if (bTypeCountKnown == TRUE)
            {
                if (un8NumChargeTypes != psStationRow->un8NumPermanentTypes)
                {
                    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 1, 
                        "++ Region %d, station %d, CHTYPEARR size is %d, CHNUMARR size is %d",
                        psStationRow->tRegion, psStationRow->tStationId, 
                        psStationRow->un8NumPermanentTypes, un8NumChargeTypes);

                    return FALSE;
                }
            }
            // We're learning the type count now
            else
            {
                // Save this count
                psStationRow->un8NumPermanentTypes = un8NumChargeTypes;
                if (psStationRow->un8NumPermanentTypes > FUEL_REFUELING_TYPES_ARRAY_SIZE)
                {
                    return FALSE;
                }
            }

            // Now, read all the entries
            for (un8Index = 0;
                 un8Index < un8NumChargeTypes; 
                 un8Index++)
            {
                SMSAPI_DEBUG_vPrint("", 2, "PNUM");

                psRFDCtrl->asTypes[un8Index].un8NumPositions = 0;
                tBitsRead = OSAL.tBufferReadHeadBits(
                    psRFDCtrl->hBuffer,
                    &psRFDCtrl->asTypes[un8Index].un8NumPositions, 0,
                    EV1_CHNUM_BITLEN);

                if (tBitsRead != EV1_CHNUM_BITLEN)
                {
                    return FALSE;
                }

                SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 5, "PNUM = %d", psRFDCtrl->asTypes[un8Index].un8NumPositions);
            }
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+AMENITIES1");

        // Read Amenities 1
        psStationRow->un32Amenities = 0;
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            SMSAPI_DEBUG_vPrint("", 2, "AMENITIES1");

            bOk = bProcessAmenities(
                psRFDCtrl, TRUE, &psStationRow->un32Amenities);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            bAmenitiesUpdated = TRUE;
        }

        SMSAPI_DEBUG_vPrint("", 2, "F+AMENITIES2");

        // Read Amenities 2
        if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
        {
            SMSAPI_DEBUG_vPrint("", 2, "AMENITIES2");

            bOk = bProcessAmenities(
                psRFDCtrl, FALSE, &psStationRow->un32Amenities);
            if (bOk == FALSE)
            {
                return FALSE;
            }

            bAmenitiesUpdated = TRUE;
        }
    }

    SMSAPI_DEBUG_vPrint("", 2, "F+EXT");

    // Read Extension
    if (bFieldPresent(psRFDCtrl->hBuffer) == TRUE)
    {
        UN8 un8NumExtensionBytes = 0;
        UN16 un16ExtensionBitSize = 0;

        SMSAPI_DEBUG_vPrint("", 2, "EXTSIZE");

        // Read the extension size
        tBitsRead = OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8NumExtensionBytes, 0,
            EV1_EXTENSION_SIZE_BITLEN);
        if (tBitsRead != EV1_EXTENSION_SIZE_BITLEN)
        {
            return FALSE;
        }

        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "EXTSIZE = %d", un8NumExtensionBytes);
        
        // The size of the extension field is the number
        // of bytes times byte size
        un16ExtensionBitSize = un8NumExtensionBytes * 8;

        // Seek past the rest of the bits
        // since we dont' understand them
        OSAL.tBufferSeekHeadBits(
            psRFDCtrl->hBuffer, (size_t)un16ExtensionBitSize);
    }

    if (bInformManager == TRUE)
    {
        // Provide the DB with the station update now
        bOk = GsEVChargingDBIntf.bStationUpdate(
            psRFDCtrl->hDBConnection,
            &psRFDCtrl->pcSQLCommandBuffer,
            &psRFDCtrl->tBufferSize,
            psStationRow, eUpdateType,
            bAmenitiesUpdated);
    }
    else
    {
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*   bProcessAmenities
*
*   Process the amenities entry for a station
*
*****************************************************************************/
static BOOLEAN bProcessAmenities (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bAmenities1,
    UN32 *pun32Amenities
        )
{
    size_t tBitsRead = 0,
           tBitOffset = 0;
    UN8 un8CurrentAmen,
        un8RawAmenValue;
    UN32 un32Temp;

    // Are we working on the first amenities field?
    if (bAmenities1 == FALSE)
    {
        // No, make sure we don't overwrite amen1 with
        // these values from amen2
        tBitOffset = FUEL_AMENITIES_ARRAY_SIZE * EV1_AMEN_FIELD_BITLEN;
    }

    for (un8CurrentAmen = 0;
         un8CurrentAmen < FUEL_AMENITIES_ARRAY_SIZE;
         un8CurrentAmen++)
    {
        // Read two bits per field
        un8RawAmenValue = 0;
        tBitsRead += OSAL.tBufferReadHeadBits(
            psRFDCtrl->hBuffer, &un8RawAmenValue, 0,
            EV1_AMEN_FIELD_BITLEN);

        // Take that value
        un32Temp = (UN32)un8RawAmenValue;

        // Place it at the correct bit position
        un32Temp <<= tBitOffset;

        // Save the result
        *pun32Amenities |= un32Temp;

        // Update the bit position
        tBitOffset += EV1_AMEN_FIELD_BITLEN;
    }

    // Make sure we read it all
    if (tBitsRead != (FUEL_AMENITIES_ARRAY_SIZE * EV1_AMEN_FIELD_BITLEN))
    {
        return FALSE;
    }

    SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 4, "Amenities = %X", *pun32Amenities);

    return TRUE;
}

/*****************************************************************************
*
*   bFieldPresent
*
*   A simple function which reads 1 bits and reports TRUE or FALSE
*   based on that bit's value.
*
*****************************************************************************/
static BOOLEAN bFieldPresent(
    OSAL_BUFFER_HDL hBuffer
        )
{
    BOOLEAN bPresent = FALSE;
    size_t tBitsRead;

    // Read the presence flag
    tBitsRead = OSAL.tBufferReadHeadBits(
        hBuffer, &bPresent, 0, EV1_PRESENCE_FLAG_BITLEN);

    if (tBitsRead != EV1_PRESENCE_FLAG_BITLEN)
    {
        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 1, 
            "Cannot read buffer, returning \"fake\" presence flag");

        // Indicate presence if we can't read the buffer
        // to force the caller to fail as well
        return TRUE;
    }

    return bPresent;
}

/*****************************************************************************
*
*   eRFDFileProcessor
*
*   The callback invoked by RFD when an update file has been fully recieved
*   and is ready for processing
*
*****************************************************************************/
static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        )
{
    EV1_RFD_CTRL_STRUCT *psRFDCtrl = (EV1_RFD_CTRL_STRUCT *)pvCallbackArg;
    RFD_PROCESS_RESULT_ENUM eResult = RFD_PROCESS_RESULT_ERROR;

    if (psRFDCtrl == NULL)
    {
        return RFD_PROCESS_RESULT_ERROR;
    }

    if (eProcessStatus == RFD_PROCESS_STATUS_BEGIN)
    {
        BOOLEAN bOk;

        // Initialize the RFD Ctrl object now
        bOk = bInitRFDCtrl(psRFDCtrl, hConnection, tProgressIndex);

        if (bOk == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 1, "Started RFD processing");

            // Process the update header now
            bOk = bProcessRFDHeader(psRFDCtrl, psRFDFile);
        }

        if (bOk == FALSE)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV1_OBJECT_NAME": Unable to process RFD file header");

            return RFD_PROCESS_RESULT_ERROR;
        }
    }
    else if (eProcessStatus == RFD_PROCESS_STATUS_STOP)
    {
        SMSAPI_DEBUG_vPrint(EV1_OBJECT_NAME, 1, "Completed RFD processing");
        
        // Free associated memory if we're stopping
        vDestroyRFDCtrl(psRFDCtrl);

        return RFD_PROCESS_RESULT_INCOMPLETE;
    }

    // Process the file contents and update the database
    eResult = eProcessNextRFDEntry(
        psRFDCtrl, psRFDFile, tFileVersion);
    if (eResult != RFD_PROCESS_RESULT_INCOMPLETE)
    {
        // Close our connection to the DB
        if (psRFDCtrl->hDBConnection != SQL_INTERFACE_INVALID_OBJECT)
        {
            SQL_INTERFACE.vDisconnect(psRFDCtrl->hDBConnection);
            psRFDCtrl->hDBConnection = SQL_INTERFACE_INVALID_OBJECT;
        }

        // Are we done?
        if (eResult == RFD_PROCESS_RESULT_COMPLETE)
        {
            BOOLEAN bFinalized;

            // Finalize the new database
            bFinalized = bFinalizeDB(psRFDCtrl);
            if (bFinalized == FALSE)
            {
                // We failed to get the DB ready for use
                eResult = RFD_PROCESS_RESULT_ERROR;
            }
        }

        // Did we experience an error?
        if (eResult  >= RFD_PROCESS_RESULT_ERROR)
        {
            SMSAPI_DEBUG_vPrintErrorFull(gpacThisFile, __LINE__,
                EV1_OBJECT_NAME": RFD processor returned error (%d)",
                eResult);

            // Delete database
            vDeleteDB(psRFDCtrl);
        }
    }

    return eResult;
}

/*****************************************************************************
*
*   bComputeLatLonFromGrid
*
*   This function is utilized a station update is received in order to
*   generate a lat/lon pair for that station based upon its location in the
*   region/sub-region/division/sub-division grid.
*
*****************************************************************************/
static BOOLEAN bComputeLatLonFromGrid (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    FUEL_STATION_ROW_STRUCT *psStationRow,
    UN8 un8SubRegion,
    UN8 un8Division,
    UN16 un16SubDivision
        )
{
    EV1_FIXED_CALC_STRUCT *psFixed =
        &psRFDCtrl->sFixed;
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT hLon;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8RegionLatOffset = EV1_REG_LAT_OFFSET(psStationRow->tRegion);
    UN8 un8RegionLonOffset = EV1_REG_LON_OFFSET(psStationRow->tRegion);
    UN32 un32Lon, un32Lat;
    UN8 un8SubRegionLon = EV1_GET_LON(un8SubRegion);
    UN8 un8SubRegionLat = EV1_GET_LAT(un8SubRegion);
    UN8 un8DivLon = EV1_GET_LON(un8Division);
    UN8 un8DivLat = EV1_GET_LAT(un8Division);
    UN8 un8SubDivLon = EV1_GET_SUBDIV_LON(un16SubDivision);
    UN8 un8SubDivLat = EV1_GET_SUBDIV_LAT(un16SubDivision);

    /*
        RX 185, Section 7.5

        lon = -168 +{((r_lon << 14) | (sr_lon << 10) | (d_lon << 6) | (sd_lon)) / 8192.0};
        lat = 12 + {((r_lat << 14) | (sr_lat << 10) | (d_lat << 6) | (sd_lat )) / 8192.0};
    */

    do
    {
        // Compute the integer math section of the lat & lon computations
        un32Lon = (
           ((un8RegionLonOffset << 14) | (un8SubRegionLon << 10) | (un8DivLon << 6) | (un8SubDivLon))
                );

        un32Lat = (
            (un8RegionLatOffset << 14) | (un8SubRegionLat << 10) | (un8DivLat << 6) | (un8SubDivLat)
                 );

        // Create a fixed for the lat & lon computations, start them out
        // with our computed values
        hLat = OSAL_FIXED.hCreateInMemory(un32Lat, 0,
            &psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * EV1_LAT_OBJ_INDEX]);
        hLon = OSAL_FIXED.hCreateInMemory(un32Lon, 0,
            &psFixed->atFixedData[OSAL_FIXED_OBJECT_SIZE * EV1_LON_OBJ_INDEX]);

        if ((hLat == OSAL_FIXED_INVALID_OBJECT) ||
            (hLon == OSAL_FIXED_INVALID_OBJECT))
        {
            // Error!
            break;
        }

        /*-- Divide those values by the computation denominator --*/
        eReturnCode = OSAL_FIXED.eDivide(hLat, psFixed->hComputeDenominator, hLat);
        eReturnCode |= OSAL_FIXED.eDivide(hLon, psFixed->hComputeDenominator, hLon);

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // Add that to the initial lat & lon
        eReturnCode |= OSAL_FIXED.eAdd( hLat, psFixed->hInitialLat, hLat );
        eReturnCode |= OSAL_FIXED.eAdd( hLon, psFixed->hInitialLon, hLon );

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // We now have fixed-point values for the lat & lon,
        // but we don't know their bin point.  We do know that
        // we must conform to the fuel manager's bin point however.
        // So, get these correctly scaled value now

        // Latitude first
        psStationRow->n32Lat = OSAL_FIXED.n32ScaledValue(hLat, LOCATION_BINPOINT);

        // Then longitude
        psStationRow->n32Lon = OSAL_FIXED.n32ScaledValue(hLon, LOCATION_BINPOINT);

        return TRUE;
    }
    while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   bCreateFixedConstants
*
*   This function creates all fixed values for the calculations that
*   this interface must perform to convert grid points to a lat/lon pair
*
*****************************************************************************/
static BOOLEAN bCreateFixedConstants (
    EV1_FIXED_CALC_STRUCT *psFixed
        )
{
    do
    {
        // Initial latitude offset
        psFixed->hInitialLat = OSAL_FIXED.hCreateFromFixed(
            EV1_INITIAL_LAT, 0);

        if (psFixed->hInitialLat == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // Initial longitude offset
        psFixed->hInitialLon = OSAL_FIXED.hCreateFromFixed(
            EV1_INITIAL_LON, 0);
        if (psFixed->hInitialLon == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        // The lat / lon computation denominator
        psFixed->hComputeDenominator = OSAL_FIXED.hCreateFromFixed(
            EV1_COMPUTE_DENOMINATOR, 0);
        if (psFixed->hComputeDenominator == OSAL_FIXED_INVALID_OBJECT)
        {
            break;
        }

        return TRUE;
    }
    while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   vDestroyFixedConstants
*
*   Destroys all fixed constants used by this interface.
*
*****************************************************************************/
static void vDestroyFixedConstants (
    EV1_FIXED_CALC_STRUCT *psFixed
        )
{
    if (psFixed->hInitialLat != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hInitialLat);
        psFixed->hInitialLat = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hInitialLon != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hInitialLon);
        psFixed->hInitialLon = OSAL_FIXED_INVALID_OBJECT;
    }

    if (psFixed->hComputeDenominator != OSAL_FIXED_INVALID_OBJECT)
    {
    	OSAL_FIXED.vDestroy(psFixed->hComputeDenominator);
        psFixed->hComputeDenominator = OSAL_FIXED_INVALID_OBJECT;
    }

    return;
}

/*****************************************************************************
*
*   bAddBrandText
*
*   This function adds a brand text entry using the provided parameters.
*
*****************************************************************************/
static BOOLEAN bAddBrandText (
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode,
    STRING_OBJECT hString
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    EV1_BRAND_TEXT_STRUCT *psText =
        (EV1_BRAND_TEXT_STRUCT *)NULL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Create the name for this text entry
    snprintf( &acName[0],
        OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
        EV1_OBJECT_NAME":Text(Code:%u)",
        un8BrandCode );

    // Allocate the text row and add it
    psText = (EV1_BRAND_TEXT_STRUCT *)
        SMSO_hCreate(
            &acName[0],
            sizeof(EV1_BRAND_TEXT_STRUCT),
            (SMS_OBJECT)psRFDCtrl, FALSE);

    if (psText == NULL)
    {
        return FALSE;
    }

    // Populate the text entry
    psText->un8Code = un8BrandCode;
    psText->hString = hString;

    // Add the new entry to the list
    eReturnCode = OSAL.eLinkedListAdd(
        psRFDCtrl->hBrandTexts, OSAL_INVALID_LINKED_LIST_ENTRY_PTR, (void *)psText );
    if (eReturnCode != OSAL_SUCCESS)
    {
        STRING_vDestroy(psText->hString);
        psText->hString = STRING_INVALID_OBJECT;

        SMSO_vDestroy((SMS_OBJECT)psText);

        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*   hGetBrandString
*
*****************************************************************************/
static STRING_OBJECT hGetBrandString(
    EV1_RFD_CTRL_STRUCT *psRFDCtrl,
    UN8 un8BrandCode
        )
{
    STRING_OBJECT hString = STRING_INVALID_OBJECT;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY;
    EV1_BRAND_TEXT_STRUCT sSearchParameters;

    // Initialize our search parameters
    sSearchParameters.hString = STRING_INVALID_OBJECT;
    sSearchParameters.un8Code = un8BrandCode;

    // Search for the code provided
    eReturnCode = OSAL.eLinkedListSearch(
        psRFDCtrl->hBrandTexts,
        &hEntry,
        &sSearchParameters );
    if (eReturnCode == OSAL_SUCCESS)
    {
        EV1_BRAND_TEXT_STRUCT *psText;

        // Extract the list contents
        psText = (EV1_BRAND_TEXT_STRUCT *)
            OSAL.pvLinkedListThis(hEntry);

        if (psText != NULL)
        {
            // Pull out the string handle
            hString = psText->hString;
        }
    }

    return hString;
}

/*****************************************************************************
*
*   n16CompareTextEntries
*
*****************************************************************************/
static N16 n16CompareTextEntries (
    EV1_BRAND_TEXT_STRUCT *psText1,
    EV1_BRAND_TEXT_STRUCT *psText2
        )
{
    if ((psText1 != NULL) && (psText2 != NULL))
    {
        // Sort by code
        if (psText1->un8Code < psText2->un8Code)
        {
            return -1;
        }

        if (psText1->un8Code > psText2->un8Code)
        {
            return 1;
        }

        return 0;
    }

    return N16_MIN;
}

/*****************************************************************************
*
*   vRemoveTextEntry
*
*****************************************************************************/
static void vRemoveTextEntry (
    EV1_BRAND_TEXT_STRUCT *psText
        )
{
    if (psText != NULL)
    {
        // Set the code to the "unbranded" code value
        psText->un8Code = 0;

        // Destroy / Clear the string handle
        STRING_vDestroy(psText->hString);
        psText->hString = STRING_INVALID_OBJECT;

        // Destroy the object
        SMSO_vDestroy((SMS_OBJECT)psText);
    }

    return;
}

/*****************************************************************************
 *
 *  bExtractVersionFromName
 *
 *  This API is used to parse a file name from an RFD file to determine
 *  base version of file and file version.  Filenames parsed
 *  by this function must be in the form UXXYY where XX is the version
 *  the file is being upgraded from and YY is the version it will be
 *  upgraded to.
 *
 *  This function is a default which may be overwritten by the client
 *
 *****************************************************************************/
static BOOLEAN bExtractVersionFromName(
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptBaseVersion,
    RFD_UPDATE_VERSION *ptNewVersion
        )
{
    char acBuffer[EV1_RFD_UPDATE_VER_STRING_LEN + 1];
    UN8 un8Idx = 0;
    size_t tFileNameStrLen = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Verify that the length of pcFileName is 7
        tFileNameStrLen = strlen(pcFileName);

        if (tFileNameStrLen != EV1_RFD_UPDATE_FILE_NAME_LEN)
        {
            break;
        }

        // Verify that the first character is 'U'
        if (pcFileName[0] != EV1_RFD_UPDATE_FILE_START_CHAR)
        {
            break;
        }

        // Parse the first three characters into a number
        for (un8Idx = 0; un8Idx < EV1_RFD_UPDATE_VER_STRING_LEN; un8Idx++)
        {
            // Offsetting un8Idx by one to account for the first char
            acBuffer[un8Idx] = pcFileName[1 + un8Idx];
        }

        acBuffer[EV1_RFD_UPDATE_VER_STRING_LEN] = '\0';

        *ptBaseVersion = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);


        // Parse the last three characters into a number
        for (un8Idx = 0; un8Idx < EV1_RFD_UPDATE_VER_STRING_LEN; un8Idx++)
        {
            // Offsetting un8Idx by one to account for the first char
            // and the first version number
            acBuffer[un8Idx] =
                pcFileName[1 + EV1_RFD_UPDATE_VER_STRING_LEN + un8Idx];
        }

        acBuffer[EV1_RFD_UPDATE_VER_STRING_LEN] = '\0';

        *ptNewVersion = (RFD_UPDATE_VERSION)
            atoi((const char *)acBuffer);

        bResult = TRUE;
    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  eRFDUpdateNeeded
 *
 *  This function informs the RFD_INTERFACE if an incoming RFD update
 *  is needed by this client.
 *
 *****************************************************************************/
static RFD_FILE_STATUS_ENUM eRFDUpdateNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        )
{
    RFD_FILE_STATUS_ENUM eStatus = RFD_FILE_STATUS_INVALID;

    if ( pvCallbackArg != NULL )
    {
        EV1_RFD_CTRL_STRUCT *psRFDCtrl;
        RFD_UPDATE_VERSION tBaseVersion = 0;
        BOOLEAN bOk = FALSE;

        psRFDCtrl = (EV1_RFD_CTRL_STRUCT *)pvCallbackArg;

        // Get the version info from the file name
        bOk = bExtractVersionFromName(
            pcFileName, &tBaseVersion, ptFileVersion);

        // Did we extract that information without error?
        if ( bOk == TRUE )
        {
            // Ignoring Base Version according to SX-9845-0221 (UIRR) ver 2.2
            // and SX-9845-0219 (Protocol) ver 2.2.
            // The update is applicable if new version is just different 
            // from the current one.
            eStatus = (psRFDCtrl->tCurrentVersion != *ptFileVersion) ?
                RFD_FILE_STATUS_APPLICABLE : RFD_FILE_STATUS_NOT_NEEDED;
        }
    }

    return eStatus;
}

/*****************************************************************************
*
*   vReleaseChargerTypeTable
*
*****************************************************************************/
static void vReleaseChargerTypeTable(
    void *pvData
        )
{
    size_t tIndex;
    EV1_CHARGER_TYPE_TABLE_STRUCT *psTable = 
        (EV1_CHARGER_TYPE_TABLE_STRUCT *)pvData;

    if (pvData == NULL)
    {
        return;
    }

    for (tIndex = 0; tIndex < EV1_CHGTYPE_MAX_MAPPINGS; tIndex++)
    {
        if (psTable->asMappings[tIndex].hLongText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psTable->asMappings[tIndex].hLongText);
        }

        if (psTable->asMappings[tIndex].hShortText != STRING_INVALID_OBJECT)
        {
            STRING_vDestroy(psTable->asMappings[tIndex].hShortText);
        }
    }

    OSAL.vLinkedListMemoryFree(psTable);

    return;
}

