/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  Electronic Program Guide (EPG) dataservice.
 *  This module includes EPG PARSER implementation for Protocol Version 1.
 *  This module implements interface API to parse received EPG Access Units.
 *
 ******************************************************************************/

#include "standard.h"
#include "osal.h"
#include "sms.h"
#include "sms_api.h"
#include "sms_obj.h"
#include "sms_api_debug.h"
#include "ds_util.h"
#include "_epg_pvn1.h"

/*****************************************************************************
 PUBLIC FUNCTIONS
 *****************************************************************************/

/*****************************************************************************
 *
 *  hEpgIntfInit
 *
 *****************************************************************************/
static EPG_PARSER_INTERFACE_OBJECT hEpgIntfInit (
        EPG_CLIENT_INTERFACE_STRUCT *psClientIntf,
        UN8 un8NumSegments,
        UN32 un32Options
            )
{
    EPG_PARSER_STRUCT *psEpgParser = EPG_PARSER_INTERFACE_INVALID_OBJECT;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN">> hEpgIntfInit(%u)\n"PC_RESET, un8NumSegments);

    do
    {
        // Check arguments
        if (psClientIntf == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: EPG Client interface is not defined.\n"PC_RESET);
            break;
        }

        if (psClientIntf->hClientObj == EPG_CLIENT_INTERFACE_INVALID_OBJECT ||
            psClientIntf->bProcessParserEvent == NULL ||
            psClientIntf->bProcessCompressedTextPayload == NULL ||
            psClientIntf->vSaveAuData == NULL ||
            psClientIntf->vSegmentGridCompleted == NULL ||
            psClientIntf->vSegmentTextCompleted == NULL ||
            psClientIntf->vWholeSegmentCompleted == NULL ||
            psClientIntf->vGetVersion == NULL ||
            psClientIntf->vVersionChanged == NULL ||
            psClientIntf->vRestartGridProcessing == NULL ||
            psClientIntf->vRestartTextProcessing == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: EPG Client interface is not initialized.\n"PC_RESET);
            break;
        }

        if (un8NumSegments == 0)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Number of segments must not be 0.\n"PC_RESET);
            break;
        }
        else if (un8NumSegments > EPG_MAX_TOTAL_NUMBER_OF_SEGMENTS)
        {
            un8NumSegments = EPG_MAX_TOTAL_NUMBER_OF_SEGMENTS;
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"WARNING: Number of segments is set to %u.\n"PC_RESET,
                un8NumSegments);
        }

        // Check if alternative options are set
        // and perform appropriate reinitialization.
        if ((un32Options & EPG_PARSER_OPTION_CHECK_SEG_VERSION) != 0
            &&  (un32Options & EPG_PARSER_OPTION_MEMORY_SAVING) != 0)
        {
            // If EPG_PARSER_OPTION_CHECK_SEG_VERSION is set,
            // EPG_PARSER_OPTION_MEMORY_SAVING has to be ignored.
            un32Options &= ~EPG_PARSER_OPTION_MEMORY_SAVING;
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"WARNING: MEMORY_SAVING option is ignored.\n"PC_RESET);
        }

        // Allocate required resources
        psEpgParser = psEpgParserCreate(un8NumSegments);
        if (psEpgParser == NULL)
        {
            break;
        }

        // Initialize required attributes
        psEpgParser->psEpgClient = psClientIntf;
        psEpgParser->un8NumSegmentsToRecieve = un8NumSegments;
        psEpgParser->un32Options = un32Options;
        if ((un32Options & EPG_PARSER_OPTION_ACCEPT_NEW_VERSION_ONLY) != 0)
        {
            psEpgParser->bAcceptNewVersionOnly = TRUE;
        }

    } while (FALSE);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< hEpgIntfInit(): %p\n"PC_RESET, psEpgParser);
    return (EPG_PARSER_INTERFACE_OBJECT) psEpgParser;
}

/*****************************************************************************
 *
 *  vEpgIntfUninit
 *
 *****************************************************************************/
static void vEpgIntfUninit (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj
            )
{
    BOOLEAN bResult = FALSE;
    EPG_PARSER_STRUCT *psEpgParser = (EPG_PARSER_STRUCT *) hEpgInterfaceObj;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN">> vEpgIntfUninit()\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"vEpgIntfUninit(): Try to lock EPG interface object...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgParser, OSAL_OBJ_TIMEOUT_NONE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"Failed to lock EPG interface object (%p) during deinitialization!\n"PC_RESET,
               psEpgParser);
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_BLUE"vEpgIntfUninit(): --> EPG interface object locked.\n"PC_RESET);
    }

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"DESTROY EPG INTERFACE OBJECT...\n"PC_RESET);
    vEpgParserDestroy(psEpgParser);
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"vEpgIntfUninit(): EPG INTERFACE OBJECT DESTROYED.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< vEpgIntfUninit()\n"PC_RESET);
    return;
}

/*****************************************************************************
 *
 *  eEpgIntfProcessMessage
 *
 *****************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgIntfProcessMessage (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        OSAL_BUFFER_HDL hPayload,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            )
{
    BOOLEAN bResult = TRUE;
    EPG_PARSER_STRUCT *psEpgParser = (EPG_PARSER_STRUCT *) hEpgInterfaceObj;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX,
        6, PC_GREEN">> eEpgIntfProcessMessage()\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"bEpgIntfProcessMessage(): Try to lock EPG interface object...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgParser, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"ERROR: Failed to lock EPG interface object (%p).\n"PC_RESET, psEpgParser);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_GREEN"<< eEpgIntfProcessMessage(): %u\n"PC_RESET, EPG_RET_FAIL);
        return EPG_RET_FAIL;
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_BLUE"bEpgIntfProcessMessage(): --> EPG interface object locked.\n"PC_RESET);
    }

    // Set Cancellation Callback
    psEpgParser->bCancellationCallback = bCancellationCallback;
    psEpgParser->pvCancellationCallbackArg = pvCancellationCallbackArg;

    // Start processing
    eResult = eEpgParserProcessMessage(psEpgParser, hPayload);

    SMSO_vUnlock((SMS_OBJECT) psEpgParser);
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"bEpgIntfProcessMessage(): <-- EPG interface object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< eEpgIntfProcessMessage(): %u\n"PC_RESET, eResult);

    return eResult;
}

/*****************************************************************************
 *
 *  eEpgIntfLoadAuData
 *
 *****************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgIntfLoadAuData (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        EPG_AU_INFO_STRUCT *psAuData,
        CANCELLATION_CHECK_CALLBACK bCancellationCallback,
        void *pvCancellationCallbackArg
            )
{
    BOOLEAN bResult = TRUE;
    EPG_PARSER_STRUCT *psEpgParser = (EPG_PARSER_STRUCT *) hEpgInterfaceObj;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN">> eEpgIntfLoadAuData()\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfLoadAuData(): Try to lock EPG interface object...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgParser, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"ERROR: Failed to lock EPG interface object (%p).\n"PC_RESET, psEpgParser);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_GREEN"<< eEpgIntfLoadAuData(): %u\n"PC_RESET, EPG_RET_FAIL);
        return EPG_RET_FAIL;
    }
    else
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_BLUE"eEpgIntfLoadAuData(): --> EPG interface object locked.\n"PC_RESET);
    }

    do
    {
        // Check input
        if (psAuData == NULL ||
            psAuData->pvData == NULL ||
            psAuData->tDataSize == 0)
        {
            eResult = EPG_RET_INVALID_DATA;
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"PROCESS AU %u:%u\n"PC_RESET,
            psAuData->un8SegmentNumber, psAuData->un32AuNumber);

        // Set Cancellation Callback
        psEpgParser->bCancellationCallback = bCancellationCallback;
        psEpgParser->pvCancellationCallbackArg = pvCancellationCallbackArg;

        // Copy data to temporary buffer
        bResult = bEpgParserCopyDataToTempBuffer(psEpgParser,
                                                 psAuData->pvData,
                                                 psAuData->tDataSize);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to copy Saved AU to temp buffer.\n"PC_RESET);
            eResult = EPG_RET_FAIL;
            break;
        }

        // Process AU
        eResult = eEpgParserProcessGridAu(psEpgParser,
                                          psAuData->un8SegmentNumber,
                                          psEpgParser->hTempBuffer);

    } while (FALSE);

    SMSO_vUnlock((SMS_OBJECT) psEpgParser);
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfLoadAuData(): <-- EPG interface object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< eEpgIntfLoadAuData(): %u\n"PC_RESET, eResult);

    return eResult;
}

/*****************************************************************************
 *
 *  eEpgIntfGetOptions
 *
 *****************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgIntfGetOptions (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 *pun32Options
            )
{
    BOOLEAN bResult = FALSE;
    EPG_PARSER_STRUCT *psEpgParser = (EPG_PARSER_STRUCT *) hEpgInterfaceObj;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN">> eEpgIntfGetOptions()\n"PC_RESET);

    // Lock EPG interface object
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfGetOptions(): Try to lock EPG interface object...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgParser, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"Failed to lock EPG interface object (%p).\n"PC_RESET, psEpgParser);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_GREEN"<< eEpgIntfGetOptions(): %u\n"PC_RESET, EPG_RET_FAIL);
        return EPG_RET_FAIL;
    }
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfGetOptions(): --> EPG interface object locked.\n"PC_RESET);

    *pun32Options = psEpgParser->un32Options;

    SMSO_vUnlock((SMS_OBJECT) psEpgParser);
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfGetOptions(): <-- EPG interface object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< eEpgIntfGetOptions(): %u\n"PC_RESET, EPG_RET_OK);
    return EPG_RET_OK;
}

/*****************************************************************************
 *
 *  eEpgIntfSetOptions
 *
 *****************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgIntfSetOptions (
        EPG_PARSER_INTERFACE_OBJECT hEpgInterfaceObj,
        UN32 un32Options
            )
{
    BOOLEAN bResult = FALSE;
    EPG_PARSER_STRUCT *psEpgParser = (EPG_PARSER_STRUCT *) hEpgInterfaceObj;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN">> eEpgIntfSetOptions(0x%X)\n"PC_RESET,
        un32Options);

    // Lock EPG interface object
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfSetOptions(): Try to lock EPG interface object...\n"PC_RESET);
    bResult = SMSO_bLock((SMS_OBJECT) psEpgParser, OSAL_OBJ_TIMEOUT_INFINITE);
    if (bResult == FALSE)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"Failed to lock EPG interface object (%p).\n"PC_RESET,
            psEpgParser);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
            PC_GREEN"<< eEpgIntfSetOptions(): %u\n"PC_RESET,
            EPG_RET_FAIL);
        return EPG_RET_FAIL;
    }
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfSetOptions(): --> EPG interface object locked.\n"PC_RESET);

    if (psEpgParser->un32Options != un32Options)
    {
        // Check if alternative options are set
        // and perform appropriate reinitialization.
        if ((un32Options & EPG_PARSER_OPTION_CHECK_SEG_VERSION) != 0
            &&  (un32Options & EPG_PARSER_OPTION_MEMORY_SAVING) != 0)
        {
            // If EPG_PARSER_OPTION_CHECK_SEG_VERSION is set,
            // EPG_PARSER_OPTION_MEMORY_SAVING should be ignored.
            un32Options &= ~EPG_PARSER_OPTION_MEMORY_SAVING;
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"WARNING: MEMORY_SAVING option is ignored.\n"PC_RESET);
        }

        // Check if MEMORY_SAVING option is changed
        bResult =
            (((psEpgParser->un32Options ^ un32Options) & EPG_PARSER_OPTION_MEMORY_SAVING) != 0);

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            PC_GREEN"Parser Options are changed from 0x%X to 0x%X.\n"PC_RESET,
               psEpgParser->un32Options, un32Options);

        // Change options
        psEpgParser->un32Options = un32Options;

        // If MEMORY_SAVING option is changed, restart processing.
        if (bResult == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"MEMORY_SAVING option is changed. Restart processing.\n"PC_RESET);

            // Clean all segments to restart processing from the beginning
            vEpgParserCleanAllSegments(psEpgParser);

            // Report about processing restart
            psEpgParser->psEpgClient->vRestartGridProcessing(psEpgParser->psEpgClient->hClientObj);
            psEpgParser->psEpgClient->vRestartTextProcessing(psEpgParser->psEpgClient->hClientObj);
        }
    }

    // Following option should be processed anyway
    if ((un32Options & EPG_PARSER_OPTION_ACCEPT_NEW_VERSION_ONLY) != 0)
    {
        psEpgParser->bAcceptNewVersionOnly = TRUE;
    }

    SMSO_vUnlock((SMS_OBJECT) psEpgParser);
    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_BLUE"eEpgIntfSetOptions(): <-- EPG interface object unlocked.\n"PC_RESET);

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"<< eEpgIntfSetOptions(): %u\n"PC_RESET,
        EPG_RET_OK);
    return EPG_RET_OK;
}

/*****************************************************************************
 PRIVATE FUNCTIONS
 *****************************************************************************/

/* Utility functions */

/*******************************************************************************
 *
 *  vEpgParserCleanParserEvent
 *
 *******************************************************************************/
static void vEpgParserCleanParserEvent (
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    if (psParserEvent->aun32AdditionalTimes != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psParserEvent->aun32AdditionalTimes);
        psParserEvent->aun32AdditionalTimes = NULL;
    }

    if (psParserEvent->atTopicId != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psParserEvent->atTopicId);
        psParserEvent->atTopicId = NULL;
    }

    OSAL.bMemSet(psParserEvent, 0, sizeof(EPG_PARSER_EVENT_STRUCT));

    return;
}

/*******************************************************************************
 *
 *  vEpgParserCleanAu
 *
 *******************************************************************************/
static void vEpgParserCleanAu (
       EPG_AU_DATA_STRUCT *psAuData
            )
{
    if (psAuData->pvAuPayload != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) psAuData->pvAuPayload);
        psAuData->pvAuPayload = NULL;
    }

    OSAL.bMemSet(psAuData, 0, sizeof(EPG_AU_DATA_STRUCT));

    return;
}

/*******************************************************************************
 *
 *  eDurationCodeFromValue
 *
 *******************************************************************************/
static EPG_PEM_DURATION_CODE_ENUM eDurationCodeFromValue (
        UN32 un32DurCodeValue
            )
{
    EPG_PEM_DURATION_CODE_ENUM eDurationCode = EPG_DURATION_CODE_UNDEFINED;

    switch (un32DurCodeValue)
    {
    case EPG_PEM_DUR_30MIN:
        eDurationCode = EPG_DURATION_CODE_30MIN;
        break;
    case EPG_PEM_DUR_1HOUR:
        eDurationCode = EPG_DURATION_CODE_1HOUR;
        break;
    case EPG_PEM_DUR_2HOURS:
        eDurationCode = EPG_DURATION_CODE_2HOURS;
        break;
    case EPG_PEM_DUR_3HOURS:
        eDurationCode = EPG_DURATION_CODE_3HOURS;
        break;
    case EPG_PEM_DUR_4HOURS:
        eDurationCode = EPG_DURATION_CODE_4HOURS;
        break;
    case EPG_PEM_DUR_5HOURS:
        eDurationCode = EPG_DURATION_CODE_5HOURS;
        break;
    case EPG_PEM_DUR_END_OF_SEGMENT:
        eDurationCode = EPG_DURATION_CODE_END_OF_SEGMENT;
        break;
    case EPG_PEM_DUR_ESCAPE_TO_ODUR:
        eDurationCode = EPG_DURATION_CODE_ESCAPE_TO_ODUR;
        break;
    default:
        eDurationCode = EPG_DURATION_CODE_UNDEFINED;
        break;
    };

    return eDurationCode;
}

/* General */

/*****************************************************************************
 *
 *  psEpgParserCreate
 *
 *****************************************************************************/
static EPG_PARSER_STRUCT *psEpgParserCreate (
        UN8 un8NumSegments
            )
{
    EPG_PARSER_STRUCT *psEpgParser = NULL;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    do
    {
        // Create the object (initially locked)
        psEpgParser =
            (EPG_PARSER_STRUCT *) SMSO_hCreate(EPG_INTF_OBJECT_NAME":Parser",
                                               sizeof(EPG_PARSER_STRUCT),
                                               SMS_INVALID_OBJECT, // No parent
                                               TRUE); // Self-locked
        if (psEpgParser == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create EPG interface object.\n"PC_RESET);
            break;
        }

        // Create segments array
        psEpgParser->asSegments =
            (EPG_SEGMENT_STRUCT *) SMSO_hCreate(EPG_INTF_OBJECT_NAME":SegmentsArray",
                                                sizeof(EPG_SEGMENT_STRUCT) * un8NumSegments,
                                                (SMS_OBJECT) psEpgParser, FALSE);
        if (psEpgParser->asSegments == NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create segments array.\n"PC_RESET);
            break;
        }

        psEpgParser->un8NumSegmentsToRecieve = un8NumSegments;

        // Request access to ISO 3309 CRC32
        eReturnCode = OSAL.eGetCRC(&(psEpgParser->hCRC), OSAL_CRC_TYPE_ISO3309_CRC32);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: CRC request error: %s\n"PC_RESET,
                OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }

        // Relinquish ownership
        SMSO_vUnlock((SMS_OBJECT) psEpgParser);

        // Success
        return psEpgParser;

    } while (FALSE);

    // Error case:
    if (psEpgParser != NULL)
    {
        vEpgParserDestroy(psEpgParser);
    }
    return NULL;
}

/*****************************************************************************
 *
 *  vEpgParserDestroy
 *
 *****************************************************************************/
static void vEpgParserDestroy (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_UNKNOWN;

    // Delete Temp Buffer
    vEpgParserDestroyTempBuffer(psEpgParser);

    // Release the CRC
    eReturnCode = OSAL.eReleaseCRC(psEpgParser->hCRC);
    if (eReturnCode != OSAL_SUCCESS)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"WARNING: Failed to release CRC.\n"PC_RESET);
    }

    // Delete segments
    vEpgParserDestroyAllSegments(psEpgParser);

    // Destroy object itself
    SMSO_vDestroy((SMS_OBJECT) psEpgParser);

    return;
}

/*****************************************************************************
 *
 *  vEpgParserDestroyAllSegments
 *
 *****************************************************************************/
static void vEpgParserDestroyAllSegments (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    if (psEpgParser->asSegments != NULL)
    {
        // Delete segments data
        vEpgParserCleanAllSegments(psEpgParser);

        // Delete segments array
        SMSO_vDestroy((SMS_OBJECT) psEpgParser->asSegments);
        psEpgParser->asSegments = NULL;
    }

    return;
}

/*****************************************************************************
 *
 *  vEpgParserCleanAllSegments
 *
 *****************************************************************************/
static void vEpgParserCleanAllSegments (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    UN8 un8SegmentNumber = 0;

    for (un8SegmentNumber = 0;
         un8SegmentNumber < psEpgParser->un8NumSegmentsToRecieve;
         un8SegmentNumber++)
    {
        vEpgParserCleanSegmentGrid(psEpgParser, un8SegmentNumber);
        vEpgParserCleanSegmentText(psEpgParser, un8SegmentNumber);
    }

    psEpgParser->un8NumSegmentsGridCollected = 0;
    psEpgParser->un8NumSegmentsTextCollected = 0;

    return;
}

/*****************************************************************************
 *
 *  vEpgParserCleanAllSegmentsGrid
 *
 *****************************************************************************/
static void vEpgParserCleanAllSegmentsGrid (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    UN8 un8SegmentNumber = 0;

    for (un8SegmentNumber = 0;
         un8SegmentNumber < psEpgParser->un8NumSegmentsToRecieve;
         un8SegmentNumber++)
    {
        vEpgParserCleanSegmentGrid(psEpgParser, un8SegmentNumber);
    }

    psEpgParser->un8NumSegmentsGridCollected = 0;

    return;
}

/*****************************************************************************
 *
 *  vEpgParserCleanSegmentGrid
 *
 *****************************************************************************/
static void vEpgParserCleanSegmentGrid (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber
            )
{
    EPG_SEG_GRID_DATA_STRUCT *psSegGridData = NULL;
    UN8 un8AuNum = 0;

    psSegGridData = &(psEpgParser->asSegments[un8SegmentNumber].sSegGridData);

    // Delete Grid AUs
    if (psSegGridData->asAuGridData != NULL)
    {
        for (un8AuNum = 0; un8AuNum < psSegGridData->un32NumAuTotal; un8AuNum++)
        {
            EPG_AU_DATA_STRUCT *psAuGridData = &(psSegGridData->asAuGridData[un8AuNum]);
            // Delete AU's payload
            if (psAuGridData->pvAuPayload != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psAuGridData->pvAuPayload);
                psAuGridData->pvAuPayload = NULL;
                psAuGridData->tPayloadSize = 0;
            }
        }

        // Delete AUs array
        SMSO_vDestroy((SMS_OBJECT) psSegGridData->asAuGridData);
        psSegGridData->asAuGridData = NULL;
    }
    psSegGridData->un32NumAuTotal = 0;
    psSegGridData->un32NumAuCollected = 0;

    return;
}

/*****************************************************************************
 *
 *  vEpgParserCleanSegmentText
 *
 *****************************************************************************/
static void vEpgParserCleanSegmentText (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber
            )
{
    EPG_SEG_TEXT_DATA_STRUCT *psSegTextData = NULL;
    UN8 un8AuNum = 0;

    psSegTextData = &(psEpgParser->asSegments[un8SegmentNumber].sSegTextData);

    // Delete Text AUs
    if (psSegTextData->asAuTextData != NULL)
    {
        for (un8AuNum = 0; un8AuNum < psSegTextData->sTextAuHeader.un32NumAuTotal; un8AuNum++)
        {
            EPG_AU_DATA_STRUCT *psAuTextData = &(psSegTextData->asAuTextData[un8AuNum]);
            // Delete AU's payload
            if (psAuTextData->pvAuPayload != NULL)
            {
                SMSO_vDestroy((SMS_OBJECT)psAuTextData->pvAuPayload);
                psAuTextData->pvAuPayload = NULL;
                psAuTextData->tPayloadSize = 0;
            }
        }

        // Delete AUs array
        SMSO_vDestroy((SMS_OBJECT) psSegTextData->asAuTextData);
        psSegTextData->asAuTextData = NULL;
    }
    psSegTextData->sTextAuHeader.un32NumAuTotal = 0;
    psSegTextData->un32NumAuCollected = 0;

    return;
}

/*****************************************************************************
 *
 *  bEpgParserCreateTempBuffer
 *
 *****************************************************************************/
static BOOLEAN bEpgParserCreateTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    // First ensure Block Pool is created
    if (psEpgParser->hTempBlockPool == OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eBlockPoolCreate(&(psEpgParser->hTempBlockPool),
                                            EPG_INTF_OBJECT_NAME":TempBlockPool",
                                            EPG_BLOCKPOOLBUF_SIZE,
                                            EPG_BLOCKPOOLBUF_NUM,
                                            OSAL_BLOCK_POOL_OPTION_NONE);
        if (eReturnCode != OSAL_SUCCESS)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to create Temp Block Pool.\n"PC_RESET);
            return FALSE;
        }
    }

    // Check if buffer is already allocated
    if (psEpgParser->hTempBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        // Already created
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"ERROR: Temp Buffer is already created.\n"PC_RESET);
        return FALSE;
    }

    // Allocate buffer
    psEpgParser->hTempBuffer = OSAL.hBufferAllocate(psEpgParser->hTempBlockPool,
                                                    FALSE, FALSE,
                                                    OSAL_BUFFER_ALLOCATE_OPTION_NONE);
    if (psEpgParser->hTempBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
            PC_RED"ERROR: Failed to allocate Temp Buffer.\n"PC_RESET);
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
 *
 *  vEpgParserDestroyTempBuffer
 *
 *****************************************************************************/
static void vEpgParserDestroyTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

    // Free Buffer
    if (psEpgParser->hTempBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        eReturnCode = OSAL.eBufferFree(psEpgParser->hTempBuffer);
        if (eReturnCode == OSAL_SUCCESS)
        {
            psEpgParser->hTempBuffer = OSAL_INVALID_BUFFER_HDL;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to free Temp Buffer.\n"PC_RESET);
        }
    }

    // Delete Block Pool
    if (psEpgParser->hTempBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eBlockPoolDelete(psEpgParser->hTempBlockPool);
        if (eReturnCode == OSAL_SUCCESS)
        {
            psEpgParser->hTempBlockPool = OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to delete Temp Block Pool.\n"PC_RESET);
        }
    }

    return;
}

/*******************************************************************************
 *
 * bEpgParserCopyDataToTempBuffer
 *
 *******************************************************************************/
static BOOLEAN bEpgParserCopyDataToTempBuffer (
        EPG_PARSER_STRUCT *psEpgParser,
        void *pvData,
        size_t tDataSize
            )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    BOOLEAN bResult = TRUE;
    size_t tBytesWritten = 0;

    do
    {
        // Verify input
        if (pvData == NULL || tDataSize == 0)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid data to copy to the Temp Buffer.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

        // Free Buffer
        if (psEpgParser->hTempBuffer != OSAL_INVALID_BUFFER_HDL)
        {
            eReturnCode = OSAL.eBufferFree(psEpgParser->hTempBuffer);
            if (eReturnCode == OSAL_SUCCESS)
            {
                psEpgParser->hTempBuffer = OSAL_INVALID_BUFFER_HDL;
            }
            else
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to free Temp Buffer.\n"PC_RESET);
                bResult = FALSE;
                break;
            }
        }

        // Reallocate buffer
        bResult = bEpgParserCreateTempBuffer(psEpgParser);
        if (bResult == FALSE)
        {
            break;
        }

        // Copy data
        tBytesWritten = OSAL.tBufferWriteHead(psEpgParser->hTempBuffer,
                                              pvData, tDataSize);
        if (tBytesWritten != tDataSize)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to write data to the Temp Buffer.\n"PC_RESET);
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  bEpgParserIsCancelled
 *
 *****************************************************************************/
static BOOLEAN bEpgParserIsCancelled (
        EPG_PARSER_STRUCT *psEpgParser
            )
{
    BOOLEAN bCancelled = FALSE;

    if (NULL != psEpgParser->bCancellationCallback)
    {
        bCancelled = psEpgParser->bCancellationCallback(psEpgParser->pvCancellationCallbackArg);
    }

    return bCancelled;
}

/* Message Processors */

/*****************************************************************************
 *
 *  bEpgParserExtractGridHeader
 *
 *****************************************************************************/
static BOOLEAN bEpgParserExtractGridHeader (
        OSAL_BUFFER_HDL hPayload,
        EPG_AU_GRID_HEADER *psGridHeader
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;

    do
    {
        // Series ID field size
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_SERSIZE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SERSIZE.\n"PC_RESET);
            break;
        }
        psGridHeader->un8SeriesIdSize = (UN8)un32ReadValue + 1;

        // Program ID field size
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_PROGSIZE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read PROGSIZE.\n"PC_RESET);
            break;
        }
        psGridHeader->un8ProgramIdSize = (UN8)un32ReadValue + 1;

        // Text Index field size
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_ISIZE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read ISIZE.\n"PC_RESET);
            break;
        }
        psGridHeader->un8TextIndexSize = (UN8)un32ReadValue + 1;

        // Topic field size
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_TSIZE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read TSIZE.\n"PC_RESET);
            break;
        }
        psGridHeader->un8TopicIdSize = (UN8)un32ReadValue + 1;

        // AUTOT and AUCT fields size
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_AU_HEADER_CTSIZE_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read CTSIZE.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present.
        if (un32ReadValue != UN32_MAX)
        {
            UN8 un8CtSize = (UN8)un32ReadValue + 1;

            // Number of Access Units in this segment
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                un8CtSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read AUTOT.\n"PC_RESET);
                break;
            }
            psGridHeader->un32NumAuTotal = un32ReadValue + 1;

            // AU Sequence Number
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                un8CtSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read AUCT.\n"PC_RESET);
                break;
            }
            psGridHeader->un32AuSequenceNumber = un32ReadValue;
        }
        else
        {
            psGridHeader->un32NumAuTotal = 1;
            psGridHeader->un32AuSequenceNumber = 0;
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  bEpgParserExtractTextHeader
 *
 *****************************************************************************/
static BOOLEAN bEpgParserExtractTextHeader (
        OSAL_BUFFER_HDL hPayload,
        EPG_AU_TEXT_HEADER *psTextHeader
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;
    size_t tBitsRead = 0;
    size_t tStuffBits = 0;

    do
    {
        // Number of text strings in compressed file
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_NOSTRING_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read NOSTRING.\n"PC_RESET);
            break;
        }
        psTextHeader->un16StringsCount = (UN16)un32ReadValue;

        // Size of uncompressed file in bytes
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_FILESIZE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read FILESIZE.\n"PC_RESET);
            break;
        }
        psTextHeader->un32DecompressedFileSize = un32ReadValue;

        // Size of AUTOT and AUCT
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_AU_HEADER_CTSIZE_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read CTSIZE.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present
        if (un32ReadValue != UN32_MAX)
        {
            UN8 un8CtSize = (UN8)un32ReadValue + 1;

            // Number of Access Units in this segment
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                un8CtSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read AUTOT.\n"PC_RESET);
                break;
            }
            psTextHeader->un32NumAuTotal = un32ReadValue + 1;

            // AU Sequence Number
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                un8CtSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read AUCT.\n"PC_RESET);
                break;
            }
            psTextHeader->un32AuSequenceNumber = un32ReadValue;

            tBitsRead = EPG_AU_HEADER_CTSIZE_BITLEN + (2 * un8CtSize);
        }
        else
        {
            psTextHeader->un32NumAuTotal = 1;
            psTextHeader->un32AuSequenceNumber = 0;
        }
        tBitsRead += EPG_CT_HEADER_BASE_BITLEN;

        // Read stuffing bits if required
        if ((tBitsRead % BITS_IN_BYTE) > 0)
        {
            tStuffBits = BITS_IN_BYTE - (tBitsRead % BITS_IN_BYTE);
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                (UN8) tStuffBits);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read stuffing bits.\n"PC_RESET);
                break;
            }
        }

    } while (FALSE);

    return bResult;
}

/*****************************************************************************
 *
 *  n8EpgParserExtractSid
 *
 *  Function returns: one of EPG_PEB_HEADER_SINC_CODE_... constants - if success
 *                    negative value - if failed
 *
 *  Value pointed by pun16Sid is modified depending on SINC code.
 *
 *****************************************************************************/
static N8 n8EpgParserExtractSid(OSAL_BUFFER_HDL hPayload,
                                SERVICE_ID *ptSid)
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;
    N8 n8Sinc = 0;

    do
    {
        // Service ID increment
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_PEB_HEADER_SINC_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SINC.\n"PC_RESET);
            n8Sinc = -1;
            break;
        }
        n8Sinc = (N8)un32ReadValue;

        // Set new SID value
        switch (n8Sinc)
        {
        case EPG_PEB_HEADER_SINC_CODE_ABSOLUTE:
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                EPG_PEB_HEADER_SID_BITLEN);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read SID.\n"PC_RESET);
                n8Sinc = -1;
                break;
            }
            *ptSid = (UN16)un32ReadValue;
            break;

        case EPG_PEB_HEADER_SINC_CODE_INC_OF_ONE:
            (*ptSid) += 1;
            break;

        case EPG_PEB_HEADER_SINC_CODE_INC_OF_TWO:
            (*ptSid) += 2;
            break;

        case EPG_PEB_HEADER_SINC_CODE_END_OF_LIST:
            break;

        default:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid SINC value.\n"PC_RESET);
            n8Sinc = -1;
            break;
        }

    } while (FALSE);

    return n8Sinc;
}

/*******************************************************************************
 *
 *  vEpgParserParseProgramFlags
 *
 *******************************************************************************/
static void vEpgParserParseProgramFlags (
        UN16 un16Flags,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{

    psParserEvent->tProgramFlags = EPG_PROGRAM_UNDEFINED;
    psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_UNDEFINED;

    // Set Program Flags
    if ((un16Flags & EPG_PEM_FLAGS_FEATURED) == EPG_PEM_FLAGS_FEATURED)
    {
        psParserEvent->tProgramFlags |= EPG_PROGRAM_FEATURED;
    }
    if ((un16Flags & EPG_PEM_FLAGS_HIGHLIGHTED) == EPG_PEM_FLAGS_HIGHLIGHTED)
    {
        psParserEvent->tProgramFlags |= EPG_PROGRAM_HIGHLIGHTED;
    }
    if ((un16Flags & EPG_PEM_FLAGS_LIVE) == EPG_PEM_FLAGS_LIVE)
    {
        psParserEvent->tProgramFlags |= EPG_PROGRAM_LIVE;
    }
    if ((un16Flags & EPG_PEM_FLAGS_NEW_THIS) == EPG_PEM_FLAGS_NEW_THIS)
    {
        psParserEvent->tProgramFlags |= EPG_PROGRAM_NEW_THIS;
    }

    // Set recording options
    switch ((un16Flags & EPG_PEM_FLAGS_RECORD_BIT_MASK))
    {
        case EPG_PEM_FLAGS_RECORD_ALWAYS:
            psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_ALWAYS;
            break;
        case EPG_PEM_FLAGS_RECORD_RESTRICTED:
            psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_RESTRICTED;
            break;
        case EPG_PEM_FLAGS_RECORD_APPROVED:
            psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_APPROVED;
            break;
        case EPG_PEM_FLAGS_RECORD_NONE:
            psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_NONE;
            break;
    }

    return;
}

/*******************************************************************************
 *
 *  bEpgParserExtractProgramFlags
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractProgramFlags (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadVal = 0;
    UN16 un16ProgramFlags = EPG_PEM_FLAGS_UNDEFINED;
    BOOLEAN bValid = FALSE;

    do
    {
        // FLAGS1
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadVal,
                                                    hPayload,
                                                    EPG_PEM_FLAGS1_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read FLAGS1.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present.
        if (un32ReadVal == UN32_MAX)
        {
            un16ProgramFlags = EPG_PEM_FLAGS_UNDEFINED;
        }
        else
        {
            un16ProgramFlags = (UN16) un32ReadVal;
            bValid = TRUE;
        }

        // FLAGS2
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadVal,
                                                    hPayload,
                                                    EPG_PEM_FLAGS2_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read FLAGS2.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present.
        if (un32ReadVal != UN32_MAX)
        {
            un16ProgramFlags =
                (UN16)(un16ProgramFlags | (un32ReadVal << EPG_PEM_FLAGS1_BITLEN));
            bValid = TRUE;
        }

        if (bValid == TRUE)
        {
            vEpgParserParseProgramFlags(un16ProgramFlags,
                                        psParserEvent);
        }
        else
        {
            psParserEvent->tProgramFlags = EPG_PROGRAM_INVALID_FLAGS;
            psParserEvent->eRecordingOption = EPG_PROGRAM_RECORD_UNDEFINED;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Read Flags = %x\n", un16ProgramFlags);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Program Flags = %x\n", psParserEvent->tProgramFlags);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Recording Option = %u\n", psParserEvent->eRecordingOption);

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractDuration
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractDuration (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent,
        BOOLEAN bMandatory
            )
{
    BOOLEAN bResult = TRUE;
    UN32 un32ReadValue = 0;

    do
    {
        if (bMandatory == FALSE)
        {
            bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                        hPayload,
                                                        EPG_PEM_DUR_BITLEN,
                                                        UN32_MAX);
        }
        else
        {
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                EPG_PEM_DUR_BITLEN);
        }
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read DUR.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present.
        if (un32ReadValue == UN32_MAX)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 15,
                "Duration code is not defined.\n");
            psParserEvent->eDurationCode = EPG_DURATION_CODE_UNDEFINED;
            psParserEvent->un32OptionalDuration = 0;
            break;
        }
        psParserEvent->eDurationCode = eDurationCodeFromValue(un32ReadValue);

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Duration code = %u -> %u\n",
            un32ReadValue,
            psParserEvent->eDurationCode);

        if (psParserEvent->eDurationCode == EPG_DURATION_CODE_ESCAPE_TO_ODUR)
        {
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                EPG_PEM_ODUR_BITLEN);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read ODUR.\n"PC_RESET);
                break;
            }
            // Convert value from LSB to seconds
            psParserEvent->un32OptionalDuration =
                un32ReadValue * EPG_PEM_ODUR_LSB_VALUE * SECONDS_IN_MINUTE;

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
                "Optional Duration = %u lsb (%u seconds)\n",
                psParserEvent->un32OptionalDuration / (EPG_PEM_ODUR_LSB_VALUE * SECONDS_IN_MINUTE),
                psParserEvent->un32OptionalDuration);
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractAdditionalTimes
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractAdditionalTimes (
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = TRUE;
    UN32 un32ReadValue = 0;
    UN32 un32Cnt = 0;

    do
    {
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_STNUM_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read STNUM.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present
        if (un32ReadValue == UN32_MAX)
        {
            psParserEvent->un8NumAdditionalStartTimes = 0;
            break;
        }
        psParserEvent->un8NumAdditionalStartTimes = (UN8)un32ReadValue + 1;

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Additional start times count = %d\n",
            psParserEvent->un8NumAdditionalStartTimes);

        // Check if Start Times array is not yet allocated
        if (psParserEvent->aun32AdditionalTimes == NULL)
        {
            // Create Start Times array
            psParserEvent->aun32AdditionalTimes =
                (UN32 *) SMSO_hCreate(EPG_INTF_OBJECT_NAME":TimesArray",
                                      sizeof(UN32) * psParserEvent->un8NumAdditionalStartTimes,
                                      SMS_INVALID_OBJECT, FALSE);
            if (psParserEvent->aun32AdditionalTimes == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to create Start Times array.\n"PC_RESET);
                bResult = FALSE;
                break;
            }
        }

        // Read all additional start times of the program
        for (un32Cnt = 0; un32Cnt < psParserEvent->un8NumAdditionalStartTimes; un32Cnt++)
        {
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                EPG_PEM_START_BITLEN);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read START time.\n"PC_RESET);
                break;
            }
            // Convert from LSB to seconds
            psParserEvent->aun32AdditionalTimes[un32Cnt] =
                un32ReadValue * EPG_PEM_START_LSB_VALUE * SECONDS_IN_MINUTE;

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
                "Additional start time #%u = %u\n",
                un32Cnt,
                psParserEvent->aun32AdditionalTimes[un32Cnt] / (EPG_PEM_START_LSB_VALUE * SECONDS_IN_MINUTE),
                psParserEvent->aun32AdditionalTimes[un32Cnt]);
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractTopics
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractTopics (
        EPG_AU_GRID_HEADER *psGridHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = TRUE;
    UN32 un32ReadValue = 0;
    UN32 un32Cnt = 0;

    do
    {
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_TCNT_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read TCNT.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present
        if (un32ReadValue == UN32_MAX)
        {
            psParserEvent->un8NumTopics = 0;
            break;
        }
        psParserEvent->un8NumTopics = (UN8)un32ReadValue + 1;

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
            "Topics IDs count = %d, with size = %d\n",
            psParserEvent->un8NumTopics,
            psGridHeader->un8TopicIdSize);

        // Check if Topics Id array is not yet allocated
        if (psParserEvent->atTopicId == NULL)
        {
            // Create Topics Id array
            psParserEvent->atTopicId =
                (TOPIC_ID *) SMSO_hCreate(EPG_INTF_OBJECT_NAME":TopicIdArray",
                                          sizeof(TOPIC_ID) * psParserEvent->un8NumTopics,
                                          SMS_INVALID_OBJECT, FALSE);
            if (psParserEvent->atTopicId == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to create Topics Id array.\n"PC_RESET);
                bResult = FALSE;
                break;
            }
        }

        for (un32Cnt = 0; un32Cnt < psParserEvent->un8NumTopics; un32Cnt++)
        {
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                psGridHeader->un8TopicIdSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read TID.\n"PC_RESET);
                break;
            }

            psParserEvent->atTopicId[un32Cnt] = (TOPIC_ID)un32ReadValue;
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
                "Topic #%d ID = %d\n",
                un32Cnt, psParserEvent->atTopicId[un32Cnt]);
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractExtraData
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractExtraData (
        OSAL_BUFFER_HDL hPayload,
        void **ppvDst,
        UN16 *pun16Size
            )
{
    BOOLEAN bResult = TRUE;
    UN32 un32ReadValue = 0;
    UN16 un16ExtDataSize = 0;
    void *pvBuf = NULL;

    if (pun16Size != NULL)
    {
        *pun16Size = 0;
    }

    do
    {
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_EXTCNT_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read EXTCNT.\n"PC_RESET);
            break;
        }
        // If default value is set, the field is not present.
        if (un32ReadValue == UN32_MAX)
        {
            // Field is not present, so ignore it.
            break;
        }
        un16ExtDataSize = ((UN16)un32ReadValue + 1) * EPG_PEM_EXTCNT_MULTIPLIER;

        if (ppvDst != NULL)
        {
            // Allocate buffer
            pvBuf = (void *) SMSO_hCreate(EPG_INTF_OBJECT_NAME":ExtraData",
                                          un16ExtDataSize,
                                          SMS_INVALID_OBJECT, FALSE);
            if (pvBuf == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to allocate Extra Data buffer.\n"PC_RESET);
                break;
            }
        }

        bResult = DS_UTIL_bExtractBytes(pvBuf, hPayload, un16ExtDataSize);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read EXTDATA.\n"PC_RESET);
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 13,
            "Additional data size = %d\n", un16ExtDataSize);

        if (ppvDst != NULL)
        {
            *ppvDst = pvBuf;
            pvBuf = NULL;
        }

        if (pun16Size != NULL)
        {
           *pun16Size = un16ExtDataSize;
        }

    } while (FALSE);

    if (pvBuf != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pvBuf);
    }

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserProcessScheduleVersion
 *
 *  Return value: TRUE - if a version is accepted, FALSE - otherwise
 *
 *******************************************************************************/
static BOOLEAN bEpgParserProcessScheduleVersion (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SCHEDULE_VERSION_STRUCT *psVersion
            )
{
    EPG_SCHEDULE_VERSION_STRUCT sCurrentVersion;
    BOOLEAN bVersionChanged = FALSE;
    BOOLEAN bVersionAccepted = TRUE;

    do
    {
        // Get Current Version
        psEpgParser->psEpgClient->vGetVersion(psEpgParser->psEpgClient->hClientObj,
                                              &sCurrentVersion);

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "CURRENT SCHEDULE:    EPOCH: %u   VERSION: %u\n",
            sCurrentVersion.un16Epoch, sCurrentVersion.un8Version);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "INCOMING SCHEDULE:   EPOCH: %u   VERSION: %u\n",
            psVersion->un16Epoch, psVersion->un8Version);

        // Check if Epoch is changed
        if (psVersion->un16Epoch != sCurrentVersion.un16Epoch)
        {
            bVersionChanged = TRUE;
            bVersionAccepted = TRUE;
            break;
        }

        // There can be a situation when schedule version is updated,
        // but several AUs with old version can be broadcasted after version change.
        if (psVersion->un8Version < sCurrentVersion.un8Version)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"VERSION CHANGE (%u->%u) IS IGNORED.\n"PC_RESET,
                   sCurrentVersion.un8Version, psVersion->un8Version);
            bVersionChanged = FALSE;
            bVersionAccepted = FALSE;
            break;
        }

        if (psVersion->un8Version > sCurrentVersion.un8Version)
        {
            // Check if SSV is supported.
            if ((psEpgParser->un32Options & EPG_PARSER_OPTION_CHECK_SEG_VERSION) == 0)
            {
                // SSV is not supported, so reset previous schedule.
                bVersionChanged = TRUE;
                bVersionAccepted = TRUE;
            }
            else
            {
                // SSV is supported. Version change should be processed when SSV is received.
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_BLUE"SSV IS SUPPORTED. VERSION CHANGE (%u->%u) IS IGNORED.\n"PC_RESET,
                       sCurrentVersion.un8Version, psVersion->un8Version);
                bVersionChanged = FALSE;
                bVersionAccepted = FALSE;
            }
        }

    } while (FALSE);

    if (bVersionChanged == TRUE)
    {
        // Version is changed from the current one.
        // Now next versions can be accepted.
        if (psEpgParser->bAcceptNewVersionOnly == TRUE)
        {
            psEpgParser->bAcceptNewVersionOnly = FALSE;
        }

        // Clean all segments
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            PC_BLUE"VERSION IS CHANGED. CLEAN ALL SEGMENTS.\n"PC_RESET);
        vEpgParserCleanAllSegments(psEpgParser);

        // Report about version change
        psEpgParser->psEpgClient->vVersionChanged(psEpgParser->psEpgClient->hClientObj,
                                                  psVersion);

        // Report about processing restart
        psEpgParser->psEpgClient->vRestartGridProcessing(psEpgParser->psEpgClient->hClientObj);
        psEpgParser->psEpgClient->vRestartTextProcessing(psEpgParser->psEpgClient->hClientObj);
    }
    else
    {
        // Version is not changed, check if it should be skipped.
        if (psEpgParser->bAcceptNewVersionOnly == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"VERSION IS NOT CHANGED. IGNORE.\n"PC_RESET);
            bVersionAccepted = FALSE;
        }
    }

    return bVersionAccepted;
}

/*******************************************************************************
 *
 *  eEpgParserProcessMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = FALSE;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;
    UN32 un32ReadValue = 0;

    do
    {
        // Read and verify CRC
        bResult = DS_UTIL_bIsCRCValid(psEpgParser->hCRC, hPayload, NULL);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: CRC is not valid.\n"PC_RESET);
            break;
        }

        // Cut the CRC from the buffer
        bResult = DS_UTIL_bCutCRC(hPayload);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to cut CRC.\n"PC_RESET);
            break;
        }

        // Read and verify protocol version
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_PVN_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read PVN.\n"PC_RESET);
            break;
        }

        if (un32ReadValue != EPG_PVN_VAL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"PVN is not supported.\n"PC_RESET);
            eResult = EPG_RET_NOT_SUPPORTED;
            break;
        }

        // Read and check Carousel ID
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_CARID_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read CARID.\n"PC_RESET);
            break;
        }

        switch (un32ReadValue)
        {
        case EPG_AU_HEADER_CARID_SCHEDULE_TODAY:
        case EPG_AU_HEADER_CARID_SCHEDULE_UPCOMING:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Schedule Message\n"PC_RESET);
            eResult = eEpgParserProcessScheduleMessage(psEpgParser, hPayload);
            break;

        case EPG_AU_HEADER_CARID_SCHEDULE_VERSIONING:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Schedule Segment Versioning Message\n"PC_RESET);
            eResult = eEpgParserProcessScheduleVersioningMessage(psEpgParser, hPayload);
            break;

        case EPG_AU_HEADER_CARID_PAM:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Program Announcement Message\n"PC_RESET);
            eResult = eEpgParserProcessPamMessage(psEpgParser, hPayload);
            break;

        case EPG_AU_HEADER_CARID_TAD:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Table Affinity Message\n"PC_RESET);
            eResult = eEpgParserProcessTadMessage(psEpgParser, hPayload);
            break;

        case EPG_AU_HEADER_CARID_PROFILE_CONFIG:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Profile Configuration Message\n"PC_RESET);
            eResult = eEpgParserProcessProfileConfigMessage(psEpgParser, hPayload);
            break;

        default:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"MSG TYPE: Unknown Message\n"PC_RESET);
            eResult = EPG_RET_INVALID_DATA;
            break;
        };

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessScheduleMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessScheduleMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;

    UN8 un8TypeFlag = EPG_AU_HEADER_TFLAG_GRID;
    EPG_SCHEDULE_VERSION_STRUCT sReceivedVersion;
    UN8 un8TotalNumberOfSegments = 0;
    UN8 un8SegmentNumber = 0;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;
    EPG_SEGMENT_STRUCT *psSegment = NULL;

    do
    {
        // 1) Read and process common data

        // Message type (TFLAG field)
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_TFLAG_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read TFLAG.\n"PC_RESET);
            break;
        }
        un8TypeFlag = (UN8) un32ReadValue;

        // Schedule version
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_SHVER_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SHVER.\n"PC_RESET);
            break;
        }
        sReceivedVersion.un8Version = (UN8)un32ReadValue;

        // Epoch
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_EPOCH_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read EPOCH.\n"PC_RESET);
            break;
        }
        sReceivedVersion.un16Epoch = (UN16)(un32ReadValue);

        // Total number of segments
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_NOSEG_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read NOSEG.\n"PC_RESET);
            break;
        }
        un8TotalNumberOfSegments = (UN8)un32ReadValue + 1;

        // Segment number
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_SHSEG_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SHSEG.\n"PC_RESET);
            break;
        }
        un8SegmentNumber = (UN8)un32ReadValue;

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "%s AU HEADER:      EPOCH=%u  SCHVER=%d  NOSEG=%u/%u  SCHSEG=%u\n",
            un8TypeFlag == EPG_AU_HEADER_TFLAG_GRID ? "GRID" : "TEXT",
            sReceivedVersion.un16Epoch,
            sReceivedVersion.un8Version,
            un8TotalNumberOfSegments,
            psEpgParser->un8NumSegmentsToRecieve,
            un8SegmentNumber);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "SEGMENTS COLLECTED:  GRID: %u   TEXT: %u   FULL: %u\n"PC_RESET,
            psEpgParser->un8NumSegmentsGridCollected,
            psEpgParser->un8NumSegmentsTextCollected,
            ((psEpgParser->un8NumSegmentsGridCollected <
            psEpgParser->un8NumSegmentsTextCollected) ?
            psEpgParser->un8NumSegmentsGridCollected :
            psEpgParser->un8NumSegmentsTextCollected));

        // Check segment number
        if (un8SegmentNumber >= psEpgParser->un8NumSegmentsToRecieve)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"Unnecessary segment %u is ignored.\n"PC_RESET,
                un8SegmentNumber);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Set epoch of segment 0 of incoming schedule
        sReceivedVersion.un16Epoch -= un8SegmentNumber;

        // Process version info. Result means that version is accepted.
        bResult = bEpgParserProcessScheduleVersion(psEpgParser, &sReceivedVersion);
        if (bResult == FALSE)
        {
            // The version is not accepted
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"Ignored segment %u with version %u.\n"PC_RESET,
                un8SegmentNumber, sReceivedVersion.un8Version);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Save general segment's info
        psSegment = &(psEpgParser->asSegments[un8SegmentNumber]);
        psSegment->un8SegmentVersion = sReceivedVersion.un8Version;
        psSegment->un16SegmentEpoch = sReceivedVersion.un16Epoch + un8SegmentNumber;
        psSegment->un8SegmentNumber = un8SegmentNumber;

        // 2) Read and process specific data depending on message type
        if (un8TypeFlag == EPG_AU_HEADER_TFLAG_GRID)
        {
            eResult = eEpgParserProcessGridMessage(psEpgParser,
                                                   psSegment,
                                                   hPayload);
        }
        else if (un8TypeFlag == EPG_AU_HEADER_TFLAG_TEXT)
        {
            eResult = eEpgParserProcessTextMessage(psEpgParser,
                                                   psSegment,
                                                   hPayload);
        }

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessGridMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessGridMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    if ((psEpgParser->un32Options & EPG_PARSER_OPTION_MEMORY_SAVING) != 0)
    {
        // Process only next nearest AU
        eResult = eEpgParserProcessNextNearestGridAu(psEpgParser,
                                                     psSegment,
                                                     hPayload);
    }
    else
    {
        // Try to save AU. If AU is already saved, EPG_RET_IGNORED is returned.
        eResult = eEpgParserSaveGridAu(psEpgParser,
                                       psSegment,
                                       hPayload);
        if (eResult == EPG_RET_OK)
        {
            // Saved segments' AUs should be parsed sequentially
            // from the latest parsed AU.
            eResult = eEpgParserParseSavedGridAus(psEpgParser, FALSE);
        }
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessNextNearestGridAu
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessNextNearestGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_SEG_GRID_DATA_STRUCT *psSegGridData = &(psSegment->sSegGridData);
    BOOLEAN bResult = FALSE;
    EPG_AU_GRID_HEADER sGridAuHeader;
    EPG_AU_INFO_STRUCT sAuData;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    // Zero stack data
    sAuData.pvData = NULL;

    do
    {
        // Check if cancelled
        bResult = bEpgParserIsCancelled(psEpgParser);
        if (bResult == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"Parsing is cancelled.\n"PC_RESET);
            eResult = EPG_RET_CANCELLED;
            break;
        }

        // Check if segment's grid data should be ignored
        if (psSegment->un8SegmentNumber != psEpgParser->un8NumSegmentsGridCollected)
        {
            // Segments should be parsed sequentially starting from segment 0.
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"SEGMENT %u GRID IS IGNORED. TARGET SEGMENT NUMBER = %u.\n"PC_RESET,
                   psSegment->un8SegmentNumber, psEpgParser->un8NumSegmentsGridCollected);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Create the temporary buffer containing payload data
        sAuData.pvData = DS_UTIL_pvCopyPayload(hPayload, &(sAuData.tDataSize));
        if (sAuData.pvData == NULL)
        {
            break;
        }

        // Read GRID specific AU header data
        bResult = bEpgParserExtractGridHeader(hPayload,
                                              &sGridAuHeader);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read Grid Header.\n"PC_RESET);
            break;
        }
        psSegGridData->un32NumAuTotal = sGridAuHeader.un32NumAuTotal;

        // We should not copy all AUs payloads.
        // AUs should be parsed sequentially starting from AU 0 of segment 0.

        // Check if received AU should be ignored
        if (sGridAuHeader.un32AuSequenceNumber != psSegGridData->un32NumAuCollected)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"GRID AU %u:%u IS IGNORED. TARGET GRID AU = %u:%u.\n"PC_RESET,
                psSegment->un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber,
                psSegment->un8SegmentNumber, psSegGridData->un32NumAuCollected);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Process payload (Program Event Blocks)
        eResult = eEpgParserProcessGridPayload(psEpgParser,
                                               psSegment->un8SegmentNumber,
                                               &sGridAuHeader,
                                               hPayload,
                                               FALSE);
        if (eResult != EPG_RET_OK)
        {
            // Clean previously collected GRID data
            vEpgParserCleanAllSegmentsGrid(psEpgParser);
            // Report about processing restart
            psEpgParser->psEpgClient->vRestartGridProcessing(psEpgParser->psEpgClient->hClientObj);
            break;
        }

        // Send AU payload data to parser's client for saving
        sAuData.eAuDataType = EPG_DATA_TYPE_GRID;
        sAuData.un8SegmentNumber = psSegment->un8SegmentNumber;
        sAuData.un32AuNumber = sGridAuHeader.un32AuSequenceNumber;
        psEpgParser->psEpgClient->vSaveAuData(psEpgParser->psEpgClient->hClientObj,
                                              &sAuData);

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5, PC_GREEN"GRID AU %u:%u IS PROCESSED.\n"PC_RESET,
               psSegment->un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber);

        // Just increase counter
        psSegGridData->un32NumAuCollected++;

        // Check if segment's GRID data is completed (all segment's GRID AUs are collected)
        if (psSegGridData->un32NumAuCollected == psSegGridData->un32NumAuTotal)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"=:=:=:=:=  SEGMENT %u GRID IS COMPLETED =:=:=:=:=\n"PC_RESET,
                psSegment->un8SegmentNumber);

            psEpgParser->un8NumSegmentsGridCollected++;

            // Report Segment's GRID completion
            psEpgParser->psEpgClient->vSegmentGridCompleted(psEpgParser->psEpgClient->hClientObj,
                                                            psSegment->un8SegmentNumber);

            // Check if a whole segment is completed
            if (psSegment->sSegTextData.un32NumAuCollected > 0//TODO: Is it possible if no TEXT AU present?
                &&
                psSegment->sSegTextData.un32NumAuCollected ==
                psSegment->sSegTextData.sTextAuHeader.un32NumAuTotal)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                    PC_GREEN"=:=:=:=:= WHOLE SEGMENT %u IS COMPLETED =:=:=:=:=\n"PC_RESET,
                    psSegment->un8SegmentNumber);

                // Report whole segment completion
                psEpgParser->psEpgClient->vWholeSegmentCompleted(psEpgParser->psEpgClient->hClientObj,
                                                                 psSegment->un8SegmentNumber);
            }
        }

        // Everything's done
        eResult = EPG_RET_OK;

    } while (FALSE);

    // Delete temporary buffer if it was allocated
    if (sAuData.pvData != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) sAuData.pvData);
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessGridAu
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = FALSE;
    EPG_AU_GRID_HEADER sGridAuHeader;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    do
    {
        // Check if cancelled
        bResult = bEpgParserIsCancelled(psEpgParser);
        if (bResult == TRUE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"Parsing is cancelled.\n"PC_RESET);
            eResult = EPG_RET_CANCELLED;
            break;
        }

        // Read GRID specific AU header data
        bResult = bEpgParserExtractGridHeader(hPayload,
                                              &sGridAuHeader);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read Grid Header.\n"PC_RESET);
            break;
        }

        // Process payload (Program Event Blocks)
        eResult = eEpgParserProcessGridPayload(psEpgParser,
                                               un8SegmentNumber,
                                               &sGridAuHeader,
                                               hPayload,
                                               TRUE);
        if (eResult != EPG_RET_OK)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"GRID AU %u:%u PROCESSING FAILED.\n"PC_RESET,
               un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber);
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"GRID AU %u:%u IS PROCESSED.\n"PC_RESET,
            un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber);

        // Everything's done
        eResult = EPG_RET_OK;

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessTextMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessTextMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    if ((psEpgParser->un32Options & EPG_PARSER_OPTION_MEMORY_SAVING) != 0)
    {
        // Process only next nearest AU
        eResult = eEpgParserProcessNextNearestTextAu(psEpgParser,
                                                     psSegment,
                                                     hPayload);
    }
    else
    {
        // Try to save AU. If AU is already saved, EPG_RET_IGNORED is returned.
        eResult = eEpgParserSaveTextAu(psEpgParser,
                                       psSegment,
                                       hPayload);
        if (eResult == EPG_RET_OK)
        {
            // Saved segments' AUs should be parsed sequentially
            // from the latest parsed AU.
            eResult = eEpgParserParseSavedTextAus(psEpgParser, FALSE);
        }
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessNextNearestTextAu
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessNextNearestTextAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_SEG_TEXT_DATA_STRUCT *psSegTextData = &(psSegment->sSegTextData);
    BOOLEAN bResult = FALSE;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    do
    {
        // Check if segment's text data should be ignored
        if (psSegment->un8SegmentNumber != psEpgParser->un8NumSegmentsTextCollected)
        {
            // Segments should be parsed sequentially starting from segment 0.
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"SEGMENT %u TEXT IS IGNORED. TARGET SEGMENT NUMBER = %u.\n"PC_RESET,
                psSegment->un8SegmentNumber, psEpgParser->un8NumSegmentsTextCollected);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Read TEXT specific AU header data
        bResult = bEpgParserExtractTextHeader(hPayload,
                                              &(psSegTextData->sTextAuHeader));
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read Text Header.\n"PC_RESET);
            break;
        }

        // We should not copy all AUs payloads.
        // AUs should be parsed sequentially starting from AU 0 of segment 0.

        // Check if received AU should be ignored
        if (psSegTextData->sTextAuHeader.un32AuSequenceNumber !=
            psSegTextData->un32NumAuCollected)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"TEXT AU %u:%u IS IGNORED. TARGET TEXT AU = %u:%u.\n"PC_RESET,
                psSegment->un8SegmentNumber, psSegTextData->sTextAuHeader.un32AuSequenceNumber,
                psSegment->un8SegmentNumber, psSegTextData->un32NumAuCollected);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // Process payload
        eResult = eEpgParserProcessTextPayload(psEpgParser,
                                               psSegment->un8SegmentNumber,
                                               &(psSegTextData->sTextAuHeader),
                                               hPayload);
        if (eResult != EPG_RET_OK)
        {
            // Nothing should be done to handle error, just skipping an AU.
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"TEXT AU %u:%u IS PROCESSED.\n"PC_RESET,
            psSegment->un8SegmentNumber,
            psSegTextData->sTextAuHeader.un32AuSequenceNumber);

        // Just increase counter
        psSegTextData->un32NumAuCollected++;

        // Check if segment's TEXT data is completed (all segment's TEXT AUs are collected)
        if (psSegTextData->un32NumAuCollected ==
            psSegTextData->sTextAuHeader.un32NumAuTotal)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"=:=:=:=:=  SEGMENT %u TEXT IS COMPLETED =:=:=:=:=\n"PC_RESET,
                psSegment->un8SegmentNumber);

            psEpgParser->un8NumSegmentsTextCollected++;

            // Report Segment's TEXT completion
            psEpgParser->psEpgClient->vSegmentTextCompleted(psEpgParser->psEpgClient->hClientObj,
                                                            psSegment->un8SegmentNumber,
                                                            psSegTextData->sTextAuHeader.un16StringsCount,
                                                            psSegTextData->sTextAuHeader.un32DecompressedFileSize);

            // Check if a whole segment is completed
            if (psSegment->sSegGridData.un32NumAuCollected > 0 &&
                psSegment->sSegGridData.un32NumAuCollected ==
                psSegment->sSegGridData.un32NumAuTotal)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                    PC_GREEN"=:=:=:=:= WHOLE SEGMENT %u IS COMPLETED =:=:=:=:=\n"PC_RESET,
                    psSegment->un8SegmentNumber);

                // Report whole segment completion
                psEpgParser->psEpgClient->vWholeSegmentCompleted(psEpgParser->psEpgClient->hClientObj,
                                                                 psSegment->un8SegmentNumber);
            }
        }

        // Everything's done
        eResult = EPG_RET_OK;

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessProfileConfigMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessProfileConfigMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_NOT_SUPPORTED;
    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessPamMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessPamMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_NOT_SUPPORTED;
    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessTadMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessTadMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_NOT_SUPPORTED;
    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessScheduleVersioningMessage
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessScheduleVersioningMessage (
        EPG_PARSER_STRUCT *psEpgParser,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;
    EPG_SCHEDULE_VERSION_STRUCT sNewVersion;
    EPG_SCHEDULE_VERSION_STRUCT sCurrentVersion;
    BOOLEAN bVersionChanged = FALSE;
    UN8 un8TotalSegments = 0;
    UN8 un8SegmentNumber = 0;
    UN8 un8SegmentVersion = 0;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    if ((psEpgParser->un32Options & EPG_PARSER_OPTION_CHECK_SEG_VERSION) == 0)
    {
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            PC_BLUE"SSV: Not supported. Ignored.\n"PC_RESET);
        return EPG_RET_IGNORED;
    }

    // Zero stack data
    bResult = OSAL.bMemSet(&sNewVersion, 0, sizeof(EPG_SCHEDULE_VERSION_STRUCT));
    if (bResult == FALSE)
    {
        return EPG_RET_FAIL;
    }

    do
    {
        // Reserved bit (skip)
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_SSV_RESERVED_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read RESERVED bit.\n"PC_RESET);
            break;
        }

        // Schedule version
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_SHVER_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SHVER.\n"PC_RESET);
            break;
        }
        sNewVersion.un8Version = (UN8)un32ReadValue;

        // Epoch
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_EPOCH_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read EPOCH.\n"PC_RESET);
            break;
        }
        sNewVersion.un16Epoch = (UN16)un32ReadValue;

        // Get Current Version
        psEpgParser->psEpgClient->vGetVersion(psEpgParser->psEpgClient->hClientObj,
                                              &sCurrentVersion);

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "SSV: Current schedule   EPOCH:%u  VERSION:%u\n",
            sCurrentVersion.un16Epoch, sCurrentVersion.un8Version);
        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
            "SSV: Incoming schedule  EPOCH:%u  VERSION:%u\n",
            sNewVersion.un16Epoch, sNewVersion.un8Version);

        // Compare Version and Epoch
        if (sNewVersion.un16Epoch != sCurrentVersion.un16Epoch ||
            sNewVersion.un8Version < sCurrentVersion.un8Version)
        {
            // All segments shall be processed again in numerical order.
            // So clean all segments (delete collected AUs).
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"SSV: Version is entirely changed. Clean all segments.\n"PC_RESET);
            bVersionChanged = TRUE;
            vEpgParserCleanAllSegments(psEpgParser);
            eResult = EPG_RET_OK;
            break;
        }
        else if (sNewVersion.un8Version == sCurrentVersion.un8Version)
        {
            // Version is not changed. Do nothing.
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_BLUE"SSV: Version is not changed.\n"PC_RESET);
            bVersionChanged = FALSE;
            eResult = EPG_RET_OK;
            break;
        }

        bVersionChanged = TRUE;

        // Check segments versions

        // Read Number of Segments
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_AU_HEADER_NOSEG_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read NOSEG.\n"PC_RESET);
            break;
        }
        un8TotalSegments = (UN8)un32ReadValue + 1;

        // Limit number of segments by minimum possible value
        // (in case if current broadcast provides less number of segments).
        if (un8TotalSegments > psEpgParser->un8NumSegmentsToRecieve)
        {
            un8TotalSegments = psEpgParser->un8NumSegmentsToRecieve;
        }

        // Read segments versions
        for (un8SegmentNumber = 0; un8SegmentNumber < un8TotalSegments; un8SegmentNumber++)
        {
            // Segment Version
            bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                                hPayload,
                                                EPG_SSV_SEGVER_BITLEN);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read SEGVER.\n"PC_RESET);
                break;
            }
            un8SegmentVersion = (UN8)un32ReadValue;

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                "SSV: Segment %u  Version %u\n",
                un8SegmentNumber, un8SegmentVersion);

            // Reset all segments with new version higher than current schedule version
            if (un8SegmentVersion > sCurrentVersion.un8Version)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_BLUE"SSV: Segment %u version is changed. Clean collected segment data.\n"PC_RESET,
                    un8SegmentNumber);
                // Clean collected AUs
                vEpgParserCleanSegmentGrid(psEpgParser, un8SegmentNumber);
                vEpgParserCleanSegmentText(psEpgParser, un8SegmentNumber);

                // Set counters to the lowest resetted segment number
                if (psEpgParser->un8NumSegmentsGridCollected > un8SegmentNumber)
                {
                    psEpgParser->un8NumSegmentsGridCollected = un8SegmentNumber;
                }
                if (psEpgParser->un8NumSegmentsTextCollected > un8SegmentNumber)
                {
                    psEpgParser->un8NumSegmentsTextCollected = un8SegmentNumber;
                }
            }
        }

        // Don't save extra data, just skip it.
        bResult = bEpgParserExtractExtraData(hPayload,
                                             NULL, NULL);
        if (bResult == FALSE)
        {
            break;
        }

        eResult = EPG_RET_OK;

    } while (FALSE);

    if (eResult == EPG_RET_OK && bVersionChanged == TRUE)
    {
        // Version is changed from the initial one.
        // Now next versions can be accepted.
        if (psEpgParser->bAcceptNewVersionOnly == TRUE)
        {
            psEpgParser->bAcceptNewVersionOnly = FALSE;
        }

        // Report about version change
        psEpgParser->psEpgClient->vVersionChanged(psEpgParser->psEpgClient->hClientObj,
                                                  &sNewVersion);
        // Report about processing restart
        psEpgParser->psEpgClient->vRestartGridProcessing(psEpgParser->psEpgClient->hClientObj);
        psEpgParser->psEpgClient->vRestartTextProcessing(psEpgParser->psEpgClient->hClientObj);

        // Reprocess all already collected grid data (if any)
        eResult = eEpgParserParseSavedGridAus(psEpgParser, TRUE);
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessGridPayload
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessGridPayload (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        BOOLEAN bIgnoreEventProcessingErrors
            )
{
    N8 n8Sinc = 0;
    SERVICE_ID tServiceId = 0;
    BOOLEAN bResult = TRUE;
    UN32 un32ReadValue = 0;
    UN16 un16NumberOfPrograms = 0;
    EPG_PARSER_EVENT_STRUCT sProgramEvent;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;

    // Zero stack data
    bResult = OSAL.bMemSet(&sProgramEvent, 0, sizeof(EPG_PARSER_EVENT_STRUCT));
    if (bResult == FALSE)
    {
        return EPG_RET_FAIL;
    }

    do
    {
        // Service ID
        n8Sinc = n8EpgParserExtractSid(hPayload,
                                       &tServiceId);
        if (n8Sinc == -1)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to get SID.\n"PC_RESET);
            eResult = EPG_RET_FAIL;
            break;
        }
        if (n8Sinc == EPG_PEB_HEADER_SINC_CODE_END_OF_LIST)
        {
            // No more Channels are listed in the Access Unit. Exit.
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
                PC_GREEN"End of list.\n"PC_RESET);
            eResult = EPG_RET_OK;
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
            PC_GREEN"CHANNEL SID = %u.\n"PC_RESET,
                tServiceId);

        // Number of Program Event Messages for this SID in this Segment
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_PEB_HEADER_SHOWNO_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read SHOWNO.\n"PC_RESET);
            eResult = EPG_RET_FAIL;
            break;
        }
        un16NumberOfPrograms = (UN16)un32ReadValue + 1;

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
            PC_GREEN"NUMBER OF PROGRAMS = %u.\n"PC_RESET,
                un16NumberOfPrograms);

        while (un16NumberOfPrograms > 0)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
                PC_GREEN"Extracting program event... Programs remained: %u\n"PC_RESET,
                    un16NumberOfPrograms);

            // Extract Program Event
            bResult = bEpgParserExtractProgramEvent(psGridAuHeader,
                                                    hPayload,
                                                    &sProgramEvent);
            if (bResult == FALSE)
            {
                eResult = EPG_RET_FAIL;
                break;
            }

            // Set Segment Number
            sProgramEvent.un8SegmentNumber = un8SegmentNumber;
            // Set SID
            sProgramEvent.tServiceId = tServiceId;

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,
                PC_GREEN"Program successfully parsed. Notify client.\n"PC_RESET);

            // Expose Program Event
            bResult = psEpgParser->psEpgClient->bProcessParserEvent(psEpgParser->psEpgClient->hClientObj,
                                                                    &sProgramEvent);
            if (bResult == FALSE)
            {
                if (bIgnoreEventProcessingErrors == TRUE)
                {
                    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                        PC_RED"WARNING: Program Event processing failure is ignored. Event Type=%u\n"PC_RESET,
                           sProgramEvent.ePemType);
                }
                else
                {
                    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                        PC_RED"ERROR: Failed to handle Program Event.\n"PC_RESET);
                    eResult = EPG_RET_FAIL;
                    break;
                }
            }

            // Clean sProgramEvent
            vEpgParserCleanParserEvent(&sProgramEvent);

            un16NumberOfPrograms--;
        }

    } while (eResult == EPG_RET_OK);

    // Clean sProgramEvent
    vEpgParserCleanParserEvent(&sProgramEvent);

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserProcessTextPayload
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserProcessTextPayload (
        EPG_PARSER_STRUCT *psEpgParser,
        UN8 un8SegmentNumber,
        EPG_AU_TEXT_HEADER *psTextAuHeader,
        OSAL_BUFFER_HDL hPayload
            )
{
    BOOLEAN bResult = FALSE;
    EPG_AU_INFO_STRUCT sAuData;
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;

    // Zero stack data
    sAuData.pvData = NULL;

    do
    {
        // Create the temporary buffer containing payload data
        sAuData.pvData = DS_UTIL_pvCopyPayload(hPayload, &(sAuData.tDataSize));
        if (sAuData.pvData == NULL)
        {
            break;
        }

        // Send Compressed Text AU payload data to parser's client
        sAuData.eAuDataType = EPG_DATA_TYPE_COMPRESSED_TEXT;
        sAuData.un8SegmentNumber = un8SegmentNumber;
        sAuData.un32AuNumber = psTextAuHeader->un32AuSequenceNumber;
        bResult = psEpgParser->psEpgClient->bProcessCompressedTextPayload(psEpgParser->psEpgClient->hClientObj,
                                                                          &sAuData);
        if (bResult == FALSE)
        {
            break;
        }

        // Everything's OK
        eResult = EPG_RET_OK;

    } while (FALSE);

    // Delete temporary buffer
    if (sAuData.pvData != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) sAuData.pvData);
    }

    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserSaveGridAu
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserSaveGridAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_FAIL;
    EPG_AU_GRID_HEADER sGridAuHeader;
    EPG_SEG_GRID_DATA_STRUCT *psSegGridData = &(psSegment->sSegGridData);
    EPG_AU_DATA_STRUCT *psAuGridData = NULL;
    void *pvTmpData = NULL;
    size_t tTmpDataSize = 0;
    BOOLEAN bResult = FALSE;

    do
    {
        // Create the temporary buffer containing payload data
        pvTmpData = DS_UTIL_pvCopyPayload(hPayload, &tTmpDataSize);
        if (pvTmpData == NULL)
        {
            break;
        }

        // Read AU header
        bResult = bEpgParserExtractGridHeader(hPayload,
                                              &sGridAuHeader);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read Grid Header.\n"PC_RESET);
            break;
        }
        psSegGridData->un32NumAuTotal = sGridAuHeader.un32NumAuTotal;

        // Check if AUs array is not yet allocated
        if (psSegGridData->asAuGridData == NULL)
        {
            // Create AUs array
            psSegGridData->asAuGridData =
                (EPG_AU_DATA_STRUCT *)
                SMSO_hCreate(EPG_INTF_OBJECT_NAME":GridAUsArray",
                             sizeof(EPG_AU_DATA_STRUCT) * psSegGridData->un32NumAuTotal,
                             SMS_INVALID_OBJECT, FALSE);
            if (psSegGridData->asAuGridData == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to create AUs array.\n"PC_RESET);
                break;
            }
        }

        // Check if AU is already saved
        psAuGridData = &(psSegGridData->asAuGridData[sGridAuHeader.un32AuSequenceNumber]);
        if (psAuGridData->pvAuPayload != NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"GRID AU %u:%u IS ALREADY SAVED.\n"PC_RESET,
                   psSegment->un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // AU is not yet saved. Save AU data.
        psAuGridData->pvAuPayload = pvTmpData;
        psAuGridData->tPayloadSize = tTmpDataSize;

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"GRID AU %u:%u IS SAVED.\n"PC_RESET,
               psSegment->un8SegmentNumber, sGridAuHeader.un32AuSequenceNumber);

        // Evetything's done
        return EPG_RET_OK;

    } while (FALSE);

    // Data is not saved
    // Delete temporary buffer if it was allocated
    if (pvTmpData != NULL)
    {
        SMSO_vDestroy((SMS_OBJECT) pvTmpData);
    }
    return eResult;
}

/*******************************************************************************
 *
 *  eEpgParserSaveTextAu
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserSaveTextAu (
        EPG_PARSER_STRUCT *psEpgParser,
        EPG_SEGMENT_STRUCT *psSegment,
        OSAL_BUFFER_HDL hPayload
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;
    EPG_SEG_TEXT_DATA_STRUCT *psSegTextData = &(psSegment->sSegTextData);
    EPG_AU_TEXT_HEADER *psTextAuHeader = &(psSegTextData->sTextAuHeader);
    EPG_AU_DATA_STRUCT *psAuTextData = NULL;
    BOOLEAN bResult = FALSE;

    do
    {
        // Read AU header
        bResult = bEpgParserExtractTextHeader(hPayload,
                                              psTextAuHeader);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read Text Header.\n"PC_RESET);
            break;
        }

        // Check if AUs array is not yet allocated
        if (psSegTextData->asAuTextData == NULL)
        {
            // Create AUs array
            psSegTextData->asAuTextData =
                (EPG_AU_DATA_STRUCT *)
                SMSO_hCreate(EPG_INTF_OBJECT_NAME":TextAUsArray",
                             sizeof(EPG_AU_DATA_STRUCT) *
                             psTextAuHeader->un32NumAuTotal,
                             SMS_INVALID_OBJECT, FALSE);
            if (psSegTextData->asAuTextData == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to create AUs array.\n"PC_RESET);
                eResult = EPG_RET_FAIL;
                break;
            }
        }

        // Check if AU is already saved
        psAuTextData = &(psSegTextData->asAuTextData[psTextAuHeader->un32AuSequenceNumber]);
        if (psAuTextData->pvAuPayload != NULL)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"TEXT AU %u:%u IS ALREADY SAVED.\n"PC_RESET,
                   psSegment->un8SegmentNumber, psTextAuHeader->un32AuSequenceNumber);
            eResult = EPG_RET_IGNORED;
            break;
        }

        // AU is not yet saved. Copy AU data.
        psAuTextData->pvAuPayload = DS_UTIL_pvCopyPayload(hPayload, &(psAuTextData->tPayloadSize));
        if (psAuTextData->pvAuPayload == NULL)
        {
            eResult = EPG_RET_FAIL;
            break;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"TEXT AU %u:%u IS SAVED.\n"PC_RESET,
            psSegment->un8SegmentNumber, psTextAuHeader->un32AuSequenceNumber);

        // Evetything's done
        return EPG_RET_OK;

    } while (FALSE);

    return eResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractProgramEvent
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractProgramEvent (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = 0;
    UN32 un32ReadValue = 0;

    do
    {
        // Program event type
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            EPG_PEM_PTYPE_BITLEN);
        if (bResult == FALSE)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Failed to read PTYPE.\n"PC_RESET);
            break;
        }

        switch (un32ReadValue)
        {
        case EPG_PEM_PTYPE_FULL_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,"PEM TYPE: FULL\n");
            psParserEvent->ePemType = EPG_PTYPE_FULL_EVENT;
            bResult = bEpgParserExtractPemFull(psGridAuHeader, hPayload, psParserEvent);
            break;

        case EPG_PEM_PTYPE_MODIFY_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,"PEM TYPE: MOD\n");
            psParserEvent->ePemType = EPG_PTYPE_MODIFY_EVENT;
            bResult = bEpgParserExtractPemMod(psGridAuHeader, hPayload, psParserEvent);
            break;

        case EPG_PEM_PTYPE_ADD_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,"PEM TYPE: ADD\n");
            psParserEvent->ePemType = EPG_PTYPE_ADD_EVENT;
            bResult = bEpgParserExtractPemAdd(psGridAuHeader, hPayload, psParserEvent);
            break;

        case EPG_PEM_PTYPE_DURATION_ONLY_EVENT:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 11,"PEM TYPE: DUR\n");
            psParserEvent->ePemType = EPG_PTYPE_DURATION_ONLY_EVENT;
            bResult = bEpgParserExtractPemDur(psGridAuHeader, hPayload, psParserEvent);
            break;

        default:
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                PC_RED"ERROR: Invalid PEM type=%u\n"PC_RESET,
                un32ReadValue);
            bResult = FALSE;
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractPemFull
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractPemFull (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;

    do
    {
        // Series ID
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8SeriesIdSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->tSeriesId = (SERIES_ID)un32ReadValue;

        // Program ID
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            psGridAuHeader->un8ProgramIdSize);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->tProgramId = (PROGRAM_ID)un32ReadValue;

        // Program flags
        bResult = bEpgParserExtractProgramFlags(hPayload,
                                                psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }
        if (psParserEvent->tProgramFlags == EPG_PROGRAM_INVALID_FLAGS)
        {
            // For FULL event there could be no INVALID flags
            psParserEvent->tProgramFlags = EPG_PROGRAM_UNDEFINED;
        }

        // Duration code
        bResult = bEpgParserExtractDuration(hPayload,
                                            psParserEvent,
                                            TRUE);
        if (bResult == FALSE)
        {
            break;
        }

        // Short name index
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            psGridAuHeader->un8TextIndexSize);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32ProgNameShortIdx = un32ReadValue;

        // Long name index
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8TextIndexSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32ProgNameLongIdx = un32ReadValue;

        // Series description index
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8TextIndexSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32SeriesDescIdx = un32ReadValue;

        // Program description index
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8TextIndexSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32ProgramDescIdx = un32ReadValue;

        // Original date
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_ORIGIN_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un16OriginalDate = (UN16)un32ReadValue;

        // Additional start times
        bResult = bEpgParserExtractAdditionalTimes(hPayload,
                                                   psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Daily repeat indicator
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_DAY_BITLEN,
                                                    0);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un8DailyRepeat = (UN8)un32ReadValue;

        // Topics
        bResult = bEpgParserExtractTopics(psGridAuHeader,
                                          hPayload,
                                          psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Don't save extra data, just skip it.
        bResult = bEpgParserExtractExtraData(hPayload,
                                             NULL, NULL);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractPemMod
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractPemMod (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;

    do
    {
        // Series ID is not present
        psParserEvent->tSeriesId = (SERIES_ID)UN32_MAX;

        // Program ID
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            psGridAuHeader->un8ProgramIdSize);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->tProgramId = (PROGRAM_ID)un32ReadValue;

        // Program flags
        bResult = bEpgParserExtractProgramFlags(hPayload,
                                                psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Duration code
        bResult = bEpgParserExtractDuration(hPayload,
                                            psParserEvent,
                                            FALSE);
        if (bResult == FALSE)
        {
            break;
        }

        // Episode description index
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8TextIndexSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32ProgramDescIdx = un32ReadValue;

        // Original date
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_ORIGIN_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un16OriginalDate = (UN16)un32ReadValue;

        // Topics
        bResult = bEpgParserExtractTopics(psGridAuHeader,
                                          hPayload,
                                          psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Don't save extra data, just skip it.
        bResult = bEpgParserExtractExtraData(hPayload,
                                             NULL, NULL);
        if (bResult == FALSE)
        {
            break;
        }

    } while (FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractPemAdd
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractPemAdd (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = FALSE;
    UN32 un32ReadValue = 0;

    do
    {
        // Series ID
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            psGridAuHeader->un8SeriesIdSize);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->tSeriesId = (SERIES_ID)un32ReadValue;

        // Program ID
        bResult = DS_UTIL_bExtractBitField(&un32ReadValue,
                                            hPayload,
                                            psGridAuHeader->un8ProgramIdSize);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->tProgramId = (PROGRAM_ID)un32ReadValue;

        // Program flags
        bResult = bEpgParserExtractProgramFlags(hPayload,
                                                psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Duration
        bResult = bEpgParserExtractDuration(hPayload,
                                            psParserEvent,
                                            FALSE);
        if (bResult == FALSE)
        {
            break;
        }

        // Episode description index
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    psGridAuHeader->un8TextIndexSize,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un32ProgramDescIdx = un32ReadValue;

        // Original date
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_ORIGIN_BITLEN,
                                                    UN32_MAX);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un16OriginalDate = (UN16)un32ReadValue;

        // Additional start times
        bResult = bEpgParserExtractAdditionalTimes(hPayload,
                                                   psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Daily repeat indicator
        bResult = DS_UTIL_bExtractOptionalBitField(&un32ReadValue,
                                                    hPayload,
                                                    EPG_PEM_DAY_BITLEN,
                                                    0);
        if (bResult == FALSE)
        {
            break;
        }
        psParserEvent->un8DailyRepeat = (UN8)un32ReadValue;

        // Topics
        bResult = bEpgParserExtractTopics(psGridAuHeader,
                                          hPayload,
                                          psParserEvent);
        if (bResult == FALSE)
        {
            break;
        }

        // Don't save extra data, just skip it.
        bResult = bEpgParserExtractExtraData(hPayload,
                                             NULL, NULL);
        if (bResult == FALSE)
        {
            break;
        }

    } while(FALSE);

    return bResult;
}

/*******************************************************************************
 *
 *  bEpgParserExtractPemDur
 *
 *******************************************************************************/
static BOOLEAN bEpgParserExtractPemDur (
        EPG_AU_GRID_HEADER *psGridAuHeader,
        OSAL_BUFFER_HDL hPayload,
        EPG_PARSER_EVENT_STRUCT *psParserEvent
            )
{
    BOOLEAN bResult = FALSE;

    // Duration only
    bResult = bEpgParserExtractDuration(hPayload,
                                        psParserEvent,
                                        TRUE);

    return bResult;
}

/*******************************************************************************
 *
 *  eEpgParserParseSavedGridAus
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserParseSavedGridAus (
        EPG_PARSER_STRUCT *psEpgParser,
        BOOLEAN bReprocessAll
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;
    EPG_AU_GRID_HEADER sGridAuHeader;
    EPG_AU_INFO_STRUCT sAuData;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"eEpgParserParseSavedGridAus(): Segments GRID collected: %u of %u.\n"PC_RESET,
        psEpgParser->un8NumSegmentsGridCollected, psEpgParser->un8NumSegmentsToRecieve);

    if (bReprocessAll == TRUE)
    {
        // Reset counter to start from the beginning
        psEpgParser->un8NumSegmentsGridCollected = 0;
    }
    else
    {
        if (psEpgParser->un8NumSegmentsGridCollected ==
            psEpgParser->un8NumSegmentsToRecieve)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
                PC_BLUE"WARNING: All GRID segments are already collected.\n"PC_RESET);
            return EPG_RET_IGNORED;
        }
    }

    // Iterate through segments
    while (psEpgParser->un8NumSegmentsGridCollected <
           psEpgParser->un8NumSegmentsToRecieve)
    {
        // Get segment
        EPG_SEGMENT_STRUCT *psSegment =
            &(psEpgParser->asSegments[psEpgParser->un8NumSegmentsGridCollected]);
        // Get segment's grid data
        EPG_SEG_GRID_DATA_STRUCT *psSegGridData = &(psSegment->sSegGridData);

        // Stop iterating if no AUs were received for required segment
        if (psSegGridData->asAuGridData == NULL ||
            psSegGridData->un32NumAuTotal == 0)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"Segment %u GRID is empty. Further parsing is impossible yet.\n"PC_RESET,
                psEpgParser->un8NumSegmentsGridCollected);
            return EPG_RET_OK;
        }

        if (bReprocessAll == TRUE)
        {
            // Reset counter to start from the beginning of a segment
            psSegGridData->un32NumAuCollected = 0;
        }

        // Iterate through Access Units
        while (psSegGridData->un32NumAuCollected <
               psSegment->sSegGridData.un32NumAuTotal)
        {
            BOOLEAN bResult = FALSE;
            // Get grid AU data
            EPG_AU_DATA_STRUCT *psAuGridData =
                &(psSegGridData->asAuGridData[psSegGridData->un32NumAuCollected]);

            // Check if cancelled
            bResult = bEpgParserIsCancelled(psEpgParser);
            if (bResult == TRUE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_GREEN"Parsing is cancelled.\n"PC_RESET);
                return EPG_RET_CANCELLED;
            }

            // Stop iterating if required AU was not received yet
            if (psAuGridData->pvAuPayload == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_GREEN"GRID AU %u:%u is not yet loaded. Further parsing is impossible yet.\n"PC_RESET,
                    psSegment->un8SegmentNumber, psSegGridData->un32NumAuCollected);
                return EPG_RET_OK;
            }

            // Copy data to temporary buffer
            bResult = bEpgParserCopyDataToTempBuffer(psEpgParser,
                                                     psAuGridData->pvAuPayload,
                                                     psAuGridData->tPayloadSize);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to copy Saved AU to temp buffer.\n"PC_RESET);
                return EPG_RET_FAIL;
            }

            // Read AU header
            bResult = bEpgParserExtractGridHeader(psEpgParser->hTempBuffer,
                                                  &sGridAuHeader);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to read Grid Header.\n"PC_RESET);
                return EPG_RET_FAIL;
            }

            // Process payload (Program Event Blocks)
            eResult = eEpgParserProcessGridPayload(psEpgParser,
                                                   psSegment->un8SegmentNumber,
                                                   &sGridAuHeader,
                                                   psEpgParser->hTempBuffer,
                                                   FALSE);
            if (eResult != EPG_RET_OK)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to parse GRID AU %u:%u.\n"PC_RESET,
                    psSegment->un8SegmentNumber, psSegGridData->un32NumAuCollected);
                // Clean failed AU, so it can be reloaded from the broadcast later
                vEpgParserCleanAu(psAuGridData);
                // Notify about GRID processing restart
                psEpgParser->psEpgClient->vRestartGridProcessing(psEpgParser->psEpgClient->hClientObj);
                // Reprocess all previously saved AUs
                eEpgParserParseSavedGridAus(psEpgParser, TRUE);
                return eResult;
            }

            // Send AU payload data to parser's client for saving
            sAuData.eAuDataType = EPG_DATA_TYPE_GRID;
            sAuData.un8SegmentNumber = psSegment->un8SegmentNumber;
            sAuData.un32AuNumber = sGridAuHeader.un32AuSequenceNumber;
            sAuData.pvData = psAuGridData->pvAuPayload;
            sAuData.tDataSize = psAuGridData->tPayloadSize;
            psEpgParser->psEpgClient->vSaveAuData(psEpgParser->psEpgClient->hClientObj,
                                                  &sAuData);

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"GRID AU %u:%u \t IS PROCESSED.\n"PC_RESET,
                psSegment->un8SegmentNumber,
                sGridAuHeader.un32AuSequenceNumber);

            psSegGridData->un32NumAuCollected++;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"=:=:=:=:=  SEGMENT %u GRID IS COMPLETED =:=:=:=:=\n"PC_RESET,
            psSegment->un8SegmentNumber);

        // Current segment is parsed. Parse next segment from the beginning
        bReprocessAll = TRUE;

        psEpgParser->un8NumSegmentsGridCollected++;

        // Notify about Segment's GRID completion
        psEpgParser->psEpgClient->vSegmentGridCompleted(psEpgParser->psEpgClient->hClientObj,
                                                        psSegment->un8SegmentNumber);

        // Check if a whole segment is completed
        if (psSegment->sSegTextData.un32NumAuCollected > 0 &&//TODO: Is it possible if no TEXT AU present?
            psSegment->sSegTextData.un32NumAuCollected ==
            psSegment->sSegTextData.sTextAuHeader.un32NumAuTotal)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"=:=:=:=:= WHOLE SEGMENT %u IS COMPLETED =:=:=:=:=\n"PC_RESET,
                psSegment->un8SegmentNumber);

            // Notify about whole segment completion
            psEpgParser->psEpgClient->vWholeSegmentCompleted(psEpgParser->psEpgClient->hClientObj,
                                                             psSegment->un8SegmentNumber);
        }
    };

    return EPG_RET_OK;
}

/*******************************************************************************
 *
 *  eEpgParserParseSavedTextAus
 *
 *******************************************************************************/
static EPG_RETURN_CODES_ENUM eEpgParserParseSavedTextAus (
        EPG_PARSER_STRUCT *psEpgParser,
        BOOLEAN bReprocessAll
            )
{
    EPG_RETURN_CODES_ENUM eResult = EPG_RET_OK;
    EPG_AU_INFO_STRUCT sAuData;

    SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
        PC_GREEN"eEpgParserParseSavedTextAus(): Segments TEXT collected: %u of %u.\n"PC_RESET,
        psEpgParser->un8NumSegmentsTextCollected, psEpgParser->un8NumSegmentsToRecieve);

    if (bReprocessAll == TRUE)
    {
        // Reset counter to start from the beginning
        psEpgParser->un8NumSegmentsTextCollected = 0;
    }
    else
    {
        if (psEpgParser->un8NumSegmentsTextCollected ==
            psEpgParser->un8NumSegmentsToRecieve)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 6,
                PC_BLUE"WARNING: All TEXT segments are already collected.\n"PC_RESET);
            return EPG_RET_IGNORED;
        }
    }

    // Iterate through segments
    while (psEpgParser->un8NumSegmentsTextCollected <
           psEpgParser->un8NumSegmentsToRecieve)
    {
        // Get segment
        EPG_SEGMENT_STRUCT *psSegment =
            &(psEpgParser->asSegments[psEpgParser->un8NumSegmentsTextCollected]);
        // Get segment's text data
        EPG_SEG_TEXT_DATA_STRUCT *psSegTextData = &(psSegment->sSegTextData);
        // Get text data related header
        EPG_AU_TEXT_HEADER *psTextAuHeader = &(psSegTextData->sTextAuHeader);

        // Stop iterating if no AUs were received for required segment
        if (psSegTextData->asAuTextData == NULL ||
            psSegTextData->sTextAuHeader.un32NumAuTotal == 0)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                PC_GREEN"Segment %u TEXT is empty. Further parsing is impossible yet.\n"PC_RESET,
                psEpgParser->un8NumSegmentsTextCollected);
            return EPG_RET_OK;
        }

        if (bReprocessAll == TRUE)
        {
            // Reset counter to start from the beginning of a segment
            psSegTextData->un32NumAuCollected = 0;
        }

        // Iterate through Access Units
        while (psSegTextData->un32NumAuCollected <
               psSegment->sSegTextData.sTextAuHeader.un32NumAuTotal)
        {
            BOOLEAN bResult = FALSE;
            // Get text AU data
            EPG_AU_DATA_STRUCT *psAuTextData =
                &(psSegTextData->asAuTextData[psSegTextData->un32NumAuCollected]);

            // Set processing AU sequence number
            psTextAuHeader->un32AuSequenceNumber = psSegTextData->un32NumAuCollected;

            // Check if cancelled
            bResult = bEpgParserIsCancelled(psEpgParser);
            if (bResult == TRUE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_GREEN"Parsing is cancelled.\n"PC_RESET);
                return EPG_RET_CANCELLED;
            }

            // Stop iterating if required AU was not received yet
            if (psAuTextData->pvAuPayload == NULL)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 1,
                    PC_GREEN"TEXT AU %u:%u is not yet loaded. Further parsing is impossible yet.\n"PC_RESET,
                    psSegment->un8SegmentNumber, psSegTextData->un32NumAuCollected);
                return EPG_RET_OK;
            }

            // Send Compressed Text AU payload data to parser's client
            sAuData.eAuDataType = EPG_DATA_TYPE_COMPRESSED_TEXT;
            sAuData.un8SegmentNumber = psSegment->un8SegmentNumber;
            sAuData.un32AuNumber = psTextAuHeader->un32AuSequenceNumber;
            sAuData.pvData = psAuTextData->pvAuPayload;
            sAuData.tDataSize = psAuTextData->tPayloadSize;
            bResult = psEpgParser->psEpgClient->bProcessCompressedTextPayload(psEpgParser->psEpgClient->hClientObj,
                                                                              &sAuData);
            if (bResult == FALSE)
            {
                SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 0,
                    PC_RED"ERROR: Failed to process TEXT AU %u:%u.\n"PC_RESET,
                    psSegment->un8SegmentNumber, psSegTextData->un32NumAuCollected);
                // Clean failed AU, so it can be reloaded from the broadcast later
                vEpgParserCleanAu(psAuTextData);
                return eResult;
            }

            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"TEXT AU %u:%u \t IS PROCESSED.\n"PC_RESET,
                psSegment->un8SegmentNumber,
                psTextAuHeader->un32AuSequenceNumber);

            psSegTextData->un32NumAuCollected++;
        }

        SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
            PC_GREEN"=:=:=:=:=  SEGMENT %u TEXT IS COMPLETED =:=:=:=:=\n"PC_RESET,
            psSegment->un8SegmentNumber);

        // Current segment is parsed. Parse next segment from the beginning.
        bReprocessAll = TRUE;

        psEpgParser->un8NumSegmentsTextCollected++;

        // Notify about Segment's TEXT completion
        psEpgParser->psEpgClient->vSegmentTextCompleted(psEpgParser->psEpgClient->hClientObj,
                                                        psSegment->un8SegmentNumber,
                                                        psTextAuHeader->un16StringsCount,
                                                        psTextAuHeader->un32DecompressedFileSize);

        // Check if a whole segment is completed
        if (psSegment->sSegGridData.un32NumAuCollected > 0 &&
            psSegment->sSegGridData.un32NumAuCollected ==
            psSegment->sSegGridData.un32NumAuTotal)
        {
            SMSAPI_DEBUG_vPrint(EPG_PARSER_DBG_PREFIX, 5,
                PC_GREEN"=:=:=:=:= WHOLE SEGMENT %u IS COMPLETED =:=:=:=:=\n"PC_RESET,
                psSegment->un8SegmentNumber);

            // Notify about whole segment completion
            psEpgParser->psEpgClient->vWholeSegmentCompleted(psEpgParser->psEpgClient->hClientObj,
                                                             psSegment->un8SegmentNumber);
        }
    };

    return EPG_RET_OK;
}
