/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _WEATHER_PVN1_H_
#define _WEATHER_PVN1_H_

#include "standard.h"

/**************/
/** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "weather_interface.h"
#include "rfd_interface_obj.h"

/***************/
/** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define WEATHER1_OBJECT_NAME "Weather1"

#define WEATHER1_DSI (440)

// The protocol version supported
// by SMS for this interface
#define WEATHER1_PVN (PVN)(1)

//Protocol related definitions
#define WEATHER1_FORECAST_EVENT_BITLEN (6)
#define WEATHER1_FORECAST_FLAG_BITLEN (1)
#define WEATHER1_FORECAST_TEMP_BITLEN (8)
#define WEATHER1_FORECAST_TMAX_BITLEN (8)
#define WEATHER1_FORECAST_TMIN_BITLEN (8)
#define WEATHER1_FORECAST_POP_BITLEN (3)
#define WEATHER1_FORECAST_PRECIP_BITLEN (7)
#define WEATHER1_FORECAST_WIND_BITLEN (8)
#define WEATHER1_FORECAST_HUMID_BITLEN (3)
#define WEATHER1_FORECAST_CLOUD_BITLEN (3)
#define WEATHER1_FORECAST_UV_BITLEN (3)
#define WEATHER1_FORECAST_AIR_BITLEN (3)
#define WEATHER1_FORECAST_POLLEN_BITLEN (4)
#define WEATHER1_FORECAST_EXTCNT_BITLEN (8)
#define WEATHER1_FORECAST_EXTDATA_ITEM_BITLEN (8)

#define WEATHER1_SKI_FLAG_BITLEN (1)
#define WEATHER1_SKI_OPSTAT_BITLEN (1)
#define WEATHER1_SKI_EVENT_BITLEN (6)
#define WEATHER1_SKI_TEMP_BITLEN (8)
#define WEATHER1_SKI_SWIND_BITLEN (3)
#define WEATHER1_SKI_SNOW_BITLEN (3)
#define WEATHER1_SKI_BMIN_BITLEN (6)
#define WEATHER1_SKI_BMAX_BITLEN (6)
#define WEATHER1_SKI_NEW_BITLEN (4)
#define WEATHER1_SKI_LIFT_BITLEN (5)
#define WEATHER1_SKI_TRAIL_BITLEN (4)
#define WEATHER1_SKI_MAKE_BITLEN (1)
#define WEATHER1_SKI_GROOM_BITLEN (1)
#define WEATHER1_SKI_NIGHT_BITLEN (1)
#define WEATHER1_SKI_BOARD_BITLEN (1)
#define WEATHER1_SKI_EXTCNT_BITLEN (8)
#define WEATHER1_SKI_EXTDATA_ITEM_BITLEN (8)

//  General access unit definitions
#define WEATHER1_CRC_BYTELEN (4)
#define WEATHER1_PVN_BITLEN (4)
#define WEATHER1_CAROUSEL_BITLEN (3)

// Location message related defines
#define WEATHER1_FCST_BITLEN (4)
#define WEATHER1_FCST_MAX (9)
#define WEATHER1_ASEQ_BITLEN (2)
#define WEATHER1_ASEQ_NEW_STATE_FIRST_LOC (2)
#define WEATHER1_ASEQ_NEW_STATE_NEW_LOC (3)
#define WEATHER1_ASEQ_SAME_STATE_NEW_LOC (1)
#define WEATHER1_ASEQ_SAME_STATE_NEXT_LOC (0)
#define WEATHER1_STATE_BITLEN (7)
#define WEATHER1_LOCID_BITLEN (6)

// Header sizes
#define WEATHER1_FORECAST_HEADER_SIZE (WEATHER1_PVN_BITLEN + \
    WEATHER1_CAROUSEL_BITLEN + WEATHER1_FCST_BITLEN)

#define WEATHER1_SKI_REPORT_HEADER_SIZE (WEATHER1_PVN_BITLEN + \
    WEATHER1_CAROUSEL_BITLEN)

// Carousel IDs
#define WEATHER1_REPORT_CAROUSEL_ID (0)
#define WEATHER1_SKI_CAROUSEL_ID (1)
#define WEATHER1_LOCATION_UPDATE_CAROUSEL_ID (2)
#define WEATHER1_METADATA_UPDATE_CAROUSEL_ID (3)

// Location update file definitions
#define WEATHER1_LOCATION_VER_BITLEN (6)
#define WEATHER1_LOCATION_UTYPE_BITLEN (2)
#define WEATHER1_LOCATION_STATE_BITLEN (7)
#define WEATHER1_LOCATION_LOCID_BITLEN (6)
#define WEATHER1_LOCATION_SKI_BITLEN (1)
#define WEATHER1_LOCATION_LOCLAT_BITLEN (20)
#define WEATHER1_LOCATION_LOCLON_BITLEN (20)
#define WEATHER1_LOCATION_LNAME_NUM_SYMBOLS (31)
#define WEATHER1_LOCATION_LCODE_NUM_SYMBOLS (5)

// defines for lat/lon calculations
#define WEATHER1_LAT_LON_DIVSOR_VALUE    (8192)
#define WEATHER1_LON_BASE_VALUE          (-50)

// As defined in rfd_receiver_config.h
#define WEATHER1_RFD_CLIENT_ID    (2)

#define WEATHER_RFD_READ_BLOCK_SIZE (2048)

// Cache file name
#define WEATHER1_CACHE_FILE_NAME "cache.bin"

// Folder for persistent storage
#define WEATHER1_PERSISTENT_FOLDER_NAME "weather"

#define WEATHER1_CACHE_SCHEMA_VER (2)

#define WEATHER1_DATA_BUFFER_SIZE (100)

// Air Quality Coding
#define WEATHER1_AIRQ_GOOD           (0)
#define WEATHER1_AIRQ_MODERATE       (1)
#define WEATHER1_AIRQ_UNHEALTHY_SENS (2)
#define WEATHER1_AIRQ_UNHEALTY       (3)
#define WEATHER1_AIRQ_VERY_UNHEATHLY (4)
#define WEATHER1_AIRQ_DANGEROUS      (5)
#define WEATHER1_AIRQ_RESERVED       (6)
#define WEATHER1_AIRQ_NA             (7)

// Cloud Cover Coding
#define WEATHER1_CLOUDS_CLEAR (0)
#define WEATHER1_CLOUDS_FEW (1)
#define WEATHER1_CLOUDS_SCATTERED (2)
#define WEATHER1_CLOUDS_BROKEN (3)
#define WEATHER1_CLOUDS_OVERCAST (4)
#define WEATHER1_CLOUDS_RSVD1 (5)
#define WEATHER1_CLOUDS_RSVD2 (6)
#define WEATHER1_CLOUDS_RSVD3 (7)

// Constant which defines the bit width of the maximum file update
// version, which is then passed to RFD_INTERFACE_hConnect.
// This is used in determining the applicability of incoming
// RFD updates.
//
// Tabular weather's RFD filename still comes in the same UXXYY form
// that the other "max 99" services do; however, Leslie French confirmed that
// the tab weather service will only use up to a maximum version of 63.
//
// Note: 63 => 11 1111 = 6 bits
#define WEATHER1_MAX_VERSION_BITLEN ( 6 )

/* Max message sizes (from SX-9845-0031)
*   Section 4.3 says:
*      The maximum size of any Access Unit is 5120 bytes.
*
*   In addition, let's make sure we have enough memory for
*   a few queued messages -- let's say 4 for weather forecast messages
*   and 4 for ski report messages (8 queued messages total).
*
*/
#define WEATHER1_PAYLOAD_MAX_SIZE (5120)
#define WEATHER1_NUM_OF_QUEUED_PAYLOADS (8)
#define WEATHER1_RFD_META_PAYLOAD_MAX_SIZE (5120)
#define WEATHER1_RFD_BLOCK_PAYLOAD_MAX_SIZE (5120)

// Number of cached payloads is equal to number of forecast types and
// 1 payload is for ski report (total 11)
#define WEATHER1_NUM_OF_CACHED_PAYLOADS (WEATHER_FCST_TYPE_MAX)

// Cache block pool
#define WEATHER1_BLOCK_POOL_BUF_SIZE (WEATHER1_PAYLOAD_MAX_SIZE)

// Number of blocks in block pool. It is equal to number of cached 
// payloads plus one block used for CRC calculation purpose
#define WEATHER1_BLOCK_POOL_BUF_NUM (WEATHER1_NUM_OF_CACHED_PAYLOADS + 1)

/**************/
/** TYPEDEFS **/
/**************/

typedef enum weather_fcst_type_enum
{
    WEATHER_FCST_TYPE_CURRENT = 0,
    WEATHER_FCST_TYPE_0_3_HOUR,
    WEATHER_FCST_TYPE_3_6_HOUR,
    WEATHER_FCST_TYPE_SUNDAY,
    WEATHER_FCST_TYPE_MONDAY,
    WEATHER_FCST_TYPE_TUESDAY,
    WEATHER_FCST_TYPE_WEDNESDAY,
    WEATHER_FCST_TYPE_THURSDAY,
    WEATHER_FCST_TYPE_FRIDAY,
    WEATHER_FCST_TYPE_SATURDAY,
    WEATHER_FCST_TYPE_SKI,
    WEATHER_FCST_TYPE_MAX
} WEATHER_FCST_TYPE_ENUM;


typedef struct weather_hash_item_data_struct
{
    UN8 un8FirstStateID;
    UN8 un8FirstAreaID;
    OSAL_CRC_RESULT tCRC;
    size_t tAUSize;
} WEATHER_HASH_ITEM_DATA_STRUCT;

typedef struct weather1_cache_file_struct
{
    UN32 un32TimestampUTCsec;
    WEATHER_HASH_ITEM_DATA_STRUCT sHash;
} WEATHER1_CACHE_FILE_STRUCT;

typedef struct weather1_object_struct
{
    // Handle to the top-level weather service
    WEATHER_SERVICE_OBJECT hWeatherService;

    // CRC handle (used to compute ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // RFD Handle
    RFD_INTERFACE_OBJECT hRFD;

    //last hash data struct is stored here
    WEATHER_HASH_ITEM_DATA_STRUCT sHash;

    //Ski reports enable flag
    BOOLEAN bSkiEnabled;

    //Folder path to store cache
    char *pacPersistentFolderPath;

    // Path to weather data cache file
    char *pacCacheFilePath;

    // Block pull used to read/write cache data
    OSAL_OBJECT_HDL hBlockPool;

    // Array of index file data
    WEATHER1_CACHE_FILE_STRUCT asCacheItem[WEATHER_FCST_TYPE_MAX];

    // Array of cached payloads
    OSAL_BUFFER_HDL ahPayload[WEATHER_FCST_TYPE_MAX];

    // Flag which determines if RAM cache now differs from
    // NVM cache and should be synchronized
    BOOLEAN bCacheSyncNeeded;

} WEATHER1_OBJECT_STRUCT;

typedef struct weather1_rfd_ctrl_struct
{
    WEATHER_SERVICE_OBJECT hWeatherService;
    RFD_INTERFACE_OBJECT hRFD;

    // Our path / connection to the DB under construction
    SQL_INTERFACE_OBJECT hDBConnection;
    STRING_OBJECT hDBPath;

    // Progress tracking
    RFD_PROGRESS_INDEX tCurProgressIndex;
    RFD_PROGRESS_INDEX tStartProgressIndex;

    // The block pool used to process RFD files
    OSAL_OBJECT_HDL hBlockPool;

    // The buffer used to access the block pool
    OSAL_BUFFER_HDL hBuffer;

    //Lat/lon divisor value for loc updates
    OSAL_FIXED_OBJECT hDegDivsor;

    //Longitude base value for loc updates
    OSAL_FIXED_OBJECT hLonBase;

    // Location row used in RFD processing
    WEATHER_LOCATION_ROW_STRUCT sLocation;

    // Buffer used to generate SQL commands
    char acBuffer[WEATHER_MAX_SQL_STRING_LENGTH];

} WEATHER1_RFD_CTRL_STRUCT;


typedef struct wind_pvn1_data_struct
{
    WIND_UNPARSED_DATA tValue;        //protocol value
    UN16 un16Direction;                //value in degrees
} WIND_PVN1_DATA_STRUCT;

typedef struct weather_event_pvn1_data_struct
{
    WEATHER_EVENT_UNPARSED_DATA tValue;        //protocol value
    UN8 un8EventCode;                        //event code value
    char *pcDescription;                    //event description string
    UN8 un8IconCode;                        //icon code related to this event
    UN8 un8AlternativeIconCode1;            //alternative icon 1 code related to this event
    UN8 un8AlternativeIconCode2;            //alternative icon 2 code related to this event
} WEATHER_EVENT_PVN1_DATA_STRUCT;

typedef struct uv_info_pvn1_data_struct
{
    UV_INFO_UNPARSED_DATA tValue;            //protocol value
    UN8 un8Low;                                //UV index range low value
    UN8 un8High;                            //UV index range high value
    UV_RISK_ENUM eRisk;                        //UV risk enum value
} UV_INFO_PVN1_DATA_STRUCT;

typedef struct new_snow_pvn1_data_struct
{
    NEW_SNOW_UNPARSED_DATA tValue;            //protocol value
    UN8 un8Low;                                //new snow range low value
    UN8 un8High;                            //new snow range high value
} NEW_SNOW_PVN1_DATA_STRUCT;

typedef struct trails_pvn1_data_struct
{
    TRAILS_UNPARSED_DATA tValue;            //protocol value
    UN8 un8Low;                                //num of trails range low value
    UN8 un8High;                            //num of trails range high value
} TRAILS_PVN1_DATA_STRUCT;

typedef struct precip_amount_pvn1_data_struct
{
    PRECIP_UNPARSED_DATA tValue;            //protocol value
    UN8 un8LowWhole;                        //whole part of low value for precipitation amount
    UN8 un8LowFrac;                            //fractional part of low value for precipitation amount
    UN8 un8LowPow;                            //negative power of ten of low value for precipitation amount
    UN8 un8HighWhole;                        //whole part of high value for precipitation amount
    UN8 un8HighFrac;                        //fractional part of high value for precipitation amount
    UN8 un8HighPow;                            //negative power of ten of high value for precipitation amount
} PRECIP_AMOUNT_PVN1_DATA_STRUCT;

typedef struct humidity_pvn1_data_struct
{
    HUMIDITY_UNPARSED_DATA tValue;            //protocol value
    UN8 un8Low;                                //low range value of humidity
    UN8 un8High;                            //high range value of humidity
} HUMIDITY_PVN1_DATA_STRUCT;

typedef struct pollen_pvn1_data_struct
{
    POLLEN_UNPARSED_DATA tValue;            //protocol value
    AIR_POLLEN_LEVEL_ENUM eLevel;            //enum value of pollen level
} POLLEN_PVN1_DATA_STRUCT;

typedef enum weather1_location_update_type_enum
{
    WEATHER_MGR_LOCATION_DELETE = 0,
    WEATHER_MGR_LOCATION_NEW,
    WEATHER_MGR_LOCATION_MODIFY,
    WEATHER_MGR_LOCATION_EOL

} WEATHER1_LOCATION_UPDATE_TYPE_ENUM;

typedef enum weather1_msg_type_enum
{
    WEATHER_FORECAST_REPORT_MSG = 0,
    WEATHER_SKI_REPORT_MSG,
    WEATHER_LOCATION_DB_UPDATE_MSG,
    WEATHER_METADATA_UPDATE_MSG,
    WEATHER_INVALID_MSG
} WEATHER1_MSG_TYPE_ENUM;

typedef struct weather1_parse_iterator_arg
{
    WEATHER1_OBJECT_STRUCT *psObj;
    WEATHER_HASH_OBJECT hHashEntry;
    WEATHER_FCST_TYPE_ENUM eFcst;
    SMS_OBJECT hParent;
    OSAL_OBJECT_HDL hObjectsList;
} WEATHER1_PARSE_ITERATOR_ARG;


/************/
/** MACROS **/
/************/

/****************/
/** PROTOTYPES **/
/****************/

// Public interface APIs

static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        );

static BOOLEAN bProcessWindData (
    WIND_UNPARSED_DATA tData,
    WIND_DATA_STRUCT *psData
        );

static BOOLEAN bProcessWeatherEventData (
    WEATHER_EVENT_UNPARSED_DATA tData,
    WEATHER_EVENT_DATA_STRUCT *psData
        );

static BOOLEAN bProcessUvInfoData (
    UV_INFO_UNPARSED_DATA tData,
    UV_INFO_DATA_STRUCT *psData
        );

static BOOLEAN bProcessNewSnowData (
    NEW_SNOW_UNPARSED_DATA tData,
    NEW_SNOW_DATA_STRUCT *psData
        );

static BOOLEAN bProcessTrailsData (
    TRAILS_UNPARSED_DATA tData,
    TRAILS_DATA_STRUCT *psData
        );

static BOOLEAN bProcessPrecipitationData (
    PRECIP_UNPARSED_DATA tData,
    PRECIP_DATA_STRUCT *psData
        );

static BOOLEAN bProcessHumidityData (
    HUMIDITY_UNPARSED_DATA tData,
    HUMIDITY_DATA_STRUCT *psData
        );

static BOOLEAN bProcessPollenData (
    POLLEN_UNPARSED_DATA tData,
    POLLEN_DATA_STRUCT *psData
        );

static WEATHER_INTERFACE_OBJECT hInit(
    WEATHER_SERVICE_OBJECT hWeatherService,
    SMS_OBJECT hParent,
    BOOLEAN bDBUpdatesEnabled,
    BOOLEAN bClearPersistent,
    UN8 un8CurDBVersion
        );

static void vUnInit(
    WEATHER_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessMessage(
        WEATHER_INTERFACE_OBJECT hInterface,
        OSAL_BUFFER_HDL *phPayload
            );

static BOOLEAN bEnableSkiReports(
    WEATHER_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bDisableSkiReports(
    WEATHER_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessTimerCall(WEATHER_INTERFACE_OBJECT hInterface);

static BOOLEAN bIsTimerNeeded(WEATHER_INTERFACE_OBJECT hInterface);

static BOOLEAN bLoadForecastData (
    WEATHER_INTERFACE_OBJECT hInterface,
    SMS_OBJECT hParent,
    OSAL_OBJECT_HDL hWeatherMsgsList
        );

static BOOLEAN bLoadSkiData (
    WEATHER_INTERFACE_OBJECT hInterface,
    SMS_OBJECT hParent,
    OSAL_OBJECT_HDL hWeatherMsgsList
        );

/* Object Private Prototypes */

static BOOLEAN bInitRFD(
    WEATHER1_OBJECT_STRUCT *psObj,
    RFD_UPDATE_VERSION tCurrentVersion
        );

static WEATHER1_RFD_CTRL_STRUCT *psCreateRFDCtrl (
    WEATHER1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bInitRFDCtrl (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex
        );

static void vDestroyRFDCtrl (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bConnectToDB (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        );

static BOOLEAN bFinalizeDB (
   WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static void vDeleteDB (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static void vUninitObject(
    WEATHER1_OBJECT_STRUCT *psObj
        );

static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        );

static BOOLEAN bProcessRFDHeader (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psRFDFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static RFD_PROCESS_RESULT_ENUM eProcessUpdateGroup (
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    BOOLEAN bInformManager
        );

static BOOLEAN bGetMessageType(
        WEATHER1_OBJECT_STRUCT *psObj,
        OSAL_BUFFER_HDL hPayload,
        WEATHER1_MSG_TYPE_ENUM *peMsgType
            );

static BOOLEAN bParseForecastData (
    UN8 *pun8Source,
    FORECAST_DATA_STRUCT *psData,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        );

static BOOLEAN bProcessHash(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    BOOLEAN bSkiFlag,
    WEATHER_FCST_TYPE_ENUM *peFcst
        );

static BOOLEAN bProcessLocationDelete(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        );

static BOOLEAN bProcessLocationModify(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        );

static BOOLEAN bProcessLocationNew(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bInformManager
        );

static BOOLEAN bProcessSkiReport (
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bProcessWeatherForecast (
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static BOOLEAN bReadLocId(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bReadLocInfo(
    WEATHER1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static LOC_ID tGetPayloadID(
    UN8 *pun8Source,
    LOC_ID tPrevID,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        );

static BOOLEAN bCheckBufferCRC(
    OSAL_OBJECT_HDL hCRC,
    OSAL_BUFFER_HDL hBuffer,
    OSAL_CRC_RESULT tCRC,
    size_t tBufferSize
        );

static void vCancelHashItems(
        WEATHER1_OBJECT_STRUCT *psObj,
        UN32 un32CurrentTime
            );

static void vDestroyLocData(
    WEATHER_LOCATION_ROW_STRUCT *psLoc
        );

static void vParseSkiMsgs(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    WEATHER_HASH_OBJECT hHashEntry
        );

static void vParseWeatherMsgs(
    WEATHER1_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    WEATHER_HASH_OBJECT hHashEntry,
    WEATHER_FCST_TYPE_ENUM eFcst
        );

OSAL_BUFFER_ITERATOR_RESULT_ENUM eParseForecasts(
    UN8 *pun8Source,
    size_t tDataSize,
    WEATHER1_PARSE_ITERATOR_ARG *psIteratorArg
        );

static BOOLEAN bParseSkiData(
    UN8 *pun8Source,
    SKI_DATA_STRUCT *psData,
    size_t tBufferSizeInBits,
    size_t *ptBitsOffset
        );

static OSAL_BUFFER_ITERATOR_RESULT_ENUM eParseSkiReports(
    UN8 *pun8Source,
    size_t tDataSize,
    WEATHER1_PARSE_ITERATOR_ARG *psIteratorArg
        );

static BOOLEAN bReadUN8(
    UN8 un8Length,
    size_t *ptPosition,
    size_t tSourceBitLength,
    UN8 *pun8Source,
    UN8 *pun8Dest
        );

static BOOLEAN bReadFlag(
    size_t *ptPosition,
    size_t tSourceBitLength,
    UN8 *pun8Source,
    UN8 *pun8Dest
        );

static void vDeleteCache (
    WEATHER1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bReadCacheFile(
    WEATHER1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bReadCachedPayload(
    WEATHER1_OBJECT_STRUCT *psObj,
    WEATHER1_CACHE_FILE_STRUCT *psCacheItem,
    size_t tOffset,
    FILE *psCacheFile,
    OSAL_BUFFER_HDL hBuffer
        );

static BOOLEAN bWriteCacheFile(
    WEATHER1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bBuildCacheFilePath(
    WEATHER1_OBJECT_STRUCT *psObj
        );

static BOOLEAN bBuildPersistentFolderPath(
    WEATHER1_OBJECT_STRUCT *psObj
        );

/* Message Processors */

/***************/
/** VARIABLES **/
/***************/

static const WIND_PVN1_DATA_STRUCT gasWindDataStruct[] =
{
    {0, 0},
    {1, 22},
    {2, 45},
    {3, 67},
    {4, 90},
    {5, 112},
    {6, 135},
    {7, 157},
    {8, 180},
    {9, 202},
    {10, 225},
    {11, 247},
    {12, 270},
    {13, 292},
    {14, 315},
    {15, 337}
};

static const WEATHER_EVENT_PVN1_DATA_STRUCT gasWeatherEventDataStruct[] =
{
    {0, 0, "No Data Available", 0, 0, 0},
    {1, 1, "Unknown Precipitation", 0, 0, 0},
    {2, 2, "Isolated Thunderstorms", 84, 0, 0},
    {3, 3, "Scattered Thunderstorms", 98, 0, 0},
    {4, 4, "Scattered Thunderstorms (night)", 84, 0, 0},
    {5, 5, "Severe Thunderstorms", 84, 108, 109},
    {6, 6, "Thunderstorms", 84, 0, 0},
    {7, 7, "Rain", 82, 0, 0},
    {8, 8, "Light Rain", 87, 0, 0},
    {9, 9, "Heavy Rain", 82, 0, 0},
    {10, 10, "Scattered Showers", 87, 0, 0},
    {11, 11, "Scattered Showers (night)", 86, 0, 0},
    {12, 12, "Showers", 87, 0, 0},
    {13, 13, "Drizzle", 76, 0, 0},
    {14, 14, "Freezing Drizzle", 90, 0, 0},
    {15, 15, "Freezing Rain", 89, 0, 0},
    {16, 16, "Wintry Mix", 79, 0, 0},
    {17, 17, "Mixed Rain And Snow", 79, 0, 0},
    {18, 18, "Mixed Rain And Sleet", 88, 0, 0},
    {19, 19, "Mixed Rain And Hail", 89, 0, 0},
    {20, 20, "Hail", 110, 0, 0},
    {21, 21, "Sleet", 88, 0, 0},
    {22, 22, "Ice Pellets", 110, 0, 0},
    {23, 23, "Flurries", 77, 0, 0},
    {24, 24, "Light Snow", 74, 0, 0},
    {25, 25, "Moderate Snow", 83, 0, 0},
    {26, 26, "Snow", 83, 0, 0},
    {27, 27, "Heavy Snow", 83, 0, 0},
    {28, 28, "Scattered Snow Showers", 74, 0, 0},
    {29, 29, "Scattered Snow Showers (night)", 96, 0, 0},
    {30, 30, "Snow Showers (night)", 74, 0, 0},
    {31, 31, "Blowing Snow", 81, 0, 0},
    {32, 32, "Blizzard", 80, 0, 0},
    {33, 33, "Sandstorm", 111, 0, 0},
    {34, 34, "Blowing Dust", 112, 0, 0},
    {35, 35, "Dust", 68, 0, 0},
    {36, 36, "Foggy", 70, 0, 0},
    {37, 37, "Light Fog", 70, 0, 0},
    {38, 38, "Moderate Fog", 70, 0, 0},
    {39, 39, "Heavy Fog", 70, 0, 0},
    {40, 40, "Mist", 95, 0, 0},
    {41, 41, "Hazy", 72, 113, 0},
    {42, 42, "Smoky", 75, 99, 0},
    {43, 43, "Blustery", 78, 0, 0},
    {44, 44, "Windy", 78, 0, 0},
    {45, 45, "Cold", 73, 0, 0},
    {46, 46, "Hot", 71, 0, 0},
    {47, 47, "Sunny", 85, 100, 0},
    {48, 48, "Mostly Sunny", 66, 106, 0},
    {49, 49, "Clear (night)", 101, 0, 0},
    {50, 50, "Mostly Clear (night)", 107, 0, 0},
    {51, 51, "Partly cloudy", 66, 106, 0},
    {52, 52, "Partly cloudy (night)", 107, 0, 0},
    {53, 53, "Mostly cloudy", 69, 104, 0},
    {54, 54, "Mostly cloudy (night)", 105, 0, 0},
    {55, 55, "Cloudy", 67, 102, 0},
    {56, 56, "Tropical Storm", 117, 118, 0},
    {57, 57, "Hurricane", 115, 116, 0},
    {58, 58, "Funnel Cloud", 103, 0, 0},
    {59, 59, "Tornado", 103, 0, 0},
    {60, 60, "Reserved", 0, 0, 0},
    {61, 61, "Reserved", 0, 0, 0},
    {62, 62, "Reserved", 0, 0, 0},
    {63, 63, "Reserved", 0, 0, 0}
};

static const UV_INFO_PVN1_DATA_STRUCT gasUvInfoDataStruct[] =
{
    {0, 0, 2, UV_RISK_LOW},
    {1, 3, 3, UV_RISK_MODERATE},
    {2, 4, 5, UV_RISK_MODERATE},
    {3, 6, 7, UV_RISK_HIGH},
    {4, 8, 8, UV_RISK_VERYHIGH},
    {5, 9, 9, UV_RISK_VERYHIGH},
    {6, 10, 10, UV_RISK_VERYHIGH},
    {7, 11, UN8_MAX, UV_RISK_EXTREME}
};

static const NEW_SNOW_PVN1_DATA_STRUCT gasNewSnowDataStruct[] =
{
    {0, 0, 0},
    {1, 0, 1},
    {2, 1, 2},
    {3, 2, 3},
    {4, 3, 4},
    {5, 4, 6},
    {6, 6, 8},
    {7, 8, 10},
    {8, 10, 12},
    {9, 12, 16},
    {10, 16, 20},
    {11, 20, 24},
    {12, 24, 30},
    {13, 30, 36},
    {14, 36, 42},
    {15, 42, UN8_MAX}
};

static const UN8 gaun8ChanceOfPrecipitation[] =
{
    /* 0x00 */ 0,
    /* 0x01 */ 10,
    /* 0x02 */ 20,
    /* 0x03 */ 30,
    /* 0x04 */ 50,
    /* 0x05 */ 70,
    /* 0x06 */ 90,
    /* 0x07 */ 100
};

static const PRECIP_AMOUNT_PVN1_DATA_STRUCT gasPrecipAmountDataStruct[] =
{
    {0, 0, 0, 0, 0, 1, 2},
    {1, 0, 1, 2, 0, 1, 1},
    {2, 0, 1, 1, 0, 25, 2},
    {3, 0, 25, 2, 0, 5, 1},
    {4, 0, 5, 1, 0, 75, 2},
    {5, 0, 75, 2, 1, 0, 0},
    {6, 1, 0, 0, 1, 5, 1},
    {7, 1, 5, 1, 2, 0, 0},
    {8, 2, 0, 0, 2, 5, 1},
    {9, 2, 5, 1, 3, 0, 0},
    {10, 3, 0, 0, 4, 0, 0},
    {11, 4, 0, 0, 5, 0, 0},
    {12, 5, 0, 0, 6, 0, 0},
    {13, 6, 0, 0, 8, 0, 0},
    {14, 8, 0, 0, 10, 0, 0},
    {15, 10, 0, 0, 12, 0, 0},
    {16, 12, 0, 0, 14, 0, 0},
    {17, 14, 0, 0, 16, 0, 0},
    {18, 16, 0, 0, 20, 0, 0},
    {19, 20, 0, 0, 24, 0, 0},
    {20, 24, 0, 0, 28, 0, 0},
    {21, 28, 0, 0, 32, 0, 0},
    {22, 32, 0, 0, 36, 0, 0},
    {23, 36, 0, 0, 40, 0, 0},
    {24, 40, 0, 0, 44, 0, 0},
    {25, 44, 0, 0, 48, 0, 0},
    {26, 48, 0, 0, 52, 0, 0},
    {27, 52, 0, 0, 56, 0, 0},
    {28, 56, 0, 0, 60, 0, 0},
    {29, 60, 0, 0, 64, 0, 0},
    {30, 64, 0, 0, 0, 0, 0},
    {31, 0, 0, 0, 0, 0, 0}
};

static const TRAILS_PVN1_DATA_STRUCT gasTrailsDataStruct[] =
{
    {0, 0, 0},
    {1, 1, 5},
    {2, 6, 10},
    {3, 11, 15},
    {4, 16, 20},
    {5, 21, 25},
    {6, 26, 30},
    {7, 31, 35},
    {8, 36, 40},
    {9, 41, 45},
    {10, 46, 50},
    {11, 51, 60},
    {12, 61, 80},
    {13, 81, 100},
    {14, 101, 150},
    {15, 151, UN8_MAX}
};


static const HUMIDITY_PVN1_DATA_STRUCT gasHumidityDataStruct[] =
{
    {0, 0, 10},
    {1, 10, 20},
    {2, 20, 30},
    {3, 30, 50},
    {4, 50, 70},
    {5, 70, 80},
    {6, 80, 90},
    {7, 90, 100}
};

static const POLLEN_PVN1_DATA_STRUCT gasPollenDataStruct[] =
{
    {0, AIR_POLLEN_LEVEL_LOW},
    {1, AIR_POLLEN_LEVEL_LOW},
    {2, AIR_POLLEN_LEVEL_LOW},
    {3, AIR_POLLEN_LEVEL_LOW},
    {4, AIR_POLLEN_LEVEL_MODERATE},
    {5, AIR_POLLEN_LEVEL_MODERATE},
    {6, AIR_POLLEN_LEVEL_MODERATE},
    {7, AIR_POLLEN_LEVEL_MODERATE},
    {8, AIR_POLLEN_LEVEL_HIGH},
    {9, AIR_POLLEN_LEVEL_HIGH},
    {10, AIR_POLLEN_LEVEL_HIGH},
    {11, AIR_POLLEN_LEVEL_HIGH},
    {12, AIR_POLLEN_LEVEL_HIGH}
};

const WEATHER_PLUGIN_INTERFACE_STRUCT GsWeatherIntf =
{
    /*.tDSI = */WEATHER1_DSI,
    /*.tMaxVersionBitlen =*/WEATHER1_MAX_VERSION_BITLEN,
    /*.tMinimumOTABufferByteSize = */tMinimumOTABufferByteSize,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProcessMessage = */bProcessMessage,
    /*.bEnableSkiReports = */bEnableSkiReports,
    /*.bDisableSkiReports = */bDisableSkiReports,
    /*.bProcessWindData = */bProcessWindData,
    /*.bProcessWeatherEventData = */bProcessWeatherEventData,
    /*.bProcessUvInfoData = */bProcessUvInfoData,
    /*.bProcessNewSnowData = */bProcessNewSnowData,
    /*.bProcessTrailsData = */bProcessTrailsData,
    /*.bProcessPrecipitationData = */bProcessPrecipitationData,
    /*.bProcessHumidityData = */bProcessHumidityData,
    /*.bProcessPollenData = */bProcessPollenData,
    /*.bIsTimerNeeded = */bIsTimerNeeded,
    /*.bProcessTimerCall = */bProcessTimerCall,
    /*.bLoadForecastData = */bLoadForecastData,
    /*.bLoadSkiData = */bLoadSkiData
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/



#endif /* _WEATHER_PVN1_H_ */
