/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _STOCK_TICKER_PROTOCOL_H_
#define _STOCK_TICKER_PROTOCOL_H_

  /**************/
 /** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "stock_ticker_interface.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define STOCK_TICKER_PROTOCOL_OBJECT_NAME "ST_PROTO"

// The protocol version supported
// by SMS for this interface
#define STOCK_TICKER_DSI (DSI)(500)

// Defines buffer size
#define STOCK_TICKER_DATA_BUFFER_SIZE (256)

//
/* TickerMessage_v2.2_2009-03-27.pdf
*   Section 5.1.1.1 says:
*       The length of each Stock Symbol Message is
*       restricted to a maximum of 1024 bytes.
*
*   Notes from Alex:
*       So, there are 4 possible message types and it should
*       be enough to have 4KiB for all of them. The last message
*       type for Provider Data is less and has maximum data about
*       1/3 KiB, so 4KiB should be enough for all message type
*       with small reserve.
*
*   However, this service has historically used 20k and we'll
*   just continue using that size for now.
*/

// Bosch ID#6: Increase the payload size as discussed with SXM
#define STOCK_TICKER_OTA_BUFFER_BYTESIZE (10 * 20 * 1024)

// Defines max symbol length in chars
#define STOCK_TICKER_SYMBOL_NAME_LENGTH (8)

// Defines stock message sync sequence
#define STOCK_TICKER_MSG_SYNC_SEQUENCE "$XMMX$"
#define STOCK_TICKER_MSG_SYNC_SEQUENCE_LEN 6

/** -- Stock Message Payload Header  -- **/
#define STOCK_TICKER_MGS_HEADER_CRC_BYTELEN (1)

/** -- Stock Symbol Message (App ID 50) -- **/
#define STOCK_TICKER_MGS_SYMBOL_SEQ_NUMBER_BYTESIZE (1)
#define STOCK_TICKER_MGS_SYMBOL_SEQ_NUMBER_OFFSET_BYTESIZE (0)

#define STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE (1)
#define STOCK_TICKER_MGS_SYMBOL_COUNT_OFFSET_BYTESIZE ( \
    STOCK_TICKER_MGS_SYMBOL_SEQ_NUMBER_BYTESIZE \
        )

#define STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE (1)
#define STOCK_TICKER_MGS_SYMBOL_SIZE_OFFSET_BYTESIZE ( \
    STOCK_TICKER_MGS_SYMBOL_COUNT_OFFSET_BYTESIZE + \
    STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE \
        )

#define STOCK_TICKER_MGS_SYMBOL_INDEX_BYTESIZE (2)
#define STOCK_TICKER_MGS_SYMBOL_INDEX_BITSIZE ( \
        STOCK_TICKER_MGS_SYMBOL_INDEX_BYTESIZE * 8 \
            )
#define STOCK_TICKER_MGS_SYMBOL_INDEX_OFFSET_BITSIZE ( \
        (STOCK_TICKER_MGS_SYMBOL_SIZE_OFFSET_BYTESIZE + \
         STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE) * 8 \
            )

#define STOCK_TICKER_MGS_SYMBOL_HDR_BYTELEN ( \
        STOCK_TICKER_MGS_SYMBOL_SEQ_NUMBER_BYTESIZE + \
        STOCK_TICKER_MGS_SYMBOL_COUNT_BYTESIZE + \
        STOCK_TICKER_MGS_SYMBOL_SIZE_BYTESIZE + \
        STOCK_TICKER_MGS_SYMBOL_INDEX_BYTESIZE )

/** -- Stock Value Message (App ID 51) -- **/
#define STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_COUNT_OFFSET_BYTESIZE (0)
#define STOCK_TICKER_MGS_VALUE_INDEX_BYTESIZE (2)
#define STOCK_TICKER_MGS_VALUE_INDEX_OFFSET_BYTESIZE ( \
        STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE \
            )
#define STOCK_TICKER_MGS_VALUE_INDEX_OFFSET_BITSIZE ( \
        STOCK_TICKER_MGS_VALUE_INDEX_OFFSET_BYTESIZE * 8 \
            )
#define STOCK_TICKER_MGS_VALUE_INDEX_BITSIZE ( \
        STOCK_TICKER_MGS_VALUE_INDEX_BYTESIZE * 8 \
            )
#define STOCK_TICKER_MGS_VALUE_HDR_BYTELEN ( \
        STOCK_TICKER_MGS_VALUE_COUNT_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_INDEX_BYTESIZE )

#define STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_SYMBOL_BYTESIZE ( \
        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE \
            )
#define STOCK_TICKER_MGS_VALUE_DELTA_DIRECTION_MASK ((UN8)(1 << 7))

/** -- Stock Extended Value Message (App ID 52) -- **/
#define STOCK_TICKER_MGS_VALUE_EX_COUNT_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_EX_COUNT_OFFSET_BYTESIZE (0)
#define STOCK_TICKER_MGS_VALUE_EX_INDEX_BYTESIZE (2)
#define STOCK_TICKER_MGS_VALUE_EX_INDEX_BITSIZE ( \
        STOCK_TICKER_MGS_VALUE_EX_INDEX_BYTESIZE * 8 \
            )
#define STOCK_TICKER_MGS_VALUE_EX_HDR_BYTELEN ( \
        STOCK_TICKER_MGS_VALUE_EX_COUNT_BYTESIZE \
            )

#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BYTESIZE (3)
#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BITSIZE ( \
    STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BYTESIZE * 8 \
        )
#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE (1)
#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE (3)
#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BITSIZE ( \
    STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE * 8 \
    )
#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE (1)

#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_VALUE_BYTESIZE ( \
        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_INT_VALUE_CHANGE_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_TRADE_FRAC_VALUE_CHANGE_BYTESIZE )

#define STOCK_TICKER_MGS_VALUE_EX_SYMBOL_BYTESIZE ( \
        STOCK_TICKER_MGS_VALUE_EX_INDEX_BYTESIZE + \
        STOCK_TICKER_MGS_VALUE_EX_SYMBOL_VALUE_BYTESIZE )

#define STOCK_TICKER_MGS_VALUE_EX_DELTA_DIRECTION_MASK ((UN32)(1 << 23))

/** -- Data Provider Message Structure (App ID 53) -- **/
#define STOCK_TICKER_MGS_DATA_PROVIDER_SEQ_NUMBER_BYTESIZE (1)
#define STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_BYTESIZE (1)
#define STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_BYTESIZE (1)
#define STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_OFFSET_BYTESIZE ( \
    STOCK_TICKER_MGS_DATA_PROVIDER_SEQ_NUMBER_BYTESIZE + \
    STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_BYTESIZE \
        )
#define STOCK_TICKER_MGS_DATA_PROVIDER_HDR_BYTESIZE ( \
        STOCK_TICKER_MGS_DATA_PROVIDER_SEQ_NUMBER_BYTESIZE + \
        STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_BYTESIZE + \
        STOCK_TICKER_MGS_DATA_PROVIDER_TEXT_SIZE_BYTESIZE \
            )

// Supported data provider references
#define STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_SPORTS_FEED (0x01)
#define STOCK_TICKER_MGS_DATA_PROVIDER_REF_NUMBER_STOCK_FEED (0x32)

// Invalid sequence identifier
#define STOCK_TICKER_INVALID_SEQID ((UN32)-1)

// Max value of the seq number
#define STOCK_TICKER_SEQ_NUMBER_MAX (UN8_MAX)
#define STOCK_TICKER_SEQ_NUMBER_MOD ((UN32)(STOCK_TICKER_SEQ_NUMBER_MAX + 1))

#define STOCK_TICKER_CRC_INVERSION_VALUE (0xFFFFFFFFL)

// Defines frac value as trigger
#define STOCK_TICKER_VALUE_FRAC_DELISTED (254)
#define STOCK_TICKER_VALUE_FRAC_NA (255)

  /**************/
 /** TYPEDEFS **/
/**************/
typedef enum stocks_msg_parser_state_enum
{
    STOCKS_MSG_PARSER_STATE_EMPTY = 0,
    STOCKS_MSG_PARSER_STATE_COLLECTING,
    STOCKS_MSG_PARSER_STATE_CHECKING,
    STOCKS_MSG_PARSER_STATE_READY,
    STOCKS_MSG_PARSER_STATE_SKIPPED
} STOCKS_MSG_PARSER_STATE_ENUM;

// Supported message types (from the protocol)
typedef enum stock_ticker_msg_protocol_type_enum
{
    STOCK_TICKER_MSG_PROTOCOL_TYPE_INVALID = 0,
    STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_SYMBOL = 50,
    STOCK_TICKER_MSG_PROTOCOL_TYPE_START =
        STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_SYMBOL,
    STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_VALUE = 51,
    STOCK_TICKER_MSG_PROTOCOL_TYPE_STOCK_VALUE_EX = 52,
    STOCK_TICKER_MSG_PROTOCOL_TYPE_DATA_PROVIDER_MSG = 53,
} STOCK_TICKER_MSG_PROTOCOL_TYPE_ENUM;

typedef struct stocks_payload_header_struct
{
    // Message type
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType;

    // Payload sequence number
    UN32 un32SeqNumber;
} STOCKS_PAYLOAD_HEADER_STRUCT;

typedef struct stocks_msg_parser_data_struct
{
    // Message type
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType;

    // State
    STOCKS_MSG_PARSER_STATE_ENUM eState;

    // Keeps reference to data accumulation buffer
    OSAL_BUFFER_HDL hBuffer;

    // Keeps reference to current buffer in case of the
    // hBuffer has full data to parse and the parse data
    // in the READY stare
    OSAL_BUFFER_HDL hCurrentBuffer;

    // Keeps currently processing payload sequence number
    UN32 un32SeqNumber;

    // Keeps currently processing message
    UN32 un32MsgSeqNumber;

    // Keeps number of processed message payloads of such type
    UN32 un32PayloadPorcessedCounter;

    // Keeps sign that the data has been read from the payload which
    // is being collected
    BOOLEAN bDataIsCollected;

    // Keeps payload pre-check data. It has valid content only if
    // bDataIsCollected is TRUE
    STOCK_TICKER_PLUGIN_PRECHECK_INTERFACE_ARG_STRUCT uData;
} STOCK_TICKER_MSG_PARSER_DATA_STRUCT;

typedef struct stock_ticker_protocol_object_struct
{
    // Handle to the top-level stock ticker service
    STOCK_TICKER_SERVICE_OBJECT hStockTickerService;

    // CRC handle (used to compute ISO 3309 CRC32 based CRCs)
    OSAL_OBJECT_HDL hCRC;

    // Keeps processing data
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *apsMsgParserDataList[STOCK_TICKER_MSG_TYPE_MAX];

    //Buffer for any data/file operations
    char *pacBuffer;
    size_t tBufferSize;
} STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT;

  /************/
 /** MACROS **/
/************/

// Verified proper value for the frac part of the stock value
#define STOCK_VALUE_IS_VALID(fracValue) \
    ((((fracValue) == 0) || (fracValue <= 99)) ? TRUE : FALSE)

// Verify the value for NA
#define STOCK_VALUE_IS_NA(fracValue) \
    ((fracValue == STOCK_TICKER_VALUE_FRAC_NA) ? TRUE : FALSE)

// Verify the value for De-listed
#define STOCK_VALUE_IS_DELISTED(fracValue) \
    ((fracValue == STOCK_TICKER_VALUE_FRAC_DELISTED) ? TRUE : FALSE)

  /****************/
 /** PROTOTYPES **/
/****************/

// Public interface APIs
static STOCK_TICKER_INTERFACE_OBJECT hInit(
    STOCK_TICKER_SERVICE_OBJECT hService,
    SMS_OBJECT hParent
        );

static void vUnInit(
    STOCK_TICKER_INTERFACE_OBJECT hInterface
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessage(
    STOCK_TICKER_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

static STOCK_TICKER_RETURN_CODE_ENUM eState (
    STOCK_TICKER_INTERFACE_OBJECT hInterface
        );

// Private function

static STOCK_TICKER_RETURN_CODE_ENUM eEnsureBufferEnough(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    size_t tBufferSize
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageHeader(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload,
    STOCKS_PAYLOAD_HEADER_STRUCT *psHeader
        );

static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockSymbol (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockValue (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageStockValueEx (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessageDataProviderMsg (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eCheckMessage (
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockSymbol(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockValue(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageStockValueEx(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        );

static STOCK_TICKER_RETURN_CODE_ENUM eProcessMessageDataProviderMsg(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psData,
    BOOLEAN bDoTouch
        );

static void vUninitObject(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj
        );

static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewName(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_ID tIndex,
    const char *pacName,
    size_t tNameLength,
    BOOLEAN bDoTouch
        );

static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewValue(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_ID tIndex,
    UN32 un32Price, UN8 un8PriceFrac,
    UN32 un32PriceDelta, UN8 un8PriceDeltaFrac,
    BOOLEAN bDoesDown
        );

static STOCK_TICKER_RETURN_CODE_ENUM eNotifyManagerByNewValueViaTouch(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eType,
    STOCK_ID tIndex
        );

static STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psCreateMsgParserData(
    SMS_OBJECT hParent,
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType
        );

static STOCK_TICKER_RETURN_CODE_ENUM eAppendPayloadToMsgParserData (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psProtoObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj,
    OSAL_BUFFER_HDL hPayload
        );

static STOCK_TICKER_RETURN_CODE_ENUM eUpdateMsgParserData(
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psProtoObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj,
    OSAL_BUFFER_HDL *phPayload,
    STOCKS_PAYLOAD_HEADER_STRUCT *psHeader
        );

static void vCompleteMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static void vResetMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static void vDestroyMsgParserData(
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psObj
        );

static STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psGetMsgParserData (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_TYPE_ENUM eMsgType
        );

static STOCK_TICKER_RETURN_CODE_ENUM eIsProcessingRequired (
    STOCKS_TICKER_PROTOCOL_OBJECT_STRUCT *psObj,
    STOCK_TICKER_MSG_PARSER_DATA_STRUCT *psMsgData
        );

  /***************/
 /** VARIABLES **/
/***************/

const STOCK_TICKER_PLUGIN_INTERFACE_STRUCT GsStockTickerIntf =
{
    /*.tDSI = */ STOCK_TICKER_DSI,
    /*.tOTABufferByteSize = */ STOCK_TICKER_OTA_BUFFER_BYTESIZE,
    /*.tSymbolNameMaxSize = */ STOCK_TICKER_SYMBOL_NAME_LENGTH,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.eProcessMessage = */ eProcessMessage,
    /*.eState */ eState
};

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif /* _STOCK_TICKER_PROTOCOL_H_ */
