/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio, Inc.          */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/

#ifndef _SAFECAM_PVN1_H_
#define _SAFECAM_PVN1_H_

/**************/
/** INCLUDES **/
/**************/

#include "standard.h"
#include "osal.h"

#include <stdio.h>

#include "sms_api.h"
#include "sql_interface_obj.h"
#include "safecam_interface.h"

/***************/
/** CONSTANTS **/
/***************/

// Object Debugging:
// If DEBUG_OBJECT isn't defined
// define it locally
#ifndef DEBUG_OBJECT
#define DEBUG_OBJECT 0
#endif

// Include the debug definitions
// header, which depends on how
// DEBUG_OBJECT is defined
#include "sms_debug_definitions.h"

// PVN 1 - Definitions
#define SAFECAM1_OBJECT_NAME "SafeCam1"

#define SAFECAM1_DSI                 (620)

// The protocol version supported
// by SMS for this interface
#define SAFECAM1_PVN (PVN)(1)

// Database version
#define SAFECAM1_DB_VERSION (1)

/* Max message sizes (from SX-9845-0051)
*   Section 4.2 says:
*       The maximum size of any Access Unit is 5120 bytes.
*
*   In addition, this service has two RFD streams running,
*   and each of those has two carousels.  This means
*   we need to have enough memory to support 4 streams
*   of 5120 bytes.
*/
#define SAFECAM1_BUFFER_BYTESIZE (5120 * 4)

// Initialize our buffer needs to 0 bytes since
// all we have is RFD carousels
#define SAFECAM1_INITIAL_BUFFER_BYTESIZE (0)

//TODO find correct size
#define SAFECAM1_RFD_READ_BLOCK_SIZE (2048)

// Protocol related definitions

// General access unit definitions
#define SAFECAM1_PVN_BITLEN (4)
#define SAFECAM1_CAROUSEL_BITLEN (3)

// Carousel IDs
#define SAFECAM1_STU_CAROUSEL_ID (0)
#define SAFECAM1_LTU_CAROUSEL_ID (1)
#define SAFECAM1_METADATA_CAROUSEL_ID (2)

// Update File Metadata Header
#define SAFECAM1_SIZE_BITLEN (32)
#define SAFECAM1_CAR_BITLEN (4)

// Update File Header
#define SAFECAM1_VER_MIN_BITLEN (16)
#define SAFECAM1_FLAGS_BITLEN (8)
#define SAFECAM1_PARTITIONS_BITLEN (8)
#define SAFECAM1_UPDATE_VER_BITLEN (16)
#define SAFECAM1_PAR_INDEX_BITLEN (32)
#define SAFECAM1_FLAG_BITLEN (1)

// Alert type data
#define SAFECAM1_ALERT_TYPE_TYPE_NUM_BITLEN (7)
#define SAFECAM1_ALERT_TYPE_TUTYPE_BITLEN (2)
#define SAFECAM1_ALERT_TYPE_TYPE_BITLEN (8)
#define SAFECAM1_ALERT_TYPE_DESC_MAX_SYMBOLS (240)

#define SAFECAM1_ALERT_TYPE_TUTYPE_DELETE (0)
#define SAFECAM1_ALERT_TYPE_TUTYPE_NEW (1)
#define SAFECAM1_ALERT_TYPE_TUTYPE_MODIFY (2)

// Alert Location Update Partition Header
#define SAFECAM1_PARTITION_RCOUNT_BITLEN (11)

// Alert Location Update Region Header
#define SAFECAM1_REGION_AREGION_BITLEN (11)
#define SAFECAM1_REGION_IDSIZE_BITLEN (4)

// Alert Location Update Data
#define SAFECAM1_ALERT_LOCATION_UTYPE_BITLEN (2)
#define SAFECAM1_ALERT_LOCATION_OSIZE_BITLEN (2)
#define SAFECAM1_ALERT_LOCATION_FLAG_BITLEN (1)
#define SAFECAM1_ALERT_LOCATION_UNIT_BITLEN (1)
#define SAFECAM1_ALERT_LOCATION_SUBNO_BITLEN (4)
#define SAFECAM1_ALERT_LOCATION_SUBID_BITLEN (4)
#define SAFECAM1_ALERT_LOCATION_TYPE_BITLEN (8)
#define SAFECAM1_ALERT_LOCATION_EXTCNT_BITLEN (8)
#define SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_BITLEN (2)
#define SAFECAM1_ALERT_LOCATION_LOC_SUBDIVISION_BITLEN (12)
#define SAFECAM1_ALERT_LOCATION_LOC_DIVISION_BITLEN (8)
#define SAFECAM1_ALERT_LOCATION_LOC_SUBREGION_BITLEN (8)

// Alert location update types
#define SAFECAM1_ALERT_LOCATION_UTYPE_DEL (0)
#define SAFECAM1_ALERT_LOCATION_UTYPE_NEW (1)
#define SAFECAM1_ALERT_LOCATION_UTYPE_MOD (2)
#define SAFECAM1_ALERT_LOCATION_UTYPE_EOF (3)

// Alert location loc preamble types
#define SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_NC (0)    // no location change
#define SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_SD (1) // sub division only
#define SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_D (2)    // division and sub division
#define SAFECAM1_ALERT_LOCATION_LOC_PREAMBLE_SR (3)    // sub region, division and sub division

// Flags bit numbers
#define SAFECAM1_ALERT_UPDATE_FLAG (7)

// Alert location description field
#define SAFECAM1_ALERT_LOCATION_DESC_MAX_SYMBOLS (48)

// Macro to create Alert Location UID
#define SAFECAM1_UID_FROM_REG_ID(aRegion, Id) \
    ((((UN32)(aRegion)) << 16) | Id)

#define SAFECAM1_GET_LON_FROM_REGION(un16Region, un8SubReg, \
    un8Division, un16SubDiv) \
    (((((UN32)un16Region) & 0x07e0) << 9) |\
    ((((UN32)un8SubReg) & 0x00F0) << 6) |\
    ((((UN32)un8Division) & 0x00F0) << 2) |\
    ((((UN32)un16SubDiv) & 0x0FC0) >> 6))


#define SAFECAM1_GET_LAT_FROM_REGION(un16Region, un8SubReg, \
    un8Division, un16SubDiv) \
    (((((UN32)un16Region) & 0x001F) << 14) |\
    ((((UN32)un8SubReg) & 0x000F) << 10) |\
    ((((UN32)un8Division) & 0x000F) << 6) |\
    (((UN32)un16SubDiv) & 0x003F))

#define SAFECAM1_GET_OFFSET_BITLEN(un8Osize) \
    (10 + 4 * un8Osize)

// Sub location data
#define SAFECAM1_SUB_LOCATION_ID_BITLEN (4)
#define SAFECAM1_SUB_LOCATION_TYPE_BITLEN (8)
#define SAFECAM1_SUB_LOCATION_DIR_BITLEN (5)
#define SAFECAM1_SUB_LOCATION_SPEED_BITLEN (5)

// This is type field value in case if sublocation has to be deleted
#define SAFECAM1_SUB_LOCATION_DELETE_FLAG (0)

// As defined in rfd_receiver_config.h
#define SAFECAM1_STU_RFD_CLIENT_ID    (3)
#define SAFECAM1_LTU_RFD_CLIENT_ID    (4)

// Baudot related constants
#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_UC  0x20
#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_LC  0x20
#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LTR_FIG 0x20

#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_UC_ESC  0x20
#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_LC_ESC  0x20
#define SAFECAM1_NUM_VALID_ROWS_FOR_BAUDOT_COL_FIG_ESC 0x12

#define SAFECAM1_BAUDOT_CODE_END     0x00
#define SAFECAM1_BAUDOT_CODE_UC_LC   0x02
#define SAFECAM1_BAUDOT_CODE_FIG_LTR 0x1f
#define SAFECAM1_BAUDOT_CODE_ESC     0x1b
#define SAFECAM1_BAUDOT_CODE_CR      0x08

// Constants used in the utilization of fixed objects
#define SAFECAM1_LAT_OBJ_INDEX (0)
#define SAFECAM1_LON_OBJ_INDEX (1)
#define SAFECAM1_SUB_LAT_OBJ_INDEX (2)
#define SAFECAM1_SUB_LON_OBJ_INDEX (3)
#define SAFECAM1_MAX_OBJ_INDEX (SAFECAM1_SUB_LON_OBJ_INDEX)
#define SAFECAM1_NUM_MEMORY_FIXED (SAFECAM1_MAX_OBJ_INDEX + 1)

// Lat / Lon calculation values
#define SAFECAM1_INITIAL_LAT (12)
#define SAFECAM1_INITIAL_LON (-168)
#define SAFECAM1_LOC_DENOMINATOR (8192)
#define SAFECAM1_OFFSET_DENOMINATOR (32768)

// Speed multiplier
#define SAFECAM1_SUB_LOCATION_SPEED_MULTIPLIER (5)

// Direction multiplier
#define SAFECAM_SUB_LOCATION_DIR_MULTIPLIER (15)

// RFD file names related definitions
#define SAFECAM1_UPDATE_VER_STRING_LEN (5)
#define SAFECAM1_UPDATE_FILE_NAME_LEN (13)
#define SAFECAM1_UPDATE_FILE_START_CHAR 'U'
#define SAFECAM1_UPDATE_FILE_LTU_CHAR 'L'
#define SAFECAM1_UPDATE_FILE_STU_CHAR 'S'

// Constant which defines the bit width of the maximum file update
// version, which is then passed to RFD_INTERFACE_hConnect.
// This is used in determining the applicability of incoming
// RFD updates.
// Note: 65535 => 1111 1111 1111 1111 = 16 bits
#define SAFECAM1_MAX_VERSION_BITLEN ( 16 )

/**************/
/** TYPEDEFS **/
/**************/

// Baudot related definitions
typedef enum {
    BAUDOT_COL_NUM_LTR_UC = 0,
    BAUDOT_COL_NUM_LTR_LC,
    BAUDOT_COL_NUM_FIG,
    BAUDOT_NUM_COLUMNS  // 3
} BAUDOT_COL_NUM_ENUM;

// Structure used to group all
// fixed object handles and memory
typedef struct safecam1_fixed_calc_struct
{
    // Fixed objects created to
    // represent constant float values
    OSAL_FIXED_OBJECT hInitialLat;
    OSAL_FIXED_OBJECT hInitialLon;
    OSAL_FIXED_OBJECT hLat;
    OSAL_FIXED_OBJECT hLon;
    OSAL_FIXED_OBJECT hLocDenominator;
    OSAL_FIXED_OBJECT hOffsetDenominator;
    OSAL_FIXED_OBJECT hKmToMilesDenominator;

    // A memory segment used for temporary
    // values used to compute lat/lon
    OSAL_FIXED_OBJECT_DATA atFixedData[
        OSAL_FIXED_OBJECT_SIZE * SAFECAM1_NUM_MEMORY_FIXED];

} SAFECAM1_FIXED_CALC_STRUCT;


typedef enum safecam1_rfd_type_enum
{
    SAFECAM1_RFD_TYPE_STU = 0,
    SAFECAM1_RFD_TYPE_LTU
} SAFECAM1_RFD_TYPE_ENUM;

typedef struct safecam1_rfd_ctrl_struct
{
    // Handle to the top-level safecam service
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService;

    // Handle of RFD interface
    RFD_INTERFACE_OBJECT hRFD;

    // The block pool used to process RFD files
    OSAL_OBJECT_HDL hBlockPool;

    // The buffer used to access the block pool
    OSAL_BUFFER_HDL hBuffer;

    // Our path / connection to the DB under construction
    SQL_INTERFACE_OBJECT hDBConnection;
    STRING_OBJECT hDBPath;

    // Buffer used to generate SQL commands
    char acBuffer[SAFECAM_MAX_SQL_STRING_LENGTH];

    // current version of data in DB
    RFD_UPDATE_VERSION tCurrentVersion;

    // Progress tracking
    RFD_PROGRESS_INDEX tCurProgressIndex;
    RFD_PROGRESS_INDEX tStartProgressIndex;

    // List of update file partitions
    OSAL_OBJECT_HDL hPartsList;

    // Partition entry which is currently processed
    OSAL_LINKED_LIST_ENTRY hCurrentPartEntry;

    // Current type of processed RFD update
    SAFECAM1_RFD_TYPE_ENUM eRFDType;

    //Update number
    N32 n32UpdateNumber;

    // Structure contains all data for calculation
    SAFECAM1_FIXED_CALC_STRUCT sFixed;

    // processed file base version
    RFD_UPDATE_VERSION tFileBaseVersion;
} SAFECAM1_RFD_CTRL_STRUCT;

typedef struct safecam1_object_struct
{
    // Handle to the top-level safecam service
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService;

    // RFD connection for LTU files
    RFD_INTERFACE_OBJECT hRFDLTU;

    // RFD connection for STU files
    RFD_INTERFACE_OBJECT hRFDSTU;

    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl;

} SAFECAM1_OBJECT_STRUCT;

typedef enum safecam1_msg_type_enum
{
    SAFECAM1_MSG_STU = 0,
    SAFECAM1_MSG_LTU,
    SAFECAM1_MSG_STU_METADATA,
    SAFECAM1_MSG_LTU_METADATA,
    SAFECAM1_INVALID_MSG
} SAFECAM1_MSG_TYPE_ENUM;


// Baudot related definitions
typedef enum safecam1_baudot_col_num_enum {
    SAFECAM1_BAUDOT_COL_NUM_LTR_UC = 0,
    SAFECAM1_BAUDOT_COL_NUM_LTR_LC,
    SAFECAM1_BAUDOT_COL_NUM_FIG,
    SAFECAM1_BAUDOT_NUM_COLUMNS  // 3
} SAFECAM1_BAUDOT_COL_NUM_ENUM;


// Structure which specifies all attributes
// of a single partition
typedef struct safecam_parition_desc_struct
{
    UN32 un32ByteSeekOffset;    // Offset from file beginning in bytes
    BOOLEAN bAlertTypeUpdate;   // Alert type update flag
    UN16 un16Version;            // Partition update version
} SAFECAM1_PARTITION_DESC_STRUCT;


typedef struct safecam1_region_data_struct {
    UN16 un16Region;
    UN8 un8SubRegion;
    UN8 un8Division;
    UN16 un16SubDivision;
} SAFECAM1_REGION_DATA_STRUCT;

/************/
/** MACROS **/
/************/

/****************/
/** PROTOTYPES **/
/****************/

// Public interface APIs

static size_t tMinimumOTABufferByteSize (
    BOOLEAN bDBUpdatesEnabled
        );

static SAFECAM_INTERFACE_OBJECT hInit(
    SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
    SMS_OBJECT hParent,
    RFD_UPDATE_VERSION tCurrentVersion
        );

static void vUnInit(
    SAFECAM_INTERFACE_OBJECT hInterface
        );

static BOOLEAN bProcessMessage(
    SAFECAM_INTERFACE_OBJECT hInterface,
    OSAL_BUFFER_HDL *phPayload
        );

/* Object Private Prototypes */

static BOOLEAN bInitRFD(
    SAFECAM1_OBJECT_STRUCT *psObj,
    RFD_UPDATE_VERSION tCurrentVersion
        );

static void vUninitObject(SAFECAM1_OBJECT_STRUCT *psObj);

static BOOLEAN bCreateFixedConstants (
    SAFECAM1_FIXED_CALC_STRUCT *psFixed
        );

static void vDestroyFixedConstants (
    SAFECAM1_FIXED_CALC_STRUCT *psFixed
        );

static RFD_PROCESS_RESULT_ENUM eRFDFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg,
    SAFECAM1_RFD_TYPE_ENUM eUpdateType
        );

static RFD_FILE_STATUS_ENUM eRFDLTUNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        );

static RFD_FILE_STATUS_ENUM eRFDSTUNeeded (
    RFD_INTERFACE_OBJECT hConnection,
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptFileVersion,
    size_t tVersionBitWidth,
    void *pvCallbackArg
        );

static BOOLEAN bRFDFileVersionReader(
    const char *pcFileName,
    RFD_UPDATE_VERSION *ptVersionMin,
    RFD_UPDATE_VERSION *ptFileVersion
        );

static void vNotifyDBExpired(
    RFD_INTERFACE_OBJECT hConnection,
    void *pvCallbackArg
        );

static BOOLEAN bConnectToDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    BOOLEAN bPerformIntegrityCheck
        );

static BOOLEAN bFinalizeDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static void vDeleteDB (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl
        );

static BOOLEAN bInitRFDCtrl (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    RFD_INTERFACE_OBJECT hRFD,
    RFD_PROGRESS_INDEX tStartingIndex,
    SAFECAM1_RFD_TYPE_ENUM eUpdateType
        );

static BOOLEAN bProcessRFDHeader (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psRFDFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessUpdateGroup (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile
        );

static RFD_PROCESS_RESULT_ENUM eProcessNextRFDEntry (
    SAFECAM1_RFD_CTRL_STRUCT *psRFDCtrl,
    FILE *psFile,
    RFD_UPDATE_VERSION tUpdateVersion
        );

static RFD_PROCESS_RESULT_ENUM eRFDSTUFileProcessor (
    RFD_INTERFACE_OBJECT hConnection,
    RFD_PROCESS_STATUS_ENUM eProcessStatus,
    FILE *psRFDFile,
    RFD_UPDATE_VERSION tFileVersion,
    RFD_PROGRESS_INDEX tProgressIndex,
    void *pvCallbackArg
        );

/* Message Processors */

static SAFECAM1_MSG_TYPE_ENUM eGetMessageType(
    OSAL_BUFFER_HDL hPayload
        );

static SAFECAM1_MSG_TYPE_ENUM eGetMetadataType(
    OSAL_BUFFER_HDL hBuffer
        );


/***************/
/** VARIABLES **/
/***************/

const SAFECAM_PLUGIN_INTERFACE_STRUCT GsSafeCamIntf =
{
    /*.tDSI = */SAFECAM1_DSI,
    /*.tMaxVersionBitlen =*/SAFECAM1_MAX_VERSION_BITLEN,
    /*.tMinimumOTABufferByteSize = */tMinimumOTABufferByteSize,
    /*.hInit = */hInit,
    /*.vUnInit = */vUnInit,
    /*.bProcessMessage = */bProcessMessage,
};

/**********************/
/** INLINE FUNCTIONS **/
/**********************/

#endif /* _SAFECAM_PVN1_H_ */
